/*
 *  Polynomial weighted fitting plugin for KST.
 *  Copyright 2004, The University of British Columbia
 *  Released under the terms of the GPL.
 */

#include <stdlib.h>
#include <math.h>
#include <gsl/gsl_multifit.h>

#define XVALUES			0
#define YVALUES			1
#define WVALUES  		2

#define	YFIT				0
#define YRESIDUALS	1
#define PARAMETERS	2
#define COVARIANCE	3

extern "C" int kstfit_polynomial_weighted(
  const double *const inArrays[], 
  const int inArrayLens[],
  const double inScalars[],
  double *outArrays[], int outArrayLens[],
  double outScalars[]);

int kstfit_polynomial_weighted(
  const double *const inArrays[], 
  const int inArrayLens[],
	const double inScalars[],
	double *outArrays[], int outArrayLens[],
	double outScalars[])
{
  gsl_matrix*	pMatrixX = NULL;
  gsl_matrix* pMatrixCovariance = NULL;
  gsl_vector*	pVectorY = NULL;
  gsl_vector* pVectorWeights = NULL;
  gsl_vector* pVectorParameters = NULL;
  gsl_multifit_linear_workspace* pWork = NULL;
  double dX;
  double dY;
  double* pResult[4];
  double dChiSq = 0.0;
  int i = 0;
  int j;
  int iOrder;
  int	iStatus = 0;
  int	iLength;
  int iReturn = -1;
  
  if (inArrayLens[YVALUES] >= 2 && inArrayLens[XVALUES] >= 2) {
    iLength = inArrayLens[YVALUES];
    if( inArrayLens[XVALUES] < iLength ) {
      iLength = inArrayLens[XVALUES];
    }
    if( inArrayLens[WVALUES] < iLength ) {
      iLength = inArrayLens[WVALUES];
    }
    
    iOrder = (int)ceil( inScalars[0] );
    
    //
    // first do some sanity checks...
    //
    if( iOrder >= 0 && iLength > iOrder )
    {
      //
      // first handle the output data arrays...
      //
      for( i=0; i<2; i++ ) {
        if( outArrayLens[i] != iLength ) {
          pResult[i] = (double*)realloc( outArrays[i], iLength * sizeof( double ) );
        } else {
          pResult[i] = outArrays[i];
        }
      }
      
      //
      // now the output parameter array...
      //
      for( ; i<3; i++ ) {
        if( outArrayLens[i] != iOrder ) {
          pResult[i] = (double*)realloc( outArrays[i], iOrder * sizeof( double ) );
        } else {
          pResult[i] = outArrays[i];
        }
      }
      
      //
      // now the covariance matrix...
      //
      for( ; i<4; i++ ) {
        if( outArrayLens[i] != iOrder * iOrder ) {
          pResult[i] = (double*)realloc( outArrays[i], iOrder * iOrder * sizeof( double ) );
        } else {
          pResult[i] = outArrays[i];
        }
      }      
      
      if( pResult[0] != NULL && 
          pResult[1] != NULL && 
          pResult[2] != NULL && 
          pResult[3] != NULL )
     {
        for( i=0; i<2; i++ ) {
          outArrays[i] 		= pResult[i];
          outArrayLens[i] = iLength;
        }
        for( ; i<3; i++ ) {
          outArrays[i] 		= pResult[i];
          outArrayLens[i] = iOrder;
        }
        for( ; i<4; i++ ) {
          outArrays[i] 		= pResult[i];
          outArrayLens[i] = iOrder * iOrder;
        }
        
        //
        // create the matrices and vectors...
        //
        pMatrixX = gsl_matrix_alloc( iLength, iOrder );
        if( pMatrixX != NULL ) {
          pVectorY = gsl_vector_alloc( iLength );
          if( pVectorY != NULL ) {
            pVectorParameters = gsl_vector_alloc( iOrder );
            if( pVectorParameters != NULL ) {
              pMatrixCovariance = gsl_matrix_alloc( iOrder, iOrder );
              if( pMatrixCovariance != NULL ) {
                pWork = gsl_multifit_linear_alloc( iLength, iOrder );
                if( pWork != NULL ) {
                  pVectorWeights = gsl_vector_alloc( iLength );
                  if( pVectorWeights != NULL ) {
                    
                    //
                    // fill in the matrices and vectors...
                    //
                    for( i=0; i<iLength; i++ ) {
                      gsl_vector_set( pVectorY, i, inArrays[YVALUES][i] );
                      gsl_vector_set( pVectorWeights, i, inArrays[WVALUES][i] );
                      for( j=0; j<iOrder; j++ ) {
                        dX = pow( inArrays[XVALUES][i], (double)j );
                        gsl_matrix_set( pMatrixX, i, j, dX );
                      }
                    }

                   
                    iStatus = gsl_multifit_wlinear( pMatrixX, 
                                                    pVectorWeights,
                                                    pVectorY, 
                                                    pVectorParameters, 
                                                    pMatrixCovariance, 
                                                    &dChiSq, 
                                                    pWork );
                    if( iStatus == 0 ) {
                      //
                      // fill in the output arrays and scalars...
                      //
                      for( i=0; i<iLength; i++ ) {
                        dY = 0.0;
                        for( j=0; j<iOrder; j++ ) {
                          dY += gsl_matrix_get( pMatrixX, i, j ) * 
                                gsl_vector_get( pVectorParameters, j );
                        }
                        outArrays[YFIT][i] = dY;
                        outArrays[YRESIDUALS][i] = inArrays[YVALUES][i] - dY;
                      }
                      
                      for( i=0; i<iOrder; i++ ) {
                        outArrays[PARAMETERS][i] = gsl_vector_get( pVectorParameters, i );
                        for( j=0; j<iOrder; j++ ) {
                          outArrays[COVARIANCE][(i*iOrder)+j] = gsl_matrix_get( pMatrixCovariance, i, j );
                        }
                      }
                      
                      outScalars[0] = dChiSq;
                      
                      iReturn = 0;
                    }
                    gsl_vector_free( pVectorWeights );
                  }
                  gsl_multifit_linear_free( pWork );
                }
                gsl_matrix_free( pMatrixCovariance );          
              }
              gsl_vector_free( pVectorParameters );          
            }
            gsl_vector_free( pVectorY );          
          }
          gsl_matrix_free( pMatrixX );          
        }
      }
    }
  }
  
  return iReturn;
}
