/***************************************************************************
                          kplview.h  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Apr 24 15:14:00 MEST 1999

    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KPLVIEW_H
#define KPLVIEW_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qpoint.h>
#include <qptrlist.h>
#include <kurl.h>
#include "kplwidget.h"

class KPrinter;
class Kpl;
class KplDoc;

/**
  * Kpl main view class.
  * Provides methods for painting and erasing the page widget and generating PostScript files.
  * @author Werner Stille
  */
class KplView : public KplWidget
{
  Q_OBJECT
public:
  /**
   * Constructor.
   * @param parent pointer to parent widget.
   * @param f zoom factor.
   */
  KplView(Kpl* parent, double f);
  /** Destructor. */
  ~KplView();
  /**
   * Contains the implementation for printing functionality.
   * @param printer printer device.
   */
  void print(KPrinter* printer) const;
  /** Adjusts scrollbars and page widget size. */
  void layout();
  /** Erases page widget. */
  inline void eraseWidget() { doPaint(false); };
  /**
   * Sets zoom factor.
   * @param f zoom factor.
   */
  inline void setZoom(double f) { fZoom = f; };
  /**
   * Returns zoom factor.
   * @return zoom factor.
   */
  inline double zoom() const { return fZoom; };

public slots:
  /** Scrolls left. */
  void slotScrollLeft();
  /** Scrolls right. */
  void slotScrollRight();
  /** Scrolls up. */
  void slotScrollUp();
  /** Scrolls down. */
  void slotScrollDown();
  /** Scrolls page up. */
  void slotScrollPageUp();
  /** Scrolls page down. */
  void slotScrollPageDown();
  /** Scrolls page left. */
  void slotScrollPageLeft();
  /** Scrolls page right. */
  void slotScrollPageRight();
  /** Scrolls to left margin. */
  void slotScrollHomeLeft();
  /** Scrolls to bottom margin. */
  void slotScrollHomeDown();
  /** Scrolls to right margin. */
  void slotScrollEndRight();
  /** Scrolls to top margin. */
  void slotScrollEndUp();
  /** Paints the page widget. */
  void paintWidget();
  /** Sets scroll bar position. */
  void setScrollBars();

protected:
  /**
   * Paints or erases page widget.
   * @param paint true for painting.
   */
  void doPaint(bool paint);
  /**
   * Checks for possible drops.
   * @param e drag enter event.
   */
  virtual void contentsDragEnterEvent(QDragEnterEvent* e);
  /**
   * Handles dropped plot or data files.
   * @param e drop event.
   */
  virtual void contentsDropEvent(QDropEvent* e);
  /**
   * Looks for moveable items.
   * @param e mouse event.
   */
  virtual void contentsMousePressEvent(QMouseEvent* e);
  /**
   * Stops moving of items.
   * @param e mouse event.
   */
  virtual void contentsMouseReleaseEvent(QMouseEvent* e);
  /**
   * Moves items.
   * @param e mouse event.
   */
  virtual void contentsMouseMoveEvent(QMouseEvent* e);
  /**
   * Draws bounding rectangle.
   * @param r rectangle
   */
  void drawBoundingRect(const QRect& r);
  /**
   * Constructs rectangle given by two points.
   * @param p1 first point
   * @param p2 second point
   * @return rectangle.
   */
  QRect sortedRect(QPoint& p1, QPoint& p2);
  Kpl* app;
  KplDoc* doc;
  QPtrList<KplItem> sel;
  double fZoom, hPos, vPos;
  QPoint startPos, dragPos;

signals:
  /**
   * Emitted when changed mouse position needs updating of the status bar.
   * @param pos mouse position.
   * @param showValues true to show Values in data units.
   */
  void mouseMoved(const QPoint& pos, bool showValues);
  /** Emitted when mouse leaves the widget. */
  void mouseLeaved();
  /**
   * Emitted when Qt drop results in a list or URLs.
   * @param list pointer to list.
   */
  void urlsDropped(KURL::List* list);
  /** Emitted when right mouse button pressed to display the context menu. */
  void rightButtonPressed();
};

#endif // KPLVIEW_H
