/*
 * KMLOFax
 *
 * A utility to process facsimiles received with a modem of the
 * ELSA MicroLink Office family.
 *
 * Copyright (C) 1999,2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <qglobal.h>

#include "mlofile.h"
#include "global.h"


#define	TIFFTAG_SUBFILETYPE							254	/* F!    LONG subfile data descriptor */
#define	    FILETYPE_PAGE								0x2	/*   F!    one page of many */
#define	TIFFTAG_IMAGEWIDTH							256	/* F! ML LONG image width in pixels (1728/2048/2432) */
#define	TIFFTAG_IMAGELENGTH							257	/* F! ML LONG image height in pixels */
#define	TIFFTAG_BITSPERSAMPLE						258	/* F!    SHORT bits per channel (sample) (1) */
#define	TIFFTAG_COMPRESSION							259	/* F! ML SHORT data compression technique */
#define	    COMPRESSION_CCITTFAX3				3		/*   F! ML CCITT Group 3 fax encoding */
#define	    COMPRESSION_CCITTFAX4				4		/*         CCITT Group 4 fax encoding */
#define	TIFFTAG_PHOTOMETRIC							262	/* F! ML SHORT photometric interpretation */
#define	    PHOTOMETRIC_MINISWHITE			0		/*      ML min value is white (Default) */
#define	    PHOTOMETRIC_MINISBLACK			1		/*         min value is black */
#define	TIFFTAG_FILLORDER								266	/* F! ML SHORT data order within a byte */
#define	    FILLORDER_MSB2LSB						1		/*         most significant -> least */
#define	    FILLORDER_LSB2MSB						2		/*      ML least significant -> most */
#define	TIFFTAG_DOCUMENTNAME						269	/* F?    ASCII name of doc. image is from */
#define	TIFFTAG_IMAGEDESCRIPTION				270	/* F?    ASCII info about image */
#define	TIFFTAG_MAKE										271	/* ?? ML ASCII scanner manufacturer name (ML:"ELSA AG") */
#define	TIFFTAG_MODEL										272	/* ?? ML ASCII scanner model name/number (ML:"MicroLink ISDN Office") */
#define	TIFFTAG_STRIPOFFSETS						273	/* F! ML LONG offsets to data strips (1 offset) */
#define	TIFFTAG_ORIENTATION							274	/* F?    image orientation */
#define	    ORIENTATION_TOPLEFT					1		/*         row 0 top, col 0 lhs */
#define	TIFFTAG_SAMPLESPERPIXEL					277	/* F!    SHORT samples per pixel (1) */
#define	TIFFTAG_ROWSPERSTRIP						278	/* F!    LONG rows per strip of data (1,ImageLength) */
#define	TIFFTAG_STRIPBYTECOUNTS					279	/* F! ML LONG bytes counts for strips (1 value) */
#define	TIFFTAG_XRESOLUTION							282	/* F! ML RATIONAL pixels/resolution in x (ML:80372/394) */
#define	TIFFTAG_YRESOLUTION							283	/* F! ML RATIONAL pixels/resolution in y (ML:77000/394, F!:98,196) */
#define	TIFFTAG_PLANARCONFIG						284	/* H     storage organization */
#define	    PLANARCONFIG_CONTIG					1		/*   H     single image plane */
#define	TIFFTAG_GROUP3OPTIONS						292	/* F! ML LONG 32 flag bits */
#define	    GROUP3OPT_2DENCODING				0x1	/*         2-dimensional coding */
#define	    GROUP3OPT_FILLBITS					0x4	/*      ML fill to byte boundary */
#define	TIFFTAG_GROUP4OPTIONS						293	/*       LONG 32 flag bits */
#define	    GROUP4OPT_UNCOMPRESSED			0x2	/*         data not compressed */
#define	TIFFTAG_RESOLUTIONUNIT					296	/* F! ML SHORT units of resolutions */
#define	    RESUNIT_INCH								2		/*      ML english */
#define	    RESUNIT_CENTIMETER					3		/*         metric */
#define	TIFFTAG_PAGENUMBER							297	/* F!    SHORT/SHORT page numbers of multi-page */
#define	TIFFTAG_SOFTWARE								305	/* F! ML ASCII name & release (ML:"Ver. 1.03 vom 24.01.2001") */
#define	TIFFTAG_DATETIME								306	/* F? ML ASCII creation date and time (F!:"YYYY:MM:DD HH:MM:SS", ML:"M/D/YYYY HH:MM:SS") */
#define	TIFFTAG_ARTIST									315	/*    ML ASCII creator of image (ML:CLIP?) */
#define	TIFFTAG_HOSTCOMPUTER						316	/*    ML ASCII machine where created (ML:Rec. station ID (MSN)) */
#define	TIFFTAG_BADFAXLINES							326	/* F? ML LONG lines w/ wrong pixel count */
#define	TIFFTAG_CLEANFAXDATA						327	/* F? ML SHORT regenerated line info */
#define	    CLEANFAXDATA_CLEAN					0		/*      ML no errors detected */
#define	    CLEANFAXDATA_REGENERATED		1		/*         receiver regenerated lines */
#define	    CLEANFAXDATA_UNCLEAN				2		/*         uncorrected errors exist */
#define	TIFFTAG_CONSECUTIVEBADFAXLINES	328	/* F? ML LONG max consecutive bad lines */
#define	TIFFTAG_FAXRECVPARAMS					34908	/*       encoded Class 2 ses. parms */
#define	TIFFTAG_FAXSUBADDRESS					34909	/*    ML received SubAddr string (_sender_) */
#define	TIFFTAG_FAXRECVTIME						34910	/*       receive time (secs) */

	

#ifndef CETX
#define CETX 03
#endif

#ifndef CDLE
#define CDLE 020
#endif


#define MODE_P    0
#define MODE_H    1
#define MODE_V0   2
#define MODE_VR_1 3
#define MODE_VR_2 4
#define MODE_VR_3 5
#define MODE_VL_1 6
#define MODE_VL_2 7
#define MODE_VL_3 8



static const hcode_t white_codes[] = {
	{0x07,  4,    2}, {0x08,  4,    3}, {0x0b,  4,    4}, {0x0c,  4,    5},
	{0x0e,  4,    6}, {0x0f,  4,    7}, {0x13,  5,    8}, {0x14,  5,    9},
	{0x07,  5,   10}, {0x08,  5,   11}, {0x1b,  5,   64}, {0x12,  5,  128},
	{0x17,  6,  192}, {0x18,  6, 1664}, {0x08,  6,   12}, {0x03,  6,   13},
	{0x34,  6,   14}, {0x35,  6,   15}, {0x2a,  6,   16}, {0x2b,  6,   17},
	{0x07,  6,    1}, {0x37,  7,  256}, {0x27,  7,   18}, {0x0c,  7,   19},
	{0x08,  7,   20}, {0x17,  7,   21}, {0x03,  7,   22}, {0x04,  7,   23},
	{0x28,  7,   24}, {0x2b,  7,   25}, {0x13,  7,   26}, {0x24,  7,   27},
	{0x18,  7,   28}, {0x02,  8,   29}, {0x03,  8,   30}, {0x1a,  8,   31},
	{0x1b,  8,   32}, {0x12,  8,   33}, {0x13,  8,   34}, {0x14,  8,   35},
	{0x15,  8,   36}, {0x16,  8,   37}, {0x17,  8,   38}, {0x28,  8,   39},
	{0x29,  8,   40}, {0x2a,  8,   41}, {0x2b,  8,   42}, {0x2c,  8,   43},
	{0x2d,  8,   44}, {0x04,  8,   45}, {0x05,  8,   46}, {0x0a,  8,   47},
	{0x0b,  8,   48}, {0x52,  8,   49}, {0x53,  8,   50}, {0x54,  8,   51},
	{0x55,  8,   52}, {0x24,  8,   53}, {0x25,  8,   54}, {0x58,  8,   55},
	{0x59,  8,   56}, {0x5a,  8,   57}, {0x5b,  8,   58}, {0x4a,  8,   59},
	{0x4b,  8,   60}, {0x32,  8,   61}, {0x33,  8,   62}, {0x34,  8,   63},
	{0x35,  8,    0}, {0x36,  8,  320}, {0x37,  8,  384}, {0x64,  8,  448},
	{0x65,  8,  512}, {0x68,  8,  576}, {0x67,  8,  640}, {0xcc,  9,  704},
	{0xcd,  9,  768}, {0xd2,  9,  832}, {0xd3,  9,  896}, {0xd4,  9,  960},
	{0xd5,  9, 1024}, {0xd6,  9, 1088}, {0xd7,  9, 1152}, {0xd8,  9, 1216},
	{0xd9,  9, 1280}, {0xda,  9, 1344}, {0xdb,  9, 1408}, {0x98,  9, 1472},
	{0x99,  9, 1536}, {0x9a,  9, 1600}, {0x9b,  9, 1728},

	/* extended make up codes */
	{0x08, 11, 1792}, {0x0c, 11, 1856}, {0x0d, 11, 1920}, {0x12, 12, 1984},
	{0x13, 12, 2048}, {0x14, 12, 2112}, {0x15, 12, 2176}, {0x16, 12, 2240},
	{0x17, 12, 2304}, {0x1c, 12, 2368}, {0x1d, 12, 2432}, {0x1e, 12, 2496},
	{0x1f, 12, 2560}, {0x00,  0,    0}
};


static const hcode_t black_codes[] = {
	{0x03,  2,    2}, {0x02,  2,    3}, {0x03,  3,    4}, {0x02,  3,    1},
	{0x03,  4,    5}, {0x02,  4,    6}, {0x03,  5,    7}, {0x05,  6,    8},
	{0x04,  6,    9}, {0x04,  7,   10}, {0x05,  7,   11}, {0x07,  7,   12},
	{0x04,  8,   13}, {0x07,  8,   14}, {0x18,  9,   15}, {0x0f, 10,   64},
	{0x17, 10,   16}, {0x18, 10,   17}, {0x08, 10,   18}, {0x37, 10,    0},
	{0x67, 11,   19}, {0x68, 11,   20}, {0x6c, 11,   21}, {0x37, 11,   22},
	{0x28, 11,   23}, {0x17, 11,   24}, {0x18, 11,   25}, {0xca, 12,   26},
	{0xcb, 12,   27}, {0xcc, 12,   28}, {0xcd, 12,   29}, {0x68, 12,   30},
	{0x69, 12,   31}, {0x6a, 12,   32}, {0x6b, 12,   33}, {0xd2, 12,   34},
	{0xd3, 12,   35}, {0xd4, 12,   36}, {0xd5, 12,   37}, {0xd6, 12,   38},
	{0xd7, 12,   39}, {0x6c, 12,   40}, {0x6d, 12,   41}, {0xda, 12,   42},
	{0xdb, 12,   43}, {0x54, 12,   44}, {0x55, 12,   45}, {0x56, 12,   46},
	{0x57, 12,   47}, {0x64, 12,   48}, {0x65, 12,   49}, {0x52, 12,   50},
	{0x53, 12,   51}, {0x24, 12,   52}, {0x37, 12,   53}, {0x38, 12,   54},
	{0x27, 12,   55}, {0x28, 12,   56}, {0x58, 12,   57}, {0x59, 12,   58},
	{0x2b, 12,   59}, {0x2c, 12,   60}, {0x5a, 12,   61}, {0x66, 12,   62},
	{0x67, 12,   63}, {0xc8, 12,  128}, {0xc9, 12,  192}, {0x5b, 12,  256},
	{0x33, 12,  320}, {0x34, 12,  384}, {0x35, 12,  448}, {0x6c, 13,  512},
	{0x6d, 13,  576}, {0x4a, 13,  640}, {0x4b, 13,  704}, {0x4c, 13,  768},
	{0x4d, 13,  832}, {0x72, 13,  896}, {0x73, 13,  960}, {0x74, 13, 1024},
	{0x75, 13, 1088}, {0x76, 13, 1152}, {0x77, 13, 1216}, {0x52, 13, 1280},
	{0x53, 13, 1344}, {0x54, 13, 1408}, {0x55, 13, 1472}, {0x5a, 13, 1536},
	{0x5b, 13, 1600}, {0x64, 13, 1664}, {0x65, 13, 1728},

	/* extended make up codes */
	{0x08, 11, 1792}, {0x0c, 11, 1856}, {0x0d, 11, 1920}, {0x12, 12, 1984},
	{0x13, 12, 2048}, {0x14, 12, 2112}, {0x15, 12, 2176}, {0x16, 12, 2240},
	{0x17, 12, 2304}, {0x1c, 12, 2368}, {0x1d, 12, 2432}, {0x1e, 12, 2496},
	{0x1f, 12, 2560}, {0x00,  0,    0}
};


static const uchar white_makeup_lut[] = {
	 0, 10, 11, 12, 21, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
	80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 13, 90, 91, 92, 93, 94,
	95, 96, 97, 98, 99,100,101,102,103
};

static const uchar white_term_lut[] = {
	68, 20,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 14, 15, 16, 17,
	18, 19,	22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35,
	36, 37, 38, 39,	40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
	52, 53, 54, 55, 56, 57,	58, 59, 60, 61, 62, 63, 64, 65, 66, 67
};

static const uchar black_makeup_lut[] = {
	 0, 15, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78,
	79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94,
	95, 96, 97, 98, 99,100,101,102,103
};

static const uchar black_term_lut[] = {
	19,  3,  0,  1,  2,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,
	16, 17, 18, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32,
	33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48,
	49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64
};



static const hcode_t mode_codes[] = {
	{0x01, 1, MODE_V0  },
	{0x01, 3, MODE_H   },
	{0x03, 3, MODE_VR_1},
	{0x02, 3, MODE_VL_1},
	{0x01, 4, MODE_P   },
	{0x03, 6, MODE_VR_2},
	{0x02, 6, MODE_VL_2},
	{0x03, 7, MODE_VR_3},
	{0x02, 7, MODE_VL_3},
	{0x00, 0, 0        }
};


static Q_UINT16 swabShort(Q_UINT16 val)
{
	char src_buff[2], dst_buff[2];
	Q_UINT16 result;

	memcpy(src_buff, &val, 2);
	dst_buff[1] = src_buff[0];
	dst_buff[0] = src_buff[1];
	memcpy(&result, dst_buff, 2);

	return result;
}


static Q_UINT32 swabLong(Q_UINT32 val)
{
	char src_buff[4], dst_buff[4];
	Q_UINT32 result;

	memcpy(src_buff, &val, 4);
	dst_buff[3] = src_buff[0];
	dst_buff[2] = src_buff[1];
	dst_buff[1] = src_buff[2];
	dst_buff[0] = src_buff[3];
	memcpy(&result, dst_buff, 4);

	return result;
}




FaxFile::FaxFile(const QString &name): QFile(name)
{
	init();
}


FaxFile::FaxFile(): QFile()
{
	init();
}


FaxFile::~FaxFile()
{
}


void FaxFile::close()
{
	QFile::close();
	init();
}


void FaxFile::setPageWidth(int width)
{
	m_width = width;
	m_bpl = ((m_width-1) >> 3) + 1;
}


int FaxFile::pageHeight()
{
	int eols;

	if (m_height)
		return m_height;

	rewindPage();

	while (skipEol()) {
		eols = 1;
		if (!getDataBits(12)) {
#ifdef MLOFILE_DEBUG
			qDebug("FaxFile: Unexpected end of file in pageHeight().");
#endif
			rewindPage();
			return m_height;
		}
		while (codeMatches(0x01, 12)) {
			cutDataBits(12);
			if ((!getDataBits(12)) || (eols++ == 6)) {
				rewindPage();
				return m_height;
			}
		}
		m_height++;
	}

	rewindPage();

	return m_height;
}


int FaxFile::pageUsedHeight()
{
	int h;
	short rle_buf[MAX_RLE_BUFF];

	if (!m_usedheight) {
		if (!m_height)
			(void)pageHeight();

		for (h=0; h < m_height; h++) {
			decodeDataLine(rle_buf, true);
			if ((rle_buf[0] != -1) && (rle_buf[1] != -1))
				m_usedheight = h + 1;
		}
	}

	rewindPage();

	return m_usedheight;
}

	
void FaxFile::rewindPage()
{
	at(0);
	initPage();
}


void FaxFile::readRleLine(short line[])
{
	decodeDataLine(line, false);
}


void FaxFile::readImgLine(uchar line[], bool lsb_first)
{
	short rle_buf[MAX_RLE_BUFF], pos = 0;
	int count = 0;

	readRleLine(rle_buf);

	memset((void *)line, 0x00, m_bpl);

	while ((pos < m_width) && (rle_buf[count] != -1)) {
		pos += rle_buf[count++];
		if ((pos >= m_width) || (rle_buf[count] == -1))
			break;
		fillDataLine(line, pos, rle_buf[count], lsb_first);
		pos += rle_buf[count++];
	}
}


int FaxFile::readHuffLine(uchar line[])
{
	short rle_buf[MAX_RLE_BUFF];

	readRleLine(rle_buf);

	return encodeDataLine(line);
}
	

void FaxFile::skipLine()
{
	(void)skipEol();
}


void FaxFile::init()
{
	m_width = 0;
	m_height = 0;
	m_usedheight = 0;
	m_bpl = 0;
	m_2d = false;
	m_msbfirst = false;
	refline[2432] = -1;

	initPage();
}


void FaxFile::initPage()
{
	data_buff = 0;
	data_width = 0;
	data_left = true;
}


int FaxFile::getDataCh()
{
	int c;

	data_left = ((c = getch()) != -1);

	return (data_left && m_msbfirst) ? (int)xchg_endian[c] : c;
}
	

bool FaxFile::getDataBits(uchar len)
{
	int c;

	while (data_width < len) {
		if ((c = getDataCh()) == -1)
			return false;
		data_buff = (data_buff << 8) | (ulong)xchg_endian[c];
		data_width += 8;
	}

	return true;
}


void FaxFile::cutDataBits(uchar width)
{
	data_width -= width;
	data_buff &= ((1 << data_width) - 1);
}


bool FaxFile::codeMatches(const uchar code, const uchar width)
{
	return (ushort)(data_buff >> (data_width - width)) == (ushort)code;
}


short FaxFile::findMatch(const hcode_t table[])
{
	int i;

	for (i=0; table[i].width; i++)
		if (codeMatches(table[i].code, table[i].width)) {
			cutDataBits(table[i].width);
			return table[i].len;
		}

	return -1;
}


bool FaxFile::skipEol()
{
	if (!getDataBits(12))
		return false;

	while (data_left && (!codeMatches(0x01, 12))) {
		if ((data_buff >> (data_width - 11)) & 0x01)
			cutDataBits(11);
		else
			cutDataBits(1);
		if (!getDataBits(12))
			return false;
	}

	if (data_left)
		cutDataBits(12);
		
	return data_left;
}


void FaxFile::fillDataLine(uchar line[], short pos, short len, bool lsb_first)
{
	short end, a, b;
	uchar a_mask, b_mask;

	if (!len)
		return;

	end = pos + len - 1;

	a = pos >> 3;
	b = end >> 3;

	if (lsb_first) {
		a_mask = (uchar)(0xff <<  (pos & 0x07));
		b_mask = (uchar)(0xff >> ((end & 0x07) ^ 0x07));
	}
	else {
		a_mask = (uchar)(0xff >>  (pos & 0x07));
		b_mask = (uchar)(0xff << ((end & 0x07) ^ 0x07));
	}

	if (a == b)
		line[a] |= (a_mask & b_mask);
	else {
		line[a++] |= a_mask;
		while (a < b)
			line[a++] = 0xff;
		line[b] |= b_mask;
	}
}


short FaxFile::getRunLength(const hcode_t table[], uchar max_width)
{
	short len1, len2;

	if (!getDataBits(max_width)) {
#ifdef MLOFILE_DEBUG
		qDebug("FaxFile: Could not get bits for make-up/terminating code.");
#endif
		return -1;
	}

	if ((len1 = findMatch(table)) < 64) {
#ifdef MLOFILE_DEBUG
		if (len1 < 0)
			qDebug("FaxFile: Could not find make-up/terminating code.");
#endif
		return len1;
	}

	if (!getDataBits(max_width)) {
#ifdef MLOFILE_DEBUG
		qDebug("FaxFile: Could not get bits for terminating code.");
#endif
		return -1;
	}

	len2 = findMatch(table);

	if ((len2 == -1) || (len2 > 63)) {
#ifdef MLOFILE_DEBUG
		qDebug("FaxFile: Could not find terminating code.");
#endif
		return -1;
	}

	return len1 | len2;
}


bool FaxFile::decode1DDataLine(short run_lengths[], bool probeonly)
{
	short pos = 0;
	int count = 0;
	bool retval = true;

	while ((pos < m_width) && (count < MAX_RLE_BUFF)) {
		if ((run_lengths[count] = getRunLength(white_codes, 12)) == -1) {
			retval = false;
			break;
		}
		pos += run_lengths[count++];
		if ((pos >= m_width) || (count >= MAX_RLE_BUFF))
			break;
		if ((run_lengths[count] = getRunLength(black_codes, 13)) == -1) {
			retval = false;
			break;
		}
		if (pos + run_lengths[count] > m_width)
			run_lengths[count] = m_width - pos;
		pos += run_lengths[count++];

		if (probeonly)
			break;
	}

	if (!count)
		run_lengths[count++] = 0;

	run_lengths[count-1] += (m_width - pos);
	run_lengths[count] = -1;
		
	return retval;
}


bool FaxFile::decode2DDataLine(short run_lengths[], bool probeonly)
{
	short mode, a0 = 0, b1 = 0;
	int count = 0, ref_count = 0, len;

	memset(run_lengths, 0, MAX_RLE_BUFF * sizeof(short));

	while ((a0 < m_width) && (count < MAX_RLE_BUFF)) {
		if (!getDataBits(7)) {
			run_lengths[count] = -1;
			return false;
		}
		mode = findMatch(mode_codes);

		while (((b1 <= a0) && (b1 < m_width)) || ((count & 1) == (ref_count & 1)))
			b1 += refline[ref_count++];

		switch (mode) {
			case MODE_P:
				b1 += refline[ref_count++];
				run_lengths[count] += b1-a0;
				a0 = b1;
				break;
			case MODE_H:
				if (count & 1) {
					len = getRunLength(black_codes, 13);
					run_lengths[count++] += len;
					a0 += len;
					len = getRunLength(white_codes, 12);
				}
				else {
					len = getRunLength(white_codes, 12);
					run_lengths[count++] += len;
					a0 += len;
					len = getRunLength(black_codes, 13);
				}
				run_lengths[count++] += len;
				a0 += len;
				break;
			case MODE_V0:
				run_lengths[count++] += b1-a0;
				a0 = b1;
				break;
			case MODE_VR_1:
				run_lengths[count++] += b1+1-a0;
				a0 = b1+1;
				break;
			case MODE_VR_2:
				run_lengths[count++] += b1+2-a0;
				a0 = b1+2;
				break;
			case MODE_VR_3:
				run_lengths[count++] += b1+3-a0;
				a0 = b1+3;
				break;
			case MODE_VL_1:
				run_lengths[count++] += b1-1-a0;
				a0 = b1-1;
				b1 -= refline[--ref_count];
				break;
			case MODE_VL_2:
				run_lengths[count++] += b1-2-a0;
				a0 = b1-2;
				b1 -= refline[--ref_count];
				break;
			case MODE_VL_3:
				run_lengths[count++] += b1-3-a0;
				a0 = b1-3;
				b1 -= refline[--ref_count];
				break;
			default:
				run_lengths[count] = -1;
				return false;
		}

		if (probeonly && (count > 1))
			break;
	}

	run_lengths[count] += (m_width - a0);

	if (run_lengths[count])
		count++;

	run_lengths[count] = -1;

	return true;
}


void FaxFile::decodeDataLine(short line[], bool probeonly)
{
	bool one_dim = true, retval;

	if (!skipEol()) {
		memcpy(line, refline, sizeof(refline));
		return;
	}

	if (m_2d) {
		if (!getDataBits(1)) {
			memcpy(line, refline, sizeof(refline));
			return;
		}
		one_dim = codeMatches(0x01, 1);
		cutDataBits(1);
	}

	if (one_dim)
		retval = decode1DDataLine(line, probeonly);
	else
		retval = decode2DDataLine(line, probeonly);
	
	if (retval)
		memcpy(refline, line, sizeof(refline));
	else
		memcpy(line, refline, sizeof(refline));
}


int FaxFile::storeDataCode(uchar dest[], int pos, const hcode_t& code)
{
	ushort data_buffer;
	uchar width, *d;
	int sh, ln;

	width = code.width;
	data_buffer = (ushort)code.code << (16-width);
	d = &dest[pos >> 3];

	sh = pos & 0x07;
	ln = 8-sh;
	*d = (*d & (0xff << ln)) | (uchar)(data_buffer >> (8+sh));

	if (ln > width)
		ln = width;
	data_buffer <<= ln;
	width -= ln;

	while (width) {
		d++;
		*d = (uchar)(data_buffer >> 8);
		
		if (width > 8) {
			width -= 8;
			data_buffer <<= 8;
		}
		else
			width = 0;
	}
	
	return pos + code.width;
}


int FaxFile::encodeDataLine(uchar dest[])
{
	static const hcode_t eol = {0x01, 12, 0};
	static hcode_t fill = {0x00, 0, 0};
	int i = 0, pos = 0;
	short length;

	while (refline[i] != -1) {
		length = refline[i];
		
		if (i & 0x01) {
			if (length > 63)
				pos = storeDataCode(dest, pos, black_codes[black_makeup_lut[length >> 6]]);
			pos = storeDataCode(dest, pos, black_codes[black_term_lut[length & 0x3f]]);
		}
		else {
			if (length > 63)
				pos = storeDataCode(dest, pos, white_codes[white_makeup_lut[length >> 6]]);
			pos = storeDataCode(dest, pos, white_codes[white_term_lut[length & 0x3f]]);
		}

		i++;
	}

	if ((pos & 0x07) != 0x04) {
		fill.width = (0x0c - (pos & 0x07)) & 0x07;
		pos = storeDataCode(dest, pos, fill);
	}

	pos = storeDataCode(dest, pos, eol);

	return pos >> 3;
}




MLOFile::MLOFile(const QString &name): FaxFile(name)
{
	init();
}


MLOFile::MLOFile(): FaxFile()
{
	init();
}


MLOFile::~MLOFile()
{
}


bool MLOFile::open(int m)
{
	if (m != IO_ReadOnly)
		return false;

	return FaxFile::open(m);
}


void MLOFile::close()
{
	FaxFile::close();
	init();
}


int MLOFile::readLine(char line[], uint maxlen)
{
	uint i = 0;
	int c;

	while (data_left)
		getDataCh();
	
	while ((c = getch()) != '\n')
		if ((c == -1) || data_left) {
			line[0] = 0;
			return 0;
		}
		
	maxlen--;
	for (i=0; i < maxlen; i++) {
		c = getch();
		if ((c == '\n') || (c == -1)) {
			if (i && (line[i-1] == '\r'))
				i--;
			break;
		}
		else
			line[i] = (char)c;
	}
	line[i] = 0;

	if (!strncmp(line, "CONNECT", 7))
		data_left = true;

	return i;
}


QString MLOFile::sender()
{
	char buff[256];

	if (!m_sender.isEmpty())
		return m_sender;

	at(0);
	init();
	data_left = false;

	while (!atEnd()) {
		readLine(buff, 256);
		if (!strncmp(buff, "+FTI:", 5)) {
			char *p, *q;

			if (!(p = strchr(buff, '\"')))
				return QString::null;
			p++;
			if (!(q = strchr(p, '\"')))
				return QString::null;
			*q = 0;
			m_sender = QString(p);

			return m_sender;
		}
	}

	return QString::null;
}


int MLOFile::pages()
{
	char buff[256];

	if (m_pages)
		return m_pages;

	FaxFile::init();

	at(0);
	data_left = false;

	while (!atEnd()) {
		readLine(buff, 256);
		if (!strncmp(buff, "CONNECT", 7))
			m_pages++;
	}

	return m_pages;
}


bool MLOFile::gotoPage(int page)
{
	char buff[256];
	int p = page;

	if (m_page != page) {
		FaxFile::init();
		m_fine = false;
		m_params = 0;
		m_length = PAGE_LEN_A4;
		m_bitrate = 2400;
	}

	at(0);
	data_left = false;

	while (!atEnd()) {
		readLine(buff, 256);
		if (!strncmp(buff, "+FCS:", 5)) {
			int vr = 0, br = 0, wd = 0, ln = 0, df = 0, ec = 0, bf = 0, st = 0;
			static int ratings[6] = { 2400, 4800, 7200, 9600, 12000, 14400 };
			static int widths[5] = { 1728, 2048, 2432, 1216, 864 };

			sscanf(buff, "+FCS:%d,%d,%d,%d,%d,%d,%d,%d", &vr, &br, &wd, &ln, &df, &ec, &bf, &st);
			m_params = (vr&1)|((br&7)<<1)|((wd&7)<<4)|((ln&3)<<7)|((df&3)<<9)|((ec&1)<<11)|((bf&1)<<12)|((st&7)<<13);
			if ((vr >= 0) && (vr < 2))
				m_fine = (vr == 1);
			if ((br >= 0) && (br < 6))
				m_bitrate = ratings[br];
			if ((wd >= 0) && (wd < 5))
				setPageWidth(widths[wd]);
			if ((ln >= 0) && (ln < 3))
				m_length = ln;
			set2DCompression(df == 1);
		}
		else if (!strncmp(buff, "CONNECT", 7)) {
			if (!(--p)) {
				refline[0] = m_width;
				refline[1] = -1;
				m_page = page;
				data_left = true;
				m_datapos = at();
				return true;
			}
		}
	}

	return false;
}


void MLOFile::rewindPage()
{
	at(m_datapos);
	initPage();
}


int MLOFile::pageHeight()
{
	if (m_height)
		return m_height;

	if (!m_page)
		if (!gotoPage(1))
			return 0;

	return FaxFile::pageHeight();
}


int MLOFile::pageUsedHeight()
{
	if (m_usedheight)
		return m_usedheight;

	if (!m_page)
		if (!gotoPage(1))
			return 0;

	return FaxFile::pageUsedHeight();
}

	
void MLOFile::init()
{
	FaxFile::init();
	data_left = false;

	m_pages = 0;
	m_page = 0;
	
	m_sender = "";
	
	m_datapos = 0;
	m_fine = false;
	m_params = 0;
	m_length = 0;
	m_bitrate = 0;
}


int MLOFile::getDataCh()
{
	int c;

	while ((c = getch()) == CDLE) {
		if ((c = getch()) == CDLE)
			return c;
		if ((c == CETX) || (c == -1)) {
			data_left = false;
			return -1;
		}
	}

	if (c == -1)
		data_left = false;

	return c;
}
	


TiffFile::TiffFile(): FaxFile()
{
	init();
}


TiffFile::TiffFile(const QString &name): FaxFile(name)
{
	init();
}


TiffFile::~TiffFile()
{
}


bool TiffFile::open(int m)
{
	Q_UINT16 magic, version;
	Q_UINT32 diroff;
	
	if (!FaxFile::open(m))
		return false;

	if (size() > 0) {
		if (!readShort(&magic) || ((magic != 0x4949) && (magic != 0x4D4D))) {
			close();
			qDebug("Bad magic number!");
			return false;
		}
		m_swabbyteorder = (magic == 0x4D4D);
		if (!readShort(&version) || (version != 42)) {
			close();
			qDebug("Bad version number %d!", version);
			return false;
		}
		if (!readLong(&diroff)) {
			close();
			qDebug("Dir offset read error!");
			return false;
		}
	}
	else {
		setMsbFirst(true);
		writeShort(0x4949);
		writeShort(42);
		writeLong(0);
		flush();
	}

	return true;
}


void TiffFile::close()
{
	FaxFile::close();
	init();
}


int TiffFile::pages()
{
	if (!m_pages) {
		Q_UINT16 count;
		Q_UINT32 pos;

		if (!at(4))
			return m_pages;
		if (!readLong(&pos))
			return m_pages;
		while (pos) {
			if (!at((Offset)pos))
				return m_pages;
			if (!readShort(&count))
				return m_pages;
			if (!at(at() + (Offset)count * 12))
				return m_pages;
			if (!readLong(&pos))
				return m_pages;
			m_pages++;
		}
	}

	return m_pages;
}

	
int TiffFile::pageUsedHeight()
{
	if (m_usedheight)
		return m_usedheight;

	if (!m_page)
		if (!gotoPage(1))
			return 0;

	return FaxFile::pageUsedHeight();
}


QString TiffFile::sender()
{
	if (!m_page)
		gotoPage(1);

	return m_sender;
}


QDateTime TiffFile::time()
{
	if (!m_page)
		gotoPage(1);

	return m_time;
}


void TiffFile::addPage(int page, int pages, int width, int height, bool fine)
{
	flush();
	if (size() & 1) {
		at(size());
		putch(0);
		flush();
	}

	setPageWidth(width);
	m_height = height;

	m_pages = pages;
	m_page = page;
	m_stripoff = size();
	m_rowsperstrip = m_height;
	m_bytesperstrip = 0;
	m_fine = fine;
	m_strip = 0;
	m_row = 0;

	rewindPage();
}


bool TiffFile::finishPage()
{	
	Offset dir_off, sender_off, make_off, model_off, xres_off, yres_off, software_off, time_off;
	Q_UINT16 count;
	Q_UINT32 pos;
	static const char *make_str = "ELSA";
	static const char *model_str = "MicroLink (ISDN) Office";
	static const char *software_str = "KMLOFax Version " VERSION;
	char time_str[20], sender_str[21];

	flush();
	if (size() & 1) {
		at(size());
		putch(0);
		flush();
	}

	sprintf(time_str, "%04d:%02d:%02d %02d:%02d:%02d", m_time.date().year(), m_time.date().month(), m_time.date().day(), m_time.time().hour(), m_time.time().minute(), m_time.time().second());
	sender_str[0] = 0;
	if (!m_sender.isEmpty()) {
		strncpy(sender_str, m_sender.latin1(), 20);
		sender_str[20] = 0;
	}

	if (!at(4))
		return false;
	if (!readLong(&pos))
		return false;
	while (pos) {
		if (!at((Offset)pos))
			return false;
		if (!readShort(&count))
			return false;
		if (!at(at() + (Offset)count * 12))
			return false;
		if (!readLong(&pos))
			return false;
	}
	at(at() - 4);
	dir_off = size();
	writeLong((Q_UINT32)dir_off);
	at(dir_off);

	make_off = dir_off + 6 + 23 * 12;
	if (make_off & 1)
		make_off++;
	model_off = make_off + strlen(make_str) + 1;
	if (model_off & 1)
		model_off++;
	xres_off = model_off + strlen(model_str) + 1;
	if (xres_off & 1)
		xres_off++;
	yres_off = xres_off + 8;
	if (yres_off & 1)
		yres_off++;
	software_off = yres_off + 8;
	if (software_off & 1)
		software_off++;
	time_off = software_off + strlen(software_str) + 1;
	if (time_off & 1)
		time_off++;
	if (strlen(sender_str) < 4)
		sender_off = (int)(sender_str[0]) | ((int)(sender_str[1]) << 8) | ((int)(sender_str[2]) << 16);
	else {
		sender_off = time_off + 20;
		if (sender_off & 1)
			sender_off++;
	}

	writeShort(23);
	writeDirEntry(TIFFTAG_SUBFILETYPE, TIFF_LONG, 1, FILETYPE_PAGE);
	writeDirEntry(TIFFTAG_IMAGEWIDTH, TIFF_LONG, 1, m_width);
	writeDirEntry(TIFFTAG_IMAGELENGTH, TIFF_LONG, 1, m_height);
	writeDirEntry(TIFFTAG_BITSPERSAMPLE, TIFF_SHORT, 1, 1);
	writeDirEntry(TIFFTAG_COMPRESSION, TIFF_SHORT, 1, COMPRESSION_CCITTFAX3);
	writeDirEntry(TIFFTAG_PHOTOMETRIC, TIFF_SHORT, 1, PHOTOMETRIC_MINISWHITE);
	writeDirEntry(TIFFTAG_FILLORDER, TIFF_SHORT, 1, FILLORDER_MSB2LSB);
	writeDirEntry(TIFFTAG_MAKE, TIFF_ASCII, strlen(make_str) + 1, make_off);
	writeDirEntry(TIFFTAG_MODEL, TIFF_ASCII, strlen(model_str) + 1, model_off);
	writeDirEntry(TIFFTAG_STRIPOFFSETS, TIFF_LONG, 1, m_stripoff);
	writeDirEntry(TIFFTAG_ORIENTATION, TIFF_SHORT, 1, ORIENTATION_TOPLEFT);
	writeDirEntry(TIFFTAG_SAMPLESPERPIXEL, TIFF_SHORT, 1, 1);
	writeDirEntry(TIFFTAG_ROWSPERSTRIP, TIFF_LONG, 1, m_height);
	writeDirEntry(TIFFTAG_STRIPBYTECOUNTS, TIFF_LONG, 1, m_bytesperstrip);
	writeDirEntry(TIFFTAG_XRESOLUTION, TIFF_RATIONAL, 1, xres_off);
	writeDirEntry(TIFFTAG_YRESOLUTION, TIFF_RATIONAL, 1, yres_off);
	writeDirEntry(TIFFTAG_GROUP3OPTIONS, TIFF_LONG, 1, GROUP3OPT_FILLBITS);
	writeDirEntry(TIFFTAG_RESOLUTIONUNIT, TIFF_SHORT, 1, RESUNIT_INCH);
	writeDirEntry(TIFFTAG_PAGENUMBER, TIFF_SHORT, 2, (uint)m_page | ((uint)m_pages << 16));
	writeDirEntry(TIFFTAG_SOFTWARE, TIFF_ASCII, strlen(software_str) + 1, software_off);
	writeDirEntry(TIFFTAG_DATETIME, TIFF_ASCII, strlen(time_str) + 1, time_off);
	writeDirEntry(TIFFTAG_CLEANFAXDATA, TIFF_SHORT, 1, CLEANFAXDATA_CLEAN);
	writeDirEntry(TIFFTAG_FAXSUBADDRESS, TIFF_ASCII, strlen(sender_str) + 1, sender_off);
	writeLong(0);

	while (at() < make_off)
		putch(0);
	writeBlock(make_str, strlen(make_str) + 1);
	while (at() < model_off)
		putch(0);
	writeBlock(model_str, strlen(model_str) + 1);
	while (at() < xres_off)
		putch(0);
	writeLong(204);
	writeLong(1);
	while (at() < yres_off)
		putch(0);
	writeLong(m_fine ? 196 : 98);
	writeLong(1);
	while (at() < software_off)
		putch(0);
	writeBlock(software_str, strlen(software_str) + 1);
	while (at() < time_off)
		putch(0);
	writeBlock(time_str, 20);
	if (strlen(sender_str) > 3) {
		while (at() < sender_off)
			putch(0);
		writeBlock(sender_str, strlen(sender_str) + 1);
	}

	flush();

	return true;	
}


void TiffFile::init()
{
	m_swabbyteorder = false;
	m_pages = 0;

	m_page = 0;
	m_stripoff = 0;
	m_rowsperstrip = 0;
	m_bytesperstrip = 0;
	m_fine = false;
	m_sender = "";
	m_time = QDateTime::currentDateTime();

	m_strip = 0;
	m_row = 0;
}

	
bool TiffFile::readShort(Q_UINT16 *v)
{
	Q_UINT16 buf;

	if (readBlock((char *)&buf, 2) != 2)
		return false;

	*v = m_swabbyteorder ? swabShort(buf) : buf;

	return true;
}


bool TiffFile::writeShort(Q_UINT16 v)
{
	return writeBlock((char *)&v, 2) == 2;
}


bool TiffFile::readLong(Q_UINT32 *v)
{
	Q_UINT32 buf;

	if (readBlock((char *)&buf, 4) != 4)
		return false;

	*v = m_swabbyteorder ? swabLong(buf) : buf;

	return true;
}


bool TiffFile::writeLong(Q_UINT32 v)
{
	return writeBlock((char *)&v, 4) == 4;
}


bool TiffFile::readDirEntry(TIFFDirEntry *entry)
{
	if (!readShort(&entry->tdir_tag))
		return false;
	if (!readShort(&entry->tdir_type))
		return false;
	if (!readLong(&entry->tdir_count))
		return false;
	if (!readLong(&entry->tdir_offset))
		return false;

	return true;
}


bool TiffFile::writeDirEntry(int tag, TIFFDataType type, int count, uint offset)
{
	if (!writeShort((Q_UINT16)tag))
		return false;
	if (!writeShort((Q_UINT16)type))
		return false;
	if (!writeLong((Q_UINT32)count))
		return false;
	if (!writeLong((Q_UINT32)offset))
		return false;
	
	return true;
}


bool TiffFile::getStrField(TIFFDirEntry *entry, QString *s)
{
	char *buff;

	if (entry->tdir_type != TIFF_ASCII)
		return false;

	buff = (char *)malloc((size_t)entry->tdir_count + 1);
	buff[entry->tdir_count] = 0;

	if (entry->tdir_count > 4) {
		Offset old_pos = at();
		if (!at((Offset)entry->tdir_offset)) {
			free(buff);
			return false;
		}
		if (readBlock(buff, (uint)entry->tdir_count) < (int)entry->tdir_count) {
			free(buff);
			return false;
		}
		at(old_pos);
	}
	else
		memcpy((void *)buff, (void *)&entry->tdir_offset, (size_t)entry->tdir_count);

	*s = QString(buff);
	free(buff);

	return true;
}


bool TiffFile::getIntField(TIFFDirEntry *entry, int *i)
{
	int tsize;

	if (entry->tdir_type == TIFF_BYTE)
		tsize = 1;
	else if (entry->tdir_type == TIFF_SHORT)
		tsize = 2;
	else if (entry->tdir_type == TIFF_LONG)
		tsize = 4;
	else
		return false;

	if (entry->tdir_count == 0)
		return false;
	
	if (entry->tdir_count == 1) {
		if (m_swabbyteorder) {
			if (entry->tdir_type == TIFF_BYTE)
				*i = (int)entry->tdir_offset >> 24;
			else if (entry->tdir_type == TIFF_SHORT)
				*i = (int)entry->tdir_offset >> 16;
			else
				*i = (int)entry->tdir_offset;
		}
		else
			*i = (int)entry->tdir_offset;
	}
	else if (tsize * entry->tdir_count > 4)
		*i = (int)entry->tdir_offset;
	else
		*i = at() - 4;

	return true;
}


bool TiffFile::getDoubleField(TIFFDirEntry *entry, double *d)
{
	Q_UINT32 v1, v2;
	Offset old_pos;

	if (entry->tdir_type != TIFF_RATIONAL)
		return false;

	old_pos = at();
	if (!at((Offset)entry->tdir_offset))
		return false;
	if (!readLong(&v1))
		return false;
	if (!readLong(&v2))
		return false;
	at(old_pos);

	if (!v2)
		return false;

	*d = (double)v1 / (double)v2;

	return true;
}


bool TiffFile::gotoPage(int page)
{
	Q_UINT16 count;
	Q_UINT32 pos;
	TIFFDirEntry dir_entry;
	int p, itmp;
	double dtmp;
	QString atmp;
#ifdef MLOFILE_DEBUG
	static const char *type_nam[6] = {"NOTYPE","BYTE","ASCII","SHORT","LONG","RATIONAL"};
#endif

	if (m_page == page) {
		rewindPage();
		return true;
	}

	FaxFile::init();
	m_fine = false;
	m_stripoff = 0;
	m_rowsperstrip = 0;
	m_bytesperstrip = 0;
	m_strip = 0;
	m_row = 0;
	
	if (!at(4))
		return false;
	count = 0;
	p = page;
	while (p--) {
		if (!at(at() + (Offset)count * 12))
			return false;
		if (!readLong(&pos))
			return false;
		if (!at((Offset)pos))
			return false;
		if (!readShort(&count))
			return false;
	}

	while (count--) {
		if (!readDirEntry(&dir_entry))
			return false;

		switch (dir_entry.tdir_tag) {
			case TIFFTAG_SUBFILETYPE: // TIFF_LONG
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: SUBFILETYPE %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				if ((dir_entry.tdir_count != 1) || (itmp != FILETYPE_PAGE))
					return false; // filetype error
				break;
			case TIFFTAG_IMAGEWIDTH: // TIFF_LONG
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: IMAGEWIDTH %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				if ((dir_entry.tdir_count != 1) || (itmp < 1))
					return false; // parameter error
				setPageWidth(itmp);
				break;
			case TIFFTAG_IMAGELENGTH: // TIFF_LONG
				if (!getIntField(&dir_entry, &m_height))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: IMAGELENGTH %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, m_height);
#endif
				if ((dir_entry.tdir_count != 1) || (m_height < 1))
					return false; // parameter error
				break;
			case TIFFTAG_BITSPERSAMPLE: // TIFF_SHORT
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: BITSPERSAMPLE %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				if ((dir_entry.tdir_count != 1) || (itmp != 1))
					return false; // filetype error
				break;
			case TIFFTAG_COMPRESSION: // TIFF_SHORT
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: COMPRESSION %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				if ((dir_entry.tdir_count != 1) || (itmp != COMPRESSION_CCITTFAX3))
					return false; // filetype error
				break;
			case TIFFTAG_PHOTOMETRIC: // TIFF_SHORT
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: PHOTOMETRIC %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				if ((dir_entry.tdir_count != 1) || (itmp != PHOTOMETRIC_MINISWHITE))
					return false; // filetype error
				break;
			case TIFFTAG_FILLORDER: // TIFF_SHORT
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: FILLORDER %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				if ((dir_entry.tdir_count != 1) || ((itmp != FILLORDER_MSB2LSB) && (itmp != FILLORDER_LSB2MSB)))
					return false; // filetype error
				setMsbFirst(itmp == FILLORDER_MSB2LSB);
				break;
			case TIFFTAG_DOCUMENTNAME: // TIFF_ASCII
				if (!getStrField(&dir_entry, &atmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: DOCUMENTNAME %s %u \"%s\"", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, atmp.latin1());
#endif
				break;
			case TIFFTAG_IMAGEDESCRIPTION: // TIFF_ASCII
				if (!getStrField(&dir_entry, &atmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: IMAGEDESCIPTION %s %u \"%s\"", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, atmp.latin1());
#endif
				break;
			case TIFFTAG_MAKE: // TIFF_ASCII
				if (!getStrField(&dir_entry, &atmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: MAKE %s %u \"%s\"", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, atmp.latin1());
#endif
				break;
			case TIFFTAG_MODEL: // TIFF_ASCII
				if (!getStrField(&dir_entry, &atmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: MODEL %s %u \"%s\"", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, atmp.latin1());
#endif
				break;
			case TIFFTAG_STRIPOFFSETS: // TIFF_LONG
				if (!getIntField(&dir_entry, (int *)&m_stripoff))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: STRIPOFFSETS %s %u %ld", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, m_stripoff);
#endif
				break;
			case TIFFTAG_ORIENTATION: // TIFF_SHORT
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: ORIENTATION %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				if ((dir_entry.tdir_count != 1) || (itmp != ORIENTATION_TOPLEFT))
					return false;
				break;
			case TIFFTAG_SAMPLESPERPIXEL: // TIFF_SHORT
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: SAMPLESPERPIXEL %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				if ((dir_entry.tdir_count != 1) || (itmp != 1))
					return false; // filetype error
				break;
			case TIFFTAG_ROWSPERSTRIP: // TIFF_LONG
				if (!getIntField(&dir_entry, &m_rowsperstrip))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: ROWSPERSTRIP %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				if ((dir_entry.tdir_count != 1) || (m_rowsperstrip < 1))
					return false; // filetype error
				break;
			case TIFFTAG_STRIPBYTECOUNTS: // TIFF_LONG
				if (!getIntField(&dir_entry, &m_bytesperstrip))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: STRIPBYTECOUNTS %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, m_bytesperstrip);
#endif
				break;
			case TIFFTAG_XRESOLUTION: // TIFF_RATIONAL
				if (!getDoubleField(&dir_entry, &dtmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: XRESOLUTION %s %u %.2f", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, dtmp);
#endif
				if (dir_entry.tdir_count != 1)
					return false;
				break;
			case TIFFTAG_YRESOLUTION: // TIFF_RATIONAL
				if (!getDoubleField(&dir_entry, &dtmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: YRESOLUTION %s %u %.2f", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, dtmp);
#endif
				if (dir_entry.tdir_count != 1)
					return false;
				m_fine = dtmp > 146.0;
				break;
			case TIFFTAG_GROUP3OPTIONS: // TIFF_LONG
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: GROUP3OPTIONS %s %u 0x%04x", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				set2DCompression((itmp & GROUP3OPT_2DENCODING) != 0);
				break;
			case TIFFTAG_RESOLUTIONUNIT: // TIFF_SHORT
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: RESOLUTIONUNIT %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				if ((dir_entry.tdir_count != 1) || (itmp != RESUNIT_INCH))
					return false; // filetype error
				break;
			case TIFFTAG_PAGENUMBER: // TIFF_SHORT
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: PAGENUMBER %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				break;
			case TIFFTAG_SOFTWARE: // TIFF_ASCII
				if (!getStrField(&dir_entry, &atmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: SOFTWARE %s %u \"%s\"", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, atmp.latin1());
#endif
				break;
			case TIFFTAG_DATETIME: // TIFF_ASCII
				int year, month, day, hour, minute, second;
				if (!getStrField(&dir_entry, &atmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: DATETIME %s %u \"%s\"", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, atmp.latin1());
#endif
				if (sscanf(atmp.latin1(), "%d:%d:%d %d:%d:%d", &year, &month, &day, &hour, &minute, &second) == 6) {
					m_time.setDate(QDate(year, month, day));
					m_time.setTime(QTime(hour, minute, second));
				}
				else if (sscanf(atmp.latin1(), "%d/%d/%d %d:%d:%d", &month, &day, &year, &hour, &minute, &second) == 6) {
					m_time.setDate(QDate(year, month, day));
					m_time.setTime(QTime(hour, minute, second));
				}
				break;
			case TIFFTAG_ARTIST: // TIFF_ASCII
				if (!getStrField(&dir_entry, &atmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: ARTIST %s %u \"%s\"", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, atmp.latin1());
#endif
				break;
			case TIFFTAG_HOSTCOMPUTER: // TIFF_ASCII
				if (!getStrField(&dir_entry, &atmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: HOSTCOMPUTER %s %u \"%s\"", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, atmp.latin1());
#endif
				break;
			case TIFFTAG_BADFAXLINES: // TIFF_LONG
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: BADFAXLINES %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				break;
			case TIFFTAG_CLEANFAXDATA: // TIFF_SHORT
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: CLEANFAXDATA %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				break;
			case TIFFTAG_CONSECUTIVEBADFAXLINES: // TIFF_LONG
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: CONSECUTIVEBADFAXLINES %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				break;
			case TIFFTAG_FAXRECVPARAMS: // TIFF_LONG
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: FAXRECVPARAMS %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				break;
			case TIFFTAG_FAXSUBADDRESS: // TIFF_ASCII
				if (!getStrField(&dir_entry, &m_sender))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: FAXSUBADDRESS %s %u \"%s\"", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, m_sender.latin1());
#endif
				break;
			case TIFFTAG_FAXRECVTIME: // TIFF_LONG
				if (!getIntField(&dir_entry, &itmp))
					return false;
#ifdef MLOFILE_DEBUG
				qDebug("Tag: FAXRECVTIME %s %u %d", type_nam[dir_entry.tdir_type], dir_entry.tdir_count, itmp);
#endif
				break;
			default:
#ifdef MLOFILE_DEBUG
				qDebug("Unhandled tag: %u %s %u %u", dir_entry.tdir_tag, type_nam[dir_entry.tdir_type], dir_entry.tdir_count, dir_entry.tdir_offset);
#endif
				break;
		}
	}

	if (!m_width || !m_height || !m_stripoff)
		return false;

	if (!m_rowsperstrip)
		m_rowsperstrip = m_height;

	m_page = page;

	rewindPage();

	return true;
}


void TiffFile::readImgLine(uchar line[], bool lsb_first)
{
	FaxFile::readImgLine(line, lsb_first);

	if (++m_row >= m_rowsperstrip) {
		m_strip++;
		gotoStrip();
	}
}


void TiffFile::rewindPage()
{
	m_strip = 0;
	gotoStrip();
}


bool TiffFile::gotoStrip()
{
	Q_UINT32 pos;

	if (m_strip * m_rowsperstrip >= m_height)
		return false;

	if (m_rowsperstrip == m_height)
		return at(m_stripoff);

	if (!at(m_stripoff + 4 * m_strip))
		return false;
	
	if (!readLong(&pos))
		return false;

	if (!at((Offset)pos))
		return false;

	m_row = 0;

	initPage();

	return true;
}


void TiffFile::writeRleLine(short line[])
{
	static hcode_t fill = {0x00, 0, 0};
	int i = 0, pos;
	short length;
	uchar dest[MAX_HUF_BUFF];
	
	dest[0] = 0;
	dest[1] = 0x01;
	pos = 16;

	for (i=0; line[i] != -1; i++) {
		length = line[i];
		if (i & 0x01) {
			if (length >= 0x40)
				pos = storeDataCode(dest, pos, black_codes[black_makeup_lut[length >> 6]]);
			pos = storeDataCode(dest, pos, black_codes[black_term_lut[length & 0x3f]]);
		}
		else {
			if (length >= 0x40)
				pos = storeDataCode(dest, pos, white_codes[white_makeup_lut[length >> 6]]);
			pos = storeDataCode(dest, pos, white_codes[white_term_lut[length & 0x3f]]);
		}
	}

	if (pos & 0x07) {
		fill.width = 0x08 - (pos & 0x07);
		pos = storeDataCode(dest, pos, fill);
	}

	pos >>= 3;

	if (!m_msbfirst)
		for (i=0; i < pos; i++)
			dest[i] = xchg_endian[dest[i]];

	writeBlock((char *)dest, (uint)pos);

	m_bytesperstrip += pos;
	m_row++;
}
