#ifndef __k2config_h__
#define __k2config_h__

#include <string.h>

#include <string>
#include <list>
#include <iostream.h>
#include <map>

class K2Config;

K2Config* k2config_parse( const char *_str );

class K2ConfigBase
{
public:
  K2ConfigBase() { m_pNext = 0L; m_pPrev = 0L; }
  virtual ~K2ConfigBase() { };

  virtual bool isA( const char *_t ) { if ( strcmp( _t, "K2ConfigBase" ) == 0 ) return true; return false; }

  virtual void setNext( K2ConfigBase *_i ) { m_pNext = _i; }
  virtual void setPrev( K2ConfigBase *_i ) { m_pPrev = _i; }
  virtual K2ConfigBase* next() { return m_pNext; }
  virtual K2ConfigBase* prev() { return m_pPrev; }

  virtual void save( ostream&, int _indent = 0 ) = 0L;
  
  void deleteAll() { if ( m_pNext ) delete m_pNext; delete this; }

protected:
  K2ConfigBase* m_pNext;
  K2ConfigBase* m_pPrev;
};

class K2ConfigTranslation : public K2ConfigBase
{
public:
  K2ConfigTranslation( const char *_lang, const char *_text, K2ConfigTranslation* _next )
                 { m_strLang = _lang; m_strText = _text; m_pNext = _next; }
  ~K2ConfigTranslation() { }

  const char* lang() { return m_strLang.c_str(); }
  const char* text() { return m_strText.c_str(); }

  virtual bool isA( const char *_t ) { if ( strcmp( _t, "K2ConfigTranslation" ) == 0 ) return true; return K2ConfigBase::isA( _t ); }

  K2ConfigTranslation* nextTranslation() { return static_cast<K2ConfigTranslation*>(m_pNext); }
  
  // HACK
  virtual void save( ostream&, int _indent = 0 ) { };

protected:
  string m_strLang;
  string m_strText;
};

class K2ConfigItem : public K2ConfigBase
{  
public:
  enum ItemType { TBOOLEAN = 1, TSTRING = 2, TLONG = 3, TFLOAT = 4, TFILE = 5, TSTRING_LIST = 6  };

  struct Value
  {
    Value() { }
    Value( const Value& _v ) { m_stringList = _v.m_stringList; m_strText = _v.m_strText; m_i = _v.m_i; m_b = _v.m_b; m_f = _v.m_f; }

    const char* text() { return m_strText.c_str(); }
    const char* fileName() { return m_strText.c_str(); }
    bool fileContent( string& _str );
    int integer() { return m_i; }
    float fp() { return m_f; }
    bool boolean() { return m_b; }
    list<string>& stringList() { return m_stringList; }

    list<string> m_stringList;
    string m_strText;
    int m_i;
    bool m_b;
    float m_f;
  };

  K2ConfigItem( const char *_name, list<string>* _lst, const char *_lang = "C" );
  K2ConfigItem( const char *_name, const char *_str, ItemType _type = TSTRING, const char *_lang = "C" );
  K2ConfigItem( const char *_name, bool _b, const char *_lang = "C" );
  K2ConfigItem( const char *_name, int _i, const char *_lang = "C" );
  K2ConfigItem( const char *_name, float _f, const char *_lang = "C" );
  ~K2ConfigItem() { }

  const char* name() { return m_strName.c_str(); }
  ItemType type() { return m_iType; }

  map<string,Value>::iterator i18n_begin() { return m_mapValues.begin(); }
  map<string,Value>::iterator i18n_end() { return m_mapValues.end(); }
  map<string,Value>::iterator i18n_find( const char *_lang ) { return m_mapValues.find( _lang ); }
  
  const char* text();
  const char* fileName() { return text(); }
  bool fileContent( string& _str );
  int integer();
  float fp();
  bool boolean();
  list<string>& stringList();

  /**
   * For internal use only
   */
  void unite( K2ConfigItem *_item );
  
  virtual bool isA( const char *_t ) { if ( strcmp( _t, "K2ConfigItem" ) == 0 ) return true; return K2ConfigBase::isA( _t ); }

  virtual void save( ostream&, int _indent = 0 );
  
protected:
  map<string,Value>::iterator findBestLang();
  
  map<string,Value> m_mapValues;
  string m_strName;
  ItemType m_iType;
};

class K2Config : public K2ConfigBase
{
public:
  struct iterator
  {
    K2ConfigBase* m_p;
    
    iterator() : m_p( 0L ) { }
    iterator( K2ConfigBase* _b ) : m_p( _b ) { }
    iterator( const iterator& _it ) : m_p( _it.m_p ) { }
    
    K2ConfigBase& operator*() { return *m_p; }
    K2ConfigBase* operator->() { return m_p; }
    bool operator==( const iterator& _it ) { return ( m_p == _it.m_p ); }
    bool operator!=( const iterator& _it ) { return ( m_p != _it.m_p ); }
    iterator& operator++() { if ( m_p ) m_p = m_p->next(); return *this; }
    iterator& operator++(int) { if ( m_p ) m_p = m_p->next(); return *this; }
    iterator& operator--() { if ( m_p && m_p->prev() ) m_p = m_p->prev(); return *this; }
    iterator& operator--(int) { if ( m_p && m_p->prev() ) m_p = m_p->prev(); return *this; }
    
    K2Config* group() { if ( m_p && m_p->isA( "K2Config" ) ) return static_cast<K2Config*>(m_p); else return 0L; }
    K2ConfigItem* item() { if ( m_p && m_p->isA( "K2ConfigItem" ) ) return static_cast<K2ConfigItem*>(m_p); else return 0L; }
  };

  typedef map<string,string>::iterator i18n_iterator;
  i18n_iterator i18n_begin() { return m_mapNames.begin(); }
  i18n_iterator i18n_end() { return m_mapNames.end(); }
  
  /**
   * Creates a toplevel group, this means that it has no name and type.
   */
  K2Config( const char *_filename );
  /**
   * Creates a toplevel group, this means that it has no name and type.
   */
  K2Config() { m_pTranslation = 0L; m_pEntries = 0L; m_errno = 0; }
  /**
   * @param _name If this is 0L or an empty string then the group becomes a
   *              toplevel group.
   */
  K2Config( const char *_name, const char *_type, K2ConfigBase* _entries = 0L, K2ConfigTranslation* _trans = 0L )
  { m_strType = _type; m_strName = _name; m_pEntries = _entries; m_pTranslation = _trans; m_errno = 0; }
  ~K2Config() { if ( m_pEntries ) m_pEntries->deleteAll(); if ( m_pTranslation ) m_pTranslation->deleteAll(); }

  void parse( const char *_str );
  
  iterator begin() { return iterator( m_pEntries ); }
  iterator end() { return iterator( 0L ); }

  void insert( iterator, K2ConfigBase* );
  void insert( K2ConfigBase* _base ) { insert( begin(), _base ); }
  /**
   * @return creates a new group at the given position. If a group of the same
   *         name and type does already exist, then the existing group is returned.
   *         In this case '_it' is ignored.
   *         If an group of this name but different type exists or if an item
   *         of this name exists, then 0L is returned.
   */
  K2Config* insertGroup( iterator _it, const char *_name, const char *_type );
  /**
   * A convenience function if one does not care about the order of the items.
   * This function is the fastest one. Dont rely on the fact that it is right now
   * the same as 'prependGroup'.
   */
  K2Config* insertGroup( const char *_name, const char *_type ) { return insertGroup( begin(), _name, _type ); }
  K2Config* prependGroup( const char *_name, const char *_type ) { return insertGroup( begin(), _name, _type ); }
  K2Config* appendGroup( const char *_name, const char *_type ) { return insertGroup( end(), _name, _type ); }

  iterator find( const char *_name, iterator _it );
  K2Config* findGroup( const char *_name, iterator _it ) {  iterator it = find( _name, _it ); return it.group(); }
  K2ConfigItem* findItem( const char *_name, iterator _it ) {  iterator it = find( _name, _it ); return it.item(); }

  iterator find( const char *_name ) { return find( _name, begin() ); }
  K2Config* findGroup( const char *_name ) { iterator it = find( _name ); return it.group(); }
  K2ConfigItem* findItem( const char *_name ) { iterator it = find( _name ); return it.item(); }
  
  void erase( iterator );
  void clear();
  
  void writeLong( const char *_name, int _l );
  void writeFloat( const char *_name, float _f );
  void writeBool( const char *_name, bool _b );
  void writeString( const char *_name, const char *_s );
  void writeFile( const char *_name, const char *_s );
  void writeStringList( const char *_name, list<string>& _lst );

  void writeLong( string& _name, int _l ) { writeLong( _name.c_str(), _l ); }
  void writeFloat( string& _name, float _f ) { writeFloat( _name.c_str(), _f ); }
  void writeBool( string& _name, bool _b ) { writeBool( _name.c_str(), _b ); }
  void writeString( string& _name, const char *_s ) { writeString( _name.c_str(), _s ); }
  void writeFile( string& _name, const char *_s ) { writeFile( _name.c_str(), _s ); }
  void writeStringList( string& _name, list<string>& _lst ) { writeStringList( _name.c_str(), _lst ); }

  void appendStringList( const char *_name, const char *_value );
  void appendStringList( string& _name, const char *_value ) { appendStringList( _name.c_str(), _value ); }

  bool readLong( const char *_name, int& _l );
  bool readFloat( const char *_name, float& _f );
  bool readBool( const char *_name, bool& _b );
  bool readString( const char *_name, string& _s );
  bool readStringList( const char *_name, list<string>& _s );
  
  /**
   * @return the amount of items and sub-groups in this group.
   */
  int size();
  
  const char* name() { return m_strName.c_str(); }
  const char* type() { return m_strType.c_str(); }
  void setName( const char *_n ) { m_strName = _n; }
  void setType( const char *_t ) { m_strType = _t; }

  virtual bool isA( const char *_t ) { if ( strcmp( _t, "K2Config" ) == 0 ) return true; return K2ConfigBase::isA( _t ); }

  int error() { return m_errno; }

  /**
   * If the object is a toplevel group, then it just saves its children.
   */
  virtual void save( ostream&, int _indent = 0 );
  virtual bool save();
  
protected:
  void makeI18N();

  struct intern_i18n_iterator
  {
    K2ConfigTranslation* m_p;

    intern_i18n_iterator() : m_p( 0L ) { }
    intern_i18n_iterator( K2ConfigTranslation* _b ) : m_p( _b ) { }
    intern_i18n_iterator( const intern_i18n_iterator& _it ) : m_p( _it.m_p ) { }
    
    K2ConfigTranslation& operator*() { return *m_p; }
    K2ConfigTranslation* operator->() { return m_p; }
    bool operator==( const intern_i18n_iterator& _it ) { return ( m_p == _it.m_p ); }
    bool operator!=( const intern_i18n_iterator& _it ) { return ( m_p != _it.m_p ); }
    intern_i18n_iterator& operator++() { if ( m_p ) m_p = static_cast<K2ConfigTranslation*>(m_p->next()); return *this; }
    intern_i18n_iterator& operator++(int) { if ( m_p ) m_p = static_cast<K2ConfigTranslation*>(m_p->next()); return *this; }
  };
  intern_i18n_iterator intern_i18n_begin() { return intern_i18n_iterator( m_pTranslation ); }
  intern_i18n_iterator intern_i18n_end() { return intern_i18n_iterator( 0L ); }
  
  string m_strName;
  map<string,string> m_mapNames;
  string m_strType;
  K2ConfigTranslation* m_pTranslation;
  K2ConfigBase* m_pEntries;
  string m_strFileName;

  int m_errno;
};

ostream& operator<<( ostream& _str, K2ConfigBase& _base );


#endif
