#include <iostream.h>

#include <qfiledlg.h>
#include <qfileinf.h>
#include <qlayout.h>
#include <qmsgbox.h>
#include <qpushbt.h>

#include "theme.h"
#include "theme_exceptions.h"
#include "ktheme.h"
#include "ktheme.moc"


KTheme::KTheme(QWidget *parent, const char *name)
    : KConfigWidget(parent, name)
{
    kthemeConfig_ = kapp->getConfig();

    loadSettings();
    initUI();
}

KTheme::~KTheme() 
{
    delete globalListBox_;
    delete localListBox_;
//    delete panner_;
}

void KTheme::initUI()
{
    QBoxLayout *topLayout = new QHBoxLayout(this, 5);

    QBoxLayout *upperLayout = new QVBoxLayout();
    topLayout->addLayout(upperLayout, 10);

    //-- Themes Label --//
    QLabel *Global_Label = new QLabel( this, "Label_1" );
    Global_Label->setText(i18n("Global:"));
    Global_Label->setFixedSize(Global_Label->sizeHint());
    QLabel *Local_Label = new QLabel( this, "Label_2" );
    Local_Label->setText(i18n("Local:"));
    Local_Label->setFixedSize(Local_Label->sizeHint());


    //-- Themes ListBox --//
    globalListBox_ = new QListBox(this, "globalListBox");
    globalListBox_->insertStrList(&theGlobalThemes_);
    connect(globalListBox_, SIGNAL(highlighted(int)),
            this,           SLOT(listBoxItemHighlighted()));

    localListBox_ = new QListBox(this, "localListBox");
    localListBox_->insertStrList(&theLocalThemes_);
    connect(localListBox_, SIGNAL(highlighted(int)),
	    this,          SLOT(listBoxItemHighlighted()));

    upperLayout->addWidget(Global_Label, 0, AlignLeft);
    upperLayout->addWidget(globalListBox_, 10);
    upperLayout->addWidget(Local_Label, 0, AlignLeft);
    upperLayout->addWidget(localListBox_, 10);

    if (!lastTheme_.isEmpty()) {
	int activeIndex;

	if (lastThemeGlobal_) {
	    if ((activeIndex = theGlobalThemes_.find(lastTheme_)) != -1)
		globalListBox_->setCurrentItem(activeIndex);
	} else {
	    if ((activeIndex = theLocalThemes_.find(lastTheme_)) != -1)
		localListBox_->setCurrentItem(activeIndex);
	}      
    }

    //-! theme listbox !-//

    QBoxLayout *buttonLayout = new QVBoxLayout();
    topLayout->addLayout(buttonLayout, 2);

    //-- Add/Save/Remove Buttons --//
    QPushButton *addLocalButton = new QPushButton(this, "addLocalButton");
    addLocalButton->setText(i18n("&Add Local"));
    addLocalButton->setMinimumSize(addLocalButton->sizeHint());
    connect (addLocalButton, SIGNAL(clicked()), this, SLOT(addLocalSelected()));

    QPushButton *addGlobalButton = new QPushButton(this, "addGlobalButton");
    addGlobalButton->setText(i18n("Add &Global"));
    addGlobalButton->setMinimumSize(addGlobalButton->sizeHint());
    if (QFileInfo(Theme::GLOBAL_THEMES_DIR).isWritable()) {
    	connect(addGlobalButton, SIGNAL(clicked()),
	       	this,            SLOT(addGlobalSelected()));
    } else {
	addGlobalButton->setEnabled(false);
    }

    buttonLayout->addWidget(addGlobalButton);
    buttonLayout->addWidget(addLocalButton);

    QPushButton *saveButton = new QPushButton(this, "saveButton");
    saveButton->setText(i18n("&Save"));
    saveButton->setMinimumSize(saveButton->sizeHint());
    buttonLayout->addWidget(saveButton);

    QPushButton *removeButton = new QPushButton(this, "removeButton");
    removeButton->setText(i18n("&Remove"));
    removeButton->setMinimumSize(removeButton->sizeHint());
    buttonLayout->addWidget(removeButton);

    //-! add/save/remove buttons !-//

    buttonLayout->addStretch(10);

//    topLayout->addStretch(1);
    topLayout->activate();
}

void KTheme::addLocalSelected()
{
    const QString selectedTheme = QFileDialog::getOpenFileName(lastDir_, 
                                                               "*.tgz", this);

    if (!selectedTheme.isEmpty()) {
	installTheme(selectedTheme, false);
	lastDir_ = QFileInfo(selectedTheme).dirPath(TRUE);
    }
}

void KTheme::addGlobalSelected()
{
    const QString selectedTheme = QFileDialog::getOpenFileName(lastDir_,
                                                               "*.tgz", this);

    if (!selectedTheme.isEmpty()) {
	installTheme(selectedTheme, true);
	lastDir_ = QFileInfo(selectedTheme).dirPath(TRUE);
    }
}

void KTheme::listBoxItemHighlighted()
{
    if (sender() == globalListBox_)
	localListBox_->clearSelection();
    else
	globalListBox_->clearSelection();
}

void KTheme::installTheme(const QString &fileName, bool isGlobal)
{
    QString themeName;
    
    try {
	themeName = Theme::install(fileName, isGlobal);

	addTheme(themeName, isGlobal);
    } catch(BadFormat) {
	QMessageBox::warning(0, i18n("ktheme"),
                             i18n("This doesn't appear to be a theme file.\n"),
                             i18n("Ok"));  
    } catch(ThemeErr) { }
}

void KTheme::addTheme(const QString &themeName, bool isGlobal)
{
    if (isGlobal) {
	theGlobalThemes_.inSort(themeName);
	globalListBox_->inSort(themeName);
    } else {
	theLocalThemes_.inSort(themeName);
	localListBox_->inSort(themeName);
    }
}

void KTheme::loadSettings()
{
    kthemeConfig_->setGroup("General");
    lastDir_         = kthemeConfig_->readEntry    ("LastDir", "");
    lastTheme_       = kthemeConfig_->readEntry    ("LastTheme", "");
    lastThemeGlobal_ = kthemeConfig_->readBoolEntry("LastThemeGlobal", false);
    pannerSepPos_    = kthemeConfig_->readNumEntry ("SeparatorPosPercent", 35);
    
    QDir *themesDir = new QDir(Theme::LOCAL_THEMES_DIR);

    if (!themesDir->exists())
	themesDir->mkdir(Theme::LOCAL_THEMES_DIR);

    theLocalThemes_ = *(themesDir->entryList(QDir::Dirs, QDir::Name));
    theLocalThemes_.remove(theLocalThemes_.find("."));
    theLocalThemes_.remove(theLocalThemes_.find(".."));

    delete themesDir;
    themesDir = new QDir(Theme::GLOBAL_THEMES_DIR);

    if (!themesDir->exists())
        // Most users won't be able to create the dir, but we don't need
	// to worry if it doesn't succeed.
	themesDir->mkdir(Theme::GLOBAL_THEMES_DIR);

    if (themesDir->exists()) {
	theGlobalThemes_ = *(themesDir->entryList(QDir::Dirs, QDir::Name));
	theGlobalThemes_.remove(theGlobalThemes_.find("."));
	theGlobalThemes_.remove(theGlobalThemes_.find(".."));
    }       
}

void KTheme::applySettings()
{
    if (globalListBox_->currentItem() != -1) {
	Theme::activate(globalListBox_->text(globalListBox_->currentItem()),
					     true);
	lastTheme_ = globalListBox_->text(globalListBox_->currentItem());
	lastThemeGlobal_ = true;
    } else if (localListBox_->currentItem() != -1) {
	Theme::activate(localListBox_->text(localListBox_->currentItem()),
					    false);
	lastTheme_ = localListBox_->text(localListBox_->currentItem());
	lastThemeGlobal_ = false;
    } else
	return;

    writeSettings();
}

void KTheme::writeSettings()
{
    kthemeConfig_->setGroup("General");

    kthemeConfig_->writeEntry("LastTheme",    lastTheme_);
    kthemeConfig_->writeEntry("LastThemeGlobal", lastThemeGlobal_ ? "true"
								  : "false");
    kthemeConfig_->writeEntry("LastDir",      lastDir_);
    kthemeConfig_->writeEntry("SeparatorPosPercent", panner_->separatorPos());

    kthemeConfig_->sync();
}
