// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_FUNCTOR_H
#define KROT_FUNCTOR_H



#include <functional>


/**
 * Safe square of a single value
 */
template< typename T >
inline T sqr( const T& x )
{
    return ( x == 0 ) ? 0 : x * x;
}

/**
 * Safe square of a single int
 */
template<>
inline int sqr<>( const int& x )
{
    return x * x;
}

/**
 * Safe square of a single long
 */
template<>
inline long sqr<>( const long& x )
{
    return x * x;
}


template< typename T >
inline T cube( const T& x )
{
    return ( x == 0 ) ? 0 : x * x * x;
}



/**
 * Calculate the sum of successive calls by plain addition :-)
 *
 * @see CorrectSum
 * @see KahanSum
 */
template< typename T >
class Sum : unary_function< T, T >
{
public:

    /**
     * Constructor
     *
     * *param init Initial value to start summing upon.
     */
    Sum( const T& init=0 ) : res( init ) {};

    /**
     * Add value.
     *
     * @param val Value to add to sum.
     */
    void operator()( const T& val ) { res += val; };

    /**
     * *return Current sum.
     */
    T result() const { return res; };

private:
    
    T res;
};



/**
 * Calculate the sum of successive calls by a more correct algorithm. This uses
 * Kahans trick, as described by e.g. berhuber: Computernumerik 1, p. 234
 */
template< typename T >
class KahanSum : unary_function< T, T >
{
public:
    
    KahanSum( const T& i=0 ) : res( i ) { err = 0; };

    void operator()( const T& x ) {
	err += x;
	T tmp = res + err;
	err = ( res - tmp ) + err;
	res = tmp;
    };
    
    /**
     * *return Current sum.
     */
    T result() const { return res; };

private:
    
    T res, err;
};



/**
 * Calculate the sum of successive calls by a more correct algorithm.
 * This uses Kahans trick ( @ref KahanSum ) in general, but has specializations
 * for float and double if double or long double, respectivly do have a higher
 * precision - so we get higher precision while still operating in hardware -
 * hopefully, better tests for that might be needed.
 */
template< typename T >
class CorrectSum : KahanSum< T >
{
    // This is the gerneral stub - just an other name for KahanSum
};



#ifdef HAVE_LONG_DOUBLE
#if ( SIZEOF_DOUBLE < SIZEOF_LONG_DOUBLE )
// Calculate the sum of double values of successive calls by using extended
// precision. This is a specialization of the general template CorrectSum
// for floats.
template<>
class CorrectSum< double > : unary_function< double, double >
{
public:
    
    CorrectSum( const double& i=0 ) : res( i ) {};

    void operator()( const double& x ) { res += x; };
    
    /**
     * *return Current sum.
     */
    double result() const { return res; };

private:
    
    long double res;
};
#endif
#endif



#if ( SIZEOF_FLOAT < SIZEOF_DOUBLE )
// Calculate the sum of float values of successive calls by using double
// precision. This is a specialization of the general template CorrectSum
// for floats.
template<>
class CorrectSum< float > : unary_function< float, float >
{
public:
    
    CorrectSum( const float& i=0 ) : res( i ) {};

    void operator()( const float& x ) { res += x; };
    
    /**
     * *return Current sum.
     */
    float result() const { return res; };

private:
    
    float res;
};
#endif



/**
 * Sum up chi2-values of two variables
 * chi^2 = ( a - b )^2
 */
template< typename T >
class Chi2 : binary_function< T, T, T >
{
public:    
    T operator()( const T& a, const T& b ) { return ( 0 == ( a - b ) ) ? 0 : ( a - b ) * ( a - b ); };
};



#endif



//* Local Variables:
//* c-file-style: "Stroustrup"
//* mode: C++
//* End:
