// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_UTIL_H
#define KROT_UTIL_H



#include <qlineedit.h>

#include <climits>



class QCheckBox;
class QFont;
class QLabel;



/**
 * Editable line input for floating point numbers.
 *
 * @author Jochen Kpper
 * @version 1998/12/02
 */
class Floated : public QLineEdit
{
    Q_OBJECT

public:

    /**
     * Constructor.
     *
     * @param parent Passed to QLineEdit constructor.
     * @param name   Passed to QLineEdit constructor.
     */
    Floated( QWidget *parent=0, const char *name=0 );

    /**
     * Constructor.
     *
     * @param value  Initial value to display.
     * @param parent Passed to QLineEdit constructor.
     * @param name   Passed to QLineEdit constructor.
     */
    Floated( const double value, QWidget *parent=0, const char *name=0 );

    /**
     * Constructor.
     *
     * We require two parameters to not accidentaly take an integer for precision
     * that the user wanted to be an value.
     *
     * @param prec   Floating point precision to use. (This is the number of digits
     * after the decimal separator.)
     * @param value  Initial value to display.
     * @param parent Passed to QLineEdit constructor.
     * @param name   Passed to QLineEdit constructor.
     */
    Floated( const int prec, const double value, QWidget *parent=0, const char *name=0 );

    /**
     * Approbiate size hint for this widget.
     * The edit area is wide enought to show 8 digit numbers without clipping.
     */
    virtual QSize sizeHint() const;
    

public slots:

    /**
     * @param value Set content of line edit.
     */
    void setValue( double value );

    /**
     * @return Value of line edit.
     */
    double value() const;
    
    
protected:

    int precision;
};



/**
 * Editable line input for floating point numbers.
 *
 * @author Jochen Kpper
 * @version 1998/12/02
 */
class Inted : public QLineEdit
{
    Q_OBJECT

public:

    /**
     * Constructor.
     *
     * @param parent Passed to QLineEdit constructor.
     * @param name   Passed to QLineEdit constructor.
     */
    Inted( QWidget *parent=0, const char *name=0 );

    /**
     * Constructor.
     *
     * @param value  Initial value to display.
     * @param parent Passed to QLineEdit constructor.
     * @param name   Passed to QLineEdit constructor.
     */
    Inted( const int value, QWidget *parent=0, const char *name=0 );

    /**
     * Constructor.
     *
     * We require two parameters to not accidentaly take an integer for precision
     * that the user wanted to be an value.
     *
     * @param min    The minimum value to accept.
     * @param max    The maximum value to accept.
     * @param value  Initial value to display.
     * @param parent Passed to QLineEdit constructor.
     * @param name   Passed to QLineEdit constructor.
     */
    Inted( const int min, const int max, const int value, QWidget *parent=0, const char *name=0 );

    /**
     * Approbiate size hint for this widget.
     * The edit area is wide enought to show 8 digit numbers without clipping.
     */
    virtual QSize sizeHint() const;
    

public slots:

    /**
     * @param value Set content of line edit.
     */
    void setValue( int value );

    /**
     * @return Value of line edit.
     */
    int value() const;
};



/**
 * Editable line input for floating point numbers with an Label in front of it.
 *
 * @author Jochen Kpper
 * @version 1998/12/02
 */
class LblFloated : public QWidget
{
    Q_OBJECT

public:
    
    LblFloated( const char *text="", const int prec=3, const double value=0,
		QWidget *parent=0, const char *name=0 );

    /**
     * Set font to use for the label text.
     *
     * @param font Font to use.
     */
    void setLabelFont( const QFont& font );
    
    /**
     * Approbiate size hint for this widget.
     * The edit area is wide enought to show 8 digit numbers without clipping.
     */
    virtual QSize sizeHint () const;
    

public slots:

    /**
     * En/disable input to this widget.
     *
     * @param enable True to enable input, false to disable
     */
    virtual void setEnabled( bool enable );
    
    /**
     * @param value Set content of line edit.
     */
    void setValue( double value );

    /**
     * @return Value of line edit.
     */
    double value() const;

    
private:

    Floated *edit;

    QLabel *label;
};



/**
 * Editable line input for integer numbers with an Label in front of it.
 *
 * @author Jochen Kpper
 * @version 1998/12/02
 */
class LblInted : public QWidget
{
    Q_OBJECT

public:
    
    LblInted( const char *text="", const int min=INT_MIN, const int max=INT_MAX, const int value=0,
	      QWidget *parent=0, const char *name=0 );

    /**
     * Set font to use for the label text.
     *
     * @param font Font to use.
     */
    void setLabelFont( const QFont& font );
    
    /**
     * Approbiate size hint for this widget.
     * The edit area is wide enought to show 8 digit numbers without clipping.
     */
    virtual QSize sizeHint () const;
    

public slots:

    /**
     * @param value Set content of line edit.
     */
    void setValue( const int value );

    /**
     * @return Value of line edit.
     */
    int value() const;

    
private:

    Inted *edit;

    QLabel *label;
};



/**
 * Editable line input for floating point numbers with an Label and an
 * checkbox in front of it.
 *
 * @author Jochen Kpper
 * @version 1998/12/02
 */
class ChkLblFloated : public QWidget
{
    Q_OBJECT

public:
    
    ChkLblFloated( const char *text="", const int prec=3, const double value=0,
		   QWidget *parent=0, const char *name=0 );

    /**
     * Set font to use for the label text.
     *
     * @param font Font to use.
     */
    void setLabelFont( const QFont& font );
    
    /**
     * Approbiate size hint for this widget.
     * The edit area is wide enought to show 8 digit numbers without clipping.
     */
    virtual QSize sizeHint () const;
    

public slots:

    /**
     * Checkmark status.
     *
     * @return True if checked, false if not checked.
     */
    bool isChecked() const;

    /**
     * Set checkmark
     *
     * @param check True to check, false to uncheck.
     */
    void setChecked( const bool check );

    /**
     * Input status.
     *
     * @param enable True to enable input, false to disable,
     */
    void setEnabled( const bool enable );

    /**
     * @param value Set content of line edit.
     */
    void setValue( const double value );

    /**
     * @return Value of line edit.
     */
    double value() const;


protected slots:

    /**
     * This slot is connected to the mark->toggled(bool) signal and all it does
     * is to emit toggled(bool).
     *
     * @param on True if checked, false if not checked.
     */
    void slotToggled( bool on );

    
signals:

    /**
     * This signal is emitted when the checkmark changes status.
     *
     * @param on True if checked, false if not checked.
     */
    void toggled( bool on );

	
protected:

    Floated *edit;

    QCheckBox *mark;
};



#include "lined_inline.h"


#endif
