// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_KROTDATA_H
#define KROT_KROTDATA_H



#include "calculationParameter.h"
#include "transition.h"

#include <kapp.h> 
#include <kmsgbox.h> 

#include <qobject.h>
#include <qstring.h>

#include <map>
#include <set>
#include <vector>



class DataSet;
class RawData;
class Spectrum;
class Simulation;



// define datatypes
typedef map< QNum, double >            mapAssignment;
typedef mapAssignment::value_type      pairAssignment;
typedef map< QString, Simulation >     mapSimulation;
typedef mapSimulation::value_type      pairSimulation;
typedef map< QString, Spectrum >       mapSpectrum;
typedef mapSpectrum::value_type        pairSpectrum;



/**
 * @short DataRange
 *
 * Describe a data range, consisting of a start and stop frequency plus a
 * minimum and maximum intensity.
 *
 * In here intensities are integers, this is what we can display.
 *
 * @author Jochen Kpper
 * @version 199/02/28
 */
class DataRange
{
public:
    
    /**
     * Constructor.
     * By default initializes the range to 0 - 0.
     */
    DataRange( int64_t start=0, int64_t stop=0, int64_t min=0, int64_t max=0 );

    /**
     * Compare all values.
     */
    bool operator==( const DataRange& range ) const;
    
    /**
     * Compare in frequency space.
     */
    bool operator<( const DataRange& two ) const;

    /**
     * Extend the range to contain the old plus the new range
     */
    void extend( const DataRange& range );
    
    /**
     * Values.
     */
    int64_t max, min, start, stop;
};



/**
 * @short DataRanges
 *
 * Container for @ref DataRange
 *
 * @author Jochen Kpper
 * @version 199/02/28
 */
class DataRanges : public vector< DataRange >
{
    
public:

    void add( const DataRange& range );
};


/**
 * @short KRot data.
 *
 * KRot configuration data and DataSet container. This class provides/manages the
 * algorithms underneath the GUI and cares about the communication of GUI and
 * calculations. Each instance of a MainWindow needs to have one of instance of
 * this class.
 *
 * @author Jochen Kpper
 * @version 1999/02/11
 */
class KRotData : public QObject
{
    Q_OBJECT

public:

    enum Kind { DATASETS, SIMULATIONS, SPECTRA, ASSIGNMENTS };

    /**
     * Default constructor.
     *
     * @param parent Passed to QObject constructor.
     * @param name Passed to QObject constructor.
     */
    KRotData( QObject *parent=0, const char *name=0 );

    /**
     * Destructor.
     */
    ~KRotData();

    /**
     * Insert a new @ref Simulation.
     */
    void add( const Simulation& sim );

    /**
     * Insert a new @ref Spectrum.
     */
    void add( const Spectrum& spec );

    /**
     * Reference to @ref Assignments container.
     *
     * @return The container of simulations.
     */
    const mapAssignment& assignments() const;

    /**
     * Averaging intervall.
     *
     * @return Number of data points to average into one displayed point.
     */
    unsigned int averaging() const;

    /**
     * Check the lowest and highest frequencies of all sets and set start/stop accordingly.
     * Here we need to get called whenever the limits of any set have changed.
     */
    void checkLimits();

    /**
     * Check wether there is any dataset loaded.
     *
     * @param val Type of datasets to check for.
     * @return True if no set of the specified kind is actually inserted, false otherwise.
     */
    bool empty( Kind val=DATASETS ) const;

    /**
     * Delete assignment of @ref QNum.
     */
    void erase( const QNum& qnum );

    /**
     * Delete that @ref DataSet with key name.
     */
    void erase( const QString& name );

    /**
     * Delete all DataSets.
     */
    void eraseAll();

    /**
     * Look up the dataset with the specifiedname. You should not call this
     * function with a none existant name as the result may be surprising.
     *
     * @param name Name of dataset to return.
     * @return Reference of the DataSet called name.
     *         Zero if non set of the given name exists.
     */
    DataSet *getDataSet( const char *name );
    /**
     *
     */
    const DataRanges& dataRanges() const;
    
    /**
     * Reference to @ref Simulation container.
     *
     * @return The container of simulations.
     */
    const mapSimulation& simulations() const;

    /**
     * Reference to @ref Simulation container.
     *
     * @return The container of simulations.
     */
    mapSimulation& simulations();

    /**
     * Reference to @ref Spectrum container.
     *
     * @return The container of spectra.
     */
    const mapSpectrum& spectra() const;

    /**
     * Reference to @ref Spectrum container.
     *
     * @return The container of spectra.
     */
    mapSpectrum& spectra();

    /**
     * Our @ref CalculationParameter
     *
     * @return Current calculation parameter
     */
    const CalculationParameter& parameter() const;
    
    /**
     * First frequency of all @ref DataSet.
     *
     * @return Lowest start frequency of all sets.
     */
    int64_t start() const;

    /**
     * Last frequency of all @ref DataSet.
     *
     * @return Highest stop frequency of all sets.
     */
    int64_t stop() const;


public slots:
 
    /**
     * Takes the given assignent and puts it into the container. If there is already a line
     * with an equal set of quantum numbers, this new one overrides the old one.
     *
     * @param asn New Assignment.
     */
    void add( const AssignedFrequency& asn );


protected slots:

    /**
     * Start arnirot calculation with given parameter and current assignments
     *
     * @param par Current parameter.
     */
    void arnirot( const CalculationParameter& par );

    /**
     * create new simulation from the given lines.
     *
     * @param par Resulting parameter, store for next calculation.
     * @param lines Simulated lines
     */
    void arnirot( const CalculationParameter par, vector< Transition > lines );

    /**
     * Save the assignments to a file.
     *
     * @param filename  Name of file the assignments shall be written to.
     */
    void exportAssignments( const QString& filename ) const;

    /**
     * Export specified simulation as lines.dat data to file.
     *
     * @param simname  Name of @ref Simulation that shall be saved.
     * @param filename Full name of file the data shall be saved to.
     */
    void exportLines( const QString& simname, const QString& filename );

    /**
     * Export specified set as xy data to file.
     *
     * @param setname  Name of @ref DataSet that shall be saved.
     * @param filename Full name of file the data shall be saved to.
     */
    void exportXY( const QString& setname, const QString& filename );

    /**
     *
     */
    void iar( CalculationParameter& par );
    
    /**
     * Import raw jba data from specified file.
     *
     * @param filename Full name of file the data shall be loaded from.
     */
    void importJba( const QString& filename );

    /**
     * Create spectrum from the khidaq object given.
     *
     * @param khidaq Calibrated raw data to create spectrum from.
     */
    void importKHiDaq( const RawData& khidaq );

    /**
     * Import simulated lines data from specified file.
     *
     * @param filename Full name of file the data shall be loaded from.
     */
    void importLines( const QString& filename );

    /**
     * Import xy data from specified file.
     *
     * @param filename Full name of file the data shall be loaded from.
     */
    void importXY( const QString& filename );

    /**
     * Needs to be called to change the averaging intervall.
     *
     * @param avp Number of data points to average into one displayed point.
     */
    void setAveraging( const unsigned int avp );



signals:

    /**
     * Emitted whenever the start/stop/min/max limits are changed.
     */
    void limits( const DataRanges& r );

    /**
     * Emitted whenever there is a new message to be displayed in the statusbar
     * message area.
     *
     * @param msg Message to display.
     */
    void message( const QString& msg );

    /**
     * Emitted whenever there is a set added or deleted from one of the containers.
     */
    void newSet();
  
    /**
     * Center display on new dataset, after one is loaded.
     */
    void position( const int64_t pos );
    

protected:
  
    // diabled assignment operator
    KRotData& operator=( const KRotData& );

    /**
     * Add the given DataRange to the available ranges
     */
    void addRange( const DataRange& range );
    

protected:

    int avg;

    mapAssignment ass;

    KConfig *config;
    
    CalculationParameter pars;
    
    DataRanges ranges;

    mapSimulation sims;

    mapSpectrum spec;
};



#include "krotData_inline.h"



#endif



//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
