// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#include "lined.h"
#include "importRawData.h"
#include "krot.h"
#include "rawData.h"

#include <iostream.h>

#include <qbuttongroup.h> 
#include <qcheckbox.h> 
#include <qlayout.h>
#include <qlineedit.h> 
#include <qvalidator.h>

#include <kapp.h>
#include <kbuttonbox.h>
#include <kconfig.h>
#include <kdebug.h>
#include <ktmainwindow.h>



DialogImportRawData::DialogImportRawData( RawData& raw, const char *name )
    : KTMainWindow( name ),
      config( kapp->getConfig() ),
      data( raw )
{
    KROT_LAUNCH( "Launching DialogImportRawData constructor." );
    setCaption( i18n( "KRot [Import data]" ) );
    view = new RawDataWorkArea( data, this, "RawDataWorkArea" );
    menu();
    tool();
    setView( view );
    connect( view, SIGNAL( assignment( const int ) ),
	     this, SLOT( setAssignment( const int ) ) );
    connect( view, SIGNAL( marker( const int ) ),
	     this, SLOT( setMarker( const int ) ) );
    return;
}



void DialogImportRawData::menu()
{
    KROT_LAUNCH( "Launching DialogImportRawData::menu" );
    config->setGroup( KROT_CONFIG_IMPORT );
    // ** file menu **
    QPopupMenu *file = new QPopupMenu;
    file->insertItem( i18n( "Import" ), this, SLOT( accept() ) );
    file->insertItem( i18n( "Cancel" ), this, SLOT( reject() ) );
    // ** traces **
    // ** settings **
    QPopupMenu *settings = new QPopupMenu;
    traces = new QPopupMenu;
    traces->setCheckable( true );
    id_etalon = traces->insertItem( i18n( "Etalon" ), this, SLOT( toggleEtalon() ) );
    id_iodine = traces->insertItem( i18n( "Iodine" ), this, SLOT( toggleIodine() ) );
    id_power  = traces->insertItem( i18n( "Power" ),  this, SLOT( togglePower()  ) );
    id_signal = traces->insertItem( i18n( "Signal" ), this, SLOT( toggleSignal() ) );
    id_marker = traces->insertItem( i18n( "Marker" ), this, SLOT( toggleMarker() ) );
    settings->insertItem( i18n( "Traces" ), traces );
    settings->insertItem( i18n( "Marker distance" ), this, SLOT( fsr() ) );
    settings->insertItem( i18n( "Linearization Model" ), this, SLOT( linModel() ) );
    // ** help menu **
    QPopupMenu *help = kapp->getHelpMenu( true, aboutKRot );
    // ** menubar **
    menuBar()->insertItem( i18n( "File" ), file );
    menuBar()->insertItem( i18n( "Settings" ), settings );
    menuBar()->insertSeparator();
    menuBar()->insertItem( i18n( "Help" ), help);
    return;
}



void DialogImportRawData::tool()
{
    KROT_LAUNCH( "Launching DialogImportRawData::toolbar" );
    toolBar()->setTitle( i18n("krot [Import raw data ToolBar]") );
    toolBar()->insertCombo( "1", ID_AVERAGING, true,
			    SIGNAL( activated( const char * ) ), this, SLOT( setAveraging( const char * ) ),
			    true,
			    i18n( "Number of data points to average over." ),
			    80 );
    toolBar()->insertComboItem( 0, "3", -1 );
    toolBar()->insertComboItem( 0, "5", -1 );
    toolBar()->insertComboItem( 0, "7", -1 );
    toolBar()->insertComboItem( 0, "9", -1 );
    toolBar()->insertComboItem( 0, "11", -1 );
    toolBar()->insertComboItem( 0, "13", -1 );
    toolBar()->insertComboItem( 0, "15", -1 );
    toolBar()->setCurrentComboItem( ID_AVERAGING, 0 );
    QIntValidator *valid = new QIntValidator( 1, 100000, this );
    toolBar()->getCombo( ID_AVERAGING )->setValidator( valid );
    return;
}



RawDataWorkArea::RawDataWorkArea( RawData& k, QWidget *parent, const char *name )
    : QWidget( parent, name ),
      data( k )
{
    KROT_LAUNCH( "Launching RawDataWorkArea constructor" );
    // create children
    drawarea = new RawDataDrawArea( data, this, "RawDataDrawArea" );
    scrollbar = new QScrollBar( QScrollBar::Horizontal, this, "RawDataScrollBar" );
    scrollbar->setRange( 0, data.size() - drawarea->width() );
    scrollbar->setSteps( 1, drawarea->width() / 2 );
    scrollbar->setValue( 0 );
    scrollbar->setFixedHeight( 16 );
    // and lay them out
    QBoxLayout *layout = new QVBoxLayout( this );
    layout->addWidget( drawarea );
    layout->addWidget( scrollbar );
    layout->activate();
    // connections
    connect( drawarea, SIGNAL( assignment( const int ) ),
	     this,     SLOT( setAssignment( const int ) ) );
    connect( drawarea, SIGNAL( marker( const int ) ),
	     this,     SLOT( setMarker( const int ) ) );
    connect( scrollbar, SIGNAL( valueChanged( int ) ),
	     drawarea,  SLOT( setStart( int ) ) );
    return;
}



void RawDataWorkArea::resizeEvent( QResizeEvent *event )
{
    KROT_LAUNCH( "Launching RawDataWorkArea::resizeEvent" );
    scrollbar->setRange( 0, data.size() - drawarea->width() );
    scrollbar->setSteps( 1, event->size().width() / 2 );
    return;
}



RawDataDrawArea::RawDataDrawArea( RawData& obj, QWidget *parent, const char *name )
    : QWidget( parent, name ),
      config( kapp->getConfig() ),
      data( obj ), start( 0 )
{
    KROT_LAUNCH( "Launching RawDataDrawArea constructor" );
    setCursor( crossCursor );
    setMouseTracking( true );
    return;
}



void RawDataDrawArea::mousePressEvent( QMouseEvent *event )
{
    KROT_LAUNCH_VERBOSE( "Launching RawDataDrawArea::mousePressEvent" );
    int button = event->button();
    if( button & LeftButton ) {
	// left button adds / removes marker
	emit marker( start + event->x() );
	update();
    }
    else if( button & MidButton ) {
	// middle button makes assignment of iodine frequency
	emit assignment( start + event->x() );
    }
    // right button does nothing yet
    return;
}



void RawDataDrawArea::paintEvent( QPaintEvent * )
{
    KROT_LAUNCH_VERBOSE( "Launching RawDataDrawArea::paintEvent" );
    config->setGroup( KROT_CONFIG_IMPORT );
    QPainter paint;
    paint.begin(this);
    if( config->readBoolEntry( config_ShowMarker, true ) ) {
	// draw marker
	paint.setPen( black );
	set< int >::const_iterator iter = data.marker().begin();
	while( iter != data.marker().end() && *iter < start )
	    iter++;
	while( iter != data.marker().end() && *iter < start + width() ) {
	    paint.moveTo( *iter - start, 0 );
	    paint.lineTo( *iter - start, height() );
	    iter++;
	}
    }
    if( config->readBoolEntry( config_ShowIodine, true ) ) {
	// draw iodine
	paint.setPen( magenta );
	if( data.max().iodine - data.min().iodine ) {
	    paint.moveTo( 0, ( ( data[ start ].iodine - data.min().iodine )
			       * height()
			       / ( data.max().iodine - data.min().iodine ) ) );
	    for( int x=1; x < width(); x++ ) {
		paint.lineTo( x,  ( static_cast< double >( data[ start + x ].iodine - data.min().iodine )
				    * height()
				    / ( data.max().iodine - data.min().iodine ) ) );
	    }
	} else {
	    paint.moveTo( 0, 0 );
	    paint.lineTo( width(), 0 );
	}
    }
    if( config->readBoolEntry( config_ShowPower, true ) ) {
	// draw power
	paint.setPen( yellow );
	if( data.max().power - data.min().power ) {
	    paint.moveTo( 0, ( height() / 10 - 1
			       - ( ( data[ start ].power - data.min().power )
				   * height() / 4096 ) ) );
	    for( int x=1; x < width(); x++ ) {
		paint.lineTo( x, ( height() / 10 - 1
				   - ( ( data[ start + x ].power - data.min().power )
				       * height() / 4096 ) ) );
	    }
	} else {
	    paint.moveTo( 0, 0 );
	    paint.lineTo( width(), 0 );
	}
    }
    if( config->readBoolEntry( config_ShowEtalon, true ) ) {
	// draw etalon
	paint.setPen( red );
	if( data.max().etalon - data.min().etalon ) {
	    paint.moveTo( 0, ( height() - 1 - ( ( data[ start ].etalon - data.min().etalon )
						* height()
						/ ( data.max().etalon - data.min().etalon ) ) ) );
	    for( int x=1; x < width(); x++ ) {
		paint.lineTo( x,  height() - 1 - ( ( data[ start + x ].etalon - data.min().etalon )
						   * height()
						   / ( data.max().etalon - data.min().etalon ) ) );
	    }
	} else {
	    paint.moveTo( 0, 0 );
	    paint.lineTo( width(), 0 );
	}
    }
    if( config->readBoolEntry( config_ShowSignal, true ) ) {
	// draw signal
	paint.setPen( blue );
	if( data.max().signal - data.min().signal ) {
	    paint.moveTo( 0, ( height() - 1 
			       - ( ( data[ start ].signal - data.min().signal )
				   * height() / ( data.max().signal - data.min().signal ) ) ) );
	    for( int x = 1; x < width(); x++ ) {
		paint.lineTo( x, ( height() - 1
				   - ( ( data[ start + x ].signal - data.min().signal )
				       * height() / ( data.max().signal - data.min().signal ) ) ) );
	    }
	} else {
	    paint.moveTo( 0, 0 );
	    paint.lineTo( width(), 0 );
	}
    }
    paint.end();
    return;
}



DialogIodineFrequency::DialogIodineFrequency( const int idx, QWidget *parent, const char *name )
    : QDialog( parent, name, true ),
      config( kapp->getConfig() )
{
    KROT_LAUNCH( "Launching DialogIodineFrequency constructor" );
    setCaption( i18n( "KRot [Iodine Frequency Input]" ) );
    config->setGroup( KROT_CONFIG_IMPORT );
    config->writeEntry( config_AbsIndex, idx );
    QGridLayout *layout = new QGridLayout( this, 4, 3, 10 );
    // label
    QLabel *label = new QLabel( i18n( "Frequency at the selected position:" ), this );
    label->setMinimumWidth( label->fontMetrics().width( i18n( "Frequency at the selected position:" ) ) );
    layout->addMultiCellWidget( label, 0, 0, 0, 1 );
    // input
    input = new Floated( 4, config->readDoubleNumEntry( config_AbsFreq, 0.0 ), this );
    input->setFrame( true );
    input->setMinimumWidth( input->sizeHint().width() );
    input->setFixedHeight( input->sizeHint().height() );
    layout->addWidget( input, 0, 2 );
    // units
    QButtonGroup *group = new QButtonGroup();
    mhz = new QRadioButton( i18n( "MHz" ), this );
    mhz->setMinimumSize( mhz->sizeHint() );
    layout->addWidget( mhz, 1, 0, AlignCenter );
    cm = new QRadioButton( i18n( "cm^-1" ), this );
    cm->setMinimumSize( cm->sizeHint() );
    layout->addWidget( cm, 1, 1, AlignCenter );
    group->insert( mhz, UNIT_MHZ );
    group->insert( cm, UNIT_CM );
    switch( config->readNumEntry( config_AbsFreqUnit, UNIT_CM ) ) {
    case UNIT_CM:
	cm->setChecked( true );
	break;
    case UNIT_MHZ:
	mhz->setChecked( true );
	break;
    }
    // doubling
    dbl = new QCheckBox( i18n( "Double it!" ), this );
    dbl->setMinimumSize( dbl->sizeHint() );
    layout->addWidget( dbl, 1, 2, AlignCenter );
    dbl->setChecked( config->readBoolEntry( config_Double, false ) );
    // separator
    QFrame *sep = new QFrame( this );
    sep->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    sep->setFixedHeight( sep->sizeHint().height() );
    layout->addMultiCellWidget( sep, 2, 2, 0, 2 );
    // buttons
    KButtonBox *bbox = new KButtonBox( this );
    bbox->addStretch( 1 );
    QPushButton *ok = bbox->addButton( i18n( "OK" ) );
    bbox->addStretch( 1 );
    QPushButton *cancel = bbox->addButton( i18n( "Cancel" ) );
    bbox->addStretch( 1 );
    bbox->layout();
    layout->addMultiCellWidget( bbox, 3, 3, 0, 2 );
    layout->activate();
    resize( sizeHint() );
    connect( ok,   SIGNAL( pressed() ),
	     this, SLOT( accept() ) );
    connect( cancel, SIGNAL( pressed() ),
	     this,   SLOT( reject() ) );
    return;
}



DialogFSR::DialogFSR( QWidget *parent, const char *name )
    : QDialog( parent, name, true ),
      config( kapp->getConfig() )
{
    KROT_LAUNCH( "Launching DialogFSR constructor" );
    setCaption( i18n( "KRot [Marker distance Input]" ) );
    config->setGroup( KROT_CONFIG_IMPORT );
    QGridLayout *layout = new QGridLayout( this, 4, 2, 10 );
    // label
    QLabel *label = new QLabel( i18n( "Distance of two marker peaks:" ), this );
    label->setMinimumSize( label->sizeHint() );
    layout->addWidget( label, 0, 0 );
    // input
    input = new Floated( 6, config->readDoubleNumEntry( config_FSR, 150.0 ), this );
    input->setFrame( true );
    input->setMinimumSize( input->sizeHint() );
    layout->addWidget( input, 0, 1 );
    // units
    mhz = new QRadioButton( i18n( "MHz" ), this );
    mhz->setFixedSize( mhz->sizeHint() );
    layout->addWidget( mhz, 1, 0, AlignCenter );
    cm = new QRadioButton( i18n( "cm^-1" ), this );
    cm->setFixedSize( cm->sizeHint() );
    layout->addWidget( cm, 1, 1, AlignCenter );
    QButtonGroup *group = new QButtonGroup();
    group->insert( mhz, UNIT_MHZ );
    group->insert( cm, UNIT_CM );
    switch( config->readNumEntry( config_UnitFSR, UNIT_MHZ ) ) {
    case UNIT_CM:
	cm->setChecked( true );
	break;
    case UNIT_MHZ:
	mhz->setChecked( true );
	break;
    }
    // separator
    QFrame *sep = new QFrame( this );
    sep->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    sep->setMinimumHeight( sep->sizeHint().height() );
    layout->addMultiCellWidget( sep, 2, 2, 0, 1 );
    // buttons
    KButtonBox *bbox = new KButtonBox( this );
    bbox->addStretch( 1 );
    QPushButton *ok = bbox->addButton( i18n( "OK" ) );
    bbox->addStretch( 1 );
    QPushButton *cancel = bbox->addButton( i18n( "Cancel" ) );
    bbox->addStretch( 1 );
    bbox->layout();
    layout->addMultiCellWidget( bbox, 3, 3, 0, 1 );
    layout->activate();
    resize( sizeHint() );
    connect( ok,   SIGNAL( pressed() ),
	     this, SLOT( accept() ) );
    connect( cancel, SIGNAL( pressed() ),
	     this,   SLOT( reject() ) );
    return;
}



DialogLinModel::DialogLinModel( QWidget *parent, const char *name )
    : QDialog( parent, name, true ),
      config( kapp->getConfig() )
{
    KROT_LAUNCH( "Launching DialogLinModel constructor" );
    setCaption( i18n( "KRot [Linearization Model]" ) );
    config->setGroup( KROT_CONFIG_IMPORT );
    QGridLayout *layout = new QGridLayout( this, 3, 1, 10 );
    // model 
    QButtonGroup *group = new QButtonGroup( i18n( "Models" ), this );
    QBoxLayout *groupLayout = new QVBoxLayout( group, 5, 0 );
    groupLayout->addSpacing( 10 );
    linear    = new QRadioButton( i18n( "linear interpolation" ), group );
    linear->setMinimumSize( linear->sizeHint() );
    groupLayout->addWidget( linear );
    quadratic = new QRadioButton( i18n( "quadratic interpolation" ), group );
    quadratic->setMinimumSize( quadratic->sizeHint() );
    groupLayout->addWidget( quadratic );
    cubic     = new QRadioButton( i18n( "cubic splines" ), group );
    cubic->setMinimumSize( cubic->sizeHint() );
    groupLayout->addWidget( cubic );
    switch( config->readNumEntry( config_LinModel, RawData::LINEAR ) ) {
    case RawData::LINEAR:
	linear->setChecked( true );
	break;
    case RawData::QUADRATIC:
	quadratic->setChecked( true );
	break;
    case RawData::CUBICSPLINE:
	cubic->setChecked( true );
	break;
    }
    quadratic->setEnabled( false );
    cubic->setEnabled( false );
    groupLayout->activate();
    layout->addWidget( group, 0, 0 );
    // separator
    QFrame *sep = new QFrame( this );
    sep->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    sep->setMinimumHeight( sep->sizeHint().height() );
    layout->addWidget( sep, 1, 0 );
    // buttons
    KButtonBox *bbox = new KButtonBox( this );
    bbox->addStretch( 1 );
    QPushButton *ok = bbox->addButton( i18n( "OK" ) );
    bbox->addStretch( 1 );
    QPushButton *cancel = bbox->addButton( i18n( "Cancel" ) );
    bbox->addStretch( 1 );
    bbox->layout();
    layout->addWidget( bbox, 2, 0 );
    layout->activate();
    resize( sizeHint() );
    connect( ok,   SIGNAL( pressed() ),
	     this, SLOT( accept() ) );
    connect( cancel, SIGNAL( pressed() ),
	     this,   SLOT( reject() ) );
    return;
}
