// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_CALCULATIONPARAMETER_H
#define KROT_CALCULATIONPARAMETER_H



class KConfig;



/**
 * Storage container for inertial constanst.
 *
 * This class is a general storage container for inertial constants.
 * It handles all needed constants for two electronic states you
 * need.
 *
 * @author Jochen Kpper
 * @version 1998/09/25
 */
class InertialConstants
{

public:
    
    enum Constant{ begin=0, A=0, B, C, DX, DY, DZ, DJ, DJK, DK, dJ, dK, end };
    
    enum State { GroundState, ExcitedState };

    /**
     * Standard constructor.
     */
    InertialConstants();

    /**
     * Copy constructor.
     *
     * @param ic The inertial constants to copy into the newly created object. 
     */
    InertialConstants( const InertialConstants& ic );

    /**
     * Assignment operator.
     *
     * @param ic The inertial constants to copy into this.
     */
    InertialConstants& operator=( const InertialConstants& ic );

    /**
     * Destructor.
     */
    ~InertialConstants();

    /**
     * If any of the centrifugal distortion constants is non-zero or shall be
     * fitted, centrifugal distortion should be included into the calculation.
     *
     * @return Is centrifugal distortion requested ?
     */
    bool centrifugalDistortion() const;
    
    /**
     * Parameter reference
     *
     * @param s State we are talking about.
     * @param c Constant you want to get.
     * @return Reference to constant.
     */
    double& constant( const State s, const Constant c );

    /**
     * Parameter value
     *
     * @param s State we are talking about.
     * @param c Constant you want to get.
     * @return Reference to constant.
     */
    double constant( const State s, const Constant c ) const;

    bool fit() const;
	
    /**
     * What constants shall be fit ?
     */
    bool fit( const State s, const Constant c ) const;

    /**
     * What constants shall be fit ?
     */
    bool& fit( const State s, const Constant c );

    /**
     * If any of the linear distortion constants is non-zero or shall be
     * fitted, centrifugal distortion should be included into the calculation.
     *
     * @return Is linear distortion requested ?
     */
    bool linearDistortion() const;

    /**
     * Read data form config file.
     *
     * This functions reads the data from the global KConfig database. It does
     * it's access to the currently selected group, so owners of this class can
     * (and have to) select the config-group to utilize.
     */
    void readConfig( KConfig *config );

    /**
     * Read data form config file.
     *
     * This functions reads the data from the global KConfig database. It does
     * it's access to the currently selected group, so owners of this class can
     * (and have to) select the config-group to utilize.
     */
    void writeConfig( KConfig *config );
    
    
protected:

    void defaultValues();
    

protected:

    bool fitValue[ 2 ][ end ];

    static const char name[ 2 ][ end ][ 8 ];
    static const char flagName[ 2 ][ end ][ 8 ];

    /**
     * inertial Parameter for both states
     * (A B C DX DY DZ DJ DJK DK dJ dK)
     */
    double value[ 2 ][ end ]; // sync with n
};



class CalculationParameter : public InertialConstants
{

public:

    /**
     * Additional parameter that do have an floating point number
     * representation. The inertial constants are defined in our base class
     * @ref InertialConstants. Integer parameter are defined in @ref ParameterInt.
     */
    enum ParameterFloat { fBegin, Temperature=0, Temperature2, TemperatureWeight,
			  AxisTiltPhi, AxisTiltTheta, AxisTiltChi,
			  Origin, LPU, HybridA, HybridB, HybridC, fEnd };

    /**
     * More parameter. These do have an integer representation. For more info,
     * see @ref ParameterFloat.
     */
    enum ParameterInt { iBegin=0, Jmin=0, Jmax, DKmax, NSSWee, NSSWeo, NSSWoe, NSSWoo, iEnd };

    /**
     * Standard constructor.
     */
    CalculationParameter();

    /**
     * Copy constructor.
     *
     * This function does perform a real copy, no fancy stuff like reference
     * counting or such. Keep performance in mind before you use it !
     */
    CalculationParameter( const CalculationParameter& );

    /**
     * Destructor.
     */
    ~CalculationParameter();

    /**
     * Assignment of one instanz to another.
     */
    CalculationParameter& operator=( const CalculationParameter& );

    bool axisTilting() const;
    
    bool fit() const;

    bool fit( const ParameterInt p ) const;

    bool& fit( const ParameterInt p );

    bool fit( const ParameterFloat p ) const;

    bool& fit( const ParameterFloat p );

    bool fit( const State s, Constant c ) const { return InertialConstants::fit( s, c ); };

    bool& fit( const State s, Constant c ) { return InertialConstants::fit( s, c ); };

    double constant( const ParameterFloat p ) const;

    double& constant( const ParameterFloat p );
    
    int constant( const ParameterInt p ) const;

    int& constant( const ParameterInt p );

    double constant( const State s, const Constant c ) const { return InertialConstants::constant( s, c ); };

    double& constant( const State s, const Constant c ) { return InertialConstants::constant( s, c ); };

    /**
     * Read data form config file.
     *
     * This functions reads the data from the global KConfig database. It does
     * it's access to the currently selected group, so owners of this class can
     * (and have to) select the config-group to utilize.
     */
    void readConfig( KConfig *config );

    /**
     * Read data form config file.
     *
     * This functions reads the data from the global KConfig database. It does
     * it's access to the currently selected group, so owners of this class can
     * (and have to) select the config-group to utilize.
     */
    void writeConfig( KConfig *config );

    
protected:

    void defaultValues();
    

protected:

    /**
     * Fit or plain simulation wanted ?
     * What parameter shall be fit ?
     */
    bool fitParameterFloat[ fEnd ];

    bool fitParameterInt[ iEnd ];

    static const char floatName[ fEnd ][ 8 ];
    static const char floatFlagName[ fEnd ][ 12 ];
    static const char intName[ iEnd ][ 8 ];
    static const char intFlagName[ iEnd ][ 12 ];
    
    int parameterInt[ iEnd ];

    double parameterFloat[ fEnd ];
};



// global helper

extern void incr( CalculationParameter::ParameterFloat& );
extern void incr( CalculationParameter::ParameterInt& );
extern void incr( InertialConstants::State& );
extern void incr( InertialConstants::Constant& );



#include "calculationParameter_inline.h"



#endif



//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
