c     This file is part of krot,
c     a program for the simulation, assignment and fit of HRLIF spectra.
c
c     Copyright (C) 1994-1999 Arnim Westphal
c     Copyright (C) 1997-1999 Jochen Kpper
c
c     If you use this program for your scientific work, please cite it according to
c     the file CITATION included with this package.
c
c     krot-arnirot
c     a program to calculate rotational resolved vibrational/vibronic bands
c
c     this program utilizes the Watson-Hamiltonian up to quartic terms
c     for reference see: James K. G. Watson, Vibrational Spectra and Structure 6, 1 (1977), ed J.R. Durig)


#include "arni.h"


c     (main) routine to calculate two state rovibronic spectra
      subroutine arnical( Jmx, dmevl, dmevc, maxnl, npr, ntheli, icqn,
     *                    swg, swe, swang, itrvec,
     *                    diaalg,
     *                    evalg, evale,
     *                    evecgr, evecgi, evecer, evecei,
     *                    Jmxcal, dKmax, cutint,
     *                    lifile, asfile,
     *                    fitges, ifit, sigma,
     *                    polori,
     *                    shorti,
     *                    cRotCg, cRotCe, cDRotC, cState, cEuler, no_yes, off_on,
     *                    frame, kind,
     *                    cBran, polax1,
     *                    lngbar, lstars, lbars, ldash,
     *                    lqn, lqn2, intens, frethe, freexp,
     *                    temp1, temp2, weight, nuspsw,
     *                    nuzero,
     *                    divis, Jmxout, normf,
     *                    rotcog, rotcoe, idelta,
     *                    ivpt,
     *                    fsrcor, olitmx,
     *                    stflag )

c     Following there is a brief description of all parameters. Be careful, these declarations are
c     pseudo code, they look like C, but represent FORTRAN coding !
c
c     asfile - input  - char[250]      - assignment file name ( only used with old format )
c     cBran  - input  - char[3][1]     - branch naming letters ( P, Q, R )
c     cDRotC - input  - char[npar][9]  - delta rotational constants names
c     cEuler - input  - char[3][5]     - Eulerian angles names
c     cRotCe - input  - char[npar][5]  - excited state rotational constants names
c     cRotCg - input  - char[npar][5]  - ground state rotational constants names
c     cState - input  - char[2][7]     - state names ( "ground" / "excited" )
c     cutint - input  - double         - intensity cutoff ( absolute value )
c     diaalg - input  - int            - choose diagonalization algorithm
c                                        ( 1: Jacobi, 2: Householder/tridi )
c     divis  - input  - double         - divisor to convert frequency units used for assignments to MHz
c     dKmax  - input  - int            - maximum delta K to calculate lines for
c     dmevl  - input  - int            - actual dimension of eigenvalue storage vectors
c     dmevc  - input  - int            - actual number of elements for eigenvector storage
c     evale  - output - double[dmevl]  - calculated excited state eigenvalues
c     evalg  - output - double[dmevl]  - calculated ground state eigenvalues
c     evecgi - output - double[dmevc]  - calculated ground state eigenvectors, imaginary part
c     evecgr - output - double[dmevc]  - calculated ground state eigenvectors, real part
c     evecei - output - double[dmevc]  - calculated excited state eigenvectors, imaginary part
c     evecer - output - double[dmevc]  - calculated excited state eigenvectors, real part
c     fitges - inout  - int            - common fitting switch and indicator for successfully performed fit
c     frame  - input  - char[10][2]    - attribute of coordinate system ( "(un)switched" )
c     freexp - output - double[maxnl]  - observed line positions ( same index as corresponding calculated line )
c     fsrcor - inout  - double         - etalon FSR correction factor: FSR(exact) = fsrcor * FSR(assumed)
c     icqn   - output - int[dmevl][3]  - set of rotational state quantum numbers ( "kets |J Ka Kc>" )
c     idelta - input  - int            - Use absolute excited state inertial constants or deltas
c                                        ( 0 absolute, 1 deltas )
c     ifit   - input  - int[npr]       - if( 1 == ifit[i] ) then fit parameter i
c     intens - output - double[maxnl]  - intensity of calculated lines
c     itrvec - input  - int            - transform transition moment flag
c     ivpt   - output - int[Jmx+1]     - index ( "pointer" ) for eigenvector coefficients
c     Jmx    - input  - int            - maximum possible J
c     Jmxcal - input  - int            - maximum J to calculate up to
c     Jmxout - input  - int            - maximum J to output calculation details
c     kind   - input  - char[9][2]     - character of matrices and vectors ( "real" or "imaginary" )
c     lbars  - input  - char[81]       - output separation line of underscores ( "_" )
c     ldash  - input  - char[81]       - output separation dashed line ( "- - -")
c     lifile - input  - char[250]      - lines file name ( for output, only used with old format )
c     lngbar - input  - char[220]      - "long bar" as separator for matrix output
c     lqn    - output - int[maxnl][6]  - set of quantum numbers for a transition ( J'Ka'Kc' J''Ka''Kc'' )
c     lqn2   - output - int[maxnl][6]  - additional lines information for subbranch discrimination
c     lstars - input  - char[81]       - output separation line of stars ( "*" )
c     maxnl  - input  - int            - maximum number of rovibronic transitions ( "lines" )
c     normf  - input  - int            - intensity scaling factor
c     no_yes - input  - char[3][2]     - logical expressions "NO" / "YES"
c     npr    - input  - int            - maximum number of rotational model parameters
c     ntheli - output - int            - number of calculated lines
c     nuspsw - input  - int[4]         - nuclear spin statistical weights of ee eo oe oo states
c     nuzero - inout  - double         - vibronic band origin
c     off_on - input  - char[3][2]     - switch status "OFF" / "ON"
c     olitmx - inout  - int            - maximum number of complete calculation and fitting cycles
c     polax1 - input  - char[1][3]     -
c     polori - input  - double[3]      - a, b, c components of transition
c     rotcoe - inout  - double[npar]   - excited state inertial parameters
c                                        ( A, B, C, DX, DY, DZ, DJ, DJK, DK, dJ, dK )
c     rotcog - inout  - double[npar]   - ground state inertial parameters
c                                        ( A, B, C, DX, DY, DZ, DJ, DJK, DK, dJ, dK )
c     shorti - input  - int            - if( 1 == shorti ) then use short I/O format, long otherwise
c     sigma  - input  - double         - common standard deviation for all frequency assignments
c                                        ( line position uncertainty )
c     swg    - input  - int            - perform axis reorientation of excited state
c     swang  - input  - int[3][2]      - axis switching angles theta,phi,chi for ground and
c                                        excited state
c     temp1  - input  - double         - rotational temperature
c     temp2  - input  - double         - rotational temperature 2 for two-temperature model
c     frethe - output - double[maxnl]  - calculated lines
c     weight - input  - double         - weight of rotational temperature 2 in two-temperature
c                                        model, weight of temperature 1 is 1.0.


      implicit none

c     maximum (possible) rotational quantum number
      integer        Jmax, Jmx
      parameter      ( Jmax = ARNIROT_JMAX )

c     maximum dimension of hamiltonian matrix
      integer        dmham
      parameter      ( dmham = 2*Jmax + 1 )
c     maximum dimension of factorized Wang hamiltonian submatrix
      integer        dmfwng
      parameter      ( dmfwng = Jmax/2 + 1 )
c     dimension of eigenvalue storage vectors
      integer        dmeval, dmevl
      parameter      ( dmeval = (Jmax + 1)*(Jmax + 1) )

c     number of elements for eigenvector storage
      integer        dmevec, dmevc
c     help variable for the calculation of dmevec
      real*8         szvec
c     last summand is 1.5 instead of 1.0 to prevent roundoff error
      parameter      ( szvec = ((4.d0/3.d0*Jmax + 4.d0)*Jmax + 11.d0/3.d0)*Jmax + 1.5d0 )
      parameter      ( dmevec = szvec )

c     maximum number of lines
      integer        maxnli, maxnl
      parameter      ( maxnli = ARNIROT_MAXNLI )
c     maximum number of rotational model parameters (npar each state plus origin)
      integer        npar, npr
      parameter      ( npar = ARNIROT_NPAR )
c     maximum number of Levenberg-Marquardt iterations
      integer        mqitmx
      parameter      ( mqitmx = 200 )

c     Boltzman k to MHz conversion factor
      real*8         k2MHz
      parameter      ( k2MHz = 20836.73969884132008768d0 )
c     lower limit for relative decrease of chisq (criterion for convergence)
      real*8         tol1, tol2
      parameter      ( tol1 = 1.d-8 )
      parameter      ( tol2 = 1.d-12 )


c     switch for diagonalization algorithm
      integer        diaalg
c     flags for complex hamiltonians and eigenvectors
      integer        fcompe, fcompg, fcompl
c     switch for fitting the parameters of ground and/or excited state
      integer        fitges
c     switch for fitting the parameters to a microwave spectrum
      integer        fitMW
c     flag for linear terms within hamiltonian
      integer        flinht
c     help variables
      integer        h1, h2
c     loop variables
      integer        i, ii, k
c     actual size of J block (2J+1)
      integer        iact
c     actual size of greatest submatrix (J/2 + 1)
      integer        iactr
c     counter for varied parameters
      integer        icount
c     rotational state kets |J Ka Kc>
      integer        icqn(dmevl,3)
c     flag for delta rotational constants
      integer        idelta
c     pointer on first eigenvalue of a given J as contained in eval?
      integer        iept
c     individual fitting switches for each rotational model parameter
      integer        ifit(2*npr+2)
      integer        ifitpt(2*npar+2)
c     sum of switch positions for excited state parameters
      integer        ifitsm
c     index limits for array copying
      integer        ilim1, ilim2
c     ordering pointer for sorting
      integer        iop(maxnli)
c     index to theoretical lines with experimental assignments (TheExpMatch)
      integer        ipttem(maxnli)
c     sign of transition moment component
      integer        isign
c     index variables for assigned to calculated lines matching
      integer        istart, itheor
c     transform transition moment flag
      integer        itrvec
c     pointer on the first element of the eigenvector block of a given J as contained in evec??
      integer        ivpt(0:Jmx)
c     current rotational quantum number
      integer        J
c     Jmax in calculation / assignment file / detailed output / temporary storage
      integer        Jmxcal, Jmxexp, Jmxout, Jmxtmp
c     Delta K max in calculation / assignment file / temporary storage
      integer        dKmax, dKmexp, dKmtmp
c     warning flag for overflow of lines data
      integer        lioflw
c     set of quantum numbers for a transition
c     lqn  : J', Ka', Kc', J", Ka", Kc"
c     lqn2 : type (1=a, 2=b, 3=c), Ka", Delta Ka, J"-Kc", Delta J, Kc" (old)
c     lqn2 : type (1=a, 2=b, 3=c), Delta J, Delta Ka, Ka", J"-Kc", Kc" (new)
      integer        lqn(maxnl,6), lqn2(maxnl,6)
c     number of Levenberg-Marquardt iterations / iterations without improvement
      integer        mqit, mqct
c     number of matching data sets (transitions)
      integer        ndata
c     number of assigned lines
      integer        nexpli
c     normalization factor for line intensities
      integer        normf
c     number of actually used rotational model parameters (<= npar)
      integer        npare, nparg
c     number of parameters to be actually fit (<= 2npar+1)
      integer        nparft
c     number of calculated lines
      integer        ntheli
c     nuclear spin statistical weights  ee eo oe oo
      integer        nuspsw(0:3)
c     current / maximum number of outer iterations
      integer        olit, olitmx
c     flag for new IO format
      integer        short0, shorti
c     program status flag
      integer        stflag
c     flags for axis reorientation
      integer        swg, swe


c     SVD solution vector x
      real*8         a(2*npar+2), atmp(2*npar+2)
c     curvature matrix (Levenberg-Marquardt method)
      real*8         alpha(2*npar+2,2*npar+2)
c     chi square (new value, old values)
      real*8         chisq, chisq0, chisq1
c     intensity cut off
      real*8         cutint, cutor
c     covariance matrix
      real*8         cvm(2*npar+2,2*npar+2)
c     sum of absolute values of Delta parameters
      real*8         delsum
c     derivatives d(eval?)/d(rotco?) corresponding to each eigenvalue
      real*8         derivg(dmeval,npar), derive(dmeval,npar)
c     derivative information assigned to calculated transition
      real*8         derlg(maxnli,npar), derle(maxnli,npar)
c     difference of chisq in two successive Marquardt iterations
      real*8         diff
c     divisor for line frequencies
      real*8         divis
c     vector space for temporary eigenvalue storage
      real*8         e(dmham)
c     real*8         er(dmfwng)
      real*8         eb(2*dmham)
c     eigenvalues of complex hamiltonian matrix (complete / real / imaginary)
      real*8         eneb(2*dmham)
      real*8         ener(dmham), enei(dmham)
#ifdef DEBUG_VERBOSE
c     eigenvalues calculated using derivatives
      real*8         ened(dmham)
#endif
c     eigenvalues (final storage)
      real*8         evalg(dmevl),  evale(dmevl)
c     eigenvectors (final storage)
c     - real part
      real*8         evecgr(dmevc), evecer(dmevc)
c     - imaginary part
      real*8         evecgi(dmevc), evecei(dmevc)
c     uncertainties of fitted parameters
      real*8         err(2*npar+2)
c     line positions: assigned / assigned+matched / calculated
      real*8         freasn(maxnli), freexp(maxnl), frethe(maxnl)
c     etalon FSR correction factor and its reciprocal
      real*8         fsrcor, fsrdiv
c     old value of fsrcor; conversion factor fsrco0/fsrcor
      real*8         fsrco0, fac1
c     summarized derivatives: -dE''/d lambda''   (1...npar)
c                              dE' /d lambda'    (npar+1...2*npar)
c                              dDE /d nuzero = 1 (2*npar+1)
c                              dDE /d fsrcor     (2*npar+2)
      real*8         funcs(maxnli,2*npar+2), funcst(maxnli,2*npar+2)
c     complex hamiltonian matrix (complete / real part / imaginary part)
      real*8         hamb(2*dmham,2*dmham)
      real*8         hamr(dmham,dmham), hami(dmham,dmham)
c     line intensities
      real*8         intens(maxnl)
c     lambda scaling factor (Levenberg-Marquardt: alamda)
      real*8         lambda
c     band origin
      real*8         nuzero
c     transition dipole components
      real*8         polori(3), polcal(3)
c     packed quantum numbers for assigned / calculated line frequencies
      real*8         pqnexp(maxnli), pqnthe(maxnli)
c     relative changes of fitted parameters, maximum and temp variable
      real*8         relchg(2*npar+2), maxchg, tmpchg
c     rotational constants
      real*8         rotcog(npr), rotcoe(npr)
c     axis switched rotational constants tensor
      real*8         rraxsw(6,2)
c     sum of squared deviations in line positions (obs - calc)
      real*8         sdsum
c     common standard deviation for all experimental lines
      real*8         sigma
c     axis SWitching Euler ANGles theta,phi,chi g.s./e.s.
      real*8         swang(3,2)
c     temperatures and weighting factor
      real*8         temp1, temp2, weight
      real*8         tempf1, tempf2
c     workspace provided for subroutine chevec
c     real*8         tmpvec(dmham)
c     F to g rotation and inverse rotation matrices (ZARE eq.(3.36))
      real*8         trang(3,3), trane(3,3), tring(3,3), trine(3,3)
c     SVD matrices U and V
      real*8         u(maxnli,2*npar+2)
      real*8         v(2*npar+2,2*npar+2)
c     matrices for eigenvectors (total complex / real part / imaginary part)
      real*8         vecb(2*dmham,2*dmham)
      real*8         vecr(dmham,dmham), veci(dmham,dmham)
c     diagonal elements of SVD matrix W
      real*8         w(2*npar+2)
c     hamiltonian matrix in Wang basis (real / imaginary)
      real*8         whamr(dmham,dmham), whami(dmham,dmham)
c     Wang transformation matrix
      real*8         wangtr(dmham,dmham)
c     right hand side vector b of linear equations system
      real*8         y(maxnli)


c     branch / parity / axes / symmetry species / logic etc. symbols
c     cf. subroutine setup
      character*1    cBran(-1:1), polax1(3)
      character*3    no_yes(0:1), off_on(0:1)
      character*5    cRotCg(npar), cRotCe(npar), cEuler(3)
      character*7    cState(2)
      character*9    cDRotC(npar), kind(0:1)
      character*10   frame(0:1)
      character*81   lstars, lbars, ldash
      character*220  lngbar
c     file names
      character*250  asfile, lifile


      ARNIROT_LAUNCH( "Launching arnical." )

c-----------------------------------------------------------------------------
c     initializations which have to be done only once
c-----------------------------------------------------------------------------
c     explicit initialization
      do k = 1, 2*npar+2
         w( k ) = 0.0
         do i = 1, 2*npar+2
            v( i, k ) = 0.0
         end do
      end do

c     determine the number of currently used rotational parameters separately for ground and excited state
c     a parameter is considered as being used if it has a non-zero value or if it is to be fitted for
      nparg = npar
      do i = npar, 1, -1
         if ( ( rotcog(i) .ne. 0 ) .or.
     *        ( fitges .ne. 0 ) .and. ( ifit(i) .ne. 0 ) )
     *        goto 100
         nparg = nparg - 1
      end do
c     note: excited state parameter values are delta values at this point, if idelta==1
  100 npare = npar
      fitMW  = 0
      delsum = 0.d0
      ifitsm = 0
      if ( idelta .eq. 1 ) then
c        determine the sum of unsigned delta values
         do i = 1, npar, 1
            delsum = delsum + dabs( rotcoe(i) )
            ifitsm = ifitsm + ifit(i+npar)
         end do
c        detect special case: fitting parameters to MW data
         if ( ( delsum .eq. 0 ) .and. ( ifitsm .eq. 0 ) .and.
     *        ( nuzero .eq. 0 ) .and. ( ifit(2*npar+1) .eq. 0 ) .and.
     *        ( fitges .ne. 0 ) ) then
            fitMW = 1
            npare = nparg
            if ( shorti .eq. 0 )
     *           write(*,*) 'This is treated as a fit to microwave data!'
         else
            do i = npar, 1, -1
               if ( ( rotcoe(i) .ne. 0 ) .or.
     *              ( rotcoe(i) .eq. 0 ) .and. ( rotcog(i) .ne. 0 ) .or.
     *              ( fitges .ne. 0 ) .and. ( ifit(i+npar) .ne. 0 ) )
     *              goto 200
               npare = npare - 1
            end do
  200       continue
         end if
      else
         do i = npar, 1, -1
            if ( ( rotcoe(i) .ne. 0 ) .or.
     *           ( fitges .ne. 0 ) .and. ( ifit(i+npar) .ne. 0 ) )
     *           goto 250
            npare = npare - 1
         end do
  250    continue
      end if

      ARNIROT_DEBUG3( "nparg = ", nparg, "  npare = ", npare )
      if ( ( nparg .eq. 0 ) .or. ( ( npare .eq. 0 ) .and. ( fitMW .eq. 0 ) ) ) then
         if ( shorti .eq. 0 ) then
#warning: this error output should be handled in another way
            write(*,*) 'All rotational parameters are zero!\n...leaving krot-arnirot'
         end if
         stflag = 2
         return
      end if

c     convert polarization percentages into square root polarization factors
      do i = 1, 3
         if ( polori(i) .ne. 0 ) then
            isign = polori(i) / dabs( polori( i ) )
            polcal(i) = isign * dsqrt( dabs( polori( i ) ) / 1.d2 )
         else
            polcal(i) = 0.d0
         end if
      end do

c     convert divisor kT of Boltzman term into single factor
      tempf1 = 1.d0 / ( k2MHz * temp1 )
      tempf2 = 1.d0 / ( k2MHz * temp2 )

c     setup vector offset pointer array
      call ivecpt( ivpt, Jmax, Jmxcal )

c-----------------------------------------------------------------------------
c     experimental lines information
c-----------------------------------------------------------------------------
      Jmxtmp = 0
      dKmtmp = 0
      chisq  = 0.d0
      chisq1 = 1.d10
      if ( fitges .eq. 1 ) then
c        input assignment file
         call expth2( maxnl, nexpli, pqnexp, freasn, asfile, Jmxexp, dKmexp, shorti )
c        reduce Jmxcal and dKmax to the smallest necessary values to save computation time
         if ( ( nexpli .gt. 0 ) .and. ( olitmx .gt. 1 ) ) then
            Jmxtmp = Jmxcal
            Jmxcal = Jmxexp
            dKmtmp = dKmax
            dKmax  = dKmexp
         end if
         if ( nexpli .le. 0 ) then
            if ( shorti .eq. 0 ) then
               write(*,*) 'WARNING: No experimental lines!\n...fitting section will not be entered'
            end if
            fitges = 0
            olitmx = 1
         end if
      end if

c-----------------------------------------------------------------------------
c     outer iteration loop
c-----------------------------------------------------------------------------
c     reduce amount of output by manipulating shorti
      short0 = shorti

      olit  = 0

  300 continue
c#ifndef DEBUG_VERBOSE
      if ( olit .lt. olitmx ) shorti = 1
c#endif
      if ( short0 .eq. 0 ) then
c        write(*,'(/,''**************************'')')
c        write(*,'(''* iteration step #'',i6,'' *'')') olit
c        write(*,'(''**************************'')')
         write(*,'(/,''iteration step #'',i6,''  chi-square: '',f15.10)') olit, chisq
      end if

c     convert deltas back to absolute constants for next iteration
      if ( idelta .eq. 1 )
     *     call delabs( rotcog, rotcoe, npar, 0 )

c     check for sensible values of rotational constants A >= B >= C > 0
      if (( rotcog(1) .lt. rotcog(2) ) .or. ( rotcog(2) .lt. rotcog(3) ) .or. ( rotcog(3) .le. 0 ) .or.
     *    ( rotcoe(1) .lt. rotcoe(2) ) .or. ( rotcoe(2) .lt. rotcoe(3) ) .or. ( rotcoe(3). le. 0 )) then
#warning: this error output should be handled in another way
         write(*,*) 'ERROR  : order of rotational constants A >= B >= C > 0 is violated!'
         stflag = 2
         return
      end if

c     section to rotate the ground and excited state rigid rotor hamiltonians
      if ( swg .eq. 1 ) then
         call rotham( rraxsw, 1, rotcog, rotcoe, npar, cState, trang, tring, swang, shorti, lstars, lbars )
      else
#warning: bug ?! workaround by means of explicit initialization
         do i = 1, 6, 1
            rraxsw( i, 1 ) = 0.d0
         end do
      end if
      if ( swe .eq. 1 ) then
         call rotham( rraxsw, 2, rotcog, rotcoe, npar, cState, trane, trine, swang, shorti, lstars, lbars )
      else
         do i = 1, 6, 1
            rraxsw( i, 2 ) = 0.d0
         end do
      end if

c     transform transition moment if specified
      if ( itrvec .eq. 1 )
     *     call murot( trang, polcal, 3, shorti )

c     * set flags: real or complex diagonalization of hamiltonian *
c     ground state
      fcompg = 0
      if ( rotcog( 5 ) .ne. 0. ) then
         ARNIROT_DEBUG_VERBOSE1( 'rotcog(5) is not 0, but ', rotcog(5) )
         fcompg = 1
      end if
      if ( rraxsw( 4, 1 ) .ne. 0. ) then
         ARNIROT_DEBUG_VERBOSE1( 'rraxsw(4,1) is not 0, but ', rraxsw(4,1) )
         fcompg = 1
      end if
      if ( rraxsw( 6, 1 ) .ne. 0. ) then
         ARNIROT_DEBUG_VERBOSE1( 'rraxsw(6,1) is not 0, but ', rraxsw(6,1) )
         fcompg = 1
      end if
c     excited state
      fcompe = 0
      if ( rotcoe( 5 ) .ne. 0. ) then
         ARNIROT_DEBUG_VERBOSE1( 'rotcoe(5) is not 0, but ', rotcoe(5) )
         fcompe = 1
      end if
      if ( rraxsw( 4, 2 ) .ne. 0. ) then
         ARNIROT_DEBUG_VERBOSE1( 'rraxsw(4,2) is not 0, but ', rraxsw(4,2) )
         fcompe = 1
      end if
      if ( rraxsw( 6, 2 ) .ne. 0. ) then
         ARNIROT_DEBUG_VERBOSE1( 'rraxsw(6,2) is not 0, but ', rraxsw(6,2) )
         fcompe = 1
      end if
c     common status flag
      fcompl = fcompg + fcompe

      ARNIROT_DEBUG_VERBOSE1( 'fcompg = ', fcompg )
      ARNIROT_DEBUG_VERBOSE1( 'fcompe = ', fcompe )
      ARNIROT_DEBUG_VERBOSE1( 'fcompl = ', fcompl )

c-----------------------------------------------------------------------------
c     ground state eigenvectors and eigenvalues
c-----------------------------------------------------------------------------
      if ( shorti .eq. 0 ) then
         write(*,'(a)') lstars
         write(*,*) 'ground state J block calculations:'
      end if

c     check if ground state hamiltonian contains linear contributions
      flinht = 0
      do i = 4, 6, 1
         if ( ( rotcog(i) .ne. 0. ) .or.
     *        ( fitges .ne. 0 ) .and. ( ifit(i) .ne. 0 ) )
     *        flinht = 1
      end do
      if ( swg .ne. 0 ) flinht = 1

c     ground state J block qn counter
      do J = 0, Jmxcal, 1
         
c        current offset
         iept = J*J

c        actual size of the J block and the greatest submatrix
         iact  = 2*J + 1
         iactr = J/2 + 1

c        produce rotational state kets for both ground and excited states
         call ket( icqn, dmeval, J )

c        calculate Wang transformation matrix
         call wang( wangtr, dmham, iact )

c        zero the hamiltonian and eigenvector work spaces
         call smat0( hamr, dmham, iact )
         call smat0( hami, dmham, iact )
c        veci is initialized later, if fcomg==1, and doesn't need to be zeroed
c        if fcomg==0 -- unless complex calculations are done for excited state
         if ( fcompe .ne. 0 )
     *        call smat0( veci, dmham, iact )
c        all other hamiltonian submatrices and eigenvector matrices don't have
c        to be zeroed since each element is filled in explicitly

c        fill ground state hamiltonian
         if ( swg .eq. 1 ) then
c           axis switched rigid rotor
            call rrhfax( J, rraxsw(1,1), rraxsw(2,1), rraxsw(3,1), hamr, dmham )
c           off-diagonal axis switching terms
            if ( rraxsw(4,1) .ne. 0. ) call jzjy( hami, dmham, J, rraxsw(4,1) )
            if ( rraxsw(5,1) .ne. 0. ) call jzjx( hamr, dmham, J, rraxsw(5,1) )
            if ( rraxsw(6,1) .ne. 0. ) call jyjx( hami, dmham, J, rraxsw(6,1) )
         else
c           rigid rotor terms not axis switched
            call watson( J, rotcog, npar, hamr, dmham )
         end if
c        linear angular momentum terms: assumed to be in axis switched frame
c        - real
         if ( rotcog(4) .ne. 0. ) call jx( hamr, dmham, J, rotcog(4) )
         if ( rotcog(6) .ne. 0. ) call jz( hamr, dmham, J, rotcog(6) )
c        - imaginary
         if ( rotcog(5) .ne. 0. ) call jy( hami, dmham, J, rotcog(5) )

#ifdef DEBUG_VERBOSE
         if ( ( shorti .eq. 0 ) .and. ( J .le. Jmxout ) ) then
c           output real hamiltonian matrix for J block
            call wriham( hamr, dmham, J, kind(0), frame(swg), lngbar, lstars )
c           output imaginary hamiltonian matrix for J block
            if ( fcompg .ne. 0 ) call wriham( hami, dmham, J, kind(1), frame(swg), lngbar, lstars )
         end if
#endif

c        transform real hamiltonian to Wang basis
         call trwang( wangtr, hamr, whamr, dmham, iact )
c        diagonalize the hamiltonian matrix: check for imaginary terms
         if ( fcompg .eq. 0 ) then
            if ( flinht .ne. 0 ) then
               call rotdi( whamr, iact, dmham, ener, vecr, e, diaalg, shorti )
            else
               call rotdip( J, iact, dmham, dmfwng, whamr, ener, vecr, diaalg, shorti, Jmxout )
            end if
         else
c           transform imaginary hamiltonian to Wang basis
            call trwang( wangtr, hami, whami, dmham, iact )
c           complex diagonalization
            call diagc ( whamr, whami, hamb, ener, enei, eneb, vecr, veci, vecb, J, dmham, eb, diaalg, shorti )
c           transform imaginary eigenvectors back to ascending K basis from Wang basis
            call bkwang( wangtr, veci, dmham, iact )
         end if
c        transform real eigenvectors back to ascending K basis from Wang basis
         call bkwang( wangtr, vecr, dmham, iact )
c !!!!!! check vector matrix for any incorrect sorting; this procedure
c        should be redundant after factorization of the Wang matrix
c        if ( swg+fcompg .eq. 0 )
c    *        call chevec( vecr, ener, tmpvec, dmham, iact )

c        store real (imaginary) eigenvectors as returned from diag routine in matrix vec
         call vecsto( evecgr, dmevec, vecr, dmham, ivpt, Jmax, J )
         if ( fcompl .ne. 0 )
     *        call vecsto( evecgi, dmevec, veci, dmham, ivpt, Jmax, J )
c        store (copy) eigenvalues as returned from diag routine into vector ener
         call dcopy( iact, ener, 1, evalg( iept + 1 ), 1 )

c        calculate energy derivatives if user wants to fit ground state parameters
         if ( fitges .eq. 1 ) then
c           Jx**2, Jy**2 and Jz**2 derivatives (rigid rotor, unswitched)
            call rrde( J, derivg, dmeval, npar, dmham, iact, hamr, vecr, veci, fcompg )
c           JyJx+JxJy, JzJx+JxJz and JzJy+JyJz derivatives (rr, axis switched)
            if ( swg .eq. 1 ) then
               call jyjxde( J, derivg, dmeval, npar, dmham, iact, hamr, vecr, veci )
               call jzjxde( J, derivg, dmeval, npar, dmham, iact, hamr, vecr, veci )
               call jzjyde( J, derivg, dmeval, npar, dmham, iact, hamr, vecr, veci )
c              rotate rr derivatives back to unrotated frame
               call rotder( derivg, dmeval, npar, trang, tring, iact, iept )
            end if
c           Jx, Jy, Jz derivatives
            if ( flinht .ne. 0 ) then
               if ( rotcog(4) .ne. 0. ) call jxde( J, derivg, dmeval, npar, dmham, iact, hamr, vecr, veci )
               if ( rotcog(5) .ne. 0. ) call jyde( J, derivg, dmeval, npar, dmham, iact, hamr, vecr, veci )
               if ( rotcog(6) .ne. 0. ) call jzde( J, derivg, dmeval, npar, dmham, iact, hamr, vecr, veci )
            end if
c           J**4, (J**2)*(Jz**2), Jz**4 etc. derivatives ( Watson hamiltonian quartic terms )
            if ( nparg .gt. 6 ) then
               call nrrde( J, derivg, dmeval, npar, dmham, iact, hamr, vecr, veci, fcompg )
            end if
         end if

#ifdef DEBUG_VERBOSE
         if ( ( shorti .eq. 0 ) .and. ( J .le. Jmxout ) ) then
c           output the real eigenvectors for J block
            call wrivec( vecr, dmham, J, icqn, dmeval, kind(0), lngbar, ldash )
c           output the imaginary eigenvectors for J block
            if ( fcompg .ne. 0 ) call wrivec( veci, dmham, J, icqn, dmeval, kind(1), lngbar, ldash )
c           output exact eigenvalues
            call eigval( ener, dmham, dmeval, J, icqn, ldash )
            if ( fitges .eq. 1 ) then
c              calculate energies with derivative approximation
               call derapp( ened, dmham, rotcog, npar, derivg, icqn, dmeval, J, lbars, ldash )
c              output the derivatives
               call derout( J, derivg, dmeval, npar, icqn, ldash )
            end if
         end if
#endif
      end do

c-----------------------------------------------------------------------------
c     excited state eigenvectors and eigenvalues
c-----------------------------------------------------------------------------
      if ( shorti .eq. 0 ) then
         write(*,'(a)') lstars
         write(*,*) 'excited state J block calculations:'
      end if

c     reduce amount of work when fitting to MW spectra
      if ( fitMW .ne. 0 ) then
         ilim1 = ( Jmxcal + 1 ) * ( Jmxcal + 1 )
         ilim2 = idint( ( ( 4.d0/3.d0*Jmxcal + 4.d0 )*Jmxcal + 11.d0/3.d0 )*Jmxcal + 1.5d0 )
         call dcopy( ilim1, evalg, 1, evale, 1 )
         call dcopy( ilim2, evecgr, 1, evecer, 1 )
         if ( fcompl .ne. 0 )
     *        call dcopy( ilim2, evecgi, 1, evecei, 1 )
         do k = 1, npar, 1
            do i = 1, ilim1, 1
               derive(i,k) = derivg(i,k)
            end do
         end do
      else
c        check if excited state hamiltonian contains linear contributions
         flinht = 0
         do i = 4, 6, 1
            if ( ( rotcoe(i) .ne. 0 ) .or.
     *           ( fitges .ne. 0 ) .and. ( ifit(i+npar) .ne. 0 ) )
     *           flinht = 1
         end do
         if ( swe .ne. 0 ) flinht = 1

c        excited state J block qn counter
         do J = 0, Jmxcal, 1

c           current offset
            iept = J*J

c           actual size of the J block and the greatest submatrix
            iact  = 2*J + 1
            iactr = J/2 + 1

c           calculate the Wang transformation matrix and its inverse
            call wang( wangtr, dmham, iact )

c           zero the hamiltonian and eigenvector work spaces
            call smat0( hamr, dmham, iact )
            call smat0( hami, dmham, iact )
c           veci is initialized later, if fcome==1, and doesn't need to be zeroed
c           if fcome==0 -- unless complex calculations have been done for ground state
            if ( fcompg .ne. 0 )
     *           call smat0( veci, dmham, iact )
c           all other hamiltonian submatrices and eigenvector matrices don't have
c           to be zeroed since each element is filled in explicitly

c           fill excited state hamiltonian
            if ( swe .eq. 1 ) then
c              axis switched rigid rotor
               call rrhfax( J, rraxsw(1,2), rraxsw(2,2), rraxsw(3,2), hamr, dmham )
c              off-diagonal axis switching terms
               if ( rraxsw(4,2) .ne. 0 ) call jzjy( hami, dmham, J, rraxsw(4,2) )
               if ( rraxsw(5,2) .ne. 0 ) call jzjx( hamr, dmham, J, rraxsw(5,2) )
               if ( rraxsw(6,2) .ne. 0 ) call jyjx( hami, dmham, J, rraxsw(6,2) )
            else
c              rigid rotor terms not axis switched
               call watson( J, rotcoe, npar, hamr, dmham )
            end if
c           linear angular momentum terms: assumed to be in axis switched frame
c           - real
            if ( rotcoe(4) .ne. 0 ) call jx( hamr, dmham, J, rotcoe(4) )
            if ( rotcoe(6) .ne. 0 ) call jz( hamr, dmham, J, rotcoe(6) )
c           - imaginary
            if ( rotcoe(5) .ne. 0 ) call jy( hami, dmham, J, rotcoe(5) )

#ifdef DEBUG_VERBOSE
            if ( ( shorti .eq. 0 ) .and. ( J .le. Jmxout ) ) then
c              output real hamiltonian matrix for J block
               call wriham( hamr, dmham, J, kind(0), frame(swe), lngbar, lstars )
c              output imaginary hamiltonian matrix for J block
               if ( fcompe .eq. 1 ) call wriham( hami, dmham, J, kind(1), frame(swe), lngbar, lstars )
            end if
#endif

c           transform real hamiltonian to Wang basis
            call trwang( wangtr, hamr, whamr, dmham, iact )
c           diagonalize the hamiltonian matrix: check for imaginary terms
            if ( fcompe .eq. 0 ) then
               if ( flinht .ne. 0 ) then
                  call rotdi( whamr, iact, dmham, ener, vecr, e, diaalg, shorti )
               else
                  call rotdip( J, iact, dmham, dmfwng, whamr, ener, vecr, diaalg, shorti, Jmxout )
               end if
            else
c              transform imaginary hamiltonian to Wang basis
               call trwang( wangtr, hami, whami, dmham, iact )
c              complex diagonalization
               call diagc ( whamr, whami, hamb, ener, enei, eneb, vecr, veci, vecb, J, dmham, eb, diaalg, shorti )
c              transform imaginary eigenvectors back to ascending K basis from Wang basis
               call bkwang( wangtr, veci, dmham, iact )
            end if
c           transform real eigenvectors back to ascending K basis from Wang basis
            call bkwang( wangtr, vecr, dmham, iact )
c !!!!!!    check vector matrix for any incorrect sorting; this procedure
c           should be redundant after factorization of the Wang matrix
c           if ( swe+fcompe .eq. 0 )
c    *           call chevec( vecr, ener, tmpvec, dmham, iact )

c           store real (imaginary) eigenvectors as returned from diag routine in matrix vec
            call vecsto( evecer, dmevec, vecr, dmham, ivpt, Jmax, J )
            if ( fcompl .ne. 0 )
     *        call vecsto( evecei, dmevec, veci, dmham, ivpt, Jmax, J )
c           store (copy) eigenvalues as returned from diag routine into vector ener
            call dcopy( iact, ener, 1, evale( iept + 1 ), 1 )
         
c           calculate energy derivatives if user wants to fit excited state parameters
            if ( fitges .eq. 1 ) then
c              Jx**2, Jy**2 and Jz**2 derivatives (rigid rotor, unswitched)
               call rrde( J, derive, dmeval, npar, dmham, iact, hamr, vecr, veci, fcompe )
c              JyJx+JxJy, JzJx+JxJz and JzJy+JyJz derivatives (rr, axis switched)
               if ( swe .eq. 1 ) then
                  call jyjxde( J, derive, dmeval, npar, dmham, iact, hamr, vecr, veci )
                  call jzjxde( J, derive, dmeval, npar, dmham, iact, hamr, vecr, veci )
                  call jzjyde( J, derive, dmeval, npar, dmham, iact, hamr, vecr, veci )
c                 rotate rr derivatives back to unrotated frame
                  call rotder( derive, dmeval, npar, trane, trine, iact, iept )
               end if
c              Jx, Jy, Jz derivatives
               if ( flinht .ne. 0 ) then
                  if ( rotcoe(4) .ne. 0. ) call jxde( J, derive, dmeval, npar, dmham, iact, hamr, vecr, veci )
                  if ( rotcoe(5) .ne. 0. ) call jyde( J, derive, dmeval, npar, dmham, iact, hamr, vecr, veci )
                  if ( rotcoe(6) .ne. 0. ) call jzde( J, derive, dmeval, npar, dmham, iact, hamr, vecr, veci )
               end if
c              J**4, (J**2)*(Jz**2), Jz**4 etc. derivatives (Watson hamiltonian quartic terms)
               if ( npare .gt. 6 ) then
                  call nrrde( J, derive, dmeval, npar, dmham, iact, hamr, vecr, veci, fcompe )
               end if
            end if

#ifdef DEBUG_VERBOSE
            if ( ( shorti .eq. 0 ) .and. ( J .le. Jmxout ) ) then
c              output the real eigenvectors for J block
               call wrivec( vecr, dmham, J, icqn, dmeval, kind(0), lngbar, ldash )
c              output the imaginary eigenvectors for J block
               if ( fcompe .eq. 1 ) call wrivec( veci, dmham, J, icqn, dmeval, kind(1), lngbar, ldash )
c              output exact eigenvalues
               call eigval( ener, dmham, dmeval, J, icqn, ldash )
               if ( fitges .eq. 1 ) then
c                 calculate energies with derivative approximation
                  call derapp( ened, dmham, rotcoe, npar, derive, icqn, dmeval, J, lbars, ldash )
c                 output the derivatives
                  call derout( J, derive, dmeval, npar, icqn, ldash )
               end if
            end if
#endif
         end do
      end if

c-----------------------------------------------------------------------------
c     line frequencies and intensities
c-----------------------------------------------------------------------------
      if ( shorti .eq. 0 ) then
         write(*,'(a)') lstars
         write(*,*) 'evaluating line intensities'
      end if

      cutor = cutint

c     start counter for number of calculated lines
  600 ntheli = 0

#ifdef DEBUG_VERBOSE
      if ( shorti .eq. 0 ) then
         open(30, file = 'debug.P', status = 'unknown')
         open(40, file = 'debug.Q', status = 'unknown')
         open(50, file = 'debug.R', status = 'unknown')
      end if
#endif

c     calculate all P, Q and R branch transitions J by J
      call pqr( Jmax, dmeval, dmevec, maxnli, npar,
     *          ntheli, icqn, lioflw, fcompl,
     *          derivg, derive, derlg, derle,
     *          evalg, evale,
     *          evecgr, evecgi, evecer, evecei,
     *          Jmxcal, dKmax, cutint,
     *          polcal,
     *          shorti,
     *          kind,
     *          cBran, polax1,
     *          lngbar, lbars, ldash,
     *          lqn, lqn2, intens, frethe,
     *          tempf1, tempf2, weight, nuspsw,
     *          nuzero,
     *          Jmxout,
     *          ivpt )

#ifdef DEBUG_VERBOSE
      if ( shorti .eq. 0 ) then
         close(30)
         close(40)
         close(50)
      end if
#endif
      
      if ( lioflw .ne. 0 ) then
c        increase cutint in a convenient way
         cutint = dmin1(cutint*2.d0, cutint + 2.d0)
         if ( shorti .eq. 0 ) then
            write(*,'(a)') lstars
            write(*,'(a,i3)') 'ERROR  : overflow in number of lines during calculation for J = ', lioflw
            write(*,'(9x,a,f9.6)') 'recalculating with new value of cutint:', cutint
         end if
         goto 600
      end if

      if ( ( cutint .ne. cutor ) .and. ( shorti .eq. 0 ) ) then
         write(*,'(a)') lstars
         write(*,'(a,f9.6)') 'WARNING: actual value for cutint is ', cutint
         write(*,'(a)') 'You should adopt this value into your input.'
      end if

c-----------------------------------------------------------------------------
c     statistical least squares fitting section
c-----------------------------------------------------------------------------
      if ( fitges .eq. 1 ) then
c        we want the fit, fitges gets back to 1 when the fit is done
         fitges = 0
         if ( shorti .eq. 0 ) then
            write(*,'(a)') lstars
            write(*,*) 'least squares analysis:'
            write(*,'(a)') lstars
         end if

c        restructure theoretical lines set in terms of packed quantum numbers
         call qnpac1( pqnthe, lqn, maxnli, ntheli )

c        sort derivative and line information by packed qn
         call linord( maxnli, npar, ntheli, nparg, npare,
     *                iop, derlg, derle, pqnthe,
     *                lqn, lqn2, intens, frethe )

c        clean array freexp
         do i = 1, ntheli, 1
            freexp(i) = 0.d0
         end do

c        establish correspondence between experimental and theoretical lines sets
c        initialize position to search in theoretical lines array at one
         istart = 1
         itheor = 1
c        loop through theoretical lines set for a match and convert frequency unit simultaneously
         do i = 1, nexpli, 1
            do k = istart, ntheli, 1
               if ( pqnexp(i) .eq. pqnthe(k) ) then
                  freexp(k) = freasn(i)*divis
                  itheor = k
               end if
            end do
            istart = itheor + 1
         end do

c        reset number of matches between experimental and calculated data
         ndata = 0
c        sum of squared deviations (observed minus calculated line positions)
         sdsum = 0.d0            
c        establish index to theoretical lines with experimental assignments
         do i = 1, ntheli, 1
            if ( freexp(i) .ne. 0 ) then
               ndata = ndata + 1
               sdsum = sdsum + ( fsrcor*freexp(i) - frethe(i) )**2
c              index the theoretical lines which are matched to experimental ones
               ipttem(ndata) = i
            end if
         end do
         if ( shorti .eq. 0 ) write(*,'(a,i6)') 'number of assignments matching calc. lines   :', ndata

c        output lines and leave this program part if zero data
         if ( ndata .le. 0 ) then
            if ( shorti .eq. 0 ) then
               write(*,*) 'WARNING: No matches between assigned and calculated lines!\n...leaving the fitting section'
            end if
            return
         end if

c        calculate reciprocal of fsrcor
         fsrdiv = 1.d0 / fsrcor

c        load the indexed derivatives into design matrix for least squares analysis
         call funcsl( funcs, maxnli, npar, ndata, ipttem, derlg, derle, fsrdiv )

c        if fitting delta constants ...
         if ( idelta .eq. 1 ) then
c           ... adapt array funcs to delta derivatives
            call delder( funcs, npar, maxnli, ndata )
c           ... convert the rotational constants to delta constants
            call delabs( rotcog, rotcoe, npar, 1 )
         end if

c        loop to load unit derivative for nuzero and experimental frequency vector
         do i = 1, ndata, 1
c           y:=uncorrected assigned experimental frequency in MHz
            y(i) = freexp(ipttem(i))
c           funcs:= dy/d(param)
c           nuzero derivative is always 1/fsrcor (=fsrdiv)
            funcs(i,2*npar+1) =  fsrdiv
c           fsrcor derivative is calculated in subroutine mrqcof
c           funcs(i,2*npar+2) = -fsrdiv * frethe(ipttem(i))
         end do

c        nparft is the number of currently varied parameters, initialized to its maximum possible value
         nparft = 2*npar+2
         if ( shorti .eq. 0 ) write(*,'(/,a,i6)') 'maximum number of parameters to be fit       :', nparft

c        counter for variant parameters, set to zero and incremented each time when nparft remains unchanged
c        finally, icount == nparft
         icount = 0

         if ( ifit(2*npar+2) .eq. 0 ) then
c           linear problem, fit by means of singular value decomposition
c           decrease number of model parameters by one (fsrcor)
            nparft = nparft - 1
c           loop through ground state contributions
            do k = 1, npar, 1
               if ( ifit(k) .eq. 0 ) then
                  nparft = nparft - 1
                  do i = 1, ndata, 1
                     y(i) = y(i) - funcs(i,k)*rotcog(k)
c                    zero that column of funcs
                     funcs(i,k) = 0.d0
                  end do
               else
                  icount = icount + 1
                  ifitpt(icount) = k
               end if
            end do
c           distinguish between 'normal' and MW fits
            if ( fitMW .eq. 0 ) then
c              normal treatment of excited state parameters
               do k = 1, npar, 1
                  if ( ifit(k+npar) .eq. 0 ) then
                     nparft = nparft - 1
                     do i = 1, ndata, 1
                        y(i) = y(i) - funcs(i,k+npar)*rotcoe(k)
c                       zero that column of funcs
                        funcs(i,k+npar) = 0.d0
                     end do
                  else
                     icount = icount + 1
                     ifitpt(icount) = k + npar
                  end if
               end do
c              nuzero contribution
               k = 2*npar + 1
               if ( ifit(k) .eq. 0 ) then
                  nparft = nparft - 1
                  do i = 1, ndata, 1
                     y(i) = y(i) - nuzero
c                    zero that column of funcs
                     funcs(i,k) = 0.d0
                  end do
               else
                  icount = icount + 1
                  ifitpt(icount) = k
               end if
            else
c              MW fit: decrease number of model parameters ( excited state parameters and nuzero are missing)
               nparft = nparft - npar - 1
c              clear the corresponding columns of funcs
               do k = npar+1, 2*npar+2, 1
                  do i = 1, ndata, 1
                     funcs(i,k) = 0.d0
                  end do
               end do
            end if
         else
c           non-linear problem, fit by means of Levenberg-Marquardt algorithm
c           distinction between active and invariant parameters only preliminary (for output)
            do k = 1, 2*npar+2, 1
               if ( ifit(k) .eq. 0 ) then
                  nparft = nparft - 1
               else
                  icount = icount + 1
                  ifitpt(icount) = k
               end if
            end do
         end if

         if ( shorti .eq. 0 ) then
            write(*,'(a,8x,a,i6)') 'number of currently fitted parameters',':', nparft
            write(*,'(/,24(i3))') (ifitpt(i), i = 1, nparft, 1)
         end if

c        if no parameters are active just output lines and finish
c        don't reset fitges to prevent program from breaking down at subroutine ideferr (utility.F)
         if ( ( nparft .eq. 0 ) .or. ( nparft .gt. ndata ) ) then
            if ( shorti .eq. 0 ) then
               if ( nparft .eq. 0 ) then
                  write(*,'(/,a)') 'WARNING: All empirical parameters are inactive!'
               else
                  write(*,'(/,a,i2,a,i2,a)') 'ERROR: ARNIROT cannot solve for ',nparft,' parameters with only ',ndata,' sets of data!'
               end if
               write(*,'(a)') '...leaving the fitting section'
            end if
            return
         end if
         if ( shorti .eq. 0 ) write(*,'(a)') lbars

c        give an initial guess for the set of fitted parameters a
c        (in principal needed only for non-linear problem, but now also for initialization of relchg)
         do i = 1, npar, 1
            a( i )      = rotcog( i )
            a( i+npar ) = rotcoe( i )
         end do
         a( 2*npar+1 ) = nuzero
         a( 2*npar+2 ) = fsrcor

c        initialize vector for relative changes with parameter start values
         do i = 1, 2*npar+2, 1
            relchg( i ) = a( i )
         end do

         if ( ifit(2*npar+2) .eq. 0 ) then
c           linear problem, fit by means of singular value decomposition
c           rearrange matrix FUNCS so that all nonzero columns are in the first NPARFT positions
c           save FUNCS to a temporary array FUNCST
            do k = 1, 2*npar+1, 1
               do i = 1, ndata, 1
                  funcst(i,k) = funcs(i,k)
               end do
            end do
c           copy into correct position
            do k = 1, nparft, 1
               do i = 1, ndata, 1
                  funcs(i,k) = funcst(i,ifitpt(k))
               end do
            end do

c           statistical analysis routine to empirically fit the rotational parameters
            call svdfit( y, sigma, ndata, a, nparft, u, v, w, maxnli, 2*npar+2, chisq, funcs, shorti )

c           calculate and output the covariance marix
            call svdvar( v, 2*npar+2, 2*npar+2, w, cvm, 2*npar+2 )

         else
c           non-linear problem, fit by means of Levenberg-Marquardt algorithm
            chisq  =  0.d0
            chisq0 =  0.d0
            lambda = -1.d0
            mqit   =  0
            mqct   =  0
#ifdef DEBUG_VERBOSE
            write(*,'(a)') 'mqit    chisq(old)       chisq(new)       lambda'
            write(*,'(a)') '---------------------------------------------------------'
#endif
c           do-while construct
  800       continue
               mqit = mqit + 1
c              save old value of fsrcor before entering mrqmin
               fsrco0 = a( 2*npar+2 )
               call mrqmin( y, sigma, ndata, a, ifit, cvm, alpha, 2*npar+2, chisq, funcs, maxnli, lambda )
#ifdef DEBUG_VERBOSE
               write(*,'(i6,3(2x,f15.9))') mqit, chisq0, chisq, lambda
#endif
c              correct the derivatives
               fac1 = fsrco0 / a( 2*npar+2 )
               do j = 1, 2*npar+1, 1
                  do i = 1, ndata, 1
                     funcs( i, j ) = funcs( i, j ) * fac1
                  end do
               end do
               diff = chisq0 - chisq
               if ( diff .gt. 0.d0 ) then
                  if ( diff .lt. chisq0*tol1 ) then
                     goto 900
                  else
                     mqit = mqit - 1
                     mqct = 0
                  end if
               else
                  mqct = mqct + 1
               end if
               chisq0 = chisq
            if ( ( mqit .lt. mqitmx ) .and. ( mqct .lt. 10 ) ) goto 800
  900       if ( shorti .eq. 0 ) then
               if ( mqit .ge. mqitmx ) then
                  write(*,*) 'mrqmin: No convergence after ', mqitmx, ' iterations.'
               else if ( mqct .ge. 10 ) then
                  write(*,*) 'mrqmin: No convergence after ', mqit, ' iterations.'
               else
                  write(*,*) 'mrqmin: Convergence after ', mqit, ' iterations.'
               end if
            end if
            call mrqmin( y, sigma, ndata, a, ifit, cvm, alpha, 2*npar+2, chisq, funcs, maxnli, 0.d0 )
         end if

         if ( shorti .eq. 0 ) then
c           output the covariance matrix
            h1 = ( nparft - 1 ) / 6
            do ii = 1, h1, 1
               write(*,'(a,i1,a,i1,a)') '\ncovariance matrix (part ',ii,' of ',h1+1,'):\n'
               do i = 1, nparft, 1
                  write(*,'(6(d13.4))') (cvm(i,k), k = 1, min0(nparft,6), 1)
               end do
            end do
            h2 = mod(nparft,6)
            if ( h2 .eq. 0 ) h2 = 6
            write(*,'(a,i1,a,i1,a)') '\ncovariance matrix (part ',h1+1,' of ',h1+1,'):\n'
            do i = 1, nparft, 1
               write(*,'(6(d13.4))') (cvm(i,k), k = h1*6+1, h1*6+h2, 1)
            end do
         end if

c        reset parameter errors (without this initialization, the program happened to terminate abnormally!)
         do i = 1, 2*npar+2, 1
            err( i ) = 0.d0
         end do
c        load the diagonal covariance matrix elements into the error vector
         do i = 1, nparft, 1
            if ( cvm(i,i) .ge. 0. ) then
               err( ifitpt(i) ) = dsqrt( cvm(i,i) )
            else
               if ( shorti .eq. 0 ) write(*,*) 'negative diagonal element #',i,' in covariance matrix'
               err( ifitpt(i) ) = dsqrt( dabs( cvm(i,i) ) )
            end if
         end do

c        rearrange A (fit parameters) back to initial order
         do i = 1, 2*npar+2, 1
            atmp( i ) = a( i )
         end do
c        load original constants back in
         do i = 1, npar, 1
            a( i )      = rotcog( i )
            a( i+npar ) = rotcoe( i )
         end do
         a( 2*npar+1 ) = nuzero
         a( 2*npar+2 ) = fsrcor

c        load in the A constants that have been changed
         if ( ifit(2*npar+2) .eq. 0 ) then
            do i = 1, nparft, 1
               a( ifitpt(i) ) = atmp( i )
            end do
         else
            do i = 1, nparft, 1
               a( ifitpt(i) ) = atmp( ifitpt(i) )
            end do
         end if

         if ( shorti .eq. 0 ) then
c           output least squares analysis summary
            write(*,'(a)') lstars
            write(*,*) 'least squares analysis summary:'
            write(*,'(a)') lstars
c           output initial and final values of constants with standard deviations
            write(*,'(a)') 'ground state constants:\n'
            write(*,'(a)') 'parameter      OLD value       NEW value                       NEW-OLD\n'
            write(*,'(a)') '[MHz]'
            do i = 1, min0( nparg, 6 ), 1
               write(*,'(a12,f15.6,1x,f15.6,''+-'',f13.6,1x,f15.6)')
     *              cRotCg(i), rotcog(i), a(i), err(i), a(i) - rotcog(i)
            end do
            if ( nparg .gt. 6 ) then
               write(*,'(a)') '[kHz]'
               do i = 7, nparg, 1
                  write(*,'(a12,f15.6,1x,f15.6,''+-'',f13.6,1x,f15.6)')
     *                 cRotCg(i), rotcog(i)*1.d3, a(i)*1.d3, err(i)*1.d3, (a(i) - rotcog(i))*1.d3
               end do
            end if
            if ( fitMW .eq. 0 ) then
               write(*,'(a)') lbars
               write(*,'(a)') 'excited state constants:\n'
               write(*,'(a)') 'parameter      OLD value       NEW value                       NEW-OLD\n'
               write(*,'(a)') '[MHz]'
               if ( idelta .eq. 0 ) then
                  do i = 1, min0( npare, 6 ), 1
                     write(*,'(a12,f15.6,1x,f15.6,''+-'',f13.6,1x,f15.6)')
     *                    cRotCe(i), rotcoe(i), a(i+npar), err(i+npar), a(i+npar) - rotcoe(i)
                  end do
               else
                  do i = 1, min0( npare, 6 ), 1
                     write(*,'(a12,f15.6,1x,f15.6,''+-'',f13.6,1x,f15.6)')
     *                    cDRotC(i), rotcoe(i), a(i+npar), err(i+npar), a(i+npar) - rotcoe(i)
                  end do
               end if
               if ( npare .gt. 6 ) then
                  write(*,'(a)') '[kHz]'
                  if ( idelta .eq. 0 ) then
                     do i = 7, npare, 1
                        write(*,'(a12,f15.6,1x,f15.6,''+-'',f13.6,1x,f15.6)')
     *                       cRotCe(i), rotcoe(i)*1.d3, a(i+npar)*1.d3, err(i+npar)*1.d3, (a(i+npar) - rotcoe(i))*1.d3
                     end do
                  else
                     do i = 7, npare, 1
                        write(*,'(a12,f15.6,1x,f15.6,''+-'',f13.6,1x,f15.6)')
     *                       cDRotC(i), rotcoe(i)*1.d3, a(i+npar)*1.d3, err(i+npar)*1.d3, (a(i+npar) - rotcoe(i))*1.d3
                     end do
                  end if
               end if
c              output nuzero data
               write(*,'(a)') lbars
               write(*,'(/,''nuzero      old:'',f15.3,'' MHz'',/,12x,''new:'',f15.3,'' +- '',f9.3,'' MHz'',/,8x,''new-old:'',f15.3,'' MHz'')')
     *              nuzero, a(2*npar+1), err(2*npar+1), a(2*npar+1) - nuzero
c              output fsrcor data
               write(*,'(a)') lbars
               write(*,'(/,''fsrcor      old:'',f15.9,/,12x,''new:'',f15.9,'' +- '',f11.9,/,8x,''new-old:'',f15.9)')
     *              fsrcor, a(2*npar+2), err(2*npar+2), a(2*npar+2) - fsrcor
            end if

            write(*,'(a)') lbars
c           uncertainty assumed for each line assignment
            write(*,'(''assumed uncertainty for each line assignment :'',f10.3,'' MHz'')') sigma
c           number of calculated lines
            write(*,'(''number of generated theoretical lines        :'',i6)') ntheli
c           number of assigned lines
            write(*,'(''number of assigned experimental lines        :'',i6)') nexpli
c           number of experimental lines which have been matched to theoretical lines
            write(*,'(''number of exp. lines matched to theor. lines :'',i6)') ndata
c           average observed minus calculated line positions
            write(*,'(''average obs. minus calc. line position [rms] :'',f10.3,'' MHz'')') dsqrt(sdsum/ndata)
c           chi-square
            write(*,'(34x,a,f17.10)') 'Chi-square :', chisq
            write(*,'(22x,a,f17.10)') 'fsrcor**2 * Chi-square :', chisq * fsrcor**2

         end if

c        equate old rotational constants to new improved least squares constants
         do i = 1, npar, 1
            rotcog( i ) = a( i )
            rotcoe( i ) = a( i+npar )
         end do
         nuzero = a( 2*npar+1 )
         fsrcor = a( 2*npar+2 )

c        indicate success of fit
         fitges = 1
      end if

c-----------------------------------------------------------------------------
c     end of outer iteration loop
c-----------------------------------------------------------------------------
c     increase loop counter and restart calculation or terminate iteration
      olit = olit + 1
      if ( fitges .eq. 1 ) then
         diff = chisq1 - chisq
         if ( ( short0 .eq. 0 ) .and. ( diff .lt. 0. ) )
     *        write(*,*) '\nWARNING: bad fit!'
         chisq1 = chisq

c        determine relative changes of fitted parameters
         maxchg = 0.d0
         do i = 1, nparft, 1
            tmpchg = relchg( ifitpt(i) )
            if ( tmpchg .ne. 0. )
     *           relchg( ifitpt(i) ) = dabs( ( tmpchg - a( ifitpt(i) ) ) / tmpchg )
            if ( relchg( ifitpt(i) ) .gt. maxchg ) maxchg = relchg( ifitpt(i) )
         end do

c        stop iteration if tolerance limit is reached
c        if ( ( diff .ge. 0. ) .and. ( diff .lt. tol2 ) ) olit = max0( olit, olitmx )
         if ( maxchg .lt. tol2 ) olit = max0( olit, olitmx )
      end if
      if ( olit .lt. olitmx ) then
         goto 300
      else
         shorti = short0
         if ( ( olit .eq. olitmx ) .and. ( olitmx .gt. 1 ) ) then
c           reset final calculation and output to full scale
            Jmxcal = Jmxtmp
            dKmax  = dKmtmp
            goto 300
         end if
      end if

      if ( fitges .eq. 1 ) then
         if ( shorti .eq. 0 ) then
c           output maximum value for J and DeltaK during iteration
            write(*,'(/,a,i6)') 'Jmax  in assignment file / during iteration  :', Jmxexp
            write(*,'(a,i6)') 'dKmax in assignment file / during iteration  :', dKmexp
c           calculate new inertial defect and uncertainties
            write(*,'(a)') lbars
            call ideferr( rotcog, rotcoe, npar, err, idelta )
         end if
      end if

      return
      end


cc Local Variables:
cc mode: FORTRAN
cc End:
