#include "xyplot.h"

#include <iostream.h>
#include <math.h>
#include <stdlib.h>

#include "debug.h"
#include "centre.h"

#include <kiconloader.h>
#include <kapp.h>



static double fract (double x)
{
  return x-(int)x;
}


static void get_scale (int n, double a, double b, double& start, double& delta)
{
  //
  // UltraSmart(tm) axis labelling calculation. Tries to get as close as
  // possible to the desired number of labels (n), using a
  // "standard" stepwidth (.1 .2 .25 .5 or 1.)
  //
  // This is possibly the most elegant way to get the optimal stepwidth.
  // If you know of a better way, please email me at
  //     schemitz@ekp.physik.uni-karlsruhe.de
  //
  // Originally written for the Xtreme project in 1997.
  //
  double mant;
  int expo;

  delta = (b-a)/n;
  expo = (int)(log10(delta));
  mant = delta/pow(10.0,expo);
  if (mant<0.15)
	delta=0.1;
  else if (mant<0.23)
	delta=0.2;
  else if (mant<0.35)
	delta=0.25;
  else if (mant<0.81)
	delta=0.5;
  else
	delta=1.0;
  delta *= pow(10.0,expo);
  start = delta*int(a/delta);
  if (fract(a/delta)>0.001)
	start += delta;

  IFDEBUG("korigin.xyplot",4)
	cout << "get_scale: n=" << n << ", a=" << a << ", b=" << b << ", start=" 
		 << start << ",delta=" << delta << endl;
  ENDDEBUG;
}


void XYPlot::paint (QPainter* p)
{
  if (!prepare()) return;
  if (!paintAxis(p)) return;
  if (!paintSets(p)) return;
  IFDEBUG("korigin.xyplot",3)
	cout << "XYPlot::paint(): all stages successful." << endl;
  ENDDEBUG;
}


bool XYPlot::prepare ()
{
  uint i;
  double xa, xb, ya, yb;

  xb = yb = 1.0E-100;
  xa = ya = 1.0E+100;
  for (i=0; i<records.count(); i++)
	if (!findSetRange(records.at(i),xa,xb,ya,yb)) return false;

  IFDEBUG("korigin.xyplot",4)
    cout << "XYPlot::prepare(): found range: "
		 << "x = [ " << xa << ", " << xb << " ], "
		 << "y = [ " << ya << ", " << yb << " ]" << endl;
  ENDDEBUG;

  if (axis->x_range.mode == 0)
	axis->x_range.from = xa, axis->x_range.to = xb;
  else
	if (axis->x_range.from == 1.0E+100)
	  axis->x_range.from = axis->x_range.to = 0.0;

  if (axis->y_range.mode == 0)
	axis->y_range.from = ya, axis->y_range.to = yb;
  else
	if (axis->y_range.from == 1.0E+100)
	  axis->y_range.from = axis->y_range.to = 0.0;

  IFDEBUG("korigin.xyplot",3)
    cout << "XYPlot::prepare(): use range: "
		 << "x = [ " << axis->x_range.from << ", " << axis->x_range.to 
		 << " ], "
		 << "y = [ " << axis->y_range.from << ", " << axis->y_range.to
		 << " ]" << endl;
  ENDDEBUG;
  return true;
}


bool XYPlot::paintAxis (QPainter* p)
{
  int height = widget()->height();
  int width = widget()->width();
  int ptsize;
  int s;
  char buffer [128];
  double x, y;
  double scalex, scaley, delta, start;
  QFont font;

  //
  // Calc. scaling factors
  //
  scalex = width / (axis->x_range.to - axis->x_range.from);
  scaley = height / (axis->y_range.to - axis->y_range.from);
  //
  // Draw X axis
  //
  p->setPen(axis->x_range.pen);
  p->drawLine(0,0,width,0);
  p->drawLine(0,height,width,height);
  if (axis->x_range.show_zero && axis->y_range.from <= 0.0 
	  && axis->y_range.to >= 0.0) {
	p->drawLine(0, y2pixel(0.0),
				width, y2pixel(0.0));
  }
  //
  // Draw Y axis
  //
  p->setPen(axis->y_range.pen);
  p->drawLine(0,0,0,height);
  p->drawLine(width,0,width,height);
  if (axis->y_range.show_zero && axis->x_range.from <= 0.0 
	  && axis->x_range.to >= 0.0) {
	p->drawLine(x2pixel(0.0),0,
				x2pixel(0.0),height);
  }
  //
  // Skip labelling if were resizing - its too slow otherwise.
  //
  if (resizing()) return true;
  //
  // Label X axis
  //
  font = axis->x_range.font;
  ptsize = font.pointSize();
  // Factor of 4 due to resolution difference between screen (75 dpi) 
  // and postscript laser (presumably 300 dpi).
  font.setPointSize(4 * ptsize);
  p->setFont(font);

  switch (axis->x_subticks.mode) {
  case 0:
	s = width / 50;
	get_scale(s,axis->x_range.from,axis->x_range.to,start,delta);
	break;
  case 1:
	delta = fabs(axis->x_subticks.size); // fabs() for sanity check
	start = delta * int(axis->x_range.from / delta);
	if (fract(axis->x_range.from / delta) > 0.001)
	  start += delta;
	break;
  case 2:
	get_scale(axis->x_subticks.steps,axis->x_range.from,
			  axis->x_range.to,start,delta);
	break;
  default:
	cout << "x_subticks unknown mode " << axis->x_subticks.mode << endl;
  }
  if (delta == 0.0)	return false;
  for (x=start; x<axis->x_range.to; x+=delta)
	{
	  int len = int(4 * axis->x_subticks.tick_len);
	  int xi = x2pixel(x);
	  if (axis->x_subticks.grid) {
		p->setPen(axis->x_subticks.grid_pen);
		p->drawLine(xi,0,xi,height);
	  }
	  p->setPen(axis->x_subticks.tick_pen);
	  p->drawLine(xi,height-len,xi,height);
	}

  switch (axis->x_ticks.mode) {
  case 0:
	s = width / 100;
	get_scale(s,axis->x_range.from,axis->x_range.to,start,delta);
	break;
  case 1:
	delta = fabs(axis->x_ticks.size); // fabs() for sanity check
	start = delta * int(axis->x_range.from / delta);
	if (fract(axis->x_range.from / delta) > 0.001)
	  start += delta;
	break;
  case 2:
	get_scale(axis->x_ticks.steps,axis->x_range.from,
			  axis->x_range.to,start,delta);
	break;
  }
  if (delta == 0.0) return false;
  for (x=start; x<axis->x_range.to; x+=delta)
	{
	  int len = int(4 * axis->x_ticks.tick_len);
	  int xi = x2pixel(x);
	  if (axis->x_ticks.grid) {
		p->setPen(axis->x_ticks.grid_pen);
		p->drawLine(xi,0,xi,height);
	  }
	  p->setPen(axis->x_ticks.tick_pen);
	  p->drawLine(xi,height-len,xi,height);
	  p->setPen(axis->x_range.color);
	  if (fabs(x)>0.0000001)
		XYPlotFormatList().at(axis->x_range.fmt)->formatter(buffer,x);
	  else
		XYPlotFormatList().at(axis->x_range.fmt)->formatter(buffer,0.0);
	  QRect area = p->boundingRect(0,0,width,height,AlignLeft|AlignTop,
								   buffer);
	  if (axis->x_range.label_at_zero)
		p->drawText(xi-area.width()/2,y2pixel(0.0)-int(0.1*area.height()),
					buffer);
	  else
		p->drawText(xi-area.width()/2,height-int(1.1*area.height()),
					buffer);
	}
  //
  // Label Y axis
  //
  font = axis->y_range.font;
  ptsize = font.pointSize();
  font.setPointSize(4 * ptsize);
  p->setFont(font);

  switch (axis->y_subticks.mode) {
  case 0:
	s = height / 50;
	get_scale(s,axis->y_range.from,axis->y_range.to,start,delta);
	break;
  case 1:
	delta = fabs(axis->y_subticks.size); // fabs() for sanity check
	start = delta * int(axis->y_range.from / delta);
	if (fract(axis->y_range.from / delta) > 0.001)
	  start += delta;
	break;
  case 2:
	get_scale(axis->y_subticks.steps,axis->y_range.from,
			  axis->y_range.to,start,delta);
	break;
  }
  if (delta == 0.0)	return false;
  for (y=start; y<axis->y_range.to; y+=delta)
	{
	  int len = int(4 * axis->y_subticks.tick_len);
	  int yi = y2pixel(y);
	  if (axis->y_subticks.grid) {
		p->setPen(axis->y_subticks.grid_pen);
		p->drawLine(0,yi,width,yi);
	  }
	  p->setPen(axis->y_subticks.tick_pen);
	  p->drawLine(0,yi,len,yi);
	}

  switch (axis->y_ticks.mode) {
  case 0:
	s = height / 25;
	get_scale(s,axis->y_range.from,axis->y_range.to,start,delta);
	break;
  case 1:
	delta = fabs(axis->y_ticks.size); // fabs() for sanity check
	start = delta * int(axis->y_range.from / delta);
	if (fract(axis->y_range.from / delta) > 0.001)
	  start += delta;
	break;
  case 2:
	get_scale(axis->y_ticks.steps,axis->y_range.from,
			  axis->y_range.to,start,delta);
	break;
  }
  if (delta == 0.0) return false;
  for (y=start; y<axis->y_range.to; y+=delta)
	{
	  int len = int(4 * axis->y_subticks.tick_len);
	  int yi = y2pixel(y);
	  if (axis->y_ticks.grid) {
		p->setPen(axis->y_ticks.grid_pen);
		p->drawLine(0,yi,width,yi);
	  }
	  p->setPen(axis->x_ticks.tick_pen);
	  p->drawLine(0,yi,len,yi);
	  p->setPen(axis->x_range.color);
	  if (fabs(y)>0.0000001)
		XYPlotFormatList().at(axis->y_range.fmt)->formatter(buffer,y);
	  else
		XYPlotFormatList().at(axis->y_range.fmt)->formatter(buffer,0.0);
	  QRect area = p->boundingRect(0,0,width,height,AlignLeft|AlignTop,
								   buffer);
	  if (axis->y_range.label_at_zero)
		p->drawText(x2pixel(0.0)+int(0.1*area.width()),yi-area.height()/2,
					buffer);
	  else
		p->drawText(int(0.1*area.width()),yi-area.height()/2,
					buffer);
	}
  //
  // Were done.
  //
  return true;
}


bool XYPlot::paintSets (QPainter* p)
{
  int i, n;
  int u, v, old_u, old_v;
  int ms;
  double x, y;
  double sx, sy;
  Column xcol;
  Column ycol;

  uint set_nr;
  XYPlotRecord* set;

  page()->getScaleFactors(sx,sy);

  for (set_nr=0; set_nr<records.count(); set_nr++)
	{
	  set = records.at(set_nr);
	  ms = set->marker_size;
	  xcol = set->input[0];
	  ycol = set->input[1];

	  if (xcol.type() != Column::columnDouble
		  || ycol.type() != Column::columnDouble)
		{
		  IFDEBUG("korigin.xyplot",1)
			cout << "XYPlot::paintSets(): no numeric data!"
				 << endl;
		  ENDDEBUG;
		  return false;
		}

	  n = xcol.dim();

	  IFDEBUG("korigin.xyplot",3)
		cout << "XYPlot::paintSet() " << ycol.title() << " vs. "
			 << xcol.title() << ", " << n << " values." << endl;
	  ENDDEBUG;

	  x = xcol[0];
	  y = ycol[0];
	  u = x2pixel(x);
	  v = y2pixel(y);
	  for (i=1; i<n; i++)
		{
		  IFDEBUG("korigin.xyplot",4)
			if (i%20 == 0) cout << "  data #" << i << "ff." << endl;
		  ENDDEBUG;
		  x = xcol[i];
		  y = ycol[i];
		  if (!isnan(x) && !isnan(y))
			if (x >= axis->x_range.from && x <= axis->x_range.to &&
				y >= axis->y_range.from && y <= axis->y_range.to) {
			  old_u = u;
			  old_v = v;
			  u = x2pixel(x);
			  v = y2pixel(y);
			  p->setPen(set->pen);
			  p->drawLine(old_u,old_v,u,v);
			  p->setPen(set->color);
			  switch (set->marker) {
			  case XYPlotRecord::NoMarker:
				break;
			  case XYPlotRecord::Square:
				p->drawRect(u-int(sx*ms),v-int(sy*ms),
							int(sx*(2*ms+1)),int(sy*(2*ms+1)));
				break;
			  case XYPlotRecord::Diamond:
				p->drawLine(u-int(sx*ms),v,u,v-int(sy*ms));
				p->drawLine(u,v-int(sy*ms),u+int(sx*ms),v);
				p->drawLine(u+int(sx*ms),v,u,v+int(sy*ms));
				p->drawLine(u,v+int(sy*ms),u-int(sx*ms),v);
				break;
			  case XYPlotRecord::Circle:
				p->drawEllipse(u-int(sx*ms),v-int(sy*ms),
							   int(sx*(2*ms+1)),int(sy*(2*ms+1)));
				break;
			  case XYPlotRecord::Point:
				p->drawPoint(u,v);
				break;
			  }
			}
		}
	}
  IFDEBUG("korigin.xyplot",3)
	cout << "XYPlot::paintSets() done." << endl;
  ENDDEBUG;
  return true;
}


bool XYPlot::findSetRange (XYPlotRecord* set, double& xmin, double& xmax,
						   double& ymin, double& ymax)
{
  int i, n;
  double x, y;
  Column xcol;
  Column ycol;
  xcol = set->input[0];
  ycol = set->input[1];
  if (xcol.type() != Column::columnDouble
	  || ycol.type() != Column::columnDouble)
	{
	  IFDEBUG("korigin.xyplot",1)
		cout << "XYPlot::findSetRange(): no numeric data: "
			 << xcol.title() << ", " << ycol.title()
			 << endl;
	  ENDDEBUG;
	  return false;
	}
  n = xcol.dim();
  for (i=0; i<n; i++)
	{
	  x = xcol[i];
	  y = ycol[i];
	  if (!isnan(x) && !isnan(y)) {
		if (x < xmin) xmin = x;
		if (x > xmax) xmax = x;
		if (y < ymin) ymin = y;
		if (y > ymax) ymax = y;
	  }
	}
  return true;
}


int XYPlot::x2pixel (double x)
{
  if (axis->log_x)
	{
	  return int(the_widget->width()*(log(x)-log(axis->x_range.from))
				 / (log(axis->x_range.to) - log(axis->x_range.from)));
	}
  else
	{
	  return int(the_widget->width()*(x-axis->x_range.from)
				 / (axis->x_range.to - axis->x_range.from));
	}
}

int XYPlot::y2pixel (double y)
{
  if (axis->log_y)
	{
	  return the_widget->height() 
		- int(the_widget->height() * (log(y) - log(axis->y_range.from))
			  / (log(axis->y_range.to)-log(axis->y_range.from)));
	}
  else
	{
	  return the_widget->height()
		- int(the_widget->height() * (y-axis->y_range.from)
			  / (axis->y_range.to - axis->y_range.from));
	}
}

double XYPlot::pixel2x (int u)
{
  if (axis->log_x)
	{
	  return exp( log(axis->x_range.from) 
				  + u * (log(axis->x_range.to)-log(axis->x_range.from)) / the_widget->width() );
	}
  else
	{
	  return axis->x_range.from + u * (axis->x_range.to-axis->x_range.from) / the_widget->width();
	}
}

double XYPlot::pixel2y (int v)
{
  if (axis->log_y)
	{
	  return exp( log(axis->y_range.from) 
				  + (the_widget->height()-v) * (log(axis->y_range.to)-log(axis->y_range.from)) / the_widget->height() );
	}
  else
	{
	  return axis->y_range.from + (the_widget->height()-v) * (axis->y_range.to-axis->y_range.from) / the_widget->height();
	}
}
