/* Emacs, this is -*- C++ -*- */

#ifndef _HISTO_H_
#define _HISTO_H_


#include <qpopmenu.h>
#include <qpixmap.h>
#include <qpainter.h>
#include <qpen.h>

#include "column.h"
#include "plot.h"



/** One Histogram Data Set.


	Every data set of the histogram has a set of properties,
	as defined in this class. (Actually, this class is more
	similar to a struct, but since it is local to HistoPlot,
	I didnt care to write 2n functions value() and setValue().)

	@author Patrick Schemitz
*/

class HistoPlotRecord
{
public:
  /// Constructor. Allocates given number of bins.
  HistoPlotRecord (int bins);

  /// Destructor. Frees bins.
  ~HistoPlotRecord ();

  /// Color of the data set.
  QColor color;

  /// The data to be plotted.
  Column data;
  
  /// Has the data set to be binned?
  bool re_bin;

  /// The bins.
  int* bin;
};



/** 1D Histogram class.


	This class represents a simple 1D Histogram. Also you can
	use it for simple analysis, its more a demonstration of how
	to write plots rather than a "real" plot.

	However, it shows how to implement your own plots.

	Functions required by the PageObject and Plot abstract base
	classes are only briefly commented. Functions special to
	HistoPlot are commented more verbosely.

	@author Patrick Schemitz
*/

class HistoPlot : public Plot
{
  Q_OBJECT

public:

  /// Constructor. Builds a HistoPlot "hull" which is filled by instantiate().
  HistoPlot ();

  /// Destructor. Shuts down a HistoPlot.
  ~HistoPlot ();

  /// Actually instantiates a HistoPlot on a Page, as required by Plot.
  void instantiate (Page*, QWidget* parent=0, const char* name=0, WFlags f=0);

  /// Clones itself, as required by Plot.
  Plot* clone ();

  /// Return name of the plot, as required by Plot.
  const char* name ();

  /// Return description of the plot, as required by PageObject.
  const char* description ();

  /// Return menu of the plot, as required by PageObject.
  QPopupMenu* menu (QPopupMenu* predefined = 0);

  /// Return icon of the plot, as required by PageObject.
  QPixmap* icon ();

  /// Stores options of the plot, as required by Plot.
  void storeOptions () {}

  /// Restores options of the plot, as required by Plot.
  void restoreOptions () {}

  const char* start_mark () { return "<histoplot>\n"; }
  const char* end_mark () { return "</histoplot>\n"; }

  /// Stores the plot, as required by PageObject.
  bool store (QFile& file) { return store(file,true); }
  bool store (QFile&, bool);

  /// Restores the plot, as required by PageObject.
  bool restore (QFile& file) { return restore(file,true); }
  bool restore (QFile&, bool);

  /// Return no. of parameters of the plot, as required by Plot.
  int paramCount ();

  /// Return parameter names of the plot, as required by Plot.
  const char* paramName (int i);

  /// Paints the plot, as required by PageObject.
  void paint (QPainter*);

  /// Handles a mouse event, as required by PageObject.
  void mousePressEvent (QMouseEvent*);

public slots:

  /** Execute Dataset Dialog.

	  The dataset dialog lets you select the color of the single
	  datasets. With X/Y plots, it would also allow you to select
	  the marker type (diamonds, circles, points, none etc).
  */
  void datasetDialog () {}

  /// Executes the options dialog, as required by Plot.
  void optionsDialog () {}

  /// Plots the selected columns of the active worksheet, as required by Plot.
  void plotSelectedColumns ();

private:

  /// The popup menu of the plot.
  QPopupMenu* the_menu;
  /// Font for axis description.
  QFont* axisFont;
  /// Color of axis and axis description.
  QColor* axisColor;
  /// Color of the frame.
  QColor* frameColor;
  /// Automagic or fixed x/y range.
  bool auto_x, auto_y;
  /// X range.
  double xmin, xmax;
  /// Y range.
  double ymin, ymax;
  /// No. of bins. Each dataset has the same number of bins.
  int bins;

  /// calc pixel x coord from double value.
  int x2pixel (double x);
  /// calc pixel y coord from double value.
  int y2pixel (double y);
  /// calc double from widget x coord.
  double pixel2x (int u);
  /// calc double from widget y coord.
  double pixel2y (int v);

  /// Container for datasets.
  QList<HistoPlotRecord> records;

  /// Set the x range to zero.
  void reset_xrange ();
  /// Set the y range to zero.
  void reset_yrange ();
  /// Find xmin and xmax from data column.
  bool find_xrange ();
  /// Find ymin and ymax from bins.
  void find_yrange ();
  /// Do the bin sorting.
  bool binning ();

private slots:

  /// slot for re-binning menu entry.
  void slotRebin ();
  /// slot for unimplemented features.
  void slotDumbaer ();

};


#endif
