
#include "ksiag_app.h"
#include "ksiag_table.h"
#include "ksiag_toplevel.h"
#include <unistd.h>
#include <qstrlist.h>
#include <qkeycode.h>
#include <kapp.h>

#define C_BIT 31
#define M_BIT 0x80
#define SIAG_CTRL(x) ((x)&C_BIT)
#define SIAG_ALT(x) ((x)|M_BIT)

char *colors[8] = {"black", "red", "green", "blue", "yellow", "purple",
    "cyan", "white"};
char *fonts[4] = {"courier", "helvetica",
    "new century schoolbook", "times"};
int fontsizes[] = {8, 10, 12, 14, 18, 24, 20, 30};
    
int grid_lines = 1;
    
KSiagTable::KSiagTable (window *win, QWidget *parent, const char *name)
    : QTableView (parent, name)
{
    int i;

    w = win;
    brush = new QBrush(SolidPattern);
    font = new QFont;
    h_total = w->buf->sh * BUFFER_ROWS;
    w_total = w->buf->sw * BUFFER_COLS;
    setNumRows(BUFFER_ROWS);
    setNumCols(BUFFER_COLS);
    setFocusPolicy(QWidget::StrongFocus);
    setFocus();
    
    // We shouldn't have any nonstandard widths/heights yet, but just in case.
    for (i=0; i < w->buf->used_lines; ++i){
        h_total -= w->buf->sh;
        h_total += w->buf->height[i];
    }
    for (i=0; i < w->buf->used_cols; ++i){
        w_total -= w->buf->sw;
        w_total += w->buf->width[i];
    }
    updateTableSize();
    blk_row = -1;
    cur_row = cur_col = 0;
    in_blk_select = 0, shift_down = 0;
}

KSiagTable::~KSiagTable()
{
    delete font;
    delete brush;
}


int KSiagTable::cellHeight (int row)
{

    return (cell_height(w->buf, row+1));
}

int KSiagTable::cellWidth (int col)
{
    return (cell_width(w->buf, col+1));
}

int KSiagTable::totalWidth()
{
    return (w_total);
}

int KSiagTable::totalHeight()
{
    return (h_total);
}

void KSiagTable::resetBuffer()
{
    int i;
    h_total = w->buf->sh * BUFFER_ROWS;
    w_total = w->buf->sw * BUFFER_COLS;
    
    for (i=0; i < w->buf->used_lines; ++i){
        h_total -= w->buf->sh;
        h_total += w->buf->height[i];
    }
    for (i=0; i < w->buf->used_cols; ++i){
        w_total -= w->buf->sw;
        w_total += w->buf->width[i];
    }
    updateTableSize();
    repaint(false);
}

window* KSiagTable::siagWindow()
{
    return (w);
}

void KSiagTable::setCursor ()
{
    int row, col;
    int y;
    row = w->point_pos.row - 1;
    col = w->point_pos.col - 1;

    rowYPos(cur_row, &y);
    repaint(0, y, width(), 2, false);
    if (row < topCell())
        emit scrollVertical(row);
    else if (row > lastRowVisible())
        emit scrollVertical(topCell()+row-lastRowVisible());
    if (col < leftCell())
        emit scrollHorizontal(col);
    else if (col > lastColVisible())
        emit scrollHorizontal(leftCell()+col-lastColVisible());
    if (row != cur_row)
    updateCell(row, col);

    // blk_x is so the we don't consider a mouse move in the cursor a block
    // select
    cur_row = blk_row = row;
    cur_col = blk_col = col;
    emit cursorChanged(w);
}

void KSiagTable::updateBlock (position blku, position blkl)
{
    int x1, y1, x2, y2;
    colXPos(blku.col-1, &x1);
    rowYPos(blku.row-1, &y1);
    colXPos(blkl.col, &x2);
    rowYPos(blkl.row, &y2);
    repaint (x1, y1, x2-x1, y2-y1, false);
}

void KSiagTable::newBlock()
{
    emit blockChanged(w);
}
    
void KSiagTable::focusInEvent (QFocusEvent *ev)
{
    if (w != w_list)
        emit activated(w);
    shift_down = 0; //Don't ask me
}

void KSiagTable::focusOutEvent (QFocusEvent *ev)
{
    ;
}
/**
 * This forces cells to be updated from right to left. It is needed so text
 * isn't overwritten when an adjoining cell is painted. It's also more
 * efficent than the QTableView paintEvent (no clipping check or anything)...
 */

void KSiagTable::paintEvent(QPaintEvent *ev)
{
    QRect r = ev->rect();
    int rcell=findCol(r.right()), lcell=findCol(r.left());
    int tcell=findRow(r.top()), bcell=findRow(r.bottom());
    int x, y;
    int cell;
    QPainter p(this);
    while (tcell <= bcell){
        for(cell=rcell; cell >= lcell; --cell){
            colXPos(cell, &x);
            rowYPos(tcell, &y);
            p.setViewport(x, y, width(), height());
            paintCell(&p, tcell, cell);
        }
        ++tcell;
    }
}

void KSiagTable::keyPressEvent(QKeyEvent *ev)
{
    int key = ev->key();

    if(ev->key() == Key_Shift)
        shift_down = 1;

    switch(key){
    case Key_Home:
        key = SIAG_CTRL('a');
        break;
    case Key_Left:
        key = SIAG_CTRL('b');
        break;
    case Key_Delete:
        key = SIAG_CTRL('d');
        break;
    case Key_End:
        key = SIAG_CTRL('e');
        break;
    case Key_Right:
        key = SIAG_CTRL('f');
        break;
    case Key_Down:
        key = SIAG_CTRL('n');
        break;
    case Key_Up:
        key = SIAG_CTRL('p');
        break;
    case Key_PageUp:
        key = SIAG_ALT('v');
        break;
    case Key_PageDown:
        key = SIAG_CTRL('v');
        break;
    default:
        if (key > 255){
            ev->ignore();
            return;
        }
        key = ev->ascii();
    }
    if (ev->state() & ControlButton)	
        key = SIAG_CTRL(key);
    if (ev->state() & AltButton)
        key = SIAG_ALT(key);
    do_cmd(key);
}

void KSiagTable::keyReleaseEvent(QKeyEvent *ev)
{
    if(ev->key() == Key_Shift)
        shift_down = 0;
}

void KSiagTable::mousePressEvent (QMouseEvent *ev)
{
    position tmp;
    switch (ev->button()){
    case LeftButton:
        tmp.row = findRow(ev->y())+1;
        tmp.col = findCol(ev->x())+1;
        if (shift_down){
            set_mark(w, tmp);
            set_block();
            repaint(false);
        }
        else{
            w->point_pos.row = tmp.row;
            w->point_pos.col = tmp.col;
            show_cur(w);
        }
        break;
    case MidButton:
        paste_block();
        break;
    case RightButton:
        KSiagApp::manager()->activeTopLevel()->
            activateShortcuts(mapToGlobal(ev->pos()));
        break;
    default:
        break;
    }
}

void KSiagTable::mouseDoubleClickEvent (QMouseEvent *ev)
{
    int row, col;
    if (ev->button() == LeftButton){
        row = findRow(ev->y())+1;
        col = findCol(ev->x())+1;
        w->point_pos.row = row;
        w->point_pos.col = col;
        w->buf->mark_pos.row = row;
        w->buf->mark_pos.col = col;
        set_block();    ;
        repaint(false);
        show_cur(w);
    }
}

void KSiagTable::mouseReleaseEvent (QMouseEvent *ev)
{
    if (ev->button() == LeftButton && in_blk_select){
        repaint();
        blk_row = -1;
        in_blk_select = 0;
    }
}
    
void KSiagTable::mouseMoveEvent (QMouseEvent *ev)
{
    int row, col;
    int x1, y1, x2, y2;
    if (ev->state() & LeftButton){
        row = findRow(ev->y());
        col = findCol(ev->x());
        if (row == blk_row && col == blk_col)
            return;
        QPainter p;
        QBrush b(white);
        blk_row = row;
        blk_col = col;
        colXPos(w->blku.col-1, &x1);
        rowYPos(w->blku.row-1, &y1);
        colXPos(w->blkl.col, &x2);
        rowYPos(w->blkl.row, &y2);
        unset_block();
        repaint(x1, y1, x2-x1, y2-y1);
        if (ev->y() < 0 && topCell()){
            emit scrollVertical(topCell()-1);
            row = topCell();
        }
        else if (ev->y() > height() || row > lastRowVisible()){
            emit scrollVertical(topCell()+1);
            row = lastRowVisible();
        }

        if (ev->x() < 0 && leftCell()){
            emit scrollHorizontal(leftCell()-1);
            col = leftCell();
        }
        else if (ev->x() > width() || col > lastColVisible()){
            emit scrollHorizontal(leftCell()+1);
            col = lastColVisible();
        }
        ++col;
        ++row;
        set_mark_row(w, row);
        set_mark_col(w, col);
        set_block();
        colXPos(w->blku.col-1, &x1);
        rowYPos(w->blku.row-1, &y1);
        colXPos(w->blkl.col, &x2);
        rowYPos(w->blkl.row, &y2);
        p.begin(this);
        p.setRasterOp(NotROP);
        p.fillRect(x1, y1, x2-x1, y2-y1,  b);
        p.end();
    }
}

void KSiagTable::paintCell (QPainter *p, int row, int col)
{
    int tf = SingleLine | DontClip;
    int fmt;
    int width = cellWidth(col);
    int height = cellHeight(row);
    int x2 = --width;
    int y2 = --height;
    static char text[1024];
    ++row, ++col;

    //if (inblock (w, row, col))
    //    p->setRasterOp(NotCopyROP);
    //else
    p->setRasterOp(CopyROP);
    brush->setColor(white);
    p->fillRect(0, 0, width+1, height+1, *brush);

    if (grid_lines){
        p->setPen(darkGray); // Cell grid
        p->drawLine(x2, 0, x2, y2);
        p->drawLine(0, y2, x2, y2);
    }
    
    ret_pvalue(text, w->buf, row, col, -1);
    fmt = ret_format(w->buf, row, col);

    if (*text != '\0'){
        font->setFamily(fonts[(ret_font(w->buf, row, col)&FONT_MASK)>>FONT_SHIFT]);
        font->setPointSize(fontsizes[fmt & SIZE_MASK]);
        font->setItalic(fmt & ITALIC);
        font->setBold(fmt & BOLD);
        font->setUnderline(fmt & ULINE);
    
        switch (fmt & HADJ_MASK){
        case HADJ_CENTER:
            tf |= AlignHCenter;
            break;
        case HADJ_RIGHT:
            tf |= AlignRight;
            break;
        default:
            tf |= AlignLeft;
        }
        switch (fmt & VADJ_MASK){
        case VADJ_BOTTOM:
            tf |= AlignBottom;
            break;
        case VADJ_TOP:
            tf |= AlignTop;
            break;
        default:
            tf |= AlignVCenter;
        }
        p->setPen(QColor(colors[ret_color(w->buf, row, col)]));
        p->setFont(*font);
        p->drawText(1, 1, width, height, tf, text);
    }
    p->setPen(QPen(black, 2));
    if (fmt & BORDER_LEFT)
        p->drawLine (1, 0, 1, y2+1);
    if (fmt & BORDER_RIGHT)
        p->drawLine (x2, 0, x2, y2+1);
    if (fmt & BORDER_TOP)
        p->drawLine (0, 0, x2+1, 0);
    if (fmt & BORDER_BOTTOM)
        p->drawLine (0, y2, x2+1, y2);

    if ((row == w->point_pos.row) && (col == w->point_pos.col)){
        cur_row = row - 1;
        cur_col = col - 1;;
        p->drawRect(2, 2, x2-3, y2-3);
    }

    if (inblock (w, row, col)){
        p->setRasterOp(NotROP);
    	p->fillRect(0, 0, width+1, height+1, *brush);
    }

}
