#include <config.h>

#include <kapp.h>
#include <klocale.h>
#include <klyx.h>
#include <qchkbox.h>
#include <qradiobt.h>

#include <stdlib.h>
#include "insetinclude.h"
#include "buffer.h"
#include "bufferlist.h"
#include "error.h"
#include "filetools.h"
#include "lyxrc.h"
#include "LyXView.h"
#include "LaTeXFeatures.h"
#include "FileInfo.h"
#include "lyx_gui_misc.h"
#include "InsertIncludeDialog.h"

extern BufferView *current_view;

extern LyXRC *lyxrc;
extern BufferList bufferlist;
extern void UpdateInset(Inset* inset, bool mark_dirty = true);

InsetInclude::InsetInclude(LString const & fname, Buffer *bf)
	: InsetCommand("include") 
{
	master = bf;
	setContents(fname);
	flag = InsetInclude::INCLUDE;
	noload = false;
}


InsetInclude::~InsetInclude()
{
}


void InsetInclude::Edit(int, int)
{
  if( current_view->currentBuffer()->isReadonly() )
	WarnReadonly();
  
  if( !klyxdialogs->insertinclude )
	klyxdialogs->insertinclude = new InsertIncludeDialog( current_view->getOwner() );

  klyxdialogs->insertinclude->setInset( this );

  klyxdialogs->insertinclude->filenameED->setText( contents.c_str() );
  klyxdialogs->insertinclude->donttypesetCB->setChecked( isNoLoad() );
  klyxdialogs->insertinclude->inputRB->setChecked( isInput() );
  klyxdialogs->insertinclude->includeRB->setChecked( isInclude() );
  klyxdialogs->insertinclude->verbatimRB->setChecked( isVerb() );
  if( isVerb() )
	klyxdialogs->insertinclude->donttypesetCB->setChecked( isVerbVisibleSpace() );
  else
	{
	  klyxdialogs->insertinclude->donttypesetCB->setChecked( false );
	  klyxdialogs->insertinclude->donttypesetCB->setEnabled( false );
	}

  (void)klyxdialogs->insertinclude->exec();
}


void InsetInclude::Write(FILE *file)
{
  fprintf(file, "Include %s\n", getCommand().c_str());
}


void InsetInclude::Read(LyXLex &lex)
{
  InsetCommand::Read(lex);
  
  if (getCmdName()=="include")
	setInclude();
  else if (getCmdName() == "input")
	setInput();
  else if (getCmdName().contains("verbatim")) {
	setVerb();
	if (getCmdName() == "verbatiminput*")
	  setVisibleSpace(true);
  }
}


LString InsetInclude::getScreenLabel() const
{
  LString temp;
  if (isInput())
	temp += i18n("Input");
  else if (isVerb()) {
	temp += i18n("Verbatim Input");
	if (isVerbVisibleSpace()) temp += '*';
  } else temp += i18n("Include");
  temp += ": ";
  
  if (contents.empty()) {
	temp+="???";
  } else {
	temp+=contents;
  }
  return temp;
}


bool InsetInclude::loadIfNeeded() const
{
  if (isNoLoad() || isVerb()) return false;
  if (!IsLyXFilename(getFileName())) return false;
  
  if (bufferlist.exists(getFileName())) return true;
  
  // the readonly flag can/will be wrong, not anymore I think.
  FileInfo finfo(getFileName());
  bool ro = !finfo.writable();
  return ( bufferlist.readFile(getFileName(), ro) != NULL );
}


int InsetInclude::Latex(FILE *file, signed char /*fragile*/)
{
  LString include_file;
  signed char dummy = 0;
  Latex(include_file, dummy);
  fprintf(file, "%s", include_file.c_str());
  return 0;
}


int InsetInclude::Latex(LString &file, signed char /*fragile*/)
{
  LString writefile, incfile;
  
  // Do nothing if no file name has been specified
  if (contents.empty())
	return 0;
  
  // Use += to force a copy of contents (JMarc)
  incfile += contents;
  
  if (loadIfNeeded()) {
	Buffer *tmp = bufferlist.getBuffer(getFileName());
	
	// write it to a file (so far the complete file)
	writefile = ChangeExtension(getFileName(), ".tex", false);
	if (!master->tmppath.empty()
		&& !master->niceFile) {
	  incfile.subst('/','@');
	  writefile = AddName(master->tmppath, incfile);
	} else
	  writefile = getFileName();
	writefile = ChangeExtension(writefile,".tex",false);
	lyxerr.debug("incfile:" + incfile, Error::LATEX);
	lyxerr.debug("writefile:" + writefile, Error::LATEX);
	
	tmp->markDepClean(master->tmppath);
	
	tmp->makeLaTeXFile(writefile,
					   OnlyPath(getMasterFilename()), 
					   master->niceFile, true);
  } 
  
  if (isVerb()) {
	file += '\\';
	file += command + '{';
	file += incfile + '}';
  } 
  else if (isInput()) {
	// \input wants file with extension (default is .tex)
	if (!IsLyXFilename(getFileName())) {
	  file += '\\';
	  file += command + '{';
	  file += incfile + '}';
	} else {
	  file += '\\';
	  file += command + '{';
	  file +=	ChangeExtension(incfile, ".tex", false)
		+ '}';
	}
  } else {
	// \include don't want extension and demands that the
	// file really have .tex
	file += '\\';
	file += command + '{';
	file +=	ChangeExtension(incfile, LString(), false)
	  + '}';
  }
  
  return 0;
}


void InsetInclude::Validate(LaTeXFeatures& features) const
{
  if (isVerb())
	features.verbatim = true;
  
  // Here we must do the fun stuff...
  // Load the file in the include if it needs
  // to be loaded:
  if (loadIfNeeded()) {
	// a file got loaded
	Buffer *tmp = bufferlist.getBuffer(getFileName());
	tmp->validate(features);
  }
}
