	/*

	Copyright (C) 1998 Stefan Westerfeld
                       stefan@space.twc.de
					   Jens Hahn
                       Jens.Hahn@t-online.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "synthmodule.h"
#include "synth_impl.h"
#include "debug.h"
#include <sys/types.h>
#include <sys/time.h>
#include <termios.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <math.h>

extern "C" {
#include "utils.h"
#include "sound.h"
}

class Synth_RC :public SynthModule
{
	float B,dB;
	float F,dF,oF,oU,U,Fsoll,Bsoll;
	float oldvalue;

	// inputs:
	enum { INVALUE, CAPACITY_B, CAPACITY_F };

	// outputs:
	enum { OUTVALUE };

public:
	void Initialize();
	void Calculate() { assert(false); }
	void CalculateBlock(unsigned long samples);
	string getParams() { return("invalue,capacity_b,capacity_f;outvalue"); }
	static void *Creator() { return new Synth_RC; };
};

ModuleClient MC_Synth_RC(SynthModule::get_MS,"Synth_RC",Synth_RC::Creator);

void Synth_RC::Initialize()
{
	haveCalculateBlock = true;
	oldvalue=0;
	B=0;
	F=0; oF=0;
	U=0; oU=0;
}

void Synth_RC::CalculateBlock(unsigned long samples)
{
	unsigned long i,hits;
	float *invalue = in[INVALUE], *capb = in[CAPACITY_B],*capf = in[CAPACITY_F];
	float *outvalue = out[OUTVALUE];

	const float zero_lower =-0.00000001;
	const float zero_upper = 0.00000001;

	if(zero_lower < invalue[0] && invalue[0] < zero_upper);
	{
		/* for comments see equalizer.cc/Synth_STD_EQUALIZER implementation */

		/*
		 * This implementation differs from the implementation there,
		 * because it is done as a kind of powersafing. If no input is
		 * found, then no output is generated.
		 */
		if(zero_lower < oldvalue && oldvalue < zero_upper)
		{
			//printf("Synth_RC: reset\n");
			oldvalue=0.0;
			B=0.0;
			F=0.0; oF=0.0;
			U=0.0; oU=0.0;
			hits = 0;
	  		for(i=0;i<samples;i++)
			{
				if(zero_lower < invalue[i] && invalue[i] < zero_upper)
				{
					// try to zero out the whole block
					outvalue[i] = 0.0;
					hits++;
				}
			}
			if(hits == samples) return;
		}
	}
/*
 * Scheduling problem: it might happen that some variables are not initialized
 * during the first turn. Since they are zero then, dividing is not so smart.
 *
 * Will be solved when rewriting scheduling stuff.
 * if(*in[CAPACITY_B] != 0 && *in[CAPACITY_F] != 0)
 * {
 * }
 *
 * Should be solved now.
 */
	for(i=0;i<samples;i++)
	{
		B = B + (invalue[i] - oldvalue);  /* input into RC */
		oldvalue = invalue[i];

		Bsoll    = U-oU;
		oU=U;
		dB       = (Bsoll-B) / capb[i];
		B       += dB;
		U		-= dB;

		Fsoll    = U;
		dF       = (Fsoll-F) / capf[i];
		F  	  	+= dF;            /* Energie dF wird ins Feld uebertragen */
		U		-= dF;

		outvalue[i] = (F - oF) * (capb[i]+capf[i]);

		oF = F;
	}
}

int hval(char n)
{
	switch(toupper(n))
	{
		case '0': return(0);
		case '1': return(1);
		case '2': return(2);
		case '3': return(3);
		case '4': return(4);
		case '5': return(5);
		case '6': return(6);
		case '7': return(7);
		case '8': return(8);
		case '9': return(9);
		case 'A': return(10);
		case 'B': return(11);
		case 'C': return(12);
		case 'D': return(13);
		case 'E': return(14);
		case 'F': return(15);
	}
	return(0);
}

#if 0
/*
 * These modules are not working (and most likely not needed as well)
 */
class Synth_STDIN :public SynthModule
{
	#define NMAX 1024
	long left, pos;
	char buffer[2*NMAX];

	const int OUTVALUE=0;
public:
	void Initialize();
	void Calculate();
	string getParams() { return(";outvalue"); }
	static SynthModule *Create() { return new Synth_STDIN; };
};

void Synth_STDIN::Initialize()
{
	left = 0;
}

void Synth_STDIN::Calculate()
{
	long l;
	unsigned char c;

	if(!left) /* nothing in buffer */
	{
		left = fread(buffer,1,2*NMAX,stdin)/2;
		if(!left) {
			*out[OUTVALUE] = 0;
			exit(1);
			return;
		}
		pos = 0;
	}

	//artsdebug("read: left = %d\n",left);
	c=buffer[pos*2+1]+128;
	l=c*256+buffer[pos*2]-32767;
	
	*out[OUTVALUE] = (float)l/(float)32768;
	pos++;
	left--;
}

class Synth_DATA :public SynthModule
{
	float value;
	const int OUTVALUE=0;
public:
	void Initialize() { value = 0; }
	void Calculate() { *out[OUTVALUE] = value; }
	string getParams() { return(";outvalue"); }
	static SynthModule *Create() { return new Synth_DATA; };

	void ProcessData(string data) {
		float newval = atof(data.c_str());
		value=newval;
	}
};
#endif

class Synth_XFADE :public SynthModule
{
	// inputs:
	enum { INVALUE1, INVALUE2, PERCENTAGE };

	// outputs:
	enum { OUTVALUE };
public:
	void Initialize() { haveCalculateBlock = true; };
	void Calculate();
	void CalculateBlock(unsigned long cycles);
	string getParams() { return("invalue1,invalue2,percentage;outvalue"); }
	static void *Creator() { return new Synth_XFADE; };
};

ModuleClient MC_Synth_XFADE(SynthModule::get_MS,"Synth_XFADE",Synth_XFADE::Creator);

void Synth_XFADE::Calculate()
{
	float p = (*in[PERCENTAGE]+1)/2;

	*out[OUTVALUE] = *in[INVALUE1]*p + *in[INVALUE2]*(1-p);
}

void Synth_XFADE::CalculateBlock(unsigned long cycles)
{
	float *percentage = in[PERCENTAGE], *in1 = in[INVALUE1], *in2 = in[INVALUE2],
				*outsig = out[OUTVALUE];

	unsigned long i;

	for(i=0;i<cycles;i++)
	{
		float p = (percentage[i]+1)/2;

		outsig[i] = in1[i]*p + in2[i]*(1-p);
	}
}

class Synth_PSCALE :public SynthModule
{
	// inputs:
	enum { INVALUE, POS, TOP};

	// outputs:
	enum { OUTVALUE };
public:
	void Initialize() { haveCalculateBlock = true; };
	void Calculate();
	void CalculateBlock(unsigned long cycles);
	string getParams() { return("invalue,pos,top;outvalue"); }
	static void *Creator() { return new Synth_PSCALE; };
};

ModuleClient MC_Synth_PSCALE(SynthModule::get_MS,"Synth_PSCALE",Synth_PSCALE::Creator);

void Synth_PSCALE::Calculate()
{
	if(*in[POS] >= *in[TOP])
		*out[OUTVALUE] = *in[INVALUE] * (1-*in[POS])/(1-*in[TOP]);
	else
		*out[OUTVALUE] = *in[INVALUE] * (*in[POS])/(*in[TOP]);
}

void Synth_PSCALE::CalculateBlock(unsigned long cycles)
{
	float *pos = in[POS], *top = in[TOP], *insig = in[INVALUE], *outsig = out[OUTVALUE];
	unsigned long i;

	for(i=0;i<cycles;i++)
	{
		if(pos[i] >= top[i])
			outsig[i] = insig[i] * (1-pos[i])/(1-top[i]);
		else
			outsig[i] = insig[i] * pos[i] / top[i];
	}
}

class Synth_NIL :public SynthModule
{
public:
	void Initialize() { haveCalculateBlock = true; };
	void Calculate() { };
	void CalculateBlock(unsigned long cycles) { };
	string getParams() { return(";"); }
	static void *Creator() { return new Synth_NIL; };
};

ModuleClient MC_Synth_NIL(SynthModule::get_MS,"Synth_NIL",Synth_NIL::Creator);

class Synth_MUL :public SynthModule
{
	// inputs:
	enum { INVALUE, FACTOR };

	// outputs:
	enum { OUTVALUE };
public:
	void Initialize() { haveCalculateBlock = true; };
	void CalculateBlock(unsigned long cycles);
	void Calculate() { *out[OUTVALUE] = (*in[INVALUE]) * (*in[FACTOR]); }
	string getParams() { return("invalue,faktor;outvalue"); }
	static void *Creator() { return new Synth_MUL; }
};

ModuleClient MC_Synth_MUL(SynthModule::get_MS,"Synth_MUL",Synth_MUL::Creator);

void Synth_MUL::CalculateBlock(unsigned long cycles)
{
	float *factor = in[FACTOR], *invalue = in[INVALUE], *outsig = out[OUTVALUE];
	unsigned long i;

	for(i=0;i<cycles;i++) outsig[i] = invalue[i] * factor[i];
}

class Synth_ADD :public SynthModule
{
	// inputs:
	enum { INVALUE, ADDIT };

	// outputs:
	enum { OUTVALUE };
public:
	void Initialize() {haveCalculateBlock = true;};
	void Calculate() { *out[OUTVALUE] = (*in[INVALUE]) + (*in[ADDIT]); }
	void CalculateBlock(unsigned long samples);
	string getParams() { return("invalue,addit;outvalue"); }
	static void *Creator() { return new Synth_ADD; }
};

ModuleClient MC_Synth_ADD(SynthModule::get_MS,"Synth_ADD",Synth_ADD::Creator);

void Synth_ADD::CalculateBlock(unsigned long samples)
{
	float *outsig=out[OUTVALUE],*sig1=in[INVALUE],*sig2=in[ADDIT];
	unsigned long i;

	for(i=0;i<samples;i++)
		outsig[i] = sig1[i] + sig2[i];
}

class Synth_DELAY :public SynthModule
{
	// inputs:
	enum { INVALUE, TIME };

	// outputs:
	enum { OUTVALUE };

	#define MAXDELAY 44100
	float dbuffer[MAXDELAY];
	int dbpos;
public:
	void Initialize();
	void Calculate();
	string getParams() { return("invalue,time;outvalue"); }
	static void *Creator() { return new Synth_DELAY; }
};

ModuleClient MC_Synth_DELAY(SynthModule::get_MS,"Synth_DELAY",Synth_DELAY::Creator);

void Synth_DELAY::Initialize()
{
	for(dbpos=0;dbpos<MAXDELAY;dbpos++) dbuffer[dbpos] = 0;

	dbpos = 0;
}

void Synth_DELAY::Calculate()
{
/** implementation completely broken when quickly changing time parameter ;) ***
	float position = (float)dbpos + (*in[TIME]*MAXDELAY); // % MAXDELAY
	if(position > MAXDELAY) position -= MAXDELAY;
	int position1 = (int)position % MAXDELAY;
	int position2 = (position1+1) % MAXDELAY;
	float error = position - position1; // precise delay calculation for
										// flanging

	dbuffer[position1] += (1-error)*(*in[INVALUE]);
	dbuffer[position2] += error*(*in[INVALUE]);
	//dbuffer[position1] += *in[INVALUE];
	*out[OUTVALUE] = dbuffer[dbpos];
	dbuffer[dbpos] = 0;
	dbpos++;
	if(dbpos == MAXDELAY) dbpos = 0;
***/
	dbuffer[dbpos] = *in[INVALUE];

	int position, position1;
	double error,int_pos;
 	error = modf(*in[TIME] * MAXDELAY,&int_pos);

	position = dbpos - (int)int_pos;
	if(position < 0) position += MAXDELAY;
	position1 = position-1;
	if(position1 < 0) position1 += MAXDELAY;

	*out[OUTVALUE] = dbuffer[position]*(1-error) + dbuffer[position1]*error;

	dbpos++;
	if(dbpos == MAXDELAY) dbpos = 0;
}

class Synth_FM_SOURCE :public SynthModule
{
	// inputs:
	enum { FREQUENCY, MODULATOR, MODLEVEL };

	// ouputs:
	enum { POSITION };

	#define SAMPLINGRATE 44100
	float pos;
public:
	void Initialize() { pos = 0; };
	void Calculate();
	string getParams() { return("frequency,modulator,modlevel;pos"); };
	static void *Creator() { return new Synth_FM_SOURCE; }
};

ModuleClient MC_Synth_FMSOURCE(SynthModule::get_MS,"Synth_FM_SOURCE",Synth_FM_SOURCE::Creator);

void Synth_FM_SOURCE::Calculate()
{
	float pinc = (*in[FREQUENCY])/(float)SAMPLINGRATE;

	pos += pinc;
	if(pos > 1) pos -= 1;
	*out[POSITION] = pos + ((*in[MODULATOR]) * (*in[MODLEVEL]));
};

class Synth_FREQUENCY :public SynthModule
{
	// inputs:
	enum { FREQUENCY };

	// outputs:
	enum { POSITION };
	#define SAMPLINGRATE 44100

	float pos;
	unsigned long ipos;
public:
	void Initialize() { pos = 0; ipos = 0; haveCalculateBlock = true;};
	void Calculate();
	void CalculateBlock(unsigned long cycles);
	string getParams() { return("frequency;pos"); };
	static void *Creator() { return new Synth_FREQUENCY; }
};

ModuleClient MC_Synth_FREQUENCY(SynthModule::get_MS,"Synth_FREQUENCY",Synth_FREQUENCY::Creator);

void Synth_FREQUENCY::Calculate()
{
	pos += (*in[FREQUENCY])/(float)SAMPLINGRATE;
	if(pos > 1) pos -= 1;
	*out[POSITION] = pos;
};

void Synth_FREQUENCY::CalculateBlock(unsigned long cycles)
{
	float *outsig = out[POSITION], *freq = in[FREQUENCY];
	unsigned long i;

	float fsample = (float)SAMPLINGRATE;
	float finc = freq[0]/fsample;

	i = 0;
	while(cycles)
	{
		if(cycles >= 8 && ((finc*8+pos) < 0.9))
		{
			pos += finc;
			outsig[i++] = pos;
			pos += finc;
			outsig[i++] = pos;
			pos += finc;
			outsig[i++] = pos;
			pos += finc;
			outsig[i++] = pos;
			pos += finc;
			outsig[i++] = pos;
			pos += finc;
			outsig[i++] = pos;
			pos += finc;
			outsig[i++] = pos;
			pos += finc;
			outsig[i++] = pos;
			cycles -= 8;
		}
		else
		{
			pos += finc;
			while(pos > 1.0) pos -= 1.0;
			outsig[i++] = pos;
			cycles--;
		}
	}
/*
	for(i=0;i<cycles;i++)
	{
		pos += freq[i]/fsample;
		if(pos > 1.0) pos -= 1.0;
		outsig[i] = pos;
	}
	while(cycles)
	{
		if(cycles > 8)
		{
			pos += finc; // *freq++/fsample;
			pos = (float)pos - (int)pos;
			*outsig++ = pos;
			pos += finc; // *freq++/fsample;
			pos = (float)pos - (int)pos;
			*outsig++ = pos;
			pos += finc; // *freq++/fsample;
			pos = (float)pos - (int)pos;
			*outsig++ = pos;
			pos += finc; // *freq++/fsample;
			pos = (float)pos - (int)pos;
			*outsig++ = pos;
			pos += finc; // *freq++/fsample;
			pos = (float)pos - (int)pos;
			*outsig++ = pos;
			pos += finc; // *freq++/fsample;
			pos = (float)pos - (int)pos;
			*outsig++ = pos;
			pos += finc; // *freq++/fsample;
			pos = (float)pos - (int)pos;
			*outsig++ = pos;
			pos += finc; // *freq++/fsample;
			pos = (float)pos - (int)pos;
			*outsig++ = pos;
			cycles -= 8;
		}
		else
		{
			pos += finc; // *freq++/fsample;
			pos = (float)pos - (int)pos;
			*outsig++ = pos;
			cycles--;
		}
	}
*/
}

#if 0
/**
 ** uses table to do sin values, but: this implementation produces noise and
 ** an interpolated version is much slower than "real" sin
 **/
class SinTable {
protected:
	/*
	   one element more for interpolation; the last element should be
	   the same as the first
	*/
	float _table[4097];
	SinTable()
	{
		int i;
		for(i=0;i<4096;i++)
			_table[i] = sin((i/4096.0)*2.0*M_PI);
		_table[4096] = _table[0];
	}
public:
	static float *table()
	{
		static SinTable *instance = 0;
		if(!instance) instance=new SinTable;
		return instance->_table;
	}
};
class Synth_WAVE_SIN :public SynthModule
{
	// inputs:
	enum { POSITION };

	// outputs:
	enum { OUTVALUE };

	float *table;
	long pos;
public:
	void Initialize() {
		pos = 0; haveCalculateBlock = true;
		table = SinTable::table();
	};
	void Calculate() { assert(false); }
	void CalculateBlock(unsigned long cycles);
	string getParams() { return("pos;outvalue"); }
	static void *Creator() { return new Synth_WAVE_SIN; }
};

ModuleClient MC_Synth_WAVE_SIN(SynthModule::get_MS,"Synth_WAVE_SIN",Synth_WAVE_SIN::Creator);

void Synth_WAVE_SIN::CalculateBlock(unsigned long cycles)
{
	unsigned long i;
	float *outsig = out[OUTVALUE], *pos = in[POSITION];

	// one could interpolate here, but it seems to be ok like that (and faster)
	for(i=0;i<cycles;i++)
		outsig[i] = table[4095 & (long)(pos[i]*4096.0)];
}
#endif

class Synth_WAVE_SIN :public SynthModule
{
	// inputs:
	enum { POSITION };

	// outputs:
	enum { OUTVALUE };

	long pos;
public:
	void Initialize() { pos = 0; haveCalculateBlock = true; };
	void Calculate() { assert(false); }
	void CalculateBlock(unsigned long cycles);
	string getParams() { return("pos;outvalue"); }
	static void *Creator() { return new Synth_WAVE_SIN; }
};

ModuleClient MC_Synth_WAVE_SIN(SynthModule::get_MS,"Synth_WAVE_SIN",Synth_WAVE_SIN::Creator);

void Synth_WAVE_SIN::CalculateBlock(unsigned long cycles)
{
	unsigned long i;
	float *outsig = out[OUTVALUE], *pos = in[POSITION];

	for(i=0;i<cycles;i++) outsig[i] = sin(pos[i]*2*M_PI);
}


class Synth_WAVE_TRI :public SynthModule
{
	//const int INVALUE=0;		// FIXME
	//const int FREQUENCY=0;

	// inputs:
	enum { POSITION };

	// outputs:
	enum { OUTVALUE };
	#define SAMPLINGRATE 44100

	long pos;
public:
	void Initialize() { pos = 0; haveCalculateBlock = true;};
	void Calculate();
	void CalculateBlock(unsigned long samples);
	string getParams() { return("pos;outvalue"); }
	static void *Creator() { return new Synth_WAVE_TRI; }
};

ModuleClient MC_Synth_WAVE_TRI(SynthModule::get_MS,"Synth_WAVE_TRI",Synth_WAVE_TRI::Creator);

void Synth_WAVE_TRI::Calculate()
{
	/*
	float end = (SAMPLINGRATE/(*in[FREQUENCY]));
	pos++;
	if(pos > end) pos = 0;
	*out[OUTVALUE] = (pos/end)*2-1;
----
	*out[OUTVALUE] = ((float)(*in[INVALUE]) - (int)(*in[INVALUE]))*2-1;
	*/
	assert(false);
}

void Synth_WAVE_TRI::CalculateBlock(unsigned long samples)
{
	float *outvalue = out[OUTVALUE], *pos = in[POSITION];

	unsigned long i;
	for(i=0;i<samples;i++) outvalue[i] = (pos[i] - 0.5) * 2.0;
}

class Synth_SEQUENCE :public SynthModule
{
	// inputs:
	enum { SPEED, PROP_SEQUENCE };

	// outputs:
	enum { FREQUENCY, POS };

	#define SAMPLINGRATE 44100

	long pos,delay;
	float *sequence;
	float *slen;
public:
	void Initialize();
	void Calculate();
	string getParams() { return("speed,_sequence;frequency,pos"); }
	static void *Creator() { return new Synth_SEQUENCE; }
};

ModuleClient MC_Synth_SEQUENCE(SynthModule::get_MS,"Synth_SEQUENCE",Synth_SEQUENCE::Creator);

void Synth_SEQUENCE::Initialize()
{
  char notea[][3]={"C-", "C#", "D-", "D#", "E-", "F-", "F#", "G-", "G#",
					"A-", "A#", "H-", "\0"};
  char noteb[][3]={"C-", "Db", "D-", "Eb", "E-", "F-", "Gb", "G-", "Ab",
					"A-", "B-", "H-", "\0"};
  float freq[]  ={261.7,277.2,293.7,311.2,329.7,349.3,370.0,392.0,415.3,440.0,
					466.2,493.9, 0 };
  float zhoch[]={1,2,4,8,16,32,64,128,256};
  int s=0,i,oktave;
  char *nptr;
  float f;
  char buffer[1024];
  strcpy(buffer,getStringProperty(PROP_SEQUENCE));

  long bufferlen = strlen(buffer);
  // FIXME: free memory of these one day.
  sequence = (float *)malloc(bufferlen*sizeof(float *));
  slen = (float *)malloc(bufferlen*sizeof(float *));
  nptr = strtok(buffer,",;");
  while(nptr)
  { 
	if(nptr[3] == ':')
		slen[s] = atof(&nptr[4]);
	else
		slen[s] = 1;
	fprintf(stderr," <%d> %s\n",s,nptr);
    oktave=atol(&nptr[2]);
    nptr[2]=0;
    f=0;
    for(i=0;notea[i][0];i++) if(strcmp(nptr,notea[i]) == 0) f=freq[i];
    for(i=0;noteb[i][0];i++) if(strcmp(nptr,noteb[i]) == 0) f=freq[i];
    f *= zhoch[oktave] / zhoch[4];
    sequence[s++]=f;
	fprintf(stderr,">%2.2f\n",f);
    nptr = strtok(NULL,",;");
  } 
  sequence[s]=0;
  delay = 0;
  pos = 0;
}

void Synth_SEQUENCE::Calculate()
{
	delay++;
	if(delay > *in[SPEED] * SAMPLINGRATE * slen[pos])
	{
		pos++;
		if(sequence[pos] == 0) pos = 0;
		delay = 0;
	}
	*out[POS] = (int)delay / (*in[SPEED] * SAMPLINGRATE * slen[pos]);
	*out[FREQUENCY] = sequence[pos];
}

class Synth_STDOUT :public SynthModule
{
	bool scaleerr;

	// inputs:
	enum { INVALUE };
public:
	void Initialize();
	void Calculate();
	string getParams() { return("invalue;"); }
	static SynthModule *Create() { return new Synth_STDOUT; };
};

void Synth_STDOUT::Initialize()
{
	scaleerr = false;
}

void Synth_STDOUT::Calculate()
{
	long syn = (long)(*in[INVALUE]*(128*256-1));
	unsigned char sb[2];

	if(abs(syn)>32767 && !scaleerr) {
		scaleerr=true;
		fprintf(stderr,"Synth_STDOUT error: overflow ; scale too big\n");
	}
	syn=syn+65536;
	sb[0] = syn % 256;
	sb[1] = syn / 256;
	fwrite(sb,2,1,stdout);
	//printf("wrote sb\n");
}

/*
 * Play module to play sample data, either mono or stereo
 */
#if 0
class Synth_PLAY :public SynthModule
{
	enum { INVALUE_L, INVALUE_R, PROP_CHANNELS };

	bool audioinit, scaleerr;
	int audiofd, byteorder, v;

	float values[8];
	unsigned char *outblock;
	unsigned long maxsamples;
	unsigned long channels;

public:
	void Initialize() { 
		artsdebug("Init: set audioinit to false");
		haveCalculateBlock = true;
		v = 0;

		int samplingrate=44100;

		channels = atoi(getStringProperty(PROP_CHANNELS));

		audiofd = sound_open();
		byteorder = get_byteorder();

		if(byteorder == ORDER_UNKNOWN)
		{
			fprintf(stderr,"Can't determine the byteorder of your system.\n");
			exit(0);
		}
		fprintf(stderr,"Synth_PLAY: audioinit: channels is %ld\n",channels);
		sound_realtime(audiofd,10);
		sound_init(audiofd,samplingrate,channels);

		maxsamples = 0;
		outblock = 0;
	}
	void getfds(int &infd, int &outfd) {
		infd = -1;
		outfd = audiofd;
	}
	void Calculate();
	void CalculateBlock(unsigned long cycles);
	string getParams() { return("invalue_left,invalue_right,_channels;"); }
	static void *Creator() { return new Synth_PLAY; };
	void DeInitialize()		{
		artsdebug("... deleting synth_play\n");
		artsdebug("Synth_PLAY: closing audio fd\n");
		sound_close(audiofd);
	}
};

ModuleClient MC_Synth_PLAY(SynthModule::get_MS,"Synth_PLAY",Synth_PLAY::Creator);

/*
unsigned char *calcsb(float value, bool *scaleerr, int byteorder)
{
	static unsigned char sb[2];

	long syn = (long)(value*(128*256-1));
	if(abs(syn)>32767 && !*scaleerr) {
		*scaleerr=true;
		fprintf(stderr,"Synth error: overflow ; scale too big\n");
	}
	syn=syn+65536;
	if(byteorder == ORDER_BIGENDIAN)
	{
		sb[0] = syn / 256;
		sb[1] = syn % 256;
	}
	else
	{
		sb[0] = syn % 256;
		sb[1] = syn / 256;
	}

	return(sb);
}
*/

#endif
unsigned char *calc8sb(float *values, bool *scaleerr, int byteorder)
{
	static unsigned char sb[16];
	int i,o=0;

	for (i=0;i<8;i++)
	{
		long syn = (long)(values[i]*32767);		// 2^16-1
		if(abs(syn)>32767 && !*scaleerr) {
			*scaleerr=true;
			fprintf(stderr,"Synth error: overflow ; scale too big\n");
		}
		syn=syn+65536;
		if(byteorder == ORDER_BIGENDIAN)
		{
			sb[o++] = syn / 256;
			sb[o++] = syn % 256;
		}
		else
		{
			sb[o++] = syn % 256;
			sb[o++] = syn / 256;
		}
	}

	return(sb);
}

#if 0
void Synth_PLAY::Calculate()
{
/*
	if(*in[CHANNELS] > 0) values[v++] = *in[INVALUE_L];
	if(*in[CHANNELS] > 1) values[v++] = *in[INVALUE_R];

	if(v == 8)
	{
		sound_write(audiofd,calc8sb(values,&scaleerr,byteorder),16);
		v = 0;
	}
*/
}

void Synth_PLAY::CalculateBlock(unsigned long samples)
{
	if(samples > maxsamples)
	{
		maxsamples = samples;
		outblock = (unsigned char *)realloc(outblock, maxsamples * 4);
													// 2 channels, 16 bit
	}

	unsigned char *sb = outblock;
	unsigned long i,size,high_byte,low_byte;
	float *left = in[INVALUE_L], *right = in[INVALUE_R];

	assert(channels);

	size = channels * 2 * samples;

	if(byteorder == ORDER_BIGENDIAN)
		{ high_byte = 0; low_byte = 1; }
	else
		{ high_byte = 1; low_byte = 0; }

	if(channels == 1)
	{
		for(i=0;i<samples;i++)
		{
			long syn = (long)(left[i]*32767)+65536;	// 2^16-1
	
			sb[high_byte] = syn / 256; //syn >> 8;    //syn / 256;
			sb[low_byte]  = syn % 256; //syn & 255;   //syn % 256;
			sb += 2;
		}
	}
	if(channels == 2)
	{
		for(i=0;i<samples;i++)
		{
			long syn = (long)(left[i]*32767)+65536;	// 2^16-1
	
			sb[high_byte] = syn / 256;
			sb[low_byte] = syn % 256;
			sb += 2;

			syn = (long)(right[i]*32767)+65536;	// 2^16-1

			sb[high_byte] = syn / 256;
			sb[low_byte]  = syn % 256;
			sb += 2;
		}
	}
	sound_write(audiofd,outblock,size);
}
#endif

class Synth_FILEPLAY :public SynthModule
{
	enum { INVALUE_L, INVALUE_R, PROP_CHANNELS };

	bool audioinit, scaleerr;
	int audiofd, byteorder, v,datalen,channels;

	float values[8];

/****

   WAV writing code and header taken from kwave. Many thanks go to
   Martin Wilz who has written this ;)
 
 ****/

	struct wavheader
	{
    	char        riffid[4];
    	long        filelength;
    	char        wavid[4];
    	char        fmtid[4];
    	long        fmtlength;
    	short int   mode;
    	short int   channels;
    	long        rate;
    	long        AvgBytesPerSec;
    	short int   BlockAlign;
    	short int   bitspersample;
	} header;

public:
	void Initialize() { 
		audiofd = open("/tmp/arts.wav",O_WRONLY|O_CREAT|O_TRUNC,0644);
		byteorder = get_byteorder();

		if(byteorder == ORDER_UNKNOWN) { fprintf(stderr,"byteorder panic!\n"); }

/* write header */

		int rate=44100;
		int bit=16;

		channels=atoi(getStringProperty(PROP_CHANNELS));
		if(channels != 1 && channels != 2)
		{
			fprintf(stderr,"ERROR (Synth_FILEPLAY): channels must be one (mono) or two (stereo)\n");
			fprintf(stderr,"                        assuming stereo output\n");
			channels = 2;
		}
		datalen=0;

        strncpy (header.riffid,"RIFF",4);
        strncpy (header.wavid,"WAVE",4);
        strncpy (header.fmtid,"fmt ",4);
        header.fmtlength=16;
        header.filelength=sizeof(struct wavheader);
        header.mode=1;
        header.channels=channels;
        header.rate=rate;
        header.AvgBytesPerSec=rate*bit/8;
        header.BlockAlign=bit/8;
        header.bitspersample=bit;

        write(audiofd,&header,sizeof (struct wavheader));
        write(audiofd,"data",4);
        write(audiofd,&datalen,4);
  
		v = 0;
	};
	void Calculate();
	string getParams() { return("invalue_left,invalue_right,_channels;"); }
	static void *Creator() { return new Synth_FILEPLAY; };
	void DeInitialize()		{

	/* rewrite header which now contains the correct size of the file */
		lseek(audiofd,0,SEEK_SET);
        header.filelength=sizeof(struct wavheader)+datalen;
        write(audiofd,&header,sizeof (struct wavheader));
        write(audiofd,"data",4);
        write(audiofd,&datalen,4);

		close(audiofd);
	}
};

void Synth_FILEPLAY::Calculate()
{
	values[v++] = *in[INVALUE_L];
	if(channels == 2) values[v++] = *in[INVALUE_R]; // stereo

	if(v == 8)
	{
		write(audiofd,calc8sb(values,&scaleerr,byteorder),16);
		datalen += 16;
		v = 0;
	}
}

ModuleClient MC_Synth_FILEPLAY(SynthModule::get_MS,"Synth_FILEPLAY",Synth_FILEPLAY::Creator);

//-----------------------------------------------------------------------------

class Synth_ENVELOPE_ADSR :public SynthModule
{
	// ins
	enum { ACTIVE, INVALUE, ATTACK, DECAY, SUSTAIN, RELEASE };
	/*
	const int ACTIVE =0;
	const int INVALUE=1;
	const int ATTACK =2;
	const int DECAY  =3;
	const int SUSTAIN=4;
	const int RELEASE=5;
	*/

	// outs
	enum { OUTVALUE, DONE };
	/*
	const int OUTVALUE=0;
	const int DONE=1;
	*/

	// other fun
	enum { NOOUT };
	/*
	const int NOOUT=0;
	*/

	int currentphase;
	float level,increment,decrement;
public:
	void Initialize()
	{
		currentphase = NOOUT;
		level = 0;
		haveCalculateBlock = true;
	}
	void CalculateBlock(unsigned long samples)
	{
		/* FIXME:
         * should be rewritten as generic envelope, would certainly
		 * be faster & more flexible
		 */
		float *active = in[ACTIVE], *insig = in[INVALUE];
		float *outsig = out[OUTVALUE], *done = out[DONE];
		unsigned long i;

		for(i=0;i<samples;i++)
		{
			done[i] = 0;
			if(active[i] < 0.5)
			{
				if(currentphase == NOOUT)
				{
					level = 0;
					done[i] = 1;
				}
				else
				{
					if(currentphase != RELEASE) {
						artsdebug("ADSR: entering release phase\n");
						currentphase = RELEASE;
						decrement = in[SUSTAIN][i] / (in[RELEASE][i] * 44100.0);
					}
					level -= decrement;
					if(level <= 0)
					{
						level = 0;
						currentphase = NOOUT;
					}
				}
			}
			else
			{
				switch(currentphase)
				{
					//quickly kill the note that is still there (channel busy ;)
					case RELEASE:
							level -= 1/200;
							if(level <= 0) {
								currentphase = NOOUT;
								level = 0;
							}
						break;
					case NOOUT:
							artsdebug("ADSR: entering attack\n");
							increment = 1 / (in[ATTACK][i] * 44100.0);
							currentphase = ATTACK;
						break;
					case ATTACK:
							level += increment;
							if (level >= 1)
							{
								level = 1;
								currentphase = DECAY;
								decrement = (1-in[SUSTAIN][i]) /
											(in[DECAY][i] * 44100.0);
							}
						break;
					case DECAY:
							level -= decrement;
							if (level <= in[SUSTAIN][i])
							{
								level = in[SUSTAIN][i];
								currentphase = SUSTAIN;
							}
						break;
					case SUSTAIN:
							level = in[SUSTAIN][i];
						break;
				}
			}
			outsig[i] = insig[i] * level;
		}
	}
	void Calculate()
	{
/* old code
		if(*in[ACTIVE] < 0.5)
		{
			if(currentphase == NOOUT)
			{
				*out[OUTVALUE] = 0;
				*out[DONE] = 1;
				return;
			}
			if(currentphase != RELEASE) {
				artsdebug("ADSR: entering release phase\n");
				currentphase = RELEASE;
				decrement = *in[SUSTAIN] / ( *in[RELEASE] * 44100);
			}
			level -= decrement;
			if(level <= 0)
			{
				level = 0;
				currentphase = NOOUT;
			}
		}
		else
		{
			switch(currentphase)
			{
				// quickly kill the note that is still there (channel busy ;)
				case RELEASE:
						level -= 1/200;
						if(level <= 0) {
							currentphase = NOOUT;
							level = 0;
						}
					break;
				case NOOUT:
						artsdebug("ADSR: entering attack\n");
						increment = 1 / (*in[ATTACK] * 44100);
						currentphase = ATTACK;
					break;
				case ATTACK:
						level += increment;
						if (level >= 1)
						{
							level = 1;
							currentphase = DECAY;
							decrement = (1-*in[SUSTAIN]) / (*in[DECAY] * 44100);
						}
					break;
				case DECAY:
						level -= decrement;
						if (level <= *in[SUSTAIN])
						{
							level = *in[SUSTAIN];
							currentphase = SUSTAIN;
						}
					break;
				case SUSTAIN:
						level = *in[SUSTAIN];
					break;
			}
		}
		*out[OUTVALUE] = *in[INVALUE] * level;
		*out[DONE] = 0;
*/
	}
	string getParams() { return("active,invalue,attack,decay,sustain,release;outvalue,done"); }
	static void *Creator() { return new Synth_ENVELOPE_ADSR; };
};

ModuleClient MC_Synth_ENVELOPE_ADSR(SynthModule::get_MS,"Synth_ENVELOPE_ADSR",Synth_ENVELOPE_ADSR::Creator);

/*
 * Debug module that outputs the value of a connection every second or so
 */

class Synth_DEBUG :public SynthModule
{
	// inputs:
	enum { INVALUE, PROP_COMMENT };

	int i;
	string comment;
public:
	void Initialize() {
		i=0;
		comment = getStringProperty(PROP_COMMENT);
	}
	void propertyChanged(unsigned long index)
	{
		assert(index == PROP_COMMENT);
		comment = getStringProperty(PROP_COMMENT);
	}
	void Calculate()
	{
		i++;
		if((i&65535) == 0)
			printf("- debug %s %f\n",comment.c_str(),*in[INVALUE]);
	}
	string getParams() { return("invalue,_comment;"); }
	static void *Creator() { return new Synth_DEBUG; };
};

ModuleClient MC_Synth_DEBUG(SynthModule::get_MS,"Synth_DEBUG",Synth_DEBUG::Creator);

class Synth_STRUCT_KILL :public SynthModule
{
	// inputs:
	enum { READY };

	bool haveKilled;
public:

	void Initialize() {haveKilled = false; haveCalculateBlock = true;}
	void Calculate() { /* don't have that */ }
	void CalculateBlock(unsigned long samples)
	{
		float *sigready = in[READY], *readyend = sigready + samples;
		while(sigready < readyend)
		{
			if(*sigready++ > 0.5)
			{
				if(!haveKilled) Synthesizer->kill(mID);
				haveKilled = true;
			}
		}
	}
	string getParams() { return("ready;"); }
	static void *Creator() { return new Synth_STRUCT_KILL; };
};

ModuleClient MC_Synth_STRUCT_KILL(SynthModule::get_MS,"Synth_STRUCT_KILL",Synth_STRUCT_KILL::Creator);

class Synth_CDELAY :public SynthModule
{
	// inputs:
	enum { INVALUE, PROP_TIME };

	// outputs:
	enum { OUTVALUE };

	#define MAXDELAY 44100
	float dbuffer[MAXDELAY];
	unsigned long dbpos;
	unsigned long delay;

public:
	void Initialize();
	void Calculate();
	void CalculateBlock(unsigned long samples);
	unsigned long PreProduce(unsigned long samples);
	string getParams() { return("invalue,_time;outvalue"); }
	static void *Creator() { return new Synth_CDELAY; }
};

ModuleClient MC_Synth_CDELAY(SynthModule::get_MS,"Synth_CDELAY",Synth_CDELAY::Creator);

void Synth_CDELAY::Initialize()
{
	float delay_in_sec = atof(getStringProperty(PROP_TIME));

	delay = (int)(delay_in_sec * (float)SAMPLINGRATE);
	if(delay < 1) delay = 1;

	for(dbpos=0;dbpos<MAXDELAY;dbpos++) dbuffer[dbpos] = 0;

	dbpos = 0;

	haveCalculateBlock = true;
}

unsigned long Synth_CDELAY::PreProduce(unsigned long samples)
{
	unsigned long i;

	if(samples > delay) samples = delay;
	for(i=0;i<samples;i++) out[OUTVALUE][i] = 0;

	delay -= samples;	
	return samples;
}

void Synth_CDELAY::Calculate()
{
	dbuffer[dbpos] = *in[INVALUE];

	int position = dbpos - delay;
	if(position < 0) position += MAXDELAY;

	*out[OUTVALUE] = dbuffer[position];

	dbpos++;
	if(dbpos == MAXDELAY) dbpos = 0;
}

void Synth_CDELAY::CalculateBlock(unsigned long samples)
{
	float *insig = in[INVALUE], *outsig = out[OUTVALUE];		// FIXME: optimize me
	unsigned long i;

	for(i=0;i<samples;i++)
	{
		dbuffer[dbpos] = insig[i];

		int position = dbpos - delay;
		if(position < 0) position += MAXDELAY;

		outsig[i] = dbuffer[position];

		dbpos++;
		if(dbpos == MAXDELAY) dbpos = 0;
	}
}

class Synth_ATAN_SATURATE :public SynthModule
{
	// inputs:
	enum { INVALUE, INSCALE };

	// outputs:
	enum { OUTVALUE };

public:
	void Initialize();
	void Calculate() {assert(false);}
	void CalculateBlock(unsigned long cycles);
	string getParams() { return("invalue,inscale;outvalue"); }
	static void *Creator() { return new Synth_ATAN_SATURATE; }
};

ModuleClient MC_Synth_ATAN_SATURATE(SynthModule::get_MS,"Synth_ATAN_SATURATE",Synth_ATAN_SATURATE::Creator);

void Synth_ATAN_SATURATE::Initialize()
{
	haveCalculateBlock=true;
}

void Synth_ATAN_SATURATE::CalculateBlock(unsigned long cycles)
{
	unsigned long i;
	float *outsig = out[OUTVALUE], *insig = in[INVALUE], *inscale = in[INSCALE];

	for(i=0;i<cycles;i++) outsig[i] = atan(insig[i]*inscale[i])/1.571;
}

// ----------------------- other authors --------------------------
// Jens Hahn

/*** not in the release

class Synth_WAVE_SOFTSAW :public SynthModule
{
	// inputs:
	enum { POSITION };

	// outputs:
	enum { OUTVALUE };
public:
	void Initialize() { haveCalculateBlock = true; };
	void Calculate();
	void CalculateBlock(unsigned long cycles);
	string getParams() { return("pos;outvalue"); }
	static void *Creator() { return new Synth_WAVE_SOFTSAW; }
};

ModuleClient MC_Synth_WAVE_SOFTSAW(SynthModule::get_MS,"Synth_WAVE_SOFTSAW",Synth_WAVE_SOFTSAW::Creator);

void Synth_WAVE_SOFTSAW::Calculate()
{
}

void Synth_WAVE_SOFTSAW::CalculateBlock(unsigned long cycles)
{
	unsigned long i;
	float *outsig = out[OUTVALUE], *pos = in[POSITION];

	// Falls im Anfangs- oder Endsektor wollen wir eine "runde" Cosinus-Welle simulieren
	if ((pos[i] < 0.1) | (pos[i] > 0.9) {
	  for(i=0;i<cycles;i++) {
	    outsig[i] = cos(pos[i]*2*M_PI);
	  }	
	}
	else	// Sonst ein harter Sgezahn
	{
	  for(i=0;i<cycles;i++) {
	    outsig[i] = 1 - (pos[i] * 2);
	  }
	}
}

**/

class Synth_WAVE_SQUARE :public SynthModule
{
	// inputs:
	enum { POSITION };

	// outputs:
	enum { OUTVALUE };
	long pos;
public:
	void Initialize() { pos = 0; haveCalculateBlock = true; };
	void Calculate() { assert(false); };
	void CalculateBlock(unsigned long cycles);
	string getParams() { return("pos;outvalue"); }
	static void *Creator() { return new Synth_WAVE_SQUARE; }
};

ModuleClient MC_Synth_WAVE_SQUARE(SynthModule::get_MS,"Synth_WAVE_SQUARE",Synth_WAVE_SQUARE::Creator);

void Synth_WAVE_SQUARE::CalculateBlock(unsigned long cycles)
{
	unsigned long i;
	float *outsig = out[OUTVALUE];
	float *pos = in[POSITION];
	float t;

	for(i=0;i<cycles;i++) {
	  t = 1;
	  if (pos[i]>=0.5) t = -1;
	  outsig[i] = t;
	}
}

class Synth_FX_CFLANGER :public SynthModule
{
	// inputs:
	enum { INVALUE, LFO, PROP_MINTIME, PROP_MAXTIME };

	// outputs:
	enum { OUTVALUE };

	#define MAXDELAY 44100
	float dbuffer[MAXDELAY];
	unsigned long dbpos;
	float center;
	float range;
public:
	void Initialize();
	void Calculate();
	void CalculateBlock(unsigned long samples);
	string getParams() { return("invalue,lfo,_mintime,_maxtime;outvalue"); }
	static void *Creator() { return new Synth_FX_CFLANGER; }
};

ModuleClient MC_Synth_FX_CFLANGER(SynthModule::get_MS,"Synth_FX_CFLANGER",Synth_FX_CFLANGER::Creator);

void Synth_FX_CFLANGER::Initialize()
{
	float t1,t2;
	float maxtime;
	float mintime;
	t1 = atof(getStringProperty(PROP_MAXTIME));
	t2 = atof(getStringProperty(PROP_MINTIME));
	// you never know who will use this...
	maxtime = max(t1,t2);
	mintime = min(t1,t2);
	center = (maxtime + mintime) / 2;
	range = maxtime - center;
	
	for(dbpos=0;dbpos<MAXDELAY;dbpos++) dbuffer[dbpos] = 0;
	dbpos = 0;
	
	haveCalculateBlock = true;
}

void Synth_FX_CFLANGER::Calculate()
{
}

void Synth_FX_CFLANGER::CalculateBlock(unsigned long samples)
{
	float *insig = in[INVALUE], *lfo = in[LFO], *outsig = out[OUTVALUE];
	unsigned long i;
	float delay,floor_delay;
	long start_pos, end_pos;
	float start_val, end_val;

	for(i=0;i<samples;i++)
	{
		dbuffer[dbpos] = insig[i];

		// Delaytime i.e. = 35ms + (+/- LFO[-1 bis 1] * 15ms) / 1000 * 44100
		delay = ((center + (lfo[i] * range))/1000.0) * (float) samplingRate;
		floor_delay = floor(delay);
		start_pos = dbpos - (long)(floor_delay);
		end_pos   = start_pos-1;
		if(start_pos < 0) start_pos += MAXDELAY;		// wrapping exception
		if(end_pos < 0) end_pos += MAXDELAY;
		start_val = dbuffer[start_pos];
		end_val = dbuffer[end_pos];
		outsig[i] = start_val + ((delay - floor_delay) * (end_val - start_val));
		dbpos++;
		if(dbpos == MAXDELAY) dbpos = 0;
	}
}


/*
// TREMOLO
//
// The tremolo module modulates the amplitude according to a LFO-Wave.
// Traditionally you would use a sine wave but why limit yourself?
// What you get is a very intense effect that cuts through most
// arrangements because of its high dynamic range.
// The tremolo effect is still one of guitarists favourite effects although
// it's not as popular as in the 1960's.

class Synth_TREMOLO :public SynthModule
{
	// inputs:
	enum { INVALUE, INLFO };

	// outputs:
	enum { OUTVALUE };

public:
	void Initialize();
	void Calculate() {assert(false);}
	void CalculateBlock(unsigned long cycles);
	string getParams() { return("invalue,inlfo;outvalue"); }
	static void *Creator() { return new Synth_TREMOLO; }
};

ModuleClient MC_Synth_TREMOLO(SynthModule::get_MS,"Synth_TREMOLO",Synth_TREMOLO::Creator);

void Synth_TREMOLO::Initialize()
{
	haveCalculateBlock=true;
}

void Synth_TREMOLO::CalculateBlock(unsigned long cycles)
{
	unsigned long i;
	float *outsig = out[OUTVALUE], *insig = in[INVALUE], *inlfo = in[INSCALE];

	// Simply modulate the amplitude
	for(i=0;i<cycles;i++) {
	  outsig[i] = insig[i] * fabs(in[i]);
	}
}
**/

#define U32 unsigned long

// AUTOPANNER
//
// An Autopanner is used to automatically pan the input signal between 
// the left and the right output. This makes mixes more lively. A standard 
// application would be a guitar or lead sound. 
// Connect a LFO, a sine or saw wave for example to "inlfo"
// and select a frequency between 0.1 and 5Hz for a traditional effect or even
// more for Special FX

class Synth_AUTOPANNER :public SynthModule
{
	// inputs:
	enum { INVALUE, INLFO };

	// outputs:
	enum { OUTVALUE_L, OUTVALUE_R };

public:
	void Initialize();
	void Calculate() {assert(false);}
	void CalculateBlock(unsigned long cycles);
	string getParams() { return("invalue,inlfo;outvalue_l, outvalue_r"); }
	static void *Creator() { return new Synth_AUTOPANNER; }
};

ModuleClient MC_Synth_AUTOPANNER(SynthModule::get_MS,"Synth_AUTOPANNER",Synth_AUTOPANNER::Creator);

void Synth_AUTOPANNER::Initialize()
{
	haveCalculateBlock=true;
}

void Synth_AUTOPANNER::CalculateBlock(unsigned long cycles)
{
 unsigned long i;
 float *outsigl = out[OUTVALUE_L], *outsigr = out[OUTVALUE_R];
 float *insig = in[INVALUE], *inlfo = in[INLFO];
 float amp_mod;

 for(i=0;i<cycles;i++) {
   // assume signal was in range -1 .. 1 before, so now make it 0..1
   // FIXME: is that nice like that?
   amp_mod = (inlfo[i]+1.0)/2.0;
   outsigl[i] = insig[i] * (1 - amp_mod);    // 0 is left
   outsigr[i] = insig[i] * amp_mod;          // 1 is right
 }
}

/*
// BRICKWALL_LIMITER
//
// A brickwall limiter is used to protect equipment (and your ears..)
// from peaks that exceed the dynamic range of your system. It doesn't
// sound good but it's better than digital distortion.

class Synth_BRICKWALL_LIMITER:public SynthModule
{
	// inputs:
	enum { INVALUE};

	// outputs:
	enum { OUTVALUE };

public:
	void Initialize();
	void Calculate() {assert(false);}
	void CalculateBlock(unsigned long cycles);
	string getParams() { return("invalue;outvalue"); }
	static void *Creator() { return new Synth_BRICKWALL_LIMITER; }
};

ModuleClient MC_Synth_BRICKWALL_LIMITER(SynthModule::get_MS,"Synth_BRICKWALL_LIMITER",Synth_BRICKWALL_LIMITER::Creator);

void Synth_BRICKWALL_LIMITER::Initialize()
{
	haveCalculateBlock=true;
}

void Synth_BRICKWALL_LIMITER::CalculateBlock(unsigned long cycles)
{
	unsigned long i;
	float *outsig = out[OUTVALUE], *insig = in[INVALUE];
	float fin;
		
	for(i=0;i<cycles;i++) {
	  fin = insig[i]
	  if (fin > 1) {
	    outsig[i] = 1;
	    continue;
	  if (fin < -1) {
	    outsig[i] = -1;
	    continue;
	  }
	  outsig[i] = fin;
	}
	  
}
*/
