    /*

    Copyright (C) 1999 Jens Hahn
                       Jens.Hahn@t-online.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "synthmodule.h"
#include <math.h>

// TREMOLO
//
// The tremolo module modulates the amplitude according to a LFO-Wave.
// Traditionally you would use a sine wave but why limit yourself?
// What you get is a very intense effect that cuts through most
// arrangements because of its high dynamic range.
// The tremolo effect is still one of guitarists favourite effects although
// it's not as popular as in the 1960's.

class Synth_TREMOLO :public SynthModule
{
 // inputs:
 enum { INVALUE, INLFO };

 // outputs:
 enum { OUTVALUE };

public:
 void Initialize();
 void Calculate() {assert(false);}
 void CalculateBlock(unsigned long cycles);
 string getParams() { return("invalue,inlfo;outvalue"); }
 static void *Creator() { return new Synth_TREMOLO; }
};

ModuleClient
MC_Synth_TREMOLO(SynthModule::get_MS,"Synth_TREMOLO",Synth_TREMOLO::Creator)
;

void Synth_TREMOLO::Initialize()
{
 haveCalculateBlock=true;
}

void Synth_TREMOLO::CalculateBlock(unsigned long cycles)
{
 unsigned long i;
 float *outsig = out[OUTVALUE], *insig = in[INVALUE], *inlfo = in[INLFO];

 // Simply modulate the amplitude
 for(i=0;i<cycles;i++) {
   outsig[i] = insig[i] * inlfo[i];
 }
}

#define U32 unsigned long

// AUTOPANNER
//
// An Autopanner is used to automatically pan the input signal between
// the left and the right output. This makes mixes more lively. A standard
// application would be a guitar or lead sound.
// Connect a LFO, a sine or saw wave for example to "inlfo"
// and select a frequency between 0.1 and 5Hz for a traditional effect or even
// more for Special FX

class Synth_AUTOPANNER :public SynthModule
{
 // inputs:
 enum { INVALUE, INLFO };

 // outputs:
 enum { OUTVALUE_L, OUTVALUE_R };

public:
 void Initialize();
 void Calculate() {assert(false);}
 void CalculateBlock(unsigned long cycles);
 string getParams() { return("invalue,inlfo;outvalue_l, outvalue_r"); }
 static void *Creator() { return new Synth_AUTOPANNER; }
};

ModuleClient
MC_Synth_AUTOPANNER(SynthModule::get_MS,"Synth_AUTOPANNER",Synth_AUTOPANNER::Creator);

void Synth_AUTOPANNER::Initialize()
{
 haveCalculateBlock=true;
}

void Synth_AUTOPANNER::CalculateBlock(unsigned long cycles)
{
 unsigned long i;
 float *outsigl = out[OUTVALUE_L], *outsigr = out[OUTVALUE_R];
 float *insig = in[INVALUE], *inlfo = in[INLFO];
 float amp_mod;

 for(i=0;i<cycles;i++) {
   // assume signal was in range -1 .. 1 before, so now make it 0..1
   // FIXME: is that nice like that?
   amp_mod = (inlfo[i]+1.0)/2.0;
   outsigl[i] = insig[i] * (1 - amp_mod);    // 0 is left
   outsigr[i] = insig[i] * amp_mod;          // 1 is right

/*
   amp_mod = fabs(inlfo[i]);
   if (inlfo[i] >= 0) {
     outsigl[i] = insig[i] * (1 - amp_mod);
     outsigr[i] = insig[i] * amp_mod;
   }
   else
   {
     outsigl[i] = insig[i] * amp_mod;
     outsigr[i] = insig[i] * (1 - amp_mod);
   }
*/
 }
}


// BRICKWALL_LIMITER
//
// A brickwall limiter is used to protect equipment (and your ears..)
// from peaks that exceed the dynamic range of your system. It doesn't
// sound good but it's better than digital distortion.

class Synth_BRICKWALL_LIMITER :public SynthModule
{
 // inputs:
 enum { INVALUE};

 // outputs:
 enum { OUTVALUE };

public:
 void Initialize();
 void Calculate() {assert(false);}
 void CalculateBlock(unsigned long cycles);
 string getParams() { return("invalue;outvalue"); }
 static void *Creator() { return new Synth_BRICKWALL_LIMITER; }
};

ModuleClient MC_Synth_BRICKWALL_LIMITER(SynthModule::get_MS,"Synth_BRICKWALL_LIMITER",Synth_BRICKWALL_LIMITER::Creator);

void Synth_BRICKWALL_LIMITER::Initialize()
{
 haveCalculateBlock=true;
}

void Synth_BRICKWALL_LIMITER::CalculateBlock(unsigned long cycles)
{
 unsigned long i;
 float *outsig = out[OUTVALUE], *insig = in[INVALUE];
 float fin;

 for(i=0;i<cycles;i++) {
   fin = insig[i];  // FIXME: should have a parameter for the clipping
                    // level (then you can use it nicely to deform wave
					// forms, like clipping a sine wave with an envelope
   if (fin > 1) {   // rewrote that without continues ;)
     outsig[i] = 1;
   }
   else if (fin < -1) {
     outsig[i] = -1;
   }
   else outsig[i] = fin;
 }
}

// CHORUS
//
// Implementation of a 1-Voice Stereo-Chorus which should be combined with some others of its
// kind, using different parameters at least for rate, center and depth
//
// Parameters
// FIXME: the units for times must be given in description
//
// RATE: Waveform with values between -1 and 1, preferably a sine wave
// CENTER: Center time of delay
// DEPTH: Amount of time to be added or subtracted, in- and decreasing with the Chorus RATE
// G: That's where it becomes mystical: Parameter to control the gain amount of the all-pass filter
//    Values between 0.1 and 1 should be tried
// PHASEDELAY: Amount of delay for the phase-shifting part.
//    Example: 100ms delay phaseshift 10kHz signal by 360 degrees. This is frequency dependent!
// LEFT_OR_RIGHT: Input a value > 0 to phase-shift left channel or < 0 to phase-shift right channel

// FIXME: crash with the following parameters:
// center 100.0000
// depth 20.000
// g 0.5000
// phasedelay 100.0000
// left_or_right -0.6

class Synth_CHORUS :public SynthModule
{
	// inputs:
	enum { INVALUE_L, INVALUE_R, RATE, CENTER, DEPTH, G, PHASEDELAY, LEFT_OR_RIGHT};

	// outputs:
	enum { OUTVALUE_L, OUTVALUE_R };

	#define SR 44100
	#define MAXDELAY SR / 2
	/* 500ms 
	 *
	 * FIXME: use samplingRate instead inside the code, and 
	 * dynamically malloc the buffers if thats necessary due to that
	 */
	float dbuffer[MAXDELAY*2];
	float dbuffer2[MAXDELAY*2];
	float dbuffer3[MAXDELAY*2];

	U32 dpos;
public:
	void Initialize();
	void Calculate() {assert(false);}
	void CalculateBlock(unsigned long cycles);
	string getParams() { return("invalue_l, invalue_r, rate, center, depth, g, phasedelay, left_or_right; outvalue_l, outvalue_r"); }
	static void *Creator() { return new Synth_CHORUS; }
};

ModuleClient MC_Synth_CHORUS(SynthModule::get_MS,"Synth_CHORUS",Synth_CHORUS::Creator);

void Synth_CHORUS::Initialize()
{
	haveCalculateBlock=true;
	for (U32 i=0; i < MAXDELAY; i++) dbuffer[i] = 0;
	for (U32 i=0; i < MAXDELAY; i++) dbuffer2[i] = 0;
	for (U32 i=0; i < MAXDELAY; i++) dbuffer3[i] = 0;
	dpos = 0;
}

void Synth_CHORUS::CalculateBlock(unsigned long cycles)
{
	unsigned long i, delay, pdelay, position, phase_pos;
	float *outsig_l = out[OUTVALUE_L], *outsig_r = out[OUTVALUE_R], *insig_l = in[INVALUE_L],
		*insig_r = in[INVALUE_R], *rate = in[RATE], *center = in[CENTER], *depth = in[DEPTH], *g = in[G],
		*phasedelay = in[PHASEDELAY], *left_or_right = in[LEFT_OR_RIGHT];
	float left, right;

	for(i=0;i<cycles;i++) {
	  dbuffer[dpos] = insig_l[i];
	  dbuffer[dpos+1] = insig_r[i];
	// Calc delay time for input delay
	  // FIXME: never assume a fixed sampling rate (such as 44100)
	  delay = (unsigned int) ((center[i] + (depth[i] * rate[i])) / 1000) * 44100;
	  position = dpos - delay;
	  if(position < 0) position += MAXDELAY;
	// Calc delay time for Phase-Shifting Allpass-Filter bestimmen
	  pdelay = (unsigned int) (phasedelay[i] / 1000) * 44100;
	  phase_pos = dpos - pdelay;
	  if(phase_pos < 0) phase_pos += MAXDELAY;

	  left = dbuffer[position];
	  right = dbuffer[position+1];
	  dbuffer3[dpos] = left;			// store in delay line for all-pass in-values
	  dbuffer3[dpos+1] = right;

	    // Hm, if that's a working all-pass???
	    // formula is: y[n] = (-g * x[n]) + x[n-D] + (g * y[n-D])
	    // whereas left/right is the signal which would be output if no phase-shift would be added,
	    // dbuffer3 contains saved input (=> already "flanged" sampels!!) and dbuffer2 calculated output samples.
	    // "g" is the gain-amount, the subtraction ("-g") cancels out comb filter's spectral effect, 
	    // while preserving echo and delay characteristics.

	  if(left_or_right[0] > 0) {		// Decide whether we shift left or right side
	    left = (-g[i] * left) + dbuffer3[phase_pos] + (g[i] * dbuffer2[phase_pos]);
	  }
	  else
	  {
	    right = (-g[i] * right) + dbuffer3[phase_pos+1] + (g[i] * dbuffer2[phase_pos+1]);
	  }

	  dbuffer2[dpos] = left;		// store in outvalue buffer for all-pass-filter
	  dbuffer2[dpos+1]= right;
	  outsig_l[i] = left;			// finally write to output buffer
	  outsig_r[i] = right;
	  dpos += 2;
	}
}
