/*
 * puzzle.cpp -- Copyright (C) 1998 by M. G"otze
 */

#include <iostream.h>
#include <stdlib.h>
#include <time.h>

extern "C" {
#include <mediatool.h>
}

#include <qbitmap.h>
#include <qbrush.h>
#include <qcolor.h>
#include <qimage.h>
#include <qpaintd.h>
#include <qpainter.h>
#include <qpen.h>
#include <qpixmap.h>
#include <qstring.h>
#include <qwidget.h>

#include <kapp.h>
#include <kaudio.h>

#include "puzzle.h"
#include "cursors.h"

const char *dataFileExt = ".gif";
const char *defImageFileName = "image0.gif";

KSlidePuzzle::KSlidePuzzle(QWidget *parent, const char *name, const char *path,
  const tOpt opt) : QWidget(parent, name), state(0), options(opt)
{
  srand(time(0));
  puz.prevImage = "";
  tempFileName = 0;
  puz.prevMousePos = QCursor::pos();
  puz.path = qstrdup(path);
  puz.mode = 1;                // default mode is classical game with 4x4 tiles
  puz.size = puz.scrambling = 0;
  setFixedSize(scn.frameSize + 2 * scn.puzOffset, scn.frameSize +
    2 * scn.puzOffset);
  for (int i = 0; i <= 4; i++)
    puz.cursors[i] = new QCursor(QBitmap(32, 32, cursorsData[2*i], TRUE),
      QBitmap(32, 32, cursorsData[2*i+1], TRUE));
  puz.cursors[5] = (QCursor*) &arrowCursor;
  puz.cursors[6] = (QCursor*) &waitCursor;
  setCursor(5);
  puz.frame = new QPixmap(getDataFileName(fnFrame));
  QBitmap *mask = new QBitmap(getDataFileName(fnFrameMask));
  puz.frame->setMask(*mask);
  delete mask;
  for (tInd i = 0; i < maxTiles * maxTiles; i++)
    puz.tiles[i] = 0;
  ani.buf = 0;
  audio = new KAudio();
  if (audio->serverStatus()) {                 // if sound initalization failed
    delete audio;
    audio = 0;
    setOption(opSoundOn, FALSE);
  }
  setMouseTracking(TRUE);
}

KSlidePuzzle::~KSlidePuzzle()
{
  // only the first five cursor pointers have been allocated!
  for (int i = 0; i <= 4; i++)
    delete puz.cursors[i];
  for (tInd i = 0; i < maxTile(); i++)
    delete puz.tiles[i];
  delete tempFileName;
  delete puz.frame;
  delete ani.buf;
  delete audio;
}

void KSlidePuzzle::paintEvent(QPaintEvent*)
{
  bitBlt(this, scn.puzOffset, scn.puzOffset, puz.frame);
  drawPuzzle();
}

void KSlidePuzzle::drawPuzzle()
{
  for (tInd i = 0; i < maxTile(); i++)
    drawTile(i);
}

void KSlidePuzzle::timerEvent(QTimerEvent*)
{
  if (isUncompleting()) {
    tInd x, y;
    spiral(ani.delta--, x, y);
    bitBlt(this, ani.x + x, ani.y + y, ani.buf);
    if (ani.delta == ani.tVal) {
      puz.pos[puz.missingTile] = noTile;
      updateMovesTos();
      setState(stUncompleting | stComplete, FALSE);
      killTimers();
      if (puz.scrambling) { 
        scramble(TRUE);
        startTimer(10);
      }
    }
  } else if (isMoving()) {
    if (ani.hor)
      bitBlt(this, ani.x + ani.delta, ani.y, ani.buf);
    else
      bitBlt(this, ani.x, ani.y + ani.delta, ani.buf);
    ani.delta += ani.step;
    if (ani.delta == ani.tVal) {
      killTimers();
      delete ani.buf;
      ani.buf = 0;
      updateMovesTos();
      setState(stMoving, FALSE);
      mouseMoveEvent(0);
      if (puz.scrambling)
        scramble(TRUE);
      else {
        if (soundOn())
          audio->play(QString(getFullSoundPath("sound1.wav")).data());
        solveCheck();
      }
    }
  } else if (isOrdering()) {
    tInd i, j;
    for (i = 0; i < maxTile(); i++) {
      if (puz.pos[i] != fixTile && i != puz.missingTile && puz.pos[i] != i) {
        for (j = lastTile(); puz.pos[j] != i; j--);
        swap(puz.pos[i], puz.pos[j]);
        drawTile(i);
        drawTile(j);
        break;
      }
    }
    if (i == maxTile()) {
      killTimers();
      setState(stOrdering, FALSE);
      if (solveCheck(FALSE))
        sigStoppedOrdering();
    }
  } else if (isCompleting()) {
    tInd x, y;
    spiral(ani.delta++, x, y);
    bitBlt(this, ani.x + x, ani.y + y, ani.buf, x, y, ani.step, ani.step,
      CopyROP, TRUE);
    if (ani.delta == ani.tVal) {
      killTimers();
      puz.pos[puz.missingTile] = puz.missingTile;
      setState(stCompleting, FALSE);
      setState(stComplete, TRUE);
      setCursor(cuNormal);
    }
  }
}

void KSlidePuzzle::mousePressEvent(QMouseEvent *event)
{
  if (isComplete() || event->button() != LeftButton)
    return;
  tInd x = (event->x() - scn.imgOffset) / tileSize();
  tInd y = (event->y() - scn.imgOffset) / tileSize();
  moveTile(coords2Index(x, y));
}

void KSlidePuzzle::mouseMoveEvent(QMouseEvent *event)
{
  if (event)  // event is 0 if called manually
    puz.prevMousePos = event->pos();
  if (isComplete() || !isIdle())
    return;
  unsigned char c = cuNormal;
  int x, y;
  if (event) {
    x = event->x() - scn.puzOffset - scn.imgOffset;
    y = event->y() - scn.puzOffset - scn.imgOffset;
  } else {
    x = puz.prevMousePos.x() - scn.puzOffset - scn.imgOffset;
    y = puz.prevMousePos.y() - scn.puzOffset - scn.imgOffset;
  }
  tInd ox = x / tileSize();
  tInd oy = y / tileSize();
  if (x >= 0 && y >= 0 && ox < puz.size && oy < puz.size) {
    c = cuNop;
    tInd z;
    if ((z = puz.movesTo[coords2Index(ox, oy)]) != noTile) {
      tInd nx, ny;
      index2Coords(z, nx, ny);
      if (abs(ox - nx) + abs(oy - ny) == 1)
        c = oy == ny ? (nx > ox ? cuRight : cuLeft) : (ny > oy ? cuDown : cuUp);
    }
  }
  if (c == puz.curCur)
    return;
  setCursor(c);
}

void KSlidePuzzle::scramble()
{
  if (!puz.scrambling)
    scramble(3 * (puz.size+1) * (puz.size+1));
  else
    scramble(0);
}

void KSlidePuzzle::order()
{
  if (!isIdle())
    return;
  setState(stOrdering, TRUE);
  setCursor(cuWait);
  sigStartedOrdering();
  startTimer(50);
}

bool KSlidePuzzle::loadImage(tOpt mode, const char *name)
{
  if (!isIdle())
    return FALSE;
  QString fileName = name;
  if (!fileName.length()) {
    if (!puz.prevImage.length() || (mode & moLoadDefImage) == moLoadDefImage)
      fileName = getFullPicPath(defImageFileName);
    else
      fileName = puz.prevImage;
  }
  mode = (mode & moKeepMode) == moKeepMode ? puz.mode : mode & 0x0f;
  tInd i, size = mode < 3 ? mode+3 : 6;
  // let's preliminarily keep the original tiles...
  QPixmap *oldTiles[maxTiles * maxTiles];
  for (i = 0; i < maxTile(); i++)
    oldTiles[i] = puz.tiles[i];
  // load image, tile borders, tile border masks
  QPixmap *image = new QPixmap(fileName);
  QPixmap *tBorder = new QPixmap(getDataFileName(fnTiles));
  QBitmap *mask = new QBitmap(getDataFileName(fnTilesMask));
  if (image->size() == QSize(imageSize, imageSize) &&
    tBorder->size() == QSize(456, 160) && mask->size() == QSize(456, 160))
  {
    tBorder->setMask(*mask);
    QBitmap *tMask = new QBitmap(tileSize(size), tileSize(size));
    // calculate horizontal offset of the required tile border
    tInd left = 0;
    for (i = 3; i < size; i++)
      left += tileSize(i);
    bitBlt(tMask, 0, 0, mask, left, 0, tileSize(size), tileSize(size));
    // create tile mask from tile border mask
    QPainter *paint = new QPainter(tMask);
    paint->fillRect(2, 2, tileSize(size)-3, tileSize(size)-3, QBrush(color1));
    delete paint;
    // split image into tiles, glue a tile border to each and define a mask
    for (i = 0; i < size * size; i++) {
      puz.tiles[i] = new QPixmap(tileSize(size), tileSize(size));
      bitBlt(puz.tiles[i], 0, 0, image, (i % size) * tileSize(size),
        (i / size) * tileSize(size), tileSize(size), tileSize(size));
      if (isFixTile(i, mode)) {
        QImage *img = new
          QImage(puz.tiles[i]->convertToImage().convertDepth(8));
        for (int j = 0; j < img->numColors(); j++)
          img->setColor(j, qRgb(int((qGray(img->color(j)) * scn.fixCoeffs[0])),
            int((qGray(img->color(j)) * scn.fixCoeffs[1])),
            int((qGray(img->color(j)) * scn.fixCoeffs[2]))));
        puz.tiles[i]->convertFromImage(*img);
        delete img;
        puz.pos[i] = fixTile;
      } else {
        bitBlt(puz.tiles[i], 0, 0, tBorder, left, 0, tileSize(size),
          tileSize(size));
        puz.tiles[i]->setMask(*tMask);
        puz.pos[i] = i;
      }
    }
    tInd x, y;
    paint = new QPainter();
    if (mode > 3) {  // mode with fixed tiles?
      for (i = 0; i < size * size; i++) {
        if (puz.pos[i] == fixTile) {
          index2Coords(i, x, y);
          paint->begin(puz.tiles[i]);
          paint->setPen(QPen(scn.borderColor, 1));
          if (x > 0 && puz.pos[i-1] != fixTile)            // left border line?
            paint->drawLine(0, 0, 0, tileSize(size));
          if (x < 5 && puz.pos[i+1] != fixTile)                       // right?
            paint->drawLine(tileSize(size)-1, 0, tileSize(size)-1,
              tileSize(size));
          if (y > 0 && puz.pos[i-6] != fixTile)                         // top?
            paint->drawLine(0, 0, tileSize(size), 0);
          if (y < 5 && puz.pos[i+6] != fixTile)                      // bottom?
            paint->drawLine(0, tileSize(size)-1, tileSize(size),
              tileSize(size)-1);
          paint->end();
        }
      }
    }
    delete paint;
    delete mask;
    delete tBorder;
    delete image;
    // delete possibly previously allocated tiles
    for (i = 0; i < maxTile(); i++)
      delete oldTiles[i];
    puz.prevImage = fileName;
    puz.mode = mode;
    puz.size = size;
    setState(stComplete, TRUE);
    drawPuzzle();
    sigStoppedOrdering();
    return TRUE;
  }
  // something's gone wrong, so let's clean up...
  delete mask;
  delete tBorder;
  delete image;
  for (i = 0; i < maxTile(); i++)
    puz.tiles[i] = oldTiles[i];
  return FALSE;
}

unsigned int KSlidePuzzle::movesTo(const tInd index)
{
  tInd target = noTile;
  if (puz.pos[index] < noTile && index < maxTile()) {
    tInd x, y;
    index2Coords(index, x, y);
    if (y > 0 && puz.pos[coords2Index(x, y-1)] == noTile)
      target = coords2Index(x, y-1);                                 // upwards
    if (y < puz.size-1 && puz.pos[coords2Index(x, y+1)] == noTile)
      target = coords2Index(x, y+1);                               // downwards
    if (x > 0 && puz.pos[index-1] == noTile)                // towards the left
      target = index-1;
    if (x < puz.size-1 && puz.pos[index+1] == noTile)      // towards the right
      target = index+1;
  }
  return target;
}

void KSlidePuzzle::updateMovesTos()
{
  for (tInd i = 0; i < maxTile(); i++)
    puz.movesTo[i] = movesTo(i);
}

void KSlidePuzzle::drawTile(const tInd index)
{
  tInd x, y;
  index2Coords(index, x, y);
  if (puz.tiles[puz.pos[index]] && puz.pos[index] < noTile) {
    bitBlt(this, tileSize() * x + scn.imgOffset, tileSize() * y +
      scn.imgOffset, puz.tiles[puz.pos[index]]);
  } else if (puz.pos[index] == fixTile) {
    bitBlt(this, tileSize() * x + scn.imgOffset, tileSize() * y +
      scn.imgOffset, puz.tiles[index]);
  } else if (puz.pos[index] == noTile) {
    QPixmap *tile = new QPixmap(tileSize(), tileSize());
    tile->fill(backgroundColor());
    bitBlt(this, tileSize() * x + scn.imgOffset, tileSize() * y +
      scn.imgOffset, tile);
    delete tile;
  }
}

void KSlidePuzzle::moveTile(const tInd index)
{
  if (getState(stInAction) || index > lastTile())
    return;
  tInd target = puz.movesTo[index];
  if (target == noTile)
    return;
  setState(stMoving, TRUE);
  setCursor(cuWait);
  tInd x, y;
  index2Coords(index, x, y);
  ani.x = x * tileSize() + scn.imgOffset;
  ani.y = y * tileSize() + scn.imgOffset;
  ani.step = 8;
  if (target < index)
    ani.step = -ani.step;
  ani.hor = abs(target - index) == 1;
  ani.delta = ani.step < 0 ? ani.step : 0;
  ani.buf = new QPixmap(tileSize() + abs(ani.step) * ani.hor, tileSize() +
    abs(ani.step) * !ani.hor);
  ani.buf->fill(backgroundColor());
  ani.tVal = ani.step < 0 ? ani.step - tileSize() : tileSize();
  int i = ani.step >= 0 ? ani.step : 0;
  bitBlt(ani.buf, i * ani.hor, i * !ani.hor, puz.tiles[puz.pos[index]]);
  ani.buf->optimize(TRUE);
  swap(puz.pos[index], puz.pos[target]);
  startTimer(10);
}

void KSlidePuzzle::scramble(const tInd moves)
{
  if (!moves) {
    if (puz.scrambling)
      sigStoppedScrambling();
    puz.scrambling = 0;
    return;
  };
  if (getState(stInAction))
    return; 
  if (!puz.scrambling) {
    puz.scrambling = moves;
    puz.prevScramble = noTile;
    sigStartedScrambling();
    if (isComplete()) {
      setState(stUncompleting, TRUE);
      setCursor(cuWait);
      chooseMissingTile();
      tInd x, y;
      index2Coords(puz.missingTile, x, y);
      ani.x = tileSize() * x + scn.imgOffset;
      ani.y = tileSize() * y + scn.imgOffset;
      ani.delta = 15; 
      ani.tVal = -1;
      ani.buf = new QPixmap(tileSize()/4, tileSize()/4);
      ani.buf->fill(backgroundColor());
      startTimer(25);
      return;
    }
  }
  tInd moveable[4], targets[4];
  tInd i, j, k = 0;
  for (i = 0; i < maxTile(); i++)
    if ((j = puz.movesTo[i]) != noTile) {
      moveable[k] = i;
      targets[k++] = j;
    }
  for (i = rand() % k; moveable[i] == puz.prevScramble; i = rand() % k);
  puz.prevScramble = targets[i];
  moveTile(moveable[i]);
  if (!--puz.scrambling)
    sigStoppedScrambling();
}

bool KSlidePuzzle::solveCheck(const bool honor)
{
  if (!isIdle()) return FALSE;
  tInd i;
  for (i = 0; i < maxTile(); i++)
    if (puz.pos[i] != fixTile && i != puz.missingTile && puz.pos[i] != i)
      return FALSE;
  tInd x, y;
  index2Coords(puz.missingTile, x, y);
  ani.x = tileSize() * x + scn.imgOffset;
  ani.y = tileSize() * y + scn.imgOffset;
  ani.delta = 0;
  ani.step = tileSize()/4;
  ani.tVal = 16;
  ani.buf = puz.tiles[puz.missingTile];
  startTimer(25);
  if (soundOn() && honor)
    audio->play(QString(getFullSoundPath("sound0.wav")).data());
  setState(stCompleting | stComplete, TRUE);
  setCursor(cuWait);
  if (honor)
    sigPuzzleSolved();
  return TRUE;
}

void KSlidePuzzle::spiral(const tInd index, tInd &x, tInd &y)
{
  static const unsigned char coords[16][2] = {
    {0, 0}, {1, 0}, {2, 0}, {3, 0}, {3, 1}, {3, 2}, {3, 3}, {2, 3},
    {1, 3}, {0, 3}, {0, 2}, {0, 1}, {1, 1}, {2, 1}, {2, 2}, {1, 2}
  };
  tInd step = tileSize() / 4;
  x = coords[index][0] * step;
  y = coords[index][1] * step;
}

bool KSlidePuzzle::isFixTile(const tInd index, const tInd mode)
{
  tInd x, y;
  index2Coords(6, index, x, y);
  switch (mode) {
    case 4:                     // puzzle-piece-like moveable range ("corners")
      return ((y == 2 || y == 3) && (x == 0 || x == 5)) ||
        ((y == 0 || y == 5) && (x == 2 || x == 3));
    case 5:                                                          // a cross
      return (x < 2 || x > 3) && (y < 2 || y > 3);
    case 6:                           // diamond-shaped range of moveable tiles
      return ((y < 2 || y > 3) && (x == 0 || x == 5)) ||
        ((y == 0 || y == 5) && (x == 1 || x == 4));
    case 7:
      return (x == 1 || x == 4) && (y == 1 || y == 4);               // "isles"
    case 8:                                                         // triangle
      return x + y > 6;
  }
  return FALSE; 
}

bool KSlidePuzzle::isCorner(const tInd index)
{
  if (index > lastTile() || puz.pos[index] >= noTile)
    return FALSE;
  tInd x, y;
  index2Coords(index, x, y);
  return ((x == 0 || x == puz.size-1) ||
    (x > 0 && puz.pos[index-1] == fixTile) ||
    (x < puz.size-1 && puz.pos[index+1] == fixTile)) &&
    ((y == 0 || y == puz.size-1) ||
    (y > 0 && puz.pos[index-puz.size] == fixTile) ||
    (y < puz.size-1 && puz.pos[index+puz.size] == fixTile));
}

void KSlidePuzzle::chooseMissingTile()
{
  if (randomMissingTile())
    for (puz.missingTile = noTile; !isCorner(puz.missingTile);
      puz.missingTile = rand() % (puz.size * puz.size));
  else
    for (puz.missingTile = lastTile(); !isCorner(puz.missingTile);
      puz.missingTile--);
}

bool KSlidePuzzle::toggleSound()
{
  setOption(opSoundOn, bool(audio) && !soundOn());
  return soundOn();
}

bool KSlidePuzzle::toggleRandomMissingTile()
{
  setOption(opRandomMissingTile, !randomMissingTile());
  return randomMissingTile();
}

char* KSlidePuzzle::getFullPicPath(const char *file)
{
  delete tempFileName;
  return tempFileName = qstrdup(QString(puz.path) + "/pics/" + file);
}

char* KSlidePuzzle::getFullSoundPath(const char *file)
{
  delete tempFileName;
  return tempFileName = qstrdup(QString(puz.path) + "/sounds/" + file);
}

char* KSlidePuzzle::getDataFileName(const tOpt file)
{
  switch (file) {
    case fnFrame:
      return getFullPicPath(QString(scn.frameFileName) + dataFileExt);
    case fnFrameMask:
      return getFullPicPath(QString(scn.frameFileName) + 'm' + dataFileExt);
    case fnTiles:
      return getFullPicPath(QString(scn.tilesFileName) + dataFileExt);
    case fnTilesMask:
      return getFullPicPath(QString(scn.tilesFileName) + 'm' + dataFileExt);
    default:
      return 0;
  }
}
