/* kgo_widget.h
 *
 * Pieter Eendebak ( pte@ddsw.nl )
 *
 */

#ifndef KGO_WIDGET_H
#define KGO_WIDGET_H

#include "go_view.h"
#include "go_board.h"
#include "dialog.h"
#include "includes.h"

#define NO_GAME -1
#define COMPUTER_GAME 0
#define TWO_PLAYER_GAME 1
#define NETWORK_GAME 2

// for loading and saving config
#define LEGENDA_STRING "Legenda"
#define MOVE_LIST_STRING "Move list"

class KNetObject ;

class KGoWidget : public QWidget
{
	Q_OBJECT

public:
	/**
	 * The constructor method for class KGoWidget
	 *
	 * @param parent Parent widget, used for QWidget
	 * @param name	Name for QWidget
	 *
	 */
	KGoWidget( QWidget *parent=0, const char *name=0 );
	
	/**
	 * The destrucor method for class KFortune
	 */
	~KGoWidget();

	KGoBoard* getGoBoard() ; 
	KGoView* getGoView() ;

	int gameType() ;

public slots:

	/**
	 * Start a new game
	 */
	void newgame() ;
	/**
	 * This function should be called when the current game ends.
	 */
	void gameEnded() ;

	/**
	 * Saving a game to disk is quite straightforward :
	 * just call the methods save() from KGoBoard and the two
	 * KGoPlayer's. Loading again is a bit more difficult 
	 * (@see #loadGame).
	 * 
	 * @short Save the current game to disk
	 */
	void saveGame() ;
	/**
	 * Loading a game is quite difficult. This method selects a file
	 * and then checks if it is a Kgo savefile. After that 
	 * control is passed to another method, depending on the game type
	 * that is being loaded from disk.
	 *
	 * The different methods should (in general) do the following things :
	 *
	 * - start a new game from the same type
	 * - load the data to the KGoBoard and the two KGoPlayer's
	 * - start the game by calling KGoboard::?????
 	 *
	 * @see #loadComputerGame
	 * @see #loadTwoPlayerGame
	 * @see #loadNetworkGame
	 * @see #saveGame
   	 *
	 * @short Load a game from disk
	 */
	void loadGame() ;

	/**
	 * Not all game types can be saved at the moment.
	 * Use this method the check if it is possible
	 *
	 * @return true if the current game can be saved
	 */ 
	bool saveAbleGame() ;

	void countScores() ;

	/**
	 * This function switches the legenda
	 * on and off.
	 */
	void toggleLegenda() ;

	/**
	 * This methods enables or disables the legenda
	 */
	void legenda( bool ) ;

	/**
	 * This method can be used to check if
	 * the legenda is turned on or off
	 *
	 * @return true if the legenda is beging displayed 
	 */ 
	bool legenda() ;

	/**
	 * This function switches the list of moves
	 * on and off.
	 */
	void toggleMovesList() ;

	/**
	 * This function enabled the list of moves
	 */
	void movesList( bool ) ;
	/**
	 * @return true if the list of moves is displayed 
	 */ 
	bool movesList() ;

	/**
	 * Load a configuration from a KConfig object
	 */
	void loadconfig( KConfig * ) ;
	/**
	 * Save the configuration to a KConfig object
	 */
	void saveconfig( KConfig * ) ;
	/**
	 * Call this to clean up the kgo_widget
	 */
	void quitapp() ;

	static void error( char * ) ;
	void notImpl() ;

signals:
	void boardStateChange( int ) ;

protected:
	void gameType(int) ;

protected slots:
	void paintEvent( QPaintEvent* ) ;
	void resize() ;

	void network_client_receiver() ;

private slots:
	void resizeEvent( QResizeEvent * ) ;
	void player_moved(int,int,int) ;
	void status( int ) ;

private:
	/**
	 * This function is called to clean up stuff
	 * from the previous game. 
	 */
	bool endgame() ;
	void computerGame() ;
	void computerCGame( int comp_player=BLACK ) ;
	void twoplayerGame() ;
	void networkGame() ;

	/* used for different kinds of save-files */
	void loadComputerGame( QDataStream & ) ;
	void loadTwoPlayerGame( QDataStream & ) ;
	void loadNetworkGame( QDataStream & ) ;

	void saveComputerGame( QDataStream & ) ;
	void saveTwoPlayerGame( QDataStream & ) ;
	void saveNetworkGame( QDataStream & ) ;

	void setPlayers( KGoPlayer *, KGoPlayer * ) ;
	void setView( KGoView *w ) ;
	void setBoard( KGoBoard *b ) ;

	// data
	KGoBoard *board ;
	KGoPlayer *white ;
	KGoPlayer *black ;
	int game_type ;
	// needed for loading & saving computer games
	int computer_color ;
	// needed for network game
	KNetObject *nobject ;

	KGoView *view ;
	QLabel *list_title ;
	// show legenda or not
	bool blegenda ;
	KMoveList *moveList ;
	// show moves list or not
	bool moves_list ;

} ;

#endif
