/* go_player.h
 *
 * Pieter Eendebak ( pte@ddsw.nl )
 *
 */

#ifndef GO_PLAYER_H
#define GO_PLAYER_H

#include <qobject.h>
#include <qsignal.h>

class KGoBoard ;
class KGoWidget ;

/**
 * This is an abstract class representing a player.
 * Possible implementations include : a computer player, a player
 * connected with the screen, a network player.
 *
 **/
class KGoPlayer : public QObject
{
	// needed for loading and saving
	friend KGoBoard ;
	friend KGoWidget ;

	Q_OBJECT

public:
	KGoPlayer( int color, KGoBoard *b, QWidget *parent=0,
		const char *name = 0 );
 	~KGoPlayer();

	/**
	 * @return true is this player is to move
	 */
	bool toMove() ;

	/**
	 * @return the color of the player to move
	 */
	int playerToMove() ;

public slots:
	/**
	 * @return the color of this player
	 */
	int getColor() ;

	/**
	 * This slot is called whenever there changes
	 * something on the board. You don't need
	 * to implement this one. To receive moves
	 * you can use the slot opponentMoved().
	 */
	void boardStateChange( int ) ;

	/**
	 * This is called when the Go board is changed
	 * not by making a move. This is also called
	 * when the game is started. The player should check
	 * at least the following items :
	 *
	 * - does the board size match ?
	 * - do the fields of the board correspond ?
	 * - which player has to move next ?
	 *
	 * Must be implemented.
	 */
	virtual void boardSetup() = 0 ;

	/**
	 * This is called when the opponent makes a move.
	 */
	virtual void opponentMoved(int, int ) = 0 ;

	/**
	 * This slot is used to notify the player that it has
	 * made an illegal move. 
	 *
	 * @param r The reason for the illegal move
	 * @param p The player that made the illegal move
	 */
	virtual void illegalMove( int r, int p) = 0 ;

	/**
	 * This can be used to get extra data on the game. Tis
	 * also means the players don't need to keep a board
	 * for themselves.
	 *
	 * @return A pointer to the KGoBoard this player belongs to
	 * @see KGoBoard
	 */
	KGoBoard* getBoard() ;
	virtual void setBoard( KGoBoard *b ) ;
	
signals:
	/**
	 * Emit this to make a new move. The signal 
	 * is connected to a KGoBoard. 
	 */
	void doMove( int, int ) ;

protected:
	/**
	 * @return true is this implementation of KGoPlayer
	 * can save itself with the method save()
	 */
	virtual bool canBeSaved() ;

	/**
	 * Load a player from disk. The default implementation 
	 * does nothing.
	 * 
	 * @param s The data stream where data can be found
	 * @param b The go board where this player will play.
	 *
	 * @return The result code 
	 */
	virtual int load( QDataStream &s, KGoBoard *b ) ;

	/**
	 * Save the data for this player to a QDataStream. 
	 * The default implementation does nothing.
	 *
	 * @return The result code 
	 */
	virtual int save( QDataStream & ) ;

private:
	// the color of this player
	int color ;
	
	// pointer to to board this player is playing with
	KGoBoard *board ;

	#ifdef DEBUG
	static int count ;
	#endif
} ;

#endif
