/* -*- C++ -*-
  $Id: XmlReader.h,v 1.2 1998/10/12 13:13:45 sattler Exp $

  This file is part of KIllustrator.
  Copyright (C) 1998 Kai-Uwe Sattler (kus@iti.cs.uni-magdeburg.de)

  modified for kvoctrain by Ewald Arnold ewald@ewald-arnold.de in April 99

*/

#ifndef XmlReader_h_
#define XmlReader_h_

#include <iostream.h>

#include "XmlElement.h"
#include "XmlTokenizer.h"

/**
 * The XMLReader class supports reading elements from a XML stream.
 *
 * @short     A class for reading XML elements from a stream.
 * @author    Kai-Uwe Sattler (kus@iti.cs.uni-magdeburg.de)
 * @version   $Id: XmlReader.h,v 1.2 1998/10/12 13:13:45 sattler Exp $
 */
class XmlReader {
public:
  /**
   * Construct a XmlReader instance for the given input stream.
   *
   * @param is   The open input stream.
   */
  XmlReader (istream& is);

  /**
   * Desctructor
   */
  ~XmlReader ();

  /**
   * Check the input stream for a valid XML header.
   * A header should look like
   * <pre>
   *  &lt;?xml version="1.0"?&gt;
   *  &lt;!doctype dtype system "dtd"&gt;
   * </pre>
   * where <tt>dtype</tt> and <tt>dtd</tt> are simple strings.
   *
   * @return  <tt>true</tt> if the header conforms to XML, otherwise
   *          <tt>false</tt>.
   */
  bool validHeader ();

  /**
   * Return the document type.
   *
   * @return   The name of the document type.
   */
  const QString& doctype () const;

  /**
   * Return the name of the document type definition.
   *
   * @return   The name of the DTD.
   */
  const QString& dtd () const;

  /**
   * Read a XML element from the stream. If the content is plain text
   * (no tag), an element with the pseudo ID <tt>#PADATA</tt> is returned 
   * and the text is available via method getText.
   *
   * @see #getText
   *
   * @param elem    The XML element which is initialized by the method.
   * @return        <tt>true</tt> for successful reading.
   */
  bool readElement (XmlElement& elem);

  /**
   * Read plain text from the stream.
   *
   * @return The text as a string.
   */
  const QString& getText ();

  /**
   * Returns current line number
   *
   * @return The current line number
   */
  int lineNumber() { return tokenizer.lineNumber(); }

protected:
  bool parseEndElement (XmlElement& elem);
  bool parseElement (const QString& id, XmlElement& elem);
  bool readAttributes (list<XmlAttribute>& attrib_list);

private:
  XmlTokenizer tokenizer;
  QString s_dtype, s_dtd;
  QString text;
};

#endif




