/***************************************************************************
                          bibletimeprinter.cpp  -  description                              
                             -------------------                                         
    begin                : Thu Jul 8 1999                                           
    copyright            : (C) 1999 by Torsten Uhlmann                         
    email                : TUhlmann@gmx.de                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include <qfont.h>
#include <qcolor.h>
#include <qprinter.h>
#include <qpainter.h>
#include <qwindowdefs.h>
#include <qstring.h>
#include <qregexp.h>
#include <qmsgbox.h>

#include <kapp.h>
#include <kconfig.h>
#include <kprocess.h>

#include "bibletimeprinter.h"

BibleTimePrinter::BibleTimePrinter(QObject *parent, const char *name) :
									 QObject(parent, name)
{
	dataFont					= QFont("times", 12);
	keyFont						= QFont("helvetica", 12);
	dataColor					= "black";
	keyColor					= "red";
	printKey					= true;
	printData					= true;
	printHeaderText		= true;
	printFooterText		= true;
	verseList					= 0;
	
	leftBorder				= 40;
	rightBorder				= 40;
	upperBorder				= 40;
	lowerBorder				= 40;
	pageWidth					= 595;
	pageHeight				= 841;
	newLineSpace			= 3;
	newParagraphSpace	= 8;
	
	headText.setStr("printed with BibleTime");
	footText.setStr("printed with BibleTime, page %d");
	
	printFlag					= AlignLeft | AlignTop | WordBreak | SingleLine;
	slotSetDefaults();
}

BibleTimePrinter::~BibleTimePrinter()
{
}

void BibleTimePrinter::slotSetDefaults()
{
	// read config values from the KConfig file
	KConfig *config = kapp->getConfig();
	config->setGroup("Printing");	
	keyColor 	= config->readColorEntry("printVersenumberColor", &keyColor) ;
	dataColor = config->readColorEntry("printTextColor", &dataColor);
	//debug("%s, %s",(config->readEntry("printHeadingFont")).data(),(config->readEntry("printTextFont")).data());
 	keyFont		= QFont(config->readEntry("printHeadingFont"),config->readNumEntry("printHeadingFontSize", 12));
	dataFont	= QFont(config->readEntry("printTextFont"),config->readNumEntry("printTextFontSize", 12));
	printHeaderText = config->readBoolEntry("printHeadline", printHeaderText);
	printFooterText = config->readBoolEntry("printFooter", printFooterText);
	newLineSpace	= config->readNumEntry("newLineSpace",	newLineSpace);
	newParagraphSpace	= config->readNumEntry("newParagraphSpace",	newParagraphSpace);	
	
	QString pf = config->readEntry("printFlag","Align left");
	if (pf.contains("center",false))
		printFlag	= AlignHCenter | AlignTop | WordBreak | SingleLine;
	else
		printFlag	= AlignLeft | AlignTop | WordBreak | SingleLine;

	// set also FootText and HeadText when in Dialog
		
	// which fonts were set
	// debug("Fonts: %s\n%s",keyFont.family(),dataFont.family());
		
}

bool BibleTimePrinter::printQueue()
{
	if (!verseList)
		return false;
	
	AdvancedPrinter btPrinter;
	QPainter btPainter;

	btPrinter.setFromTo(0,0);
	// do as long as user choose Preview
	do
	{
		int ret = btPrinter.setup();
		if (ret != true)
		{
			emit sigPrintingAborted();
			return false;
		}
	
		initPrintValues(&btPrinter);
		btPainter.begin(&btPrinter);
		
		// now we can print our data
		btPrinter.setCreator("BibleTime " VERSION);
		btPrinter.setHeaderFreeString(i18n( "BibleTime, Version " VERSION ));
	
		// print according to number of copies
		int numCopies = btPrinter.numCopies();
		for (int i=1;i <= numCopies;i++)
		{
			verseList->first();
			totalPageNumber = 0;
			currentPageNumber = 1;
			printHeader(&btPrinter,&btPainter);
			while (verseList->current())
			{
				if (!printOneItem(&btPrinter,&btPainter,verseList->current()))
				{
					emit sigPrintingAborted();
					break;
				}
				verseList->next();
			}
			printFooter(&btPrinter,&btPainter);
			if (i<numCopies)
			{
				btPrinter.newPage();
				currentPageNumber = 1;
				cR.setLeft(leftBorder);
				cR.setTop(upperBorder);
				cR.setRight(pageWidth - leftBorder - rightBorder);
				cR.setBottom(pageHeight - upperBorder - lowerBorder);
			}
		}
		btPainter.end();
		if (btPrinter.getPreview())
			startPreviewer(&btPrinter);
			
	} while (btPrinter.getPreview());
	
	// reset versList pointer
	verseList = 0;
	emit sigPrintingDone();
	return true;
}

void BibleTimePrinter::slotSetPrintData(KeyVerseList *l)
{
	verseList = l;
}

void BibleTimePrinter::slotPrintKey(bool k)
{
	printKey = k;
}

void BibleTimePrinter::slotPrintData(bool d)
{
	printData = d;
}

void BibleTimePrinter::initPrintValues(AdvancedPrinter *btPrinter)
{
	pageWidth  = btPrinter->pageFullSize().width;
	pageHeight = btPrinter->pageFullSize().height;
	
	//debug("Height: %d, Width: %d", pageHeight, pageWidth);
	
	// overwrite borders with values from AdvancedPrinter
	SPageMargin margin = btPrinter->pageMarginMM();
	leftBorder  = margin.left;
	rightBorder = margin.right;
	upperBorder = margin.top;
	lowerBorder = margin.bottom;
	
	//debug("Borders are: Top=%d, Bottom=%d, Left=%d, Right=%d",upperBorder,lowerBorder,leftBorder,rightBorder);
	
	cR.setLeft(leftBorder);
	cR.setTop(upperBorder);
	cR.setRight(pageWidth - leftBorder - rightBorder);
	cR.setBottom(pageHeight - upperBorder - lowerBorder);
}

// we get one item here, do not separate key and data
// if item does not fit one page, pagebreak and try again
// items, that are larger then one page are not handled at the moment
bool BibleTimePrinter::printOneItem(AdvancedPrinter *btPrinter,QPainter *btPainter,KeyVerse *keyVerse)
{
	bool 			done;
	bool 			newPage 					= false;
	bool 			doesFit 					= true;
	int  			newPageThreshold 	= 300;
	unsigned 	lenPrinted				= 0;
	unsigned  lenNew						= 0;
	QString 	data 							= keyVerse->data;
	QRect			bBoxNew;
	
	do
	{
		done = true;
		QRect bBox =  btPainter->boundingRect(cR.x(),cR.y(),cR.width(),cR.height(),printFlag,data.data());
		if (keyVerse->printKey)
		{
			// add one line + newLinespace for the Key
			bBox.setHeight(bBox.height() + newLineSpace + keyFont.pointSize());
		}
		// check if bBox fits in cR
		if (bBox.height() > cR.height())
		{
			newPage = true;
			if (bBox.height() > cR.bottom())
			{
				// bBox is larger than a whole page
				done = false;
				if (cR.height() > newPageThreshold)
					newPage = false;
			}
		} else
			newPage = false;
		if (newPage)
		{
			printFooter(btPrinter, btPainter);
			btPrinter->newPage();
			currentPageNumber++;
			printHeader(btPrinter, btPainter);
			cR.setLeft(leftBorder);
			cR.setTop(upperBorder);
			cR.setRight(pageWidth - leftBorder - rightBorder);
			cR.setBottom(pageHeight - upperBorder - lowerBorder);
		}
		if (keyVerse->printKey)
		{
			btPainter->setFont(keyFont);
			btPainter->setPen(keyColor);
			cR.setY(cR.y() + keyFont.pointSize());
			btPainter->drawText(cR.x(),cR.y(),keyVerse->key.data());
			cR.setY(cR.y() + newLineSpace);
		}
		// now we have the bounding box for the data to fit in
		if (!done)
		{
			// take the first words from data until bBoxNew > cR
			lenPrinted = 0;
			doesFit = true;
			btPainter->setFont(dataFont);
			while (doesFit)
			{
				lenNew = data.find(QRegExp("\\s+"),lenPrinted + 1);
				//debug("%d, text is %s",lenPrinted,data.left(lenNew).data());
				bBoxNew = btPainter->boundingRect(cR.x(),cR.y(),cR.width(),cR.height(),printFlag,data.data(),lenNew);
				if (bBoxNew.height() > cR.height())
					doesFit = false;
				else
					lenPrinted = lenNew;
			}
		}
		if (keyVerse->printData)
		{
			btPainter->setFont(dataFont);
			btPainter->setPen(dataColor);
			if (!done)
				btPainter->drawText ( cR.x(), cR.y(), cR.width(), cR.height(), printFlag, data.data(),lenPrinted, &bBox);
			else
				btPainter->drawText ( cR.x(), cR.y(), cR.width(), cR.height(), printFlag, data.data(),qstrlen(data.data()), &bBox);
			// bBox is now set to what we actually used.
			// add bBox.height() to cR.y();
			cR.setY( cR.y() + bBox.height() + newParagraphSpace );
		}
		// disable key if another page is needed
		if (!done)
		{
			keyVerse->printKey = false;
			// remove the first lenPrinted chars from data
			data.remove(0,lenPrinted);
		}
	} while (!done);
	return true;
}

void BibleTimePrinter::printHeader(AdvancedPrinter *btPrinter, QPainter *btPainter)
{
	btPrinter->printHeader(btPainter);
}

void BibleTimePrinter::printFooter(AdvancedPrinter *btPrinter, QPainter *btPainter)
{
	btPrinter->printFooter(btPainter);
	btPainter->drawText(0,0," ");
	btPainter->drawText(pageWidth,pageHeight," ");
}

/** start the application that handles previews */
void BibleTimePrinter::startPreviewer(AdvancedPrinter *btPrinter){
	QString file = btPrinter->outputFileName();
	if (file.isEmpty()) {
		QMessageBox::information(0,i18n("Preview File"),
									i18n("You must specify a file!"),
									QMessageBox::Ok | QMessageBox::Default);
  	return;	
  }

  bool success = false;
	QString command = btPrinter->getPreviewApplication() + (QString)" " + file;
 	// now spawn process
  KShellProcess	process;
 	process.clearArguments();
  //debug("execute: %s",command.data());
 	process << command.data();
	success = process.start(KProcess::DontCare, KProcess::AllOutput);
 	if (!success)
	{
  	QMessageBox::information(0,i18n("External Process"),
    	i18n("The Preview Application could not be started!"),
 	  	QMessageBox::Ok | QMessageBox::Default);
 	  return;
 	}
	return;
}

