/* Miscellaneous index routines for use with GNATS. 
   Copyright (C) 1993, 1995 Free Software Foundation, Inc.
   Contributed by Brendan Kehoe (brendan@cygnus.com) and
   by Tim Wicinski (wicinski@barn.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"
#include "gnatsd.h"

/* Where to keep the static index if necessary.  */
Index *index_chain;

/* The last time the index was modified.  */
time_t index_mtime;

/* The filename of the index.  */
char *index_filename = (char *)NULL;

void
create_index_entry (buf)
     char *buf;
{
  char *resp, *r;
  char *tim;
  time_t arrival_date, last_modified, closed_date;
#ifdef GNATS_RELEASE_BASED
  time_t date_required;
#endif

  resp = field_value (RESPONSIBLE);
  if (! *resp)
    return;

#ifdef GNATS_RELEASE_BASED
  tim = field_value (DATE_REQUIRED);
  if (tim && *tim)
    {
      date_required = get_date (tim, NULL);
      if (! date_required)
	return;
    }
  else
    date_required = 0;
#endif

  tim = field_value (ARRIVAL_DATE);
  if (tim && *tim)
    {
      arrival_date = get_date (tim, NULL);
      if (! arrival_date)
	return;
    }
  else
    arrival_date = 0;
  
  tim = field_value (LAST_MODIFIED);
  if (tim && *tim)
    {
      last_modified = get_date (tim, NULL);
      if (! last_modified)
	return;
    }
  else
    last_modified = 0;
  
  tim = field_value (CLOSED_DATE);
  if (tim && *tim)
    {
      closed_date = get_date (tim, NULL);
      if (! closed_date)
	return;
    }
  else
    closed_date = 0;
  
  resp = (char *) strdup (resp);

  r = (char *) strchr (resp, ' ');
  if (r != NULL)
    *r = '\0';

#ifdef GNATS_RELEASE_BASED  	
  sprintf (buf, "%s/%s|%s|%s|%s|%s|%s|%s|%d|%s|%s|%d|%d|%d|%s|%s|%s|%s\n",
           field_value (CATEGORY), field_value (NUMBER), field_value (SUBMITTER),
	   resp, field_value (STATE), field_value (CONFIDENTIAL), 
	   field_value (SEVERITY), field_value (PRIORITY),
           date_required, field_value (QUARTER), field_value (KEYWORDS),
           arrival_date, last_modified, closed_date,
           field_value (CLASS), field_value (ORIGINATOR),
           field_value (RELEASE), field_value (SYNOPSIS));
#else
  sprintf (buf, "%s/%s|%s|%s|%s|%s|%s|%s|%d|%d|%d|%s|%s|%s|%s\n",
           field_value (CATEGORY), field_value (NUMBER), field_value (SUBMITTER),
	   resp, field_value (STATE), field_value (CONFIDENTIAL), 
	   field_value (SEVERITY), field_value (PRIORITY),
           arrival_date, last_modified, closed_date,
           field_value (CLASS), field_value (ORIGINATOR),
           field_value (RELEASE), field_value (SYNOPSIS));
#endif

  xfree (resp);
}


/* Return the next index entry from FP.  */
Index *
next_index_entry (fp)
     FILE *fp;
{
  char *start, *end, *b;
  char *buf = (char *) xmalloc (STR_MAXLONG);
  Index *p = (Index *) xmalloc (sizeof (Index));

  b = p->buf = buf;
  do
    if (fgets (buf, STR_MAXLONG, fp) == NULL)
      goto no_entry;
  while (buf[0] == '#');

  start = b;
  end = (char *) strchr (start, '/');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->category = start;

  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->number = start;

  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->submitter = start;

  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->responsible = start;

  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->state = start;

  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->confidential = start;

  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->severity = start;

  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->priority = start;

#ifdef GNATS_RELEASE_BASED
  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->date_required = start;
  
  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->quarter = start;
  
  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->keywords = start;
#endif

  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->arrival_date = start;

  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->last_modified = start;

  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->closed_date = start;

  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->class = start;

  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->originator = start;

  start = end + 1;
  end = (char *) strchr (start, '|');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->release = start;

  start = end + 1;
  end = (char *) strchr (start, '\n');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->synopsis = start;

  p->next = NULL;
  return p;

no_entry:
  xfree ((char *) p);
  xfree (buf);
  return (Index *) NULL;
}

/* Find problem report NUMBER in the index file, returning its category.  */
char *
find_pr_category (fp, number)
     FILE *fp;
     char *number;
{
  char *start, *end, *b;
  static char buf[STR_MAXLONG];
  static char *category;

  b = &buf[0];
  category = (char *)NULL;

  while (1)
    {
      do
	if (fgets (&buf[0], STR_MAXLONG, fp) == NULL)
	  return NULL;
      while (buf[0] == '#');

      start = b;
      end = (char *) strchr (start, '/');
      if (end == NULL)
	return NULL;
      *end = '\0';
      category = start;

      start = end + 1;
      end = (char *) strchr (start, '|');
      if (end == NULL)
	return NULL;
      *end = '\0';

      if (strcmp (start, number) == 0)
	return category;
    }

  return NULL;
}

/* get_index - reads in the entire index. useful for searching or
   modifying index records.  */
Index*
get_index ()
{
  FILE *fp = open_index ();
  Index *i;
  Index *current_chain = NULL;
  Index *index_chain_end = NULL;

  if (fp == (FILE *) NULL)
    return (Index *) NULL;

  while ((i = next_index_entry (fp)))
    {
      if (index_chain_end == NULL)
        current_chain = index_chain_end = i;
      else
        {
          index_chain_end->next = i;
          index_chain_end = i;
        }

      i->next = NULL;
    }

  close_index (fp);
  return current_chain;
}

time_t
stat_index (name)
     char *name;
{
  struct stat buf;
  char *path;
  int i;

  if (! name)
    path = index_filename;
  else
    path = name;

  i = stat (path, &buf);
  if (i < 0)
    {
      if (! is_daemon)
	fprintf (stderr, "%s: can't stat the index file %s\n",
		 program_name, path);
      return (time_t)-1;
    }

  return buf.st_mtime;
}

int
check_index ()
{
  struct stat buf;
  int i;

  i = stat (index_filename, &buf);
  /* We should probably send something here.  */
  if (i < 0)
    return 0;

  if (buf.st_mtime > index_mtime)
    {
      index_chain = get_index ();
      if (index_chain == NULL)
	{
	  printf ("%d GNATS server cannot read the index.\r\n", CODE_FILE_ERROR);
	  exit (1);
	}
      return 1;
    }

  return 0;
}

FILE *
open_index ()
{
  FILE *fp;
  time_t t;

  if (! index_filename)
    {
      index_filename = (char *) xmalloc (PATH_MAX);
      sprintf (index_filename, "%s/gnats-adm/%s", gnats_root, INDEX);
    }

  fp = fopen (index_filename, "r");
  if (fp == NULL)
    {
/*       if (! is_daemon) */
/* 	fprintf (stderr, "%s: can't open the index file %s\n", */
/* 		 program_name, index_filename); */
      xfree (index_filename);
      index_filename = (char *)NULL;
      return (FILE *) NULL;
    }

  t = stat_index (index_filename);
  if (t != (time_t)-1)
    index_mtime = t;
  return fp;
}

void
close_index (fp)
     FILE *fp;
{
  fclose (fp);
}

void
free_index (i)
     Index *i;
{
  Index *j;

  for ( ; i; i = j)
    {
      j = i->next;
      free_index_entry (i);
    }
}

void
free_index_entry (i)
     Index *i;
{
  xfree (i->buf);
  xfree ((char *)i);
}

/* Don't call this directly; call lookup_pr_path instead */
char *
get_category (p)
     char *p;
{
  FILE *fp;
  char *path, *category = NULL;

  if (index_chain)
    {
      Index *j;
      for (j = index_chain ; j ; j = j->next)
	if (strcmp (p, j->number) == 0)
	  {
	    category = j->category;
	    break;
	  }
    }
  else
    {
      fp = open_index ();
      if (fp == (FILE *)NULL)
	return NULL;
      category = find_pr_category (fp, p);
      close_index (fp);
    }

  if (category == NULL)
    return NULL;

  path = (char *) xmalloc (PATH_MAX);
  sprintf (path, "%s/%s/%s", gnats_root, category, p);

  return path;
}

char *
lookup_pr_path (p)
     char *p;
{
  char *path;
  FILE *fp;

  /* If they gave it to us with the category, remove it. */
  if (( strrchr (p, '/')) != NULL)
    p = strrchr (p, '/') + 1;
  path = get_category (p);
  if ((path == NULL) || ((fp = fopen (path, "r")) == (FILE *) NULL))
    {
      if (is_daemon)
        printf ("%d Invalid PR %s.\r\n", CODE_INVALID_PR,
                get_cat_prid_from_path ((path == NULL) ? p : path));
      else 
        fprintf (stderr, "%s: Invalid PR %s.\n", program_name,
                 get_cat_prid_from_path ((path == NULL) ? p : path));
      return NULL;
    }

  fclose (fp);
  return path;
}
