(*
Module: Known_Hosts
  Parses SSH known_hosts files

Author: Raphaël Pinson <raphink@gmail.com>

About: Reference
  This lens manages OpenSSH's known_hosts files. See `man 8 sshd` for reference.

About: License
  This file is licenced under the LGPL v2+, like the rest of Augeas.

About: Lens Usage
  Sample usage of this lens in augtool:

    * Get a key by name from ssh_known_hosts
      > print /files/etc/ssh_known_hosts/*[.="foo.example.com"]
      ...

    * Change a host's key
      > set /files/etc/ssh_known_hosts/*[.="foo.example.com"]/key "newkey"

About: Configuration files
  This lens applies to SSH known_hosts files. See <filter>.

*)

module Known_Hosts =

autoload xfm

(* View: marker
  The marker is optional, but if it is present then it must be one of
  “@cert-authority”, to indicate that the line contains a certification
  authority (CA) key, or “@revoked”, to indicate that the key contained
  on the line is revoked and must not ever be accepted.
  Only one marker should be used on a key line.
*)
let marker = [ key /@(revoked|cert-authority)/ . Sep.space ]

(* View: hostnames
  Hostnames is a comma-separated list of patterns
  (‘*’ and ‘?’ act as wildcards); each pattern in turn is matched
  against the canonical host name (when authenticating a client)
  or against the user-supplied name (when authenticating a server).
  A pattern may also be preceded by ‘!’ to indicate negation:
  if the host name matches a negated pattern, it is not accepted
  (by that line) even if it matched another pattern on the line.
  A hostname or address may optionally be enclosed within ‘[’ and ‘]’
  brackets then followed by ‘:’ and a non-standard port number.

  Alternately, hostnames may be stored in a hashed form which hides
  host names and addresses should the file's contents be disclosed.
  Hashed hostnames start with a ‘|’ character. Only one hashed hostname
  may appear on a single line and none of the above negation or wildcard
  operators may be applied.
*)
let hostnames =
     let pattern = [ label "pattern" . store Rx.neg1 ]
  in Build.opt_list pattern Sep.comma

(* View: type
  Bits, exponent, and modulus are taken directly from the RSA host key;
  they can be obtained, for example, from /etc/ssh/ssh_host_key.pub.
  The optional comment field continues to the end of the line, and is not used.
*)
let type = [ label "type" . store Rx.neg1 ]

(* View: entry
  Each line in these files contains the following fields:
  markers (optional), hostnames, bits, exponent, modulus, comment.
  The fields are separated by spaces.
 *)
let entry =
     let key = [ label "key" . store Rx.neg1 ]
  in [ Util.indent . seq "entry" . marker?
     . hostnames . Sep.space . type . Sep.space . key
     . Util.comment_or_eol ]

(* View: lns
     The known_hosts lens *)
let lns = (Util.empty | Util.comment | entry)*

(* Variable: filter *)
let filter = incl "/etc/ssh/ssh_known_hosts"
           . incl (Sys.getenv("HOME") . "/.ssh/known_hosts")

let xfm = transform lns filter
