/***************************************************************************
    qgswidgetwrapper.h
     --------------------------------------
    Date                 : 14.5.2014
    Copyright            : (C) 2013 Matthias Kuhn
    Email                : matthias dot kuhn at gmx dot ch
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef QGSWIDGETWRAPPER_H
#define QGSWIDGETWRAPPER_H

#include <QObject>
#include <QMap>
#include <QVariant>

class QgsVectorLayer;

#include "qgseditorwidgetconfig.h"
#include "qgsattributeeditorcontext.h"

/**
 * Manages an editor widget
 * Widget and wrapper share the same parent
 *
 * A wrapper controls one attribute editor widget and is able to create a default
 * widget or use a pre-existent widget. It is able to set the widget to the value implied
 * by a field of a vector layer, or return the value it currently holds. Every time it is changed
 * it has to emit a valueChanged signal. If it fails to do so, there is no guarantee that the
 * changed status of the widget will be saved.
 *
 */
class GUI_EXPORT QgsWidgetWrapper : public QObject
{
    Q_OBJECT
  public:
    /**
     * Create a new widget wrapper
     *
     * @param vl        The layer on which the field is
     * @param editor    An editor widget. Can be NULL if one should be autogenerated.
     * @param parent    A parent widget for this widget wrapper and the created widget.
     */
    explicit QgsWidgetWrapper( QgsVectorLayer* vl, QWidget* editor = 0, QWidget* parent = 0 );

    /**
     * @brief Access the widget managed by this wrapper
     *
     * @return The widget
     */
    QWidget* widget();

    /**
     * @brief Access the widget managed by this wrapper and cast it to a given type
     * Example: QPushButton* pb = wrapper->widget<QPushButton*>();
     *
     * @return The widget as template type or NULL, if it cannot be cast to this type.
     */
    template <class T>
    T* widget() { return dynamic_cast<T>( mWidget ); }

    /**
     * Will set the config of this wrapper to the specified config.
     *
     * @param config The config for this wrapper
     */
    void setConfig( const QgsEditorWidgetConfig& config );

    /**
     * Set the context in which this widget is shown
     *
     * @param context context information
     */
    void setContext( const QgsAttributeEditorContext context );

    /**
     * Use this inside your overriden classes to access the configuration.
     *
     * @param key         The configuration option you want to load
     * @param defaultVal  Default value
     *
     * @return the value assigned to this configuration option
     */
    QVariant config( QString key, QVariant defaultVal = QVariant() );

    /**
     * Returns the whole config
     *
     * @return The configuration
     */
    const QgsEditorWidgetConfig config();

    /**
     * Returns information about the context in which this widget is shown
     *
     * @return context information
     */
    const QgsAttributeEditorContext& context();

    /**
     * Access the QgsVectorLayer, you are working on
     *
     * @return The layer
     *
     * @see field()
     */
    QgsVectorLayer* layer();

    /**
     * Will return a wrapper for a given widget
     * @param widget The widget which was created by a wrapper
     * @return The wrapper for the widget or NULL
     */
    static QgsWidgetWrapper* fromWidget( QWidget* widget );

  protected:
    /**
     * This method should create a new widget with the provided parent. This will only be called
     * if the form did not already provide a widget, so it is not guaranteed to be called!
     * You should not do initialisation stuff, which also has to be done for custom editor
     * widgets inside this method. Things like filling comboboxes and assigning other data which
     * will also be used to make widgets on forms created in the QtDesigner usable should be assigned
     * in {@link initWidget(QWidget*)}.
     *
     * @param parent You should set this parent on the created widget.
     * @return A new widget
     */
    virtual QWidget* createWidget( QWidget* parent ) = 0;

    /**
     * This method should initialize the editor widget with runtime data. Fill your comboboxes here.
     *
     * @param editor The widget which will represent this attribute editor in a form.
     */
    virtual void initWidget( QWidget* editor );

  public slots:
    /**
     * Is called, when the value of the widget needs to be changed. Update the widget representation
     * to reflect the new value.
     *
     * @param feature The new feature
     */
    virtual void setFeature( const QgsFeature& feature ) = 0;

    /**
     * Is used to enable or disable the edit functionality of the managed widget.
     * By default this will not change the enabled state of the widget
     *
     * @param enabled  Enable or Disable?
     */
    virtual void setEnabled( bool enabled );

  private:
    QgsAttributeEditorContext mContext;
    QgsEditorWidgetConfig mConfig;
    QWidget* mWidget;
    QWidget* mParent;
    QgsVectorLayer* mLayer;
    bool mInitialized;
};

// We'll use this class inside a QVariant in the widgets properties
Q_DECLARE_METATYPE( QgsWidgetWrapper* )

#endif // QGSWIDGETWRAPPER_H
