/*
 * Decompiled with CFR 0.152.
 */
package net.sf.freecol.server.generator;

import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.logging.Logger;
import net.sf.freecol.common.model.Game;
import net.sf.freecol.common.model.LandMap;
import net.sf.freecol.common.model.Map;
import net.sf.freecol.common.model.Region;
import net.sf.freecol.common.model.Resource;
import net.sf.freecol.common.model.ResourceType;
import net.sf.freecol.common.model.Specification;
import net.sf.freecol.common.model.Tile;
import net.sf.freecol.common.model.TileImprovement;
import net.sf.freecol.common.model.TileImprovementType;
import net.sf.freecol.common.model.TileItemContainer;
import net.sf.freecol.common.model.TileType;
import net.sf.freecol.common.option.OptionGroup;
import net.sf.freecol.common.util.RandomChoice;
import net.sf.freecol.common.util.RandomUtils;
import net.sf.freecol.server.generator.River;
import net.sf.freecol.server.generator.RiverSection;
import net.sf.freecol.server.model.ServerRegion;

public class TerrainGenerator {
    private static final Logger logger = Logger.getLogger(TerrainGenerator.class.getName());
    public static final int LAND_REGIONS_SCORE_VALUE = 1000;
    public static final int LAND_REGION_MIN_SCORE = 5;
    public static final int PACIFIC_SCORE_VALUE = 100;
    public static final int LAND_REGION_MAX_SIZE = 75;
    private final Game game;
    private final Game importGame;
    private final Random random;
    private final OptionGroup mapOptions;
    private final Specification spec;
    private ArrayList<TileType> landTileTypes = null;
    private ArrayList<TileType> oceanTileTypes = null;
    private ServerRegion[] geographicRegions = null;

    public TerrainGenerator(Game game, Game importGame, Random random) {
        this.game = game;
        this.importGame = importGame;
        this.random = random;
        this.mapOptions = game.getMapGeneratorOptions();
        this.spec = game.getSpecification();
    }

    private int limitToRange(int value, int lower, int upper) {
        return Math.max(lower, Math.min(value, upper));
    }

    private int getApproximateLandCount() {
        return this.mapOptions.getInteger("model.option.mapWidth") * this.mapOptions.getInteger("model.option.mapHeight") * this.mapOptions.getInteger("model.option.landMass") / 100;
    }

    private void setGeographicRegion(ServerRegion sr) {
        if (this.geographicRegions == null) {
            return;
        }
        for (ServerRegion gr : this.geographicRegions) {
            if (!gr.containsCenter(sr)) continue;
            sr.setParent(gr);
            gr.addChild(sr);
            gr.setSize(gr.getSize() + sr.getSize());
            break;
        }
    }

    private Tile createTile(int x, int y, LandMap landMap, int latitude) {
        return landMap.isLand(x, y) ? new Tile(this.game, this.getRandomLandTileType(latitude), x, y) : new Tile(this.game, this.getRandomOceanTileType(latitude), x, y);
    }

    private TileType getRandomLandTileType(int latitude) {
        if (this.landTileTypes == null) {
            this.landTileTypes = new ArrayList();
            for (TileType type : this.spec.getTileTypeList()) {
                if (type.isElevation() || type.isWater()) continue;
                this.landTileTypes.add(type);
            }
        }
        return this.getRandomTileType(this.landTileTypes, latitude);
    }

    private TileType getRandomOceanTileType(int latitude) {
        if (this.oceanTileTypes == null) {
            this.oceanTileTypes = new ArrayList();
            for (TileType type : this.spec.getTileTypeList()) {
                if (!type.isWater() || !type.isHighSeasConnected() || type.isDirectlyHighSeasConnected()) continue;
                this.oceanTileTypes.add(type);
            }
        }
        return this.getRandomTileType(this.oceanTileTypes, latitude);
    }

    private TileType getRandomTileType(List<TileType> candidates, int latitude) {
        TileType type;
        int forestChance = this.mapOptions.getInteger("model.option.forestNumber");
        int temperaturePreference = this.mapOptions.getInteger("model.option.temperature");
        int poleTemperature = -20;
        int equatorTemperature = 40;
        switch (temperaturePreference) {
            case 0: {
                poleTemperature = -20;
                equatorTemperature = 25;
                break;
            }
            case 1: {
                poleTemperature = -20;
                equatorTemperature = 30;
                break;
            }
            case 2: {
                poleTemperature = -10;
                equatorTemperature = 35;
                break;
            }
            case 3: {
                poleTemperature = -5;
                equatorTemperature = 40;
                break;
            }
            case 4: {
                poleTemperature = 0;
                equatorTemperature = 40;
                break;
            }
        }
        Specification spec = this.game.getSpecification();
        int temperatureRange = equatorTemperature - poleTemperature;
        int localeTemperature = poleTemperature + (90 - Math.abs(latitude)) * temperatureRange / 90;
        int temperatureDeviation = 7;
        localeTemperature += RandomUtils.randomInt(logger, "Temperature", this.random, temperatureDeviation * 2) - temperatureDeviation;
        localeTemperature = this.limitToRange(localeTemperature, -20, 40);
        int localeHumidity = spec.getRangeOption("model.option.humidity").getValue();
        int humidityDeviation = 20;
        localeHumidity += RandomUtils.randomInt(logger, "Humidity", this.random, humidityDeviation * 2) - humidityDeviation;
        localeHumidity = this.limitToRange(localeHumidity, 0, 100);
        ArrayList<TileType> candidateTileTypes = new ArrayList<TileType>(candidates);
        int i = 0;
        while (i < candidateTileTypes.size()) {
            type = (TileType)candidateTileTypes.get(i);
            if (!type.withinRange(TileType.RangeType.TEMPERATURE, localeTemperature)) {
                candidateTileTypes.remove(i);
                continue;
            }
            ++i;
        }
        switch (candidateTileTypes.size()) {
            case 0: {
                throw new RuntimeException("No TileType for temperature==" + localeTemperature);
            }
            case 1: {
                return (TileType)candidateTileTypes.get(0);
            }
        }
        i = 0;
        while (i < candidateTileTypes.size()) {
            type = (TileType)candidateTileTypes.get(i);
            if (!type.withinRange(TileType.RangeType.HUMIDITY, localeHumidity)) {
                candidateTileTypes.remove(i);
                continue;
            }
            ++i;
        }
        switch (candidateTileTypes.size()) {
            case 0: {
                throw new RuntimeException("No TileType for humidity==" + localeHumidity);
            }
            case 1: {
                return (TileType)candidateTileTypes.get(0);
            }
        }
        boolean forested = RandomUtils.randomInt(logger, "Forest", this.random, 100) < forestChance;
        i = 0;
        while (i < candidateTileTypes.size()) {
            TileType type2 = (TileType)candidateTileTypes.get(i);
            if (type2.isForested() != forested) {
                candidateTileTypes.remove(i);
                continue;
            }
            ++i;
        }
        i = candidateTileTypes.size();
        switch (i) {
            case 0: {
                throw new RuntimeException("No TileType for forested==" + forested);
            }
            case 1: {
                return (TileType)candidateTileTypes.get(0);
            }
        }
        return (TileType)candidateTileTypes.get(RandomUtils.randomInt(logger, "Forest tile", this.random, i));
    }

    private void createOceanRegions(Map map) {
        Tile t;
        int y;
        ServerRegion pacific = (ServerRegion)map.getRegion("model.region.pacific");
        ServerRegion northPacific = (ServerRegion)map.getRegion("model.region.northPacific");
        ServerRegion southPacific = (ServerRegion)map.getRegion("model.region.southPacific");
        ServerRegion atlantic = (ServerRegion)map.getRegion("model.region.atlantic");
        ServerRegion northAtlantic = (ServerRegion)map.getRegion("model.region.northAtlantic");
        ServerRegion southAtlantic = (ServerRegion)map.getRegion("model.region.southAtlantic");
        int present = 0;
        if (pacific == null) {
            pacific = new ServerRegion(this.game, "model.region.pacific", Region.RegionType.OCEAN, null);
            pacific.setDiscoverable(true);
            map.putRegion(pacific);
            pacific.setScoreValue(100);
        }
        if (northPacific == null) {
            northPacific = new ServerRegion(this.game, "model.region.northPacific", Region.RegionType.OCEAN, pacific);
            northPacific.setDiscoverable(false);
            map.putRegion(northPacific);
        } else {
            ++present;
        }
        if (southPacific == null) {
            southPacific = new ServerRegion(this.game, "model.region.southPacific", Region.RegionType.OCEAN, pacific);
            southPacific.setDiscoverable(false);
            map.putRegion(southPacific);
        } else {
            ++present;
        }
        if (atlantic == null) {
            atlantic = new ServerRegion(this.game, "model.region.atlantic", Region.RegionType.OCEAN, null);
            atlantic.setPrediscovered(true);
            atlantic.setDiscoverable(false);
            map.putRegion(atlantic);
        }
        if (northAtlantic == null) {
            northAtlantic = new ServerRegion(this.game, "model.region.northAtlantic", Region.RegionType.OCEAN, atlantic);
            northAtlantic.setPrediscovered(true);
            northAtlantic.setDiscoverable(false);
            map.putRegion(northAtlantic);
        } else {
            ++present;
        }
        if (southAtlantic == null) {
            southAtlantic = new ServerRegion(this.game, "model.region.southAtlantic", Region.RegionType.OCEAN, atlantic);
            southAtlantic.setPrediscovered(true);
            southAtlantic.setDiscoverable(false);
            map.putRegion(southAtlantic);
        } else {
            ++present;
        }
        if (present == 4) {
            return;
        }
        int maxx = map.getWidth();
        int midx = maxx / 2;
        int maxy = map.getHeight();
        int midy = maxy / 2;
        Tile tNP = null;
        Tile tSP = null;
        Tile tNA = null;
        Tile tSA = null;
        for (y = midy - 1; y >= 0; --y) {
            if (tNP == null && !(t = map.getTile(0, y)).isLand()) {
                tNP = t;
            }
            if (tNA == null && !(t = map.getTile(maxx - 1, y)).isLand()) {
                tNA = t;
            }
            if (tNP != null && tNA != null) break;
        }
        for (y = midy; y < maxy; ++y) {
            if (tSP == null && !(t = map.getTile(0, y)).isLand()) {
                tSP = t;
            }
            if (tSA == null && !(t = map.getTile(maxx - 1, y)).isLand()) {
                tSA = t;
            }
            if (tSP != null && tSA != null) break;
        }
        int nNP = 0;
        int nSP = 0;
        int nNA = 0;
        int nSA = 0;
        Rectangle rNP = new Rectangle(0, 0, midx, midy);
        Rectangle rSP = new Rectangle(0, midy, midx, maxy - midy);
        Rectangle rNA = new Rectangle(midx, 0, maxx - midx, midy);
        Rectangle rSA = new Rectangle(midx, midy, maxx - midx, maxy - midy);
        if (tNP != null) {
            nNP += this.fillOcean(map, tNP, northPacific, rNP);
        }
        if (tSP != null) {
            nSP += this.fillOcean(map, tSP, southPacific, rSP);
        }
        if (tNA != null) {
            nNA += this.fillOcean(map, tNA, northAtlantic, rNA);
        }
        if (tSA != null) {
            nSA += this.fillOcean(map, tSA, southAtlantic, rSA);
        }
        Rectangle rN = new Rectangle(0, 0, maxx, midy);
        Rectangle rS = new Rectangle(0, midy, maxx, maxy - midy);
        if (tNP != null) {
            nNP += this.fillOcean(map, tNP, northPacific, rN);
        }
        if (tSP != null) {
            nSP += this.fillOcean(map, tSP, southPacific, rS);
        }
        if (tNA != null) {
            nNA += this.fillOcean(map, tNA, northAtlantic, rN);
        }
        if (tSA != null) {
            nSA += this.fillOcean(map, tSA, southAtlantic, rS);
        }
        Rectangle rAll = new Rectangle(0, 0, maxx, maxy);
        if (tNP != null) {
            nNP += this.fillOcean(map, tNP, northPacific, rAll);
        }
        if (tSP != null) {
            nSP += this.fillOcean(map, tSP, southPacific, rAll);
        }
        if (tNA != null) {
            nNA += this.fillOcean(map, tNA, northAtlantic, rAll);
        }
        if (tSA != null) {
            nSA += this.fillOcean(map, tSA, southAtlantic, rAll);
        }
        if (nNP <= 0) {
            logger.warning("No North Pacific tiles found");
        }
        if (nSP <= 0) {
            logger.warning("No South Pacific tiles found");
        }
        if (nNA <= 0) {
            logger.warning("No North Atlantic tiles found");
        }
        if (nSA <= 0) {
            logger.warning("No South Atlantic tiles found");
        }
        logger.info("Ocean regions complete: " + nNP + " North Pacific, " + nSP + " South Pacific, " + nNA + " North Atlantic, " + nSP + " South Atlantic");
    }

    private int fillOcean(Map map, Tile tile, ServerRegion region, Rectangle bounds) {
        LinkedList<Tile> q = new LinkedList<Tile>();
        int n = 0;
        boolean[][] visited = new boolean[map.getWidth()][map.getHeight()];
        visited[tile.getX()][tile.getY()] = true;
        q.add(tile);
        while ((tile = (Tile)q.poll()) != null) {
            region.addTile(tile);
            ++n;
            for (Map.Direction direction : Map.Direction.values()) {
                Tile t = map.getAdjacentTile(tile, direction);
                if (t == null || visited[t.getX()][t.getY()] || !bounds.contains(t.getX(), t.getY())) continue;
                visited[t.getX()][t.getY()] = true;
                if (t.getRegion() != null && t.getRegion() != region || t.isLand()) continue;
                q.add(t);
            }
        }
        return n;
    }

    private ServerRegion[] getStandardRegions(Map map) {
        ServerRegion antarctic;
        int arcticHeight = 2;
        int antarcticHeight = map.getHeight() - 2 - 1;
        ServerRegion arctic = (ServerRegion)map.getRegion("model.region.arctic");
        if (arctic == null) {
            arctic = new ServerRegion(this.game, "model.region.arctic", Region.RegionType.LAND, null);
            arctic.setPrediscovered(true);
            map.putRegion(arctic);
            for (int x = 0; x < map.getWidth(); ++x) {
                for (int y = 0; y < 2; ++y) {
                    Tile tile;
                    if (!map.isValid(x, y) || !(tile = map.getTile(x, y)).isLand()) continue;
                    arctic.addTile(tile);
                }
            }
        }
        if ((antarctic = (ServerRegion)map.getRegion("model.region.antarctic")) == null) {
            antarctic = new ServerRegion(this.game, "model.region.antarctic", Region.RegionType.LAND, null);
            antarctic.setPrediscovered(true);
            map.putRegion(antarctic);
            for (int x = 0; x < map.getWidth(); ++x) {
                for (int y = antarcticHeight; y < map.getHeight(); ++y) {
                    Tile tile;
                    if (!map.isValid(x, y) || !(tile = map.getTile(x, y)).isLand()) continue;
                    antarctic.addTile(tile);
                }
            }
        }
        int thirdWidth = map.getWidth() / 3;
        int twoThirdWidth = 2 * thirdWidth;
        int thirdHeight = map.getHeight() / 3;
        int twoThirdHeight = 2 * thirdHeight;
        ServerRegion northWest = (ServerRegion)map.getRegion("model.region.northWest");
        if (northWest == null) {
            northWest = new ServerRegion(this.game, "model.region.northWest", Region.RegionType.LAND, null);
            map.putRegion(northWest);
        }
        northWest.setBounds(new Rectangle(0, 0, thirdWidth, thirdHeight));
        northWest.setPrediscovered(true);
        ServerRegion north = (ServerRegion)map.getRegion("model.region.north");
        if (north == null) {
            north = new ServerRegion(this.game, "model.region.north", Region.RegionType.LAND, null);
            map.putRegion(north);
        }
        north.setBounds(new Rectangle(thirdWidth, 0, thirdWidth, thirdHeight));
        north.setPrediscovered(true);
        ServerRegion northEast = (ServerRegion)map.getRegion("model.region.northEast");
        if (northEast == null) {
            northEast = new ServerRegion(this.game, "model.region.northEast", Region.RegionType.LAND, null);
            map.putRegion(northEast);
        }
        northEast.setBounds(new Rectangle(twoThirdWidth, 0, thirdWidth, thirdHeight));
        northEast.setPrediscovered(true);
        ServerRegion west = (ServerRegion)map.getRegion("model.region.west");
        if (west == null) {
            west = new ServerRegion(this.game, "model.region.west", Region.RegionType.LAND, null);
            map.putRegion(west);
        }
        west.setBounds(new Rectangle(0, thirdHeight, thirdWidth, thirdHeight));
        west.setPrediscovered(true);
        ServerRegion center = (ServerRegion)map.getRegion("model.region.center");
        if (center == null) {
            center = new ServerRegion(this.game, "model.region.center", Region.RegionType.LAND, null);
            map.putRegion(center);
        }
        center.setBounds(new Rectangle(thirdWidth, thirdHeight, thirdWidth, thirdHeight));
        center.setPrediscovered(true);
        ServerRegion east = (ServerRegion)map.getRegion("model.region.east");
        if (east == null) {
            east = new ServerRegion(this.game, "model.region.east", Region.RegionType.LAND, null);
            map.putRegion(east);
        }
        east.setBounds(new Rectangle(twoThirdWidth, thirdHeight, thirdWidth, thirdHeight));
        east.setPrediscovered(true);
        ServerRegion southWest = (ServerRegion)map.getRegion("model.region.southWest");
        if (southWest == null) {
            southWest = new ServerRegion(this.game, "model.region.southWest", Region.RegionType.LAND, null);
            map.putRegion(southWest);
        }
        southWest.setBounds(new Rectangle(0, twoThirdHeight, thirdWidth, thirdHeight));
        southWest.setPrediscovered(true);
        ServerRegion south = (ServerRegion)map.getRegion("model.region.south");
        if (south == null) {
            south = new ServerRegion(this.game, "model.region.south", Region.RegionType.LAND, null);
            map.putRegion(south);
        }
        south.setBounds(new Rectangle(thirdWidth, twoThirdHeight, thirdWidth, thirdHeight));
        south.setPrediscovered(true);
        ServerRegion southEast = (ServerRegion)map.getRegion("model.region.southEast");
        if (southEast == null) {
            southEast = new ServerRegion(this.game, "model.region.southEast", Region.RegionType.LAND, null);
            map.putRegion(southEast);
        }
        southEast.setBounds(new Rectangle(twoThirdWidth, twoThirdHeight, thirdWidth, thirdHeight));
        southEast.setPrediscovered(true);
        return new ServerRegion[]{northWest, north, northEast, west, center, east, southWest, south, southEast};
    }

    private void createLandRegions(Map map) {
        int c;
        int y;
        int continents = 0;
        boolean[][] landmap = new boolean[map.getWidth()][map.getHeight()];
        int[][] continentmap = new int[map.getWidth()][map.getHeight()];
        int landsize = 0;
        for (int x = 0; x < map.getWidth(); ++x) {
            for (y = 0; y < map.getHeight(); ++y) {
                continentmap[x][y] = 0;
                landmap[x][y] = false;
                if (!map.isValid(x, y)) continue;
                Tile tile = map.getTile(x, y);
                boolean bl = landmap[x][y] = tile.isLand() && tile.getRegion() == null;
                if (!tile.isLand()) continue;
                ++landsize;
            }
        }
        for (int y2 = 0; y2 < map.getHeight(); ++y2) {
            for (int x = 0; x < map.getWidth(); ++x) {
                if (!landmap[x][y2]) continue;
                ++continents;
                boolean[][] continent = Map.floodFill(landmap, x, y2);
                for (int yy = 0; yy < map.getHeight(); ++yy) {
                    for (int xx = 0; xx < map.getWidth(); ++xx) {
                        if (!continent[xx][yy]) continue;
                        continentmap[xx][yy] = continents;
                        landmap[xx][yy] = false;
                    }
                }
            }
        }
        logger.info("Number of individual landmasses is " + continents);
        int[] continentsize = new int[continents + 1];
        for (y = 0; y < map.getHeight(); ++y) {
            for (int x = 0; x < map.getWidth(); ++x) {
                int n = continentmap[x][y];
                continentsize[n] = continentsize[n] + 1;
            }
        }
        int oldcontinents = continents;
        for (int c2 = 1; c2 <= oldcontinents; ++c2) {
            if (continentsize[c2] <= 75) continue;
            boolean[][] splitcontinent = new boolean[map.getWidth()][map.getHeight()];
            int splitX = 0;
            int splitY = 0;
            for (int x = 0; x < map.getWidth(); ++x) {
                for (int y3 = 0; y3 < map.getHeight(); ++y3) {
                    if (continentmap[x][y3] == c2) {
                        splitcontinent[x][y3] = true;
                        splitX = x;
                        splitY = y3;
                        continue;
                    }
                    splitcontinent[x][y3] = false;
                }
            }
            while (continentsize[c2] > 75) {
                int targetsize = 75;
                if (continentsize[c2] < 150) {
                    targetsize = continentsize[c2] / 2;
                }
                ++continents;
                boolean[][] newregion = Map.floodFill(splitcontinent, splitX, splitY, targetsize);
                for (int x = 0; x < map.getWidth(); ++x) {
                    for (int y4 = 0; y4 < map.getHeight(); ++y4) {
                        if (newregion[x][y4]) {
                            continentmap[x][y4] = continents;
                            splitcontinent[x][y4] = false;
                            int n = c2;
                            continentsize[n] = continentsize[n] - 1;
                        }
                        if (!splitcontinent[x][y4]) continue;
                        splitX = x;
                        splitY = y4;
                    }
                }
            }
        }
        logger.info("Number of land regions being created: " + continents);
        ServerRegion[] landregions = new ServerRegion[continents + 1];
        int landIndex = 1;
        for (c = 1; c <= continents; ++c) {
            String id;
            while (map.getRegion(id = "model.region.land" + Integer.toString(landIndex++)) != null) {
            }
            landregions[c] = new ServerRegion(this.game, id, Region.RegionType.LAND, null);
            landregions[c].setDiscoverable(true);
            map.putRegion(landregions[c]);
        }
        for (int y5 = 0; y5 < map.getHeight(); ++y5) {
            for (int x = 0; x < map.getWidth(); ++x) {
                if (continentmap[x][y5] <= 0) continue;
                Tile tile = map.getTile(x, y5);
                landregions[continentmap[x][y5]].addTile(tile);
            }
        }
        for (c = 1; c <= continents; ++c) {
            ServerRegion sr = landregions[c];
            int score = Math.max((int)((float)sr.getSize() / (float)landsize * 1000.0f), 5);
            sr.setScoreValue(score);
            this.setGeographicRegion(sr);
            logger.fine("Created land region " + sr.getNameKey() + " (size " + sr.getSize() + ", score " + sr.getScoreValue() + ", parent " + (sr.getParent() == null ? "(null)" : sr.getParent().getNameKey()) + ")");
        }
        for (ServerRegion gr : this.geographicRegions) {
            logger.fine("Geographic region " + gr.getNameKey() + " (size " + gr.getSize() + ", children " + gr.getChildren().size() + ")");
        }
    }

    private void createMountains(Map map) {
        int tries;
        float randomHillsRatio = 0.5f;
        int maximumLength = Math.max(this.mapOptions.getInteger("model.option.mapWidth"), this.mapOptions.getInteger("model.option.mapHeight")) / 10;
        int number = (int)((1.0f - randomHillsRatio) * (float)(this.getApproximateLandCount() / this.mapOptions.getInteger("model.option.mountainNumber")));
        logger.info("Number of mountain tiles is " + number);
        logger.fine("Maximum length of mountain ranges is " + maximumLength);
        TileType hills = this.spec.getTileType("model.tile.hills");
        TileType mountains = this.spec.getTileType("model.tile.mountains");
        if (hills == null || mountains == null) {
            throw new RuntimeException("Both Hills and Mountains TileTypes must be defined");
        }
        int counter = 0;
        block0: for (tries = 0; tries < 100; ++tries) {
            if (counter >= number) continue;
            Tile startTile = map.getRandomLandTile(this.random);
            if (startTile == null) {
                return;
            }
            if (startTile.getType() == hills || startTile.getType() == mountains) continue;
            for (Tile t : startTile.getSurroundingTiles(3)) {
                if (t.getType() != mountains) continue;
                continue block0;
            }
            for (Tile t : startTile.getSurroundingTiles(2)) {
                if (t.isLand()) continue;
                continue block0;
            }
            ServerRegion mountainRegion = new ServerRegion(this.game, "model.region.mountain" + tries, Region.RegionType.MOUNTAIN, startTile.getRegion());
            mountainRegion.setDiscoverable(true);
            mountainRegion.setClaimable(true);
            map.putRegion(mountainRegion);
            Map.Direction direction = Map.Direction.getRandomDirection("getLand", this.random);
            int length = maximumLength - RandomUtils.randomInt(logger, "MLen", this.random, maximumLength / 2);
            for (int index = 0; index < length; ++index) {
                Tile nextTile = startTile.getNeighbourOrNull(direction);
                if (nextTile == null || !nextTile.isLand()) continue;
                nextTile.setType(mountains);
                mountainRegion.addTile(nextTile);
                ++counter;
                for (Tile neighbour : nextTile.getSurroundingTiles(1)) {
                    if (!neighbour.isLand() || neighbour.getType() == mountains) continue;
                    int r = RandomUtils.randomInt(logger, "MSiz", this.random, 8);
                    if (r == 0) {
                        neighbour.setType(mountains);
                        mountainRegion.addTile(neighbour);
                        ++counter;
                        continue;
                    }
                    if (r <= 2) continue;
                    neighbour.setType(hills);
                    mountainRegion.addTile(neighbour);
                }
            }
            int scoreValue = 2 * mountainRegion.getSize();
            mountainRegion.setScoreValue(scoreValue);
            logger.fine("Created mountain region (direction " + (Object)((Object)direction) + ", length " + length + ", size " + mountainRegion.getSize() + ", score value " + scoreValue + ").");
        }
        logger.info("Added " + counter + " mountain range tiles.");
        number = (int)((float)this.getApproximateLandCount() * randomHillsRatio) / this.mapOptions.getInteger("model.option.mountainNumber");
        counter = 0;
        block5: for (tries = 0; tries < 1000; ++tries) {
            if (counter >= number) continue;
            Tile t = map.getRandomLandTile(this.random);
            if (t == null) {
                return;
            }
            if (t.getType() == hills || t.getType() == mountains) continue;
            for (Tile tile : t.getSurroundingTiles(3)) {
                if (tile.getType() != mountains) continue;
                continue block5;
            }
            for (Tile tile : t.getSurroundingTiles(1)) {
                if (tile.isLand()) continue;
                continue block5;
            }
            boolean m = RandomUtils.randomInt(logger, "MorH", this.random, 4) == 0;
            t.setType(m ? mountains : hills);
            ++counter;
        }
        logger.info("Added " + counter + " random hills tiles.");
    }

    private void createRivers(Map map) {
        TileImprovementType riverType = this.spec.getTileImprovementType("model.improvement.river");
        int number = this.getApproximateLandCount() / this.mapOptions.getInteger("model.option.riverNumber");
        int counter = 0;
        HashMap<Tile, River> riverMap = new HashMap<Tile, River>();
        ArrayList<River> rivers = new ArrayList<River>();
        block0: for (int i = 0; i < number; ++i) {
            block1: for (int tries = 0; tries < 100; ++tries) {
                Tile tile = map.getRandomLandTile(this.random);
                if (tile == null) {
                    return;
                }
                if (!riverType.isTileTypeAllowed(tile.getType())) continue;
                for (Tile neighborTile : tile.getSurroundingTiles(2)) {
                    if (neighborTile.isLand()) continue;
                    continue block1;
                }
                if (riverMap.get(tile) != null) continue;
                ServerRegion riverRegion = new ServerRegion(this.game, "model.region.river" + i, Region.RegionType.RIVER, tile.getRegion());
                riverRegion.setDiscoverable(true);
                riverRegion.setClaimable(true);
                River river = new River(map, riverMap, riverRegion, this.random);
                if (river.flowFromSource(tile)) {
                    logger.fine("Created new river with length " + river.getLength());
                    map.putRegion(riverRegion);
                    rivers.add(river);
                    ++counter;
                    continue block0;
                }
                logger.fine("Failed to generate river.");
                continue block0;
            }
        }
        logger.info("Created " + counter + " rivers of maximum " + number);
        for (River river : rivers) {
            ServerRegion region = river.getRegion();
            int scoreValue = 0;
            for (RiverSection section : river.getSections()) {
                scoreValue += section.getSize();
            }
            region.setScoreValue(scoreValue *= 2);
            logger.fine("Created river region (length " + river.getLength() + ", score value " + scoreValue + ").");
        }
    }

    private void createLakeRegions(Map map) {
        TileType lakeType = this.spec.getTileType("model.tile.lake");
        ArrayList<Tile> lakes = new ArrayList<Tile>();
        StringBuilder sb = new StringBuilder("Lakes at:");
        for (int y = 0; y < map.getHeight(); ++y) {
            for (int x = 0; x < map.getWidth(); ++x) {
                Tile tile;
                if (!map.isValid(x, y) || (tile = map.getTile(x, y)).isLand() || map.getTile(x, y).getRegion() != null) continue;
                lakes.add(tile);
                sb.append(" ").append(Integer.toString(x)).append(",").append(Integer.toString(y));
            }
        }
        logger.fine(sb.toString());
        for (ServerRegion sr : TerrainGenerator.makeLakes(map, lakes)) {
            this.setGeographicRegion(sr);
        }
    }

    public static List<ServerRegion> makeLakes(Map map, List<Tile> lakes) {
        Game game = map.getGame();
        TileType lakeType = map.getSpecification().getTileType("model.tile.lake");
        ArrayList<Tile> todo = new ArrayList<Tile>();
        ArrayList<ServerRegion> result = new ArrayList<ServerRegion>();
        int lakeCount = 0;
        while (!lakes.isEmpty()) {
            String id;
            Tile tile = lakes.get(0);
            if (tile.getRegion() != null) continue;
            while (game.getFreeColGameObject(id = "model.region.inlandLake" + lakeCount) != null) {
                ++lakeCount;
            }
            ServerRegion lakeRegion = new ServerRegion(game, id, Region.RegionType.LAKE, null);
            map.putRegion(lakeRegion);
            lakeRegion.setPrediscovered(false);
            todo.clear();
            todo.add(tile);
            while (!todo.isEmpty()) {
                Tile t = (Tile)todo.remove(0);
                if (!lakes.contains(t)) continue;
                t.setRegion(lakeRegion);
                t.setType(lakeType);
                lakes.remove(t);
                for (Map.Direction d : Map.Direction.allDirections) {
                    Tile t0 = map.getAdjacentTile(t, d);
                    if (t0 == null) continue;
                    todo.add(t0);
                }
            }
            result.add(lakeRegion);
        }
        return result;
    }

    private void perhapsAddBonus(Tile t, boolean generateBonus) {
        Specification spec = t.getSpecification();
        TileImprovementType fishBonusLandType = spec.getTileImprovementType("model.improvement.fishBonusLand");
        TileImprovementType fishBonusRiverType = spec.getTileImprovementType("model.improvement.fishBonusRiver");
        int bonusNumber = this.mapOptions.getInteger("model.option.bonusNumber");
        if (t.isLand()) {
            if (generateBonus && RandomUtils.randomInt(logger, "Land Resource", this.random, 100) < bonusNumber) {
                t.addResource(this.createResource(t));
            }
        } else {
            int adjacentLand = 0;
            boolean adjacentRiver = false;
            for (Map.Direction direction : Map.Direction.values()) {
                Tile otherTile = t.getNeighbourOrNull(direction);
                if (otherTile == null || !otherTile.isLand()) continue;
                ++adjacentLand;
                if (!otherTile.hasRiver()) continue;
                adjacentRiver = true;
            }
            if (adjacentLand > 2) {
                t.add(new TileImprovement(this.game, t, fishBonusLandType));
            }
            if (!t.hasRiver() && adjacentRiver) {
                t.add(new TileImprovement(this.game, t, fishBonusRiverType));
            }
            if (t.getType().isHighSeasConnected()) {
                if (generateBonus && adjacentLand > 1 && RandomUtils.randomInt(logger, "Sea resource", this.random, 10 - adjacentLand) == 0) {
                    t.addResource(this.createResource(t));
                }
            } else if (RandomUtils.randomInt(logger, "Water resource", this.random, 100) < bonusNumber) {
                t.addResource(this.createResource(t));
            }
        }
    }

    private Resource createResource(Tile tile) {
        int maxValue;
        if (tile == null) {
            return null;
        }
        ResourceType resourceType = (ResourceType)RandomChoice.getWeightedRandom(null, null, tile.getType().getWeightedResources(), this.random);
        if (resourceType == null) {
            return null;
        }
        int minValue = resourceType.getMinValue();
        int quantity = minValue == (maxValue = resourceType.getMaxValue()) ? maxValue : minValue + RandomUtils.randomInt(logger, "Rsiz", this.random, maxValue - minValue + 1);
        return new Resource(this.game, tile, resourceType, quantity);
    }

    public static void encodeStyle(Tile tile) {
        Tile t;
        EnumMap<Map.Direction, Boolean> connections = new EnumMap<Map.Direction, Boolean>(Map.Direction.class);
        Iterator<Map.Direction> i$ = Map.Direction.corners.iterator();
        while (i$.hasNext()) {
            Map.Direction d;
            t = tile.getNeighbourOrNull(d = i$.next());
            connections.put(d, t != null && t.isLand());
        }
        for (Map.Direction d : Map.Direction.longSides) {
            t = tile.getNeighbourOrNull(d);
            if (t != null && t.isLand()) {
                connections.put(d, Boolean.TRUE);
                connections.put(d.getNextDirection(), Boolean.FALSE);
                connections.put(d.getPreviousDirection(), Boolean.FALSE);
                continue;
            }
            connections.put(d, Boolean.FALSE);
        }
        int result = 0;
        int index = 0;
        for (Map.Direction d : Map.Direction.corners) {
            if (((Boolean)connections.get((Object)d)).booleanValue()) {
                result += (int)Math.pow(2.0, index);
            }
            ++index;
        }
        for (Map.Direction d : Map.Direction.longSides) {
            if (((Boolean)connections.get((Object)d)).booleanValue()) {
                result += (int)Math.pow(2.0, index);
            }
            ++index;
        }
        tile.setStyle(result);
    }

    public Map createMap(LandMap landMap) {
        int width = landMap.getWidth();
        int height = landMap.getHeight();
        boolean importTerrain = this.importGame != null && this.mapOptions.getBoolean("model.option.importTerrain");
        boolean importBonuses = this.importGame != null && this.mapOptions.getBoolean("model.option.importBonuses");
        boolean mapHasLand = false;
        Map map = new Map(this.game, width, height);
        int minimumLatitude = this.mapOptions.getInteger("model.option.minimumLatitude");
        int maximumLatitude = this.mapOptions.getInteger("model.option.maximumLatitude");
        minimumLatitude = this.limitToRange(minimumLatitude, -90, 90);
        maximumLatitude = this.limitToRange(maximumLatitude, -90, 90);
        map.setMinimumLatitude(Math.min(minimumLatitude, maximumLatitude));
        map.setMaximumLatitude(Math.max(minimumLatitude, maximumLatitude));
        HashMap<String, ServerRegion> regionMap = new HashMap<String, ServerRegion>();
        if (importTerrain) {
            ServerRegion region;
            String ids = "";
            for (Region r : this.importGame.getMap().getRegions()) {
                region = new ServerRegion(this.game, r);
                map.putRegion(region);
                regionMap.put(r.getId(), region);
                ids = ids + " " + region.getNameKey();
            }
            for (Region r : this.importGame.getMap().getRegions()) {
                region = (ServerRegion)regionMap.get(r.getId());
                Region x = r.getParent();
                if (x != null) {
                    x = (Region)regionMap.get(x.getId());
                }
                region.setParent(x);
                for (Region c : r.getChildren()) {
                    x = (Region)regionMap.get(c.getId());
                    if (x == null) continue;
                    region.addChild(x);
                }
            }
            logger.info("Imported regions: " + ids);
        }
        ArrayList<Tile> fixRegions = new ArrayList<Tile>();
        for (int y = 0; y < height; ++y) {
            int latitude = map.getLatitude(y);
            for (int x = 0; x < width; ++x) {
                Tile t;
                if (landMap.isLand(x, y)) {
                    mapHasLand = true;
                }
                Tile importTile = null;
                if (importTerrain && this.importGame.getMap().isValid(x, y) && (importTile = this.importGame.getMap().getTile(x, y)) != null && importTile.isLand() == landMap.isLand(x, y)) {
                    Region r;
                    String id = importTile.getType().getId();
                    t = new Tile(this.game, this.spec.getTileType(id), x, y);
                    if (importTile.getMoveToEurope() != null) {
                        t.setMoveToEurope(importTile.getMoveToEurope());
                    }
                    if (importTile.getTileItemContainer() != null) {
                        TileItemContainer container = new TileItemContainer(this.game, t);
                        container.copyFrom(importTile.getTileItemContainer(), importBonuses, true);
                        t.setTileItemContainer(container);
                    }
                    if ((r = importTile.getRegion()) == null) {
                        fixRegions.add(t);
                    } else {
                        ServerRegion ours = (ServerRegion)regionMap.get(r.getId());
                        if (ours == null) {
                            logger.warning("Could not set tile region " + r.getId() + " for tile: " + t);
                            fixRegions.add(t);
                        } else {
                            ours.addTile(t);
                        }
                    }
                } else {
                    t = this.createTile(x, y, landMap, latitude);
                }
                map.setTile(t, x, y);
            }
        }
        this.game.setMap(map);
        this.geographicRegions = this.getStandardRegions(map);
        if (importTerrain) {
            if (!fixRegions.isEmpty()) {
                this.createOceanRegions(map);
                this.createLakeRegions(map);
                this.createLandRegions(map);
            }
        } else {
            this.createOceanRegions(map);
            map.resetHighSeas(this.mapOptions.getInteger("model.option.distanceToHighSea"), this.mapOptions.getInteger("model.option.maximumDistanceToEdge"));
            if (mapHasLand) {
                this.createMountains(map);
                this.createRivers(map);
                this.createLakeRegions(map);
                this.createLandRegions(map);
            }
        }
        for (Tile tile : map.getAllTiles()) {
            this.perhapsAddBonus(tile, !importBonuses);
            if (tile.isLand()) continue;
            TerrainGenerator.encodeStyle(tile);
        }
        map.resetContiguity();
        map.resetHighSeasCount();
        return map;
    }
}

