package Font::TTF::OTTags;

=head1 NAME

Font::TTF::OTTags - Utilities for TrueType/OpenType tags

=head1 SYNOPSIS

  use Font::TTF::OTTags qw( %tttags %ttnames %iso639 readtagsfile );
  
  # Look at built-in stuff:
  $script_tag = $tttags{'SCRIPT'}{'Cypriot Syllabary'};  # returns 'cprt'
  $lang_name = $ttnames{'LANGUAGE'}{'ALT '};             # returns 'Altai'
  
  # Map iso639 tag to/from OT lang tag
  @isotags = $iso639{'ALT '};    # returns [ 'atv', 'alt' ]
  $lang_tag = $iso639{'atv'};    # returns 'ALT '
  	
  # Read latest tags file to add to built-in definitions
  readtagsfile ("C:\\Program Files\\Microsoft VOLT\\TAGS.txt");

First-level keys to %tttags and %ttnames are:

=over

=item SCRIPT

retrieve script tag or name

=item LANGUAGE 

retrieve language tag or name

=item FEATURE 

retrieve feature tag or name

=back

Built-in data has been derived from Microsofts tag registry at
L<http://www.microsoft.com/typography/otspec/ttoreg.htm>, 
updated to draft v1.5 of the OpenType Spec.

=head1 METHODS

=cut

use strict;
use vars qw( %tttags %ttnames %iso639 @EXPORT_OK @ISA );
require Exporter;
@ISA = qw( Exporter );
@EXPORT_OK = qw( %tttags %ttnames %iso639 readtagsfile);


# All data below derived Microsoft OpenType specification 1.6

%tttags = (

'SCRIPT' => {
    "Arabic" => 'arab',
    "Armenian" => 'armn',
    "Balinese" => 'bali',
    "Bengali" => 'beng',
    "Bengali v.2" => 'bng2',
    "Bopomofo" => 'bopo',
    "Braille" => 'brai',
    "Buginese" => 'bugi',
    "Buhid" => 'buhd',
    "Byzantine Music" => 'byzm',
    "CJK Ideographic" => 'hani',
    "Canadian Syllabics" => 'cans',
    "Carian" => 'cari',
    "Cham" => 'cham',
    "Cherokee" => 'cher',
    "Coptic" => 'copt',
    "Cypriot Syllabary" => 'cprt',
    "Cyrillic" => 'cyrl',
    "Default" => 'DFLT',
    "Deseret" => 'dsrt',
    "Devanagari" => 'deva',
    "Devanagari v.2" => 'dev2',
    "Ethiopic" => 'ethi',
    "Georgian" => 'geor',
    "Glagolitic" => 'glag',
    "Gothic" => 'goth',
    "Greek" => 'grek',
    "Gujarati" => 'gujr',
    "Gujarati v.2" => 'gjr2',
    "Gurmukhi" => 'guru',
    "Gurmukhi v.2" => 'gur2',
    "Hangul" => 'hang',
    "Hangul Jamo" => 'jamo',
    "Hanunoo" => 'hano',
    "Hebrew" => 'hebr',
    "Hiragana" => 'kana',
    "Javanese" => 'java',
    "Kannada" => 'knda',
    "Kannada v.2" => 'knd2',
    "Katakana" => 'kana',
    "Kayah Li" => 'kali',
    "Kharosthi" => 'khar',
    "Khmer" => 'khmr',
    "Lao" => 'lao ',
    "Latin" => 'latn',
    "Lepcha" => 'lepc',
    "Limbu" => 'limb',
    "Linear B" => 'linb',
    "Lycian" => 'lyci',
    "Lydian" => 'lydi',
    "Malayalam" => 'mlym',
    "Malayalam v.2" => 'mlm2',
    "Mathematical Alphanumeric Symbols" => 'math',
    "Mongolian" => 'mong',
    "Musical Symbols" => 'musc',
    "Myanmar" => 'mymr',
    "N'Ko" => 'nko ',
    "New Tai Lue" => 'talu',
    "Ogham" => 'ogam',
    "Ol Chiki" => 'olck',
    "Old Italic" => 'ital',
    "Old Persian Cuneiform" => 'xpeo',
    "Oriya" => 'orya',
    "Oriya v.2" => 'ory2',
    "Osmanya" => 'osma',
    "Phags-pa" => 'phag',
    "Phoenician" => 'phnx',
    "Rejang" => 'rjng',
    "Runic" => 'runr',
    "Saurashtra" => 'saur',
    "Shavian" => 'shaw',
    "Sinhala" => 'sinh',
    "Sumero-Akkadian Cuneiform" => 'xsux',
    "Sundanese" => 'sund',
    "Syloti Nagri" => 'sylo',
    "Syriac" => 'syrc',
    "Tagalog" => 'tglg',
    "Tagbanwa" => 'tagb',
    "Tai Le" => 'tale',
    "Tamil" => 'taml',
    "Tamil v.2" => 'tml2',
    "Telugu" => 'telu',
    "Telugu v.2" => 'tel2',
    "Thaana" => 'thaa',
    "Thai" => 'thai',
    "Tibetan" => 'tibt',
    "Tifinagh" => 'tfng',
    "Ugaritic Cuneiform" => 'ugar',
    "Vai" => 'vai ',
    "Yi" => 'yi  ',
    },

'LANGUAGE' => {
    "Aari" => 'ARI ',
    "Abaza" => 'ABA ',
    "Abkhazian" => 'ABK ',
    "Adyghe" => 'ADY ',
    "Afar" => 'AFR ',
    "Afrikaans" => 'AFK ',
    "Agaw" => 'AGW ',
    "Albanian" => 'SQI ',
    "Alsatian" => 'ALS ',
    "Altai" => 'ALT ',
    "Amharic" => 'AMH ',
    "Arabic" => 'ARA ',
    "Arakanese" => 'ARK ',
    "Armenian" => 'HYE ',
    "Assamese" => 'ASM ',
    "Athapaskan" => 'ATH ',
    "Avar" => 'AVR ',
    "Awadhi" => 'AWA ',
    "Aymara" => 'AYM ',
    "Azeri" => 'AZE ',
    "Badaga" => 'BAD ',
    "Baghelkhandi" => 'BAG ',
    "Balante" => 'BLN ',
    "Balkar" => 'BAL ',
    "Balochi" => 'BLI ',
    "Balti" => 'BLT ',
    "Bambara" => 'BMB ',
    "Bamileke" => 'BML ',
    "Bashkir" => 'BSH ',
    "Basque" => 'EUQ ',
    "Baule" => 'BAU ',
    "Belarussian" => 'BEL ',
    "Bemba" => 'BEM ',
    "Bench" => 'BCH ',
    "Bengali" => 'BEN ',
    "Berber" => 'BBR ',
    "Beti" => 'BTI ',
    "Bhili" => 'BHI ',
    "Bhojpuri" => 'BHO ',
    "Bible Cree" => 'BCR ',
    "Bikol" => 'BIK ',
    "Bilen" => 'BIL ',
    "Blackfoot" => 'BKF ',
    "Bosnian" => 'BOS ',
    "Brahui" => 'BRH ',
    "Braj Bhasha" => 'BRI ',
    "Breton" => 'BRE ',
    "Bulgarian" => 'BGR ',
    "Burmese" => 'BRM ',
    "Carrier" => 'CRR ',
    "Catalan" => 'CAT ',
    "Cebuano" => 'CEB ',
    "Chaha Gurage" => 'CHG ',
    "Chattisgarhi" => 'CHH ',
    "Chechen" => 'CHE ',
    "Cherokee" => 'CHR ',
    "Chichewa" => 'CHI ',
    "Chin" => 'QIN ',
    "Chinese Hong Kong" => 'ZHH ',
    "Chinese Phonetic" => 'ZHP ',
    "Chinese Simplified" => 'ZHS ',
    "Chinese Traditional" => 'ZHT ',
    "Chipewyan" => 'CHP ',
    "Chukchi" => 'CHK ',
    "Church Slavonic" => 'CSL ',
    "Chuvash" => 'CHU ',
    "Comorian" => 'CMR ',
    "Coptic" => 'COP ',
    "Corsican" => 'COS ',
    "Cree" => 'CRE ',
    "Crimean Tatar" => 'CRT ',
    "Croatian" => 'HRV ',
    "Czech" => 'CSY ',
    "Dangme" => 'DNG ',
    "Danish" => 'DAN ',
    "Dargwa" => 'DAR ',
    "Dari" => 'DRI ',
    "Dhivehi" => 'DIV ',
    "Dhivehi (deprecated)" => 'DHV ',
    "Dinka" => 'DNK ',
    "Djerma" => 'DJR ',
    "Dogri" => 'DGR ',
    "Dungan" => 'DUN ',
    "Dutch" => 'NLD ',
    "Dzongkha" => 'DZN ',
    "Eastern Cree" => 'ECR ',
    "Ebira" => 'EBI ',
    "Edo" => 'EDO ',
    "Efik" => 'EFI ',
    "English" => 'ENG ',
    "Erzya" => 'ERZ ',
    "Esperanto" => 'NTO ',
    "Estonian" => 'ETI ',
    "Even" => 'EVN ',
    "Evenki" => 'EVK ',
    "Ewe" => 'EWE ',
    "Faroese" => 'FOS ',
    "Farsi" => 'FAR ',
    "Fijian" => 'FJI ',
    "Filipino" => 'PIL ',
    "Finnish" => 'FIN ',
    "Flemish" => 'FLE ',
    "Fon" => 'FON ',
    "Forest Nenets" => 'FNE ',
    "French" => 'FRA ',
    "French Antillean" => 'FAN ',
    "Frisian" => 'FRI ',
    "Friulian" => 'FRL ',
    "Fulani" => 'FUL ',
    "Futa" => 'FTA ',
    "Ga" => 'GAD ',
    "Gaelic" => 'GAE ',
    "Gagauz" => 'GAG ',
    "Galician" => 'GAL ',
    "Garhwali" => 'GAW ',
    "Garo" => 'GRO ',
    "Garshuni" => 'GAR ',
    "Ge'ez" => 'GEZ ',
    "Georgian" => 'KAT ',
    "German" => 'DEU ',
    "Gilyak" => 'GIL ',
    "Gondi" => 'GON ',
    "Greek" => 'ELL ',
    "Greenlandic" => 'GRN ',
    "Guarani" => 'GUA ',
    "Gujarati" => 'GUJ ',
    "Gumuz" => 'GMZ ',
    "Haitian" => 'HAI ',
    "Halam" => 'HAL ',
    "Hammer-Banna" => 'HBN ',
    "Harari" => 'HRI ',
    "Harauti" => 'HAR ',
    "Hausa" => 'HAU ',
    "Hawaiin" => 'HAW ',
    "Hebrew" => 'IWR ',
    "High Mari" => 'HMA ',
    "Hiligaynon" => 'HIL ',
    "Hindi" => 'HIN ',
    "Hindko" => 'HND ',
    "Ho" => 'HO  ',
    "Hungarian" => 'HUN ',
    "Icelandic" => 'ISL ',
    "Igbo" => 'IBO ',
    "Ijo" => 'IJO ',
    "Ilokano" => 'ILO ',
    "Inari Sami" => 'ISM ',
    "Indonesian" => 'IND ',
    "Ingush" => 'ING ',
    "Inuktitut" => 'INU ',
    "Irish" => 'IRI ',
    "Irish Traditional" => 'IRT ',
    "Italian" => 'ITA ',
    "Japanese" => 'JAN ',
    "Javanese" => 'JAV ',
    "Judezmo" => 'JUD ',
    "Jula" => 'JUL ',
    "Kabardian" => 'KAB ',
    "Kachchi" => 'KAC ',
    "Kalenjin" => 'KAL ',
    "Kalmyk" => 'KLM ',
    "Kamba" => 'KMB ',
    "Kannada" => 'KAN ',
    "Kanuri" => 'KNR ',
    "Karachay" => 'KAR ',
    "Karaim" => 'KRM ',
    "Karakalpak" => 'KRK ',
    "Karelian" => 'KRL ',
    "Karen" => 'KRN ',
    "Kashmiri" => 'KSH ',
    "Kazakh" => 'KAZ ',
    "Kebena" => 'KEB ',
    "Khakass" => 'KHA ',
    "Khanty-Kazim" => 'KHK ',
    "Khanty-Shurishkar" => 'KHS ',
    "Khanty-Vakhi" => 'KHV ',
    "Khasi" => 'KSI ',
    "Khmer" => 'KHM ',
    "Khowar" => 'KHW ',
    "Khutsuri Georgian" => 'KGE ',
    "Kikongo" => 'KON ',
    "Kikuyu" => 'KIK ',
    "Kildin Sami" => 'KSM ',
    "Kirghiz" => 'KIR ',
    "Kisii" => 'KIS ',
    "Kodagu" => 'KOD ',
    "Kokni" => 'KKN ',
    "Komi-Permyak" => 'KOP ',
    "Komi-Zyrian" => 'KOZ ',
    "Komo" => 'KMO ',
    "Komso" => 'KMS ',
    "Konkani" => 'KOK ',
    "Koorete" => 'KRT ',
    "Korean" => 'KOR ',
    "Korean Old Hangul" => 'KOH ',
    "Koryak" => 'KYK ',
    "Kpelle" => 'KPL ',
    "Krio" => 'KRI ',
    "Kui" => 'KUI ',
    "Kulvi" => 'KUL ',
    "Kumaoni" => 'KMN ',
    "Kumyk" => 'KUM ',
    "Kurdish" => 'KUR ',
    "Kurukh" => 'KUU ',
    "Kuy" => 'KUY ',
    "L-Cree" => 'LCR ',
    "Ladakhi" => 'LDK ',
    "Ladin" => 'LAD ',
    "Lahuli" => 'LAH ',
    "Lak" => 'LAK ',
    "Lambani" => 'LAM ',
    "Lao" => 'LAO ',
    "Latin" => 'LAT ',
    "Latvian" => 'LVI ',
    "Laz" => 'LAZ ',
    "Lezgi" => 'LEZ ',
    "Limbu" => 'LMB ',
    "Lingala" => 'LIN ',
    "Lithuanian" => 'LTH ',
    "Lomwe" => 'LMW ',
    "Low Mari" => 'LMA ',
    "Lower Sorbian" => 'LSB ',
    "Luba" => 'LUB ',
    "Luganda" => 'LUG ',
    "Luhya" => 'LUH ',
    "Lule Sami" => 'LSM ',
    "Luo" => 'LUO ',
    "Luxembourgish" => 'LTZ ',
    "Macedonian" => 'MKD ',
    "Maithili" => 'MTH ',
    "Majang" => 'MAJ ',
    "Makua" => 'MAK ',
    "Malagasy" => 'MLG ',
    "Malay" => 'MLY ',
    "Malayalam Reformed" => 'MLR ',
    "Malayalam Traditional" => 'MAL ',
    "Male" => 'MLE ',
    "Malinke" => 'MLN ',
    "Maltese" => 'MTS ',
    "Manchu" => 'MCH ',
    "Mandinka" => 'MND ',
    "Maninka" => 'MNK ',
    "Manipuri" => 'MNI ',
    "Mansi" => 'MAN ',
    "Manx Gaelic" => 'MNX ',
    "Maori" => 'MRI ',
    "Mapudungun" => 'MAP ',
    "Marathi" => 'MAR ',
    "Marwari" => 'MAW ',
    "Mbundu" => 'MBN ',
    "Me'en" => 'MEN ',
    "Mende" => 'MDE ',
    "Mizo" => 'MIZ ',
    "Mohawk" => 'MOH ',
    "Moksha" => 'MOK ',
    "Moldavian" => 'MOL ',
    "Mon" => 'MON ',
    "Mongolian" => 'MNG ',
    "Moose Cree" => 'MCR ',
    "Moroccan" => 'MOR ',
    "Mundari" => 'MUN ',
    "N'Ko" => 'NKO ',
    "N-Cree" => 'NCR ',
    "Naga-Assamese" => 'NAG ',
    "Nagari" => 'NGR ',
    "Nanai" => 'NAN ',
    "Naskapi" => 'NAS ',
    "Ndebele" => 'NDB ',
    "Ndonga" => 'NDG ',
    "Nepali" => 'NEP ',
    "Newari" => 'NEW ',
    "Nisi" => 'NIS ',
    "Niuean" => 'NIU ',
    "Nkole" => 'NKL ',
    "Nogai" => 'NOG ',
    "Northern Sami" => 'NSM ',
    "Northern Tai" => 'NTA ',
    "Norway House Cree" => 'NHC ',
    "Norwegian" => 'NOR ',
    "Nynorsk" => 'NYN ',
    "Occitan" => 'OCI ',
    "Oji-Cree" => 'OCR ',
    "Ojibway" => 'OJB ',
    "Oriya" => 'ORI ',
    "Oromo" => 'ORO ',
    "Ossetian" => 'OSS ',
    "Palaung" => 'PLG ',
    "Palestinian Aramaic" => 'PAA ',
    "Pali" => 'PAL ',
    "Palpa" => 'PAP ',
    "Pashto" => 'PAS ',
    "Phonetic transcription-Americanist conventions" => 'APPH',
    "Phonetic transcription-IPA conventions" => 'IPPH',
    "Polish" => 'PLK ',
    "Polytonic Greek" => 'PGR ',
    "Portuguese" => 'PTG ',
    "Provencal" => 'PRO ',
    "Punjabi" => 'PAN ',
    "R-Cree" => 'RCR ',
    "Rajasthani" => 'RAJ ',
    "Rhaeto-Romanic" => 'RMS ',
    "Riang" => 'RIA ',
    "Romanian" => 'ROM ',
    "Romany" => 'ROY ',
    "Ruanda" => 'RUA ',
    "Russian" => 'RUS ',
    "Russian Buriat" => 'RBU ',
    "Rusyn" => 'RSY ',
    "Sadri" => 'SAD ',
    "Samoan" => 'SMO ',
    "Sango" => 'SGO ',
    "Sanskrit" => 'SAN ',
    "Santali" => 'SAT ',
    "Saraiki" => 'SRK ',
    "Sayisi" => 'SAY ',
    "Sekota" => 'SEK ',
    "Selkup" => 'SEL ',
    "Sena" => 'SNA ',
    "Serbian" => 'SRB ',
    "Serer" => 'SRR ',
    "Shan" => 'SHN ',
    "Sibe" => 'SIB ',
    "Sidamo" => 'SID ',
    "Silte Gurage" => 'SIG ',
    "Sindhi" => 'SND ',
    "Sinhalese" => 'SNH ',
    "Skolt Sami" => 'SKS ',
    "Slavey" => 'SLA ',
    "Slovak" => 'SKY ',
    "Slovenian" => 'SLV ',
    "Sodo Gurage" => 'SOG ',
    "Somali" => 'SML ',
    "Soninke" => 'SNK ',
    "Sotho" => 'SOT ',
    "South Slavey" => 'SSL ',
    "Southern Sami" => 'SSM ',
    "Spanish" => 'ESP ',
    "Suri" => 'SUR ',
    "Sutu" => 'SXT ',
    "Svan" => 'SVA ',
    "Swadaya Aramaic" => 'SWA ',
    "Swahili" => 'SWK ',
    "Swazi" => 'SWZ ',
    "Swedish" => 'SVE ',
    "Syriac" => 'SYR ',
    "TH-Cree" => 'TCR ',
    "Tabasaran" => 'TAB ',
    "Tahitian" => 'THT ',
    "Tai Lue" => 'XBD ',
    "Tajiki" => 'TAJ ',
    "Tamil" => 'TAM ',
    "Tatar" => 'TAT ',
    "Telugu" => 'TEL ',
    "Temne" => 'TMN ',
    "Thai" => 'THA ',
    "Tibetan" => 'TIB ',
    "Tigre" => 'TGR ',
    "Tigrinya" => 'TGY ',
    "Todo" => 'TOD ',
    "Tonga" => 'TNG ',
    "Tongan" => 'TGN ',
    "Tsonga" => 'TSG ',
    "Tswana" => 'TNA ',
    "Tulu" => 'TUL ',
    "Tundra Nenets" => 'TNE ',
    "Turkish" => 'TRK ',
    "Turkmen" => 'TKM ',
    "Turoyo Aramaic" => 'TUA ',
    "Tuvin" => 'TUV ',
    "Twi" => 'TWI ',
    "Udmurt" => 'UDM ',
    "Ukrainian" => 'UKR ',
    "Upper Sorbian" => 'USB ',
    "Urdu" => 'URD ',
    "Uyghur" => 'UYG ',
    "Uzbek" => 'UZB ',
    "Venda" => 'VEN ',
    "Vietnamese" => 'VIT ',
    "Wa" => 'WA  ',
    "Wagdi" => 'WAG ',
    "Welsh" => 'WEL ',
    "West-Cree" => 'WCR ',
    "Wolof" => 'WLF ',
    "Woods Cree" => 'DCR ',
    "Xhosa" => 'XHS ',
    "Y-Cree" => 'YCR ',
    "Yakut" => 'YAK ',
    "Yi Classic" => 'YIC ',
    "Yi Modern" => 'YIM ',
    "Yiddish" => 'JII ',
    "Yoruba" => 'YBA ',
    "Zande" => 'ZND ',
    "Zulu" => 'ZUL ',
    },

'FEATURE' => {
    "Above-base Forms" => 'abvf',
    "Above-base Mark Positioning" => 'abvm',
    "Above-base Substitutions" => 'abvs',
    "Access All Alternates" => 'aalt',
    "Akhands" => 'akhn',
    "Alternate Annotation Forms" => 'nalt',
    "Alternate Half Widths" => 'halt',
    "Alternate Vertical Half Metrics" => 'vhal',
    "Alternate Vertical Metrics" => 'valt',
    "Alternative Fractions" => 'afrc',
    "Below-base Forms" => 'blwf',
    "Below-base Mark Positioning" => 'blwm',
    "Below-base Substitutions" => 'blws',
    "Capital Spacing" => 'cpsp',
    "Case-Sensitive Forms" => 'case',
    "Centered CJK Punctuation" => 'cpct',
    "Character Variants 01" => 'cv01',
    "Character Variants 02" => 'cv02',
    "Character Variants 03" => 'cv03',
    "Character Variants 04" => 'cv04',
    "Character Variants 05" => 'cv05',
    "Character Variants 06" => 'cv06',
    "Character Variants 07" => 'cv07',
    "Character Variants 08" => 'cv08',
    "Character Variants 09" => 'cv09',
    "Character Variants 10" => 'cv10',
    "Character Variants 11" => 'cv11',
    "Character Variants 12" => 'cv12',
    "Character Variants 13" => 'cv13',
    "Character Variants 14" => 'cv14',
    "Character Variants 15" => 'cv15',
    "Character Variants 16" => 'cv16',
    "Character Variants 17" => 'cv17',
    "Character Variants 18" => 'cv18',
    "Character Variants 19" => 'cv19',
    "Character Variants 20" => 'cv20',
    "Character Variants 21" => 'cv21',
    "Character Variants 22" => 'cv22',
    "Character Variants 23" => 'cv23',
    "Character Variants 24" => 'cv24',
    "Character Variants 25" => 'cv25',
    "Character Variants 26" => 'cv26',
    "Character Variants 27" => 'cv27',
    "Character Variants 28" => 'cv28',
    "Character Variants 29" => 'cv29',
    "Character Variants 30" => 'cv30',
    "Character Variants 31" => 'cv31',
    "Character Variants 32" => 'cv32',
    "Character Variants 33" => 'cv33',
    "Character Variants 34" => 'cv34',
    "Character Variants 35" => 'cv35',
    "Character Variants 36" => 'cv36',
    "Character Variants 37" => 'cv37',
    "Character Variants 38" => 'cv38',
    "Character Variants 39" => 'cv39',
    "Character Variants 40" => 'cv40',
    "Character Variants 41" => 'cv41',
    "Character Variants 42" => 'cv42',
    "Character Variants 43" => 'cv43',
    "Character Variants 44" => 'cv44',
    "Character Variants 45" => 'cv45',
    "Character Variants 46" => 'cv46',
    "Character Variants 47" => 'cv47',
    "Character Variants 48" => 'cv48',
    "Character Variants 49" => 'cv49',
    "Character Variants 50" => 'cv50',
    "Character Variants 51" => 'cv51',
    "Character Variants 52" => 'cv52',
    "Character Variants 53" => 'cv53',
    "Character Variants 54" => 'cv54',
    "Character Variants 55" => 'cv55',
    "Character Variants 56" => 'cv56',
    "Character Variants 57" => 'cv57',
    "Character Variants 58" => 'cv58',
    "Character Variants 59" => 'cv59',
    "Character Variants 60" => 'cv60',
    "Character Variants 61" => 'cv61',
    "Character Variants 62" => 'cv62',
    "Character Variants 63" => 'cv63',
    "Character Variants 64" => 'cv64',
    "Character Variants 65" => 'cv65',
    "Character Variants 66" => 'cv66',
    "Character Variants 67" => 'cv67',
    "Character Variants 68" => 'cv68',
    "Character Variants 69" => 'cv69',
    "Character Variants 70" => 'cv70',
    "Character Variants 71" => 'cv71',
    "Character Variants 72" => 'cv72',
    "Character Variants 73" => 'cv73',
    "Character Variants 74" => 'cv74',
    "Character Variants 75" => 'cv75',
    "Character Variants 76" => 'cv76',
    "Character Variants 77" => 'cv77',
    "Character Variants 78" => 'cv78',
    "Character Variants 79" => 'cv79',
    "Character Variants 80" => 'cv80',
    "Character Variants 81" => 'cv81',
    "Character Variants 82" => 'cv82',
    "Character Variants 83" => 'cv83',
    "Character Variants 84" => 'cv84',
    "Character Variants 85" => 'cv85',
    "Character Variants 86" => 'cv86',
    "Character Variants 87" => 'cv87',
    "Character Variants 88" => 'cv88',
    "Character Variants 89" => 'cv89',
    "Character Variants 90" => 'cv90',
    "Character Variants 91" => 'cv91',
    "Character Variants 92" => 'cv92',
    "Character Variants 93" => 'cv93',
    "Character Variants 94" => 'cv94',
    "Character Variants 95" => 'cv95',
    "Character Variants 96" => 'cv96',
    "Character Variants 97" => 'cv97',
    "Character Variants 98" => 'cv98',
    "Character Variants 99" => 'cv99',
    "Conjunct Form After Ro" => 'cfar',
    "Conjunct Forms" => 'cjct',
    "Contextual Alternates" => 'calt',
    "Contextual Ligatures" => 'clig',
    "Contextual Swash" => 'cswh',
    "Cursive Positioning" => 'curs',
    "Denominators" => 'dnom',
    "Discretionary Ligatures" => 'dlig',
    "Distances" => 'dist',
    "Expert Forms" => 'expt',
    "Final Glyph on Line Alternates" => 'falt',
    "Fractions" => 'frac',
    "Full Widths" => 'fwid',
    "Glyph Composition / Decomposition" => 'ccmp',
    "Halant Forms" => 'haln',
    "Half Forms" => 'half',
    "Half Widths" => 'hwid',
    "Hangul" => 'hngl',
    "Historical Forms" => 'hist',
    "Historical Ligatures" => 'hlig',
    "Hojo Kanji Forms (JIS X 0212-1990 Kanji Forms)" => 'hojo',
    "Horizontal Kana Alternates" => 'hkna',
    "Initial Forms" => 'init',
    "Isolated Forms" => 'isol',
    "Italics" => 'ital',
    "JIS2004 Forms" => 'jp04',
    "JIS78 Forms" => 'jp78',
    "JIS83 Forms" => 'jp83',
    "JIS90 Forms" => 'jp90',
    "Justification Alternates" => 'jalt',
    "Kerning" => 'kern',
    "Leading Jamo Forms" => 'ljmo',
    "Left Bounds" => 'lfbd',
    "Left-to-right alternates" => 'ltra',
    "Left-to-right mirrored forms" => 'ltrm',
    "Lining Figures" => 'lnum',
    "Localized Forms" => 'locl',
    "Mark Positioning" => 'mark',
    "Mark Positioning via Substitution" => 'mset',
    "Mark to Mark Positioning" => 'mkmk',
    "Mathematical Greek" => 'mgrk',
    "Medial Forms" => 'medi',
    "Medial Forms #2" => 'med2',
    "NLC Kanji Forms" => 'nlck',
    "Nukta Forms" => 'nukt',
    "Numerators" => 'numr',
    "Oldstyle Figures" => 'onum',
    "Optical Bounds" => 'opbd',
    "Optical size" => 'size',
    "Ordinals" => 'ordn',
    "Ornaments" => 'ornm',
    "Petite Capitals" => 'pcap',
    "Petite Capitals From Capitals" => 'c2pc',
    "Post-base Forms" => 'pstf',
    "Post-base Substitutions" => 'psts',
    "Pre-Base Forms" => 'pref',
    "Pre-base Substitutions" => 'pres',
    "Proportional Alternate Vertical Metrics" => 'vpal',
    "Proportional Alternate Widths" => 'palt',
    "Proportional Figures" => 'pnum',
    "Proportional Kana" => 'pkna',
    "Proportional Widths" => 'pwid',
    "Quarter Widths" => 'qwid',
    "Rakar Forms" => 'rkrf',
    "Randomize" => 'rand',
    "Reph Forms" => 'rphf',
    "Required Ligatures" => 'rlig',
    "Right Bounds" => 'rtbd',
    "Right-to-left alternates" => 'rtla',
    "Right-to-left mirrored forms" => 'rtlm',
    "Ruby Notation Forms" => 'ruby',
    "Scientific Inferiors" => 'sinf',
    "Simplified Forms" => 'smpl',
    "Slashed Zero" => 'zero',
    "Small Capitals" => 'smcp',
    "Small Capitals From Capitals" => 'c2sc',
    "Standard Ligatures" => 'liga',
    "Stylistic Alternates" => 'salt',
    "Stylistic Set 1" => 'ss01',
    "Stylistic Set 10" => 'ss10',
    "Stylistic Set 11" => 'ss11',
    "Stylistic Set 12" => 'ss12',
    "Stylistic Set 13" => 'ss13',
    "Stylistic Set 14" => 'ss14',
    "Stylistic Set 15" => 'ss15',
    "Stylistic Set 16" => 'ss16',
    "Stylistic Set 17" => 'ss17',
    "Stylistic Set 18" => 'ss18',
    "Stylistic Set 19" => 'ss19',
    "Stylistic Set 2" => 'ss02',
    "Stylistic Set 20" => 'ss20',
    "Stylistic Set 3" => 'ss03',
    "Stylistic Set 4" => 'ss04',
    "Stylistic Set 5" => 'ss05',
    "Stylistic Set 6" => 'ss06',
    "Stylistic Set 7" => 'ss07',
    "Stylistic Set 8" => 'ss08',
    "Stylistic Set 9" => 'ss09',
    "Subscript" => 'subs',
    "Superscript" => 'sups',
    "Swash" => 'swsh',
    "Tabular Figures" => 'tnum',
    "Terminal Forms" => 'fina',
    "Terminal Forms #2" => 'fin2',
    "Terminal Forms #3" => 'fin3',
    "Third Widths" => 'twid',
    "Titling" => 'titl',
    "Traditional Forms" => 'trad',
    "Traditional Name Forms" => 'tnam',
    "Trailing Jamo Forms" => 'tjmo',
    "Unicase" => 'unic',
    "Vattu Variants" => 'vatu',
    "Vertical Alternates and Rotation" => 'vrt2',
    "Vertical Kana Alternates" => 'vkna',
    "Vertical Kerning" => 'vkrn',
    "Vertical Writing" => 'vert',
    "Vowel Jamo Forms" => 'vjmo',
    },

);

%iso639 = (
    'ABA ' => 'abq',
    'ABK ' => 'abk',
    'ADY ' => 'ady',
    'AFK ' => 'afr',
    'AFR ' => 'aar',
    'AGW ' => 'ahg',
    'ALS ' => 'gsw',
    'ALT ' => 'atv alt',
    'AMH ' => 'amh',
    'ARA ' => 'ara',
    'ARI ' => 'aiw',
    'ARK ' => 'mhv rmz rki',
    'ASM ' => 'asm',
    'ATH ' => 'apk apj apl apm apw nav bea sek bcr caf crx clc gwi haa chp dgr scs xsl srs ing hoi koy hup ktw mvb wlk coq ctc gce tol tuu kkz tgx tht aht tfn taa tau tcb kuu tce ttm txc',
    'AVR ' => 'ava',
    'AWA ' => 'awa',
    'AYM ' => 'aym',
    'AZE ' => 'aze',
    'BAD ' => 'bfq',
    'BAG ' => 'bfy',
    'BAL ' => 'krc',
    'BAU ' => 'bci',
    'BCH ' => 'bcq',
    'BEL ' => 'bel',
    'BEM ' => 'bem',
    'BEN ' => 'ben',
    'BGR ' => 'bul',
    'BHI ' => 'bhi bhb',
    'BHO ' => 'bho',
    'BIK ' => 'bik',
    'BIL ' => 'byn',
    'BKF ' => 'bla',
    'BLI ' => 'bal',
    'BLN ' => 'bjt ble',
    'BLT ' => 'bft',
    'BMB ' => 'bam',
    'BOS ' => 'bos',
    'BRE ' => 'bre',
    'BRH ' => 'brh',
    'BRI ' => 'bra',
    'BRM ' => 'mya',
    'BSH ' => 'bak',
    'BTI ' => 'btb',
    'CAT ' => 'cat',
    'CEB ' => 'ceb',
    'CHE ' => 'che',
    'CHG ' => 'sgw',
    'CHH ' => 'hne',
    'CHI ' => 'nya',
    'CHK ' => 'ckt',
    'CHP ' => 'chp',
    'CHR ' => 'chr',
    'CHU ' => 'chv',
    'CMR ' => 'swb wlc wni zdj',
    'COP ' => 'cop',
    'COS ' => 'cos',
    'CRE ' => 'cre',
    'CRR ' => 'crx caf',
    'CRT ' => 'crh',
    'CSL ' => 'chu',
    'CSY ' => 'ces',
    'DAN ' => 'dan',
    'DAR ' => 'dar',
    'DCR ' => 'cwd',
    'DEU ' => 'deu',
    'DGR ' => 'doi',
    'DHV ' => 'div',
    'DIV ' => 'div',
    'DJR ' => 'dje',
    'DNG ' => 'ada',
    'DNK ' => 'din',
    'DRI ' => 'prs',
    'DUN ' => 'dng',
    'DZN ' => 'dzo',
    'EBI ' => 'igb',
    'ECR ' => 'crj crl',
    'EDO ' => 'bin',
    'EFI ' => 'efi',
    'ELL ' => 'ell',
    'ENG ' => 'eng',
    'ERZ ' => 'myv',
    'ESP ' => 'spa',
    'ETI ' => 'est',
    'EUQ ' => 'eus',
    'EVK ' => 'evn',
    'EVN ' => 'eve',
    'EWE ' => 'ewe',
    'FAN ' => 'acf',
    'FAR ' => 'fas',
    'FIN ' => 'fin',
    'FJI ' => 'fij',
    'FLE ' => 'vls',
    'FNE ' => 'enf',
    'FON ' => 'fon',
    'FOS ' => 'fao',
    'FRA ' => 'fra',
    'FRI ' => 'fry',
    'FRL ' => 'fur',
    'FTA ' => 'fuf',
    'FUL ' => 'ful',
    'GAD ' => 'gaa',
    'GAE ' => 'gla',
    'GAG ' => 'gag',
    'GAL ' => 'glg',
    'GAW ' => 'gbm',
    'GEZ ' => 'gez',
    'GIL ' => 'niv',
    'GMZ ' => 'guk',
    'GON ' => 'gon',
    'GRN ' => 'kal',
    'GRO ' => 'grt',
    'GUA ' => 'grn',
    'GUJ ' => 'guj',
    'HAI ' => 'hat',
    'HAL ' => 'flm',
    'HAR ' => 'hoj',
    'HAU ' => 'hau',
    'HAW ' => 'haw',
    'HBN ' => 'amf',
    'HIL ' => 'hil',
    'HIN ' => 'hin',
    'HMA ' => 'mrj',
    'HND ' => 'hno hnd',
    'HO  ' => 'hoc',
    'HRI ' => 'har',
    'HRV ' => 'hrv',
    'HUN ' => 'hun',
    'HYE ' => 'hye',
    'IBO ' => 'ibo',
    'IJO ' => 'ijc',
    'ILO ' => 'ilo',
    'IND ' => 'ind',
    'ING ' => 'inh',
    'INU ' => 'iku',
    'IRI ' => 'gle',
    'IRT ' => 'gle',
    'ISL ' => 'isl',
    'ISM ' => 'smn',
    'ITA ' => 'ita',
    'IWR ' => 'heb',
    'JAN ' => 'jpn',
    'JAV ' => 'jav',
    'JII ' => 'yid',
    'JUD ' => 'lad',
    'JUL ' => 'dyu',
    'KAB ' => 'kbd',
    'KAC ' => 'kfr',
    'KAL ' => 'kln',
    'KAN ' => 'kan',
    'KAR ' => 'krc',
    'KAT ' => 'kat',
    'KAZ ' => 'kaz',
    'KEB ' => 'ktb',
    'KGE ' => 'kat',
    'KHA ' => 'kjh',
    'KHK ' => 'kca',
    'KHM ' => 'khm',
    'KHS ' => 'kca',
    'KHV ' => 'kca',
    'KHW ' => 'khw',
    'KIK ' => 'kik',
    'KIR ' => 'kir',
    'KIS ' => 'kqs kss',
    'KKN ' => 'kex',
    'KLM ' => 'xal',
    'KMB ' => 'kam',
    'KMN ' => 'kfy',
    'KMO ' => 'kmw',
    'KMS ' => 'kxc',
    'KNR ' => 'kau',
    'KOD ' => 'kfa',
    'KOH ' => 'okm',
    'KOK ' => 'kok',
    'KON ' => 'ktu',
    'KOP ' => 'koi',
    'KOR ' => 'kor',
    'KOZ ' => 'kpv',
    'KPL ' => 'kpe',
    'KRI ' => 'kri',
    'KRK ' => 'kaa',
    'KRL ' => 'krl',
    'KRM ' => 'kdr',
    'KRN ' => 'kar',
    'KRT ' => 'kqy',
    'KSH ' => 'kas',
    'KSI ' => 'kha',
    'KSM ' => 'sjd',
    'KUI ' => 'kxu',
    'KUL ' => 'kfx',
    'KUM ' => 'kum',
    'KUR ' => 'kur',
    'KUU ' => 'kru',
    'KUY ' => 'kdt',
    'KYK ' => 'kpy',
    'LAD ' => 'lld',
    'LAH ' => 'bfu',
    'LAK ' => 'lbe',
    'LAM ' => 'lmn',
    'LAO ' => 'lao',
    'LAT ' => 'lat',
    'LAZ ' => 'lzz',
    'LCR ' => 'crm',
    'LDK ' => 'lbj',
    'LEZ ' => 'lez',
    'LIN ' => 'lin',
    'LMA ' => 'mhr',
    'LMB ' => 'lif',
    'LMW ' => 'ngl',
    'LSB ' => 'dsb',
    'LSM ' => 'smj',
    'LTH ' => 'lit',
    'LTZ ' => 'ltz',
    'LUB ' => 'lua lub',
    'LUG ' => 'lug',
    'LUH ' => 'luy',
    'LUO ' => 'luo',
    'LVI ' => 'lav',
    'MAJ ' => 'mpe',
    'MAK ' => 'vmw',
    'MAL ' => 'mal',
    'MAN ' => 'mns',
    'MAP ' => 'arn',
    'MAR ' => 'mar',
    'MAW ' => 'mwr',
    'MBN ' => 'kmb',
    'MCH ' => 'mnc',
    'MCR ' => 'crm',
    'MDE ' => 'men',
    'MEN ' => 'mym',
    'MIZ ' => 'lus',
    'MKD ' => 'mkd',
    'MLE ' => 'mdy',
    'MLG ' => 'mlg',
    'MLN ' => 'mlq',
    'MLR ' => 'mal',
    'MLY ' => 'msa',
    'MND ' => 'mnk',
    'MNG ' => 'mon',
    'MNI ' => 'mni',
    'MNK ' => 'man',
    'MNX ' => 'glv',
    'MOH ' => 'moh',
    'MOK ' => 'mdf',
    'MOL ' => 'mol',
    'MON ' => 'mnw',
    'MRI ' => 'mri',
    'MTH ' => 'mai',
    'MTS ' => 'mlt',
    'MUN ' => 'unr',
    'NAG ' => 'nag',
    'NAN ' => 'gld',
    'NAS ' => 'nsk',
    'NCR ' => 'csw',
    'NDB ' => 'nbl nde',
    'NDG ' => 'ndo',
    'NEP ' => 'nep',
    'NEW ' => 'new',
    'NHC ' => 'csw',
    'NIS ' => 'dap',
    'NIU ' => 'niu',
    'NKL ' => 'nyn',
    'NKO ' => 'nqo',
    'NLD ' => 'nld',
    'NOG ' => 'nog',
    'NOR ' => 'nob',
    'NSM ' => 'sme',
    'NTA ' => 'nod',
    'NTO ' => 'epo',
    'NYN ' => 'nno',
    'OCI ' => 'oci',
    'OCR ' => 'ojs',
    'OJB ' => 'oji',
    'ORI ' => 'ori',
    'ORO ' => 'orm',
    'OSS ' => 'oss',
    'PAA ' => 'sam',
    'PAL ' => 'pli',
    'PAN ' => 'pan',
    'PAP ' => 'plp',
    'PAS ' => 'pus',
    'PGR ' => 'ell',
    'PIL ' => 'fil',
    'PLG ' => 'pce rbb pll',
    'PLK ' => 'pol',
    'PRO ' => 'pro',
    'PTG ' => 'por',
    'QIN ' => 'bgr cnh cnw czt sez tcp csy ctd flm pck tcz zom cmr dao hlt cka cnk mrh mwg cbl cnb csh',
    'RAJ ' => 'raj',
    'RBU ' => 'bxr',
    'RCR ' => 'atj',
    'RIA ' => 'ria',
    'RMS ' => 'roh',
    'ROM ' => 'ron',
    'ROY ' => 'rom',
    'RSY ' => 'rue',
    'RUA ' => 'kin',
    'RUS ' => 'rus',
    'SAD ' => 'sck',
    'SAN ' => 'san',
    'SAT ' => 'sat',
    'SAY ' => 'chp',
    'SEK ' => 'xan',
    'SEL ' => 'sel',
    'SGO ' => 'sag',
    'SHN ' => 'shn',
    'SIB ' => 'sjo',
    'SID ' => 'sid',
    'SIG ' => 'xst',
    'SKS ' => 'sms',
    'SKY ' => 'slk',
    'SLA ' => 'scs',
    'SLV ' => 'slv',
    'SML ' => 'som',
    'SMO ' => 'smo',
    'SNA ' => 'she',
    'SND ' => 'snd',
    'SNH ' => 'sin',
    'SNK ' => 'snk',
    'SOG ' => 'gru',
    'SOT ' => 'nso sot',
    'SQI ' => 'sqi',
    'SRB ' => 'srp',
    'SRK ' => 'skr',
    'SRR ' => 'srr',
    'SSL ' => 'xsl',
    'SSM ' => 'sma',
    'SUR ' => 'suq',
    'SVA ' => 'sva',
    'SVE ' => 'swe',
    'SWA ' => 'aii',
    'SWK ' => 'swa',
    'SWZ ' => 'ssw',
    'SXT ' => 'ngo',
    'SYR ' => 'syr',
    'TAB ' => 'tab',
    'TAJ ' => 'tgk',
    'TAM ' => 'tam',
    'TAT ' => 'tat',
    'TCR ' => 'cwd',
    'TEL ' => 'tel',
    'TGN ' => 'ton',
    'TGR ' => 'tig',
    'TGY ' => 'tir',
    'THA ' => 'tha',
    'THT ' => 'tah',
    'TIB ' => 'bod',
    'TKM ' => 'tuk',
    'TMN ' => 'tem',
    'TNA ' => 'tsn',
    'TNE ' => 'enh',
    'TNG ' => 'toi',
    'TOD ' => 'xal',
    'TRK ' => 'tur',
    'TSG ' => 'tso',
    'TUA ' => 'tru',
    'TUL ' => 'tcy',
    'TUV ' => 'tyv',
    'TWI ' => 'aka',
    'UDM ' => 'udm',
    'UKR ' => 'ukr',
    'URD ' => 'urd',
    'USB ' => 'hsb',
    'UYG ' => 'uig',
    'UZB ' => 'uzb',
    'VEN ' => 'ven',
    'VIT ' => 'vie',
    'WA  ' => 'wbm',
    'WAG ' => 'wbr',
    'WCR ' => 'crk',
    'WEL ' => 'cym',
    'WLF ' => 'wol',
    'XBD ' => 'khb',
    'XHS ' => 'xho',
    'YAK ' => 'sah',
    'YBA ' => 'yor',
    'YIM ' => 'iii',
    'ZHH ' => 'zho',
    'ZHP ' => 'zho',
    'ZHS ' => 'zho',
    'ZHT ' => 'zho',
    'ZND ' => 'zne',
    'ZUL ' => 'zul',
);

{
    foreach my $s (qw ( SCRIPT LANGUAGE FEATURE ) )
    {
        map { $ttnames{$s}{$tttags{$s}{$_}} = $_ }  keys %{$tttags{$s}};
    }
	
	# For ISO639 info, the raw data is a space-separated list of ISO639
	# language IDs. We convert that list to an array.
	
	foreach my $tag (keys %iso639)
	{
		my $list = $iso639{$tag};
		$iso639{$tag} = [ split(' ', $list) ];
		# Also set the reverse mapping:
		map { $iso639{$_} = $tag } @{$iso639{$tag}};
	}
}


=head2 readtagsfile ( filename )

Read a file in the syntax of Tags.txt (included with Microsoft VOLT) to obtain additional/replacement tag definitions.

Returns 0 if cannot open the file; else 1.

=cut

sub readtagsfile
{
    my $fname = shift;
    open (TAGS, $fname) or return 0;
	my ($what, $name, $tag);
	while (<TAGS>)
	{
		($what, $name, $tag) = (m/"([^"]*)", "([^"]*)", "([^"]*)"/);  #"
		$ttnames{$what}{$tag} = $name;
		$tttags{$what}{$name} = $tag;
	}
	close TAGS;
    return 1;	
}

1;

=head1 AUTHOR

Bob Hallissy. L<Bob_Hallissy@sil.org>


=head1 LICENSING

Copyright (c) 1998-2013, SIL International (http://www.sil.org) 

This module is released under the terms of the Artistic License 2.0. 
For details, see the full text of the license in the file LICENSE.



=cut