/*
 * Copyright 2005 - 2009  Zarafa B.V.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License, version 3, 
 * as published by the Free Software Foundation with the following additional 
 * term according to sec. 7:
 *  
 * According to sec. 7 of the GNU Affero General Public License, version
 * 3, the terms of the AGPL are supplemented with the following terms:
 * 
 * "Zarafa" is a registered trademark of Zarafa B.V. The licensing of
 * the Program under the AGPL does not imply a trademark license.
 * Therefore any rights, title and interest in our trademarks remain
 * entirely with us.
 * 
 * However, if you propagate an unmodified version of the Program you are
 * allowed to use the term "Zarafa" to indicate that you distribute the
 * Program. Furthermore you may use our trademarks where it is necessary
 * to indicate the intended purpose of a product or service provided you
 * use it in accordance with honest practices in industrial or commercial
 * matters.  If you want to propagate modified versions of the Program
 * under the name "Zarafa" or "Zarafa Server", you may only do so if you
 * have a written permission by Zarafa B.V. (to acquire a permission
 * please contact Zarafa at trademark@zarafa.com).
 * 
 * The interactive user interface of the software displays an attribution
 * notice containing the term "Zarafa" and/or the logo of Zarafa.
 * Interactive user interfaces of unmodified and modified versions must
 * display Appropriate Legal Notices according to sec. 5 of the GNU
 * Affero General Public License, version 3, when you propagate
 * unmodified or modified versions of the Program. In accordance with
 * sec. 7 b) of the GNU Affero General Public License, version 3, these
 * Appropriate Legal Notices must retain the logo of Zarafa or display
 * the words "Initial Development by Zarafa" if the display of the logo
 * is not reasonably feasible for technical reasons. The use of the logo
 * of Zarafa in Legal Notices is allowed for unmodified and modified
 * versions of the software.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#ifndef ECFIFOBUFFER_H
#define ECFIFOBUFFER_H

#include <list>
#include <pthread.h>

#include "ZarafaCode.h"

class ECFifoBlock;

#define EC_FB_SWCACHE_SIZE	8192	// Small Write Cache Size

// Thread safe buffer for FIFO operations
class ECFifoBuffer
{
public:
	ECFifoBuffer();
	ECFifoBuffer(unsigned int ulMaxSize);
	~ECFifoBuffer();
	
	// Writes cbBuf bytes of lpszBuf into the buffer.
	ECRESULT Write(const void *lpBuf, unsigned int cbBuf, unsigned int ulTimeoutMs);

	// Read upto cbBuf bytes from the buffer into lpszBuf.
	// lpcbRead will be set to the amount of bytes read.
	ECRESULT Read(void *lpBuf, unsigned int cbBuf, unsigned int ulTimeoutMs, unsigned int *lpcbRead);

	// Close the buffer causing the Read to abort when no data is left.
	ECRESULT Close();

	// Check if the buffer is closed
	bool IsClosed() const;

	// Check if the buffer is empty
	bool IsEmpty() const;

	// Get the size of the buffer
	unsigned long Size();
	
private:
	typedef std::list<ECFifoBlock*>		BlockList;
	typedef BlockList::iterator			BlockIter;

private:
	// prohibit copy
	ECFifoBuffer(const ECFifoBuffer &);
	ECFifoBuffer& operator=(const ECFifoBuffer &);
	
private:
	unsigned long	m_ulMaxSize;
	unsigned long 	m_ulSize;
	bool			m_bClosed;
	BlockList		m_sBlockList;

	// Small Write Cache
	char			*m_lpszSWCache;
	unsigned long	m_cbSWCache;

	pthread_mutex_t	m_hMutex;
	pthread_cond_t	m_hCond;
};


// inlines
inline bool ECFifoBuffer::IsClosed() const {
	return m_bClosed;
}

inline bool ECFifoBuffer::IsEmpty() const {
	return m_ulSize == 0;
}

inline unsigned long ECFifoBuffer::Size() {
	return m_ulSize;
}

#endif // ndef ECFIFOBUFFER_H
