<?php

require_once( 'PHPUnit.php' );
require_once( '../includes/Defines.php' );
require_once( '../includes/Profiling.php' );
require_once( '../includes/GlobalFunctions.php' );

class GlobalTest extends PHPUnit_TestCase {
	function GlobalTest( $name ) {
		$this->PHPUnit_TestCase( $name );
	}
	
	function setUp() {
		$this->save = array();
		$saveVars = array( 'wgReadOnlyFile' );
		foreach( $saveVars as $var ) {
			if( isset( $GLOBALS[$var] ) ) {
				$this->save[$var] = $GLOBALS[$var];
			}
		}
		$GLOBALS['wgReadOnlyFile'] = '/tmp/testReadOnly-' . mt_rand();
	}
	
	function tearDown() {
		foreach( $this->save as $var => $data ) {
			$GLOBALS[$var] = $data;
		}
	}
	
	function testDecodeLatin() {
		$this->assertEquals(
			"\xe9cole",
			do_html_entity_decode( '&eacute;cole', ENT_COMPAT, 'iso-8859-1' ) );
	}

	function testDecodeUnicode() {
		$this->assertEquals(
			"\xc3\xa9cole",
			do_html_entity_decode( '&eacute;cole', ENT_COMPAT, 'utf-8' ) );
	}

	function testRandom() {
		# This could hypothetically fail, but it shouldn't ;)
		$this->assertFalse(
			wfRandom() == wfRandom() );
	}
	
	function testUrlencode() {
		$this->assertEquals(
			"%E7%89%B9%E5%88%A5:Contributions/Foobar",
			wfUrlencode( "\xE7\x89\xB9\xE5\x88\xA5:Contributions/Foobar" ) );
	}
	
	function testUtf8Sequence1() {
		$this->assertEquals(
			'A',
			wfUtf8Sequence( 65 ) );
	}
	
	function testUtf8Sequence2() {
		$this->assertEquals(
			"\xc4\x88",
			wfUtf8Sequence( 0x108 ) );
	}

	function testUtf8Sequence3() {
		$this->assertEquals(
			"\xe3\x81\x8b",
			wfUtf8Sequence( 0x304b ) );
	}

	function testUtf8Sequence4() {
		$this->assertEquals(
			"\xf0\x90\x91\x90",
			wfUtf8Sequence( 0x10450 ) );
	}
	
	function testMungeToUtf8() {
		$this->assertEquals(
			"\xc4\x88io bonas dans l'\xc3\xa9cole!",
			wfMungeToUtf8( "&#x108;io bonas dans l'&#233;cole!" ) );
	}
	
	function testUtf8ToHTML() {
		$this->assertEquals(
			"&#264;io bonas dans l'&#233;cole!",
			wfUtf8ToHTML( "\xc4\x88io bonas dans l'\xc3\xa9cole!" ) );
	}
	
	function testReadOnlyEmpty() {
		$this->assertFalse( wfReadOnly() );
	}
	
	function testReadOnlySet() {
		$f = fopen( $GLOBALS['wgReadOnlyFile'], "wt" );
		fwrite( $f, 'Message' );
		fclose( $f );
		$this->assertTrue( wfReadOnly() );
		
		unlink( $GLOBALS['wgReadOnlyFile'] );
		$this->assertFalse( wfReadOnly() );
	}
	
	function testQuotedPrintable() {
		$this->assertEquals(
			"=?UTF-8?Q?=C4=88u=20legebla=3F?=",
			wfQuotedPrintable( "\xc4\x88u legebla?", "UTF-8" ) );
	}
	
	function testTime() {
		$start = wfTime();
		$this->assertType( 'double', $start );
		$end = wfTime();
		$this->assertTrue( $end > $start, "Time is running backwards!" );
	}
	
	function testArrayToCGI() {
		$this->assertEquals(
			"baz=AT%26T&foo=bar",
			wfArrayToCGI(
				array( 'baz' => 'AT&T', 'ignore' => '' ),
				array( 'foo' => 'bar', 'baz' => 'overridden value' ) ) );
	}
	
	function testMimeTypeMatch() {
		$this->assertEquals(
			'text/html',
			mimeTypeMatch( 'text/html',
				array( 'application/xhtml+xml' => 1.0,
				       'text/html'             => 0.7,
				       'text/plain'            => 0.3 ) ) );
		$this->assertEquals(
			'text/*',
			mimeTypeMatch( 'text/html',
				array( 'image/*' => 1.0,
				       'text/*'  => 0.5 ) ) );
		$this->assertEquals(
			'*/*',
			mimeTypeMatch( 'text/html',
				array( '*/*' => 1.0 ) ) );
		$this->assertNull(
			mimeTypeMatch( 'text/html',
				array( 'image/png'     => 1.0,
				       'image/svg+xml' => 0.5 ) ) );
	}
	
	function testNegotiateType() {
		$this->assertEquals(
			'text/html',
			wfNegotiateType(
				array( 'application/xhtml+xml' => 1.0,
				       'text/html'             => 0.7,
				       'text/plain'            => 0.5,
				       'text/*'                => 0.2 ),
				array( 'text/html'             => 1.0 ) ) );
		$this->assertEquals(
			'application/xhtml+xml',
			wfNegotiateType(
				array( 'application/xhtml+xml' => 1.0,
				       'text/html'             => 0.7,
				       'text/plain'            => 0.5,
				       'text/*'                => 0.2 ),
				array( 'application/xhtml+xml' => 1.0,
				       'text/html'             => 0.5 ) ) );
		$this->assertEquals(
			'text/html',
			wfNegotiateType(
				array( 'text/html'             => 1.0,
				       'text/plain'            => 0.5,
				       'text/*'                => 0.5,
				       'application/xhtml+xml' => 0.2 ),
				array( 'application/xhtml+xml' => 1.0,
				       'text/html'             => 0.5 ) ) );
		$this->assertEquals(
			'text/html',
			wfNegotiateType(
				array( 'text/*'                => 1.0,
				       'image/*'               => 0.7,
				       '*/*'                   => 0.3 ),
				array( 'application/xhtml+xml' => 1.0,
				       'text/html'             => 0.5 ) ) );
		$this->assertNull(
			wfNegotiateType(
				array( 'text/*'                => 1.0 ),
				array( 'application/xhtml+xml' => 1.0 ) ) );
	}
	
	function testTimestamp() {
		$t = gmmktime( 12, 34, 56, 1, 15, 2001 );
		$this->assertEquals(
			'20010115123456',
			wfTimestamp( TS_MW, $t ),
			'TS_UNIX to TS_MW' );
		$this->assertEquals(
			979562096,
			wfTimestamp( TS_UNIX, $t ),
			'TS_UNIX to TS_UNIX' );
		$this->assertEquals(
			'2001-01-15 12:34:56',
			wfTimestamp( TS_DB, $t ),
			'TS_UNIX to TS_DB' );
		
		$this->assertEquals(
			'20010115123456',
			wfTimestamp( TS_MW, '20010115123456' ),
			'TS_MW to TS_MW' );
		$this->assertEquals(
			979562096,
			wfTimestamp( TS_UNIX, '20010115123456' ),
			'TS_MW to TS_UNIX' );
		$this->assertEquals(
			'2001-01-15 12:34:56',
			wfTimestamp( TS_DB, '20010115123456' ),
			'TS_MW to TS_DB' );
		
		$this->assertEquals(
			'20010115123456',
			wfTimestamp( TS_MW, '2001-01-15 12:34:56' ),
			'TS_DB to TS_MW' );
		$this->assertEquals(
			979562096,
			wfTimestamp( TS_UNIX, '2001-01-15 12:34:56' ),
			'TS_DB to TS_UNIX' );
		$this->assertEquals(
			'2001-01-15 12:34:56',
			wfTimestamp( TS_DB, '2001-01-15 12:34:56' ),
			'TS_DB to TS_DB' );
	}
	
	/* TODO: many more! */
}

?>