\name{EEF.profile}
\alias{EEF.profile}
\alias{EL.profile}
\title{ Empirical Likelihoods}
\description{
 Construct the empirical log likelihood or empirical exponential 
family log likelihood for a mean.}
\usage{
EEF.profile(y, tmin=min(y) + 0.1, tmax=max(y) - 0.1, n.t=25, 
            u=function(y, t) { y-t})
EL.profile(y, tmin = min(y) + 0.1, tmax = max(y) - 0.1, n.t = 25, 
           u = function(y, t) y - t)
}
\arguments{
  \item{y}{A vector or matrix of data}
  \item{tmin}{ The minimum value of the range over which the 
    likelihood should be computed.  This must be larger than 
    \code{min(y)}.}
  \item{tmax}{ The maximum value of the range over which the 
    likelihood should be computed.  This must be smaller than 
    \code{max(y)}.}
  \item{n.t}{ The number of points between \code{tmin} and
    \code{tmax} at which the value of the log-liklihood should be
    computed.}
  \item{u}{A function of the data and the parameter.}
}
\details{
These functions calculate the log likelihood for a mean using either
an empirical likelihood or an empirical exponential family likelihood.
They are supplied as part of the package \code{boot} for demonstration
purposes with the practicals in chapter 10 of Davison and Hinkley (1997).
The functions are not intended for general use and are not supported
as part of the \code{boot}package.  For more general and more robust
code to calculate empirical likelihoods see Professor A. B. Owen's 
empirical likelihood home page at the URL
\url{http://www-stat.stanford.edu/~owen/empirical/}.}
\value{
A matrix with \code{n.t} rows.  The first column contains the
values of the parameter used.  The second colmn of the output
of \code{EL.profile} contains the values of the empirical
log likelihood. The second and third columns of the output of
\code{EEF.profile} contain two versions of the empirical
exponential family log-likelihood.  The final column of the
output matrix contains the values of the Lagrange multiplier
used in the optimization procedure.
}
\references{ 
Davison, A. C. and Hinkley, D. V. (1997)
\emph{Bootstrap Methods and Their Application}. Cambridge University
Press.
}
\author{Angelo J. Canty}
\keyword{htest}

\eof
\name{Imp.Estimates}
\alias{Imp.Estimates}
\alias{imp.moments}
\alias{imp.prob}
\alias{imp.quantile}
\alias{imp.reg}
\title{
Importance Sampling Estimates
}
\description{
Central moment, tail probability, and quantile estimates for a statistic
under importance resampling.
}
\usage{
imp.moments(boot.out=NULL, index=1, t=boot.out$t[, index], 
         w=NULL, def=TRUE, q=NULL)
imp.prob(boot.out=NULL, index=1, t0=boot.out$t0[index], 
         t=boot.out$t[, index], w=NULL, def=TRUE, q=NULL)
imp.quantile(boot.out=NULL, alpha=NULL, index=1, 
         t=boot.out$t[, index], w=NULL, def=TRUE, q=NULL)
}
\arguments{
\item{boot.out}{
A object of class \code{"boot"} generated by a call to \code{boot} or \code{tilt.boot}. 
Use of these functions makes sense only when the bootstrap resampling used
unequal weights for the observations.  If the importance weights \code{w} are not 
supplied then \code{boot.out} is a required argument. It is also required if
\code{t} is not supplied.
}
\item{alpha}{
The alpha levels for the required quantiles.  The default is to calculate
the 1\%, 2.5\%, 5\%, 10\%, 90\%, 95\%, 97.5\% and 99\% quantiles.
}
\item{index}{
The index of the variable of interest in the output of \code{boot.out$statistic}.
This is not used if the argument \code{t} is supplied.
}
\item{t0}{
The values at which tail probability estimates are required.  For each
value \code{t0[i]} the function will estimate the bootstrap cdf evaluated at
\code{t0[i]}.  If \code{imp.prob} is called without the argument \code{t0} then the bootstrap
cdf evaluated at the observed value of the statistic is found.
}
\item{t}{
The bootstrap replicates of a statistic.  By default these are taken from
the bootstrap output object \code{boot.out} but they can be supplied separately
if required (e.g. when the statistic of interest is a function of the
calculated values in \code{boot.out}).  Either \code{boot.out} or \code{t} must be supplied.
}
\item{w}{
The importance resampling weights for the bootstrap replicates.  If they are
not supplied then \code{boot.out} must be supplied, in
which case the importance weights are calculated by a call to \code{imp.weights}.
}
\item{def}{
A logical value indicating whether a defensive mixture is to be used for weight
calculation.  This is used only if \code{w} is missing and it is passed unchanged 
to \code{imp.weights} to calculate \code{w}.
}
\item{q}{
A vector of probabilities specifying the resampling distribution from which 
any estimates should be found.  In general this would correspond to the usual
bootstrap resampling distribution which gives equal weight to each of the 
original observations. The estimates depend on this distribution only through 
the importance weights \code{w} so this argument is ignored if \code{w} is supplied.  If
\code{w} is missing then \code{q} is passed as an argument to \code{imp.weights} and used to
find \code{w}.
}}
\value{
A list with the following components :



\item{alpha}{
The \code{alpha} levels used for the quantiles, if \code{imp.quantile} is used.
}
\item{t0}{
The values at which the tail probabilities are estimated, if \code{imp.prob}
is used.
}
\item{raw}{
The raw importance resampling estimates.  For \code{imp.moments} this has length 2,
the first component being the estimate of the mean and the second being the
variance estimate.  For \code{imp.prob}, \code{raw} is of the same length as \code{t0}, and
for \code{imp.quantile} it is of the same length as \code{alpha}.
}
\item{rat}{
The ratio importance resampling estimates.  In this method the weights \code{w} are
rescaled to have average value one before they are used.  The format of this
vector is the same as \code{raw}.
}
\item{reg}{
The regression importance resampling estimates.  In this method the weights
which are used are derived from a regression of \code{t*w} on \code{w}.  This choice of
weights can be shown to minimize the variance of the weights and also the
Euclidean distance of the weights from the uniform weights.  The format of this 
vector is the same as \code{raw}.
}}
\references{
Davison, A. C. and Hinkley, D. V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Hesterberg, T. (1995) Weighted average importance sampling and defensive 
mixture distributions. \emph{Technometrics}, \bold{37}, 185--194.


Johns, M.V.  (1988) Importance sampling for bootstrap confidence intervals.
\emph{Journal of the American Statistical Association}, \bold{83}, 709--714.
}
\seealso{
\code{\link{boot}}, \code{\link{exp.tilt}}, \code{\link{imp.weights}}, \code{\link{smooth.f}}, \code{\link{tilt.boot}}
}
\examples{
# Example 9.8 of Davison and Hinkley (1997) requires tilting the 
# resampling distribution of the studentized statistic to be centred 
# at the observed value of the test statistic, 1.84.  In this example
# we show how certain estimates can be found using resamples taken from
# the tilted distribution.
data(gravity)
grav1 <- gravity[as.numeric(gravity[,2])>=7,]
grav.fun <- function(dat, w, orig)
{    strata <- tapply(dat[, 2], as.numeric(dat[, 2]))
     d <- dat[, 1]
     ns <- tabulate(strata)
     w <- w/tapply(w, strata, sum)[strata]
     mns <- tapply(d * w, strata, sum)
     mn2 <- tapply(d * d * w, strata, sum)
     s2hat <- sum((mn2 - mns^2)/ns)
     as.vector(c(mns[2]-mns[1],s2hat,(mns[2]-mns[1]-orig)/sqrt(s2hat)))
}
grav.z0 <- grav.fun(grav1,rep(1,26),0)
grav.L <- empinf(data=grav1, statistic=grav.fun, stype="w", 
                 strata=grav1[,2], index=3, orig=grav.z0[1])
grav.tilt <- exp.tilt(grav.L,grav.z0[3],strata=grav1[,2])
grav.tilt.boot <- boot(grav1, grav.fun, R=199, stype="w", 
                       strata=grav1[,2], weights=grav.tilt$p,
                       orig=grav.z0[1])
# Since the weights are needed for all calculations, we shall calculate
# them once only.
grav.w <- imp.weights(grav.tilt.boot)
grav.mom <- imp.moments(grav.tilt.boot, w=grav.w, index=3)
grav.p <- imp.prob(grav.tilt.boot, w=grav.w, index=3, t0=grav.z0[3])
grav.q <- imp.quantile(grav.tilt.boot, w=grav.w, index=3, 
                       alpha=c(0.9,0.95,0.975,0.99))
}
\keyword{htest}
\keyword{nonparametric}
% Converted by Sd2Rd version 1.15.

\eof
\name{abc.ci}
\alias{abc.ci}
\title{
Nonparametric ABC Confidence Intervals
}
\description{
Calculate equi-tailed two-sided nonparametric approximate bootstrap confidence 
intervals for a parameter, given a set of data and an estimator of the 
parameter, using numerical differentiation.  
}
\usage{
abc.ci(data, statistic, index=1, strata=rep(1, n), conf=0.95, 
       eps=0.001/n, \dots)
}
\arguments{
\item{data}{
A data set expressed as a vector, matrix or data frame.
}
\item{statistic}{
A function which returns the statistic of interest.  The function must
take at least 2 arguments; the first argument should be the data and the
second a vector of weights.  The weights passed to \code{statistic} will be 
normalized to sum to 1 within each stratum.  Any other arguments should be 
passed to \code{abc.ci} as part of the \code{\dots{}} argument.  
}
\item{index}{
If \code{statistic} returns a vector of length greater than 1, then this indicates
the position of the variable of interest within that vector.
}
\item{strata}{
A factor or numerical vector indicating to which sample each
observation belongs in multiple sample problems.  The default 
is the one-sample case.  
}
\item{conf}{
A scalar or vector containing the confidence level(s) of the required
interval(s).
}
\item{eps}{
The value of epsilon to be used for the numerical differentiation.
}
\item{...}{
Any other arguments for \code{statistic}.  These will be passed unchanged to
\code{statistic} each time it is called within \code{abc.ci}.
}}
\value{
A \code{length(conf)} by 3 matrix where each row contains the confidence level 
followed by the lower and upper end-points of the ABC interval at that
level.
}
\details{
This function is based on the function \code{abcnon} written by R. Tibshirani.
A listing of the original function is available in DiCiccio and Efron (1996).
The function uses numerical differentiation for the first and second
derivatives of the statistic and then uses these values to approximate
the bootstrap BCa intervals.  The total number of evaluations of the
statistic is \code{2*n+2+2*length(conf)} where \code{n} is the number of data points
(plus calculation of the original value of the statistic).  The function 
works for the multiple sample case
without the need to rewrite the statistic in an artificial form since
the stratified normalization is done internally by the function.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}, Chapter 5. 
Cambridge University Press. 


DiCiccio, T. J. and Efron B. (1992) More accurate confidence intervals in 
exponential families. \emph{Biometrika}, \bold{79}, 231--245.


DiCiccio, T. J.  and Efron  B. (1996) Bootstrap confidence intervals (with
Discussion). 
\emph{Statistical Science}, \bold{11}, 189--228.
}
\seealso{
\code{\link{boot.ci}}
}
\examples{
# 90\% and 95\% confidence intervals for the correlation 
# coefficient between the columns of the bigcity data


data(bigcity)
abc.ci(bigcity, corr, conf=c(0.90,0.95))


data(gravity)
# A 95\% confidence interval for the difference between the means of
# the last two samples in gravity
mean.diff <- function(y, w)
{    gp1 <- 1:table(as.numeric(y$series))[1]
     sum(y[gp1,1] * w[gp1]) - sum(y[-gp1,1] * w[-gp1])
}
grav1 <- gravity[as.numeric(gravity[,2])>=7,]
abc.ci(grav1, mean.diff, strata=grav1$series)


}
\keyword{nonparametric}
\keyword{htest}
% Converted by Sd2Rd version 1.15.

\eof
\name{acme}
\alias{acme}
\title{
Monthly Excess Returns
}
\description{
The \code{acme} data frame has 60 rows and 3 columns.


The excess return for the Acme Cleveland Corporation are recorded along with
those for all stocks listed on the New York and American Stock Exchanges were
recorded over a five year period.  These excess returns are relative to the
return on a risk-less investment such a U.S. Treasury bills.  
}
\usage{
data(acme)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{month}}{
A character string representing the month of the observation.
}
\item{\code{market}}{
The excess return of the market as a whole.
}
\item{\code{acme}}{
The excess return for the Acme Cleveland Corporation.
}}}
\source{
The data were obtained from


Simonoff, J.S. and Tsai, C.-L. (1994) Use of modified profile likelihood for 
improved tests of constancy of variance in regression. 
\emph{Applied Statistics}, \bold{43}, 353--370.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) \emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{aids}
\alias{aids}
\title{
Delay in AIDS Reporting in England and Wales
}
\description{
The \code{aids} data frame has 570 rows and 6 columns.


Although all cases of AIDS in England and Wales must be reported to the
Communicable Disease Surveillance Centre, there is often a considerable delay
between the time of diagnosis and the time that it is reported.  In estimating
the prevalence of AIDS, account must be taken of the unknown number of cases
which have been diagnosed but not reported.  The data set here records the
reported cases of AIDS diagnosed from July 1983 and until the end of 1992.  
The data are cross-classified by the date of diagnosis and the time delay in 
the reporting of the cases.  
}
\usage{
data(aids)
}
\format{
This data frame contains the following columns:
\describe{
\item{\code{year}}{
The year of the diagnosis.
}
\item{\code{quarter}}{
The quarter of the year in which diagnosis was made.
}
\item{\code{delay}}{
The time delay (in months) between diagnosis and reporting. 0 means that the
case was reported within one month.  Longer delays are grouped in 3 month
intervals and the value of \code{delay} is the midpoint of the interval (therefore
a value of \code{2} indicates that reporting was delayed for between 1 and 3 
months).
}
\item{\code{dud}}{
An indicator of censoring. These are categories for which full information is
not yet available and the number recorded is a lower bound only.
}
\item{\code{time}}{
The time interval of the diagnosis. That is the number of quarters from July
1983 until the end of the quarter in which these cases were diagnosed.
}
\item{\code{y}}{
The number of AIDS cases reported.
}}}
\source{
The data were obtained from


De Angelis, D. and Gilks, W.R. (1994) Estimating acquired immune
deficiency syndrome accounting for reporting delay.
\emph{Journal of the Royal Statistical Society, A}, \bold{157}, 31--40.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997)
\emph{Bootstrap Methods and Their Application}.
Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{aircondit}
\alias{aircondit}
\alias{aircondit7}
\title{
Failures of Air-conditioning Equipment
}
\description{
Proschan (1963) reported on the times between failures of the air-conditioning
equipment in 10 Boeing 720 aircraft. The \code{aircondit} data frame contains 
the intervals for the ninth aircraft while \code{aircondit7} contains those for the
seventh aircraft. 


Both data frames have just one column.  Note that the data have been sorted 
into increasing order.
}
\usage{
data(aircondit)
}
\format{
The data frames contain the following column:


\describe{
\item{\code{hours}}{
The time interval in hours between successive failures of the air-conditioning
equipment
}}}
\source{
The data were taken from


Cox, D.R. and Snell, E.J. (1981) 
\emph{Applied Statistics: Principles and Examples}. Chapman and Hall.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Proschan, F. (1963) Theoretical explanation of observed decreasing failure 
rate. \emph{Technometrics}, \bold{5}, 375-383.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{amis}
\alias{amis}
\title{
Car Speeding and Warning Signs
}
\description{
The \code{amis} data frame has 8437 rows and 4 columns.


In a study into the effect that warning signs have on speeding patterns, 
Cambridgeshire County Council considered 14 pairs of locations.
The locations were paired to account for factors such as traffic volume
and type of road.  One site in each pair had a sign erected warning of the
dangers of speeding and asking drivers to slow down.  No action was taken at
the second site.  Three sets of measurements were taken at each site.  Each set
of measurements was nominally of the speeds of 100 cars but not all sites have
exactly 100 measurements. These speed measurements were taken before the 
erection of the sign, shortly after the erection of the sign, and again after 
the sign had been in place for some time.  
}
\usage{
data(amis)
}
\format{
This data frame contains the following columns:
\describe{
\item{\code{speed}}{
Speeds of cars (in miles per hour).
}
\item{\code{period}}{
A numeric column indicating the time that the reading was taken.
A value of 1 indicates a reading taken before the sign was erected, 
a 2 indicates a reading
taken shortly after erection of the sign and a 3 indicates a reading taken 
after the sign had been in place for some time.
}
\item{\code{warning}}{
A numeric column indicating whether the location of the reading was chosen to
have a warning sign erected. A value of 1 indicates presence of a sign and a
value of 2 indicates that no sign was erected.
}
\item{\code{pair}}{
A numeric column giving the pair number at which the reading was taken. Pairs
were numbered from 1 to 14.
}}}
\source{
The data were kindly made available by Mr. Graham Amis, Cambridgeshire County
Council, U.K.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{aml}
\alias{aml}
\title{
Remission Times for Acute Myelogenous Leukaemia
}
\description{
The \code{aml} data frame has 23 rows and 3 columns.


A clinical trial to evaluate the efficacy of maintenance chemotherapy for
acute myelogenous leukaemia was 
conducted by Embury et al. (1977) at Stanford University.  After reaching a
stage of remission through treatment by chemotherapy, patients were randomized
into two groups. The first group received maintenance chemotherapy and the 
second group did not.  The aim of the study was to see if maintenance 
chemotherapy increased the length of the remission.  The data here formed a
preliminary analysis which was conducted in October 1974.
}
\usage{
data(aml)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{time}}{
The length of the complete remission (in weeks).
}
\item{\code{cens}}{
An indicator of right censoring.  1 indicates that the patient had a relapse
and so \code{time} is the length of the remission. 0 indicates that the patient
had left the study or was still in remission in October 1974, that is the
length of remission is right-censored.
}
\item{\code{group}}{
The group into which the patient was randomized.  Group 1 received
maintenance chemotherapy, group 2 did not.
}}}
\source{
The data were obtained from


Miller, R.G. (1981) \emph{Survival Analysis}. John Wiley.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Embury, S.H, Elias, L., Heller, P.H., Hood, C.E., Greenberg, P.L. and
Schrier, S.L. (1977) Remission maintenance therapy in  acute myelogenous
leukaemia. \emph{Western Journal of Medicine}, \bold{126}, 267-272.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{beaver}
\alias{beaver}
\title{
Beaver Body Temperature Data
}
\description{
The \code{beaver} data frame has 100 rows and 4 columns.  It is a multivariate
time series of class \code{"ts"} and also inherits from class \code{"data.frame"}.

This data set is part of a long study into body temperature regulation in
beavers.  Four adult female beavers were live-trapped and had a 
temperature-sensitive radiotransmitter surgically implanted.  Readings were
taken every 10 minutes.  The location of the beaver was also recorded and
her activity level was dichotomized by whether she was in the retreat or 
outside of it since high-intensity activities only occur outside of the 
retreat.

The data in this data frame are those readings for one of the beavers on a day
in autumn.  
}
\usage{
data(beaver)
}
\format{
  This data frame contains the following columns:
  \describe{
\item{\code{day}}{
The day number.  The data includes only data from day 307 and early 308.
}
\item{\code{time}}{
The time of day formatted as hour-minute.
}
\item{\code{temp}}{
The body temperature in degrees Celsius.
}
\item{\code{activ}}{
The dichotomized activity indicator. \code{1} indicates that the beaver is outside
of the retreat and therefore engaged in high-intensity activity.
}}}
\source{
The data were obtained from

Reynolds, P.S. (1994) Time-series analyses of beaver body temperatures.
In \emph{Case Studies in Biometry}. N. Lange, L. Ryan, L. Billard, 
D. Brillinger, L. Conquest and J. Greenhouse (editors), 211--228. John Wiley.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{bigcity}
\alias{bigcity}
\alias{city}
\title{
Population of U.S. Cities
}
\description{
The \code{bigcity} data frame has 49 rows and 2 columns. 


The \code{city} data frame has 10 rows and 2 columns.  


The measurements are the
population (in 1000's) of 49 U.S. cities in 1920 and 1930.  The 49 cities are 
a random sample taken from the 196 largest cities in 1920.  The \code{city} data 
frame consists of the first 10 observations in \code{bigcity}. 
}
\usage{
data(bigcity)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{u}}{
The 1920 population.
}
\item{\code{x}}{
The 1930 population.
}}}
\source{
The data were obtained from


Cochran, W.G. (1977) \emph{Sampling Techniques}. Third edition. John Wiley
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) \emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{boot-practicals}
\alias{nested.corr}
\alias{lik.CI}
\title{ Functions for Bootstrap Practicals}
\description{
 Functions for use with the practicals in Davison and Hinkley (1997)}
\usage{
nested.corr(data, w, t0, M)
lik.CI(like, lim)
}
\details{
\code{nested.corr} is meant for use with the double bootstrap
in practical 5.5 of Davison and Hinkley (1997).

\code{lik.CI} is meant for use with practicals 10.1 and 10.2 of
Davison and Hinkley (1997). 
}
\references{ 
Davison, A. C. and Hinkley, D. V. (1997)
\emph{Bootstrap MEthods and Their Application}. Cambridge University
Press.
}
\author{ Angelo J. Canty }
\keyword{internal}

\eof
\name{boot}
\alias{boot}
\alias{boot.return}
\title{
Bootstrap Resampling 
}
\description{
Generate \code{R} bootstrap replicates of a statistic applied to data.  Both
parametric and nonparametric resampling are possible.  For the nonparametric
bootstrap, possible resampling methods are the ordinary bootstrap, the 
balanced bootstrap, antithetic resampling, and permutation.
For nonparametric multi-sample problems stratified resampling is used.  
This is specified by including a vector of strata in the call to boot.
Importance resampling weights may be specified.
}
\usage{
boot(data, statistic, R, sim="ordinary", stype="i", 
     strata=rep(1,n), L=NULL, m=0, weights=NULL, 
     ran.gen=function(d, p) d, mle=NULL, \dots)
}
\arguments{
\item{data}{
The data as a vector, matrix or data frame.  If it is a matrix or data frame
then each row is considered as one multivariate observation.
}
\item{statistic}{
A function which when applied to data returns a vector containing the
statistic(s) of interest.  When \code{sim="parametric"}, the first argument to
\code{statistic} must be the data.  For each replicate a simulated dataset returned
by \code{ran.gen} will be passed.  In all other cases \code{statistic} must take at
least two
arguments.  The first argument passed will always be the original data. The
second will be a vector of
indices, frequencies or weights which define the bootstrap sample.
Further, if predictions are required, then a third argument is required
which would be a vector of the random indices used to generate the
bootstrap predictions.  Any further arguments can be passed to \code{statistic} 
through the \code{\dots{}} argument.
}
\item{R}{
The number of bootstrap replicates.  Usually this will be a
single positive integer.  For importance resampling, some
resamples may use one set of weights and others use a different set of weights.
In this case \code{R} would be a vector of integers where each component gives the
number of resamples from each of the rows of weights.
}
\item{sim}{
A character string indicating the type of simulation required.  Possible
values are \code{"ordinary"} (the default), \code{"parametric"}, \code{"balanced"},
\code{"permutation"},
or \code{"antithetic"}.  Importance resampling is specified by including importance
weights; the type of importance resampling must still be specified but may only
be \code{"ordinary"} or \code{"balanced"} in this case.  
}
\item{stype}{
A character string indicating what the second argument of statistic represents.
Possible values of stype are \code{"i"} (indices - the default), \code{"f"} (frequencies),
or \code{"w"} (weights).
}
\item{strata}{
An integer vector or factor specifying the strata for multi-sample problems.  
This may be specified for any simulation, but is ignored when 
\code{sim} is \code{"parametric"}. When \code{strata} is supplied for a nonparametric
bootstrap, the simulations are done within the specified strata.
}
\item{L}{
Vector of influence values evaluated at the observations.  This is used only
when \code{sim} is \code{"antithetic"}.  If not supplied, they are calculated 
through a call to \code{empinf}.  This will use the infinitesimal jackknife
provided that \code{stype} is \code{"w"}, otherwise the usual jackknife is used.
}
\item{m}{
The number of predictions which are to be made at each bootstrap replicate.
This is most useful for (generalized) linear models.  This can only be used
when \code{sim} is \code{"ordinary"}.  \code{m} will usually be a
single integer but, if there are strata, it may be a vector with length
equal to the number of strata, specifying how many of the errors for prediction
should come from each strata.  The actual predictions should be 
returned as the final part of the output of \code{statistic}, which should also
take a vector of indices of the errors to be used for the predictions.
}
\item{weights}{
Vector or matrix of importance weights. If a vector then it should have as many 
elements as there are observations in \code{data}.  When simulation from more than 
one set of weights is required, \code{weights} should be a matrix where each row of 
the matrix is one set of importance weights.  If \code{weights} is a matrix then \code{R} 
must be a vector of length \code{nrow(weights)}.  This parameter is ignored if \code{sim} 
is not \code{"ordinary"} or \code{"balanced"}. 
}
\item{ran.gen}{
This function is used only when \code{sim} is \code{"parametric"} when it
describes how random values are to be generated.  It should be a function of
two arguments.  The first argument should be the observed data and the
second argument consists of any other information needed (e.g. parameter
estimates).  The second argument may be a list, allowing any number of items
to be passed to \code{ran.gen}.  The returned value should be a simulated
data set of the same form as the observed data which will be passed to
statistic to get a bootstrap replicate.  It is important that the returned
value be of the same shape and type as the original dataset.  If \code{ran.gen}
is not specified, the default is a function which returns the original \code{data} 
in which case all simulation should be included as part of \code{statistic}.  Use
of \code{sim="parametric"} with a suitable \code{ran.gen} allows the user to implement
any types of nonparametric resampling which are not supported directly.
}
\item{mle}{
The second argument to be passed to \code{ran.gen}.  Typically these will be maximum
likelihood estimates of the parameters.  For efficiency \code{mle} is often a list
containing all of the objects needed by \code{ran.gen} which can be calculated 
using the original data set only.
}
\item{...}{
Any other arguments for \code{statistic} which are passed unchanged
each time it is called.  Any such arguments to \code{statistic} must follow the 
arguments which \code{statistic} is required to have for the simulation.
}}
\value{
The returned value is an object of class \code{"boot"}, containing the following 
components :

\item{t0}{
The observed value of \code{statistic} applied to \code{data}. 
}
\item{t}{
A matrix with \code{R} rows each of which is a bootstrap replicate of \code{statistic}.
}
\item{R}{
The value of \code{R} as passed to \code{boot}.
}
\item{data}{
The \code{data} as passed to \code{boot}.
}
\item{seed}{
The value of \code{.Random.seed} when \code{boot} was called.  
}
\item{statistic}{
The function \code{statistic} as passed to \code{boot}.
}
\item{sim}{
Simulation type used.
}
\item{stype}{
Statistic type as passed to \code{boot}.
}
\item{call}{
The original call to \code{boot}.
}
\item{strata}{
The strata used.  This is the vector passed to \code{boot}, if it was supplied
or a vector of ones if there were no strata.  It is not returned if \code{sim} is 
\code{"parametric"}.
}
\item{weights}{
The importance sampling weights as passed to \code{boot} or the empirical 
distribution
function weights if no importance sampling weights were specified.  It is 
omitted if \code{sim} is not one of \code{"ordinary"} or \code{"balanced"}.
}
\item{pred.i}{
If predictions are required (\code{m>0}) this is the matrix of indices at which
predictions were calculated as they were passed to statistic.  Omitted if
\code{m} is \code{0} or \code{sim} is not \code{"ordinary"}.
}
\item{L}{
The influence values used when \code{sim} is \code{"antithetic"}.  If no such values were
specified and \code{stype} is not \code{"w"} then \code{L} is returned as consecutive integers 
corresponding to the assumption that data is ordered by influence values.
This component is omitted when \code{sim} is not \code{"antithetic"}.
}
\item{ran.gen}{
The random generator function used if \code{sim} is \code{"parametric"}. This component 
is omitted for any other value of \code{sim}.
}
\item{mle}{
The parameter estimates passed to \code{boot} when \code{sim} is \code{"parametric"}.  It is 
omitted for all other values of \code{sim}.
}}
\details{
The statistic to be bootstrapped can be as simple or complicated as desired
as long as its arguments correspond to the dataset and (for a nonparametric
bootstrap) a vector of indices, frequencies or weights.  \code{statistic} is treated
as a black box by the \code{boot} function and is not checked to ensure that these
conditions are met.  


The first order balanced bootstrap is
described in Davison, Hinkley and Schechtman (1986).  The antithetic bootstrap
is described by Hall (1989) and is experimental, particularly
when used with strata.  The other non-parametric
simulation types are the ordinary bootstrap (possibly with unequal 
probabilities), and permutation which returns random permutations of cases.
All of these methods work independently within strata
if that argument is supplied.


For the parametric bootstrap it is necessary for the user to specify how the
resampling is to be conducted.  The best way of accomplishing this is to 
specify the function \code{ran.gen} which will return a simulated data set from the
observed data set and a set of parameter estimates specified in \code{mle}.
}
\references{
There are many references explaining the bootstrap and its variations.
Among them are :

Booth, J.G., Hall, P. and Wood, A.T.A. (1993) Balanced importance resampling 
for the bootstrap. \emph{Annals of Statistics}, \bold{21}, 286--298.

Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.

Davison, A.C., Hinkley, D.V. and Schechtman, E. (1986) Efficient bootstrap 
simulation. \emph{Biometrika}, \bold{73}, 555--566.

Efron, B. and Tibshirani, R. (1993) \emph{An Introduction to the Bootstrap}.
Chapman & Hall.

Gleason, J.R. (1988) Algorithms for balanced bootstrap simulations.
\emph{ American Statistician}, \bold{42}, 263--266.

Hall, P. (1989) Antithetic resampling for the bootstrap. \emph{Biometrika},
\bold{73}, 713--724.

Hinkley, D.V. (1988) Bootstrap methods (with Discussion). 
\emph{Journal of the  Royal Statistical Society, B}, \bold{50}, 312--337, 355--370.

Hinkley, D.V. and Shi, S. (1989) Importance sampling and the nested bootstrap.
\emph{Biometrika}, \bold{76}, 435--446.

Johns M.V. (1988) Importance sampling for bootstrap confidence intervals.
\emph{Journal of the American Statistical Association}, \bold{83}, 709--714.

Noreen, E.W. (1989) \emph{Computer Intensive Methods for Testing Hypotheses}. 
John Wiley & Sons.
}
\seealso{
  \code{\link{boot.array}}, \code{\link{boot.ci}},
  \code{\link{censboot}}, \code{\link{empinf}},
  \code{\link{jack.after.boot}}, \code{\link{tilt.boot}}, \code{\link{tsboot}}
}
\examples{
# usual bootstrap of the ratio of means using the city data
data(city)
ratio <- function(d, w)
     sum(d$x * w)/sum(d$u * w)
boot(city, ratio, R=999, stype="w")


# Stratified resampling for the difference of means.  In this
# example we will look at the difference of means between the final
# two series in the gravity data.
data(gravity)
diff.means <- function(d, f)
{    n <- nrow(d)
     gp1 <- 1:table(as.numeric(d$series))[1]
     m1 <- sum(d[gp1,1] * f[gp1])/sum(f[gp1])
     m2 <- sum(d[-gp1,1] * f[-gp1])/sum(f[-gp1])
     ss1 <- sum(d[gp1,1]^2 * f[gp1]) - 
            (m1 *  m1 * sum(f[gp1]))
     ss2 <- sum(d[-gp1,1]^2 * f[-gp1]) - 
            (m2 *  m2 * sum(f[-gp1]))
     c(m1-m2, (ss1+ss2)/(sum(f)-2))
}
grav1 <- gravity[as.numeric(gravity[,2])>=7,]
boot(grav1, diff.means, R=999, stype="f", strata=grav1[,2])


#  In this example we show the use of boot in a prediction from 
#  regression based on the nuclear data.  This example is taken 
#  from Example 6.8 of Davison and Hinkley (1997).  Notice also 
#  that two extra arguments to statistic are passed through boot.
data(nuclear)
nuke <- nuclear[,c(1,2,5,7,8,10,11)]
nuke.lm <- glm(log(cost)~date+log(cap)+ne+ ct+log(cum.n)+pt, data=nuke)
nuke.diag <- glm.diag(nuke.lm)
nuke.res <- nuke.diag$res*nuke.diag$sd
nuke.res <- nuke.res-mean(nuke.res)


#  We set up a new data frame with the data, the standardized 
#  residuals and the fitted values for use in the bootstrap.
nuke.data <- data.frame(nuke,resid=nuke.res,fit=fitted(nuke.lm))


#  Now we want a prediction of plant number 32 but at date 73.00
new.data <- data.frame(cost=1, date=73.00, cap=886, ne=0,
                       ct=0, cum.n=11, pt=1)
new.fit <- predict(nuke.lm, new.data)


nuke.fun <- function(dat, inds, i.pred, fit.pred, x.pred)
{
     assign(".inds", inds, envir=.GlobalEnv)
     lm.b <- glm(fit+resid[.inds] ~date+log(cap)+ne+ct+
          log(cum.n)+pt, data=dat)
     pred.b <- predict(lm.b,x.pred)
     remove(".inds", envir=.GlobalEnv)
     c(coef(lm.b), pred.b-(fit.pred+dat$resid[i.pred]))
}


nuke.boot <- boot(nuke.data, nuke.fun, R=999, m=1, 
     fit.pred=new.fit, x.pred=new.data)
#  The bootstrap prediction error would then be found by
mean(nuke.boot$t[,8]^2)
#  Basic bootstrap prediction limits would be
new.fit-sort(nuke.boot$t[,8])[c(975,25)]




#  Finally a parametric bootstrap.  For this example we shall look 
#  at the air-conditioning data.  In this example our aim is to test 
#  the hypothesis that the true value of the index is 1 (i.e. that 
#  the data come from an exponential distribution) against the 
#  alternative that the data come from a gamma distribution with
#  index not equal to 1.
air.fun <- function(data)
{    ybar <- mean(data$hours)
     para <- c(log(ybar),mean(log(data$hours)))
     ll <- function(k) {
          if (k <= 0) out <- 1e200 # not NA
          else out <- lgamma(k)-k*(log(k)-1-para[1]+para[2])
         out
     }
     khat <- nlm(ll,ybar^2/var(data$hours))$estimate
     c(ybar, khat)
}


air.rg <- function(data, mle)
#  Function to generate random exponential variates.  mle will contain 
#  the mean of the original data
{    out <- data
     out$hours <- rexp(nrow(out), 1/mle)
     out
}

data(aircondit)
air.boot <- boot(aircondit, air.fun, R=999, sim="parametric",
     ran.gen=air.rg, mle=mean(aircondit$hours))


# The bootstrap p-value can then be approximated by
sum(abs(air.boot$t[,2]-1) > abs(air.boot$t0[2]-1))/(1+air.boot$R)
}
\keyword{nonparametric}
\keyword{htest}
% Converted by Sd2Rd version 1.15.

\eof
\name{boot.array}
\alias{boot.array}
\title{
Bootstrap Resampling Arrays
}
\description{
This function takes a bootstrap object calculated by one of the functions
\code{boot}, \code{censboot}, or \code{tilt.boot} and returns the frequency (or index)
array for the the bootstrap resamples.
}
\usage{
boot.array(boot.out, indices=<<see below>>)
}
\arguments{
\item{boot.out}{
An object of class \code{"boot"} returned by one of the generation functions for
such an object.
}
\item{indices}{
A logical argument which specifies whether to return the frequency array
or the raw index array.  The default is \code{indices=F} unless \code{boot.out} was 
created by \code{tsboot} in which case the default is \code{indices=T}.
}}
\value{
A matrix with \code{boot.out$R} rows and \code{n} columns where \code{n} is the number of
observations in \code{boot.out$data}.  If \code{indices} is \code{FALSE} then this will give
the frequency of each of the original observations in each bootstrap resample.
If \code{indices} is \code{TRUE} it will give the indices of the bootstrap resamples in
the order in which they would have been passed to the statistic.  
}
\section{Side Effects}{
This function temporarily resets \code{.Random.seed} to the value in \code{boot.out$seed}
and then returns it to its original value at the end of the function.
}
\details{
The process by which the original index array was generated is repeated with
the same value of \code{.Random.seed}.  If the frequency array is required then 
\code{freq.array} is called to convert the index array to a frequency array. 


A resampling array can only be returned when such a concept makes sense.  In 
particular it cannot be found for any parametric or model-based resampling
schemes.  Hence for objects generated by \code{censboot} the only resampling
scheme for which such an array can be found is ordinary case resampling.  
Similarly if \code{boot.out$sim} is \code{"parametric"} in the case of \code{boot}
or \code{"model"} in the case of \code{tsboot} the array cannot be found.  Note also
that for post-blackened bootstraps from \code{tsboot} the indices found will relate
to those prior to any post-blackening and so will not be useful.


Frequency arrays are used in many post-bootstrap calculations such as
the jackknife-after-bootstrap and finding importance sampling weights.
They are also used to find empirical influence values through the
regression method.
}
\seealso{
\code{\link{boot}}, \code{\link{censboot}}, \code{\link{freq.array}}, \code{\link{tilt.boot}}, \code{\link{tsboot}}
}
\examples{
#  A frequency array for a nonparametric bootstrap
data(city)
city.boot <- boot(city, corr, R=40, stype="w")
boot.array(city.boot)


perm.cor <- function(d,i) 
     cor(d$x,d$u[i])
city.perm <- boot(city, perm.cor, R=40, sim="permutation")
boot.array(city.perm, indices=TRUE)
}
\keyword{nonparametric}
% Converted by Sd2Rd version 1.15.

\eof
\name{boot.ci}
\alias{boot.ci}
\title{
  Nonparametric Bootstrap Confidence Intervals
}
\description{
  This function generates 5 different types of equi-tailed two-sided
  nonparametric confidence intervals.  These are the first order normal 
  approximation, the basic bootstrap interval, the studentized bootstrap 
  interval, the bootstrap percentile interval, and the adjusted bootstrap 
  percentile (BCa) interval.  All or a subset of these intervals can be 
  generated.  
}
\usage{
boot.ci(boot.out, conf = 0.95, type = "all", 
        index = 1:min(2,length(boot.out$t0)), var.t0 = NULL, 
        var.t = NULL, t0 = NULL, t = NULL, L = NULL, h = function(t) t,
        hdot = function(t) rep(1,length(t)), hinv = function(t) t, \dots)
}
\arguments{
  \item{boot.out}{
    An object of class \code{"boot"} containing the output of a bootstrap
    calculation.  
  }
  \item{conf}{
    A scalar or vector containing the confidence level(s) of the required
    interval(s).
  }
  \item{type}{
    A vector of character strings representing the type of intervals
    required. The value should be any subset of the values
    \code{c("norm","basic", "stud", "perc", "bca")} or simply \code{"all"}
    which will compute all five types of intervals.
  }
  \item{index}{
    This should be a vector of length 1 or 2.  The first element of
    \code{index} indicates the position of the variable of interest in
    \code{boot.out$t0} and the relevant column in \code{boot.out$t}.  The
    second element indicates the position of the variance of the variable of
    interest.  If both \code{var.t0} and \code{var.t} are supplied then the
    second element of \code{index} (if present) is ignored.  The default is
    that the variable of interest is in position 1 and its variance is in
    position 2 (as long as there are 2 positions in \code{boot.out$t0}).
  }
  \item{var.t0}{
    If supplied, a value to be used as an estimate of the variance of
    the statistic for the normal approximation and studentized intervals.
    If it is not supplied and \code{length(index)} is 2 then \code{var.t0}
    defaults to \code{boot.out$t0[index[2]]} otherwise \code{var.t0} is
    undefined.  For studentized intervals \code{var.t0} must be defined.
    For the normal approximation, if \code{var.t0} is undefined it defaults
    to \code{var(t)}.  If a transformation is supplied through the argument
    \code{h} then \code{var.t0} should be the variance of the untransformed
    statistic.
  }
  \item{var.t}{
    This is a vector (of length \code{boot.out$R}) of variances of the
    bootstrap replicates of the variable of interest.  It is used only for
    studentized intervals.  If it is not supplied and \code{length(index)}
    is 2 then \code{var.t} defaults to \code{boot.out$t[,index[2]]},
    otherwise its value is undefined which will cause an error for
    studentized intervals.  If a transformation is supplied through the
    argument \code{h} then \code{var.t} should be the variance of the
    untransformed bootstrap statistics.
  }
  \item{t0}{
    The observed value of the statistic of interest.  The default value
    is \code{boot.out$t0[index[1]]}.  Specification of \code{t0} and
    \code{t} allows the user to get intervals for a transformed statistic
    which may not be in the bootstrap output object.  See the second example
    below.  An alternative way of achieving this would be to supply the
    functions \code{h}, \code{hdot}, and \code{hinv} below.
  }
  \item{t}{
    The bootstrap replicates of the statistic of interest.  It must be a
    vector of length \code{boot.out$R}.  It is an error to supply one of
    \code{t0} or \code{t} but not the other.  Also if studentized
    intervals are required and \code{t0} and \code{t} are supplied then
    so should be \code{var.t0} and \code{var.t}.  The default value is
    \code{boot.out$t[,index]}.
  }
  \item{L}{
    The empirical influence values of the statistic of interest for the
    observed data.  These are used only for BCa intervals.  If a
    transformation is supplied through the parameter \code{h} then
    \code{L} should be the influence values for \code{t}; the values for
    \code{h(t)} are derived from these and \code{hdot} within the
    function. If \code{L} is not supplied then the values are calculated
    using \code{empinf} if they are needed.
  }
  \item{h}{
    A function defining a transformation.  The intervals are calculated
    on the scale of \code{h(t)} and the inverse function \code{hinv}
    applied to the resulting intervals.  It must be a function of one
    variable only and for a vector argument, it must return a vector of
    the same length, i.e. \code{h(c(t1,t2,t3))} should return
    \code{c(h(t1),h(t2),h(t3))}. The default is the identity function.
  }
  \item{hdot}{
    A function of one argument returning the derivative of \code{h}.  It
    is a required argument if \code{h} is supplied and normal,
    studentized or BCa intervals are required.  The function is used for
    approximating the variances of \code{h(t0)} and \code{h(t)} using
    the delta method, and also for finding the empirical influence
    values for BCa intervals.  Like \code{h} it should be able to take a
    vector argument and return a vector of the same length.  The default
    is the constant function 1.
  }
  \item{hinv}{
    A function, like \code{h}, which returns the inverse of \code{h}.
    It is used to transform the intervals calculated on the scale of
    \code{h(t)} back to the original scale. The default is the identity
    function.  If \code{h} is supplied but \code{hinv} is not, then the
    intervals returned will be on the transformed scale.
  }
  \item{\dots}{
    Any extra arguments that \code{boot.out$statistic} is
    expecting. These arguments are needed only if BCa intervals are required
    and \code{L} is not supplied since in that case \code{L} is calculated
    through a call to \code{empinf} which calls \code{boot.out$statistic}.
  }
}
\details{
  The formulae on which the calculations are based can be found in
  Chapter 5 of Davison and Hinkley (1997).  Function \code{boot} must be
  run prior to running this function to create the object to be passed as
  \code{boot.out}.

  Variance estimates are required for studentized intervals.  The variance
  of the observed statistic is optional for normal theory intervals.  If
  it is not supplied then the bootstrap estimate of variance is used.  The
  normal intervals also use the bootstrap bias correction.

  Interpolation on the normal quantile scale is used when a non-integer order
  statistic is required.  If the order statistic used is the smallest or
  largest of the R values in boot.out a warning is generated and such
  intervals should not be considered reliable.  
}
\value{
  An object of type \code{"bootci"} which contains the intervals.
  It has components
  \item{R}{
    The number of bootstrap replicates on which the intervals were based.
  }
  \item{t0}{
    The observed value of the statistic on the same scale as the intervals.
  }
  \item{call}{
    The call to \code{boot.ci} which generated the object.

    It will also contain one or more of the following components depending
    on the value of \code{type} used in the call to \code{bootci}.
  }
  \item{normal}{
    A matrix of intervals calculated using the normal approximation.  It will
    have 3 columns, the first being the level and the other two being the upper
    and lower endpoints of the intervals.
  }
  \item{basic}{
    The intervals calculated using the basic bootstrap method.
  }
  \item{student}{
    The intervals calculated using the studentized bootstrap method.
  }
  \item{percent}{
    The intervals calculated using the bootstrap percentile method.
  }
  \item{bca}{
    The intervals calculated using the adjusted bootstrap percentile
    (BCa) method.

    These latter four components will be matrices with 5 columns,  the
    first column containing the level, the next two containing the
    indices of the order statistics used in the calculations and the
    final two the calculated endpoints themselves. 
  }
}
\references{
  Davison, A.C. and Hinkley, D.V. (1997) 
  \emph{Bootstrap Methods and Their Application}, Chapter 5.
  Cambridge University Press.

  DiCiccio, T.J. and Efron  B. (1996) Bootstrap confidence intervals (with 
  Discussion). \emph{Statistical Science}, \bold{11}, 189--228.

  Efron, B. (1987) Better bootstrap confidence intervals (with Discussion).
  \emph{Journal of the American Statistical Association}, \bold{82}, 171--200.
}
\seealso{
  \code{\link{abc.ci}}, \code{\link{boot}},
  \code{\link{empinf}}, \code{\link{norm.ci}}
}
\examples{
# confidence intervals for the city data
data(city)
ratio <- function(d, w) sum(d$x * w)/sum(d$u * w)
city.boot <- boot(city, ratio, R = 999, stype = "w",sim = "ordinary")
boot.ci(city.boot, conf = c(0.90,0.95),
        type = c("norm","basic","perc","bca"))

# studentized confidence interval for the two sample 
# difference of means problem using the final two series
# of the gravity data. 
data(gravity)
diff.means <- function(d, f)
{    n <- nrow(d)
     gp1 <- 1:table(as.numeric(d$series))[1]
     m1 <- sum(d[gp1,1] * f[gp1])/sum(f[gp1])
     m2 <- sum(d[-gp1,1] * f[-gp1])/sum(f[-gp1])
     ss1 <- sum(d[gp1,1]^2 * f[gp1]) - (m1 *  m1 * sum(f[gp1]))
     ss2 <- sum(d[-gp1,1]^2 * f[-gp1]) - (m2 *  m2 * sum(f[-gp1]))
     c(m1-m2, (ss1+ss2)/(sum(f)-2))
}
grav1 <- gravity[as.numeric(gravity[,2])>=7,]
grav1.boot <- boot(grav1, diff.means, R=999, stype="f", strata=grav1[,2])
boot.ci(grav1.boot, type=c("stud","norm"))

# Nonparametric confidence intervals for mean failure time 
# of the air-conditioning data as in Example 5.4 of Davison
# and Hinkley (1997)
data(aircondit)
mean.fun <- function(d, i) 
{    m <- mean(d$hours[i])
     n <- length(i)
     v <- (n-1)*var(d$hours[i])/n^2
     c(m, v)
}
air.boot <- boot(aircondit, mean.fun, R=999)
boot.ci(air.boot, type = c("norm", "basic", "perc", "stud"))

# Now using the log transformation
# There are two ways of doing this and they both give the
# same intervals.

# Method 1
boot.ci(air.boot, type = c("norm", "basic", "perc", "stud"), 
        h = log, hdot = function(x) 1/x)

# Method 2
vt0 <- air.boot$t0[2]/air.boot$t0[1]^2
vt <- air.boot$t[,2]/air.boot$t[,1]^2
boot.ci(air.boot, type = c("norm", "basic", "perc", "stud"), 
        t0 = log(air.boot$t0[1]), t = log(air.boot$t[,1]),
        var.t0 = vt0, var.t = vt)
}
\keyword{nonparametric}
\keyword{htest}


\eof
\name{brambles}
\alias{brambles}
\title{
Spatial Location of Bramble Canes
}
\description{
The \code{brambles} data frame has 823 rows and 3 columns.


The location of living bramble canes in a 9m square plot was recorded.  
We take 9m to be the unit of distance so that the plot can be thought of
as a unit square. The bramble canes were also classified by their age.
}
\usage{
data(brambles)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{x}}{
The x coordinate of the position of the cane in the plot.
}
\item{\code{y}}{
The y coordinate of the position of the cane in the plot.
}
\item{\code{age}}{
The age classification of the canes; \code{0} indicates a newly emerged cane,
\code{1} indicates a one year old cane and \code{2} indicates a two year old cane.
}}}
\source{
The data were obtained from


Diggle, P.J. (1983) \emph{Statistical Analysis of Spatial Point Patterns}.
Academic Press.
}
\references}{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{breslow}
\alias{breslow}
\title{
Smoking Deaths Among Doctors
}
\description{
The \code{breslow} data frame has 10 rows and 5 columns.


In 1961 Doll and Hill sent out a questionnaire to all men on the British 
Medical Register enquiring about their smoking habits. Almost 70\% of 
such men replied.  Death certificates were obtained for medical practitioners
and causes of death were assigned on the basis of these certificates.  The
\code{breslow} data set contains the person-years of observations and deaths from
coronary artery disease accumulated during the first ten years of the study.
}
\usage{
data(breslow)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{age}}{
The mid-point of the 10 year age-group for the doctors.
}
\item{\code{smoke}}{
An indicator of whether the doctors smoked (1) or not (0).
}
\item{\code{n}}{
The number of person-years in the category.
}
\item{\code{y}}{
The number of deaths attributed to coronary artery disease.
}
\item{\code{ns}}{
The number of smoker years in the category (\code{smoke*n}).
}}}
\source{
The data were obtained from


Breslow, N.E. (1985) Cohort Analysis in Epidemiology. In 
\emph{A Celebration of Statistics} 
A.C. Atkinson and S.E. Fienberg (editors), 109--143.
Springer-Verlag.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Doll, R. and Hill, A.B. (1966) Mortality of British doctors in relation to 
smoking: Observations on coronary thrombosis. 
\emph{National Cancer Institute Monograph}, \bold{19}, 205-268.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{calcium}
\alias{calcium}
\title{
Calcium Uptake Data
}
\description{
The \code{calcium} data frame has 27 rows and 2 columns.


Howard Grimes from the Botany Department, North Carolina State University,
conducted an experiment for biochemical analysis of intracellular storage
and transport of calcium across plasma membrane.  Cells were
suspended in a solution of radioactive calcium for a certain length of time and
then the amount of radioactive calcium that was absorbed by the cells was
measured.  The experiment was repeated independently with 9 different
times of suspension each replicated 3 times.  
}
\usage{
data(calcium)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{time}}{
The time (in minutes) that the cells were suspended in the solution.
}
\item{\code{cal}}{
The amount of calcium uptake (nmoles/mg).
}}}
\source{
The data were obtained from


Rawlings, J.O. (1988) \emph{Applied Regression Analysis}. 
Wadsworth and Brooks/Cole Statistics/Probability Series.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{cane}
\alias{cane}
\title{
Sugar-cane Disease Data
}
\description{
  The \code{cane} data frame has 180 rows and 5 columns. 
  The data frame represents a randomized
  block design with 45 varieties of sugar-cane and 4 blocks.}
\details{
  The aim of the experiment was to classify the varieties into
  resistant, intermediate and susceptible to a disease called "coal of
  sugar-cane" (carvao da cana-de-acucar).  This is a disease that is
  common in sugar-cane plantations in certain areas of Brazil.
  
  For each plot, fifty pieces of sugar-cane stem were put in a solution
  containing the disease agent and then some were planted in the plot.
  After a fixed period of time, the total number of shoots and the number
  of diseased shoots were recorded.
}
\usage{
data(cane)
}
\format{
This data frame contains the following columns:

\describe{
\item{\code{n}}{
The total number of shoots in each plot.
}
\item{\code{r}}{
The number of diseased shoots.
}
\item{\code{x}}{
The number of pieces of the stems, out of 50, planted in each plot.
}
\item{\code{var}}{
A factor indicating the variety of sugar-cane in each plot.
}
\item{\code{block}}{
A factor for the blocks.
}}
}
\source{
  The data were kindly supplied by Dr. C.G.B. Demetrio of Escola
  Superior de Agricultura, Universidade de Sao Paolo, Brazil.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 0.3-1.

\eof
\name{capability}
\alias{capability}
\title{
Simulated Manufacturing Process Data
}
\description{
The \code{capability} data frame has 75 rows and 1 columns.


The data are simulated successive observations from a process in equilibrium.
The process is assumed to have specification limits (5.49, 5.79).
}
\usage{
data(capability)
}
\format{
This data frame contains the following column:


\describe{
\item{\code{y}}{
The simulated measurements.
}}}
\source{
The data were obtained from


Bissell, A.F. (1990) How reliable is your capability index?
\emph{Applied Statistics}, \bold{39}, 331--340.
}
\references{
Canty, A.J. and Davison, A.C. (1996) Implementation of saddlepoint 
approximations to resampling distributions. To appear in 
\emph{Computing Science and Statistics; Proceedings of the 28th Symposium on the Interface}.


Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{catsM}
\alias{catsM}
\title{
Weight Data for Domestic Cats
}
\description{
The \code{catsM} data frame has 97 rows and 3 columns.

144 adult (over 2kg in weight) cats used for experiments with the drug
digitalis had their heart and body weight recorded.  47 of the cats were
female and 97 were male.  The \code{catsM} data frame consists of the data for
the male cats.  The full data are in dataset \code{\link[MASS]{cats}}
in package \code{MASS}.
}
\usage{
data(cats)
}
\format{
This data frames contain the following columns:

\describe{
\item{\code{Sex}}{
A factor for the sex of the cat (levels are \code{F} and \code{M}).
}
\item{\code{Bwt}}{
Body weight in kg.
}
\item{\code{Hwt}}{
Heart weight in kg.
}}}
\seealso{
  \code{\link[MASS]{cats}}
}
\source{
The data were obtained from

Fisher, R.A. (1947) The analysis of covariance method for the relation
between a part and the whole. \emph{Biometrics}, \bold{3}, 65--68.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Venables, W.N. and Ripley, B.D. (1994) 
\emph{Modern Applied Statistics with S-Plus}. Springer-Verlag.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{cav}
\alias{cav}
\title{
Position of Muscle Caveolae
}
\description{
The \code{cav} data frame has 138 rows and 2 columns.


The data gives the positions of the individual caveolae in a square region
with sides of length 500 units.  This grid was originally on a 2.65mum
square of muscle fibre.  The data are those points falling in the lower left 
hand quarter of the region used for the
dataset \code{caveolae.dat} in the \code{spatial} library by B.D. Ripley (1994) which
is available from Statlib
}
\usage{
data(cav)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{x}}{
The x coordinate of the caveola's position in the region.
}
\item{\code{y}}{
The y coordinate of the caveola's position in the region.
}}}
\references{
Appleyard, S.T., Witkowski, J.A., Ripley, B.D., Shotton, D.M. and Dubowicz, V.
(1985) A novel procedure for pattern analysis of features present on freeze 
fractured plasma membranes. \emph{Journal of Cell Science}, \bold{74}, 105--117.


Davison, A.C. and Hinkley, D.V. (1997) \emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{cd4}
\alias{cd4}
\title{
CD4 Counts for HIV-Positive Patients
}
\description{
The \code{cd4} data frame has 20 rows and 2 columns.


CD4 cells are carried in the blood as part of the human immune system.  One of
the effects of the HIV virus is that these cells die.  The count of CD4 cells is
used in determining the onset of full-blown AIDS in a patient.  In this study of
the effectiveness of a new anti-viral drug on HIV, 20 HIV-positive patients had
their CD4 counts recorded and then were put on a course of treatment with this
drug.  After using the drug for one year, their CD4 counts were again recorded.
The aim of the experiment was to show that patients taking the drug had
increased CD4 counts which is not generally seen in HIV-positive patients.
}
\usage{
data(cd4)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{baseline}}{
The CD4 counts (in 100's) on admission to the trial.
}
\item{\code{oneyear }}{
The CD4 counts (in 100's) after one year of treatment with the new drug.
}}}
\source{
The data were obtained from


DiCiccio, T.J. and Efron  B. (1996) Bootstrap confidence intervals (with 
Discussion). \emph{Statistical Science}, \bold{11}, 189--228.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{cd4.nested}
\alias{cd4.nested}
\title{
Nested Bootstrap of cd4 data
}
\description{
This is an example of a nested bootstrap for the correlation 
coefficient of the \code{cd4} data frame.  It is used in a practical
in Chapter 5 of Davison and Hinkley (1997).
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\seealso{
\code{\link{cd4}}
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{censboot}
\alias{censboot}
\alias{cens.return}
\title{
Bootstrap for Censored Data
}
\description{  
This function applies types of bootstrap resampling which have
been suggested to deal with right-censored data.  It can also do model-based
resampling using a Cox regression model.
}
\usage{
censboot(data, statistic, R, F.surv, G.surv, strata=matrix(1,n,2),
         sim="ordinary", cox=NULL, index=c(1, 2), \dots)
}
\arguments{
  \item{data}{
The data frame or matrix containing the data.  It must have at least two
columns, one of which contains the times and the other the censoring
indicators.  It is allowed to have as many other columns as desired
(although efficiency is reduced for large numbers of columns) except for
\code{sim="weird"} when it should only have two columns - the times and
censoring indicators.  The columns of \code{data} referenced by the
components of \code{index} are taken to be the times and censoring
indicators.
}
\item{statistic}{
A function which operates on the data frame and returns the required
statistic.  Its first argument must be the data. Any other arguments
that it requires can be passed using the \code{\dots{}} argument.  In
the case of \code{sim="weird"}, the data passed to \code{statistic} only
contains the times and censoring indicator regardless of the actual
number of columns in \code{data}. In all other cases the data passed to
statistic will be of the same form as the original data.  When
\code{sim="weird"}, the actual number of observations in the resampled
data sets may not be the same as the number in \code{data}.  For this
reason, if \code{sim="weird"} and \code{strata} is supplied,
\code{statistic} should also take a numeric vector indicating the
strata.  This allows the statistic to depend on the strata if required.
}
\item{R}{
The number of bootstrap replicates.
}
\item{F.surv}{
An object returned from a call to \code{survfit} giving the survivor
function for the data. This is a required argument unless
\code{sim="ordinary"} or \code{sim="model"} and \code{cox} is missing.
}
\item{G.surv}{
Another object returned from a call to \code{survfit} but with the
censoring indicators reversed to give the product-limit estimate of the
censoring distribution.  Note that for consistency the uncensored times
should be reduced by a small amount in the call to \code{survfit}.  This
is a required argument whenever \code{sim="cond"} or when
\code{sim="model"} and \code{cox} is supplied.
}
\item{strata}{
The strata used in the calls to \code{survfit}.  It can be a vector or a
matrix with 2 columns.  If it is a vector then it is assumed to be the
strata for the survival distribution, and the censoring distribution is
assumed to be the same for all observations.  If it is a matrix then the
first column is the strata for the survival distribution and the second
is the strata for the censoring distribution.  When \code{sim="weird"}
only the strata for the survival distribution are used since the
censoring times are considered fixed.  When \code{sim="ordinary"}, only
one set of strata is used to stratify the observations, this is taken to
be the first column of \code{strata} when it is a matrix.
}
\item{sim}{
  The simulation type.  Possible types are \code{"ordinary"} (case
resampling), \code{"model"} (equivalent to \code{"ordinary"} if
\code{cox} is missing, otherwise it is model based resampling),
\code{"weird"} (the weird bootstrap - this cannot be used if \code{cox}
is supplied), and \code{"cond"} (the conditional bootstrap, in which
censoring times are resampled from the conditional censoring
distribution).
}
\item{cox}{
An object returned from \code{coxph}.  If it is supplied, then \code{F.surv} should have
been generated by a call of the form \code{survfit(cox)}.  
}
\item{index}{
A vector of length two giving the positions of the columns in \code{data} which
correspond to the times and censoring indicators respectively.
}
\item{...}{
Any other arguments which are passed to \code{statistic}.
}}
\value{
An object of class \code{"boot"} containing the following components:
\item{t0}{
The value of \code{statistic} when applied to the original data.
}
\item{t}{
A matrix of bootstrap replicates of the values of \code{statistic}.
}
\item{R}{
The number of bootstrap replicates performed.
}
\item{sim}{
The simulation type used.  This will usually be the input value of \code{sim}
unless that was \code{"model"} but \code{cox} was not supplied, in which case it will
be \code{"ordinary"}.
}
\item{data}{
The data used for the bootstrap. This will generally be the input
value of \code{data} unless \code{sim="weird"}, in which case it will just be the
columns containing the times and the censoring indicators.
}
\item{seed}{
The value of \code{.Random.seed} when \code{censboot} was called.
}
\item{statistic}{
The input value of \code{statistic}.
}
\item{strata}{
The strata used in the resampling.  When \code{sim="ordinary"} this will be a vector
which stratifies the observations, when \code{sim="weird"} it is the strata for the
survival distribution and in all other cases it is a matrix containing the 
strata for the survival distribution and the censoring distribution.
}
\item{call}{
The original call to \code{censboot}.
}}
\details{
The various types of resampling are described in Davison and Hinkley (1997)
in sections 3.5 and 7.3.  The simplest is case resampling which simply 
resamples with replacement from the observations.  

The conditional bootstrap simulates failure times from the estimate of
the survival distribution.  Then, for each observation its simulated
censoring time is equal to the observed censoring time if the
observation was censored and generated from the estimated censoring
distribution conditional on being greater than the observed failure time
if the observation was uncensored.  If the largest value is censored
then it is given a nominal failure time of \code{Inf} and conversely if
it is uncensored it is given a nominal censoring time of \code{Inf}.
This is necessary to allow the largest observation to be in the
resamples.

If a Cox regression model is fitted to the data and supplied, then the
failure times are generated from the survival distribution using that
model.  In this case the censoring times can either be simulated from
the estimated censoring distribution (\code{sim="model"}) or from the
conditional censoring distribution as in the previous paragraph
(\code{sim="cond"}).

The weird bootstrap holds the censored observations as fixed and also
the observed failure times.  It then generates the number of events at
each failure time using a binomial distribution with mean 1 and
denominator the number of failures that could have occurred at that time
in the original data set.  In our implementation we insist that there is
a least one simulated event in each stratum for every bootstrap dataset.


When there are strata involved and \code{sim} is either \code{"model"}
or \code{"cond"} the situation becomes more difficult.  Since the strata
for the survival and censoring distributions are not the same it is
possible that for some observations both the simulated failure time and
the simulated censoring time are infinite.  To see this consider an
observation in stratum 1F for the survival distribution and stratum 1G
for the censoring distribution.  Now if the largest value in stratum 1F
is censored it is given a nominal failure time of \code{Inf}, also if
the largest value in stratum 1G is uncensored it is given a nominal
censoring time of \code{Inf} and so both the simulated failure and
censoring times could be infinite.  When this happens the simulated
value is considered to be a failure at the time of the largest observed
failure time in the stratum for the survival distribution.
}
\references{ Andersen, P.K., Borgan, O., Gill, R.D. and Keiding,
N. (1993) \emph{Statistical Models Based on Counting
Processes}. Springer-Verlag.

Burr, D. (1994) A comparison of certain bootstrap confidence intervals
in the Cox model. \emph{Journal of the American Statistical
Association}, \bold{89}, 1290--1302.

Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.

Efron, B. (1981) Censored data and the bootstrap. 
\emph{Journal of the  American Statistical Association}, \bold{76}, 312--319.

Hjort, N.L. (1985) Bootstrapping Cox's regression model. Technical report 
NSF-241, Dept. of Statistics, Stanford University.
}
\seealso{
\code{\link{boot}}, 
\code{\link[survival]{coxph}}, \code{\link[survival]{survfit}}
}
\examples{
data(aml, package="boot")
library(survival)
# Example 3.9 of Davison and Hinkley (1997) does a bootstrap on some
# remission times for patients with a type of leukaemia.  The patients
# were divided into those who received maintenance chemotherapy and 
# those who did not.  Here we are interested in the median remission 
# time for the two groups.
aml.fun <- function(data) {
     surv <- survfit(Surv(time, cens)~group, data=data)
     out <- NULL
     st <- 1
     for (s in 1:length(surv$strata)) {
          inds <- st:(st+surv$strata[s]-1)
          md <- min(surv$time[inds[1-surv$surv[inds]>=0.5]])
          st <- st+surv$strata[s]
          out <- c(out,md)
     }
}
aml.case <- censboot(aml,aml.fun,R=499,strata=aml$group)

# Now we will look at the same statistic using the conditional 
# bootstrap and the weird bootstrap.  For the conditional bootstrap 
# the survival distribution is stratified but the censoring 
# distribution is not. 

aml.s1 <- survfit(Surv(time,cens)~group, data=aml)
aml.s2 <- survfit(Surv(time-0.001*cens,1-cens)~1, data=aml)
aml.cond <- censboot(aml,aml.fun,R=499,strata=aml$group,
     F.surv=aml.s1,G.surv=aml.s2,sim="cond")


# For the weird bootstrap we must redefine our function slightly since
# the data will not contain the group number.
aml.fun1 <- function(data,str) {
     surv <- survfit(Surv(data[,1],data[,2])~str)
     out <- NULL
     st <- 1
     for (s in 1:length(surv$strata)) {
          inds <- st:(st+surv$strata[s]-1)
          md <- min(surv$time[inds[1-surv$surv[inds]>=0.5]])
          st <- st+surv$strata[s]
          out <- c(out,md)
     }
}
aml.wei <- censboot(cbind(aml$time,aml$cens),aml.fun1,R=499,
     strata=aml$group, F.surv=aml.s1,sim="weird")

# Now for an example where a cox regression model has been fitted
# the data we will look at the melanoma data of Example 7.6 from 
# Davison and Hinkley (1997).  The fitted model assumes that there
# is a different survival distribution for the ulcerated and 
# non-ulcerated groups but that the thickness of the tumour has a
# common effect.  We will also assume that the censoring distribution
# is different in different age groups.  The statistic of interest
# is the linear predictor.  This is returned as the values at a
# number of equally spaced points in the range of interest.
data(melanoma, package="boot")
library(splines)# for ns
mel.cox <- coxph(Surv(time,status==1)~ns(thickness,df=4)+strata(ulcer),
     data=melanoma)
mel.surv <- survfit(mel.cox)
agec <- cut(melanoma$age,c(0,39,49,59,69,100))
mel.cens <- survfit(Surv(time-0.001*(status==1),status!=1)~
     strata(agec),data=melanoma)
mel.fun <- function(d) { 
     t1 <- ns(d$thickness,df=4)
     cox <- coxph(Surv(d$time,d$status==1) ~ t1+strata(d$ulcer))
     eta <- unique(cox$linear.predictors)
     u <- unique(d$thickness)
     sp <- smooth.spline(u,eta,df=20)
     th <- seq(from=0.25,to=10,by=0.25)
     predict(sp,th)$y
}
mel.str<-cbind(melanoma$ulcer,agec)
# this is slow!
mel.mod <- censboot(melanoma,mel.fun,R=999,F.surv=mel.surv,
     G.surv=mel.cens,cox=mel.cox,strata=mel.str,sim="model")
proc.time()
# To plot the original predictor and a 95\% pointwise envelope for it
mel.env <- envelope(mel.mod)$point
plot(seq(0.25,10,by=0.25),mel.env[1,], ylim=c(-2,2),
     xlab="thickness (mm)", ylab="linear predictor",type="n")
lines(seq(0.25,10,by=0.25),mel.env[1,],lty=2)
lines(seq(0.25,10,by=0.25),mel.env[2,],lty=2)
lines(seq(0.25,10,by=0.25),mel.mod$t0,lty=1)
}
\author{Angelo J. Canty}
\keyword{survival}

\eof
\name{channing}
\alias{channing}
\title{
Channing House Data
}
\description{
The \code{channing} data frame has 462 rows and 5 columns.


Channing House is a retirement centre in Palo Alto, California.  These data 
were
collected between the opening of the house in 1964 until July 1, 1975.  In that
time 97 men and 365 women passed through the centre.  For each of these, their
age on entry and also on leaving or death was recorded.  A large number of the
observations were censored mainly due to the resident being alive on July 1,
1975 when the data was collected.  Over the time of the study 130 women and
46 men died at Channing House.  Differences between the survival of the sexes,
taking age into account, was one of the primary concerns of this study.
}
\usage{
data(channing)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{sex}}{
A factor for the sex of each resident (\code{"Male"} or \code{"Female"}).
}
\item{\code{entry}}{
The residents age (in months) on entry to the centre
}
\item{\code{exit}}{
The age (in months) of the resident on death, leaving the centre or July 1, 
1975 whichever event occurred first.
}
\item{\code{time}}{
The length of time (in months) that the resident spent at Channing House.
(\code{time=exit-entry})
}
\item{\code{cens}}{
The indicator of right censoring.  1 indicates that the resident died at
Channing House, 0 indicates that they left the house prior to July 1, 1975 or
that they were still alive and living in the centre at that date.
}}}
\source{
The data were obtained from


Hyde, J. (1980) Testing survival with incomplete observations. 
\emph{Biostatistics Casebook}.  
R.G. Miller, B. Efron, B.W. Brown and L.E. Moses (editors), 
31--46. John Wiley.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{claridge}
\alias{claridge}
\title{
Genetic Links to Left-handedness
}
\description{
The \code{claridge} data frame has 37 rows and 2 columns.  


The data are from an experiment which was designed to look for a relationship 
between a certain genetic characteristic
and handedness.  The 37 subjects were women who had a son with mental
retardation due to inheriting a defective X-chromosome.  For each such mother
a genetic measurement of their DNA was made.  Larger values of this measurement
are known to be linked to the defective gene and it was hypothesized that 
larger values might also be linked to a progressive shift away from 
right-handednesss.  Each woman also filled in a 
questionnaire regarding which hand they used for various tasks.  From these
questionnaires a measure of hand preference was found for each mother.  The 
scale of
this measure goes from 1, indicating someone who always favours their right 
hand, to 8, indicating someone who always favours their left hand. Between 
these two extremes are people who favour one hand for some tasks and the other 
for other tasks.
}
\usage{
data(claridge)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{dnan}}{
The genetic measurement on each woman's DNA.
}
\item{\code{hand}}{
The measure of left-handedness on an integer scale from 1 to 8.
}}}
\source{
The data were kindly made available by Dr. Gordon S. Claridge from the
Department of Experimental Psychology, University of Oxford.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{cloth}
\alias{cloth}
\title{
Number of Flaws in Cloth
}
\description{
The \code{cloth} data frame has 32 rows and 2 columns.
}
\usage{
data(cloth)
}
\format{
This data frame contains the following columns:
\describe{
\item{\code{x}}{
The length of the roll of cloth.
}
\item{\code{y}}{
The number of flaws found in the roll.
}}}
\source{
The data were obtained from


Bissell, A.F. (1972) A negative binomial model with varying element size.
\emph{Biometrika}, \bold{59}, 435--441.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{co.transfer}
\alias{co.transfer}
\title{
Carbon Monoxide Transfer
}
\description{
The \code{co.transfer} data frame has 7 rows and 2 columns.  Seven smokers with
chickenpox had their levels of carbon monoxide transfer measured on entry
to hospital and then again after 1 week.  The main question being whether
one week of hospitalization  has changed the carbon monoxide transfer factor.
}
\usage{
data(co.transfer)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{entry}}{
Carbon monoxide transfer factor on entry to hospital.
}
\item{\code{week}}{
Carbon monoxide transfer one week after admittance to hospital.
}}}
\source{
The data were obtained from


Hand, D.J., Daly, F., Lunn, A.D., McConway, K.J. and Ostrowski, E (1994)
\emph{A Handbook of Small Data Sets}. Chapman and Hall.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Ellis, M.E., Neal, K.R. and Webb, A.K. (1987) Is smoking a risk factor for
pneumonia in patients with chickenpox? \emph{British Medical Journal},
\bold{294}, 1002.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{coal}
\alias{coal}
\title{
Dates of Coal Mining Disasters 
}
\description{
The \code{coal} data frame has 191 rows and 1 columns.


This data frame gives the dates of 191 explosions in coal mines which
resulted in 10 or more fatalities.  The time span of the data is from March 15,
1851 until March 22 1962.  
}
\usage{
data(coal)
}
\format{
This data frame contains the following column:


\describe{
\item{\code{date}}{
The date of the disaster.  The integer part of \code{date} gives the year.  The day
is represented as the fraction of the year that had elapsed on that day.
}}}
\source{
The data were obtained from


Hand, D.J., Daly, F., Lunn, A.D., McConway, K.J. and Ostrowski, E. (1994)
\emph{A Handbook of Small Data Sets}, Chapman and Hall.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Jarrett, R.G. (1979) A note on the intervals between coal-mining disasters.
\emph{Biometrika}, \bold{66}, 191-193.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{control}
\alias{control}
\title{
  Control Variate Calculations
}
\description{
  This function will find control variate estimates from a bootstrap
  output object.  It can either find the adjusted bias estimate using
  post-simulation balancing or it can estimate the bias, variance, third
  cumulant and quantiles, using the linear approximation as a control
  variate.
}
\usage{
control(boot.out, L = NULL, distn = NULL, index = 1, t0 = NULL, t = NULL, 
        bias.adj = FALSE, alpha = NULL, \dots)
}
\arguments{
  \item{boot.out}{
    A bootstrap output object returned from \code{boot}.  The bootstrap
    replicates must have been generated using the usual nonparametric
    bootstrap.
  }
  \item{L}{
    The empirical influence values for the statistic of interest.  If
    \code{L} is not supplied then \code{empinf} is called to calculate
    them from \code{boot.out}.
  }
  \item{distn}{
    If present this must be the output from \code{smooth.spline} giving
    the distribution function of the linear approximation.  This is used
    only if \code{bias.adj} is \code{FALSE}.  Normally this would be
    found using a saddlepoint approximation. If it is not supplied in
    that case then it is calculated by \code{saddle.distn}.
  }
  \item{index}{
    The index of the variable of interest in the output of
    \code{boot.out$statistic}.
  }
  \item{t0}{
    The observed value of the statistic of interest on the original data
    set \code{boot.out$data}.  This argument is used only if
    \code{bias.adj} is \code{FALSE}. The input value is ignored if
    \code{t} is not also supplied.  The default value is is
    \code{boot.out$t0[index]}.
  }
  \item{t}{
    The bootstrap replicate values of the statistic of interest.  This
    argument is used only if \code{bias.adj} is \code{FALSE}.  The input
    is ignored if \code{t0} is not supplied also.  The default value is
    \code{boot.out$t[,index]}.
  }
  \item{bias.adj}{
    A logical variable which if \code{TRUE} specifies that the adjusted
    bias estimate using post-simulation balance is all that is required.
    If \code{bias.adj} is \code{FALSE} (default) then the linear
    approximation to the statistic is calculated and used as a control
    variate in estimates of the bias, variance and third cumulant as
    well as quantiles.
  }
  \item{alpha}{
    The alpha levels for the required quantiles if \code{bias.adj} is
    \code{FALSE}.
  }
  \item{\dots}{
    Any additional arguments that \code{boot.out$statistic} requires.
    These are passed unchanged every time \code{boot.out$statistic} is
    called.  \code{boot.out$statistic} is called once if \code{bias.adj}
    is \code{TRUE}, otherwise it may be called by \code{empinf} for
    empirical influence calculations if \code{L} is not supplied.
  }
}
\value{
  If \code{bias.adj} is \code{TRUE} then the returned value is the
  adjusted bias estimate.

  If \code{bias.adj} is \code{FALSE} then the returned value is a list
  with the following components
  
  \item{L}{
    The empirical influence values used.  These are the input values if
    supplied, and otherwise they are the values calculated by
    \code{empinf}.
  }
  \item{tL}{
    The linear approximations to the bootstrap replicates \code{t} of
    the statistic of interest.
  }
  \item{bias}{
    The control estimate of bias using the linear approximation to
    \code{t} as a control variate.
  }
  \item{var}{
    The control estimate of variance using the linear approximation to
    \code{t} as a control variate.
  }
  \item{k3}{
    The control estimate of the third cumulant using the linear
    approximation to \code{t} as a control variate.
  }
  \item{quantiles}{
    A matrix with two columns; the first column are the alpha levels
    used for the quantiles and the second column gives the corresponding
    control estimates of the quantiles using the linear approximation to
    \code{t} as a control variate.
  }
  \item{distn}{
    An output object from \code{smooth.spline} describing the
    saddlepoint approximation to the bootstrap distribution of the
    linear approximation to \code{t}.  If \code{distn} was supplied on
    input then this is the same as the input otherwise it is calculated
    by a call to \code{saddle.distn}.
  }
}
\details{
  If \code{bias.adj} is \code{FALSE} then the linear approximation to
  the statistic is found and evaluated at each bootstrap replicate.
  Then using the equation \emph{T* = Tl*+(T*-Tl*)}, moment estimates can
  be found.  For quantile estimation the distribution of the linear
  approximation to \code{t} is approximated very accurately by
  saddlepoint methods, this is then combined with the bootstrap
  replicates to approximate the bootstrap distribution of \code{t} and
  hence to estimate the bootstrap quantiles of \code{t}.
}
\references{
  Davison, A.C. and Hinkley, D.V. (1997) 
  \emph{Bootstrap Methods and Their Application}. Cambridge University Press.
  
  Davison, A.C., Hinkley, D.V. and Schechtman, E. (1986) Efficient bootstrap 
  simulation. \emph{Biometrika}, \bold{73}, 555--566.

  Efron, B. (1990) More efficient bootstrap computations.
  \emph{Journal of the American Statistical Association}, \bold{55}, 79--89.
}
\seealso{
\code{\link{boot}}, \code{\link{empinf}}, \code{\link{k3.linear}}, \code{\link{linear.approx}}, \code{\link{saddle.distn}}, \code{\link{smooth.spline}}, \code{\link{var.linear}}
}
\examples{
# Use of control variates for the variance of the air-conditioning data
mean.fun <- function(d, i)
{    m <- mean(d$hours[i])
     n <- nrow(d)
     v <- (n-1)*var(d$hours[i])/n^2
     c(m, v)
}
data(aircondit)
air.boot <- boot(aircondit, mean.fun, R = 999)
control(air.boot, index = 2, bias.adj = TRUE)
air.cont <- control(air.boot, index = 2)
# Now let us try the variance on the log scale.
air.cont1 <- control(air.boot, t0=log(air.boot$t0[2]), t=log(air.boot$t[,2]))
}
\keyword{nonparametric}

\eof
\name{corr}
\alias{corr}
\title{
Correlation Coefficient
}
\description{
Calculates the weighted correlation given a data set and a set of weights.
}
\usage{
corr(d, w=rep(1,nrow(d))/nrow(d))
}
\arguments{
\item{d}{
A matrix with two columns corresponding to the two variables whose correlation
we wish to calculate.
}
\item{w}{
A vector of weights to be applied to each pair of observations.  The default
is equal weights for each pair.  Normalization takes place within the function
so \code{sum(w)} need not equal 1.
}}
\value{
The correlation coefficient between \code{d[,1]} and \code{d[,2]}.
}
\details{
This function finds the correlation coefficient in weighted form.  This is
often useful in bootstrap methods since it allows for numerical differentiation
to get the empirical influence values.  It is also necessary to have the 
statistic in this form to find ABC intervals.
}
\seealso{
\code{\link{cor}}
}
\keyword{math}
\keyword{multivariate}
% Converted by Sd2Rd version 1.15.

\eof
\name{cum3}
\alias{cum3}
\title{
Calculate Third Order Cumulants
}
\description{
Calculates an estimate of the third cumulant, or skewness, of a vector. 
Also, if more than one vector is specified, a product-moment of order 3 is
estimated. 
}
\usage{
cum3(a, b=a, c=a, unbiased=TRUE)
}
\arguments{
\item{a}{
A vector of observations.
}
\item{b}{
Another vector of observations,  if not supplied it is set to the value of \code{a}.
If supplied then it must be the same length as \code{a}.
}
\item{c}{
Another vector of observations,  if not supplied it is set to the value of \code{a}.
If supplied then it must be the same length as \code{a}.
}
\item{unbiased}{
A logical value indicating whether the unbiased estimator should be used.  
}}
\value{
The required estimate.
}
\details{
The unbiased estimator uses a multiplier of \code{n/((n-1)*(n-2))} where \code{n} is
the sample size, if \code{unbiased} is \code{FALSE} then a multiplier of \code{1/n} is used.
This is multiplied by \code{sum((a-mean(a))*(b-mean(b))*(c-mean(c)))} to give the
required estimate.
}
\keyword{math}
\keyword{multivariate}
% Converted by Sd2Rd version 1.15.

\eof
\name{cv.glm}
\alias{cv.glm}
\title{
Cross-validation for Generalized Linear Models
}
\description{
This function calculates the estimated K-fold cross-validation prediction 
error for generalized linear models.
}
\usage{
cv.glm(data, glmfit, cost, K)
}
\arguments{
\item{data}{
A matrix or data frame containing the data.  The rows should be cases and
the columns correspond to variables, one of which is the response.
}
\item{glmfit}{
An object of class \code{"glm"} containing the results of a generalized linear
model fitted to \code{data}.
}
\item{cost}{
A function of two vector arguments specifying the cost function for the 
cross-validation.  The first argument to \code{cost} should correspond to the
observed responses and the second argument should correspond to the predicted
or fitted responses from the generalized linear model.  \code{cost} must return a
non-negative scalar value.  The defualt is the average squared error function.
}
\item{K}{
The number of groups into which the data should be split to estimate the
cross-validation prediction error.  The value of \code{K} must be such that all
groups are of approximately equal size.  If the supplied value of \code{K} does
not satisfy this criterion then it will be set to the closest integer which
does and a warning is generated specifying the value of \code{K} used.  The default
is to set \code{K} equal to the number of observations in \code{data} which gives the
usual leave-one-out cross-validation.
}}
\value{
The returned value is a list with the following components.

\item{call}{
The original call to \code{cv.glm}.
}
\item{K}{
The value of \code{K} used for the K-fold cross validation.
}
\item{delta}{
A vector of length two.  The first component is the raw cross-validation
estimate of prediction error.  The second component is the adjusted
cross-validation estimate.  The adjustment is designed to compensate for the
bias introduced by not using leave-one-out cross-validation.
}
\item{seed}{
The value of \code{.Random.seed} when \code{cv.glm} was called. 
}}
\section{Side Effects}{
The value of \code{.Random.seed} is updated.
}
\details{
The data is divided randomly into \code{K} groups.  For each group the generalized
linear model is fit to \code{data} omitting that group, then the function \code{cost}
is applied to the observed responses in the group that was omitted from the fit
and the prediction made by the fitted models for those observations.


When \code{K} is the number of observations leave-one-out cross-validation is used
and all the possible splits of the data are used.  When \code{K} is less than
the number of observations the \code{K} splits to be used are found by randomly
partitioning the data into \code{K} groups of approximately equal size.  In this
latter case a certain amount of bias is introduced.  This can be reduced by
using a simple adjustment (see equation 6.48 in Davison and Hinkley, 1997).
The second value returned in \code{delta} is the estimate adjusted by this method.


}
\references{
Brieman, L., Friedman, J.H., Olshen, R.A. and Stone, C.J. (1984)
\emph{Classification and Regression Trees}. Wadsworth.


Burman, P. (1989) A comparitive study of ordinary cross-validation, 
\emph{v}-fold cross-validation and repeated learning-testing methods.
\emph{Biometrika}, \bold{76}, 503--514


Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Efron, B. (1986) How biased is the apparent error rate of a prediction rule?
\emph{Journal of the American Statistical Association}, \bold{81}, 461--470.


Stone, M.  (1974) Cross-validation choice and assessment of statistical
predictions (with Discussion). 
\emph{Journal of the Royal Statistical Society, B}, \bold{36}, 111--147.
}
\seealso{
\code{\link{glm}}, \code{\link{glm.diag}}, \code{\link{predict}}
}
\examples{
# leave-one-out and 6-fold cross-validation prediction error for 
# the mammals data set.
data(mammals, package="MASS")
mammals.glm <- glm(log(brain)~log(body),data=mammals)
cv.err <- cv.glm(mammals,mammals.glm)
cv.err.6 <- cv.glm(mammals, mammals.glm, K=6)


# As this is a linear model we could calculate the leave-one-out 
# cross-validation estimate without any extra model-fitting.
muhat <- mammals.glm$fitted
mammals.diag <- glm.diag(mammals.glm)
cv.err <- mean((mammals.glm$y-muhat)^2/(1-mammals.diag$h)^2)


# leave-one-out and 11-fold cross-validation prediction error for 
# the nodal data set.  Since the response is a binary variable an
# appropriate cost function is
cost <- function(r, pi=0) mean(abs(r-pi)>0.5)


data(nodal)
nodal.glm <- glm(r~stage+xray+acid,binomial,data=nodal)
cv.err <- cv.glm(nodal, nodal.glm, cost, K=nrow(nodal))$delta 
cv.11.err <- cv.glm(nodal, nodal.glm, cost, K=11)$delta 


}
\keyword{regression}
% Converted by Sd2Rd version 1.15.

\eof
\name{darwin}
\alias{darwin}
\title{
Darwin's Plant Height Differences
}
\description{
The \code{darwin} data frame has 15 rows and 1 columns.


Charles Darwin conducted an experiment to examine the superiority of
cross-fertilized plants over self-fertilized plants.  15 pairs of plants
were used. Each pair consisted of one cross-fertilized plant and one
self-fertilized plant which germinated at the same time and grew in the
same pot. The plants were measured at a fixed time after planting and
the difference in heights between the cross- and self-fertilized plants are
recorded in eighths of an inch.
}
\usage{
data(darwin)
}
\format{
This data frame contains the following column:


\describe{
\item{\code{y}}{
The difference in heights for the pairs of plants (in units of 0.125 inches).
}}}
\source{
The data were obtained from


Fisher, R.A. (1935) \emph{Design of Experiments}. Oliver and Boyd.
}
\references{
Darwin, C. (1876) 
\emph{The Effects of Cross- and Self-fertilisation in the Vegetable Kingdom}. 
John Murray.


Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{dogs}
\alias{dogs}
\non_function{}
\title{ Cardiac Data for Domestic Dogs }
\usage{data(dogs)}
\description{
The \code{dogs} data frame has 7 rows and 2 columns.

Data on the cardiac oxygen consumption and left ventricular
pressure were gathered on 7 domestic dogs.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{mvo}{Cardiac Oxygen Consumption}
    \item{lvp}{Left Ventricular Pressure}
  }
}
\references{
Davison, A. C. and Hinkley, D. V. (1997)
\emph{Bootstrap Methods and Their Application}. Cambridge University
Press.
}
\keyword{datasets}

\eof
\name{downs.bc}
\alias{downs.bc}
\title{
Incidence of Down's Syndrome in British Columbia
}
\description{
The \code{downs.bc} data frame has 30 rows and 3 columns.


Down's syndrome is a genetic disorder caused by an extra chromosome 21 or
a part of chromosome 21 being translocated to another chromosome.  The 
incidence of Down's syndrome is highly dependent on the mother's age and rises
sharply after age 30.  In the 1960's a large scale study of the effect of
maternal age on the incidence of Down's syndrome was conducted at the
British Columbia Health Surveillance Registry.  These are the data which was
collected in that study. 


Mothers were classified by age. Most groups correspond to the age in years but
the first group comprises all mothers with ages in the range 15-17 and the 
last is those with ages 46-49.  No data for
mothers over 50 or below 15 were collected. 
}
\usage{
data(downs.bc)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{age}}{
The average age of all mothers in the age category.
}
\item{\code{m}}{
The total number of live births to mothers in the age category.
}
\item{\code{r}}{
The number of cases of Down's syndrome.
}}}
\source{
The data were obtained from


Geyer, C.J. (1991) Constrained maximum likelihood exemplified by isotonic 
convex logistic regression. 
\emph{Journal of the American Statistical Association}, \bold{86}, 717--724.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{ducks}
\alias{ducks}
\title{
Behavioral and Plumage Characteristics of Hybrid Ducks
}
\description{
The \code{ducks} data frame has 11 rows and 2 columns.


Each row of the data frame represents a male duck who is a second
generation cross of mallard and pintail ducks.  For 11 such ducks a
behavioural and plumage index were calculated.  These were measured on
scales devised for this experiment which was to examine whether there
was any link between which species the ducks resembled physically and
which they resembled in behaviour.  The scale for the physical
appearance ranged from 0 (identical in appearance to a mallard) to 20
(identical to a pintail).  The behavioural traits of the ducks were on
a scale from 0 to 15 with lower numbers indicating closer to
mallard-like in behaviour.
}
\usage{
data(ducks)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{plumage}}{
The index of physical appearance based on the plumage of individual ducks.
}
\item{\code{behaviour}}{
The index of behavioural characteristics of the ducks.
}}}
\source{
The data were obtained from


Larsen, R.J. and Marx, M.L. (1986) 
\emph{An Introduction to Mathematical Statistics and its Applications} 
(Second Edition). Prentice-Hall.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Sharpe, R.S., and Johnsgard, P.A. (1966)
Inheritance of behavioral characters in
\eqn{F_2}{F2} mallard  x  pintail
(\emph{Anas Platyrhynchos L.  x  Anas Acuta L.}) 
hybrids. \emph{Behaviour}, \bold{27}, 259-272.
}
\keyword{datasets}

\eof
\name{empinf}
\alias{empinf}
\title{
Empirical Influence Values
}
\description{
This function calculates the empirical influence values for a statistic
applied to a data set.  It allows four types of calculation, namely
the infinitesimal jackknife (using numerical differentiation), the usual
jackknife estimates, the "positive" jackknife estimates and a method which
estimates the empirical influence values using regression of bootstrap 
replicates of the statistic.  All methods can be used with
one or more samples.
}
\usage{
empinf(boot.out=NULL, data=NULL, statistic=NULL, 
       type=NULL, stype="w", index=1, t=NULL,
       strata=rep(1, n), eps=0.001, \dots)
}
\synopsis{
empinf(boot.out=NULL, data=NULL, statistic=NULL,
       type=NULL, stype=NULL ,index=1, t=NULL,
       strata=rep(1, n), eps=0.001, ...)
}
\arguments{
\item{boot.out}{
A bootstrap object created by the function \code{boot}.  If \code{type} is \code{"reg"} then 
this argument is required.  For any of the other types it is 
an optional argument.  If it is included when optional then the values of 
\code{data}, \code{statistic}, \code{stype}, and \code{strata} are taken from the components of 
\code{boot.out} and any values passed to \code{empinf} directly are ignored.
}
\item{data}{
A vector, matrix or data frame containing
the data for which empirical influence values are required.  It is a required
argument if \code{boot.out} is not supplied.  If \code{boot.out} is supplied then \code{data}
is set to \code{boot.out$data} and any value supplied is ignored.  
}
\item{statistic}{
The statistic for which empirical influence values are required.  It must be
a function of at least two arguments, the data set and 
a vector of weights, frequencies or indices.  The nature of the second
argument is given by the value of \code{stype}.  Any other arguments that it 
takes must be supplied to \code{empinf} and will be passed to \code{statistic} unchanged.
This is a required argument if \code{boot.out} is not supplied, otherwise its value
is taken from \code{boot.out} and any value supplied here will be ignored.
}
\item{type}{
The calculation type to be used for the empirical influence values.  
Possible values of \code{type} are \code{"inf"} (infinitesimal jackknife), \code{"jack"}
(usual jackknife), \code{"pos"} (positive jackknife), and \code{"reg"} (regression
estimation).  The default value depends on the other arguments.  If \code{t} is supplied then the default value of \code{type} is \code{"reg"} and \code{boot.out}
should be present so that its frequency array can be found.  It \code{t} is not 
supplied then if \code{stype} is \code{"w"}, the default value of \code{type} is \code{"inf"}; otherwise, if \code{boot.out} is present the default is \code{"reg"}.  If 
none of these conditions apply then the default is \code{"jack"}.  
Note that it is an error for \code{type} to be
\code{"reg"} if \code{boot.out} is missing or to be  \code{"inf"} if \code{stype} is not \code{"w"}.
}
\item{stype}{
A character variable giving the nature of the second argument to \code{statistic}.
It can take on three values: \code{"w"} (weights), \code{"f"} (frequencies), or \code{"i"}
(indices).  If \code{boot.out} is supplied the value of \code{stype} is set to 
\code{boot.out$stype} and any value supplied here is ignored.  Otherwise it is an
optional argument which defaults to \code{"w"}.  If \code{type} is \code{"inf"} then \code{stype} 
MUST be \code{"w"}.
}
\item{index}{
An integer giving the position of the variable of interest in the output of
\code{statistic}.
}
\item{t}{
A vector of length \code{boot.out$R} which gives the bootstrap replicates of the
statistic of interest.  \code{t} is used only when \code{type} is \code{reg} and it defaults
to \code{boot.out$t[,index]}.  
}
\item{strata}{
An integer vector or a factor specifying the strata for multi-sample problems.
If \code{boot.out} is supplied  the value of \code{strata} is set to \code{boot.out$strata}.  
Otherwise it is an optional argument which has default corresponding to the 
single sample situation.
}
\item{eps}{
This argument is used only if \code{type} is \code{"inf"}.  In that case the value of
epsilon to be used for numerical differentiation will be \code{eps} divided by
the number of observations in \code{data}.
}
\item{...}{
Any other arguments that \code{statistic} takes.  They will be passed unchanged to
\code{statistic} every time that it is called.
}}
\section{Warning}{
All arguments to \code{empinf} must be passed using the \code{name=value} convention.  If
this is not followed then unpredictable errors can occur.
}
\value{
A vector of the empirical influence values of \code{statistic} applied to \code{data}.
The values will be in the same order as the observations in data.
}
\details{
If \code{type} is \code{"inf"} then numerical differentiation is used to approximate the
empirical influence values.  This makes sense only for statistics which
are written in weighted form (i.e. \code{stype} is \code{"w"}).  If \code{type} is \code{"jack"}
then the
usual leave-one-out jackknife estimates of the empirical influence are 
returned.  If \code{type} is \code{"pos"} then the positive (include-one-twice) jackknife
values are used.  If \code{type} is \code{"reg"} then a bootstrap object must be supplied.
The regression method then works by regressing the bootstrap replicates of
\code{statistic} on the frequency array from which they were derived.  The 
bootstrap frequency array is obtained through a call to \code{boot.array}.  Further
details of the methods are given in Section 2.7 of Davison and Hinkley (1997).


Empirical influence values are often used frequently in nonparametric bootstrap
applications.  For this reason many other functions call \code{empinf} when they
are required.  Some examples of their use are for nonparametric delta estimates
of variance, BCa intervals and finding linear approximations to statistics for 
use as control variates.  They are also used for antithetic bootstrap 
resampling.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Efron, B. (1982) \emph{The Jackknife, the Bootstrap and Other Resampling Plans}.
CBMS-NSF Regional Conference Series in Applied Mathematics, \bold{38}, SIAM.


Fernholtz, L.T. (1983) \emph{von Mises Calculus for Statistical Functionals}.
Lecture Notes in Statistics, \bold{19}, Springer-Verlag.
}
\seealso{
\code{\link{boot}}, \code{\link{boot.array}}, \code{\link{boot.ci}}, \code{\link{control}}, \code{\link{jack.after.boot}}, \code{\link{linear.approx}}, \code{\link{var.linear}}
}
\examples{
# The empirical influence values for the ratio of means in 
# the city data.
data(city)
ratio <- function(d, w) sum(d$x *w)/sum(d$u*w)
empinf(data=city,statistic=ratio)
city.boot <- boot(city,ratio,499,stype="w")
empinf(boot.out=city.boot,type="reg")


# A statistic that may be of interest in the difference of means
# problem is the t-statistic for testing equality of means.  In 
# the bootstrap we get replicates of the difference of means and 
# the variance of that statistic and then want to use this output
# to get the empirical influence values of the t-statistic.
data(gravity)
grav1 <- gravity[as.numeric(gravity[,2])>=7,]
grav.fun <- function(dat, w)
{    strata <- tapply(dat[, 2], as.numeric(dat[, 2]))
     d <- dat[, 1]
     ns <- tabulate(strata)
     w <- w/tapply(w, strata, sum)[strata]
     mns <- tapply(d * w, strata, sum)
     mn2 <- tapply(d * d * w, strata, sum)
     s2hat <- sum((mn2 - mns^2)/ns)
     c(mns[2]-mns[1],s2hat)
}


grav.boot <- boot(grav1, grav.fun, R=499, stype="w", strata=grav1[,2])


# Since the statistic of interest is a function of the bootstrap
# statistics, we must calculate the bootstrap replicates and pass
# them to empinf using the t argument.
grav.z <- (grav.boot$t[,1]-grav.boot$t0[1])/sqrt(grav.boot$t[,2])
empinf(boot.out=grav.boot,t=grav.z)
}
\keyword{nonparametric}
\keyword{math}

\eof
\name{envelope}
\alias{envelope}
\title{
Confidence Envelopes for Curves
}
\description{
This function calculates overall and pointwise confidence envelopes for a 
curve based on bootstrap replicates of the curve evaluated at a number of
fixed points.
}
\usage{
envelope(boot.out=NULL, mat=NULL, level=0.95, index=1:ncol(mat))
}
\arguments{
\item{boot.out}{
An object of class \code{"boot"} for which \code{boot.out$t} contains the replicates
of the curve at a number of fixed points.
}
\item{mat}{
A matrix of bootstrap replicates of the values of the curve at a number of
fixed points.  This is a required argument if \code{boot.out} is not supplied
and is set to \code{boot.out$t} otherwise.
}
\item{level}{
The confidence level of the envelopes required.  The default is to
find 95\% confidence envelopes.  It can be a scalar or a vector of length 2.  
If it is scalar then both the pointwise and the overall
envelopes are found at that level.  If is a vector then the first element gives 
the level for the pointwise envelope and the second gives the level for the
overall envelope.
}
\item{index}{
The numbers of the columns of \code{mat} which contain the bootstrap replicates.
This can be used to ensure that other statistics which may have been calculated
in the bootstrap are not considered as values of the function.
}}
\value{
A list with the following components :



\item{point}{
A matrix with two rows corresponding to the values of the upper and lower
pointwise confidence envelope at the same points as the bootstrap replicates
were calculated.
}
\item{overall}{
A matrix similar to \code{point} but containing the envelope which controls the
overall error.
}
\item{k.pt}{
The quantiles used for the pointwise envelope.
}
\item{err.pt}{
A vector with two components, the first gives the pointwise error rate for the
pointwise envelope, and the second the overall error rate for that envelope.
}
\item{k.ov}{
The quantiles used for the overall envelope.
}
\item{err.ov}{
A vector with two components, the first gives the pointwise error rate for the
overall envelope, and the second the overall error rate for that envelope.
}
\item{err.nom}{
A vector of length 2 giving the nominal error rates for the pointwise and the
overall envelopes.
}}
\details{
The pointwise envelope is found by simply looking at the quantiles of the
replicates at each point.  The overall error for that envelope is then
calculated using equation (4.17) of Davison and Hinkley (1997).  A sequence
of pointwise envelopes is then found until one of them has overall error
approximately equal to the level required.  If no such envelope can be
found then the envelope returned will just contain the extreme values of each
column of \code{mat}.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\seealso{
\code{\link{boot}}, \code{\link{boot.ci}}
}
\examples{
# Testing whether the final series of measurements of the gravity data
# may come from a normal distribution.  This is done in Examples 4.7 
# and 4.8 of Davison and Hinkley (1997).
data(gravity)
grav1 <- gravity$g[gravity$series==8]
grav.z <- (grav1-mean(grav1))/sqrt(var(grav1))
grav.gen <- function(dat,mle)
     rnorm(length(dat))
grav.qqboot <- boot(grav.z,sort,R=999,sim="parametric",ran.gen=grav.gen)
grav.qq <- qqnorm(grav.z,plot=FALSE)
grav.qq <- lapply(grav.qq,sort)
plot(grav.qq,ylim=c(-3.5,3.5),ylab="Studentized Order Statistics",
     xlab="Normal Quantiles")
grav.env <- envelope(grav.qqboot,level=0.9)
lines(grav.qq$x,grav.env$point[1,],lty=4)
lines(grav.qq$x,grav.env$point[2,],lty=4)
lines(grav.qq$x,grav.env$overall[1,],lty=1)
lines(grav.qq$x,grav.env$overall[2,],lty=1)
}
\keyword{dplot}
\keyword{htest}
% Converted by Sd2Rd version 1.15.

\eof
\name{exp.tilt}
\alias{exp.tilt}
\title{
Exponential Tilting
}
\description{
This function calculates exponentially tilted multinomial distributions 
such that the resampling distributions of the linear approximation to a
statistic have the required means.
}
\usage{
exp.tilt(L, theta=NULL, t0=0, lambda=NULL,
         strata=rep(1, length(L)))
}
\arguments{
\item{L}{
The empirical influence values for the statistic of interest based on the 
observed data.  The length of \code{L} should be the same as the size of the 
original data set.  Typically \code{L} will be calculated by a call to \code{empinf}.
}
\item{theta}{
The value at which the tilted distribution is to be centred.  This is not
required if \code{lambda} is supplied but is needed otherwise.
}
\item{t0}{
The current value of the statistic.  The default is that the statistic equals 0.
}
\item{lambda}{
The Lagrange multiplier(s).  For each value of \code{lambda} a multinomial 
distribution
is found with probabilities proportional to \code{exp(lambda * L)}.  In general
\code{lambda} is not known and so \code{theta} would be supplied, and the corresponding
value of \code{lambda} found.  If both \code{lambda} and \code{theta} are supplied then
\code{lambda} is ignored and the multipliers for tilting to \code{theta} are found.
}
\item{strata}{
A vector or factor of the same length as \code{L} giving the strata for the
observed data and the empirical influence values \code{L}.
}}
\value{
A list with the following components :



\item{p}{
The tilted probabilities.  There will be \code{m} distributions where \code{m} is the
length of \code{theta} (or \code{lambda} if supplied).  If \code{m} is 1 then \code{p} is a vector
of \code{length(L)} probabilities.  If \code{m} is greater than 1 then \code{p} is a matrix
with \code{m} rows, each of which contain \code{length(L)} probabilities.  In this case
the vector \code{p[i,]} is the distribution tilted to \code{theta[i]}.  \code{p} is
in the form required by the argument \code{weights} of the function \code{boot} for
importance resampling.
}
\item{lambda}{
The Lagrange multiplier used in the equation to determine the tilted
probabilities.  \code{lambda} is a vector of the same length as \code{theta}.
}
\item{theta}{
The values of \code{theta} to which the distributions have been tilted.  In general
this will be the input value of \code{theta} but if \code{lambda} was supplied then 
this is the vector of the corresponding \code{theta} values.
}}
\details{
Exponential tilting involves finding a set of weights for a data set to
ensure that the bootstrap distribution of the linear approximation to a 
statistic of interest has mean \code{theta}.  The weights chosen to achieve this
are given by \code{p[j]} proportional to 
\code{exp(lambda*L[j]/n)}, where \code{n} is the number of data points.  
\code{lambda} is then 
chosen to make the mean of the bootstrap
distribution, of the linear approximation to the statistic of interest, equal
to the the required value \code{theta}.  Thus \code{lambda} is defined as the 
solution of a nonlinear equation.   
The equation is solved by minimizing the Euclidean distance between 
the left and right hand sides of the equation using the function \code{nlmin}.
If this minimum is not equal to zero then the method fails.


Typically exponential tilting is used to find suitable weights for importance
resampling.  If a small tail probability or quantile of the distribution of
the statistic of interest is required then a more efficient simulation is to
centre the resampling distribution close to the point of interest and
then use the functions \code{imp.prob} or \code{imp.quantile} to estimate the required
quantity.


Another method of achieving a similar shifting of the distribution is through
the use of \code{smooth.f}.  The function \code{tilt.boot} uses \code{exp.tilt} or \code{smooth.f}
to find the weights for a tilted bootstrap.
}
\references{
Davison, A. C. and Hinkley, D. V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Efron, B. (1981) Nonparametric standard errors and confidence intervals 
(with Discussion). \emph{Canadian Journal of Statistics}, \bold{9}, 139--172.
}
\seealso{
\code{\link{empinf}}, \code{\link{imp.prob}}, \code{\link{imp.quantile}}, \code{\link{optim}}, \code{\link{smooth.f}}, \code{\link{tilt.boot}}
}
\examples{
# Example 9.8 of Davison and Hinkley (1997) requires tilting the resampling
# distribution of the studentized statistic to be centred at the observed
# value of the test statistic 1.84.  This can be achieved as follows.
data(gravity)
grav1 <- gravity[as.numeric(gravity[,2])>=7,]
grav.fun <- function(dat, w, orig)
{    strata <- tapply(dat[, 2], as.numeric(dat[, 2]))
     d <- dat[, 1]
     ns <- tabulate(strata)
     w <- w/tapply(w, strata, sum)[strata]
     mns <- tapply(d * w, strata, sum)
     mn2 <- tapply(d * d * w, strata, sum)
     s2hat <- sum((mn2 - mns^2)/ns)
     as.vector(c(mns[2]-mns[1],s2hat,(mns[2]-mns[1]-orig)/sqrt(s2hat)))
}
grav.z0 <- grav.fun(grav1,rep(1,26),0)
grav.L <- empinf(data=grav1, statistic=grav.fun, stype="w", 
                 strata=grav1[,2], index=3, orig=grav.z0[1])
grav.tilt <- exp.tilt(grav.L, grav.z0[3], strata=grav1[,2])
boot(grav1, grav.fun, R=499, stype="w", weights=grav.tilt$p,
     strata=grav1[,2], orig=grav.z0[1])
}
\keyword{nonparametric}
\keyword{smooth}
% Converted by Sd2Rd version 1.15.

\eof
\name{fir}
\alias{fir}
\title{
Counts of Balsam-fir Seedlings
}
\description{
The \code{fir} data frame has 50 rows and 3 columns.


The number of balsam-fir seedlings in each quadrant of a grid of 50 five foot 
square quadrants were counted.  The grid consisted of 5 rows of 10 quadrants in
each row.
}
\usage{
data(fir)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{count}}{
The number of seedlings in the quadrant.
}
\item{\code{row}}{
The row number of the quadrant. 
}
\item{\code{col}}{
The quadrant number within the row.
}}}
\source{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{freq.array}
\alias{freq.array}
\title{
Bootstrap Frequency Arrays
}
\description{
Take a matrix of indices for nonparametric bootstrap resamples and
return the frequencies of the original observations in each resample.
}
\usage{
freq.array(i.array)
}
\arguments{
\item{i.array}{
This will be an matrix of integers between 1 and n, where n is the number
of observations in a data set.  The matrix will have n columns and R rows
where R is the number of bootstrap resamples.  Such matrices are found by
\code{boot} when doing nonparametric bootstraps.  They can also be found after a
bootstrap has been run through the function \code{boot.array}.
}}
\value{
A matrix of the same dimensions as the input matrix.  Each row of the 
matrix corresponds to a single bootstrap resample.  Each column of the
matrix corresponds to one of the original observations and specifies its
frequency in each bootstrap resample.  Thus the first column tells us how
often the first observation appeared in each bootstrap resample.  Such
frequency arrays are often useful for diagnostic purposes such as the
jackknife-after-bootstrap plot.  They are also necessary for the regression
estimates of empirical influence values and for finding importance sampling 
weights.
}
\seealso{
\code{\link{boot.array}}
}
\keyword{nonparametric}
% Converted by Sd2Rd version 1.15.

\eof
\name{frets}
\alias{frets}
\title{
Head Dimensions in Brothers
}
\description{
The \code{frets} data frame has 25 rows and 4 columns.


The data consist of measurements of the length and breadth of the heads of 
pairs of adult brothers in 25 randomly sampled families.  All measurements
are expressed in millimetres.
}
\usage{
data(frets)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{l1}}{
The head length of the eldest son.
}
\item{\code{b1}}{
The head breadth of the eldest son.
}
\item{\code{l2}}{
The head length of the second son.
}
\item{\code{b2}}{
The head breadth of the second son.
}}}
\source{
The data were obtained from


Frets, G.P. (1921) Heredity of head form in man. \emph{Genetica}, \bold{3},
193.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Whittaker, J. (1990) \emph{Graphical Models in Applied Multivariate Statistics}.
John Wiley.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{glm.diag}
\alias{glm.diag}
\title{
Generalized Linear Model Diagnostics
}
\description{
Calculates jackknife deviance residuals, standardized deviance residuals, 
standardized Pearson residuals, approximate Cook statistic, leverage and
estimated dispersion.
}
\usage{
glm.diag(glmfit)
}
\arguments{
\item{glmfit}{
\code{glmfit} is a \code{glm.object} - the result of a call to \code{glm()}
}}
\value{
Returns a list with the following components

\item{res}{
The vector of jackknife deviance residuals.
}
\item{rd}{
The vector of standardized deviance residuals.
}
\item{rp}{
The vector of standardized Pearson residuals.
}
\item{cook}{
The vector of approximate Cook statistics.
}
\item{h}{
The vector of leverages of the observations.
}
\item{sd}{
The value used to standardize the residuals.  This is the the estimate of
residual standard deviation in the Gaussian family and is the square root of
the estimated shape parameter in the Gamma family.  In all other cases it is 1.
}}
\references{
Davison, A.C. and Snell, E.J.  (1991)  Residuals and diagnostics.  
In \emph{Statistical Theory and Modelling: In Honour of Sir David Cox}.
D.V. Hinkley, N. Reid and E.J. Snell (editors), 83--106.  Chapman and Hall. 
}
\seealso{
\code{\link{glm}}, \code{\link{glm.diag.plots}}, \code{\link{summary.glm}}
}
\note{
See the helpfile for \code{glm.diag.plots} for an example of the use of \code{glm.diag}.
}
\keyword{regression}
\keyword{dplot}
% Converted by Sd2Rd version 1.15.

\eof
\name{glm.diag.plots}
\alias{glm.diag.plots}
\title{
Diagnostics plots for generalized linear models
}
\description{
Makes plot of jackknife deviance residuals against linear predictor, 
normal scores plots of standardized deviance residuals, plot of approximate Cook statistics against leverage/(1-leverage), and case plot of Cook statistic.
}
\usage{
glm.diag.plots(glmfit, glmdiag=glm.diag(glmfit), 
     subset=NULL, iden=FALSE, labels=NULL, ret=FALSE)
}
\arguments{
\item{glmfit}{
\code{glm.object} : the result of a call to \code{glm()}
}
\item{glmdiag}{
Diagnostics of \code{glmfit} obtained from a call to \code{glm.diag}.  If
it is not supplied then it is calculated.  
}
\item{subset}{
Subset of \code{data} for which \code{glm} fitting performed: should be the same as the 
\code{subset} option used in the call to \code{glm()} which generated \code{glmfit}.  Needed 
only if the \code{subset=} option was used in the call to \code{glm}.  
}
\item{iden}{
A logical argument. If \code{TRUE} then, after the plots are drawn, the user will
be prompted for an integer between 0 and 4.  A positive integer will select
a plot and invoke \code{identify()} on that plot.  After exiting \code{identify()}, the
user is again prompted, this loop continuing until the user responds to the
prompt with 0.  If \code{iden} is \code{FALSE} (default) the user cannot interact with the plots.
}
\item{labels}{
A vector of labels for use with \code{identify()} if \code{iden} is \code{TRUE}.  If it is not 
supplied then the labels are derived from \code{glmfit}.
}
\item{ret}{
A logical argument indicating if \code{glmdiag} should be returned.  The default is
\code{FALSE}.
}}
\value{
If \code{ret} is \code{TRUE} then the value of \code{glmdiag} is returned otherwise there is
no returned value.
}
\details{
The diagnostics required for the plots are calculated by \code{glm.diag}.  These are
then used to produce the four plots on the current graphics device.


The plot on the top left is a plot of the jackknife deviance residuals 
against the fitted values.


The plot on the top right is a normal QQ plot of the standardized deviance 
residuals.  The dotted line is the expected line if the standardized residuals
are normally distributed, i.e. it is the line with intercept 0 and slope 1.


The bottom two panels are plots of the Cook statistics.  On the left is a plot
of the Cook statistics against the standardized leverages.  In general there
will be two dotted lines on this plot.  The horizontal line is at 8/(n-2p)
where n is the number of observations and p is the number of parameters 
estimated.  Points above this line may be points with high influence on the
model.  The vertical line is at 2p/(n-2p) and points to the right of this
line have high leverage compared to the variance of the raw residual at that 
point.  If all points are below the horizontal line or to the left of the
vertical line then the line is not shown.


The final plot again shows the Cook statistic this time plotted against case
number enabling us to find which observations are influential.


Use of \code{iden=T} is encouraged for proper exploration of these four plots as
a guide to how well the model fits the data and whether certain observations
have an unduly large effect on parameter estimates.
}
\section{Side Effects}{
The current device is cleared and four plots are plotted by use of
\code{split.screen(c(2,2))}.  If \code{iden} is \code{TRUE}, interactive identification of 
points is enabled.  All screens are closed, but not cleared, on termination of 
the function.
}
\references{
Davison, A. C. and Hinkley, D. V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Davison, A.C. and Snell, E.J.  (1991)  Residuals and diagnostics.  In 
\emph{Statistical Theory and Modelling: In Honour of Sir David Cox}
D.V. Hinkley, N. Reid, and E.J. Snell (editors), 83--106. Chapman and Hall.
}
\seealso{
\code{\link{glm}}, \code{\link{glm.diag}}, \code{\link{identify}}
}
\examples{
# In this example we look at the leukaemia data which was looked at in 
# Example 7.1 of Davison and Hinkley (1997)
data(leuk, package="MASS")
leuk.mod <- glm(time~ag-1+log10(wbc),family=Gamma(log),data=leuk)
leuk.diag <- glm.diag(leuk.mod)
glm.diag.plots(leuk.mod,leuk.diag)
}
\keyword{regression}
\keyword{dplot}
\keyword{hplot}

\eof
\name{gravity}
\alias{gravity}
\alias{grav}
\title{
Acceleration Due to Gravity
}
\description{
The \code{gravity} data frame has 81 rows and 2 columns.

The \code{grav} data set has 26 rows and 2 columns.

Between May 1934 and July 1935, the National Bureau of Standards in
Washington D.C. conducted a series of experiments to estimate the
acceleration due to gravity, \emph{g}, at Washington.  Each experiment
produced a number of replicate estimates of \emph{g} using the same
methodology.  Although the basic method remained the same for all
experiments, that of the reversible pendulum, there were changes in
configuration.

The \code{gravity} data frame contains the data from all eight
experiments.  The \code{grav} data frame contains the data from the
experiments 7 and 8.  The data are expressed as deviations from 980.000
in centimetres per second squared.
}
\usage{
data(gravity)
}
\format{
This data frame contains the following columns:

\describe{
\item{\code{g}}{
The deviation of the estimate from 980.000 centimetres per second squared.
}
\item{\code{series}}{
A factor describing from which experiment the estimate was derived.
}}}
\source{
The data were obtained from

Cressie, N. (1982) Playing safe with misweighted means. 
\emph{Journal of the American Statistical Association}, \bold{77}, 754--759.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{hirose}
\alias{hirose}
\title{
Failure Time of PET Film
}
\description{
The \code{hirose} data frame has 44 rows and 3 columns.


PET film is used in electrical insulation.  In this accelerated life test
the failure times for 44 samples in gas insulated transformers.  4 different
voltage levels were used.
}
\usage{
data(hirose)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{volt}}{
The voltage (in kV).
}
\item{\code{time}}{
The failure or censoring time in hours.
}
\item{\code{cens}}{
The censoring indicator; \code{1} means right-censored data.
}}}
\source{
The data were obtained from


Hirose, H. (1993) Estimation of threshold stress in accelerated life-testing.
\emph{IEEE Transactions on Reliability}, \bold{42}, 650--657.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{imp.weights}
\alias{imp.weights}
\title{
Importance Sampling Weights
}
\description{
This function calculates the importance sampling weight required to correct
for simulation from a distribution with probabilities \code{p} when estimates 
are required assuming that simulation was from an alternative distribution 
with probabilities \code{q}.
}
\usage{
imp.weights(boot.out, def=TRUE, q=NULL)
}
\arguments{
\item{boot.out}{
A object of class \code{"boot"} generated by \code{boot} or \code{tilt.boot}.  Typically the 
bootstrap simulations would have
been done using importance resampling and we wish to do our calculations
under the assumption of sampling with equal probabilities.
}
\item{def}{
A logical variable indicating whether the defensive mixture distribution
weights should be calculated.  This makes sense only in the case where the 
replicates in \code{boot.out} were simulated under a number of different 
distributions.  If this is the case then the defensive mixture weights use a 
mixture of the distributions used in the bootstrap.  The alternative is to 
calculate the weights for each replicate using knowledge of the distribution 
from which the bootstrap resample was generated.
}
\item{q}{
A vector of probabilities specifying the resampling distribution from which 
we require inferences to be made. In general this would correspond to the usual
bootstrap resampling distribution which gives equal weight to each of the 
original observations and this is the default.  \code{q} must have length equal
to the number of observations in the \code{boot.out$data} and all elements of \code{q}
must be positive.
}}
\value{
A vector of importance weights of the same length as \code{boot.out$t}.  These 
weights can then be used to reweight \code{boot.out$t} so that estimates can be 
found as if the simulations were from a distribution with probabilities \code{q}.
}
\details{
The importance sampling weight for a bootstrap replicate with frequency
vector \code{f} is given by \code{prod((q/p)^f)}.  This reweights the replicates so that
estimates can be found as if the bootstrap resamples were generated according
to the probabilities \code{q} even though, in fact, they came from the 
distribution \code{p}.
}
\references{
Davison, A. C. and Hinkley, D. V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Hesterberg, T. (1995) Weighted average importance sampling and defensive 
mixture distributions. \emph{Technometrics}, \bold{37}, 185--194.


Johns, M.V.  (1988) Importance sampling for bootstrap confidence intervals.
\emph{Journal of the American Statistical Association}, \bold{83}, 709--714.
}
\seealso{
\code{\link{boot}}, \code{\link{exp.tilt}}, \code{\link{imp.moments}}, \code{\link{smooth.f}}, \code{\link{tilt.boot}}
}
\note{
See the example in the help for \code{imp.moments} for an example of using 
\code{imp.weights}.
}
\keyword{nonparametric}
% Converted by Sd2Rd version 1.15.

\eof
\name{inv.logit}
\alias{inv.logit}
\title{
Inverse Logit Function
}
\description{
Given a numeric object return the inverse logit of the values.
}
\usage{
inv.logit(x)
}
\arguments{
\item{x}{
A numeric object. Missing values (\code{NA}s) are allowed.
}}
\value{
An object of the same type as \code{x} containing the inverse logits of the
input values.  
}
\details{
The inverse logit is defined by \code{exp(x)/(1+exp(x))}.  Values in \code{x} of
\code{-Inf} or \code{Inf} return logits of 0 or 1 respectively.  Any \code{NA}s in the input 
will also be \code{NA}s in the output.
}
\seealso{
\code{\link{logit}}, \code{\link{plogis}} which is the underlying function.
}
\keyword{math}
% Converted by Sd2Rd version 1.15.

\eof
\name{islay}
\alias{islay}
\title{
Jura Quartzite Azimuths on Islay
}
\description{
The \code{islay} data frame has 18 rows and 1 columns.


Measurements were taken of paleocurrent azimuths from the Jura Quartzite on
the Scottish island of Islay.
}
\usage{
data(islay)
}
\format{
This data frame contains the following column:


\describe{
\item{\code{theta}}{
The angle of the azimuth in degrees East of North.
}}}
\source{
The data were obtained from


Hand, D.J., Daly, F., Lunn, A.D., McConway, K.J. and Ostrowski, E. (1994)
\emph{A Handbook of Small Data Sets}, Chapman and Hall.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Till, R. (1974) \emph{Statistical Methods for the Earth Scientist}. Macmillan.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{jack.after.boot}
\alias{jack.after.boot}
\title{
Jackknife-after-Bootstrap Plots
}
\description{
This function calculates the jackknife influence values from a bootstrap
output object and plots the corresponding jackknife-after-bootstrap plot.
}
\usage{
jack.after.boot(boot.out, index=1, t=NULL, L=NULL, useJ=TRUE, 
                stinf=TRUE, alpha=NULL, main="", ylab=NULL, \dots)
}
\arguments{
\item{boot.out}{
An object of class \code{"boot"} which would normally be created by a call
to \code{\link{boot}}. It should represent a nonparametric bootstrap.
For reliable results \code{boot.out$R} should be reasonably large.
}
\item{index}{
The index of the statistic of interest in the output of \code{boot.out$statistic}.
}
\item{t}{
A vector of length \code{boot.out$R} giving the bootstrap replicates of the statistic
of interest.  This is useful if the statistic of interest is a function of 
the calculated bootstrap output.  If it is not supplied then the default is
\code{boot.out$t[,index]}.
}
\item{L}{
The empirical influence values for the statistic of interest.  These are used
only if \code{useJ} is \code{FALSE}.  If they are not supplied and are needed, they are
calculated by a call to \code{empinf}.  If \code{L} is supplied then it is assumed that
they are the infinitesimal jackknife values.
}
\item{useJ}{
A logical variable indicating if the jackknife influence values calculated from
the bootstrap replicates should be used.  If \code{FALSE} the empirical influence
values are used.  The default is \code{TRUE}.
}
\item{stinf}{
A logical variable indicating whether to standardize the jackknife values
before plotting them.  If \code{TRUE} then the jackknife values used are divided by 
their standard error.
}
\item{alpha}{
The quantiles at which the plots are required.  The default is 
\code{c(0.05, 0.1, 0.16, 0.5, 0.84, 0.9, 0.95)}.  
}
\item{main}{
A character string giving the main title for the plot.
}
\item{ylab}{
The label for the Y axis.  If the default values of \code{alpha} are used and \code{ylab}
is not supplied then a label indicating which percentiles are plotted is used.
If \code{alpha} is supplied then the default label will not say which percentiles
were used.
}
\item{...}{
Any extra arguments required by \code{boot.out$statistic}.  These are required only
if \code{useJ} is \code{FALSE} and \code{L} is not supplied, in which case they are passed to
\code{empinf} for use in calculation of the empirical influence values.
}}
\value{
There is no returned value but a plot is generated on the current graphics
display.
}
\section{Side Effects}{
A plot is created on the current graphics device.
}
\details{
The centred jackknife quantiles for each observation are estimated from those 
bootstrap samples in which  the particular observation did not appear.  These 
are then plotted against the influence values.  If \code{useJ} is \code{TRUE} then the
influence values are found in the same way as the difference between the 
mean of the statistic in the samples excluding the observations and the mean in 
all samples.  If \code{useJ} is \code{FALSE} then empirical influence values are 
calculated by calling \code{empinf}.


The resulting plots are useful diagnostic tools for looking at the way
individual observations affect the bootstrap output.


The plot will consist of a number of horizontal dotted lines which correspond
to the quantiles of the centred bootstrap distribution.  For each data point
the quantiles of the bootstrap distribution calculated by omitting that point
are plotted against the (possibly standardized) jackknife values.  The 
observation number is printed below the plots.  To make it easier to see 
the effect of omitting points on quantiles, the plotted quantiles are joined
by line segments.  These plots provide a useful diagnostic tool in
establishing the effect of individual observations on the bootstrap 
distribution.  See the references below for some guidelines on the 
interpretation of the plots.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) \emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Efron, B. (1992) Jackknife-after-bootstrap standard errors and influence 
functions (with Discussion). 
\emph{Journal of the Royal Statistical Society, B}, \bold{54}, 83--127.
}
\seealso{
\code{\link{boot}}, \code{\link{empinf}}
}
\examples{
#  To draw the jackknife-after-bootstrap plot for the head size data as in
#  Example 3.24 of Davison and Hinkley (1997)
pcorr <- function( x )
{ 
#  function to find the correlations and partial correlations between
#  the four measurements.
     v <- cor(x)
     v.d <- diag(var(x))
     iv <- solve(v)
     iv.d <- sqrt(diag(iv))
     iv <- - diag(1/iv.d) \%*\% iv \%*\% diag(1/iv.d)
     q <- NULL
     n <- nrow(v)
     for (i in 1:(n-1)) 
          q <- rbind( q, c(v[i,1:i],iv[i,(i+1):n]) )
     q <- rbind( q, v[n,] )
     diag(q) <- round(diag(q))
     q
}


frets.fun <- function( data, i )
{    d <- data[i,]
     v <- pcorr( d )
     c(v[1,],v[2,],v[3,],v[4,])
}
data(frets)
frets.boot <- boot(log(as.matrix(frets)), frets.fun, R=999)
#  we will concentrate on the partial correlation between head breadth
#  for the first son and head length for the second.  This is the 7th
#  element in the output of frets.fun so we set index=7
jack.after.boot(frets.boot,useJ=FALSE,stinf=FALSE,index=7)
}
\keyword{hplot}
\keyword{nonparametric}
% Converted by Sd2Rd version 1.15.

\eof
\name{k3.linear}
\alias{k3.linear}
\title{
Linear Skewness Estimate
}
\description{
Estimates the skewness of a statistic from its empirical influence values.
}
\usage{
k3.linear(L, strata=NULL)
}
\arguments{
\item{L}{
Vector of the empirical influence values of a statistic.  These will usually
be calculated by a call to \code{empinf}.
}
\item{strata}{
A numeric vector or factor specifying which observations (and hence which
components of \code{L}) come from which strata.
}}
\value{
The skewness estimate calculated from \code{L}.
}
\references{
Davison, A. C. and Hinkley, D. V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\seealso{
\code{\link{empinf}}, \code{\link{linear.approx}}, \code{\link{var.linear}}
}
\examples{
#  To estimate the skewness of the ratio of means for the city data.
data(city)
ratio <- function(d,w)
     sum(d$x * w)/sum(d$u * w)
k3.linear(empinf(data=city,statistic=ratio))
}
\keyword{nonparametric}
% Converted by Sd2Rd version 1.15.

\eof
\name{linear.approx}
\alias{linear.approx}
\title{
Linear Approximation of Bootstrap Replicates
}
\description{
This function takes a bootstrap object and for each bootstrap replicate
it calculates the linear approximation to the statistic of interest for 
that bootstrap sample.
}
\usage{
linear.approx(boot.out, L=NULL, index=1, type=NULL, t0=NULL, 
              t=NULL, \dots)
}
\arguments{
\item{boot.out}{
An object of class \code{"boot"} representing a nonparametric bootstrap.  It will
usually be created by the function \code{boot}.
}
\item{L}{
A vector containing the empirical influence values for the statistic of
interest.  If it is not supplied then \code{L} is calculated through a call 
to \code{empinf}.
}
\item{index}{
The index of the variable of interest within the output of 
\code{boot.out$statistic}. 
}
\item{type}{
This gives the type of empirical influence values to be calculated.  It is
not used if \code{L} is supplied.  The possible types of empirical influence
values are described in the helpfile for \code{empinf}.
}
\item{t0}{
The observed value of the statistic of interest.  The input value is used only 
if one of \code{t} or \code{L} is also supplied.  The default value is 
\code{boot.out$t0[index]}.  If \code{t0} is supplied but neither \code{t} nor \code{L} are supplied
then \code{t0} is set to \code{boot.out$t0[index]} and a warning is generated.
}
\item{t}{
A vector of bootstrap replicates of the statistic of interest.  If \code{t0} is 
missing then \code{t} is not used, otherwise it is used to calculate the empirical 
influence values (if they are not supplied in \code{L}).  
}
\item{...}{
Any extra arguments required by \code{boot.out$statistic}.  These are needed if
\code{L} is not supplied as they are used by \code{empinf} to calculate empirical 
influence values.
}}
\value{
A vector of length \code{boot.out$R} with the linear approximations to the
statistic of interest for each of the bootstrap samples.  
}
\details{
The linear approximation to a bootstrap replicate with frequency vector \code{f}
is given by \code{t0 + sum(L * f)/n} in the one sample with an easy extension
to the stratified case.  The frequencies are found by calling \code{boot.array}.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\seealso{
\code{\link{boot}}, \code{\link{empinf}}, \code{\link{control}}
}
\examples{
# Using the city data let us look at the linear approximation to the 
# ratio statistic and its logarithm. We compare these with the 
# corresponding plots for the bigcity data 


data(city)
data(bigcity)
ratio <- function(d, w)
     sum(d$x * w)/sum(d$u * w)
city.boot <- boot(city, ratio, R=499, stype="w")
bigcity.boot <- boot(bigcity, ratio, R=499, stype="w")
par(pty="s")
par(mfrow=c(2,2))


# The first plot is for the city data ratio statistic.
city.lin1 <- linear.approx(city.boot)
lim <- range(c(city.boot$t,city.lin1))
plot(city.boot$t, city.lin1, xlim=lim,ylim=lim, 
     main="Ratio; n=10", xlab="t*", ylab="tL*")
abline(0,1)


# Now for the log of the ratio statistic for the city data.
city.lin2 <- linear.approx(city.boot,t0=log(city.boot$t0), 
                           t=log(city.boot$t))
lim <- range(c(log(city.boot$t),city.lin2))
plot(log(city.boot$t), city.lin2, xlim=lim, ylim=lim, 
     main="Log(Ratio); n=10", xlab="t*", ylab="tL*")
abline(0,1)


# The ratio statistic for the bigcity data.
bigcity.lin1 <- linear.approx(bigcity.boot)
lim <- range(c(bigcity.boot$t,bigcity.lin1))
plot(bigcity.lin1,bigcity.boot$t, xlim=lim,ylim=lim,
     main="Ratio; n=49", xlab="t*", ylab="tL*")
abline(0,1)


# Finally the log of the ratio statistic for the bigcity data.
bigcity.lin2 <- linear.approx(bigcity.boot,t0=log(bigcity.boot$t0), 
                              t=log(bigcity.boot$t))
lim <- range(c(log(bigcity.boot$t),bigcity.lin2))
plot(bigcity.lin2,log(bigcity.boot$t), xlim=lim,ylim=lim,
     main="Log(Ratio); n=49", xlab="t*", ylab="tL*")
abline(0,1)

par(mfrow=c(1,1))
}
\keyword{nonparametric}
% Converted by Sd2Rd version 1.15.

\eof
\name{lines.saddle.distn}
\alias{lines.saddle.distn}
\title{
  Add a Saddlepoint Approximation to a Plot
}
\description{
  This function adds a line corresponding to a saddlepoint density or 
  distribution function approximation to the current plot.
}
\usage{
\method{lines}{saddle.distn}(x, dens = TRUE, h = function(u) u, J = function(u) 1, 
     npts = 50, lty = 1, \dots)
}
\arguments{
  \item{x}{
    An object of class \code{"saddle.distn"}  (see
    \code{\link{saddle.distn.object}} representing a saddlepoint
    approximation to a distribution.
  }
  \item{dens}{
    A logical variable indicating whether the saddlepoint density
    (\code{TRUE}; the default) or the saddlepoint distribution function
    (\code{FALSE}) should be plotted.
  }
  \item{h}{
    Any transformation of the variable that is required.  Its first argument
    must be the value at which the approximation is being performed and the
    function must be vectorized.
  }
  \item{J}{
    When \code{dens=TRUE} this function specifies the Jacobian for any
    transformation that may be necessary.  The first argument of \code{J}
    must the value at which the approximation is being performed and the
    function must be vectorized. If \code{h} is supplied \code{J} must
    also be supplied and both must have the same argument list.
  }
  \item{npts}{
    The number of points to be used for the plot.  These points will be evenly
    spaced over the range of points used in finding the saddlepoint
    approximation.
  }
  \item{lty}{
    The line type to be used.
  }
  \item{\dots}{
    Any additional arguments to \code{h} and \code{J}.
  }
}
\value{
  \code{sad.d} is returned invisibly.
}
\section{Side Effects}{
  A line is added to the current plot.
}
\details{
  The function uses \code{smooth.spline} to produce the saddlepoint
  curve.  When \code{dens=TRUE} the spline is on the log scale and when
  \code{dens=FALSE} it is on the probit scale.
}
\seealso{
  \code{\link{saddle.distn}}
}
\references{
  Davison, A.C. and Hinkley, D.V. (1997) 
  \emph{Bootstrap Methods and Their Application}. Cambridge University Press. 
}
\examples{
# In this example we show how a plot such as that in Figure 9.9 of
# Davison and Hinkley (1997) may be produced.  Note the large number of
# bootstrap replicates required in this example.
expdata <- rexp(12)
vfun <- function(d, i) 
{    n <- length(d)
     (n-1)/n*var(d[i])
}
exp.boot <- boot(expdata,vfun, R = 9999)
exp.L <- (expdata-mean(expdata))^2 - exp.boot$t0
exp.tL <- linear.approx(exp.boot, L = exp.L)
hist(exp.tL, nclass = 50, prob = TRUE)
exp.t0 <- c(0,sqrt(var(exp.boot$t)))
exp.sp <- saddle.distn(A = exp.L/12,wdist = "m", t0 = exp.t0)

# The saddlepoint approximation in this case is to the density of
# t-t0 and so t0 must be added for the plot.
lines(exp.sp,h = function(u,t0) u+t0, J = function(u,t0) 1,
      t0 = exp.boot$t0)
}
\keyword{aplot}
\keyword{smooth}
\keyword{nonparametric}

\eof
\name{logit}
\alias{logit}
\title{
Logit of Proportions
}
\description{
This function calculates the logit of proportions.
}
\usage{
logit(p)
}
\arguments{
\item{p}{
A numeric Splus object, all of whose values are in the range [0,1].  Missing
values (\code{NA}s) are allowed.
}}
\value{
A numeric object of the same type as \code{p} containing the logits of the input
values.
}
\details{
If any elements of \code{p} are outside the unit interval then an error message
is generated.  Values of \code{p} equal to 0 or 1 (to within machine precision)
will return \code{-Inf} or \code{Inf} respectively.  Any \code{NA}s in the input will also
be \code{NA}s in the output.
}
\seealso{
\code{\link{inv.logit}}
}
\keyword{math}
% Converted by Sd2Rd version 1.15.

\eof
\name{manaus}
\alias{manaus}
\title{
Average Heights of the Rio Negro river at Manaus
}
\description{
  The \code{manaus} time series is of class \code{"ts"} and has
  1080 observations on one variable.

  The data values are monthly averages of the daily stages (heights) of
  the Rio Negro at Manaus.  Manaus is 18km upstream from the confluence of
  the Rio Negro with the Amazon but because of the tiny slope of the water
  surface and the lower courses of its flatland affluents, they may be
  regarded as a good approximation of the water level in the Amazon at the
  confluence.  The data here cover 90 years from January 1903 until
  December 1992.

  The Manaus gauge is tied in with an arbitrary bench mark of 100m set in
  the steps of the Municipal Prefecture; gauge readings are usually
  referred to sea level, on the basis of a mark on the steps leading to
  the Parish Church (Matriz), which is assumed to lie at an altitude of
  35.874 m according to observations made many years ago under the
  direction of Samuel Pereira, an engineer in charge of the Manaus
  Sanitation Committee Whereas such an altitude cannot, by any means, be
  considered to be a precise datum point, observations have been
  provisionally referred to it.  The measurements are in metres.
}
\source{
The data were kindly made available by Professors H. O'Reilly Sternberg and 
D. R. Brillinger of the University of California at Berkeley.  
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) \emph{Bootstrap Methods and Their Application}. Cambridge University Press.

Sternberg, H. O'R. (1987) Aggravation of floods in the Amazon river as a
consequence of deforestation? \emph{Geografiska Annaler}, \bold{69A}, 201-219.

Sternberg, H. O'R. (1995) Waters and wetlands of Brazilian Amazonia: An
uncertain future. In \emph{The Fragile Tropics of Latin America:
Sustainable Management of Changing Environments}, Nishizawa, T. and
Uitto, J.I. (editors), United Nations University Press, 113-179.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{melanoma}
\alias{melanoma}
\title{
Survival from Malignant Melanoma
}
\description{
The \code{melanoma} data frame has 205 rows and 7 columns.


The data consist of measurements made on patients with malignant melanoma.
Each patient had their tumour removed by surgery at the Department of Plastic
Surgery, University Hospital of Odense, Denmark during the period 1962 to 1977.
The surgery consisted of complete removal of the tumour together with about
2.5cm of the surrounding skin.  Among the measurements taken were the thickness
of the tumour and whether it was ulcerated or not.  These are thought to be
important prognostic variables in that patients with a thick and/or ulcerated
tumour have an increased chance of death from melanoma.
Patients were followed until the end of 1977.  
}
\usage{
data(melanoma)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{time}}{
Survival time in days since the operation, possibly censored.
}
\item{\code{status}}{
The patients status at the end of the study. 1 indicates that they had died
from melanoma, 2 indicates that they were still alive and 3 indicates that they
had died from causes unrelated to their melanoma.
}
\item{\code{sex}}{
The patients sex; 1=male, 2=female.
}
\item{\code{age}}{
Age in years at the time of the operation.
}
\item{\code{year}}{
Year of operation.
}
\item{\code{thickness}}{
Tumour thickness in mm.
}
\item{\code{ulcer}}{
Indicator of ulceration; 1=present, 0=absent.
}}}
\note{
This dataset is not related to the dataset in the \code{trellis} library
with the same name.
}
\source{
The data were obtained from


Andersen, P.K., Borgan, O., Gill, R.D. and Keiding, N. (1993)
\emph{Statistical Models Based on Counting Processes}. Springer-Verlag.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) \emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Venables, W.N. and Ripley, B.D. (1994) 
\emph{Modern Applied Statistics with S-Plus}. Springer-Verlag.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{motor}
\alias{motor}
\title{
Data from a Simulated Motorcycle Accident
}
\description{
The \code{motor} data frame has 94 rows and 4 columns.  The rows are
obtained by removing replicate values of \code{time} from the dataset 
\code{\link[MASS]{mcycle}}.  Two extra columns are added to allow for strata with
a different residual variance in each stratum.
}
\usage{
data(motor)
}
\format{
This data frame contains the following columns:
\describe{
\item{\code{times}}{
The time in milliseconds since impact.
}
\item{\code{accel}}{
The recorded head acceleration (in g).
}
\item{\code{strata}}{
A numeric column indicating to which of the three strata (numbered 1, 2 and 3)
the observations belong.
}
\item{\code{v}}{
An estimate of the residual variance for the observation.  \code{v} is constant
within the strata but a different
estimate is used for each of the three strata.
}}}
\source{
The data were obtained from


Silverman, B.W. (1985) Some aspects of the spline smoothing approach to
non-parametric curve fitting. 
\emph{Journal of the Royal Statistical Society, B}, \bold{47}, 1--52.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Venables, W.N. and Ripley, B.D. (1994) 
\emph{Modern Applied Statistics with S-Plus}. Springer-Verlag.
}
\seealso{
\code{\link[MASS]{mcycle}}
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{neuro}
\alias{neuro}
\title{
Neurophysiological Point Process Data
}
\description{
\code{neuro} is a matrix containing times of observed firing of a neuron in windows
of 250ms either side of the application of a stimulus to a human subject.
Each row of the matrix is a replication of the experiment and there were a 
total of 469 replicates.
}
\note{
There are a lot of missing values in the matrix as different numbers of 
firings 
were observed in different replicates.  The number of firings observed varied 
from 2 to 6.
}
\source{
The data were collected and kindly made available by Dr. S.J. Boniface of the
Neurophysiology Unit at the Radcliffe Infirmary, Oxford.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Ventura, V., Davison, A.C. and Boniface, S.J. (1997) A stochastic model for 
the effect of magnetic brain stimulation on a motoneurone. To appear in
\emph{Applied Statistics}. 
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{nitrofen}
\alias{nitrofen}
\title{
Toxicity of Nitrofen in Aquatic Systems
}
\description{
The \code{nitrofen} data frame has 50 rows and 5 columns.


Nitrofen is a herbicide that was used extensively for the control of 
broad-leaved and grass weeds in cereals and rice. Although it is relatively
non-toxic to adult mammals, nitrofen is a significant tetragen and mutagen.
It is also acutely toxic and reproductively toxic to cladoceran zooplankton.
Nitrofen is no longer in commercial use in the U.S., having been the first
pesticide to be withdrawn due to tetrogenic effects.


The data here come from an experiment to measure the reproductive toxicity
of nitrofen on a species of zooplankton (\emph{Ceriodaphnia dubia}).  50 animals
were randomized into batches of 10 and each batch was put in a solution with
a measured concentration of nitrofen.  Then the number of live offspring in
each of the three broods to each animal was recorded.
}
\usage{
data(nitrofen)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{conc}}{
The nitrofen concentration in the solution (mug/litre).
}
\item{\code{brood1}}{
The number of live offspring in the first brood.
}
\item{\code{brood2}}{
The number of live offspring in the second brood.
}
\item{\code{brood3}}{
The number of live offspring in the third brood.
}
\item{\code{total}}{
The total number of live offspring in the first three broods.
}}}
\source{
The data were obtained from


Bailer, A.J. and Oris, J.T. (1994) Assessing toxicity of pollutants in aquatic 
systems. In \emph{Case Studies in Biometry}. N. Lange, L. Ryan, L. Billard,
D. Brillinger, L. Conquest and J. Greenhouse (editors), 25--40. John Wiley.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{nodal}
\alias{nodal}
\title{
Nodal Involvement in Prostate Cancer
}
\description{
The \code{nodal} data frame has 53 rows and 7 columns.


The treatment strategy for a patient diagnosed with cancer of the prostate
depend highly on whether the cancer has spread to the surrounding lymph nodes.
It is common to operate on the patient to get samples from the nodes which can
then be analysed under a microscope but clearly it would be preferable if an
accurate assessment of nodal involvement could be made without surgery.


For a sample of 53 prostate cancer patients, a number of possible predictor
variables were measured before surgery.  The patients then had surgery to 
determine nodal involvement.  It was required to see if nodal involvement could 
be accurately predicted from the predictor variables and which ones were
most important.
}
\usage{
data(nodal)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{m}}{
A column of ones.
}
\item{\code{r}}{
An indicator of nodal involvement. 
}
\item{\code{aged}}{
The patients age dichotomized into less than 60 (\code{0}) and 60 or over \code{1}.
}
\item{\code{stage}}{
A measurement of the size and position of the tumour observed by palpatation
with the fingers via the rectum.  A value of \code{1} indicates a more serious
case of the cancer.
}
\item{\code{grade}}{
Another indicator of the seriousness of the cancer, this one is determined by
a pathology reading of a biopsy taken by needle before surgery.
A value of \code{1} indicates a more serious case of the cancer.
}
\item{\code{xray}}{
A third measure of the seriousness of the cancer taken from an X-ray reading.
A value of \code{1} indicates a more serious case of the cancer.
}
\item{\code{acid}}{
The level of acid phosphatase in the blood serum.
}}}
\source{
The data were obtained from


Brown, B.W. (1980) Prediction analysis for binary data. In 
\emph{Biostatistics Casebook}.
R.G. Miller, B. Efron, B.W. Brown and L.E. Moses (editors),
3--18. John Wiley.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{norm.ci}
\alias{norm.ci}
\title{
Normal Approximation Confidence Intervals
}
\description{
Using the normal approximation to a statistic, calculate equi-tailed two-sided 
confidence intervals.
}
\usage{
norm.ci(boot.out=NULL, conf=0.95, index=1, var.t0=NULL, 
        t0=NULL, t=NULL, L=NULL, h=function(t) t, 
        hdot=function(t) 1, hinv=function(t) t)
}
\arguments{
\item{boot.out}{
A bootstrap output object returned from a call to \code{boot}.  If \code{t0} is
missing then \code{boot.out} is a required argument.  It is also required if
both \code{var.t0} and \code{t} are missing.
}
\item{conf}{
A scalar or vector containing the confidence level(s) of the required 
interval(s).
}
\item{index}{
The index of the statistic of interest within the output of a call to
\code{boot.out$statistic}.  It is not used if \code{boot.out} is missing, in which
case \code{t0} must be supplied.
}
\item{var.t0}{
The variance of the statistic of interest.  If it is not supplied then 
\code{var(t)} is used.
}
\item{t0}{
The observed value of the statistic of interest.  If it is missing then it is
taken from \code{boot.out} which is required in that case.
}
\item{t}{
Bootstrap replicates of the variable of interest.  These are used to estimate 
the variance of the statistic of interest if \code{var.t0} is not supplied.  The
default value is \code{boot.out$t[,index]}.
}
\item{L}{
The empirical influence values for the statistic of interest.  These are used
to calculate \code{var.t0} if neither \code{var.t0} nor \code{boot.out} are supplied.  If a
transformation is supplied through \code{h} then the influence values must be for the
untransformed statistic \code{t0}.
}
\item{h}{
A function defining a monotonic transformation,  the intervals are calculated
on the scale of \code{h(t)} and the inverse function \code{hinv} is applied to the 
resulting intervals.  \code{h} must be a function of one variable only and must
be vectorized. The default is the identity function.
}
\item{hdot}{
A function of one argument returning the derivative of \code{h}.  It is a required
argument if \code{h} is supplied and is used for approximating the variance of
\code{h(t0)}.  The default is the constant function 1.
}
\item{hinv}{
A function, like \code{h}, which returns the inverse of \code{h}.  It is used to transform
the intervals calculated on the scale of \code{h(t)} back to the original scale.
The default is the identity function.  If \code{h} is supplied but \code{hinv} is not,
then the intervals returned will be on the transformed scale.
}}
\value{
If \code{length(conf)} is 1 then a vector containing the confidence level and the
endpoints of the interval is returned.  Otherwise, the returned value is a 
matrix where each row corresponds to a different confidence level.
}
\details{
It is assumed that the statistic of interest has an approximately normal
distribution with variance \code{var.t0} and so a confidence interval of length
\code{2*qnorm((1+conf)/2)*sqrt(var.t0)} is found.  If \code{boot.out} or \code{t} are
supplied then the interval is bias-corrected using the bootstrap bias estimate,
and so the interval would be centred at \code{2*t0-mean(t)}.  Otherwise the interval
is centred at \code{t0}.
}
\note{
This function is primarily designed to be called by \code{boot.ci} to calculate 
the normal approximation after a bootstrap but it can also be used without
doing any bootstrap calculations as long as \code{t0} and \code{var.t0} can be supplied. See the examples below.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\seealso{
\code{\link{boot.ci}}
}
\examples{
#  In Example 5.1 of Davison and Hinkley (1997), normal approximation 
#  confidence intervals are found for the air-conditioning data.
data(aircondit)
air.mean <- mean(aircondit$hours)
air.n <- nrow(aircondit)
air.v <- air.mean^2/air.n
norm.ci(t0=air.mean, var.t0=air.v)
exp(norm.ci(t0=log(air.mean), var.t0=1/air.n)[2:3])


# Now a more complicated example - the ratio estimate for the city data.
data(city)
ratio <- function(d, w)
     sum(d$x * w)/sum(d$u *w)
city.v <- var.linear(empinf(data=city, statistic=ratio))
norm.ci(t0=ratio(city,rep(0.1,10)), var.t0=city.v)


}
\keyword{htest}
% Converted by Sd2Rd version 1.15.

\eof
\name{nuclear}
\alias{nuclear}
\title{
Nuclear Power Station Construction Data
}
\description{
The \code{nuclear} data frame has 32 rows and 11 columns.


The data relate to the construction of 32 light water reactor (LWR) plants 
constructed in the U.S.A in the late 1960's and early 1970's.  The data was
collected with the aim of predicting the cost of construction of further
LWR plants.  6 of the power plants had partial turnkey guarantees and it is
possible that, for these plants, some manufacturers' subsidies may be hidden 
in the quoted capital costs.
}
\usage{
data(nuclear)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{cost}}{
The capital cost of construction in millions of dollars adjusted to 1976 base.
}
\item{\code{date}}{
The date on which the construction permit was issued.  The data are measured in
years since January 1 1990 to the nearest month.
}
\item{\code{t1}}{
The time between application for and issue of the construction permit.
}
\item{\code{t2}}{
The time between issue of operating license and construction permit.
}
\item{\code{cap}}{
The net capacity of the power plant (MWe).
}
\item{\code{pr}}{
A binary variable where \code{1} indicates the prior existence of a LWR plant at
the same site.
}
\item{\code{ne}}{
A binary variable where \code{1} indicates that the plant was constructed in the 
north-east region of the U.S.A.
}
\item{\code{ct}}{
A binary variable where \code{1} indicates the use of a cooling tower in the plant.
}
\item{\code{bw}}{
A binary variable where \code{1} indicates that the nuclear steam supply system was
manufactured by Babcock-Wilcox.
}
\item{\code{cum.n}}{
The cumulative number of power plants constructed by each architect-engineer.
}
\item{\code{pt}}{
A binary variable where \code{1} indicates those plants with partial turnkey
guarantees.
}}}
\source{
The data were obtained from


Cox, D.R. and Snell, E.J. (1981) 
\emph{Applied Statistics: Principles and Examples}. Chapman and Hall.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{paulsen}
\alias{paulsen}
\title{
Neurotransmission in Guinea Pig Brains
}
\description{
The \code{paulsen} data frame has 346 rows and 1 columns.  


Sections were prepared from the brain of adult guinea pigs. Spontaneous
currents that flowed into individual brain cells were then recorded and
the peak amplitude of each current measured.  The aim of the experiment 
was to see if the current flow was
quantal in nature (i.e. that it is not a single burst but instead is built up
of many smaller bursts of current).  If the current was indeed quantal then it
would be expected that the distribution of the current amplitude would be
multimodal with modes at regular intervals.  The modes would be expected to
decrease in magnitude for higher current amplitudes.
}
\usage{
data(paulsen)
}
\format{
This data frame contains the following column:


\describe{
\item{\code{y}}{
The current flowing into individual brain cells. The currents are
measured in pico-amperes.
}}}
\source{
The data were kindly made available by Dr. O. Paulsen from the Department
of Pharmacology at the University of Oxford. 


Paulsen, O. and Heggelund, P. (1994) The quantal size at retinogeniculate 
synapses determined from spontaneous and evoked EPSCs in guinea-pig thalamic 
slices. \emph{Journal of Physiology}, \bold{480}, 505--511. 
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{plot.boot}
\alias{plot.boot}
\title{
Plots of the Output of a Bootstrap Simulation
}
\description{
This takes a bootstrap object and produces plots for the bootstrap
replicates of the variable of interest.  
}
\usage{
\method{plot}{boot}(x, index=1, t0=NULL, t=NULL, jack=FALSE, qdist="norm", 
     nclass=NULL, df, \dots)
}
\arguments{
\item{x}{
An object of class \code{"boot"} returned from one of the bootstrap generation
functions.
}
\item{index}{
The index of the variable of interest within the output of \code{boot.out}.  This
is ignored if \code{t} and \code{t0} are supplied.
}
\item{t0}{
The original value of the statistic.  This defaults to \code{boot.out$t0[index]}
unless \code{t} is supplied when it defaults to \code{NULL}. In that case no vertical 
line is drawn on the histogram.
}
\item{t}{
The bootstrap replicates of the statistic.  Usually this will take on its
default value of \code{boot.out$t[,index]}, however it may be useful sometimes 
to supply a different set of values which are a function of \code{boot.out$t}.
}
\item{jack}{
A logical value indicating whether a jackknife-after-bootstrap plot is 
required.  The default is not to produce such a plot.
}
\item{qdist}{
The distribution against which the Q-Q plot should be drawn.  At present 
\code{"norm"} (normal distribution - the default) and \code{"chisq"} (chi-squared 
distribution) are the only possible values.
}
\item{nclass}{
An integer giving the number of classes to be used in the bootstrap histogram.
The default is the integer between 10 and 100 closest to
\code{ceiling(length(t)/25)}. 
}
\item{df}{
If \code{qdist} is \code{"chisq"} then this is the degrees of freedom for the chi-squared
distribution to be used.  It is a required argument in that case.
}
\item{...}{
When \code{jack} is \code{TRUE} additional parameters to \code{jack.after.boot} can be 
supplied.  See the help file for \code{jack.after.boot} for details of the
possible parameters.
}}
\value{
\code{boot.out} is returned invisibly.
}
\section{Side Effects}{
All screens are closed and cleared and a number of plots are produced on the 
current graphics device.  Screens are closed but not cleared at termination
of this function.
}
\details{
This function will generally produce two side-by-side plots.  The left plot
will be a histogram of the bootstrap replicates.  Usually the breaks of the
histogram will be chosen so that \code{t0} is at a breakpoint and all intervals
are of equal length.  A vertical dotted line indicates the position of \code{t0}.
This cannot be done if \code{t} is supplied but \code{t0} is not and so, in that case, 
the breakpoints are computed by \code{hist} using the \code{nclass} argument and no
vertical line is drawn.


The second plot is a Q-Q plot of the bootstrap replicates.  The order 
statistics
of the replicates can be plotted against normal or chi-squared quantiles.  In
either case the expected line is also plotted.  For the normal, this will
have intercept \code{mean(t)} and slope \code{sqrt(var(t))} while for the chi-squared
it has intercept 0 and slope 1.  


If \code{jack} is \code{TRUE} a third plot is produced beneath these two.  That plot
is the jackknife-after-bootstrap plot.  This plot may only be requested
when nonparametric simulation has been used.  See \code{jack.after.boot} for further
details of this plot.
}
\seealso{
\code{\link{boot}}, \code{\link{jack.after.boot}}, \code{\link{print.boot}}
}
\examples{
# We fit an exponential model to the air-conditioning data and use
# that for a parametric bootstrap.  Then we look at plots of the
# resampled means.
air.rg <- function(data, mle)
     rexp(length(data), 1/mle)

data(aircondit)
air.boot <- boot(aircondit$hours, mean, R=999, sim="parametric",
                 ran.gen=air.rg, mle=mean(aircondit$hours))
plot(air.boot)


# In the difference of means example for the last two series of the 
# gravity data
data(gravity)
grav1 <- gravity[as.numeric(gravity[,2])>=7,]
grav.fun <- function(dat, w)
{    strata <- tapply(dat[, 2], as.numeric(dat[, 2]))
     d <- dat[, 1]
     ns <- tabulate(strata)
     w <- w/tapply(w, strata, sum)[strata]
     mns <- tapply(d * w, strata, sum)
     mn2 <- tapply(d * d * w, strata, sum)
     s2hat <- sum((mn2 - mns^2)/ns)
     c(mns[2]-mns[1],s2hat)
}


grav.boot <- boot(grav1, grav.fun, R=499, stype="w", strata=grav1[,2])
plot(grav.boot)
# now suppose we want to look at the studentized differences.
grav.z <- (grav.boot$t[,1]-grav.boot$t0[1])/sqrt(grav.boot$t[,2])
plot(grav.boot,t=grav.z,t0=0)


# In this example we look at the one of the partial correlations for the
# head dimensions in the dataset frets.
pcorr <- function( x )
{ 
#  Function to find the correlations and partial correlations between
#  the four measurements.
     v <- cor(x);
     v.d <- diag(var(x));
     iv <- solve(v);
     iv.d <- sqrt(diag(iv));
     iv <- - diag(1/iv.d) \%*\% iv \%*\% diag(1/iv.d);
     q <- NULL; 
     n <- nrow(v);
     for (i in 1:(n-1)) 
          q <- rbind( q, c(v[i,1:i],iv[i,(i+1):n]) );
     q <- rbind( q, v[n,] );
     diag(q) <- round(diag(q));
     q
}


frets.fun <- function( data, i )
{    d <- data[i,];
     v <- pcorr( d );
     c(v[1,],v[2,],v[3,],v[4,])
}
data(frets)
frets.boot <- boot(log(as.matrix(frets)), frets.fun, R=999)
plot(frets.boot, index=7, jack=TRUE, stinf=FALSE, useJ=FALSE)
}
\keyword{hplot}
\keyword{nonparametric}
% Converted by Sd2Rd version 1.15.

\eof
\name{poisons}
\alias{poisons}
\title{
Animal Survival Times
}
\description{
The \code{poisons} data frame has 48 rows and 3 columns.


The data form a 3x4 factorial experiment, the factors being three poisons
and four treatments.  Each combination of the two factors was used for
four animals, the allocation to animals having been completely randomized.
}
\usage{
data(poisons)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{time}}{
The survival time of the animal in units of 10 hours.
}
\item{\code{poison}}{
A factor with levels \code{1}, \code{2} and \code{3} giving the type of poison used.
}
\item{\code{treat}}{
A factor with levels \code{A}, \code{B}, \code{C} and \code{D} giving the treatment.
}}}
\source{
The data were obtained from


Box, G.E.P. and Cox, D.R. (1964) An analysis of transformations (with
Discussion). 
\emph{ Journal of the Royal Statistical Society, B}, \bold{26}, 211--252.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{polar}
\alias{polar}
\title{
Pole Positions of New Caledonian Laterites
}
\description{
The \code{polar} data frame has 50 rows and 2 columns.


The data are the pole positions from a paleomagnetic study of New Caledonian
laterites.
}
\usage{
data(polar)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{lat}}{
The latitude (in degrees) of the pole position.  Note that all latitudes
are negative as the axis is taken to be in the lower hemisphere.
}
\item{\code{long}}{
The longitude (in degrees) of the pole position.
}}}
\source{
The data were obtained from


Fisher, N.I., Lewis, T. and Embleton, B.J.J. (1987) 
\emph{Statistical Analysis of Spherical Data}. Cambridge University Press.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{print.boot}
\alias{print.boot}
\title{
Print a Summary of a Bootstrap Object
}
\description{
This is a method for the function print for objects of the class \code{"boot"}.
}
\usage{
\method{print}{boot}(x, digits = getOption("digits"), 
      index = 1:ncol(boot.out$t), \dots)
}
\arguments{
  \item{x}{
    A bootstrap output object of class \code{"boot"} generated by one
    of the bootstrap functions.
  }
  \item{digits}{
    The number of digits to be printed in the summary statistics.
  }
  \item{index}{
    Indices indicating for which elements of the bootstrap output
    summary statistics are required.
  }
  \item{\dots}{further arguments passed to or from other methods.}
}
\value{
  The bootstrap object is returned invisibly.
}
\details{
  For each statistic calculated in the bootstrap the original value and
  the bootstrap estimates of its bias and standard error are printed.
  If \code{boot.out$t0} is missing (such as when it was created by a
  call to \code{tsboot} with \code{orig.t=FALSE}) the bootstrap mean and
  standard error are printed. If resampling was done using importance
  resampling weights, then the bootstrap estimates are reweighted as if
  uniform resampling had been done.  The ratio importance sampling
  estimates are used and if there were a number of distributions then
  defensive mixture distributions are used.  In this case an extra
  column with the mean of the observed bootstrap statistics is also
  printed.
}
\seealso{
  \code{\link{boot}}, \code{\link{censboot}}, \code{\link{imp.moments}},
  \code{\link{plot.boot}}, \code{\link{tilt.boot}}, \code{\link{tsboot}}
}
\keyword{nonparametric}
\keyword{htest}

\eof
\name{print.bootci}
\alias{print.bootci}
\title{
Print Bootstrap Confidence Intervals
}
\description{
This is a method for the function \code{print()} to print objects of the
class \code{"bootci"}.
}
\usage{
\method{print}{bootci}(x, hinv = NULL, ...)
}
\arguments{
  \item{x}{
    The output from a call to \code{boot.ci}.
  }
  \item{hinv}{
    A transformation to be made to the interval end-points before they are
    printed.
  }
  \item{\dots}{further arguments passed to or from other methods.}
}
\value{
The object \code{ci.out} is returned invisibly.
}
\details{
This function prints out the results from \code{boot.ci} in a "nice" format.
It also notes whether the scale of the intervals is the original scale
of the input to \code{boot.ci} or a different scale and whether the calculations
were done on a transformed scale.  It also looks at the order statistics
that were used in calculating the intervals.  If the smallest or largest
values were used then it prints a message 


\code{Warning : Intervals used Extreme Quantiles}


Such intervals should be considered very unstable and not relied upon for
inferences.  Even if the extreme values are not used, it is possible that the
intervals are unstable if they used quantiles close to the extreme values.
The function alerts the user to intervals which use the upper
or lower 10 order statistics with the message


\code{Some intervals may be unstable}
}
\seealso{
\code{\link{boot.ci}}
}
\keyword{print}
\keyword{htest}
% Converted by Sd2Rd version 0.3-1.

\eof
\name{print.saddle.distn}
\alias{print.saddle.distn}
\title{
Print Quantiles of Saddlepoint Approximations
}
\description{
This is a method for the function \code{print()} to print objects of class
\code{"saddle.distn"}.
}
\usage{
\method{print}{saddle.distn}(x, \dots)
}
\arguments{
  \item{x}{
    An object of class \code{"saddle.distn"} created by a call to
    \code{saddle.distn}.
  }
  \item{\dots}{further arguments passed to or from other methods.}
 
}
\value{
The input is returned invisibly.
}
\details{
The quantiles of the saddlepoint approximation to the distribution are printed
along with the original call and some other useful information.
}
\seealso{
\code{\link{lines.saddle.distn}}, \code{\link{saddle.distn}}
}
\keyword{print}
\keyword{smooth}
\keyword{nonparametric}
% Converted by Sd2Rd version 0.3-1.

\eof
\name{print.simplex}
\alias{print.simplex}
\title{
Print Solution to Linear Programming Problem
}
\description{
This is a method for the function \code{print()} to print objects of class
\code{"simplex"}.
}
\usage{
\method{print}{simplex}(x, \dots)
}
\arguments{
  \item{x}{
    An object of class \code{"simplex"} created by calling the
    function \code{simplex} to solve a linear programming problem.
  }
  \item{\dots}{further arguments passed to or from other methods.}
}
\value{
\code{x} is returned silently.
}
\details{
The coefficients of the objective function are printed.
If a solution to the linear programming problem was found then the solution
and the optimal value of the objective function are printed.  If a feasible
solution was found but the maximum number of iterations was exceeded then the
last feasible solution and the objective function value at that point are
printed.  If no feasible solution could be found then a message stating that is
printed.
}
\seealso{
\code{\link{simplex}}
}
\keyword{print}
\keyword{optimize}
% Converted by Sd2Rd version 0.3-1.

\eof
\name{remission}
\alias{remission}
\title{
Cancer Remission and Cell Activity
}
\description{
The \code{remission} data frame has 27 rows and 3 columns.
}
\usage{
data(remission)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{LI}}{
A measure of cell activity.
}
\item{\code{m}}{
The number of patients in each group (all values are actually 1 here).
}
\item{\code{r}}{
The number of patients (out of \code{m}) who went into remission.
}}}
\source{
The data were obtained from


Freeman, D.H. (1987) \emph{Applied Categorical Data Analysis}. Marcel Dekker.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{saddle}
\alias{saddle}
\title{
Saddlepoint Approximations for Bootstrap Statistics
}
\description{
This function calculates a saddlepoint approximation to the distribution of
a linear combination of \bold{W} at a particular point \code{u}, where \bold{W}
is a vector of random variables.  The distribution 
of \bold{W} may be multinomial (default), Poisson or binary.  Other
distributions are possible also if the 
adjusted cumulant generating function and its second derivative are given.  
Conditional saddlepoint approximations to the distribution of one linear 
combination given the values of
other linear combinations of \bold{W} can be calculated for \bold{W} having
binary or Poisson distributions.  
}
\usage{
saddle(A=NULL, u=NULL, wdist="m", type="simp", d=NULL, d1=1, 
       init=rep(0.1, d), mu=rep(0.5, n), LR=FALSE, strata=NULL, 
       K.adj=NULL, K2=NULL)
}
\arguments{
\item{A}{
A vector or matrix of known coefficients of the linear combinations of \bold{W}.
It is a required argument unless \code{K.adj} and \code{K2} are supplied, in which case 
it is ignored.
}
\item{u}{
The value at which it is desired to calculate the saddlepoint approximation to
the distribution of the linear combination of \bold{W}. It is a required
argument unless \code{K.adj} and \code{K2} are supplied, in which case it is ignored.
}
\item{wdist}{
The distribution of \bold{W}.  This can be one of \code{"m"} (multinomial), \code{"p"} 
(Poisson), \code{"b"} (binary) or "o" (other).  If K.adj and K2 are given  
\code{wdist} is set to "o".
}
\item{type}{
The type of saddlepoint approximation.  Possible types are \code{"simp"} for simple
saddlepoint and \code{"cond"} for the conditional saddlepoint.  When \code{wdist} is
\code{"o"} or \code{"m"}, \code{type} is automatically set to \code{"simp"}, which is the only
type of saddlepoint currently implemented for those distributions.
}
\item{d}{
This specifies the dimension of the whole statistic.  This argument is required
only when \code{wdist="o"} and defaults to 1 if not supplied in that case.  For
other distributions it is set to \code{ncol(A)}.
}
\item{d1}{
When \code{type} is \code{"cond"} this is the dimension of the statistic of interest
which must be less than \code{length(u)}.  Then the saddlepoint approximation to the
conditional distribution of the first \code{d1} linear combinations given the values
of the remaining combinations is found.  Conditional distribution function
approximations can only be found if the value of \code{d1} is 1. 
}
\item{init}{
Used if \code{wdist} is either \code{"m"} or \code{"o"}, this gives initial values to 
\code{nlmin} which is used to solve the saddlepoint equation.  
}
\item{mu}{
The values of the parameters of the distribution of \bold{W} when \code{wdist} is
\code{"m"}, \code{"p"} \code{"b"}.  \code{mu} must be of the same length as W (i.e. \code{nrow(A)}).
The default is that all values of \code{mu} are equal and so the elements of \bold{W}
are identically distributed.
}
\item{LR}{
If \code{TRUE} then the Lugananni-Rice approximation to the cdf is used,
otherwise the approximation used is based on Barndorff-Nielsen's r*. 
}
\item{strata}{
The strata for stratified data.
}
\item{K.adj}{
The adjusted cumulant generating function used when \code{wdist} is \code{"o"}.  
This is a function of a single parameter, \code{zeta},  which calculates 
\code{K(zeta)-u\%*\%zeta}, where \code{K(zeta)} is the cumulant generating function 
of \bold{W}.
}
\item{K2}{
This is a function of a single parameter \code{zeta} which returns the 
matrix of second derivatives of \code{K(zeta)} for use when \code{wdist} is \code{"o"}.  
If \code{K.adj} is given then this must be given also.  It is called only once
with the calculated solution to the saddlepoint equation being passed as 
the argument.  This argument is ignored if \code{K.adj} is not supplied.
}}
\value{
A list consisting of the following components

\item{spa}{
The saddlepoint approximations.  The first value is the density approximation
and the second value is the distribution function approximation.
}
\item{zeta.hat}{
The solution to the saddlepoint equation.  For the conditional saddlepoint
this is the solution to the saddlepoint equation for the numerator.
}
\item{zeta2.hat}{
If \code{type} is \code{"cond"} this is the solution to the saddlepoint equation for the 
denominator.  This component is not returned for any other value of \code{type}.
}}
\details{
If \code{wdist} is \code{"o"} or \code{"m"}, the saddlepoint equations are solved using
\code{nlmin} to minimize \code{K.adj} with respect to its parameter \code{zeta}. 
For the Poisson and binary cases, a generalized linear model is fitted such 
that the parameter 
estimates solve the saddlepoint equations.  The response variable 'y' for the
\code{glm} must satisfy the equation \code{t(A)\%*\%y=u} (\code{t()} being the transpose 
function).  Such a vector can be found
as a feasible solution to a linear programming problem.  This is done
by a call to \code{simplex}.  The covariate matrix for the \code{glm} is given by \code{A}. 
}
\references{
Booth, J.G. and Butler, R.W. (1990) Randomization distributions and 
saddlepoint approximations in generalized linear models. 
\emph{Biometrika}, \bold{77}, 787--796.


Canty, A.J. and Davison, A.C. (1997) Implementation of saddlepoint 
approximations to resampling distributions.  
\emph{Computing Science and Statistics; Proceedings of the 28th Symposium on the Interface}, 248--253.


Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and their Application}. Cambridge University Press.


Jensen, J.L. (1995) \emph{Saddlepoint Approximations}. Oxford University Press.
}
\seealso{
  \code{\link{saddle.distn}}, \code{\link{simplex}}
}
\examples{
# To evaluate the bootstrap distribution of the mean failure time of 
# air-conditioning equipment at 80 hours
data(aircondit)
saddle(A=aircondit$hours/12, u=80)

# Alternatively this can be done using a conditional poisson
saddle(A=cbind(aircondit$hours/12,1), u=c(80,12), wdist="p", type="cond")

# To use the Lugananni-Rice approximation to this
saddle(A=cbind(aircondit$hours/12,1), u=c(80,12), wdist="p", type="cond", 
       LR = TRUE)

# Example 9.16 of Davison and Hinkley (1997) calculates saddlepoint 
# approximations to the distribution of the ratio statistic for the
# city data. Since the statistic is not in itself a linear combination
# of random Variables, its distribution cannot be found directly.  
# Instead the statistic is expressed as the solution to a linear 
# estimating equation and hence its distribution can be found.  We
# get the saddlepoint approximation to the pdf and cdf evaluated at
# t=1.25 as follows.
jacobian <- function(dat,t,zeta)
{
     p <- exp(zeta*(dat$x-t*dat$u))
     abs(sum(dat$u*p)/sum(p))
}
data(city)
city.sp1 <- saddle(A=city$x-1.25*city$u, u=0)
city.sp1$spa[1] <- jacobian(city, 1.25, city.sp1$zeta.hat) * city.sp1$spa[1]
city.sp1
}
\keyword{smooth}
\keyword{nonparametric}
% Converted by Sd2Rd version 1.15.

\eof
\name{saddle.distn}
\alias{saddle.distn}
\title{
  Saddlepoint Distribution Approximations for Bootstrap Statistics
}
\description{
  Approximate an entire distribution using saddlepoint methods.  This
  function can calculate simple and conditional saddlepoint distribution
  approximations for a univariate quantity of interest.  For the simple
  saddlepoint the quantity of interest is a linear combination of
  \bold{W} where \bold{W} is a vector of random variables.  For the
  conditional saddlepoint we require the distribution of one linear
  combination given the values of any number of other linear
  combinations. The distribution of \bold{W} must be one of multinomial,
  Poisson or binary.  The primary use of this function is to calculate
  quantiles of bootstrap distributions using saddlepoint approximations.
  Such quantiles are required by the library function \code{control} to
  approximate the distribution of the linear approximation to a
  statistic.
}
\usage{
saddle.distn(A, u=NULL, alpha=NULL, wdist="m", 
             type="simp", npts=20, t=NULL, t0=NULL, 
             init=rep(0.1, d), mu=rep(0.5, n), LR=FALSE, 
             strata=NULL, \dots)
}
\arguments{
  \item{A}{
    This is a matrix of known coefficients or a function which returns
    such a matrix.  If a function then its first argument must be the
    point \code{t} at which a saddlepoint is required.   The most common
    reason for A being a function would be if the statistic is not
    itself a linear combination of the \bold{W} but is the solution to a
    linear estimating equation.
  }
  \item{u}{
    If \code{A} is a function then \code{u}  must also be a function
    returning a vector with length equal to the number of columns of the
    matrix returned by \code{A}. Usually all components other than the
    first will be constants as the other components are the values of
    the conditioning variables. If \code{A} is a matrix with more than
    one column (such as when \code{wdist="cond"}) then \code{u} should
    be a vector with length one less than \code{ncol(A)}.  In this case
    \code{u} specifies the values of the conditioning variables.  If
    \code{A} is a matrix with one column or a vector then \code{u} is
    not used.
  }
  \item{alpha}{
    The alpha levels for the quantiles of the distribution which should be
    returned.  By default the 0.1, 0.5, 1, 2.5, 5, 10, 20, 50, 80, 90,
    95, 97.5, 99, 99.5 and 99.9 percentiles are calculated. 
  }
  \item{wdist}{
    The distribution of \bold{W}.  Possible values are \code{"m"}
    (multinomial), \code{"p"} (Poisson), or \code{"b"} (binary).
  }
  \item{type}{
    The type of saddlepoint to be used.  Possible values are
    \code{"simp"} (simple saddlepoint) and \code{"cond"} (conditional).
    If \code{wdist} is \code{"m"}, \code{type} is set to \code{"simp"}.
  }
  \item{npts}{
    The number of points at which the saddlepoint approximation should be
    calculated and then used to fit the spline.
  }
  \item{t}{
    A vector of points at which the saddlepoint approximations are
    calculated. These points should extend beyond the extreme quantiles
    required but still be in the possible range of the bootstrap
    distribution.  The observed value of the statistic should not be
    included in \code{t} as the distribution function approximation
    breaks down at that point.  The points should, however cover the
    entire effective range of the distribution including close to the
    centre. If \code{t} is supplied then \code{npts} is set to
    \code{length(t)}. When \code{t} is not supplied, the function
    attempts to find the effective range of the distribution and then
    selects points to cover this range.
  }
  \item{t0}{
    If \code{t} is not supplied then a vector of length 2 should be
    passed as \code{t0}. The first component of \code{t0} should be the
    centre of the distribution and the second should be an estimate of
    spread (such as a standard error). These two are then used to find
    the effective range of the distribution. The range finding mechanism
    does rely on an accurate estimate of location in \code{t0[1]}.
  }
  \item{init}{
    When \code{wdist} is \code{"m"}, this vector should contain the
    initial values to be passed to \code{nlmin} when it is called to
    solve the saddlepoint equations.
  }
  \item{mu}{
    The vector of parameter values for the distribution.  The
    default is that the components of \bold{W} are identically distributed.
  }
  \item{LR}{
    A logical flag.  When \code{LR} is \code{TRUE} the Lugananni-Rice
    cdf approximations are calculated and used to fit the spline.
    Otherwise the cdf approximations used are based on
    Barndorff-Nielsen's r*.
  }
  \item{strata}{
    A vector giving the strata when the rows of A relate to stratified
    data.  This is used only when \code{wdist} is \code{"m"}.
  }
  \item{\dots}{
    When \code{A} and \code{u} are functions any additional arguments
    are passed unchanged each time one of them is called.
  }
}
\value{
  The returned value is an object of class \code{"saddle.distn"}.  See the help
  file for \code{\link{saddle.distn.object}} for a description of such
  an object.
}
\details{
  The range at which the saddlepoint is used is such that the cdf
  approximation at the endpoints is more extreme than required by the
  extreme values of \code{alpha}.  The lower endpoint is found by
  evaluating the saddlepoint at the points \code{t0[1]-2*t0[2]},
  \code{t0[1]-4*t0[2]}, \code{t0[1]-8*t0[2]} etc.  until a point is
  found with a cdf approximation less than \code{min(alpha)/10}, then a
  bisection method is used to find the endpoint which has cdf
  approximation in the range (\code{min(alpha)/1000},
  \code{min(alpha)/10}). Then a number of, equally spaced, points are
  chosen between the lower endpoint and \code{t0[1]} until a total of
  \code{npts/2} approximations have been made. The remaining
  \code{npts/2} points are chosen to the right of \code{t0[1]} in a
  similar manner.  Any points which are very close to the centre of the
  distribution are then omitted as the cdf approximations are not
  reliable at the centre. A smoothing spline is then fitted to the
  probit of the saddlepoint distribution function approximations at the
  remaining points and the required quantiles are predicted from the
  spline.

  Sometimes the function will terminate with the message
  \code{"Unable to find range"}. There are two main reasons why this may
  occur.  One is that the distribution is too discrete and/or the
  required quantiles too extreme, this can cause the function to be
  unable to find a point within the allowable range which is beyond the
  extreme quantiles.  Another possibility is that the value of
  \code{t0[2]} is too small and so too many steps are required to find
  the range. The first problem cannot be solved except by asking for
  less extreme quantiles, although for very discrete distributions the
  approximations may not be very good.  In the second case using a
  larger value of \code{t0[2]} will usually solve the problem.
}
\references{
  Booth, J.G. and Butler, R.W. (1990) Randomization distributions and 
  saddlepoint approximations in generalized linear models. 
  \emph{Biometrika}, \bold{77}, 787--796.

  Canty, A.J. and Davison, A.C. (1997) Implementation of saddlepoint 
  approximations to resampling distributions. 
  \emph{Computing Science and Statistics; Proceedings of the 28th
    Symposium on the Interface} 248--253.

  Davison, A.C. and Hinkley, D.V. (1997) 
  \emph{Bootstrap Methods and their Application}. Cambridge University Press.

  Jensen, J.L. (1995) \emph{Saddlepoint Approximations}. Oxford University Press.
}
\seealso{
  \code{\link{lines.saddle.distn}}, \code{\link{saddle}},
  \code{\link{saddle.distn.object}}, \code{\link{smooth.spline}}
}
\examples{
#  The bootstrap distribution of the mean of the air-conditioning 
#  failure data: fails to find value on R (and probably on S too)
data(aircondit)
air.t0 <- c(mean(aircondit$hours), sqrt(var(aircondit$hours)/12))
\dontrun{saddle.distn(A = aircondit$hours/12, t0 = air.t0)}

# alternatively using the conditional poisson
saddle.distn(A = cbind(aircondit$hours/12, 1), u = 12, wdist = "p",
             type = "cond", t0 = air.t0)

# Distribution of the ratio of a sample of size 10 from the bigcity 
# data, taken from Example 9.16 of Davison and Hinkley (1997).
data(city); data(bigcity)
ratio <- function(d, w) sum(d$x *w)/sum(d$u * w)
city.v <- var.linear(empinf(data = city, statistic = ratio))
bigcity.t0 <- c(mean(bigcity$x)/mean(bigcity$u), sqrt(city.v))
Afn <- function(t, data) cbind(data$x - t*data$u, 1)
ufn <- function(t, data) c(0,10)
saddle.distn(A = Afn, u = ufn, wdist = "b", type = "cond",
             t0 = bigcity.t0, data = bigcity)

# From Example 9.16 of Davison and Hinkley (1997) again, we find the 
# conditional distribution of the ratio given the sum of city$u.
Afn <- function(t, data) cbind(data$x-t*data$u, data$u, 1)
ufn <- function(t, data) c(0, sum(data$u), 10)
city.t0 <- c(mean(city$x)/mean(city$u), sqrt(city.v))
saddle.distn(A = Afn, u = ufn, wdist = "p", type = "cond", t0 = city.t0, 
             data = city)
}
\keyword{nonparametric}
\keyword{smooth}
\keyword{dplot}
% Converted by Sd2Rd version 1.15.

\eof
\name{saddle.distn.object}
\alias{saddle.distn.object}
\title{
Saddlepoint Distribution Approximation Objects
}
\description{
  Class of objects that result from calculating saddlepoint distribution
  approximations by a call to \code{saddle.distn}.
}
\section{Generation}{
  This class of objects is returned from calls to the function
  \code{\link{saddle.distn}}.
}
\section{Methods}{
  The class \code{"saddle.distn"} has methods for the functions
  \code{\link{lines}} and \code{\link{print}}.
}
\section{Structure}{
  Objects of class \code{"saddle.distn"} are implemented as a list with
  the following components.

  \item{quantiles}{
    A matrix with 2 columns.  The first column contains the
    probabilities \code{alpha} and the second column contains the
    estimated quantiles of the distribution at those probabilities
    derived from the spline.
  }
  \item{points}{
    A matrix of evaluations of the saddlepoint approximation.  The first
    column contains the values of \code{t} which were used, the second
    and third contain the density and cdf approximations at those points
    and the rest of the columns contain the solutions to the saddlepoint
    equations.  When \code{type} is \code{"simp"}, there is only one of
    those.  When \code{type} is \code{"cond"} there are \code{2*d-1}
    where \code{d} is the number of columns in \code{A} or the output of
    \code{A(t,\dots{})}. The first \code{d} of these correspond to the
    numerator and the remainder correspond to the denominator.
  }
  \item{distn}{
    An object of class \code{smooth.spline}.  This corresponds to the
    spline fitted to the saddlepoint cdf approximations in points in
    order to approximate the entire distribution.  For the structure of
    the object see \code{smooth.spline}.
  }
  \item{call}{
    The original call to \code{saddle.distn} which generated the object.
  }
  \item{LR}{
    A logical variable indicating whether the Lugananni-Rice
    approximations were used.
  }
}
\seealso{
  \code{\link{lines.saddle.distn}}, \code{\link{saddle.distn}},
  \code{\link{print.saddle.distn}}
}
\keyword{nonparametric}
\keyword{methods}
\keyword{smooth}

\eof
\name{salinity}
\alias{salinity}
\title{
Water Salinity and River Discharge
}
\description{
The \code{salinity} data frame has 28 rows and 4 columns.


Biweekly averages of the water salinity and river discharge in Pamlico
Sound, North Carolina were recorded between the years 1972 and 1977.
The data in this set consists only of those measurements in March, April
and May.
}
\usage{
data(salinity)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{sal}}{
The average salinity of the water over two weeks.
}
\item{\code{lag}}{
The average salinity of the water lagged two weeks.  Since only spring is used,
the value of \code{lag} is not always equal to the previous value of \code{sal}.
}
\item{\code{trend}}{
A factor indicating in which of the 6 biweekly periods between March and May,
the observations were taken. The levels of the factor are from 0 to 5 with
0 being the first two weeks in March.
}
\item{\code{dis}}{
The amount of river discharge during the two weeks for which \code{sal} is the
average salinity.
}}}
\source{
The data were obtained from


Ruppert, D. and Carroll, R.J. (1980) Trimmed least squares estimation in the 
linear model. 
\emph{Journal of the American Statistical Association}, \bold{75}, 828--838.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{simplex}
\alias{simplex}
\title{
Simplex Method for Linear Programming Problems
}
\description{
  This function will optimize the linear function \code{a\%*\%x} subject
  to the constraints \code{A1\%*\%x <= b1}, \code{A2\%*\%x >= b2},
  \code{A3\%*\%x = b3} and \code{x >= 0}.  Either maximization or
  minimization is possible but the default is minimization.
}
\usage{
simplex(a, A1 = NULL, b1 = NULL, A2 = NULL, b2 = NULL, A3 = NULL,
        b3 = NULL, maxi = FALSE, n.iter = n + 2 * m, eps = 1e-10)
}
\arguments{
  \item{a}{
    A vector of length \code{n} which gives the coefficients of the
    objective function.
  }
  \item{A1}{
    An \code{m1} by \code{n} matrix of coefficients for the "<=" type of
    constraints.
  }
  \item{b1}{
    A vector of length \code{m1} giving the right hand side of the "<="
    constraints. This argument is required if \code{A1} is given and
    ignored otherwise.  All values in \code{b1} must be non-negative.
  }
  \item{A2}{
    An \code{m2} by \code{n} matrix of coefficients for the ">=" type of
    constraints.
  }
  \item{b2}{
    A vector of length \code{m2} giving the right hand side of the ">="
    constraints. This argument is required if \code{A2} is given and
    ignored otherwise.  All values in \code{b2} must be non-negative.
    Note that the constraints \code{x >= 0} are included automatically
    and so should not be repeated here.
  }
  \item{A3}{
    An \code{m3} by \code{n} matrix of coefficients for the equality
    constraints.
  }
  \item{b3}{
    A vector of length \code{m3} giving the right hand side of equality
    constraints. This argument is required if \code{A3} is given and
    ignored otherwise.  All values in \code{b3} must be non-negative.
  }
  \item{maxi}{
    A logical flag which specifies minimization if \code{FALSE}
    (default) and maximization otherwise.  If \code{maxi} is \code{TRUE}
    then the maximization problem is recast as a minimization problem by
    changing the objective function coefficients to their negatives.
  }
  \item{n.iter}{
    The maximum number of iterations to be conducted in each phase of
    the simplex method.  The default is \code{n+2*(m1+m2+m3)}.
  }
  \item{eps}{
    The floating point tolerance to be used in tests of equality.
  }
}
\value{
  An object of class \code{"simplex"}: see \code{\link{simplex.object}}.
}
\details{
  The method employed by this function is the two phase tableau simplex
  method. If there are ">=" or equality constraints an initial feasible
  solution is not easy to find.  To find a feasible solution an
  artificial variable is introduced into each ">=" or equality
  constraint and an auxiliary objective function is defined as the sum
  of these artificial variables.  If a feasible solution to the set of
  constraints exists then the auxiliary objective will be minimized when
  all of the artificial variables are 0. These are then discarded and
  the original problem solved starting at the solution to the auxiliary
  problem.  If the only constraints are of the "<=" form, the origin is
  a feasible solution and so the first stage can be omitted.
}
\note{
  The method employed here is suitable only for relatively small
  systems.  Also if possible the number of constraints should be reduced
  to a minimum in order to speed up the execution time which is
  approximately proportional to the cube of the number of constraints.
  In particular if there are any constraints of the form \code{x[i] >=
    b2[i]} they should be omitted by setting \code{x[i] = x[i]-b2[i]},
  changing all the constraints and the objective function accordingly
  and then transforming back after the solution has been found.
}
\references{
  Gill, P.E., Murray, W. and Wright, M.H. (1991)
  \emph{Numerical Linear Algebra and Optimization Vol. 1}. Addison-Wesley.

  Press, W.H., Teukolsky, S.A., Vetterling, W.T. and Flannery, B.P. (1992)
  \emph{Numerical Recipes: The Art of Scientific Computing (Second Edition)}.
  Cambridge University Press.
}
\examples{
#  This example is taken from Exercise 7.5 of Gill, Murray,
#  and Wright (1991).
enj <- c(200, 6000, 3000, -200)
fat <- c(800, 6000, 1000, 400)
vitx <- c(50, 3, 150, 100)
vity <- c(10, 10, 75, 100)
vitz <- c(150, 35, 75, 5)
simplex(a = enj, A1 = fat, b1 = 13800, A2 = rbind(vitx, vity, vitz),
        b2 = c(600, 300, 550), maxi = TRUE)
}
\keyword{optimize}

\eof
\name{simplex.object}
\alias{simplex.object}
\title{
Linear Programming Solution Objects
}
\description{
  Class of objects that result from solving a linear programming
  problem using \code{simplex}.
}
\section{Generation}{
This class of objects is returned from calls to the function \code{simplex}.
}
\section{Methods}{
The class \code{"saddle.distn"} has a method for the function \code{print}.
}
\section{Structure}{
  Objects of class \code{"simplex"} are implemented as a list with the
  following components.

  \item{soln}{
    The values of \code{x} which optimize the objective function under
    the specified constraints provided those constraints are jointly feasible.
  }
  \item{solved}{
    This indicates whether the problem was solved.  A value of \code{-1}
    indicates that no feasible solution could be found.  A value of
    \code{0} that the maximum number of iterations was reached without
    termination of the second stage.  This may indicate an unbounded
    function or simply that more iterations are needed. A value of
    \code{1} indicates that an optimal solution has been found.
  }
  \item{value}{
    The value of the objective function at \code{soln}.
  }
  \item{val.aux}{
    This is \code{NULL} if a feasible solution is found. Otherwise it is
    a positive value giving the value of the auxiliary objective
    function when it was minimized.
  }
  \item{obj}{
    The original coefficients of the objective function.
  }
  \item{a}{
    The objective function coefficients re-expressed such that the basic
    variables have coefficient zero.
  }
  \item{a.aux}{
    This is \code{NULL} if a feasible solution is found. Otherwise it is the
    re-expressed auxiliary objective function at the termination of the first
    phase of the simplex method.
  }
  \item{A}{
    The final constraint matrix which is expressed in terms of the
    non-basic variables.  If a feasible solution is found then this will
    have dimensions \code{m1+m2+m3} by \code{n+m1+m2}, where the final
    \code{m1+m2} columns correspond to slack and surplus variables.  If
    no feasible solution is found there will be an additional
    \code{m1+m2+m3} columns for the artificial variables introduced to
    solve the first phase of the problem.
  }
  \item{basic}{
    The indices of the basic (non-zero) variables in the solution.
    Indices between \code{n+1} and \code{n+m1} correspond to slack
    variables, those between \code{n+m1+1} and \code{n+m2} correspond to
    surplus variables and those greater than \code{n+m2} are artificial
    variables.  Indices greater than \code{n+m2} should occur only if
    \code{solved} is \code{-1} as the artificial variables are discarded in
    the second stage of the simplex method.
  }
  \item{slack}{
    The final values of the \code{m1} slack variables which arise when
    the "<=" constraints are re-expressed as the equalities
    \code{A1\%*\%x + slack = b1}.
  }
  \item{surplus}{
    The final values of the \code{m2} surplus variables which arise when
    the "<=" constraints are re-expressed as the equalities \code{A2\%*\%x -
      surplus = b2}.
  }
  \item{artificial}{
    This is NULL if a feasible solution can be found.  If no solution
    can be found then this contains the values of the \code{m1+m2+m3}
    artificial variables which minimize their sum subject to the
    original constraints.  A feasible solution exists only if all of the
    artificial variables can be made 0 simultaneously.
  }
}
\seealso{
  \code{\link{print.simplex}}, \code{\link{simplex}}
}
\keyword{optimize}
\keyword{methods}
% Converted by Sd2Rd version 0.3-1.

\eof
\name{smooth.f}
\alias{smooth.f}
\title{
Smooth Distributions on Data Points
}
\description{
This function uses the method of frequency smoothing to find a distribution 
on a data set which has a required value, \code{theta},  of the statistic of 
interest.  The method results in distributions which vary smoothly with 
\code{theta}.
}
\usage{
smooth.f(theta, boot.out, index=1, t=boot.out$t[, index], width=0.5)
}
\arguments{
\item{theta}{
The required value for the statistic of interest.  If \code{theta} is a vector,
a separate distribution will be found for each element of \code{theta}.
}
\item{boot.out}{
A bootstrap output object returned by a call to \code{boot}.  
}
\item{index}{
The index of the variable of interest in the output of \code{boot.out$statistic}.
This argument is ignored if \code{t} is supplied.  \code{index} must be a scalar.
}
\item{t}{
The bootstrap values of the statistic of interest.  This must be a vector of
length \code{boot.out$R} and the values must be in the same order as the bootstrap
replicates in \code{boot.out}.
}
\item{width}{
The standardized width for the kernel smoothing.  The smoothing uses a
value of \code{width*s} for epsilon, where \code{s} is the bootstrap estimate of the 
standard error of the statistic of interest.  \code{width} should take a value in 
the range (0.2, 1) to produce a reasonable
smoothed distribution.  If \code{width} is too large then the distribution becomes
closer to uniform. 
}}
\value{
If \code{length(theta)} is 1 then a vector with the same length as the data set
\code{boot.out$data} is returned.  The value in position \code{i} is the probability 
to be given to the data point in position \code{i} so that the distribution has 
parameter value approximately equal to \code{theta}.
If \code{length(theta)} is bigger than 1 then the returned value is a matrix with 
\code{length(theta)} rows each of which corresponds to a distribution with the 
parameter value approximately equal to the corresponding value of \code{theta}.
}
\details{
The new distributional weights are found by applying a normal kernel smoother
to the observed values of \code{t} weighted by the observed frequencies in the
bootstrap simulation.  The resulting distribution may not have
parameter value exactly equal to the required value \code{theta} but it will 
typically have a value which is close to \code{theta}.  The details of how this
method works can be found in Davison, Hinkley and Worton (1995) and Section
3.9.2 of Davison and Hinkley (1997).
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) \emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Davison, A.C., Hinkley, D.V. and Worton, B.J. (1995) Accurate and efficient 
construction of bootstrap likelihoods. \emph{Statistics and Computing}, 
\bold{5}, 257--264.
}
\seealso{
\code{\link{boot}}, \code{\link{exp.tilt}}, \code{\link{tilt.boot}}
}
\examples{
# Example 9.8 of Davison and Hinkley (1997) requires tilting the resampling
# distribution of the studentized statistic to be centred at the observed
# value of the test statistic 1.84.  In the book exponential tilting was used
# but it is also possible to use smooth.f.
data(gravity)
grav1 <- gravity[as.numeric(gravity[,2])>=7,]
grav.fun <- function(dat, w, orig)
{    strata <- tapply(dat[, 2], as.numeric(dat[, 2]))
     d <- dat[, 1]
     ns <- tabulate(strata)
     w <- w/tapply(w, strata, sum)[strata]
     mns <- tapply(d * w, strata, sum)
     mn2 <- tapply(d * d * w, strata, sum)
     s2hat <- sum((mn2 - mns^2)/ns)
     c(mns[2]-mns[1],s2hat,(mns[2]-mns[1]-orig)/sqrt(s2hat),)
}
grav.z0 <- grav.fun(grav1,rep(1,26),0)
grav.boot <- boot(grav1, grav.fun, R=499, stype="w", 
                  strata=grav1[,2], orig=grav.z0[1])
grav.sm <- smooth.f(grav.z0[3], grav.boot, index=3)


# Now we can run another bootstrap using these weights
grav.boot2 <- boot(grav1, grav.fun, R=499, stype="w", 
                   strata=grav1[,2], orig=grav.z0[1],
                   weights=grav.sm)


# Estimated p-values can be found from these as follows
mean(grav.boot$t[,3] >= grav.z0[3])
imp.prob(grav.boot2,t0=-grav.z0[3],t=-grav.boot2$t[,3])


# Note that for the importance sampling probability we must 
# multiply everything by -1 to ensure that we find the correct
# probability.  Raw resampling is not reliable for probabilities
# greater than 0.5. Thus
1-imp.prob(grav.boot2,index=3,t0=grav.z0[3])$raw
# can give very strange results (negative probabilities).
}
\keyword{smooth}
\keyword{nonparametric}
% Converted by Sd2Rd version 1.15.

\eof
\name{sunspot}
\alias{sunspot}
\title{
Annual Mean Sunspot Numbers
}
\description{
\code{sunspot} is a time series of class \code{"rts"} and contains 289 observations.


The Zurich sunspot numbers have been analyzed in almost all books on time
series analysis as well as numerous papers.  The data set, usually attributed
to Rudolf Wolf, consists of means of daily relative numbers of sunspot 
sightings.
The relative number for a day is given by k(f+10g) where g is the number of
sunspot groups observed, f is the total number of spots within the groups and
k is a scaling factor relating the observer and telescope to a baseline. The
relative numbers are then averaged to give an annual figure.   
See Insenman (1983) for a discussion of the relative numbers. The figures are
for the years 1700-1988.
}
\note{
These data are not related to dataset of the same name in the \code{trellis} 
library.  They are, however, similar to the \code{S-Plus} dataset \code{sunspots}
except that this set contains annual averages instead of monthly averages
and it is for a longer period of time.
}
\source{
The data were obtained from


Tong, H. (1990) \emph{Nonlinear Time Series: A Dynamical System Approach}.
Oxford University Press
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Inzenman, A.J. (1983) J.R. Wolf and H.A. Wolfer: An historical note on the
Zurich sunspot relative numbers. 
\emph{Journal of the Royal Statistical Society, A}, \bold{146},
311-318.


Waldmeir, M. (1961) \emph{The Sunspot Activity in the Years 1610-1960}.
Schulthess and Co.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{survival}
\alias{survival}
\title{
Survival of Rats after Radiation Doses
}
\description{
The \code{survival} data frame has 14 rows and 2 columns.


The data measured the survival percentages of batches of rats who were given 
varying doses of radiation.  At each of 6 doses there were two or three
replications of the experiment.
}
\usage{
data(survival)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{dose}}{
The dose of radiation administered (rads).
}
\item{\code{surv}}{
The survival rate of the batches expressed as a percentage.
}}}
\source{
The data were obtained from


Efron, B. (1988) Computer-intensive methods in statistical regression.
\emph{SIAM Review}, \bold{30}, 421--449.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{tau}
\alias{tau}
\title{
Tau Particle Decay Modes
}
\description{
The \code{tau} data frame has 60 rows and 2 columns.


The tau particle is a heavy electron-like particle discovered in the 1970's
by Martin Perl at the Stanford Linear Accelerator Center.  Soon after its
production the tau particle decays into various collections of more stable
particles.  About 86\% of the time the decay involves just one charged particle.
This rate has been measured independently 13 times.  


The one-charged-particle event is made up of four major modes of decay as well
as a collection of other events.  The four main types of decay are denoted
rho, pi, e and mu.  These rates have been measured independently 6, 7, 14 and
19 times respectively.  Due to physical constraints each experiment can only
estimate the composite one-charged-particle decay rate or the rate of one of
the major modes of decay.  


Each experiment consists of a major research project involving many years work.
One of the goals of the experiments was to estimate the rate of decay due to 
events other than the four main modes of decay.  These are uncertain events and
so cannot themselves be observed directly.
}
\usage{
data(tau)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{rate}}{
The decay rate expressed as a percentage.
}
\item{\code{decay}}{
The type of decay measured in the experiment.  It is a factor with levels
\code{1}, \code{rho}, \code{pi}, \code{e} and \code{mu}.
}}}
\source{
The data were obtained from


Efron, B. (1992) Jackknife-after-bootstrap standard errors and influence 
functions (with Discussion). 
\emph{Journal of the Royal Statistical Society, B}, \bold{54}, 83--127.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Hayes, K.G., Perl, M.L. and Efron, B. (1989) Application of the bootstrap
statistical method to the tau-decay-mode problem. 
\emph{Physical Review, D}, \bold{39},
274-279.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{tilt.boot}
\alias{tilt.boot}
\title{
Non-parametric Tilted Bootstrap
}
\description{
This function will run an initial bootstrap with equal resampling 
probabilities (if required) and will use the output of the initial run to 
find resampling probabilities which put the
value of the statistic at required values.  It then runs an importance
resampling bootstrap using the calculated probabilities as the resampling
distribution.
}
\usage{
tilt.boot(data, statistic, R, sim="ordinary", stype="i", 
          strata=rep(1, n), L=NULL, theta=NULL, 
          alpha=c(0.025, 0.975), tilt=TRUE, width=0.5, 
          index=1, \dots)
}
\arguments{
\item{data}{
The data as a vector, matrix or data frame.  If it is a matrix or data frame 
then each row is considered as one (multivariate) observation.
}
\item{statistic}{
A function which when applied to data returns a vector containing the
statistic(s) of interest.  It must take at least two arguments.  The first
argument will always be \code{data} and the second should be a vector of indices,
weights or frequencies describing the bootstrap sample.  Any other arguments
must be supplied to \code{tilt.boot} and will be passed unchanged to statistic each
time it is called.
}
\item{R}{
The number of bootstrap replicates required.  This will generally be a vector,
the first value stating how many uniform bootstrap simulations are to be 
performed at the initial stage.  The remaining values of \code{R} are the number of 
simulations to be performed resampling from each reweighted distribution.  
The first value of
\code{R} must always be present, a value of 0 implying that no uniform resampling is
to be carried out.  Thus \code{length(R)} should always equal \code{1+length(theta)}.
}
\item{sim}{
This is a character string indicating the type of bootstrap simulation required.
There are only two possible values that this can take: \code{"ordinary"} and
\code{"balanced"}.  If other simulation types are required for the initial
un-weighted bootstrap then it will be necessary to run
\code{boot}, calculate the weights appropriately, and run \code{boot} again using the
calculated weights.
}
\item{stype}{
A character string indicating the type of second argument expected by 
\code{statistic}.  The possible values that \code{stype} can take are \code{"i"} (indices), 
\code{"w"} (weights) and \code{"f"} (frequencies).
}
\item{strata}{
An integer vector or factor representing the strata for multi-sample problems.
}
\item{L}{
The empirical influence values for the statistic of interest.  They are used
only for exponential tilting when \code{tilt} is \code{TRUE}.  If \code{tilt} is \code{TRUE} and 
they are not supplied then \code{tilt.boot} uses \code{empinf} to calculate them.
}
\item{theta}{
The required parameter value(s) for the tilted distribution(s).  There should be
one value of \code{theta} for each of the non-uniform distributions.  If \code{R[1]}
is 0 \code{theta} is a required argument.  Otherwise \code{theta} values can be estimated 
from the initial uniform bootstrap and the values in \code{alpha}.
}
\item{alpha}{
The alpha level to which tilting is required.  This parameter is ignored if
\code{R[1]} is 0 or if \code{theta} is supplied, otherwise it is used to find the values 
of \code{theta} as quantiles of the initial uniform bootstrap.  In this case \code{R[1]} 
should  be large enough that \code{min(c(alpha, 1-alpha))*R[1] > 5}, if this is not 
the case then a warning is generated to the effect that the \code{theta} are extreme 
values and so the tilted output may be unreliable.  
}
\item{tilt}{
A logical variable which if \code{TRUE} (the default) indicates that exponential 
tilting should be used, otherwise local frequency smoothing (\code{smooth.f}) is 
used.  If \code{tilt} is \code{FALSE} then \code{R[1]} must be positive.  In fact in this case 
the value of \code{R[1]} should be fairly large (in the region of 500 or more).
}
\item{width}{
This argument is used only if \code{tilt} is \code{FALSE}, in which case it is passed 
unchanged to \code{smooth.f} as the standardized bandwidth for the smoothing 
operation.  The value should generally be in the range (0.2, 1). See \code{smooth.f}
for for more details.
}
\item{index}{
The index of the statistic of interest in the output from \code{statistic}.  By 
default the first element of the output of \code{statistic} is used.
}
\item{...}{
Any additional arguments required by \code{statistic}.  These are passed unchanged to
\code{statistic} each time it is called.
}}
\value{
An object of class \code{"boot"} with the following components

\item{t0}{
The observed value of the statistic on the original data.
}
\item{t}{
The values of the bootstrap replicates of the statistic.  There will be \code{sum(R)}
of these, the first \code{R[1]} corresponding to the uniform bootstrap and the 
remainder to the tilted bootstrap(s).
}
\item{R}{
The input vector of the number of bootstrap replicates.
}
\item{data}{
The original data as supplied.
}
\item{statistic}{
The \code{statistic} function as supplied.
}
\item{sim}{
The simulation type used in the bootstrap(s), it can either be \code{"ordinary"} or
\code{"balanced"}.
}
\item{stype}{
The type of statistic supplied, it is the same as the input value \code{stype}.
}
\item{call}{
A copy of the original call to \code{tilt.boot}.
}
\item{strata}{
The strata as supplied.
}
\item{weights}{
The matrix of weights used.  If \code{R[1]} is greater than 0 then the first row 
will be the uniform weights and each subsequent row the tilted weights.  
If \code{R[1]} equals 0 then the uniform
weights are omitted and only the tilted weights are output.
}
\item{theta}{
The values of \code{theta} used for the tilted distributions.  These are either the
input values or the values derived from the uniform bootstrap and \code{alpha}.
}}
\references{
Booth, J.G., Hall, P. and Wood, A.T.A. (1993) Balanced importance resampling 
for the bootstrap. \emph{Annals of Statistics}, \bold{21}, 286--298.


Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.


Hinkley, D.V. and Shi, S. (1989) Importance sampling and the nested bootstrap. 
\emph{Biometrika}, \bold{76}, 435--446.
}
\seealso{
\code{\link{boot}}, \code{\link{exp.tilt}}, \code{\link{Imp.Estimates}}, \code{\link{imp.weights}}, \code{\link{smooth.f}}
}
\examples{
# Note that these examples can take a while to run.


# Example 9.9 of Davison and Hinkley (1997).
data(gravity)
grav1 <- gravity[as.numeric(gravity[,2])>=7,]
grav.fun <- function(dat, w, orig)
{    strata <- tapply(dat[, 2], as.numeric(dat[, 2]))
     d <- dat[, 1]
     ns <- tabulate(strata)
     w <- w/tapply(w, strata, sum)[strata]
     mns <- tapply(d * w, strata, sum)
     mn2 <- tapply(d * d * w, strata, sum)
     s2hat <- sum((mn2 - mns^2)/ns)
     c(mns[2]-mns[1],s2hat,(mns[2]-mns[1]-orig)/sqrt(s2hat),)
}
grav.z0 <- grav.fun(grav1,rep(1,26),0)
tilt.boot(grav1, grav.fun, R=c(249,375,375), stype="w", 
          strata=grav1[,2], tilt=TRUE, index=3, orig=grav.z0[1]) 


#  Example 9.10 of Davison and Hinkley (1997) requires a balanced 
#  importance resampling bootstrap to be run.  In this example we 
#  show how this might be run.  
acme.fun <- function(data, i, bhat)
{    d <- data[i,]
     n <- nrow(d)
     d.lm <- glm(d$acme~d$market)
     beta.b <- coef(d.lm)[2]
     d.diag <- glm.diag(d.lm)
     SSx <- (n-1)*var(d$market)
     tmp <- (d$market-mean(d$market))*d.diag$res*d.diag$sd
     sr <- sqrt(sum(tmp^2))/SSx
     c(beta.b, sr, (beta.b-bhat)/sr)
}
data(acme)
acme.b <- acme.fun(acme,1:nrow(acme),0)
acme.boot1 <- tilt.boot(acme, acme.fun, R=c(499, 250, 250), 
                        stype="i", sim="balanced", alpha=c(0.05, 0.95), 
                        tilt=TRUE, index=3, bhat=acme.b[1])
}
\keyword{nonparametric}
% Converted by Sd2Rd version 1.15.

\eof
\name{tsboot}
\alias{tsboot}
\alias{ts.return}
\title{
Bootstrapping of Time Series
}
\description{
Generate \code{R} bootstrap replicates of a statistic applied to a time series.  The
replicate time series can be generated using fixed or random block lengths or
can be model based replicates.   
}
\usage{
tsboot(tseries, statistic, R, l=NULL, sim="model", endcorr=TRUE, 
       n.sim=NROW(tseries), orig.t=TRUE, ran.gen, 
       ran.args=NULL, norm=TRUE, \dots)
}
\arguments{
\item{tseries}{
  A univariate or multivariate time series.
}
\item{statistic}{
  A function which when applied to \code{tseries} returns a vector
  containing the statistic(s) of interest.  Each time \code{statistic} is
  called it is passed a time series of length \code{n.sim} which is of the
  same class as the original \code{tseries}.  Any other arguments which
  \code{statistic} takes must remain constant for each bootstrap replicate
  and should be supplied through the \dots{} argument to \code{tsboot}.
}
\item{R}{
  A positive integer giving the number of bootstrap replicates required.  
}
\item{sim}{
  The type of simulation required to generate the replicate time series.  The
  possible input values are \code{"model"} (model based resampling),
  \code{"fixed"} (block resampling with fixed block lengths of
  \code{l}), \code{"geom"} (block resampling with block lengths
  having a geometric distribution with mean \code{l}) or
  \code{"scramble"} (phase scrambling).  
}
\item{l}{
  If \code{sim} is \code{"fixed"} then \code{l} is the fixed block
  length used in generating the replicate time series.  If \code{sim} is
  \code{"geom"} then \code{l} is the mean of the geometric distribution
  used to generate the block lengths. \code{l} should be a positive
  integer less than the length of \code{tseries}.  This argument is not
  required when \code{sim} is \code{"model"} but it is required for all
  other simulation types.
}
\item{endcorr}{
  A logical variable indicating whether end corrections are to be
  applied when \code{sim} is \code{"fixed"}.  When \code{sim} is
  \code{"geom"}, \code{endcorr} is automatically set to \code{TRUE};
  \code{endcorr} is not used when \code{sim} is \code{"model"} or
  \code{"scramble"}.
}
\item{n.sim}{
  The length of the simulated time series.  Typically this will be equal
  to the length of the original time series but there are situations when
  it will be larger.  One obvious situation is if prediction is required.
  Another situation in which \code{n.sim} is larger than the original
  length is if \code{tseries} is a residual time series from fitting some
  model to the original time series. In this case, \code{n.sim} would
  usually be the length of the original time series.
}
\item{orig.t}{
  A logical variable which indicates whether \code{statistic} should be
  applied to \code{tseries} itself as well as the bootstrap replicate
  series.  If \code{statistic} is expecting a longer time series than
  \code{tseries} or if applying \code{statistic} to \code{tseries} will
  not yield any useful information then \code{orig.t} should be set to
  \code{FALSE}.
}
\item{ran.gen}{
  This is a function of three arguments.  The first argument is a time
  series.  If \code{sim} is code{"model"} then it will always be
  \code{tseries} that is passed.  For other simulation types it is the
  result of selecting \code{n.sim} observations from \code{tseries} by
  some scheme and converting the result back into a time series of the
  same form as \code{tseries} (although of length \code{n.sim}).  The
  second argument to \code{ran.gen} is always the value \code{n.sim}, and
  the third argument is \code{ran.args}, which is used to supply any other
  objects needed by \code{ran.gen}.  If \code{sim} is \code{"model"} then
  the generation of the replicate time series will be done in
  \code{ran.gen} (for example through use of \code{\link{arima.sim}}).
  For the other simulation types \code{ran.gen} is used for
  \dQuote{post-blackening}.  The default is that the function simply returns
  the time series passed to it.
}
\item{ran.args}{
This will be supplied to \code{ran.gen} each time it is called.  If \code{ran.gen} needs 
any extra arguments then they should be supplied as components of \code{ran.args}.  
Multiple arguments may be passed by making \code{ran.args} a list.  If \code{ran.args} 
is \code{NULL} then it should not be used within \code{ran.gen} but note that \code{ran.gen} 
must still have its third argument.  
}
\item{norm}{
A logical argument indicating whether normal margins should be used for
phase scrambling.  If \code{norm} is \code{FALSE} then margins corresponding to the exact
empirical margins are used.
}
\item{...}{
Any extra arguments to \code{statistic} may be supplied here.
}}
\value{
An object of class \code{"boot"} with the following components.

\item{t0}{
If \code{orig.t} is \code{TRUE} then \code{t0} is the result of \code{statistic(tseries,\dots{})} 
otherwise it is \code{NULL}.
}
\item{t}{
The results of applying \code{statistic} to the replicate time series. 
}
\item{R}{
The value of \code{R} as supplied to \code{tsboot}.
}
\item{tseries}{
The original time series.
}
\item{statistic}{
The function \code{statistic} as supplied.
}
\item{sim}{
The simulation type used in generating the replicates.
}
\item{endcorr}{
The value of \code{endcorr} used.  The value is meaningful only when \code{sim} is
\code{"fixed"}; it is ignored for model based simulation or phase scrambling
and is always set to \code{TRUE} if \code{sim} is \code{"geom"}.
}
\item{n.sim}{
The value of \code{n.sim} used.
}
\item{l}{
The value of \code{l} used for block based resampling.  This will be \code{NULL} if 
block based resampling was not used.
}
\item{ran.gen}{
The \code{ran.gen} function used for generating the series or for "post-blackening".
}
\item{ran.args}{
The extra arguments passed to \code{ran.gen}.
}
\item{call}{
The original call to \code{tsboot}.
}}
\details{
If \code{sim} is \code{"fixed"} then each replicate time series is found by taking 
blocks of length \code{l}, from the original time series and putting them 
end-to-end until a new series of length \code{n.sim} is created.  When \code{sim} is 
\code{"geom"} a similar approach is taken except that now the block lengths are 
generated from a geometric distribution with mean \code{l}.   Post-blackening can 
be carried out on these replicate time series by including the function 
\code{ran.gen} in the call to \code{tsboot} and having \code{tseries} as a time series of 
residuals.  


Model based resampling is very similar to the parametric bootstrap
and all simulation must be in one of the user specified functions.  This 
avoids the complicated problem of choosing the block length but relies on an 
accurate model choice being made.


Phase scrambling is described in Section 8.2.4 of Davison and Hinkley (1997).
The types of statistic for which this method produces reasonable results is
very limited and the other methods seem to do better in most situations.
Other types of resampling in the frequency domain
can be accomplished using the function \code{boot} with the argument
\code{sim="parametric"}.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.

Kunsch, H.R. (1989) The jackknife and the bootstrap for general stationary
observations. \emph{Annals of Statistics}, \bold{17}, 1217--1241.

Politis, D.N. and Romano, J.P. (1994) The stationary bootstrap. 
\emph{Journal of the American Statistical Association}, \bold{89}, 1303--1313.
}
\seealso{
  \code{\link{boot}}, \code{\link[ts]{arima.sim}}
}
\examples{
library(ts)
data(lynx)
lynx.fun <- function(tsb) 
{    ar.fit <- ar(tsb, order.max=25)
     c(ar.fit$order, mean(tsb), tsb)
}

# the stationary bootstrap with mean block length 20
lynx.1 <- tsboot(log(lynx), lynx.fun, R=99, l=20, sim="geom")

# the fixed block bootstrap with length 20
lynx.2 <- tsboot(log(lynx), lynx.fun, R=99, l=20, sim="fixed")

# Now for model based resampling we need the original model
# Note that for all of the bootstraps which use the residuals as their
# data, we set orig.t to FALSE since the function applied to the residual
# time series will be meaningless.
lynx.ar <- ar(log(lynx))
lynx.model <- list(order=c(lynx.ar$order,0,0),ar=lynx.ar$ar)
lynx.res <- lynx.ar$resid[!is.na(lynx.ar$resid)]
lynx.res <- lynx.res - mean(lynx.res)

lynx.sim <- function(res,n.sim, ran.args) {
# random generation of replicate series using arima.sim 
     rg1 <- function(n, res)
          sample(res, n, replace=TRUE)
     ts.orig <- ran.args$ts
     ts.mod <- ran.args$model
     mean(ts.orig)+ts(arima.sim(model=ts.mod, n=n.sim,
                      rand.gen=rg1, res=as.vector(res)))
}

lynx.3 <- tsboot(lynx.res, lynx.fun, R=99, sim="model", n.sim=114,
                 orig.t=FALSE, ran.gen=lynx.sim, 
                 ran.args=list(ts=log(lynx), model=lynx.model))

#  For "post-blackening" we need to define another function
lynx.black <- function(res, n.sim, ran.args) 
{    ts.orig <- ran.args$ts
     ts.mod <- ran.args$model
     mean(ts.orig) + ts(arima.sim(model=ts.mod,n=n.sim,innov=res))
}

# Now we can run apply the two types of block resampling again but this
# time applying post-blackening.
lynx.1b <- tsboot(lynx.res, lynx.fun, R=99, l=20, sim="fixed",
                  n.sim=114, orig.t=FALSE, ran.gen=lynx.black, 
                  ran.args=list(ts=log(lynx), model=lynx.model))

lynx.2b <- tsboot(lynx.res, lynx.fun, R=99, l=20, sim="geom",
                  n.sim=114, orig.t=FALSE, ran.gen=lynx.black, 
                  ran.args=list(ts=log(lynx), model=lynx.model))

# To compare the observed order of the bootstrap replicates we
# proceed as follows.
table(lynx.1$t[,1])
table(lynx.1b$t[,1])
table(lynx.2$t[,1])
table(lynx.2b$t[,1])
table(lynx.3$t[,1])
# Notice that the post-blackened and model-based bootstraps preserve
# the true order of the model (11) in many more cases than the others.
}
\keyword{nonparametric}
\keyword{ts}

\eof
\name{tuna}
\alias{tuna}
\title{
Tuna Sighting Data
}
\description{
The \code{tuna} data frame has 64 rows and 1 columns.  


The data come from an aerial
line transect survey of Southern Bluefin Tuna in the Great Australian Bight.
An aircraft with two spotters on board flies randomly allocated line transects.
Each school of tuna sighted is counted and its perpendicular distance from the
transect measured.  The survey was conducted in summer when tuna tend to stay
on the surface.
}
\usage{
data(tuna)
}
\format{
This data frame contains the following column:


\describe{
\item{\code{y}}{
The perpendicular distance, in miles, from the transect for 64 independent 
sightings of tuna schools.
}}}
\source{
The data were obtained from


Chen, S.X. (1996) Empirical likelihood confidence intervals for
nonparametric density estimation. \emph{Biometrika}, \bold{83}, 329--341.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{urine}
\alias{urine}
\title{
Urine Analysis Data
}
\description{
The \code{urine} data frame has 79 rows and 7 columns.


79 urine specimens were analyzed in an effort to determine if certain physical
characteristics of the urine might be related to the formation of calcium
oxalate crystals.  
}
\usage{
data(urine)
}
\format{
This data frame contains the following columns:


\describe{
\item{\code{r}}{
Indicator of the presence of calcium oxalate crystals.
}
\item{\code{gravity}}{
The specific gravity of the urine.
}
\item{\code{ph}}{
The pH reading of the urine.
}
\item{\code{osmo}}{
The osmolarity of the urine. Osmolarity is proportional to the concentration
of molecules in solution.
}
\item{\code{cond}}{
The conductivity of the urine.  Conductivity is proportional to the 
concentration of charged ions in solution.
}
\item{\code{urea}}{
The urea concentration in millimoles per litre.
}
\item{\code{calc}}{
The calcium concentration in millimoles per litre.
}}}
\source{
The data were obtained from


Andrews, D.F. and Herzberg, A.M. (1985) 
\emph{Data: A Collection of Problems from Many Fields for the Student and Research Worker}. Springer-Verlag.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) 
\emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
\name{var.linear}
\alias{var.linear}
\title{
Linear Variance Estimate
}
\description{
Estimates the variance of a statistic from its empirical influence values.
}
\usage{
var.linear(L, strata=NULL)
}
\arguments{
\item{L}{
Vector of the empirical influence values of a statistic.  These will usually
be calculated by a call to \code{empinf}.
}
\item{strata}{
A numeric vector or factor specifying which  observations (and hence empirical
influence values) come from which strata.
}}
\value{
The variance estimate calculated from \code{L}.
}
\references{
Davison, A. C. and Hinkley, D. V. (1997) \emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\seealso{
\code{\link{empinf}}, \code{\link{linear.approx}}, \code{\link{k3.linear}}
}
\examples{
#  To estimate the variance of the ratio of means for the city data.
data(city)
ratio <- function(d,w)
     sum(d$x * w)/sum(d$u * w)
var.linear(empinf(data=city,statistic=ratio))
}
\keyword{nonparametric}
% Converted by Sd2Rd version 1.15.

\eof
\name{wool}
\alias{wool}
\title{
Australian Relative Wool Prices
}
\description{
\code{wool} is a time series of class \code{"ts"} and contains 309 observations.

Each week that the market is open the Australian Wool Corporation set a floor
price which determines their policy on intervention and is therefore a 
reflection of the overall price of wool for the week in question.  Actual prices
paid can vary considerably about the floor price.  The series here is the log
of the ratio between the price for fine grade wool and the floor price, each
market week between July 1976 and Jun 1984.
}
\source{
The data were obtained from


Diggle, P.J. (1990) \emph{Time Series: A Biostatistical Introduction}.
Oxford University Press.
}
\references{
Davison, A.C. and Hinkley, D.V. (1997) \emph{Bootstrap Methods and Their Application}. Cambridge University Press.
}
\keyword{datasets}
% Converted by Sd2Rd version 1.15.

\eof
