\name{asVector}
\alias{asVector}
%\alias{asVector.xyVector}
\title{Coerce an Object to a Vector}
\description{
  This is a generic function.  Methods for this function coerce objects
  of given classes to vectors.
}
\usage{
asVector(object)
}
\arguments{
  \item{object}{An object.}
}
\value{
  a vector
}
\details{
  Methods for vector coercion in new classes must be created for the
  \code{asVector} generic instead of \code{as.vector}.  The
  \code{as.vector} function is internal and not easily extended.
  Currently the only class with an \code{asVector} method is the
  \code{xyVector} class.
}
\author{Douglas Bates and Bill Venables}
\seealso{
  \code{\link{xyVector}}
}
\examples{
data( women )
ispl <- interpSpline( weight ~ height,  women )
pred <- predict(ispl)
class(pred)
utils::str(pred)
asVector(pred)
}
\keyword{models}
\eof
\name{backSpline}
\alias{backSpline}
%\alias{backSpline.nbSpline}
%\alias{backSpline.npolySpline}
\title{Monotone Inverse Spline}
\description{
  Create a monotone inverse of a monotone natural spline.
}
\usage{
backSpline(object)
}
\arguments{
  \item{object}{an object that inherits from class \code{nbSpline} or
    \code{npolySpline}.  That is, the object must represent a natural
    interpolation spline but it can be either in the B-spline
    representation or the piecewise polynomial one.  The spline is
    checked to see if it represents a monotone function.
  }
}
\value{
  An object of class \code{polySpline} that contains the piecewise
  polynomial representation of a function that has the appropriate
  values and derivatives at the knot positions to be an inverse of the
  spline represented by \code{object}.  Technically this object is not a
  spline because the second derivative is not constrained to be
  continuous at the knot positions.  However, it is often a much better
  approximation to the inverse than fitting an interpolation spline to
  the y/x pairs.
}
\author{Douglas Bates and Bill Venables}
\seealso{
  \code{\link{interpSpline}}
}
\examples{
data( women )
ispl <- interpSpline( women$height, women$weight )
bspl <- backSpline( ispl )
plot( bspl )                   # plots over the range of the knots
points( women$weight, women$height )
}
\keyword{ models }
\eof
\name{bs}
\alias{bs}
%\alias{makepredictcall.bs}
\title{Generate a Basis for Polynomial Splines}
\description{
  Generate the B-spline basis matrix for a polynomial spline.
}
\usage{
bs(x, df = NULL, knots = NULL, degree = 3, intercept = FALSE,
   Boundary.knots = range(x))
}
\arguments{
  \item{x}{the predictor variable.}
  \item{df}{degrees of freedom; one can specify \code{df} rather than
    \code{knots}; \code{bs()} then chooses \code{df-degree-1} knots at
    suitable quantiles of \code{x}.}
  \item{knots}{the \emph{internal} breakpoints that define the
    spline.  The default is \code{NULL}, which results in a basis for
    ordinary polynomial regression. Typical values are the mean or
    median for one knot, quantiles for more knots. See also
    \code{Boundary.knots}.}
  \item{degree}{degree of the piecewise polynomial---default is 3 for
    cubic splines.}
  \item{intercept}{if \code{TRUE}, an intercept is included in the
    basis; default is \code{FALSE}.}
  \item{Boundary.knots}{boundary points at which to anchor the B-spline
    basis (default the range of the data). If both \code{knots} and
    \code{Boundary.knots} are supplied, the basis parameters do not
    depend on \code{x}. Data can extend beyond \code{Boundary.knots}.}
}
\value{
  A matrix of dimension \code{length(x) * df}, where either \code{df}
  was supplied or if \code{knots} were supplied,
  \code{df = length(knots) + 3 + intercept}.  Attributes are returned
  that  correspond to the arguments to \code{bs}, and explicitly give
  the \code{knots}, \code{Boundary.knots} etc for use by
  \code{predict.bs()}.

  \code{bs()} is based on the function \code{\link{spline.des}()}.
  It generates a basis matrix for
  representing the family of piecewise polynomials with the specified
  interior knots and degree, evaluated at the values of \code{x}.  A
  primary use is in modeling formulas to directly specify a piecewise
  polynomial term in a model.

%   Beware of making predictions with new \code{x} values when \code{df}
%   is used as an argument.  Either use \code{safe.predict.gam()}, or else
%   specify \code{knots} and \code{Boundary.knots}.
}
\seealso{
  \code{\link{ns}}, \code{\link{poly}}, \code{\link[stats]{smooth.spline}},
  \code{\link{predict.bs}}, \code{\link{SafePrediction}}
}
\references{
  Hastie, T. J. (1992)
  Generalized additive models.
  Chapter 7 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\examples{
data(women)
bs(women$height, df = 5)
summary(fm1 <- lm(weight ~ bs(height, df = 5), data = women))

## example of safe prediction
plot(women, xlab = "Height (in)", ylab = "Weight (lb)")
ht <- seq(57, 73, len = 200)
lines(ht, predict(fm1, data.frame(height=ht)))

## Consistency:
x <- c(1:3,5:6)
stopifnot(identical(bs(x), bs(x, df = 3)),
          !is.null(kk <- attr(bs(x), "knots")),# not true till 1.5.1
          length(kk) == 0)
}
\keyword{smooth}
\eof
\name{interpSpline}
\alias{interpSpline}
%\alias{interpSpline.default}
%\alias{interpSpline.formula}
%\alias{print.bSpline}
%\alias{print.ppolySpline}
\title{Create an Interpolation Spline}
\description{
  Create an interpolation spline, either from \code{x} and \code{y}
  vectors, or from a formula/data.frame combination.
}
\usage{
interpSpline(obj1, obj2, bSpline = FALSE, period = NULL,
             na.action = na.fail)
}
\arguments{
 \item{obj1}{Either a numeric vector of \code{x} values or a formula.}
 \item{obj2}{If \code{obj1} is numeric this should be a numeric vector
   of the same length.  If \code{obj1} is a formula this can be an
   optional data frame in which to evaluate the names in the formula.}
 \item{bSpline}{If \code{TRUE} the b-spline representation is returned,
   otherwise the piecewise polynomial representation is returned.
   Defaults to \code{FALSE}.}
 \item{period}{An optional positive numeric value giving a period for a
   periodic interpolation spline.}
 \item{na.action}{a optional function which indicates what should happen
   when the data contain \code{NA}s.  The default action
   (\code{na.omit}) is to omit any incomplete observations.  The
   alternative action \code{na.fail} causes \code{interpSpline} to print
   an error message and terminate if there are any incomplete
   observations.}
}
\value{
  An object that inherits from class \code{spline}. The object can be in
  the B-spline representation, in which case it will be of class
  \code{nbSpline} for natural B-spline, or in the piecewise polynomial
  representation, in which case it will be of class \code{npolySpline}.
}
\author{Douglas Bates and Bill Venables}
\seealso{
  \code{\link{splineKnots}},
  \code{\link{splineOrder}},
  \code{\link{periodicSpline}}.
}
\examples{
data( women )
ispl <- interpSpline( women$height, women$weight )
ispl2 <- interpSpline( weight ~ height,  women )
# ispl and ispl2 should be the same
plot( predict( ispl, seq( 55, 75, len = 51 ) ), type = "l" )
points( women$height, women$weight )
plot( ispl )    # plots over the range of the knots
points( women$height, women$weight )
splineKnots( ispl )
}
\keyword{ models }
\eof
\name{ns}
\alias{ns}
%\alias{makepredictcall.ns}
\title{Generate a Basis Matrix for Natural Cubic Splines}
\description{
  Generate the B-spline basis matrix for a natural cubic spline.
}
\usage{
ns(x, df = NULL, knots = NULL, intercept = FALSE,
   Boundary.knots = range(x))
}
\arguments{
  \item{x}{the predictor variable.}
  \item{df}{degrees of freedom. One can supply \code{df} rather than
    knots; \code{ns()} then chooses \code{df - 1 - intercept} knots at
    suitably chosen quantiles of \code{x}.}
  \item{knots}{breakpoints that define the spline. The default is no
    knots; together with the natural boundary conditions this results in
    a basis for linear regression on \code{x}.  Typical values are the
    mean or median for one knot, quantiles for more knots. See also
    \code{Boundary.knots}.}
  \item{intercept}{if \code{TRUE}, an intercept is included in the
    basis; default is \code{FALSE}.}
  \item{Boundary.knots}{boundary points at which to impose the natural
    boundary conditions and anchor the B-spline basis (default the range
    of the data).  If both \code{knots} and \code{Boundary.knots} are
    supplied, the basis parameters do not depend on \code{x}. Data can
    extend beyond \code{Boundary.knots}}
}
\value{
    A matrix of dimension \code{length(x) * df} where either \code{df} was
    supplied or if \code{knots} were supplied,
    \code{df = length(knots) + 1 + intercept}.
    Attributes are returned that correspond to the arguments to \code{ns},
    and explicitly give the \code{knots}, \code{Boundary.knots} etc for
    use by \code{predict.ns()}.

    \code{ns()} is based on the function \code{\link{spline.des}}.  It
    generates a basis matrix for representing the family of
    piecewise-cubic splines with the specified sequence of
    interior knots, and the natural boundary conditions.  These enforce
    the constraint that the function is linear beyond the boundary knots,
    which can either be supplied, else default to the extremes of the
    data.  A primary use is in modeling formula to directly specify a
    natural spline term in a model.

%     Beware of making predictions with new \code{x} values when \code{df}
%     is used as an argument. Either use \code{safe.predict.gam()}, or else
%     specify \code{knots} and \code{Boundary.knots}.
}
\references{
  Hastie, T. J. (1992)
  Generalized additive models.
  Chapter 7 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{bs}},
  \code{\link{poly}},
  \code{\link{predict.ns}}, \code{\link{SafePrediction}}
}
\examples{
data(women)
ns(women$height, df = 5)
summary(fm1 <- lm(weight ~ ns(height, df = 5), data = women))

## example of safe prediction
plot(women, xlab = "Height (in)", ylab = "Weight (lb)")
ht <- seq(57, 73, len = 200)
lines(ht, predict(fm1, data.frame(height=ht)))
}
\keyword{smooth}
\eof
\name{periodicSpline}
\alias{periodicSpline}
%\alias{periodicSpline.default}
%\alias{periodicSpline.formula}
\title{Create a Periodic Interpolation Spline}
\description{
  Create a periodic interpolation spline, either from \code{x} and
  \code{y} vectors, or from a formula/data.frame combination.
}
\usage{
periodicSpline(obj1, obj2, knots, period = 2*pi, ord = 4)
}
\arguments{
 \item{obj1}{either a numeric vector of \code{x} values or a formula.}
 \item{obj2}{if \code{obj1} is numeric this should be a numeric vector
   of the same length.  If \code{obj1} is a formula this can be an
   optional data frame in which to evaluate the names in the formula.}
 \item{knots}{optional numeric vector of knot positions.}
 \item{period}{positive numeric value giving the period for the
   periodic spline.  Defaults to \code{2 * pi}.}
 \item{ord}{integer giving the order of the spline, at least 2.  Defaults
   to 4.  See \code{\link{splineOrder}} for a definition of the order of
   a spline.}
}
\value{
  An object that inherits from class \code{spline}.  The object can be in
  the B-spline representation, in which case it will be a
  \code{pbSpline} object, or in the piecewise polynomial representation
  (a \code{ppolySpline} object).
}
\author{Douglas Bates and Bill Venables}
\seealso{
  \code{\link{splineKnots}},
  \code{\link{interpSpline}}
}
\examples{
xx <- seq( -pi, pi, len = 16 )[-1]
yy <- sin( xx )
frm <- data.frame( xx, yy )
( pispl <- periodicSpline( xx, yy, period = 2 * pi ) )
pispl2 <- periodicSpline( yy ~ xx, frm, period = 2 * pi )
stopifnot(all.equal(pispl, pispl2))# pispl and pispl2 are the same

plot( pispl )          # displays over one period
points( yy ~ xx, col = "brown")
plot( predict( pispl, seq(-3*pi, 3*pi, len = 101) ), type = "l" )
}
\keyword{models}
\eof
\name{polySpline}
\alias{polySpline}
%\alias{polySpline.bSpline}
%\alias{polySpline.nbSpline}
%\alias{polySpline.pbSpline}
%\alias{polySpline.polySpline}
\alias{as.polySpline}
%\alias{predict.polySpline}
%\alias{print.polySpline}
\title{Piecewise Polynomial Spline Representation}
\description{
  Create the piecewise polynomial representation of a spline object.
}
\usage{
polySpline(object, \dots)
as.polySpline(object, \dots)
}
\arguments{
 \item{object}{An object that inherits from class \code{spline}.}
 \item{\dots}{Optional additional arguments.  At present no additional
     arguments are used.}
}
\value{
  An object that inherits from class \code{polySpline}.  This is the
  piecewise polynomial representation of a univariate spline function.
  It is defined by a set of distinct numeric values called knots.  The
  spline function is a polynomial function between each successive pair
  of knots.  At each interior knot the polynomial segments on each side
  are constrained to have the same value of the function and some of its
  derivatives.
}
\author{Douglas Bates and Bill Venables}
\seealso{
  \code{\link{interpSpline}},
  \code{\link{periodicSpline}},
  \code{\link{splineKnots}},
  \code{\link{splineOrder}}
}
\examples{
data( women )
ispl <- polySpline( interpSpline( weight ~ height,  women, bSpline = TRUE ) )
print( ispl )   # print the piecewise polynomial representation
plot( ispl )    # plots over the range of the knots
points( women$height, women$weight )
}
\keyword{ models }
\eof
\name{predict.bSpline}
\alias{predict.bSpline}
\alias{predict.nbSpline}
\alias{predict.pbSpline}
\alias{predict.npolySpline}
\alias{predict.ppolySpline}
%\alias{plot.spline}
\title{Evaluate a spline at new values of x}
\description{
  The \code{predict} methods for the classes that inherit from the
  virtual classes \code{bSpline} and \code{polySpline} are used to
  evaluate the spline or its derivatives.  The \code{plot} method for a
  spline object first evaluates \code{predict} with the \code{x}
  argument missing, then plots the resulting \code{xyVector} with
  \code{type = "l"}.}
\usage{
\method{predict}{bSpline}(object, x, nseg=50, deriv=0, \dots)
%plot.spline(x, \dots)
}
% \synopsis{
% predict.bSpline(object, x, nseg=50, deriv=0, \dots)
% predict.nbSpline(object, x, nseg=50, deriv=0, \dots)
% predict.pbSpline(object, x, nseg=50, deriv=0, \dots)
% predict.npolySpline(object, x, nseg=50, deriv=0, \dots)
% predict.ppolySpline(object, x, nseg=50, deriv=0, \dots)
% plot.spline(x, \dots)
% }
\arguments{
  \item{object}{An object that inherits from the \code{bSpline} or the
    \code{polySpline} class.  For \code{plot.spline} this argument is
    called \code{x}.}
  \item{x}{A numeric vector of \code{x} values at which to evaluate the
    spline.  If this argument is missing a suitable set of \code{x}
    values is generated as a sequence of \code{nseq} segments spanning
    the range of the knots.  For \code{plot.spline} the \code{x}
    argument is as described under \code{object} above.}
  \item{nseg}{A positive integer giving the number of segments in a set
    of equally-spaced \code{x} values spanning the range of the knots
    in \code{object}.  This value is only used if \code{x} is missing.}
  \item{deriv}{An integer between 0 and \code{splineOrder(object) - 1}
    specifying the derivative to evaluate.}
  \item{\dots}{
    \code{predict}: further arguments passed to or from other methods.

    \code{plot}: additional graphical parameters (see \code{link{par}}).
  }
}
\value{
  an \code{xyVector} with components
  \item{x}{the supplied or inferred numeric vector of \code{x} values}
  \item{y}{the value of the spline (or its \code{deriv}'th derivative)
      at the \code{x} vector}
}
\author{Douglas Bates and Bill Venables}
\seealso{
  \code{\link{xyVector}},
  \code{\link{interpSpline}},
  \code{\link{periodicSpline}}
}
\examples{
data( women )
ispl <- interpSpline( weight ~ height,  women )
opar <- par(mfrow = c(2, 2), las = 1)
plot(predict(ispl, nseg = 201),     # plots over the range of the knots
     main = "Original data with interpolating spline", type = "l",
     xlab = "height", ylab = "weight") 
points(women$height, women$weight, col = 4)
plot(predict(ispl, nseg = 201, deriv = 1),
     main = "First derivative of interpolating spline", type = "l",
     xlab = "height", ylab = "weight") 
plot(predict(ispl, nseg = 201, deriv = 2),
     main = "Second derivative of interpolating spline", type = "l",
     xlab = "height", ylab = "weight") 
plot(predict(ispl, nseg = 401, deriv = 3),
     main = "Third derivative of interpolating spline", type = "l",
     xlab = "height", ylab = "weight") 
par(opar)
}
\keyword{models}
\eof
\name{predict.bs}
\title{Evaluate a Spline Basis}
\alias{predict.bs}
\alias{predict.ns}
\description{
  Evaluate a predefined spline basis at given values.
}
\usage{
\method{predict}{bs}(object, newx, \dots)
%% the 'ns' one now looks identical
}
% \synopsis{
% predict.bs(object, newx, \dots)
% predict.ns(object, newx, \dots)
% }
\arguments{
  \item{object}{the result of a call to \code{\link{bs}} or
    \code{\link{ns}} having attributes describing \code{knots},
    \code{degree}, etc.}
  \item{newx}{the \code{x} values at which evaluations are required.}
  \item{\dots}{Optional additional arguments.  Presently no additional
    arguments are used.}
}
\value{
  An object just like \code{object}, except evaluated at the new values
  of \code{x}.

  These are methods for the generic function \code{\link{predict}} for
  objects inheriting from classes \code{"bs"} or \code{"ns"}.  See
  \code{\link{predict}} for the general behavior of this function.
}
\seealso{
  \code{\link{bs}}, \code{\link{ns}}, \code{\link{poly}}.
  %%-- FIXME: These are not in R (and wont ever ?!): \code{lo}, \code{s}
}
\examples{
data(women)
basis <- ns(women$height, df = 5)
newX <- seq(58, 72, len = 51)
# evaluate the basis at the new data
predict(basis, newX)
}
\keyword{smooth}
\eof
\name{splineDesign}
\alias{splineDesign}
\alias{spline.des}
\title{Design Matrix for B-splines}
\description{
  Evaluate the design matrix for the B-splines defined by \code{knots}
  at the values in \code{x}.
}
\usage{
splineDesign(knots, x, ord = 4, derivs)
spline.des(knots, x, ord = 4, derivs)
}
\arguments{
  \item{knots}{a numeric vector of knot positions with non-decreasing
    values.}
  \item{x}{a numeric vector of values at which to evaluate the B-spline
    functions or derivatives. The values in \code{x} must be between
    \code{knots[ord]} and \code{knots[ length(knots) + 1 - ord ]}.}
  \item{ord}{a positive integer giving the order of the spline function.
    This is the number of coefficients in each piecewise polynomial
    segment, thus a cubic spline has order 4.  Defaults to 4.}
  \item{derivs}{an integer vector of the same length as \code{x} and with
    values between \code{0} and \code{ord - 1}.  The derivative of the
    given order is evaluated at the \code{x} positions.  Defaults to a
    vector of zeroes of the same length as \code{x}.}
}
\value{
  A matrix with \code{length( x )} rows and \code{length( knots ) - ord}
  columns.  The i'th row of the matrix contains the coefficients of the
  B-splines (or the indicated derivative of the B-splines) defined by
  the \code{knot} vector and evaluated at the i'th value of \code{x}.
  Each B-spline is defined by a set of \code{ord} successive knots so
  the total number of B-splines is \code{length(knots)-ord}.
}
\note{The older \code{spline.des} function takes the same arguments but
  returns a list with several components including \code{knots},
  \code{ord}, \code{derivs}, and \code{design}.  The \code{design}
  component is the same as the value of the \code{splineDesign}
  function.
}
\author{Douglas Bates and Bill Venables}
\examples{
splineDesign(knots = 1:10, x = 4:7)
}
\keyword{ models }
\eof
%% $Id: splineKnots.Rd,v 1.4 2003/03/03 18:02:04 ripley Exp $
\name{splineKnots}
\alias{splineKnots}
%\alias{splineKnots.spline}
\title{Knot Vector from a Spline}
\description{
  Return the knot vector corresponding to a spline object.
}
\usage{
splineKnots(object)
}
\arguments{
  \item{object}{an object that inherits from class \code{"spline"}.}
}
\value{
  A non-decreasing numeric vector of knot positions.
}
\author{Douglas Bates and Bill Venables}
\examples{
data( women )
ispl <- interpSpline( weight ~ height, women )
splineKnots( ispl )
}
\keyword{ models }
\eof
\name{splineOrder}
\alias{splineOrder}
%\alias{splineOrder.bSpline}
%\alias{splineOrder.polySpline}
\title{Determine the Order of a Spline}
\description{
  Return the order of a spline object.
}
\usage{
splineOrder(object)
}
\arguments{
 \item{object}{An object that inherits from class \code{"spline"}.}
}
\details{
  The order of a spline is the number of coefficients in each piece of
  the piecewise polynomial representation.  Thus a cubic spline has
  order 4.
}
\value{
  A positive integer.
}
\author{Douglas Bates and Bill Venables}
\seealso{
  \code{\link{splineKnots}},
  \code{\link{interpSpline}},
  \code{\link{periodicSpline}}
}
\examples{
data( women )
splineOrder( interpSpline( weight ~ height, women ) )
}
\keyword{ models }
\eof
%% $Id: xyVector.Rd,v 1.8 2003/03/03 18:02:04 ripley Exp $
\name{xyVector}
\alias{xyVector}
%\alias{as.data.frame.xyVector}
%\alias{plot.xyVector}
\title{Construct an xyVector Object}
\description{
  Create an object to represent a set of x-y pairs.  The resulting
  object can be treated as a matrix or as a data frame or as a vector.
  When treated as a vector it reduces to the \code{y} component only.

  The result of functions such as \code{predict.spline} is returned as
  an \code{xyVector} object so the x-values used to generate the
  y-positions are retained, say for purposes of generating plots.
}
\usage{
xyVector(x, y)
}
\arguments{
  \item{x}{a numeric vector}
  \item{y}{a numeric vector of the same length as \code{x}}
}
\value{
  An object of class \code{xyVector} with components
  \item{x}{a numeric vector}
  \item{y}{a numeric vector of the same length as \code{x}}
}
\author{Douglas Bates and Bill Venables}
\examples{
data( women )
ispl <- interpSpline( weight ~ height, women )
weights <- predict( ispl, seq( 55, 75, len = 51 ))
class( weights )
plot( weights, type = "l", xlab = "height", ylab = "weight" )
points( women$height, women$weight )
weights
}
\keyword{ models }
\eof
