%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
%% Copyright (C) 1990, Ellemtel Telecommunications Systems Laboratories
%% File    : lib.erl
%% Author  : Robert Virding & a cast of thousands
%% Purpose : Some extra useful library functions.

-module(lib).
-copyright('Copyright (c) 1991-97 Ericsson Telecom AB').
-vsn('$Revision: /main/release/free/1').

-export([flush_receive/0,error_message/2, progname/0, nonl/1, send/2,
	 sendw/2, eval_str/1]).

flush_receive() ->
    receive
	Any ->
	    flush_receive()
    after
	0 ->
	    ok
    end.

%%
%% Functions for doing standard system format i/o.
%%

error_message(Format, Args) ->
    io:format("** ~s **\n", [io_lib:format(Format, Args)]).

%% Return the name of the script that starts (this) erlang 
%%

progname() ->
    case init:get_argument(progname) of
	{ok, [[Prog]]} ->
	    list_to_atom(Prog);
	Other ->
	    no_prog_name
    end.

nonl([10]) -> [];
nonl([]) -> [];
nonl([H|T]) -> [H|nonl(T)].


send(To, Msg) -> To ! Msg.

sendw(To, Msg) ->
    To ! {self(), Msg},
    receive 
	Reply -> Reply
    end.



%% eval_str(InStr) -> {ok, OutStr} | {error, ErrStr'}
%%   InStr must represent a body

-define(result(F,D), lists:flatten(io_lib:format(F, D))).

eval_str(Str) when list(Str) ->
    case erl_scan:tokens([], Str, 0) of
	{more, {['"'], _, _}} ->
	    {error, "Unterminated (\") string"};
	{more, {['\''],_,_}} ->
	    {error, "Unterminated (\') string"};
	{more, _} ->
	    {error, "Incomplete form (missing .<cr>)??"};
	{done, {ok, Toks, _}, Rest} ->
	    case all_white(Rest) of
		true ->
		    case erl_parse:parse_exprs(Toks) of
			{ok, Exprs} ->
			    case catch erl_eval:exprs(Exprs, []) of
				{value, Val, _} ->
				    {ok, Val};
				Other ->
				    {error, ?result("*** eval: ~p", [Other])}
			    end;
			{error, {Line, Mod, Args}} ->
			    Msg = ?result("*** ~s",[apply(Mod,format_error,
							     [Args])]),
			    {error, Msg}
		    end;
		false ->
		    {error, ?result("Non-white space found after "
				    "end-of-form :~s", [Rest])}
		end
    end;
eval_str(Bin) when binary(Bin) ->
    eval_str(binary_to_list(Bin)).

all_white([$ |T])  -> all_white(T);
all_white([$\n|T]) -> all_white(T);
all_white([$\t|T]) -> all_white(T);
all_white([])      -> true;
all_white(_)       -> false.

		     
