/* libspectrum.h: the library for dealing with ZX Spectrum emulator files
   Copyright (c) 2001-2003 Philip Kendall, Darren Salt

   $Id: libspectrum.h.in,v 1.101.2.1 2004/01/26 22:41:25 pak21 Exp $

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   Author contact information:

   E-mail: pak21-fuse@srcf.ucam.org
   Postal address: 15 Crescent Road, Wokingham, Berks, RG40 2DB, England

*/

/* NB: This file is autogenerated from libspectrum.h.in. Do not edit
   unless you know what you're doing */

#ifndef LIBSPECTRUM_LIBSPECTRUM_H
#define LIBSPECTRUM_LIBSPECTRUM_H

#ifdef __cplusplus
extern "C" {
#endif				/* #ifdef __cplusplus */

#include <stdarg.h>
#include <stddef.h>
#include <stdlib.h>

#include <gcrypt.h>

/* Standard typedefs */
typedef unsigned char libspectrum_byte;
typedef   signed char libspectrum_signed_byte;
typedef unsigned short libspectrum_word;
typedef   signed short libspectrum_signed_word;
typedef unsigned int libspectrum_dword;
typedef   signed int libspectrum_signed_dword;
typedef unsigned long libspectrum_qword;
typedef   signed long libspectrum_signed_qword;

/* glib replacement (if necessary) */

/*
 * General libspectrum routines
 */

/* Error handling */

/* The various errors which can occur */
typedef enum libspectrum_error {

  LIBSPECTRUM_ERROR_NONE = 0,

  LIBSPECTRUM_ERROR_WARNING,

  LIBSPECTRUM_ERROR_MEMORY,
  LIBSPECTRUM_ERROR_UNKNOWN,
  LIBSPECTRUM_ERROR_CORRUPT,
  LIBSPECTRUM_ERROR_SIGNATURE,
  LIBSPECTRUM_ERROR_SLT,	/* .slt data found at end of a .z80 file */
  LIBSPECTRUM_ERROR_INVALID,	/* Invalid parameter supplied */

  LIBSPECTRUM_ERROR_LOGIC = -1,

} libspectrum_error;

/* Initialisation */

libspectrum_error libspectrum_init( void );

/* Version checking */

int libspectrum_check_version( const char *version );
const char *libspectrum_version( void );
const char *libspectrum_gcrypt_version( void );

/* Error handling */
typedef libspectrum_error
(*libspectrum_error_function_t)( libspectrum_error error,
				 const char *format, va_list ap );

extern libspectrum_error_function_t libspectrum_error_function;

libspectrum_error
libspectrum_default_error_function( libspectrum_error error,
				    const char *format, va_list ap );

/* Attempt to identify a given file */

/* Various types of file we might manage to identify */
typedef enum libspectrum_id_t {

  /* These types present in all versions of libspectrum */

  LIBSPECTRUM_ID_UNKNOWN = 0,		/* Unidentified file */
  LIBSPECTRUM_ID_RECORDING_RZX,		/* RZX input recording */
  LIBSPECTRUM_ID_SNAPSHOT_SNA,		/* .sna snapshot */
  LIBSPECTRUM_ID_SNAPSHOT_Z80,		/* .z80 snapshot */
  LIBSPECTRUM_ID_TAPE_TAP,		/* Z80-style .tap tape image */
  LIBSPECTRUM_ID_TAPE_TZX,		/* TZX tape image */

  /* Below here, present only in 0.1.1 and later */

  LIBSPECTRUM_ID_DISK_DSK,		/* .dsk +3 disk image */
  LIBSPECTRUM_ID_DISK_SCL,		/* .scl TR-DOS disk image */
  LIBSPECTRUM_ID_DISK_TRD,		/* .trd TR-DOS disk image */
  LIBSPECTRUM_ID_CARTRIDGE_DCK,		/* .dck Timex cartridge image */

  /* Below here, present only in 0.2.0 and later */

  LIBSPECTRUM_ID_TAPE_WARAJEVO,		/* Warajevo-style .tap tape image */ 

  LIBSPECTRUM_ID_SNAPSHOT_PLUSD,	/* +D snapshot */
  LIBSPECTRUM_ID_SNAPSHOT_SP,		/* .sp snapshot */
  LIBSPECTRUM_ID_SNAPSHOT_SNP,		/* .snp snapshot */
  LIBSPECTRUM_ID_SNAPSHOT_ZXS,		/* .zxs snapshot (zx32) */
  LIBSPECTRUM_ID_SNAPSHOT_SZX,		/* .szx snapshot (Spectaculator) */

  /* Below here, present only in 0.2.1 and later */

  LIBSPECTRUM_ID_COMPRESSED_BZ2,	/* bzip2 compressed file */
  LIBSPECTRUM_ID_COMPRESSED_GZ,		/* gzip compressed file */

} libspectrum_id_t;

/* And 'classes' of file */
typedef enum libspectrum_class_t {

  LIBSPECTRUM_CLASS_UNKNOWN,

  LIBSPECTRUM_CLASS_CARTRIDGE_TIMEX,	/* Timex cartridges */
  LIBSPECTRUM_CLASS_DISK_PLUS3,		/* +3 disk */
  LIBSPECTRUM_CLASS_DISK_TRDOS,		/* TR-DOS disk */
  LIBSPECTRUM_CLASS_RECORDING,		/* Input recording */
  LIBSPECTRUM_CLASS_SNAPSHOT,		/* Snapshot */
  LIBSPECTRUM_CLASS_TAPE,		/* Tape */

  /* Below here, present only in 0.2.1 and later */

  LIBSPECTRUM_CLASS_COMPRESSED,		/* A compressed file */

} libspectrum_class_t;

libspectrum_error
libspectrum_identify_file( libspectrum_id_t *type, const char *filename,
                           const unsigned char *buffer, size_t length );

libspectrum_error
libspectrum_identify_file_with_class(
  libspectrum_id_t *type, libspectrum_class_t *libspectrum_class,
  const char *filename, const unsigned char *buffer, size_t length );

libspectrum_error
libspectrum_identify_file_raw( libspectrum_id_t *type, const char *filename,
			       const unsigned char *buffer, size_t length );

libspectrum_error
libspectrum_identify_class( libspectrum_class_t *libspectrum_class,
                            libspectrum_id_t type );

/* Different Spectrum variants and their capabilities */

/* The machine types we can handle */
typedef enum libspectrum_machine {

  LIBSPECTRUM_MACHINE_48,
  LIBSPECTRUM_MACHINE_TC2048,
  LIBSPECTRUM_MACHINE_128,
  LIBSPECTRUM_MACHINE_PLUS2,
  LIBSPECTRUM_MACHINE_PENT,
  LIBSPECTRUM_MACHINE_PLUS2A,
  LIBSPECTRUM_MACHINE_PLUS3,

  /* Used by libspectrum_tape_guess_hardware if we can't work out what
     hardware should be used */
  LIBSPECTRUM_MACHINE_UNKNOWN,

  LIBSPECTRUM_MACHINE_16,
  LIBSPECTRUM_MACHINE_TC2068,

} libspectrum_machine;

const char* libspectrum_machine_name( libspectrum_machine type );

/* The various capabilities of the different machines */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_AY;		/* AY-3-8192 */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_128_MEMORY;
                                                  /* 128-style memory paging */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_PLUS3_MEMORY;
                                                   /* +3-style memory paging */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_PLUS3_DISK;
                                                      /* +3-style disk drive */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_TIMEX_MEMORY;
                                                /* Timex-style memory paging */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_TIMEX_VIDEO;
                                                  /* Timex-style video modes */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_TRDOS_DISK;
                                                   /* TRDOS-style disk drive */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_TIMEX_DOCK;
                                           /* T[SC]2068-style cartridge port */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_SINCLAIR_JOYSTICK;
                                            /* Sinclair-style joystick ports */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_KEMPSTON_JOYSTICK;
                                            /* Kempston-style joystick ports */

/* Get the capabilities of a machine */
int libspectrum_machine_capabilities( libspectrum_machine type );

/* Get the timings of a machine */
libspectrum_dword
libspectrum_timings_processor_speed( libspectrum_machine machine );
libspectrum_dword libspectrum_timings_ay_speed( libspectrum_machine machine );
libspectrum_word
libspectrum_timings_left_border( libspectrum_machine machine );
libspectrum_word
libspectrum_timings_horizontal_screen( libspectrum_machine machine );
libspectrum_word
libspectrum_timings_right_border( libspectrum_machine machine );
libspectrum_word
libspectrum_timings_horizontal_retrace( libspectrum_machine machine );
libspectrum_word libspectrum_timings_top_border( libspectrum_machine machine );
libspectrum_word
libspectrum_timings_vertical_screen( libspectrum_machine machine );
libspectrum_word
libspectrum_timings_bottom_border( libspectrum_machine machine );
libspectrum_word
libspectrum_timings_vertical_retrace( libspectrum_machine machine );
libspectrum_word
libspectrum_timings_top_left_pixel( libspectrum_machine machine );
libspectrum_word
libspectrum_timings_tstates_per_line( libspectrum_machine machine );
libspectrum_word
libspectrum_timings_lines_per_frame( libspectrum_machine machine );
libspectrum_dword
libspectrum_timings_tstates_per_frame( libspectrum_machine machine );

/* Creator information */

typedef struct libspectrum_creator libspectrum_creator;

libspectrum_error libspectrum_creator_alloc( libspectrum_creator **creator );
libspectrum_error libspectrum_creator_free( libspectrum_creator *creator );

libspectrum_error
libspectrum_creator_set_program( libspectrum_creator *creator,
				 const libspectrum_byte *program );
const libspectrum_byte*
libspectrum_creator_program( libspectrum_creator *creator );

libspectrum_error libspectrum_creator_set_major( libspectrum_creator *creator,
						 libspectrum_word major );
libspectrum_word
libspectrum_creator_major( libspectrum_creator *creator );

libspectrum_error libspectrum_creator_set_minor( libspectrum_creator *creator,
						 libspectrum_word minor );
libspectrum_word
libspectrum_creator_minor( libspectrum_creator *creator );

libspectrum_error
libspectrum_creator_set_competition_code( libspectrum_creator *creator,
					  libspectrum_dword competition_code );
libspectrum_dword
libspectrum_creator_competition_code( libspectrum_creator *creator );

libspectrum_error
libspectrum_creator_set_custom( libspectrum_creator *creator,
				libspectrum_byte *data, size_t length );

libspectrum_byte*
libspectrum_creator_custom( libspectrum_creator *creator );

size_t
libspectrum_creator_custom_length( libspectrum_creator *creator );

/*
 * Snap handling routines
 */

typedef struct libspectrum_snap libspectrum_snap;

libspectrum_error libspectrum_snap_alloc( libspectrum_snap **snap );
libspectrum_error libspectrum_snap_free( libspectrum_snap *snap );

/* Read in a snapshot, optionally guessing what type it is */
libspectrum_error
libspectrum_snap_read( libspectrum_snap *snap, const libspectrum_byte *buffer,
		       size_t length, libspectrum_id_t type,
		       const char *filename );

/* Write a snapshot */
libspectrum_error
libspectrum_snap_write( libspectrum_byte **buffer, size_t *length,
			int *out_flags, libspectrum_snap *snap,
	 		libspectrum_id_t type, libspectrum_creator *creator,
			int in_flags );

/* The flags that can be given to libspectrum_snap_write() */
extern const int LIBSPECTRUM_FLAG_SNAPSHOT_NO_COMPRESSION;

/* The flags that may be returned from libspectrum_snap_write() */
extern const int LIBSPECTRUM_FLAG_SNAPSHOT_MINOR_INFO_LOSS;
extern const int LIBSPECTRUM_FLAG_SNAPSHOT_MAJOR_INFO_LOSS;

/* Accessor functions */
libspectrum_machine libspectrum_snap_machine( libspectrum_snap *snap );
void libspectrum_snap_set_machine( libspectrum_snap *snap, libspectrum_machine machine );
libspectrum_byte libspectrum_snap_a( libspectrum_snap *snap );
void libspectrum_snap_set_a( libspectrum_snap *snap, libspectrum_byte a );
libspectrum_byte libspectrum_snap_f( libspectrum_snap *snap );
void libspectrum_snap_set_f( libspectrum_snap *snap, libspectrum_byte f );
libspectrum_word libspectrum_snap_bc( libspectrum_snap *snap );
void libspectrum_snap_set_bc( libspectrum_snap *snap, libspectrum_word bc );
libspectrum_word libspectrum_snap_de( libspectrum_snap *snap );
void libspectrum_snap_set_de( libspectrum_snap *snap, libspectrum_word de );
libspectrum_word libspectrum_snap_hl( libspectrum_snap *snap );
void libspectrum_snap_set_hl( libspectrum_snap *snap, libspectrum_word hl );
libspectrum_byte libspectrum_snap_a_( libspectrum_snap *snap );
void libspectrum_snap_set_a_( libspectrum_snap *snap, libspectrum_byte a_ );
libspectrum_byte libspectrum_snap_f_( libspectrum_snap *snap );
void libspectrum_snap_set_f_( libspectrum_snap *snap, libspectrum_byte f_ );
libspectrum_word libspectrum_snap_bc_( libspectrum_snap *snap );
void libspectrum_snap_set_bc_( libspectrum_snap *snap, libspectrum_word bc_ );
libspectrum_word libspectrum_snap_de_( libspectrum_snap *snap );
void libspectrum_snap_set_de_( libspectrum_snap *snap, libspectrum_word de_ );
libspectrum_word libspectrum_snap_hl_( libspectrum_snap *snap );
void libspectrum_snap_set_hl_( libspectrum_snap *snap, libspectrum_word hl_ );
libspectrum_word libspectrum_snap_ix( libspectrum_snap *snap );
void libspectrum_snap_set_ix( libspectrum_snap *snap, libspectrum_word ix );
libspectrum_word libspectrum_snap_iy( libspectrum_snap *snap );
void libspectrum_snap_set_iy( libspectrum_snap *snap, libspectrum_word iy );
libspectrum_byte libspectrum_snap_i( libspectrum_snap *snap );
void libspectrum_snap_set_i( libspectrum_snap *snap, libspectrum_byte i );
libspectrum_byte libspectrum_snap_r( libspectrum_snap *snap );
void libspectrum_snap_set_r( libspectrum_snap *snap, libspectrum_byte r );
libspectrum_word libspectrum_snap_sp( libspectrum_snap *snap );
void libspectrum_snap_set_sp( libspectrum_snap *snap, libspectrum_word sp );
libspectrum_word libspectrum_snap_pc( libspectrum_snap *snap );
void libspectrum_snap_set_pc( libspectrum_snap *snap, libspectrum_word pc );
libspectrum_byte libspectrum_snap_iff1( libspectrum_snap *snap );
void libspectrum_snap_set_iff1( libspectrum_snap *snap, libspectrum_byte iff1 );
libspectrum_byte libspectrum_snap_iff2( libspectrum_snap *snap );
void libspectrum_snap_set_iff2( libspectrum_snap *snap, libspectrum_byte iff2 );
libspectrum_byte libspectrum_snap_im( libspectrum_snap *snap );
void libspectrum_snap_set_im( libspectrum_snap *snap, libspectrum_byte im );
libspectrum_dword libspectrum_snap_tstates( libspectrum_snap *snap );
void libspectrum_snap_set_tstates( libspectrum_snap *snap, libspectrum_dword tstates );
int libspectrum_snap_halted( libspectrum_snap *snap );
void libspectrum_snap_set_halted( libspectrum_snap *snap, int halted );
int libspectrum_snap_last_instruction_ei( libspectrum_snap *snap );
void libspectrum_snap_set_last_instruction_ei( libspectrum_snap *snap, int last_instruction_ei );
libspectrum_byte libspectrum_snap_out_ula( libspectrum_snap *snap );
void libspectrum_snap_set_out_ula( libspectrum_snap *snap, libspectrum_byte out_ula );
libspectrum_byte libspectrum_snap_out_128_memoryport( libspectrum_snap *snap );
void libspectrum_snap_set_out_128_memoryport( libspectrum_snap *snap, libspectrum_byte out_128_memoryport );
libspectrum_byte libspectrum_snap_out_plus3_memoryport( libspectrum_snap *snap );
void libspectrum_snap_set_out_plus3_memoryport( libspectrum_snap *snap, libspectrum_byte out_plus3_memoryport );
libspectrum_byte libspectrum_snap_out_ay_registerport( libspectrum_snap *snap );
void libspectrum_snap_set_out_ay_registerport( libspectrum_snap *snap, libspectrum_byte out_ay_registerport );
libspectrum_byte libspectrum_snap_ay_registers( libspectrum_snap *snap, int idx );
void libspectrum_snap_set_ay_registers( libspectrum_snap *snap, int idx, libspectrum_byte ay_registers );
libspectrum_byte libspectrum_snap_out_scld_hsr( libspectrum_snap *snap );
void libspectrum_snap_set_out_scld_hsr( libspectrum_snap *snap, libspectrum_byte out_scld_hsr );
libspectrum_byte libspectrum_snap_out_scld_dec( libspectrum_snap *snap );
void libspectrum_snap_set_out_scld_dec( libspectrum_snap *snap, libspectrum_byte out_scld_dec );
libspectrum_byte* libspectrum_snap_pages( libspectrum_snap *snap, int idx );
void libspectrum_snap_set_pages( libspectrum_snap *snap, int idx, libspectrum_byte* pages );
libspectrum_byte* libspectrum_snap_slt( libspectrum_snap *snap, int idx );
void libspectrum_snap_set_slt( libspectrum_snap *snap, int idx, libspectrum_byte* slt );
size_t libspectrum_snap_slt_length( libspectrum_snap *snap, int idx );
void libspectrum_snap_set_slt_length( libspectrum_snap *snap, int idx, size_t slt_length );
libspectrum_byte* libspectrum_snap_slt_screen( libspectrum_snap *snap );
void libspectrum_snap_set_slt_screen( libspectrum_snap *snap, libspectrum_byte* slt_screen );
int libspectrum_snap_slt_screen_level( libspectrum_snap *snap );
void libspectrum_snap_set_slt_screen_level( libspectrum_snap *snap, int slt_screen_level );

/* DEPRECATED: use libspectrum_snap_read() instead */
libspectrum_error
libspectrum_sna_read( libspectrum_snap *snap,
	              const libspectrum_byte *buffer, size_t buffer_length );

/* DEPRECATED: use libspectrum_snap_read() instead */
libspectrum_error
libspectrum_z80_read( libspectrum_snap *snap,
	              const libspectrum_byte *buffer, size_t buffer_length );

/* DEPRECATED: use libspectrum_snap_write() instead */
libspectrum_error
libspectrum_z80_write( libspectrum_byte **buffer, size_t *length,
		       libspectrum_snap *snap );

/*
 * Tape handling routines
 */

/* The various types of block available
   The values here must be the same as used in the .tzx format */
typedef enum libspectrum_tape_type {
  LIBSPECTRUM_TAPE_BLOCK_ROM = 0x10,
  LIBSPECTRUM_TAPE_BLOCK_TURBO,
  LIBSPECTRUM_TAPE_BLOCK_PURE_TONE,
  LIBSPECTRUM_TAPE_BLOCK_PULSES,
  LIBSPECTRUM_TAPE_BLOCK_PURE_DATA,
  LIBSPECTRUM_TAPE_BLOCK_RAW_DATA,

  LIBSPECTRUM_TAPE_BLOCK_PAUSE = 0x20,
  LIBSPECTRUM_TAPE_BLOCK_GROUP_START,
  LIBSPECTRUM_TAPE_BLOCK_GROUP_END,
  LIBSPECTRUM_TAPE_BLOCK_JUMP,
  LIBSPECTRUM_TAPE_BLOCK_LOOP_START,
  LIBSPECTRUM_TAPE_BLOCK_LOOP_END,

  LIBSPECTRUM_TAPE_BLOCK_SELECT = 0x28,

  LIBSPECTRUM_TAPE_BLOCK_STOP48 = 0x2a,

  LIBSPECTRUM_TAPE_BLOCK_COMMENT = 0x30,
  LIBSPECTRUM_TAPE_BLOCK_MESSAGE,
  LIBSPECTRUM_TAPE_BLOCK_ARCHIVE_INFO,
  LIBSPECTRUM_TAPE_BLOCK_HARDWARE,

  LIBSPECTRUM_TAPE_BLOCK_CUSTOM = 0x35,

  LIBSPECTRUM_TAPE_BLOCK_CONCAT = 0x5a,
} libspectrum_tape_type;

typedef struct libspectrum_tape_block libspectrum_tape_block;

/* Some flags */
extern const int LIBSPECTRUM_TAPE_FLAGS_BLOCK;	/* End of block */
extern const int LIBSPECTRUM_TAPE_FLAGS_STOP;	/* Stop tape */
extern const int LIBSPECTRUM_TAPE_FLAGS_STOP48;	/* Stop tape if in 48K mode */

/* The states which a block can be in */
typedef enum libspectrum_tape_state_type {

  LIBSPECTRUM_TAPE_STATE_INVALID = 0,

  LIBSPECTRUM_TAPE_STATE_PILOT,	/* Pilot pulses */
  LIBSPECTRUM_TAPE_STATE_SYNC1,	/* First sync pulse */
  LIBSPECTRUM_TAPE_STATE_SYNC2,	/* Second sync pulse */
  LIBSPECTRUM_TAPE_STATE_DATA1,	/* First edge of a data bit */
  LIBSPECTRUM_TAPE_STATE_DATA2,	/* Second edge of a data bit */
  LIBSPECTRUM_TAPE_STATE_PAUSE,	/* The pause at the end of a block */

} libspectrum_tape_state_type;

/* Routines to manipulate tape blocks */
libspectrum_error
libspectrum_tape_block_alloc( libspectrum_tape_block **block,
			      libspectrum_tape_type type );
libspectrum_error libspectrum_tape_block_free( libspectrum_tape_block *block );
libspectrum_tape_type
libspectrum_tape_block_type( libspectrum_tape_block *block );
libspectrum_error
libspectrum_tape_block_set_type( libspectrum_tape_block *block,
				 libspectrum_tape_type type );

libspectrum_error libspectrum_tape_block_init( libspectrum_tape_block *block );

/* Accessor functions */
libspectrum_dword libspectrum_tape_block_bit_length( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_bit_length( libspectrum_tape_block *block, libspectrum_dword bit_length );
libspectrum_dword libspectrum_tape_block_bit0_length( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_bit0_length( libspectrum_tape_block *block, libspectrum_dword bit0_length );
libspectrum_dword libspectrum_tape_block_bit1_length( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_bit1_length( libspectrum_tape_block *block, libspectrum_dword bit1_length );
size_t libspectrum_tape_block_bits_in_last_byte( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_bits_in_last_byte( libspectrum_tape_block *block, size_t bits_in_last_byte );
size_t libspectrum_tape_block_count( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_count( libspectrum_tape_block *block, size_t count );
libspectrum_byte* libspectrum_tape_block_data( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_data( libspectrum_tape_block *block, libspectrum_byte* data );
size_t libspectrum_tape_block_data_length( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_data_length( libspectrum_tape_block *block, size_t data_length );
int libspectrum_tape_block_ids( libspectrum_tape_block *block, size_t idx );
libspectrum_error libspectrum_tape_block_set_ids( libspectrum_tape_block *block, int *ids );
int libspectrum_tape_block_offset( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_offset( libspectrum_tape_block *block, int offset );
int libspectrum_tape_block_offsets( libspectrum_tape_block *block, size_t idx );
libspectrum_error libspectrum_tape_block_set_offsets( libspectrum_tape_block *block, int *offsets );
libspectrum_dword libspectrum_tape_block_pause( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_pause( libspectrum_tape_block *block, libspectrum_dword pause );
libspectrum_dword libspectrum_tape_block_pilot_length( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_pilot_length( libspectrum_tape_block *block, libspectrum_dword pilot_length );
size_t libspectrum_tape_block_pilot_pulses( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_pilot_pulses( libspectrum_tape_block *block, size_t pilot_pulses );
libspectrum_dword libspectrum_tape_block_pulse_length( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_pulse_length( libspectrum_tape_block *block, libspectrum_dword pulse_length );
libspectrum_dword libspectrum_tape_block_pulse_lengths( libspectrum_tape_block *block, size_t idx );
libspectrum_error libspectrum_tape_block_set_pulse_lengths( libspectrum_tape_block *block, libspectrum_dword *pulse_lengths );
libspectrum_tape_state_type libspectrum_tape_block_state( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_state( libspectrum_tape_block *block, libspectrum_tape_state_type state );
libspectrum_dword libspectrum_tape_block_sync1_length( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_sync1_length( libspectrum_tape_block *block, libspectrum_dword sync1_length );
libspectrum_dword libspectrum_tape_block_sync2_length( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_sync2_length( libspectrum_tape_block *block, libspectrum_dword sync2_length );
libspectrum_byte* libspectrum_tape_block_text( libspectrum_tape_block *block );
libspectrum_error libspectrum_tape_block_set_text( libspectrum_tape_block *block, libspectrum_byte* text );
libspectrum_byte* libspectrum_tape_block_texts( libspectrum_tape_block *block, size_t idx );
libspectrum_error libspectrum_tape_block_set_texts( libspectrum_tape_block *block, libspectrum_byte* *texts );
int libspectrum_tape_block_types( libspectrum_tape_block *block, size_t idx );
libspectrum_error libspectrum_tape_block_set_types( libspectrum_tape_block *block, int *types );
int libspectrum_tape_block_values( libspectrum_tape_block *block, size_t idx );
libspectrum_error libspectrum_tape_block_set_values( libspectrum_tape_block *block, int *values );

/* A linked list of tape blocks */
typedef struct libspectrum_tape libspectrum_tape;

libspectrum_error libspectrum_tape_alloc( libspectrum_tape **tape );
libspectrum_error libspectrum_tape_clear( libspectrum_tape *tape );
libspectrum_error libspectrum_tape_free( libspectrum_tape *tape );

/* Read in a tape file, optionally guessing what sort of file it is */
libspectrum_error
libspectrum_tape_read( libspectrum_tape *tape, const libspectrum_byte *buffer,
		       size_t length, libspectrum_id_t type,
		       const char *filename );

/* Does this tape structure actually contain a tape? */
int libspectrum_tape_present( const libspectrum_tape *tape );

libspectrum_error
libspectrum_tape_get_next_edge( libspectrum_dword *tstates, int *flags,
	                        libspectrum_tape *tape );

/* Get the current block from the tape */
libspectrum_tape_block*
libspectrum_tape_current_block( libspectrum_tape *tape );

/* Peek at the next block on the tape */
libspectrum_tape_block*
libspectrum_tape_peek_next_block( libspectrum_tape *tape );

/* Cause the next block on the tape to be active, initialise it
   and return it */
libspectrum_tape_block*
libspectrum_tape_select_next_block( libspectrum_tape *tape );

/* Get the position on the tape of the current block */
libspectrum_error libspectrum_tape_position( int *n, libspectrum_tape *tape );

/* Select the nth block on the tape */
libspectrum_error
libspectrum_tape_nth_block( libspectrum_tape *tape, int n );

libspectrum_error
libspectrum_tape_block_description( char *buffer, size_t length,
	                            libspectrum_tape_block *block );

/* Append a block to the current tape */
libspectrum_error
libspectrum_tape_append_block( libspectrum_tape *tape,
			       libspectrum_tape_block *block );

/*** Routines for .tap format files ***/

libspectrum_error
libspectrum_tap_read( libspectrum_tape *tape, const libspectrum_byte *buffer,
		      const size_t length );
libspectrum_error
libspectrum_tap_write( libspectrum_byte **buffer, size_t *length,
		       libspectrum_tape *tape );

/*** Routines for .tzx format files ***/

libspectrum_error
libspectrum_tzx_read( libspectrum_tape *tape, const libspectrum_byte *buffer,
		      const size_t length );

libspectrum_error
libspectrum_tzx_write( libspectrum_byte **buffer, size_t *length,
		       libspectrum_tape *tape );

/*** Routines for Warajevo .tap format files ***/

libspectrum_error
libspectrum_warajevo_read( libspectrum_tape *tape,
			   const libspectrum_byte *buffer, size_t length );

/* Something to step through all the blocks in a tape */
typedef struct _GSList *libspectrum_tape_iterator;

libspectrum_tape_block*
libspectrum_tape_iterator_init( libspectrum_tape_iterator *iterator,
				libspectrum_tape *tape );

libspectrum_tape_block*
libspectrum_tape_iterator_next( libspectrum_tape_iterator *iterator );

/*
 * Input recording routines
 */

typedef struct libspectrum_rzx libspectrum_rzx;

libspectrum_error libspectrum_rzx_alloc( libspectrum_rzx **rzx );
libspectrum_error libspectrum_rzx_free( libspectrum_rzx *rzx );

libspectrum_error libspectrum_rzx_store_frame( libspectrum_rzx *rzx,
					       size_t instructions, 
					       size_t count,
					       libspectrum_byte *in_bytes );

libspectrum_error libspectrum_rzx_start_playback( libspectrum_rzx *rzx );
libspectrum_error libspectrum_rzx_playback_frame( libspectrum_rzx *rzx,
						  int *finished );
libspectrum_error libspectrum_rzx_playback( libspectrum_rzx *rzx,
					    libspectrum_byte *byte );

/* Get and set the tstate counter */
size_t libspectrum_rzx_tstates( libspectrum_rzx *rzx );
size_t libspectrum_rzx_set_tstates( libspectrum_rzx *rzx, size_t tstates );

/* Get the current frame's instruction count */
size_t libspectrum_rzx_instructions( libspectrum_rzx *rzx );

typedef struct libspectrum_rzx_dsa_key {

  /* The standard DSA parameters, represented as hex strings with MSB first.
     Set x = NULL for a public key */
  const char *p, *q, *g, *y, *x;

} libspectrum_rzx_dsa_key;

typedef struct libspectrum_rzx_signature {

  /* The key ID (low 32 bits of 'y') */
  libspectrum_dword key_id;

  /* Where in the buffer is the signed data? */
  const libspectrum_byte *start; ptrdiff_t length;

  /* The DSA signature parameters 'r' and 's' */
  gcry_mpi_t r, s;

} libspectrum_rzx_signature;

libspectrum_error
libspectrum_rzx_read( libspectrum_rzx *rzx, libspectrum_snap **snap,
	              const libspectrum_byte *buffer, size_t length,
		      libspectrum_rzx_signature *signature );

libspectrum_error
libspectrum_rzx_write( libspectrum_byte **buffer, size_t *length,
		       libspectrum_rzx *rzx, libspectrum_snap *snap,
		       libspectrum_creator *creator, int compress,
		       libspectrum_rzx_dsa_key *key );

libspectrum_error libspectrum_rzx_alloc( libspectrum_rzx **rzx );

/*
 * Timex DOCK/EXROM handling routines
 */

/* The types of memory which this bank will page into */
typedef enum libspectrum_dck_bank {
  LIBSPECTRUM_DCK_BANK_DOCK = 0,
  LIBSPECTRUM_DCK_BANK_EXROM = 254,
  LIBSPECTRUM_DCK_BANK_HOME = 255
} libspectrum_dck_bank; 

/* The type of one 8Kb page */
typedef enum libspectrum_dck_page_type {
  LIBSPECTRUM_DCK_PAGE_NULL = 0,
  LIBSPECTRUM_DCK_PAGE_RAM_EMPTY,
  LIBSPECTRUM_DCK_PAGE_ROM,
  LIBSPECTRUM_DCK_PAGE_RAM
} libspectrum_dck_page_type;

/* One set of 8 x 8Kb pages */
typedef struct libspectrum_dck_block {
  libspectrum_dck_bank bank;
  libspectrum_dck_page_type access[8];
  libspectrum_byte *pages[8];
} libspectrum_dck_block;

/* A number of 8 x 8Kb page sets */
/* FIXME: Remove the arbitrary limit on the number of banks */
typedef struct libspectrum_dck {
  libspectrum_dck_block *dck[256];	/* dck block data */
} libspectrum_dck;

libspectrum_error libspectrum_dck_alloc( libspectrum_dck **dck );
libspectrum_error libspectrum_dck_free( libspectrum_dck *dck, int keep_pages );

/* Read in a DCK file */

libspectrum_error
libspectrum_dck_read( libspectrum_dck *dck, const libspectrum_byte *buffer,
                      size_t length );

libspectrum_error
libspectrum_dck_read2( libspectrum_dck *dck, const libspectrum_byte *buffer,
                       size_t length, const char *filename );

/*
 * Crypto functions
 */

libspectrum_error
libspectrum_verify_signature( libspectrum_rzx_signature *signature,
			      libspectrum_rzx_dsa_key *key );
libspectrum_error
libspectrum_signature_free( libspectrum_rzx_signature *signature );

#ifdef __cplusplus
};
#endif				/* #ifdef __cplusplus */

#endif				/* #ifndef LIBSPECTRUM_LIBSPECTRUM_H */
