%% From package:nlme  AIC.Rd,v 1.4 2000/07/03 18:22:45 bates
%% with additional "k = 2" argument (MM)
\name{AIC}
\alias{AIC}
%\alias{AIC.default}
%\alias{AIC.logLik}
\title{Akaike's An Information Criterion}
\description{
  Generic function calculating the Akaike information criterion for
  one or several fitted model objects for which a log-likelihood value
  can be obtained, according to the formula
  \eqn{-2 \mbox{log-likelihood} + k n_{par}}{-2*log-likelihood + k*npar},
  where \eqn{n_{par}}{npar} represents the number of parameters in the
  fitted model, and \eqn{k = 2} for the usual AIC, or \eqn{k = \log(n)}
  (\eqn{n} the number of observations) for the so-called BIC or SBC
  (Schwarz's Bayesian criterion).
}
\usage{
AIC(object, \dots, k = 2)
}
\arguments{
  \item{object}{a fitted model object, for which there exists a
    \code{logLik} method to extract the corresponding log-likelihood, or
    an object inheriting from class \code{logLik}.}
  \item{\dots}{optionally more fitted model objects.}
  \item{k}{numeric, the \dQuote{penalty} per parameter to be used; the
    default \code{k = 2} is the classical AIC.}
}
\details{
  The default method for \code{AIC}, \code{AIC.default()} entirely
  relies on the existence of a \code{\link{logLik}} method
  computing the log-likelihood for the given class.

  When comparing fitted objects, the smaller the AIC, the better the fit.
}
\value{
  If just one object is provided, returns a numeric value
  with the corresponding AIC (or BIC, or \dots, depending on \code{k});
  if more than one object are provided, returns a \code{data.frame} with
  rows corresponding to the objects and columns representing the number
  of parameters in the model (\code{df}) and the AIC.
}
\references{
  Sakamoto, Y., Ishiguro, M., and Kitagawa G. (1986).
  \emph{Akaike Information Criterion Statistics}.
  D. Reidel Publishing Company.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{
  \code{\link{extractAIC}}, \code{\link{logLik}}.
}
\examples{
data(swiss)
lm1 <- lm(Fertility ~ . , data = swiss)
AIC(lm1)
stopifnot(all.equal(AIC(lm1),
                    AIC(logLik(lm1))))
## a version of BIC or Schwarz' BC :
AIC(lm1, k = log(nrow(swiss)))
}
\keyword{models}
\eof
\name{ARMAacf}
\alias{ARMAacf}
\title{Compute Theoretical ACF for an ARMA Process}
\description{
 Compute the theoretical autocorrelation function or partial
 autocorrelation function for an ARMA process.
}
\usage{
ARMAacf(ar = numeric(0), ma = numeric(0), lag.max = r, pacf = FALSE)
}
\arguments{
  \item{ar}{numeric vector of AR coefficients}
  \item{ma}{numeric vector of MA coefficients}
  \item{lag.max}{integer. Maximum lage required. Defaults to
    \code{max(p, q+1)}, where \code{p, q} are the numbers of AR and MA
    terms respectively.}
  \item{pacf}{logical. Should the partial autocorrelations be returned?}
}
\details{
  The methods used follow Brockwell \& Davis (1991, section 3.3).  Their
  equations (3.3.8) are solved for the autocovariances at lags
  \eqn{0, \dots, \max(p, q+1)}, and the remaining autocorrelations are
  given by a recursive filter.
}
\value{
  A vector of (partial) autocorrelations, named by the lags.  
}

\references{
  Brockwell, P. J. and Davis, R. A. (1991) \emph{Time Series: Theory and
    Methods}, Second Edition.  Springer.
}
\seealso{\code{\link{arima}}, \code{\link{ARMAtoMA}}, \code{\link{filter}}.}

\examples{
ARMAacf(c(1.0, -0.25), 1.0, lag.max = 10)
## Example from Brockwell & Davis (1991, pp.92-4)
## answer 2^(-n) * (32/3 + 8 * n) /(32/3)
n <- 1:10; 2^(-n) * (32/3 + 8 * n) /(32/3)
ARMAacf(c(1.0, -0.25), 1.0, lag.max = 10, pacf = TRUE)
ARMAacf(c(1.0, -0.25), lag.max = 10, pacf = TRUE)
}
\keyword{ts}
\eof
\name{ARMAtoMA}
\alias{ARMAtoMA}
\title{Convert ARMA Process to Infinite MA Process}
\description{
  Convert ARMA process to infinite MA process.
}
\usage{
ARMAtoMA(ar = numeric(0), ma = numeric(0), lag.max)
}
\arguments{
  \item{ar}{numeric vector of AR coefficients}
  \item{ma}{numeric vector of MA coefficients}
  \item{lag.max}{Largest MA(Inf) coefficient required.}
}
\value{
  A vector of coefficients.
}

\references{
  Brockwell, P. J. and Davis, R. A. (1991) \emph{Time Series: Theory and
    Methods}, Second Edition.  Springer.
}

\seealso{\code{\link{arima}}, \code{\link{ARMAacf}}.}

\examples{
ARMAtoMA(c(1.0, -0.25), 1.0, 10)
## Example from Brockwell & Davis (1991, p.92)
## answer (1 + 3*n)*2^(-n)
n <- 1:10; (1 + 3*n)*2^(-n)
}
\keyword{ts}

\eof
\name{AirPassengers}
\docType{data}
\alias{AirPassengers}
\title{Monthly Airline Passenger Numbers 1949-1960}
\description{
 The classic Box \& Jenkins airline data.  Monthly totals of
 international airline passengers, 1949 to 1960.
}
\usage{data(AirPassengers)}
\format{
  A monthly time series, in thousands.
}
\source{
  Box, G. E. P., Jenkins, G. M. and Reinsel, G. C. (1976)
  \emph{Time Series Analysis, Forecasting and Control.}
  Third Edition. Holden-Day. Series G.
}
 \examples{
\dontrun{## These are quite slow and so not run by example(AirPassengers)

data(AirPassengers)
## The classic 'airline model', by full ML
(fit <- arima(log10(AirPassengers), c(0, 1, 1),
              seasonal = list(order=c(0, 1 ,1), period=12)))
update(fit, method = "CSS")
update(fit, x=window(log10(AirPassengers), start = 1954))
pred <- predict(fit, n.ahead = 24)
tl <- pred$pred - 1.96 * pred$se
tu <- pred$pred + 1.96 * pred$se
ts.plot(AirPassengers, 10^tl, 10^tu, log = "y", lty = c(1,2,2))

## full ML fit is the same if the series is reversed, CSS fit is not
ap0 <- rev(log10(AirPassengers))
attributes(ap0) <- attributes(AirPassengers)
arima(ap0, c(0, 1, 1), seasonal = list(order=c(0, 1 ,1), period=12))
arima(ap0, c(0, 1, 1), seasonal = list(order=c(0, 1 ,1), period=12),
      method = "CSS")

## Structural Time Series
ap <- log10(AirPassengers) - 2
(fit <- StructTS(ap, type= "BSM"))
par(mfrow=c(1,2))
plot(cbind(ap, fitted(fit)), plot.type = "single")
plot(cbind(ap, tsSmooth(fit)), plot.type = "single")
}}
\keyword{datasets}
\eof
\name{BJsales}
\docType{data}
\alias{BJsales}
\alias{BJsales.lead}
\title{Sales Data with Leading Indicator}
\description{
  The sales time series \code{BJsales} and leading indicator
  \code{BJsales.lead} each contain 150 observations.
  The objects are of class \code{"ts"}.
}
\usage{
data(BJsales)
}
\source{
  The data are given in Box & Jenkins (1976).
  Obtained from the Time Series Data Library at
  \url{http://www-personal.buseco.monash.edu.au/~hyndman/TSDL/}
}
\references{
  G. E. P. Box and G. M. Jenkins (1976):
  \emph{Time Series Analysis, Forecasting and Control},
  Holden-Day, San Francisco, p. 537.

  P. J. Brockwell and R. A. Davis (1991):
  \emph{Time Series: Theory and Methods},
  Second edition, Springer Verlag, NY, pp. 414.
}
\keyword{datasets}
\eof
\name{BOD}
\docType{data}
\alias{BOD}
\title{ Biochemical Oxygen Demand }
\description{
  The \code{BOD} data frame has 6 rows and 2 columns giving the
  biochemical oxygen demand versus time in an evaluation of water
  quality.
}
\usage{data(BOD)}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Time}{
      A numeric vector giving the time of the measurement (days).
    }
    \item{demand}{
      A numeric vector giving the biochemical oxygen demand (mg/l).
    }
  }
}
\source{
  Bates, D.M. and Watts, D.G. (1988),
  \emph{Nonlinear Regression Analysis and Its Applications},
  Wiley, Appendix A1.4.

  Originally from Marske (1967), \emph{Biochemical
    Oxygen Demand Data Interpretation Using Sum of Squares Surface}
  M.Sc. Thesis, University of Wisconsin -- Madison.
}
\examples{
data(BOD)
# simplest form of fitting a first-order model to these data
fm1 <- nls(demand ~ A*(1-exp(-exp(lrc)*Time)), data = BOD,
   start = c(A = 20, lrc = log(.35)))
coef(fm1)
print(fm1)
# using the plinear algorithm
fm2 <- nls(demand ~ (1-exp(-exp(lrc)*Time)), data = BOD,
   start = c(lrc = log(.35)), algorithm = "plinear", trace = TRUE)
# using a self-starting model
fm3 <- nls(demand ~ SSasympOrig(Time, A, lrc), data = BOD)
summary( fm3 )
}
\keyword{datasets}
\eof
\name{Beta}
\alias{Beta}
\alias{dbeta}
\alias{pbeta}
\alias{qbeta}
\alias{rbeta}
\title{The Beta Distribution}
\concept{incomplete beta function}
\description{
  Density, distribution function, quantile function and random
  generation for the Beta distribution with parameters \code{shape1} and
  \code{shape2} (and optional non-centrality parameter \code{ncp}).
}
\usage{
dbeta(x, shape1, shape2, ncp=0, log = FALSE)
pbeta(q, shape1, shape2, ncp=0, lower.tail = TRUE, log.p = FALSE)
qbeta(p, shape1, shape2,        lower.tail = TRUE, log.p = FALSE)
rbeta(n, shape1, shape2)
}
\arguments{
  \item{x, q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{shape1, shape2}{positive parameters of the Beta distribution.}
  \item{ncp}{non-centrality parameter.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dbeta} gives the density, \code{pbeta} the distribution
  function, \code{qbeta} the quantile function, and \code{rbeta}
  generates random deviates.
}
\details{
  The Beta distribution with parameters \code{shape1} \eqn{= a} and
  \code{shape2} \eqn{= b} has density
  \deqn{f(x)=\frac{\Gamma(a+b)}{\Gamma(a)\Gamma(b)}{x}^{a} {(1-x)}^{b}%
  }{Gamma(a+b)/(Gamma(a)Gamma(b))x^(a-1)(1-x)^(b-1)}
  for \eqn{a > 0}, \eqn{b > 0} and \eqn{0 \le x \le 1}{0 <= x <= 1}
  where the boundary values at \eqn{x=0} or \eqn{x=1} are defined as
  by continuity (as limits).

  \code{pbeta} is closely related to the incomplete beta function.  As
  defined by Abramowitz and Stegun 6.6.1
  \deqn{B_x(a,b) = \int_0^x t^{a-1} (1-t)^{b-1} dt,}{B_x(a,b) =
    integral_0^x t^(a-1) (1-t)^(b-1) dt,}
  and 6.6.2 \eqn{I_x(a,b) = B_x(a,b) / B(a,b)} where
  \eqn{B(a,b) = B_1(a,b)} is the Beta function (\code{\link{beta}}).

  \eqn{I_x(a,b)} is \code{pbeta(x,a,b)}.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.

  Abramowitz, M. and Stegun, I. A. (1972)
  \emph{Handbook of Mathematical Functions.} New York: Dover.
  Chapter 6: Gamma and Related Functions.
}
\seealso{
  \code{\link{beta}} for the Beta function, and \code{\link{dgamma}} for
  the Gamma distribution.
}
\examples{
x <- seq(0, 1, length=21)
dbeta(x, 1, 1)
pbeta(x, 1, 1)
}
\keyword{distribution}
\eof
\name{Binomial}
\alias{Binomial}
\alias{dbinom}
\alias{pbinom}
\alias{qbinom}
\alias{rbinom}
\title{The Binomial Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the binomial distribution with parameters \code{size}
  and \code{prob}.
}
\usage{
dbinom(x, size, prob, log = FALSE)
pbinom(q, size, prob, lower.tail = TRUE, log.p = FALSE)
qbinom(p, size, prob, lower.tail = TRUE, log.p = FALSE)
rbinom(n, size, prob)
}
\arguments{
  \item{x, q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{size}{number of trials.}
  \item{prob}{probability of success on each trial.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dbinom} gives the density, \code{pbinom} gives the distribution
  function, \code{qbinom} gives the quantile function and \code{rbinom}
  generates random deviates.

  If \code{size} is not an integer, \code{NaN} is returned.
}
\details{
  The binomial distribution with \code{size} \eqn{= n} and
  \code{prob} \eqn{= p} has density
  \deqn{p(x) = {n \choose x} {p}^{x} {(1-p)}^{n-x}}{
    p(x) = choose(n,x) p^x (1-p)^(n-x)}
  for \eqn{x = 0, \ldots, n}.

  If an element of \code{x} is not integer, the result of \code{dbinom}
  is zero, with a warning.
  \eqn{p(x)} is computed using Loader's algorithm, see the reference below.

  The quantile is defined as the smallest value \eqn{x} such that
  \eqn{F(x) \ge p}, where \eqn{F} is the distribution function.
}
\seealso{
  \code{\link{dnbinom}} for the negative binomial, and
  \code{\link{dpois}} for the Poisson distribution.
}
\references{
  Catherine Loader (2000). \emph{Fast and Accurate Computation of
    Binomial Probabilities}; manuscript available from
  \url{http://cm.bell-labs.com/cm/ms/departments/sia/catherine/dbinom}
}
\examples{
# Compute P(45 < X < 55) for X Binomial(100,0.5)
sum(dbinom(46:54, 100, 0.5))

## Using "log = TRUE" for an extended range :
n <- 2000
k <- seq(0, n, by = 20)
plot (k, dbinom(k, n, pi/10, log=TRUE), type='l', ylab="log density",
      main = "dbinom(*, log=TRUE) is better than  log(dbinom(*))")
lines(k, log(dbinom(k, n, pi/10)), col='red', lwd=2)
## extreme points are omitted since dbinom gives 0.
mtext("dbinom(k, log=TRUE)", adj=0)
mtext("extended range", adj=0, line = -1, font=4)
mtext("log(dbinom(k))", col="red", adj=1)
}
\keyword{distribution}
\eof
\name{Cauchy}
\alias{Cauchy}
\alias{dcauchy}
\alias{pcauchy}
\alias{qcauchy}
\alias{rcauchy}
\title{The Cauchy Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the Cauchy distribution with location parameter
  \code{location} and scale parameter \code{scale}.
}
\usage{
dcauchy(x, location = 0, scale = 1, log = FALSE)
pcauchy(q, location = 0, scale = 1, lower.tail = TRUE, log.p = FALSE)
qcauchy(p, location = 0, scale = 1, lower.tail = TRUE, log.p = FALSE)
rcauchy(n, location = 0, scale = 1)
}
\arguments{
  \item{x, q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{location, scale}{location and scale parameters.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dcauchy}, \code{pcauchy}, and \code{qcauchy} are respectively
  the density, distribution function and quantile function of the Cauchy
  distribution.  \code{rcauchy} generates random deviates from the
  Cauchy.
}
\details{
  If \code{location} or \code{scale} are not specified, they assume
  the default values of \code{0} and \code{1} respectively.

  The Cauchy distribution with location \eqn{l} and scale \eqn{s} has
  density
  \deqn{f(x) = \frac{1}{\pi s}
    \left( 1 + \left(\frac{x - l}{s}\right)^2 \right)^{-1}%
  }{f(x) = 1 / (pi s (1 + ((x-l)/s)^2))}
  for all \eqn{x}.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{dt}} for the t distribution which generalizes
  \code{dcauchy(*, l = 0, s = 1)}.
}
\examples{
dcauchy(-1:4)
}
\keyword{distribution}
\eof
\name{ChickWeight}
\docType{data}
\alias{ChickWeight}
\title{Weight versus age of chicks on different diets}
\description{
  The \code{ChickWeight} data frame has 578 rows and 4 columns from an
  experiment on the effect of diet on early growth of chicks.
}
\usage{data(ChickWeight)}
\format{
  This data frame contains the following columns:
  \describe{
    \item{weight}{
      a numeric vector giving the body weight of the chick (gm).
    }
    \item{Time}{
      a numeric vector giving the number of days since birth when
      the measurement was made.
    }
    \item{Chick}{
      an ordered factor with levels
      \code{18} < \dots < \code{48}
      giving a unique identifier for the chick.  The ordering of
      the levels groups chicks on the same diet together and
      orders them according to their final weight (lightest to
      heaviest) within diet.
    }
    \item{Diet}{
      a factor with levels 1,\dots,4 indicating which
      experimental diet the chick received.
    }
  }
}
\details{
  The body weights of the chicks were measured at birth and every
  second day thereafter until day 20.  They were also measured on day
  21.  There were four groups on chicks on different protein diets.
}
\source{
  Crowder, M. and Hand, D. (1990), \emph{Analysis of Repeated Measures},
  {Chapman and Hall} (example 5.3)

  Hand, D. and Crowder, M. (1996), \emph{Practical Longitudinal Data
    Analysis}, Chapman and Hall (table A.2)

  Pinheiro, J. C. and Bates, D. M. (2000) \emph{Mixed-effects Models in
    S and S-PLUS}, Springer.
}
\examples{
data(ChickWeight)
coplot(weight ~ Time | Chick, data = ChickWeight,
    type = "b", show = FALSE)
## fit a representative chick
fm1 <- nls(weight ~ SSlogis( Time, Asym, xmid, scal ),
    data = ChickWeight, subset = Chick == 1)
summary( fm1 )
}
\keyword{datasets}
\eof
\name{Chisquare}
\alias{Chisquare}
\alias{dchisq}
\alias{pchisq}
\alias{qchisq}
\alias{rchisq}
\title{The (non-central) Chi-Squared Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the chi-squared (\eqn{\chi^2}{chi^2}) distribution with
  \code{df} degrees of freedom and optional non-centrality parameter
  \code{ncp}.
}
\usage{
dchisq(x, df, ncp=0, log = FALSE)
pchisq(q, df, ncp=0, lower.tail = TRUE, log.p = FALSE)
qchisq(p, df, ncp=0, lower.tail = TRUE, log.p = FALSE)
rchisq(n, df, ncp=0)
}
\arguments{
  \item{x, q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{df}{degrees of freedom (non-negative, but can be non-integer).}
  \item{ncp}{non-centrality parameter (non-negative).  Note that
    \code{ncp} values larger than about 1417 are not allowed currently
    for \code{pchisq} and \code{qchisq}.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dchisq} gives the density, \code{pchisq} gives the distribution
  function, \code{qchisq} gives the quantile function, and \code{rchisq}
  generates random deviates.
}
\details{
  The chi-squared distribution with \code{df}\eqn{= n} degrees of
  freedom has density
  \deqn{f_n(x) = \frac{1}{{2}^{n/2} \Gamma (n/2)} {x}^{n/2-1} {e}^{-x/2}}{%
    f_n(x) = 1 / (2^(n/2) Gamma(n/2))  x^(n/2-1) e^(-x/2)}
  for \eqn{x > 0}.  The mean and variance are \eqn{n} and \eqn{2n}.

  The non-central chi-squared distribution with \code{df}\eqn{= n}
  degrees of freedom and non-centrality parameter \code{ncp}
  \eqn{= \lambda} has density
  \deqn{
    f(x) = e^{-\lambda / 2}
      \sum_{r=0}^\infty \frac{(\lambda/2)^r}{r!}\, f_{n + 2r}(x)}{%
    f(x) = exp(-lambda/2) SUM_{r=0}^infty ((lambda/2)^r / r!) dchisq(x, df + 2r)
  }
  for \eqn{x \ge 0}.  For integer \eqn{n}, this is the distribution of
  the sum of squares of \eqn{n} normals each with variance one,
  \eqn{\lambda} being the sum of squares of the normal means; further,
  \cr
  \eqn{E(X) = n + \lambda}, \eqn{Var(X) = 2(n + 2*\lambda)}, and
  \eqn{E((X - E(X))^3) = 8(n + 3*\lambda)}.

  Note that the degrees of freedom \code{df}\eqn{= n}, can be
  non-integer, and for non-centrality \eqn{\lambda > 0}, even \eqn{n = 0};
  see the reference, chapter 29.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.

  Johnson, Kotz and Balakrishnan (1995). \emph{Continuous Univariate
    Distributions}, Vol \bold{2}; Wiley NY;
}
\seealso{
  A central chi-squared distribution with \eqn{n} degrees of freedom
  is the same as a Gamma distribution with \code{shape} \eqn{\alpha =
    n/2}{a = n/2} and \code{scale} \eqn{\sigma = 2}{s = 2}.  Hence, see
  \code{\link{dgamma}} for the Gamma distribution.
}
\examples{
dchisq(1, df=1:3)
pchisq(1, df= 3)
pchisq(1, df= 3, ncp = 0:4)# includes the above

x <- 1:10
## Chi-squared(df = 2) is a special exponential distribution
all.equal(dchisq(x, df=2), dexp(x, 1/2))
all.equal(pchisq(x, df=2), pexp(x, 1/2))

## non-central RNG -- df=0 is ok for ncp > 0:  Z0 has point mass at 0!
Z0 <- rchisq(100, df = 0, ncp = 2.)
graphics::stem(Z0)

\dontrun{## visual testing
## do P-P plots for 1000 points at various degrees of freedom
L <- 1.2; n <- 1000; pp <- ppoints(n)
op <- par(mfrow = c(3,3), mar= c(3,3,1,1)+.1, mgp= c(1.5,.6,0),
          oma = c(0,0,3,0))
for(df in 2^(4*rnorm(9))) {
  plot(pp, sort(pchisq(rr <- rchisq(n,df=df, ncp=L), df=df, ncp=L)),
       ylab="pchisq(rchisq(.),.)", pch=".")
  mtext(paste("df = ",formatC(df, digits = 4)), line= -2, adj=0.05)
  abline(0,1,col=2)
}
mtext(expression("P-P plots : Noncentral  "*
                 chi^2 *"(n=1000, df=X, ncp= 1.2)"),
      cex = 1.5, font = 2, outer=TRUE)
par(op)}
}
\keyword{distribution}
\eof
\name{DNase}
\docType{data}
\alias{DNase}
\title{Elisa assay of DNase}
\description{
  The \code{DNase} data frame has 176 rows and 3 columns of data
  obtained during development of an ELISA assay for the recombinant
  protein DNase in rat serum.
}
\usage{data(DNase)}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Run}{
      an ordered factor with levels \code{10} < \dots < \code{3}
      indicating the assay run.
    }
    \item{conc}{
      a numeric vector giving the known concentration of the
      protein. 
    }
    \item{density}{
      a numeric vector giving the measured optical density
      (dimensionless) in the assay.  Duplicate optical density
      measurements were obtained. 
    }
  }
}
\source{
  Davidian, M. and Giltinan, D. M. (1995) \emph{Nonlinear Models for
    Repeated Measurement Data}, Chapman & Hall (section 5.2.4, p. 134)

  Pinheiro, J. C. and Bates, D. M. (2000) \emph{Mixed-effects Models in
    S and S-PLUS}, Springer.
}
\examples{
data(DNase)
coplot(density ~ conc | Run, data = DNase,
    show = FALSE, type = "b")
coplot(density ~ log(conc) | Run, data = DNase,
    show = FALSE, type = "b")
## fit a representative run
fm1 <- nls(density ~ SSlogis( log(conc), Asym, xmid, scal ),
    data = DNase, subset = Run == 1)
## compare with a four-parameter logistic
fm2 <- nls(density ~ SSfpl( log(conc), A, B, xmid, scal ),
    data = DNase, subset = Run == 1)
summary(fm2)
anova(fm1, fm2)
}
\keyword{datasets}
\eof
\name{EuStockMarkets}
\docType{data}
\alias{EuStockMarkets}
\title{
  Daily Closing Prices of Major European Stock Indices, 1991--1998
}
\description{
  Contains the daily closing prices of major European stock indices:
  Germany DAX (Ibis), Switzerland SMI, France CAC, and UK FTSE.  The
  data are sampled in business time, i.e., weekends and holidays are
  omitted.
}
\usage{
data(EuStockMarkets)
}
\format{
  A multivariate time series with 1860 observations on 4 variables.
  The object is of class \code{"mts"}.
}
\source{
  The data were kindly provided by Erste Bank AG, Vienna, Austria.
}
\keyword{datasets}
\eof
\name{Exponential}
\alias{Exponential}
\alias{dexp}
\alias{pexp}
\alias{qexp}
\alias{rexp}
\title{The Exponential Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the exponential distribution with rate \code{rate}
  (i.e., mean \code{1/rate}).
}
\usage{
dexp(x, rate = 1, log = FALSE)
pexp(q, rate = 1, lower.tail = TRUE, log.p = FALSE)
qexp(p, rate = 1, lower.tail = TRUE, log.p = FALSE)
rexp(n, rate = 1)
}
\arguments{
  \item{x, q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{rate}{vector of rates.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dexp} gives the density,
  \code{pexp} gives the distribution function,
  \code{qexp} gives the quantile function, and
  \code{rexp} generates random deviates.
}
\details{
  If \code{rate} is not specified, it assumes the default value of
  \code{1}.

  The exponential distribution with rate \eqn{\lambda} has density
  \deqn{
    f(x) = \lambda {e}^{- \lambda x}}{
    f(x) = lambda e^(- lambda x)}
  for \eqn{x \ge 0}.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{exp}} for the exponential function,
  \code{\link{dgamma}} for the gamma distribution and
  \code{\link{dweibull}} for the Weibull distribution, both of which
  generalize the exponential.
}
\note{
  The cumulative hazard \eqn{H(t) = - \log(1 - F(t))}{H(t) = - log(1 - F(t))}
  is \code{-pexp(t, r, lower = FALSE, log = TRUE)}.
}
\examples{
dexp(1) - exp(-1) #-> 0
}
\keyword{distribution}
\eof
\name{FDist}
\alias{FDist}
\alias{df}
\alias{pf}
\alias{qf}
\alias{rf}
\title{The F Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the F distribution with \code{df1} and \code{df2}
  degrees of freedom (and optional non-centrality parameter
  \code{ncp}).
}
\usage{
df(x, df1, df2, log = FALSE)
pf(q, df1, df2, ncp=0, lower.tail = TRUE, log.p = FALSE)
qf(p, df1, df2,        lower.tail = TRUE, log.p = FALSE)
rf(n, df1, df2)
}
\arguments{
  \item{x, q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{df1, df2}{degrees of freedom.}
  \item{ncp}{non-centrality parameter.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{df} gives the density,
  \code{pf} gives the distribution function
  \code{qf} gives the quantile function, and
  \code{rf} generates random deviates.
}
\details{
  The F distribution with \code{df1 =} \eqn{n_1}{n1} and \code{df2 =}
  \eqn{n_2}{n2} degrees of freedom has density
  \deqn{
    f(x) = \frac{\Gamma(n_1/2 + n_2/2)}{\Gamma(n_1/2)\Gamma(n_2/2)}
    \left(\frac{n_1}{n_2}\right)^{n_1/2} x^{n_1/2 -1}
    \left(1 + \frac{n_1 x}{n_2}\right)^{-(n_1 + n_2) / 2}%
  }{f(x) = Gamma((n1 + n2)/2) / (Gamma(n1/2) Gamma(n2/2))
    (n1/n2)^(n1/2) x^(n1/2 - 1)
    (1 + (n1/n2) x)^-(n1 + n2)/2}
  for \eqn{x > 0}.

  It is the distribution of the ratio of the mean squares of
  \eqn{n_1}{n1} and \eqn{n_2}{n2} independent standard normals, and hence
  of the ratio of two independent chi-squared variates each divided by its
  degrees of freedom.  Since the ratio of a normal and the root
  mean-square of \eqn{m} independent normals has a Student's \eqn{t_m}
  distribution, the square of a \eqn{t_m} variate has a F distribution on
  1 and \eqn{m} degrees of freedom.

  The non-central F distribution is again the ratio of mean squares of
  independent normals of unit variance, but those in the numerator are
  allowed to have non-zero means and \code{ncp} is the sum of squares of
  the means.  See \code{\link{Chisquare}} for further details on
  non-central distributions.  
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{dchisq}} for chi-squared and \code{\link{dt}} for Student's
  t distributions.
}
\examples{
## the density of the square of a t_m is 2*dt(x, m)/(2*x)
# check this is the same as the density of F_{1,m}
x <- seq(0.001, 5, len=100)
all.equal(df(x^2, 1, 5), dt(x, 5)/x)

## Identity:  qf(2*p - 1, 1, df)) == qt(p, df)^2)  for  p >= 1/2
p <- seq(1/2, .99, length=50); df <- 10
rel.err <- function(x,y) ifelse(x==y,0, abs(x-y)/mean(abs(c(x,y))))
quantile(rel.err(qf(2*p - 1, df1=1, df2=df), qt(p, df)^2), .90)# ~= 7e-9
}
\keyword{distribution}
\eof
\name{GammaDist}
\alias{GammaDist}
\alias{dgamma}
\alias{pgamma}
\alias{qgamma}
\alias{rgamma}
\concept{incomplete gamma function}
\title{The Gamma Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the Gamma distribution with parameters \code{shape} and
  \code{scale}.
}
\usage{
dgamma(x, shape, rate = 1, scale = 1/rate, log = FALSE)
pgamma(q, shape, rate = 1, scale = 1/rate, lower.tail = TRUE, log.p = FALSE)
qgamma(p, shape, rate = 1, scale = 1/rate, lower.tail = TRUE, log.p = FALSE)
rgamma(n, shape, rate = 1, scale = 1/rate)
}
\arguments{
  \item{x, q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{rate}{an alternative way to specify the scale.}
  \item{shape, scale}{shape and scale parameters.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dgamma} gives the density,
  \code{pgamma} gives the distribution function
  \code{qgamma} gives the quantile function, and
  \code{rgamma} generates random deviates.
}
\details{
  If \code{scale} is omitted, it assumes the default value of \code{1}.

  The Gamma distribution with parameters \code{shape} \eqn{=\alpha}{= a}
  and \code{scale} \eqn{=\sigma}{= s} has density
  \deqn{
    f(x)= \frac{1}{{\sigma}^{\alpha}\Gamma(\alpha)} {x}^{\alpha-1} e^{-x/\sigma}%
  }{f(x)= 1/(s^a Gamma(a)) x^(a-1) e^-(x/s)}
  for \eqn{x > 0}, \eqn{\alpha > 0}{a > 0} and \eqn{\sigma > 0}{s > 0}.
  The mean and variance are
  \eqn{E(X) = \alpha\sigma}{E(X) = a*s} and
  \eqn{Var(X) = \alpha\sigma^2}{Var(X) = a*s^2}.

  \code{pgamma()} uses algorithm AS 239, see the references.
}
\note{
  The S parametrization is via \code{shape} and \code{rate}: S has no
  \code{scale} parameter.

  The cumulative hazard \eqn{H(t) = - \log(1 - F(t))}{H(t) = - log(1 - F(t))}
  is \code{-pgamma(t, ..., lower = FALSE, log = TRUE)}.

  \code{pgamma} is closely related to the incomplete gamma function.  As
  defined by Abramowitz and Stegun 6.5.1
  \deqn{P(a,x) = \frac{1}{\Gamma(a)} \int_0^x t^{a-1} e^{-t} dt}{P(a,x) =
      1/Gamma(a) integral_0^x t^(a-1) exp(-t) dt}
  \eqn{P(a, x)} is \code{pgamma(x, a)}.  Other authors (for example
  Karl Pearson in his 1922 tables) omit the normalizing factor,
  defining the incomplete gamma function as \code{pgamma(x, a) * gamma(a)}.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.

  Shea, B. L. (1988)
  Algorithm AS 239,  Chi-squared and Incomplete Gamma Integral,
  \emph{Applied Statistics (JRSS C)} \bold{37}, 466--473.

  Abramowitz, M. and Stegun, I. A. (1972)
  \emph{Handbook of Mathematical Functions.} New York: Dover.
  Chapter 6: Gamma and Related Functions.
}
\seealso{
  \code{\link{gamma}} for the Gamma function, \code{\link{dbeta}} for
  the Beta distribution and \code{\link{dchisq}} for the chi-squared
  distribution which is a special case of the Gamma distribution.
}
\examples{
-log(dgamma(1:4, shape=1))
p <- (1:9)/10
pgamma(qgamma(p,shape=2), shape=2)
1 - 1/exp(qgamma(p, shape=1))
}
\keyword{distribution}
\eof
\name{Geometric}
\alias{Geometric}
\alias{dgeom}
\alias{pgeom}
\alias{qgeom}
\alias{rgeom}
\title{The Geometric Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the geometric distribution with parameter \code{prob}.
}
\usage{
dgeom(x, prob, log = FALSE)
pgeom(q, prob, lower.tail = TRUE, log.p = FALSE)
qgeom(p, prob, lower.tail = TRUE, log.p = FALSE)
rgeom(n, prob)
}
\arguments{
  \item{x, q}{vector of quantiles representing the number of failures in
    a sequence of Bernoulli trials before success occurs.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{prob}{probability of success in each trial.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dgeom} gives the density,
  \code{pgeom} gives the distribution function,
  \code{qgeom} gives the quantile function, and
  \code{rgeom} generates random deviates.
}
\details{
  The geometric distribution with \code{prob} \eqn{= p} has density
  \deqn{p(x) = p {(1-p)}^{x}}{p(x) = p (1-p)^x}
  for \eqn{x = 0, 1, 2, \ldots}{x = 0, 1, 2, ...}

  If an element of \code{x} is not integer, the result of \code{pgeom}
  is zero, with a warning.

  The quantile is defined as the smallest value \eqn{x} such that
  \eqn{F(x) \ge p}, where \eqn{F} is the distribution function.
}
\seealso{
  \code{\link{dnbinom}} for the negative binomial which generalizes
  the geometric distribution.
}
\examples{
qgeom((1:9)/10, prob = .2)
Ni <- rgeom(20, prob = 1/4); table(factor(Ni, 0:max(Ni)))
}
\keyword{distribution}
\eof
\name{Harman23.cor}
\docType{data}
\alias{Harman23.cor}
\title{Harman Example 2.3}
\description{
  A correlation matrix of eight physical measurements on 305 girls between
  ages seven and seventeen.
}
\usage{data(Harman23.cor)}
\source{
  Harman, H. H. (1976)
  \emph{Modern Factor Analysis}, Third Edition Revised,
  University of Chicago Press, Table 2.3.
}
\examples{
data(Harman23.cor)
(Harman23.FA <- factanal(factors = 1, covmat = Harman23.cor))
for(factors in 2:4) print(update(Harman23.FA, factors = factors))
}
\keyword{datasets}
\eof
\name{Harman74.cor}
\docType{data}
\alias{Harman74.cor}
\title{Harman Example 7.4}
\description{
  A correlation matrix of 24 psychological tests given to 145 seventh and
  eight-grade children in a Chicago suburb by Holzinger and Swineford.
}
\usage{data(Harman74.cor)}
\source{
  Harman, H. H. (1976)
  \emph{Modern Factor Analysis}, Third Edition Revised,
  University of Chicago Press, Table 7.4.
}
\examples{
data(Harman74.cor)
(Harman74.FA <- factanal(factors = 1, covmat = Harman74.cor))
for(factors in 2:5) print(update(Harman74.FA, factors = factors))
Harman74.FA <- factanal(factors = 5, covmat = Harman74.cor,
                        rotation="promax")
print(Harman74.FA$loadings, sort = TRUE)
}
\keyword{datasets}
\eof
\name{HoltWinters}
\alias{HoltWinters}
\title{Holt-Winters Filtering}
\description{
  Computes Holt-Winters Filtering of a given time series.
  Unknown parameters are determined by minimizing the squared
  prediction error.
}
\usage{
HoltWinters(x, alpha = NULL, beta = NULL, gamma = NULL,
            seasonal = c("additive", "multiplicative"),
            start.periods = 3, l.start = NULL, b.start = NULL,
            s.start = NULL,
            optim.start = c(alpha = 0.3, beta = 0.1, gamma = 0.1),
            optim.control = list())
}
\arguments{
  \item{x}{An object of class \code{ts}}
  \item{alpha}{\eqn{alpha} parameter of Holt-Winters Filter}
  \item{beta}{\eqn{beta} parameter of Holt-Winters Filter. If set to 0,
    the function will do exponential smoothing.}
  \item{gamma}{\eqn{gamma} parameter used for the seasonal component.
    If set to 0, an non-seasonal model is fitted.}
  \item{seasonal}{Character string to select an \code{"additive"} 
    (the default) or \code{"multiplicative"} seasonal model. The first
    few characters are sufficient. (Only takes effect if 
    \code{gamma} is non-zero).}
  \item{start.periods}{Start periods used in the autodetection of start
    values. Must be at least 3.}
  \item{l.start}{Start value for level (a[0]).}
  \item{b.start}{Start value for trend (b[0]).}
  \item{s.start}{Vector of start values for the seasonal component
    (\eqn{s_1[0]...s_p[0]}{s_1[0]...s_p[0]})}
  \item{optim.start}{Vector with named components \code{alpha},
    \code{beta}, and \code{gamma} containing the starting values for the
    optimizer. Only the values needed must be specified.}
  \item{optim.control}{Optional list with additional control parameters
    passed to \code{optim}.}
}
\details{
  The additive Holt-Winters prediction function (for time series with
  period length p) is
  \deqn{\hat Y[t+h] = a[t] + h b[t] + s[t + 1 + (h - 1) \bmod p],}{
    Yhat[t+h] = a[t] + h * b[t] + s[t + 1 + (h - 1) mod p],}
  where \eqn{a[t]}, \eqn{b[t]} and \eqn{s[t]} are given by
  \deqn{a[t] = \alpha (Y[t] - s[t-p])  + (1-\alpha) (a[t-1] + b[t-1])}{
  a[t] = \alpha (Y[t] - s[t-p])  + (1-\alpha) (a[t-1] + b[t-1])}
  \deqn{b[t] = \beta (a[t] -a[t-1]) + (1-\beta)  b[t-1]}{
  b[t] = \beta (a[t] - a[t-1]) + (1-\beta) b[t-1]}
  \deqn{s[t] = \gamma (Y[t] - a[t])   + (1-\gamma) s[t-p]}{
  s[t] = \gamma (Y[t] - a[t]) + (1-\gamma) s[t-p]}

  The multiplicative Holt-Winters prediction function (for time series
  with period length p) is
  \deqn{\hat Y[t+h] = (a[t] + h b[t]) \times s[t + 1 + (h - 1) \bmod p].}{
    Yhat[t+h] = (a[t] + h * b[t]) * s[t + 1 + (h - 1) mod p],}
  where \eqn{a[t]}, \eqn{b[t]} and \eqn{s[t]} are given by
  \deqn{a[t] = \alpha (Y[t] / s[t-p])  + (1-\alpha) (a[t-1] + b[t-1])}{
  a[t] = \alpha (Y[t] / s[t-p])  + (1-\alpha) (a[t-1] + b[t-1])}
  \deqn{b[t] = \beta (a[t] - a[t-1]) + (1-\beta) b[t-1]}{
  b[t] = \beta (a[t] - a[t-1]) + (1-\beta) b[t-1]}
  \deqn{s[t] = \gamma (Y[t] / a[t])   + (1-\gamma) s[t-p]}{
  s[t] = \gamma  (Y[t] / a[t])  + (1-\gamma) s[t-p]}

  The function tries to find the optimal values of \eqn{\alpha} and/or
  \eqn{\beta} and/or \eqn{\gamma} by minimizing the squared
  one-step prediction error if they are omitted.

  For seasonal models, start values for \code{a}, \code{b} and \code{s} are detected by
  performing a simple decomposition in trend and seasonal component using
  moving averages (see function \code{decompose}) on the
  \code{start.periods} first periods (a simple linear regression on the
  trend component is used for starting level and trend.). For
  level/trend-models (no seasonal component), start values for a and b
  are x[2] and x[2] - x[1], respectively. For level-only models
  (ordinary exponential smoothing), the start value for a is x[1].
}
\value{
  An object of class \code{"HoltWinters"}, a list with components:
  \item{fitted}{A multiple time series with one column for the
    filtered series as well as for the level, trend and seasonal
    components, estimated contemporaneously (that is at time t and not
    at the end of the series).}
  \item{x}{The original series}
  \item{alpha}{alpha used for filtering}
  \item{beta}{beta used for filtering}
  \item(gamma){gamma used for filtering}
  \item{coefficients}{A vector with named components \code{a, b, s1, ..., sp}
    containing the estimated values for the level, trend and seasonal
    components}
  \item{seasonal}{The specified \code{seasonal}-parameter}
  \item{SSE}{The final sum of squared errors achieved in optimizing}
  \item{call}{The call used}
}
\references{
  C. C  Holt (1957)
  Forecasting seasonals and trends by exponentially weighted
  moving averages,
  ONR Research Memorandum, Carnigie Institute 52.

  P. R.  Winters (1960)
  Forecasting sales by exponentially weighted moving averages,
  \emph{Management Science} \bold{6}, 324--342.
}
\author{
  David Meyer \email{david.meyer@ci.tuwien.ac.at}
}
\seealso{
  \code{\link{predict.HoltWinters}},\code{\link{optim}}
}

\examples{
data(co2)

## Seasonal Holt-Winters
(m <- HoltWinters(co2))
plot(m)
plot(fitted(m))

data(AirPassengers)
(m <- HoltWinters(AirPassengers, seasonal = "mult"))
plot(m)

## Non-Seasonal Holt-Winters
data(uspop)
x <- uspop + rnorm(uspop, sd = 5)
m <- HoltWinters(x, gamma = 0)
plot(m)

## Exponential Smoothing
m2 <- HoltWinters(x, gamma = 0, beta = 0)
lines(fitted(m2)[,1], col = 3)
}
\keyword{ts}
\eof
\name{Hypergeometric}
\alias{Hypergeometric}
\alias{dhyper}
\alias{phyper}
\alias{qhyper}
\alias{rhyper}
\title{The Hypergeometric Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the hypergeometric distribution.
}
\usage{
dhyper(x, m, n, k, log = FALSE)
phyper(q, m, n, k, lower.tail = TRUE, log.p = FALSE)
qhyper(p, m, n, k, lower.tail = TRUE, log.p = FALSE)
rhyper(nn, m, n, k)
}
\arguments{
  \item{x, q}{vector of quantiles representing the number of white balls
    drawn without replacement from an urn which contains both black and
    white balls.}
  \item{m}{the number of white balls in the urn.}
  \item{n}{the number of black balls in the urn.}
  \item{k}{the number of balls drawn from the urn.}
  \item{p}{probability, it must be between 0 and 1.}
  \item{nn}{number of observations. If \code{length(nn) > 1}, the length
    is taken to be the number required.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dhyper} gives the density,
  \code{phyper} gives the distribution function,
  \code{qhyper} gives the quantile function, and
  \code{rhyper} generates random deviates.
}
\details{
  The hypergeometric distribution is used for sampling \emph{without}
  replacement.  The density of this distribution with parameters
  \code{m}, \code{n} and \code{k} (named \eqn{Np}, \eqn{N-Np}, and
  \eqn{n}, respectively in the reference below) is given by
  \deqn{
    p(x) = \left. {m \choose x}{n \choose k-x} \right/ {m+n \choose k}%
  }{p(x) =      choose(m, x) choose(n, k-x) / choose(m+n, k)}
  for \eqn{x = 0, \ldots, k}{x = 0, ..., k}.
}
\references{
  Johnson, N. L., Kotz, S., and Kemp, A. W. (1992)
  \emph{Univariate Discrete Distributions},
  Second Edition. New York: Wiley.
}
\examples{
m <- 10; n <- 7; k <- 8
x <- 0:(k+1)
rbind(phyper(x, m, n, k), dhyper(x, m, n, k))
all(phyper(x, m, n, k) == cumsum(dhyper(x, m, n, k)))# FALSE
## but error is very small:
signif(phyper(x, m, n, k) - cumsum(dhyper(x, m, n, k)), dig=3)
}
\keyword{distribution}
\eof
\name{IQR}
\alias{IQR}
\title{The Interquartile Range}
\usage{
IQR(x, na.rm = FALSE)
}
\description{computes interquartile range of the \code{x} values.}
\arguments{
  \item{x}{a numeric vector.}
  \item{na.rm}{logical. Should missing values be removed?}
}
\details{
  Note that this function computes the quartiles using the
  \code{\link{quantile}} function rather than following
  Tukey's recommendations,
  i.e., \code{IQR(x) = quantile(x,3/4) - quantile(x,1/4)}.

  For normally \eqn{N(m,1)} distributed \eqn{X}, the expected value of
  \code{IQR(X)} is \code{2*qnorm(3/4) = 1.3490}, i.e., for a normal-consistent
  estimate of the standard deviation, use \code{IQR(x) / 1.349}.
}
\references{
  Tukey, J. W. (1977).
  \emph{Exploratory Data Analysis.}
  Reading: Addison-Wesley.
}
\seealso{
 \code{\link{fivenum}}, \code{\link{mad}} which is more robust,
 \code{\link{range}}, \code{\link{quantile}}.
}
\examples{
data(rivers)
IQR(rivers)
}
\keyword{univar}
\keyword{robust}
\keyword{distribution}

\eof
\name{Indometh}
\docType{data}
\alias{Indometh}
\title{Pharmacokinetics of Indomethicin}
\description{
  The \code{Indometh} data frame has 66 rows and 3 columns of data on
  the pharmacokinetics of indomethicin.
}
\usage{data(Indometh)}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Subject}{
      an ordered factor with containing the subject codes.  The
      ordering is according to increasing maximum response.
    }
    \item{time}{
      a numeric vector of times at which blood samples were drawn (hr).
    }
    \item{conc}{
      a numeric vector of plasma concentrations of indomethicin (mcg/ml).
    }
  }
}
\details{
  Each of the six subjects were given an intravenous injection of
  indomethicin.
}
\source{
  Kwan, Breault, Umbenhauer, McMahon and Duggan (1976),
  \dQuote{Kinetics of Indomethicin absorption, elimination, and
    enterohepatic circulation in man},
  \emph{Journal of Pharmacokinetics and Biopharmaceutics}, \bold{4},
  255--280.

  Davidian, M. and Giltinan, D. M. (1995) \emph{Nonlinear Models for
    Repeated Measurement Data}, Chapman & Hall (section 5.2.4, p. 134)

  Pinheiro, J. C. and Bates, D. M. (2000) \emph{Mixed-effects Models in
    S and S-PLUS}, Springer.
}
\examples{
data(Indometh)
fm1 <- nls(conc ~ SSbiexp(time, A1, lrc1, A2, lrc2),
           data = Indometh, subset = Subject == 1)
summary(fm1)
}
\keyword{datasets}
\eof
\name{JohnsonJohnson}
\docType{data}
\alias{JohnsonJohnson}
\title{Quarterly Earnings per Johnson \& Johnson Share}
\description{
  Quarterly earnings (dollars) per Johnson \& Johnson share 1960--80.
}
\usage{data(JohnsonJohnson)}
\format{
  A quarterly time series
}
\source{
  Shumway, R. H. and Stoffer, D. S. (2000)
  \emph{Time Series Analysis and its Applications}.
  Second Edition.  Springer.  Example 1.1.
}
\examples{
data(JohnsonJohnson)
JJ <- log10(JohnsonJohnson)
plot(JJ)
(fit <- StructTS(JJ, type="BSM"))
tsdiag(fit)
sm <- tsSmooth(fit)
plot(cbind(JJ, sm[, 1], sm[, 3]-0.5), plot.type = "single",
     col = c("black", "green", "blue"))
abline(h = -0.5, col = "grey60")

monthplot(fit)
}
\keyword{datasets}
\eof
\name{KalmanLike}
\alias{KalmanLike}
\alias{KalmanRun}
\alias{KalmanSmooth}
\alias{KalmanForecast}
\alias{makeARIMA}
\title{Kalman Filtering}
\description{
  Use Kalman Filtering to find the (Gaussian) log-likelihood, or for
  forecasting or smoothing.
}
\usage{
KalmanLike(y, mod, nit = 0)
KalmanRun(y, mod, nit = 0)
KalmanSmooth(y, mod, nit = 0)
KalmanForecast(n.ahead = 10, mod)
makeARIMA(phi, theta, Delta, kappa = 1e6)
}
\arguments{
  \item{y}{a univariate time series.}
  \item{mod}{A list describing the state-space model: see Details.}
  \item{nit}{The time at which the initialization is computed.
    \code{nit = 0} implies that the initialization is for a one-step
    prediction, so \code{Pn} should not be computed at the first step.}
  \item{n.ahead}{The number of steps ahead for which prediction is
    required.}
  \item{phi, theta}{numeric vectors of length \eqn{\ge 0}{>=0} giving AR
    and MA parameters.}
  \item{Delta}{vector of differencing coefficients, so an ARMA model is
    fitted to \code{y[t] - Delta[1]*y[t-1] - \dots}.}
  \item{kappa}{the prior variance (as a multiple of the innovations
    variance) for the past observations in a differenced model.}
}
\details{
  These functions work with a general univariate state-space model
  with state vector \code{a}, transitions \code{a <- T a + R e},
  \eqn{e \sim {\cal N}(0, \kappa Q)}{e \~ N(0, kappa Q)} and observation
  equation \code{y = Z'a + eta},
  \eqn{(eta\equiv\eta), \eta \sim {\cal N}(0, \kappa h)}{eta ~ N(0, \kappa h)}.
  The likelihood is a profile likelihood after estimation of \eqn{\kappa}.

  The model is specified as a list with at least components
  \describe{
    \item{\code{T}}{the transition matrix}
    \item{\code{Z}}{the observation coeficients}
    \item{\code{h}}{the observation variance}
    \item{\code{V}}{\code{RQR'}}
    \item{\code{a}}{the current state estimate}
    \item{\code{P}}{the current estimate of the state uncertainty matrix}
    \item{\code{Pn}}{the estimate at time \eqn{t-1} of the state
      uncertainty matrix}
  }

  \code{KalmanSmooth} is the workhorse function for
  \code{\link{tsSmooth}}.

  \code{makeARIMA} constructs the state-space model for an ARIMA model.
}

\value{
  For \code{KalmanLike}, a list with components \code{Lik} (the
  log-likelihood less some constants) and \code{s2}, the estimate of
  of \eqn{\kappa}.

  For \code{KalmanRun}, a list with components \code{values}, a vector
  of length 2 giving the output of \code{KalmanLike}, \code{resid} (the
  residuals) and \code{states}, the contemporaneous state estimates,
  a matrix with one row for each time.

  For \code{KalmanSmooth}, a list with two components.
  Component \code{smooth} is a \code{n} by \code{p} matrix of state
  estimates based on all the observations, with one row for each time.
  Component \code{var} is a \code{n} by \code{p} by \code{p} array of
  variance matrices.

  For \code{KalmanForecast}, a list with components \code{pred}, the
  predictions, and \code{var}, the unscaled variances of the prediction
  errors (to be muliplied by \code{s2}).

  For \code{makeARIMA}, a model list including components for
  its arguments.
}

\section{Warning}{
  These functions are designed to be called from other functions which
  check the validity of the arguments passed, so very little checking is
  done.
}

\references{
  Durbin, J. and Koopman, S. J. (2001) \emph{Time Series Analysis by
    State Space Methods.}  Oxford University Press.
}

\seealso{
   \code{\link{arima}}, \code{\link{StructTS}}. \code{\link{tsSmooth}}.
 }
\keyword{ts}
\eof
\name{LakeHuron}
\docType{data}
\alias{LakeHuron}
\title{Level of Lake Huron 1875--1972}
\description{
  Annual measurements of the level, in feet, of Lake Huron 1875--1972.
}
\usage{data(LakeHuron)}
\format{
  A time series of length 98.
}
\source{
  Brockwell, P. J. & Davis, R. A. (1991).
  \emph{Time Series and Forecasting Methods}.
  Second edition. Springer, New York. Series A, page 555.

  Brockwell, P. J. & Davis, R. A. (1996).
  \emph{Introduction to Time Series and Forecasting}.
  Springer, New York.
  Sections 5.1 and 7.6.
}
\keyword{datasets}
\eof
\name{Loblolly}
\docType{data}
\alias{Loblolly}
\title{Growth of Loblolly pine trees}
\description{
  The \code{Loblolly} data frame has 84 rows and 3 columns of records of
  the growth of Loblolly pine trees.
}
\usage{data(Loblolly)}
\format{
  This data frame contains the following columns:
  \describe{
    \item{height}{
      a numeric vector of tree heights (ft).
    }
    \item{age}{
      a numeric vector of tree ages (yr).
    }
    \item{Seed}{
      an ordered factor indicating the seed source for the tree.
      The ordering is according to increasing maximum height.
    }
  }
}
\source{
  Kung, F. H. (1986),
  \dQuote{Fitting logistic growth curve with predetermined carrying
    capacity}, \emph{Proceedings of the Statistical Computing Section,
    American Statistical Association}, 340--343.

  Pinheiro, J. C. and Bates, D. M. (2000)
  \emph{Mixed-effects Models in S and S-PLUS}, Springer.
}
\examples{
data(Loblolly)
plot(height ~ age, data = Loblolly, subset = Seed == 329,
     xlab = "Tree age (yr)", las = 1,
     ylab = "Tree height (ft)",
     main = "Loblolly data and fitted curve (Seed 329 only)")
fm1 <- nls(height ~ SSasymp(age, Asym, R0, lrc),
           data = Loblolly, subset = Seed == 329)
summary(fm1)
age <- seq(0, 30, len = 101)
lines(age, predict(fm1, list(age = age)))
}
\keyword{datasets}
\eof
\name{Logistic}
\alias{Logistic}
\alias{dlogis}
\alias{plogis}
\alias{qlogis}
\alias{rlogis}
\title{The Logistic Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the logistic distribution with parameters
  \code{location} and \code{scale}.
}
\usage{
dlogis(x, location = 0, scale = 1, log = FALSE)
plogis(q, location = 0, scale = 1, lower.tail = TRUE, log.p = FALSE)
qlogis(p, location = 0, scale = 1, lower.tail = TRUE, log.p = FALSE)
rlogis(n, location = 0, scale = 1)
}
\arguments{
  \item{x, q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{location, scale}{location and scale parameters.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}

\value{
  \code{dlogis} gives the density,
  \code{plogis} gives the distribution function,
  \code{qlogis} gives the quantile function, and
  \code{rlogis} generates random deviates.
}
\details{
  If \code{location} or \code{scale} are omitted, they assume the
  default values of \code{0} and \code{1} respectively.

  The Logistic distribution with \code{location} \eqn{= \mu}{= m} and
  \code{scale} \eqn{= \sigma}{= s} has distribution function
  \deqn{
    F(x) = \frac{1}{1 + e^{-(x-\mu)/\sigma}}%
  }{F(x) = 1 / (1 + exp(-(x-m)/s))}  and density
  \deqn{
    f(x)= \frac{1}{\sigma}\frac{e^{(x-\mu)/\sigma}}{(1 + e^{(x-\mu)/\sigma})^2}%
  }{f(x) = 1/s exp((x-m)/s) (1 + exp((x-m)/s))^-2.}

  It is a long-tailed distribution with mean \eqn{\mu}{m} and variance
  \eqn{\pi^2/3 \sigma^2}{pi^2 /3 s^2}.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\examples{
var(rlogis(4000, 0, s = 5))# approximately (+/- 3)
pi^2/3 * 5^2
}
\keyword{distribution}
\eof
\name{Lognormal}
\alias{Lognormal}
\alias{dlnorm}
\alias{plnorm}
\alias{qlnorm}
\alias{rlnorm}
\title{The Log Normal Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the log normal distribution whose logarithm has mean
  equal to \code{meanlog} and standard  deviation equal to \code{sdlog}.
}
\usage{
dlnorm(x, meanlog = 0, sdlog = 1, log = FALSE)
plnorm(q, meanlog = 0, sdlog = 1, lower.tail = TRUE, log.p = FALSE)
qlnorm(p, meanlog = 0, sdlog = 1, lower.tail = TRUE, log.p = FALSE)
rlnorm(n, meanlog = 0, sdlog = 1)
}
\arguments{
  \item{x, q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{meanlog, sdlog}{mean and standard deviation of the distribution
    on the log scale with default values of \code{0} and \code{1} respectively.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dlnorm} gives the density,
  \code{plnorm} gives the distribution function,
  \code{qlnorm} gives the quantile function, and
  \code{rlnorm} generates random deviates.
}
\details{
  The log normal distribution has density
  \deqn{
    f(x) = \frac{1}{\sqrt{2\pi}\sigma x} e^{-(\log(x) - \mu)^2/2 \sigma^2}%
  }{f(x) = 1/(sqrt(2 pi) sigma x) e^-((log x - mu)^2 / (2 sigma^2))}
  where \eqn{\mu} and \eqn{\sigma} are the mean and standard
  deviation of the logarithm.
  The mean is \eqn{E(X) = exp(\mu + 1/2 \sigma^2)}, and the variance
  \eqn{Var(X) = exp(2\mu + \sigma^2)(exp(\sigma^2) - 1)}{%
    Var(X) = exp(2*mu + sigma^2)*(exp(sigma^2) - 1)} and
  hence the coefficient of variation is
  \eqn{\sqrt{exp(\sigma^2) - 1}}{sqrt(exp(sigma^2) - 1)} which is
    approximately \eqn{\sigma} when that is small (e.g., \eqn{\sigma < 1/2}).
%% Mode = exp(max(0, mu - sigma^2)); Median = exp(mu)
}
\note{
  The cumulative hazard \eqn{H(t) = - \log(1 - F(t))}{H(t) = - log(1 - F(t))}
  is \code{-plnorm(t, r, lower = FALSE, log = TRUE)}.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{dnorm}} for the normal distribution.
}
\examples{
dlnorm(1) == dnorm(0)
}
\keyword{distribution}
\eof
\name{Multinomial}
\alias{Multinomial}
\alias{rmultinom}
\alias{dmultinom}
\title{The Multinomial Distribution}
\description{
  Generate multinomially distributed random number vectors and
  compute multinomial \dQuote{density} probabilities.
}
\usage{
rmultinom(n, size, prob)
dmultinom(x, size = NULL, prob, log = FALSE)
}
\arguments{
  \item{x}{vector of length \eqn{K} of integers in \code{0:size}.}
  %%FUTURE:  matrix of \eqn{K} rows or ...
  \item{n}{number of random vectors to draw.}
  \item{size}{integer, say \eqn{N}, specifying the total number
    of objects that are put into \eqn{K} boxes in the typical multinomial
    experiment. For \code{dmultinom}, it defaults to \code{sum(x)}.}
  \item{prob}{numeric non-negative vector of length \eqn{K}, specifying
    the probability for the \eqn{K} classes; is internally normalized to
    sum 1.}
  \item{log}{logical; if TRUE, log probabilities are computed.}
}
\note{\code{dmultinom} is currently \emph{not vectorized} at all and has
  no C interface (API); this may be amended in the future.% yes, DO THIS!
}
\details{
  If \code{x} is a $K$-component vector, \code{dmultinom(x, prob)} is
  the probability
  \deqn{P(X_1=x_1,\ldots,X_K=x_k) = C \times \prod_{j=1}^K \pi_j^{x_j}}{%
    P(X[1]=x[1],\ldots,X[K]=x[k]) = C * prod(j=1,..,K) p[j]^x[j]}
  where \eqn{C} is the \dQuote{multinomial coefficient}
  \eqn{C = N! / (x_1! \cdots x_K!)}{C = N! / (x[1]! * \dots * x[K]!)}
  and \eqn{N = \sum_{j=1}^K x_j}{N = sum(j=1,..,K) x[j]}.
  \cr
  By definition, each component \eqn{X_j}{X[j]} is binomially distributed as
  \code{Bin(size, prob[j])} for \eqn{j = 1,\ldots,K}.

  The \code{rmultinom()} algorithm draws binomials from
  \eqn{Bin(n_j,P_j)}{Bin(n[j], P[j])} sequentially, where
  \eqn{n_1 = N}{n[1] = N} (N := \code{size}),
  \eqn{P_1 = \pi_1}{P[1] = p[1]} (\eqn{\pi}{p} is \code{prob} scaled to sum 1),
  and for \eqn{j \ge 2}{j >= 2}, recursively
  \eqn{n_j = N - \sum_{k=1}^{j-1} n_k}{%
    n[j]= N - sum(k=1, .., j-1) n[k]} and
  \eqn{P_j = \pi_j / (1 - \sum_{k=1}^{j-1} \pi_k)}{%
    P[j]= p[j] / (1 - sum(p[1:(j-1)]))}.
}
\value{
  For \code{rmultinom()},
  an integer \code{K x n} matrix where each column is a random vector
  generated according to the desired multinomial law, and hence summing
  to \code{size}.  Whereas the \emph{transposed} result would seem more
  natural at first, the returned matrix is more efficient because of
  columnwise storage.
}
%%\author{Martin Maechler}
\seealso{\code{\link{rbinom}} which is a special case conceptually.
%% but does not return 2-vectors
}
\examples{
rmultinom(10, size = 12, prob=c(0.1,0.2,0.8))

pr <- c(1,3,6,10) # normalization not necessary for generation
rmultinom(10, 20, prob = pr)

## all possible outcomes of Multinom(N = 3, K = 3)
X <- t(as.matrix(expand.grid(0:3, 0:3))); X <- X[, colSums(X) <= 3]
X <- rbind(X, 3:3 - colSums(X)); dimnames(X) <- list(letters[1:3], NULL)
X
round(apply(X, 2, function(x) dmultinom(x, prob = c(1,2,5))), 3)
}
\keyword{distribution}
\eof
% $Id: NLSstAsymptotic.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{NLSstAsymptotic}
\title{Fit the Asymptotic Regression Model}
\alias{NLSstAsymptotic}
\alias{NLSstAsymptotic.sortedXyData}
\usage{
NLSstAsymptotic(xy)
}
\arguments{
 \item{xy}{a \code{sortedXyData} object}
}
\description{
  Fits the asymptotic regression model, in the form \code{b0 +
      b1*(1-exp(-exp(lrc) * x)} to the \code{xy} data.  
  This can be used as a building block in determining starting estimates
  for more complicated models.
}
\value{
  A numeric value of length 3 with components labelled \code{b0},
  \code{b1}, and \code{lrc}.  \code{b0} is the estimated intercept on
  the \code{y}-axis, \code{b1} is the estimated difference between the
  asymptote and the \code{y}-intercept, and \code{lrc} is the estimated
  logarithm of the rate constant.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{\code{\link{SSasymp}}}
\examples{
data( Loblolly )
Lob.329 <- Loblolly[ Loblolly$Seed == "329", ]
NLSstAsymptotic(sortedXyData(expression(age), expression(height), Lob.329 ))
}
\keyword{manip}
\eof
% $Id: NLSstClosestX.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{NLSstClosestX}
\title{Inverse Interpolation}
\alias{NLSstClosestX}
\alias{NLSstClosestX.sortedXyData}
\usage{
NLSstClosestX(xy, yval)
}
\arguments{
 \item{xy}{a \code{sortedXyData} object}
 \item{yval}{a numeric value on the \code{y} scale}
}
\description{
  Use inverse linear interpolation to approximate the \code{x} value at
  which the function represented by \code{xy} is equal to \code{yval}.
}
\value{
  A single numeric value on the \code{x} scale.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{\code{\link{sortedXyData}}, \code{\link{NLSstLfAsymptote}},
  \code{\link{NLSstRtAsymptote}}, \code{\link{selfStart}}}
\examples{
data( DNase )
DNase.2 <- DNase[ DNase$Run == "2", ]
DN.srt <- sortedXyData( expression(log(conc)), expression(density), DNase.2 )
NLSstClosestX( DN.srt, 1.0 )
}
\keyword{manip}
\eof
% $Id: NLSstLfAsymptote.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{NLSstLfAsymptote}
\title{Horizontal Asymptote on the Left Side}
\alias{NLSstLfAsymptote}
\alias{NLSstLfAsymptote.sortedXyData}
\description{
  Provide an initial guess at the horizontal asymptote on the left side
  (i.e., small values of \code{x}) of the graph of \code{y} versus
  \code{x} from the \code{xy} object.  Primarily used within
  \code{initial} functions for self-starting nonlinear regression
  models.
}
\usage{
NLSstLfAsymptote(xy)
}
\arguments{
 \item{xy}{a \code{sortedXyData} object}
}
\value{
  A single numeric value estimating the horizontal asymptote for small
  \code{x}. 
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{
  \code{\link{sortedXyData}},
  \code{\link{NLSstClosestX}},
  \code{\link{NLSstRtAsymptote}},
  \code{\link{selfStart}}
}
\examples{
data( DNase )
DNase.2 <- DNase[ DNase$Run == "2", ]
DN.srt <- sortedXyData( expression(log(conc)), expression(density), DNase.2 )
NLSstLfAsymptote( DN.srt )
}
\keyword{manip}
\eof
% $Id: NLSstRtAsymptote.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{NLSstRtAsymptote}
\title{Horizontal Asymptote on the Right Side}
\alias{NLSstRtAsymptote}
\alias{NLSstRtAsymptote.sortedXyData}
\description{
  Provide an initial guess at the horizontal asymptote on the right side
  (i.e., large values of \code{x}) of the graph of \code{y} versus
  \code{x} from the \code{xy} object.  Primarily used within
  \code{initial} functions for self-starting nonlinear regression
  models.
}
\usage{
NLSstRtAsymptote(xy)
}
\arguments{
 \item{xy}{a \code{sortedXyData} object}
}
\value{
  A single numeric value estimating the horizontal asymptote for large
  \code{x}.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{
  \code{\link{sortedXyData}},
  \code{\link{NLSstClosestX}},
  \code{\link{NLSstRtAsymptote}},
  \code{\link{selfStart}}
}
\examples{
data( DNase )
DNase.2 <- DNase[ DNase$Run == "2", ]
DN.srt <- sortedXyData( expression(log(conc)), expression(density), DNase.2 )
NLSstRtAsymptote( DN.srt )
}
\keyword{manip}
\eof
\name{NegBinomial}
\alias{NegBinomial}
\alias{dnbinom}
\alias{pnbinom}
\alias{qnbinom}
\alias{rnbinom}
\title{The Negative Binomial Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the negative binomial distribution with parameters
  \code{size} and \code{prob}.
}
\usage{
dnbinom(x, size, prob, mu, log = FALSE)
pnbinom(q, size, prob, mu, lower.tail = TRUE, log.p = FALSE)
qnbinom(p, size, prob, mu, lower.tail = TRUE, log.p = FALSE)
rnbinom(n, size, prob, mu)
}
\arguments{
  \item{x}{vector of (non-negative integer) quantiles.}
  \item{q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{size}{target for number of successful trials, or dispersion
    parameter (the shape parameter of the gamma mixing distribution).}
  \item{prob}{probability of success in each trial.}
  \item{mu}{alternative parametrization via mean: see Details}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dnbinom} gives the density,
  \code{pnbinom} gives the distribution function,
  \code{qnbinom} gives the quantile function, and
  \code{rnbinom} generates random deviates.
}
\details{
  The negative binomial distribution with \code{size} \eqn{= n} and
  \code{prob} \eqn{= p} has density
  \deqn{
    p(x) = \frac{\Gamma(x+n)}{\Gamma(n) x!} p^n (1-p)^x}{%
    p(x) = Gamma(x+n)/(Gamma(n) x!) p^n (1-p)^x}
  for \eqn{x = 0, 1, 2, \ldots}

  This represents the number of failures
  which occur in a sequence of Bernoulli trials before a target number
  of successes is reached.

  A negative binomial distribution can arise as a mixture of Poisson
  distributions with mean distributed as a
  gamma (\code{\link{pgamma}}) distribution with scale parameter
  \code{(1 - prob)/prob} and shape parameter \code{size}.  (This
  definition allows non-integer values of \code{size}.)
  In this model \code{prob} = \code{scale/(1+scale)}, and the mean is
  \code{size * (1 - prob)/prob)}
  
  The alternative parametrization (often used in ecology) is by the
  \emph{mean} \code{mu}, and \code{size}, the \emph{dispersion parameter},
  where \code{prob} = \code{size/(size+mu)}.  In this parametrization the
  variance is \code{mu + mu^2/size}.

  If an element of \code{x} is not integer, the result of \code{dnbinom}
  is zero, with a warning.

  The quantile is defined as the smallest value \eqn{x} such that
  \eqn{F(x) \ge p}, where \eqn{F} is the distribution function.
}
\seealso{
  \code{\link{dbinom}} for the binomial, \code{\link{dpois}} for the
  Poisson and \code{\link{dgeom}} for the geometric distribution, which
  is a special case of the negative binomial.
}
\examples{
x <- 0:11
dnbinom(x, size = 1, prob = 1/2) * 2^(1 + x) # == 1
126 /  dnbinom(0:8, size  = 2, prob  = 1/2) #- theoretically integer

## Cumulative ('p') = Sum of discrete prob.s ('d');  Relative error :
summary(1 - cumsum(dnbinom(x, size = 2, prob = 1/2)) /
                  pnbinom(x, size  = 2, prob = 1/2))

x <- 0:15
size <- (1:20)/4
persp(x,size, dnb <- outer(x,size,function(x,s)dnbinom(x,s, pr= 0.4)),
      xlab = "x", ylab = "s", zlab="density", theta = 150)
title(tit <- "negative binomial density(x,s, pr = 0.4)  vs.  x & s")

image  (x,size, log10(dnb), main= paste("log [",tit,"]"))
contour(x,size, log10(dnb),add=TRUE)

## Alternative parametrization
x1 <- rnbinom(500, mu = 4, size = 1)
x2 <- rnbinom(500, mu = 4, size = 10)
x3 <- rnbinom(500, mu = 4, size = 100)
h1 <- hist(x1, breaks = 20, plot = FALSE)
h2 <- hist(x2, breaks = h1$breaks, plot = FALSE)
h3 <- hist(x3, breaks = h1$breaks, plot = FALSE)
barplot(rbind(h1$counts, h2$counts, h3$counts),
        beside = TRUE, col = c("red","blue","cyan"),
        names.arg = round(h1$breaks[-length(h1$breaks)]))
}
\keyword{distribution}
\eof
\name{Nile}
\docType{data}
\alias{Nile}
\title{Flow of the River Nile}
\usage{data(Nile)}
\description{
  Measurements of the annual flow of the river Nile at Ashwan 1871--1970.
}
\format{
  A time series of length 100.
}
\source{
  Durbin, J. and Koopman, S. J. (2001) \emph{Time Series Analysis by
    State Space Methods.}  Oxford University Press.
  \url{http://www.ssfpack.com/dkbook/}
}
\references{
  Balke, N. S. (1993) Detecting level shifts in time series.
  \emph{Journal of Business and Economic Statistics} \bold{11}, 81--92.
  
  Cobb, G. W. (1978) The problem of the Nile: conditional solution to a
  change-point problem.  \emph{Biometrika} \bold{65}, 243--51.
}
\examples{
data(Nile)
par(mfrow = c(2,2))
plot(Nile)
acf(Nile)
pacf(Nile)
ar(Nile) # selects order 2
cpgram(ar(Nile)$resid)
par(mfrow = c(1,1))
arima(Nile, c(2, 0, 0))

## Now consider missing values, following Durbin & Koopman
NileNA <- Nile
NileNA[c(21:40, 61:80)] <- NA
arima(NileNA, c(2, 0, 0))
plot(NileNA)
pred <- predict(arima(window(NileNA, 1871, 1890), c(2, 0, 0)), n.ahead = 20)
lines(pred$pred, lty = 3, col = "red")
lines(pred$pred + 2*pred$se, lty=2, col="blue")
lines(pred$pred - 2*pred$se, lty=2, col="blue")
pred <- predict(arima(window(NileNA, 1871, 1930), c(2, 0, 0)), n.ahead = 20)
lines(pred$pred, lty = 3, col = "red")
lines(pred$pred + 2*pred$se, lty=2, col="blue")
lines(pred$pred - 2*pred$se, lty=2, col="blue")

## Structural time series models
par(mfrow = c(3, 1))
plot(Nile)
## local level model
(fit <- StructTS(Nile, type = "level"))
lines(fitted(fit), lty = 2)              # contempareneous smoothing
lines(tsSmooth(fit), lty = 2, col = 4)   # fixed-interval smoothing
plot(residuals(fit)); abline(h = 0, lty = 3)
## local trend model
(fit2 <- StructTS(Nile, type = "trend")) ## constant trend fitted
pred <- predict(fit, n.ahead = 30)
## with 50\% confidence interval
ts.plot(Nile, pred$pred, pred$pred + 0.67*pred$se, pred$pred -0.67*pred$se)

## Now consider missing values
plot(NileNA)
(fit3 <- StructTS(NileNA, type = "level"))
lines(fitted(fit3), lty = 2)
lines(tsSmooth(fit3), lty = 3)
plot(residuals(fit3)); abline(h = 0, lty = 3)
}
\keyword{datasets}
\eof
\name{Normal}
\alias{Normal}
\alias{dnorm}
\alias{pnorm}
\alias{qnorm}
\alias{rnorm}
\title{The Normal Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the normal distribution with mean equal to \code{mean}
  and standard deviation equal to \code{sd}.
}
\usage{
dnorm(x, mean=0, sd=1, log = FALSE)
pnorm(q, mean=0, sd=1, lower.tail = TRUE, log.p = FALSE)
qnorm(p, mean=0, sd=1, lower.tail = TRUE, log.p = FALSE)
rnorm(n, mean=0, sd=1)
}
\arguments{
  \item{x,q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{mean}{vector of means.}
  \item{sd}{vector of standard deviations.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dnorm} gives the density,
  \code{pnorm} gives the distribution function,
  \code{qnorm} gives the quantile function, and
  \code{rnorm} generates random deviates.
}
\details{
  If \code{mean} or \code{sd} are not specified they assume the default
  values of \code{0} and \code{1}, respectively.

  The normal distribution has density
  \deqn{
    f(x) =
    \frac{1}{\sqrt{2\pi}\sigma} e^{-(x-\mu)^2/2\sigma^2}}{
    f(x) = 1/(sqrt(2 pi) sigma) e^-((x - mu)^2/(2 sigma^2))
  }
  where \eqn{\mu}{mu} is the mean of the distribution and
  \eqn{\sigma}{sigma} the standard deviation.

  \code{qnorm} is based on Wichura's algorithm AS 241 which provides
  precise results up to about 16 digits.
}
\seealso{
  \code{\link{runif}} and \code{\link{.Random.seed}} about random number
  generation, and \code{\link{dlnorm}} for the \emph{Log}normal distribution.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.

  Wichura, M. J. (1988)
  Algorithm AS 241: The Percentage Points of the Normal Distribution.
  \emph{Applied Statistics}, \bold{37}, 477--484.
}
\examples{
dnorm(0) == 1/ sqrt(2*pi)
dnorm(1) == exp(-1/2)/ sqrt(2*pi)
dnorm(1) == 1/ sqrt(2*pi*exp(1))

## Using "log = TRUE" for an extended range :
par(mfrow=c(2,1))
plot(function(x)dnorm(x, log=TRUE), -60, 50, main = "log { Normal density }")
curve(log(dnorm(x)), add=TRUE, col="red",lwd=2)
mtext("dnorm(x, log=TRUE)", adj=0); mtext("log(dnorm(x))", col="red", adj=1)

plot(function(x)pnorm(x, log=TRUE), -50, 10, main = "log { Normal Cumulative }")
curve(log(pnorm(x)), add=TRUE, col="red",lwd=2)
mtext("pnorm(x, log=TRUE)", adj=0); mtext("log(pnorm(x))", col="red", adj=1)

## if you want the so-called 'error function'
erf <- function(x) 2 * pnorm(x * sqrt(2)) - 1
## (see Abrahamowitz and Stegun 29.2.29)
## and the so-called 'complementary error function'
erfc <- function(x) 2 * pnorm(x * sqrt(2), lower=FALSE)
}
\keyword{distribution}
\eof
\name{Orange}
\docType{data}
\alias{Orange}
\title{Growth of orange trees}
\description{
  The \code{Orange} data frame has 35 rows and 3 columns of records of
  the growth of orange trees.
}
\usage{data(Orange)}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Tree}{
      an ordered factor indicating the tree on which the measurement is
      made.  The ordering is according to increasing maximum diameter.
    }
    \item{age}{
      a numeric vector giving the age of the tree (days since 1968/12/31)
    }
    \item{circumference}{
      a numeric vector of trunk circumferences (mm).  This is probably
      \dQuote{circumference at breast height}, a standard measurement in
      forestry.
    }
  }
}
\source{
  Draper, N. R. and Smith, H. (1998), \emph{Applied Regression Analysis
    (3rd ed)}, Wiley (exercise 24.N).

  Pinheiro, J. C. and Bates, D. M. (2000) \emph{Mixed-effects Models
    in S and S-PLUS}, Springer.
}
\examples{
data(Orange)
coplot(circumference ~ age | Tree, data = Orange, show = FALSE)
fm1 <- nls(circumference ~ SSlogis(age, Asym, xmid, scal),
           data = Orange, subset = Tree == 3)
plot(circumference ~ age, data = Orange, subset = Tree == 3,
     xlab = "Tree age (days since 1968/12/31)",
     ylab = "Tree circumference (mm)", las = 1,
     main = "Orange tree data and fitted model (Tree 3 only)")
age <- seq(0, 1600, len = 101)
lines(age, predict(fm1, list(age = age)))
}
\keyword{datasets}
\eof
\name{Poisson}
\alias{Poisson}
\alias{dpois}
\alias{ppois}
\alias{qpois}
\alias{rpois}
\title{The Poisson Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the Poisson distribution with parameter \code{lambda}.
}
\usage{
dpois(x, lambda, log = FALSE)
ppois(q, lambda, lower.tail = TRUE, log.p = FALSE)
qpois(p, lambda, lower.tail = TRUE, log.p = FALSE)
rpois(n, lambda)
}
\arguments{
  \item{x}{vector of (non-negative integer) quantiles.}
  \item{q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of random values to return.}
  \item{lambda}{vector of positive means.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dpois} gives the (log) density,
  \code{ppois} gives the (log) distribution function,
  \code{qpois} gives the quantile function, and
  \code{rpois} generates random deviates.
}
\details{
  The Poisson distribution has density
%-- please leave the linebreaking for the next two ! --
  \deqn{p(x) = \frac{\lambda^x e^{-\lambda}}{x!}}{%
        p(x) = lambda^x exp(-lambda)/x!}
  for \eqn{x = 0, 1, 2, \ldots}.  The mean and variance are
  \eqn{E(X) = Var(X) = \lambda}.

  If an element of \code{x} is not integer, the result of \code{dpois}
  is zero, with a warning.
  \eqn{p(x)} is computed using Loader's algorithm, see the reference in
  \code{\link{dbinom}}.

  The quantile is left continuous: \code{qgeom(q, prob)} is the largest
  integer \eqn{x} such that \eqn{P(X \le x) < q}.

  Setting \code{lower.tail = FALSE} allows to get much more precise
  results when the default, \code{lower.tail = TRUE} would return 1, see
  the example below.
}
\seealso{
  \code{\link{dbinom}} for the binomial and \code{\link{dnbinom}} for
  the negative binomial distribution.
}
\examples{
-log(dpois(0:7, lambda=1) * gamma(1+ 0:7)) # == 1
Ni <- rpois(50, lam= 4); table(factor(Ni, 0:max(Ni)))

1 - ppois(10*(15:25), lambda=100)		# becomes 0 (cancellation)
    ppois(10*(15:25), lambda=100, lower=FALSE)	# no cancellation

par(mfrow = c(2, 1))
x <- seq(-0.01, 5, 0.01)
plot(x, ppois(x, 1), type="s", ylab="F(x)", main="Poisson(1) CDF")
plot(x, pbinom(x, 100, 0.01),type="s", ylab="F(x)",
     main="Binomial(100, 0.01) CDF")
}
\keyword{distribution}
\eof
\name{Puromycin}
\docType{data}
\alias{Puromycin}
\title{Reaction velocity of an enzymatic reaction}
\description{
  The \code{Puromycin} data frame has 23 rows and 3 columns of the
  reaction velocity versus substrate concentration in an enzymatic
  reaction involving untreated cells or cells treated with Puromycin.
}
\usage{data(Puromycin)}
\format{
  This data frame contains the following columns:
  \describe{
    \item{conc}{
      a numeric vector of substrate concentrations (ppm)
    }
    \item{rate}{
      a numeric vector of instantaneous reaction rates (counts/min/min)
    }
    \item{state}{
      a factor with levels
      \code{treated} 
      \code{untreated} 
    }
  }
}
\details{
  Data on the \dQuote{velocity} of an enzymatic reaction were obtained
  by Treloar (1974).  The number of counts per minute of radioactive
  product from the reaction was measured as a function of substrate
  concentration in parts per million (ppm) and from these counts the
  initial rate, or \dQuote{velocity,} of the reaction was calculated
  (counts/min/min).  The experiment was conducted once with the enzyme
  treated with Puromycin, and once with the enzyme untreated.
}
\source{
  Bates, D.M. and Watts, D.G. (1988),
  \emph{Nonlinear Regression Analysis and Its Applications},
  Wiley, Appendix A1.3.

  Treloar, M. A. (1974), \emph{Effects of Puromycin on
    Galactosyltransferase in Golgi Membranes}, M.Sc. Thesis, U. of
  Toronto. 
}
\examples{
data(Puromycin)
plot(rate ~ conc, data = Puromycin, las = 1,
     xlab = "Substrate concentration (ppm)",
     ylab = "Reaction velocity (counts/min/min)",
     pch = as.integer(Puromycin$state),
     col = as.integer(Puromycin$state),
     main = "Puromycin data and fitted Michaelis-Menten curves")
## simplest form of fitting the Michaelis-Menten model to these data
fm1 <- nls(rate ~ Vm * conc/(K + conc), data = Puromycin,
           subset = state == "treated",
           start = c(Vm = 200, K = 0.05), trace = TRUE)
fm2 <- nls(rate ~ Vm * conc/(K + conc), data = Puromycin,
           subset = state == "untreated",
           start = c(Vm = 160, K = 0.05), trace = TRUE)
summary(fm1)
summary(fm2)
## using partial linearity
fm3 <- nls(rate ~ conc/(K + conc), data = Puromycin,
           subset = state == "treated", start = c(K = 0.05),
           algorithm = "plinear", trace = TRUE)
## using a self-starting model
fm4 <- nls(rate ~ SSmicmen(conc, Vm, K), data = Puromycin,
           subset = state == "treated")
summary(fm4)
## add fitted lines to the plot
conc <- seq(0, 1.2, len = 101)
lines(conc, predict(fm1, list(conc = conc)), lty = 1, col = 1)
lines(conc, predict(fm2, list(conc = conc)), lty = 2, col = 2)
legend(0.8, 120, levels(Puromycin$state),
       col = 1:2, lty = 1:2, pch = 1:2)
}
\keyword{datasets}
\eof
% $Id: SSasymp.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{SSasymp}
\title{Asymptotic Regression Model}
\usage{
SSasymp(input, Asym, R0, lrc)
}
\alias{SSasymp}
\arguments{
 \item{input}{a numeric vector of values at which to evaluate the model.}
 \item{Asym}{a numeric parameter representing the horizontal asymptote on
   the right side (very large values of \code{input}).}
 \item{R0}{a numeric parameter representing the response when
   \code{input} is zero.}
 \item{lrc}{a numeric parameter representing the natural logarithm of
   the rate constant.}
}
\description{
  This \code{selfStart} model evaluates the asymptotic regression
  function and its gradient.  It has an \code{initial} attribute that
  will evaluate initial estimates of the parameters \code{Asym}, \code{R0},
  and \code{lrc} for a given set of data.
}
\value{
  a numeric vector of the same length as \code{input}.  It is the value of
  the expression \code{Asym+(R0-Asym)*exp(-exp(lrc)*input)}.  If all of
  the arguments \code{Asym}, \code{R0}, and \code{lrc} are
  names of objects, the gradient matrix with respect to these names is
  attached as an attribute named \code{gradient}.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{\code{\link{nls}}, \code{\link{selfStart}}
}
\examples{
data( Loblolly )
Lob.329 <- Loblolly[ Loblolly$Seed == "329", ]
SSasymp( Lob.329$age, 100, -8.5, -3.2 )  # response only
Asym <- 100
resp0 <- -8.5
lrc <- -3.2
SSasymp( Lob.329$age, Asym, resp0, lrc ) # response and gradient
getInitial(height ~ SSasymp( age, Asym, resp0, lrc), data = Lob.329)
## Initial values are in fact the converged values
fm1 <- nls(height ~ SSasymp( age, Asym, resp0, lrc), data = Lob.329)
summary(fm1)
\dontshow{
  xx <- seq(0, 5, len = 101)
  yy <- 5 - 4 * exp(-xx/(2*log(2)))
  par(mar = c(0, 0, 4.1, 0))
  plot(xx, yy, type = "l", axes = FALSE, ylim = c(0,6), xlim = c(-1, 5),
       xlab = "", ylab = "", lwd = 2,
       main = "Parameters in the SSasymp model")
  usr <- par("usr")
  arrows(usr[1], 0, usr[2], 0, length = 0.1, angle = 25)
  arrows(0, usr[3], 0, usr[4], length = 0.1, angle = 25)
  text(usr[2] - 0.2, 0.1, "x", adj = c(1, 0))
  text(-0.1, usr[4], "y", adj = c(1, 1))
  abline(h = 5, lty = 2, lwd = 0)
  arrows(-0.8, 2.1, -0.8, 0, length = 0.1, angle = 25)
  arrows(-0.8, 2.9, -0.8, 5, length = 0.1, angle = 25)
  text(-0.8, 2.5, expression(phi[1]), adj = c(0.5, 0.5))
  segments(-0.4, 1, 0, 1, lty = 2, lwd = 0.75)
  arrows(-0.3, 0.25, -0.3, 0, length = 0.07, angle = 25)
  arrows(-0.3, 0.75, -0.3, 1, length = 0.07, angle = 25)
  text(-0.3, 0.5, expression(phi[2]), adj = c(0.5, 0.5))
  segments(1, 3.025, 1, 4, lty = 2, lwd = 0.75)
  arrows(0.2, 3.5, 0, 3.5, length = 0.08, angle = 25)
  arrows(0.8, 3.5, 1, 3.5, length = 0.08, angle = 25)
  text(0.5, 3.5, expression(t[0.5]), adj = c(0.5, 0.5))
}
}
\keyword{models}
\eof
% $Id: SSasympOff.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{SSasympOff}
\title{Asymptotic Regression Model with an Offset}
\usage{
SSasympOff(input, Asym, lrc, c0)
}
\alias{SSasympOff}
\arguments{
 \item{input}{a numeric vector of values at which to evaluate the model.}
 \item{Asym}{a numeric parameter representing the horizontal asymptote on
   the right side (very large values of \code{input}).}
 \item{lrc}{a numeric parameter representing the natural logarithm of
   the rate constant.}
 \item{c0}{a numeric parameter representing the \code{input} for which the
   response is zero.}
}
\description{
  This \code{selfStart} model evaluates an alternative parameterization
  of the asymptotic
  regression function and the gradient with respect to those parameters.
  It has an \code{initial}
  attribute that creates initial estimates of the parameters
  \code{Asym}, \code{lrc}, and \code{c0}.
}
\value{
  a numeric vector of the same length as \code{input}.  It is the value of
  the expression \code{Asym*(1 - exp(-exp(lrc)*(input - c0)))}.  If all of
  the arguments \code{Asym}, \code{lrc}, and \code{c0} are
  names of objects, the gradient matrix with respect to these names is
  attached as an attribute named \code{gradient}.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{\code{\link{nls}}, \code{\link{selfStart}}
}
\examples{
data( CO2 )
CO2.Qn1 <- CO2[CO2$Plant == "Qn1", ]
SSasympOff( CO2.Qn1$conc, 32, -4, 43 )  # response only
Asym <- 32; lrc <- -4; c0 <- 43
SSasympOff( CO2.Qn1$conc, Asym, lrc, c0 ) # response and gradient
getInitial(uptake ~ SSasymp( conc, Asym, lrc, c0), data = CO2.Qn1)
## Initial values are in fact the converged values
fm1 <- nls(uptake ~ SSasymp( conc, Asym, lrc, c0), data = CO2.Qn1)
summary(fm1)
\dontshow{
  xx <- seq(0.5, 5, len = 101)
  yy <- 5 * (1 -  exp(-(xx - 0.5)/(2*log(2))))
  par(mar = c(0, 0, 4.0, 0))
  plot(xx, yy, type = "l", axes = FALSE, ylim = c(0,6), xlim = c(-1, 5),
       xlab = "", ylab = "", lwd = 2,
       main = "Parameters in the SSasympOff model")
  usr <- par("usr")
  arrows(usr[1], 0, usr[2], 0, length = 0.1, angle = 25)
  arrows(0, usr[3], 0, usr[4], length = 0.1, angle = 25)
  text(usr[2] - 0.2, 0.1, "x", adj = c(1, 0))
  text(-0.1, usr[4], "y", adj = c(1, 1))
  abline(h = 5, lty = 2, lwd = 0)
  arrows(-0.8, 2.1, -0.8, 0, length = 0.1, angle = 25)
  arrows(-0.8, 2.9, -0.8, 5, length = 0.1, angle = 25)
  text(-0.8, 2.5, expression(phi[1]), adj = c(0.5, 0.5))
  segments(0.5, 0, 0.5, 3, lty = 2, lwd = 0.75)
  text(0.5, 3.1, expression(phi[3]), adj = c(0.5, 0))
  segments(1.5, 2.525, 1.5, 3, lty = 2, lwd = 0.75)
  arrows(0.7, 2.65, 0.5, 2.65, length = 0.08, angle = 25)
  arrows(1.3, 2.65, 1.5, 2.65, length = 0.08, angle = 25)
  text(1.0, 2.65, expression(t[0.5]), adj = c(0.5, 0.5))
}
}
\keyword{models}
\eof
% $Id: SSasympOrig.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{SSasympOrig}
\title{Asymptotic Regression Model through the Origin}
\usage{
SSasympOrig(input, Asym, lrc)
}
\alias{SSasympOrig}
\arguments{
 \item{input}{a numeric vector of values at which to evaluate the model.}
 \item{Asym}{a numeric parameter representing the horizontal asymptote.}
 \item{lrc}{a numeric parameter representing the natural logarithm of
   the rate constant.}
}
\description{
  This \code{selfStart} model evaluates the asymptotic regression
  function through the origin and its gradient.  It has an
  \code{initial} attribute that will evaluate initial estimates of the
  parameters \code{Asym} and \code{lrc} for a given set of data.
}
\value{
  a numeric vector of the same length as \code{input}.  It is the value of
  the expression \code{Asym*(1 - exp(-exp(lrc)*input))}.  If all of
  the arguments \code{Asym} and \code{lrc} are
  names of objects, the gradient matrix with respect to these names is
  attached as an attribute named \code{gradient}.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{\code{\link{nls}}, \code{\link{selfStart}}
}
\examples{
data( Loblolly )
Lob.329 <- Loblolly[ Loblolly$Seed == "329", ]
SSasympOrig( Lob.329$age, 100, -3.2 )  # response only
Asym <- 100; lrc <- -3.2
SSasympOrig( Lob.329$age, Asym, lrc ) # response and gradient
getInitial(height ~ SSasympOrig(age, Asym, lrc), data = Lob.329)
## Initial values are in fact the converged values
fm1 <- nls(height ~ SSasympOrig( age, Asym, lrc), data = Lob.329)
summary(fm1)
\dontshow{
  xx <- seq(0, 5, len = 101)
  yy <- 5 * (1- exp(-xx/(2*log(2))))
  par(mar = c(0, 0, 3.5, 0))
  plot(xx, yy, type = "l", axes = FALSE, ylim = c(0,6), xlim = c(-1, 5),
       xlab = "", ylab = "", lwd = 2,
       main = "Parameters in the SSasympOrig model")
  usr <- par("usr")
  arrows(usr[1], 0, usr[2], 0, length = 0.1, angle = 25)
  arrows(0, usr[3], 0, usr[4], length = 0.1, angle = 25)
  text(usr[2] - 0.2, 0.1, "x", adj = c(1, 0))
  text(-0.1, usr[4], "y", adj = c(1, 1))
  abline(h = 5, lty = 2, lwd = 0)
  arrows(-0.8, 2.1, -0.8, 0, length = 0.1, angle = 25)
  arrows(-0.8, 2.9, -0.8, 5, length = 0.1, angle = 25)
  text(-0.8, 2.5, expression(phi[1]), adj = c(0.5, 0.5))
  segments(1, 2.525, 1, 3.5, lty = 2, lwd = 0.75)
  arrows(0.2, 3.0, 0, 3.0, length = 0.08, angle = 25)
  arrows(0.8, 3.0, 1, 3.0, length = 0.08, angle = 25)
  text(0.5, 3.0, expression(t[0.5]), adj = c(0.5, 0.5))
}
}
\keyword{models}
\eof
% $Id: SSbiexp.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{SSbiexp}
\title{Biexponential model}
\usage{
SSbiexp(input, A1, lrc1, A2, lrc2)
}
\alias{SSbiexp}
\arguments{
 \item{input}{a numeric vector of values at which to evaluate the model.}
 \item{A1}{a numeric parameter representing the multiplier of the first
   exponential.}
 \item{lrc1}{a numeric parameter representing the natural logarithm of
   the rate constant of the first exponential.}
 \item{A2}{a numeric parameter representing the multiplier of the second
   exponential.}
 \item{lrc2}{a numeric parameter representing the natural logarithm of
   the rate constant of the second exponential.}
}
\description{
  This \code{selfStart} model evaluates the biexponential model function
  and its gradient.  It has an \code{initial} attribute that 
  creates initial estimates of the parameters \code{A1}, \code{lrc1},
  \code{A2}, and \code{lrc2}.
}
\value{
  a numeric vector of the same length as \code{input}.  It is the value of
  the expression
  \code{A1*exp(-exp(lrc1)*input)+A2*exp(-exp(lrc2)*input)}.
  If all of the arguments \code{A1}, \code{lrc1}, \code{A2}, and
  \code{lrc2} are names of objects, the gradient matrix with respect to
  these names is attached as an attribute named \code{gradient}.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{\code{\link{nls}}, \code{\link{selfStart}}
}
\examples{
data( Indometh )
Indo.1 <- Indometh[Indometh$Subject == 1, ]
SSbiexp( Indo.1$time, 3, 1, 0.6, -1.3 )  # response only
A1 <- 3; lrc1 <- 1; A2 <- 0.6; lrc2 <- -1.3
SSbiexp( Indo.1$time, A1, lrc1, A2, lrc2 ) # response and gradient
getInitial(conc ~ SSbiexp(time, A1, lrc1, A2, lrc2), data = Indo.1)
## Initial values are in fact the converged values
fm1 <- nls(conc ~ SSbiexp(time, A1, lrc1, A2, lrc2), data = Indo.1)
summary(fm1)
\dontshow{
  xx <- seq(0, 5, len = 101)
  y1 <- 3.5 * exp(-4*xx)
  y2 <- 1.5 * exp(-xx)
  yy <- y1 + y2
  par(mar = c(0, 0, 3.5, 0))
  plot(xx, yy, type = "n", axes = FALSE, ylim = c(-0.2,6), xlim = c(0, 5),
       xlab = "", ylab = "", main = "Components of the SSbiexp model")
  usr <- par("usr")
  lines(xx, y1, lty = 2, lwd = 0.75)
  lines(xx, y2, lty = 3, lwd = 0.75)
  lines(xx, yy, lwd = 2)
  segments(1:5, 0, 1:5, -0.15)
  segments(0, 1:5, -0.06, 1:5)
  text(-0.1, 1:5, 1:5, adj = c(1, 0.25), cex = 0.7)
  text(1:5, -0.15, 1:5, adj = c(0.5, 1), cex = 0.7)
  arrows(usr[1], 0, usr[2], 0, length = 0.1, angle = 25)
  arrows(0, usr[3], 0, usr[4], length = 0.1, angle = 25)
  text(usr[2] - 0.2, 0.1, "x", adj = c(1, 0))
  text(-0.1, usr[4], "y", adj = c(1, 1))
}
}
\keyword{models}
\eof
% $Id: SSfol.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{SSfol}
\title{First-order Compartment Model}
\usage{
SSfol(Dose, input, lKe, lKa, lCl)
}
\alias{SSfol}
\arguments{
  \item{Dose}{a numeric value representing the initial dose.}
  \item{input}{a numeric vector at which to evaluate the model.}
  \item{lKe}{a numeric parameter representing the natural logarithm of
    the elimination rate constant.}
  \item{lKa}{a numeric parameter representing the natural logarithm of
    the absorption rate constant.}
  \item{lCl}{a numeric parameter representing the natural logarithm of
    the clearance.}
}
\description{
  This \code{selfStart} model evaluates the first-order compartment
  function and its gradient.  It has an \code{initial} attribute that 
  creates initial estimates of the parameters \code{lKe}, \code{lKa},
  and \code{lCl}.
}
\value{
  a numeric vector of the same length as \code{input}, which is the
  value of the expression
  \code{Dose * exp(lKe+lKa-lCl) * (exp(-exp(lKe)*input)-exp(-exp(lKa)*input)) / (exp(lKa)-exp(lKe))}.

  If all of the arguments \code{lKe}, \code{lKa}, and \code{lCl} are
  names of objects, the gradient matrix with respect to these names is
  attached as an attribute named \code{gradient}.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{\code{\link{nls}}, \code{\link{selfStart}}
}
\examples{
data( Theoph )
Theoph.1 <- Theoph[ Theoph$Subject == 1, ]
SSfol( Theoph.1$Dose, Theoph.1$Time, -2.5, 0.5, -3 )  # response only
lKe <- -2.5; lKa <- 0.5; lCl <- -3
SSfol( Theoph.1$Dose, Theoph.1$Time, lKe, lKa, lCl ) # response and gradient
getInitial(conc ~ SSfol(Dose, Time, lKe, lKa, lCl), data = Theoph.1)
## Initial values are in fact the converged values
fm1 <- nls(conc ~ SSfol(Dose, Time, lKe, lKa, lCl), data = Theoph.1)
summary(fm1)
}
\keyword{models}
\eof
% $Id: SSfpl.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{SSfpl}
\title{Four-parameter Logistic Model}
\usage{
SSfpl(input, A, B, xmid, scal)
}
\alias{SSfpl}
\arguments{
 \item{input}{a numeric vector of values at which to evaluate the model.}
 \item{A}{a numeric parameter representing the horizontal asymptote on
   the left side (very small values of \code{input}).}
 \item{B}{a numeric parameter representing the horizontal asymptote on
   the right side (very large values of \code{input}).}
 \item{xmid}{a numeric parameter representing the \code{input} value at the
   inflection point of the curve.  The value of \code{SSfpl} will be
   midway between \code{A} and \code{B} at \code{xmid}.}
 \item{scal}{a numeric scale parameter on the \code{input} axis.}
}
\description{
  This \code{selfStart} model evaluates the four-parameter logistic
  function and its gradient.  It has an \code{initial} attribute that
  will evaluate initial estimates of the parameters \code{A}, \code{B},
  \code{xmid}, and \code{scal} for a given set of data.
}
\value{
  a numeric vector of the same length as \code{input}.  It is the value of
  the expression \code{A+(B-A)/(1+exp((xmid-input)/scal))}.  If all of
  the arguments \code{A}, \code{B}, \code{xmid}, and \code{scal} are
  names of objects, the gradient matrix with respect to these names is
  attached as an attribute named \code{gradient}.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{\code{\link{nls}}, \code{\link{selfStart}}
}
\examples{
data( ChickWeight )
Chick.1 <- ChickWeight[ChickWeight$Chick == 1, ]
SSfpl( Chick.1$Time, 13, 368, 14, 6 )  # response only
A <- 13; B <- 368; xmid <- 14; scal <- 6
SSfpl( Chick.1$Time, A, B, xmid, scal ) # response and gradient
getInitial(weight ~ SSfpl(Time, A, B, xmid, scal), data = Chick.1)
## Initial values are in fact the converged values
fm1 <- nls(weight ~ SSfpl(Time, A, B, xmid, scal), data = Chick.1)
summary(fm1)
\dontshow{
  xx <- seq(-0.5, 5, len = 101)
  yy <- 1 + 4 / ( 1 + exp((2-xx)))
  par(mar = c(0, 0, 3.5, 0))
  plot(xx, yy, type = "l", axes = FALSE, ylim = c(0,6), xlim = c(-1, 5),
       xlab = "", ylab = "", lwd = 2,
       main = "Parameters in the SSfpl model")
  usr <- par("usr")
  arrows(usr[1], 0, usr[2], 0, length = 0.1, angle = 25)
  arrows(0, usr[3], 0, usr[4], length = 0.1, angle = 25)
  text(usr[2] - 0.2, 0.1, "x", adj = c(1, 0))
  text(-0.1, usr[4], "y", adj = c(1, 1))
  abline(h = 5, lty = 2, lwd = 0)
  arrows(-0.8, 2.1, -0.8, 0, length = 0.1, angle = 25)
  arrows(-0.8, 2.9, -0.8, 5, length = 0.1, angle = 25)
  text(-0.8, 2.5, expression(phi[1]), adj = c(0.5, 0.5))
  abline(h = 1, lty = 2, lwd = 0)
  arrows(-0.3, 0.25, -0.3, 0, length = 0.07, angle = 25)
  arrows(-0.3, 0.75, -0.3, 1, length = 0.07, angle = 25)
  text(-0.3, 0.5, expression(phi[2]), adj = c(0.5, 0.5))
  segments(2, 0, 2, 3.3, lty = 2, lwd = 0.75)
  text(2, 3.3, expression(phi[3]), adj = c(0.5, 0))
  segments(3, 1+4/(1+exp(-1)) - 0.025, 3, 2.5, lty = 2, lwd = 0.75)
  arrows(2.3, 2.7, 2.0, 2.7, length = 0.08, angle = 25)
  arrows(2.7, 2.7, 3.0, 2.7, length = 0.08, angle = 25)
  text(2.5, 2.7, expression(phi[4]), adj = c(0.5, 0.5))
}
}
\keyword{models}
\eof
% $Id: SSgompertz.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{SSgompertz}
\title{Gompertz Growth Model}
\usage{
SSgompertz(x, Asym, b2, b3)
}
\alias{SSgompertz}
\arguments{
 \item{x}{a numeric vector of values at which to evaluate the model.}
 \item{Asym}{a numeric parameter representing the asymptote.}
 \item{b2}{a numeric parameter related to the value of the function at
   \code{x = 0}}
 \item{b3}{a numeric parameter related to the scale the \code{x} axis.}
}
\description{
  This \code{selfStart} model evaluates the Gompertz growth model
  and its gradient.  It has an \code{initial} attribute that
  creates initial estimates of the parameters \code{Asym},
  \code{b2}, and \code{b3}.
}
\value{
  a numeric vector of the same length as \code{input}.  It is the value of
  the expression \code{Asym*exp(-b2*b3^x)}.  If all of
  the arguments \code{Asym}, \code{b2}, and \code{b3} are
  names of objects the gradient matrix with respect to these names is attached as
  an attribute named \code{gradient}.
}
\author{Douglas Bates}
\seealso{\code{\link{nls}}, \code{\link{selfStart}}
}
\examples{
data(DNase )
DNase.1 <- subset(DNase, Run == 1)
SSlogis(log(DNase.1$conc), 4.5, 2.3, 0.7)  # response only
Asym <- 4.5; b2 <- 2.3; b3 <- 0.7
SSgompertz(log(DNase.1$conc), Asym, b2, b3 ) # response and gradient
getInitial(density ~ SSgompertz(log(conc), Asym, b2, b3),
           data = DNase.1)
## Initial values are in fact the converged values
fm1 <- nls(density ~ SSgompertz(log(conc), Asym, b2, b3),
           data = DNase.1)
summary(fm1)
}
\keyword{models}
\eof
% $Id: SSlogis.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{SSlogis}
\title{Logistic Model}
\usage{
SSlogis(input, Asym, xmid, scal)
}
\alias{SSlogis}
\arguments{
 \item{input}{a numeric vector of values at which to evaluate the model.}
 \item{Asym}{a numeric parameter representing the asymptote.}
 \item{xmid}{a numeric parameter representing the \code{x} value at the
   inflection point of the curve.  The value of \code{SSlogis} will be
   \code{Asym/2} at \code{xmid}.}
 \item{scal}{a numeric scale parameter on the \code{input} axis.}
}
\description{
  This \code{selfStart} model evaluates the logistic
  function and its gradient.  It has an \code{initial} attribute that
  creates initial estimates of the parameters \code{Asym},
  \code{xmid}, and \code{scal}.
}
\value{
  a numeric vector of the same length as \code{input}.  It is the value of
  the expression \code{Asym/(1+exp((xmid-input)/scal))}.  If all of
  the arguments \code{Asym}, \code{xmid}, and \code{scal} are
  names of objects the gradient matrix with respect to these names is attached as
  an attribute named \code{gradient}.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{\code{\link{nls}}, \code{\link{selfStart}}
}
\examples{
data( ChickWeight )
Chick.1 <- ChickWeight[ChickWeight$Chick == 1, ]
SSlogis( Chick.1$Time, 368, 14, 6 )  # response only
Asym <- 368; xmid <- 14; scal <- 6
SSlogis( Chick.1$Time, Asym, xmid, scal ) # response and gradient
getInitial(weight ~ SSlogis(Time, Asym, xmid, scal), data = Chick.1)
## Initial values are in fact the converged values
fm1 <- nls(weight ~ SSlogis(Time, Asym, xmid, scal), data = Chick.1)
summary(fm1)
\dontshow{
  xx <- seq(-0.5, 5, len = 101)
  yy <- 5 / ( 1 + exp((2-xx)))
  par(mar = c(0, 0, 3.5, 0))
  plot(xx, yy, type = "l", axes = FALSE, ylim = c(0,6), xlim = c(-1, 5),
       xlab = "", ylab = "", lwd = 2,
       main = "Parameters in the SSlogis model")
  usr <- par("usr")
  arrows(usr[1], 0, usr[2], 0, length = 0.1, angle = 25)
  arrows(0, usr[3], 0, usr[4], length = 0.1, angle = 25)
  text(usr[2] - 0.2, 0.1, "x", adj = c(1, 0))
  text(-0.1, usr[4], "y", adj = c(1, 1))
  abline(h = 5, lty = 2, lwd = 0)
  arrows(-0.8, 2.1, -0.8, 0, length = 0.1, angle = 25)
  arrows(-0.8, 2.9, -0.8, 5, length = 0.1, angle = 25)
  text(-0.8, 2.5, expression(phi[1]), adj = c(0.5, 0.5))
  segments(2, 0, 2, 4.0, lty = 2, lwd = 0.75)
  text(2, 4.0, expression(phi[2]), adj = c(0.5, 0))
  segments(3, 5/(1+exp(-1)) + 0.025, 3, 4.0, lty = 2, lwd = 0.75)
  arrows(2.3, 3.8, 2.0, 3.8, length = 0.08, angle = 25)
  arrows(2.7, 3.8, 3.0, 3.8, length = 0.08, angle = 25)
  text(2.5, 3.8, expression(phi[3]), adj = c(0.5, 0.5))
}
}
\keyword{models}
\eof
% $Id: SSmicmen.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{SSmicmen}
\title{Michaelis-Menten Model}
\usage{
SSmicmen(input, Vm, K)
}
\alias{SSmicmen}
\arguments{
 \item{input}{a numeric vector of values at which to evaluate the model.}
 \item{Vm}{a numeric parameter representing the maximum value of the response.}
 \item{K}{a numeric parameter representing the \code{input} value at
   which half the maximum response is attained.  In the field of enzyme
   kinetics this is called the Michaelis parameter.}
}
\description{
  This \code{selfStart} model evaluates the Michaelis-Menten model and
  its gradient.  It has an \code{initial} attribute that
  will evaluate initial estimates of the parameters \code{Vm} and \code{K}
}
\value{
  a numeric vector of the same length as \code{input}.  It is the value of
  the expression \code{Vm*input/(K+input)}.  If both
  the arguments \code{Vm} and \code{K} are
  names of objects, the gradient matrix with respect to these names is
  attached as an attribute named \code{gradient}.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{\code{\link{nls}}, \code{\link{selfStart}}
}
\examples{
data( Puromycin )
PurTrt <- Puromycin[ Puromycin$state == "treated", ]
SSmicmen( PurTrt$conc, 200, 0.05 )  # response only
Vm <- 200; K <- 0.05
SSmicmen( PurTrt$conc, Vm, K ) # response and gradient
getInitial(rate ~ SSmicmen(conc, Vm, K), data = PurTrt)
## Initial values are in fact the converged values
fm1 <- nls(rate ~ SSmicmen(conc, Vm, K), data = PurTrt)
summary( fm1 )
## Alternative call using the subset argument
fm2 <- nls(rate ~ SSmicmen(conc, Vm, K), data = Puromycin,
           subset = state == "treated")
summary(fm2)
\dontshow{
  xx <- seq(0, 5, len = 101)
  yy <- 5 * xx/(1+xx)
  par(mar = c(0, 0, 3.5, 0))
  plot(xx, yy, type = "l", axes = FALSE, ylim = c(0,6), xlim = c(-1, 5),
       xlab = "", ylab = "", lwd = 2,
       main = "Parameters in the SSmicmen model")
  usr <- par("usr")
  arrows(usr[1], 0, usr[2], 0, length = 0.1, angle = 25)
  arrows(0, usr[3], 0, usr[4], length = 0.1, angle = 25)
  text(usr[2] - 0.2, 0.1, "x", adj = c(1, 0))
  text(-0.1, usr[4], "y", adj = c(1, 1))
  abline(h = 5, lty = 2, lwd = 0)
  arrows(-0.8, 2.1, -0.8, 0, length = 0.1, angle = 25)
  arrows(-0.8, 2.9, -0.8, 5, length = 0.1, angle = 25)
  text(-0.8, 2.5, expression(phi[1]), adj = c(0.5, 0.5))
  segments(1, 0, 1, 2.7, lty = 2, lwd = 0.75)
  text(1, 2.7, expression(phi[2]), adj = c(0.5, 0))
}
}
\keyword{models}
\eof
% $Id: SSweibull.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{SSweibull}
\title{Weibull growth curve model}
\usage{
SSweibull(x, Asym, Drop, lrc, pwr)
}
\alias{SSweibull}
\arguments{
 \item{x}{a numeric vector of values at which to evaluate the model.}
 \item{Asym}{a numeric parameter representing the horizontal asymptote on
   the right side (very small values of \code{x}).}
 \item{Drop}{a numeric parameter representing the change from
   \code{Asym} to the \code{y} intercept.}
 \item{lrc}{a numeric parameter representing the natural logarithm of
   the rate constant.}
 \item{pwr}{a numeric parameter representing the power to which \code{x}
   is raised.}
}
\description{
  This \code{selfStart} model evaluates the Weibull model for growth
  curve data and its gradient.  It has an \code{initial} attribute that
  will evaluate initial estimates of the parameters \code{Asym}, \code{Drop},
  \code{lrc}, and \code{pwr} for a given set of data.
}
\value{
  a numeric vector of the same length as \code{x}.  It is the value of
  the expression \code{Asym-Drop*exp(-exp(lrc)*x^pwr)}.  If all of
  the arguments \code{Asym}, \code{Drop}, \code{lrc}, and \code{pwr} are
  names of objects, the gradient matrix with respect to these names is
  attached as an attribute named \code{gradient}.
}
\details{
  This model is a generalization of the \code{\link{SSasymp}} model in
  that it reduces to \code{SSasymp} when \code{pwr} is unity.
  }
\author{Douglas Bates}
\references{
  Ratkowsky, David A. (1983), \emph{Nonlinear Regression Modeling},
  Dekker. (section 4.4.5)
}
\seealso{\code{\link{nls}}, \code{\link{selfStart}}, \code{\link{SSasymp}}
}
\examples{
data(ChickWeight)
Chick.6 <- subset(ChickWeight, (Chick == 6) & (Time > 0))
SSweibull(Chick.6$Time, 160, 115, -5.5, 2.5 )  # response only
Asym <- 160; Drop <- 115; lrc <- -5.5; pwr <- 2.5
SSweibull(Chick.6$Time, Asym, Drop, lrc, pwr)  # response and gradient
getInitial(weight ~ SSweibull(Time, Asym, Drop, lrc, pwr), data = Chick.6)
## Initial values are in fact the converged values
fm1 <- nls(weight ~ SSweibull(Time, Asym, Drop, lrc, pwr), data = Chick.6)
summary(fm1)
}
\keyword{models}
\eof
\name{SignRank}
\alias{SignRank}
\alias{dsignrank}
\alias{psignrank}
\alias{qsignrank}
\alias{rsignrank}
\title{Distribution of the Wilcoxon Signed Rank Statistic}
\description{
  Density, distribution function, quantile function and random
  generation for the distribution of the Wilcoxon Signed Rank statistic
  obtained from a sample with size \code{n}.
}
\usage{
dsignrank(x, n, log = FALSE)
psignrank(q, n, lower.tail = TRUE, log.p = FALSE)
qsignrank(p, n, lower.tail = TRUE, log.p = FALSE)
rsignrank(nn, n)
}
\arguments{
  \item{x,q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{nn}{number of observations. If \code{length(nn) > 1}, the length
    is taken to be the number required.}
  \item{n}{number(s) of observations in the sample(s).  A positive
    integer, or a vector of such integers.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dsignrank} gives the density,
  \code{psignrank} gives the distribution function,
  \code{qsignrank} gives the quantile function, and
  \code{rsignrank} generates random deviates.
}
\details{
  This distribution is obtained as follows.  Let \code{x} be a sample of
  size \code{n} from a continuous distribution symmetric about the
  origin.  Then the Wilcoxon signed rank statistic is the sum of the
  ranks of the absolute values \code{x[i]} for which \code{x[i]} is
  positive.  This statistic takes values between \eqn{0} and
  \eqn{n(n+1)/2}, and its mean and variance are \eqn{n(n+1)/4} and
  \eqn{n(n+1)(2n+1)/24}, respectively.

  If either of the first two arguments is a vector, the recycling rule is
  used to do the calculations for all combinations of the two up to
  the length of the longer vector.
}
\author{Kurt Hornik \email{hornik@ci.tuwien.ac.at}}
\seealso{
  \code{\link{wilcox.test}} to calculate the statistic from data, find p
  values and so on.
  
  \code{\link{dwilcox}} etc, for the distribution of \emph{two-sample} Wilcoxon
  rank sum statistic.
}
\examples{
par(mfrow=c(2,2))
for(n in c(4:5,10,40)) {
  x <- seq(0, n*(n+1)/2, length=501)
  plot(x, dsignrank(x,n=n), type='l', main=paste("dsignrank(x,n=",n,")"))
}
}
\keyword{distribution}
\eof
\name{StructTS}
\alias{StructTS}
\alias{print.StructTS}
\alias{predict.StructTS}
\title{Fit Structural Time Series}
\description{
  Fit a structural model for a time series by maximum likelihood.
}
\usage{
StructTS(x, type = c("level", "trend", "BSM"), init = NULL,
         fixed = NULL, optim.control = NULL)
}
\arguments{
  \item{x}{a univariate numeric time series. Missing values are allowed.}

  \item{type}{the class of structural model.  If omitted, a BSM is used
    for a time series with \code{frequency(x) > 1}, and a local trend
    model otherwise.}

  \item{init}{initial values of the variance parameters.}

  \item{fixed}{optional numeric vector of the same length as the total
    number of parameters.  If supplied, only non-\code{NA} entries in
    \code{fixed} will be varied.  Probably most useful for setting
    variances to zero.}

  \item{optim.control}{List of control parameters for
    \code{\link{optim}}.  Method \code{"L-BFGS-B"} is used.}
}
\details{
 \emph{Structural time series} models are (linear Gaussian) state-space
 models for (univariate) time series based on a decomposition of the
 series into a number of components. They are specified by a set of
 error variances, some of which may be zero.

 The simplest model is the \emph{local level} model specified by
 \code{type = "level"}.  This has an underlying level \eqn{\mu_t}{m[t]} which
 evolves by
 \deqn{\mu_{t+1} = \mu_t + \xi_t,  \qquad \xi_t \sim N(0, \sigma^2_\xi)}{%
   m[t+1] = m[t] + xi[t], xi[t] ~ N(0, sigma^2_xi)}
 The observations are
 \deqn{x_t = \mu_t + \epsilon_t, \qquad \epsilon_t \sim  N(0, \sigma^2_\epsilon)}{%
   x[t] = m[t] + eps[t], eps[t] ~  N(0, sigma^2_eps)}
 There are two parameters, \eqn{\sigma^2_\xi}{sigma^2_xi}
 and \eqn{\sigma^2_\epsilon}{sigma^2_eps}.  It is an ARIMA(0,1,1) model,
 but with restrictions on the parameter set.

 The \emph{local linear trend model}, \code{type = "trend"}, has the same
 measurement equation, but with a time-varying slope in the dynamics for
 \eqn{\mu_t}{m[t]}, given by
 \deqn{
   \mu_{t+1} = \mu_t + \nu_t + \xi_t, \qquad  \xi_t \sim N(0, \sigma^2_\xi)
 }{%
   m[t+1] = m[t] + n[t] + xi[t], xi[t] ~ N(0, sigma^2_xi)
 }
 \deqn{
   \nu_{t+1} = \nu_t + \zeta_t, \qquad \zeta_t \sim N(0, \sigma^2_\zeta)
 }{%
   n[t+1] = n[t] + zeta[t],  zeta[t] ~ N(0, sigma^2_zeta)
 }
 with three variance parameters.  It is not uncommon to find
 \eqn{\sigma^2_\zeta = 0}{sigma^2_zeta = 0} (which reduces to the local
 level model) or \eqn{\sigma^2_\xi = 0}{sigma^2_xi = 0}, which ensures a
 smooth trend.  This is a restricted ARIMA(0,2,2) model.

 The \emph{basic structural model}, \code{type = "BSM"}, is a local
 trend model with an additional seasonal component. Thus the measurement
 equation is
 \deqn{x_t = \mu_t + \gamma_t + \epsilon_t, \qquad \epsilon_t \sim  N(0, \sigma^2_\epsilon)}{%
   x[t] = m[t] + s[t] + eps[t], exp[t] ~  N(0, sigma^2_eps)}
 where \eqn{\gamma_t}{s[t]} is a seasonal component with dynamics
 \deqn{
   \gamma_{t+1} = -\gamma_t + \cdots + \gamma_{t-s+2} + \omega_t, \qquad
   \omega_t \sim N(0, \sigma^2_\omega)
 }{%
   s[t+1] = -s[t] - \dots - s[t - s + 2] + w[t],  w[t] ~ N(0, sigma^2_w)
 }
 The boundary case \eqn{\sigma^2_\omega = 0}{sigma^2_w = 0} corresponds
 to a deterministic (but arbitrary) seasonal pattern.  (This is
 sometimes known as the \sQuote{dummy variable} version of the BSM.)
}
\value{
  A list of class \code{"StructTS"} with components:

  \item{coef}{the estimated variances of the components.}
  \item{loglik}{the maximized log-likelihood.  Note that as all these
    models are non-stationary this includes a diffuse prior for some
    observations and hence is not comparable with \code{\link{arima}}
    nor different types of structural models.}
  \item{data}{the time series \code{x}.}
  \item{residuals}{the standardized residuals.}
  \item{fitted}{a multiple time series with one component for the level,
    slope and seasonal components, estimated contemporaneously (that is
    at time \eqn{t} and not at the end of the series.}
  \item{call}{the matched call.}
  \item{series}{the name of the series \code{x}.}
  \item{convergence}{the value returned by \code{\link{optim}}.}
  \item{model, model0}{Lists representing the Kalman Filter used in the
    fitting.  See \code{\link{KalmanLike}}.  \code{model0} is the
    initial state of the filter, \code{model} its final state.}
  \item{xtsp}{the \code{tsp} attributes of \code{x}.}
}
\references{
  Brockwell, P. J. & Davis, R. A. (1996).
  \emph{Introduction to Time Series and Forecasting}.
  Springer, New York.
  Sections 8.2 and 8.5.

  Durbin, J. and Koopman, S. J. (2001) \emph{Time Series Analysis by
    State Space Methods.}  Oxford University Press.
  
  Harvey, A. C. (1989)
  \emph{Forecasting, Structural Time Series Models and the Kalman Filter}.
  Cambridge University Press.

  Harvey, A. C. (1993) \emph{Time Series Models}.
  2nd Edition, Harvester Wheatsheaf.
}
\note{
  Optimization of structural models is a lot harder than many of the
  references admit. For example, the \code{\link{AirPassengers}} data
  are considered in Brockwell \& Davis (1996): their solution appears to
  be a local maximum, but nowhere near as good a fit as that produced by
  \code{StructTS}.  It is quite common to find fits with one or more
  variances zero, and this can include \eqn{\sigma^2_\epsilon}{sigma^2_eps}.
}

\seealso{
  \code{\link{KalmanLike}}, \code{\link{tsSmooth}}.
}

\examples{
## see also JohnsonJohnson, Nile and AirPassengers

data(treering)
trees <- window(treering, start=0)
(fit <- StructTS(trees, type = "level"))
plot(trees)
lines(fitted(fit), col = "green")
tsdiag(fit)

data(UKgas)
(fit <- StructTS(log10(UKgas), type = "BSM"))
par(mfrow = c(4, 1))
plot(log10(UKgas))
plot(cbind(fitted(fit), resids=resid(fit)), main = "UK gas consumption")
}
\keyword{ts}
\eof
\name{TDist}
\alias{TDist}
\alias{dt}
\alias{pt}
\alias{qt}
\alias{rt}
\title{The Student t Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the t distribution with \code{df} degrees of freedom
  (and optional noncentrality parameter \code{ncp}).
}
\usage{
dt(x, df, ncp=0, log = FALSE)
pt(q, df, ncp=0, lower.tail = TRUE, log.p = FALSE)
qt(p, df,        lower.tail = TRUE, log.p = FALSE)
rt(n, df)
}
\arguments{
  \item{x, q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{df}{degrees of freedom (\eqn{> 0}, maybe non-integer).}
  \item{ncp}{non-centrality parameter \eqn{\delta};
    currently for \code{pt()} and \code{dt()}, only for \code{ncp <= 37.62}.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dt} gives the density,
  \code{pt} gives the distribution function,
  \code{qt} gives the quantile function, and
  \code{rt} generates random deviates.
}
\details{
  The \eqn{t} distribution with \code{df} \eqn{= \nu}{= n} degrees of
  freedom has density
  \deqn{
    f(x) = \frac{\Gamma ((\nu+1)/2)}{\sqrt{\pi \nu} \Gamma (\nu/2)}
    (1 + x^2/\nu)^{-(\nu+1)/2}%
  }{f(x) = Gamma((n+1)/2) / (sqrt(n pi) Gamma(n/2)) (1 + x^2/n)^-((n+1)/2)}
  for all real \eqn{x}.
  It has mean \eqn{0} (for \eqn{\nu > 1}{n > 1}) and
  variance \eqn{\frac{\nu}{\nu-2}}{n/(n-2)} (for \eqn{\nu > 2}{n > 2}).

  The general \emph{non-central} \eqn{t}
  with parameters \eqn{(\nu,\delta)}{(df,Del)} \code{= (df, ncp)}
  is defined as a the distribution of
  \eqn{T_{\nu}(\delta) := \frac{U + \delta}{\chi_{\nu}/\sqrt{\nu}}}{%
      T(df,Del) := (U + Del) / (Chi(df) / sqrt(df)) }
  where \eqn{U} and \eqn{\chi_{\nu}}{Chi(df)}  are independent random
  variables, \eqn{U \sim {\cal N}(0,1)}{U \~ N(0,1)}, and
%%fails  \eqn{{\chi_{\nu}}^2}{(Chi(df))^2}
  \eqn{\chi^2_\nu}{Chi(df)^2}
  is chi-squared, see \code{\link{pchisq}}.

  The most used applications are power calculations for \eqn{t}-tests:\cr
  Let \eqn{T= \frac{\bar{X} - \mu_0}{S/\sqrt{n}}}{T= (mX - m0) / (S/sqrt(n))}
  where
  \eqn{\bar{X}}{mX} is the \code{\link{mean}} and \eqn{S} the sample standard
  deviation (\code{\link{sd}}) of \eqn{X_1,X_2,\dots,X_n} which are i.i.d.
%%fails \eqn{{\cal N}(\mu,\sigma^2)}{N(mu,sigma^2)}
        \eqn{       N(\mu,\sigma^2)}{N(mu,sigma^2)}.
  Then \eqn{T} is distributed as non-centrally \eqn{t} with
  \code{df}\eqn{= n-1}
  degrees of freedom and \bold{n}on-\bold{c}entrality \bold{p}arameter
  \code{ncp}\eqn{= (\mu - \mu_0) \sqrt{n}/\sigma}{= (mu - m0) * sqrt(n)/sigma}.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole. (except non-central versions.)

  Lenth, R. V. (1989). \emph{Algorithm AS 243} ---
    Cumulative distribution function of the non-central \eqn{t} distribution,
    \emph{Appl.\ Statist.} \bold{38}, 185--189.
}
\seealso{\code{\link{df}} for the F distribution.}
\examples{
1 - pt(1:5, df = 1)
qt(.975, df = c(1:10,20,50,100,1000))

tt <- seq(0,10, len=21)
ncp <- seq(0,6, len=31)
ptn <- outer(tt,ncp, function(t,d) pt(t, df = 3, ncp=d))
image(tt,ncp,ptn, zlim=c(0,1),main=t.tit <- "Non-central t - Probabilities")
persp(tt,ncp,ptn, zlim=0:1, r=2, phi=20, theta=200, main=t.tit,
      xlab = "t", ylab = "noncentrality parameter", zlab = "Pr(T <= t)")

op <- par(yaxs="i")
plot(function(x) dt(x, df = 3, ncp = 2), -3, 11, ylim = c(0, 0.32),
     main="Non-central t - Density")
par(op)
}
\keyword{distribution}
\eof
\name{Theoph}
\docType{data}
\alias{Theoph}
\title{Pharmacokinetics of theophylline}
\description{
  The \code{Theoph} data frame has 132 rows and 5 columns of data from
  an experiment on the pharmacokinetics of theophylline.}
\usage{data(Theoph)}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Subject}{
      an ordered factor with levels \code{1}, \dots, \code{12}
      identifying the subject on whom the observation was made.  The
      ordering is by increasing maximum concentration of theophylline
      observed.
    }
    \item{Wt}{
      weight of the subject (kg).
    }
    \item{Dose}{
      dose of theophylline administered orally to the subject (mg/kg).
    }
    \item{Time}{
      time since drug administration when the sample was drawn (hr).
    }
    \item{conc}{
      theophylline concentration in the sample (mg/L).
    }
  }
}
\details{
  Boeckmann, Sheiner and Beal (1994) report data from a study by
  Dr. Robert Upton of the kinetics of the anti-asthmatic drug
  theophylline.  Twelve subjects were given oral doses of theophylline
  then serum concentrations were measured at 11 time points over the
  next 25 hours.

  These data are analyzed in Davidian and Giltinan (1995) and Pinheiro
  and Bates (2000) using a two-compartment open pharmacokinetic model,
  for which a self-starting model function, \code{SSfol}, is available.

}
\source{
  Boeckmann, A. J., Sheiner, L. B. and Beal, S. L. (1994), \emph{NONMEM
    Users Guide: Part V}, NONMEM Project Group, University of
  California, San Francisco.
    
  Davidian, M. and Giltinan, D. M. (1995) \emph{Nonlinear Models for
    Repeated Measurement Data}, Chapman & Hall (section 5.5, p. 145 and
  section 6.6, p. 176)

  Pinheiro, J. C. and Bates, D. M. (2000) \emph{Mixed-effects Models in
    S and S-PLUS}, Springer (Appendix A.29)
}
\seealso{\code{\link{SSfol}}}
\examples{
data(Theoph)
coplot(conc ~ Time | Subject, data = Theoph, show = FALSE)
Theoph.4 <- subset(Theoph, Subject == 4)
fm1 <- nls(conc ~ SSfol(Dose, Time, lKe, lKa, lCl),
           data = Theoph.4)
summary(fm1)
plot(conc ~ Time, data = Theoph.4,
     xlab = "Time since drug administration (hr)",
     ylab = "Theophylline concentration (mg/L)",
     main = "Observed concentrations and fitted model",
     sub  = "Theophylline data - Subject 4 only",
     las = 1, col = 4)
xvals <- seq(0, par("usr")[2], len = 55)
lines(xvals, predict(fm1, newdata = list(Time = xvals)),
      col = 4)
}
\keyword{datasets}
\eof
\name{Tukey}
\alias{Tukey}
\title{The Studentized Range Distribution}
\usage{
ptukey(q, nmeans, df, nranges = 1, lower.tail = TRUE, log.p = FALSE)
qtukey(p, nmeans, df, nranges = 1, lower.tail = TRUE, log.p = FALSE)
}
\alias{ptukey}
\alias{qtukey}
\arguments{
  \item{q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{nmeans}{sample size for range (same for each group).}
  \item{df}{degrees of freedom for \eqn{s} (see below).}
  \item{nranges}{number of \emph{groups} whose \bold{maximum} range is
    considered.}
  \item{log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\description{
  Functions on the distribution of
  the studentized range, \eqn{R/s}, where \eqn{R} is the range of a
  standard normal sample of size \eqn{n} and \eqn{s^2} is independently
  distributed as chi-squared with \eqn{df} degrees of freedom, see
  \code{\link{pchisq}}.
}
\details{
  If \eqn{n_g =}{ng =}\code{nranges} is greater than one, \eqn{R} is
  the \emph{maximum} of \eqn{n_g}{ng} groups of \code{nmeans}
  observations each.
}
\value{
  \code{ptukey} gives the distribution function and \code{qtukey} its
  inverse, the quantile function.
}
\note{
  A Legendre 16-point formula is used for the integral of \code{ptukey}.
  The computations are relatively expensive, especially for
  \code{qtukey} which uses a simple secant method for finding the
  inverse of \code{ptukey}.
  \code{qtukey} will be accurate to the 4th decimal place.
}
\references{
  Copenhaver, Margaret Diponzio and Holland, Burt S. (1988)
  Multiple comparisons of simple effects in
  the two-way analysis of variance with fixed effects.
  \emph{Journal of Statistical Computation and Simulation}, \bold{30}, 1--15.
}
\seealso{
  \code{\link{pnorm}} and \code{\link{qnorm}} for the corresponding
  functions for the normal distribution.
}
\examples{
if(interactive())
  curve(ptukey(x, nm=6, df=5), from=-1, to=8, n=101)
(ptt <- ptukey(0:10, 2, df= 5))
(qtt <- qtukey(.95, 2, df= 2:11))
## The precision may be not much more than about 8 digits:
summary(abs(.95 - ptukey(qtt,2, df = 2:11)))
}
\keyword{distribution}
\eof
\name{TukeyHSD}
\alias{TukeyHSD}
\alias{TukeyHSD.aov}
\alias{print.TukeyHSD}
\alias{plot.TukeyHSD}
\title{Compute Tukey Honest Significant Differences}
\description{
  Create a set of confidence intervals on the differences between the
  means of the levels of a factor with the specified family-wise
  probability of coverage.  The intervals are based on the Studentized
  range statistic, Tukey's \sQuote{Honest Significant Difference}
  method.  There is a \code{plot} method.
}
\usage{
TukeyHSD(x, which, ordered = FALSE, conf.level = 0.95, \dots)
}
\arguments{
 \item{x}{A fitted model object, usually an \code{\link{aov}} fit.}
 \item{which}{A list of terms in the fitted model for which the
   intervals should be calculated.  Defaults to all the terms.}
 \item{ordered}{A logical value indicating if the levels of the factor
   should be ordered according to increasing average in the sample
   before taking differences.  If \code{ordered} is true then
   the calculated differences in the means will all be positive.  The
   significant differences will be those for which the \code{lwr} end
   point is positive.}
 \item{conf.level}{A numeric value between zero and one giving the
   family-wise confidence level to use.}
 \item{\dots}{Optional additional arguments.  None are used at present.}
}
\details{
  When comparing the means for the levels of a factor in an analysis of
  variance, a simple comparison using t-tests will inflate the
  probability of declaring a significant difference when it is not in
  fact present.  This because the intervals are calculated with a
  given coverage probability for each interval but the interpretation of
  the coverage is usually with respect to the entire family of
  intervals.

  John Tukey introduced intervals based on the range of the
  sample means rather than the individual differences.  The intervals
  returned by this function are based on this Studentized range
  statistics.

  Technically the intervals constructed in this way would only apply to
  balanced designs where there are the same number of observations made
  at each level of the factor.  This function incorporates an adjustment
  for sample size that produces sensible intervals for mildly unbalanced
  designs.
}
\value{
  A list with one component for each term requested in \code{which}.
  Each component is a matrix with columns \code{diff} giving the
  difference in the observed means, \code{lwr} giving the lower
  end point of the interval, and \code{upr} giving the upper end point.
}
\references{
  Miller, R. G. (1981)
  \emph{Simultaneous Statistical Inference}. Springer.

  Yandell, B. S. (1997)
  \emph{Practical Data Analysis for Designed Experiments}.
  Chapman \& Hall.
}
\author{
  Douglas Bates
}
\seealso{
  \code{\link{aov}}, \code{\link{qtukey}}, \code{\link{model.tables}}
}
\examples{
data(warpbreaks)
summary(fm1 <- aov(breaks ~ wool + tension, data = warpbreaks))
TukeyHSD(fm1, "tension", ordered = TRUE)
plot(TukeyHSD(fm1, "tension"))
}
\keyword{models}
\keyword{design}
\eof
\name{UKDriverDeaths}
\docType{data}
\alias{UKDriverDeaths}
\alias{Seatbelts}
\title{
  Road Casualties in Great Britain 1969--84
}
\description{
  \code{UKDriverDeaths} is a time series giving the monthly totals
  of car drivers in
  Great Britain killed or seriously injured Jan 1969 to Dec 1984.
  Compulsory wearing of seat belts was introduced on 31 Jan 1983.

  \code{Seatbelts} is more information on the same problem.
}
\usage{
data(UKDriverDeaths)
data(Seatbelts)
}
\format{
  \code{Seatbelts} is a multiple time series, with columns
  \describe{
    \item{\code{DriversKilled}}{car drivers killed.}
    \item{\code{drivers}}{same as \code{UKDriverDeaths}.}
    \item{\code{front}}{front-seat passengers killed or seriously injured.}
    \item{\code{rear}}{rear-seat passengers killed or seriously injured.}
    \item{\code{kms}}{distance driven.}
    \item{\code{PetrolPrice}}{petrol price.}
    \item{\code{VanKilled}}{number of van (\sQuote{light goods vehicle})
      drivers.}
    \item{\code{law}}{0/1: was the law in effect that month?}
  }
}
\source{
  Harvey, A.C. (1989)
  \emph{Forecasting, Structural Time Series Models and the Kalman Filter.}
  Cambridge University Press, pp. 519--523.

  Durbin, J. and Koopman, S. J. (2001) \emph{Time Series Analysis by
    State Space Methods.}  Oxford University Press.
  \url{http://www.ssfpack.com/dkbook/}
}
\references{
  Harvey, A. C. and Durbin, J. (1986) The effects of seat belt
  legislation on British road casualties: A case study in structural
  time series modelling. \emph{Journal of the Royal Statistical Society}
  series B, \bold{149}, 187--227.
}
\examples{
data(UKDriverDeaths)
## work with pre-seatbelt period to identify a model, use logs
work <- window(log10(UKDriverDeaths), end = 1982+11/12)
par(mfrow = c(3,1))
plot(work); acf(work); pacf(work)
par(mfrow = c(1,1))
(fit <- arima(work, c(1,0,0), seasonal = list(order= c(1,0,0))))
z <- predict(fit, n.ahead = 24)
ts.plot(log10(UKDriverDeaths), z$pred, z$pred+2*z$se, z$pred-2*z$se,
        lty = c(1,3,2,2), col = c("black", "red", "blue", "blue"))

## now see the effect of the explanatory variables
data(Seatbelts)
X <- Seatbelts[, c("kms", "PetrolPrice", "law")]
X[, 1] <- log10(X[, 1]) - 4
arima(log10(Seatbelts[, "drivers"]), c(1,0,0),
      seasonal = list(order= c(1,0,0)), xreg = X)
}
\keyword{datasets}
\eof
% file UKLungDeaths.Rd
% copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{UKLungDeaths}
\docType{data}
\alias{UKLungDeaths}
\alias{ldeaths}
\alias{fdeaths}
\alias{mdeaths}
\title{
Monthly Deaths from Lung Diseases in the UK
}
\usage{
data(UKLungDeaths)
}
\description{
    Three time series giving the monthly deaths from bronchitis,
    emphysema and asthma in the UK, 1974--1979,
    both sexes (\code{ldeaths}), males (\code{mdeaths}) and
    females (\code{fdeaths}).
}
\source{
P. J. Diggle (1990)
\emph{Time Series: A Biostatistical Introduction.}
Oxford, table A.3
}
\examples{
data(UKLungDeaths)
plot(ldeaths) 
plot(mdeaths, fdeaths) 
## Better labels:
yr <- floor(tt <- time(mdeaths))
plot(mdeaths, fdeaths,
     xy.labels = paste(month.abb[12*(tt - yr)], yr-1900, sep="'"))
}
\keyword{datasets}
\eof
\name{UKgas}
\docType{data}
\alias{UKgas}
\title{UK Quarterly Gas Consumption}
\usage{data(UKgas)}
\description{
  Quarterly UK gas consumption from 1960Q1 to 1986Q4, in millions of therms.
}
\format{
  A quarterly time series of length 108.
}
\source{
  Durbin, J. and Koopman, S. J. (2001) \emph{Time Series Analysis by
    State Space Methods.}  Oxford University Press.
  \url{http://www.ssfpack.com/dkbook/}
}
\examples{
data(UKgas)
## maybe str(UKgas) ; plot(UKgas) ...
}
\keyword{datasets}
\eof
% file USAccDeaths.Rd
% copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{USAccDeaths}
\docType{data}
\alias{USAccDeaths}
\title{
  Accidental Deaths in the US 1973--1978
}
\description{
  A time series giving the monthly totals of accidental deaths in the
  USA.  The values for the first six months of 1979 are 7798 7406 8363
  8460 9217 9316.
}
\usage{data(USAccDeaths)}
\source{
  P. J. Brockwell and R. A. Davis (1991)
  \emph{Time Series: Theory and Methods.}
  Springer, New York.
}
\keyword{datasets}
\eof
\name{Uniform}
\alias{Uniform}
\title{The Uniform Distribution}
\usage{
dunif(x, min=0, max=1, log = FALSE)
punif(q, min=0, max=1, lower.tail = TRUE, log.p = FALSE)
qunif(p, min=0, max=1, lower.tail = TRUE, log.p = FALSE)
runif(n, min=0, max=1)
}
\alias{dunif}
\alias{punif}
\alias{qunif}
\alias{runif}
\arguments{
  \item{x,q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{min,max}{lower and upper limits of the distribution.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\description{
  These functions provide information about the uniform distribution
  on the interval from \code{min} to \code{max}.  \code{dunif} gives the
  density, \code{punif} gives the distribution function \code{qunif}
  gives the quantile function and \code{runif} generates random
  deviates.
}
\details{
  If \code{min} or \code{max} are not specified they assume the default
  values of \code{0} and \code{1} respectively.

  The uniform distribution has density
  \deqn{f(x) = \frac{1}{max-min}}{f(x) = 1/(max-min)}
  for \eqn{min \le x \le max}.

  For the case of \eqn{u := min == max}, the limit case of
  \eqn{X \equiv u}{X == u} is assumed.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{.Random.seed}} about random number generation,
  \code{\link{rnorm}}, etc for other distributions.
}
\examples{
u <- runif(20)

## The following relations always hold :
punif(u) == u
dunif(u) == 1

var(runif(10000))#- ~ = 1/12 = .08333
}
\keyword{distribution}
\eof
\name{WWWusage}
\docType{data}
\alias{WWWusage}
\title{Internet Usage per Minute}
\usage{data(WWWusage)}
\description{
  A time series of the numbers of users connected to the Internet
  through a server every minute.
}
\format{
  A time series of length 100.
}
\source{
  Durbin, J. and Koopman, S. J. (2001) \emph{Time Series Analysis by
    State Space Methods.}  Oxford University Press.
  \url{http://www.ssfpack.com/dkbook/} 
}
\references{
  Makridakis, S., Wheelwright, S. C. and Hyndman, R. J. (1998)
  \emph{Forecasting: Methods and Applications.} Wiley.
}
\examples{
data(WWWusage)
work <- diff(WWWusage)
par(mfrow = c(2,1)); plot(WWWusage); plot(work)
\dontrun{
aics <- matrix(, 6, 6, dimnames=list(p=0:5, q=0:5))
for(q in 1:5) aics[1, 1+q] <- arima(WWWusage, c(0,1,q),
    optim.control = list(maxit = 500))$aic
for(p in 1:5)
   for(q in 0:5) aics[1+p, 1+q] <- arima(WWWusage, c(p,1,q),
       optim.control = list(maxit = 500))$aic
round(aics - min(aics, na.rm=TRUE), 2)
}}
\keyword{datasets}
\eof
\name{Weibull}
\alias{Weibull}
\alias{dweibull}
\alias{pweibull}
\alias{qweibull}
\alias{rweibull}
\title{The Weibull Distribution}
\description{
  Density, distribution function, quantile function and random
  generation for the Weibull distribution with parameters \code{shape}
  and \code{scale}.
}
\usage{
dweibull(x, shape, scale = 1, log = FALSE)
pweibull(q, shape, scale = 1, lower.tail = TRUE, log.p = FALSE)
qweibull(p, shape, scale = 1, lower.tail = TRUE, log.p = FALSE)
rweibull(n, shape, scale = 1)
}
\arguments{
  \item{x, q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{shape, scale}{shape and scale parameters, the latter defaulting to 1.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dweibull} gives the density,
  \code{pweibull} gives the distribution function,
  \code{qweibull} gives the quantile function, and
  \code{rweibull} generates random deviates.
}
\details{
  The Weibull distribution with \code{shape} parameter \eqn{a} and
  \code{scale} parameter \eqn{\sigma}{b} has density given by
  \deqn{f(x) = (a/\sigma) {(x/\sigma)}^{a-1} \exp (-{(x/\sigma)}^{a})}{%
        f(x) = (a/b) (x/b)^(a-1) exp(- (x/b)^a)}
  for \eqn{x > 0}.
  The cumulative is
  \eqn{F(x) = 1 - \exp(-{(x/\sigma)}^a)}{F(x) = 1 - exp(- (x/b)^a)}, the
  mean is \eqn{E(X) = \sigma \Gamma(1 + 1/a)}{E(X) = b Gamma(1 + 1/a)}, and
  the \eqn{Var(X) = \sigma^2(\Gamma(1 + 2/a)-(\Gamma(1 + 1/a))^2)}{%
           Var(X) = b^2 * (gamma(1 + 2/a) - (gamma(1 + 1/a))^2)}.
}
\note{
  The cumulative hazard \eqn{H(t) = - \log(1 - F(t))}{H(t) = - log(1 - F(t))}
  is \code{-pweibull(t, a, b, lower = FALSE, log = TRUE)} which is just
  \eqn{H(t) = {(t/b)}^a}.
}
\seealso{
  \code{\link{dexp}} for the Exponential which is a special case of a
  Weibull distribution.
}
\examples{
x <- c(0,rlnorm(50))
all.equal(dweibull(x, shape = 1), dexp(x))
all.equal(pweibull(x, shape = 1, scale = pi), pexp(x, rate = 1/pi))
## Cumulative hazard H():
all.equal(pweibull(x, 2.5, pi, lower=FALSE, log=TRUE), -(x/pi)^2.5, tol=1e-15)
all.equal(qweibull(x/11, shape = 1, scale = pi), qexp(x/11, rate = 1/pi))
}
\keyword{distribution}
\eof
\name{Wilcoxon}
\alias{Wilcoxon}
\alias{dwilcox}
\alias{pwilcox}
\alias{qwilcox}
\alias{rwilcox}
\title{Distribution of the Wilcoxon Rank Sum Statistic}
\description{
  Density, distribution function, quantile function and random
  generation for the distribution of the Wilcoxon rank sum statistic
  obtained from samples with size \code{m} and \code{n}, respectively.
}
\usage{
dwilcox(x, m, n, log = FALSE)
pwilcox(q, m, n, lower.tail = TRUE, log.p = FALSE)
qwilcox(p, m, n, lower.tail = TRUE, log.p = FALSE)
rwilcox(nn, m, n)
}
\arguments{
  \item{x, q}{vector of quantiles.}
  \item{p}{vector of probabilities.}
  \item{nn}{number of observations. If \code{length(nn) > 1}, the length
    is taken to be the number required.}
  \item{m, n}{numbers of observations in the first and second sample,
    respectively.  Can be vectors of positive integers.}
  \item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}
  \item{lower.tail}{logical; if TRUE (default), probabilities are
    \eqn{P[X \le x]}{P[X <= x]}, otherwise, \eqn{P[X > x]}{P[X > x]}.}
}
\value{
  \code{dwilcox} gives the density,
  \code{pwilcox} gives the distribution function,
  \code{qwilcox} gives the quantile function, and
  \code{rwilcox} generates random deviates.
}
\details{
  This distribution is obtained as follows.  Let \code{x} and \code{y}
  be two random, independent samples of size \code{m} and \code{n}.
  Then the Wilcoxon rank sum statistic is the number of all pairs
  \code{(x[i], y[j])} for which \code{y[j]} is not greater than
  \code{x[i]}.  This statistic takes values between \code{0} and
  \code{m * n}, and its mean and variance are \code{m * n / 2} and
  \code{m * n * (m + n + 1) / 12}, respectively.

  If any of the first three arguments are vectors, the recycling rule is
  used to do the calculations for all combinations of the three up to
  the length of the longest vector.
}
\note{
  S-PLUS uses a different (but equivalent) definition of the Wilcoxon
  statistic.
}
\author{Kurt Hornik \email{hornik@ci.tuwien.ac.at}}
\seealso{
  \code{\link{wilcox.test}} to calculate the statistic from data, find p
  values and so on.
  
  \code{\link{dsignrank}} etc, for the distribution of the
  \emph{one-sample} Wilcoxon signed rank statistic.
}
\examples{
x <- -1:(4*6 + 1)
fx <- dwilcox(x, 4, 6)
Fx <- pwilcox(x, 4, 6)

layout(rbind(1,2),width=1,heights=c(3,2))
plot(x, fx,type='h', col="violet",
     main= "Probabilities (density) of Wilcoxon-Statist.(n=6,m=4)")
plot(x, Fx,type="s", col="blue",
     main= "Distribution of Wilcoxon-Statist.(n=6,m=4)")
abline(h=0:1, col="gray20",lty=2)
layout(1)# set back

N <- 200
hist(U <- rwilcox(N, m=4,n=6), breaks=0:25 - 1/2, border="red", col="pink",
     sub = paste("N =",N))
mtext("N * f(x),  f() = true \"density\"", side=3, col="blue")
 lines(x, N*fx, type='h', col='blue', lwd=2)
points(x, N*fx, cex=2)

## Better is a Quantile-Quantile Plot
qqplot(U, qw <- qwilcox((1:N - 1/2)/N, m=4,n=6),
       main = paste("Q-Q-Plot of empirical and theoretical quantiles",
                     "Wilcoxon Statistic,  (m=4, n=6)",sep="\n"))
n <- as.numeric(names(print(tU <- table(U))))
text(n+.2, n+.5, labels=tU, col="red")
}
\keyword{distribution}
\eof
\name{ability.cov}
\docType{data}
\alias{ability.cov}
\title{Ability and Intelligence Tests}
\description{
  Six tests were given to 112 individuals. The covariance matrix is
  given in this object.
}
\details{
  The tests are described as
  \describe{
    \item{general:}{a non-verbal measure of general intelligence using
      Cattell's culture-fair test.}
    \item{picture:}{a picture-completion test}
    \item{blocks:}{block design}
    \item{maze:}{mazes}
    \item{reading:}{reading comprehension}
    \item{vocab:}{vocabulary}
  }
  
  Bartholomew gives both covariance and correlation matrices,
  but these are inconsistent.  Neither are in the original paper.
}
\usage{data(ability.cov)}
\source{
  Barthlomew, D. J. (1987) \emph{Latent Variable Analysis and Factor
    Analysis.} Griffin.  

  Barthlomew, D. J.  and Knott, M. (1990) \emph{Latent Variable Analysis
    and Factor Analysis.} Second Edition, Arnold.  
}
\references{
  Smith, G. A. and Stanley G. (1983)
  Clocking \eqn{g}: relating intelligence and measures of timed
  performance. \emph{Intelligence}, \bold{7}, 353--368.
}
\examples{
data(ability.cov)
(ability.FA <- factanal(factors = 1, covmat=ability.cov))
update(ability.FA, factors=2)
update(ability.FA, factors=2, rotation="promax")
}
\keyword{datasets}
\eof
\name{acf}
\alias{acf}
\alias{ccf}
\alias{pacf}
\alias{pacf.default}
\alias{[.acf}
\title{Auto- and Cross- Covariance and -Correlation Function Estimation}
\description{
  The function \code{acf} computes (and by default plots) estimates of
  the autocovariance or autocorrelation function.  Function \code{pacf}
  is the function used for the partial autocorrelations.  Function
  \code{ccf} computes the cross-correlation or cross-covariance of two
  univariate series.
}
\usage{
acf(x, lag.max = NULL,
    type = c("correlation", "covariance", "partial"),
    plot = TRUE, na.action = na.fail, demean = TRUE, \dots)

pacf(x, lag.max, plot, na.action, \dots)

\method{pacf}{default}(x, lag.max = NULL, plot = TRUE, na.action = na.fail, 
    ...) 

ccf(x, y, lag.max = NULL, type = c("correlation", "covariance"),
    plot = TRUE, na.action = na.fail, \dots)

acf.obj[i, j]
}
\arguments{
  \item{x, y}{a univariate or multivariate (not \code{ccf}) numeric time
    series object or a numeric vector or matrix.}
  \item{lag.max}{maximum number of lags at which to calculate the acf.
    Default is \eqn{10\log_{10}(N/m)}{10*log10(N/m)} where \eqn{N} is the
    number of observations and \eqn{m} the number of series.}
  \item{type}{character string giving the type of acf to be computed.
    Allowed values are
    \code{"correlation"} (the default), \code{"covariance"} or
    \code{"partial"}.}
  \item{plot}{logical. If \code{TRUE} (the default) the acf is plotted.}
  \item{na.action}{function to be called to handle missing
    values. \code{na.pass} can be used.}
  \item{demean}{logical.  Should the covariances be about the sample
    means?}
  \item{\dots}{further arguments to be passed to \code{plot.acf}.}

  \item{acf.obj}{an object of class \code{"acf"} resulting from a call
    to \code{acf}.}
  \item{i}{a set of lags to retain.}
  \item{j}{a set of series to retain.}
}
\value{
  An object of class \code{"acf"}, which is a list with the following
  elements:

  \item{lag}{A three dimensional array containing the lags at which
    the acf is estimated.}
  \item{acf}{An array with the same dimensions as \code{lag} containing
    the estimated acf.}
  \item{type}{The type of correlation (same as the \code{type}
    argument).}
  \item{n.used}{The number of observations in the time series.}
  \item{series}{The name of the series \code{x}.}
  \item{snames}{The series names for a multivariate time series.}

  The result is returned invisibly if \code{plot} is \code{TRUE}.
}
\details{
  For \code{type} = \code{"correlation"} and \code{"covariance"}, the
  estimates are based on the sample covariance.

  By default, no missing values are allowed.  If the \code{na.action}
  function passes through missing values (as \code{na.pass} does), the
  covariances are computed from the complete cases.  This means that the
  estimate computed may well not be a valid autocorrelation sequence,
  and may contain missing values.  Missing values are not allowed when
  computing the PACF of a multivariate time series.

  The partial correlation coefficient is estimated by fitting
  autoregressive models of successively higher orders up to
  \code{lag.max}.

  The generic function \code{plot} has a method for objects of class
  \code{"acf"}.

  The lag is returned and plotted in units of time, and not numbers of
  observations.

  There are \code{print} and subsetting methods for objects of class
  \code{"acf"}.
}
\author{
  Original: Paul Gilbert, Martyn Plummer.
  Extensive modifications and univariate case of \code{pacf} by
  B.D. Ripley.
}
\seealso{
  \code{\link{plot.acf}}
}
\examples{
## Examples from Venables & Ripley
data(lh)
acf(lh)
acf(lh, type = "covariance")
pacf(lh)

data(UKLungDeaths)
acf(ldeaths)
acf(ldeaths, ci.type = "ma")
acf(ts.union(mdeaths, fdeaths))
ccf(mdeaths, fdeaths) # just the cross-correlations.

data(presidents) # contains missing values
acf(presidents, na.action = na.pass)
pacf(presidents, na.action = na.pass)
}
\keyword{ts}
\eof
\name{acf2AR}
\alias{acf2AR}
\title{Compute an AR Process Exactly Fitting an ACF}
\description{
  Compute an AR process exactly fitting an autocorrelation function.
}
\usage{
acf2AR(acf)
}
\arguments{
  \item{acf}{An autocorrelation or autocovariance sequence.}
}
\value{
  A matrix, with one row for the computed AR(p) coefficients for
  \code{1 <= p <= length(acf)}.
}
\seealso{
  \code{\link{ARMAacf}}, \code{\link{ar.yw}} which does this from an
  empirical ACF.
}
\examples{
(Acf <- ARMAacf(c(0.6, 0.3, -0.2)))
acf2AR(Acf)
}
\keyword{ts}
\eof
\name{add1}
\alias{add1}
\alias{add1.default}
\alias{add1.lm}
\alias{add1.glm}
%\alias{add1.mlm}
\alias{drop1}
\alias{drop1.default}
\alias{drop1.lm}
\alias{drop1.glm}
%\alias{drop1.mlm}
\title{Add or Drop All Possible Single Terms to a Model}
\usage{
add1(object, scope, \dots)

\method{add1}{default}(object, scope, scale = 0, test = c("none", "Chisq"),
     k = 2, trace = FALSE, \dots)

\method{add1}{lm}(object, scope, scale = 0, test = c("none", "Chisq", "F"),
     x = NULL, k = 2, \dots)

\method{add1}{glm}(object, scope, scale = 0, test = c("none", "Chisq", "F"),
     x = NULL, k = 2, \dots)

drop1(object, scope, \dots)

\method{drop1}{default}(object, scope, scale = 0, test = c("none", "Chisq"),
      k = 2, trace = FALSE, \dots)

\method{drop1}{lm}(object, scope, scale = 0, all.cols = TRUE,
      test=c("none", "Chisq", "F"),k = 2, \dots)

\method{drop1}{glm}(object, scope, scale = 0, test = c("none", "Chisq", "F"),
      k = 2, \dots)
}
\arguments{
  \item{object}{a fitted model object.}
  \item{scope}{a formula giving the terms to be considered for adding or
    dropping.}
  \item{scale}{an estimate of the residual mean square to be
    used in computing \eqn{C_p}{Cp}. Ignored if \code{0} or \code{NULL}.}
  \item{test}{should the results include a test statistic relative to the
    original model?  The F test is only appropriate for \code{\link{lm}} and
    \code{\link{aov}} models or perhaps for \code{\link{glm}} fits with
    estimated dispersion.
    The \eqn{\chi^2}{Chisq} test can be an exact test
    (\code{lm} models with known scale) or a likelihood-ratio test or a
    test of the reduction in scaled deviance depending on the method.}
  \item{k}{the penalty constant in AIC / \eqn{C_p}{Cp}.}
  \item{trace}{if \code{TRUE}, print out progress reports.}
  \item{x}{a model matrix containing columns for the fitted model and all
    terms in the upper scope.  Useful if \code{add1} is to be called
    repeatedly.}
  \item{all.cols}{(Provided for compatibility with S.) Logical to specify
    whether all columns of the design matrix should be used. If
    \code{FALSE} then non-estimable columns are dropped, but the result
    is not usually statistically meaningful.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  Compute all the single terms in the \code{scope} argument that can be
  added to or dropped from the model, fit those models and compute a
  table of the changes in fit.
}
\details{
  For \code{drop1} methods, a missing \code{scope} is taken to be all
  terms in the model. The hierarchy is respected when considering terms
  to be added or dropped: all main effects contained in a second-order
  interaction must remain, and so on.

  The methods for \code{\link{lm}} and \code{\link{glm}} are more
  efficient in that they do not recompute the model matrix and call the
  \code{fit} methods directly.

  The default output table gives AIC, defined as minus twice log
  likelihood plus \eqn{2p} where \eqn{p} is the rank of the model (the
  number of effective parameters).  This is only defined up to an
  additive constant (like log-likelihoods).  For linear Gaussian models
  with fixed scale, the constant is chosen to give Mallows' \eqn{C_p}{Cp},
  \eqn{RSS/scale + 2p - n}.  Where \eqn{C_p}{Cp} is used,
  the column is labelled as \code{Cp} rather than \code{AIC}.
}
\value{
  An object of class \code{"anova"} summarizing the differences in fit
  between the models.
}
\author{
  The design was inspired by the S functions of the same names described
  in Chambers (1992).
}
\references{
  Chambers, J. M. (1992)
  \emph{Linear models.}
  Chapter 4 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\note{
  These are not fully equivalent to the functions in S.  There is no
  \code{keep} argument, and the methods used are not quite so
  computationally efficient.
    
  Their authors' definitions of Mallows' \eqn{C_p}{Cp} and Akaike's AIC
  are used, not those of the authors of the models chapter of S.
}
\section{Warning}{
  The model fitting must apply the models to the same dataset. Most
  methods will attempt to use a subset of the data with no missing
  values for any of the variables if \code{na.action=na.omit}, but
  this may give biased results.  Only use
  these functions with data containing missing values with great care.
}
\seealso{
  \code{\link{step}}, \code{\link{aov}}, \code{\link{lm}},
  \code{\link{extractAIC}}, \code{\link{anova}}
}
\examples{
example(step)#-> swiss
add1(lm1, ~ I(Education^2) + .^2)
drop1(lm1, test="F")  # So called 'type II' anova

example(glm)
drop1(glm.D93, test="Chisq")
drop1(glm.D93, test="F")
}
\keyword{models}
\eof
\name{addmargins}

\alias{addmargins}
\concept{totals}
\concept{margins}

\title{
  Puts arbitrary margins on multidimensional tables or arrays.
} 

\description{
  For a given table one can specify which of the classifying factors to
  expand by one or more levels to hold margins to be calculated. One may for
  example form sums and means over the first dimension and medians over the
  second. The resulting table will then have two extra levels for the first
  dimension and one extra level for the second. The default is to sum over
  all margins in the table. Other possibilities may give results that
  depend on the order in which the margins are computed. This is flagged
  in the printed output from the function. 
}

\usage{
  addmargins( A, margin = 1:length(dim(A)), FUN = sum, quiet=FALSE )
}

\arguments{
  \item{A}{A table or array. The function uses the presence of the
    \code{"dim"} and \code{"dimnames"} attributes of \code{A}}
  \item{margin}{Vector of dimensions over which to form margins. Margins
    are formed in the order in which dimensions are specified in
    \code{margin}.} 
  \item{FUN}{List of the same length as \code{margin}, each element of
    the list being either a function or a list of functions. Names of
    the list elements will appear as levels in dimnames of the result.
    Unnamed list elements will have names constructed:  the name
    of a function or a constructed name based on the position in the table.}
  \item{quiet}{Logical which suppresses the message telling the order in
    which the margins were computed.}
}

\details{
  If the functions used to form margins are not commutative the result
  depends on the order in which margins are computed. Annotation
  of margins is done via naming the \code{FUN} list.
}

\value{
  A table with the same number of dimensions as A, but with extra levels
  of the dimensions mentioned in \code{margin}. The number of levels
  added to each dimension is the length of the entries in \code{FUN}.
  A message with the order of computation of margins is printed.
}

\author{
  Bendix Carstensen, Steno Diabetes Center \& Department of
  Biostatistics, University of Copenhagen,
  \url{http://www.biostat.ku.dk/~bxc}, autumn 2003.  
  Margin naming enhanced by Duncan Murdoch.
}

\seealso{\code{\link{table}}, \code{\link{ftable}}, \code{\link{margin.table}}.}

\examples{
Aye <- sample( c("Yes","Si","Oui"), 177, replace=TRUE )
Bee <- sample( c("Hum","Buzz"), 177, replace=TRUE )
Sea <- sample( c("White","Black","Red","Dead"), 177, replace=TRUE )
A <- table( Aye, Bee, Sea )
A
addmargins( A )
ftable( A )
ftable( addmargins( A ) )

# Non commutative functions - note differences between resulting tables:
ftable( addmargins( A, c(1,3), FUN=list( Sum=sum, list( Min=min, Max=max ) ) ) )
ftable( addmargins( A, c(3,1), FUN=list( list( Min=min, Max=max ), Sum=sum ) ) )

# Weird function needed to return the N when computing percentages
sqsm <- function( x ) sum( x )^2/100 
B <- table( Sea, Bee )
round( sweep( addmargins( B, 1, list( list( All=sum, N=sqsm ) ) ), 2, 
                    apply( B, 2, sum )/100, "/" ), 1 )
round( sweep( addmargins( B, 2, list( list( All=sum, N=sqsm ) ) ), 1, 
                    apply( B, 1, sum )/100, "/" ), 1 )

# A total over Bee requires formation of the Bee-margin first:
mB <-  addmargins( B, 2, FUN=list(list(Total=sum)) )
round( ftable( sweep( addmargins( mB, 1, list( list( All=sum, N=sqsm ) ) ), 2, 
                apply( mB, 2, sum )/100, "/" ) ), 1 )
}

\keyword{ manip }
\keyword{ methods }
\keyword{ array }
\eof
\name{aggregate}
\alias{aggregate}
\alias{aggregate.default}
\alias{aggregate.data.frame}
\alias{aggregate.ts}
\title{Compute Summary Statistics of Data Subsets}
\usage{
aggregate(x, \dots)

\method{aggregate}{default}(x, \dots)

\method{aggregate}{data.frame}(x, by, FUN, \dots)

\method{aggregate}{ts}(x, nfrequency = 1, FUN = sum, ndeltat = 1,
          ts.eps = getOption("ts.eps"), \dots)
}
\description{
  Splits the data into subsets, computes summary statistics for each,
  and returns the result in a convenient form.
}
\arguments{
  \item{x}{an R object.}
  \item{by}{a list of grouping elements, each as long as the variables
    in \code{x}.  Names for the grouping variables are provided if
    they are not given.  The elements of the list will be coerced to
    factors (if they are not already factors).}
  \item{FUN}{a scalar function to compute the summary statistics which
    can be applied to all data subsets.}
  \item{nfrequency}{new number of observations per unit of time; must
    be a divisor of the frequency of \code{x}.}
  \item{ndeltat}{new fraction of the sampling period between
    successive observations; must be a divisor of the sampling
    interval of \code{x}.}
  \item{ts.eps}{tolerance used to decide if \code{nfrequency} is a
    sub-multiple of the original frequency.}
  \item{\dots}{further arguments passed to or used by methods.}
}
\details{
  \code{aggregate} is a generic function with methods for data frames
  and time series.
  
  The default method \code{aggregate.default} uses the time series
  method if \code{x} is a time series, and otherwise coerces \code{x}
  to a data frame and calls the data frame method.

  \code{aggregate.data.frame} is the data frame method.  If \code{x}
  is not a data frame, it is coerced to one.  Then, each of the
  variables (columns) in \code{x} is split into subsets of cases
  (rows) of identical combinations of the components of \code{by}, and
  \code{FUN} is applied to each such subset with further arguments in
  \code{\dots} passed to it.
  (I.e., \code{tapply(VAR, by, FUN, \dots, simplify = FALSE)} is done
  for each variable \code{VAR} in \code{x}, conveniently wrapped into
  one call to \code{lapply()}.)
  Empty subsets are removed, and the result is reformatted into a data
  frame containing the variables in \code{by} and \code{x}.  The ones
  arising from \code{by} contain the unique combinations of grouping
  values used for determining the subsets, and the ones arising from
  \code{x} the corresponding summary statistics for the subset of the
  respective variables in \code{x}.
  
  \code{aggregate.ts} is the time series method.  If \code{x} is not a
  time series, it is coerced to one.  Then, the variables in \code{x}
  are split into appropriate blocks of length
  \code{frequency(x) / nfrequency}, and \code{FUN} is applied to each
  such block, with further (named) arguments in \code{\dots} passed to
  it.  The result returned is a time series with frequency
  \code{nfrequency} holding the aggregated values.
}
\author{Kurt Hornik}
\seealso{
  \code{\link{apply}}, \code{\link{lapply}}, \code{\link{tapply}}.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\examples{
data(state)

## Compute the averages for the variables in 'state.x77', grouped
## according to the region (Northeast, South, North Central, West) that
## each state belongs to.
aggregate(state.x77, list(Region = state.region), mean)

## Compute the averages according to region and the occurrence of more
## than 130 days of frost.
aggregate(state.x77,
          list(Region = state.region,
               Cold = state.x77[,"Frost"] > 130),
          mean)
## (Note that no state in 'South' is THAT cold.)

data(presidents)
## Compute the average annual approval ratings for American presidents.
aggregate(presidents, nf = 1, FUN = mean)
## Give the summer less weight.
aggregate(presidents, nf = 1, FUN = weighted.mean, w = c(1, 1, 0.5, 1))
}
\keyword{category}
\keyword{array}
\eof
\name{airmiles}
\docType{data}
\alias{airmiles}
\title{Passenger Miles on Commercial US Airlines, 1937--1960}
\description{
  The revenue passenger miles flown by commercial airlines in
  the United States for each year from 1937 to 1960.
}
\usage{data(airmiles)}
\format{
  A time series of 24 observations; yearly, 1937--1960.
}
\source{F.A.A. Statistical Handbook of Aviation.}
\references{
  Brown, R. G. (1963)
  \emph{Smoothing, Forecasting and Prediction of Discrete Time Series}.
  Prentice-Hall.
}
\examples{
data(airmiles)
plot(airmiles, main = "airmiles data",
     xlab = "Passenger-miles flown by U.S. commercial airlines", col = 4)
}
\keyword{datasets}
\eof
\name{alias}
\title{Find Aliases (Dependencies) in a Model}
\usage{
alias(object, \dots)

\method{alias}{formula}(object, data, \dots)

\method{alias}{lm}(object, complete = TRUE, partial = FALSE,
      partial.pattern = FALSE, \dots)
}
\alias{alias}
\alias{alias.formula}
\alias{alias.lm}
%\alias{print.mtable}
\arguments{
  \item{object}{A fitted model object, for example from \code{lm} or
    \code{aov}, or a formula for \code{alias.formula}.}
  \item{data}{Optionally, a data frame to search for the objects
    in the formula.}
  \item{complete}{Should information on complete aliasing be included?}
  \item{partial}{Should information on partial aliasing be included?}
  \item{partial.pattern}{Should partial aliasing be presented in a
    schematic way? If this is done, the results are presented in a
    more compact way, usually giving the deciles of the coefficients.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  Find aliases (linearly dependent terms) in a linear model specified by
  a formula.
}
\details{
  Although the main method is for class \code{"lm"}, \code{alias} is
  most useful for experimental designs and so is used with fits from
  \code{aov}.
  Complete aliasing refers to effects in linear models that cannot be estimated
  independently of the terms which occur earlier in the model and so
  have their coefficients omitted from the fit. Partial aliasing refers
  to effects that can be estimated less precisely because of
  correlations induced by the design.
}
\value{
  A list (of \code{\link[base]{class} "listof"}) containing components
  \item{Model}{Description of the model; usually the formula.}
  \item{Complete}{A matrix with columns corresponding to effects that
    are linearly dependent on the rows; may be of class \code{"mtable"}
    which has its own \code{\link{print}} method.}
  \item{Partial}{The correlations of the estimable effects, with a zero
    diagonal.}
}
\note{The aliasing pattern may depend on the contrasts in use: Helmert
  contrasts are probably most useful.

  The defaults are different from those in S.
}
\references{
  Chambers, J. M., Freeny, A and Heiberger, R. M. (1992)
  \emph{Analysis of variance; designed experiments.}
  Chapter 5 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\author{
  The design was inspired by the S function of the same name described
  in Chambers \emph{et al.} (1992).
}
\examples{
had.VR <- "package:MASS" \%in\% search()
## The next line is for fractions() which gives neater results
if(!had.VR) res <- require(MASS)
## From Venables and Ripley (2002) p.165.
N <- c(0,1,0,1,1,1,0,0,0,1,1,0,1,1,0,0,1,0,1,0,1,1,0,0)
P <- c(1,1,0,0,0,1,0,1,1,1,0,0,0,1,0,1,1,0,0,1,0,1,1,0)
K <- c(1,0,0,1,0,1,1,0,0,1,0,1,0,1,1,0,0,0,1,1,1,0,1,0)
yield <- c(49.5,62.8,46.8,57.0,59.8,58.5,55.5,56.0,62.8,55.8,69.5,55.0,
           62.0,48.8,45.5,44.2,52.0,51.5,49.8,48.8,57.2,59.0,53.2,56.0)
npk <- data.frame(block=gl(6,4), N=factor(N), P=factor(P),
                  K=factor(K), yield=yield)

op <- options(contrasts=c("contr.helmert", "contr.poly"))
npk.aov <- aov(yield ~ block + N*P*K, npk)
alias(npk.aov)
if(!had.VR && res) detach(package:MASS)
options(op)# reset
}
\keyword{models}
\eof
\name{anova}
\alias{anova}
\alias{print.anova}
\concept{regression}
\title{Anova Tables}
\usage{
anova(object, \dots)
}
\description{
  Compute analysis of variance (or deviance) tables for one or more
  fitted model objects.
}
\arguments{
  \item{object}{an object containing the results returned by a model
    fitting function (e.g., \code{lm} or \code{glm}).} 
  \item{\dots}{additional objects of the same type.}
}
\value{
  This (generic) function returns an object of class \code{anova}.
  These objects represent analysis-of-variance and analysis-of-deviance tables.
  When given a single argument it produces a table which
  tests whether the model terms are significant.

  When given a sequence of objects, \code{anova} tests
  the models against one another in the order specified.

  The print method for \code{anova} objects prints
  tables in a \dQuote{pretty} form.
}
\section{Warning}{
  The comparison between two or more models will only be valid if they
  are fitted to the same dataset. This may be a problem if there are
  missing values and \R's default of \code{na.action = na.omit} is used.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical Models in S}, Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{coefficients}}, \code{\link{effects}},
  \code{\link{fitted.values}}, \code{\link{residuals}},
  \code{\link{summary}}, \code{\link{drop1}}, \code{\link{add1}}.
}
\keyword{regression}
\keyword{models}
\eof
\name{anova.glm}
\title{Analysis of Deviance for Generalized Linear Model Fits}
\usage{
\method{anova}{glm}(object, \dots, dispersion = NULL, test = NULL)
}
\alias{anova.glm}
\alias{anova.glmlist}
%\alias{anova.glm.null}
\description{
  Compute an analysis of deviance table for one or more generalized
  linear model fits.
}
\arguments{
  \item{object, \dots}{objects of class \code{glm}, typically
    the result of a call to \code{\link{glm}}, or a list of
    \code{objects} for the \code{"glmlist"} method.}
  \item{dispersion}{the dispersion parameter for the fitting family.
    By default it is obtained from \code{glm.obj}.}
  \item{test}{a character string, (partially) matching one of \code{"Chisq"},
    \code{"F"} or \code{"Cp"}. See \code{\link{stat.anova}}.}
}
\details{
  Specifying a single object gives a sequential analysis of deviance
  table for that fit.  That is, the reductions in the residual deviance
  as each term of the formula is added in turn are given in as
  the rows of a table, plus the residual deviances themselves.

  If more than one object is specified, the table has a row for the
  residual degrees of freedom and deviance for each model.
  For all but the first model, the change in degrees of freedom and
  deviance is also given. (This only makes statistical sense if the
  models are nested.)  It is conventional to list the models from
  smallest to largest, but this is up to the user.

  The table will optionally contain test statistics (and P values)
  comparing the reduction in deviance for the row to the residuals.
  For models with known dispersion (e.g., binomial and Poisson fits)
  the chi-squared test is most appropriate, and for those with
  dispersion estimated by moments (e.g., \code{gaussian},
  \code{quasibinomial} and \code{quasipoisson} fits) the F test is
  most appropriate.  Mallows' \eqn{C_p}{Cp} statistic is the residual
  deviance plus twice the estimate of \eqn{\sigma^2}{sigma^2} times
  the residual degrees of freedom, which is closely related to AIC (and
  a multiple of it if the dispersion is known).
}
\value{
  An object of class \code{"anova"} inheriting from class \code{"data.frame"}.
}
\section{Warning}{
  The comparison between two or more models by \code{anova} or
  \code{anova.glmlist} will only be valid if they
  are fitted to the same dataset. This may be a problem if there are
  missing values and \R's default of \code{na.action = na.omit} is used,
  and \code{anova.glmlist} will detect this with an error.
}
\references{
  Hastie, T. J. and Pregibon, D. (1992)
  \emph{Generalized linear models.}
  Chapter 6 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{glm}}, \code{\link{anova}}.

  \code{\link{drop1}} for
  so-called \sQuote{type II} anova where each term is dropped one at a
  time respecting their hierarchy.
}
\examples{
## --- Continuing the Example from  '?glm':%\code{\link{glm}}:
\dontshow{example("glm", echo = FALSE)}
anova(glm.D93)
anova(glm.D93, test = "Cp")
anova(glm.D93, test = "Chisq")
}
\keyword{models}
\keyword{regression}

\eof
\name{anova.lm}
\title{ANOVA for Linear Model Fits}
\usage{
\method{anova}{lm}(object, \dots)

anova.lmlist(object, \dots, scale = 0, test = "F")
}
\alias{anova.lm}
%\alias{anova.lm.null}
\alias{anova.lmlist}
\alias{anova.mlm}
\description{
  Compute an analysis of variance table for one or more linear model fits.
}
\arguments{
  \item{object, \dots}{objects of class \code{lm}, usually, a result of a
    call to \code{\link{lm}}.}
  \item{test}{a character string specifying the test statistic to be
    used. Can be one of \code{"F"}, \code{"Chisq"} or \code{"Cp"},
    with partial matching allowed, or \code{NULL} for no test.}
  \item{scale}{numeric. An estimate of the noise variance
    \eqn{\sigma^2}{sigma^2}. If zero this will be estimated from the
    largest model considered.
  }
}
\details{
  Specifying a single object gives a sequential analysis of variance
  table for that fit.  That is, the reductions in the residual sum of
  squares as each term of the formula is added in turn are given in as
  the rows of a table, plus the residual sum of squares.

  The table will contain F statistics (and P values) comparing the
  mean square for the row to the residual mean square.

  If more than one object is specified, the table has a row for the
  residual degrees of freedom and sum of squares for each model.
  For all but the first model, the change in degrees of freedom and sum
  of squares is also given. (This only make statistical sense if the
  models are nested.)  It is conventional to list the models from
  smallest to largest, but this is up to the user.

  Optionally the table can include test statistics.  Normally the
  F statistic is most appropriate, which compares the mean square for a
  row to the residual sum of squares for the largest model considered.
  If \code{scale} is specified chi-squared tests can be used. Mallows'
  \eqn{C_p}{Cp} statistic is the residual sum of squares plus twice the
  estimate of \eqn{\sigma^2}{sigma^2} times the residual degrees of freedom.
}
\value{
  An object of class \code{"anova"} inheriting from class \code{"data.frame"}.
}
\note{
  Versions of \R prior to 1.2.0 based F tests on pairwise comparisons,
  and this behaviour can still be obtained by a direct call to
  \code{anovalist.lm}.
}
\section{Warning}{
  The comparison between two or more models will only be valid if they
  are fitted to the same dataset. This may be a problem if there are
  missing values and \R's default of \code{na.action = na.omit} is used,
  and \code{anova.lmlist} will detect this with an error.
}
\references{
  Chambers, J. M. (1992)
  \emph{Linear models.}
  Chapter 4 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\seealso{
  The model fitting function \code{\link{lm}}, \code{\link{anova}}.

  \code{\link{drop1}} for
  so-called \sQuote{type II} anova where each term is dropped one at a
  time respecting their hierarchy.
}
\examples{
## sequential table
data(LifeCycleSavings)
fit <- lm(sr ~ ., data = LifeCycleSavings)
anova(fit)

## same effect via separate models
fit0 <- lm(sr ~ 1, data = LifeCycleSavings)
fit1 <- update(fit0, . ~ . + pop15)
fit2 <- update(fit1, . ~ . + pop75)
fit3 <- update(fit2, . ~ . + dpi)
fit4 <- update(fit3, . ~ . + ddpi)
anova(fit0, fit1, fit2, fit3, fit4, test="F")

anova(fit4, fit2, fit0, test="F") # unconventional order
}
\keyword{regression}
\keyword{models}
\eof
\name{ansari.test}
\alias{ansari.test}
\alias{ansari.test.default}
\alias{ansari.test.formula}
\title{Ansari-Bradley Test}
\description{
  Performs the Ansari-Bradley two-sample test for a difference in scale
  parameters.
}
\usage{
ansari.test(x, \dots)

\method{ansari.test}{default}(x, y, alternative = c("two.sided", "less", "greater"),
            exact = NULL, conf.int = FALSE, conf.level = 0.95, \dots)

\method{ansari.test}{formula}(formula, data, subset, na.action, \dots)
}
\arguments{
  \item{x}{numeric vector of data values.}
  \item{y}{numeric vector of data values.}
  \item{alternative}{indicates the alternative hypothesis and must be
    one of \code{"two.sided"}, \code{"greater"} or \code{"less"}.  You
    can specify just the initial letter.}
  \item{exact}{a logical indicating whether an exact p-value should be
    computed.}
  \item{conf.int}{a logical,indicating whether a confidence interval
    should be computed.}
  \item{conf.level}{confidence level of the interval.}
  \item{formula}{a formula of the form \code{lhs ~ rhs} where \code{lhs}
    is a numeric variable giving the data values and \code{rhs} a factor
    with two levels giving the corresponding groups.}
  \item{data}{an optional data frame containing the variables in the
    model formula.}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.  Defaults to
    \code{getOption("na.action")}.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\details{
  Suppose that \code{x} and \code{y} are independent samples from
  distributions with densities \eqn{f((t-m)/s)/s} and \eqn{f(t-m)},
  respectively, where \eqn{m} is an unknown nuisance parameter and
  \eqn{s}, the ratio of scales, is the parameter of interest.  The
  Ansari-Bradley test is used for testing the null that \eqn{s} equals
  1, the two-sided alternative being that \eqn{s \ne 1}{s != 1} (the
  distributions differ only in variance), and the one-sided alternatives
  being \eqn{s > 1} (the distribution underlying \code{x} has a larger
  variance, \code{"greater"}) or \eqn{s < 1} (\code{"less"}).

  By default (if \code{exact} is not specified), an exact p-value is
  computed if both samples contain less than 50 finite values and there
  are no ties.  Otherwise, a normal approximation is used.

  Optionally, a nonparametric confidence interval and an estimator for
  \eqn{s} are computed.  If exact p-values are available, an exact
  confidence interval is obtained by the algorithm described in Bauer
  (1972), and the Hodges-Lehmann estimator is employed.  Otherwise, the
  returned confidence interval and point estimate are based on normal
  approximations.
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the Ansari-Bradley test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{null.value}{the ratio of scales \eqn{s} under the null, 1.}
  \item{alternative}{a character string describing the alternative
    hypothesis.}
  \item{method}{the string \code{"Ansari-Bradley test"}.}
  \item{data.name}{a character string giving the names of the data.}
  \item{conf.int}{a confidence interval for the scale parameter.
    (Only present if argument \code{conf.int = TRUE}.)}
  \item{estimate}{an estimate of the ratio of scales.
    (Only present if argument \code{conf.int = TRUE}.)}
}
\note{
  To compare results of the Ansari-Bradley test to those of the F test
  to compare two variances (under the assumption of normality), observe
  that \eqn{s} is the ratio of scales and hence \eqn{s^2} is the ratio
  of variances (provided they exist), whereas for the F test the ratio
  of variances itself is the parameter of interest.  In particular,
  confidence intervals are for \eqn{s} in the Ansari-Bradley test but
  for \eqn{s^2} in the F test.
}
\references{
  Myles Hollander & Douglas A. Wolfe (1973),
  \emph{Nonparametric statistical inference}.
  New York: John Wiley & Sons.
  Pages 83--92.

  David F. Bauer (1972),
  Constructing confidence sets using rank statistics.
  \emph{Journal of the American Statistical Association}
  \bold{67}, 687--690.
}
\seealso{
  \code{\link{fligner.test}} for a rank-based (nonparametric)
  \eqn{k}-sample test for homogeneity of variances;
  \code{\link{mood.test}} for another rank-based two-sample test for a
  difference in scale parameters;
  \code{\link{var.test}} and \code{\link{bartlett.test}} for parametric
  tests for the homogeneity in variance.
}
\examples{
## Hollander & Wolfe (1973, p. 86f):
## Serum iron determination using Hyland control sera
ramsay <- c(111, 107, 100, 99, 102, 106, 109, 108, 104, 99,
            101, 96, 97, 102, 107, 113, 116, 113, 110, 98)
jung.parekh <- c(107, 108, 106, 98, 105, 103, 110, 105, 104,
            100, 96, 108, 103, 104, 114, 114, 113, 108, 106, 99)
ansari.test(ramsay, jung.parekh)

ansari.test(rnorm(10), rnorm(10, 0, 2), conf.int = TRUE)
}
\keyword{htest}
\eof
\name{aov}
\alias{aov}
\alias{print.aov}
\alias{print.aovlist}
\alias{Error}
\title{Fit an Analysis of Variance Model}
\description{
  Fit an analysis of variance model by a call to \code{lm} for each stratum.
}
\usage{
aov(formula, data = NULL, projections = FALSE, qr = TRUE,
    contrasts = NULL, \dots)
}
\arguments{
  \item{formula}{A formula specifying the model.}
  \item{data}{A data frame in which the variables specified in the
    formula will be found. If missing, the variables are searched for in
    the standard way.}
  \item{projections}{Logical flag: should the projections be returned?}
  \item{qr}{Logical flag: should the QR decomposition be returned?}
  \item{contrasts}{A list of contrasts to be used for some of the factors
    in the formula. These are not used for any \code{Error} term, and
    supplying contrasts for factors only in the \code{Error} term will give
    a warning.}
  \item{\dots}{Arguments to be passed to \code{lm}, such as \code{subset}
    or \code{na.action}.}
}
\details{
  This provides a wrapper to \code{lm} for fitting linear models to
  balanced or unbalanced experimental designs.

  The main difference from \code{lm} is in the way \code{print},
  \code{summary} and so on handle the fit: this is expressed in the
  traditional language of the analysis of variance rather than that of
  linear models.

  If the formula contains a single \code{Error} term, this is used to
  specify error strata, and appropriate models are fitted within each
  error stratum.

  The formula can specify multiple responses.

  Weights can be specified by a \code{weights} argument, but should not
  be used with an \code{Error} term, and are incompletely supported
  (e.g., not by \code{\link{model.tables}}).
}
\note{
  \code{aov} is designed for balanced designs, and the results can be
  hard to interpret without balance: beware that missing values in the
  reponse(s) will likely lose the balance.  If there are two or more
  error strata, the methods used are statistically inefficient without
  balance, and it may be better to use \code{\link[nlme]{lme}}.

  Balance can be checked with the \code{\link{replications}} function.
}
\value{
  An object of class \code{c("aov", "lm")} or for multiple responses
  of class \code{c("maov", "aov", "mlm", "lm")} or for multiple error
  strata of class \code{"aovlist"}.  There are
  \code{\link{print}} and \code{\link{summary}} methods available for these.
}
\references{
  Chambers, J. M., Freeny, A and Heiberger, R. M. (1992)
  \emph{Analysis of variance; designed experiments.}
  Chapter 5 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\author{
  The design was inspired by the S function of the same name described
  in Chambers \emph{et al.} (1992).
}
\seealso{
  \code{\link{lm}}, \code{\link{summary.aov}},
  \code{\link{replications}}, \code{\link{alias}},
  \code{\link{proj}}, \code{\link{model.tables}}, \code{\link{TukeyHSD}}
}
\examples{
## From Venables and Ripley (2002) p.165.
N <- c(0,1,0,1,1,1,0,0,0,1,1,0,1,1,0,0,1,0,1,0,1,1,0,0)
P <- c(1,1,0,0,0,1,0,1,1,1,0,0,0,1,0,1,1,0,0,1,0,1,1,0)
K <- c(1,0,0,1,0,1,1,0,0,1,0,1,0,1,1,0,0,0,1,1,1,0,1,0)
yield <- c(49.5,62.8,46.8,57.0,59.8,58.5,55.5,56.0,62.8,55.8,69.5,55.0,
           62.0,48.8,45.5,44.2,52.0,51.5,49.8,48.8,57.2,59.0,53.2,56.0)
npk <- data.frame(block=gl(6,4), N=factor(N), P=factor(P),
                  K=factor(K), yield=yield)

op <- options(contrasts=c("contr.helmert", "contr.treatment"))
( npk.aov <- aov(yield ~ block + N*P*K, npk) )
summary(npk.aov)
coefficients(npk.aov)

## to show the effects of re-ordering terms contrast the two fits
aov(yield ~ block + N * P + K, npk)
aov(terms(yield ~ block + N * P + K, keep.order=TRUE), npk)


## as a test, not particularly sensible statistically
npk.aovE <- aov(yield ~  N*P*K + Error(block), npk)
npk.aovE
summary(npk.aovE)
options(op)# reset to previous
}
\keyword{models}
\keyword{regression}
\eof
\name{ar}
\alias{ar}
\alias{ar.burg}
\alias{ar.burg.default}
\alias{ar.burg.mts}
\alias{ar.yw}
\alias{ar.yw.default}
\alias{ar.yw.mts}
\alias{ar.mle}
\alias{print.ar}
\alias{predict.ar}
\title{Fit Autoregressive Models to Time Series}
\usage{
ar(x, aic = TRUE, order.max = NULL,
   method=c("yule-walker", "burg", "ols", "mle", "yw"), na.action,
   series, \dots)

ar.burg(x, \dots)
\method{ar.burg}{default}(x, aic = TRUE, order.max = NULL,
        na.action = na.fail, demean = TRUE, series, var.method = 1, \dots)
\method{ar.burg}{mts}(x, aic = TRUE, order.max = NULL,
        na.action = na.fail, demean = TRUE, series, var.method = 1, \dots)

ar.yw(x, \dots)
\method{ar.yw}{default}(x, aic = TRUE, order.max = NULL,
      na.action = na.fail, demean = TRUE, series, \dots)
\method{ar.yw}{mts}(x, aic = TRUE, order.max = NULL,
      na.action = na.fail, demean = TRUE, series, \dots)

ar.mle(x, aic = TRUE, order.max = NULL, na.action = na.fail, demean = TRUE,
       series, \dots)

\method{predict}{ar}(object, newdata, n.ahead = 1, se.fit = TRUE, \dots)
}
\arguments{
  \item{x}{A univariate or multivariate time series.}

  \item{aic}{Logical flag.  If \code{TRUE} then the Akaike Information
    Criterion is used to choose the order of the autoregressive
    model. If \code{FALSE}, the model of order \code{order.max} is
    fitted.}

  \item{order.max}{Maximum order (or order) of model to fit. Defaults
    to \eqn{10\log_{10}(N)}{10*log10(N)} where \eqn{N} is the number
    of observations except for \code{method="mle"} where it is the
    minimum of this quantity and 12.}

  \item{method}{Character string giving the method used to fit the
    model.  Must be one of the strings in the default argument
    (the first few characters are sufficient).  Defaults to
    \code{"yule-walker"}.}

  \item{na.action}{function to be called to handle missing values.}

  \item{demean}{should a mean be estimated during fitting?}

  \item{series}{names for the series.  Defaults to
    \code{deparse(substitute(x))}.}

  \item{var.method}{the method to estimate the innovations variance
    (see Details).}

  \item{\dots}{additional arguments for specific methods.}

  \item{object}{a fit from \code{ar}.}

  \item{newdata}{data to which to apply the prediction.}

  \item{n.ahead}{number of steps ahead at which to predict.}

  \item{se.fit}{logical: return estimated standard errors of the
    prediction error?}
}
\description{
  Fit an autoregressive time series model to the data, by default
  selecting the complexity by AIC.
}
\details{
  For definiteness, note that the AR coefficients have the sign in

  \deqn{x_t - \mu = a_1(x_{t-1} - \mu) + \cdots +  a_p(x_{t-p} - \mu) + e_t}{\code{(x[t] - m) = a[1]*(x[t-1] - m) + \dots +  a[p]*(x[t-p] - m) + e[t]}}

  \code{ar} is just a wrapper for the functions \code{ar.yw},
  \code{ar.burg}, \code{\link{ar.ols}} and \code{ar.mle}.

  Order selection is done by AIC if \code{aic} is true. This is
  problematic, as of the methods here only \code{ar.mle} performs
  true maximum likelihood estimation. The AIC is computed as if the variance
  estimate were the MLE, omitting the determinant term from the
  likelihood. Note that this is not the same as the Gaussian likelihood
  evaluated at the estimated parameter values. In \code{ar.yw} the
  variance matrix of the innovations is computed from the fitted
  coefficients and the autocovariance of \code{x}.

  \code{ar.burg} allows two methods to estimate the innovations
  variance and hence AIC. Method 1 is to use the update given by
  the Levinson-Durbin recursion (Brockwell and Davis, 1991, (8.2.6)
  on page 242), and follows S-PLUS. Method 2 is the mean of the sum
  of squares of the forward and backward prediction errors
  (as in Brockwell and Davis, 1996, page 145). Percival and Walden
  (1998) discuss both. In the multivariate case the estimated
  coefficients will depend (slightly) on the variance estimation method.

  Remember that \code{ar} includes by default a constant in the model, by
  removing the overall mean of \code{x} before fitting the AR model,
  or (\code{ar.mle}) estimating a constant to subtract.
}
\value{
  For \code{ar} and its methods a list of class \code{"ar"} with
  the following elements:
  \item{order}{The order of the fitted model.  This is chosen by
    minimizing the AIC if \code{aic=TRUE}, otherwise it is \code{order.max}.}
  \item{ar}{Estimated autoregression coefficients for the fitted model.}
  \item{var.pred}{The prediction variance: an estimate of the portion of the
    variance of the time series that is not explained by the
    autoregressive model.}
  \item{x.mean}{The estimated mean of the series used in fitting and for
    use in prediction.}
  \item{x.intercept}{(\code{ar.ols} only.) The intercept in the model for
    \code{x - x.mean}.}
  \item{aic}{The value of the \code{aic} argument.}
  \item{n.used}{The number of observations in the time series.}
  \item{order.max}{The value of the \code{order.max} argument.}
  \item{partialacf}{The estimate of the partial autocorrelation function
    up to lag \code{order.max}.}
  \item{resid}{residuals from the fitted model, conditioning on the
    first \code{order} observations. The first \code{order} residuals
    are set to \code{NA}. If \code{x} is a time series, so is \code{resid}.}
  \item{method}{The value of the \code{method} argument.}
  \item{series}{The name(s) of the time series.}
  \item{asy.var.coef}{(univariate case.) The asymptotic-theory
    variance matrix of the coefficient estimates.}

  For \code{predict.ar}, a time series of predictions, or if
  \code{se.fit = TRUE}, a list with components \code{pred}, the
  predictions, and \code{se}, the estimated standard errors. Both
  components are time series.
}
\author{
  Martyn Plummer. Univariate case of \code{ar.yw}, \code{ar.mle}
  and C code for univariate case of \code{ar.burg} by B. D. Ripley.
}

\note{
  Only the univariate case of \code{ar.mle} is implemented.

  Fitting by \code{method="mle"} to long series can be very slow.
}

\seealso{
  \code{\link{ar.ols}}, \code{\link{arima0}} for ARMA models.
}

\references{
  Brockwell, P. J. and Davis, R. A. (1991) \emph{Time
    Series and Forecasting Methods.}  Second edition. Springer, New
  York. Section 11.4.

  Brockwell, P. J. and Davis, R. A. (1996) \emph{Introduction to Time
    Series and Forecasting.} Springer, New York. Sections 5.1 and 7.6.

  Percival, D. P. and Walden, A. T. (1998) \emph{Spectral Analysis
    for Physical Applications.} Cambridge University Press.

  Whittle, P. (1963) On the fitting of multivariate autoregressions
  and the approximate canonical factorization of a spectral density
  matrix. \emph{Biometrika} \bold{40}, 129--134.
}

\examples{
data(lh)
ar(lh)
ar(lh, method="burg")
ar(lh, method="ols")
ar(lh, FALSE, 4) # fit ar(4)

data(sunspot)
(sunspot.ar <- ar(sunspot.year))
predict(sunspot.ar, n.ahead=25)
## try the other methods too

data(BJsales)
ar(ts.union(BJsales, BJsales.lead))
## Burg is quite different here, as is OLS (see ar.ols)
ar(ts.union(BJsales, BJsales.lead), method="burg")
}
\keyword{ts}
\eof
\name{ar.ols}
\alias{ar.ols}
\title{Fit Autoregressive Models to Time Series by OLS}
\usage{
ar.ols(x, aic = TRUE, order.max = NULL, na.action = na.fail,
       demean = TRUE, intercept = demean, series, \dots)
}
\arguments{
  \item{x}{A univariate or multivariate time series.}

  \item{aic}{Logical flag.  If \code{TRUE} then the Akaike Information
    Criterion is used to choose the order of the autoregressive
    model. If \code{FALSE}, the model of order \code{order.max} is
    fitted.}

  \item{order.max}{Maximum order (or order) of model to fit. Defaults
    to \eqn{10\log_{10}(N)}{10*log10(N)} where \eqn{N} is the number
    of observations.}

  \item{na.action}{function to be called to handle missing values.}

  \item{demean}{should the AR model be for \code{x} minus its mean?}

  \item{intercept}{should a separate intercept term be fitted?}

  \item{series}{names for the series.  Defaults to
    \code{deparse(substitute(x))}.}

  \item{\dots}{further arguments to be passed to or from methods.}
}
\description{
  Fit an autoregressive time series model to the data by ordinary
  least squares, by default selecting the complexity by AIC.
}
\details{
  \code{ar.ols} fits the general AR model to a possibly non-stationary
  and/or multivariate system of series \code{x}. The resulting
  unconstrained least squares estimates are consistent, even if
  some of the series are non-stationary and/or co-integrated.
  For definiteness, note that the AR coefficients have the sign in

  \deqn{x_t - \mu = a_0 + a_1(x_{t-1} - \mu) + \cdots + a_p(x_{t-p} - \mu) + e_t}{\code{(x[t] - m) = a[0] + a[1]*(x[t-1] - m) + \dots +  a[p]*(x[t-p] - m) + e[t]}}

  where \eqn{a_0}{a[0]} is zero unless \code{intercept} is true, and
  \eqn{\mu}{m} is the sample mean if \code{demean} is true, zero
  otherwise.

  Order selection is done by AIC if \code{aic} is true. This is
  problematic, as \code{ar.ols} does not perform
  true maximum likelihood estimation. The AIC is computed as if
  the variance estimate (computed from the variance matrix of the
  residuals) were the MLE, omitting the determinant term from the
  likelihood. Note that this is not the same as the Gaussian
  likelihood evaluated at the estimated parameter values.

  Some care is needed if \code{intercept} is true and \code{demean} is
  false. Only use this is the series are roughly centred on
  zero. Otherwise the computations may be inaccurate or fail entirely.
}
\value{
  A list of class \code{"ar"} with the following elements:
  \item{order}{The order of the fitted model.  This is chosen by
    minimizing the AIC if \code{aic=TRUE}, otherwise it is
    \code{order.max}.}
  \item{ar}{Estimated autoregression coefficients for the fitted
    model.}
  \item{var.pred}{The prediction variance: an estimate of the portion of
    the variance of the time series that is not explained by the
    autoregressive model.}
  \item{x.mean}{The estimated mean (or zero if \code{demean} is false)
    of the series used in fitting and for use in prediction.}
  \item{x.intercept}{The intercept in the model for
    \code{x - x.mean}, or zero if \code{intercept} is false.}
  \item{aic}{The value of the \code{aic} argument.}
  \item{n.used}{The number of observations in the time series.}
  \item{order.max}{The value of the \code{order.max} argument.}
  \item{partialacf}{\code{NULL}.  For compatibility with \code{ar}.}
  \item{resid}{residuals from the fitted model, conditioning on the
    first \code{order} observations.  The first \code{order} residuals
    are set to \code{NA}. If \code{x} is a time series, so is
    \code{resid}.}
  \item{method}{The character string \code{"Unconstrained LS"}.}
  \item{series}{The name(s) of the time series.}
  \item{asy.se.coef}{The asymptotic-theory standard errors of the
    coefficient estimates.}
}
\author{Adrian Trapletti, Brian Ripley.}
\seealso{
  \code{\link{ar}}
}
\references{
  Luetkepohl, H. (1991): \emph{Introduction to Multiple Time Series
    Analysis.} Springer Verlag, NY, pp. 368--370.
}
\examples{
data(lh)
ar(lh, method="burg")
ar.ols(lh)
ar.ols(lh, FALSE, 4) # fit ar(4)

data(BJsales)
ar.ols(ts.union(BJsales, BJsales.lead))

data(EuStockMarkets)
x <- diff(log(EuStockMarkets))
ar.ols(x, order.max=6, demean=FALSE, intercept=TRUE)
}
\keyword{ts}
\eof
\name{arima}
\alias{arima}
%\alias{print.Arima}
%\alias{coef.Arima}
%\alias{vcov.Arima}
%\alias{logLik.Arima}
\title{ARIMA Modelling of Time Series}
\description{
  Fit an ARIMA model to a univariate time series.
}
\usage{
arima(x, order = c(0, 0, 0),
      seasonal = list(order = c(0, 0, 0), period = NA),
      xreg = NULL, include.mean = TRUE, transform.pars = TRUE,
      fixed = NULL, init = NULL, method = c("CSS-ML", "ML", "CSS"),
      n.cond, optim.control = list(), kappa = 1e6)
}
\arguments{
  \item{x}{a univariate time series}

  \item{order}{A specification of the non-seasonal part of the ARIMA
    model: the three components \eqn{(p, d, q)} are the AR order, the
    degree of differencing, and the MA order.}

  \item{seasonal}{A specification of the seasonal part of the ARIMA
    model, plus the period (which defaults to \code{frequency(x)}).
    This should be a list with components \code{order} and
    \code{period}, but a specification of just a numeric vector of
    length 3 will be turned into a suitable list with the specification
    as the \code{order}.}

  \item{xreg}{Optionally, a vector or matrix of external regressors,
    which must have the same number of rows as \code{x}.}

  \item{include.mean}{Should the ARIMA model include
    a mean term? The default is \code{TRUE} for undifferenced series,
    \code{FALSE} for differenced ones (where a mean would not affect
    the fit nor predictions).}

  \item{transform.pars}{Logical.  If true, the AR parameters are
    transformed to ensure that they remain in the region of
    stationarity.  Not used for \code{method = "CSS"}.}

  \item{fixed}{optional numeric vector of the same length as the total
    number of parameters.  If supplied, only \code{NA} entries in
    \code{fixed} will be varied.  \code{transform.pars = TRUE}
    will be overridden (with a warning) if any AR parameters are fixed.
    It may be wise to set \code{transform.pars = FALSE} when fixing
    MA parameters, especially near non-invertibility.
  }

  \item{init}{optional numeric vector of initial parameter
    values.  Missing values will be filled in, by zeroes except for
    regression coefficients.  Values already specified in \code{fixed}
    will be ignored.}

  \item{method}{Fitting method: maximum likelihood or minimize
    conditional sum-of-squares.  The default (unless there are missing
    values) is to use conditional-sum-of-squares to find starting
    values, then maximum likelihood.}

  \item{n.cond}{Only used if fitting by conditional-sum-of-squares: the
    number of initial observations to ignore.  It will be ignored if
    less than the maximum lag of an AR term.}

  \item{optim.control}{List of control parameters for \code{\link{optim}}.}

  \item{kappa}{the prior variance (as a multiple of the innovations
    variance) for the past observations in a differenced model.  Do not
    reduce this.}
}
\details{
  Different definitions of ARMA models have different signs for the
  AR and/or MA coefficients. The definition here has

  \deqn{X_t = a_1X_{t-1} + \cdots + a_pX_{t-p} + e_t + b_1e_{t-1} + \dots + b_qe_{t-q}}{\code{X[t] = a[1]X[t-1] + \dots + a[p]X[t-p] + e[t] + b[1]e[t-1] + \dots + b[q]e[t-q]}}

  and so the MA coefficients differ in sign from those of
  S-PLUS.  Further, if \code{include.mean} is true, this formula
  applies to \eqn{X-m} rather than \eqn{X}.  For ARIMA models with
  differencing, the differenced series follows a zero-mean ARMA model.
  If a \code{xreg} term is included, a linear regression (with a
  constant term if \code{include.mean} is true) is fitted with an ARMA
  model for the error term.

  The variance matrix of the estimates is found from the Hessian of
  the log-likelihood, and so may only be a rough guide.

  Optimization is done by \code{\link{optim}}. It will work
  best if the columns in \code{xreg} are roughly scaled to zero mean
  and unit variance, but does attempt to estimate suitable scalings.
}
\section{Fitting methods}{
  The exact likelihood is computed via a state-space representation of
  the ARIMA process, and the innovations and their variance found by a
  Kalman filter.  The initialization of the differenced ARMA process uses
  stationarity and is based on Gardner \emph{et al.} (1980).  For a
  differenced process the non-stationary components are given a diffuse
  prior (controlled by \code{kappa}).  Observations which are still
  controlled by the diffuse prior (determined by having a Kalman gain of
  at least \code{1e4}) are excluded from the likelihood calculations.
  (This gives comparable results to \code{\link{arima0}} in the absence
  of missing values, when the observations excluded are precisely those
  dropped by the differencing.)

  Missing values are allowed, and are handled exactly in method \code{"ML"}.

  If \code{transform.pars} is true, the optimization is done using an
  alternative parametrization which is a variation on that suggested by
  Jones (1980) and ensures that the model is stationary.  For an AR(p)
  model the parametrization is via the inverse tanh of the partial
  autocorrelations: the same procedure is applied (separately) to the
  AR and seasonal AR terms.  The MA terms are not constrained to be
  invertible during optimization, but they will be converted to
  invertible form after optimization if \code{transform.pars} is true.

  Conditional sum-of-squares is provided mainly for expositional
  purposes.  This computes the sum of squares of the fitted innovations
  from observation \code{n.cond} on, (where \code{n.cond} is at least
  the maximum lag of an AR term), treating all earlier innovations to
  be zero.  Argument \code{n.cond} can be used to allow comparability
  between different fits.  The \dQuote{part log-likelihood} is the first
  term, half the log of the estimated mean square.  Missing values
  are allowed, but will cause many of the innovations to be missing.

  When regressors are specified, they are orthogonalized prior to
  fitting unless any of the coefficients is fixed.  It can be helpful to
  roughly scale the regressors to zero mean and unit variance.
}
\value{
  A list of class \code{"Arima"} with components:

  \item{coef}{a vector of AR, MA and regression coefficients, which can
    be extracted by the \code{\link{coef}} method.}

  \item{sigma2}{the MLE of the innovations variance.}

  \item{var.coef}{the estimated variance matrix of the coefficients
    \code{coef}, which can be extracted by the \code{\link{vcov}} method.}

  \item{loglik}{the maximized log-likelihood (of the differenced data),
    or the approximation to it used.}

  \item{arma}{A compact form of the specification, as a vector giving
    the number of AR, MA, seasonal AR and seasonal MA coefficients,
    plus the period and the number of non-seasonal and seasonal
    differences.}

  \item{aic}{the AIC value corresponding to the log-likelihood. Only
    valid for \code{method = "ML"} fits.}

  \item{residuals}{the fitted innovations.}

  \item{call}{the matched call.}

  \item{series}{the name of the series \code{x}.}

  \item{convergence}{the value returned by \code{\link{optim}}.}

  \item{n.cond}{the number of initial observations not used in the fitting.}

  \item{model}{A list representing the Kalman Filter used in the
    fitting.  See \code{\link{KalmanLike}}.}
}
\references{
  Brockwell, P. J. and Davis, R. A. (1996) \emph{Introduction to Time
    Series and Forecasting.} Springer, New York. Sections 3.3 and 8.3.

  Durbin, J. and Koopman, S. J. (2001) \emph{Time Series Analysis by
    State Space Methods.}  Oxford University Press.

  Gardner, G, Harvey, A. C. and Phillips, G. D. A. (1980) Algorithm
  AS154. An algorithm for exact maximum likelihood estimation of
  autoregressive-moving average models by means of Kalman filtering.
  \emph{Applied Statistics} \bold{29}, 311--322.

  Harvey, A. C. (1993) \emph{Time Series Models},
  2nd Edition, Harvester Wheatsheaf, sections 3.3 and 4.4.

  Jones, R. H. (1980) Maximum likelihood fitting of ARMA models to time
  series with missing observations. \emph{Technometrics} \bold{20} 389--395.
}

\note{
  The results are likely to be different from S-PLUS's
  \code{arima.mle}, which computes a conditional likelihood and does
  not include a mean in the model.  Further, the convention used by
  \code{arima.mle} reverses the signs of the MA coefficients.

  \code{arima} is very similar to \code{\link{arima0}} for
  ARMA models or for differenced models without missing values,
  but handles differenced models with missing values exactly.
  It is somewhat slower than \code{arima0}, particularly for seasonally
  differenced models.
}

\seealso{
  \code{\link{predict.Arima}}, \code{\link{arima.sim}} for simulating
  from an ARIMA model, \code{\link{tsdiag}}, \code{\link{arima0}},
  \code{\link{ar}}
}

\examples{
data(lh)
arima(lh, order = c(1,0,0))
arima(lh, order = c(3,0,0))
arima(lh, order = c(1,0,1))

arima(lh, order = c(3,0,0), method = "CSS")

data(USAccDeaths)
arima(USAccDeaths, order = c(0,1,1), seasonal = list(order=c(0,1,1)))
arima(USAccDeaths, order = c(0,1,1), seasonal = list(order=c(0,1,1)),
      method = "CSS") # drops first 13 observations.
# for a model with as few years as this, we want full ML

data(LakeHuron)
arima(LakeHuron, order = c(2,0,0), xreg = time(LakeHuron)-1920)

data(presidents) # contains NAs
## graphs in example(acf) suggest order 1 or 3
(fit1 <- arima(presidents, c(1, 0, 0)))
tsdiag(fit1)
(fit3 <- arima(presidents, c(3, 0, 0)))  # smaller AIC
tsdiag(fit3)
}
\keyword{ts}
\eof
\name{arima.sim}
\alias{arima.sim}
\title{Simulate from an ARIMA Model}
\description{
  Simulate from an ARIMA model.
}
\usage{
arima.sim(model, n, rand.gen = rnorm, innov = rand.gen(n, \dots),
          n.start = NA, \dots)
}
\arguments{
  \item{model}{A list with component \code{ar} and/or \code{ma} giving
    the AR and MA coefficients respectively.  Optionally a component
    \code{order} can be used. An empty list gives an ARIMA(0, 0, 0)
    model, that is white noise.}
  \item{n}{length of output series, before un-differencing.}
  \item{rand.gen}{optional: a function to generate the innovations.}
  \item{innov}{an optional times series of innovations.  If not
    provided, \code{rand.gen} is used.}
  \item{n.start}{length of \dQuote{burn-in} period.  If \code{NA}, the
    default, a reasonable value is computed.}
  \item{\dots}{additional arguments for \code{rand.gen}.  Most usefully,
    the standard deviation of the innovations generated by \code{rnorm}
    can be specified by \code{sd}.}
}
\details{
  See \code{\link{arima}} for the precise definition of an ARIMA model.

  The ARMA model is checked for stationarity.

  ARIMA models are specified via the \code{order} component of
  \code{model}, in the same way as for \code{\link{arima}}.  Other
  aspects of the \code{order} component are ignored, but inconsistent
  specifications of the MA and AR orders are detected.  The
  un-differencing assumes previous values of zero, and to remind the
  user of this, those values are returned.
}
\value{
  A time-series object of class \code{"ts"}.
}
\seealso{
  \code{\link{arima}}
}
\examples{
arima.sim(n = 63, list(ar = c(0.8897, -0.4858), ma = c(-0.2279, 0.2488)),
          sd = sqrt(0.1796))
# mildly long-tailed
arima.sim(n = 63, list(ar=c(0.8897, -0.4858), ma=c(-0.2279, 0.2488)),
          rand.gen = function(n, ...) sqrt(0.1796) * rt(n, df = 5))

# An ARIMA simulation
ts.sim <- arima.sim(list(order = c(1,1,0), ar = 0.7), n = 200)
ts.plot(ts.sim)
}
\keyword{ts}
\eof
\name{arima0}
\alias{arima0}
\alias{print.arima0}
\alias{predict.arima0}
\title{ARIMA Modelling of Time Series -- Preliminary Version}
\description{
  Fit an ARIMA model to a univariate time series, and forecast from
  the fitted model.
}
\usage{
arima0(x, order = c(0, 0, 0),
       seasonal = list(order = c(0, 0, 0), period = NA),
       xreg = NULL, include.mean = TRUE, delta = 0.01,
       transform.pars = TRUE, fixed = NULL, init = NULL,
       method = c("ML", "CSS"), n.cond, optim.control = list())

\method{predict}{arima0}(object, n.ahead = 1, newxreg, se.fit = TRUE, \dots)
}
\arguments{
  \item{x}{a univariate time series}

  \item{order}{A specification of the non-seasonal part of the ARIMA
    model: the three components \eqn{(p, d, q)} are the AR order, the
    degree of differencing, and the MA order.}

  \item{seasonal}{A specification of the seasonal part of the ARIMA
    model, plus the period (which defaults to \code{frequency(x)}).
    This should be a list with components \code{order} and
    \code{period}, but a specification of just a numeric vector of
    length 3 will be turned into a suitable list with the specification
    as the \code{order}.}

  \item{xreg}{Optionally, a vector or matrix of external regressors,
    which must have the same number of rows as \code{x}.}

  \item{include.mean}{Should the ARIMA model include
    a mean term? The default is \code{TRUE} for undifferenced series,
    \code{FALSE} for differenced ones (where a mean would not affect
    the fit nor predictions).}

  \item{delta}{A value to indicate at which point \sQuote{fast
      recursions} should be used.  See the Details section.}

  \item{transform.pars}{Logical.  If true, the AR parameters are
    transformed to ensure that they remain in the region of
    stationarity.  Not used for \code{method = "CSS"}.}

  \item{fixed}{optional numeric vector of the same length as the total
    number of parameters.  If supplied, only \code{NA} entries in
    \code{fixed} will be varied.  \code{transform.pars = TRUE}
    will be overridden (with a warning) if any ARMA parameters are
    fixed.}

  \item{init}{optional numeric vector of initial parameter
    values.  Missing values will be filled in, by zeroes except for
    regression coefficients.  Values already specified in \code{fixed}
    will be ignored.}

  \item{method}{Fitting method: maximum likelihood or minimize
    conditional sum-of-squares.}

  \item{n.cond}{Only used if fitting by conditional-sum-of-squares: the
    number of initial observations to ignore.  It will be ignored if
    less than the maximum lag of an AR term.}

  \item{optim.control}{List of control parameters for \code{\link{optim}}.}

  \item{object}{The result of an \code{arima0} fit.}

  \item{newxreg}{New values of \code{xreg} to be used for
    prediction. Must have at least \code{n.ahead} rows.}

  \item{n.ahead}{The number of steps ahead for which prediction is required.}

  \item{se.fit}{Logical: should standard errors of prediction be returned?}

  \item{\dots}{arguments passed to or from other methods.}
}
\details{
  Different definitions of ARMA models have different signs for the
  AR and/or MA coefficients. The definition here has

  \deqn{X_t = a_1X_{t-1} + \cdots + a_pX_{t-p} + e_t + b_1e_{t-1} + \dots + b_qe_{t-q}}{\code{X[t] = a[1]X[t-1] + \dots + a[p]X[t-p] + e[t] + b[1]e[t-1] + \dots + b[q]e[t-q]}}

  and so the MA coefficients differ in sign from those of
  S-PLUS.  Further, if \code{include.mean} is true, this formula
  applies to \eqn{X-m} rather than \eqn{X}.  For ARIMA models with
  differencing, the differenced series follows a zero-mean ARMA model.

  The variance matrix of the estimates is found from the Hessian of
  the log-likelihood, and so may only be a rough guide, especially for
  fits close to the boundary of invertibility.

  Optimization is done by \code{\link{optim}}. It will work
  best if the columns in \code{xreg} are roughly scaled to zero mean
  and unit variance, but does attempt to estimate suitable scalings.

  Finite-history prediction is used. This is only statistically
  efficient if the MA part of the fit is invertible, so
  \code{predict.arima0} will give a warning for non-invertible MA
  models.
}
\section{Fitting methods}{
  The exact likelihood is computed via a state-space representation of
  the ARMA process, and the innovations and their variance found by a
  Kalman filter based on Gardner \emph{et al.} (1980).  This has
  the option to switch to \sQuote{fast recursions} (assume an
  effectively infinite past) if the innovations variance is close
  enough to its asymptotic bound. The argument \code{delta} sets the
  tolerance: at its default value the approximation is normally
  negligible and the speed-up considerable.  Exact computations can be
  ensured by setting \code{delta} to a negative value.

  If \code{transform.pars} is true, the optimization is done using an
  alternative parametrization which is a variation on that suggested by
  Jones (1980) and ensures that the model is stationary.  For an AR(p)
  model the parametrization is via the inverse tanh of the partial
  autocorrelations: the same procedure is applied (separately) to the
  AR and seasonal AR terms.  The MA terms are also constrained to be
  invertible during optimization by the same transformation if
  \code{transform.pars} is true.  Note that the MLE for MA terms does
  sometimes occur for MA polynomials with unit roots: such models can be
  fitted by using \code{transform.pars = FALSE} and specifying a good
  set of initial values (often obtainable from a fit with
  \code{transform.pars = TRUE}).

  As from \R 1.5.0 missing values are allowed, but any missing values
  will force \code{delta} to be ignored and full recursions used.
  Note that missing values will be propogated by differencing, so the
  procedure used in this function is not fully efficient in that case.

  Conditional sum-of-squares is provided mainly for expositional
  purposes.  This computes the sum of squares of the fitted innovations
  from observation
  \code{n.cond} on, (where \code{n.cond} is at least the maximum lag of
  an AR term), treating all earlier innovations to be zero.  Argument
  \code{n.cond} can be used to allow comparability between different
  fits.  The \dQuote{part log-likelihood} is the first term, half the
  log of the estimated mean square.  Missing values are allowed, but
  will cause many of the innovations to be missing.

  When regressors are specified, they are orthogonalized prior to
  fitting unless any of the coefficients is fixed.  It can be helpful to
  roughly scale the regressors to zero mean and unit variance.
}
\value{
  For \code{arima0}, a list of class \code{"arima0"} with components:

  \item{coef}{a vector of AR, MA and regression coefficients,}

  \item{sigma2}{the MLE of the innovations variance.}

  \item{var.coef}{the estimated variance matrix of the coefficients
    \code{coef}.}

  \item{loglik}{the maximized log-likelihood (of the differenced data),
    or the approximation to it used.}

  \item{arma}{A compact form of the specification, as a vector giving
    the number of AR, MA, seasonal AR and seasonal MA coefficients,
    plus the period and the number of non-seasonal and seasonal
    differences.}

  \item{aic}{the AIC value corresponding to the log-likelihood. Only
    valid for \code{method = "ML"} fits.}

  \item{residuals}{the fitted innovations.}

  \item{call}{the matched call.}

  \item{series}{the name of the series \code{x}.}

  \item{convergence}{the value returned by \code{\link{optim}}.}

  \item{n.cond}{the number of initial observations not used in the fitting.}

  For \code{predict.arima0}, a time series of predictions, or if
  \code{se.fit = TRUE}, a list with components \code{pred}, the
  predictions, and \code{se}, the estimated standard errors. Both
  components are time series.
}
\references{
  Brockwell, P. J. and Davis, R. A. (1996) \emph{Introduction to Time
    Series and Forecasting.} Springer, New York. Sections 3.3 and 8.3.

  Gardner, G, Harvey, A. C. and Phillips, G. D. A. (1980) Algorithm
  AS154. An algorithm for exact maximum likelihood estimation of
  autoregressive-moving average models by means of Kalman filtering.
  \emph{Applied Statistics} \bold{29}, 311--322.

  Harvey, A. C. (1993) \emph{Time Series Models},
  2nd Edition, Harvester Wheatsheaf, sections 3.3 and  4.4.

  Harvey, A. C. and McKenzie, C. R. (1982) Algorithm AS182.
  An algorithm for finite sample prediction from ARIMA processes.
  \emph{Applied Statistics} \bold{31}, 180--187.

  Jones, R. H. (1980) Maximum likelihood fitting of ARMA models to time
  series with missing observations. \emph{Technometrics} \bold{20} 389--395.
}

\note{
  This is a preliminary version, and will be replaced by \code{\link{arima}}.

  The standard errors of prediction exclude the uncertainty in the
  estimation of the ARMA model and the regression coefficients.

  The results are likely to be different from S-PLUS's
  \code{arima.mle}, which computes a conditional likelihood and does
  not include a mean in the model.  Further, the convention used by
  \code{arima.mle} reverses the signs of the MA coefficients.
}

\seealso{\code{\link{arima}}, \code{\link{ar}}, \code{\link{tsdiag}}}

\examples{
data(lh)
\dontrun{arima0(lh, order = c(1,0,0))}
arima0(lh, order = c(3,0,0))
arima0(lh, order = c(1,0,1))
predict(arima0(lh, order = c(3,0,0)), n.ahead = 12)

arima0(lh, order = c(3,0,0), method = "CSS")

data(USAccDeaths)
# for a model with as few years as this, we want full ML
(fit <- arima0(USAccDeaths, order = c(0,1,1),
               seasonal = list(order=c(0,1,1)), delta = -1))
predict(fit, n.ahead = 6)

data(LakeHuron)
arima0(LakeHuron, order = c(2,0,0), xreg = time(LakeHuron)-1920)
\dontrun{
data(presidents) # contains NAs
## graphs in example(acf) suggest order 1 or 3
(fit1 <- arima0(presidents, c(1, 0, 0), delta = -1))  # avoid warning
tsdiag(fit1)
(fit3 <- arima0(presidents, c(3, 0, 0), delta = -1))  # smaller AIC
tsdiag(fit3)}
}
\keyword{ts}
\eof
\name{as.hclust}
\alias{as.hclust}
\alias{as.hclust.default}
\alias{as.hclust.twins}
\title{Convert Objects to Class hclust}
\usage{
as.hclust(x, \dots)
%as.hclust.default(x, \dots)
%as.hclust.twins(x, \dots)
}
\arguments{
  \item{x}{Hierarchical clustering object}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  Converts objects from other hierarchical clustering functions to
  class \code{"hclust"}.
}
\details{
  Currently there is only support for converting objects of
  class \code{"twins"} as produced by the functions \code{diana} and
  \code{agnes} from the package \pkg{cluster}.  The default method
  throws an error unless passed an \code{"hclust"} object.
}
\value{
  An object of class \code{"hclust"}.
}
\seealso{
  \code{\link{hclust}}, \code{\link[cluster]{diana}},
  \code{\link[cluster]{agnes}}
}
\examples{
x <- matrix(rnorm(30), ncol=3)
hc <- hclust(dist(x), method="complete")

if(require(cluster, quietly=TRUE)) {# is a recommended package
  ag <- agnes(x, method="complete")
  hcag <- as.hclust(ag)
  ## The dendrograms order slightly differently:
  op <- par(mfrow=c(1,2))
  plot(hc) ;  mtext("hclust", side=1)
  plot(hcag); mtext("agnes",  side=1)
}
}
\keyword{multivariate}
\keyword{cluster}
\eof
% $Id: asOneSidedFormula.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{asOneSidedFormula}
\title{Convert to One-Sided Formula}
\usage{
asOneSidedFormula(object)
}
\alias{asOneSidedFormula}
\arguments{
 \item{object}{a one-sided formula, an expression, a numeric value, or a
   character string.}
}
\description{
  Names, expressions, numeric values, and character strings are converted to
  one-sided formulas. If \code{object} is a formula, it must be
  one-sided, in which case it is returned unaltered.
}
\value{
  a one-sided formula representing \code{object}
}
\author{Jose Pinheiro and Douglas Bates}

\seealso{\code{\link{formula}}}

\examples{
asOneSidedFormula("age")
asOneSidedFormula(~ age)
}
\keyword{models}
\eof
% file austres.Rd
% copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{austres}
\docType{data}
\alias{austres}
\title{
  Quarterly Time Series of the Number of Australian Residents
}
\description{
  Numbers (in thousands) of Australian residents measured quarterly from
  March 1971 to March 1994.  The object is of class \code{"ts"}.
}
\usage{data(austres)}
\source{
  P. J. Brockwell and R. A. Davis (1996)
  \emph{Introduction to Time Series and Forecasting.}
  Springer
}
\keyword{datasets}
\eof
\name{ave}
\title{Group Averages Over Level Combinations of Factors}
\usage{
ave(x, \dots, FUN = mean)
}
\alias{ave}
\arguments{
  \item{x}{A numeric.}
  \item{\dots}{Grouping variables, typically factors, all of the same
    \code{length} as \code{x}.}
  \item{FUN}{Function to apply for each factor level combination.}
}
\description{
  Subsets of \code{x[]} are averaged, where each subset consist of those
  observations with the same factor levels.
}
\value{
  A numeric vector, say \code{y} of length \code{length(x)}.
  If \code{\dots} is \code{g1,g2}, e.g.,
  \code{y[i]} is equal to \code{FUN(x[j]}, for all \code{j} with
  \code{g1[j]==g1[i]} and \code{g2[j]==g2[i])}.
}
\seealso{\code{\link{mean}}, \code{\link{median}}.}
\examples{
ave(1:3)# no grouping -> grand mean

data(warpbreaks)
attach(warpbreaks)
ave(breaks, wool)
ave(breaks, tension)
ave(breaks, tension, FUN = function(x)mean(x, trim=.1))
plot(breaks, main =
     "ave( Warpbreaks )  for   wool  x  tension  combinations")
lines(ave(breaks, wool, tension            ), type='s', col = "blue")
lines(ave(breaks, wool, tension, FUN=median), type='s', col = "green")
legend(40,70, c("mean","median"), lty=1,col=c("blue","green"), bg="gray90")
detach()
}
\keyword{univar}

\eof
\name{bandwidth}
\alias{bw.nrd0}
\alias{bw.nrd}
\alias{bw.ucv}
\alias{bw.bcv}
\alias{bw.SJ}
\concept{bandwidth}
\title{Bandwidth Selectors for Kernel Density Estimation}
\synopsis{
bw.nrd0(x)
bw.nrd(x)
bw.ucv(x, nb = 1000, lower = 0.1 * hmax, upper = hmax)
bw.bcv(x, nb = 1000, lower = 0.1 * hmax, upper = hmax)
bw.SJ(x, nb = 1000, lower = 0.1 * hmax, upper = hmax, method=c("ste", "dpi"))
}
\usage{
bw.nrd0(x)
bw.nrd(x)
bw.ucv(x, nb = 1000, lower, upper)
bw.bcv(x, nb = 1000, lower, upper)
bw.SJ(x, nb = 1000, lower, upper, method = c("ste", "dpi"))
}
\arguments{
  \item{x}{A data vector.}
  \item{nb}{number of bins to use.}
  \item{lower, upper}{Range over which to minimize.  The default is
    almost always satisfactory.}
  \item{method}{Either \code{"ste"} ("solve-the-equation") or
    \code{"dpi"} ("direct plug-in").}
}
\description{
  Bandwidth selectors for gaussian windows in \code{\link{density}}.
}

\details{
  \code{bw.nrd0} implements a rule-of-thumb for
  choosing the bandwidth of a Gaussian kernel density estimator.
  It defaults to 0.9 times the
  minimum of the standard deviation and the interquartile range divided by
  1.34 times the sample size to the negative one-fifth power
  (= Silverman's \dQuote{rule of thumb}, Silverman (1986, page 48, eqn (3.31))
  \emph{unless} the quartiles coincide when a positive result
  will be guaranteed.

  \code{bw.nrd} is the more common variation given by Scott (1992),
  using factor 1.06.

  \code{bw.ucv} and \code{bw.bcv} implement unbiased and
  biased cross-validation respectively.

  \code{bw.SJ} implements the methods of Sheather & Jones (1991)
  to select the bandwidth using pilot estimation of derivatives.
}
\value{
  A bandwidth on a scale suitable for the \code{bw} argument
  of \code{density}.
}
\seealso{
  \code{\link{density}}.

  \code{\link[MASS]{bandwidth.nrd}}, \code{\link[MASS]{ucv}},
  \code{\link[MASS]{bcv}} and \code{\link[MASS]{width.SJ}} in
  package \pkg{MASS}, which are all scaled to the \code{width} argument
  of \code{density} and so give answers four times as large.
}
\references{
  Scott, D. W. (1992)
  \emph{Multivariate Density Estimation: Theory, Practice, and
    Visualization.}
  Wiley.

  Sheather, S. J. and Jones, M. C. (1991)
  A reliable data-based bandwidth selection method for kernel density
  estimation.
  \emph{Journal of the Royal Statistical Society series B},
  \bold{53}, 683--690.

  Silverman, B. W. (1986)
  \emph{Density Estimation}.
  London: Chapman and Hall.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S}.
  Springer.
}
\examples{
data(precip)
plot(density(precip, n = 1000))
rug(precip)
lines(density(precip, bw="nrd"), col = 2)
lines(density(precip, bw="ucv"), col = 3)
lines(density(precip, bw="bcv"), col = 4)
lines(density(precip, bw="SJ-ste"), col = 5)
lines(density(precip, bw="SJ-dpi"), col = 6)
legend(55, 0.035,
       legend = c("nrd0", "nrd", "ucv", "bcv", "SJ-ste", "SJ-dpi"),
       col = 1:6, lty = 1)
}
\keyword{distribution}
\keyword{smooth}
\eof
\name{bartlett.test}
\alias{bartlett.test}
\alias{bartlett.test.default}
\alias{bartlett.test.formula}
\title{Bartlett Test for Homogeneity of Variances}
\description{
  Performs Bartlett's test of the null that the variances in each of the
  groups (samples) are the same.
}
\usage{
bartlett.test(x, \dots)

\method{bartlett.test}{default}(x, g, \dots)

\method{bartlett.test}{formula}(formula, data, subset, na.action, \dots)
}
\arguments{
  \item{x}{a numeric vector of data values, or a list of numeric data
    vectors representing the respective samples, or fitted linear model
    objects (inheriting from class \code{"lm"}).}
  \item{g}{a vector or factor object giving the group for the
    corresponding elements of \code{x}.
    Ignored if \code{x} is a list.}
  \item{formula}{a formula of the form \code{lhs ~ rhs} where \code{lhs}
    gives the data values and \code{rhs} the corresponding groups.}
  \item{data}{an optional data frame containing the variables in the
    model formula.}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.  Defaults to
    \code{getOption("na.action")}.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\details{
  If \code{x} is a list, its elements are taken as the samples or fitted
  linear models to be compared for homogeneity of variances.  In this
  case, the elements must either all be numeric data vectors or fitted
  linear model objects, \code{g} is ignored, and one can simply use
  \code{bartlett.test(x)} to perform the test.  If the samples are not
  yet contained in a list, use \code{bartlett.test(list(x, ...))}.

  Otherwise, \code{x} must be a numeric data vector, and \code{g} must
  be a vector or factor object of the same length as \code{x} giving the
  group for the corresponding elements of \code{x}.
}
\value{
  A list of class \code{"htest"} containing the following components:
  \item{statistic}{Bartlett's K-squared test statistic.}
  \item{parameter}{the degrees of freedom of the approximate chi-squared
    distribution of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{method}{the character string
    \code{"Bartlett test for homogeneity of variances"}.}
  \item{data.name}{a character string giving the names of the data.}
}
\seealso{
  \code{\link{var.test}} for the special case of comparing variances in
  two samples from normal distributions;
  \code{\link{fligner.test}} for a rank-based (nonparametric)
  \eqn{k}-sample test for homogeneity of variances;
  \code{\link{ansari.test}} and \code{\link{mood.test}} for two rank
  based two-sample tests for difference in scale.
}
\references{
  Bartlett, M. S. (1937).
  Properties of sufficiency and statistical tests.
  \emph{Proceedings of the Royal Statistical Society Series A}
  \bold{160}, 268--282.
}
\examples{
% FIXME: could use something more interesting here
data(InsectSprays)
plot(count ~ spray, data = InsectSprays)
bartlett.test(InsectSprays$count, InsectSprays$spray)
bartlett.test(count ~ spray, data = InsectSprays)
}
\keyword{htest}
\eof
% file beavers.Rd
% copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{beavers}
\docType{data}
\alias{beavers}
\alias{beaver1}
\alias{beaver2}
\title{Body Temperature Series of Two Beavers}
\usage{data(beavers)}
\description{
  Reynolds (1994) describes a small part of a study of the long-term
  temperature dynamics of beaver \emph{Castor canadensis} in
  north-central Wisconsin.  Body temperature was measured by telemetry
  every 10 minutes for four females, but data from a one period of
  less than a day for each of two animals is used there. 
}
\format{
  The \code{beaver1} data frame has 114 rows and 4 columns on body
  temperature measurements at 10 minute intervals.

  The \code{beaver2} data frame has 100 rows and 4 columns on body
  temperature measurements at 10 minute intervals.

  The variables are as follows:
  \describe{
    \item{day}{Day of observation (in days since the beginning of
      1990), December 12--13 (\code{beaver1}) and November 3--4
      (\code{beaver2}).}
    \item{time}{Time of observation, in the form \code{0330} for
      3:30am}
    \item{temp}{Measured body temperature in degrees Celsius.}
    \item{activ}{Indicator of activity outside the retreat.}
  }
}
\note{
  The observation at 22:20 is missing in \code{beaver1}.
}
\source{
  P. S. Reynolds (1994) Time-series analyses of beaver body
  temperatures.  Chapter 11 of Lange, N., Ryan, L., Billard, L.,
  Brillinger, D., Conquest, L. and Greenhouse, J. eds (1994)
  \emph{Case Studies in Biometry.}
  New York: John Wiley and Sons.
}
\examples{
data(beavers)
%% consider converting times to POSIXct (using 'yday' ?)

(yl <- range(beaver1$temp, beaver2$temp))

beaver.plot <- function(bdat, ...) {
  nam <- deparse(substitute(bdat))  
  attach(bdat)
  # Hours since start of day:
  hours <- time \%/\% 100 + 24*(day - day[1]) + (time \%\% 100)/60
  plot (hours, temp, type = "l", ...,
        main = paste(nam, "body temperature"))
  abline(h = 37.5, col = "gray", lty = 2)
  is.act <- activ == 1
  points(hours[is.act], temp[is.act], col = 2, cex = .8)
}
op <- par(mfrow = c(2,1), mar = c(3,3,4,2), mgp = .9* 2:0)
 beaver.plot(beaver1, ylim = yl)
 beaver.plot(beaver2, ylim = yl)
par(op)
}
\keyword{datasets}
\eof
\name{binom.test}
\alias{binom.test}
\title{Exact Binomial Test}
\description{
  Performs an exact test of a simple null hypothesis about the
  probability of success in a Bernoulli experiment.
}
\usage{
binom.test(x, n, p = 0.5,
           alternative = c("two.sided", "less", "greater"),
           conf.level = 0.95)
}
\arguments{
  \item{x}{number of successes, or a vector of length 2 giving the
    numbers of successes and failures, respectively.}
  \item{n}{number of trials; ignored if \code{x} has length 2.}
  \item{p}{hypothesized probability of success.}
  \item{alternative}{indicates the alternative hypothesis and must be
    one of \code{"two.sided"}, \code{"greater"} or \code{"less"}.
    You can specify just the initial letter.}
  \item{conf.level}{confidence level for the returned confidence
    interval.}
}
\details{
  Confidence intervals are obtained by a procedure first given in
  Clopper and Pearson (1934).  This guarantees that the confidence level
  is at least \code{conf.level}, but in general does not give the
  shortest-length confidence intervals.
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the number of successes.}
  \item{parameter}{the number of trials.}
  \item{p.value}{the p-value of the test.}
  \item{conf.int}{a confidence interval for the probability of success.}
  \item{estimate}{the estimated probability of success.}
  \item{null.value}{the probability of success under the null,
    \code{p}.}
  \item{alternative}{a character string describing the alternative
    hypothesis.}
  \item{method}{the character string \code{"Exact binomial test"}.}
  \item{data.name}{a character string giving the names of the data.}
}
\references{
  Clopper, C. J. & Pearson, E. S. (1934).
  The use of confidence or fiducial limits illustrated in the case of
  the binomial.
  \emph{Biometrika}, \bold{26}, 404--413.
  
  William J. Conover (1971),
  \emph{Practical nonparametric statistics}.
  New York: John Wiley & Sons.
  Pages 97--104.

  Myles Hollander & Douglas A. Wolfe (1973),
  \emph{Nonparametric statistical inference}.
  New York: John Wiley & Sons.
  Pages 15--22.
}
\seealso{
  \code{\link{prop.test}} for a general (approximate) test for equal or
  given proportions.
}
\examples{
## Conover (1971), p. 97f.
## Under (the assumption of) simple Mendelian inheritance, a cross
##  between plants of two particular genotypes produces progeny 1/4 of
##  which are "dwarf" and 3/4 of which are "giant", respectively.
##  In an experiment to determine if this assumption is reasonable, a
##  cross results in progeny having 243 dwarf and 682 giant plants.
##  If "giant" is taken as success, the null hypothesis is that p =
##  3/4 and the alternative that p != 3/4.
binom.test(c(682, 243), p = 3/4)
binom.test(682, 682 + 243, p = 3/4)   # The same.
## => Data are in agreement with the null hypothesis.
}
\keyword{htest}
\eof
\name{biplot}
\alias{biplot}
\alias{biplot.default}
\title{Biplot of Multivariate Data}
\usage{
biplot(x, \dots)

\method{biplot}{default}(x, y, var.axes = TRUE, col, cex = rep(par("cex"), 2),
       xlabs = NULL, ylabs = NULL, expand = 1,
       xlim  = NULL, ylim  = NULL, arrow.len = 0.1,
       main = NULL, sub = NULL, xlab = NULL, ylab = NULL, \dots)
}
\arguments{
  \item{x}{The \code{biplot}, a fitted object. For \code{biplot.default},
    the first set of points (a two-column matrix), usually associated
    with observations.}
  \item{y}{The second set of points (a two-column matrix), usually associated
    with variables.}
  \item{var.axes}{If \code{TRUE} the second set of points have arrows
    representing them as (unscaled) axes.}
  \item{col}{A vector of length 2 giving the colours for the first and
    second set of points respectively (and the corresponding axes). If a
    single colour is specified it will be used for both sets.  If
    missing the default colour is looked for in the
    \code{\link{palette}}: if there it and the next colour as used,
    otherwise the first two colours of the paletter are used.}
  \item{cex}{The character expansion factor used for labelling the
    points. The labels can be of different sizes for the two sets by
    supplying a vector of length two.}
  \item{xlabs}{A vector of character strings to label the first set of
    points: the default is to use the row dimname of \code{x}, or
    \code{1:n} is the dimname is \code{NULL}.}
  \item{ylabs}{A vector of character strings to label the second set of
    points: the default is to use the row dimname of \code{y}, or
    \code{1:n} is the dimname is \code{NULL}.}
  \item{expand}{An expansion factor to apply when plotting the second set
    of points relative to the first. This can be used to tweak the
    scaling of the two sets to a physically comparable scale.}
  \item{arrow.len}{The length of the arrow heads on the axes plotted in
    \code{var.axes} is true. The arrow head can be suppressed by
    \code{arrow.len = 0}.}
  \item{xlim, ylim}{Limits for the x and y axes in the units of the
    first set of variables.}
  \item{main, sub, xlab, ylab, \dots}{graphical parameters.}
}
\description{
  Plot a biplot on the current graphics device.
}
\details{
  A biplot is plot which aims to represent both the observations and
  variables of a matrix of multivariate data on the same plot. There are
  many variations on biplots (see the references) and perhaps the most
  widely used one is implemented by \code{\link{biplot.princomp}}.
  The function \code{biplot.default} merely provides the
  underlying code to plot two sets of variables on the same figure.

  Graphical parameters can also be given to \code{biplot}.
}
\section{Side Effects}{
  a plot is produced on the current graphics device.
}
\references{
  K. R. Gabriel (1971).  The biplot graphical display of matrices with
  application to principal component analysis. \emph{Biometrika}
  \bold{58}, 453--467.

  J.C. Gower and D. J. Hand (1996). \emph{Biplots}. Chapman & Hall.
}
\seealso{\code{\link{biplot.princomp}}, also for examples.}
\keyword{hplot}
\keyword{multivariate}
\eof
\name{biplot.princomp}
\alias{biplot.princomp}
\alias{biplot.prcomp}
\title{
Biplot for Principal Components
}
\description{
  Produces a biplot (in the strict sense) from the output of
  \code{\link{princomp}} or \code{\link{prcomp}}
}
\usage{
\method{biplot}{prcomp}(x, choices = 1:2, scale = 1, pc.biplot = FALSE, \dots)
\method{biplot}{princomp}(x, choices = 1:2, scale = 1, pc.biplot = FALSE, \dots)
}
\arguments{
  \item{x}{an object of class \code{"princomp"}.}
  \item{choices}{
    length 2 vector specifying the components to plot. Only the default
    is a biplot in the strict sense.
  }
  \item{scale}{
    The variables are scaled by \code{lambda ^ scale} and the
    observations are scaled by \code{lambda ^ (1-scale)} where
    \code{lambda} are the singular values as computed by
    \code{\link{princomp}}. Normally \code{0 <= scale <= 1}, and a warning 
    will be issued if the specified \code{scale} is outside this range.
  }
  \item{pc.biplot}{
    If true, use what Gabriel (1971) refers to as a "principal component
    biplot", with \code{lambda = 1} and observations scaled up by sqrt(n) and
    variables scaled down by sqrt(n).  Then inner products between
    variables approximate covariances and distances between observations
    approximate Mahalanobis distance.
  }
  \item{\dots}{optional arguments to be passed to
    \code{\link{biplot.default}}.}
}
\section{Side Effects}{
    a plot is produced on the current graphics device.
}
\details{
    This is a method for the generic function \code{biplot}.  There is
    considerable confusion over the precise definitions: those of the
    original paper, Gabriel (1971), are followed here.  Gabriel and
    Odoroff (1990) use the same definitions, but their plots actually
    correspond to \code{pc.biplot = TRUE}.
}
\references{
    Gabriel, K. R. (1971).
    The biplot graphical display of matrices with applications to
    principal component analysis.
    \emph{Biometrika}, \bold{58}, 453--467.

    Gabriel, K. R. and Odoroff, C. L. (1990).
    Biplots in biomedical research.
    \emph{Statistics in Medicine}, \bold{9}, 469--485.
}
\seealso{
    \code{\link{biplot}},
    \code{\link{princomp}}.
}
\examples{
data(USArrests)
biplot(princomp(USArrests))
}
\keyword{multivariate}
\keyword{hplot}
\eof
\name{birthday}
\alias{qbirthday}
\alias{pbirthday}
\title{Probability of coincidences}
\description{
  Computes approximate answers to a generalised \dQuote{birthday paradox}
  problem. \code{pbirthday} computes the probability of a coincidence
  and \code{qbirthday} computes the number of observations needed to
  have a specified probability of coincidence.
}
\usage{
qbirthday(prob = 0.5, classes = 365, coincident = 2)
pbirthday(n, classes = 365, coincident = 2)
}
\arguments{
  \item{classes}{How many distinct categories the people could fall into}
  \item{prob}{The desired probability of coincidence}
  \item{n}{The number of people}
  \item{coincident}{The number of people to fall in the same category}
}
\value{
  \item{qbirthday}{
    Number of people needed for a probability \code{prob} that \code{k} of 
    them have the same one out of \code{classes} equiprobable labels.
  }

  \item{pbirthday}{Probability of the specified coincidence}  
}
\details{
 The birthday paradox is that a very small number of people, 23,
 suffices to have  a 50-50 chance that two of them have the same
 birthday.  This function generalises the calculation to probabilities
 other than 0.5, numbers of coincident events other than 2, and numbers
 of classes other than 365.

 This formula is approximate, as the example below shows.  For
 \code{coincident=2} the exact computation is straightforward and may be
 preferable. 
}
\references{
  Diaconis P, Mosteller F., \dQuote{Methods for studying coincidences}.
  JASA 84:853-861}
\examples{
 ## the standard version
qbirthday() 
 ## same 4-digit PIN number 
qbirthday(classes=10^4)
 ## 0.9 probability of three coincident birthdays
qbirthday(coincident=3,prob=0.9)
## Chance of 4 coincident birthdays in 150 people
pbirthday(150,coincident=4)
## Accuracy compared to exact calculation 
x1<- sapply(10:100, pbirthday)
x2<-1-sapply(10:100, function(n)prod((365:(365-n+1))/rep(365,n)))
par(mfrow=c(2,2))
plot(x1,x2,xlab="approximate",ylab="exact")
abline(0,1)
plot(x1,x1-x2,xlab="approximate",ylab="error")
abline(h=0)
plot(x1,x2,log="xy",xlab="approximate",ylab="exact")
abline(0,1)
plot(1-x1,1-x2,log="xy",xlab="approximate",ylab="exact")
abline(0,1)
}
\keyword{distribution}
\eof
\name{Box.test}
\alias{Box.test}
\title{Box-Pierce and Ljung-Box Tests}
\description{
  Compute the Box--Pierce or Ljung--Box test statistic for examining the
  null hypothesis of independence in a given time series.
}
\usage{
Box.test(x, lag = 1, type = c("Box-Pierce", "Ljung-Box"))
}
\arguments{
  \item{x}{a numeric vector or univariate time series.}
  \item{lag}{the statistic will be based on \code{lag} autocorrelation
    coefficients.}
  \item{type}{test to be performed: partial matching is used.}
}
\note{
  Missing values are not handled.
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the test statistic.}
  \item{parameter}{the degrees of freedom of the approximate chi-squared
    distribution of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{method}{a character string indicating which type of test was
    performed.}
  \item{data.name}{a character string giving the name of the data.}
}
\references{
  Box, G. E. P. and Pierce, D. A. (1970),
  Distribution of residual correlations in autoregressive-integrated
  moving average time series models.
  \emph{Journal of the American Statistical Association}, \bold{65},
  1509--1526.
    
  Ljung, G. M. and Box, G. E. P. (1978),
  On a measure of lack of fit in time series models.
  \emph{Biometrika} \bold{65}, 553--564.
  
  Harvey, A. C. (1993)
  \emph{Time Series Models}.
  2nd Edition, Harvester Wheatsheaf, NY, pp. 44, 45.
}
\author{A. Trapletti}
\examples{
x <- rnorm (100)
Box.test (x, lag = 1)
Box.test (x, lag = 1, type="Ljung")
}
\keyword{ts}
\eof
\name{cancor}
\title{Canonical Correlations}
\alias{cancor}
\usage{
cancor(x, y, xcenter = TRUE, ycenter = TRUE)
}
\arguments{
  \item{x}{numeric matrix (\eqn{n \times p_1}{n * p1}), containing the
    x coordinates.}
  \item{y}{numeric matrix (\eqn{n \times p_2}{n * p2}), containing the
    y coordinates.}
  \item{xcenter}{logical or numeric vector of length \eqn{p_1}{p1},
    describing any centering to be done on the x values before the
    analysis.  If \code{TRUE} (default), subtract the column means.
    If \code{FALSE}, do not adjust the columns.  Otherwise, a vector
    of values to be subtracted from the columns.} 
  \item{ycenter}{analogous to \code{xcenter}, but for the y values.}
}
\description{
  Compute the canonical correlations between two data matrices.
}
\details{
  The canonical correlation analysis seeks linear combinations of the
  \code{y} variables which are well explained by linear combinations
  of the \code{x} variables. The relationship is symmetric as
  \sQuote{well explained} is measured by correlations.
}
\value{
  A list containing the following components:
  \item{cor}{correlations.}
  \item{xcoef}{estimated coefficients for the \code{x} variables.}
  \item{ycoef}{estimated coefficients for the \code{y} variables.}
  \item{xcenter}{the values used to adjust the \code{x} variables.}
  \item{ycenter}{the values used to adjust the \code{x} variables.}
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.

  Hotelling H. (1936).
  Relations between two sets of variables.
  \emph{Biometrika}, \bold{28}, 321--327.
  
  Seber, G. A. F. (1984).
  \emph{Multivariate Observations}.
  New York: Wiley, p. 506f.
}
\seealso{
  \code{\link{qr}}, \code{\link{svd}}.
}
\examples{
data(LifeCycleSavings)
pop <- LifeCycleSavings[, 2:3]
oec <- LifeCycleSavings[, -(2:3)]
cancor(pop, oec)

x <- matrix(rnorm(150), 50, 3)
y <- matrix(rnorm(250), 50, 5)
(cxy <- cancor(x, y))
all(abs(cor(x \%*\% cxy$xcoef,
            y \%*\% cxy$ycoef)[,1:3] - diag(cxy $ cor)) < 1e-15)
all(abs(cor(x \%*\% cxy$xcoef) - diag(3)) < 1e-15)
all(abs(cor(y \%*\% cxy$ycoef) - diag(5)) < 1e-15)
}
\keyword{multivariate}
\eof
\name{case/variable.names}
\alias{case.names}
\alias{case.names.lm}
%\alias{case.names.default}
\alias{variable.names}
\alias{variable.names.lm}
%\alias{variable.names.default}
\title{Case and Variable Names of Fitted Models}
\description{
  Simple utilities returning (non-missing) case names, and
  (non-eliminated) variable names.
}
\usage{
case.names(object, \dots)
\method{case.names}{lm}(object, full = FALSE, \dots)

variable.names(object, \dots)
\method{variable.names}{lm}(object, full = FALSE, \dots)
}
\arguments{
 \item{object}{an \R object, typically a fitted model.}
 \item{full}{logical; if \code{TRUE}, all names (including zero weights,
   \dots) are returned.}
 \item{\dots}{further arguments passed to or from other methods.}
}
\value{
  A character vector.
}
\seealso{\code{\link{lm}}}
\examples{
x <- 1:20
y <-  x + (x/4 - 2)^3 + rnorm(20, s=3)
names(y) <- paste("O",x,sep=".")
ww <- rep(1,20); ww[13] <- 0
summary(lmxy <- lm(y ~ x + I(x^2)+I(x^3) + I((x-10)^2),
                   weights = ww), cor = TRUE)
variable.names(lmxy)
variable.names(lmxy, full= TRUE)# includes the last
case.names(lmxy)
case.names(lmxy, full = TRUE)# includes the 0-weight case
}
\keyword{regression}
\keyword{models}
\eof
\name{.checkMFClasses}
\alias{.checkMFClasses}
\alias{.MFclass}
\alias{.getXlevels}
\title{Functions to Check the Type of Variables passed to Model Frames}
\description{
  \code{.checkMFClasses} checks if the variables used in a predict
  method agree in type with those used for fitting.

  \code{.MFclass} categories variables for this purpose.
}
\usage{
.checkMFClasses(cl, m, ordNotOK = FALSE)
.MFclass(x)
.getXlevels(Terms, m)
}
\arguments{
  \item{cl}{a character vector of class descriptions to match.}
  \item{m}{a model frame.}
  \item{x}{any \R object.}
  \item{ordNotOK}{logical: are ordered factors different?}
  \item{Terms}{a \code{terms} object.}
}
\details{
  For applications involving \code{model.matrix} such as linear models
  we do not need to differentiate between ordered factors and factors as
  although these affect the coding, the coding used in the fit is
  already recorded and imposed during prediction.  However, other
  applications may treat ordered factors differently:
  \code{\link[rpart]{rpart}} does, for example.
}
\value{
  \code{.MFclass} returns a character string, one of \code{"logical"},
  \code{"ordered"}, \code{"factor"}, \code{"numeric"}, \code{"nmatrix.*"}
  (a numeric matrix with a number of columns appended) or \code{"other"}.

  \code{.getXlevels} returns a named character vector, or \code{NULL}.
}
\keyword{internal}
\eof
\name{chisq.test}
\alias{chisq.test}
\title{Pearson's Chi-squared Test for Count Data}
\description{
  \code{chisq.test} performs chi-squared tests on contingency tables.
}
\usage{
chisq.test(x, y = NULL, correct = TRUE,
           p = rep(1/length(x), length(x)),
           simulate.p.value = FALSE, B = 2000)
}
\arguments{
  \item{x}{a vector or matrix.}
  \item{y}{a vector; ignored if \code{x} is a matrix.}
  \item{correct}{a logical indicating whether to apply continuity
    correction when computing the test statistic.}
  \item{p}{a vector of probabilities of the same length of \code{x}.}
  \item{simulate.p.value}{a logical indicating whether to compute
    p-values by Monte Carlo simulation.}
  \item{B}{an integer specifying the number of replicates used in the
    Monte Carlo simulation.}
}
\details{
  If \code{x} is a matrix with one row or column, or if \code{x} is a
  vector and \code{y} is not given, \code{x} is treated as a
  one-dimensional contingency table.  In this case, the hypothesis
  tested is whether the population probabilities equal those in
  \code{p}, or are all equal if \code{p} is not given.

  If \code{x} is a matrix with at least two rows and columns, it is
  taken as a two-dimensional contingency table, and hence its entries
  should be nonnegative integers.  Otherwise, \code{x} and \code{y} must
  be vectors or factors of the same length; incomplete cases are
  removed, the objects are coerced into factor objects, and the
  contingency table is computed from these.  Then, Pearson's chi-squared
  test of the null that the joint distribution of the cell counts in a
  2-dimensional contingency table is the product of the row and column
  marginals is performed.  If \code{simulate.p.value} is \code{FALSE},
  the p-value is computed from the asymptotic chi-squared distribution
  of the test statistic; continuity correction is only used in the
  2-by-2 case if \code{correct} is \code{TRUE}.  Otherwise, if
  \code{simulate.p.value} is \code{TRUE}, the p-value is computed by
  Monte Carlo simulation with \code{B} replicates.  This is done by
  random sampling from the set of all contingency tables with given
  marginals, and works only if the marginals are positive.  (A C
  translation of the algorithm of Patefield (1981) is used.)
}
\value{
  A list with class \code{"htest"} containing the following
  components:
  \item{statistic}{the value the chi-squared test statistic.}
  \item{parameter}{the degrees of freedom of the approximate
    chi-squared distribution of the test statistic, \code{NA} if the
    p-value is computed by Monte Carlo simulation.}
  \item{p.value}{the p-value for the test.}
  \item{method}{a character string indicating the type of test
    performed, and whether Monte Carlo simulation or continuity
    correction was used.}
  \item{data.name}{a character string giving the name(s) of the data.}
  \item{observed}{the observed counts.}
  \item{expected}{the expected counts under the null hypothesis.}
  \item{residuals}{the Pearson residuals, \code{(observed - expected)
      / sqrt(expected)}.}
}
\references{
  Patefield, W. M. (1981)
  Algorithm AS159.  An efficient method of generating r x c tables
  with given row and column totals.
  \emph{Applied Statistics} \bold{30}, 91--97.
} 
\examples{
data(InsectSprays)              # Not really a good example
chisq.test(InsectSprays$count > 7, InsectSprays$spray)
                                # Prints test summary
chisq.test(InsectSprays$count > 7, InsectSprays$spray)$obs
                                # Counts observed
chisq.test(InsectSprays$count > 7, InsectSprays$spray)$exp
                                # Counts expected under the null

## Effect of simulating p-values
x <- matrix(c(12, 5, 7, 7), nc = 2)
chisq.test(x)$p.value           # 0.4233
chisq.test(x, simulate.p.value = TRUE, B = 10000)$p.value
                                # around 0.29!

## Testing for population probabilities
## Case A. Tabulated data
x <- c(A = 20, B = 15, C = 25)
chisq.test(x)
chisq.test(as.table(x))		# the same
## Case B. Raw data
x <- trunc(5 * runif(100))
chisq.test(table(x))            # NOT 'chisq.test(x)'!
}
\keyword{htest}
\keyword{distribution}
\eof
\name{clearNames}
\alias{clearNames}
\title{Remove the Names from an Object}
\usage{
clearNames(object)
}
\arguments{
 \item{object}{ an object that may have a \code{names} attribute }
}
\description{
 This function sets the \code{names} attribute of \code{object} to
 \code{NULL} and returns the object.
}
\value{
  An object similar to \code{object} but without names.
}
\author{ Douglas Bates and Saikat DebRoy}
\seealso{\code{\link{setNames}}}
\examples{
data( women )
lapply( women, mean )               # has a names attribute
clearNames( lapply( women, mean ) ) # removes the names
}
\keyword{list}
\eof
\name{cmdscale}
\alias{cmdscale}
\title{Classical (Metric) Multidimensional Scaling}
\usage{
cmdscale(d, k = 2, eig = FALSE, add = FALSE, x.ret = FALSE)
}
\description{
  Classical multidimensional scaling of a data matrix.
  Also known as \emph{principal coordinates analysis} (Gower, 1966).
}
\arguments{
  \item{d}{a distance structure such as that returned by \code{dist}
    or a full symmetric matrix containing the dissimilarities.}
  \item{k}{the dimension of the space which the data are to be
    represented in; must be in \eqn{\{1,2,\ldots,n-1\}}.}
  \item{eig}{indicates whether eigenvalues should be returned.}
  \item{add}{logical indicating if an additive constant \eqn{c*} should
    be computed, and added to the non-diagonal dissimilarites such that
    all \eqn{n-1} eigenvalues are non-negative.}
  \item{x.ret}{indicates whether the doubly centered symmetric distance
    matrix should be returned.}
}
\details{
  Multidimensional scaling takes a set of dissimilarities and returns a
  set of points such that the distances between the points are
  approximately equal to the dissimilarities.

  The functions \code{isoMDS} and \code{sammon} in package \pkg{MASS}
  provide alternative ordination techniques.

  When \code{add = TRUE}, an additive constant \eqn{c*} is computed, and
  the dissimilarities \eqn{d_{ij} + c*}{d[i,j] + c*} are used instead of
  the original \eqn{d_{ij}}{d[i,j]}'s.

  Whereas S (Becker \emph{et al.}, 1988) computes this constant using
  an approximation suggested by Torgerson, \R uses the analytical
  solution of Cailliez (1983), see also Cox and Cox (1994).
}
\value{
  If \code{eig = FALSE} and \code{x.ret = FALSE} (default), a matrix
  with \code{k} columns whose rows give the coordinates of the points
  chosen to represent the dissimilarities.

  Otherwise, a list containing the following components.
  \item{points}{a matrix with \code{k} columns whose rows give the
    coordinates of the points chosen to represent the dissimilarities.}
  \item{eig}{the \eqn{n-1} eigenvalues computed during the scaling process if
    \code{eig} is true.}
  \item{x}{the doubly centered distance matrix if \code{x.ret} is true.}
  \item{GOF}{a numeric vector of length 2, equal to say
    \eqn{(g_1,g_2)}{(g.1,g.2)}, where
    \eqn{g_i = (\sum_{j=1}^k \lambda_j)/ (\sum_{j=1}^n T_i(\lambda_j))}{%
         g.i = (sum{j=1..k} lambda[j]) / (sum{j=1..n} T.i(lambda[j]))}, where
       \eqn{\lambda_j}{lambda[j]} are the eigenvalues (sorted
       decreasingly),
       \eqn{T_1(v) = \left| v \right|}{T.1(v) = abs(v)}, and
       \eqn{T_2(v) = max( v, 0 )}{T.2(v) = max(v, 0)}.
     }
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.

  Cailliez, F. (1983)
  The analytical solution of the additive constant problem.
  \emph{Psychometrika} \bold{48}, 343--349.

  Cox, T. F. and Cox, M. A. A. (1994)
  \emph{Multidimensional Scaling}.
  Chapman and Hall.

  Gower, J. C. (1966)  
  Some distance properties of latent root and vector 
  methods used in multivariate analysis.  
  \emph{Biometrika} \bold{53}, 325--328.

  Mardia, K. V., Kent, J. T. and Bibby, J. M. (1979).  Chapter 14 of
  \emph{Multivariate Analysis}, London: Academic Press.

  Seber, G. A. F. (1984).
  \emph{Multivariate Observations}.
  New York: Wiley.

  Torgerson, W. S. (1958).
  \emph{Theory and Methods of Scaling}.
  New York: Wiley.
}
\seealso{
  \code{\link{dist}}.
  Also \code{\link[MASS]{isoMDS}} and \code{\link[MASS]{sammon}}
  in package \pkg{MASS}.
}
\examples{
data(eurodist)
loc <- cmdscale(eurodist)
x <- loc[,1]
y <- -loc[,2]
plot(x, y, type="n", xlab="", ylab="", main="cmdscale(eurodist)")
text(x, y, names(eurodist), cex=0.8)

cmdsE <- cmdscale(eurodist, k=20, add = TRUE, eig = TRUE, x.ret = TRUE)
str(cmdsE)
}
\keyword{multivariate}
\eof
\name{co2}
\docType{data}
\alias{co2}
\title{Mauna Loa Atmospheric CO2 Concentration}
\description{
  Atmospheric concentrations of CO\eqn{_2}{2} are expressed in parts per
  million (ppm) and reported in the preliminary 1997 SIO manometric mole
  fraction scale.
}
\usage{data(co2)}
\format{
  A time series of 468 observations; monthly from 1959 to 1997.
}
\details{
  The values for February, March and April of 1964 were missing and have
  been obtained by interpolating linearly between the values for January
  and May of 1964.
}
\source{
  Keeling, C. D. and  Whorf, T. P.,
  Scripps Institution of Oceanography (SIO),
  University of California,
  La Jolla, California USA 92093-0220.

  \url{ftp://cdiac.esd.ornl.gov/pub/maunaloa-co2/maunaloa.co2}.
}
\references{
  Cleveland, W. S. (1993)
  \emph{Visualizing Data}.
  New Jersey: Summit Press.
}
\examples{
data(co2)
plot(co2, ylab = expression("Atmospheric concentration of CO"[2]),
     las = 1)
title(main = "co2 data set")
}
\keyword{datasets}
\eof
\name{coef}
\title{Extract Model Coefficients}
\usage{
coef(object, \dots)
coefficients(object, \dots)
}
\alias{coef}
\alias{coefficients}
%\alias{coef.default}
%\alias{coef.lm}
%\alias{coef.glm}
%\alias{coef.aov}
%\alias{coef.listof}
\arguments{
  \item{object}{an object for which the extraction of model coefficients is
    meaningful.}
  \item{\dots}{other arguments.}
}
\value{
  Coefficients extracted from the model object \code{object}.
}
\description{
  \code{coef} is a generic function which extracts model coefficients
  from objects returned by modeling functions.  \code{coefficients} is
  an \emph{alias} for it.
}
\details{
  All object classes which are returned by model fitting functions
  should provide a \code{coef} method or use the default one.
  (Note that the method is \code{coef} and not \code{coefficients}.)

  Class \code{"aov"} has a \code{coef} method that does not report
  aliased coefficients (see \code{\link{alias}}).
}
\seealso{
  \code{\link{fitted.values}} and \code{\link{residuals}} for related methods;
  \code{\link{glm}}, \code{\link{lm}} for model fitting.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical Models in S}.
  Wadsworth \& Brooks/Cole.
}
\examples{
x <- 1:5; coef(lm(c(1:3,7,6) ~ x))
}
\keyword{regression}
\keyword{models}
\eof
\name{complete.cases}
\alias{complete.cases}
\title{Find Complete Cases}
\description{
  Return a logical vector indicating which cases are complete, i.e.,
  have no missing values.
}
\usage{
complete.cases(\dots)
}
\arguments{
  \item{\dots}{a sequence of vectors, matrices and data frames.}
}
\value{
  A logical vector specifying which observations/rows have no missing
  values across the entire sequence.
}
\seealso{
  \code{\link{is.na}},
  \code{\link{na.omit}},
  \code{\link{na.fail}}.
}
\examples{
data(airquality)
x <- airquality[, -1] # x is a regression design matrix
y <- airquality[,  1] # y is the corresponding response

stopifnot(complete.cases(y) != is.na(y))
ok <- complete.cases(x,y)
sum(!ok) # how many are not "ok" ?
x <- x[ok,]
y <- y[ok]
}
\keyword{NA}
\keyword{logic}
\eof
\name{confint}
\alias{confint}
%\alias{confint.lm}
\title{Confidence Intervals for Model Parameters}
\description{
  Computes confidence intervals for one or more parameters in a fitted
  model.  Base has a method for objects inheriting from class
  \code{"lm"}.
}
\usage{
confint(object, parm, level = 0.95, \dots)
}
\arguments{
\item{object}{
a fitted model object.
}
\item{parm}{
a specification of which parameters are to be given confidence
intervals, either a vector of numbers or a vector of names. If
missing, all parameters are considered.
}
\item{level}{
the confidence level required.
}
\item{\dots}{
additional argument(s) for methods
}}
\value{
A matrix (or vector) with columns giving lower and upper confidence
limits for each parameter. These will be labelled as (1-level)/2 and
1 - (1-level)/2 in \% (by default 2.5\% and 97.5\%).
}
\details{
  \code{confint} is a generic function with no default method.
  For objects of class \code{"lm"} the direct formulae based on t values
  are used.

  Package \pkg{MASS} contains methods for \code{"glm"} and \code{"nls"}
  objects.
}
\seealso{
  \code{\link[MASS]{confint.nls}}
}
\examples{
data(mtcars)
fit <- lm(100/mpg ~ disp + hp + wt + am, data=mtcars)
confint(fit)
confint(fit, "wt")
}
\keyword{models}
\eof
\name{constrOptim}
\alias{constrOptim}
\title{Linearly constrained optimisation}
\description{
Minimise a function subject to linear inequality constraints using an
adaptive barrier algorithm.  
}
\usage{
constrOptim(theta, f, grad, ui, ci, mu = 1e-04, control = list(),
            method = if(is.null(grad)) "Nelder-Mead" else "BFGS",
            outer.iterations = 100, outer.eps = 1e-05, \dots)
}
\arguments{
  \item{theta}{Starting value: must be in the feasible region.}
  \item{f}{Function to minimise.}
  \item{grad}{Gradient of \code{f}.}
  \item{ui}{Constraints (see below).}
  \item{ci}{Constraints (see below).}
  \item{mu}{(Small) tuning parameter.}
  \item{control}{Passed to \code{\link{optim}}.}
  \item{method}{Passed to \code{\link{optim}}.}
  \item{outer.iterations}{Iterations of the barrier algorithm.}
  \item{outer.eps}{Criterion for relative convergence of the barrier
    algorithm.}
  \item{\dots}{ Other arguments passed to \code{\link{optim}} }
}
\details{
  The feasible region is defined by \code{ui \%*\% theta - ci >= 0}. The
  starting value must be in the interior of the feasible region, but the
  minimum may be on the boundary.
  
  A logarithmic barrier is added to enforce the constraints and then
  \code{\link{optim}} is called. The barrier function is chosen so that
  the objective function should decrease at each outer iteration. Minima
  in the interior of the feasible region are typically found quite
  quickly, but a substantial number of outer iterations may be needed
  for a minimum on the boundary.

 The tuning parameter \code{mu} multiplies the barrier term. Its precise
  value is often relatively unimportant. As \code{mu} increases the
  augmented objective function becomes closer to the original objective
  function but also less smooth near the boundary of the feasible
  region.

  Any \code{optim} method that permits infinite values for the objective
  function may be used (currently all but "L-BFGS-B").  The gradient
  function must be supplied except with \code{method="Nelder-Mead"}.
  
  As with \code{optim}, the default is to minimise and maximisation can
  be performed by setting \code{control$fnscale} to a negative value.
  
}
\value{
  As for \code{\link{optim}}, but with two extra components:
  \code{barrier.value} giving the value of the barrier function at the
  optimum and \code{outer.iterations} gives the
  number of outer iterations (calls to \code{optim})
}

\references{
  K. Lange \emph{Numerical Analysis for Statisticians.} Springer
  2001, p185ff
}

\seealso{
  \code{\link{optim}}, especially \code{method="L-BGFS-B"} which
  does box-constrained optimisation.
}

\examples{
## from optim
fr <- function(x) {   ## Rosenbrock Banana function
    x1 <- x[1]
    x2 <- x[2]
    100 * (x2 - x1 * x1)^2 + (1 - x1)^2
}
grr <- function(x) { ## Gradient of 'fr'
    x1 <- x[1]
    x2 <- x[2]
    c(-400 * x1 * (x2 - x1 * x1) - 2 * (1 - x1),
       200 *      (x2 - x1 * x1))
}

optim(c(-1.2,1), fr, grr)
#Box-constraint, optimum on the boundary
constrOptim(c(-1.2,0.9), fr, grr, ui=rbind(c(-1,0),c(0,-1)), ci=c(-1,-1))
#  x<=0.9,  y-x>0.1
constrOptim(c(.5,0), fr, grr, ui=rbind(c(-1,0),c(1,-1)), ci=c(-0.9,0.1))


## Solves linear and quadratic programming problems
## but needs a feasible starting value
#
# from example(solve.QP) in 'quadprog'
# no derivative
fQP <- function(b) {-sum(c(0,5,0)*b)+0.5*sum(b*b)}
Amat       <- matrix(c(-4,-3,0,2,1,0,0,-2,1),3,3)
bvec       <- c(-8,2,0)
constrOptim(c(2,-1,-1), fQP, NULL, ui=t(Amat),ci=bvec)
# derivative
gQP <- function(b) {-c(0,5,0)+b}
constrOptim(c(2,-1,-1), fQP, gQP, ui=t(Amat), ci=bvec)

## Now with maximisation instead of minimisation
hQP <- function(b) {sum(c(0,5,0)*b)-0.5*sum(b*b)}
constrOptim(c(2,-1,-1), hQP, NULL, ui=t(Amat), ci=bvec,
            control=list(fnscale=-1))
}
\keyword{optimize}

\eof
\name{contrast}
\alias{contr.helmert}
\alias{contr.poly}
\alias{contr.sum}
\alias{contr.treatment}
\title{Contrast Matrices}
\description{
  Return a matrix of contrasts.
}
\usage{
contr.helmert(n, contrasts = TRUE)
contr.poly(n, scores = 1:n, contrasts = TRUE)
contr.sum(n, contrasts = TRUE)
contr.treatment(n, base = 1, contrasts = TRUE)
}
\arguments{
  \item{n}{a vector of levels for a factor, or the number of levels.}
  \item{contrasts}{a logical indicating whether contrasts should be
    computed.}
  \item{scores}{the set of values over which orthogonal polynomials are
    to be computed.}
  \item{base}{an integer specifying which group is considered the
    baseline group. Ignored if \code{contrasts} is \code{FALSE}.}
}
\details{
  These functions are used for creating contrast matrices for use in
  fitting analysis of variance and regression models.  The columns of
  the resulting matrices contain contrasts which can be used for coding
  a factor with \code{n} levels.  The returned value contains the
  computed contrasts.  If the argument \code{contrasts} is \code{FALSE}
  then a square indicator matrix is returned.

  \code{cont.helmert} returns Helmert contrasts, which contrast the
  second level with the first, the third with the average of the first
  two, and so on.  \code{contr.poly} returns contrasts based on
  orthogonal polynomials. \code{contr.sum} uses \dQuote{sum to zero
  contrasts}.

  \code{contr.treatment} contrasts each level with the baseline level
  (specified by \code{base}): the baseline level is omitted.  Note that
  this does not produce \dQuote{contrasts} as defined in the standard
  theory for linear models as they are not orthogonal to the constant.
}
\value{
  A matrix with \code{n} rows and \code{k} columns, with \code{k=n-1} if
  \code{contrasts} is \code{TRUE} and \code{k=n} if \code{contrasts} is
  \code{FALSE}.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical models.}
  Chapter 2 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{contrasts}},
  \code{\link{C}},
  and
  \code{\link{aov}},
  \code{\link{glm}},
  \code{\link{lm}}.
}
\examples{
(cH <- contr.helmert(4))
apply(cH, 2,sum) # column sums are 0!
crossprod(cH) # diagonal -- columns are orthogonal
contr.helmert(4, contrasts = FALSE) # just the 4 x 4 identity matrix

(cT <- contr.treatment(5))
all(crossprod(cT) == diag(4)) # TRUE: even orthonormal

(cP <- contr.poly(3)) # Linear and Quadratic
zapsmall(crossprod(cP), dig=15) # orthonormal up to fuzz
}
\keyword{design}
\keyword{regression}
\keyword{array}
\eof
\name{contrasts}
\alias{contrasts}
\alias{contrasts<-}
\title{Get and Set Contrast Matrices}
\description{
  Set and view the contrasts associated with a factor.
}
\usage{
contrasts(x, contrasts = TRUE)
contrasts(x, how.many) <- value
}
\arguments{
  \item{x}{a factor or a logical variable.}
  \item{contrasts}{logical. See Details.}
  \item{how.many}{How many contrasts should be made. Defaults to one
    less than the number of levels of \code{x}.  This need not be the
    same as the number of columns of \code{ctr}.}
  \item{value}{either a matrix whose columns give coefficients for
    contrasts in the levels of \code{x}, or the (quoted) name of a
    function which computes such matrices.}
}
\details{
  If contrasts are not set for a factor the default functions from
  \code{\link{options}("contrasts")} are used.

  A logical vector \code{x} is converted into a two-level factor with
  levels \code{c(FALSE, TRUE)} (regardless of which levels occur in the
  variable).

  The argument \code{contrasts} is ignored if \code{x} has a matrix
  \code{contrasts} attribute set.  Otherwise if \code{contrasts = TRUE}
  it is passed to a contrasts function such as
  \code{\link{contr.treatment}} and if \code{contrasts = FALSE}
  an identity matrix is returned.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical models.}
  Chapter 2 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{C}},
  \code{\link{contr.helmert}},
  \code{\link{contr.poly}},
  \code{\link{contr.sum}},
  \code{\link{contr.treatment}};
  \code{\link{glm}},
  \code{\link{aov}},
  \code{\link{lm}}.
}
\examples{
example(factor)
fff <- ff[, drop=TRUE]  # reduce to 5 levels.
contrasts(fff) # treatment contrasts by default
contrasts(C(fff, sum))
contrasts(fff, contrasts = FALSE) # the 5x5 identity matrix

contrasts(fff) <- contr.sum(5); contrasts(fff)  # set sum contrasts
contrasts(fff, 2) <- contr.sum(5); contrasts(fff)  # set 2 contrasts
# supply 2 contrasts, compute 2 more to make full set of 4.
contrasts(fff) <- contr.sum(5)[,1:2]; contrasts(fff)
}
\keyword{design}
\keyword{regression}
\eof
\name{cophenetic}
\alias{cophenetic}
\title{Cophenetic Distances for a Hierarchical Clustering}
\description{
  Computes the cophenetic distances for a hierarchical clustering.
}
\usage{
cophenetic(x)
}
\arguments{
  \item{x}{an object of class \code{\link{hclust}} or with a method
    for \code{\link{as.hclust}()} such as \code{\link[cluster]{agnes}}.}
}
\details{
  The cophenetic distance between two observations that have been
  clustered is defined to be the intergroup dissimilarity at which the
  two observations are first combined into a single cluster.
  Note that this distance has many ties and restrictions.

  It can be argued that a dendrogram is an appropriate summary of some
  data if the correlation between the original distances and the
  cophenetic distances is high.  Otherwise, it should simply be viewed as
  the description of the output of the clustering algorithm.
}
\value{
  An object of class \code{dist}.
}
\references{
  Sneath, P.H.A. and Sokal, R.R (1973)
  \emph{Numerical Taxonomy: The Principles and Practice of Numerical
    Classification}, p. 278 ff;
  Freeman, San Francisco.
}
\author{Robert Gentleman}

\seealso{\code{\link{dist}}, \code{\link{hclust}}}

\examples{
 data(USArrests)
 d1 <- dist(USArrests)
 hc <- hclust(d1, "ave")
 d2 <- cophenetic(hc)
 cor(d1,d2) # 0.7659

## Example from Sneath & Sokal, Fig. 5-29, p.279
d0 <- c(1,3.8,4.4,5.1, 4,4.2,5, 2.6,5.3, 5.4)
attributes(d0) <- list(Size = 5, diag=TRUE)
class(d0) <- "dist"
names(d0) <- letters[1:5]
d0
str(upgma <- hclust(d0, method = "average"))
plot(upgma, hang = -1)
#
(d.coph <- cophenetic(upgma))
cor(d0, d.coph) # 0.9911
}
\keyword{cluster}
\keyword{multivariate}

\eof
\name{cor}
\title{Correlation, Variance and Covariance (Matrices)}
\usage{
var(x, y = NULL, na.rm = FALSE, use)

cov(x, y = NULL, use = "all.obs",
    method = c("pearson", "kendall", "spearman"))

cor(x, y = NULL, use = "all.obs",
     method = c("pearson", "kendall", "spearman"))

cov2cor(V)
}
\alias{var}
\alias{cov}
\alias{cor}
\alias{cov2cor}
\description{
  \code{var}, \code{cov} and \code{cor} compute the variance of \code{x}
  and the covariance or correlation of \code{x} and \code{y} if these
  are vectors.	If \code{x} and \code{y} are matrices then the
  covariances (or correlations) between the columns of \code{x} and the
  columns of \code{y} are computed.

  \code{cov2cor} scales a covariance matrix into the corresponding
  correlation matrix \emph{efficiently}.
}
\arguments{
  \item{x}{a numeric vector, matrix or data frame.}
  \item{y}{\code{NULL} (default) or a vector, matrix or data frame with
    compatible dimensions to \code{x}.	The default is equivalent to
    \code{y = x} (but more efficient).}
  \item{na.rm}{logical. Should missing values be removed?}
  \item{use}{an optional character string giving a
    method for computing covariances in the presence
    of missing values.	This must be (an abbreviation of) one of the strings
    \code{"all.obs"}, \code{"complete.obs"} or \code{"pairwise.complete.obs"}.}
  \item{method}{a character string indicating which correlation
    coefficient (or covariance) is to be computed.  One of
    \code{"pearson"} (default), \code{"kendall"}, or \code{"spearman"},
    can be abbreviated.}
  \item{V}{symmetric numeric matrix, usually positive definite such as a
    covariance matrix.}
}
\value{For \code{r <- cor(*, use = "all.obs")}, it is now guaranteed that
  \code{all(r <= 1)}.
}
\details{
  For \code{cov} and \code{cor} one must \emph{either} give a matrix or
  data frame for \code{x} \emph{or} give both \code{x} and \code{y}.

  \code{var} is just another interface to \code{cov}, where
  \code{na.rm} is used to determine the default for \code{use} when that
  is unspecified.  If \code{na.rm} is \code{TRUE} then the complete
  observations (rows) are used (\code{use = "complete"}) to compute the
  variance.  Otherwise (\code{use = "all"}), \code{var} will give an
  error if there are missing values.

  If \code{use} is \code{"all.obs"}, then the presence
  of missing observations will produce an error.
  If \code{use} is \code{"complete.obs"} then missing values
  are handled by casewise deletion.  Finally, if \code{use} has the
  value \code{"pairwise.complete.obs"} then the correlation between
  each pair of variables is computed using all complete pairs
  of observations on those variables.
  This can result in covariance or correlation matrices which are not
  positive semidefinite.

  The denominator \eqn{n - 1} is used which gives an unbiased estimator
  of the (co)variance for i.i.d. observations.
  These functions return \code{\link{NA}} when there is only one
  observation (whereas S-PLUS has been returning \code{NaN}), and %from \R 1.2.3
  fail if \code{x} has length zero.

  For \code{cor()}, if \code{method} is \code{"kendall"} or
  \code{"spearman"}, Kendall's \eqn{\tau}{tau} or Spearman's
  \eqn{\rho}{rho} statistic is used to estimate a rank-based measure of
  association.  These are more robust and have be recommended if the
  data do not necessarily come from a bivariate normal distribution.\cr
  For \code{cov()}, a non-Pearson method is unusual but available for
  the sake of completeness.  Note that \code{"spearman"} basically
  computes \code{cor(R(x), R(y))} (or \code{cov(.,.)}) where
  \code{R(u) := rank(u, na.last="keep")}. Notice also that the ranking
  is (currently) done removing only cases that are missing on the
  variable itself, which may not be what you expect if you let
  \code{use} be \code{"complete.obs"} or \code{"pairwise.complete.obs"}.

  Scaling a covariance matrix into a correlation one can be achieved in
  many ways, mathematically most appealing by multiplication with a
  diagonal matrix from left and right, or more efficiently by using
  \code{\link{sweep}(.., FUN = "/")} twice.  The \code{cov2cor} function
  is even a bit more efficient, and provided mostly for didactical
  reasons.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{cor.test}} (package \pkg{ctest}) for confidence
  intervals (and tests).
  
  \code{\link{cov.wt}} for \emph{weighted} covariance computation.

  \code{\link{sd}} for standard deviation (vectors).
}
\examples{
var(1:10)# 9.166667

var(1:5,1:5)# 2.5

## Two simple vectors
cor(1:10,2:11)# == 1

## Correlation Matrix of Multivariate sample:
data(longley)
(Cl <- cor(longley))
## Graphical Correlation Matrix:
symnum(Cl) # highly correlated

## Spearman's rho  and  Kendall's tau
symnum(clS <- cor(longley, method = "spearman"))
symnum(clK <- cor(longley, method = "kendall"))
## How much do they differ?
i <- lower.tri(Cl)
cor(cbind(P = Cl[i], S = clS[i], K = clK[i]))


## cov2cor() scales a covariance matrix by its diagonal
##	     to become the correlation matrix.
cov2cor # see the function definition {and learn ..}
stopifnot(all.equal(Cl, cov2cor(cov(longley))),
          all.equal(cor(longley, method="kendall"),
            cov2cor(cov(longley, method="kendall"))))

##--- Missing value treatment:
data(swiss)
C1 <- cov(swiss)
range(eigen(C1, only=TRUE)$val) # 6.19	1921
swM <- swiss
swM[1,2] <- swM[7,3] <- swM[25,5] <- NA # create 3 "missing"
try(cov(swM)) # Error: missing obs...
C2 <- cov(swM, use = "complete")
range(eigen(C2, only=TRUE)$val) # 6.46	1930
C3 <- cov(swM, use = "pairwise")
range(eigen(C3, only=TRUE)$val) # 6.19	1938

(scM <- symnum(cor(swM, method = "kendall", use = "complete")))
## Kendall's tau doesn't change much: identical symnum codings!
identical(scM, symnum(cor(swiss, method = "kendall")))

all.equal(cov2cor(cov(swM, method = "kendall", use = "pairwise")),
                  cor(swM, method = "kendall", use = "pairwise"))
}
\keyword{univar}
\keyword{multivariate}
\keyword{array}
\eof
\name{cor.test}
\alias{cor.test}
\alias{cor.test.default}
\alias{cor.test.formula}
\concept{Kendall correlation coefficient}
\concept{Kendall's tau}
\concept{Pearson correlation coefficient}
\concept{Spearman correlation coefficient}
\concept{Spearman's rho}
\title{Test for Association/Correlation Between Paired Samples}
\description{
  Test for association between paired samples, using one of
  Pearson's product moment correlation coefficient,
  Kendall's \eqn{\tau}{tau} or Spearman's \eqn{\rho}{rho}.
}
\usage{
cor.test(x, \dots)

\method{cor.test}{default}(x, y,
         alternative = c("two.sided", "less", "greater"),
         method = c("pearson", "kendall", "spearman"),
         exact = NULL, conf.level = 0.95, \dots)

\method{cor.test}{formula}(formula, data, subset, na.action, \dots)
}
\arguments{
  \item{x, y}{numeric vectors of data values.  \code{x} and \code{y}
    must have the same length.}
  \item{alternative}{indicates the alternative hypothesis and must be
    one of \code{"two.sided"}, \code{"greater"} or \code{"less"}.  You
    can specify just the initial letter.  \code{"greater"} corresponds
    to positive association, \code{"less"} to negative association.}
  \item{method}{a character string indicating which correlation
    coefficient is to be  used for the test.  One of \code{"pearson"},
    \code{"kendall"}, or \code{"spearman"}, can be abbreviated.}
  \item{exact}{a logical indicating whether an exact p-value should be
    computed.  Only used for Kendall's \eqn{\tau}{tau}.
    See the Details for the meaning of \code{NULL} (the default).}
  \item{conf.level}{confidence level for the returned confidence
    interval.  Currently only used for the Pearson product moment
    correlation coefficient if there are at least 4 complete pairs of
    observations.}
  \item{formula}{a formula of the form \code{~ u + v}, where each of
    \code{u} and \code{v} are numeric variables giving the data values
    for one sample.  The samples must be of the same length.}
  \item{data}{an optional data frame containing the variables in the
    model formula.}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.  Defaults to
    \code{getOption("na.action")}.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the test statistic.}
  \item{parameter}{the degrees of freedom of the test statistic in the
    case that it follows a t distribution.}
  \item{p.value}{the p-value of the test.}
  \item{estimate}{the estimated measure of association, with name
    \code{"cor"}, \code{"tau"}, or \code{"rho"} correspoding
    to the method employed.}
  \item{null.value}{the value of the association measure under the
    null hypothesis, always \code{0}.}
  \item{alternative}{a character string describing the alternative
    hypothesis.}
  \item{method}{a character string indicating how the association was
    measured.}
  \item{data.name}{a character string giving the names of the data.}
  \item{conf.int}{a confidence interval for the measure of association.
    Currently only given for Pearson's product moment correlation
    coefficient in case of at least 4 complete pairs of observations.}
}
\details{
  The three methods each estimate the association between paired samples
  and compute a test of the value being zero.  They use different
  measures of association, all in the range \eqn{[-1, 1]} with \eqn{0}
  indicating no association.  These are sometimes referred to as tests
  of no \emph{correlation}, but that term is often confined to the
  default method.

  If \code{method} is \code{"pearson"}, the test statistic is based on
  Pearson's product moment correlation coefficient \code{cor(x, y)} and
  follows a t distribution with \code{length(x)-2} degrees of freedom
  if the samples follow independent normal distributions.  If there are
  at least 4 complete pairs of observation, an asymptotic confidence
  interval is given based on Fisher's Z transform.

  If \code{method} is \code{"kendall"} or \code{"spearman"}, Kendall's
  \eqn{\tau}{tau} or Spearman's \eqn{\rho}{rho} statistic is used to
  estimate a rank-based measure of association.  These tests may be used
  if the data do not necessarily come from a bivariate normal
  distribution.

  For Kendall's test, by default (if \code{exact} is NULL), an exact
  p-value is computed if there are less than 50 paired samples containing
  finite values and there are no ties.  Otherwise, the test statistic is
  the estimate scaled to zero mean and unit variance, and is approximately
  normally distributed.

  For Spearman's test, p-values are computed using algorithm AS 89.
}
\references{
  D. J. Best & D. E. Roberts (1975),
  Algorithm AS 89: The Upper Tail Probabilities of Spearman's
  \eqn{\rho}{rho}.
  \emph{Applied Statistics}, \bold{24}, 377--379.

  Myles Hollander & Douglas A. Wolfe (1973),
  \emph{Nonparametric statistical inference}.
  New York: John Wiley & Sons.
  Pages 185--194 (Kendall and Spearman tests).
}
\examples{
## Hollander & Wolfe (1973), p. 187f.
## Assessment of tuna quality.  We compare the Hunter L measure of
##  lightness to the averages of consumer panel scores (recoded as
##  integer values from 1 to 6 and averaged over 80 such values) in
##  9 lots of canned tuna.

x <- c(44.4, 45.9, 41.9, 53.3, 44.7, 44.1, 50.7, 45.2, 60.1)
y <- c( 2.6,  3.1,  2.5,  5.0,  3.6,  4.0,  5.2,  2.8,  3.8)

##  The alternative hypothesis of interest is that the
##  Hunter L value is positively associated with the panel score.

cor.test(x, y, method = "kendall", alternative = "greater")
## => p=0.05972

cor.test(x, y, method = "kendall", alternative = "greater",
         exact = FALSE) # using large sample approximation
## => p=0.04765

## Compare this to
cor.test(x, y, method = "spearm", alternative = "g")
cor.test(x, y,                    alternative = "g")

## Formula interface.
data(USJudgeRatings)
pairs(USJudgeRatings)
cor.test(~ CONT + INTG, data = USJudgeRatings)
}
\keyword{htest}
\eof
\name{cov.wt}
\alias{cov.wt}
\title{Weighted Covariance Matrices}
\usage{
cov.wt(x, wt = rep(1/nrow(x), nrow(x)), cor = FALSE, center = TRUE)
}
\description{
  Returns a list containing estimates of the weighted covariance matrix
  and the mean of the data, and optionally of the (weighted) correlation
  matrix.}
\arguments{
  \item{x}{a matrix or data frame.  As usual, rows are observations and
    columns are variables.}
  \item{wt}{a non-negative and non-zero vector of weights for each
    observation.  Its length must equal the number of rows of \code{x}.}
  \item{cor}{A logical indicating whether the estimated correlation
    weighted matrix will be returned as well.}
  \item{center}{Either a logical or a numeric vector specifying the
    centers to be used when computing covariances.  If \code{TRUE}, the
    (weighted) mean of each variable is used, if \code{FALSE}, zero is
    used.  If \code{center} is numeric, its length must equal the number
    of columns of \code{x}.}
}
\value{
  A list containing the following named components:
  \item{cov}{the estimated (weighted) covariance matrix}
  \item{center}{an estimate for the center (mean) of the data.}
  \item{n.obs}{the number of observations (rows) in \code{x}.}
  \item{wt}{the weights used in the estimation.  Only returned if given
    as an argument.}
  \item{cor}{the estimated correlation matrix.  Only returned if
    \code{cor} is \code{TRUE}.}
}
\details{
  The covariance matrix is divided by one minus the sum of squares of
  the weights, so if the weights are the default (\eqn{1/n}) the conventional
  unbiased estimate of the covariance matrix with divisor \eqn{(n - 1)}
  is obtained.  This differs from the behaviour in S-PLUS.
}
\seealso{\code{\link{cov}} and \code{\link{var}}.
}
\keyword{multivariate}
\eof
% file cpgram.Rd
% copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{cpgram}
\alias{cpgram}
\title{
  Plot Cumulative Periodogram
}
\description{
  Plots a cumulative periodogram.
}
\usage{
cpgram(ts, taper=0.1, main=
       paste("Series: ", deparse(substitute(ts))), ci.col="blue")
}
\arguments{
  \item{ts}{a univariate time series}
  \item{taper}{proportion tapered in forming the periodogram}
  \item{main}{main title}
  \item{ci.col}{colour for confidence band.}
}
\value{
  None.
}
\section{Side Effects}{
  Plots the cumulative periodogram in a square plot.
}
\author{B.D. Ripley}
\note{
  From package \pkg{MASS}.
}
\examples{
par(pty = "s", mfrow = c(1,2))
data(lh)
cpgram(lh)
lh.ar <- ar(lh, order.max = 9)
cpgram(lh.ar$resid, main = "AR(3) fit to lh")

data(UKLungDeaths)
cpgram(ldeaths)
}
\keyword{ts}
\keyword{hplot}
\eof
\name{cutree}
\alias{cutree}
\title{Cut a tree into groups of data}
\description{
  Cuts a tree, e.g., as resulting from \code{\link{hclust}}, into several
  groups either by specifying the desired number(s) of groups or the cut
  height(s).
}
\usage{
cutree(tree, k = NULL, h = NULL)
}
\arguments{
 \item{tree}{a tree as produced by \code{\link{hclust}}. \code{cutree()}
   only expects a list with components \code{merge}, \code{height}, and
   \code{labels}, of appropriate content each.}
 \item{k}{an integer scalar or vector with the desired number of groups}
 \item{h}{numeric scalar or vector with heights where the tree should
   be cut.}
 At least one of \code{k} or \code{h} must be specified, \code{k}
 overrides \code{h} if both are given.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\value{
  \code{cutree} returns a vector with group memberships if \code{k} or
  \code{h} are scalar, otherwise a matrix with group meberships is returned
  where each column corresponds to the elements of \code{k} or \code{h},
  respectively (which are also used as column names).
}
\seealso{
  \code{\link{hclust}}, \code{\link{dendrogram}} for cutting trees themselves.
}
\examples{
data(USArrests)

hc <- hclust(dist(USArrests))

cutree(hc, k=1:5)#k = 1 is trivial
cutree(hc, h=250)

## Compare the 2 and 3 grouping:
g24 <- cutree(hc, k = c(2,4))
table(g24[,"2"], g24[,"4"])
}
\keyword{multivariate}
\keyword{cluster}

\eof
\name{decompose}
\alias{decompose}
\alias{plot.decomposed.ts}
\title{
  Classical Seasonal Decomposition by Moving Averages
}
\description{
  Decompose a time series into seasonal, trend and irregular components
  using moving averages.  Deals with additive or multiplicative
  seasonal component.
}
\usage{
decompose(x, type = c("additive", "multiplicative"), filter = NULL)
}
\arguments{
  \item{x}{A time series.}
  \item{type}{The type of seasonal component.}
  \item{filter}{A vector of filter coefficients in reverse time order (as for
    AR or MA coefficients), used for filtering out the seasonal
    component. If \code{NULL}, a moving average with symmetric window is
  performed.}
}
\details{
The additive model used is:

Y[t] = T[t] + S[t] + e[t]

The multiplicative model used is:

Y[t] = T[t] * S[t] + e[t]
}
\note{
  The function \code{\link{stl}} provides a much more sophisticated
  decomposition.
}
\value{
  An object of class \code{"decomposed.ts"} with following components:
  \item{seasonal}{The seasonal component (i.e., the repeated seasonal figure)}
  \item{figure}{The estimated seasonal figure only}
  \item{trend}{The trend component}
  \item{random}{The remainder part}
  \item{type}{The value of \code{type}}
}
\author{
  David Meyer \email{david.meyer@ci.tuwien.ac.at}
}
\seealso{\code{\link{stl}}}

\examples{
data(co2)
m <- decompose(co2)
m$figure
plot(m)
}
\keyword{ts}
\eof
\name{delete.response}
\title{Modify Terms Objects}
\usage{
delete.response(termobj)
reformulate(termlabels, response = NULL)
drop.terms(termobj, dropx = NULL, keep.response = FALSE)
}
\alias{reformulate}
\alias{drop.terms}
\alias{delete.response}
\alias{[.terms}
\arguments{
  \item{termobj}{A \code{terms} object}
  \item{termlabels}{character vector giving the right-hand side of a
    model formula.}
  \item{response}{character string, symbol or call giving the left-hand side of
    a model formula.}
  \item{dropx}{vector of positions of variables to drop from the
    right-hand side of the model.}
  \item{keep.response}{Keep the response in the resulting object?}
}
\description{
  \code{delete.response} returns a \code{terms} object for the same
  model but with no response variable.

  \code{drop.terms} removes variables from the right-hand side of the
  model. There is also a \code{"[.terms"} method to perform the same
  function (with \code{keep.response=TRUE}).

  \code{reformulate} creates a formula from a character vector.
}
\value{
  \code{delete.response} and \code{drop.terms} return a \code{terms}
  object.

  \code{reformulate} returns a \code{formula}.
}
\seealso{\code{\link{terms}}}

\examples{
ff <- y ~ z + x + w
tt <- terms(ff)
tt
delete.response(tt)
drop.terms(tt, 2:3, keep.response = TRUE)
tt[-1]
tt[2:3]
reformulate(attr(tt, "term.labels"))

## keep LHS :
reformulate("x*w", ff[[2]])
fS <- surv(ft, case) ~ a + b
reformulate(c("a", "b*f"), fS[[2]])

stopifnot(identical(      ~ var, reformulate("var")),
          identical(~ a + b + c, reformulate(letters[1:3])),
          identical(  y ~ a + b, reformulate(letters[1:2], "y"))
         )
}
\keyword{programming}
\eof
\name{dendrogram}
\title{General Tree Structures}
\alias{dendrogram}% the class
\alias{as.dendrogram}
\alias{as.dendrogram.hclust}
\alias{cut.dendrogram}
\alias{[[.dendrogram}
\alias{print.dendrogram}
\alias{rev.dendrogram}
\alias{str.dendrogram}
\alias{plot.dendrogram}
%\alias{plotNode}
%\alias{plotNodeLimit}
%\alias{isLeaf}
\description{
  Class \code{"dendrogram"} provides general functions for handling
  tree-like structures.  It is intended as a replacement for similar
  functions in hierarchical clustering and classification/regression
  trees, such that all of these can use the same engine for plotting or
  cutting trees.

  The code is still in testing stage and the API may
  change in the future.
}
\usage{
as.dendrogram(object, \dots)
\method{as.dendrogram}{hclust}(object, hang = -1, \dots)

\method{plot}{dendrogram}(x, type = c("rectangle", "triangle"),
      center = FALSE, edge.root = isLeaf(x) || !is.null(attr(x,"edgetext")),
      nodePar = NULL, edgePar = list(), xlab = "", ylab = "",
      horiz = FALSE, frame.plot = FALSE, \dots)

\method{cut}{dendrogram}(x, h, \dots)

\method{print}{dendrogram}(x, digits, \dots)

\method{rev}{dendrogram}(x)

\method{str}{dendrogram}(object, max.level = 0, digits.d = 3,
    give.attr = FALSE, wid = getOption("width"),
    nest.lev = 0, indent.str = "", \dots)
}
\arguments{
  \item{object}{any \R object that can be made into one of class
    \code{"dendrogram"}.}
  \item{x}{object of class \code{"dendrogram"}.}
  \item{hang}{numeric scalar indicating how the \emph{height} of leaves
    should be computed from the heights of their parents; see
    \code{\link{plot.hclust}}.}
  \item{type}{type of plot.}
  \item{center}{logical; if \code{TRUE}, nodes are plotted centered with
    respect to the leaves in the branch.  Otherwise (default), plot them
    in the middle of all direct child nodes.}
  \item{edge.root}{logical; if true, draw an edge to the root node.}
  \item{nodePar}{a \code{list} of plotting parameters to use for the
    nodes (see \code{\link{points}}) or \code{NULL} by default which
    does not draw symbols at the nodes.  The list may contain components
    named \code{pch}, \code{cex}, \code{col}, and/or \code{bg} each of
    which can have length two for specifying separate attributes for
    \emph{inner} nodes and \emph{leaves}.}
  \item{edgePar}{a \code{list} of plotting parameters to use for the
    edge (see \code{\link{lines}}).  The list may contain components
    named \code{col}, \code{lty} and/or \code{lwd}.}
  \item{horiz}{logical indicating if the dendrogram should be draw
    \emph{horizontally} or not.}
  \item{frame.plot}{logical indicating if a box around the plot should
    be drawn, see \code{\link{plot.default}}.}
  \item{h}{height at which the tree is cut.}
  \item{\dots, xlab, ylab}{graphical parameters, or arguments for
    other methods.}
  \item{digits}{integer specifiying the precision for printing, see
    \code{\link{print.default}}.}
  \item{max.level, digits.d, give.attr, wid, nest.lev, indent.str}{%
    arguments to \code{str}, see \code{\link{str.default}()}.  Note that
    by default \code{give.attr = FALSE}, it still shows \code{height}
    and \code{members} attributes for each node.}
}
\details{
  Warning: This documentation is preliminary.

  The dendrogram is directly represented as a nested list where each
  component corresponds to a branch of the tree.  Hence, the first
  branch of tree \code{z} is \code{z[[1]]}, the second branch of the
  corresponding subtree is \code{z[[1]][[2]]} etc.. Each node of the tree
  carries some information needed for efficient plotting or cutting as
  attributes:
  \describe{
    \item{\code{members}}{total number of leaves in the branch}
    \item{\code{height}}{numeric non-negative height at which the node
      is plotted.}
    \item{\code{midpoint}}{numeric horizontal distance of the node from
      the left border (the leftmost leaf) of the branch (unit 1 between
      all leaves).  This is used for \code{plot(*, center=FALSE)}.}
    \item{\code{label}}{character; the label of the node}
    \item{\code{edgetext}}{character; the label for the edge leading to
      the node}
    \item{\code{nodePar}}{a named list of length one vectors specifying
      node-specific attributes for \code{\link{points}} plotting, see
      the \code{nodePar} argument above.}
    \item{\code{edgePar}}{a named list of length one vectors specifying
      attributes for \code{\link{segments}} plotting of the edge leading
      to the node, see the \code{edgePar} argument above.}
    \item{\code{leaf}}{logical, if \code{TRUE}, the node is a leaf of
      the tree.}%  This will often be a \code{\link{character}} which can
    %      be used for plotting instead of the \code{text} attribute.}
  }

  \code{cut.dendrogram()} returns a list with components \code{$upper}
  and \code{$lower}, the first is a truncated version of the original
  tree, also of class \code{dendrogram}, the latter a list with the
  branches obtained from cutting the tree, each a \code{dendrogram}.

  There are \code{\link{[[}}, \code{\link{print}}, and \code{\link{str}}
  methods for \code{"dendrogram"} objects where the first one
  (extraction) ensures that selecting sub-branches keeps the class.

  Objects of class \code{"hclust"} can be converted to class
  \code{"dendrogram"} using method \code{as.dendrogram}.

  \code{isLeaf()}, \code{plotNode()} and \code{plotNodeLimit()} are helper
  functions.
}
\note{
  When using \code{type = "triangle"}, \code{center = TRUE} often looks
  better.
}
\examples{
data(USArrests)
hc <- hclust(dist(USArrests), "ave")
(dend1 <- as.dendrogram(hc)) # "print()" method
str(dend1) # "str()" method

op <- par(mfrow= c(2,2), mar = c(3,3,1,1))
plot(dend1)
## "triangle" type and show inner nodes:
plot(dend1, nodePar=list(pch = c(1,NA),cex=0.8), type = "t", center=TRUE)
plot(dend1, edgePar=list(col = 1:2, lty = 2:3), edge.root = TRUE)
plot(dend1, nodePar=list(pch = 2:1,cex=.4*2:1, col = 2:3), horiz = TRUE)

dend2 <- cut(dend1, h=70)
plot(dend2$upper)
## leafs are wrong horizontally:%% FIXME ?!?
plot(dend2$upper, nodePar=list(pch = c(1,7), col = 2:1))
##  dend2$lower is *NOT* a dendrogram, but a list of .. :
plot(dend2$lower[[3]], nodePar=list(col=4), horiz = TRUE, type = "tr")
## "inner" and "leaf" edges in different type & color :
plot(dend2$lower[[2]], nodePar=list(col=1),# non empty list
     edgePar = list(lty=1:2, col=2:1), edge.root=TRUE)
par(op)
}
\keyword{multivariate}
\keyword{tree}% FIXME: want as.dendrogram.tree() etc!
\keyword{hplot}% only for plot.()
\eof
\name{density}
\alias{density}
\alias{print.density}
\title{Kernel Density Estimation}
\usage{
density(x, bw = "nrd0", adjust = 1,
        kernel = c("gaussian", "epanechnikov", "rectangular", "triangular",
                   "biweight", "cosine", "optcosine"),
        window = kernel, width,
        give.Rkern = FALSE,
        n = 512, from, to, cut = 3, na.rm = FALSE)
}
\arguments{
  \item{x}{the data from which the estimate is to be computed.}

  \item{bw}{the smoothing bandwidth to be used.  The kernels are scaled
    such that this is the standard deviation of the smoothing kernel.
    (Note this differs from the reference books cited below, and from S-PLUS.)

    \code{bw} can also be a character string giving a rule to choose the
    bandwidth.  See \code{\link{bw.nrd}}.
  
    The specified (or computed) value of \code{bw} is multiplied by
    \code{adjust}.
  }
  \item{adjust}{the bandwidth used is actually \code{adjust*bw}.
    This makes it easy to specify values like \dQuote{half the default}
    bandwidth.}

  \item{kernel, window}{a character string giving the smoothing kernel
    to be used. This must be one of \code{"gaussian"},
    \code{"rectangular"}, \code{"triangular"}, \code{"epanechnikov"},
    \code{"biweight"}, \code{"cosine"} or \code{"optcosine"}, with default
    \code{"gaussian"}, and may be abbreviated to a unique prefix (single
    letter).
    
    \code{"cosine"} is smoother than \code{"optcosine"}, which is the
    usual \dQuote{cosine} kernel in the literature and almost MSE-efficient.
    However, \code{"cosine"} is the version used by S.
  }
  
  \item{width}{this exists for compatibility with S; if given, and
    \code{bw} is not, will set \code{bw} to \code{width} if this is a
    character string, or to a kernel-dependent multiple of \code{width}
    if this is numeric.}

  \item{give.Rkern}{logical; if true, \emph{no} density is estimated, and
    the \dQuote{canonical bandwidth} of the chosen \code{kernel} is returned
    instead.}

  \item{n}{the number of equally spaced points at which the density
    is to be estimated.  When \code{n > 512}, it is rounded up to the next
    power of 2 for efficiency reasons (\code{\link{fft}}).}

  \item{from,to}{the left and right-most points of the grid at which the
    density is to be estimated.}

  \item{cut}{by default, the values of \code{left} and \code{right} are
    \code{cut} bandwidths beyond the extremes of the data. This allows the
    estimated density to drop to approximately zero at the extremes.}

  \item{na.rm}{logical; if \code{TRUE}, missing values are removed
    from \code{x}. If \code{FALSE} any missing values cause an error.}
}
\description{
  The function \code{density} computes kernel density estimates
  with the given kernel and bandwidth.
}
\details{
  The algorithm used in \code{density} disperses the mass of the
  empirical distribution function over a regular grid of at least 512
  points and then uses the fast Fourier transform to convolve this
  approximation with a discretized version of the kernel and then uses
  linear approximation to evaluate the density at the specified points.

  The statistical properties of a kernel are determined by
  \eqn{\sigma^2_K = \int t^2 K(t) dt}{sig^2 (K) = int(t^2 K(t) dt)}
  which is always \eqn{= 1} for our kernels (and hence the bandwidth
  \code{bw} is the standard deviation of the kernel) and
  \eqn{R(K) = \int K^2(t) dt}{R(K) = int(K^2(t) dt)}.\cr
  MSE-equivalent bandwidths (for different kernels) are proportional to
  \eqn{\sigma_K R(K)}{sig(K) R(K)} which is scale invariant and for our
  kernels equal to \eqn{R(K)}.  This value is returned when
  \code{give.Rkern = TRUE}.  See the examples for using exact equivalent
  bandwidths.

  Infinite values in \code{x} are assumed to correspond to a point mass at
  \code{+/-Inf} and the density estimate is of the sub-density on
  \code{(-Inf, +Inf)}.
}
\value{
  If \code{give.Rkern} is true, the number \eqn{R(K)}, otherwise
  an object with class \code{"density"} whose
  underlying structure is a list containing the following components.
  \item{x}{the \code{n} coordinates of the points where the density is
    estimated.}
  \item{y}{the estimated density values.}
  \item{bw}{the bandwidth used.}
  \item{N}{the sample size after elimination of missing values.}
  \item{call}{the call which produced the result.}
  \item{data.name}{the deparsed name of the \code{x} argument.}
  \item{has.na}{logical, for compatibility (always \code{FALSE}).}
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole (for S version).

  Scott, D. W. (1992)
  \emph{Multivariate Density Estimation. Theory, Practice and Visualization}.
  New York: Wiley.

  Sheather, S. J. and Jones M. C. (1991)
  A reliable data-based bandwidth selection method for kernel density
  estimation.
  \emph{J. Roy. Statist. Soc.} \bold{B}, 683--690.

  Silverman, B. W. (1986)
  \emph{Density Estimation}.
  London: Chapman and Hall.

  Venables, W. N. and Ripley, B. D. (1999)
  \emph{Modern Applied Statistics with S-PLUS}.
  New York: Springer.
}
\seealso{
  \code{\link{bw.nrd}},
  \code{\link{plot.density}}, \code{\link{hist}}.
}
\examples{
plot(density(c(-20,rep(0,98),20)), xlim = c(-4,4))# IQR = 0

# The Old Faithful geyser data
data(faithful)
d <- density(faithful$eruptions, bw = "sj")
d
plot(d)

plot(d, type = "n")
polygon(d, col = "wheat")

## Missing values:
x <- xx <- faithful$eruptions
x[i.out <- sample(length(x), 10)] <- NA
doR <- density(x, bw = 0.15, na.rm = TRUE)
lines(doR, col = "blue")
points(xx[i.out], rep(0.01, 10))


(kernels <- eval(formals(density)$kernel))

## show the kernels in the R parametrization
plot (density(0, bw = 1), xlab = "",
      main="R's density() kernels with bw = 1")
for(i in 2:length(kernels))
   lines(density(0, bw = 1, kern =  kernels[i]), col = i)
legend(1.5,.4, legend = kernels, col = seq(kernels),
       lty = 1, cex = .8, y.int = 1)

## show the kernels in the S parametrization
plot(density(0, from=-1.2, to=1.2, width=2, kern="gaussian"), type="l",
     ylim = c(0, 1), xlab="", main="R's density() kernels with width = 1")
for(i in 2:length(kernels))
   lines(density(0, width=2, kern =  kernels[i]), col = i)
legend(0.6, 1.0, legend = kernels, col = seq(kernels), lty = 1)

(RKs <- cbind(sapply(kernels, function(k)density(kern = k, give.Rkern = TRUE))))
100*round(RKs["epanechnikov",]/RKs, 4) ## Efficiencies

if(interactive()) {
data(precip)
bw <- bw.SJ(precip) ## sensible automatic choice
plot(density(precip, bw = bw, n = 2^13),
     main = "same sd bandwidths, 7 different kernels")
for(i in 2:length(kernels))
   lines(density(precip, bw = bw, kern =  kernels[i], n = 2^13), col = i)

## Bandwidth Adjustment for "Exactly Equivalent Kernels"
h.f <- sapply(kernels, function(k)density(kern = k, give.Rkern = TRUE))
(h.f <- (h.f["gaussian"] / h.f)^ .2)
## -> 1, 1.01, .995, 1.007,... close to 1 => adjustment barely visible..

plot(density(precip, bw = bw, n = 2^13),
     main = "equivalent bandwidths, 7 different kernels")
for(i in 2:length(kernels))
   lines(density(precip, bw = bw, adjust = h.f[i], kern =  kernels[i],
         n = 2^13), col = i)
legend(55, 0.035, legend = kernels, col = seq(kernels), lty = 1)
}
}
\keyword{distribution}
\keyword{smooth}
\eof
\name{deriv}
\alias{D}
\alias{deriv}
\alias{deriv.default}
\alias{deriv.formula}
\alias{deriv3}
\alias{deriv3.default}
\alias{deriv3.formula}
\title{Symbolic and Algorithmic Derivatives of Simple Expressions}
\description{
  Compute derivatives of simple expressions, symbolically.
}
\synopsis{
D(expr, name)
deriv(expr, ...)
deriv.default(expr, namevec, function.arg = NULL, tag = ".expr", hessian = FALSE, ...)
deriv.formula(expr, namevec, function.arg = NULL, tag = ".expr", hessian = FALSE, ...)
deriv3(expr, ...)
deriv3.default(expr, namevec, function.arg = NULL, tag = ".expr", hessian = TRUE, ...)
deriv3.formula(expr, namevec, function.arg = NULL, tag = ".expr", hessian = TRUE, ...)
}
\usage{
    D (expr, name)
 deriv(expr, namevec, function.arg, tag = ".expr", hessian = FALSE)
deriv3(expr, namevec, function.arg, tag = ".expr", hessian = TRUE)
}
\arguments{
  \item{expr}{\code{\link{expression}} or \code{\link{call}} to
    be differentiated.}
  \item{name,namevec}{character vector, giving the variable names (only
    one for \code{D()}) with respect to which derivatives will be
    computed.}
  \item{function.arg}{If specified, a character vector of arguments for
    a function return, or a function (with empty body) or \code{TRUE},
    the latter indicating that a function with argument names
    \code{namevec} should be used.}
  \item{tag}{character; the prefix to be used for the locally created
    variables in result.}
  \item{hessian}{a logical value indicating whether the second derivatives
    should be calculated and incorporated in the return value.}
}
\details{
  \code{D} is modelled after its S namesake for taking simple symbolic
  derivatives.

  \code{deriv} is a \emph{generic} function with a default and a
  \code{\link{formula}} method.  It returns a \code{\link{call}} for
  computing the \code{expr} and its (partial) derivatives,
  simultaneously.  It uses so-called \dQuote{\emph{algorithmic
      derivatives}}.  If \code{function.arg} is a function,
  its arguments can have default values, see the \code{fx} example below.

  Currently, \code{deriv.formula} just calls \code{deriv.default} after
  extracting the expression to the right of \code{~}.

  \code{deriv3} and its methods are equivalent to \code{deriv} and its
  methods except that \code{hessian} defaults to \code{TRUE} for
  \code{deriv3}.
}
\value{
  \code{D} returns a call and therefore can easily be iterated
  for higher derivatives.

  \code{deriv} and \code{deriv3} normally return an
  \code{\link{expression}} object whose evaluation returns the function
  values with a \code{"gradient"} attribute containing the gradient
  matrix.  If \code{hessian} is \code{TRUE} the evaluation also returns
  a \code{"hessian"} attribute containing the Hessian array.
  
  If \code{function.arg} is specified, \code{deriv} and \code{deriv3}
  return a function with those arguments rather than an expression.
}
\references{
  Griewank, A.  and  Corliss, G. F. (1991)
  \emph{Automatic Differentiation of Algorithms: Theory, Implementation,
    and Application}.
  SIAM proceedings, Philadelphia.

  Bates, D. M. and Chambers, J. M. (1992)
  \emph{Nonlinear models.}
  Chapter 10 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{nlm}} and \code{\link{optim}} for numeric minimization
  which could make use of derivatives,
}
\examples{
## formula argument :
dx2x <- deriv(~ x^2, "x") ; dx2x
\dontrun{expression({
         .value <- x^2
         .grad <- array(0, c(length(.value), 1), list(NULL, c("x")))
         .grad[, "x"] <- 2 * x
         attr(.value, "gradient") <- .grad
         .value
})}
mode(dx2x)
x <- -1:2
eval(dx2x)

## Something 'tougher':
trig.exp <- expression(sin(cos(x + y^2)))
( D.sc <- D(trig.exp, "x") )
all.equal(D(trig.exp[[1]], "x"), D.sc)

( dxy <- deriv(trig.exp, c("x", "y")) )
y <- 1
eval(dxy)
eval(D.sc)

## function returned:
deriv((y ~ sin(cos(x) * y)), c("x","y"), func = TRUE)

## function with defaulted arguments:
(fx <- deriv(y ~ b0 + b1 * 2^(-x/th), c("b0", "b1", "th"),
             function(b0, b1, th, x = 1:7){} ) )
fx(2,3,4)

## Higher derivatives
deriv3(y ~ b0 + b1 * 2^(-x/th), c("b0", "b1", "th"),
     c("b0", "b1", "th", "x") )

## Higher derivatives:
DD <- function(expr,name, order = 1) {
   if(order < 1) stop("'order' must be >= 1")
   if(order == 1) D(expr,name)
   else DD(D(expr, name), name, order - 1)
}
DD(expression(sin(x^2)), "x", 3)
## showing the limits of the internal "simplify()" :
\dontrun{
-sin(x^2) * (2 * x) * 2 + ((cos(x^2) * (2 * x) * (2 * x) + sin(x^2) *
    2) * (2 * x) + sin(x^2) * (2 * x) * 2)
}
}
\keyword{math}
\keyword{nonlinear}


\eof
\name{deviance}
\alias{deviance}
%\alias{deviance.default}
%\alias{deviance.lm}
%\alias{deviance.glm}
%\alias{deviance.mlm}
\title{Model Deviance}
\description{
  Returns the deviance of a fitted model object.
}
\usage{
deviance(object, \dots)
}
\arguments{
  \item{object}{an object for which the deviance is desired.}
  \item{\dots}{additional optional argument.}
}
\details{
  This is a generic function which can be used to extract deviances for
  fitted models.  Consult the individual modeling functions for details
  on how to use this function.
}
\value{
  The value of the deviance extracted from the object \code{object}.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical Models in S.}
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{df.residual}},
  \code{\link{extractAIC}},
  \code{\link{glm}},
  \code{\link{lm}}.
}
\keyword{models}
\eof
\name{df.residual}
\alias{df.residual}
%\alias{df.residual.default}
\title{Residual Degrees-of-Freedom}
\description{
  Returns the residual degrees-of-freedom extracted from a fitted model
  object.
}
\usage{
df.residual(object, \dots)
}
\arguments{
  \item{object}{an object for which the degrees-of-freedom are desired.}
  \item{\dots}{additional optional arguments.}
}
\details{
  This is a generic function which can be used to extract residual
  degrees-of-freedom for fitted models.  Consult the individual modeling
  functions for details on how to use this function.

  The default method just extracts the \code{df.residual} component.
}
\value{
  The value of the residual degrees-of-freedom extracted from the object
  \code{x}.
}
\seealso{
  \code{\link{deviance}}, \code{\link{glm}}, \code{\link{lm}}.
}
\keyword{models}
\keyword{regression}
\eof
\name{diffinv}
\alias{diffinv}
\alias{diffinv.default}
\alias{diffinv.ts}
\title{Discrete Integration: Inverse of Differencing}
\description{
  Computes the inverse function of the lagged differences function
  \code{\link{diff}}. 
}
\synopsis{
diffinv(x, ...)
diffinv.default(x, lag = 1, differences = 1,
    xi = rep(0, lag * differences * dim(as.matrix(x))[2]), ...) 
diffinv.ts(x, lag = 1, differences = 1,
    xi = rep(0, lag * differences * NCOL(x)), ...) 
}
\usage{
diffinv(x, lag = 1, differences = 1,
        xi = rep(0.0, lag*differences*NCOL(x)), \dots)
}
\arguments{
  \item{x}{a numeric vector, matrix, or time series.}
  \item{lag}{a scalar lag parameter.}
  \item{differences}{an integer representing the order of the
    difference.}
  \item{xi}{a numeric vector, matrix, or time series containing the
    initial values for the integrals.}
  \item{\dots}{arguments passed to or from other methods.}
}
\details{
  \code{diffinv} is a generic function with methods for class \code{"ts"}
  and \code{default} for vectors and matrices.

  Missing values are not handled.
}
\value{
  A numeric vector, matrix, or time series representing the discrete
  integral of \code{x}. 
}
\author{A. Trapletti}
\seealso{
  \code{\link{diff}}
}
\examples{
s <- 1:10
d <- diff(s)
diffinv(d, xi = 1)
}
\keyword{ts}
\eof
\name{discoveries}
\docType{data}
\alias{discoveries}
\title{Yearly Numbers of Important Discoveries}
\description{
  The numbers of \dQuote{great} inventions and scientific
  discoveries in each year from 1860 to 1959.
}
\usage{data(discoveries)}
\format{A time series of 100 values.}
\source{
  The World Almanac and Book of Facts, 1975 Edition, pages 315--318.
}
\references{
  McNeil, D. R. (1977)
  \emph{Interactive Data Analysis}.
  Wiley.
}
\examples{
data(discoveries)
plot(discoveries, ylab = "Number of important discoveries",
     las = 1)
title(main = "discoveries data set")
}
\keyword{datasets}
\eof
\name{dist}
\title{Distance Matrix Computation}
\usage{
dist(x, method = "euclidean", diag = FALSE, upper = FALSE, p = 2)

as.dist(m, diag = FALSE, upper = FALSE)

\method{print}{dist}(x, diag = NULL, upper = NULL,
      digits = getOption("digits"), justify = "none", right = TRUE, \dots)

\method{as.matrix}{dist}(x)
}
\alias{dist}
\alias{print.dist}
\alias{format.dist}
\alias{as.matrix.dist}
\alias{names.dist}
\alias{names<-.dist}
\alias{as.dist}
\arguments{
  \item{x}{a numeric matrix, data frame or \code{"dist"} object.}
  \item{method}{the distance measure to be used. This must be one of
    \code{"euclidean"}, \code{"maximum"}, \code{"manhattan"},
    \code{"canberra"}, \code{"binary"} or \code{"minkowski"}.
    Any unambiguous substring can be given.}
  \item{diag}{logical value indicating whether the diagonal of the
    distance matrix should be printed by \code{print.dist}.}
  \item{upper}{logical value indicating whether the upper triangle of the
    distance matrix should be printed by \code{print.dist}.}
  \item{p}{The power of the Minikowski distance.}
  \item{m}{A matrix of distances to be converted to a \code{"dist"}
    object (only the lower triangle is used, the rest is ignored).}
  \item{digits, justify}{passed to \code{\link{format}} inside of
    \code{print()}.}
  \item{right, \dots}{further arguments, passed to the (next)
    \code{print} method.}
}
\description{
  This function computes and returns the distance matrix computed by
  using the specified distance measure to compute the distances between
  the rows of a data matrix.
}
\details{
  Available distance measures are (written for two vectors \eqn{x} and
  \eqn{y}):
  \describe{
    \item{\code{euclidean}:}{Usual square distance between the two
      vectors (2 norm).}

    \item{\code{maximum}:}{Maximum distance between two components of \eqn{x}
      and \eqn{y} (supremum norm)}

    \item{\code{manhattan}:}{Absolute distance between the two vectors
      (1 norm).}

    \item{\code{canberra}:}{\eqn{\sum_i |x_i - y_i| / |x_i + y_i|}{%
	sum(|x_i - y_i| / |x_i + y_i|)}.  Terms with zero numerator and
      denominator are omitted from the sum and treated as if the values
      were missing.
    }

    \item{\code{binary}:}{(aka \emph{asymmetric binary}): The vectors
      are regarded as binary bits, so non-zero elements are \sQuote{on}
      and zero elements are \sQuote{off}.  The distance is the
      \emph{proportion} of bits in which only one is on amongst those in
      which at least one is on.}

    \item{\code{minkowki}:}{The \eqn{p} norm, the \eqn{p}th root of the
      sum of the \eqn{p}th powers of the differences of the components.}
  }

  Missing values are allowed, and are excluded from all computations
  involving the rows within which they occur.
  Further, when \code{Inf} values are involved, all pairs of values are
  excluded when their contribution to the distance gave \code{NaN} or
  \code{NA}.\cr
  If some columns are excluded in calculating a Euclidean, Manhattan,
  Canberra or Minkowski distance, the sum is scaled up proportionally
  to the number of columns used.  If all pairs are excluded when calculating a
  particular distance, the value is \code{NA}.

  The functions \code{as.matrix.dist()} and \code{as.dist()} can be used
  for conversion between objects of class \code{"dist"} and conventional
  distance matrices and vice versa.
}
\value{
  An object of class \code{"dist"}.

  The lower triangle of the distance matrix stored by columns in a
  vector, say \code{do}. If \code{n} is the number of
  observations, i.e., \code{n <- attr(do, "Size")}, then
  for \eqn{i < j <= n}, the dissimilarity between (row) i and j is
  \code{do[n*(i-1) - i*(i-1)/2 + j-i]}.
  The length of the vector is \eqn{n*(n-1)/2}, i.e., of order \eqn{n^2}.

  The object has the following attributes (besides \code{"class"} equal
  to \code{"dist"}):
  \item{Size}{integer, the number of observations in the dataset.}
  \item{Labels}{optionally, contains the labels, if any, of the
    observations of the dataset.}
  \item{Diag, Upper}{logicals corresponding to the arguments \code{diag}
    and \code{upper} above, specifying how the object should be printed.}
  \item{call}{optionally, the \code{\link{call}} used to create the
    object.}
  \item{method}{optionally, the distance method used; resulting form
    \code{\link{dist}()}, the (\code{\link{match.arg}()}ed) \code{method}
    argument.}
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.

  Mardia, K. V., Kent, J. T. and Bibby, J. M. (1979)
  \emph{Multivariate Analysis.} Academic Press.

  Borg, I. and Groenen, P. (1997)
  \emph{Modern Multidimensional Scaling.  Theory and Applications.}
  Springer.
}
\seealso{
  \code{\link[cluster]{daisy}} in the \pkg{cluster} package with more
  possibilities in the case of \emph{mixed} (contiuous / categorical)
  variables.
  \code{\link{hclust}}.
}
\examples{
x <- matrix(rnorm(100), nrow=5)
dist(x)
dist(x, diag = TRUE)
dist(x, upper = TRUE)
m <- as.matrix(dist(x))
d <- as.dist(m)
stopifnot(d == dist(x))
names(d) <- LETTERS[1:5]
print(d, digits = 3)

## example of binary and canberra distances.
x <- c(0, 0, 1, 1, 1, 1)
y <- c(1, 0, 1, 1, 0, 1)
dist(rbind(x,y), method="binary")
## answer 0.4 = 2/5
dist(rbind(x,y), method="canberra")
## answer 2 * (6/5)

## Examples involving "Inf" :
## These will not work on a non-IEEE 754 system.
## 1)
x[6] <- Inf
(m2 <- rbind(x,y))
dist(m2, method="binary")# warning, answer 0.5 = 2/4
## These all give "Inf":
stopifnot(Inf == dist(m2, method= "euclidean"),
          Inf == dist(m2, method= "maximum"),
          Inf == dist(m2, method= "manhattan"))
##  "Inf" is same as very large number:
x1 <- x; x1[6] <- 1e100
stopifnot(dist(cbind(x ,y), method="canberra") ==
    print(dist(cbind(x1,y), method="canberra")))

## 2)
y[6] <- Inf #-> 6-th pair is excluded
dist(rbind(x,y), method="binary")# warning; 0.5
dist(rbind(x,y), method="canberra")#  3
dist(rbind(x,y), method="maximum")  # 1
dist(rbind(x,y), method="manhattan")# 2.4
}
\keyword{multivariate}
\keyword{cluster}
\eof
\name{dummy.coef}
\title{Extract Coefficients in Original Coding}
\usage{
dummy.coef(object, \dots)

\method{dummy.coef}{lm}(object, use.na = FALSE, \dots)

\method{dummy.coef}{aovlist}(object, use.na = FALSE, \dots)
}
\alias{dummy.coef}
\alias{dummy.coef.lm}
\alias{dummy.coef.aovlist}
\alias{print.dummy.coef}
\alias{print.dummy.coef.list}
\arguments{
  \item{object}{a linear model fit.}
  \item{use.na}{logical flag for coefficients in a singular model. If
    \code{use.na} is true, undetermined coefficients will be missing; if
    false they will get one possible value.}
  \item{\dots}{arguments passed to or from other methods.}
}
\description{
  This extracts coefficients in terms of the original levels of the
  coefficients rather than the coded variables.
}
\details{
  A fitted linear model has coefficients for the contrasts of the factor
  terms, usually one less in number than the number of levels.  This
  function re-expresses the coefficients in the original coding; as the
  coefficients will have been fitted in the reduced basis, any implied
  constraints (e.g., zero sum for \code{contr.helmert} or \code{contr.sum}
  will be respected.  There will be little point in using
  \code{dummy.coef} for \code{contr.treatment} contrasts, as the missing
  coefficients are by definition zero.

  The method used has some limitations, and will give incomplete results
  for terms such as \code{poly(x, 2))}.  However, it is adequate for
  its main purpose, \code{aov} models.
}
\value{
  A list giving for each term the values of the coefficients. For a
  multistratum \code{aov} model, such a list for each stratum.
}

\section{Warning}{
  This function is intended for human inspection of the
  output: it should not be used for calculations.  Use coded variables
  for all calculations.

  The results differ from S for singular values, where S can be incorrect.
}

\seealso{\code{\link{aov}}, \code{\link{model.tables}}}

\examples{
options(contrasts=c("contr.helmert", "contr.poly"))
## From Venables and Ripley (2002) p.165.
N <- c(0,1,0,1,1,1,0,0,0,1,1,0,1,1,0,0,1,0,1,0,1,1,0,0)
P <- c(1,1,0,0,0,1,0,1,1,1,0,0,0,1,0,1,1,0,0,1,0,1,1,0)
K <- c(1,0,0,1,0,1,1,0,0,1,0,1,0,1,1,0,0,0,1,1,1,0,1,0)
yield <- c(49.5,62.8,46.8,57.0,59.8,58.5,55.5,56.0,62.8,55.8,69.5,
55.0, 62.0,48.8,45.5,44.2,52.0,51.5,49.8,48.8,57.2,59.0,53.2,56.0)

npk <- data.frame(block=gl(6,4), N=factor(N), P=factor(P),
                  K=factor(K), yield=yield)
npk.aov <- aov(yield ~ block + N*P*K, npk)
dummy.coef(npk.aov)

npk.aovE <- aov(yield ~  N*P*K + Error(block), npk)
dummy.coef(npk.aovE)
}
\keyword{models}
\eof
\name{ecdf}
\alias{ecdf}
\alias{plot.ecdf}
\alias{print.ecdf}
\title{Empirical Cumulative Distribution Function}
\usage{
ecdf(x)

\method{plot}{ecdf}(x, \dots, ylab="Fn(x)", verticals = FALSE,
     col.01line = "gray70")

\method{print}{ecdf}(x, digits= getOption("digits") - 2, \dots)
}
\arguments{
  \item{x}{numeric vector of \dQuote{observations} in \code{ecdf};  for
    the methods, an object of class \code{"ecdf"}, typically.}
  \item{\dots}{arguments to be passed to subsequent methods, i.e.,
    \code{\link{plot.stepfun}} for the \code{plot} method.}
  \item{ylab}{label for the y-axis.}
  \item{verticals}{see \code{\link{plot.stepfun}}.}
  \item{col.01line}{numeric or character specifying the color of the
    horizontal lines at y=0 and 1, see \code{\link{colors}}.}
  \item{digits}{number of significant digits to use, see
    \code{\link{print}}.}
}
\description{
  Compute or plot an empirical cumulative distribution function.
}
\details{
  The e.c.d.f. (empirical cumulative distribution function)
  \eqn{F_n}{Fn} is a step
  function with jump \eqn{1/n} at each observation (possibly with
  multiple jumps at one place if there are ties).

  For observations
  \code{x}\eqn{= (}\eqn{x_1,x_2}{x1,x2}, \ldots \eqn{x_n)}{xn)},
  \eqn{F_n}{Fn} is the fraction of observations less or equal to \eqn{t},
  i.e.,
  \deqn{F_n(t) = \#\{x_i\le t\}\ / n
               = \frac1 n\sum_{i=1}^n \mathbf{1}_{[x_i \le t]}.}{
    Fn(t) = #\{x_i \le t\} / n  =  1/n sum(i=1,n) Indicator(xi <= t).}

  The function \code{plot.ecdf} which implements the \code{\link{plot}}
  method for \code{ecdf} objects, is implemented via a call to
  \code{\link{plot.stepfun}}; see its documentation.
}
\value{
  For \code{ecdf}, a function of class \code{"ecdf"}, inheriting from the
  \code{"\link{stepfun}"} class.
}
\author{
  Martin Maechler, \email{maechler@stat.math.ethz.ch}.
}
\seealso{\code{\link{stepfun}}, the more general class of step functions,
  \code{\link{approxfun}} and \code{\link{splinefun}}.
}
\examples{
##-- Simple didactical  ecdf  example:
Fn <- ecdf(rnorm(12))
Fn; summary(Fn)
12*Fn(knots(Fn)) == 1:12 ## == 1:12  if and only if there are no ties !

y <- round(rnorm(12),1); y[3] <- y[1]
Fn12 <- ecdf(y)
Fn12
print(knots(Fn12), dig=2)
12*Fn12(knots(Fn12)) ## ~= 1:12  if there where no ties

summary(Fn12)
summary.stepfun(Fn12)
print(ls.Fn12 <- ls(env= environment(Fn12)))
##[1] "f"  "method"  "n"  "ties"   "x"  "y"  "yleft"  "yright"

12 * Fn12((-20:20)/10)

###----------------- Plotting --------------------------

op <- par(mfrow=c(3,1), mgp=c(1.5, 0.8,0), mar= .1+c(3,3,2,1))

F10 <- ecdf(rnorm(10))
summary(F10)

plot(F10)
plot(F10, verticals= TRUE, do.p = FALSE)

plot(Fn12)# , lwd=2) dis-regarded
xx <- unique(sort(c(seq(-3,2, length=201), knots(Fn12))))
lines(xx, Fn12(xx), col='blue')
abline(v=knots(Fn12),lty=2,col='gray70')

plot(xx, Fn12(xx), type='b', cex=.1)#- plot.default
plot(Fn12, col.h='red', add= TRUE)  #- plot method
abline(v=knots(Fn12),lty=2,col='gray70')
plot(Fn12, verticals=TRUE, col.p='blue', col.h='red',col.v='bisque')
par(op)

##-- this works too (automatic call to  ecdf(.)):
plot.ecdf(rnorm(24))
}
\keyword{dplot}
\keyword{hplot}

\eof
\name{eff.aovlist}
\title{Compute Efficiencies of Multistratum Analysis of Variance}
\usage{
eff.aovlist(aovlist)
}
\alias{eff.aovlist}
\arguments{
  \item{aovlist}{
    The result of a call to \code{aov} with an \code{Error} term.
  }
}
\description{
  Computes the efficiencies of fixed-effect terms in an analysis of
  variance model with multiple strata.
}
\details{
  Fixed-effect terms in an analysis of variance model with multiple strata
  may be estimable in more than one stratum, in which case there is less
  than complete information in each. The efficiency is the fraction of
  the maximum possible precision (inverse variance) obtainable by
  estimating in just that stratum.

  This is used to pick strata in which to estimate terms in
  \code{model.tables.aovlist} and elsewhere.
}
\value{
  A matrix giving for each non-pure-error stratum (row) the efficiencies
  for each fixed-effect term in the model.
}

\seealso{\code{\link{aov}}, \code{\link{model.tables.aovlist}},
  \code{\link{se.contrast.aovlist}} }

\examples{
## for balanced designs all efficiencies are zero or one.
## so as a statistically meaningless test:
options(contrasts=c("contr.helmert", "contr.poly"))
## From Venables and Ripley (2002) p.165.
N <- c(0,1,0,1,1,1,0,0,0,1,1,0,1,1,0,0,1,0,1,0,1,1,0,0)
P <- c(1,1,0,0,0,1,0,1,1,1,0,0,0,1,0,1,1,0,0,1,0,1,1,0)
K <- c(1,0,0,1,0,1,1,0,0,1,0,1,0,1,1,0,0,0,1,1,1,0,1,0)
yield <- c(49.5,62.8,46.8,57.0,59.8,58.5,55.5,56.0,62.8,55.8,69.5,
55.0, 62.0,48.8,45.5,44.2,52.0,51.5,49.8,48.8,57.2,59.0,53.2,56.0)

npk <- data.frame(block=gl(6,4), N=factor(N), P=factor(P),
                  K=factor(K), yield=yield)
npk.aovE <- aov(yield ~  N*P*K + Error(block), npk)
eff.aovlist(npk.aovE)
}
\keyword{models}
\eof
\name{effects}
\title{Effects from Fitted Model}
\usage{
effects(object, \dots)

\method{effects}{lm}(object, set.sign=FALSE, \dots)
}
\alias{effects}
\alias{effects.lm}
\alias{effects.glm}
\arguments{
 \item{object}{an \R object; typically, the result of a model fitting function
   such as \code{\link{lm}}.}
 \item{set.sign}{logical. If \code{TRUE}, the sign of the effects
   corresponding to coefficients in the model will be set to agree with the
   signs of the corresponding coefficients, otherwise the sign is
   arbitrary.}
 \item{\dots}{arguments passed to or from other methods.}
}
\description{
  Returns (orthogonal) effects from a fitted model, usually a linear
  model. This is a generic function, but currently only has a methods for
  objects inheriting from classes \code{"lm"} and \code{"glm"}.
}
\details{
  For a linear model fitted by \code{\link{lm}} or \code{\link{aov}},
  the effects are the uncorrelated single-degree-of-freedom values
  obtained by projecting the data onto the successive orthogonal
  subspaces generated by the QR decomposition during the fitting
  process. The first \eqn{r} (the rank of the model) are associated with
  coefficients and the remainder span the space of residuals (but are
  not associated with particular residuals).

  Empty models do not have effects.
}
\value{
  A (named) numeric vector of the same length as
  \code{\link{residuals}}, or a matrix if there were multiple responses
  in the fitted model, in either case of class \code{"coef"}.

  The first \eqn{r} rows are labelled by the corresponding coefficients,
  and the remaining rows are unlabelled.  Note that in rank-deficient
  models the \dQuote{corresponding} coefficients will be in a different
  order if pivoting occurred.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical Models in S.}
  Wadsworth \& Brooks/Cole.
}
\seealso{\code{\link{coef}}}
\examples{
y <- c(1:3,7,5)
x <- c(1:3,6:7)
( ee <- effects(lm(y ~ x)) )
c(round(ee - effects(lm(y+10 ~ I(x-3.8))),3))# just the first is different
}
\keyword{models}
\keyword{regression}
\eof
\name{embed}
\title{Embedding a Time Series}
\usage{
embed (x, dimension = 1) 
}
\alias{embed}
\arguments{
  \item{x}{a numeric vector, matrix, or time series.}
  \item{dimension}{a scalar representing the embedding dimension.}
}
\description{
  Embeds the time series \code{x} into a low-dimensional
  Euclidean space.
}
\details{
  Each row of the resulting matrix consists of sequences
  \code{x[t]}, \code{x[t-1]}, \dots, \code{x[t-dimension+1]}, where
  \code{t} is the original index of \code{x}. If \code{x} is a matrix,
  i.e., \code{x} contains more than one variable, then \code{x[t]}
  consists of the \code{t}th observation on each variable.
}
\value{
  A matrix containing the embedded time series \code{x}.
}
\author{A. Trapletti, B.D. Ripley}
\examples{
x <- 1:10
embed (x, 3)
}
\keyword{ts}

\eof
\name{expand.model.frame}
\alias{expand.model.frame}
\title{Add new variables to a model frame}
\description{
  Evaluates new variables as if they had been part of the formula of the
  specified model.  This ensures that the same \code{na.action} and
  \code{subset} arguments are applied and allows, for example, \code{x}
  to be recovered for a model using \code{sin(x)} as a predictor.
}
\usage{
expand.model.frame(model, extras, envir=environment(formula(model)),
                   na.expand = FALSE)
}
\arguments{
  \item{model}{a fitted model}
  \item{extras}{one-sided formula or vector of character strings
    describing new variables to be added}
  \item{envir}{an environment to evaluate things in}
  \item{na.expand}{logical; see below}
}
\details{
  If \code{na.expand=FALSE} then \code{NA} values in the extra variables
  will be passed to the \code{na.action} function used in
  \code{model}.  This may result in a shorter data frame (with
  \code{\link{na.omit}}) or an error (with \code{\link{na.fail}}).  If
  \code{na.expand=TRUE} the returned data frame will have precisely the
  same rows as \code{model.frame(model)}, but the columns corresponding to
  the extra variables may contain \code{NA}.
}
\value{
  A data frame.
}
\seealso{\code{\link{model.frame}},\code{\link{predict}}}

\examples{
data(trees)
model <- lm(log(Volume) ~ log(Girth) + log(Height), data=trees)
expand.model.frame(model, ~ Girth) # prints data.frame like

dd <- data.frame(x=1:5, y=rnorm(5), z=c(1,2,NA,4,5))
model <- glm(y ~ x, data=dd, subset=1:4, na.action=na.omit)
expand.model.frame(model, "z", na.expand=FALSE) # = default
expand.model.frame(model, "z", na.expand=TRUE)
}
\keyword{manip}
\keyword{regression}
\eof
%-- This page by Martin Maechler,  improvements welcome!
\name{extractAIC}
\title{Extract AIC from a Fitted Model}
%
\alias{extractAIC}
%\alias{extractAIC.aov}  
%\alias{extractAIC.lm}
%\alias{extractAIC.glm}    
%\alias{extractAIC.coxph}  
%\alias{extractAIC.negbin}
%\alias{extractAIC.survreg}
\usage{
extractAIC(fit, scale,     k = 2, \dots)  
}
\arguments{
  \item{fit}{fitted model, usually the result of a fitter like
    \code{\link{lm}}.}
 \item{scale}{optional numeric specifying the scale parameter of the
   model, see \code{scale} in \code{\link{step}}.
 }
 \item{k}{numeric specifying the \dQuote{weight} of the
   \emph{equivalent degrees of freedom} (\eqn{\equiv}{=:}\code{edf})
   part in the AIC formula.}
 \item{\dots}{further arguments (currently unused in base \R).}
}
%-- Source in ../R/add.R
\description{
  Computes the (generalized) Akaike \bold{A}n \bold{I}nformation
  \bold{C}riterion for a fitted parametric model.
}
\details{
  This is a generic function, with methods in base \R for \code{"aov"},
  \code{"coxph"}, \code{"glm"}, \code{"lm"}, \code{"negbin"}
  and \code{"survreg"} classes.
  
  The criterion used is 
 \deqn{AIC = - 2\log L +  k \times \mbox{edf},}{AIC = - 2*log L +  k * edf,}
  where \eqn{L} is the likelihood
  and \code{edf} the equivalent degrees of freedom (i.e., the number of
  parameters for usual parametric models) of \code{fit}.
 
  For linear models with unknown scale (i.e., for \code{\link{lm}} and
  \code{\link{aov}}), \eqn{-2\log L} is computed from the
  \emph{deviance} and uses a different additive constant to \code{\link{AIC}}.
      
  \code{k = 2} corresponds to the traditional AIC, using \code{k =
    log(n)} provides the BIC (Bayes IC) instead.
  
  For further information, particularly about \code{scale}, see
  \code{\link{step}}.
}
\note{
  These functions are used in \code{\link{add1}},
  \code{\link{drop1}} and \code{\link{step}} and that may be their
  main use.
}
\value{
  A numeric vector of length 2, giving

  \item{edf}{the \dQuote{\bold{e}quivalent \bold{d}egrees of \bold{f}reedom}
  of the fitted model \code{fit}.}

\item{AIC}{the (generalized) Akaike Information Criterion for \code{fit}.}
}
%-- Source in ../R/add.R
\author{B. D. Ripley}
\references{
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.}
  New York: Springer (4th ed).
}
\seealso{
  \code{\link{AIC}}, \code{\link{deviance}}, \code{\link{add1}},
  \code{\link{step}}
}
\examples{
example(glm)
extractAIC(glm.D93)#>>  5  15.129
}
\keyword{models}

\eof
\name{factanal}
\alias{factanal}
\alias{factanal.fit.mle}
\title{Factor Analysis}
\description{
  Perform maximum-likelihood factor analysis on a covariance matrix or
  data matrix.
}
\usage{
factanal(x, factors, data = NULL, covmat = NULL, n.obs = NA,
         subset, na.action,
         start = NULL, scores = c("none", "regression", "Bartlett"),
         rotation = "varimax", control = NULL, \dots)
}
\arguments{
  \item{x}{A formula or a numeric matrix or an object that can be
    coerced to a numeric matrix.}
  \item{factors}{The number of factors to be fitted.}
  \item{data}{A data frame, used only if \code{x} is a formula.}
  \item{covmat}{A covariance matrix, or a covariance list as returned by
    \code{\link{cov.wt}}.  Of course, correlation matrices are covariance
    matrices.}
  \item{n.obs}{The number of observations, used if \code{covmat} is a
    covariance matrix.}
  \item{subset}{A specification of the cases to be used, if \code{x} is
    used as a matrix or formula.}
  \item{na.action}{The \code{na.action} to be used if \code{x} is
    used as a formula.}
  \item{start}{\code{NULL} or a matrix of starting values, each column
    giving an initial set of uniquenesses.}
  \item{scores}{Type of scores to produce, if any.  The default is none,
    \code{"regression"} gives Thompson's scores, \code{"Bartlett"} given
    Bartlett's weighted least-squares scores. Partial matching allows
    these names to be abbreviated.}
  \item{rotation}{character. \code{"none"} or the name of a function
    to be used to rotate the factors: it will be called with first
    argument the loadings matrix, and should return a list with component
    \code{loadings} giving the rotated loadings, or just the rotated loadings.}
  \item{control}{A list of control values,
    \describe{
      \item{nstart}{The number of starting values to be tried if
	\code{start = NULL}. Default 1.}
      \item{trace}{logical. Output tracing information? Default \code{FALSE}.}
      \item{lower}{The lower bound for uniquenesses during
	optimization. Should be > 0. Default 0.005.}
      \item{opt}{A list of control values to be passed to
	\code{\link{optim}}'s \code{control} argument.}
      \item{rotate}{a list of additional arguments for the rotation function.}
    }
  }
  \item{\dots}{Components of \code{control} can also be supplied as
    named arguments to \code{factanal}.}
}
\details{
  The factor analysis model is
  \deqn{x = \Lambda f + e}{ x = Lambda f + e}
  for a \eqn{p}--element row-vector \eqn{x}, a \eqn{p \times k}{p x k}
  matrix of \emph{loadings}, a \eqn{k}--element vector of \emph{scores}
  and a \eqn{p}--element vector of errors.  None of the components
  other than \eqn{x} is observed, but the major restriction is that the
  scores be uncorrelated and of unit variance, and that the errors be
  independent with variances \eqn{\Phi}{Phi}, the
  \emph{uniquenesses}.  Thus factor analysis is in essence a model for
  the covariance matrix of \eqn{x},
  \deqn{\Sigma = \Lambda^\prime\Lambda + \Psi}{Sigma = Lambda'Lambda + Psi}
  There is still some indeterminacy in the model for it is unchanged
  if \eqn{\Lambda}{Lambda} is replaced by \eqn{G\Lambda}{G Lambda} for
  any orthogonal matrix \eqn{G}.  Such matrices \eqn{G} are known as
  \emph{rotations} (although the term is applied also to non-orthogonal
  invertible matrices).
  
  If \code{covmat} is supplied it is used.  Otherwise \code{x} is used if
  it is a matrix, or a formula \code{x} is used with \code{data} to
  construct a model matrix, and that is used to construct a covariance
  matrix.  (It makes no sense for the formula to have a response,
  and all the variables must be numeric.)  Once a covariance matrix is found or
  calculated from \code{x}, it is converted to a correlation matrix for
  analysis.  The correlation matrix is returned as component
  \code{correlation} of the result.

  The fit is done by optimizing the log likelihood assuming multivariate
  normality over the uniquenesses.  (The maximizing loadings for given
  uniquenesses can be found analytically: Lawley & Maxwell (1971,
  p. 27).)  All the starting values supplied in \code{start} are tried
  in turn and the best fit obtained is used.  If \code{start = NULL}
  then the first fit is started at the value suggested by Jreskog
  (1963) and given by Lawley & Maxwell (1971, p. 31), and then
  \code{control$nstart - 1} other values are tried, randomly selected
  as equal values of the uniquenesses.

  The uniquenesses are technically constrained to lie in \eqn{[0, 1]},
  but near-zero values are problematical, and the optimization is
  done with a lower bound of \code{control$lower}, default 0.005
  (Lawley & Maxwell, 1971, p. 32).

  Scores can only be produced if a data matrix is supplied and used.
  The first method is the regression method of Thomson (1951), the
  second the weighted least squares method of Bartlett (1937, 8).
  Both are estimates of the unobserved scores \eqn{f}.  Thomson's method
  regresses (in the population) the unknown \eqn{f} on \eqn{x} to yield
  \deqn{\hat f = \Lambda^\prime \Sigma^{-1} x}{hat f = Lambda' Sigma^-1 x}
  and then substitutes the sample estimates of the quantities on the
  right-hand side.  Bartlett's method minimizes the sum of squares of
  standardized errors over the choice of \eqn{f}, given (the fitted)
  \eqn{\Lambda}{Lambda}.

  If \code{x} is a formula then the standard NA-handling is applied to
  the scores (if requested): see \code{\link{napredict}}.
}
\value{
  An object of class \code{"factanal"} with components
  \item{loadings}{A matrix of loadings, one column for each factor.  The
    factors are ordered in decreasing order of sums of squares of
    loadings, and given the sign that will make the sum of the loadings
    positive.}
  \item{uniquenesses}{The uniquenesses computed.}
  \item{correlation}{The correlation matrix used.}
  \item{criteria}{The results of the optimization: the value of the
    negative log-likelihood and information on the iterations used.}
  \item{factors}{The argument \code{factors}.}
  \item{dof}{The number of degrees of freedom of the factor analysis model.}
  \item{method}{The method: always \code{"mle"}.}
  \item{scores}{If requested, a matrix of scores.}
  \item{n.obs}{The number of observations if available, or \code{NA}.}
  \item{call}{The matched call.}
  \item{na.action}{If relevant.}
  \item{STATISTIC, PVAL}{The significance-test statistic and P value, if
    if can be computed.}
}

\note{
  There are so many variations on factor analysis that it is hard to
  compare output from different programs.  Further, the optimization in
  maximum likelihood factor analysis is hard, and many other examples we
  compared had less good fits than produced by this function.  In
  particular, solutions which are Heywood cases (with one or more
  uniquenesses essentially zero) are much often common than most texts
  and some other programs would lead one to believe.
}

\references{
  Bartlett, M. S. (1937) The statistical conception of mental factors.
  \emph{British Journal of Psychology}, \bold{28}, 97--104.

  Bartlett, M. S. (1938) Methods of estimating mental
  factors. \emph{Nature}, \bold{141}, 609--610.
  
  Jreskog, K. G. (1963) \emph{Statistical Estimation in Factor
    Analysis.}  Almqvist and Wicksell.
  
  Lawley, D. N. and Maxwell, A. E. (1971) \emph{Factor Analysis as a
    Statistical Method.} Second edition. Butterworths.

  Thomson, G. H. (1951) \emph{The Factorial Analysis of Human Ability.}
  London University Press.
}

\seealso{
  \code{\link{print.loadings}},
  \code{\link{varimax}}, \code{\link{princomp}},
  \code{\link{ability.cov}}, \code{\link{Harman23.cor}},
  \code{\link{Harman74.cor}}
}

\examples{
# A little demonstration, v2 is just v1 with noise,
# and same for v4 vs. v3 and v6 vs. v5
# Last four cases are there to add noise
# and introduce a positive manifold (g factor)
v1 <- c(1,1,1,1,1,1,1,1,1,1,3,3,3,3,3,4,5,6)
v2 <- c(1,2,1,1,1,1,2,1,2,1,3,4,3,3,3,4,6,5)
v3 <- c(3,3,3,3,3,1,1,1,1,1,1,1,1,1,1,5,4,6)
v4 <- c(3,3,4,3,3,1,1,2,1,1,1,1,2,1,1,5,6,4)
v5 <- c(1,1,1,1,1,3,3,3,3,3,1,1,1,1,1,6,4,5)
v6 <- c(1,1,1,2,1,3,3,3,4,3,1,1,1,2,1,6,5,4)
m1 <- cbind(v1,v2,v3,v4,v5,v6)
cor(m1)
factanal(m1, factors=3) # varimax is the default
factanal(m1, factors=3, rotation="promax")
# The following shows the g factor as PC1
prcomp(m1)

## formula interface
factanal(~v1+v2+v3+v4+v5+v6, factors = 3,
         scores = "Bartlett")$scores

## a realistic example from Barthlomew (1987, pp. 61-65)
example(ability.cov)
}
\keyword{multivariate}
\eof
\name{factor.scope}
\title{Compute Allowed Changes in Adding to or Dropping from a Formula}
\usage{
add.scope(terms1, terms2)
drop.scope(terms1, terms2)
factor.scope(factor, scope)
}
\alias{add.scope}
\alias{drop.scope}
\alias{factor.scope}
\arguments{
 \item{terms1}{the terms or formula for the base model.}
 \item{terms2}{the terms or formula for the upper (\code{add.scope}) or
   lower (\code{drop.scope}) scope. If missing for \code{drop.scope} it is
   taken to be the null formula, so all terms (except any intercept) are
   candidates to be dropped.}
 \item{factor}{the \code{"factor"} attribute of the terms of the base object.}
 \item{scope}{a list with one or both components \code{drop} and
   \code{add} giving the \code{"factor"} attribute of the lower and
   upper scopes respectively.}
}
\description{
 \code{add.scope} and \code{drop.scope} compute those terms that can be
 individually added to or dropped from a model while respecting the
 hierarchy of terms.
}
\details{
 \code{factor.scope} is not intended to be called directly by users.
}
\value{
  For \code{add.scope} and \code{drop.scope} a character vector of
  terms labels. For \code{factor.scope}, a list with components
  \code{drop} and \code{add}, character vectors of terms labels.
}

\seealso{\code{\link{add1}}, \code{\link{drop1}},
  \code{\link{aov}}, \code{\link{lm}}}

\examples{
add.scope( ~ a + b + c + a:b,  ~ (a + b + c)^3)
# [1] "a:c" "b:c"
drop.scope( ~ a + b + c + a:b)
# [1] "c"   "a:b"
}
\keyword{models}
\eof
\name{family}
\title{Family Objects for Models}
\usage{
family(object, \dots)

binomial(link = "logit")
gaussian(link ="identity")
Gamma(link = "inverse")
inverse.gaussian(link = "1/mu^2")
poisson(link = "log")
quasi(link = "identity", variance = "constant")
quasibinomial(link = "logit")
quasipoisson(link = "log")
}
\alias{family}
\alias{binomial}
\alias{gaussian}
\alias{Gamma}
\alias{inverse.gaussian}
\alias{poisson}
\alias{quasi}
\alias{quasibinomial}
\alias{quasipoisson}
\alias{print.family}
\arguments{
  \item{link}{a specification for the model link function.
    The \code{gaussian} family accepts the links \code{"identity"},
    \code{"log"} and \code{"inverse"};
    the \code{binomial} family the links \code{"logit"},
    \code{"probit"}, \code{"log"} and \code{"cloglog"} (complementary
    log-log);
    the \code{Gamma} family the links \code{"inverse"}, \code{"identity"}
     and \code{"log"};
    the \code{poisson} family the links \code{"log"}, \code{"identity"}, 
    and \code{"sqrt"} and the \code{inverse.gaussian} family the links
    \code{"1/mu^2"}, \code{"inverse"}, \code{"inverse"} and \code{"log"}.
    
    The \code{quasi} family allows the links \code{"logit"}, \code{"probit"},
    \code{"cloglog"},  \code{"identity"}, \code{"inverse"},
    \code{"log"}, \code{"1/mu^2"} and \code{"sqrt"}.
    The function \code{\link{power}} can also be used to create a
    power link function for the \code{quasi} family.
  }
  \item{variance}{for all families, other than \code{quasi}, the
    variance function is determined by the family.  The \code{quasi}
    family will accept the specifications \code{"constant"},
    \code{"mu(1-mu)"}, \code{"mu"}, \code{"mu^2"} and \code{"mu^3"} for
    the variance function.}
  \item{object}{the function \code{family} accesses the \code{family}
    objects which are stored within objects created by modelling
    functions (e.g., \code{glm}).}
  \item{\dots}{further arguments passed to methods.}
}
\description{
  Family objects provide a convenient way to specify the details of the
  models used by functions such as \code{\link{glm}}.  See the
  documentation for \code{\link{glm}} for the details on how such model
  fitting takes place.
}
\details{
  The \code{quasibinomial} and \code{quasipoisson} families differ from
  the \code{binomial} and \code{poisson} families only in that the
  dispersion parameter is not fixed at one, so they can \dQuote{model}
  over-dispersion.  For the binomial case see McCullagh and Nelder
  (1989, pp. 124--8).  Although they show that there is (under some
  restrictions) a model with
  variance proportional to mean as in the quasi-binomial model, note
  that \code{glm} does not compute maximum-likelihood estimates in that
  model.  The behaviour of S is closer to the quasi- variants. 
}
\references{
  McCullagh P. and Nelder, J. A. (1989)
  \emph{Generalized Linear Models.}
  London: Chapman and Hall.

  Dobson, A. J. (1983)
  \emph{An Introduction to Statistical Modelling.}
  London: Chapman and Hall.

  Cox, D. R. and  Snell, E. J. (1981).
  \emph{Applied Statistics; Principles and Examples.}
  London: Chapman and Hall.

  Hastie, T. J. and Pregibon, D. (1992)
  \emph{Generalized linear models.}
  Chapter 6 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\author{
  The design was inspired by S functions of the same names described
  in Hastie \& Pregibon (1992).
}
\seealso{
  \code{\link{glm}}, \code{\link{power}}.
}
\examples{
nf <- gaussian()# Normal family
nf
str(nf)# internal STRucture

gf <- Gamma()
gf
str(gf)
gf$linkinv
gf$variance(-3:4) #- == (.)^2

## quasipoisson. compare with example(glm)
counts <- c(18,17,15,20,10,20,25,13,12)
outcome <- gl(3,1,9)
treatment <- gl(3,3)
d.AD <- data.frame(treatment, outcome, counts)
glm.qD93 <- glm(counts ~ outcome + treatment, family=quasipoisson())
glm.qD93
anova(glm.qD93, test="F")
summary(glm.qD93)
## for Poisson results use
anova(glm.qD93, dispersion = 1, test="Chisq")
summary(glm.qD93, dispersion = 1)


## tests of quasi
x <- rnorm(100)
y <- rpois(100, exp(1+x))
glm(y ~x, family=quasi(var="mu", link="log"))
# which is the same as
glm(y ~x, family=poisson)
glm(y ~x, family=quasi(var="mu^2", link="log"))
\dontrun{glm(y ~x, family=quasi(var="mu^3", link="log")) # should fail}
y <- rbinom(100, 1, plogis(x))
# needs to set a starting value for the next fit
glm(y ~x, family=quasi(var="mu(1-mu)", link="logit"), start=c(0,1))
}
\keyword{models}
\eof
\name{filter}
\alias{filter}
\title{Linear Filtering on a Time Series}
\usage{
filter(x, filter, method = c("convolution", "recursive"),
       sides = 2, circular = FALSE, init)
}
\arguments{
  \item{x}{a univariate or multivariate time series.}

  \item{filter}{a vector of filter coefficients in reverse time order
    (as for AR or MA coefficients).}

  \item{method}{Either \code{"convolution"} or \code{"recursive"} (and
    can be abbreviated). If \code{"convolution"} a moving average is
    used: if \code{"recursive"} an autoregression is used.}

  \item{sides}{for convolution filters only. If \code{sides=1} the
    filter coefficients are for past values only; if \code{sides=2}
    they are centred around lag 0. In this case the length of the
    filter should be odd, but if it is even, more of the filter
    is forward in time than backward.}

  \item{circular}{for convolution filters only.  If \code{TRUE}, wrap
    the filter around the ends of the series, otherwise assume
    external values are missing (\code{NA}).}

  \item{init}{for recursive filters only. Specifies the initial values
    of the time series just prior to the start value, in reverse
    time order. The default is a set of zeros.}
}
\description{
  Applies linear filtering to a univariate time series or to each series
  separately of a multivariate time series.
}
\value{
  A time series object.
}
\details{
  Missing values are allowed in \code{x} but not in \code{filter}
  (where they would lead to missing values everywhere in the output).

  Note that there is an implied coefficient 1 at lag 0 in the
  recursive filter, which gives
  \deqn{y_i = x_i + f_1y_{i-1} + \cdots + f_py_{i-p}}{\code{y[i] =
      x[i] + f[1]*y[i-1] + \dots + f[p]*y[i-p]}}
  No check is made to see if recursive filter is invertible:
  the output may diverge if it is not.

  The convolution filter is
  \deqn{y_i = f_1x_{i+o} + \cdots + f_px_{i+o-p-1}}{\code{y[i] = f[1]*x[i+o] + \dots + f[p]*x[i+o-p-1]}}

  where \code{o} is the offset: see \code{sides} for how it is determined.
}
\note{
  \code{\link{convolve}(, type="filter")} uses the FFT for computations
  and so \emph{may} be faster for long filters on univariate series,
  but it does not return a time series (and so the  time alignment is
  unclear), nor does it handle missing values.  \code{filter} is
  faster for a filter of length 100 on a series of length 1000,
  for example.
}

\seealso{
  \code{\link{convolve}}, \code{\link{arima.sim}}
}

\examples{
x <- 1:100
filter(x, rep(1, 3))
filter(x, rep(1, 3), sides = 1)
filter(x, rep(1, 3), sides = 1, circular = TRUE)
data(presidents)
filter(presidents, rep(1,3))
}
\keyword{ts}
\eof
\name{fisher.test}
\alias{fisher.test}
\title{Fisher's Exact Test for Count Data}
\description{
  Performs Fisher's exact test for testing the null of independence of
  rows and columns in a contingency table with fixed marginals.
}
\usage{
fisher.test(x, y = NULL, workspace = 200000, hybrid = FALSE, control = list(),
            or = 1, alternative = "two.sided", conf.level = 0.95)
}
\arguments{
  \item{x}{either a two-dimensional contingency table in matrix form,
    or a factor object.}
  \item{y}{a factor object; ignored if \code{x} is a matrix.}
  \item{workspace}{an integer specifying the size of the workspace
    used in the network algorithm.}
  \item{hybrid}{a logical indicating whether the exact probabilities
    (default) or a hybrid approximation thereof should be computed.
    In the hybrid case, asymptotic chi-squared probabilities are only
    used provided that the \dQuote{Cochran} conditions are satisfied.}
  \item{control}{a list with named components for low level algorithm control.}
  \item{or}{the hypothesized odds ratio.  Only used in the 2 by 2 case.}
  \item{alternative}{indicates the alternative hypothesis and must be
    one of \code{"two.sided"}, \code{"greater"} or \code{"less"}.
    You can specify just the initial letter.  Only used in the 2 by
    2 case.}
  \item{conf.level}{confidence level for the returned confidence
    interval.  Only used in the 2 by 2 case.}
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{p.value}{the p-value of the test.}
  \item{conf.int}{a confidence interval for the odds ratio.
    Only present in the 2 by 2 case.}
  \item{estimate}{an estimate of the odds ratio.  Note that the
    \emph{conditional} Maximum Likelihood Estimate (MLE) rather than the
    unconditional MLE (the sample odds ratio) is used.
    Only present in the 2 by 2 case.}
  \item{null.value}{the odds ratio under the null, \code{or}.
    Only present in the 2 by 2 case.}
  \item{alternative}{a character string describing the alternative
    hypothesis.}
  \item{method}{the character string
    \code{"Fisher's Exact Test for Count Data"}.}
  \item{data.name}{a character string giving the names of the data.}
}
\details{
  If \code{x} is a matrix, it is taken as a two-dimensional contingency
  table, and hence its entries should be nonnegative integers.
  Otherwise, both \code{x} and \code{y} must be vectors of the same
  length.  Incomplete cases are removed, the vectors are coerced into
  factor objects, and the contingency table is computed from these.

  In the one-sided 2 by 2 cases, p-values are obtained directly using
  the hypergeometric distribution.  Otherwise, computations are based on
  a C version of the FORTRAN subroutine FEXACT which implements the
  network developed by Mehta and Patel (1986) and improved by Clarkson,
  Fan & Joe (1993).  The FORTRAN code can be obtained from
  \url{http://www.netlib.org/toms/643}.  Note this fails (with an error
  message) when the entries of the table are too large.

  In the 2 by 2 case, the null of conditional independence is equivalent
  to the hypothesis that the odds ratio equals one.  Exact inference can
  be based on observing that in general, given all marginal totals
  fixed, the first element of the contingency table has a non-central
  hypergeometric distribution with non-centrality parameter given by the
  odds ratio (Fisher, 1935).
}
\references{
  Alan Agresti (1990).
  \emph{Categorical data analysis}.
  New York: Wiley.
  Pages 59--66.

  Fisher, R. A. (1935).
  The logic of inductive inference.
  \emph{Journal of the Royal Statistical Society Series A} \bold{98},
  39--54.

  Fisher, R. A. (1962).
  Confidence limits for a cross-product ratio.
  \emph{Australian Journal of Statistics} \bold{4}, 41.

  Cyrus R. Mehta & Nitin R. Patel (1986).
  Algorithm 643. FEXACT: A Fortran subroutine for Fisher's exact test
  on unordered \eqn{r*c} contingency tables.
  \emph{ACM Transactions on Mathematical Software}, \bold{12},
  154--161.

  Douglas B. Clarkson, Yuan-an Fan & Harry Joe (1993).
  A Remark on Algorithm 643: FEXACT: An Algorithm for Performing
  Fisher's Exact Test in \eqn{r \times c}{r x c} Contingency Tables.
  \emph{ACM Transactions on Mathematical Software}, \bold{19},
  484--488.
}
\seealso{
  \code{\link{chisq.test}}
}
\examples{
## Agresti (1990), p. 61f, Fisher's Tea Drinker
## A British woman claimed to be able to distinguish whether milk or
##  tea was added to the cup first.  To test, she was given 8 cups of
##  tea, in four of which milk was added first.  The null hypothesis
##  is that there is no association between the true order of pouring
##  and the women's guess, the alternative that there is a positive
##  association (that the odds ratio is greater than 1).
TeaTasting <-
matrix(c(3, 1, 1, 3),
       nr = 2,
       dimnames = list(Guess = c("Milk", "Tea"),
                       Truth = c("Milk", "Tea")))
fisher.test(TeaTasting, alternative = "greater")
## => p=0.2429, association could not be established

## Fisher (1962), Convictions of like-sex twins in criminals
Convictions <-
matrix(c(2, 10, 15, 3),
       nr = 2,
       dimnames =
       list(c("Dizygotic", "Monozygotic"),
            c("Convicted", "Not convicted")))
Convictions
fisher.test(Convictions, alternative = "less")
fisher.test(Convictions, conf.level = 0.95)$conf.int
fisher.test(Convictions, conf.level = 0.99)$conf.int
}
\keyword{htest}
\eof
\name{fitted}
\title{Extract Model Fitted Values}
\usage{
fitted(object, \dots)
fitted.values(object, \dots)
}
\alias{fitted.values}
\alias{fitted}
%%- methods:
\alias{fitted.default}
%\alias{fitted.lm}
%\alias{fitted.glm}
\arguments{
  \item{object}{an object for which the extraction of model fitted values is
    meaningful.}
  \item{\dots}{other arguments.}
}
\description{
  \code{fitted} is a generic function which extracts fitted values from
  objects returned by modeling functions.  \code{fitted.values} is an
  alias for it.

  All object classes which are returned by model fitting functions
  should provide a \code{fitted} method.  (Note that the generic is
  \code{fitted} and not \code{fitted.values}.)

  Methods can make use of \code{\link{napredict}} methods to compensate
  for the omission of missing values.  The default, \code{lm} and
  \code{glm} methods do.
}
\value{
  Fitted values extracted from the object \code{x}.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical Models in S}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{coefficients}}, \code{\link{glm}}, \code{\link{lm}},
  \code{\link{residuals}}.
}
\keyword{models}
\keyword{regression}
\eof
\name{fivenum}
\alias{fivenum}
\title{Tukey Five-Number Summaries}
\description{
  Returns Tukey's five number summary (minimum, lower-hinge, median,
  upper-hinge, maximum) for the input data.
}
\usage{
fivenum(x, na.rm = TRUE)
}
\arguments{
  \item{x}{numeric, maybe including \code{\link{NA}}s and
    \eqn{\pm}{+/-}\code{\link{Inf}}s.}
  \item{na.rm}{logical; if \code{TRUE}, all \code{\link{NA}} and
    \code{\link{NaN}}s are dropped, before the statistics are computed.}
}
\value{
  A numeric vector of length 5 containing the summary information.  See
  \code{\link{boxplot.stats}} for more details.
}
\seealso{
  \code{\link{IQR}},
  \code{\link{boxplot.stats}},
  \code{\link{median}},
  \code{\link{quantile}},
  \code{\link{range}}.
}
\examples{
fivenum(c(rnorm(100),-1:1/0))
}
\keyword{univar}
\keyword{robust}
\keyword{distribution}

\eof
\name{fligner.test}
\alias{fligner.test}
\alias{fligner.test.default}
\alias{fligner.test.formula}
\title{Fligner-Killeen Test for Homogeneity of Variances}
\description{
  Performs a Fligner-Killeen (median) test of the null that the
  variances in each of the groups (samples) are the same.
}
\usage{
fligner.test(x, \dots)

\method{fligner.test}{default}(x, g, \dots)

\method{fligner.test}{formula}(formula, data, subset, na.action, \dots)
}
\arguments{
  \item{x}{a numeric vector of data values, or a list of numeric data
    vectors.}
  \item{g}{a vector or factor object giving the group for the
    corresponding elements of \code{x}.
    Ignored if \code{x} is a list.}
  \item{formula}{a formula of the form \code{lhs ~ rhs} where \code{lhs}
    gives the data values and \code{rhs} the corresponding groups.}
  \item{data}{an optional data frame containing the variables in the
    model formula.}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.  Defaults to
    \code{getOption("na.action")}.}
  \item{\dots}{further arguments to be passed to or from methods.}  
}
\details{
  If \code{x} is a list, its elements are taken as the samples to be
  compared for homogeneity of variances, and hence have to be numeric
  data vectors.  In this case, \code{g} is ignored, and one can simply
  use \code{fligner.test(x)} to perform the test.  If the samples are
  not yet contained in a list, use \code{fligner.test(list(x, ...))}.

  Otherwise, \code{x} must be a numeric data vector, and \code{g} must
  be a vector or factor object of the same length as \code{x} giving the
  group for the corresponding elements of \code{x}.
  
  The Fligner-Killeen (median) test has been determined in a simulation
  study as one of the many tests for homogeneity of variances which is
  most robust against departures from normality, see Conover, Johnson &
  Johnson (1981).  It is a \eqn{k}-sample simple linear rank which uses
  the ranks of the absolute values of the centered samples and weights
  \eqn{a(i) = \mathrm{qnorm}((1 + i/(n+1))/2)}{a(i) = qnorm((1 +
    i/(n+1))/2)}.  The version implemented here uses median centering in
  each of the samples (F-K:med \eqn{X^2} in the reference).
}
\value{
  A list of class \code{"htest"} containing the following components:
  \item{statistic}{the Fligner-Killeen:med \eqn{X^2} test statistic.}
  \item{parameter}{the degrees of freedom of the approximate chi-squared
    distribution of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{method}{the character string
    \code{"Fligner-Killeen test for homogeneity of variances"}.}
  \item{data.name}{a character string giving the names of the data.}
}
\references{
  William J. Conover & Mark E. Johnson & Myrle M. Johnson (1981).
  A comparative study of tests for homogeneity of variances, with
  applications to the outer continental shelf bidding data.
  \emph{Technometrics} \bold{23}, 351--361.
}
\seealso{
  \code{\link{ansari.test}} and \code{\link{mood.test}} for rank-based
  two-sample test for a difference in scale parameters;
  \code{\link{var.test}} and \code{\link{bartlett.test}} for parametric
  tests for the homogeneity of variances.
}
\examples{
% FIXME: could use something more interesting here
data(InsectSprays)
plot(count ~ spray, data = InsectSprays)
fligner.test(InsectSprays$count, InsectSprays$spray)
fligner.test(count ~ spray, data = InsectSprays)
## Compare this to bartlett.test()
}
\keyword{htest}
\eof
\name{formula}
\title{Model Formulae}
\usage{
formula(x, \dots)
as.formula(object, env = parent.frame())
}
\alias{formula}
\alias{formula.default}
\alias{formula.formula}
\alias{formula.terms}
\alias{formula.data.frame}
\alias{as.formula}
\alias{print.formula}
\alias{[.formula}
\description{
  The generic function \code{formula} and its specific methods provide a
  way of extracting formulae which have been included in other objects.

  \code{as.formula} is almost identical, additionally preserving
  attributes when \code{object} already inherits from
  \code{"formula"}. The default value of the \code{env} argument is
  used only when the formula would otherwise lack an environment.
}
\arguments{
  \item{x, object}{\R object.}
  \item{\dots}{further arguments passed to or from other methods.}
  \item{env}{the environment to associate with the result.}
}
\details{
  The models fit by, e.g., the \code{\link{lm}} and \code{\link{glm}} functions
  are specified in a compact symbolic form.
  The \code{~} operator is basic in the formation of such models.
  An expression of the form \code{y ~ model} is interpreted
  as a specification that the response \code{y} is modelled
  by a linear predictor specified symbolically by \code{model}.
  Such a model consists of a series of terms separated
  by \code{+} operators.
  The terms themselves consist of variable and factor
  names separated by \code{:} operators.
  Such a term is interpreted as the interaction of
  all the variables and factors appearing in the term.

  In addition to \code{+} and \code{:}, a number of other operators are
  useful in model formulae.  The \code{*} operator denotes factor
  crossing: \code{a*b} interpreted as \code{a+b+a:b}.  The \code{^}
  operator indicates crossing to the specified degree.  For example
  \code{(a+b+c)^2} is identical to \code{(a+b+c)*(a+b+c)} which in turn
  expands to a formula containing the main effects for \code{a},
  \code{b} and \code{c} together with their second-order interactions.
  The \code{\%in\%} operator indicates that the terms on its left are
  nested within those on the right.  For example \code{a + b \%in\% a}
  expands to the formula \code{a + a:b}.  The \code{-} operator removes
  the specified terms, so that \code{(a+b+c)^2 - a:b} is identical to
  \code{a + b + c + b:c + a:c}.  It can also used to remove the intercept
  term: \code{y ~ x - 1} is a line through the origin.  A model with no
  intercept can be also specified as \code{y ~ x + 0} or \code{y ~ 0 + x}.

  While formulae usually involve just variable and factor
  names, they can also involve arithmetic expressions.
  The formula \code{log(y) ~ a + log(x)} is quite legal.
  When such arithmetic expressions involve
  operators which are also used symbolically
  in model formulae, there can be confusion between
  arithmetic and symbolic operator use.

  To avoid this confusion, the function \code{\link{I}()}
  can be used to bracket those portions of a model
  formula where the operators are used in their
  arithmetic sense.  For example, in the formula
  \code{y ~ a + I(b+c)}, the term \code{b+c} is to be
  interpreted as the sum of \code{b} and \code{c}.

  As from \R 1.8.0 variable names can be quoted by backticks \code{`like
    this`} in formulae, although there is no guarantee that all code
  using formulae will accept such non-syntactic names.

  When \code{formula} is called on a fitted model object, either a
  specific method is used (such as that for class \code{"nls"}) of the
  default method.  The default first looks for a \code{"formula"}
  component of the object (and evaluates it), then a \code{"terms"}
  component, then a \code{formula} parameter of the call (and evaluates
  its value) and finally a \code{"formula"} attribute.
}
\section{Environments}{
  A formula object has an associated environment, and
  this environment (rather than the parent
  environment) is used by \code{\link{model.frame}} to evaluate variables
  that are not found in the supplied \code{data} argument.

  Formulas created with the \code{~} operator use the
  environment in which they were created. Formulas created with
  \code{as.formula} will use the \code{env} argument for their
  environment. Pre-existing formulas extracted with
  \code{as.formula} will only have their environment changed if
  \code{env} is given explicitly.
}
\value{
  All the functions above produce an object of class \code{"formula"}
  which contains a symbolic model formula.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical models.}
  Chapter 2 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{I}}.
  
  For formula manipulation: \code{\link{terms}}, and \code{\link{all.vars}};
  for typical use: \code{\link{lm}}, \code{\link{glm}}, and
  \code{\link{coplot}}.
}
\examples{
class(fo <- y ~ x1*x2) # "formula"
fo
typeof(fo)# R internal : "language"
terms(fo)

environment(fo)
environment(as.formula("y ~ x"))
environment(as.formula("y ~ x",env=new.env()))


## Create a formula for a model with a large number of variables:
xnam <- paste("x", 1:25, sep="")
(fmla <- as.formula(paste("y ~ ", paste(xnam, collapse= "+"))))
}
\keyword{models}
\eof
% $Id: formula.nls.Rd,v 1.2 2003/12/11 17:27:18 ripley Exp $
\name{formula.nls}
\alias{formula.nls}
\title{Extract Model Formula from nls Object}
\usage{
\method{formula}{nls}(x, \dots)
}
\arguments{
 \item{x}{an object inheriting from class \code{"nls"}, representing
   a nonlinear least squares fit.}
 \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  Returns the model used to fit \code{object}. 
}
\value{
  a formula representing the model used to obtain \code{object}.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{\code{\link{nls}}, \code{\link{formula}}}
\examples{
data(Orange)
fm1 <- nls(circumference ~ A/(1+exp((B-age)/C)), Orange,
  start = list(A=160, B=700, C = 350))
formula(fm1)
}
\keyword{models}
\eof
\name{freeny}
\docType{data}
\alias{freeny}
\alias{freeny.x}
\alias{freeny.y}
\title{Freeny's Revenue Data}
\description{
  Freeny's data on quarterly revenue and explanatory variables.
}
\usage{data(freeny)}
\format{
  There are three \sQuote{freeny} data sets.

  \code{freeny.y} is a time series with 39 observations on quarterly
  revenue from (1962,2Q) to (1971,4Q).
  
  \code{freeny.x} is a matrix of explanatory variables.  The columns
  are \code{freeny.y} lagged 1 quarter, price index, income level, and
  market potential.

  Finally, \code{freeny} is a data frame with variables \code{y},
  \code{lag.quarterly.revenue}, \code{price.index}, \code{income.level},
  and \code{market.potential} obtained from the above two data objects.
}
\source{
  A. E. Freeny (1977)
  \emph{A Portable Linear Regression Package with Test Programs}.
  Bell Laboratories memorandum.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\examples{
data(freeny)
summary(freeny)
pairs(freeny, main = "freeny data")
summary(fm1 <- lm(y ~ ., data = freeny))
opar <- par(mfrow = c(2, 2), oma = c(0, 0, 1.1, 0),
            mar = c(4.1, 4.1, 2.1, 1.1))
plot(fm1)
par(opar)
}
\keyword{datasets}
\eof
\name{friedman.test}
\alias{friedman.test}
\alias{friedman.test.default}
\alias{friedman.test.formula}
\title{Friedman Rank Sum Test}
\description{
  Performs a Friedman rank sum test with unreplicated blocked data.
}
\usage{
friedman.test(y, \dots)

\method{friedman.test}{default}(y, groups, blocks, \dots)

\method{friedman.test}{formula}(formula, data, subset, na.action, \dots)
}
\arguments{
  \item{y}{either a numeric vector of data values, or a data matrix.}
  \item{groups}{a vector giving the group for the corresponding
    elements of \code{y} if this is a vector;  ignored if \code{y}
    is a matrix.  If not a factor object, it is coerced to one.}
  \item{blocks}{a vector giving the block for the corresponding
    elements of \code{y} if this is a vector;  ignored if \code{y}
    is a matrix.  If not a factor object, it is coerced to one.}
  \item{formula}{a formula of the form \code{a ~ b | c}, where \code{a},
    \code{b} and \code{c} give the data values and corresponding groups
    and blocks, respectively.}
  \item{data}{an optional data frame containing the variables in the
    model formula.}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.  Defaults to
    \code{getOption("na.action")}.}
  \item{\dots}{further arguments to be passed to or from methods.}    
}
\details{
  \code{friedman.test} can be used for analyzing unreplicated complete
  block designs (i.e., there is exactly one observation in \code{y}
  for each combination of levels of \code{groups} and \code{blocks})
  where the normality assumption may be violated.

  The null hypothesis is that apart from an effect of \code{blocks},
  the location parameter of \code{y} is the same in each of the
  \code{groups}. 

  If \code{y} is a matrix, \code{groups} and \code{blocks} are
  obtained from the column and row indices, respectively.  \code{NA}'s
  are not allowed in \code{groups} or \code{blocks};  if \code{y}
  contains \code{NA}'s, corresponding blocks are removed.
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of Friedman's chi-squared statistic.}
  \item{parameter}{the degrees of freedom of the approximate
    chi-squared distribution of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{method}{the character string \code{"Friedman rank sum test"}.}
  \item{data.name}{a character string giving the names of the data.}
}
\references{
  Myles Hollander & Douglas A. Wolfe (1973),
  \emph{Nonparametric statistical inference}.
  New York: John Wiley & Sons.
  Pages 139--146.
}
\seealso{
  \code{\link{quade.test}}.
}
\examples{
## Hollander & Wolfe (1973), p. 140ff.
## Comparison of three methods ("round out", "narrow angle", and
##  "wide angle") for rounding first base.  For each of 18 players
##  and the three method, the average time of two runs from a point on
##  the first base line 35ft from home plate to a point 15ft short of
##  second base is recorded.
RoundingTimes <-
matrix(c(5.40, 5.50, 5.55,
         5.85, 5.70, 5.75,
         5.20, 5.60, 5.50,
         5.55, 5.50, 5.40,
         5.90, 5.85, 5.70,
         5.45, 5.55, 5.60,
         5.40, 5.40, 5.35,
         5.45, 5.50, 5.35,
         5.25, 5.15, 5.00,
         5.85, 5.80, 5.70,
         5.25, 5.20, 5.10,
         5.65, 5.55, 5.45,
         5.60, 5.35, 5.45,
         5.05, 5.00, 4.95,
         5.50, 5.50, 5.40,
         5.45, 5.55, 5.50,
         5.55, 5.55, 5.35,
         5.45, 5.50, 5.55,
         5.50, 5.45, 5.25,
         5.65, 5.60, 5.40,
         5.70, 5.65, 5.55,
         6.30, 6.30, 6.25),
       nr = 22,
       byrow = TRUE,
       dimnames = list(1 : 22,
                       c("Round Out", "Narrow Angle", "Wide Angle")))
friedman.test(RoundingTimes)
## => strong evidence against the null that the methods are equivalent
##    with respect to speed

data(warpbreaks)
wb <- aggregate(warpbreaks$breaks,
                by = list(w = warpbreaks$wool,
                          t = warpbreaks$tension),
                FUN = mean)
wb
friedman.test(wb$x, wb$w, wb$t)
friedman.test(x ~ w | t, data = wb)
}
\keyword{htest}
\eof
\name{ftable}
\alias{ftable}
\alias{ftable.default}
\alias{print.ftable}
\title{Flat Contingency Tables}
\description{Create \dQuote{flat} contingency tables.}
\usage{
ftable(x, \dots)

\method{ftable}{default}(\dots, exclude = c(NA, NaN), row.vars = NULL, col.vars = NULL)
}
\arguments{
  \item{x, \dots}{\R objects which can be interpreted as factors (including
    character strings), or a list (or data frame) whose components can
    be so interpreted, or a contingency table object of class
    \code{"table"} or \code{"ftable"}.}
  \item{exclude}{values to use in the exclude argument of \code{factor}
    when interpreting non-factor objects.}
  \item{row.vars}{a vector of integers giving the numbers of the
    variables, or a character vector giving the names of the variables
    to be used for the rows of the flat contingency table.}
  \item{col.vars}{a vector of integers giving the numbers of the
    variables, or a character vector giving the names of the variables
    to be used for the columns of the flat contingency table.}
}
\value{
  \code{ftable} returns an object of class \code{"ftable"}, which is a
  matrix with counts of each combination of the levels of variables with
  information on the names and levels of the (row and columns) variables
  stored as attributes \code{"row.vars"} and \code{"col.vars"}.
}
\details{
  \code{ftable} creates \dQuote{flat} contingency tables.  Similar to the
  usual contingency tables, these contain the counts of each combination
  of the levels of the variables (factors) involved.  This information
  is then re-arranged as a matrix whose rows and columns correspond to
  unique combinations of the levels of the row and column variables (as
  specified by \code{row.vars} and \code{col.vars}, respectively).  The
  combinations are created by looping over the variables in reverse
  order (so that the levels of the \dQuote{left-most} variable vary the
  slowest).  Displaying a contingency table in this flat matrix form
  (via \code{print.ftable}, the print method for objects of class
  \code{"ftable"}) is often preferable to showing it as a
  higher-dimensional array.

  \code{ftable} is a generic function.  Its default method,
  \code{ftable.default}, first creates a contingency table in array
  form from all arguments except \code{row.vars} and \code{col.vars}.
  If the first argument is of class \code{"table"}, it represents a
  contingency table and is used as is; if it is a flat table of class
  \code{"ftable"}, the information it contains is converted to the usual
  array representation using \code{as.ftable}.  Otherwise, the arguments
  should be \R objects which can be interpreted as factors (including
  character strings), or a list (or data frame) whose components can be
  so interpreted, which are cross-tabulated using \code{\link{table}}.
  Then, the arguments \code{row.vars} and \code{col.vars} are used to
  collapse the contingency table into flat form.  If neither of these
  two is given, the last variable is used for the columns.  If both are
  given and their union is a proper subset of all variables involved,
  the other variables are summed out.

  When the arguments are \R expressions interpreted as factors,
  additional arguments will be passed to \code{table} to control how
  the variable names are displayed; see the last example below.

  Function \code{\link{ftable.formula}} provides a formula method for
  creating flat contingency tables.
}
\seealso{
  \code{\link{ftable.formula}} for the formula interface (which allows a
  \code{data = .} argument);
  \code{\link{read.ftable}} for information on reading, writing and
  coercing flat contingency tables;
  \code{\link{table}} for \dQuote{ordinary} cross-tabulation;
  \code{\link{xtabs}} for formula-based cross-tabulation.
}
\examples{
## Start with a contingency table.
data(Titanic)
ftable(Titanic, row.vars = 1:3)
ftable(Titanic, row.vars = 1:2, col.vars = "Survived")
ftable(Titanic, row.vars = 2:1, col.vars = "Survived")

## Start with a data frame.
data(mtcars)
x <- ftable(mtcars[c("cyl", "vs", "am", "gear")])
x
ftable(x, row.vars = c(2, 4))

## Start with expressions, use table()'s "dnn" to change labels
ftable(mtcars$cyl, mtcars$vs, mtcars$am, mtcars$gear, row.vars = c(2, 4), 
       dnn = c("Cylinders", "V/S", "Transmission", "Gears"))
 }
\keyword{category}
\eof
\name{ftable.formula}
\alias{ftable.formula}
\title{Formula Notation for Flat Contingency Tables}
\description{Produce or manipulate a flat contingency table using
  formula notation.}
\usage{
\method{ftable}{formula}(formula, data = NULL, subset, na.action, \dots)
}
\arguments{
  \item{formula}{a formula object with both left and right hand sides
    specifying the column and row variables of the flat table.}
  \item{data}{a data frame, list or environment containing the variables
    to be cross-tabulated, or a contingency table (see below).}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.
    Ignored if \code{data} is a contingency table.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.
    Ignored if \code{data} is a contingency table.}
  \item{\dots}{further arguments to the default ftable method may also
    be passed as arguments, see \code{\link{ftable.default}}.}
}
\details{
  This is a method of the generic function \code{\link{ftable}}.

  The left and right hand side of \code{formula} specify the column and
  row variables, respectively, of the flat contingency table to be
  created.  Only the \code{+} operator is allowed for combining the
  variables.  A \code{.} may be used once in the formula to indicate
  inclusion of all the \dQuote{remaining} variables.

  If \code{data} is an object of class \code{"table"} or an array with
  more than 2 dimensions, it is taken as a contingency table, and hence
  all entries should be nonnegative.  Otherwise, if it is not a flat
  contingency table (i.e., an object of class \code{"ftable"}), it
  should be a data frame or matrix, list or environment containing the
  variables to be cross-tabulated.  In this case, \code{na.action} is
  applied to the data to handle missing values, and, after possibly
  selecting a subset of the data as specified by the \code{subset}
  argument, a contingency table is computed from the variables.

  The contingency table is then collapsed to a flat table, according to
  the row and column variables specified by \code{formula}.
}
\value{
  A flat contingency table which contains the counts of each combination
  of the levels of the variables, collapsed into a matrix for suitably
  displaying the counts.
}
\seealso{
  \code{\link{ftable}},
  \code{\link{ftable.default}};
  \code{\link{table}}.
}
\examples{
data(Titanic)
Titanic
x <- ftable(Survived ~ ., data = Titanic)
x
ftable(Sex ~ Class + Age, data = x)
}
\keyword{category}
\eof
% $Id: getInitial.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{getInitial}
\title{Get Initial Parameter Estimates}
\alias{getInitial}
\alias{getInitial.default}
\alias{getInitial.formula}
\alias{getInitial.selfStart}
\usage{
getInitial(object, data, \dots)
}
\arguments{
  \item{object}{a formula or a \code{selfStart} model that defines a
    nonlinear regression model}
  \item{data}{a data frame in which the expressions in the formula or
    arguments to the \code{selfStart} model can be evaluated}
  \item{\dots}{optional additional arguments}
}
\description{
  This function evaluates initial parameter estimates for a nonlinear
  regression model.  If \code{data} is a parameterized data frame or
  \code{pframe} object, its \code{parameters} attribute is returned.
  Otherwise the object is examined to see if it contains a call to a
  \code{selfStart} object whose \code{initial} attribute can be
  evaluated.
}
\value{
    A named numeric vector or list of starting estimates for the
    parameters.  The construction of many \code{selfStart} models is such
    that these "starting" estimates are, in fact, the converged parameter
    estimates.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{\code{\link{nls}}, %\code{\link{nlsList}},
    \code{\link{selfStart}},
    \code{\link{selfStart.default}}, \code{\link{selfStart.formula}}
}
\examples{
data(Puromycin)
PurTrt <- Puromycin[ Puromycin$state == "treated", ]
getInitial( rate ~ SSmicmen( conc, Vm, K ), PurTrt )
}
\keyword{models}
\keyword{nonlinear}
\keyword{manip}
\eof
\name{glm}
\title{Fitting Generalized Linear Models}
\alias{glm}
\alias{glm.fit}
\alias{weights.glm}
\alias{print.glm}
\concept{regression}
\concept{logistic}
\concept{log-linear}
\concept{loglinear}
\description{
  \code{glm} is used to fit generalized linear models, specified by
  giving a symbolic description of the linear predictor and a
  description of the error distribution.
}
\usage{
glm(formula, family = gaussian, data, weights, subset,
    na.action, start = NULL, etastart, mustart,
    offset, control = glm.control(\dots), model = TRUE,
    method = "glm.fit", x = FALSE, y = TRUE, contrasts = NULL, \dots)

glm.fit(x, y, weights = rep(1, nobs),
        start = NULL, etastart = NULL, mustart = NULL,
        offset = rep(0, nobs), family = gaussian(),
        control = glm.control(), intercept = TRUE)

\method{weights}{glm}(object, type = c("prior", "working"), \dots)
}
\arguments{
  \item{formula}{a symbolic description of the model to be fit.
    The details of model specification are given below.}

  \item{family}{a description of the error distribution and link
    function to be used in the model. This can be a character string
    naming a family function, a family function or the result of a call
    to a family function.  (See \code{\link{family}} for details of
    family functions.)}

  \item{data}{an optional data frame containing the variables
    in the model.  If not found in \code{data}, the variables are taken
    from \code{environment(formula)}, typically the environment from
    which \code{glm} is called.}

  \item{weights}{an optional vector of weights to be used
    in the fitting process.}

  \item{subset}{an optional vector specifying a subset of observations
    to be used in the fitting process.}

  \item{na.action}{a function which indicates what should happen
    when the data contain \code{NA}s.  The default is set by
    the \code{na.action} setting of \code{\link{options}}, and is
    \code{\link{na.fail}} if that is unset.  The \dQuote{factory-fresh}
    default is \code{\link{na.omit}}.}

  \item{start}{starting values for the parameters in the linear predictor.}

  \item{etastart}{starting values for the linear predictor.}

  \item{mustart}{starting values for the vector of means.}

  \item{offset}{this can be used to specify an \emph{a priori}
    known component to be included in the linear predictor
    during fitting.}

  \item{control}{a list of parameters for controlling the fitting
    process.  See the documentation for \code{\link{glm.control}}
    for details.}

  \item{model}{a logical value indicating whether \emph{model frame}
    should be included as a component of the returned value.}

  \item{method}{the method to be used in fitting the model.
    The default method \code{"glm.fit"} uses iteratively reweighted
    least squares (IWLS).  The only current alternative is
    \code{"model.frame"} which returns the model frame and does no fitting.}

  \item{x, y}{For \code{glm}:
    logical values indicating whether the response
    vector and model matrix used in the fitting process
    should be returned as components of the returned value.

    For \code{glm.fit}: \code{x} is a design matrix of dimension \code{n
      * p}, and \code{y} is a vector of observations of length \code{n}.
  }

  \item{contrasts}{an optional list. See the \code{contrasts.arg}
    of \code{model.matrix.default}.}

  \item{object}{an object inheriting from class \code{"glm"}.}
  \item{type}{character, partial matching allowed.  Type of weights to
    extract from the fitted model object.}

  \item{intercept}{logical. Should an intercept be included?}

  \item{\dots}{further arguments passed to or from other methods.}
}
\details{
  A typical predictor has the form \code{response ~ terms} where
  \code{response} is the (numeric) response vector and \code{terms} is a
  series of terms which specifies a linear predictor for \code{response}.
  For \code{binomial} models the response can also be specified as a
  \code{\link{factor}} (when the first level denotes failure and all
  others success) or as a two-column matrix with the columns giving the
  numbers of successes and failures.  A terms specification of the form
  \code{first + second} indicates all the terms in \code{first} together
  with all the terms in \code{second} with duplicates removed.  The terms in
  the formula will be re-ordered so that main effects come first,
  followed by the interactions, all second-order, all third-order and so
  on: to avoid this pass a \code{terms} object as the formula.

  A specification of the form \code{first:second} indicates the
  the set of terms obtained by taking the interactions of
  all terms in \code{first} with all terms in \code{second}.
  The specification \code{first*second} indicates the \emph{cross}
  of \code{first} and \code{second}.
  This is the same as \code{first + second + first:second}.

  \code{glm.fit} and \code{glm.fit.null} are the workhorse functions:
  the former calls the latter for a null model (with no intercept).

  If more than one of \code{etastart}, \code{start} and \code{mustart}
  is specified, the first in the list will be used.

  All of \code{weights}, \code{subset}, \code{offset}, \code{etastart}
  and \code{mustart} are evaluated in the same way as variables in
  \code{formula}, that is first in \code{data} and then in the
  environment of \code{formula}.
}
\value{
  \code{glm} returns an object of class inheriting from \code{"glm"}
  which inherits from the class \code{"lm"}. See later in this section.

  The function \code{\link{summary}} (i.e., \code{\link{summary.glm}}) can
  be used to obtain or print a summary of the results and the function
  \code{\link{anova}} (i.e., \code{\link{anova.glm}})
  to produce an analysis of variance table.

  The generic accessor functions \code{\link{coefficients}},
  \code{effects}, \code{fitted.values} and \code{residuals} can be used to
  extract various useful features of the value returned by \code{glm}.

  \code{weights} extracts a vector of weights, one for each case in the
  fit (after subsetting and \code{na.action}).

  An object of class \code{"glm"} is a list containing at least the
  following components:

  \item{coefficients}{a named vector of coefficients}
  \item{residuals}{the \emph{working} residuals, that is the residuals
    in the final iteration of the IWLS fit.}
  \item{fitted.values}{the fitted mean values, obtained by transforming
    the linear predictors by the inverse of the link function.}
  \item{rank}{the numeric rank of the fitted linear model.}
  \item{family}{the \code{\link{family}} object used.}
  \item{linear.predictors}{the linear fit on link scale.}
  \item{deviance}{up to a constant, minus twice the maximized
    log-likelihood.  Where sensible, the constant is chosen so that a
    saturated model has deviance zero.}
  \item{aic}{Akaike's \emph{An Information Criterion}, minus twice the
    maximized log-likelihood plus twice the number of coefficients (so
    assuming that the dispersion is known.}
  \item{null.deviance}{The deviance for the null model, comparable with
    \code{deviance}. The null model will include the offset, and an
    intercept if there is one in the model}
  \item{iter}{the number of iterations of IWLS used.}
  \item{weights}{the \emph{working} weights, that is the weights
    in the final iteration of the IWLS fit.}
  \item{prior.weights}{the case weights initially supplied.}
  \item{df.residual}{the residual degrees of freedom.}
  \item{df.null}{the residual degrees of freedom for the null model.}
  \item{y}{the \code{y} vector used. (It is a vector even for a binomial
    model.)}
  \item{converged}{logical. Was the IWLS algorithm judged to have converged?}
  \item{boundary}{logical. Is the fitted value on the boundary of the
    attainable values?}
  \item{call}{the matched call.}
  \item{formula}{the formula supplied.}
  \item{terms}{the \code{\link{terms}} object used.}
  \item{data}{the \code{data argument}.}
  \item{offset}{the offset vector used.}
  \item{control}{the value of the \code{control} argument used.}
  \item{method}{the name of the fitter function used, in \R always
    \code{"glm.fit"}.}
  \item{contrasts}{(where relevant) the contrasts used.}
  \item{xlevels}{(where relevant) a record of the levels of the factors
    used in fitting.}

  In addition, non-empty fits will have components \code{qr}, \code{R}
  and \code{effects} relating to the final weighted linear fit.

  Objects of class \code{"glm"} are normally of class \code{c("glm",
    "lm")}, that is inherit from class \code{"lm"}, and well-designed
  methods for class \code{"lm"} will be applied to the weighted linear
  model at the final iteration of IWLS.  However, care is needed, as
  extractor functions for class \code{"glm"} such as
  \code{\link{residuals}} and \code{weights} do \bold{not} just pick out
  the component of the fit with the same name.

  If a \code{\link{binomial}} \code{glm} model is specified by giving a
  two-column response, the weights returned by \code{prior.weights} are
  the total numbers of cases (factored by the supplied case weights) and
  the component \code{y} of the result is the proportion of successes.
}
\seealso{
  \code{\link{anova.glm}}, \code{\link{summary.glm}}, etc. for
  \code{glm} methods,
  and the generic functions \code{\link{anova}}, \code{\link{summary}},
  \code{\link{effects}}, \code{\link{fitted.values}},
  and \code{\link{residuals}}. Further, \code{\link{lm}} for
  non-generalized \emph{linear} models.

  \code{\link{esoph}}, \code{\link{infert}} and
  \code{\link{predict.glm}} have examples of fitting binomial glms.
}
\author{
  The original \R implementation of \code{glm} was written by Simon
  Davies working for Ross Ihaka at the University of Auckland, but has
  since been extensively re-written by members of the R Core team.

  The design was inspired by the S function of the same name described
  in Hastie \& Pregibon (1992).
}
\references{
  Dobson, A. J. (1990)
  \emph{An Introduction to Generalized Linear Models.}
  London: Chapman and Hall.

  Hastie, T. J. and Pregibon, D. (1992)
  \emph{Generalized linear models.}
  Chapter 6 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.

  McCullagh P. and Nelder, J. A. (1989)
  \emph{Generalized Linear Models.}
  London: Chapman and Hall.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.}
  New York: Springer.
}

\examples{
## Dobson (1990) Page 93: Randomized Controlled Trial :
counts <- c(18,17,15,20,10,20,25,13,12)
outcome <- gl(3,1,9)
treatment <- gl(3,3)
print(d.AD <- data.frame(treatment, outcome, counts))
glm.D93 <- glm(counts ~ outcome + treatment, family=poisson())
anova(glm.D93)
summary(glm.D93)

## an example with offsets from Venables & Ripley (2002, p.189)
\dontshow{"anorexia" <-
structure(list(Treat = structure(c(2, 2, 2, 2, 2, 2, 2, 2, 2,
2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1, 1, 1, 1,
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
1, 1, 1, 1, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3
), .Label = c("CBT", "Cont", "FT"), class = "factor"), Prewt = c(80.7,
89.4, 91.8, 74, 78.1, 88.3, 87.3, 75.1, 80.6, 78.4, 77.6, 88.7,
81.3, 78.1, 70.5, 77.3, 85.2, 86, 84.1, 79.7, 85.5, 84.4, 79.6,
77.5, 72.3, 89, 80.5, 84.9, 81.5, 82.6, 79.9, 88.7, 94.9, 76.3,
81, 80.5, 85, 89.2, 81.3, 76.5, 70, 80.4, 83.3, 83, 87.7, 84.2,
86.4, 76.5, 80.2, 87.8, 83.3, 79.7, 84.5, 80.8, 87.4, 83.8, 83.3,
86, 82.5, 86.7, 79.6, 76.9, 94.2, 73.4, 80.5, 81.6, 82.1, 77.6,
83.5, 89.9, 86, 87.3), Postwt = c(80.2, 80.1, 86.4, 86.3, 76.1,
78.1, 75.1, 86.7, 73.5, 84.6, 77.4, 79.5, 89.6, 81.4, 81.8, 77.3,
84.2, 75.4, 79.5, 73, 88.3, 84.7, 81.4, 81.2, 88.2, 78.8, 82.2,
85.6, 81.4, 81.9, 76.4, 103.6, 98.4, 93.4, 73.4, 82.1, 96.7,
95.3, 82.4, 72.5, 90.9, 71.3, 85.4, 81.6, 89.1, 83.9, 82.7, 75.7,
82.6, 100.4, 85.2, 83.6, 84.6, 96.2, 86.7, 95.2, 94.3, 91.5,
91.9, 100.3, 76.7, 76.8, 101.6, 94.9, 75.2, 77.8, 95.5, 90.7,
92.5, 93.8, 91.7, 98)), .Names = c("Treat", "Prewt", "Postwt"
), class = "data.frame", row.names = c("1", "2", "3", "4", "5",
"6", "7", "8", "9", "10", "11", "12", "13", "14", "15", "16",
"17", "18", "19", "20", "21", "22", "23", "24", "25", "26", "27",
"28", "29", "30", "31", "32", "33", "34", "35", "36", "37", "38",
"39", "40", "41", "42", "43", "44", "45", "46", "47", "48", "49",
"50", "51", "52", "53", "54", "55", "56", "57", "58", "59", "60",
"61", "62", "63", "64", "65", "66", "67", "68", "69", "70", "71",
"72"))}
%% "else" (not dontshow)
\dontrun{
## Need the anorexia data from a recent version of the package 'MASS':
library(MASS)
data(anorexia)
}
anorex.1 <- glm(Postwt ~ Prewt + Treat + offset(Prewt),
            family = gaussian, data = anorexia)
summary(anorex.1)

# A Gamma example, from McCullagh & Nelder (1989, pp. 300-2)
clotting <- data.frame(
    u = c(5,10,15,20,30,40,60,80,100),
    lot1 = c(118,58,42,35,27,25,21,19,18),
    lot2 = c(69,35,26,21,18,16,13,12,12))
summary(glm(lot1 ~ log(u), data=clotting, family=Gamma))
summary(glm(lot2 ~ log(u), data=clotting, family=Gamma))

\dontrun{
## for an example of the use of a terms object as a formula
demo(glm.vr)
}}
\keyword{models}
\keyword{regression}
\eof
\name{glm.control}
\alias{glm.control}
\title{Auxiliary for Controlling GLM Fitting}
\description{
  Auxiliary function as user interface for \code{\link{glm}} fitting.
  Typically only used when calling \code{\link{glm}} or \code{\link{glm.fit}}.
}
\usage{
glm.control(epsilon=1e-8, maxit=25, trace=FALSE)
}
\arguments{
  \item{epsilon}{positive convergence tolerance \eqn{epsilon};
    the iterations converge when \eqn{|dev - devold|/(|dev| + 0.1) < epsilon}.}
  \item{maxit}{integer giving the maximal number of IWLS iterations.}
  \item{trace}{logical indicating if output should be produced for each
    iteration.}
}
\details{
  If \code{epsilon} is small, it is also used as the tolerance for the
  least squares solution.

  When \code{trace} is true, calls to \code{\link{cat}} produce the
  output for each IWLS iteration. Hence, \code{\link{options}(digits = *)}
  can be used to increase the precision, see the example.
}
\value{
  A list with the arguments as components.
}
\references{
  Hastie, T. J. and Pregibon, D. (1992)
  \emph{Generalized linear models.}
  Chapter 6 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\seealso{\code{\link{glm.fit}}, the fitting procedure used by
  \code{\link{glm}}.}
\examples{
### A variation on  example(glm) :

## Annette Dobson's example ...
counts <- c(18,17,15,20,10,20,25,13,12)
outcome <- gl(3,1,9)
treatment <- gl(3,3)
oo <- options(digits = 12) # to see more when tracing :
glm.D93X <- glm(counts ~ outcome + treatment, family=poisson(),
                trace = TRUE, epsilon = 1e-14)
options(oo)
coef(glm.D93X) # the last two are closer to 0 than in ?glm's  glm.D93
# put less so than in R < 1.8.0 when the default was 1e-4
}
\keyword{optimize}
\keyword{models}

\eof
\name{glm.summaries}
\alias{family.glm}
\alias{residuals.glm}
\title{Accessing Generalized Linear Model Fits}
\usage{
\method{family}{glm}(object, \dots)

\method{residuals}{glm}(object, type = c("deviance", "pearson", "working",
                      "response", "partial"), \dots)
}
\arguments{
  \item{object}{an object of class \code{glm}, typically the result of
    a call to \code{\link{glm}}.}
  \item{type}{the type of residuals which should be returned.
    The alternatives are: \code{"deviance"} (default), \code{"pearson"},
    \code{"working"}, \code{"response"}, and \code{"partial"}.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  These functions are all \code{\link{methods}} for class \code{glm} or
  \code{summary.glm} objects.
}
\details{
  The references define the types of residuals: Davison \& Snell is a
  good reference for the usages of each.
  
  The partial residuals are a matrix of working residuals, with each
  column formed by omitting a term from the model.
}
\seealso{
  \code{\link{glm}} for computing \code{glm.obj}, \code{\link{anova.glm}};
  the corresponding \emph{generic} functions, \code{\link{summary.glm}},
  \code{\link{coef}}, \code{\link{deviance}},
  \code{\link{df.residual}},
  \code{\link{effects}}, \code{\link{fitted}},
  \code{\link{residuals}}.
}
\references{
  Davison, A. C. and Snell, E. J. (1991)
  \emph{Residuals and diagnostics.}  In: Statistical Theory
  and Modelling. In Honour of Sir David Cox, FRS, eds.
  Hinkley, D. V., Reid, N. and Snell, E. J., Chapman \& Hall.
  
  Hastie, T. J. and Pregibon, D. (1992)
  \emph{Generalized linear models.}
  Chapter 6 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.

  McCullagh P. and Nelder, J. A. (1989)
  \emph{Generalized Linear Models.}
  London: Chapman and Hall.
}
\keyword{models}
\keyword{regression}

\eof
\name{hclust}
\title{Hierarchical Clustering}
\alias{hclust}
\alias{plot.hclust}
\alias{plclust}
\alias{print.hclust}
\description{
  Hierarchical cluster analysis on a set of dissimilarities and
  methods for analyzing it.
}
\usage{
hclust(d, method = "complete", members=NULL)

\method{plot}{hclust}(x, labels = NULL, hang = 0.1,
     axes = TRUE, frame.plot = FALSE, ann = TRUE,
     main = "Cluster Dendrogram",
     sub = NULL, xlab = NULL, ylab = "Height", \dots)

plclust(tree, hang = 0.1, unit = FALSE, level = FALSE, hmin = 0,
        square = TRUE, labels = NULL, plot. = TRUE,
        axes = TRUE, frame.plot = FALSE, ann = TRUE,
        main = "", sub = NULL, xlab = NULL, ylab = "Height")
}
\arguments{
  \item{d}{a dissimilarity structure as produced by \code{dist}.}

  \item{method}{the agglomeration method to be used. This should
    be (an unambiguous abbreviation of) one of
    \code{"ward"}, \code{"single"}, \code{"complete"},
    \code{"average"}, \code{"mcquitty"}, \code{"median"} or
    \code{"centroid"}.}

  \item{members}{\code{NULL} or a vector with length size of
    \code{d}. See the Details section.}

  \item{x,tree}{an object of the type produced by \code{hclust}.}

  \item{hang}{The fraction of the plot height by which labels should hang
    below the rest of the plot.
    A negative value will cause the labels to hang down from 0.}

  \item{labels}{A character vector of labels for the leaves of the
    tree. By default the row names or row numbers of the original data are
    used. If \code{labels=FALSE} no labels at all are plotted.}

  \item{axes, frame.plot, ann}{logical flags as in \code{\link{plot.default}}.}
  \item{main, sub, xlab, ylab}{character strings for
    \code{\link{title}}.  \code{sub} and \code{xlab} have a non-NULL
    default when there's a \code{tree$call}.}
  \item{\dots}{Further graphical arguments.}

  \item{unit}{logical.  If true, the splits are plotted at
    equally-spaced heights rather than at the height in the object.}

  \item{hmin}{numeric.  All heights less than \code{hmin} are regarded
    as being \code{hmin}: this can be used to suppress detail at the
    bottom of the tree.}
  \item{level, square, plot.}{as yet unimplemented arguments of
    \code{plclust} for S-PLUS compatibility.}
}
\value{
  An object of class \bold{hclust} which describes the
  tree produced by the clustering process.
  The object is a list with components:
  \item{merge}{an \eqn{n-1} by 2 matrix.
    Row \eqn{i} of \code{merge} describes the merging of clusters
    at step \eqn{i} of the clustering.
    If an element \eqn{j} in the row is negative,
    then observation \eqn{-j} was merged at this stage.
    If \eqn{j} is positive then the merge
    was with the cluster formed at the (earlier) stage \eqn{j}
    of the algorithm.
    Thus negative entries in \code{merge} indicate agglomerations
    of singletons, and positive entries indicate agglomerations
    of non-singletons.}

  \item{height}{a set of \eqn{n-1} non-decreasing real values.
    The clustering \emph{height}: that is, the value of
    the criterion associated with the clustering
    \code{method} for the particular agglomeration.}

  \item{order}{a vector giving the permutation of the original
    observations suitable for plotting, in the sense that a cluster
    plot using this ordering and matrix \code{merge} will not have
    crossings of the branches.}

  \item{labels}{labels for each of the objects being clustered.}

  \item{call}{the call which produced the result.}

  \item{method}{the cluster method that has been used.}

  \item{dist.method}{the distance that has been used to create \code{d}
    (only returned if the distance object has a \code{"method"}
    attribute).}

  There are \code{\link{print}}, \code{\link{plot}} and \code{identify}
  (see \code{\link{identify.hclust}}) methods and the
  \code{\link{rect.hclust}()} function for \code{hclust} objects.
  The \code{plclust()} function is basically the same as the plot method,
  \code{plot.hclust}, primarily for back compatibility with S-plus. Its
  extra arguments are not yet implemented.
}
\details{
  This function performs a hierarchical cluster analysis
  using a set of dissimilarities for the \eqn{n} objects being
  clustered.  Initially, each object is assigned to its own
  cluster and then the algorithm proceeds iteratively,
  at each stage joining the two most similar clusters,
  continuing until there is just a single cluster.
  At each stage distances between clusters are recomputed
  by the Lance--Williams dissimilarity update formula
  according to the particular clustering method being used.

  A number of different clustering methods are provided.
  \emph{Ward's} minimum variance method aims at finding compact,
  spherical clusters.  The \emph{complete linkage} method finds
  similar clusters. The \emph{single linkage} method
  (which is closely related to the minimal spanning tree)
  adopts a \sQuote{friends of friends} clustering strategy.
  The other methods can be regarded as aiming
  for clusters with characteristics somewhere between
  the single and complete link methods.

  If \code{members!=NULL}, then \code{d} is taken to be a
  dissimilarity matrix between clusters instead of dissimilarities
  between singletons and \code{members} gives the number of observations
  per cluster.  This way the hierarchical cluster algorithm can be
  \dQuote{started in the middle of the dendrogram}, e.g., in order to
  reconstruct the part of the tree above a cut (see examples).
  Dissimilarities between clusters can be efficiently computed (i.e.,
  without \code{hclust} itself) only for a limited number of
  distance/linkage combinations, the simplest one being squared
  Euclidean distance and centroid linkage.  In this case the
  dissimilarities between the clusters are the squared Euclidean
  distances between cluster means.

  In hierarchical cluster displays, a decision is needed at each merge to
  specify which subtree should go on the left and which on the right.
  Since, for \eqn{n} observations there are \eqn{n-1} merges,
  there are \eqn{2^{(n-1)}} possible orderings for the leaves
  in a cluster tree, or dendrogram.
  The algorithm used in \code{hclust} is to order the subtree so that
  the tighter cluster is on the left (the last, i.e., most recent,
  merge of the left subtree is at a lower value than the last
  merge of the right subtree).
  Single observations are the tightest clusters possible,
  and merges involving two observations place them in order by their
  observation sequence number.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole. (S version.)

  Everitt, B. (1974).
  \emph{Cluster Analysis}.
  London: Heinemann Educ. Books.

  Hartigan, J. A. (1975).
  \emph{Clustering  Algorithms}.
  New York: Wiley.

  Sneath, P. H. A. and R. R. Sokal (1973).
  \emph{Numerical Taxonomy}.
  San Francisco: Freeman.

  Anderberg, M. R. (1973).
  \emph{Cluster Analysis for Applications}.
  Academic Press: New York.

  Gordon, A. D. (1999).
  \emph{Classification}. Second Edition.
  London: Chapman and Hall / CRC

  Murtagh, F. (1985).
  \dQuote{Multidimensional Clustering Algorithms}, in
  \emph{COMPSTAT Lectures 4}.
  Wuerzburg: Physica-Verlag
  (for algorithmic details of algorithms used).
}
\author{
  The \code{hclust} function is based on Fortran code
  contributed to STATLIB by F. Murtagh.
}
\seealso{
  \code{\link{identify.hclust}}, \code{\link{rect.hclust}},
  \code{\link{cutree}}, \code{\link{dendrogram}}, \code{\link{kmeans}}.
}
\examples{
data(USArrests)
hc <- hclust(dist(USArrests), "ave")
plot(hc)
plot(hc, hang = -1)

## Do the same with centroid clustering and squared Euclidean distance,
## cut the tree into ten clusters and reconstruct the upper part of the
## tree from the cluster centers.
hc <- hclust(dist(USArrests)^2, "cen")
memb <- cutree(hc, k = 10)
cent <- NULL
for(k in 1:10){
  cent <- rbind(cent, colMeans(USArrests[memb == k, , drop = FALSE]))
}
hc1 <- hclust(dist(cent)^2, method = "cen", members = table(memb))
opar <- par(mfrow = c(1, 2))
plot(hc,  labels = FALSE, hang = -1, main = "Original Tree")
plot(hc1, labels = FALSE, hang = -1, main = "Re-start from 10 clusters")
par(opar)
}
\keyword{multivariate}
\keyword{cluster}
\eof
\name{heatmap}
\alias{heatmap}
\title{ Draw a Heat Map }
\description{
  A heat map is a false color image (basically
  \code{\link{image}(t(x))}) with a dendrogram added to the left side
  and to the top.  Typically, reordering of the rows and columns
  according to some set of values (row or column means) within the
  restrictions imposed by the dendrogram is carried out.
}
\usage{
heatmap(x, Rowv=NULL, Colv=if(symm)"Rowv" else NULL,
	distfun = dist, hclustfun = hclust, add.expr,
	symm = FALSE, revC = identical(Colv, "Rowv"),
	scale=c("row", "column", "none"), na.rm = TRUE,
	margins = c(5, 5), ColSideColors, RowSideColors,
	cexRow = 0.2 + 1/log10(nr), cexCol = 0.2 + 1/log10(nc),
	labRow = NULL, labCol = NULL, main = NULL, xlab = NULL, ylab = NULL,
        keep.dendro = FALSE, verbose = getOption("verbose"), \dots)
}
\arguments{
  \item{x}{numeric matrix of the values to be plotted. }
  \item{Rowv}{determines if and how the \emph{row} dendrogram should be
    computed and reordered.  Either a \code{\link{dendrogram}} or a
    vector of values used to reorder the row dendrogram or
    \code{\link{NA}} to suppress any row dendrogram (and reordering) or
    by default, \code{\link{NULL}}, see \emph{Details} below.}
  \item{Colv}{determines if and how the \emph{column} dendrogram should be
    reordered.	Has the same options as the \code{Rowv} argument above and
    \emph{additionally} when \code{x} is a square matrix, \code{Colv =
      "Rowv"} means that columns should be treated identically to the rows.}
  \item{distfun}{function used to compute the distance (dissimilarity)
    between both rows and columns.  Defaults to \code{\link{dist}}.}
  \item{hclustfun}{function used to compute the hierarchical clustering
    when \code{Rowv} or \code{Colv} are not dendrograms.  Defaults to
    \code{\link{hclust}}.}
  \item{add.expr}{expression that will be evaluated after the call to
    \code{image}.  Can be used to add components to the plot.}
  \item{symm}{logical indicating if \code{x} should be treated
    \bold{symm}etrically; can only be true when \code{x} is a square matrix.}
  \item{revC}{logical indicating if the column order should be
    \code{\link{rev}}ersed for plotting, such that e.g., for the
    symmetric case, the symmetry axis is as usual.}
  \item{scale}{character indicating if the values should be centered and
    scaled in either the row direction or the column direction, or
    none.  The default is \code{"row"} if \code{symm} false, and
    \code{"none"} otherwise.}
  \item{na.rm}{logical indicating whether \code{NA}'s should be removed.}
  \item{margins}{numeric vector of length 2 containing the margins
    (see \code{\link{par}(mar= *)}) for column and row names, respectively.}
  \item{ColSideColors}{(optional) character vector of length \code{ncol(x)}
    containing the color names for a horizontal side bar that may be used to
    annotate the columns of \code{x}.}
  \item{RowSideColors}{(optional) character vector of length \code{nrow(x)}
    containing the color names for a vertical side bar that may be used to
    annotate the rows of \code{x}.}
  \item{cexRow, cexCol}{positive numbers, used as \code{cex.axis} in
    for the row or column axis labeling.  The defaults currently only
    use number of rows or columns, respectively.}
  \item{labRow, labCol}{character vectors with row and column labels to
    use; these default to \code{rownames(x)} or \code{colnames(x)},
    respectively.}
  \item{main, xlab, ylab}{main, x- and y-axis titles; defaults to none.}
  \item{keep.dendro}{logical indicating if the dendrogram(s) should be
    kept as part of the result (when \code{Rowv} and/or \code{Colv} are
    not NA).}
  \item{verbose}{logical indicating if information should be printed.}
  \item{\dots}{additional arguments passed on to \code{\link{image}},
    e.g., \code{col} specifying the colors.} %
}
\details{
  If either \code{Rowv} or \code{Colv} are dendrograms they are honored
  (and not reordered).	Otherwise, dendrograms are computed as
  \code{dd <- as.dendrogram(hclustfun(distfun(X)))} where \code{X} is
  either \code{x} or \code{t(x)}.

  If either is a vector (of \dQuote{weights}) then the appropriate
  dendrogram is reordered according to the supplied values subject to
  the constraints imposed by the dendrogram, by \code{\link{reorder}(dd,
    Rowv)}, in the row case.
  %%
  If either is missing, as by default, then the ordering of the
  corresponding dendrogram is by the mean value of the rows/columns,
  i.e., in the case of rows, \code{Rowv <- rowMeans(x, na.rm=na.rm)}.
  %%
  If either is \code{\link{NULL}}, \emph{no reordering} will be done for
  the corresponding side.

  By default (\code{scale = "row"}) the rows are scaled to have mean
  zero and standard deviation one.  There is some empirical evidence
  from genomic plotting that this is useful.

  The default colors are not pretty.  Consider using enhancements such
  as the \pkg{RColorBrewer} package,
  \url{http://cran.r-project.org/src/contrib/PACKAGES.html#RColorBrewer}.
}
\note{
  Unless \code{Rowv = NA} (or \code{Colw = NA}), the original rows and
  columns are reordered \emph{in any case} to match the dendrogram,
  e.g., the rows by \code{\link{order.dendrogram}(Rowv)} where
  \code{Rowv} is the (possibly \code{\link{reorder}()}ed) row
  dendrogram.

  \code{heatmap()} uses \code{\link{layout}} and draws the
  \code{\link{image}} in the lower right corner of a 2x2 layout.
  Consequentially, it can \bold{not} be used in a multi column/row
  layout, i.e., when \code{\link{par}(mfrow= *)} or \code{(mfcol= *)}
  has been called.
}
\value{
  Invisibly, a list with components
  \item{rowInd}{\bold{r}ow index permutation vector as returned by
    \code{\link{order.dendrogram}}.}
  \item{colInd}{\bold{c}olumn index permutation vector.}
  \item{Rowv}{the row dendrogram; only if input \code{Rowv} was not NA
    and \code{keep.dendro} is true.}
  \item{Colv}{the column dendrogram; only if input \code{Colv} was not NA
    and \code{keep.dendro} is true.}
}
\author{Andy Liaw, original; R. Gentleman, M. Maechler, W. Huber, revisions.}

\seealso{\code{\link{image}}, \code{\link{hclust}}}

\examples{
require(graphics)
data(mtcars)
x  <- as.matrix(mtcars)
rc <- rainbow(nrow(x), start=0, end=.3)
cc <- rainbow(ncol(x), start=0, end=.3)
hv <- heatmap(x, col = cm.colors(256), scale="column",
	      RowSideColors = rc, ColSideColors = cc, margin=c(5,10),
	      xlab = "specification variables", ylab= "Car Models",
	      main = "heatmap(<Mtcars data>, ..., scale = \"column\")")
str(hv) # the two re-ordering index vectors

## no column dendrogram (nor reordering) at all:
heatmap(x, Colv = NA, col = cm.colors(256), scale="column",
	RowSideColors = rc, margin=c(5,10),
	xlab = "specification variables", ylab= "Car Models",
	main = "heatmap(<Mtcars data>, ..., scale = \"column\")")
\testonly{
## no row dendrogram (nor reordering) at all:
heatmap(x, Rowv = NA, col = cm.colors(256), scale="column",
	ColSideColors = cc, margin=c(5,10),
	xlab = "xlab", ylab= "ylab")# no main
}
## "no nothing"
heatmap(x, Rowv = NA, Colv = NA, scale="column",
	main = "heatmap(*, NA, NA) ~= image(t(x))")
%% also want example using the `add.exp' argument!

data(attitude)
round(Ca <- cor(attitude), 2)
symnum(Ca) # simple graphic
heatmap(Ca,		 symm = TRUE, margin=c(6,6))# with reorder()
heatmap(Ca, Rowv=FALSE, symm = TRUE, margin=c(6,6))# _NO_ reorder()

## For variable clustering, rather use distance based on cor():
data(USJudgeRatings)
symnum( cU <- cor(USJudgeRatings) )

hU <- heatmap(cU, Rowv = FALSE, symm = TRUE, col = topo.colors(16),
	     distfun = function(c) as.dist(1 - c), keep.dendro = TRUE)
## The Correlation matrix with same reordering:
round(100 * cU[hU[[1]], hU[[2]]])
## The column dendrogram:
str(hU$Colv)
}
\keyword{hplot}
\eof
\name{identify.hclust}
\alias{identify.hclust}
\title{Identify Clusters in a Dendrogram}
\description{
  \code{identify.hclust} reads the position of the graphics pointer when the
  (first) mouse button is pressed.  It then cuts the tree at the
  vertical position of the pointer and highlights the cluster containing
  the horizontal position of the pointer.  Optionally a function is
  applied to the index of data points contained in the cluster.
}
\usage{
\method{identify}{hclust}(x, FUN = NULL, N = 20, MAXCLUSTER = 20, DEV.FUN = NULL, \dots)
}
\arguments{
  \item{x}{an object of the type produced by \code{hclust}.}
  \item{FUN}{(optional) function to be applied to the index numbers of
    the data points in a cluster (see Details below).}
  \item{N}{the maximum number of clusters to be identified.}
  \item{MAXCLUSTER}{the maximum number of clusters that can be produced
    by a cut (limits the effective vertical range of the pointer). }
  \item{DEV.FUN}{(optional) integer scalar. If specified, the
    corresponding graphics device is made active before \code{FUN} is
    applied.}
  \item{\dots}{further arguments to \code{FUN}.}
}
\details{
  By default clusters can be identified using the mouse and an
  \code{\link{invisible}} list of indices of the respective data points
  is returned.

  If \code{FUN} is not \code{NULL}, then the index vector of data points
  is passed to this function as first argument, see the examples
  below.  The active graphics device for \code{FUN} can be specified using
  \code{DEV.FUN}.

  The identification process is terminated by pressing any mouse
  button other than the first, see also \code{\link{identify}}.
}
\value{
  Either a list of data point index vectors or a list of return values
  of \code{FUN}.
}
\seealso{
  \code{\link{hclust}},
  \code{\link{rect.hclust}}
}
\examples{
%-- in the future -- when example() will
%   show comments, and always have a pager and ... :
% ## Open a default device; if it's an interactive one, continue example:
% get(devNam <- getOption("device"))()
% if (dev.interactive()) {
\dontrun{
  data(USArrests)
  hca <- hclust(dist(USArrests))
  plot(hca)
  (x <- identify(hca)) ##  Terminate with 2nd mouse button !!

  data(iris)
  hci <- hclust(dist(iris[,1:4]))
  plot(hci)
  identify(hci, function(k) print(table(iris[k,5])))

  # open a new device (one for dendrogram, one for bars):
  get(getOption("device"))()# << make that narrow (& small) and *beside* 1st one
  nD <- dev.cur()           # to be for the barplot
  dev.set(dev.prev())# old one for dendrogram
  plot(hci)
  ## select subtrees in dendrogram and "see" the species distribution:
  identify(hci, function(k) barplot(table(iris[k,5]),col=2:4), DEV.FUN = nD)
}
% dev.off()
}
\keyword{cluster}
\keyword{iplot}
\eof
\name{influence.measures}
\title{Regression Deletion Diagnostics}
\alias{influence.measures}
\alias{print.infl}
\alias{summary.infl}
\alias{hat}
\alias{hatvalues}
\alias{hatvalues.lm}
\alias{rstandard}
\alias{rstandard.lm}
\alias{rstandard.glm}
\alias{rstudent}
\alias{rstudent.lm}
\alias{rstudent.glm}
\alias{dfbeta}
\alias{dfbeta.lm}
\alias{dfbetas}
\alias{dfbetas.lm}
\alias{dffits}
\alias{covratio}
\alias{cooks.distance}
\alias{cooks.distance.lm}
\alias{cooks.distance.glm}
\usage{
influence.measures(model)

rstandard(model, \dots)
\method{rstandard}{lm}(model, infl = lm.influence(model, do.coef=FALSE),
          sd = sqrt(deviance(model)/df.residual(model)), \dots)
\method{rstandard}{glm}(model, infl = lm.influence(model, do.coef=FALSE), \dots)

rstudent(model, \dots)
\method{rstudent}{lm}(model, infl = lm.influence(model, do.coef=FALSE),
	 res = infl$wt.res, \dots)
\method{rstudent}{glm}(model, infl = influence(model, do.coef=FALSE), \dots)

dffits(model, infl = , res = )

dfbeta(model, \dots)
\method{dfbeta}{lm}(model, infl = lm.influence(model, do.coef=TRUE), \dots)

dfbetas(model, \dots)
\method{dfbetas}{lm}(model, infl = lm.influence(model, do.coef=TRUE), \dots)

covratio(model, infl = lm.influence(model, do.coef=FALSE),
         res = weighted.residuals(model))

cooks.distance(model, \dots)
\method{cooks.distance}{lm}(model, infl = lm.influence(model, do.coef=FALSE),
               res = weighted.residuals(model),
               sd = sqrt(deviance(model)/df.residual(model)),
               hat = infl$hat, \dots)
\method{cooks.distance}{glm}(model, infl = influence(model, do.coef=FALSE),
               res = infl$pear.res, dispersion = summary(model)$dispersion,
               hat = infl$hat, \dots)

hatvalues(model, \dots)
\method{hatvalues}{lm}(model, infl = lm.influence(model, do.coef=FALSE), \dots)

hat(x, intercept = TRUE)
}
\arguments{
  \item{model}{an \R object, typically returned by \code{\link{lm}} or
    \code{\link{glm}}.}
  \item{infl}{influence structure as returned by
    \code{\link{lm.influence}} or \code{\link{influence}} (the latter
    only for the \code{glm} method of \code{rstudent} and
    \code{cooks.distance}).}
  \item{res}{(possibly weighted) residuals, with proper default.}
  \item{sd}{standard deviation to use, see default.}
  \item{dispersion}{dispersion (for \code{\link{glm}} objects) to use,
    see default.}
  \item{hat}{hat values \eqn{H_{ii}}{H[i,i]}, see default.}

  \item{x}{the \eqn{X} or design matrix.}
  \item{intercept}{should an intercept column be pre-prended to \code{x}?}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  This suite of functions can be used to compute some of the regression
  (leave-one-out deletion) diagnostics for linear and generalized linear
  models discussed in Belsley, Kuh and Welsch (1980), Cook and Weisberg (1982),
  etc.
}
\details{
  The primary high-level function is \code{influence.measures} which produces a
  class \code{"infl"} object tabular display showing the DFBETAS for
  each model variable, DFFITS, covariance ratios, Cook's distances and
  the diagonal elements of the hat matrix.  Cases which are influential
  with respect to any of these measures are marked with an asterisk.

  The functions \code{dfbetas}, \code{dffits},
  \code{covratio} and \code{cooks.distance} provide direct access to the
  corresponding diagnostic quantities.  Functions \code{rstandard} and
  \code{rstudent} give the standardized and Studentized residuals
  respectively. (These re-normalize the residuals to have unit variance,
  using an overall and leave-one-out measure of the error variance
  respectively.)

  Values for generalized linear models are approximations, as described
  in Williams (1987) (except that Cook's distances are scaled as
  \eqn{F} rather than as chi-square values).

  The optional \code{infl}, \code{res} and \code{sd} arguments are there
  to encourage the use of these direct access functions, in situations
  where, e.g., the underlying basic influence measures (from
  \code{\link{lm.influence}} or the generic \code{\link{influence}}) are
  already available.

  Note that cases with \code{weights == 0} are \emph{dropped} from all
  these functions, but that if a linear model has been fitted with
  \code{na.action = na.exclude}, suitable values are filled in for the
  cases excluded during fitting.

  The function \code{hat()} exists mainly for S (version 2)
  compatibility; we recommend using \code{hatvalues()} instead.
}
\note{
  For \code{hatvalues}, \code{dfbeta}, and \code{dfbetas}, the method
  for linear models also works for generalized linear models.
}
\author{
  Several R core team members and John Fox, originally in his \file{car}
  package.
}
\references{
  Belsley, D. A., Kuh, E. and Welsch, R. E. (1980)
  \emph{Regression Diagnostics}.
  New York: Wiley.

  Cook, R. D. and Weisberg, S. (1982)
  \emph{Residuals and Influence in Regression}.
  London: Chapman and Hall.

  Williams, D. A. (1987)
  Generalized linear model diagnostics using the deviance and single
  case deletions. \emph{Applied Statistics} \bold{36}, 181--191.

  Fox, J. (1997)
  \emph{Applied Regression, Linear Models, and Related Methods}. Sage.

  Fox, J. (2002)
  \emph{An R and S-Plus Companion to Applied Regression}.
  Sage Publ.; \url{http://www.socsci.mcmaster.ca/jfox/Books/Companion/}.
}
\seealso{\code{\link{influence}} (containing \code{\link{lm.influence}}).
}
\examples{
## Analysis of the life-cycle savings data
## given in Belsley, Kuh and Welsch.
data(LifeCycleSavings)
lm.SR <- lm(sr ~ pop15 + pop75 + dpi + ddpi, data = LifeCycleSavings)

inflm.SR <- influence.measures(lm.SR)
which(apply(inflm.SR$is.inf, 1, any)) # which observations 'are' influential
summary(inflm.SR) # only these
inflm.SR          # all
plot(rstudent(lm.SR) ~ hatvalues(lm.SR)) # recommended by some

## The 'infl' argument is not needed, but avoids recomputation:
rs <- rstandard(lm.SR)
iflSR <- influence(lm.SR)
identical(rs, rstandard(lm.SR, infl = iflSR))
## to "see" the larger values:
1000 * round(dfbetas(lm.SR, infl = iflSR), 3)

## Huber's data [Atkinson 1985]
xh <- c(-4:0, 10)
yh <- c(2.48, .73, -.04, -1.44, -1.32, 0)
summary(lmH <- lm(yh ~ xh))
(im <- influence.measures(lmH))
plot(xh,yh, main = "Huber's data: L.S. line and influential obs.")
abline(lmH); points(xh[im$is.inf], yh[im$is.inf], pch=20, col=2)
}
\keyword{regression}
\eof
\name{integrate}
\alias{integrate}
\alias{print.integrate}
\title{Integration of One-Dimensional Functions}
\description{
  Adaptive quadrature of functions of one variable over a finite or
  infinite interval.
}
\usage{
integrate(f, lower, upper, subdivisions=100,
          rel.tol = .Machine$double.eps^0.25, abs.tol = rel.tol,
          stop.on.error = TRUE, keep.xy = FALSE, aux = NULL, \dots)
}
\arguments{
  \item{f}{an \R function taking a numeric first argument and returning
    a numeric vector of the same length.  Returning a non-finite element will
    generate an error.}
  \item{lower, upper}{the limits of integration.  Can be infinite.}
  \item{subdivisions}{the maximum number of subintervals.}
  \item{rel.tol}{relative accuracy requested.}
  \item{abs.tol}{absolute accuracy requested.}
  \item{stop.on.error}{logical. If true (the default) an error stops the
    function.  If false some errors will give a result with a warning in
    the \code{message} component.}
  \item{keep.xy}{unused.  For compatibility with S.}
  \item{aux}{unused.  For compatibility with S.}
  \item{\dots}{additional arguments to be passed to \code{f}.  Remember
    to use argument names \emph{not} matching those of \code{integrate(.)}!}
}
\details{
  If one or both limits are infinite, the
  infinite range is mapped onto a finite interval.

  For a finite interval, globally adaptive interval subdivision is used
  in connection with extrapolation by the Epsilon algorithm.

  \code{rel.tol} cannot be less than \code{max(50*.Machine$double.eps,
    0.5e-28)} if \code{abs.tol <= 0}.
}
\note{
  Like all numerical integration routines, these evaluate the function
  on a finite set of points.  If the function is approximately constant
  (in particular, zero) over nearly all its range it is possible that
  the result and error estimate may be seriously wrong.

  When integrating over infinite intervals do so explicitly, rather than
  just using a large number as the endpoint.  This increases the chance
  of a correct answer -- any function whose integral over an infinite
  interval is finite must be near zero for most of that interval.
}
\value{
  A list of class \code{"integrate"} with components
  \item{value}{the final estimate of the integral.}
  \item{abs.error}{estimate of the modulus of the absolute error.}
  \item{subdivisions}{the number of subintervals produced in the
    subdivision process.}
  \item{message}{\code{"OK"} or a character string giving the error message.}
  \item{call}{the matched call.}
}
\references{
  Based on QUADPACK routines \code{dqags} and \code{dqagi} by
  R. Piessens and E. deDoncker-Kapenga, available from Netlib.

  See\cr
  R. Piessens, E. deDoncker-Kapenga, C. Uberhuber, D. Kahaner (1983)
  \emph{Quadpack: a Subroutine Package for Automatic Integration};
  Springer Verlag.
}
\seealso{
  The function \code{\link[adapt]{adapt}} in the \pkg{adapt} package on
  CRAN, for multivariate integration.
}
\examples{
integrate(dnorm, -1.96, 1.96)
integrate(dnorm, -Inf, Inf)

## a slowly-convergent integral
integrand <- function(x) {1/((x+1)*sqrt(x))}
integrate(integrand, lower = 0, upper = Inf)

## don't do this if you really want the integral from 0 to Inf
integrate(integrand, lower = 0, upper = 10)
integrate(integrand, lower = 0, upper = 100000)
integrate(integrand, lower = 0, upper = 1000000, stop.on.error = FALSE)

try(integrate(function(x) 2, 0, 1))  ## no vectorizable function
integrate(function(x) rep(2, length(x)), 0, 1)  ## correct

## integrate can fail if misused
integrate(dnorm,0,2)
integrate(dnorm,0,20)
integrate(dnorm,0,200)
integrate(dnorm,0,2000)
integrate(dnorm,0,20000) ## fails on many systems
integrate(dnorm,0,Inf)   ## works
}
\keyword{math}
\keyword{utilities}
\eof
\name{interaction.plot}
\alias{interaction.plot}
\title{Two-way Interaction Plot}
\description{
  Plots the mean (or other summary) of the response for two-way
  combinations of factors, thereby illustrating possible interactions.
}
\usage{
interaction.plot(x.factor, trace.factor, response, fun = mean,
                 type = c("l", "p"), legend = TRUE,
                 trace.label=deparse(substitute(trace.factor)), fixed=FALSE,
                 xlab = deparse(substitute(x.factor)), ylab = ylabel,
                 ylim = range(cells, na.rm=TRUE),
                 lty = nc:1, col = 1, pch = c(1:9, 0, letters),
                 xpd = NULL, leg.bg = par("bg"), leg.bty = "n",
                 xtick = FALSE, xaxt = par("xaxt"), axes = TRUE, \dots)
}
\arguments{
  \item{x.factor}{a factor whose levels will form the x axis.}
  \item{trace.factor}{another factor whose levels will form the traces.}
  \item{response}{a numeric variable giving the response}
  \item{fun}{the function to compute the summary. Should return a single
    real value.}
  \item{type}{the type of plot: lines or points.}
  \item{legend}{logical. Should a legend be included?}
  \item{trace.label}{overall label for the legend.}
  \item{fixed}{logical.  Should the legend be in the order of the levels
    of \code{trace.factor} or in the order of the traces at their right-hand ends?}
  \item{xlab,ylab}{the x and y label of the plot each with a sensible default.}
  \item{ylim}{numeric of length 2 giving the y limits for the plot.}
  \item{lty}{line type for the lines drawn, with sensible default.}
  \item{col}{the color to be used for plotting.}
  \item{pch}{a vector of plotting symbols or characters, with sensible
    default.}
  \item{xpd}{determines clipping behaviour for the \code{\link{legend}}
    used, see \code{\link{par}(xpd)}.  Per default, the legend is
    \emph{not} clipped at the figure border.}
  \item{leg.bg, leg.bty}{arguments passed to \code{\link{legend}()}.}
  \item{xtick}{logical. Should tick marks be used on the x axis?}
  \item{xaxt, axes, \dots}{graphics parameters to be passed to the plotting routines.}
}
\details{
  By default the levels of \code{x.factor} are plotted on the x axis in
  their given order, with extra space left at the right for the legend
  (if specified). If \code{x.factor} is an ordered factor and the levels
  are numeric, these numeric values are used for the x axis.

  The response and hence its summary can contain missing values. If so,
  the missing values and the line segments joining them are omitted from
  the plot (and this can be somewhat disconcerting).

  The graphics parameters \code{xlab}, \code{ylab}, \code{ylim},
  \code{lty}, \code{col} and \code{pch} are given suitable defaults
  (and \code{xlim} and \code{xaxs} are set and cannot be overriden).
  The defaults are to cycle through the line types, use the foreground
  colour, and to use the symbols 1:9, 0, and the capital letters to plot
  the traces.
}
\note{
  Some of the argument names and the precise behaviour are chosen for
  S-compatibility.
}
\references{
  Chambers, J. M., Freeny, A and Heiberger, R. M. (1992)
  \emph{Analysis of variance; designed experiments.}
  Chapter 5 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}

\examples{
data(ToothGrowth)
attach(ToothGrowth)
interaction.plot(dose, supp, len, fixed=TRUE)
dose <- ordered(dose)
interaction.plot(dose, supp, len, fixed=TRUE, col = 2:3, leg.bty = "o")
detach()

data(OrchardSprays)
with(OrchardSprays, {
  interaction.plot(treatment, rowpos, decrease)
  interaction.plot(rowpos, treatment, decrease, cex.axis=0.8)
  ## order the rows by their mean effect
  rowpos <- factor(rowpos, levels=sort.list(tapply(decrease, rowpos, mean)))
  interaction.plot(rowpos, treatment, decrease, col = 2:9, lty = 1)
})

data(esoph)
with(esoph, {
  interaction.plot(agegp, alcgp, ncases/ncontrols)
  interaction.plot(agegp, tobgp, ncases/ncontrols, trace.label="tobacco",
		   fixed=TRUE, xaxt = "n")
})
}
\keyword{hplot}
\eof
\name{is.empty.model}
\alias{is.empty.model}
\title{Check if a Model is Empty}
\description{
\R model notation allows models with no intercept and no predictors. These require special handling internally.  \code{is.empty.model()} checks whether an object describes an empty model.
}
\usage{
is.empty.model(x)
}
\arguments{
 \item{x}{A \code{terms} object or an object with a \code{terms} method.}
}

\value{
  \code{TRUE} if the model is empty
}

\seealso{ \code{\link{lm}},\code{\link{glm}}}

\examples{
y <- rnorm(20)
is.empty.model(y ~ 0)
is.empty.model(y ~ -1)
is.empty.model(lm(y ~ 0))
}
\keyword{models}
\eof
\name{isoreg}
\title{Isotonic / Monotone Regression}
\alias{isoreg}
%\alias{fitted.isoreg}
%\alias{residuals.isoreg}
\description{
  Compute the isotonic (monotonely increasing nonparametric) least
  squares regression which is piecewise constant.
}
\usage{
isoreg(x, y = NULL)
}
\arguments{
  \item{x, y}{in \code{isoreg}, coordinate vectors of the regression
    points.  Alternatively a single \dQuote{plotting} structure can be
    specified: see \code{\link{xy.coords}}.
  }
  \item{\dots}{potentially further arguments passed to methods.}
}
\details{
  The algorithm determines the convex minorant \eqn{m(x)} of the
  \emph{cumulative} data (i.e., \code{cumsum(y)}) which is piecewise
  linear and the result is \eqn{m'(x)}, a step function with level
  changes at locations where the convex \eqn{m(x)} touches the
  cumulative data polygon and changes slope.\cr
  \code{\link[stats]{as.stepfun}()} returns a \code{\link[stats]{stepfun}}
  object which can be more parsimonious.
}
\value{
  \code{isoreg()} returns an object of class \code{isoreg} which is
  basically a list with components
  \item{x}{original (constructed) abscissa values \code{x}.}
  \item{y}{corresponding y values.}
  \item{yf}{fitted values corresponding to \emph{ordered} x values.}
  \item{yc}{cumulative y values corresponding to \emph{ordered} x values.}
  \item{iKnots}{integer vector giving indices where the fitted curve jumps,
    i.e., where the convex minorant has kinks.}
  \item{isOrd}{logical indicating if original x values were ordered
    increasingly already.}
  \item{ord}{\code{if(!isOrd)}: integer permutation \code{\link{order}(x)} of
    \emph{original} \code{x}.}
  \item{call}{the \code{\link{call}} to \code{isoreg()} used.}
}
\note{
  The code should be improved to accept \emph{weights} additionally and
  solve the corresponding weighted least squares problem.\cr
  \dQuote{Patches are welcome!}
}
\references{
  Barlow, R. E., Bartholomew, D. J., Bremner, J. M., and Brunk, H. D. (1972)
  \emph{Statistical inference under order restrictions}; Wiley, London.

  Robertson, T., Wright,F. T. and Dykstra, R. L. (1988)
  \emph{Order Restricted Statistical Inference}; Wiley, New York.
}
%%\author{Original C code by Brian Ripley; all else: Martin Maechler}
\seealso{the plotting method \code{\link{plot.isoreg}} with more examples;
  \code{\link[MASS]{isoMDS}()} from the \pkg{MASS} package internally
  uses isotonic regression.
}
\examples{
(ir <- isoreg(c(1,0,4,3,3,5,4,2,0)))
plot(ir, plot.type = "row")

(ir3 <- isoreg(y3 <- c(1,0,4,3,3,5,4,2, 3)))# last "3", not "0"
(fi3 <- as.stepfun(ir3))
(ir4 <- isoreg(1:10, y4 <- c(5, 9, 1:2, 5:8, 3, 8)))
cat("R^2 =", formatC(sum(residuals(ir4)^2) / (9*var(y4)), dig=2),"\n")
}
\keyword{regression}
\keyword{smooth}
\eof
\name{kernapply}
\title{Apply Smoothing Kernel}
\alias{kernapply}
\alias{kernapply.default}
\alias{kernapply.ts}
\alias{kernapply.tskernel}
\alias{kernapply.vector}
\synopsis{
kernapply(x, ...)
kernapply.default(x, k, circular = FALSE, ...)
kernapply.ts(x, k, circular = FALSE, ...)
kernapply.vector(x, k, circular = FALSE, ...)
kernapply.tskernel(x, k2, ...)
}
\usage{
kernapply(x, k, circular = FALSE, \dots) 
kernapply(k1, k2)
}
\arguments{
  \item{k, k1, k2}{smoothing \code{"tskernel"} objects.}
  \item{x}{an input vector, matrix, or time series to be smoothed.}
  \item{circular}{a logical indicating whether the input sequence to be
    smoothed is treated as circular, i.e., periodic.}
  \item{\dots}{arguments passed to or from other methods.}
}
\description{
  \code{kernapply} computes the convolution between an input sequence 
  and a specific kernel.
}
\value{
  A smoothed version of the input sequence.
}
\author{A. Trapletti}
\seealso{
  \code{\link{kernel}}, \code{\link{convolve}}, \code{\link{filter}},
  \code{\link{spectrum}}
}
\examples{
## see 'kernel' for examples
}
\keyword{ts}
\eof
\name{kernel}
\alias{kernel}
\alias{bandwidth.kernel}
\alias{df.kernel}
\alias{is.tskernel}
%\alias{print.tskernel}
%\alias{plot.tskernel}
%\alias{[.tskernel}
\title{Smoothing Kernel Objects}
\description{
  The \code{"tskernel"} class is designed to represent discrete
  symmetric normalized smoothing kernels.  These kernels can be used to
  smooth vectors, matrices, or time series objects.
}
\usage{
kernel(coef, m, r, name)

df.kernel(k)
bandwidth.kernel(k)
is.tskernel(k)
%
%\method{print}{tskernel}(x, digits = max(3,getOption("digits")-3), ...)
%\method{plot}{tskernel}(x, ...)
}
\arguments{
  \item{coef}{the upper half of the smoothing kernel coefficients
    (inclusive of coefficient zero) \emph{or} the name of a kernel
    (currently \code{"daniell"}, \code{"dirichlet"}, \code{"fejer"} or
    \code{"modified.daniell"}.}
  \item{m}{the kernel dimension. The number of kernel coefficients is
    \code{2*m+1}.}
  \item{name}{the name of the kernel.}
  \item{r}{the kernel order for a Fejer kernel.}
%  \item{digits}{the number of digits to format real numbers.}
  \item{k}{a \code{"tskernel"} object.}
%  \item{\dots}{arguments passed to or from other methods.}
}
\details{
  \code{kernel} is used to construct a general kernel or named specific
  kernels.  The modified Daniell kernel halves the end coefficients (as
  used by S-PLUS). 
  
  \code{df.kernel} returns the \dQuote{equivalent degrees of freedom} of
  a smoothing kernel as defined in Brockwell and Davies (1991), page
  362, and \code{bandwidth.kernel} returns the equivalent bandwidth as
  defined in Bloomfield (1991), p. 201, with a continuity correction.
}
\value{
  \code{kernel} returns a list with class \code{"tskernel"}, and
  components the coefficients \code{coef} and the kernel dimension
  \code{m}.  An additional attribute is \code{"name"}.
}
\author{A. Trapletti; modifications by B.D. Ripley}
\seealso{
    \code{\link{kernapply}}
}
\references{
  Bloomfield, P. (1976) \emph{Fourier Analysis of Time Series: An
    Introduction.} Wiley.

  Brockwell, P.J. and Davis, R.A. (1991) \emph{Time Series: Theory and
    Methods.} Second edition. Springer, pp. 350--365.
}
\examples{
data(EuStockMarkets)    # Demonstrate a simple trading strategy for the 
x <- EuStockMarkets[,1]  # financial time series German stock index DAX.
k1 <- kernel("daniell", 50)  # a long moving average
k2 <- kernel("daniell", 10)  # and a short one
plot(k1) 
plot(k2)
x1 <- kernapply(x, k1)
x2 <- kernapply(x, k2)
plot(x)
lines(x1, col = "red")    # go long if the short crosses the long upwards
lines(x2, col = "green")  # and go short otherwise

data(sunspot)     # Reproduce example 10.4.3 from Brockwell and Davies (1991)
spectrum(sunspot.year, kernel=kernel("daniell", c(11,7,3)), log="no")
}
\keyword{ts}

\eof
\name{kmeans}
\alias{kmeans}
\title{
K-Means Clustering
}
\description{
  Perform k-means clustering on a data matrix.
}
\usage{
kmeans(x, centers, iter.max = 10)
}
\arguments{
\item{x}{
  A numeric matrix of data, or an object that can be coerced to such a
  matrix (such as a numeric vector or a data frame with all numeric columns).
}
\item{centers}{
  Either the number of clusters or a set of initial cluster centers.
  If the first, a random set of rows in \code{x} are chosen as the initial
  centers.
}
\item{iter.max}{
  The maximum number of iterations allowed.
}}
\details{
  The data given by \code{x} is clustered by the k-means algorithm.
  When this terminates, all cluster centres are at the mean of
  their Voronoi sets (the set of data points which are nearest to
  the cluster centre).

  The algorithm of Hartigan and Wong (1979) is used.
}
\value{
  A list with components:

  \item{cluster}{
    A vector of integers indicating the cluster to which each point is
    allocated.
  }
  \item{centers}{A matrix of cluster centres.}
  \item{withinss}{The within-cluster sum of squares for each cluster.}
  \item{size}{The number of points in each cluster.}
}
\references{
  Hartigan, J.A. and Wong, M.A. (1979).
  A K-means clustering algorithm.
  \emph{Applied Statistics} \bold{28}, 100--108.
}
\examples{
# a 2-dimensional example
x <- rbind(matrix(rnorm(100, sd = 0.3), ncol = 2),
           matrix(rnorm(100, mean = 1, sd = 0.3), ncol = 2))
cl <- kmeans(x, 2, 20)
plot(x, col = cl$cluster)
points(cl$centers, col = 1:2, pch = 8)
}
\keyword{multivariate}
\keyword{cluster}
\eof
\name{kruskal.test}
\alias{kruskal.test}
\alias{kruskal.test.default}
\alias{kruskal.test.formula}
\title{Kruskal-Wallis Rank Sum Test}
\description{
  Performs a Kruskal-Wallis rank sum test.
}
\usage{
kruskal.test(x, \dots)

\method{kruskal.test}{default}(x, g, \dots)

\method{kruskal.test}{formula}(formula, data, subset, na.action, \dots)
}
\arguments{
  \item{x}{a numeric vector of data values, or a list of numeric data
    vectors.}
  \item{g}{a vector or factor object giving the group for the
    corresponding elements of \code{x}.  Ignored if \code{x} is a
    list.}
  \item{formula}{a formula of the form \code{lhs ~ rhs} where \code{lhs}
    gives the data values and \code{rhs} the corresponding groups.}
  \item{data}{an optional data frame containing the variables in the
    model formula.}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.  Defaults to
    \code{getOption("na.action")}.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\details{
  \code{kruskal.test} performs a Kruskal-Wallis rank sum test of the
  null that the location parameters of the distribution of \code{x}
  are the same in each group (sample).  The alternative is that they
  differ in at least one.
  
  If \code{x} is a list, its elements are taken as the samples to be
  compared, and hence have to be numeric data vectors.  In this case,
  \code{g} is ignored, and one can simply use \code{kruskal.test(x)}
  to perform the test.  If the samples are not yet contained in a
  list, use \code{kruskal.test(list(x, ...))}.
  
  Otherwise, \code{x} must be a numeric data vector, and \code{g} must
  be a vector or factor object of the same length as \code{x} giving
  the group for the corresponding elements of \code{x}.
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the Kruskal-Wallis rank sum statistic.}
  \item{parameter}{the degrees of freedom of the approximate
    chi-squared distribution of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{method}{the character string \code{"Kruskal-Wallis rank sum test"}.}
  \item{data.name}{a character string giving the names of the data.}
}
\references{
  Myles Hollander & Douglas A. Wolfe (1973),
  \emph{Nonparametric statistical inference}.
  New York: John Wiley & Sons.
  Pages 115--120.
}
\seealso{
  The Wilcoxon rank sum test (\code{\link{wilcox.test}}) as the special
  case for two samples;
  \code{\link{lm}} together with \code{\link{anova}} for performing
  one-way location analysis under normality assumptions; with Student's
  t test (\code{\link{t.test}}) as the special case for two samples.
}
\examples{
## Hollander & Wolfe (1973), 116.
## Mucociliary efficiency from the rate of removal of dust in normal
##  subjects, subjects with obstructive airway disease, and subjects
##  with asbestosis.
x <- c(2.9, 3.0, 2.5, 2.6, 3.2)	# normal subjects
y <- c(3.8, 2.7, 4.0, 2.4)      # with obstructive airway disease
z <- c(2.8, 3.4, 3.7, 2.2, 2.0) # with asbestosis
kruskal.test(list(x, y, z))
## Equivalently,
x <- c(x, y, z)
g <- factor(rep(1:3, c(5, 4, 5)),
            labels = c("Normal subjects",
                       "Subjects with obstructive airway disease",
                       "Subjects with asbestosis"))
kruskal.test(x, g)

## Formula interface.
data(airquality)
boxplot(Ozone ~ Month, data = airquality)
kruskal.test(Ozone ~ Month, data = airquality)
}
\keyword{htest}
\eof
\name{ks.test}
\alias{ks.test}
\title{Kolmogorov-Smirnov Tests}
\description{
  Performs one or two sample Kolmogorov-Smirnov tests.
}
\usage{
ks.test(x, y, \dots, alternative = c("two.sided", "less", "greater"),
        exact = NULL)
}
\arguments{
  \item{x}{a numeric vector of data values.}
  \item{y}{either a numeric vector of data values, or a character string
    naming a distribution function.}
  \item{\dots}{parameters of the distribution specified (as a character
    string) by \code{y}.}
  \item{alternative}{indicates the alternative hypothesis and must be
    one of \code{"two.sided"} (default), \code{"less"}, or
    \code{"greater"}.  You can specify just the initial letter.}
  \item{exact}{\code{NULL} or a logical indicating whether an exact
    p-value should be computed.  See Details for the meaning of \code{NULL}.
    Only used in the two-sided two-sample case.}
  
}
\details{
  If \code{y} is numeric, a two-sample test of the null hypothesis
  that \code{x} and \code{y} were drawn from the same \emph{continuous}
  distribution is performed.

  Alternatively, \code{y} can be a character string naming a continuous
  distribution function.  In this case, a one-sample test is carried
  out of the null that the distribution function which generated
  \code{x} is distribution \code{y} with parameters specified by \code{\dots}.

  The presence of ties generates a warning, since continuous distributions
  do not generate them.

  The possible values \code{"two.sided"}, \code{"less"} and
  \code{"greater"} of \code{alternative} specify the null hypothesis
  that the true distribution function of \code{x} is equal to, not less
  than or not greater than the hypothesized distribution function
  (one-sample case) or the distribution function of \code{y} (two-sample
  case), respectively.

  Exact p-values are only available for the two-sided two-sample test
  with no ties.  In that case, if \code{exact = NULL} (the default) an
  exact p-value is computed if the product of the sample sizes is less
  than 10000.  Otherwise, asymptotic distributions are used whose
  approximations may be inaccurate in small samples.

  If a single-sample test is used, the parameters specified in
  \code{\dots} must be pre-specified and not estimated from the data.
  There is some more refined distribution theory for the KS test with
  estimated parameters (see Durbin, 1973), but that is not implemented
  in \code{ks.test}.
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{alternative}{a character string describing the alternative
    hypothesis.}
  \item{method}{a character string indicating what type of test was
    performed.}
  \item{data.name}{a character string giving the name(s) of the data.}
}
\references{
  William J. Conover (1971),
  \emph{Practical nonparametric statistics}.
  New York: John Wiley & Sons.
  Pages 295--301 (one-sample \dQuote{Kolmogorov} test),
  309--314 (two-sample \dQuote{Smirnov} test).

  Durbin, J. (1973)
  \emph{Distribution theory for tests based on the sample distribution
    function}.  SIAM.
}
\seealso{
  \code{\link{shapiro.test}} which performs the Shapiro-Wilk test for
  normality.
}
\examples{
x <- rnorm(50)
y <- runif(30)
# Do x and y come from the same distribution?
ks.test(x, y)
# Does x come from a shifted gamma distribution with shape 3 and scale 2?
ks.test(x+2, "pgamma", 3, 2) # two-sided
ks.test(x+2, "pgamma", 3, 2, alternative = "gr")
}
\keyword{htest}
\eof
\name{ksmooth}
\alias{ksmooth}
\title{Kernel Regression Smoother}
\description{
  The Nadaraya-Watson kernel regression estimate.
}
\usage{
ksmooth(x, y, kernel = c("box", "normal"), bandwidth = 0.5,
        range.x = range(x), n.points = max(100, length(x)), x.points)
}
\arguments{
 \item{x}{input x values}
 \item{y}{input y values}
 \item{kernel}{the kernel to be used.}
 \item{bandwidth}{the bandwidth. The kernels are scaled so that their
   quartiles (viewed as probability densities) are at
   \eqn{\pm}{+/-} \code{0.25*bandwidth}.}
 \item{range.x}{the range of points to be covered in the output.}
 \item{n.points}{the number of points at which to evaluate the fit.}
 \item{x.points}{points at which to evaluate the smoothed fit. If
   missing, \code{n.points} are chosen uniformly to cover \code{range.x}.}
}
\value{
  A list with components
  \item{x}{values at which the smoothed fit is evaluated. Guaranteed to
    be in increasing order.}
  \item{y}{fitted values corresponding to \code{x}.}
}
\note{This function is implemented purely for compatibility with S,
  although it is nowhere near as slow as the S function. Better kernel
  smoothers are available in other packages.}
\examples{
data(cars)
with(cars, {
    plot(speed, dist)
    lines(ksmooth(speed, dist, "normal", bandwidth=2), col=2)
    lines(ksmooth(speed, dist, "normal", bandwidth=5), col=3)
})
}
\keyword{smooth}
\eof
\name{lag}
\alias{lag}
\alias{lag.default}
\title{Lag a Time Series}
\description{
  Compute a lagged version of a time series, shifting the time base
  back by a given number of observations.
}
\usage{
lag(x, \dots)

\method{lag}{default}(x, k = 1, \dots)
}
\arguments{
  \item{x}{A vector or matrix or univariate or multivariate time series}
  \item{k}{The number of lags (in units of observations).}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\details{
  Vector or matrix arguments \code{x} are coerced to time series.

  \code{lag} is a generic function; this page documents its default
  method.
}
\value{
  A time series object.
}

\note{
  Note the sign of \code{k}: a series lagged by a positive \code{k}
  starts \emph{earlier}.
}

\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{\code{\link{diff}}, \code{\link{deltat}} }

\examples{
data(UKLungDeaths)
lag(ldeaths, 12) # starts one year earlier
}
\keyword{ts}
\eof
\name{lag.plot}
\alias{lag.plot}
\title{Time Series Lag Plots}
\description{
  Plot time series against lagged versions of themselves.
  Helps visualizing \dQuote{auto-dependence} even when auto-correlations
  vanish. 
}
\usage{
lag.plot(x, lags = 1, layout = NULL, set.lags = 1:lags,
         main = NULL, asp = 1,
         font.main=par("font.main"), cex.main=par("cex.main"),
         diag = TRUE, diag.col="gray", type="p", oma =NULL, ask =NULL,
         do.lines = n <= 150, labels = do.lines, \dots)
}
\arguments{
  \item{x}{time-series (univariate or multivariate)}
  \item{lags}{number of lag plots desired, see arg \code{set.lags}.}
  \item{layout}{the layout of multiple plots, basically the \code{mfrow}
    \code{\link{par}()} argument.  The default uses about a square
    layout (see \code{\link{n2mfrow}} such that all plots are on one page.}
  \item{set.lags}{positive integer vector allowing to specify the set of
  lags used; defaults to \code{1:lags}.}
  \item{main}{character with a main header title to be done on the top
    of each page.}
  \item{asp}{Aspect ratio to be fixed, see \code{\link{plot.default}}.}
  \item{font.main, cex.main}{attributes for the title, see \code{par()}.}
  \item{diag}{logical indicating if the x=y diagonal should be drawn.}
  \item{diag.col}{color to be used for the diagonal \code{if(diag)}.}
  \item{type}{plot type to be used, but see \code{\link{plot.ts}} about
    its restricted meaning.}
  \item{oma}{outer margins, see \code{\link{par}}.}
  \item{ask}{logical; if true, the user is asked before a new page is started.}
  \item{do.lines}{logical indicating if lines should be drawn.}
  \item{labels}{logical indicating if labels should be used.}
  \item{\dots}{Further arguments to \code{\link{plot.ts}}.}
}
\note{
  It is more flexible and has different default behaviour than
  the S version.  We use \code{main =} instead of \code{head = } for
  internal consistency.
}
\author{Martin Maechler}
\seealso{
  \code{\link{plot.ts}} which is the basic work horse.
}
\examples{
data(nhtemp)
lag.plot(nhtemp, 8, diag.col = "forest green")
lag.plot(nhtemp, 5, main="Average Temperatures in New Haven")
## ask defaults to TRUE when we have more than one page:
lag.plot(nhtemp, 6, layout = c(2,1), asp = NA,
         main = "New Haven Temperatures", col.main = "blue")

## Multivariate (but non-stationary! ...)
data(freeny)
lag.plot(freeny.x, lag = 3)
\dontrun{
data(sunspots) # no lines for long series :
lag.plot(sqrt(sunspots), set = c(1:4, 9:12), pch = ".", col = "gold")
}
}
\keyword{hplot}
\keyword{ts}
\eof
% file lh.Rd
% copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{lh}
\docType{data}
\alias{lh}
\title{
  Luteinizing Hormone in Blood Samples
}
\description{
  A regular time series giving the luteinizing hormone in blood
  samples at 10 mins intervals from a human female, 48 samples.
}
\usage{data(lh)}
\source{
  P.J. Diggle (1990)
  \emph{Time Series: A Biostatistical Introduction.}
  Oxford, table A.1, series 3
}
\keyword{datasets}
\eof
\name{line}
\alias{line}
%\alias{coef.tukeyline}
%\alias{coefficients.tukeyline}
\alias{residuals.tukeyline}
%\alias{fitted.tukeyline}
%\alias{fitted.values.tukeyline}
%\alias{print.tukeyline}
\title{Robust Line Fitting}
\description{
  Fit a line robustly as recommended in \emph{Exploratory Data Analysis}.
}
\usage{
line(x, y)
}
\arguments{
  \item{x,y}{the arguments can be any way of specifying x-y pairs.}
}
\value{
  An object of class \code{"tukeyline"}.

  Methods are available for the generic functions \code{coef},
  \code{residuals}, \code{fitted}, and \code{print}.
}
\references{
  Tukey, J. W. (1977).
  \emph{Exploratory Data Analysis},
  Reading Massachusetts: Addison-Wesley.
}
\seealso{
  \code{\link{lm}}.
}
\examples{
data(cars)
plot(cars)
(z <- line(cars))
abline(coef(z))
## Tukey-Anscombe Plot :
plot(residuals(z) ~ fitted(z), main = deparse(z$call))
}
\keyword{robust}
\keyword{regression}
\eof
\name{lm}
\alias{lm}
\alias{print.lm}
\concept{regression}
\title{Fitting Linear Models}
\description{
  \code{lm} is used to fit linear models.
  It can be used to carry out regression,
  single stratum analysis of variance and
  analysis of covariance (although \code{\link{aov}} may provide a more
  convenient interface for these).
}
\usage{
lm(formula, data, subset, weights, na.action,
   method = "qr", model = TRUE, x = FALSE, y = FALSE, qr = TRUE,
   singular.ok = TRUE, contrasts = NULL, offset, \dots)
}
\arguments{
  \item{formula}{a symbolic description of the model to be fit.
    The details of model specification are given below.}

  \item{data}{an optional data frame containing the variables
    in the model.  If not found in \code{data}, the variables are taken
    from \code{environment(formula)}, typically the environment from which
    \code{lm} is called.}

  \item{subset}{an optional vector specifying a subset of observations
    to be used in the fitting process.}

  \item{weights}{an optional vector of weights to be used
    in the fitting process. If specified, weighted least squares is used
    with weights \code{weights} (that is, minimizing \code{sum(w*e^2)});
    otherwise ordinary least squares is used.}

  \item{na.action}{a function which indicates what should happen
    when the data contain \code{NA}s.  The default is set by
    the \code{na.action} setting of \code{\link{options}}, and is
    \code{\link{na.fail}} if that is unset. The \dQuote{factory-fresh}
    default is \code{\link{na.omit}}.}

  \item{method}{the method to be used; for fitting, currently only
    \code{method="qr"} is supported; \code{method="model.frame"} returns
    the model frame (the same as with \code{model = TRUE}, see below).}

  \item{model, x, y, qr}{logicals.  If \code{TRUE} the corresponding
    components of the fit (the model frame, the model matrix, the
    response, the QR decomposition) are returned.}

  \item{singular.ok}{logical. If \code{FALSE} (the default in S but
    not in \R) a singular fit is an error.}

  \item{contrasts}{an optional list. See the \code{contrasts.arg}
    of \code{model.matrix.default}.}

  \item{offset}{this can be used to specify an \emph{a priori}
    known component to be included in the linear predictor
    during fitting.  An \code{\link{offset}} term can be included in the
    formula instead or as well, and if both are specified their sum is used.}

  \item{\dots}{additional arguments to be passed to the low level
    regression fitting functions (see below).}
}
\details{
  Models for \code{lm} are specified symbolically.  A typical model has
  the form \code{response ~ terms} where \code{response} is the (numeric)
  response vector and \code{terms} is a series of terms which specifies a
  linear predictor for \code{response}.  A terms specification of the form
  \code{first + second} indicates all the terms in \code{first} together
  with all the terms in \code{second} with duplicates removed.  A
  specification of the form \code{first:second} indicates the set of
  terms obtained by taking the interactions of all terms in \code{first}
  with all terms in \code{second}.  The specification \code{first*second}
  indicates the \emph{cross} of \code{first} and \code{second}.  This is
  the same as \code{first + second + first:second}.  If \code{response} 
  is a matrix a linear model is fitted to each column of the matrix.
  See \code{\link{model.matrix}} for some further details.  The terms in
  the formula will be re-ordered so that main effects come first,
  followed by the interactions, all second-order, all third-order and so
  on: to avoid this pass a \code{terms} object as the formula.

  A formula has an implied intercept term.  To remove this use either
  \code{y ~ x - 1} or \code{y ~ 0 + x}.  See \code{\link{formula}} for
  more details of allowed formulae.
  
  \code{lm} calls the lower level functions \code{\link{lm.fit}}, etc,
  see below, for the actual numerical computations.  For programming
  only, you may consider doing likewise.

  All of \code{weights}, \code{subset} and \code{offset} are evaluated
  in the same way as variables in \code{formula}, that is first in
  \code{data} and then in the environment of \code{formula}.
}
\value{
  \code{lm} returns an object of \code{\link[base]{class}} \code{"lm"} or for
  multiple responses of class \code{c("mlm", "lm")}.

  The functions \code{summary} and \code{\link{anova}} are used to
  obtain and print a summary and analysis of variance table of the
  results.  The generic accessor functions \code{coefficients},
  \code{effects}, \code{fitted.values} and \code{residuals} extract
  various useful features of the value returned by \code{lm}.

  An object of class \code{"lm"} is a list containing at least the
  following components:

  \item{coefficients}{a named vector of coefficients}
  \item{residuals}{the residuals, that is response minus fitted values.}
  \item{fitted.values}{the fitted mean values.}
  \item{rank}{the numeric rank of the fitted linear model.}
  \item{weights}{(only for weighted fits) the specified weights.}
  \item{df.residual}{the residual degrees of freedom.}
  \item{call}{the matched call.}
  \item{terms}{the \code{\link{terms}} object used.}
  \item{contrasts}{(only where relevant) the contrasts used.}
  \item{xlevels}{(only where relevant) a record of the levels of the
    factors used in fitting.}
  \item{y}{if requested, the response used.}
  \item{x}{if requested, the model matrix used.}
  \item{model}{if requested (the default), the model frame used.}

  In addition, non-null fits will have components \code{assign},
  \code{effects} and (unless not requested) \code{qr} relating to the linear
  fit, for use by extractor functions such as \code{summary} and
  \code{\link{effects}}.
}
\seealso{
  \code{\link{summary.lm}} for summaries and \code{\link{anova.lm}} for
  the ANOVA table; \code{\link{aov}} for a different interface.

  The generic functions \code{\link{coef}}, \code{\link{effects}},
  \code{\link{residuals}}, \code{\link{fitted}}, \code{\link{vcov}}.

  \code{\link{predict.lm}} (via \code{\link{predict}}) for prediction,
  including confidence and prediction intervals.

  \code{\link{lm.influence}} for regression diagnostics, and
  \code{\link{glm}} for \bold{generalized} linear models.

  The underlying low level functions,
  \code{\link{lm.fit}} for plain, and \code{\link{lm.wfit}} for weighted
  regression fitting.
}
\references{
  Chambers, J. M. (1992)
  \emph{Linear models.}
  Chapter 4 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.

  Wilkinson, G. N. and Rogers, C. E. (1973)
  Symbolic descriptions of factorial models for analysis of variance.
  \emph{Applied Statistics}, \bold{22}, 392--9.
}
\author{
  The design was inspired by the S function of the same name described
  in Chambers (1992).  The implementation of model formula by Ross Ihaka
  was based on Wilkinson \& Rogers (1973).
}

\note{
  Offsets specified by \code{offset} will not be included in predictions
  by \code{\link{predict.lm}}, whereas those specified by an offset term
  in the formula will be.
}
\examples{
## Annette Dobson (1990) "An Introduction to Generalized Linear Models".
## Page 9: Plant Weight Data.
ctl <- c(4.17,5.58,5.18,6.11,4.50,4.61,5.17,4.53,5.33,5.14)
trt <- c(4.81,4.17,4.41,3.59,5.87,3.83,6.03,4.89,4.32,4.69)
group <- gl(2,10,20, labels=c("Ctl","Trt"))
weight <- c(ctl, trt)
anova(lm.D9 <- lm(weight ~ group))
summary(lm.D90 <- lm(weight ~ group - 1))# omitting intercept
summary(resid(lm.D9) - resid(lm.D90)) #- residuals almost identical

opar <- par(mfrow = c(2,2), oma = c(0, 0, 1.1, 0))
plot(lm.D9, las = 1)      # Residuals, Fitted, ...
par(opar)

## model frame :
stopifnot(identical(lm(weight ~ group, method = "model.frame"),
                    model.frame(lm.D9)))
}
\keyword{regression}
\eof
\name{lm.influence}
\title{Regression Diagnostics}
\usage{
influence(model, \dots)
\method{influence}{lm}(model, do.coef = TRUE, \dots)
\method{influence}{glm}(model, do.coef = TRUE, \dots)

lm.influence(model, do.coef = TRUE)
}
\alias{lm.influence}
\alias{influence}
\alias{influence.lm}
\alias{influence.glm}
\arguments{
  \item{model}{an object as returned by \code{\link{lm}}.}
  \item{do.coef}{logical indicating if the changed \code{coefficients}
    (see below) are desired.  These need \eqn{O(n^2 p)} computing time.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  This function provides the basic quantities which are
  used in forming a wide variety of diagnostics for
  checking the quality of regression fits.
}
\details{
  The \code{\link{influence.measures}()} and other functions listed in
  \bold{See Also} provide a more user oriented way of computing a
  variety of regression diagnostics.  These all build on
  \code{lm.influence}.

  An attempt is made to ensure that computed hat values that are
  probably one are treated as one, and the corresponding rows in
  \code{sigma} and \code{coefficients} are \code{NaN}.  (Dropping such a
  case would normally result in a variable being dropped, so it is not
  possible to give simple drop-one diagnostics.)
}
\value{
  A list containing the following components of the same length or
  number of rows \eqn{n}, which is the number of non-zero weights.
  Cases omitted in the fit are omitted unless a \code{\link{na.action}}
  method was used (such as \code{\link{na.exclude}}) which restores them.
  
  \item{hat}{a vector containing the diagonal of the \dQuote{hat} matrix.}
  \item{coefficients}{(unless \code{do.coef} is false) a matrix whose
    i-th row contains the change in the estimated coefficients which
    results when the i-th case is dropped from the regression.  Note
    that aliased coefficients are not included in the matrix.}
  \item{sigma}{a vector whose i-th element contains the estimate
    of the residual standard deviation obtained when the i-th
    case is dropped from the regression.}
  \item{wt.res}{a vector of \emph{weighted} (or for class \code{glm}
    rather \emph{deviance}) residuals.}
}
\note{
  The \code{coefficients} returned by the \R version
  of \code{lm.influence} differ from those computed by S.
  Rather than returning the coefficients which result
  from dropping each case, we return the changes in the coefficients.
  This is more directly useful in many diagnostic measures.\cr
  Since these need \eqn{O(n^2 p)} computing time, they can be omitted by
  \code{do.coef = FALSE}.

  Note that cases with \code{weights == 0} are \emph{dropped} (contrary
  to the situation in S).

  If a model has been fitted with \code{na.action=na.exclude} (see
  \code{\link{na.exclude}}), cases excluded in the fit \emph{are}
  considered here.
}
\references{
  See the list in the documentation for \code{\link{influence.measures}}.

  Chambers, J. M. (1992)
  \emph{Linear models.}
  Chapter 4 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\seealso{
\code{\link{summary.lm}} for \code{\link{summary}} and related methods;\cr
\code{\link{influence.measures}},\cr
\code{\link{hat}} for the hat matrix diagonals,\cr
\code{\link{dfbetas}},
\code{\link{dffits}},
\code{\link{covratio}},
\code{\link{cooks.distance}},
\code{\link{lm}}.
}
\examples{
## Analysis of the life-cycle savings data
## given in Belsley, Kuh and Welsch.
data(LifeCycleSavings)
summary(lm.SR <- lm(sr ~ pop15 + pop75 + dpi + ddpi,
                    data = LifeCycleSavings),
        corr = TRUE)
str(lmI <- lm.influence(lm.SR))

## For more "user level" examples, use example(influence.measures)
}
\keyword{regression}
\eof
\name{lm.summaries}
\alias{family.lm}
\alias{formula.lm}
\alias{residuals.lm}
\alias{weights}
\alias{weights.default}
\title{Accessing Linear Model Fits}
\usage{
\method{family}{lm}(object, \dots)

\method{formula}{lm}(x, \dots)

\method{residuals}{lm}(object,
          type = c("working", "response", "deviance","pearson", "partial"),
          \dots)

weights(object, \dots)
}
\arguments{
  \item{object, x}{an object inheriting from class \code{lm}, usually
    the result of a call to \code{\link{lm}} or \code{\link{aov}}.}
  \item{\dots}{further arguments passed to or from other methods.}
  \item{type}{the type of residuals which should be returned.}
}
\description{
  All these functions are \code{\link{methods}} for class \code{"lm"}  objects.
}
\details{
  The generic accessor functions \code{coef}, \code{effects},
  \code{fitted} and \code{residuals} can be used to extract
  various useful features of the value returned by \code{lm}.

  The working and response residuals are \dQuote{observed - fitted}.  The
  deviance and pearson residuals are weighted residuals, scaled by the
  square root of the weights used in fitting.  The partial residuals
  are a matrix with each column formed by omitting a term from the
  model.  In all these, zero weight cases are never omitted (as opposed
  to the standardized \code{\link{rstudent}} residuals).

}
\seealso{
  The model fitting function \code{\link{lm}}, \code{\link{anova.lm}}.

  \code{\link{coef}}, \code{\link{deviance}},
  \code{\link{df.residual}},
  \code{\link{effects}}, \code{\link{fitted}},
  \code{\link{glm}} for \bold{generalized} linear models,
  \code{\link{influence}} (etc on that page) for regression diagnostics,
  \code{\link{weighted.residuals}},
  \code{\link{residuals}}, \code{\link{residuals.glm}},
  \code{\link{summary.lm}}.
}
\references{
  Chambers, J. M. (1992)
  \emph{Linear models.}
  Chapter 4 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\examples{
\dontshow{example("lm", echo = FALSE)}
##-- Continuing the  lm(.) example:
coef(lm.D90)# the bare coefficients

## The 2 basic regression diagnostic plots [plot.lm(.) is preferred]
plot(resid(lm.D90), fitted(lm.D90))# Tukey-Anscombe's
abline(h=0, lty=2, col = 'gray')

qqnorm(residuals(lm.D90))
}
\keyword{regression}
\keyword{models}
\eof
\name{lm.fit}
\title{Fitter Functions for Linear Models}
\usage{
lm.fit (x, y,    offset = NULL, method = "qr", tol = 1e-7,
       singular.ok = TRUE, \dots)

lm.wfit(x, y, w, offset = NULL, method = "qr", tol = 1e-7,
        singular.ok = TRUE, \dots)
}
\alias{lm.fit}
\alias{lm.wfit}
\description{
  These are the basic computing engines called by \code{\link{lm}} used
  to fit linear models.  These should usually \emph{not} be used
  directly unless by experienced users.
}
\arguments{
  \item{x}{design matrix of dimension \code{n * p}.}
  \item{y}{vector of observations of length \code{n}.}
  \item{w}{vector of weights (length \code{n}) to be used in the fitting
    process for the \code{wfit} functions.  Weighted least squares is
    used with weights \code{w}, i.e., \code{sum(w * e^2)} is minimized.}
  \item{offset}{numeric of length \code{n}).  This can be used to
    specify an \emph{a priori} known component to be included in the
    linear predictor during fitting.}

  \item{method}{currently, only \code{method="qr"} is supported.}

  \item{tol}{tolerance for the \code{\link{qr}} decomposition.  Default
    is 1e-7.}

  \item{singular.ok}{logical. If \code{FALSE}, a singular model is an
    error.}
  
  \item{\dots}{currently disregarded.}
}
\details{
  The functions \code{lm.\{w\}fit.null} are called by \code{lm.fit} or
  \code{lm.wfit} respectively, when \code{x} has zero columns.
}
\value{
%% S(-PLUS) returns an object of class "lm"
%% such that print.lm, summary,... work; but that'd need more changes for R.
  a list with components
  \item{coefficients}{\code{p} vector}
  \item{residuals}{\code{n} vector}
  \item{fitted.values}{\code{n} vector}
  \item{effects}{(not null fits)\code{n} vector of orthogonal single-df
    effects.  The first \code{rank} of them correspond to non-aliased
    coeffcients, and are named accordingly.}
  \item{weights}{\code{n} vector --- \emph{only} for the \code{*wfit*}
    functions.}
  \item{rank}{integer, giving the rank}
  \item{df.residual}{degrees of freedom of residuals}
  \item{qr}{(not null fits) the QR decomposition, see \code{\link{qr}}.}
}
\seealso{
  \code{\link{lm}} which you should use for linear least squares regression,
  unless you know better.
}
\examples{
%% FIXME: Do something more sensible (non-random data) !!
set.seed(129)
n <- 7 ; p <- 2
X <- matrix(rnorm(n * p), n,p) # no intercept!
y <- rnorm(n)
w <- rnorm(n)^2

str(lmw <- lm.wfit(x=X, y=y, w=w))

str(lm. <- lm.fit (x=X, y=y))

%% do an example which sets 'tol' and gives a difference!
}
\keyword{regression}
\keyword{array}
\eof
\name{loadings}
\alias{loadings}
\alias{print.loadings}
\alias{print.factanal}
\title{Print Loadings in Factor Analysis}
\description{
  Extract or print loadings in factor analysis (or principal
  components analysis).
}
\usage{
loadings(x)

\method{print}{loadings}(x, digits = 3, cutoff = 0.1, sort = FALSE, \dots)

\method{print}{factanal}(x, digits = 3, \dots)
}
\arguments{
  \item{x}{an object of class \code{"factanal"} or \code{"princomp"}
    or the \code{loadings} component of such an object.}
  \item{digits}{number of decimal places to use in printing uniquenesses
    and loadings.}
  \item{cutoff}{loadings smaller than this (in absolute value) are suppressed.}
  \item{sort}{logical. If true, the variables are sorted by their
    importance on each factor.  Each variable with any loading larger
    than 0.5 (in modulus) is assigned to the factor with the largest
    loading, and the variables are printed in the order of the factor
    they are assigned to, then those unassigned.}
  \item{\dots}{further arguments for other methods, such as
    \code{cutoff} and \code{sort} for \code{print.factanal}.}
}

\seealso{
  \code{\link{factanal}}, \code{\link{princomp}}
}

\keyword{multivariate}
\keyword{print}
\eof
\name{loess}
\title{Local Polynomial Regression Fitting}
\usage{
loess(formula, data, weights, subset, na.action, model = FALSE,
      span = 0.75, enp.target, degree = 2,
      parametric = FALSE, drop.square = FALSE, normalize = TRUE,
      family = c("gaussian", "symmetric"),
      method = c("loess", "model.frame"),
      control = loess.control(\dots), \dots)
}
\alias{loess}
%\alias{anova.loess}
%\alias{print.loess}
%\alias{summary.loess}
%\alias{print.summary.loess}
\arguments{
  \item{formula}{a formula specifying the numeric response and one to four
    numeric predictors (best specified via an interaction, but can also
    be specified additively).}
  \item{data}{an optional data frame within which to look first for the
    response, predictors and weights.}
  \item{weights}{optional weights for each case.}
  \item{subset}{an optional specification of a subset of the data to be
    used.}
  \item{na.action}{the action to be taken with missing values in the
    response or predictors.  The default is given by
    \code{getOption("na.action")}.}
  \item{model}{should the model frame be returned?}
  \item{span}{the parameter \eqn{\alpha} which controls the degree of
    smoothing.}
  \item{enp.target}{an alternative way to specify \code{span}, as the
    approximate equivalent number of parameters to be used.}
  \item{degree}{the degree of the polynomials to be used, up to 2.}
  \item{parametric}{should any terms be fitted globally rather than
    locally?  Terms can be specified by name, number or as a logical
    vector of the same length as the number of predictors.}
  \item{drop.square}{for fits with more than one predictor and
    \code{degree=2}, should the quadratic term (and cross-terms) be
    dropped for particular predictors?  Terms are specified in the same
    way as for \code{parametric}.}
  \item{normalize}{should the predictors be normalized to a common scale
    if there is more than one?  The normalization used is to set the
    10\% trimmed standard deviation to one.  Set to false for spatial
    coordinate predictors and others know to be a common scale.}
  \item{family}{if \code{"gaussian"} fitting is by least-squares, and if
    \code{"symmetric"} a re-descending M estimator is used with Tukey's
    biweight function.}
  \item{method}{fit the model or just extract the model frame.}
  \item{control}{control parameters: see \code{\link{loess.control}}.}
  \item{\dots}{control parameters can also be supplied directly.}
}
\description{
  Fit a polynomial surface determined by one or more numerical
  predictors, using local fitting.
}
\details{
  Fitting is done locally.  That is, for the fit at point \eqn{x}, the
  fit is made using points in a neighbourhood of \eqn{x}, weighted by
  their distance from \eqn{x} (with differences in \sQuote{parametric}
  variables being ignored when computing the distance). The size of the
  neighbourhood is controlled by \eqn{\alpha} (set by \code{span} or
  \code{enp.target}).  For \eqn{\alpha < 1}, the neighbourhood includes
  proportion \eqn{\alpha} of the points, and these have tricubic weighting
  (proportional to \eqn{(1 - \mathrm{(dist/maxdist)}^3)^3}{(1 -
    (dist/maxdist)^3)^3}. For \eqn{\alpha > 1}, all points are used, with
  the \sQuote{maximum distance} assumed to be
  \eqn{\alpha^{1/p}}{alpha^1/p} times the actual maximum distance for
  \eqn{p} explanatory variables.

  For the default family, fitting is by (weighted) least squares. For
  \code{family="symmetric"} a few iterations of an M-estimation
  procedure with Tukey's biweight are used. Be aware that as the initial
  value is the least-squares fit, this need not be a very resistant fit.

  It can be important to tune the control list to achieve acceptable
  speed. See \code{\link{loess.control}} for details.
}
\value{
  An object of class \code{"loess"}.
}
\references{
  W.S. Cleveland, E. Grosse and W.M. Shyu (1992) Local regression
  models. Chapter 8 of \emph{Statistical Models in S} eds J.M. Chambers
  and T.J. Hastie, Wadsworth & Brooks/Cole.
}
\author{
  B.D. Ripley, based on the \code{cloess} package of Cleveland,
  Grosse and Shyu avaliable at \url{http://www.netlib.org/a/}.
}
\note{
  As this is based on the \code{cloess} package available at
  \code{netlib}, it is similar to but not identical to the \code{loess}
  function of S. In particular, conditioning is not implemented.

  The memory usage of this implementation of \code{loess} is roughly
  quadratic in the number of points, with 1000 points taking about 10Mb.
}
\seealso{
  \code{\link{loess.control}},
  \code{\link{predict.loess}}.

  \code{\link{lowess}}, the ancestor of \code{loess} (with
  different defaults!).
}
\examples{
data(cars)
cars.lo <- loess(dist ~ speed, cars)
predict(cars.lo, data.frame(speed = seq(5, 30, 1)), se = TRUE)
# to allow extrapolation
cars.lo2 <- loess(dist ~ speed, cars,
  control = loess.control(surface = "direct"))
predict(cars.lo2, data.frame(speed = seq(5, 30, 1)), se = TRUE)
}
\keyword{smooth}
\keyword{loess}
\eof
\name{loess.control}
\title{Set Parameters for Loess}
\usage{
loess.control(surface = c("interpolate", "direct"),
              statistics = c("approximate", "exact"),
              trace.hat = c("exact", "approximate"),
              cell = 0.2, iterations = 4, \dots)
}
\alias{loess.control}
\arguments{
 \item{surface}{should be fitted surface be computed exactly or via
   interpolation from a kd tree?}
 \item{statistics}{should the statistics be computed exactly or
   approximately? Exact computation can be very slow.}
 \item{trace.hat}{should the trace of the smoother matrix be computed
   exactly or approximately? It is recommended to use the approximation
   for more than about 1000 data points.}
 \item{cell}{if interpolation is used this controls the accuracy of the
   approximation via the maximum number of points in a  cell in the kd
   tree. Cells with more than \code{floor(n*span*cell)} points are subdivided.}
 \item{iterations}{the number of iterations used in robust fitting.}
  \item{\dots}{further arguments which are ignored.}
}
\description{
  Set control parameters for \code{loess} fits.
}
\value{
  A list with components
  \item{surface}{}
  \item{statistics}{}
  \item{trace.hat}{}
  \item{cell}{}
  \item{iterations}{}
  with meanings as explained under \sQuote{Arguments}.
}
\seealso{\code{\link{loess}}}
\keyword{smooth}
\keyword{loess}
\eof
\name{logLik}
\alias{logLik}
\alias{print.logLik}
\alias{str.logLik}
% \alias{as.data.frame.logLik}
\alias{logLik.lm}
\title{Extract Log-Likelihood}
\usage{
logLik(object, \dots)

\method{logLik}{lm}(object, REML = FALSE, \dots)
%
%\method{as.data.frame}{logLik}(x, row.names = NULL, optional = FALSE)
%notyet print(logLik(*), ...)
%notyet   str(logLik(*), ...)
}
\arguments{
 \item{object}{any object from which a log-likelihood value, or a
   contribution to a log-likelihood value, can be extracted.}
 \item{\dots}{some methods for this generic function require additional
   arguments.}
  \item{REML}{an optional logical value.  If \code{TRUE} the restricted
    log-likelihood is returned, else, if \code{FALSE}, the
    log-likelihood is returned.  Defaults to \code{FALSE}.}
% \item{x}{an object of class \code{logLik}.}
% \item{row.names, optional}{arguments to the \code{\link{as.data.frame}}
%   method; see its documentation.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects.  Classes which already have methods for
  this function include: \code{glm}, \code{lm}, \code{nls}
  and \code{gls}, \code{lme} and others in package \pkg{nlme}.
%   \code{corStruct}, \code{lmList}, \code{lmeStruct}, \code{reStruct}, and
%   \code{varFunc}.
}
\details{
  For a \code{glm} fit the \code{\link{family}} does not have to specify
  how to calculate the log-likelihood, so this is based on the
  family's function to compute
  the AIC.  For \code{\link{gaussian}}, \code{\link{Gamma}} and
  \code{\link{inverse.gaussian}}
  families it assumed that the dispersion of the GLM is estimated and
  has been included in the AIC, and for all other families it is assumed
  that the dispersion is known.

  Note that this procedure is not completely accurate for the gamma and
  inverse gaussian families, as the estimate of dispersion used is not
  the MLE.
}
\value{
  Returns an object, say \code{r}, of class \code{logLik} which is a
  number with attributes, \code{attr(r, "df")} (\bold{d}egrees of
  \bold{f}reedom) giving the number of parameters in the model.
  There's a simple \code{print} method for \code{logLik} objects.

  The details depend on the method function used; see the appropriate
  documentation.
}
\seealso{
  \code{\link[nlme]{logLik.gls}}, \code{\link[nlme]{logLik.lme}}, etc.
}
\references{
  For \code{logLik.lm}:
  
  Harville, D.A. (1974).
  Bayesian inference for variance components using only error contrasts.
  \emph{Biometrika}, \bold{61}, 383--385.
}
\author{Jose Pinheiro and Douglas Bates}
\examples{
x <- 1:5
lmx <- lm(x ~ 1)
logLik(lmx) # using print.logLik() method
str(logLik(lmx))

## lm method
data(attitude)
(fm1 <- lm(rating ~ ., data = attitude))
logLik(fm1)
logLik(fm1, REML = TRUE)

res <- try(data(Orthodont, package="nlme"))
if(!inherits(res, "try-error")) {
  fm1 <- lm(distance ~ Sex * age, Orthodont)
  print(logLik(fm1))
  print(logLik(fm1, REML = TRUE))
}
}
\keyword{models}
\eof
\name{loglin}
\alias{loglin}
\title{Fitting Log-Linear Models}
\usage{
loglin(table, margin, start = rep(1, length(table)), fit = FALSE,
       eps = 0.1, iter = 20, param = FALSE, print = TRUE)
}
\description{
  \code{loglin} is used to fit log-linear models to multidimensional
  contingency tables by Iterative Proportional Fitting.  
}
\arguments{
  \item{table}{a contingency table to be fit, typically the output from
    \code{table}.}
  \item{margin}{a list of vectors with the marginal totals to be fit.

    (Hierarchical) log-linear models can be specified in terms of these
    marginal totals which give the \dQuote{maximal} factor subsets contained
    in the model.  For example, in a three-factor model,
    \code{list(c(1, 2), c(1, 3))} specifies a model which contains
    parameters for the grand mean, each factor, and the 1-2 and 1-3
    interactions, respectively (but no 2-3 or 1-2-3 interaction), i.e.,
    a model where factors 2 and 3 are independent conditional on factor
    1 (sometimes represented as \sQuote{[12][13]}).
     
    The names of factors (i.e., \code{names(dimnames(table))}) may be
    used rather than numeric indices.
  }
  \item{start}{a starting estimate for the fitted table.  This optional
    argument is important for incomplete tables with structural zeros
    in \code{table} which should be preserved in the fit.  In this
    case, the corresponding entries in \code{start} should be zero and
    the others can be taken as one.}
  \item{fit}{a logical indicating whether the fitted values should be
    returned.}
  \item{eps}{maximum deviation allowed between observed and fitted
    margins.}
  \item{iter}{maximum number of iterations.}
  \item{param}{a logical indicating whether the parameter values should
    be returned.}
  \item{print}{a logical.  If \code{TRUE}, the number of iterations and
    the final deviation are printed.}
}
\value{
  A list with the following components.
  \item{lrt}{the Likelihood Ratio Test statistic.}
  \item{pearson}{the Pearson test statistic (X-squared).}
  \item{df}{the degrees of freedom for the fitted model.  There is no
    adjustment for structural zeros.}
  \item{margin}{list of the margins that were fit.  Basically the same
    as the input \code{margin}, but with numbers replaced by names
    where possible.} 
  \item{fit}{An array like \code{table} containing the fitted values.
    Only returned if \code{fit} is \code{TRUE}.}
  \item{param}{A list containing the estimated parameters of the
    model.  The \dQuote{standard} constraints of zero marginal sums
    (e.g., zero row and column sums for a two factor parameter) are
    employed.  Only returned if \code{param} is \code{TRUE}.}
}
\details{
  The Iterative Proportional Fitting algorithm as presented in
  Haberman (1972) is used for fitting the model.  At most \code{iter}
  iterations are performed, convergence is taken to occur when the
  maximum deviation between observed and fitted margins is less than
  \code{eps}.  All internal computations are done in double precision;
  there is no limit on the number of factors (the dimension of the
  table) in the model.

  Assuming that there are no structural zeros, both the Likelihood
  Ratio Test and Pearson test statistics have an asymptotic chi-squared
  distribution with \code{df} degrees of freedom.
    
  Package \pkg{MASS} contains \code{loglm}, a front-end to \code{loglin}
  which allows the log-linear model to be specified and fitted in a
  formula-based manner similar to that of other fitting functions such
  as \code{lm} or \code{glm}.
}

\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.

  Haberman, S. J. (1972)
  Log-linear fit for contingency tables---Algorithm AS51.
  \emph{Applied Statistics}, \bold{21}, 218--225.

  Agresti, A. (1990)
  \emph{Categorical data analysis}.
  New York: Wiley.
}
\author{Kurt Hornik}
\seealso{\code{\link{table}}}
\examples{
data(HairEyeColor)
## Model of joint independence of sex from hair and eye color.
fm <- loglin(HairEyeColor, list(c(1, 2), c(1, 3), c(2, 3)))
fm
1 - pchisq(fm$lrt, fm$df)
## Model with no three-factor interactions fits well.
}
\keyword{category}
\keyword{models}
\eof
\name{lowess}
\title{Scatter Plot Smoothing}
\usage{
lowess(x, y = NULL, f = 2/3, iter=3, delta = 0.01 * diff(range(xy$x[o])))
}
\alias{lowess}
\arguments{
  \item{x, y}{vectors giving the coordinates of the points in the scatter plot.
    Alternatively a single plotting structure can be specified.}
  \item{f}{the smoother span. This gives the proportion of points in
    the plot which influence the smooth at each value.
    Larger values give more smoothness.}
  \item{iter}{the number of robustifying iterations which should be
    performed.
    Using smaller values of \code{iter} will make \code{lowess} run faster.}
  \item{delta}{values of \code{x} which lie within \code{delta}
    of each other are replaced by a single value in the output from
    \code{lowess}.  Defaults to 1/100th of the range of \code{x}.}
}
\description{
  This function performs the computations for the
  \emph{LOWESS} smoother (see the reference below).
  \code{lowess} returns a list containing components
  \code{x} and \code{y} which give the coordinates of the smooth.
  The smooth should be added to a plot of the original
  points with the function \code{lines}.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.

  Cleveland, W. S. (1979)
  Robust locally weighted regression and smoothing scatterplots.
  \emph{J. Amer. Statist. Assoc.} \bold{74}, 829--836.

  Cleveland, W. S. (1981)
  LOWESS: A program for smoothing scatterplots by robust locally weighted
  regression. \emph{The American Statistician}, \bold{35}, 54.
}
\seealso{\code{\link{loess}}), a newer
  formula based version of \code{lowess} (with different defaults!).
}
\examples{
data(cars)
plot(cars, main = "lowess(cars)")
lines(lowess(cars), col = 2)
lines(lowess(cars, f=.2), col = 3)
legend(5, 120, c(paste("f = ", c("2/3", ".2"))), lty = 1, col = 2:3)
}
\keyword{smooth}
\eof
\name{ls.diag}
\title{Compute Diagnostics for `lsfit' Regression Results}
\usage{
ls.diag(ls.out)
}
\alias{ls.diag}
\arguments{
  \item{ls.out}{Typically the result of \code{\link{lsfit}()}}
}
\description{
  Computes basic statistics, including standard errors, t- and p-values
  for the regression coefficients.
}
\value{
  A \code{list} with the following numeric components.
  \item{std.dev}{The standard deviation of the errors, an estimate of
    \eqn{\sigma}{sigma}.}
  \item{hat}{diagonal entries \eqn{h_{ii}} of the hat matrix \eqn{H}}
  \item{std.res}{standardized residuals}
  \item{stud.res}{studentized residuals}
  \item{cooks}{Cook's distances}
  \item{dfits}{DFITS statistics}
  \item{correlation}{correlation matrix}
  \item{std.err}{standard errors of the regression coefficients}
  \item{cov.scaled}{Scaled covariance matrix of the coefficients}
  \item{cov.unscaled}{Unscaled covariance matrix of the coefficients}
}
\references{
  Belsley, D. A., Kuh, E. and Welsch, R. E. (1980)
  \emph{Regression Diagnostics.}
  New York: Wiley.
}
\seealso{
  \code{\link{hat}} for the hat matrix diagonals,
  \code{\link{ls.print}},
  \code{\link{lm.influence}}, \code{\link{summary.lm}},
  \code{\link{anova}}.
}
\examples{
\dontshow{example("lm", echo = FALSE)}
##-- Using the same data as the lm(.) example:
lsD9 <- lsfit(x = as.numeric(gl(2, 10, 20)), y = weight)
dlsD9 <- ls.diag(lsD9)
str(dlsD9, give.attr=FALSE)
abs(1 - sum(dlsD9$hat) / 2) < 10*.Machine$double.eps # sum(h.ii) = p
plot(dlsD9$hat, dlsD9$stud.res, xlim=c(0,0.11))
abline(h = 0, lty = 2, col = "lightgray")
}
\keyword{regression}
\eof
\name{ls.print}
\title{Print `lsfit' Regression Results}
\usage{
ls.print(ls.out, digits = 4, print.it = TRUE)
}
\alias{ls.print}
\arguments{
  \item{ls.out}{Typically the result of \code{\link{lsfit}()}}
  \item{digits}{The number of significant digits used for printing}
  \item{print.it}{a logical indicating whether the result should also be
    printed}
}
\description{
  Computes basic statistics, including standard errors, t- and p-values
  for the regression coefficients and prints them if \code{print.it} is
  \code{TRUE}.
}
\value{
  A list with the components
  \item{summary}{The ANOVA table of the regression}
  \item{coef.table}{matrix with regression coefficients, standard
    errors, t- and p-values}
}
\note{
  Usually, you'd rather use \code{summary(lm(\ldots))} and
  \code{anova(lm(\ldots))} for obtaining similar output.
}
\seealso{
  \code{\link{ls.diag}}, \code{\link{lsfit}}, also for examples;
  \code{\link{lm}}, \code{\link{lm.influence}} which usually are
  preferable.
}
\keyword{regression}
\eof
\name{lsfit}
\title{Find the Least Squares Fit}
\usage{
lsfit(x, y, wt=NULL, intercept=TRUE, tolerance=1e-07, yname=NULL)
}
\alias{lsfit}
\arguments{
\item{x}{a matrix whose rows correspond to cases and whose columns
  correspond to variables.}
\item{y}{the responses, possibly a matrix if you want to fit multiple
  left hand sides.}
\item{wt}{an optional vector of weights for performing weighted least squares.}
\item{intercept}{whether or not an intercept term should be used.}
\item{tolerance}{the tolerance to be used in the matrix decomposition.}
\item{yname}{names to be used for the response variables.}
}
\description{
The least squares estimate of \bold{\eqn{\beta}{b}} in the model
\deqn{\bold{Y} = \bold{X \beta} + \bold{\epsilon}}{y = X b + e}
is found.
}
\details{
If weights are specified then a weighted least squares is performed
with the weight given to the \emph{j}th case specified by the \emph{j}th
entry in \code{wt}.

If any observation has a missing value in any field, that observation
is removed before the analysis is carried out.
This can be quite inefficient if there is a lot of missing data.

The implementation is via a modification of the LINPACK subroutines
which allow for multiple left-hand sides.
}
\value{
A list with the following named components:
\item{coef}{the least squares estimates of the coefficients in
  the model (\bold{\eqn{\beta}{b}} as stated above).}
\item{residuals}{residuals from the fit.}
\item{intercept}{indicates whether an intercept was fitted.}
\item{qr}{the QR decomposition of the design matrix.}
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
\code{\link{lm}} which usually is preferable;
\code{\link{ls.print}}, \code{\link{ls.diag}}.
}
\examples{
\dontshow{example("lm", echo = FALSE)}
##-- Using the same data as the lm(.) example:
lsD9 <- lsfit(x = unclass(gl(2,10)), y = weight)
ls.print(lsD9)
}
\keyword{regression}
\eof
\name{lynx}
\docType{data}
\alias{lynx}
\title{Annual Canadian Lynx trappings 1821--1934}
\description{
  Annual numbers of lynx trappings for 1821--1934 in Canada. Taken from
  Brockwell & Davis (1991), this appears to be the series considered
  by Campbell & Walker (1977).
}
\usage{data(lynx)}
\source{
  Brockwell, P. J. and Davis, R. A. (1991) \emph{Time
    Series and Forecasting Methods.}  Second edition.
  Springer. Series G (page 557).
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.

  Campbell, M. J.and A. M.  Walker (1977).  A Survey of
  statistical work on the Mackenzie River series of annual
  Canadian lynx trappings for the years  1821--1934 and
  a new analysis.
  \emph{Journal of the Royal Statistical Society series A},
  \bold{140}, 411--431.
}
\keyword{datasets}
\eof
\name{mad}
\title{Median Absolute Deviation}
\usage{
mad(x, center = median(x), constant = 1.4826, na.rm = FALSE,
    low = FALSE, high = FALSE)
}
\alias{mad}
\arguments{
  \item{x}{a numeric vector.}
  \item{center}{Optionally, the centre: defaults to the median.}
  \item{constant}{scale factor.}
  \item{na.rm}{if \code{TRUE} then \code{NA} values are stripped
    from \code{x} before computation takes place.}
  \item{low}{if \code{TRUE}, compute the \dQuote{lo-median}, i.e., for even
    sample size, do not average the two middle values, but take the
    smaller one.}
  \item{high}{if \code{TRUE}, compute the \dQuote{hi-median}, i.e., take the
    larger of the two middle values for even sample size.}
}
\description{
  Compute the median absolute deviation, i.e., the (lo-/hi-) median of
  the absolute deviations from the median, and (by default) adjust by a
  factor for asymptotically normal consistency.
}
\details{
  The actual value calculated is \code{constant * cMedian(abs(x - center))}
  with the default value of \code{center} being \code{median(x)}, and
  \code{cMedian} being the usual, the \dQuote{low} or \dQuote{high} median, see
  the arguments description for \code{low} and \code{high} above.

  The default \code{constant = 1.4826} (approximately
  \eqn{1/\Phi^{-1}(\frac 3 4)}{1/ Phi^(-1)(3/4)} = \code{1/qnorm(3/4)})
  ensures consistency, i.e.,
  \deqn{E[mad(X_1,\dots,X_n)] = \sigma}
  for \eqn{X_i} distributed as \eqn{N(\mu,\sigma^2)} and large \eqn{n}.

  If \code{na.rm} is \code{TRUE} then \code{NA}
  values are stripped from \code{x} before computation takes place.
  If this is not done then an \code{NA} value in
  \code{x} will cause \code{mad} to return \code{NA}.
}
\seealso{
  \code{\link{IQR}} which is simpler but less robust,
  \code{\link{median}}, \code{\link{var}}.
}
\examples{
mad(c(1:9))
print(mad(c(1:9),     constant=1)) ==
      mad(c(1:8,100), constant=1)       # = 2 ; TRUE
x <- c(1,2,3, 5,7,8)
sort(abs(x - median(x)))
c(mad(x, co=1), mad(x, co=1, lo = TRUE), mad(x, co=1, hi = TRUE))
}
\keyword{univar}
\keyword{robust}

\eof
\name{mahalanobis}
\title{Mahalanobis Distance}
\usage{
mahalanobis(x, center, cov, inverted=FALSE, tol.inv = 1e-7)
}
\alias{mahalanobis}
\arguments{
  \item{x}{vector or matrix of data with, say, \eqn{p} columns.}
  \item{center}{mean vector of the distribution or second data vector of
    length \eqn{p}.}
  \item{cov}{covariance matrix (\eqn{p \times p}{p x p}) of the distribution.}
  \item{inverted}{logical.  If \code{TRUE}, \code{cov} is supposed to
    contain the \emph{inverse} of the covariance matrix.}
  \item{tol.inv}{tolerance to be used for computing the inverse (if
    \code{inverted} is false), see \code{\link{solve}}.}
}
\description{
  Returns the Mahalanobis distance of all rows in \code{x} and the
  vector \eqn{\mu=}\code{center} with respect to
  \eqn{\Sigma=}\code{cov}.
  This is (for vector \code{x}) defined as
  \deqn{
    D^2 = (x - \mu)' \Sigma^{-1} (x - \mu)
  }
}
\author{Friedrich Leisch}
\seealso{\code{\link{cov}}, \code{\link{var}}}
\examples{
ma <- cbind(1:6, 1:3)
(S <-  var(ma))
mahalanobis(c(0,0), 1:2, S)

x <- matrix(rnorm(100*3), ncol = 3)
stopifnot(mahalanobis(x, 0, diag(ncol(x))) == rowSums(x*x))
	##- Here, D^2 = usual Euclidean distances
Sx <- cov(x)
D2 <- mahalanobis(x, rowMeans(x), Sx)
plot(density(D2, bw=.5), main="Mahalanobis distances, n=100, p=3"); rug(D2)
qqplot(qchisq(ppoints(100), df=3), D2,
       main = expression("Q-Q plot of Mahalanobis" * ~D^2 *
                         " vs. quantiles of" * ~ chi[3]^2))
abline(0, 1, col = 'gray')
}
\keyword{multivariate}
\eof
\name{make.link}
\alias{make.link}
\title{Create a Link for GLM families}
\usage{
make.link(link)
}
\arguments{
    \item{link}{character or numeric; one of \code{"logit"},
        \code{"probit"}, \code{"cloglog"}, \code{"identity"},
        \code{"log"},  \code{"sqrt"},  \code{"1/mu^2"},
        \code{"inverse"}, or number, say \eqn{\lambda}{lambda} resulting
        in power link \eqn{=\mu ^ \lambda}{mu ^ lambda}.}
}
%%-- implementation in ../R/family.R
\description{
    This function is used with the \code{\link{family}} functions in
    \code{\link{glm}()}.
    Given a link, it returns a link function, an inverse link
    function, the derivative \eqn{d\mu / d\eta}{dmu/deta} and a function
    for domain checking.
}
\value{
    A list with components
    \item{linkfun}{Link function \code{function(mu)}}
    \item{linkinv}{Inverse link function \code{function(eta)}}
    \item{mu.eta}{Derivative \code{function(eta)} \eqn{d\mu / d\eta}{dmu/deta}}
    \item{valideta}{\code{function(eta)}\{ \code{TRUE} if all of
        \code{eta} is in the domain of \code{linkinv} \}.}
}
\seealso{\code{\link{glm}}, \code{\link{family}}.}
\examples{
str(make.link("logit"))

l2 <- make.link(2)
l2$linkfun(0:3)# 0 1 4 9
l2$mu.eta(eta= 1:2)#= 1/(2*sqrt(eta))
}
\keyword{models}

\eof
\name{make.tables}
\title{Create model.tables}
\alias{make.tables.aovproj}
\alias{make.tables.aovprojlist}
\usage{
make.tables.aovproj    (proj.cols, mf.cols, prjs, mf,
                        fun = "mean", prt = FALSE, \dots)

make.tables.aovprojlist(proj.cols, strata.cols, model.cols, projections,
                        model, eff, fun = "mean", prt = FALSE, \dots)  
}
\description{
 These are support functions for (the methods of)
 \code{\link{model.tables}} and probably not much of use otherwise.
}
%% \details{
%%              ___ SOURCE in ../R/model.tables.R ___
%% }
\seealso{\code{\link{model.tables}}}
\keyword{internal}
\eof
\name{makepredictcall}
\alias{makepredictcall}
\alias{makepredictcall.default}
\alias{SafePrediction}
\title{Utility Function for Safe Prediction}
\description{
  A utility to help \code{\link{model.frame.default}} create the right
  matrices when predicting from models with terms like \code{poly} or
  \code{ns}. 
}
\usage{
makepredictcall(var, call)
}
\arguments{
  \item{var}{A variable.}
  \item{call}{The term in the formula, as a call.}
}
\details{
  This is a generic function with methods for \code{poly}, \code{bs} and
  \code{ns}: the default method handles \code{scale}.  If
  \code{model.frame.default} encounters such a term when
  creating a model frame, it modifies the \code{predvars} attribute of
  the terms supplied to replace the term with one that will work for
  predicting new data.  For example \code{makepredictcall.ns} adds
  arguments for the knots and intercept.

  To make use of this, have your model-fitting function return the
  \code{terms} attribute of the model frame, or copy the \code{predvars}
  attribute of the \code{terms} attribute of the model frame to your
  \code{terms} object.

  To extend this, make sure the term creates variables with a class,
  and write a suitable method for that class.
}
\value{
  A replacement for \code{call} for the \code{predvars} attribute of
  the terms.
}
\seealso{
  \code{\link{model.frame}}, \code{\link{poly}}, \code{\link{scale}},
  \code{\link[splines]{bs}}, \code{\link[splines]{ns}}, \code{\link{cars}}
}
\examples{
## using poly: this did not work in R < 1.5.0
data(women)
fm <- lm(weight ~ poly(height, 2), data = women)
plot(women, xlab = "Height (in)", ylab = "Weight (lb)")
ht <- seq(57, 73, len = 200)
lines(ht, predict(fm, data.frame(height=ht)))

## see also example(cars)

## see bs and ns for spline examples.
}
\keyword{models}
\eof
\name{manova}
\alias{manova}
\title{Multivariate Analysis of Variance}
\description{
  A class for the multivariate analysis of variance.
}
\usage{
manova(\dots)
}
\arguments{
  \item{\dots}{Arguments to be passed to \code{\link{aov}}.}
}
\details{
  Class \code{"manova"} differs from class \code{"aov"} in selecting a
  different \code{summary} method.  Function \code{manova} calls
  \code{\link{aov}} and then add class \code{"manova"} to the result
  object for each stratum.
}
\value{
  See \code{\link{aov}} and the comments in Details here.
}
\note{
  \code{manova} does not support multistratum analysis of variance, so
  the formula should not include an \code{Error} term.
}
\references{
  Krzanowski, W. J. (1988) \emph{Principles of Multivariate Analysis. A
    User's Perspective.} Oxford.

  Hand, D. J. and Taylor, C. C.  (1987)
  \emph{Multivariate Analysis of Variance and Repeated Measures.}
  Chapman and Hall.
}

\seealso{
  \code{\link{aov}}, \code{\link{summary.manova}}, the latter containing
  examples.
}
\keyword{models}
\eof
\name{mantelhaen.test}
\alias{mantelhaen.test}
\title{Cochran-Mantel-Haenszel Chi-Squared Test for Count Data}
\description{
  Performs a Cochran-Mantel-Haenszel chi-squared test of the null that
  two nominal variables are conditionally independent in each stratum,
  assuming that there is no three-way interaction.
}
\usage{
mantelhaen.test(x, y = NULL, z = NULL,
                alternative = c("two.sided", "less", "greater"),
                correct = TRUE, exact = FALSE, conf.level = 0.95)
}
\arguments{
  \item{x}{either a 3-dimensional contingency table in array form where
    each dimension is at least 2 and the last dimension corresponds to
    the strata, or a factor object with at least 2 levels.}
  \item{y}{a factor object with at least 2 levels; ignored if \code{x}
    is an array.}
  \item{z}{a factor object with at least 2 levels identifying to which
    stratum the corresponding elements in \code{x} and \code{y} belong;
    ignored if \code{x} is an array.}
  \item{alternative}{indicates the alternative hypothesis and must be
    one of \code{"two.sided"}, \code{"greater"} or \code{"less"}.
    You can specify just the initial letter.
    Only used in the 2 by 2 by \eqn{K} case.}
  \item{correct}{a logical indicating whether to apply continuity
    correction when computing the test statistic.
    Only used in the 2 by 2 by \eqn{K} case.}
  \item{exact}{a logical indicating whether the Mantel-Haenszel test or
    the exact conditional test (given the strata margins) should be
    computed.
    Only used in the 2 by 2 by \eqn{K} case.}
  \item{conf.level}{confidence level for the returned confidence
    interval.
    Only used in the 2 by 2 by \eqn{K} case.}
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{Only present if no exact test is performed.  In the
    classical case of a 2 by 2 by \eqn{K} table (i.e., of dichotomous
    underlying variables), the Mantel-Haenszel chi-squared statistic;
    otherwise, the generalized Cochran-Mantel-Haenszel statistic.}
  \item{parameter}{the degrees of freedom of the approximate chi-squared
    distribution of the test statistic (\eqn{1} in the classical case).
    Only present if no exact test is performed.}
  \item{p.value}{the p-value of the test.}
  \item{conf.int}{a confidence interval for the common odds ratio.
    Only present in the 2 by 2 by \eqn{K} case.}
  \item{estimate}{an estimate of the common odds ratio.  If an exact
    test is performed, the conditional Maximum Likelihood Estimate is
    given; otherwise, the Mantel-Haenszel estimate.
    Only present in the 2 by 2 by \eqn{K} case.}
  \item{null.value}{the common odds ratio under the null of
    independence, \code{1}.
    Only present in the 2 by 2 by \eqn{K} case.}
  \item{alternative}{a character string describing the alternative
    hypothesis.
    Only present in the 2 by 2 by \eqn{K} case.}
  \item{method}{a character string indicating the method employed, and whether
    or not continuity correction was used.}
  \item{data.name}{a character string giving the names of the data.}
}
\details{
  If \code{x} is an array, each dimension must be at least 2, and
  the entries should be nonnegative integers.  \code{NA}'s are not
  allowed.  Otherwise, \code{x}, \code{y} and \code{z} must have the
  same length.  Triples containing \code{NA}'s are removed.  All
  variables must take at least two different values.
}
\note{
  The asymptotic distribution is only valid if there is no three-way
  interaction.  In the classical 2 by 2 by \eqn{K} case, this is
  equivalent to the conditional odds ratios in each stratum being
  identical.  Currently, no inference on homogeneity of the odds ratios
  is performed.

  See also the example below.
}
\references{
  Alan Agresti (1990).
  \emph{Categorical data analysis}.
  New York: Wiley.
  Pages 230--235.
}
\examples{
## Agresti (1990), pages 231--237, Penicillin and Rabbits
## Investigation of the effectiveness of immediately injected or 1.5
##  hours delayed penicillin in protecting rabbits against a lethal
##  injection with beta-hemolytic streptococci.
Rabbits <-
array(c(0, 0, 6, 5,
        3, 0, 3, 6,
        6, 2, 0, 4,
        5, 1, 6, 0,
        2, 5, 0, 0),
      dim = c(2, 2, 5),
      dimnames = list(
          Delay = c("None", "1.5h"),
          Response = c("Cured", "Died"),
          Penicillin.Level = c("1/8", "1/4", "1/2", "1", "4")))
Rabbits
## Classical Mantel-Haenszel test
mantelhaen.test(Rabbits)
## => p = 0.047, some evidence for higher cure rate of immediate
##               injection
## Exact conditional test
mantelhaen.test(Rabbits, exact = TRUE)
## => p - 0.040
## Exact conditional test for one-sided alternative of a higher
## cure rate for immediate injection
mantelhaen.test(Rabbits, exact = TRUE, alternative = "greater")
## => p = 0.020

## UC Berkeley Student Admissions
data(UCBAdmissions)
mantelhaen.test(UCBAdmissions)
## No evidence for association between admission and gender
## when adjusted for department.  However,
apply(UCBAdmissions, 3, function(x) (x[1,1]*x[2,2])/(x[1,2]*x[2,1]))
## This suggests that the assumption of homogeneous (conditional)
## odds ratios may be violated.  The traditional approach would be
## using the Woolf test for interaction:
woolf <- function(x) {
  x <- x + 1 / 2
  k <- dim(x)[3]
  or <- apply(x, 3, function(x) (x[1,1]*x[2,2])/(x[1,2]*x[2,1]))
  w <-  apply(x, 3, function(x) 1 / sum(1 / x))
  1 - pchisq(sum(w * (log(or) - weighted.mean(log(or), w)) ^ 2), k - 1)
}
woolf(UCBAdmissions)
## => p = 0.003, indicating that there is significant heterogeneity.
## (And hence the Mantel-Haenszel test cannot be used.)
}  
\keyword{htest}
\eof
\name{mcnemar.test}
\alias{mcnemar.test}
\title{McNemar's Chi-squared Test for Count Data}
\description{
  Performs McNemar's chi-squared test for symmetry of rows and columns
  in a two-dimensional contingency table.
}
\usage{
mcnemar.test(x, y = NULL, correct = TRUE)
}
\arguments{
  \item{x}{either a two-dimensional contingency table in matrix form,
    or a factor object.}
  \item{y}{a factor object; ignored if \code{x} is a matrix.}
  \item{correct}{a logical indicating whether to apply continuity
    correction when computing the test statistic.} 
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of McNemar's statistic.}
  \item{parameter}{the degrees of freedom of the approximate
    chi-squared distribution of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{method}{a character string indicating the type of test
    performed, and whether continuity correction was used.}
  \item{data.name}{a character string giving the name(s) of the data.}
}
\details{
  The null is that the probabilities of being classified into cells
  \code{[i,j]} and \code{[j,i]} are the same.

  If \code{x} is a matrix, it is taken as a two-dimensional contingency
  table, and hence its entries should be nonnegative integers.
  Otherwise, both \code{x} and \code{y} must be vectors of the same
  length.  Incomplete cases are removed, the vectors are coerced into
  factor objects, and the contingency table is computed from these.

  Continuity correction is only used in the 2-by-2 case if
  \code{correct} is \code{TRUE}.
}
\references{
  Alan Agresti (1990).
  \emph{Categorical data analysis}.
  New York: Wiley.
  Pages 350--354.
}
\examples{
## Agresti (1990), p. 350.
## Presidential Approval Ratings.
##  Approval of the President's performance in office in two surveys,
##  one month apart, for a random sample of 1600 voting-age Americans.
Performance <-
matrix(c(794, 86, 150, 570),
       nr = 2,
       dimnames = list("1st Survey" = c("Approve", "Disapprove"),
                       "2nd Survey" = c("Approve", "Disapprove")))
Performance
mcnemar.test(Performance)
## => very strong association between the two successive ratings
}
\keyword{htest}
\eof
\name{median}
\title{Median Value}
\usage{
median(x, na.rm=FALSE)
}
\alias{median}
\arguments{
  \item{x}{a numeric vector containing the values
    whose median is to be computed.}
  \item{na.rm}{a logical value indicating whether \code{NA}
    values should be stripped before the computation proceeds.}
  }
\description{
  Compute the sample median of the vector of values given as its argument.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{\code{\link{quantile}} for general quantiles.}
\examples{
median(1:4)# = 2.5 [even number]
median(c(1:3,100,1000))# = 3 [odd, robust]
}
\keyword{univar}
\keyword{robust}
\eof
\name{medpolish}
\alias{medpolish}
%\alias{plot.medpolish}
%\alias{print.medpolish}
\title{Median Polish of a Matrix}
\description{
  Fits an additive model using Tukey's \emph{median polish} procedure.
}
\usage{
medpolish(x, eps = 0.01, maxiter = 10, trace.iter = TRUE,
          na.rm = FALSE)
}
\arguments{
  \item{x}{a numeric matrix.}
  \item{eps}{real number greater than 0. A tolerance for convergence:
    see \bold{Details}.}
  \item{maxiter}{the maximum number of iterations}
  \item{trace.iter}{logical. Should progress in convergence be reported?}
  \item{na.rm}{logical. Should missing values be removed?}
}
\details{
  The model fitted is additive (constant + rows + columns). The
  algorithm works by alternately removing the row and column medians,
  and continues until the proportional reduction in the sum
  of absolute residuals is less than \code{eps}
  or until there have been \code{maxiter} iterations.
  The sum of absolute residuals is printed at
  each iteration of the fitting process, if \code{trace.iter} is \code{TRUE}.
  If \code{na.rm} is \code{FALSE} the presence of any \code{NA} value in
  \code{x} will cause an error, otherwise \code{NA} values are ignored.

  \code{medpolish} returns an object of class \code{medpolish} (see below).
  There are printing and plotting methods for this
  class, which are invoked via by the generics
  \code{\link{print}} and \code{\link{plot}}.
}
\value{
  An object of class \code{medpolish} with the following named components:
  \item{overall}{the fitted constant term.}
  \item{row}{the fitted row effects.}
  \item{col}{the fitted column effects.}
  \item{residuals}{the residuals.}
  \item{name}{the name of the dataset.}
}
\seealso{\code{\link{median}}; \code{\link{aov}} for a \emph{mean}
  instead of \emph{median} decomposition.}
\references{
  Tukey, J. W. (1977).
  \emph{Exploratory Data Analysis},
  Reading Massachusetts: Addison-Wesley.
}
\examples{
## Deaths from sport parachuting;  from ABC of EDA, p.224:
deaths <-
    rbind(c(14,15,14),
          c( 7, 4, 7),
          c( 8, 2,10),
          c(15, 9,10),
          c( 0, 2, 0))
dimnames(deaths) <- list(c("1-24", "25-74", "75-199", "200++", "NA"),
                         paste(1973:1975))
deaths
(med.d <- medpolish(deaths))
plot(med.d)
## Check decomposition:
all(deaths == med.d$overall + outer(med.d$row,med.d$col, "+") + med.d$resid)
}
\keyword{robust}
\eof
\name{model.extract}
\title{Extract Components from a Model Frame}
\usage{
model.extract(frame, component)
model.offset(x)
model.response(data, type = "any")
model.weights(x)
}
\alias{model.extract}
\alias{model.offset}
\alias{model.response}
\alias{model.weights}
\arguments{
 \item{frame, x, data}{A model frame.}
 \item{component}{literal character string or name. The name of a
   component to extract, such as \code{"weights"}, \code{"subset"}.}
 \item{type}{One of \code{"any"}, \code{"numeric"}, \code{"double"}.
   Using the either of latter two coerces the result to have
   storage mode \code{"double"}.}
}
\description{Returns the response, offset, subset, weights or other
  special components of a model frame passed as optional arguments to
  \code{\link{model.frame}}.
}
\details{
  \code{model.extract} is provided for compatibility with S, which does
  not have the more specific functions.  It is also useful to extract
  e.g. the \code{etastart} and \code{mustart} components of a
  \code{\link{glm}} fit.

  \code{model.offset} and \code{model.response} are equivalent to
  \code{model.frame(, "offset")} and \code{model.frame(, "response")}
  respectively.

  \code{model.weights} is slightly different from
  \code{model.frame(, "weights")} in not naming the vector it returns.
}
\value{The specified component of the model frame, usually a vector.}

\seealso{\code{\link{model.frame}}, \code{\link{offset}}}

\examples{
data(esoph)
a <- model.frame(cbind(ncases,ncontrols) ~ agegp+tobgp+alcgp, data=esoph)
model.extract(a, "response")
stopifnot(model.extract(a, "response") == model.response(a))

a <- model.frame(ncases/(ncases+ncontrols) ~ agegp+tobgp+alcgp,
                 data = esoph, weights = ncases+ncontrols)
model.response(a)
model.extract(a, "weights")

a <- model.frame(cbind(ncases,ncontrols) ~ agegp,
                 something = tobgp, data = esoph)
names(a)
stopifnot(model.extract(a, "something") == esoph$tobgp)
}
\keyword{manip}
\keyword{programming}
\keyword{models}
\eof
\name{model.frame}
\alias{model.frame}
\alias{model.frame.default}
\alias{model.frame.lm}
\alias{model.frame.glm}
\alias{model.frame.aovlist}
\title{Extracting the ``Environment'' of a Model Formula}
\usage{
model.frame(formula, \dots)

\method{model.frame}{default}(formula, data = NULL,
           subset = NULL, na.action = na.fail,
           drop.unused.levels = FALSE, xlev = NULL, \dots)

\method{model.frame}{aovlist}(formula, data = NULL, \dots)

\method{model.frame}{glm}(formula, \dots)

\method{model.frame}{lm}(formula, \dots)
}
\arguments{
  \item{formula}{a model \code{\link{formula}} or \code{\link{terms}}
    object or an \R object.}

  \item{data}{\code{data.frame}, list, \code{environment} or object
    coercible to \code{data.frame} containing the variables in
    \code{formula}.  Neither a matrix nor an array will be accepted.}

  \item{subset}{a specification of the rows to be used: defaults to all
    rows. This can be any valid indexing vector (see
    \code{\link{[.data.frame}}) for the rows of \code{data} or if that is not
    supplied, a data frame made up of the variables used in \code{formula}.}

  \item{na.action}{how \code{NA}s are treated. The default is first,
    any \code{na.action} attribute of \code{data}, second
    a \code{na.action} setting of \code{\link{options}}, and third
    \code{\link{na.fail}} if that is unset.  The \dQuote{factory-fresh}
    default is \code{\link{na.omit}}.}

  \item{drop.unused.levels}{should factors have unused levels dropped?
    Defaults to \code{FALSE}.}

  \item{xlev}{a named list of character vectors giving the full set of levels
    to be assumed for each factor.}

  \item{\dots}{further arguments such as \code{data}, \code{na.action},
    \code{subset}.  Any additional arguments such as \code{offset} and
    \code{weights} which reach the default method are used to create
    further columns in the model frame, with parenthesised names such as
    \code{"(offset)"}.}
}
\description{
  \code{model.frame} (a generic function) and its methods return a
  \code{\link{data.frame}} with the variables needed to use
  \code{formula} and any \code{\dots} arguments.
}
\details{
  Exactly what happens depends on the class and attributes of the object
  \code{formula}.  If this is an object of fitted-model class such as
  \code{"lm"}, the method will either returned the saved model frame
  used when fitting the model (if any, often selected by argument
  \code{model = TRUE}) or pass the call used when fitting on to the
  default method.  The default method itself can cope with rather
  standard model objects such as those of classes
  \code{"\link[MASS]{lqs}"} and \code{"\link[ppr]{ppr}"} if no other
  arguments are supplied.
  
  The rest of this section applies only to the default method.

  If either \code{formula} or \code{data} is already a model frame (a
  data frame with a \code{"terms"} attribute and the other is missing,
  the model frame is returned.  Unless \code{formula} is a terms object,
  \code{terms} is called on it.  (If you wish to use the
  \code{keep.order} argument of \code{terms.formula}, pass a terms
  object rather than a formula.)

  All the variables in \code{formula}, \code{subset} and in \code{\dots}
  are looked for first in \code{data} and then in the environment of
  \code{formula} (see the help for \code{\link{formula}()} for further
  details) and collected into a data frame. Then the \code{subset}
  expression is evaluated, and it is is used as a row index to the data
  frame. Then the \code{na.action} function is applied to the data frame
  (and may well add attributes).  The levels of any factors in the data
  frame are adjusted according to the \code{drop.unused.levels} and
  \code{xlev} arguments.
}
\value{
  A \code{\link{data.frame}} containing the variables used in
  \code{formula} plus those specified \code{\dots}.
}
\seealso{\code{\link{model.matrix}} for the \dQuote{design matrix},
  \code{\link{formula}} for formulas  and
  \code{\link{expand.model.frame}} for model.frame manipulation.
}
\references{
  Chambers, J. M. (1992)
  \emph{Data for models.}
  Chapter 3 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\examples{
data(cars)
data.class(model.frame(dist ~ speed, data = cars))
}
\keyword{models}
\eof
\name{model.matrix}
\alias{model.matrix}
\alias{model.matrix.default}
\alias{model.matrix.lm}
%\alias{model.matrix.lm.null}
%\alias{model.matrix.glm.null}
\title{Construct Design Matrices}
\usage{
model.matrix(object, \dots)

\method{model.matrix}{default}(object, data = environment(object),
             contrasts.arg = NULL, xlev = NULL, \dots)
}
\arguments{
  \item{object}{an object of an appropriate class. For the default
    method, a model formula or terms object.}
  \item{data}{a data frame created with \code{\link{model.frame}}.}
  \item{contrasts.arg}{A list, whose entries are contrasts suitable for
    input to the \code{\link{contrasts}} replacement function and
    whose names are the names of columns of \code{data} containing
    \code{\link{factor}}s.}
  \item{xlev}{to be used as argument of \code{\link{model.frame}} if
    \code{data} has no \code{"terms"} attribute.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  \code{model.matrix} creates a design matrix.
}
\details{
  \code{model.matrix} creates a design matrix from the description given
  in \code{terms(formula)}, using the data in \code{data} which must
  contain columns with the same names as would be created by a call to
  \code{model.frame(formula)} or, more precisely, by evaluating
  \code{attr(terms(formula), "variables")}.  There may be other columns
  and the order is not important.  If \code{contrasts} is specified it
  overrides the default factor coding for that variable.

  In interactions, the variable whose levels vary fastest is the first
  one to appear in the formula (and not in the term), so in \code{~ a +
    b + b:a} the interaction will have \code{a} varying fastest.

  By convention, if the response variable also appears on the
  right-hand side of the formula it is dropped (with a warning),
  although interactions involving the term are retained.
}
\value{
  The design matrix for a regression model with the specified formula
  and data.
}
\references{
  Chambers, J. M. (1992)
  \emph{Data for models.}
  Chapter 3 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{model.frame}}, \code{\link{model.extract}},
  \code{\link{terms}}
}
\examples{
data(trees)
ff <- log(Volume) ~ log(Height) + log(Girth)
str(m <- model.frame(ff, trees))
mat <- model.matrix(ff, m)

dd <- data.frame(a = gl(3,4), b = gl(4,1,12))# balanced 2-way
options("contrasts")
model.matrix(~ a + b, dd)
model.matrix(~ a + b, dd, contrasts = list(a="contr.sum"))
model.matrix(~ a + b, dd, contrasts = list(a="contr.sum", b="contr.poly"))
m.orth <- model.matrix(~a+b, dd, contrasts = list(a="contr.helmert"))
crossprod(m.orth)# m.orth is  ALMOST  orthogonal
}
\keyword{models}
\eof
\name{model.tables}
\alias{model.tables}
\alias{model.tables.aov}
\alias{model.tables.aovlist}
%\alias{print.tables.aov}
\title{Compute Tables of Results from an Aov Model Fit}
\description{
  Computes summary tables for model fits, especially complex \code{aov}
  fits.
}
\usage{
model.tables(x, \dots)

\method{model.tables}{aov}(x, type = "effects", se = FALSE, cterms, \dots)

\method{model.tables}{aovlist}(x, type = "effects", se = FALSE, \dots)
}
\arguments{
  \item{x}{a model object, usually produced by \code{aov}}
  \item{type}{type of table: currently only \code{"effects"} and
    \code{"means"} are implemented.}
  \item{se}{should standard errors be computed?}
  \item{cterms}{A character vector giving the names of the terms for
    which tables should be computed. The default is all tables.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\details{
  For \code{type = "effects"} give tables of the coefficients for each
  term, optionally with standard errors.

  For \code{type = "means"} give tables of the mean response for each
  combinations of levels of the factors in a term.
}
\value{
  An object of class \code{"tables.aov"}, as list which may contain components
  \item{tables}{A list of tables for each requested term.}
  \item{n}{The replication information for each term.}
  \item{se}{Standard error information.}
}
\section{Warning}{
  The implementation is incomplete, and only the simpler cases have been
  tested thoroughly.

  Weighted \code{aov} fits are not supported.
}

\seealso{
  \code{\link{aov}}, \code{\link{proj}},
  \code{\link{replications}}, \code{\link{TukeyHSD}},
  \code{\link{se.contrast}}
}

\examples{
## From Venables and Ripley (2002) p.165.
N <- c(0,1,0,1,1,1,0,0,0,1,1,0,1,1,0,0,1,0,1,0,1,1,0,0)
P <- c(1,1,0,0,0,1,0,1,1,1,0,0,0,1,0,1,1,0,0,1,0,1,1,0)
K <- c(1,0,0,1,0,1,1,0,0,1,0,1,0,1,1,0,0,0,1,1,1,0,1,0)
yield <- c(49.5,62.8,46.8,57.0,59.8,58.5,55.5,56.0,62.8,55.8,69.5,
55.0, 62.0,48.8,45.5,44.2,52.0,51.5,49.8,48.8,57.2,59.0,53.2,56.0)

npk <- data.frame(block=gl(6,4), N=factor(N), P=factor(P),
                  K=factor(K), yield=yield)
npk.aov <- aov(yield ~ block + N*P*K, npk)
model.tables(npk.aov, "means", se=TRUE)

## as a test, not particularly sensible statistically
options(contrasts=c("contr.helmert", "contr.treatment"))
npk.aovE <- aov(yield ~  N*P*K + Error(block), npk)
model.tables(npk.aovE,  se=TRUE)
model.tables(npk.aovE, "means")
}
\keyword{models}
\eof
\name{monthplot}
\alias{monthplot}
\alias{monthplot.default}
\alias{monthplot.ts}
\alias{monthplot.stl}
\alias{monthplot.StructTS}

\title{
  Plot a Seasonal or other Subseries
}
\description{
  These functions plot seasonal (or other) subseries of a time series.
  For each season (or other category), a time series is plotted.
}
\synopsis{
monthplot(x, \dots)
monthplot.StructTS(x, labels = NULL, ylab = choice, choice = "sea", \dots)
monthplot.stl(x, labels = NULL, ylab = choice, choice = "seasonal", \dots)
monthplot.ts(x, labels = NULL, times = time(x), phase = cycle(x),
             ylab = deparse(substitute(x)), \dots)
monthplot.default(x, labels = 1:12,
              times = 1:length(x),
              phase = (times - 1)\%\%length(labels) + 1,
              base = mean,
              xlim = c(0.55, f + 0.45), ylim = range(x, na.rm = TRUE),
              axes = TRUE, xlab = "", ylab = deparse(substitute(x)),
              type = "l", box = TRUE, add = FALSE, \dots)
}
\usage{
monthplot(x, labels = NULL, times, phase, base, choice, \ldots)
}
\arguments{
  \item{x}{Time series or related object.}
  \item{labels}{Labels to use for each \dQuote{season}.}
  \item{times}{Time of each observation.}
  \item{phase}{Indicator for each \dQuote{season}.}
  \item{base}{Function to use for reference line for subseries.}
  \item{choice}{Which series of an \code{stl} or \code{StructTS} object?}
  \item{\dots}{Graphical parameters.}
}
\details{
  These functions extract subseries from a time series and plot them
  all in one frame.  The \code{\link{ts}}, \code{\link{stl}}, and
  \code{\link{StructTS}} methods use the internally recorded frequency and
  start and finish times to set the scale and the seasons.  The default
  method assumes observations come in groups of 12 (though this can be
  changed).

  If the \code{labels} are not given but the \code{phase} is given, then
  the \code{labels} default to the unique values of the \code{phase}.  If
  both are given, then the \code{phase} values are assumed to be indices
  into the \code{labels} array, i.e., they should be in the range
  from 1 to \code{length(labels)}.
}
\value{
  These functions are executed for their side effect of
  drawing a seasonal subseries plot on the current graphical
  window.
}
\author{Duncan Murdoch}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{ts}}, \code{\link{stl}}, \code{\link{StructTS}}
}

\examples{
## The CO2 data
data(co2)
fit <- stl(log(co2), s.window = 20, t.window = 20)
plot(fit)
op <- par(mfrow = c(2,2))
monthplot(co2, ylab = "data", cex.axis = 0.8)
monthplot(fit, choice = "seasonal", cex.axis = 0.8)
monthplot(fit, choice = "trend", cex.axis = 0.8)
monthplot(fit, choice = "remainder", type = "h", cex.axis = 0.8)
par(op)

## The CO2 data, grouped quarterly
quarter <- (cycle(co2) - 1) \%/\% 3
monthplot(co2, phase = quarter)

## see also JohnsonJohnson
}
\keyword{ hplot }
\keyword{ ts }
\eof
\name{mood.test}
\alias{mood.test}
\alias{mood.test.default}
\alias{mood.test.formula}
\title{Mood Two-Sample Test of Scale}
\description{
  Performs Mood's two-sample test for a difference in scale parameters.
}
\usage{
mood.test(x, \dots)

\method{mood.test}{default}(x, y, alternative = c("two.sided", "less", "greater"), \dots)

\method{mood.test}{formula}(formula, data, subset, na.action, \dots)
}
\arguments{
  \item{x, y}{numeric vectors of data values.}
  \item{alternative}{indicates the alternative hypothesis and must be
    one of \code{"two.sided"} (default), \code{"greater"} or
    \code{"less"} all of which can be abbreviated.}
  \item{formula}{a formula of the form \code{lhs ~ rhs} where \code{lhs}
    is a numeric variable giving the data values and \code{rhs} a factor
    with two levels giving the corresponding groups.}
  \item{data}{an optional data frame containing the variables in the
    model formula.}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.  Defaults to
    \code{getOption("na.action")}.}
  \item{\dots}{further arguments to be passed to or from methods.}  
}
\details{
  The underlying model is that the two samples are drawn from
  \eqn{f(x-l)} and \eqn{f((x-l)/s)/s}, respectively, where \eqn{l} is a
  common location parameter and \eqn{s} is a scale parameter.

  The null hypothesis is \eqn{s = 1}.

  There are more useful tests for this problem.
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{alternative}{a character string describing the alternative
    hypothesis.}
  \item{method}{the character string \code{"Mood two-sample test of scale"}.}
  \item{data.name}{a character string giving the names of the data.}
}
\references{
  William J. Conover (1971),
  \emph{Practical nonparametric statistics}.
  New York: John Wiley & Sons.
  Pages 234f.
}
\seealso{
  \code{\link{fligner.test}} for a rank-based (nonparametric) k-sample
  test for homogeneity of variances;
  \code{\link{ansari.test}} for another rank-based two-sample test for a
  difference in scale parameters;
  \code{\link{var.test}} and \code{\link{bartlett.test}} for parametric
  tests for the homogeneity in variance.
}
\examples{
## Same data as for the Ansari-Bradley test:
## Serum iron determination using Hyland control sera
ramsay <- c(111, 107, 100, 99, 102, 106, 109, 108, 104, 99,
            101, 96, 97, 102, 107, 113, 116, 113, 110, 98)
jung.parekh <- c(107, 108, 106, 98, 105, 103, 110, 105, 104,
            100, 96, 108, 103, 104, 114, 114, 113, 108, 106, 99)
mood.test(ramsay, jung.parekh)
## Compare this to ansari.test(ramsay, jung.parekh)
}
\keyword{htest}
\eof
\name{na.action}
\title{NA Action}
\usage{
na.action(object, \dots)
%na.action.default(object, \dots)
}
\alias{na.action}
\alias{na.action.default}
\arguments{
  \item{object}{any object whose \code{\link{NA}} action is given.}
  \item{\dots}{further arguments special methods could require.}
}
\description{
  \code{na.action} is a generic function, and \code{na.action.default} its
  default method.
}
\value{
  The \dQuote{NA action} which should be applied to \code{object} whenever
  \code{\link{NA}}s are not desired.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical Models in S.}
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{options}("na.action")}, \code{\link{na.omit}},
  \code{\link{na.fail}}
}
\examples{
na.action(c(1, NA))
}
\keyword{NA}
\keyword{methods}
\eof
\name{na.contiguous}
\alias{na.contiguous}
\title{Find Longest Contiguous Stretch of non-NAs}
\description{
  Find the longest consecutive stretch of non-missing values in a time
  series object.  (In the event of a tie, the first such stretch.)
}
\usage{
na.contiguous(frame)
}
\arguments{
  \item{frame}{a univariate or multivariate time series.}
}
\value{
  A time series without missing values.  The class of \code{frame} will
  be preserved.
}
\seealso{
  \code{\link{na.omit}} and \code{\link{na.omit.ts}};
  \code{\link{na.fail}}
}
\examples{
data(presidents)
na.contiguous(presidents)
}
\keyword{ts}
\eof
\name{na.fail}
\alias{na.fail}
\alias{na.fail.default}
\alias{na.omit}
\alias{na.omit.data.frame}
\alias{na.omit.default}
\alias{na.exclude}
\alias{na.exclude.data.frame}
\alias{na.exclude.default}
\alias{na.pass}
\title{Handle Missing Values in Objects}
\usage{
na.fail(object, \dots)
na.omit(object, \dots)
na.exclude(object, \dots)
na.pass(object, \dots)
}
\arguments{
  \item{object}{an \R object, typically a data frame}
  \item{\dots}{further arguments special methods could require.}
}
\description{
  These generic functions are useful for dealing with \code{\link{NA}}s
  in e.g., data frames.
  \code{na.fail} returns the object if it does not contain any
  missing values, and signals an error otherwise.
  \code{na.omit} returns the object with incomplete cases removed.
  \code{na.pass} returns the object unchanged.
}
\details{
  At present these will handle vectors, matrices and data frames
  comprising vectors and matrices (only).

  If \code{na.omit} removes cases, the row numbers of the cases form the
  \code{"na.action"} attribute of the result, of class \code{"omit"}.

  \code{na.exclude} differs from \code{na.omit} only in the class of the
  \code{"na.action"} attribute of the result, which is
  \code{"exclude"}.  This gives different behaviour in functions making
  use of \code{\link{naresid}} and \code{\link{napredict}}: when
  \code{na.exclude} is used the residuals and predictions are padded to
  the correct length by inserting \code{NA}s for cases omitted by
  \code{na.exclude}.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical Models in S.}
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{na.action}};
  \code{\link{options}} with argument \code{na.action} for setting
  \dQuote{NA actions};
  and \code{\link{lm}} and \code{\link{glm}} for functions using these.
}
\examples{
DF <- data.frame(x = c(1, 2, 3), y = c(0, 10, NA))
na.omit(DF)
m <- as.matrix(DF)
na.omit(m)
stopifnot(all(na.omit(1:3) == 1:3))  # does not affect objects with no NA's
try(na.fail(DF))#> Error: missing values in ...

options("na.action")
}
\keyword{NA}
\eof
\name{naresid}
\alias{naresid}
\alias{naresid.default}
\alias{naresid.exclude}
\alias{napredict}
\alias{napredict.default}
\alias{napredict.exclude}
\title{
Adjust for Missing Values
}
\description{
  Use missing value information to adjust residuals and predictions.
}
\usage{
naresid(omit, x, \dots)
napredict(omit, x, \dots)
}
\arguments{
  \item{omit}{an object produced by an \code{\link{na.action}} function,
    typically the \code{"na.action"} attribute of the result of
    \code{\link{na.omit}} or \code{\link{na.exclude}}.}
  \item{x}{a vector, data frame, or matrix to be adjusted based upon the
    missing value information.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\value{
  These return a similar object to \code{x}.
}
\details{
  These are utility functions used to allow \code{\link{predict}} and
  \code{\link{resid}} methods for modelling functions to compensate for
  the removal of \code{NA}s in the fitting process.  They are used by
  the default, \code{"lm"} and \code{"glm"} methods, and by further
  methods in packages \pkg{MASS}, \pkg{rpart} and \pkg{survival}.

  The default methods do nothing.  The default method for the \code{na.exclude}
  action is to pad the object with \code{NA}s in the correct positions to
  have the same number of rows as the original data frame.

  Currently \code{naresid} and \code{napredict} are identical, but
  future methods need not be.  \code{naresid} is used for residuals, and
  \code{napredict} for fitted values and predictions.
}
\note{
  Packages \code{rpart} and \code{survival5} used to contain versions of
  these functions that had an \code{na.omit} action equivalent to that
  now used for \code{na.exclude}.
}
\keyword{NA}
\keyword{models}
\eof
\name{naprint}
\alias{naprint}
\alias{naprint.default}
\alias{naprint.exclude}
\alias{naprint.omit}
\title{
Adjust for Missing Values
}
\description{
  Use missing value information to report the effects of an \code{na.action}.
}
\usage{
naprint(x, \dots)
}
\arguments{
  \item{x}{
    An object produced by an \code{na.action} function.
  }
  \item{\dots}{further arguments passed to or from other methods.}
}
\details{
  This is a generic function, and the exact information differs by
  method. \code{naprint.omit} reports the number of rows omitted:
  \code{naprint.default} reports an empty string.
}
\value{
  A character string providing information on missing values, for
  example the number.
}
\keyword{NA}
\keyword{models}
\eof
\name{nhtemp}
\docType{data}
\alias{nhtemp}
\title{Average Yearly Temperatures in New Haven}
\description{
  The mean annual temperature in degrees Fahrenheit in New Haven,
  Connecticut, from 1912 to 1971.
}
\usage{data(nhtemp)}
\format{A time series of 60 observations.}
\source{
  Vaux, J. E. and Brinker, N. B. (1972)
  \emph{Cycles}, \bold{1972}, 117--121.
}
\references{
  McNeil, D. R. (1977)
  \emph{Interactive Data Analysis}.
  New York: Wiley.
}
\examples{
data(nhtemp)
plot(nhtemp, main = "nhtemp data",
  ylab = "Mean annual temperature in New Haven, CT (deg. F)")
}
\keyword{datasets}
\eof
\name{nlm}
\alias{nlm}
\title{Non-Linear Minimization}
\concept{optimization}
\usage{
nlm(f, p, hessian = FALSE, typsize=rep(1, length(p)), fscale=1,
    print.level = 0, ndigit=12, gradtol = 1e-6,
    stepmax = max(1000 * sqrt(sum((p/typsize)^2)), 1000),
    steptol = 1e-6, iterlim = 100, check.analyticals = TRUE, \dots)
}
\description{
  This function carries out a minimization of the function \code{f}
  using a Newton-type algorithm.  See the references for details.
}
\arguments{
  \item{f}{the function to be minimized.  If the function value has
    an attribute called \code{gradient} or both \code{gradient} and
    \code{hessian} attributes, these will be used in the calculation of
    updated parameter values.  Otherwise, numerical derivatives are
    used. \code{\link{deriv}} returns a function with suitable
    \code{gradient} attribute.  This should be a function of a vector of
    the length of \code{p} followed by any other arguments specified
    by the \code{\dots} argument.}
  \item{p}{starting parameter values for the minimization.}
  \item{hessian}{if \code{TRUE}, the hessian of \code{f}
    at the minimum is returned.}
  \item{typsize}{an estimate of the size of each parameter
    at the minimum.}
  \item{fscale}{an estimate of the size of \code{f} at the minimum.}
  \item{print.level}{this argument determines the level of printing
    which is done during the minimization process.  The default
    value of \code{0} means that no printing occurs, a value of \code{1}
    means that initial and final details are printed and a value
    of 2 means that full tracing information is printed.}
  \item{ndigit}{the number of significant digits in the function \code{f}.}
  \item{gradtol}{a positive scalar giving the tolerance at which the
    scaled gradient is considered close enough to zero to
    terminate the algorithm.  The scaled gradient is a
    measure of the relative change in \code{f} in each direction
    \code{p[i]} divided by the relative change in \code{p[i]}.}
  \item{stepmax}{a positive scalar which gives the maximum allowable
    scaled step length.  \code{stepmax} is used to prevent steps which
    would cause the optimization function to overflow, to prevent the
    algorithm from leaving the area of interest in parameter space, or to
    detect divergence in the algorithm. \code{stepmax} would be chosen
    small enough to prevent the first two of these occurrences, but should
    be larger than any anticipated reasonable step.}
  \item{steptol}{A positive scalar providing the minimum allowable
    relative step length.}
  \item{iterlim}{a positive integer specifying the maximum number of
    iterations to be performed before the program is terminated.}
  \item{check.analyticals}{a logical scalar specifying whether the
    analytic gradients and Hessians, if they are supplied, should be
    checked against numerical derivatives at the initial parameter
    values. This can help detect incorrectly formulated gradients or
    Hessians.}
  \item{\dots}{additional arguments to \code{f}.}
}
\details{
  If a gradient or hessian is supplied but evaluates to the wrong mode
  or length, it will be ignored if \code{check.analyticals = TRUE} (the
  default) with a warning.  The hessian is not even checked unless the
  gradient is present and passes the sanity checks.

  From the three methods available in the original source, we always use
  method \dQuote{1} which is line search.
}
\value{
  A list containing the following components:
  \item{minimum}{the value of the estimated minimum of \code{f}.}
  \item{estimate}{the point at which the minimum value of
    \code{f} is obtained.}
  \item{gradient}{the gradient at the estimated minimum of \code{f}.}
  \item{hessian}{the hessian at the estimated minimum of \code{f} (if
    requested).}
  \item{code}{an integer indicating why the optimization process terminated.
    \describe{
      \item{1:}{relative gradient is close to zero, current iterate is
	probably solution.}
      \item{2:}{successive iterates within tolerance, current iterate
	is probably solution.}
      \item{3:}{last global step failed to locate a point lower than
	\code{estimate}.  Either \code{estimate} is an approximate local
	minimum of the function or \code{steptol} is too small.}
      \item{4:}{iteration limit exceeded.}
      \item{5:}{maximum step size \code{stepmax} exceeded five consecutive
	times.  Either the function is unbounded below,
	becomes asymptotic to a finite value from above in
	some direction or \code{stepmax} is too small.}
    }
  }
  \item{iterations}{the number of iterations performed.}
}
\references{
  Dennis, J. E. and Schnabel, R. B. (1983) \emph{Numerical Methods for
    Unconstrained Optimization and Nonlinear Equations.} Prentice-Hall,
  Englewood Cliffs, NJ.

  Schnabel, R. B., Koontz, J. E. and Weiss, B. E. (1985) A modular
  system of algorithms for unconstrained minimization.
  \emph{ACM Trans. Math. Software}, \bold{11}, 419--440.
}
\seealso{
  \code{\link{optim}}.  \code{\link{optimize}} for one-dimensional
  minimization and \code{\link{uniroot}} for root finding.
  \code{\link{deriv}} to calculate analytical derivatives.

  For nonlinear regression, \code{\link{nls}} may be better.
}
\examples{
f <- function(x) sum((x-1:length(x))^2)
nlm(f, c(10,10))
nlm(f, c(10,10), print.level = 2)
str(nlm(f, c(5), hessian = TRUE))

f <- function(x, a) sum((x-a)^2)
nlm(f, c(10,10), a=c(3,5))
f <- function(x, a)
{
    res <- sum((x-a)^2)
    attr(res, "gradient") <- 2*(x-a)
    res
}
nlm(f, c(10,10), a=c(3,5))

## more examples, including the use of derivatives.
\dontrun{demo(nlm)}
}
\keyword{nonlinear}
\keyword{optimize}
\eof
\name{nls}
\alias{nls}
%\alias{anova.nls}
%\alias{anovalist.nls}
%\alias{coef.nls}
%\alias{deviance.nls}
%\alias{df.residual.nls}
%\alias{fitted.nls}
%\alias{logLik.nls}
%\alias{print.nls}
%\alias{residuals.nls}
%\alias{summary.nls}
%\alias{print.summary.nls}
%\alias{weights.nls}
%\alias{vcov.nls}
\title{Nonlinear Least Squares}
\description{
  Determine the nonlinear least-squares estimates of the nonlinear model
  parameters and return a class \code{nls} object.
}
\usage{
nls(formula, data = parent.frame(), start, control = nls.control(),
    algorithm = "default", trace = FALSE, subset,
    weights, na.action, model = FALSE)
}
\arguments{
 \item{formula}{a nonlinear model formula including variables and parameters}
 \item{data}{an optional data frame in which to evaluate the variables in
     \code{formula}}
 \item{start}{a named list or named numeric vector of starting estimates}
 \item{control}{an optional list of control settings.  See
     \code{nls.control} for the names of the settable control values and
     their effect.}
 \item{algorithm}{character string specifying the algorithm to use.
     The default algorithm is a Gauss-Newton algorithm. The other 
     alternative is "plinear", the Golub-Pereyra algorithm for
     partially linear least-squares models.}
 \item{trace}{logical value indicating if a trace of the iteration
     progress should be printed.  Default is \code{FALSE}.  If
     \code{TRUE} the residual sum-of-squares and the parameter values
     are printed at the conclusion of each iteration.  When the
     \code{"plinear"} algorithm is used, the conditional estimates of
     the linear parameters are printed after the nonlinear parameters.}
 \item{subset}{an optional vector specifying a subset of observations
   to be used in the fitting process.}
 \item{weights}{an optional numeric vector of (fixed) weights.  When
   present, the objective function is weighted least squares. \emph{not
     yet implemented}}
\item{na.action}{a function which indicates what should happen
    when the data contain \code{NA}s.}
\item{model}{logical.  If true, the model frame is returned as part of
  the object.}
}
\details{
  \bold{Do not use \code{nls} on artificial "zero-residual" data.}

  The \code{nls} function uses a relative-offset convergence criterion
  that compares the numerical imprecision at the current parameter
  estimates to the residual sum-of-squares.  This performs well on data of
  the form \deqn{y=f(x,\theta)+\epsilon}{y = f(x, theta) + eps} (with
  \code{var(eps) > 0}).  It
  fails to indicate convergence on data of the form
  \deqn{y=f(x,\theta)}{y = f(x, theta)} because the criterion amounts to
  comparing two components of the round-off error.  If you wish to test
  \code{nls} on artificial data please add a noise component, as shown
  in the example below.
  
  An \code{nls} object is a type of fitted model object.  It has methods
  for the generic functions \code{\link{coef}}, \code{\link{formula}},
  \code{\link{resid}}, \code{\link{print}}, \code{\link{summary}},
  \code{\link{AIC}}, \code{\link{fitted}} and \code{\link{vcov}}.

  Variables in \code{formula} are looked for first in \code{data}, then
  the environment of \code{formula} (added in 1.9.1) and finally along
  the search path.  Functions in \code{formula} are searched for first
  in the environment of \code{formula} (added in 1.9.1) and then along
  the search path.
}
\value{
  A list of
  \item{m}{an \code{nlsModel} object incorporating the model}
  \item{data}{the expression that was passed to \code{nls} as the data
    argument.  The actual data values are present in the environment of
    the \code{m} component.}
}
\references{
  Bates, D.M. and Watts, D.G. (1988)
  \emph{Nonlinear Regression Analysis and Its Applications},
  Wiley

  Bates, D. M. and Chambers, J. M. (1992)
  \emph{Nonlinear models.}
  Chapter 10 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\author{Douglas M. Bates and Saikat DebRoy}
\seealso{\code{\link{nlsModel}}}
\examples{
data( DNase )
DNase1 <- DNase[ DNase$Run == 1, ]
## using a selfStart model
fm1DNase1 <- nls( density ~ SSlogis( log(conc), Asym, xmid, scal ), DNase1 )
summary( fm1DNase1 )
## using conditional linearity
fm2DNase1 <- nls( density ~ 1/(1 + exp(( xmid - log(conc) )/scal ) ),
                  data = DNase1,
                  start = list( xmid = 0, scal = 1 ),
                  alg = "plinear", trace = TRUE )
summary( fm2DNase1 )
## without conditional linearity
fm3DNase1 <- nls( density ~ Asym/(1 + exp(( xmid - log(conc) )/scal ) ),
                  data = DNase1,
                  start = list( Asym = 3, xmid = 0, scal = 1 ),
                  trace = TRUE )
summary( fm3DNase1 )
## weighted nonlinear regression
data(Puromycin)
Treated <- Puromycin[Puromycin$state == "treated", ]
weighted.MM <- function(resp, conc, Vm, K)
{
    ## Purpose: exactly as white book p.451 -- RHS for nls()
    ##  Weighted version of Michaelis-Menten model
    ## ------------------------------------------------------------
    ## Arguments: 'y', 'x' and the two parameters (see book)
    ## ------------------------------------------------------------
    ## Author: Martin Maechler, Date: 23 Mar 2001, 18:48

    pred <- (Vm * conc)/(K + conc)
    (resp - pred) / sqrt(pred)
}

Pur.wt <- nls( ~ weighted.MM(rate, conc, Vm, K), data = Treated,
              start = list(Vm = 200, K = 0.1),
              trace = TRUE)

## The two examples below show that you can fit a model to
## artificial data with noise but not to artificial data
## without noise.
x = 1:10
y = x                                  # perfect fit
yeps = y + rnorm(length(y), sd = 0.01) # added noise
nls(yeps ~ a + b*x, start = list(a = 0.12345, b = 0.54321),
     trace = TRUE)
\dontrun{
nls(y ~ a + b*x, start = list(a = 0.12345, b = 0.54321),
     trace = TRUE)
}       
}
\keyword{nonlinear}
\keyword{regression}
\keyword{models}
\eof
\name{nls.control}
\alias{nls.control}
\title{Control the Iterations in nls}
\description{
  Allow the user to set some characteristics of the \code{nls}
  nonlinear least squares algorithm.
}
\usage{
nls.control(maxiter=50, tol=1e-05, minFactor=1/1024)
}
\arguments{
  \item{maxiter}{A positive integer specifying the maximum number of
    iterations allowed.}
  \item{tol}{A positive numeric value specifying the tolerance level for
    the relative offset convergence criterion.}
  \item{minFactor}{A positive numeric value specifying the minimum
    step-size factor allowed on any step in the iteration.  The
    increment is calculated with a Gauss-Newton algorithm and
    successively halved until the residual sum of squares has been
    decreased or until the step-size factor has been reduced below this
    limit.}
}
\value{
  A \code{list} with exactly three components:
  \item{maxiter}{}
  \item{tol}{}
  \item{minFactor}{}
  with meanings as explained under \sQuote{Arguments}.
}
\references{
  Bates and Watts (1988),
  \emph{Nonlinear Regression Analysis and Its Applications}, Wiley.
}
\author{Douglas Bates and Saikat DebRoy}
\seealso{\code{\link{nls}}}
\examples{
nls.control(minFactor = 1/2048)
}
\keyword{nonlinear}
\keyword{regression}
\keyword{models}
\eof
\name{nlsModel}
\alias{nlsModel}
\alias{nlsModel.plinear}
\title{Create an nlsModel Object}
\usage{
nlsModel(form, data, start)
}
\arguments{
 \item{form}{a nonlinear model formula}
 \item{data}{a data frame or a list in which to evaluate the variables
     from the model formula}
 \item{start}{a named list or named numeric vector of starting
   estimates for the parameters in the model}
}
\description{
  This is the constructor for \code{nlsModel} objects, which are
  function closures for several functions in a list.  The closure
  includes a nonlinear model formula, data values for the formula, as
  well as parameters and their values.
}
\details{
  An \code{nlsModel} object is primarily used within the \code{nls}
  function.  It encapsulates the model, the data, and the parameters in
  an environment and provides several methods to access characteristics
  of the model.  It forms an important component of the object returned
  by the \code{nls} function.

  See \code{\link{nls}} for where elements of the formula \code{form}
  are looked for.  In normal use all the variables will be in \code{data}.
}
\value{
    The value is a list of functions that share a common environment.
    \item{resid}{returns the residual vector evaluated at the current
	parameter values}
    \item{fitted}{returns the fitted responses and their gradient at the
	current parameter values}
    \item{formula}{returns the model formula}
    \item{deviance}{returns the residual sum-of-squares at the current
	parameter values}
    \item{gradient}{returns the gradient of the model function at the
	current parameter values}
    \item{conv}{returns the relative-offset convergence criterion
	evaluated at the current parmeter values}
    \item{incr}{returns the parameter increment calculated according to
	the Gauss-Newton formula}
    \item{setPars}{a function with one argument, \code{pars}.  It sets the
	parameter values for the \code{nlsModel} object and returns a
	logical value denoting a singular gradient array.}
    \item{getPars}{returns the current value of the model parameters as a
	numeric vector}
    \item{getAllPars}{returns the current value of the model parameters as a
	numeric vector}
    \item{getEnv}{returns the environment shared by these functions,
      which contains copies of all the variables in \code{data} and has
      as parent the environment of \code{form}.}
    \item{trace}{the function that is called at each iteration if
	tracing is enabled}
    \item{Rmat}{the upper triangular factor of the gradient array at the
	current parameter values}
    \item{predict}{takes as argument \code{newdata},a \code{data.frame} and
	returns the predicted response for \code{newdata}.}
}
\references{
  Bates, D.M. and Watts, D.G. (1988), \emph{Nonlinear Regression Analysis
    and Its Applications}, Wiley
}
\author{Douglas M. Bates and Saikat DebRoy}
\seealso{
  \code{\link{nls}}
}
\examples{
data( DNase )
DNase1 <- DNase[ DNase$Run == 1, ]
mod <-
 nlsModel(density ~ SSlogis( log(conc), Asym, xmid, scal ),
          DNase1, list( Asym = 3, xmid = 0, scal = 1 ))
mod$getPars()     # returns the parameters as a list
mod$deviance()    # returns the residual sum-of-squares
mod$resid()       # returns the residual vector and the gradient
mod$incr()        # returns the suggested increment
mod$setPars( unlist(mod$getPars()) + mod$incr() )  # set new parameter values
mod$getPars()     # check the parameters have changed
mod$deviance()    # see if the parameter increment was successful
mod$trace()       # check the tracing
mod$Rmat()        # R matrix from the QR decomposition of the gradient
}
\keyword{nonlinear}
\keyword{models}
\eof
\name{nottem}
\docType{data}
\alias{nottem}
\title{
  Average Monthly Temperatures at Nottingham, 1920--1939
}
\description{
  A time series object containing average air temperatures at
  Nottingham Castle in degrees Fahrenheit for 20 years.
}
\usage{data(nottem)}
\source{
  Anderson, O. D. (1976)
  \emph{Time Series Analysis and Forecasting: The Box-Jenkins approach.}
  Butterworths. Series R.
}
\examples{
\dontrun{data(nottem)
nott <- window(nottem, end=c(1936,12))
fit <- arima(nott,order=c(1,0,0), list(order=c(2,1,0), period=12))
nott.fore <- predict(fit, n.ahead=36)
ts.plot(nott, nott.fore$pred, nott.fore$pred+2*nott.fore$se,
        nott.fore$pred-2*nott.fore$se, gpars=list(col=c(1,1,4,4)))
}}
\keyword{datasets}
\eof
\name{numericDeriv}
\alias{numericDeriv}
\title{Evaluate derivatives numerically}
\description{
    \code{numericDeriv} numerically evaluates the gradient of an expression.
}
\usage{
numericDeriv(expr, theta, rho=parent.frame())
}
\arguments{
  \item{expr}{The expression to be differentiated.  The value of this
      expression should be a numeric vector.}
  \item{theta}{A character vector of names of variables used in \code{expr}}
  \item{rho}{An environment containing all the variables needed to
      evaluate \code{expr}}
}
\details{
    This is a front end to the C function \code{numeric_deriv}, which is
    described in \emph{Writing R Extensions}.
}
\value{
    The value of \code{eval(expr, env = rho)} plus a matrix
    attribute called \code{gradient}.  The columns of this matrix are
    the derivatives of the value with respect to the variables listed in
    \code{theta}.
}
\author{Saikat DebRoy \email{saikat@stat.wisc.edu}}
\examples{
myenv <- new.env()
assign("mean", 0., env = myenv)
assign("sd", 1., env = myenv)
assign("x", seq(-3., 3., len = 31), env = myenv)
numericDeriv(quote(pnorm(x, mean, sd)), c("mean", "sd"), myenv)
}
\keyword{models}

\eof
\name{offset}
\title{Include an Offset in a Model Formula}
\usage{
offset(object)
}
\alias{offset}
\arguments{
 \item{object}{An offset to be included in a model frame}
}
\description{
  An offset is a term to be added to a linear predictor, such as in a
  generalised linear model, with known coefficient 1 rather than an
  estimated coefficient.
}
\value{
  The input value.
}
\details{
  There can be more than one offset in a model formula, but \code{-} is
  not supported for \code{offset} terms (and is equivalent to \code{+}). 
}

\seealso{
  \code{\link{model.offset}}, \code{\link{model.frame}}.
  
  For examples see \code{\link{glm}}, \code{\link[MASS]{Insurance}}.
}
\keyword{models}
\eof
\name{oneway.test}
\alias{oneway.test}
\title{Test for Equal Means in a One-Way Layout}
\description{
  Test whether two or more samples from normal distributions have the
  same means.  The variances are not necessarily assumed to be equal.
}
\usage{
oneway.test(formula, data, subset, na.action, var.equal = FALSE)
}
\arguments{
  \item{formula}{a formula of the form \code{lhs ~ rhs} where \code{lhs}
    gives the sample values and \code{rhs} the corresponding groups.}
  \item{data}{an optional data frame containing the variables in the
    model formula.}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.  Defaults to
    \code{getOption("na.action")}.}
  \item{var.equal}{a logical variable indicating whether to treat the
    variances in the samples as equal.  If \code{TRUE}, then a simple F
    test for the equality of means in a one-way analysis of variance is 
    preformed.  If \code{FALSE}, an approximate method of Welch (1951)
    is used, which generalizes the commonly known 2-sample Welch test to
    the case of arbitrarily many samples.}
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the test statistic.}
  \item{parameter}{the degrees of freedom of the exact or approximate F
    distribution of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{method}{a character string indicating the test performed.}
  \item{data.name}{a character string giving the names of the data.}
}
\references{
  B. L. Welch (1951),
  On the comparison of several mean values: an alternative approach.
  \emph{Biometrika}, \bold{38}, 330--336.
}
\seealso{
  The standard t test (\code{\link{t.test}}) as the special case for two
  samples;
  the Kruskal-Wallis test \code{\link{kruskal.test}} for a nonparametric
  test for equal location parameters in a one-way layout.
}
\examples{
data(sleep)
## Not assuming equal variances
oneway.test(extra ~ group, data = sleep)
## Assuming equal variances
oneway.test(extra ~ group, data = sleep, var.equal = TRUE)
## which gives the same result as
anova(lm(extra ~ group, data = sleep))
}
\keyword{htest}
\eof
\name{optim}
\alias{optim}
\title{General-purpose Optimization}
\concept{minimization}
\concept{maximization}
\description{
  General-purpose optimization based on Nelder--Mead, quasi-Newton and
  conjugate-gradient algorithms. It includes an option for
  box-constrained optimization and simulated annealing.
}
\usage{
optim(par, fn, gr = NULL,
      method = c("Nelder-Mead", "BFGS", "CG", "L-BFGS-B", "SANN"),
      lower = -Inf, upper = Inf,
      control = list(), hessian = FALSE, \dots)
}
\arguments{
 \item{par}{Initial values for the parameters to be optimized over.}
 \item{fn}{A function to be minimized (or maximized), with first
   argument the vector of parameters over which minimization is to take
   place.  It should return a scalar result.}
 \item{gr}{A function to return the gradient for the \code{"BFGS"},
   \code{"CG"} and \code{"L-BFGS-B"} methods.  If it is \code{NULL}, a
   finite-difference approximation will be used.

   For the \code{"SANN"} method it specifies a function to generate a new
   candidate point.  If it is \code{NULL} a default Gaussian Markov
   kernel is used.
 }
 \item{method}{The method to be used. See \bold{Details}.}
 \item{lower, upper}{Bounds on the variables for the \code{"L-BFGS-B"} method.}
 \item{control}{A list of control parameters. See \bold{Details}.}
 \item{hessian}{Logical. Should a numerically differentiated Hessian
   matrix be returned?}
 \item{\dots}{Further arguments to be passed to \code{fn} and \code{gr}.}
}
\details{
  By default this function performs minimization, but it will maximize
  if \code{control$fnscale} is negative.

  The default method is an implementation of that of Nelder and Mead
  (1965), that uses only function values and is robust but relatively
  slow. It will work reasonably well for non-differentiable functions.

  Method \code{"BFGS"} is a quasi-Newton method (also known as a variable
  metric algorithm), specifically that published simultaneously in 1970
  by Broyden, Fletcher, Goldfarb and Shanno. This uses function values
  and gradients to build up a picture of the surface to be optimized.

  Method \code{"CG"} is a conjugate gradients method based on that by
  Fletcher and Reeves (1964) (but with the option of Polak--Ribiere or
  Beale--Sorenson updates).  Conjugate gradient methods will generally
  be more fragile that the BFGS method, but as they do not store a
  matrix they may be successful in much larger optimization problems.

  Method \code{"L-BFGS-B"} is that of Byrd \emph{et. al.} (1995) which
  allows \emph{box constraints}, that is each variable can be given a lower
  and/or upper bound. The initial value must satisfy the constraints.
  This uses a limited-memory modification of the BFGS quasi-Newton
  method. If non-trivial bounds are supplied, this method will be
  selected, with a warning.

  Nocedal and Wright (1999) is a comprehensive reference for the
  previous three methods.
  
  Method \code{"SANN"} is by default a variant of simulated annealing
  given in Belisle (1992). Simulated-annealing belongs to the class of
  stochastic global optimization methods. It uses only function values
  but is relatively slow. It will also work for non-differentiable
  functions. This implementation uses the Metropolis function for the
  acceptance probability. By default the next candidate point is
  generated from a Gaussian Markov kernel with scale proportional to the
  actual temperature. If a function to generate a new candidate point is
  given, method \code{"SANN"} can also be used to solve combinatorial
  optimization problems. Temperatures are decreased according to the
  logarithmic cooling schedule as given in Belisle (1992, p. 890). Note
  that the \code{"SANN"} method depends critically on the settings of
  the control parameters. It is not a general-purpose method but can be 
  very useful in getting to a good value on a very rough surface.

  Function \code{fn} can return \code{NA} or \code{Inf} if the function
  cannot be evaluated at the supplied value, but the initial value must
  have a computable finite value of \code{fn}.
  (Except for method \code{"L-BFGS-B"} where the values should always be
  finite.)

  \code{optim} can be used recursively, and for a single parameter
  as well as many.

  The \code{control} argument is a list that can supply any of the
  following components:
  \describe{
    \item{\code{trace}}{Non-negative integer. If positive,
      tracing information on the
      progress of the optimization is produced. Higher values may
      produce more tracing information: for method \code{"L-BFGS-B"}
      there are six levels of tracing.  (To understand exactly what
      these do see the source code: higher levels give more detail.)}
    \item{\code{fnscale}}{An overall scaling to be applied to the value
      of \code{fn} and \code{gr} during optimization. If negative,
      turns the problem into a maximization problem. Optimization is
      performed on \code{fn(par)/fnscale}.}
    \item{\code{parscale}}{A vector of scaling values for the parameters.
	Optimization is performed on \code{par/parscale} and these should be
	comparable in the sense that a unit change in any element produces
	about a unit change in the scaled value.}
    \item{\code{ndeps}}{A vector of step sizes for the finite-difference
      approximation to the gradient, on \code{par/parscale}
      scale. Defaults to \code{1e-3}.}
    \item{\code{maxit}}{The maximum number of iterations. Defaults to
      \code{100} for the derivative-based methods, and
      \code{500} for \code{"Nelder-Mead"}. For \code{"SANN"}
      \code{maxit} gives the total number of function evaluations. There is
      no other stopping criterion. Defaults to \code{10000}.}
    \item{\code{abstol}}{The absolute convergence tolerance. Only
      useful for non-negative functions, as a tolerance for reaching zero.}
    \item{\code{reltol}}{Relative convergence tolerance.  The algorithm
      stops if it is unable to reduce the value by a factor of
      \code{reltol * (abs(val) + reltol)} at a step.  Defaults to
      \code{sqrt(.Machine$double.eps)}, typically about \code{1e-8}.}
    \item{\code{alpha}, \code{beta}, \code{gamma}}{Scaling parameters
      for the \code{"Nelder-Mead"} method. \code{alpha} is the reflection
      factor (default 1.0), \code{beta} the contraction factor (0.5) and
      \code{gamma} the expansion factor (2.0).}
    \item{\code{REPORT}}{The frequency of reports for the \code{"BFGS"}
      and \code{"L-BFGS-B"} methods if \code{control$trace} is positive.
      Defaults to every 10 iterations.}
    \item{\code{type}}{for the conjugate-gradients method. Takes value
      \code{1} for the Fletcher--Reeves update, \code{2} for
      Polak--Ribiere and \code{3} for Beale--Sorenson.}
    \item{\code{lmm}}{is an integer giving the number of BFGS updates
      retained in the \code{"L-BFGS-B"} method, It defaults to \code{5}.}
    \item{\code{factr}}{controls the convergence of the \code{"L-BFGS-B"}
      method. Convergence occurs when the reduction in the objective is
      within this factor of the machine tolerance. Default is \code{1e7},
      that is a tolerance of about \code{1e-8}.}
    \item{\code{pgtol}}{helps controls the convergence of the \code{"L-BFGS-B"}
      method. It is a tolerance on the projected gradient in the current
      search direction. This defaults to zero, when the check is
      suppressed.}
    \item{\code{temp}}{controls the \code{"SANN"} method. It is the
      starting temperature for the cooling schedule. Defaults to
      \code{10}.}
    \item{\code{tmax}}{is the number of function evaluations at each
      temperature for the \code{"SANN"} method. Defaults to \code{10}.}
  }
}
\value{
  A list with components:
  \item{par}{The best set of parameters found.}
  \item{value}{The value of \code{fn} corresponding to \code{par}.}
  \item{counts}{A two-element integer vector giving the number of calls
    to \code{fn} and \code{gr} respectively. This excludes those calls needed
    to compute the Hessian, if requested, and any calls to \code{fn} to
    compute a finite-difference approximation to the gradient.}
  \item{convergence}{An integer code. \code{0} indicates successful
    convergence. Error codes are
    \describe{
      \item{\code{1}}{indicates that the iteration limit \code{maxit}
      had been reached.}
      \item{\code{10}}{indicates degeneracy of the Nelder--Mead simplex.}
      \item{\code{51}}{indicates a warning from the \code{"L-BFGS-B"}
      method; see component \code{message} for further details.}
      \item{\code{52}}{indicates an error from the \code{"L-BFGS-B"}
      method; see component \code{message} for further details.}
    }
  }
  \item{message}{A character string giving any additional information
    returned by the optimizer, or \code{NULL}.}
  \item{hessian}{Only if argument \code{hessian} is true. A symmetric
    matrix giving an estimate of the Hessian at the solution found. Note
    that this is the Hessian of the unconstrained problem even if the
    box constraints are active.}
}
\references{
  Belisle, C. J. P. (1992) Convergence theorems for a class of simulated
  annealing algorithms on \eqn{R^d}{Rd}. \emph{J Applied Probability},
  \bold{29}, 885--895.

  Byrd, R. H., Lu, P., Nocedal, J. and Zhu, C.  (1995) A limited
  memory algorithm for bound constrained optimization.
  \emph{SIAM J. Scientific Computing}, \bold{16}, 1190--1208.

  Fletcher, R. and Reeves, C. M. (1964) Function minimization by
  conjugate gradients. \emph{Computer Journal} \bold{7}, 148--154.
  
  Nash, J. C. (1990) \emph{Compact Numerical Methods for
    Computers. Linear Algebra and Function Minimisation.} Adam Hilger.

  Nelder, J. A. and Mead, R. (1965) A simplex algorithm for function
  minimization. \emph{Computer Journal} \bold{7}, 308--313.

  Nocedal, J. and Wright, S. J. (1999) \emph{Numerical Optimization}.
  Springer.
}
\note{
  \code{optim} will work with one-dimensional \code{par}s, but the
  default method does not work well (and will warn).  Use
  \code{\link{optimize}} instead.
  
  The code for methods \code{"Nelder-Mead"}, \code{"BFGS"} and
  \code{"CG"} was based originally on Pascal code in Nash (1990) that was
  translated by \code{p2c} and then hand-optimized.  Dr Nash has agreed
  that the code can be made freely available.

  The code for method \code{"L-BFGS-B"} is based on Fortran code by Zhu,
  Byrd, Lu-Chen and Nocedal obtained from Netlib (file
  \file{opt/lbfgs\_bcm.shar}: another version is in \file{toms/778}).

  The code for method \code{"SANN"} was contributed by A. Trapletti.
}

\seealso{\code{\link{nlm}}, \code{\link{optimize}}, \code{\link{constrOptim}}}

\examples{
fr <- function(x) {   ## Rosenbrock Banana function
    x1 <- x[1]
    x2 <- x[2]
    100 * (x2 - x1 * x1)^2 + (1 - x1)^2
}
grr <- function(x) { ## Gradient of 'fr'
    x1 <- x[1]
    x2 <- x[2]
    c(-400 * x1 * (x2 - x1 * x1) - 2 * (1 - x1),
       200 *      (x2 - x1 * x1))
}
optim(c(-1.2,1), fr)
optim(c(-1.2,1), fr, grr, method = "BFGS")
optim(c(-1.2,1), fr, NULL, method = "BFGS", hessian = TRUE)
optim(c(-1.2,1), fr, grr, method = "CG")
optim(c(-1.2,1), fr, grr, method = "CG", control=list(type=2))
optim(c(-1.2,1), fr, grr, method = "L-BFGS-B")

flb <- function(x)
    { p <- length(x); sum(c(1, rep(4, p-1)) * (x - c(1, x[-p])^2)^2) }
## 25-dimensional box constrained
optim(rep(3, 25), flb, NULL, "L-BFGS-B",
      lower=rep(2, 25), upper=rep(4, 25)) # par[24] is *not* at boundary

## "wild" function , global minimum at about -15.81515
fw <- function (x)
    10*sin(0.3*x)*sin(1.3*x^2) + 0.00001*x^4 + 0.2*x+80
plot(fw, -50, 50, n=1000, main = "optim() minimising 'wild function'")

res <- optim(50, fw, method="SANN",
             control=list(maxit=20000, temp=20, parscale=20))
res
## Now improve locally
(r2 <- optim(res$par, fw, method="BFGS"))
points(r2$par, r2$val, pch = 8, col = "red", cex = 2)

## Combinatorial optimization: Traveling salesman problem
library(stats) # normally loaded

data(eurodist)
eurodistmat <- as.matrix(eurodist)

distance <- function(sq) {  # Target function
    sq2 <- embed(sq, 2)
    return(sum(eurodistmat[cbind(sq2[,2],sq2[,1])]))
}

genseq <- function(sq) {  # Generate new candidate sequence 
    idx <- seq(2, NROW(eurodistmat)-1, by=1)
    changepoints <- sample(idx, size=2, replace=FALSE)
    tmp <- sq[changepoints[1]]
    sq[changepoints[1]] <- sq[changepoints[2]]
    sq[changepoints[2]] <- tmp
    return(sq)
}

sq <- c(1,2:NROW(eurodistmat),1)  # Initial sequence
distance(sq)

set.seed(2222) # chosen to get a good soln quickly
res <- optim(sq, distance, genseq, method="SANN",
             control = list(maxit=6000, temp=2000, trace=TRUE))
res  # Near optimum distance around 12842

loc <- cmdscale(eurodist)
rx <- range(x <- loc[,1])
ry <- range(y <- -loc[,2])
tspinit <- loc[sq,]
tspres <- loc[res$par,]
s <- seq(NROW(tspres)-1)

plot(x, y, type="n", asp=1, xlab="", ylab="",
     main="initial solution of traveling salesman problem")
arrows(tspinit[s,1], -tspinit[s,2], tspinit[s+1,1], -tspinit[s+1,2],
       angle=10, col="green") 
text(x, y, names(eurodist), cex=0.8)

plot(x, y, type="n", asp=1, xlab="", ylab="",
     main="optim() 'solving' traveling salesman problem") 
arrows(tspres[s,1], -tspres[s,2], tspres[s+1,1], -tspres[s+1,2],
       angle=10, col="red")
text(x, y, names(eurodist), cex=0.8)
}
\keyword{nonlinear}
\keyword{optimize}
\eof
\name{optimize}
\title{One Dimensional Optimization}
\usage{
optimize(f = , interval = , lower = min(interval),
        upper = max(interval), maximum = FALSE,
        tol = .Machine$double.eps^0.25, \dots)
optimise(f = , interval = , lower = min(interval),
        upper = max(interval), maximum = FALSE,
        tol = .Machine$double.eps^0.25, \dots)
}
\alias{optimize}
\alias{optimise}
\arguments{
  \item{f}{the function to be optimized. The function is
    either minimized or maximized over its first argument
    depending on the value of \code{maximum}.}
  \item{interval}{a vector containing the end-points of the interval
    to be searched for the minimum.}
  \item{lower}{the lower end point of the interval
    to be searched.}
  \item{upper}{the upper end point of the interval
    to be searched.}
  \item{maximum}{logical. Should we maximize or minimize (the default)?}
  \item{tol}{the desired accuracy.}
  \item{\dots}{additional arguments to \code{f}.}
}
\description{
  The function \code{optimize} searches the interval from
  \code{lower} to \code{upper} for a minimum or maximum of
  the function \code{f} with respect to its first argument.

  \code{optimise} is an alias for \code{optimize}.
}
\details{
  The method used is a combination of golden section search and
  successive parabolic interpolation.  Convergence is never much slower
  than that for a Fibonacci search.  If \code{f} has a continuous second
  derivative which is positive at the minimum (which is not at \code{lower} or
  \code{upper}), then convergence is superlinear, and usually of the
  order of about 1.324.

  The function \code{f} is never evaluated at two points closer together
  than \eqn{\epsilon}{eps *}\eqn{ |x_0| + (tol/3)}, where
  \eqn{\epsilon}{eps} is approximately \code{sqrt(\link{.Machine}$double.eps)}
  and \eqn{x_0} is the final abscissa \code{optimize()$minimum}.\cr
  If \code{f} is a unimodal function and the computed values of \code{f}
  are always unimodal when separated by at least \eqn{\epsilon}{eps *}
  \eqn{ |x| + (tol/3)}, then \eqn{x_0} approximates the abscissa of the
  global minimum of \code{f} on the interval \code{lower,upper} with an
  error less than \eqn{\epsilon}{eps *}\eqn{ |x_0|+ tol}.\cr
  If \code{f} is not unimodal, then \code{optimize()} may approximate a
  local, but perhaps non-global, minimum to the same accuracy.

  The first evaluation of \code{f} is always at
  \eqn{x_1 = a + (1-\phi)(b-a)} where \code{(a,b) = (lower, upper)} and
  \eqn{\phi = (\sqrt 5 - 1)/2 = 0.61803..} is the golden section ratio.
  Almost always, the second evaluation is at \eqn{x_2 = a + \phi(b-a)}.
  Note that a local minimum inside \eqn{[x_1,x_2]} will be found as
  solution, even when \code{f} is constant in there, see the last
  example.

  It uses a C translation of Fortran code (from Netlib) based on the
  Algol 60 procedure \code{localmin} given in the reference.
}
\value{
  A list with components \code{minimum} (or \code{maximum})
  and \code{objective} which give the location of the minimum (or maximum)
  and the value of the function at that point.
}
\references{
  Brent, R. (1973)
  \emph{Algorithms for Minimization without Derivatives.}
  Englewood Cliffs N.J.: Prentice-Hall.
}
\seealso{
  \code{\link{nlm}}, \code{\link{uniroot}}.
}
\examples{
f <- function (x,a) (x-a)^2
xmin <- optimize(f, c(0, 1), tol = 0.0001, a = 1/3)
xmin

## See where the function is evaluated:
optimize(function(x) x^2*(print(x)-1), l=0, u=10)

## "wrong" solution with unlucky interval and piecewise constant f():
f  <- function(x) ifelse(x > -1, ifelse(x < 4, exp(-1/abs(x - 1)), 10), 10)
fp <- function(x) { print(x); f(x) }

plot(f, -2,5, ylim = 0:1, col = 2)
optimize(fp, c(-4, 20))# doesn't see the minimum
optimize(fp, c(-7, 20))# ok
}
\keyword{optimize}
\eof
\name{order.dendrogram}
\alias{order.dendrogram}
\title{ Get the ordering of the leaves in a dendrogram }
\description{
  This function returns the order (index) for the leaves in a
  dendrogram.  These indices can then be used to access the appropriate
  components of any additional data.
}
\usage{
order.dendrogram(x)
}
\arguments{
  \item{x}{A dendrogram (see \code{\link{as.dendrogram}}).}
}
\details{
  The indices for the leaves in left to right order are retrieved.
}
\value{
  A vector with length equal to the number of leaves in the dendrogram
  is returned.  Each element is the index into the original data (from
  which the dendrogram was computed)
}
\author{R. Gentleman}

\seealso{\code{\link{reorder}}, \code{\link{dendrogram}}.
}
\examples{
  set.seed(123)
  x <- rnorm(10)
  hc <- hclust(dist(x))
  hc$order
  dd <- as.dendrogram(hc)
  order.dendrogram(dd)
}
\keyword{manip}
\eof
\name{p.adjust}
\alias{p.adjust}
\alias{p.adjust.methods}
\title{Adjust p-values for multiple comparisons}
\description{Given a set of p-values, returns p-values adjusted using
  one of several methods.}
\usage{
p.adjust(p, method=p.adjust.methods, n=length(p))

p.adjust.methods
}
\arguments{
  \item{p}{vector of p-values}
  \item{method}{correction method}
  \item{n}{number of comparisons}
}
\value{A vector of corrected p-values.}
\details{
  The adjustment methods include the Bonferroni correction
  (\code{"bonferroni"}) in which the p-values are multiplied by the
  number of comparisons.  Four less conservative corrections are also
  included by Holm (1979) (\code{"holm"}), Hochberg (1988)
  (\code{"hochberg"}), Hommel (1988) (\code{"hommel"}) and Benjamini &
  Hochberg (1995) (\code{"fdr"}), respectively.
  A pass-through option (\code{"none"}) is also included.
  The set of methods are contained in the \code{p.adjust.methods} vector
  for the benefit of methods that need to have the method as an option
  and pass it on to \code{p.adjust}.

  The first four methods are designed to give strong control of the
  family wise error rate.  There seems no reason to use the unmodified
  Bonferroni correction because it is dominated by Holm's method, which
  is also valid under arbitrary assumptions.

  Hochberg's and Hommel's methods are valid when the hypothesis tests
  are independent or when they are non-negatively associated (Sarkar,
  1998; Sarkar and Chang, 1997).  Hommel's method is more powerful than
  Hochberg's, but the difference is usually small and the Hochberg
  p-values are faster to compute.

  The \code{"fdr"} method of Benjamini and Hochberg (1995) controls the
  false discovery rate, the expected proportion of false discoveries
  amongst the rejected hypotheses.  The false discovery rate is a less
  stringent condition than the family wise error rate, so Benjamini and
  Hochberg's method is more powerful than the other methods.
}

\references{
  Benjamini, Y., and Hochberg, Y. (1995).
  Controlling the false discovery rate: a practical and powerful
  approach to multiple testing.
  \emph{Journal of the Royal Statistical Society Series} B, \bold{57},
  289--300.

  Holm, S. (1979).
  A simple sequentially rejective multiple test procedure.
  \emph{Scandinavian Journal of Statistics}, \bold{6}, 65--70.

  Hommel, G. (1988).
  A stagewise rejective multiple test procedure based on a modified
  Bonferroni test.
  \emph{Biometrika}, \bold{75}, 383--386.

  Hochberg, Y. (1988).
  A sharper Bonferroni procedure for multiple tests of significance.
  \emph{Biometrika}, \bold{75}, 800--803.

  Shaffer, J. P. (1995).
  Multiple hypothesis testing.
  \emph{Annual Review of Psychology}, \bold{46}, 561--576.
  (An excellent review of the area.)

  Sarkar, S. (1998).
  Some probability inequalities for ordered MTP2 random variables: a
  proof of Simes conjecture.
  \emph{Annals of Statistics}, \bold{26}, 494--504.

  Sarkar, S., and Chang, C. K. (1997).
  Simes' method for multiple hypothesis testing with positively
  dependent test statistics.
  \emph{Journal of the American Statistical Association}, \bold{92},
  1601--1608.

  Wright, S. P. (1992).
  Adjusted P-values for simultaneous inference.
  \emph{Biometrics}, \bold{48}, 1005--1013.
  (Explains the adjusted P-value approach.)
}

\seealso{
  \code{pairwise.*} functions in the \pkg{stats} package, such
  as \code{\link{pairwise.t.test}}.
}

\examples{
x <- rnorm(50, m=c(rep(0,25),rep(3,25)))
p <- 2*pnorm( -abs(x))

round(p, 3)

round(p.adjust(p), 3)
round(p.adjust(p,"bonferroni"), 3)

round(p.adjust(p,"fdr"), 3)
}
\keyword{htest}
\eof
\name{pairwise.prop.test}
\alias{pairwise.prop.test}
\title{ Pairwise comparisons of proportions}
\description{
  Calculate pairwise comparisons between pairs of proportions with
  correction for multiple testing 
}
\usage{
pairwise.prop.test(x, n, p.adjust.method=p.adjust.methods, ...)
}
\arguments{
 \item{x}{ Vector of counts of successes or a matrix with 2 columns
          giving the counts of successes and failures, respectively. }
 \item{n}{ Vector of counts of trials; ignored if \code{x} is a matrix.}
 \item{p.adjust.method}{Method for adjusting p values (see \code{\link{p.adjust}}) }
 \item{\dots}{ Additional arguments to pass to \code{prop.test} }
}
\value{
  Object of class \code{"pairwise.htest"}
}
\seealso{ \code{\link{prop.test}}, \code{\link{p.adjust}}}
\examples{
smokers  <- c( 83, 90, 129, 70 )
patients <- c( 86, 93, 136, 82 )
pairwise.prop.test(smokers, patients)
}
\keyword{htest}
\eof
\name{pairwise.t.test}
\alias{pairwise.t.test}
\title{ Pairwise t tests}
\description{
 Calculate pairwise comparisons between group levels with corrections
 for multiple testing 
}
\usage{
pairwise.t.test(x, g, p.adjust.method=p.adjust.methods, pool.sd=TRUE, \dots)
}
\arguments{
 \item{x}{ Response vector }
 \item{g}{ Grouping vector or factor }
 \item{p.adjust.method}{ Method for adjusting p values (see \code{\link{p.adjust}}) }
 \item{pool.sd}{ Switch to allow/disallow the use of a pooled SD }
 \item{\dots}{ Additional arguments to pass to \code{t.test} }
}
\value{
  Object of class \code{"pairwise.htest"}
}
\seealso{ \code{\link{t.test}}, \code{\link{p.adjust}}}

\examples{
data(airquality)
attach(airquality)
Month <- factor(Month, labels = month.abb[5:9])
pairwise.t.test(Ozone, Month)
pairwise.t.test(Ozone, Month, p.adj = "bonf")
pairwise.t.test(Ozone, Month, pool.sd = FALSE)
detach()
}
\keyword{htest}
\eof
\name{pairwise.table}
\alias{pairwise.table}
\title{Tabulate p values for pairwise comparisons}
\description{
 Creates  table of p values for pairwise comparisons
 with corrections for multiple testing.
}
\usage{
pairwise.table(compare.levels, level.names, p.adjust.method)
}
\arguments{
 \item{compare.levels}{ Function to compute (raw) p value given indices
   \code{i} and \code{j} }
 \item{level.names}{ Names of the group levels}
 \item{p.adjust.method}{Method for multiple testing adjustment}
}
\details{
  Functions that do multiple group comparisons create separate
  \code{compare.levels} functions (assumed to be symmetrical in \code{i}
  and \code{j}) and passes them to this function.
}
\value{
  Table of p values in lower triangular form.
}
\seealso{ \code{\link{pairwise.t.test}}, et al.}

\keyword{htest}
\eof
\name{pairwise.wilcox.test}
\alias{pairwise.wilcox.test}
\title{ Pairwise Wilcoxon rank sum tests}
\description{
  Calculate pairwise comparisons between group levels with corrections
  for multiple testing.
}
\usage{
pairwise.wilcox.test(x, g, p.adjust.method=p.adjust.methods, \dots)
}
\arguments{
  \item{x}{ Response vector }
  \item{g}{ Grouping vector or factor }
  \item{p.adjust.method}{ Method for adjusting p values (see
    \code{\link{p.adjust}}) }
  \item{\dots}{Additional arguments to pass to \code{\link{wilcox.test}}.}
}
\value{
  Object of class \code{"pairwise.htest"}
}
\seealso{ \code{\link{wilcox.test}}, \code{\link{p.adjust}}}

\examples{
data(airquality)
attach(airquality)
Month <- factor(Month, labels = month.abb[5:9])
## These give warnings because of ties :
pairwise.wilcox.test(Ozone, Month)
pairwise.wilcox.test(Ozone, Month, p.adj = "bonf")
detach()
}
\keyword{htest}
\eof
\name{plot.HoltWinters}
\alias{plot.HoltWinters}
\title{Plot function for HoltWinters objects}
\description{
  Produces a chart of the original time series along with the fitted
  values. Optionally, predicted values (and their confidence bounds) can
  also be plotted.
}
\usage{
\method{plot}{HoltWinters}(x, predicted.values = NA, intervals = TRUE,
        separator = TRUE, col = 1, col.predicted = 2,
        col.intervals = 4, col.separator = 1, lty = 1,
        lty.predicted = 1, lty.intervals = 1, lty.separator = 3,
        ylab = "Observed / Fitted", main = "Holt-Winters filtering",
        ylim = NULL, ...)
}
\arguments{
  \item{x}{Object of class \code{"HoltWinters"}}
  \item{predicted.values}{Predicted values as returned by \code{predict.HoltWinters}}
  \item{intervals}{If \code{TRUE}, the prediction intervals are plotted (default).}
  \item{separator}{If \code{TRUE}, a separating line between fitted and predicted values is plotted (default).}
  \item{col, lty}{Color/line type of original data (default: black solid).}
  \item{col.predicted, lty.predicted}{Color/line type of
    fitted and predicted values (default: red solid).}
  \item{col.intervals, lty.intervals}{Color/line type of prediction
    intervals (default: blue solid).}
  \item{col.separator, lty.separator}{Color/line type of
    observed/predicted values separator (default: black dashed).}
  \item{ylab}{Label of the y-axis.}
  \item{main}{Main title.}
  \item{ylim}{Limits of the y-axis. If \code{NULL}, the range is chosen
    such that the plot contains the original series, the fitted values,
    and the predicted values if any.}
  \item{\dots}{Other graphics parametes.}
}
\references{
  C.C Holt (1957)
  Forecasting seasonals and trends by exponentially weighted
  moving averages,
  ONR Research Memorandum, Carnigie Institute 52.
  
  P.R Winters (1960)
  Forecasting sales by exponentially weighted moving averages,
  \emph{Management Science} \bold{6}, 324--342.
}
\author{
  David Meyer \email{david.meyer@ci.tuwien.ac.at}
}
\seealso{
  \code{\link{HoltWinters}}, \code{\link{predict.HoltWinters}}
}

\keyword{ts}
\eof
\name{plot.acf}
\alias{plot.acf}
\title{Plot Autocovariance and Autocorrelation Functions}
\description{
  Plot method for objects of class \code{"acf"}.
}
\usage{
\method{plot}{acf}(x, ci = 0.95, type = "h", xlab = "Lag", ylab = NULL,
     ylim = NULL, main = NULL, ci.col="blue", ci.type = c("white", "ma"),
     max.mfrow = 6, ask = Npgs > 1 && dev.interactive(),
     mar = if(nser > 2) c(3,2,2,0.8) else par("mar"),
     oma = if(nser > 2) c(1,1.2,1,1) else par("oma"),
     mgp = if(nser > 2) c(1.5,0.6,0) else par("mgp"),
     xpd = par("xpd"), cex.main = if(nser > 2) 1 else par("cex.main"),
     verbose = getOption("verbose"),
     \dots)
}
\arguments{
  \item{x}{an object of class \code{"acf"}.}
  \item{ci}{coverage probability for confidence interval.  Plotting of
    the confidence interval is suppressed if \code{ci} is zero or
    negative.}
  \item{type}{the type of plot to be drawn, default to histogram like
    vertical lines.}
  \item{xlab}{the x label of the plot.}
  \item{ylab}{the y label of the plot.}
  \item{ylim}{numeric of length 2 giving the y limits for the plot.}
  \item{main}{overall title for the plot.}
  \item{ci.col}{colour to plot the confidence interval lines.}
  \item{ci.type}{should the confidence limits assume a white noise
    input or for lag \eqn{k} an MA(\eqn{k-1}) input?}
  \item{max.mfrow}{positive integer; for multivariate \code{x}
    indicating how many rows and columns of plots should be put on one
    page, using \code{\link{par}(mfrow = c(m,m))}.}
  \item{ask}{logical; if \code{TRUE}, the user is asked before a new
    page is started.}
  \item{mar, oma, mgp, xpd, cex.main}{graphics parameters as in
    \code{\link{par}(*)}, by default adjusted to use smaller than
    default margins for multivariate \code{x} only.
    % drop the following line after year 2002
    \code{xpd = NA} used to be the default for R version <= 1.4.0.
  }
  \item{verbose}{logical.  Should \R report extra information on
    progress?}
  \item{\dots}{graphics parameters to be passed to the plotting
    routines.}
}
\note{
  The confidence interval plotted in \code{plot.acf} is based on an
  \emph{uncorrelated} series and should be treated with appropriate
  caution.  Using \code{ci.type = "ma"} may be less potentially
  misleading.
}
\seealso{\code{\link{acf}} which calls \code{plot.acf} by default.
}
\examples{
%%-- FIXME: use less trivial data :
z4  <- ts(matrix(rnorm(400), 100, 4), start=c(1961, 1), frequency=12)
z7  <- ts(matrix(rnorm(700), 100, 7), start=c(1961, 1), frequency=12)
acf(z4)
acf(z7, max.mfrow = 7)# squeeze on 1 page
acf(z7) # multi-page
}
\keyword{hplot}
\keyword{ts}
\eof
\name{plot.density}
\alias{plot.density}
\title{Plot Method for Kernel Density Estimation}
\usage{
\method{plot}{density}(x, main = NULL, xlab = NULL, ylab = "Density", type = "l",
     zero.line = TRUE, \dots)
}
\arguments{
\item{x}{a \dQuote{density} object.}
\item{main, xlab, ylab, type}{plotting parameters with useful defaults.}
\item{\dots}{further plotting parameters.}
\item{zero.line}{logical; if \code{TRUE}, add a base line at \eqn{y = 0}}
}
\description{
  The \code{plot} method for density objects.
}
\value{
  None.
}
\references{
}
\seealso{
  \code{\link{density}}.
}
\keyword{dplot}
\eof
\name{plot.isoreg}
\alias{plot.isoreg}
%\alias{print.isoreg}
\title{Plot Method for isoreg  Objects}
\description{
  The \code{\link{plot}} method for
  \R objects of class \code{\link{isoreg}}.
}
\usage{
\method{plot}{isoreg}(x, plot.type = c("single", "row.wise", "col.wise"),
      main = paste("Isotonic regression", deparse(x$call)),
      main2 = "Cumulative Data and Convex Minorant",
      xlab = "x0", ylab = "x$y",
      par.fit = list(col = "red", cex = 1.5, pch = 13, lwd = 1.5),
      mar = if (both) 0.1 + c(3.5, 2.5, 1, 1) else par("mar"),
      mgp = if (both) c(1.6, 0.7, 0) else par("mgp"),
      grid = length(x$x) < 12, \dots)
%\method{print}{isoreg}(x, digits = getOption("digits"), \dots)
}
\arguments{
  \item{x}{an \code{\link{isoreg}} object.}
  \item{plot.type}{character indicating which type of plot is desired.
    The first (default) only draws the data and the fit, where the
    others add a plot of the cumulative data and fit.}
  \item{main}{main title of plot, see \code{\link{title}}.}
  \item{main2}{title for second (cumulative) plot.}
  \item{xlab, ylab}{x- and y- axis annotation.}
  \item{par.fit}{a \code{\link{list}} of arguments (for
    \code{\link{points}} and \code{\link{lines}}) for drawing the fit.}
  \item{mar, mgp}{graphical parameters, see \code{\link{par}}, mainly
    for the case of two plots.}
  \item{grid}{logical indicating if grid lines should be drawn.  If
    true, \code{\link{grid}()} is used for the first plot, where as
    vertical lines are drawn at \dQuote{touching} points for the
    cumulative plot.}
  \item{\dots}{further arguments passed to and from methods.}
%  \item{digits}{integer indicating number of digits to use for printing.}
}
\seealso{\code{\link{isoreg}} for computation of \code{isoreg} objects.}
\examples{
example(isoreg) # for the examples there

## 'same' plot as above, "proving" that only ranks of 'x' are important
plot(isoreg(2^(1:9), c(1,0,4,3,3,5,4,2,0)), plot.t = "row", log = "x")

plot(ir3, plot.type = "row", ylab = "y3")
plot(isoreg(y3 - 4), plot.t="r", ylab = "y3 - 4")
plot(ir4, plot.type = "ro",  ylab = "y4", xlab = "x = 1:n")

## experiment a bit with these (C-c C-j):
plot(isoreg(sample(9),  y3), plot.type="row")
plot(isoreg(sample(9),  y3), plot.type="col.wise")

plot(ir <- isoreg(sample(10), sample(10, replace = TRUE)), plot.t = "r")
}
\keyword{hplot}
\keyword{print}
\eof
\name{plot.lm}
\alias{plot.lm}
\alias{plot.mlm}%which is .NotYetImplemented()
\title{Plot Diagnostics for an lm Object}
\usage{
\method{plot}{lm}(x, which = 1:4,
     caption = c("Residuals vs Fitted", "Normal Q-Q plot",
                 "Scale-Location plot", "Cook's distance plot"),
     panel = points,
     sub.caption = deparse(x$call), main = "",
     ask = prod(par("mfcol")) < length(which) && dev.interactive(),
     \dots,
     id.n = 3, labels.id = names(residuals(x)), cex.id = 0.75)
}
\arguments{
  \item{x}{\code{lm} object, typically result of \code{\link{lm}} or
    \code{\link{glm}}.}
  \item{which}{If a subset of the plots is required, specify a subset of
    the numbers \code{1:4}.}
  \item{caption}{Captions to appear above the plots}
  \item{panel}{Panel function.  A useful alternative to
    \code{\link{points}} is \code{\link{panel.smooth}}.}
  \item{sub.caption}{common title---above figures if there are multiple;
    used as \code{sub} (s.\code{\link{title}}) otherwise.}
  \item{main}{title to each plot---in addition to the above
    \code{caption}.}
  \item{ask}{logical; if \code{TRUE}, the user is \emph{ask}ed before
    each plot, see \code{\link{par}(ask=.)}.}
  \item{\dots}{other parameters to be passed through to plotting
    functions.}
  \item{id.n}{number of points to be labelled in each plot, starting
    with the most extreme.}
  \item{labels.id}{vector of labels, from which the labels for extreme
    points will be chosen.  \code{NULL} uses observation numbers.}
  \item{cex.id}{magnification of point labels.}
}
\description{
  Four plots (selectable by \code{which}) are currently provided: a plot
  of residuals against fitted values, a Scale-Location plot of
  \eqn{\sqrt{| residuals |}} against fitted values, a Normal Q-Q plot,
  and a plot of Cook's distances versus row labels.
}
\details{
  \code{sub.caption}---by default the function call---is shown as
  a subtitle (under the x-axis title) on each plot when plots are on
  separate pages, or as a subtitle in the outer margin (if any) when
  there are multiple plots per page.

  The \dQuote{Scale-Location} plot, also called \dQuote{Spread-Location} or
  \dQuote{S-L} plot, takes the square root of the absolute residuals in
  order to diminish skewness (\eqn{\sqrt{| E |}} is much less skewed
  than \eqn{| E |} for Gaussian zero-mean \eqn{E}).

  This \sQuote{S-L} and the Q-Q plot use \emph{standardized} residuals
  which have identical variance (under the hypothesis).  They are given
  as \eqn{R_i / (s \times \sqrt{1 - h_{ii}})}{R[i] / (s*sqrt(1 - h.ii))}
  where \eqn{h_{ii}}{h.ii} are the diagonal entries of the hat matrix,
% bug in Rdconv: "$" and \link inside \code fails; '\$' doesn't help :
  \code{\link{influence}()}\code{$hat}, see also \code{\link{hat}}.
}
\references{
  Belsley, D. A., Kuh, E. and Welsch, R. E. (1980)
  \emph{Regression Diagnostics.}  New York: Wiley.

  Cook, R. D. and Weisberg, S. (1982)
  \emph{Residuals and Influence in Regression.}
  London: Chapman and Hall.

  Hinkley, D. V. (1975) On power transformations to
  symmetry. \emph{Biometrika} \bold{62}, 101--111.

  McCullagh, P. and Nelder, J. A. (1989)
  \emph{Generalized Linear Models.}
  London: Chapman and Hall.
}
\author{
  John Maindonald and Martin Maechler.
}
\seealso{\code{\link{termplot}}, \code{\link{lm.influence}},
  \code{\link{cooks.distance}}.
}
\examples{
## Analysis of the life-cycle savings data
## given in Belsley, Kuh and Welsch.
data(LifeCycleSavings)
plot(lm.SR <- lm(sr ~ pop15 + pop75 + dpi + ddpi, data = LifeCycleSavings))

## 4 plots on 1 page; allow room for printing model formula in outer margin:
par(mfrow = c(2, 2), oma = c(0, 0, 2, 0))
plot(lm.SR)
plot(lm.SR, id.n = NULL)               # no id's
plot(lm.SR, id.n = 5, labels.id = NULL)# 5 id numbers

## Fit a smooth curve, where applicable:
plot(lm.SR, panel = panel.smooth)
## Gives a smoother curve
plot(lm.SR, panel = function(x,y) panel.smooth(x, y, span = 1))

par(mfrow=c(2,1))# same oma as above
plot(lm.SR, which = 1:2, sub.caption = "Saving Rates, n=50, p=5")
}
\keyword{hplot}
\keyword{regression}
\eof
% file modreg/man/plot.ppr.Rd
% copyright (C) 1995-8 B. D. Ripley
%
\name{plot.ppr}
\alias{plot.ppr}
\title{
Plot Ridge Functions for Projection Pursuit Regression Fit
}
\description{
Plot ridge functions for projection pursuit regression fit.
}
\usage{
\method{plot}{ppr}(x, ask, type = "o", \dots)
}
\arguments{
  \item{x}{
    A fit of class \code{"ppr"} as produced by a call to \code{ppr}.
  }
  \item{ask}{
    the graphics parameter \code{ask}: see \code{par} for details.
    If set to \code{TRUE} will ask between the plot of each cross-section.
  }
  \item{type}{
    the type of line to draw
  }
  \item{\dots}{
    further graphical parameters
  }
}
\value{
  None
}
\section{Side Effects}{
  A series of plots are drawn on the current graphical device, one for
  each term in the fit.
}
\seealso{
  \code{\link{ppr}}, \code{\link{par}}
}
\examples{
data(rock)
attach(rock)
area1 <- area/10000; peri1 <- peri/10000
par(mfrow=c(3,2))# maybe: , pty="s")
rock.ppr <- ppr(log(perm) ~ area1 + peri1 + shape,
                data = rock, nterms = 2, max.terms = 5)
plot(rock.ppr, main="ppr(log(perm)~ ., nterms=2, max.terms=5)")
plot(update(rock.ppr, bass=5), main = "update(..., bass = 5)")
plot(update(rock.ppr, sm.method="gcv", gcvpen=2),
     main = "update(..., sm.method=\"gcv\", gcvpen=2)")
detach()
}
\keyword{hplot}
% Converted by Sd2Rd version 0.2-a5.
\eof
\name{plot.profile.nls}
\alias{plot.profile.nls}
\title{Plot a profile.nls Object}
\description{
  Displays a series of plots of the profile t function and interpolated
  confidence intervals for the parameters in a nonlinear regression
  model that has been fit with \code{nls} and profiled with
  \code{profile.nls}.
}
\usage{
\method{plot}{profile.nls}(x, levels, conf= c(99, 95, 90, 80, 50)/100,
     nseg = 50, absVal =TRUE, \dots)
}
\arguments{
 \item{x}{an object of class \code{"profile.nls"} }
 \item{levels}{levels, on the scale of the absolute value of a t
     statistic, at which to interpolate intervals.  Usually \code{conf}
     is used instead of giving \code{levels} explicitly.}
 \item{conf}{a numeric vector of confidence levels for profile-based
     confidence intervals on the parameters.  Defaults to \code{c(0.99,
	 0.95, 0.90, 0.80, 0.50).}}
 \item{nseg}{an integer value giving the number of segments to use in
     the spline interpolation of the profile t curves. Defaults to 50.}
 \item{absVal}{a logical value indicating whether or not the plots
     should be on the scale of the absolute value of the profile t.
     Defaults to \code{TRUE}.} 
 \item{\dots}{other arguments to the \code{plot} function can be passed here.}
}
\references{
  Bates, D.M. and Watts, D.G. (1988),
  \emph{Nonlinear Regression Analysis and Its Applications},
  Wiley (chapter 6)
}
\author{Douglas M. Bates and Saikat DebRoy}
\seealso{
  \code{\link{nls}},
  \code{\link{profile}},
  \code{\link{profile.nls}}
}
\examples{
data( BOD )
# obtain the fitted object
fm1 <- nls(demand ~ SSasympOrig( Time, A, lrc ), data = BOD)
# get the profile for the fitted model
pr1 <- profile( fm1 )
opar <- par(mfrow = c(2,2), oma = c(1.1, 0, 1.1, 0), las = 1)
plot(pr1, conf = c(95, 90, 80, 50)/100)
plot(pr1, conf = c(95, 90, 80, 50)/100, absVal = FALSE)
mtext("Confidence intervals based on the profile sum of squares",
      side = 3, outer = TRUE)
mtext("BOD data - confidence levels of 50\%, 80\%, 90\% and 95\%",
      side = 1, outer = TRUE)
par(opar)
}
\keyword{nonlinear}
\keyword{regression}
\keyword{models}
\eof
\name{plot.spec}
\alias{plot.spec}
\alias{plot.spec.coherency}
\alias{plot.spec.phase}
\title{Plotting Spectral Densities}
\description{
  Plotting method for objects of class \code{"spec"}.  For multivariate
  time series it plots the marginal spectra of the series or pairs plots
  of the coherency and phase of the cross-spectra.
}
\synopsis{
plot.spec(x, add = FALSE, ci = 0.95, log = c("yes", "dB", "no"), 
    xlab = "frequency", ylab = NULL, type = "l",
    ci.col = "blue", main = NULL, sub = NULL,
    plot.type = c("marginal", "coherency", "phase"), \dots) 
plot.spec.phase(x, ci = 0.95, xlab = "frequency",
    ylab = "phase", ylim = c(-pi, pi), type = "l",
    main = NULL, ci.lty = 3, ci.col = "blue", \dots) 
plot.spec.coherency(x, ci = 0.95, xlab = "frequency",
    ylab = "squared coherency", ylim = c(0, 1), type = "l",
    main = NULL, ci.lty = 3, ci.col = "blue", \dots)
}
\usage{
\method{plot}{spec}(x, add = FALSE, ci = 0.95, log = c("yes", "dB", "no"),
     xlab = "frequency", ylab, type = "l", ci.col = "blue",
     main = NULL, sub = NULL,
     plot.type = c("marginal", "coherency", "phase"),
     ci.lty = 3, \dots)
}
\arguments{
  \item{x}{an object of class \code{"spec"}.}
  \item{add}{logical.  If \code{TRUE}, add to already existing plot.}
  \item{ci}{Coverage probability for confidence interval.  Plotting of
    the confidence bar is omitted unless \code{ci} is strictly
    positive.}
  \item{log}{If \code{"dB"}, plot on log10 (decibel) scale (as S-PLUS),
    otherwise use conventional log scale or linear scale.  Logical
    values are also accepted.  The default is \code{"yes"} unless
    \code{options(ts.S.compat = TRUE)} has been set, when it is
    \code{"dB"}.}
  \item{xlab}{the x label of the plot.}  
  \item{ylab}{the y label of the plot.}
  \item{type}{the type of plot to be drawn, defaults to lines.}
  \item{ci.col}{Colour for plotting confidence bar or confidence
    intervals for coherency and phase.}
  \item{main}{overall title for the plot.}
  \item{sub}{a sub title for the plot.}
  \item{plot.type}{For multivariate time series, the type of plot
    required.  Only the first character is needed.}
  \item{ci.lty}{line type for confidence intervals for coherency and
    phase.}
  \item{\dots}{Further graphical parameters.}
}
\seealso{
  \code{\link{spectrum}}
}
\keyword{hplot}
\keyword{ts}
\eof
\name{plot.stepfun}
\alias{plot.stepfun}
\alias{lines.stepfun}
\title{Plot Step Functions}
\description{
  Method of the generic \code{\link{plot}} for \code{\link{stepfun}}
  objects and utility for plotting piecewise constant functions.
}
\usage{
\method{plot}{stepfun}(x, xval, xlim, ylim,
     xlab = "x", ylab = "f(x)", main = NULL,
     add = FALSE, verticals = TRUE, do.points = TRUE,
     pch = par("pch"), col.points=par("col"), cex.points=par("cex"),
     col.hor = par("col"), col.vert= par("col"),
     lty = par("lty"), lwd = par("lwd"), \dots)

\method{lines}{stepfun}(x, \dots)
}
\arguments{
  \item{x}{an \R object inheriting from \code{"stepfun"}.}
  \item{xval}{numeric vector of abscissa values at which to evaluate
    \code{x}.  Defaults to \code{\link{knots}(x)} restricted to \code{xlim}.}
  \item{xlim,ylim}{numeric(2) each; range of \code{x} or \code{y} values
    to use.  Both have sensible defaults.}
  \item{xlab,ylab}{labels of x and y axis.}
  \item{main}{main title.}
  \item{add}{logical; if \code{TRUE} only \emph{add} to an existing plot.}
  \item{verticals}{logical;  if \code{TRUE}, draw vertical lines at steps.}
  \item{do.points}{logical;  if \code{true}, also draw points at the
    (\code{xlim} restricted) knot locations.}
  \item{pch}{character; point character if \code{do.points}.}
  \item{col.points}{character or integer code; color of points if
    \code{do.points}.}
  \item{cex.points}{numeric; character expansion factor if \code{do.points}.}
  \item{col.hor}{color of horizontal lines.}
  \item{col.vert}{color of vertical lines.}
  \item{lty, lwd}{line type and thickness for all lines.}
  \item{\dots}{further arguments of \code{\link{plot}(.)}, or if\code{(add)}
    \code{\link{segments}(.)}.}
}
\value{
  A list with two components
  \item{t}{abscissa (x) values, including the two outermost ones.}
  \item{y}{y values \sQuote{in between} the \code{t[]}.}
}
\author{Martin Maechler \email{maechler@stat.math.ethz.ch}, 1990,
    1993; ported to \R, 1997.}
\seealso{\code{\link{ecdf}} for empirical distribution functions as
  special step functions,
  \code{\link{approxfun}} and \code{\link{splinefun}}.
}
\examples{
y0 <- c(1,2,4,3)
sfun0  <- stepfun(1:3, y0, f = 0)
sfun.2 <- stepfun(1:3, y0, f = .2)
sfun1  <- stepfun(1:3, y0, right = TRUE)

tt <- seq(0,3, by=0.1)
op <- par(mfrow=c(2,2))
plot(sfun0); plot(sfun0, xval=tt, add=TRUE, col.h="bisque")
plot(sfun.2);plot(sfun.2,xval=tt, add=TRUE, col.h="orange")
plot(sfun1);lines(sfun1, xval=tt, col.h="coral")
##-- This is  revealing :
plot(sfun0, verticals= FALSE,
     main = "stepfun(x, y0, f=f)  for f = 0, .2, 1")
for(i in 1:3)
  lines(list(sfun0,sfun.2,stepfun(1:3,y0,f = 1))[[i]], col.h=i, col.v=i)
legend(2.5, 1.9, paste("f =", c(0,0.2,1)), col=1:3, lty=1, y.inter=1); par(op)

# Extend and/or restrict 'viewport':
plot(sfun0, xlim = c(0,5), ylim = c(0, 3.5),
     main = "plot(stepfun(*), xlim= . , ylim = .)")

##-- this works too (automatic call to  ecdf(.)):
plot.stepfun(rt(50, df=3), col.vert = "gray20")
}
\keyword{hplot}
\eof
\name{plot.ts}
\alias{plot.ts}
\alias{lines.ts}
\title{Plotting Time-Series Objects}
\description{
  Plotting method for objects inheriting from class \code{"ts"}.
}
\usage{
\method{plot}{ts}(x, y = NULL, plot.type = c("multiple", "single"),
        xy.labels, xy.lines, panel = lines, nc, yax.flip = FALSE,
        mar.multi = c(0, 5.1, 0, if(yax.flip) 5.1 else 2.1),
        oma.multi = c(6, 0, 5, 0), axes = TRUE, \dots)

\method{lines}{ts}(x, \dots)
}
\arguments{
  \item{x, y}{time series objects, usually inheriting from class \code{"ts"}.}

  \item{plot.type}{for multivariate time series, should the series by
    plotted separately (with a common time axis) or on a single plot?}

  \item{xy.labels}{logical, indicating if \code{\link{text}()} labels
    should be used for an x-y plot, \emph{or} character, supplying a
    vector of labels to be used.  The default is to label for up to 150
    points, and not for more.}

  \item{xy.lines}{logical, indicating if \code{\link{lines}}
    should be drawn for an x-y plot.  Defaults to the value of
    \code{xy.labels} if that is logical, otherwise to \code{TRUE}.}

  \item{panel}{a \code{function(x, col, bg, pch, type, ...)} which gives the
    action to be carried out in each panel of the display for
    \code{plot.type="multiple"}.  The default is \code{lines}.}

  \item{nc}{the number of columns to use when \code{type="multiple"}.
    Defaults to 1 for up to 4 series, otherwise to 2.}
  \item{yax.flip}{logical indicating if the y-axis (ticks and numbering)
    should flip from side 2 (left) to 4 (right) from series to series
    when \code{type="multiple"}.}
  \item{mar.multi, oma.multi}{the (default) \code{\link{par}} settings
    for \code{plot.type="multiple"}.  Modify with care!}
  \item{axes}{logical indicating if x- and y- axes should be drawn.}
  \item{\dots}{additional graphical arguments, see \code{\link{plot}},
    \code{\link{plot.default}} and \code{\link{par}}.}
}
\details{
  If \code{y} is missing, this function creates a time series
  plot, for multivariate series of one of two kinds depending on
  \code{plot.type}.

  If \code{y} is present, both \code{x} and \code{y} must be univariate,
  and a \dQuote{scatter} plot \code{y ~ x} will be drawn, enhanced by
  using \code{\link{text}} if \code{xy.labels} is
  \code{\link{TRUE}} or \code{character}, and \code{\link{lines}} if
  \code{xy.lines} is \code{TRUE}.
}
\seealso{
  \code{\link{ts}} for basic time series construction and access
  functionality.
}
\examples{
## Multivariate
z <- ts(matrix(rt(200 * 8, df = 3), 200, 8), start=c(1961, 1), frequency=12)
plot(z, yax.flip = TRUE)
plot(z, axes= FALSE, ann= FALSE, frame.plot= TRUE,
      mar.mult= c(0,0,0,0), oma.mult= c(1,1,5,1))
title("plot(ts(..), axes=FALSE, ann=FALSE, frame.plot=TRUE, mar..., oma...)")

z <- window(z[,1:3], end = c(1969,12))
plot(z, type = "b")    # multiple
plot(z, plot.type="single", lty=1:3, col=4:2)

## A phase plot:
data(nhtemp)
plot(nhtemp, c(nhtemp[-1], NA), cex = .8, col="blue",
     main = "Lag plot of New Haven temperatures")
## a clearer way to do this would be
\dontrun{
plot(nhtemp, lag(nhtemp, 1), cex = .8, col="blue",
     main = "Lag plot of New Haven temperatures")
}

data(sunspots)
## xy.lines and xy.labels are FALSE for large series:
plot(lag(sunspots, 1), sunspots, pch = ".")

data(EuStockMarkets)
SMI <- EuStockMarkets[, "SMI"]
plot(lag(SMI,  1), SMI, pch = ".")
plot(lag(SMI, 20), SMI, pch = ".", log = "xy",
     main = "4 weeks lagged SMI stocks -- log scale", xy.lines= TRUE)
}
\keyword{hplot}
\keyword{ts}
\eof
\name{poly}
\alias{poly}
\alias{polym}
\alias{predict.poly}
\alias{makepredictcall.poly}
\title{Compute Orthogonal Polynomials}
\usage{
poly(x, \dots, degree = 1, coefs = NULL)
polym(\dots, degree = 1)

\method{predict}{poly}(object, newdata, \dots)
}
\description{
  Returns or evaluates orthogonal polynomials of degree 1 to
  \code{degree} over the specified set of points \code{x}. These are all
  orthogonal to the constant polynomial of degree 0.
}
\arguments{
 \item{x, newdata}{a numeric vector at which to evaluate the
   polynomial. \code{x} can also be a matrix.}
 \item{degree}{the degree of the polynomial}
 \item{coefs}{for prediction, coefficients from a previous fit.}
 \item{object}{an object inheriting from class \code{"poly"}, normally
   the result of a call to \code{poly} with a single vector argument.}
 \item{\dots}{\code{poly, polym}: further vectors.\cr
   \code{predict.poly}: arguments to be passed to or from other methods.
 }
}
\value{
  For \code{poly} with a single vector argument:\cr
  A matrix with rows corresponding to points in \code{x} and columns
  corresponding to the degree, with attributes \code{"degree"} specifying
  the degrees of the columns and \code{"coefs"} which contains the
  centering and normalization constants used in constructing the
  orthogonal polynomials.  The matrix is given class
  \code{c("poly", "matrix")} as from \R 1.5.0.

  Other cases of \code{poly} and \code{polym}, and \code{predict.poly}:
  a matrix.
}
\details{
  Although formally \code{degree} should be named (as it follows
  \code{\dots}), an unnamed second argument of length 1 will be
  interpreted as the degree.
  
  The orthogonal polynomial is summarized by the coefficients, which can
  be used to evaluate it via the three-term recursion given in Kennedy
  \& Gentle (1980, pp. 343-4), and used in the \dQuote{predict} part of
  the code.
}
\note{
  This routine is intended for statistical purposes such as
  \code{contr.poly}: it does not attempt to orthogonalize to
  machine accuracy.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical Models in S}.
  Wadsworth \& Brooks/Cole.

  Kennedy, W. J. Jr and Gentle, J. E. (1980)
  \emph{Statistical Computing} Marcel Dekker.
}

\seealso{\code{\link{contr.poly}}}

\examples{
(z <- poly(1:10, 3))
predict(z, seq(2, 4, 0.5))
poly(seq(4, 6, 0.5), 3, coefs = attr(z, "coefs"))

polym(1:4, c(1, 4:6), degree=3) # or just poly()
poly(cbind(1:4, c(1, 4:6)), degree=3)
}
\keyword{math}
\eof
\name{power}
\alias{power}
\title{Create a Power Link Object}
\usage{
power(lambda = 1)
}
\arguments{
    \item{lambda}{a real number.}
}
\description{
    Creates a link object based on the link function \eqn{\eta = \mu ^
        \lambda}{eta = mu^lambda}.
}
\details{
    If \code{lambda} is non-negative, it is taken as zero, and the log
    link is obtained.  The default \code{lambda = 1} gives the identity
    link.
}
\value{
    A list with components \code{linkfun}, \code{linkinv},
    \code{mu.eta}, and \code{valideta}.
    See \code{\link{make.link}} for information on their meaning.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical Models in S.}
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{make.link}},
  \code{\link{family}}

  To raise a number to a power, see \code{\link{Arithmetic}}.

  To calculate the power of a test, see various functions in the 
  \pkg{stats} package, e.g., \code{\link{power.t.test}}.
}
\examples{
power()
quasi(link=power(1/3))[c("linkfun", "linkinv")]
}
\keyword{models}
\eof
\name{power.anova.test}
\alias{power.anova.test}
\title{Power calculations for balanced one-way analysis of variance tests}
\usage{
power.anova.test(groups = NULL, n = NULL, between.var = NULL,
                 within.var = NULL, sig.level = 0.05, power = NULL)
}
\arguments{
 \item{groups}{Number of groups}
 \item{n}{Number of observations (per group)}
 \item{between.var}{Between group variance}
 \item{within.var}{Within group variance}
 \item{sig.level}{Significance level (Type I error probability)}
 \item{power}{Power of test (1 minus Type II error probability)}
}
\description{
 Compute power of test or determine parameters to obtain target power.
}

\details{
  Exactly one of the parameters \code{groups}, \code{n}, \code{between.var},
  \code{power}, \code{within.var}, and \code{sig.level} must be passed as NULL,
  and that parameter is determined from the others. Notice that
  \code{sig.level} has non-NULL default so NULL must be explicitly
  passed if you want it computed.
}
\value{
  Object of class \code{"power.htest"}, a list of the arguments
  (including the computed one) augmented with \code{method} and
  \code{note} elements. 
}

\author{Claus Ekstrm}
\note{
  \code{uniroot} is used to solve power equation for unknowns, so
  you may see errors from it, notably about inability to bracket the
  root when invalid arguments are given.
}
\seealso{\code{\link{anova}}, \code{\link{lm}}, \code{\link{uniroot}}}

\examples{
power.anova.test(groups=4, n=5, between.var=1, within.var=3)
# Power = 0.3535594

power.anova.test(groups=4, between.var=1, within.var=3,
                 power=.80)
# n = 11.92613

## Assume we have prior knowledge of the group means:
groupmeans <- c(120, 130, 140, 150)
power.anova.test(groups = length(groupmeans),
                 between.var=var(groupmeans),
                 within.var=500, power=.90) # n = 15.18834

}
\keyword{htest}
\eof
\name{power.prop.test}
\alias{power.prop.test}
\title{Power calculations two sample test for of proportions}
\description{
  Compute power of test, or determine parameters to obtain target power.
}
\usage{
power.prop.test(n=NULL, p1=NULL, p2=NULL, sig.level=0.05,
                power=NULL,
                alternative=c("two.sided", "one.sided"), strict=FALSE)
}
\arguments{
  \item{n}{Number of observations (per group)}
  \item{p1}{probability in one group}
  \item{p2}{probability in other group}
  \item{sig.level}{Significance level (Type I error probability)}
  \item{power}{Power of test (1 minus Type II error probability)}
  \item{alternative}{One- or two-sided test}
  \item{strict}{Use strict interpretation in two-sided case}
}
\details{
  Exactly one of the parameters \code{n}, \code{p1}, \code{p2},
  \code{power}, and \code{sig.level} must be passed as NULL, and that
  parameter is determined from the others. Notice that \code{sig.level}
  has a non-NULL default so NULL must be explicitly passed if you want
  it  computed.

  If \code{strict=TRUE} is used, the power will include the probability of
  rejection in the opposite direction of the true effect, in the two-sided 
  case. Without this the power will be half the significance level if the 
  true difference is zero.
}
\value{
  Object of class \code{"power.htest"}, a list of the arguments
  (including the computed one) augmented with \code{method} and
  \code{note} elements. 
}
\author{Peter Dalgaard.  Based on previous work by Claus Ekstrm}
\note{
  \code{uniroot} is used to solve power equation for unknowns, so
  you may see errors from it, notably about inability to bracket the
  root when invalid arguments are given. If one of them is computed
  \code{p1 < p2} will hold, although this is not enforced when both are
  specified.
}

\seealso{\code{\link{prop.test}}, \code{\link{uniroot}}}


\examples{
 power.prop.test(n=50, p1=.50, p2=.75)
 power.prop.test(p1=.50, p2=.75, power=.90)
 power.prop.test(n=50, p1=.5, power=.90)
}
\keyword{ htest }
\eof
\name{power.t.test}
\alias{power.t.test}
\title{Power calculations for one and two sample t tests}
\usage{
power.t.test(n=NULL, delta=NULL, sd=1, sig.level=0.05, power=NULL,
             type=c("two.sample", "one.sample", "paired"),
             alternative=c("two.sided", "one.sided"), strict=FALSE)
}
\arguments{
 \item{n}{Number of observations (per group)}
 \item{delta}{True difference in means}
 \item{sd}{Standard deviation}
 \item{sig.level}{Significance level (Type I error probability)}
 \item{power}{Power of test (1 minus Type II error probability)}
 \item{type}{Type of t test}
 \item{alternative}{One- or two-sided test}
 \item{strict}{Use strict interpretation in two-sided case}
}
\description{
 Compute power of test, or determine parameters to obtain target power.
}

\details{
  Exactly one of the parameters \code{n}, \code{delta}, \code{power},
  \code{sd}, and \code{sig.level} must be passed as NULL, and that
  parameter is determined from the others. Notice that the last two have
  non-NULL defaults so NULL must be explicitly passed if you want to
  compute them.

  If \code{strict=TRUE} is used, the power will include the probability of
  rejection in the opposite direction of the true effect, in the two-sided 
  case. Without this the power will be half the significance level if the 
  true difference is zero.
}
\value{
  Object of class \code{"power.htest"}, a list of the arguments
  (including the computed one) augmented with \code{method} and
  \code{note} elements. 
}

\author{Peter Dalgaard.  Based on previous work by Claus Ekstrm}
\note{
  \code{uniroot} is used to solve power equation for unknowns, so
  you may see errors from it, notably about inability to bracket the
  root when invalid arguments are given.
}
\seealso{\code{\link{t.test}}, \code{\link{uniroot}}}

\examples{
 power.t.test(n=20, delta=1)
 power.t.test(power=.90, delta=1)
 power.t.test(power=.90, delta=1, alt="one.sided")
}
\keyword{htest}


\eof
\name{PP.test}
\alias{PP.test}
\title{Phillips-Perron Test for Unit Roots}
\usage{
PP.test(x, lshort = TRUE)
}
\arguments{
  \item{x}{a numeric vector or univariate time series.}
  \item{lshort}{a logical indicating whether the short or long version
    of the truncation lag parameter is used.}
}
\description{
  Computes the Phillips-Perron test for the null hypothesis that
  \code{x} has a unit root against a stationary alternative.
}
\details{
  The general regression equation which incorporates a constant and a
  linear trend is used and the corrected t-statistic for a first order
  autoregressive coefficient equals one is computed.  To estimate
  \code{sigma^2} the Newey-West estimator is used.  If \code{lshort}
  is \code{TRUE}, then the truncation lag parameter is set to
  \code{trunc(4*(n/100)^0.25)}, otherwise
  \code{trunc(12*(n/100)^0.25)} is used.  The \eqn{p}-values are
  interpolated from Table 4.2, page 103 of Banerjee \emph{et al.}
  (1993).
  
  Missing values are not handled.
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the test statistic.}
  \item{parameter}{the truncation lag parameter.}
  \item{p.value}{the \eqn{p}-value of the test.}
  \item{method}{a character string indicating what type of test was
    performed.} 
  \item{data.name}{a character string giving the name of the data.}
}
\references{
  A. Banerjee, J. J. Dolado, J. W. Galbraith, and D. F. Hendry (1993)
  \emph{Cointegration, Error Correction, and the Econometric Analysis
    of Non-Stationary Data}, Oxford University Press, Oxford.
  
  P. Perron (1988) Trends and random walks in macroeconomic time
  series. \emph{Journal of Economic Dynamics and Control} \bold{12},
  297--332.
}
\author{A. Trapletti}
\examples{
x <- rnorm(1000)
PP.test(x)
y <- cumsum(x) # has unit root
PP.test(y)
}
\keyword{ts}
\eof
\name{ppoints}
\title{Ordinates for Probability Plotting}
\usage{
ppoints(n, a = ifelse(n <= 10, 3/8, 1/2))
}
\alias{ppoints}
\arguments{
  \item{n}{either the number of points generate or a vector of
    observations.}
  \item{a}{the offset fraction to be used; typically in \eqn{(0,1)}.}
}
\description{
  Generates the sequence of \dQuote{probability} points
  \code{(1:m - a)/(m + (1-a)-a)}
  where \code{m} is either \code{n}, if \code{length(n)==1}, or
  \code{length(n)}.
}
\details{
 If \eqn{0 < a < 1}, the resulting values are within \eqn{(0,1)}
 (excluding boundaries).
 In any case, the resulting sequence is symmetric in \eqn{[0,1]}, i.e.,
 \code{p + rev(p) == 1}.

 \code{ppoints()} is used in \code{qqplot} and \code{qqnorm} to generate
 the set of probabilities at which to evaluate the inverse distribution.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
\code{\link{qqplot}}, \code{\link{qqnorm}}.
}
\examples{
ppoints(4) # the same as  ppoints(1:4)
ppoints(10)
ppoints(10, a=1/2)
}
\keyword{dplot}
\keyword{arith}
\keyword{distribution}
\eof
% file stats/man/ppr.Rd
% copyright (C) 1995-8 B. D. Ripley
% copyright (C) 2000-3   The R Core Development Team
%
\name{ppr}
\alias{ppr}
\alias{ppr.default}
\alias{ppr.formula}
%\alias{predict.ppr}
%\alias{print.ppr}
%\alias{summary.ppr}
%\alias{print.summary.ppr}
\title{Projection Pursuit Regression}
\description{
  Fit a projection pursuit regression model.
}
\usage{
ppr(x, \dots)

\method{ppr}{formula}(formula, data, weights, subset, na.action,
    contrasts = NULL, \dots, model = FALSE)

\method{ppr}{default}(x, y, weights = rep(1,n),
    ww = rep(1,q), nterms, max.terms = nterms, optlevel = 2,
    sm.method = c("supsmu", "spline", "gcvspline"),
    bass = 0, span = 0, df = 5, gcvpen = 1, \dots)
}
\arguments{
  \item{formula}{
    a formula specifying one or more numeric response variables and the
    explanatory variables.
  }
  \item{x}{
    numeric matrix of explanatory variables.  Rows represent observations, and
    columns represent variables.  Missing values are not accepted.
  }
  \item{y}{
    numeric matrix of response variables.  Rows represent observations, and
    columns represent variables.  Missing values are not accepted.
  }
  \item{nterms}{number of terms to include in the final model.}
  \item{data}{
    data frame from which variables specified in \code{formula} are
    preferentially to be taken.
  }
  \item{weights}{a vector of weights \code{w_i} for each \emph{case}.}
  \item{ww}{
    a vector of weights for each \emph{response}, so the fit criterion is
    the sum over case \code{i} and responses \code{j} of
    \code{w_i ww_j (y_ij - fit_ij)^2} divided by the sum of \code{w_i}.
  }
  \item{subset}{
    an index vector specifying the cases to be used in the training
    sample.  (NOTE: If given, this argument must be named.)
  }
  \item{na.action}{
    a function to specify the action to be taken if \code{\link{NA}}s are
    found. The default action is given by \code{getOption("na.action")}.
    (NOTE: If given, this argument must be named.)
  }
  \item{contrasts}{
    the contrasts to be used when any factor explanatory variables are coded.
  }
  \item{max.terms}{
    maximum number of terms to choose from when building the model.
  }
  \item{optlevel}{
    integer from 0 to 3 which determines the thoroughness of an
    optimization routine in the SMART program. See the \bold{Details} section.
  }
  \item{sm.method}{
    the method used for smoothing the ridge functions.  The default is to
    use Friedman's super smoother \code{supsmu}.  The alternatives are to use
    the smoothing spline code underlying \code{smooth.spline}, either with a
    specified (equivalent) degrees of freedom for each ridge functions, or
    to allow the smoothness to be chosen by GCV.
  }
  \item{bass}{
    super smoother bass tone control used with automatic span selection
    (see \code{supsmu}); the range of values is 0 to 10, with larger values
    resulting in increased smoothing.
  }
  \item{span}{
    super smoother span control (see \code{supsmu}).  The default, \code{0},
    results in automatic span selection by local cross validation. \code{span}
    can also take a value in \code{(0, 1]}.
  }
  \item{df}{
    if \code{sm.method} is \code{"spline"} specifies the smoothness of
    each ridge term via the requested equivalent degrees of freedom.
  }
  \item{gcvpen}{
    if \code{sm.method} is \code{"gcvspline"} this is the penalty used
    in the GCV selection for each degree of freedom used.
  }
  \item{\dots}{arguments to be passed to or from other methods.}
  \item{model}{logical.  If true, the model frame is returned.}
}
\value{
A list with the following components, many of which are for use by the
method functions.

\item{call}{the matched call}
\item{p}{the number of explanatory variables (after any coding)}
\item{q}{the number of response variables}
\item{mu}{the argument \code{nterms}}
\item{ml}{the argument \code{max.terms}}
\item{gof}{the overall residual (weighted) sum of squares for the
  selected model}
\item{gofn}{the overall residual (weighted) sum of squares against the
  number of terms, up to \code{max.terms}.  Will be invalid (and zero)
  for less than \code{nterms}.}
\item{df}{the argument \code{df}}
\item{edf}{if \code{sm.method} is \code{"spline"} or \code{"gcvspline"}
  the equivalent number of degrees of freedom for each ridge term used.}
\item{xnames}{the names of the explanatory variables}
\item{ynames}{the names of the response variables}
\item{alpha}{a matrix of the projection directions, with a column for
  each ridge term}
\item{beta}{a matrix of the coefficients applied for each response to
  the ridge terms: the rows are the responses and the columns the ridge terms}
\item{yb}{the weighted means of each response}
\item{ys}{the overall scale factor used: internally the responses are
  divided by \code{ys} to have unit total weighted sum of squares.}
\item{fitted.values}{the fitted values, as a matrix if \code{q > 1}.}
\item{residuals}{the residuals, as a matrix if \code{q > 1}.}
\item{smod}{internal work array, which includes the ridge functions
  evaluated at the training set points.}
\item{model}{(only if \code{model=TRUE}) the model frame.}
}
\details{
  The basic method is given by Friedman (1984), and is essentially the
  same code used by S-PLUS's \code{ppreg}.  This code is extremely
  sensitive to the compiler used.

  The algorithm first adds up to \code{max.terms} ridge terms one at a
  time; it will use less if it is unable to find a term to add that makes
  sufficient difference.  It then removes the least
  \dQuote{\emph{important}} term at each step until \code{nterm} terms
  are left.

  The levels of optimization (argument \code{optlevel})
  differ in how thoroughly the models are refitted during this process.
  At level 0 the existing ridge terms are not refitted.  At level 1
  the projection directions are not refitted, but the ridge
  functions and the regression coefficients are.
%
  Levels 2 and 3 refit all the terms and are equivalent for one
  response; level 3 is more careful to re-balance the contributions
  from each regressor at each step and so is a little less likely to
  converge to a saddle point of the sum of squares criterion.
}
\references{
  Friedman, J. H. and Stuetzle, W. (1981)
  Projection pursuit regression.
  \emph{Journal of the American Statistical Association},
  \bold{76}, 817--823.

  Friedman, J. H. (1984)
  SMART User's Guide.
  Laboratory for Computational Statistics, Stanford University Technical
  Report No. 1.

  Venables, W. N. & Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.}  Springer.
}
\seealso{
  \code{\link{plot.ppr}}, \code{\link{supsmu}}, \code{\link{smooth.spline}}
}
\examples{
# Note: your numerical values may differ
data(rock)
attach(rock)
area1 <- area/10000; peri1 <- peri/10000
rock.ppr <- ppr(log(perm) ~ area1 + peri1 + shape,
                data = rock, nterms = 2, max.terms = 5)
rock.ppr
# Call:
# ppr.formula(formula = log(perm) ~ area1 + peri1 + shape, data = rock,
#     nterms = 2, max.terms = 5)
#
# Goodness of fit:
#  2 terms  3 terms  4 terms  5 terms
# 8.737806 5.289517 4.745799 4.490378

summary(rock.ppr)
# .....  (same as above)
# .....
#
# Projection direction vectors:
#       term 1      term 2
# area1  0.34357179  0.37071027
# peri1 -0.93781471 -0.61923542
# shape  0.04961846  0.69218595
#
# Coefficients of ridge terms:
#    term 1    term 2
# 1.6079271 0.5460971

par(mfrow=c(3,2))# maybe: , pty="s")
plot(rock.ppr, main="ppr(log(perm)~ ., nterms=2, max.terms=5)")
plot(update(rock.ppr, bass=5), main = "update(..., bass = 5)")
plot(update(rock.ppr, sm.method="gcv", gcvpen=2),
     main = "update(..., sm.method=\"gcv\", gcvpen=2)")
cbind(perm=rock$perm, prediction=round(exp(predict(rock.ppr)), 1))
detach()
}
\keyword{regression}

\eof
\name{prcomp}
\alias{prcomp}
\alias{plot.prcomp}
\alias{print.prcomp}
\alias{summary.prcomp}
\alias{print.summary.prcomp}
\concept{PCA}
\title{Principal Components Analysis}
\description{
  Performs a principal components analysis on the given data matrix
  and returns the results as an object of class \code{prcomp}.}
\usage{
prcomp(x, retx = TRUE, center = TRUE, scale. = FALSE, tol = NULL)
}
\arguments{
  \item{x}{a numeric or complex matrix (or data frame) which provides
    the data for the principal components analysis.}
  \item{retx}{a logical value indicating whether the rotated variables
    should be returned.}
  \item{center}{a logical value indicating whether the variables
    should be shifted to be zero centered. Alternately, a vector of
    length equal the number of columns of \code{x} can be supplied.
    The value is passed to \code{scale}.}
  \item{scale.}{a logical value indicating whether the variables should
    be scaled to have unit variance before the analysis takes
    place. The default is \code{FALSE} for consistency with S, but
    in general scaling is advisable.  Alternatively, a vector of length
    equal the number of columns of \code{x} can be supplied.  The
    value is passed to \code{\link{scale}}.}
  \item{tol}{a value indicating the magnitude below which components
    should be omitted. (Components are omitted if their
    standard deviations are less than or equal to \code{tol} times the
    standard deviation of the first component.)
    With the default null setting, no components
    are omitted.  Other settings for tol could be \code{tol = 0} or
    \code{tol = sqrt(.Machine$double.eps)}, which would omit
    essentially constant components.} 
}
\value{
  \code{prcomp} returns an list with class \code{"prcomp"}
  containing the following components:
  \item{sdev}{the standard deviations of the principal components
    (i.e., the square roots of the eigenvalues of the
    covariance/correlation matrix, though the calculation
    is actually done with the singular values of the data matrix).}
  \item{rotation}{the matrix of variable loadings (i.e., a matrix
    whose columns contain the eigenvectors).  The function
    \code{princomp} returns this in the element \code{loadings}.}
  \item{x}{if \code{retx} is true the value of the rotated data (the
    centred (and scaled if requested) data multiplied by the
    \code{rotation} matrix) is returned.}
}
\details{
  The calculation is done by a singular value decomposition of the
  (centered and possibly scaled) data matrix, not by using
  \code{eigen} on the covariance matrix.  This
  is generally the preferred method for numerical accuracy.  The
  \code{print} method for the these objects prints the results in a nice
  format and the \code{plot} method produces a scree plot.
}
\note{
  The signs of the columns of the rotation matrix are arbitrary, and
  so may differ between different programs for PCA, and even between
  different builds of \R.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.

  Mardia, K. V., J. T. Kent, and J. M. Bibby (1979)
  \emph{Multivariate Analysis}, London: Academic Press.

  Venables, W. N. and B. D. Ripley (1997, 9)
  \emph{Modern Applied Statistics with S-PLUS}, Springer-Verlag.
}
\seealso{
  \code{\link{biplot.prcomp}},
  \code{\link{princomp}}, \code{\link{cor}}, \code{\link{cov}},
  \code{\link{svd}}, \code{\link{eigen}}.
}
\examples{
## the variances of the variables in the
## USArrests data vary by orders of magnitude, so scaling is appropriate
data(USArrests)
prcomp(USArrests)  # inappropriate
prcomp(USArrests, scale = TRUE)
plot(prcomp(USArrests))
summary(prcomp(USArrests, scale = TRUE))
biplot(prcomp(USArrests, scale = TRUE))
}
\keyword{multivariate}
\eof
\name{predict.HoltWinters}
\alias{predict.HoltWinters}
\title{prediction function for fitted Holt-Winters models}
\description{
  Computes predictions and prediction intervals for models fitted by
  the Holt-Winters method.
}
\usage{
\method{predict}{HoltWinters}(object, n.ahead=1, prediction.interval = FALSE,
                    level = 0.95, ...)
}
\arguments{
  \item{object}{An object of class \code{HoltWinters}.}
  \item{n.ahead}{Number of future periods to predict.}
  \item{prediction.interval}{logical. If \code{TRUE}, the lower and
    upper bounds of the corresponding prediction intervals are computed.}
  \item{level}{Confidence level for the prediction interval.}
  \item{\dots}{arguments passed to or from other methods.}
}
\value{
  A time series of the predicted values. If prediction intervals are
  requested, a multiple time series is returned with columns \code{fit},
  \code{lwr} and \code{upr} for the predicted values and the lower and
  upper bounds respectively.
}
\references{
  C.C Holt (1957)
  Forecasting seasonals and trends by exponentially weighted
  moving averages,
  ONR Research Memorandum, Carnigie Institute 52.
  
  P.R Winters (1960)
  Forecasting sales by exponentially weighted moving averages,
  \emph{Management Science} \bold{6}, 324--342.
}
\author{
  David Meyer \email{david.meyer@ci.tuwien.ac.at}
}
\seealso{\code{\link{HoltWinters}}}

\examples{
data(co2)
m <- HoltWinters(co2)
p <- predict(m, 50, prediction.interval = TRUE)
plot(m, p)
}
\keyword{ts}


\eof
\name{predict}
\title{Model Predictions}
\usage{
predict (object, \dots)
}
\alias{predict}
\arguments{
  \item{object}{a model object for which prediction is desired.}
  \item{\dots}{additional arguments affecting the predictions produced.}
}
\description{
  \code{predict} is a generic function for predictions from the results of
  various model fitting functions.  The function invokes particular
  \emph{methods} which depend on the \code{\link[base]{class}} of
  the first argument.
}
\details{
  Most prediction methods which similar to fitting linear models have an
  argument \code{newdata} specifiying the explanatory variables to be
  used for prediction.  Some considerable attempts are made to match up
  the columns in \code{newdata} to those used for fitting, for example
  that they are of comparable types and that any factors have the same
  level set in the same order (or can be transformed to be so).

  Time series prediction methods have an argument \code{n.ahead}
  specifying how many time steps ahead to predict.

  Many methods have a logical argument \code{se.fit} saying if standard
  errors are to returned.
}
 \value{
  The form of the value returned by \code{predict} depends on the
  class of its argument.  See the documentation of the
  particular methods for details of what is produced by that method.
}
\seealso{
  \code{\link{predict.glm}},
  \code{\link{predict.lm}},
  \code{\link{predict.loess}},
  \code{\link{predict.nls}},
  \code{\link{predict.poly}},
  \code{\link[stats:princomp]{predict.princomp}},
  \code{\link{predict.smooth.spline}}.

  For time-series prediction,
  \code{\link[stats:ar]{predict.ar}},
  \code{\link{predict.Arima}},
  \code{\link[stats:arima0]{predict.arima0}},
  \code{\link{predict.HoltWinters}},
  \code{\link[stats:StructTS]{predict.StructTS}}.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical Models in S}.
  Wadsworth \& Brooks/Cole.
}
\examples{
## All the "predict" methods found
## NB most of the methods in the standard packages are hidden.
for(fn in methods("predict"))
   try({
       f <- eval(substitute(getAnywhere(fn)$objs[[1]], list(fn = fn)))
       cat(fn, ":\n\t", deparse(args(f)), "\n")
       }, silent = TRUE)
}
\keyword{methods}
\eof
\name{predict.Arima}
\alias{predict.Arima}
\title{Forecast from ARIMA fits}
\description{
  Forecast from models fitted by \code{\link{arima}}.
}
\usage{
\method{predict}{Arima}(object, n.ahead = 1, newxreg = NULL,
        se.fit = TRUE, \dots)
}
\arguments{
  \item{object}{The result of an \code{arima} fit.}
  \item{n.ahead}{The number of steps ahead for which prediction is required.}
  \item{newxreg}{New values of \code{xreg} to be used for
    prediction. Must have at least \code{n.ahead} rows.}
  \item{se.fit}{Logical: should standard errors of prediction be returned?}
  \item{\dots}{arguments passed to or from other methods.}
}
\details{
  Finite-history prediction is used, via \code{\link{KalmanForecast}}.
  This is only statistically efficient if the MA part of the fit is
  invertible, so \code{predict.Arima} will give a warning for
  non-invertible MA models.

  The standard errors of prediction exclude the uncertainty in the
  estimation of the ARMA model and the regression coefficients.
  According to Harvey (1993, pp. 58--9) the effect is small.
}
\value{
  A time series of predictions, or if \code{se.fit = TRUE}, a list
  with components \code{pred}, the predictions, and \code{se},
  the estimated standard errors.  Both components are time series.
}
\references{
  Durbin, J. and Koopman, S. J. (2001) \emph{Time Series Analysis by
    State Space Methods.}  Oxford University Press.
  
  Harvey, A. C. and McKenzie, C. R. (1982) Algorithm AS182.
  An algorithm for finite sample prediction from ARIMA processes.
  \emph{Applied Statistics} \bold{31}, 180--187.

  Harvey, A. C. (1993) \emph{Time Series Models},
  2nd Edition, Harvester Wheatsheaf, sections 3.3 and 4.4.
}

\seealso{
  \code{\link{arima}}
}

\examples{
data(lh)
predict(arima(lh, order = c(3,0,0)), n.ahead = 12)

data(USAccDeaths)
(fit <- arima(USAccDeaths, order = c(0,1,1),
              seasonal = list(order=c(0,1,1))))
predict(fit, n.ahead = 6)
}
\keyword{ts}
\eof
\name{predict.glm}
\alias{predict.glm}
\concept{regression}
\title{Predict Method for GLM Fits}
\usage{
\method{predict}{glm}(object, newdata = NULL,
            type = c("link", "response", "terms"),
            se.fit = FALSE, dispersion = NULL, terms = NULL,
            na.action = na.pass, \dots)
}
\arguments{
  \item{object}{a fitted object of class inheriting from \code{"glm"}.}
  \item{newdata}{optionally, a new data frame from which to make the
    predictions.  If omitted, the fitted linear predictors are used.}
  \item{type}{the type of prediction required.  The default is on the
    scale of the linear predictors; the alternative \code{"response"}
    is on the scale of the response variable.  Thus for a default
    binomial model the default predictions are of log-odds (probabilities
    on logit scale) and \code{type = "response"} gives the predicted
    probabilities.  The \code{"terms"} option returns a matrix giving the
    fitted values of each term in the model formula on the linear predictor
    scale.

    The value of this argument can be abbreviated.
  }
  \item{se.fit}{logical switch indicating if standard errors are required.}
  \item{dispersion}{the dispersion of the GLM fit to be assumed in
    computing the standard errors.  If omitted, that returned by
    \code{summary} applied to the object is used.}
  \item{terms}{with \code{type="terms"} by default all terms are returned.
    A character vector specifies which terms are to be returned}
  \item{na.action}{function determining what should be done with missing
    values in \code{newdata}.  The default is to predict \code{NA}.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  Obtains predictions and optionally estimates standard errors of those
  predictions from a fitted generalized linear model object.
}
\value{
  If \code{se = FALSE}, a vector or matrix of predictions.  If \code{se
    = TRUE}, a list with components
  \item{fit}{Predictions}
  \item{se.fit}{Estimated standard errors}
  \item{residual.scale}{A scalar giving the square root of the
    dispersion used in computing the standard errors.}
}

\seealso{
  \code{\link{glm}}, \code{\link{SafePrediction}}
}

\examples{
## example from Venables and Ripley (2002, pp. 190-2.)
ldose <- rep(0:5, 2)
numdead <- c(1, 4, 9, 13, 18, 20, 0, 2, 6, 10, 12, 16)
sex <- factor(rep(c("M", "F"), c(6, 6)))
SF <- cbind(numdead, numalive=20-numdead)
budworm.lg <- glm(SF ~ sex*ldose, family=binomial)
summary(budworm.lg)

plot(c(1,32), c(0,1), type = "n", xlab = "dose",
     ylab = "prob", log = "x")
text(2^ldose, numdead/20, as.character(sex))
ld <- seq(0, 5, 0.1)
lines(2^ld, predict(budworm.lg, data.frame(ldose=ld,
   sex=factor(rep("M", length(ld)), levels=levels(sex))),
   type = "response"))
lines(2^ld, predict(budworm.lg, data.frame(ldose=ld,
   sex=factor(rep("F", length(ld)), levels=levels(sex))),
   type = "response"))
}
\keyword{models}
\keyword{regression}
\eof
\name{predict.lm}
\title{Predict method for Linear Model Fits}
\alias{predict.lm}
\alias{predict.mlm}
\concept{regression}
\description{
  Predicted values based on linear model object
}
\usage{
\method{predict}{lm}(object, newdata, se.fit = FALSE, scale = NULL, df = Inf, 
        interval = c("none", "confidence", "prediction"),
        level = 0.95, type = c("response", "terms"),
        terms = NULL, na.action = na.pass, \dots)
}
\arguments{
  \item{object}{Object of class inheriting from \code{"lm"}}
  \item{newdata}{Data frame in which to predict}
  \item{se.fit}{A switch indicating if standard errors are required.}
  \item{scale}{Scale parameter for std.err. calculation}
  \item{df}{Degrees of freedom for scale}
  \item{interval}{Type of interval calculation}
  \item{level}{Tolerance/confidence level}
  \item{type}{Type of prediction (response or model term)}
  \item{terms}{If \code{type="terms"}, which terms (default is all terms)}
  \item{na.action}{function determining what should be done with missing
    values in \code{newdata}.  The default is to predict \code{NA}.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\details{
  \code{predict.lm} produces predicted values, obtained by evaluating
  the regression function in the frame \code{newdata} (which defaults to
  \code{model.frame(object)}.  If the logical \code{se.fit} is
  \code{TRUE}, standard errors of the predictions are calculated.  If
  the numeric argument \code{scale} is set (with optional \code{df}), it
  is used as the residual standard deviation in the computation of the
  standard errors, otherwise this is extracted from the model fit.
  Setting \code{intervals} specifies computation of confidence or
  prediction (tolerance) intervals at the specified \code{level}.

  If the fit is rank-deficient, some of the columns of the design matrix
  will have been dropped.  Prediction from such a fit only makes sense
  if \code{newdata} is contained in the same subspace as the original
  data. That cannot be checked accurately, so a warning is issued.
}
\value{
  \code{predict.lm} produces a vector of predictions or a matrix of
  predictions and bounds with column names \code{fit}, \code{lwr}, and
  \code{upr} if \code{interval} is set.  If \code{se.fit} is
  \code{TRUE}, a list with the following components is returned: 
  \item{fit}{vector or matrix as above}
  \item{se.fit}{standard error of predictions}
  \item{residual.scale}{residual standard deviations}
  \item{df}{degrees of freedom for residual}
}
\note{
  Offsets specified by \code{offset} in the fit by \code{\link{lm}}
  will not be included in predictions, whereas those specified by an
  offset term in the formula will be.
}
\seealso{
  The model fitting function \code{\link{lm}}, \code{\link{predict}},
  \code{\link{SafePrediction}}
}
\examples{
## Predictions
x <- rnorm(15)
y <- x + rnorm(15)
predict(lm(y ~ x))
new <- data.frame(x = seq(-3, 3, 0.5))
predict(lm(y ~ x), new, se.fit = TRUE)
pred.w.plim <- predict(lm(y ~ x), new, interval="prediction")
pred.w.clim <- predict(lm(y ~ x), new, interval="confidence")
matplot(new$x,cbind(pred.w.clim, pred.w.plim[,-1]),
        lty=c(1,2,2,3,3), type="l", ylab="predicted y")
}
\keyword{regression}

\eof
\name{predict.loess}
\alias{predict.loess}
\title{Predict Loess Curve or Surface}
\description{
 Predictions from a \code{loess} fit, optionally with standard errors.
}
\usage{
\method{predict}{loess}(object, newdata = NULL, se = FALSE, \dots)
}
\arguments{
 \item{object}{an object fitted by \code{loess}.}
 \item{newdata}{an optional data frame specifying points at which to do
   the predictions. If missing, the original data points are used.}
 \item{se}{should standard errors be computed?}
 \item{\dots}{arguments passed to or from other methods.}
}
\details{
  The standard errors calculation is slower than prediction.

  When the fit was made using \code{surface="interpolate"} (the
  default), \code{predict.loess} will not extrapolate -- so points outside
  an axis-aligned hypercube enclosing the original data will have
  missing (\code{NA}) predictions and standard errors.
}
\value{
  If \code{se = FALSE}, a vector giving the prediction for each row of
  \code{newdata} (or the original data). If \code{se = TRUE}, a list
  containing components
  \item{fit}{the predicted values.}
  \item{se}{an estimated standard error for each predicted value.}
  \item{residual.scale}{the estimated scale of the residuals used in
    computing the standard errors.}
  \item{df}{an estimate of the effective degrees of freedom used in
    estimating the residual scale, intended for use with t-based
    confidence intervals. }
  If \code{newdata} was the result of a call to
  \code{\link{expand.grid}}, the predictions (and s.e.'s if requested)
  will be an array of the appropriate dimensions.
}
\author{
  B.D. Ripley, based on the \code{cloess} package of Cleveland,
  Grosse and Shyu.
}

\seealso{\code{\link{loess}}}

\examples{
data(cars)
cars.lo <- loess(dist ~ speed, cars)
predict(cars.lo, data.frame(speed=seq(5, 30, 1)), se=TRUE)
# to get extrapolation
cars.lo2 <- loess(dist ~ speed, cars,
  control=loess.control(surface="direct"))
predict(cars.lo2, data.frame(speed=seq(5, 30, 1)), se=TRUE)
}
\keyword{smooth}
\eof
\name{predict.nls}
\title{Predicting from Nonlinear Least Squares Fits}
\alias{predict.nls}
\usage{
\method{predict}{nls}(object, newdata , se.fit = FALSE, scale = NULL, df = Inf, 
        interval = c("none", "confidence", "prediction"),
        level = 0.95, \dots)
}
\description{
  \code{predict.nls} produces predicted values, obtained by evaluating
  the regression function in the frame \code{newdata}.  If the logical
  \code{se.fit} is \code{TRUE}, standard errors of the predictions are
  calculated.  If the numeric argument \code{scale} is set (with
  optional \code{df}), it is used as the residual standard deviation in
  the computation of the standard errors, otherwise this is extracted
  from the model fit.  Setting \code{intervals} specifies computation of
  confidence or prediction (tolerance) intervals at the specified
  \code{level}.

  At present \code{se.fit} and \code{interval} are ignored.
}
\arguments{
 \item{object}{An object that inherits from class \code{nls}.}
 \item{newdata}{A named list or data frame with values of the input
     variables for the model in \code{object}.  If \code{newdata} is
     missing the fitted values at the original data points are returned.}
 \item{se.fit}{A logical value indicating if the standard errors of the
     predictions should be calculated.  Defaults to \code{FALSE}.  At
     present this argument is ignored.}
 \item{scale}{A numeric scalar.  If it is set (with optional
     \code{df}), it is used as the residual standard deviation in the
     computation of the standard errors, otherwise this information is
     extracted from the model fit. At present this argument is ignored.}
 \item{df}{A positive numeric scalar giving the number of degrees of
     freedom for the \code{scale} estimate. At present this argument is
     ignored.}
 \item{interval}{A character string indicating if prediction intervals
     or a confidence interval on the mean responses are to be
     calculated. At present this argument is ignored.}
 \item{level}{A numeric scalar between 0 and 1 giving the confidence
     level for the intervals (if any) to be calculated.  At present
     this argument is ignored.} 
 \item{\dots}{Additional optional arguments.  At present no optional
     arguments are used.}
}
\value{
  \code{predict.nls} produces a vector of predictions.
  When implemented, \code{interval} will produce a matrix of
  predictions and bounds with column names \code{fit}, \code{lwr}, and
  \code{upr}.  When implemented, if \code{se.fit} is
  \code{TRUE}, a list with the following components will be returned:

  \item{fit}{vector or matrix as above}
  \item{se.fit}{standard error of predictions}
  \item{residual.scale}{residual standard deviations}
  \item{df}{degrees of freedom for residual}
}
\seealso{
  The model fitting function \code{\link{nls}},
  \code{\link{predict}}.
}
\examples{
data( BOD )
fm <- nls(demand ~ SSasympOrig(Time, A, lrc), data = BOD)
predict(fm)              # fitted values at observed times
## Form data plot and smooth line for the predictions
opar <- par(las = 1)
plot(demand ~ Time, data = BOD, col = 4,
     main = "BOD data and fitted first-order curve",
     xlim = c(0,7), ylim = c(0, 20) )
tt <- seq(0, 8, length = 101)
lines(tt, predict(fm, list(Time = tt)))
par(opar)
}
\keyword{nonlinear}
\keyword{regression}
\keyword{models}
\eof
\name{predict.smooth.spline}
\alias{predict.smooth.spline}
%\alias{predict.smooth.spline.fit}
\title{Predict from Smoothing Spline Fit}
\description{
  Predict a smoothing spline fit at new points, return the derivative if
  desired. The predicted fit is linear beyond the original data.
}
\usage{
\method{predict}{smooth.spline}(object, x, deriv = 0, \dots)
}
\arguments{
  \item{object}{a fit from \code{smooth.spline}.}
  \item{x}{the new values of x.}
  \item{deriv}{integer; the order of the derivative required.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\value{
  A list with components
  \item{x}{The input \code{x}.}
  \item{y}{The fitted values or derivatives at \code{x}.}
}
\seealso{\code{\link{smooth.spline}}
}
\examples{
data(cars)
attach(cars)
cars.spl <- smooth.spline(speed, dist, df=6.4)
\dontshow{
print.default(cars.spl)
}

## "Proof" that the derivatives are okay, by comparing with approximation
diff.quot <- function(x,y) {
  ## Difference quotient (central differences where available)
  n <- length(x); i1 <- 1:2; i2 <- (n-1):n
  c(diff(y[i1]) / diff(x[i1]), (y[-i1] - y[-i2]) / (x[-i1] - x[-i2]),
    diff(y[i2]) / diff(x[i2]))
}

xx <- unique(sort(c(seq(0,30, by = .2), kn <- unique(speed))))
i.kn <- match(kn, xx)# indices of knots within xx
op <- par(mfrow = c(2,2))
plot(speed, dist, xlim = range(xx), main = "Smooth.spline & derivatives")
lines(pp <- predict(cars.spl, xx), col = "red")
points(kn, pp$y[i.kn], pch = 3, col="dark red")
mtext("s(x)", col = "red")
for(d in 1:3){
  n <- length(pp$x)
  plot(pp$x, diff.quot(pp$x,pp$y), type = 'l', xlab="x", ylab="",
       col = "blue", col.main = "red",
       main= paste("s",paste(rep("'",d), collapse=""),"(x)", sep=""))
  mtext("Difference quotient approx.(last)", col = "blue")
  lines(pp <- predict(cars.spl, xx, deriv = d), col = "red")
\dontshow{
  print(pp)
}
  points(kn, pp$y[i.kn], pch = 3, col="dark red")
  abline(h=0, lty = 3, col = "gray")
}
detach(); par(op)
}
\keyword{smooth}
\eof
\name{preplot}
\alias{preplot}
\title{Pre-computations for a Plotting Objeect}
\description{
    Compute an object to be used for plots relating to the given model object.
}
\usage{
preplot(object, \dots)
}
\arguments{
 \item{object}{a fitted model object.}
 \item{\dots}{additional arguments for specific methods.}
}
\details{
    Only the generic function is currently provided in base \R, but some
    add-on packages have methods. Principally here for S compatibility. 
}
\value{
    An object set up to make a plot that describes \code{object}.
}
\keyword{models}
\eof
\name{presidents}
\docType{data}
\alias{presidents}
\title{Quarterly Approval Ratings of US Presidents}
\description{
  The (approximately) quarterly approval rating for the President of the
  United states from the first quarter of 1945 to the last quarter of
  1974.
}
\usage{data(presidents)}
\format{
  A time series of 120 values.
}
\source{
  The Gallup Organisation.
}
\details{
  The data are actually a fudged version of the approval ratings.  See
  McNeil's book for details.
}
\references{
  McNeil, D. R. (1977)
  \emph{Interactive Data Analysis}.
  New York: Wiley.
}
\examples{
data(presidents)
plot(presidents, las = 1, ylab = "Approval rating (\%)",
     main = "presidents data")
}
\keyword{datasets}
\eof
\name{princomp}
\alias{princomp}
\alias{princomp.formula}
\alias{princomp.default}
\alias{plot.princomp}
\alias{print.princomp}
\alias{predict.princomp}
\title{Principal Components Analysis}
\concept{PCA}
\usage{
princomp(x, \dots)

\method{princomp}{formula}(formula, data = NULL, subset, na.action, \dots)

\method{princomp}{default}(x, cor = FALSE, scores = TRUE, covmat = NULL,
         subset = rep(TRUE, nrow(as.matrix(x))), \dots)

\method{predict}{princomp}(object, newdata, \dots)
}
\arguments{
  \item{formula}{a formula with no response variable.}
  \item{data}{an optional data frame containing the variables in the
    formula \code{formula}.  By default the variables are taken from
    \code{environment(formula)}.}
  \item{x}{a numeric matrix or data frame which provides the data for the
    principal components analysis.}
  \item{subset}{an optional vector used to select rows (observations) of the
    data matrix \code{x}.}
  \item{na.action}{a function which indicates what should happen
    when the data contain \code{NA}s.  The default is set by
    the \code{na.action} setting of \code{\link{options}}, and is
    \code{\link{na.fail}} if that is unset. The \dQuote{factory-fresh}
    default is \code{\link{na.omit}}.}
  \item{cor}{a logical value indicating whether the calculation should
    use the correlation matrix or the covariance matrix.}
  \item{scores}{a logical value indicating whether the score on each
    principal component should be calculated.}
  \item{covmat}{a covariance matrix, or a covariance list as returned by
    \code{\link{cov.wt}}, \code{\link[MASS]{cov.mve}} or
    \code{\link[MASS]{cov.mcd}}.
    If supplied, this is used rather than the covariance matrix of
    \code{x}.}
  \item{\dots}{arguments passed to or from other methods. If \code{x} is
    a formula one might specify \code{cor} or \code{scores}.}
  \item{object}{Object of class inheriting from \code{"princomp"}}
  \item{newdata}{Data frame in which to predict}
}
\description{
  \code{princomp} performs a principal components analysis on the given
  numeric data matrix and returns the results as an object of class
  \code{princomp}.
}
\value{
  \code{princomp} returns a list with class \code{"princomp"}
  containing the following components:
  \item{sdev}{the standard deviations of the principal components.}
  \item{loadings}{the matrix of variable loadings (i.e., a matrix
    whose columns contain the eigenvectors).  This is of class
    \code{"loadings"}: see \code{\link{loadings}} for its \code{print}
    method.}
  \item{center}{the means that were subtracted.}
  \item{scale}{the scalings applied to each variable.}
  \item{n.obs}{the number of observations.}
  \item{scores}{if \code{scores = TRUE}, the scores of the supplied
    data on the principal components.  These are non-null only if
    \code{x} was supplied, and if \code{covmat} was also supplied if it
    was a covariance list.}
  \item{call}{the matched call.}
  \item{na.action}{If relevant.}
}
\details{
  \code{princomp} is a generic function with \code{"formula"} and
  \code{"default"} methods.

  The calculation is done using \code{\link{eigen}} on the correlation or
  covariance matrix, as determined by \code{\link{cor}}.  This is done for
  compatibility with the S-PLUS result.  A preferred method of
  calculation is to use \code{\link{svd}} on \code{x}, as is done in
  \code{prcomp}.

  Note that the default calculation uses divisor \code{N} for the
  covariance matrix.

  The \code{\link{print}} method for the these objects prints the
  results in a nice format and the \code{\link{plot}} method produces
  a scree plot (\code{\link{screeplot}}).  There is also a
  \code{\link{biplot}} method.

  If \code{x} is a formula then the standard NA-handling is applied to
  the scores (if requested): see \code{\link{napredict}}.

  \code{princomp} only handles so-called R-mode PCA, that is feature
  extraction of variables. If a data matrix is supplied (possibly via a
  formula) it is required that there are at least as many units as
  variables.  For Q-mode PCA use \code{\link{prcomp}}.
}
\note{
  The signs of the columns of the loadings and scores are arbitrary, and
  so may differ between different programs for PCA, and even between
  different builds of \R.
}
\references{
  Mardia, K. V., J. T. Kent and J. M. Bibby (1979).
  \emph{Multivariate Analysis}, London: Academic Press.

  Venables, W. N. and B. D. Ripley (2002).
  \emph{Modern Applied Statistics with S}, Springer-Verlag.
}
\seealso{
  \code{\link{summary.princomp}}, \code{\link{screeplot}},
  \code{\link{biplot.princomp}},
  \code{\link{prcomp}}, \code{\link{cor}}, \code{\link{cov}},
  \code{\link{eigen}}.
}
\examples{
## The variances of the variables in the
## USArrests data vary by orders of magnitude, so scaling is appropriate
data(USArrests)
(pc.cr <- princomp(USArrests))  # inappropriate
princomp(USArrests, cor = TRUE) # =^= prcomp(USArrests, scale=TRUE)
## Similar, but different:
## The standard deviations differ by a factor of sqrt(49/50)

summary(pc.cr <- princomp(USArrests, cor = TRUE))
loadings(pc.cr)  ## note that blank entries are small but not zero
plot(pc.cr) # shows a screeplot.
biplot(pc.cr)

## Formula interface
princomp(~ ., data = USArrests, cor = TRUE)
# NA-handling
USArrests[1, 2] <- NA
pc.cr <- princomp(~ Murder + Assault + UrbanPop,
                  data = USArrests, na.action=na.exclude, cor = TRUE)
pc.cr$scores
}
\keyword{multivariate}
\eof
\name{print.power.htest}
\alias{print.power.htest}
\title{Print method for power calculation object}
\description{
  Print object of class \code{"power.htest"} in nice layout.
}
\usage{
\method{print}{power.htest}(x, \dots)
}
\arguments{
  \item{x}{Object of class \code{"power.htest"}.}
  \item{\dots}{further arguments to be passed to or from methods.}  
}
\details{
  A \code{power.htest} object is just a named list of numbers and
  character strings, supplemented with \code{method} and \code{note}
  elements.  The \code{method} is displayed as a title, the \code{note}
  as a footnote, and the remaining elements are given in an aligned
  \sQuote{name = value} format.
}
\value{
  none
}
\author{Peter Dalgaard}

\seealso{
  \code{\link{power.t.test}},
  \code{\link{power.prop.test}}
}
\keyword{htest}
\eof
\name{print.ts}
\title{Printing Time-Series Objects}
\alias{print.ts}
\description{
  Print method for time series objects.
}
\usage{
\method{print}{ts}(x, calendar, \dots)
}
\arguments{
  \item{x}{a time series object.}
  \item{calendar}{enable/disable the display of information about
    month names, quarter names or year when printing.  The default is
    \code{TRUE} for a frequency of 4 or 12, \code{FALSE} otherwise.}
  \item{\dots}{additional arguments to \code{\link{print}}.}
}
\details{
  This is the \code{\link{print}} methods for objects inheriting from
  class \code{"ts"}.
}
\seealso{
  \code{\link{print}},
  \code{\link{ts}}.
}
\examples{
print(ts(1:10, freq = 7, start = c(12, 2)), calendar = TRUE)
}
\keyword{ts}
\eof
\name{printCoefmat}
\alias{printCoefmat}
\title{Print Coefficient Matrices}
\description{
  Utility function to be used in \dQuote{higher level} \code{\link{print}}
  methods, such as \code{\link{print.summary.lm}},
  \code{\link{print.summary.glm}} and \code{\link{print.anova}}.  The
  goal is to provide a flexible interface with smart defaults such
  that often, only \code{x} needs to be specified.
}
\usage{
printCoefmat(x, digits=max(3, getOption("digits") - 2),
             signif.stars = getOption("show.signif.stars"),
             dig.tst = max(1, min(5, digits - 1)),
             cs.ind = 1:k, tst.ind = k + 1, zap.ind = integer(0),
             P.values = NULL,
             has.Pvalue = nc >= 4 && substr(colnames(x)[nc],1,3) == "Pr(",
             eps.Pvalue = .Machine$double.eps,
             na.print = "NA", \dots)
}
\arguments{
  \item{x}{a numeric matrix like object, to be printed.}
  \item{digits}{minimum number of significant digits to be used for
    most numbers.}
  \item{signif.stars}{logical; if \code{TRUE}, P-values are additionally
    encoded visually as \dQuote{significance stars} in order to help scanning
    of long coefficient tables.  It defaults to the
    \code{show.signif.stars} slot of \code{\link{options}}.}
  \item{dig.tst}{minimum number of significant digits for the test statistics,
    see \code{tst.ind}.}
  \item{cs.ind}{indices (integer) of column numbers which are (like)
    \bold{c}oefficients and \bold{s}tandard errors to be formatted
    together.}
  \item{tst.ind}{indices (integer) of column numbers for test
    statistics.}
  \item{zap.ind}{indices (integer) of column numbers which should be
    formatted by \code{\link{zapsmall}}, i.e., by \dQuote{zapping} values
    close to 0.}
  \item{P.values}{logical or \code{NULL}; if \code{TRUE}, the last
    column of \code{x} is formatted by \code{\link{format.pval}} as P
    values.  If \code{P.values = NULL}, the default, it is set to
    \code{TRUE} only if \code{\link{options}("show.coef.Pvalue")} is
    \code{TRUE} \emph{and} \code{x} has at least 4 columns \emph{and}
    the last column name of \code{x} starts with \code{"Pr("}.}
  \item{has.Pvalue}{logical; if \code{TRUE}, the last column of \code{x}
    contains P values; in that case, it is printed if and only if
    \code{P.values} (above) is true.}
  \item{eps.Pvalue}{number,..}
  \item{na.print}{a character string to code \code{\link{NA}} values in
    printed output.}
 \item{\dots}{further arguments for \code{\link{print}}.}
}
\value{
  Invisibly returns its argument, \code{x}.
}
\author{Martin Maechler}
\seealso{
  \code{\link{print.summary.lm}},
  \code{\link{format.pval}},
  \code{\link{format}}.
}
\examples{
cmat <- cbind(rnorm(3, 10), sqrt(rchisq(3, 12)))
cmat <- cbind(cmat, cmat[,1]/cmat[,2])
cmat <- cbind(cmat, 2*pnorm(-cmat[,3]))
colnames(cmat) <- c("Estimate", "Std.Err", "Z value", "Pr(>z)")
printCoefmat(cmat[,1:3])
printCoefmat(cmat)
options(show.coef.Pvalues = FALSE)
printCoefmat(cmat, digits=2)
printCoefmat(cmat, digits=2, P.values = TRUE)
options(show.coef.Pvalues = TRUE)# revert
}
\keyword{print}
\eof
\name{profile}
\alias{profile}
\title{Generic Function for Profiling Models}
\usage{
profile(fitted, \dots)
}
\arguments{
  \item{fitted}{the original fitted model object.}
  \item{\dots}{additional parameters. See documentation on individual
    methods.}
}
\description{
  Investigates behavior of objective function near the solution
  represented by \code{fitted}.

  See documentation on method functions for further details.
}
\value{
  A list with an element for each parameter being profiled. See the
  individual methods for further details.
}
\seealso{
  \code{\link{profile.nls}},
  \code{\link[MASS:confint]{profile.glm}} in package \pkg{MASS},
  \ldots

  For profiling code, see \code{\link{Rprof}}.
}
\keyword{models}
\eof
% $Id: profile.nls.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{profile.nls}
\alias{profile.nls}
\title{ Method for Profiling nls Objects}
\usage{
\method{profile}{nls}(fitted, which=1:npar, maxpts=100, alphamax=0.01,
        delta.t=cutoff/5, \dots)
}
\arguments{
  \item{fitted}{ the original fitted model object.}
  \item{which}{ the original model parameters which should be
    profiled. By default, all parameters are profiled.}
  \item{maxpts}{ maximum number of points to be used for profiling each
    parameter.}
  \item{alphamax}{ maximum significance level allowed for the profile
    t-statistics.}
  \item{delta.t}{ suggested change on the scale of the profile
    t-statistics. Default value chosen to allow profiling at about
    10 parameter values.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  Investigates behavior of the log-likelihood function near the solution
  represented by \code{fitted}.
}
\value{
  A list with an element for each parameter being profiled. The elements
  are data-frames with two variables 
  \item{par.vals}{ a matrix of parameter values for each fitted model.}
  \item{tau}{ The profile t-statistics.}
}
\details{
  The profile t-statistics is defined as the square root of change in
  sum-of-squares divided by residual standard error with an
  appropriate sign.
}
\references{
  Bates, D.M. and Watts, D.G. (1988), \emph{Nonlinear Regression Analysis
	and Its Applications}, Wiley (chapter 6)}
\author{Douglas M. Bates and Saikat DebRoy}
\seealso{\code{\link{nls}}, \code{\link{profile}},
    \code{\link{profiler.nls}}, \code{\link{plot.profile.nls}}
}
\examples{
data( BOD )
# obtain the fitted object
fm1 <- nls(demand ~ SSasympOrig( Time, A, lrc ), data = BOD)
# get the profile for the fitted model
pr1 <- profile( fm1 )
# profiled values for the two parameters
pr1$A
pr1$lrc
}
\keyword{nonlinear}
\keyword{regression}
\keyword{models}
\eof
% $Id: profiler.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{profiler}
\alias{profiler}
\title{ Constructor for Profiler Objects for Nonlinear Models}
\usage{
profiler(fitted, \dots)
}
\arguments{
 \item{fitted}{ the original fitted model object.}
 \item{\dots}{  Additional parameters. See documentation on individual
     methods.}
}
\description{
    Create a profiler object for the model object \code{fitted}.
}
\value{
    An object of class \code{"profiler"} which is a list with function
    elements
    \item{getFittedPars()}{
	the parameters in \code{fitted}
    }
    \item{setDefault(varying, params)}{
	this is used for changing the default settings for
	profiling. In absence of both parameters, the default is set
	to the original fitted parameters with all parameters
	varying. The arguments are

	\code{varying}: a logical, integer or character vector
	giving parameters to be varied.
	\code{params}: the default value at which profiling is to
	take place.
    }
    \item{getProfile(varying, params)}{
	this can be used in conjunction with \code{setDefault}
	without any arguments. Alternatively, the parameters to be
	varied and the values for fixed parameters can be specified
	using the arguments. The arguments are

	\code{varying}: a logical vector giving parameters to be
	varied. This can be omitted if params is a named list or
	numeric vector.

	\code{params}: values for parameters to be held fixed.

        It returns a list with elements
    
	\code{parameters}: the parameter values for the profiled
	optimum.

	\code{fstat}: a profile statistics. See individual methods
	for details.

	\code{varying}: a logical vector indicating parameters
	    which were varied.
    }
}
\author{Douglas M. Bates and Saikat DebRoy}
\seealso{\code{\link{profiler.nls}}, \code{\link{profile}}}
\examples{
# see documentation on individual methods
}
\keyword{nonlinear}
\keyword{models}
\eof
% $Id: profiler.nls.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{profiler.nls}
\alias{profiler.nls}
\title{ Constructor for Profiler Objects from nls Objects}
\description{
  Create a profiler object for the model object \code{fitted} of class
  \code{\link{nls}}.
}
\usage{
\method{profiler}{nls}(fitted, \dots)
}
\arguments{
  \item{fitted}{the original fitted model object of class
    \code{\link{nls}}.}
  \item{\dots}{  Additional parameters. None are used.}
}
\value{
  An object of class \code{profiler.nls} which is a list with function
  elements
  \item{getFittedModel()}{
    the \code{\link{nlsModel}} object corresponding to \code{fitted}
  }
  \item{getFittedPars()}{
    See documentation for \code{\link{profiler}}
  }
  \item{setDefault(varying, params)}{
    See documentation for \code{\link{profiler}}
  }
  \item{getProfile(varying, params)}{
    In the returned list, \code{fstat} is the ratio of change in
    sum-of-squares and the residual standard error.
    
    For other details, see documentation for \code{\link{profiler}}
  }
}
\section{Warning}{
  When using setDefault and getProfile together, the internal state of
  the fitted model may get changed. So after completing the profiling
  for a parameter, the internal states should be restored by a call to
  setDefault without any arguments. For example see below or the source
  for \code{\link{profile.nls}}.
}
\references{
  Bates, D.M. and Watts, D.G. (1988),
  \emph{Nonlinear Regression Analysis and Its Applications},
  Wiley
}
\author{Douglas M. Bates and Saikat DebRoy}
\seealso{
  \code{\link{nls}},
  \code{\link{nlsModel}},
  \code{\link{profiler}},
  \code{\link{profile.nls}}
}
\examples{
data( BOD )
## obtain the fitted object
fm1 <- nls(demand ~ SSasympOrig( Time, A, lrc ), data = BOD)
## get the profile for the fitted model
prof1 <- profiler( fm1 )
## profile with A fixed at 16.0
prof1$getProfile(c(FALSE, TRUE), 16.0)
## vary lrc
prof1$setDefault(varying = c(FALSE, TRUE))
## fix A at 14.0 and starting estimate of lrc at -0.2
prof1$setDefault(params = c(14.0, -0.2))
## and get the profile
prof1$getProfile()
## finally, set defaults back to original estimates
prof1$setDefault()
}
\keyword{nonlinear}
\keyword{regression}
\keyword{models}
\eof
\name{proj}
\title{Projections of Models}
\usage{
proj(object, \dots)  

\method{proj}{aov}(object, onedf = FALSE, unweighted.scale = FALSE, \dots)  

\method{proj}{aovlist}(object, onedf = FALSE, unweighted.scale = FALSE, \dots)  

\method{proj}{default}(object, onedf = TRUE, \dots)  

\method{proj}{lm}(object, onedf = FALSE, unweighted.scale = FALSE, \dots)  
}
\alias{proj}
\alias{proj.default}
\alias{proj.lm}
\alias{proj.aov}
\alias{proj.aovlist}
\arguments{
 \item{object}{An object of class \code{"lm"} or a class inheriting from
   it, or an object with a similar structure including in particular
   components \code{qr} and \code{effects}.}
 \item{onedf}{A logical flag. If \code{TRUE}, a projection is returned for all
   the columns of the model matrix. If \code{FALSE}, the single-column
   projections are collapsed by terms of the model (as represented in
   the analysis of variance table).}
 \item{unweighted.scale}{If the fit producing \code{object} used
   weights, this determines if the projections correspond to weighted or
   unweighted observations.}
 \item{\dots}{Swallow and ignore any other arguments.}
}
\description{
 \code{proj} returns a matrix or list of matrices giving the projections
 of the data onto the terms of a linear model.  It is most frequently
 used for \code{\link{aov}} models.
}
\details{
 A projection is given for each stratum of the object, so for \code{aov}
 models with an \code{Error} term the result is a list of projections.
}
\value{
  A projection matrix or (for multi-stratum objects) a list of
  projection matrices.

  Each projection is a matrix with a row for each observations and
  either a column for each term (\code{onedf = FALSE}) or for each
  coefficient (\code{onedf = TRUE}). Projection matrices from the
  default method have orthogonal columns representing the projection of
  the response onto the column space of the Q matrix from the QR
  decomposition.  The fitted values are the sum of the projections, and
  the sum of squares for each column is the reduction in sum of squares
  from fitting that column (after those to the left of it).

  The methods for \code{lm} and \code{aov} models add a column to the
  projection matrix giving the residuals (the projection of the data
  onto the orthogonal complement of the model space).

  Strictly, when \code{onedf = FALSE} the result is not a projection,
  but the columns represent sums of projections onto the columns of the
  model matrix corresponding to that term. In this case the matrix does
  not depend on the coding used.
}
\references{
  Chambers, J. M., Freeny, A and Heiberger, R. M. (1992)
  \emph{Analysis of variance; designed experiments.}
  Chapter 5 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\author{
  The design was inspired by the S function of the same name described
  in Chambers \emph{et al.} (1992).
}
\seealso{\code{\link{aov}}, \code{\link{lm}}, \code{\link{model.tables}}
}
\examples{
N <- c(0,1,0,1,1,1,0,0,0,1,1,0,1,1,0,0,1,0,1,0,1,1,0,0)
P <- c(1,1,0,0,0,1,0,1,1,1,0,0,0,1,0,1,1,0,0,1,0,1,1,0)
K <- c(1,0,0,1,0,1,1,0,0,1,0,1,0,1,1,0,0,0,1,1,1,0,1,0)
yield <- c(49.5,62.8,46.8,57.0,59.8,58.5,55.5,56.0,62.8,55.8,69.5,
55.0, 62.0,48.8,45.5,44.2,52.0,51.5,49.8,48.8,57.2,59.0,53.2,56.0)

npk <- data.frame(block=gl(6,4), N=factor(N), P=factor(P),
                  K=factor(K), yield=yield)
npk.aov <- aov(yield ~ block + N*P*K, npk)
proj(npk.aov)

## as a test, not particularly sensible
options(contrasts=c("contr.helmert", "contr.treatment"))
npk.aovE <- aov(yield ~  N*P*K + Error(block), npk)
proj(npk.aovE)
}
\keyword{models}
\eof
\name{prop.test}
\title{Test for Equal or Given Proportions}
\usage{
prop.test(x, n, p = NULL,
          alternative = c("two.sided", "less", "greater"),
          conf.level = 0.95, correct = TRUE)
}
\alias{prop.test}
\arguments{
  \item{x}{a vector of counts of successes or a matrix with 2 columns
    giving the counts of successes and failures, respectively.}
  \item{n}{a vector of counts of trials; ignored if \code{x} is a
    matrix.}
  \item{p}{a vector of probabilities of success.  The length of
    \code{p} must be the same as the number of groups specified by
    \code{x}, and its elements must be greater than 0 and less than 1.}
  \item{alternative}{a character string specifying the alternative
    hypothesis, must be one of \code{"two.sided"} (default),
    \code{"greater"} or \code{"less"}.  You can specify just the initial
    letter.  Only used for testing the null that a single proportion
    equals a given value, or that two proportions are equal; ignored
    otherwise.}
  \item{conf.level}{confidence level of the returned confidence
    interval.  Must be a single number between 0 and 1.  Only used
    when testing the null that a single proportion equals a given
    value, or that two proportions are equal; ignored otherwise.}
  \item{correct}{a logical indicating whether Yates' continuity
    correction should be applied.}
}
\description{
  \code{prop.test} can be used for testing the null that the
  proportions (probabilities of success) in several groups are the
  same, or that they equal certain given values.
}
\details{
  Only groups with finite numbers of successes and failures are used.
  Counts of successes and failures must be nonnegative and hence not
  greater than the corresponding numbers of trials which must be
  positive.  All finite counts should be integers.

  If \code{p} is \code{NULL} and there is more than one group, the null
  tested is that the proportions in each group are the same.  If there
  are two groups, the alternatives are that the probability of success
  in the first group is less than, not equal to, or greater than the
  probability of success in the second group, as specified by
  \code{alternative}.  A confidence interval for the difference of
  proportions with confidence level as specified by \code{conf.level}
  and clipped to \eqn{[-1,1]} is returned.  Continuity correction is
  used only if it does not exceed the difference of the sample
  proportions in absolute value.  Otherwise, if there are more than 2
  groups, the alternative is always \code{"two.sided"}, the returned
  confidence interval is \code{NULL}, and continuity correction is never
  used.

  If there is only one group, then the null tested is that the
  underlying probability of success is \code{p}, or .5 if \code{p} is
  not given.  The alternative is that the probability of success if less
  than, not equal to, or greater than \code{p} or 0.5, respectively, as
  specified by \code{alternative}.  A confidence interval for the
  underlying proportion with confidence level as specified by
  \code{conf.level} and clipped to \eqn{[0,1]} is returned.  Continuity
  correction is used only if it does not exceed the difference between
  sample and null proportions in absolute value. The confidence interval 
  is computed by inverting the score test.

  Finally, if \code{p} is given and there are more than 2 groups, the
  null tested is that the underlying probabilities of success are those
  given by \code{p}.  The alternative is always \code{"two.sided"}, the
  returned confidence interval is \code{NULL}, and continuity correction
  is never used.
}
\value{
  A list with class \code{"htest"} containing the following
  components:
  \item{statistic}{the value of Pearson's chi-squared test statistic.}
  \item{parameter}{the degrees of freedom of the approximate
    chi-squared distribution of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{estimate}{a vector with the sample proportions \code{x/n}.}
  \item{conf.int}{a confidence interval for the true proportion if
    there is one group, or for the difference in proportions if
    there are 2 groups and \code{p} is not given, or \code{NULL}
    otherwise.  In the cases where it is not \code{NULL}, the
    returned confidence interval has an asymptotic confidence level
    as specified by \code{conf.level}, and is appropriate to the
    specified alternative hypothesis.}
  \item{null.value}{the value of \code{p} if specified by the null, or
    \code{NULL} otherwise.}
  \item{alternative}{a character string describing the alternative.}
  \item{method}{a character string indicating the method used, and
    whether Yates' continuity correction was applied.}
  \item{data.name}{a character string giving the names of the data.}
}
\seealso{\code{\link{binom.test}} for an \emph{exact} test of a binomial
  hypothesis.}
\examples{
heads <- rbinom(1, size=100, pr = .5)
prop.test(heads, 100)          # continuity correction TRUE by default
prop.test(heads, 100, correct = FALSE)

## Data from Fleiss (1981), p. 139.
## H0: The null hypothesis is that the four populations from which
##     the patients were drawn have the same true proportion of smokers.
## A:  The alternative is that this proportion is different in at
##     least one of the populations.

smokers  <- c( 83, 90, 129, 70 )
patients <- c( 86, 93, 136, 82 )
prop.test(smokers, patients)
}
\keyword{htest}
\eof
\name{prop.trend.test}
\title{Test for trend in proportions}
\usage{
prop.trend.test(x, n, score = 1:length(x))
}
\alias{prop.trend.test}
\arguments{
  \item{x}{ Number of events }
  \item{n}{ Number of trials }
  \item{score}{ Group score }
}
\description{
  Performs chi-squared test for trend in proportions, i.e., a test
  asymptotically optimal for local alternatives where the log odds vary
  in proportion with \code{score}.  By default, \code{score} is chosen
  as the group numbers.
}
\value{
  An object of class \code{"htest"} with title, test statistic, p-value,
  etc.
}
\author{ Peter Dalgaard }
\note{ This really should get integrated with \code{prop.test} }
\seealso{ \code{\link{prop.test}}}
\examples{
smokers  <- c( 83, 90, 129, 70 )
patients <- c( 86, 93, 136, 82 )
prop.test(smokers, patients)
prop.trend.test(smokers, patients)
prop.trend.test(smokers, patients,c(0,0,0,1))
}
\keyword{htest}
\eof
\name{qqnorm}
\title{Quantile-Quantile Plots}
\usage{
qqnorm(y, \dots)
\method{qqnorm}{default}(y, ylim, main = "Normal Q-Q Plot",
       xlab = "Theoretical Quantiles",
       ylab = "Sample Quantiles", plot.it = TRUE, datax = FALSE,
       \dots)
qqline(y, datax = FALSE, \dots)
qqplot(x, y, plot.it = TRUE, xlab = deparse(substitute(x)),
       ylab = deparse(substitute(y)), \dots)
}
\alias{qqnorm}
\alias{qqnorm.default}
\alias{qqplot}
\alias{qqline}

\arguments{
  \item{x}{The first sample for \code{qqplot}.}
  \item{y}{The second or only data sample.}
  \item{xlab, ylab, main}{plot labels.}
  \item{plot.it}{logical. Should the result be plotted?}
  \item{datax}{logical. Should data values be on the x-axis?}
  \item{ylim, \dots}{graphical parameters.}
}
\description{
  \code{qqnorm} is a generic functions the default method of which
  produces a normal QQ plot of the values in \code{y}.
  \code{qqline} adds a line to a normal quantile-quantile plot which
  passes through the first and third quartiles.

  \code{qqplot} produces a QQ plot of two datasets.

  Graphical parameters may be given as arguments to \code{qqnorm},
  \code{qqplot} and \code{qqline}.
}
\value{
  For \code{qqnorm} and \code{qqplot}, a list with components
  \item{x}{The x coordinates of the points that were/would be plotted}
  \item{y}{The original \code{y} vector, i.e., the corresponding y
    coordinates \emph{including \code{\link{NA}}s}.}
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
\code{\link{ppoints}}.
}
\examples{
y <- rt(200, df = 5)
qqnorm(y); qqline(y, col = 2)
qqplot(y, rt(300, df = 5))
data(precip)
qqnorm(precip, ylab = "Precipitation [in/yr] for 70 US cities")
}
\keyword{hplot}
\keyword{distribution}
\eof
\name{quade.test}
\alias{quade.test}
\alias{quade.test.default}
\alias{quade.test.formula}
\title{Quade Test}
\description{
  Performs a Quade test with unreplicated blocked data.
}
\usage{
quade.test(y, \dots)

\method{quade.test}{default}(y, groups, blocks, \dots)

\method{quade.test}{formula}(formula, data, subset, na.action, \dots)
}
\arguments{
  \item{y}{either a numeric vector of data values, or a data matrix.}
  \item{groups}{a vector giving the group for the corresponding	elements
    of \code{y} if this is a vector;  ignored if \code{y} is a matrix.
    If not a factor object, it is coerced to one.}
  \item{blocks}{a vector giving the block for the corresponding	elements
    of \code{y} if this is a vector;  ignored if \code{y} is a matrix.
    If not a factor object, it is coerced to one.}
  \item{formula}{a formula of the form \code{a ~ b | c}, where \code{a},
    \code{b} and \code{c} give the data values and corresponding groups
    and blocks, respectively.}
  \item{data}{an optional data frame containing the variables in the
    model formula.}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.  Defaults to
    \code{getOption("na.action")}.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\details{
  \code{quade.test} can be used for analyzing unreplicated complete
  block designs (i.e., there is exactly one observation in \code{y}
  for each combination of levels of \code{groups} and \code{blocks})
  where the normality assumption may be violated.

  The null hypothesis is that apart from an effect of \code{blocks},
  the location parameter of \code{y} is the same in each of the
  \code{groups}.

  If \code{y} is a matrix, \code{groups} and \code{blocks} are obtained
  from the column and row indices, respectively.  \code{NA}'s are not
  allowed in \code{groups} or \code{blocks};  if \code{y} contains
  \code{NA}'s, corresponding blocks are removed.
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of Quade's F statistic.}
  \item{parameter}{a vector with the numerator and denominator degrees 
    of freedom of the approximate F distribution of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{method}{the character string \code{"Quade test"}.}
  \item{data.name}{a character string giving the names of the data.}
}
\references{
  D. Quade (1979),
  Using weighted rankings in the analysis of complete blocks with
  additive block effects.
  \emph{Journal of the American Statistical Association}, \bold{74},
  680--683.
    
  William J. Conover (1999),
  \emph{Practical nonparametric statistics}.
  New York: John Wiley & Sons.
  Pages 373--380.
}
\seealso{
  \code{\link{friedman.test}}.
}
\examples{
## Conover (1999, p. 375f):
## Numbers of five brands of a new hand lotion sold in seven stores
## during one week.
y <- matrix(c( 5,  4,  7, 10, 12,
               1,  3,  1,  0,  2,
              16, 12, 22, 22, 35,
               5,  4,  3,  5,  4,
              10,  9,  7, 13, 10,
              19, 18, 28, 37, 58,
              10,  7,  6,  8,  7),
            nr = 7, byrow = TRUE,
            dimnames =
            list(Store = as.character(1:7),
                 Brand = LETTERS[1:5]))
y
quade.test(y)
}
\keyword{htest}
\eof
\name{quantile}
\title{Sample Quantiles}
\alias{quantile}
\alias{quantile.default}
\description{
  The generic function \code{quantile} produces sample quantiles
  corresponding to the given probabilities.
  The smallest observation corresponds to a probability of 0 and the
  largest to a probability of 1.
}
\usage{
quantile(x, \dots)

\method{quantile}{default}(x, probs = seq(0, 1, 0.25), na.rm = FALSE,
         names = TRUE, \dots)
}
\arguments{
  \item{x}{numeric vectors whose sample quantiles are wanted.}
  \item{probs}{numeric vector with values in \eqn{[0,1]}.}
  \item{na.rm}{logical; if true, any \code{\link{NA}} and \code{NaN}'s
    are removed from \code{x} before the quantiles are computed.}
  \item{names}{logical; if true, the result has a \code{\link{names}}
    attribute.  Set to \code{FALSE} for speedup with many \code{probs}.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\details{
  A vector of length \code{length(probs)} is returned;
  if \code{names = TRUE}, it has a \code{\link{names}} attribute.

  \code{quantile(x,p)} as a function of \code{p} linearly interpolates
  the points \eqn{( (i-1)/(n-1), ox[i] )}, where
  \code{ox <- sort(x)} and \code{n <- length(x)}.

  This gives \code{quantile(x, p) == (1-f)*ox[i] + f*ox[i+1]}, where
  \code{r <- 1 + (n-1)*p}, \code{i <- floor(r)}, \code{f <- r - i}
  \emph{and} \code{ox[n+1] :=  ox[n]}.

  \code{\link{NA}} and \code{\link{NaN}} values in \code{probs} are
  propagated to the result.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link[stats]{ecdf}} (in the \pkg{stats} package) for
  empirical distributions of which \code{quantile} is the
  \dQuote{inverse};
  \code{\link{boxplot.stats}} and \code{\link{fivenum}} for computing
  \dQuote{versions} of quartiles, etc.
}
\examples{
quantile(x <- rnorm(1001))# Extremes & Quartiles by default
quantile(x,  probs=c(.1,.5,1,2,5,10,50, NA)/100)
}
\keyword{univar}
\eof
\name{read.ftable}
\alias{read.ftable}
\alias{write.ftable}
\alias{as.table.ftable}
\title{Manipulate Flat Contingency Tables}
\description{Read, write and coerce \dQuote{flat} contingency tables.}
\usage{
read.ftable(file, sep = "", quote = "\"",
            row.var.names, col.vars, skip = 0)
write.ftable(x, file = "", quote = TRUE, digits = getOption("digits"))

\method{as.table}{ftable}(x, \dots)
}
\arguments{
  \item{file}{either a character string naming a file or a connection
    which the data are to be read from or written to.  \code{""}
    indicates input from the console for reading and output to the
    console for writing.}
  \item{sep}{the field separator string.  Values on each line of the file
    are separated by this string.}
  \item{quote}{a character string giving the set of quoting characters
    for \code{read.ftable}; to disable quoting altogether, use
    \code{quote=""}.  For \code{write.table}, a logical indicating
    whether strings in the data will be surrounded by double quotes.}
  \item{row.var.names}{a character vector with the names of the row
    variables, in case these cannot be determined automatically.}
  \item{col.vars}{a list giving the names and levels of the column
    variables, in case these cannot be determined automatically.}
  \item{skip}{the number of lines of the data file to skip before
    beginning to read data.}
  \item{x}{an object of class \code{"ftable"}.}
  \item{digits}{an integer giving the number of significant digits to
    use for (the cell entries of) \code{x}.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\details{
  \code{read.ftable} reads in a flat-like contingency table from a
  file.  If the file contains the written representation of a flat
  table (more precisely, a header with all information on names and
  levels of column variables, followed by a line with the names of the
  row variables), no further arguments are needed.  Similarly, flat
  tables with only one column variable the name of which is the only
  entry in the first line are handled automatically.  Other variants can
  be dealt with by skipping all header information using \code{skip},
  and providing the names of the row variables and the names and levels
  of the column variable using \code{row.var.names} and \code{col.vars},
  respectively.  See the examples below.

  Note that flat tables are characterized by their \dQuote{ragged}
  display of row (and maybe also column) labels.  If the full grid of
  levels of the row variables is given, one should instead use
  \code{read.table} to read in the data, and create the contingency
  table from this using \code{\link{xtabs}}.

  \code{write.ftable} writes a flat table to a file, which is useful for
  generating \dQuote{pretty} ASCII representations of contingency tables.

  \code{as.table.ftable} converts a contingency table in flat matrix
  form to one in standard array form.  This is a method for the generic
  function \code{as.table}.
}
\seealso{
  \code{\link{ftable}} for more information on flat contingency tables.
}
\references{
  Agresti, A. (1990)
  \emph{Categorical data analysis}.
  New York: Wiley.
}  
\examples{
## Agresti (1990), page 157, Table 5.8.
## Not in ftable standard format, but o.k.
file <- tempfile()
cat("             Intercourse\n",
    "Race  Gender     Yes  No\n",
    "White Male        43 134\n",
    "      Female      26 149\n",
    "Black Male        29  23\n",
    "      Female      22  36\n",
    file = file)
file.show(file)
ft <- read.ftable(file)
ft
unlink(file)

## Agresti (1990), page 297, Table 8.16.
## Almost o.k., but misses the name of the row variable.
file <- tempfile()
cat("                      \"Tonsil Size\"\n",
    "            \"Not Enl.\" \"Enl.\" \"Greatly Enl.\"\n",
    "Noncarriers       497     560           269\n",
    "Carriers           19      29            24\n",
    file = file)
file.show(file)
ft <- read.ftable(file, skip = 2,
                  row.var.names = "Status",
                  col.vars = list("Tonsil Size" =
                      c("Not Enl.", "Enl.", "Greatly Enl.")))
ft
unlink(file)
}
\keyword{category}
\eof
\name{rect.hclust}
\alias{rect.hclust}
\title{Draw Rectangles Around Hierarchical Clusters}
\description{
  Draws rectangles around the branches of a dendrogram highlighting the
  corresponding clusters. First the dendrogram is cut at a certain
  level, then a rectangle is drawn around selected branches.
}
\usage{
rect.hclust(tree, k = NULL, which = NULL, x = NULL, h = NULL,
            border = 2, cluster = NULL)
}
\arguments{
 \item{tree}{an object of the type produced by \code{hclust}.}
 \item{k, h}{Scalar. Cut the dendrogram such that either exactly
   \code{k} clusters are produced or by cutting at height \code{h}.}
 \item{which, x}{A vector selecting the clusters around which a
   rectangle should be drawn. \code{which} seleccts clusters by number
   (from left to right in the tree), \code{x} selects clusters
   containing the respective horizontal coordinates. Default is
   \code{which = 1:k}.}
 \item{border}{Vector with border colors for the rectangles.}
 \item{cluster}{Optional vector with cluster memberships as returned by
   \code{cutree(hclust.obj, k = k)}, can be specified for efficiency if
   already computed.}
}
\value{
  (Invisibly) returns a list where each element contains a vector of
  data points contained in the respective cluster.}
\seealso{
  \code{\link{hclust}}, \code{\link{identify.hclust}}.
}
\examples{
data(USArrests)
hca <- hclust(dist(USArrests))
plot(hca)
rect.hclust(hca, k=3, border="red")
x <- rect.hclust(hca, h=50, which=c(2,7), border=3:4)
x
}
\keyword{aplot}
\keyword{cluster}
\eof
\name{relevel}
\alias{relevel}
\alias{relevel.default}
\alias{relevel.factor}
\alias{relevel.ordered}
\title{Reorder Levels of Factor}
\usage{
relevel(x, ref, \dots)
}
\arguments{
    \item{x}{An unordered factor.}
    \item{ref}{The reference level.}
    \item{\dots}{Additional arguments for future methods.}
}
\description{
    The levels of a factor are re-ordered so that the level specified by
    \code{ref} is first and the others are moved down. This is useful
    for \code{contr.treatment} contrasts which take the first level as
    the reference.
}
\value{
  A factor of the same length as \code{x}.
}

\seealso{\code{\link{factor}}, \code{\link{contr.treatment}}}

\examples{
data(warpbreaks)
warpbreaks$tension <- relevel(warpbreaks$tension, ref="M")
summary(lm(breaks ~ wool + tension, data=warpbreaks))
}
\keyword{utilities}
\keyword{models}
\eof
\name{reorder}
\alias{reorder}
\alias{reorder.dendrogram}
\title{ Reorder a dendrogram }
\description{
  There are many different orderings of a dendrogram that are consistent
  with the structure imposed.  This function takes a dendrogram and a
  vector of values and reorders the dendrogram in the order of the
  supplied vector, maintaining the constraints on the dendrogram.
}
\usage{
reorder(x, \dots)

\method{reorder}{dendrogram}(x, wts, \dots)
}
\arguments{
  \item{x}{the object to be reordered}
  \item{wts}{numeric weights (arbitrary values) for reordering.}
  \item{\dots}{additional arguments}
}
\details{
  Using the weights \code{wts}, the leaves of the dendrogram are
  reordered so as to be in an order as consistent as possible with the
  weights.  Afterward, at each node, the branches are ordered in
  increasing weights (where the weight of a branch is defined as the sum
  of the weights of its sub branches).
}
\value{
  From \code{reorder.dendrogram} a dendrogram, where each node has a
  further attribute \code{value} with its corresponding weight.
}
\author{R. Gentleman}
\seealso{\code{\link{heatmap}}, \code{\link{cophenetic}} }

\examples{
  set.seed(123)
  x <- rnorm(10)
  hc <- hclust(dist(x))
  dd <- as.dendrogram(hc)
  dd.reorder <- reorder(dd, 10:1)

  op <- par(mfcol = 1:2)
  plot(dd, main = "random dendrogram `dd'")
  plot(dd.reorder, main = "reorder(dd, 10:1)")
  par(op)
}
\keyword{manip}
\eof
\name{replications}
\title{Number of Replications of Terms}
\usage{
replications(formula, data=NULL, na.action)
}
\alias{replications}
\arguments{
  \item{formula}{a formula or a terms object or a data frame.}
  \item{data}{a data frame used  to  find  the  objects in \code{formula}.}
  \item{na.action}{function for handling missing values.  Defaults to
    a \code{na.action} attribute of \code{data}, then
    a setting of the option \code{na.action}, or \code{na.fail} if that
    is not set.}
}
\description{
  Returns a vector or a list of the number of replicates for
  each term in the formula.
}
\details{
  If \code{formula} is a data frame and \code{data} is missing,
  \code{formula} is used for \code{data} with the formula \code{~ .}.
}
\value{
  A vector or list with one entry for each term in the formula giving
  the number(s) of replications for each level. If all levels are
  balanced (have the same number of replications) the result is a
  vector, otherwise it is a list with a component for each terms,
  as a vector, matrix or array as required.

  A test for balance is \code{!is.list(replications(formula,data))}.
}
\references{
  Chambers, J. M., Freeny, A and Heiberger, R. M. (1992)
  \emph{Analysis of variance; designed experiments.}
  Chapter 5 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\author{
  The design was inspired by the S function of the same name described
  in Chambers \emph{et al.} (1992).
}

\seealso{\code{\link{model.tables}}}

\examples{
## From Venables and Ripley (2002) p.165.
N <- c(0,1,0,1,1,1,0,0,0,1,1,0,1,1,0,0,1,0,1,0,1,1,0,0)
P <- c(1,1,0,0,0,1,0,1,1,1,0,0,0,1,0,1,1,0,0,1,0,1,1,0)
K <- c(1,0,0,1,0,1,1,0,0,1,0,1,0,1,1,0,0,0,1,1,1,0,1,0)
yield <- c(49.5,62.8,46.8,57.0,59.8,58.5,55.5,56.0,62.8,55.8,69.5,
55.0, 62.0,48.8,45.5,44.2,52.0,51.5,49.8,48.8,57.2,59.0,53.2,56.0)

npk <- data.frame(block=gl(6,4), N=factor(N), P=factor(P),
                  K=factor(K), yield=yield)
replications(~ . - yield, npk)
}
\keyword{models}
\eof
\name{reshape}
\alias{reshape}
\title{Reshape Grouped Data}
\description{
  This function reshapes a data frame between \sQuote{wide} format with
  repeated measurements in separate columns of the same record and
  \sQuote{long} format with the repeated measurements in separate
  records.
}
\usage{
reshape(data, varying = NULL, v.names = NULL, timevar = "time",
	idvar = "id", ids = 1:NROW(data),
	times = seq(length = length(varying[[1]])),
	drop = NULL, direction, new.row.names = NULL,
	split = list(regexp="\\.", include=FALSE))
}
\arguments{
  \item{data}{a data frame}
  \item{varying}{names of sets of variables in the wide format that
    correspond to single variables in long format
    (\sQuote{time-varying}).  A list of vectors (or optionally a matrix
    for \code{direction="wide"}).  See below for more details and
    options.}
  \item{v.names}{names of variables in the long format that correspond
    to multiple variables in the wide format.}
  \item{timevar}{the variable in long format that differentiates multiple
    records from the same group or individual.}
  \item{idvar}{the variable in long format that identifies multiple
    records from the same group/individual.  This variable may also be
    present in wide format.}
  \item{ids}{the values to use for a newly created \code{idvar}
    variable in long format.}
  \item{times}{the values to use for a newly created \code{timevar}
    variable in long format.}
  \item{drop}{a vector of names of variables to drop before reshaping}
  \item{direction}{character string, either \code{"wide"} to reshape to
    wide format, or \code{"long"} to reshape to long format.}
  \item{new.row.names}{logical; if \code{TRUE} and \code{direction="wide"},
    create new row names in long format from the values of the id and
    time variables.}
  \item{split}{information for guessing the \code{varying},
    \code{v.names}, and \code{times} arguments.  See below for details.}
}
\details{
  The arguments to this function are described in terms of longitudinal
  data, as that is the application motivating the functions.  A \sQuote{wide}
  longitudinal dataset will have one record for each individual with
  some time-constant variables that occupy single columns and some
  time-varying variables that occupy a column for each time point.  In
  \sQuote{long} format there will be multiple records for each individual, with
  some variables being constant across these records and others varying
  across the records.  A \sQuote{long} format dataset also needs a \sQuote{time}
  variable identifying which time point each record comes from and an
  \sQuote{id} variable showing which records refer to the same person.

  If the data frame resulted from a previous \code{reshape} then the
  operation can be reversed by specifying just the \code{direction}
  argument.  The other arguments are stored as attributes on the data frame.

  If \code{direction="long"} and no \code{varying} or \code{v.names}
  arguments are supplied it is assumed that all variables except
  \code{idvar} and \code{timevar} are time-varying. They are all
  expanded into multiple variables in wide format.

  If \code{direction="wide"} the \code{varying} argument can be a vector
  of column names or column numbers (converted to column names). The
  function will attempt to guess the \code{v.names} and \code{times} from
  these names.  The default is variable names like \code{x.1},
  \code{x.2},where \code{split=list(regexp="\\.",include=FALSE)} to
  specifies to split at the dot and drop it from the name. To have alphabetic
  followed by numeric times use
  \code{split=list(regexp="[A-Za-z][0-9]",include=TRUE)}. This splits
  between the alphabetic and numeric parts of the name and does not drop
  the regular expression.

}
\value{
  The reshaped data frame with added attributes to simplify reshaping
  back to the original form.
}
\seealso{\code{\link{stack}}, \code{\link{aperm}}}

\examples{
data(Indometh)
summary(Indometh)
wide <- reshape(Indometh, v.names="conc", idvar="Subject",
                timevar="time", direction="wide")
wide

reshape(wide, direction="long")
reshape(wide, idvar="Subject", varying=list(names(wide)[2:12]),
        v.names="conc", direction="long")

## times need not be numeric
df <- data.frame(id=rep(1:4,rep(2,4)), visit=I(rep(c("Before","After"),4)),
              x=rnorm(4), y=runif(4))
df
reshape(df, timevar="visit", idvar="id", direction="wide")
## warns that y is really varying
reshape(df, timevar="visit", idvar="id", direction="wide", v.names="x")


##  unbalanced 'long' data leads to NA fill in 'wide' form
df2 <- df[1:7,]
df2
reshape(df2, timevar="visit", idvar="id", direction="wide")

## Alternative regular expressions for guessing names
df3 <- data.frame(id=1:4, age=c(40,50,60,50), dose1=c(1,2,1,2),
                  dose2=c(2,1,2,1), dose4=c(3,3,3,3))
reshape(df3, direction="long", varying=3:5,
        split=list(regexp="[a-z][0-9]", include=TRUE))


## an example that isn't longitudinal data
data(state)
state.x77 <- as.data.frame(state.x77)
long <- reshape(state.x77, idvar="state", ids=row.names(state.x77),
                times=names(state.x77), timevar="Characteristic",
                varying=list(names(state.x77)), direction="long")

reshape(long, direction="wide")

reshape(long, direction="wide", new.row.names=unique(long$state))
}
\keyword{manip}

\eof
\name{residuals}
\title{Extract Model Residuals}
\usage{
residuals(object, \dots)
resid(object, \dots)
}
\alias{residuals}
\alias{resid}
\alias{residuals.default}
\arguments{
  \item{object}{an object for which the extraction of model residuals is
    meaningful.}
  \item{\dots}{other arguments.}
}
\value{
  Residuals extracted from the object \code{object}.
}
\description{
  \code{residuals} is a generic function which extracts model residuals
  from objects returned by modeling functions.

  The abbreviated form \code{resid} is an alias for \code{residuals}.
  It is intended to encourage users to access object components through
  an accessor function rather than by directly referencing an object
  slot.

  All object classes which are returned by model fitting functions
  should provide a \code{residuals} method.  (Note that the method is
  \samp{residuals} and not \samp{resid}.)

  Methods can make use of \code{\link{naresid}} methods to compensate
  for the omission of missing values.  The default method does.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical Models in S}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{coefficients}}, \code{\link{fitted.values}},
  \code{\link{glm}}, \code{\link{lm}}.
}
\keyword{models}
\keyword{regression}
\eof
% file MASS/rock.d
% copyright (C) 1994-8 W. N. Venables and B. D. Ripley
%
% Manually converted to Rd data file description format by KH.
\name{rock}
\docType{data}
\alias{rock}
\title{Measurements on Petroleum Rock Samples}
\description{
    Measurements on 48 rock samples from a petroleum reservoir.}
\usage{data(rock)}
\format{
    A data frame with 48 rows and 4 numeric columns.

    \tabular{rll}{
	[,1] \tab area  \tab area of pores space, in pixels
	out of 256 by 256 \cr
	[,2] \tab peri  \tab perimeter in pixels \cr
	[,3] \tab shape \tab perimeter/sqrt(area) \cr
	[,4] \tab perm  \tab permeability in milli-Darcies
    }
}
\details{
    Twelve core samples from petroleum reservoirs were sampled by 4
    cross-sections.  Each core sample was measured for permeability, and
    each cross-section has total area of pores, total perimeter of
    pores, and shape.
}
\source{
    Data from BP Research, image analysis by Ronit Katz, U. Oxford.
}
\keyword{datasets}
\eof
\name{runmed}
\title{Running Medians -- Robust Scatter Plot Smoothing}
\alias{runmed}
\description{
  Compute running medians of odd span.  This is the \dQuote{most robust}
  scatter plot smoothing possible.  For efficiency (and historical
  reason), you can use one of two different algorithms giving identical
  results.
}
\usage{
runmed(x, k, endrule = c("median","keep","constant"),
       algorithm = NULL, print.level = 0)
}
\arguments{
  \item{x}{numeric vector, the \dQuote{dependent} variable to be
    smoothed.}
  \item{k}{integer width of median window; must be odd.  Turlach had a
    default of \code{k <- 1 + 2 * min((n-1)\%/\% 2, ceiling(0.1*n))}.
    Use \code{k = 3} for \dQuote{minimal} robust smoothing eliminating
    isolated outliers.}
  \item{endrule}{character string indicating how the values at the
    beginning and the end (of the data) should be treated.
    \describe{
      \item{\code{"keep"}}{keeps the first and last \eqn{k_2}{k2} values
	at both ends, where \eqn{k_2}{k2} is the half-bandwidth \code{k2
	  = k \%/\% 2},
	i.e., \code{y[j] = x[j]} for \eqn{j \in \{1,\ldots,k_2;
	  n-k_2+1,\ldots,n\}}{j = 1,..,k2 and (n-k2+1),..,n};}
      \item{\code{"constant"}}{copies \code{median(y[1:k2])} to the first
	values and analogously for the last ones making the smoothed ends
	\emph{constant};}
      \item{\code{"median"}}{the default, smoothes the ends by using
	symmetrical medians of subsequently smaller bandwidth, but for
	the very first and last value where Tukey's robust end-point
	rule is applied, see \code{\link{smoothEnds}}.}
    }
  }
  \item{algorithm}{character string (partially matching \code{"Turlach"} or
    \code{"Stuetzle"}) or the default \code{NULL}, specifying which algorithm
    should be applied.  The default choice depends on \code{n = length(x)}
    and \code{k} where \code{"Turlach"} will be used for larger problems.}
  \item{print.level}{integer, indicating verboseness of algorithm;
    should rarely be changed by average users.}
}
\value{vector of smoothed values of the same length as \code{x} with an
  \code{\link{attr}}ibute \code{k} containing (the \sQuote{oddified})
  \code{k}.
}
\details{
  Apart from the end values, the result \code{y = runmed(x, k)} simply has
  \code{y[j] = median(x[(j-k2):(j+k2)])} (k = 2*k2+1), computed very
  efficiently.

  The two algorithms are internally entirely different:
  \describe{
    \item{"Turlach"}{is the Hrdle-Steiger
      algorithm (see Ref.) as implemented by Berwin Turlach.
      A tree algorithm is used, ensuring performance \eqn{O(n \log
	k)}{O(n * log(k))} where \code{n <- length(x)} which is
      asymptotically optimal.}
    \item{"Stuetzle"}{is the (older) Stuetzle-Friedman implementation
      which makes use of median \emph{updating} when one observation
      enters and one leaves the smoothing window.  While this performs as
      \eqn{O(n \times k)}{O(n * k)} which is slower asymptotically, it is
      considerably faster for small \eqn{k} or \eqn{n}.}
  }
}
\references{
  Hrdle, W. and Steiger, W. (1995)
  [Algorithm AS 296] Optimal median smoothing,
  \emph{Applied Statistics} \bold{44}, 258--264.

  Jerome H. Friedman and Werner Stuetzle (1982)
  \emph{Smoothing of Scatterplots};
  Report, Dep. Statistics, Stanford U., Project Orion 003.

  Martin Maechler (2003)
  Fast Running Medians: Finite Sample and Asymptotic Optimality;
  working paper available from the author.
}
\author{Martin Maechler \email{maechler@stat.math.ethz.ch},
  based on Fortran code from Werner Stuetzle and S-plus and C code from
  Berwin Turlach.
}
\seealso{
  \code{\link{smoothEnds}} which implements Tukey's end point rule and
  is called by default from \code{runmed(*, endrule = "median")}.
  \code{\link[eda]{smooth}} (from \pkg{eda} package) uses running
  medians of 3 for its compound smoothers.
}
\examples{
example(nhtemp)#> data(nhtemp)
myNHT <- as.vector(nhtemp)
myNHT[20] <- 2 * nhtemp[20]
plot(myNHT, type="b", ylim = c(48,60), main = "Running Medians Example")
lines(runmed(myNHT, 7), col = "red")

## special: multiple y values for one x
data(cars)
plot(cars, main = "'cars' data and runmed(dist, 3)")
lines(cars, col = "light gray", type = "c")
with(cars, lines(speed, runmed(dist, k = 3), col = 2))
%% FIXME: Show how to do it properly ! tapply(*, unique(.), median)

## nice quadratic with a few outliers
y <- ys <- (-20:20)^2
y [c(1,10,21,41)] <- c(150, 30, 400, 450)
all(y == runmed(y, 1)) # 1-neigborhood <==> interpolation
plot(y) ## lines(y, lwd=.1, col="light gray")
lines(lowess(seq(y),y, f = .3), col = "brown")
lines(runmed(y, 7), lwd=2, col = "blue")
lines(runmed(y,11), lwd=2, col = "red")

## Lowess is not robust
y <- ys ; y[21] <- 6666 ; x <- seq(y)
col <- c("black", "brown","blue")
plot(y, col=col[1])
lines(lowess(x,y, f = .3), col = col[2])
%% predict(loess(y ~ x, span = 0.3, degree=1, family = "symmetric"))
%% gives 6-line warning but does NOT break down
lines(runmed(y, 7),      lwd=2, col = col[3])
legend(length(y),max(y), c("data", "lowess(y, f = 0.3)", "runmed(y, 7)"),
       xjust = 1, col = col, lty = c(0, 1,1), pch = c(1,NA,NA))
}
\keyword{smooth}
\keyword{robust}
\eof
\name{scatter.smooth}
\title{Scatter Plot with Smooth Curve Fitted by Loess}
\alias{scatter.smooth}
\alias{loess.smooth}
\description{
  Plot and add a smooth curve computed by \code{loess} to a scatter plot.
}
\usage{
scatter.smooth(x, y, span = 2/3, degree = 1,
    family = c("symmetric", "gaussian"),
    xlab = deparse(substitute(x)), ylab = deparse(substitute(y)),
    ylim = range(y, prediction$y), evaluation = 50, \dots)

loess.smooth(x, y, span = 2/3, degree = 1,
    family = c("symmetric", "gaussian"), evaluation = 50, \dots)
}
\arguments{
  \item{x}{x coordinates for scatter plot.}
  \item{y}{y coordinates for scatter plot.}
  \item{span}{smoothness parameter for \code{loess}.}
  \item{degree}{degree of local polynomial used.}
  \item{family}{if \code{"gaussian"} fitting is by least-squares, and if
    \code{family="symmetric"} a re-descending M estimator is used.} 
  \item{xlab}{label for x axis.}
  \item{ylab}{label for y axis.}
  \item{ylim}{the y limits of the plot.}
  \item{evaluation}{number of points at which to evaluate the smooth
    curve.}
  \item{\dots}{graphical parameters.}
}
\details{
  \code{loess.smooth} is an auxiliary function which evaluates the
  \code{loess} smooth at \code{evaluation} equally spaced points
  covering the range of \code{x}.
}
\value{
  For \code{scatter.smooth}, none.

  For \code{loess.smooth}, a list with two components, \code{x} (the
  grid of evaluation points) and \code{y} (the smoothed values at the
  grid points).
}
\seealso{\code{\link{loess}}}
\examples{
data(cars)
attach(cars)
scatter.smooth(speed, dist)
detach()
}
\keyword{smooth}
\eof
\name{screeplot}
\alias{screeplot}
\title{Screeplot of PCA Results}
\usage{
screeplot(x, npcs = min(10, length(x$sdev)),
          type = c("barplot", "lines"), main = deparse(substitute(x)), \dots)
}
\arguments{
  \item{x}{an object of class \code{"princomp"}, as
    from \code{\link{princomp}()}.}
  \item{npcs}{the number of principal components to be plotted.}
  \item{type}{the type of plot.}
  \item{main, \dots}{graphics parameters.}
}
\description{
  \code{screeplot} plots the variances against the number of the
  principal component. This is also the \code{plot} method for class
  \code{"princomp"}.
}
\references{
  Mardia, K. V., J. T. Kent and J. M. Bibby (1979).
  \emph{Multivariate Analysis}, London: Academic Press.

  Venables, W. N. and B. D. Ripley (2002).
  \emph{Modern Applied Statistics with S}, Springer-Verlag.
}
\seealso{
  \code{\link{princomp}}.
}
\examples{
## The variances of the variables in the
## USArrests data vary by orders of magnitude, so scaling is appropriate
data(USArrests)
(pc.cr <- princomp(USArrests, cor = TRUE))  # inappropriate
screeplot(pc.cr)

data(Harman74.cor)
fit <- princomp(covmat=Harman74.cor)
screeplot(fit)
screeplot(fit, npcs=24, type="lines")
}
\keyword{multivariate}
\eof
\name{sd}
\alias{sd}
\title{Standard Deviation}
\description{
  This function computes the standard deviation of the values in
  \code{x}.
  If \code{na.rm} is \code{TRUE} then missing values are removed before
  computation proceeds.
  If \code{x} is a matrix or a data frame, a vector of the standard
  deviation of the columns is returned.
}
\usage{
sd(x, na.rm = FALSE)
}
\arguments{
  \item{x}{a numeric vector, matrix or data frame.}
  \item{na.rm}{logical. Should missing values be removed?}
}
\seealso{
  \code{\link{var}} for its square, and \code{\link{mad}}, the most
  robust alternative.
}
\examples{
sd(1:2) ^ 2
}
\keyword{univar}
\eof
\name{se.aov}
\alias{se.aov}
\alias{se.aovlist}
\title{Internal Functions Used by model.tables}
\usage{
se.aov(object, n, type = "means")
se.aovlist(object, dn.proj, dn.strata, factors, mf, efficiency,
           n, type = "diff.means", \dots)
}
\description{
  Internal function for use by \code{\link{model.tables}}.
}
\seealso{\code{\link{model.tables}}}

\keyword{internal}
\eof
\name{se.contrast}
\alias{se.contrast}
\alias{se.contrast.aov}
\alias{se.contrast.aovlist}
\title{Standard Errors for Contrasts in Model Terms}
\usage{
se.contrast(object, \dots)
\method{se.contrast}{aov}(object, contrast.obj,
           coef = contr.helmert(ncol(contrast))[, 1],
           data = NULL, \dots)
}
\arguments{
 \item{object}{A suitable fit, usually from \code{aov}.}
 \item{contrast.obj}{The contrasts for which standard errors are
   requested.  This can be specified via a list or via a matrix.  A
   single contrast can be specified by a list of logical vectors giving
   the cells to be contrasted.  Multiple contrasts should be specified
   by a matrix, each column of which is a numerical contrast vector
   (summing to zero).
   }
 \item{coef}{used when \code{contrast.obj} is a list; it should be a
   vector of the same length as the list with zero sum.  The default
   value is the first Helmert contrast, which contrasts the first and
   second cell means specified by the list.}
 \item{data}{The data frame used to evaluate \code{contrast.obj}.}
 \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  Returns the standard errors for one or more contrasts in an \code{aov}
  object.
}
\details{
  Contrasts are usually used to test if certain means are
  significantly different; it can be easier to use \code{se.contrast}
  than compute them directly from the coefficients.

  In multistratum models, the contrasts can appear in more than one
  stratum; the contrast and standard error are computed in the lowest
  stratum and adjusted for efficiencies and comparisons between strata.

  Suitable matrices for use with \code{coef} can be found by
  calling \code{\link{contrasts}} and indexing the columns by a factor.
}
 \value{
  A vector giving the standard errors for each contrast.
}

\seealso{
  \code{\link{contrasts}},
  \code{\link{model.tables}}
}
\examples{
## From Venables and Ripley (2002) p.165.
N <- c(0,1,0,1,1,1,0,0,0,1,1,0,1,1,0,0,1,0,1,0,1,1,0,0)
P <- c(1,1,0,0,0,1,0,1,1,1,0,0,0,1,0,1,1,0,0,1,0,1,1,0)
K <- c(1,0,0,1,0,1,1,0,0,1,0,1,0,1,1,0,0,0,1,1,1,0,1,0)
yield <- c(49.5,62.8,46.8,57.0,59.8,58.5,55.5,56.0,62.8,55.8,69.5,
55.0, 62.0,48.8,45.5,44.2,52.0,51.5,49.8,48.8,57.2,59.0,53.2,56.0)

npk <- data.frame(block = gl(6,4), N = factor(N), P = factor(P),
                  K = factor(K), yield = yield)
options(contrasts=c("contr.treatment", "contr.poly"))
npk.aov1 <- aov(yield ~ block + N + K, npk)
se.contrast(npk.aov1, list(N=="0", N=="1"), data=npk)
# or via a matrix
cont <- matrix(c(-1,1), 2, 1, dimnames=list(NULL, "N"))
se.contrast(npk.aov1, cont[N, , drop=FALSE]/12, data=npk)

## test a multi-stratum model
npk.aov2 <- aov(yield ~ N + K + Error(block/(N + K)), npk)
se.contrast(npk.aov2, list(N == "0", N == "1"))
}
\keyword{models}
\eof
\name{selfStart}
\alias{selfStart}
\alias{selfStart.default}
\alias{selfStart.formula}
\title{Construct Self-starting Nonlinear Models}
\description{
  Construct self-starting nonlinear models.
}
\usage{
selfStart(model, initial, parameters, template)
}
\arguments{
  \item{model}{a function object defining a nonlinear model or
    a nonlinear formula object of the form \code{~expression}.}
 \item{initial}{a function object, taking three arguments: \code{mCall},
   \code{data}, and \code{LHS}, representing, respectively, a matched
   call to the function \code{model}, a data frame in
   which to interpret the variables in \code{mCall}, and the expression
   from the left-hand side of the model formula in the call to \code{nls}.
   This function should return initial values for the parameters in
   \code{model}.}
  \item{parameters}{a character vector specifying the terms on the right
    hand side of \code{model} for which initial estimates should be
    calculated.  Passed as the \code{namevec} argument to the
    \code{deriv} function.}
  \item{template}{an optional prototype for the calling sequence of the
    returned object, passed as the \code{function.arg} argument to the
    \code{deriv} function.  By default, a template is generated with the
    covariates in \code{model} coming first and the parameters in
    \code{model} coming last in the calling sequence.}
}
\details{
  This function is generic; methods functions can be written to handle
  specific classes of objects.
}
\value{
  a function object of class \code{"selfStart"}, for the \code{formula}
  method obtained by applying
  \code{deriv} to the right hand side of the \code{model} formula.  An
  \code{initial} attribute (defined by the \code{initial} argument) is
  added to the function to calculate starting estimates for the
  parameters in the model automatically.
}
\author{Jose Pinheiro and Douglas Bates}

\seealso{
  \code{\link{nls}}
}
\examples{
## self-starting logistic model

SSlogis <- selfStart(~ Asym/(1 + exp((xmid - x)/scal)),
  function(mCall, data, LHS)
  {
    xy <- sortedXyData(mCall[["x"]], LHS, data)
    if(nrow(xy) < 4) {
      stop("Too few distinct x values to fit a logistic")
    }
    z <- xy[["y"]]
    if (min(z) <= 0) { z <- z + 0.05 * max(z) } # avoid zeroes
    z <- z/(1.05 * max(z))		# scale to within unit height
    xy[["z"]] <- log(z/(1 - z))		# logit transformation
    aux <- coef(lm(x ~ z, xy))
    parameters(xy) <- list(xmid = aux[1], scal = aux[2])
    pars <- as.vector(coef(nls(y ~ 1/(1 + exp((xmid - x)/scal)), 
  			     data = xy, algorithm = "plinear")))
    value <- c(pars[3], pars[1], pars[2])
    names(value) <- mCall[c("Asym", "xmid", "scal")]
    value
  }, c("Asym", "xmid", "scal"))

# 'first.order.log.model' is a function object defining a first order
# compartment model 
# 'first.order.log.initial' is a function object which calculates initial
# values for the parameters in 'first.order.log.model'

# self-starting first order compartment model
\dontrun{
SSfol <- selfStart(first.order.log.model, first.order.log.initial)
}
}    
\keyword{models}
\eof
\name{setNames}
\alias{setNames}
\title{Set the Names in an Object}
\usage{
setNames(object, nm)
}
\arguments{
 \item{object}{an object for which a \code{names} attribute will be meaningful }
 \item{nm}{a character vector of names to assign to the object}
}
\description{
  This is a convenience function that sets the names on an object and
  returns the object.  It is most useful at the end of a function
  definition where one is creating the object to be returned and would
  prefer not to store it under a name just so the names can be assigned.
}
\value{
  An object of the same sort as \code{object} with the new names assigned.
}
\author{Douglas M. Bates and Saikat DebRoy }
\seealso{\code{\link{clearNames}}}
\examples{
setNames( 1:3, c("foo", "bar", "baz") )
# this is just a short form of
tmp <- 1:3
names(tmp) <-  c("foo", "bar", "baz")
tmp
}
\keyword{list}
\eof
\name{shapiro.test}
\alias{shapiro.test}
\title{Shapiro-Wilk Normality Test}
\description{
  Performs the Shapiro-Wilk test for normality.
}
\usage{
shapiro.test(x)
}
\arguments{
  \item{x}{a numeric vector of data values, the number of which must be
    between 3 and 5000.  Missing values are allowed.}
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the Shapiro-Wilk statistic.}
  \item{p.value}{the p-value for the test.}
  \item{method}{the character string \code{"Shapiro-Wilk normality test"}.}
  \item{data.name}{a character string giving the name(s) of the data.}
}
\references{
  Patrick Royston (1982)
  An Extension of Shapiro and Wilk's \eqn{W} Test for Normality to Large
  Samples.
  \emph{Applied Statistics}, \bold{31}, 115--124.

  Patrick Royston (1982)
  Algorithm AS 181: The \eqn{W} Test for Normality.
  \emph{Applied Statistics}, \bold{31}, 176--180.

  Patrick Royston (1995)
  A Remark on Algorithm AS 181: The \eqn{W} Test for Normality.
  \emph{Applied Statistics}, \bold{44}, 547--551.
}
\seealso{
  \code{\link[base:qqnorm]{qqnorm}} for producing a normal quantile-quantile plot.
}
\examples{
% FIXME: could use something more interesting here
shapiro.test(rnorm(100, mean = 5, sd = 3))
shapiro.test(runif(100, min = 2, max = 4))
}
\keyword{htest}
\eof
\name{smooth}
\alias{smooth}
%\alias{print.tukeysmooth}
%\alias{summary.tukeysmooth}
\title{Tukey's (Running Median) Smoothing}
\description{
  Tukey's smoothers, \emph{3RS3R}, \emph{3RSS}, \emph{3R}, etc.
}
\usage{
smooth(x, kind = c("3RS3R", "3RSS", "3RSR", "3R", "3", "S"),
       twiceit = FALSE,
       endrule = "Tukey", do.ends = FALSE)
}
\arguments{
  \item{x}{a vector or time series}
  \item{kind}{a character string indicating the kind of smoother required;
    defaults to \code{"3RS3R"}.}
  \item{twiceit}{logical, indicating if the result should be \dQuote{twiced}.
    Twicing a smoother \eqn{S(y)} means \eqn{S(y) + S(y - S(y))}, i.e.,
    adding smoothed residuals to the smoothed values.  This decreases
    bias (increasing variance).}
  \item{endrule}{a character string indicating the rule for smoothing at the
    boundary.  Either \code{"Tukey"} (default) or \code{"copy"}.}
  \item{do.ends}{logical, indicating if the 3-splitting of ties should
    also happen at the boundaries (\dQuote{ends}).  This is only used for
    \code{kind = "S"}.}
}
\details{
  \emph{\code{3}} is Tukey's short notation for running \code{\link{median}}s
  of length \bold{3},
  \cr
  \emph{\code{3R}} stands for \bold{R}epeated \emph{\code{3}} until
  convergence, and
  \cr
  \emph{\code{S}} for \bold{S}plitting of horizontal stretches of length 2 or 3.

  Hence, \emph{\code{3RS3R}} is a concatenation of \code{3R}, \code{S}
  and \code{3R}, \emph{\code{3RSS}} similarly,
  whereas \emph{\code{3RSR}} means first \code{3R}
  and then \code{(S and 3)} \bold{R}epeated until convergence -- which
  can be bad.
}
\value{
  An object of class \code{"tukeysmooth"} (which has \code{print} and
  \code{summary} methods) and is a vector or time series containing the
  smoothed values with additional attributes.
}
\note{
  S and S-PLUS use a different (somewhat better) Tukey smoother in
  \code{smooth(*)}.
  Note that there are other smoothing methods which provide
  rather better results.  These were designed for hand calculations
  and may be used mainly for didactical purposes.

  Since \R version 1.2, \code{smooth} \emph{does} really implement
  Tukey's end-point rule correctly (see argument \code{endrule}).

  \code{kind = "3RSR"} has been the default till \R-1.1,
  but it can have very bad properties, see the examples.

  Note that repeated application of \code{smooth(*)} \emph{does}
  smooth more, for the \code{"3RS*"} kinds.
}
\references{
  Tukey, J. W. (1977).
  \emph{Exploratory Data Analysis},
  Reading Massachusetts: Addison-Wesley.
}
\seealso{
  \code{\link{lowess}};
  \code{\link{loess}},
  \code{\link{supsmu}} and
  \code{\link{smooth.spline}}.
}
\examples{
## see also   demo(smooth) !

x1 <- c(4, 1, 3, 6, 6, 4, 1, 6, 2, 4, 2) # very artificial
(x3R <- smooth(x1, "3R")) # 2 iterations of "3"
smooth(x3R, kind = "S")

sm.3RS <- function(x, ...)
   smooth(smooth(x, "3R", ...), "S", ...)

y <- c(1,1, 19:1)
plot(y, main = "misbehaviour of \"3RSR\"", col.main = 3)
lines(sm.3RS(y))
lines(smooth(y))
lines(smooth(y, "3RSR"), col = 3, lwd = 2)# the horror

x <- c(8:10,10, 0,0, 9,9)
plot(x, main = "breakdown of  3R  and  S  and hence  3RSS")
matlines(cbind(smooth(x,"3R"),smooth(x,"S"), smooth(x,"3RSS"),smooth(x)))

data(presidents)
presidents[is.na(presidents)] <- 0 # silly
summary(sm3 <- smooth(presidents, "3R"))
summary(sm2 <- smooth(presidents,"3RSS"))
summary(sm  <- smooth(presidents))

all.equal(c(sm2),c(smooth(smooth(sm3, "S"), "S"))) # 3RSS  === 3R S S
all.equal(c(sm), c(smooth(smooth(sm3, "S"), "3R")))# 3RS3R === 3R S 3R

plot(presidents, main = "smooth(presidents0, *) :  3R and default 3RS3R")
lines(sm3,col = 3, lwd = 1.5)
lines(sm, col = 2, lwd = 1.25)
}
\keyword{robust}
\keyword{smooth}
\eof
\name{smooth.spline}
\alias{smooth.spline}
%\alias{print.smooth.spline}
\title{Fit a Smoothing Spline}
\description{
  Fits a cubic smoothing spline to the supplied data.
}
\usage{
smooth.spline(x, y = NULL, w = NULL, df, spar = NULL,
              cv = FALSE, all.knots = FALSE, nknots = NULL,
              df.offset = 0, penalty = 1, control.spar = list())
}
\arguments{
 \item{x}{a vector giving the values of the predictor variable, or  a
  list or a two-column matrix specifying x and y. }
 \item{y}{responses. If \code{y} is missing, the responses are assumed
   to be specified by \code{x}.}
 \item{w}{optional vector of weights of the same length as \code{x};
   defaults to all 1.}
 \item{df}{the desired equivalent number of degrees of freedom (trace of
   the smoother matrix).}
 \item{spar}{smoothing parameter, typically (but not necessarily) in
   \eqn{(0,1]}.  The coefficient \eqn{\lambda} of the integral of the
   squared second derivative in the fit (penalized log likelihood)
   criterion is a monotone function of \code{spar}, see the details
   below.}
 \item{cv}{ordinary (\code{TRUE}) or \dQuote{generalized} cross-validation
   (GCV) when \code{FALSE}.}
 \item{all.knots}{if \code{TRUE}, all distinct points in \code{x} are used as
   knots.  If \code{FALSE} (default), a subset of \code{x[]} is used,
   specifically \code{x[j]} where the \code{nknots} indices are evenly
   spaced in \code{1:n}, see also the next argument \code{nknots}.}
 \item{nknots}{integer giving the number of knots to use when
   \code{all.knots=FALSE}.  Per default, this is less than \eqn{n}, the
   number of unique \code{x} values for \eqn{n > 49}.}
 \item{df.offset}{allows the degrees of freedom to be increased by
   \code{df.offset} in the GCV criterion.}
 \item{penalty}{the coefficient of the penalty for degrees of freedom
   in the GCV criterion.}
 \item{control.spar}{optional list with named components controlling the
   root finding when the smoothing parameter \code{spar} is computed,
   i.e., missing or \code{NULL}, see below.

   \bold{Note} that this is partly \emph{experimental} and may change
   with general spar computation improvements!

   \describe{
     \item{low:}{lower bound for \code{spar}; defaults to -1.5 (used to
       implicitly default to 0 in \R versions earlier than 1.4).}
     \item{high:}{upper bound for \code{spar}; defaults to +1.5.}
     \item{tol:}{the absolute precision (\bold{tol}erance) used; defaults
     to 1e-4 (formerly 1e-3).}
     \item{eps:}{the relative precision used; defaults to 2e-8 (formerly
       0.00244).}
     \item{trace:}{logical indicating if iterations should be traced.}
     \item{maxit:}{integer giving the maximal number of iterations;
       defaults to 500.}
   }
   Note that \code{spar} is only searched for in the interval
   \eqn{[low, high]}.
 }
}
\details{
  The \code{x} vector should contain at least four distinct values.
  \emph{Distinct} here means \dQuote{distinct after rounding to 6 significant
  digits}, i.e., \code{x} will be transformed to
  \code{unique(sort(signif(x, 6)))}, and \code{y} and \code{w} are
  pooled accordingly.

  The computational \eqn{\lambda} used (as a function of
  \eqn{s=spar}{\code{spar}}) is
  \eqn{\lambda = r * 256^{3 s - 1}}{lambda = r * 256^(3*spar - 1)}
  where
  \eqn{r = tr(X' W X) / tr(\Sigma)},
  \eqn{\Sigma} is the matrix given by
  \eqn{\Sigma_{ij} = \int B_i''(t) B_j''(t) dt}{%
    Sigma[i,j] = Integral B''[i](t) B''[j](t) dt},
  \eqn{X} is given by \eqn{X_{ij} = B_j(x_i)}{X[i,j] = B[j](x[i])},
  \eqn{W} is the diagonal matrix of weights (scaled such that
  its trace is \eqn{n}, the original number of observations)
  and \eqn{B_k(.)}{B[k](.)} is the \eqn{k}-th B-spline.

  Note that with these definitions, \eqn{f_i = f(x_i)}, and the B-spline
  basis representation \eqn{f = X c} (i.e., \eqn{c} is
  the vector of spline coefficients), the penalized log likelihood is
  \eqn{L = (y - f)' W (y - f) + \lambda c' \Sigma c}, and hence
  \eqn{c} is the solution of the (ridge regression)
  \eqn{(X' W X + \lambda \Sigma) c = X' W y}.

  If \code{spar} is missing or \code{NULL}, the value of \code{df} is used to
  determine the degree of smoothing.  If both are missing, leave-one-out
  cross-validation (ordinary or \dQuote{generalized} as determined by
  \code{cv}) is used to determine \eqn{\lambda}.
  Note that from the above relation,
%%  lam      = r * 256^(3s - 1)
%%  log(lam) = log(r) + (3s - 1) * log(256)
%% (log(lam) - log(r)) / log(256)  = 3s - 1
%% s = [1 +  {log(lam) - log(r)} / {8 log(2)} ] / 3
%%   = 1/3 + {log(lam) - log(r)} / {24 log(2)}
%%   = 1/3 - log(r)/{24 log(2)} +  log(lam) / {24 log(2)}
%%   =               s0         + 0.0601 * log(lam)
  \code{spar} is \eqn{s = s0 + 0.0601 * \bold{\log}\lambda}{%
                   spar = s0 + 0.0601 * log(lambda)},
  which is intentionally \emph{different} from the S-plus implementation
  of \code{smooth.spline} (where \code{spar} is proportional to
  \eqn{\lambda}).  In \R's (\eqn{\log \lambda}) scale, it makes more
  sense to vary \code{spar} linearly.

  Note however that currently the results may become very unreliable
  for \code{spar} values smaller than about -1 or -2.  The same may
  happen for values larger than 2 or so. Don't think of setting
  \code{spar} or the controls \code{low} and \code{high} outside such a
  safe range, unless you know what you are doing!

  The \dQuote{generalized} cross-validation method will work correctly when
  there are duplicated points in \code{x}.  However, it is ambiguous what
  leave-one-out cross-validation means with duplicated points, and the
  internal code uses an approximation that involves leaving out groups
  of duplicated points.  \code{cv=TRUE} is best avoided in that case.
}
\note{
  The default \code{all.knots = FALSE} and \code{nknots = NULL} entails
  using only \eqn{O(n^{0.2})}
  knots instead of \eqn{n} for \eqn{n > 49}.  This cuts speed and memory
  requirements, but not drastically anymore since \R version 1.5.1 where
  it is only \eqn{O(n_k) + O(n)}{O(nk) + O(n)} where \eqn{n_k}{nk} is
  the number of knots.
  In this case where not all unique \code{x} values are
  used as knots, the result is not a smoothing spline in the strict
  sense, but very close unless a small smoothing parameter (or large
  \code{df}) is used.
}
\value{
  An object of class \code{"smooth.spline"} with components
  \item{x}{the \emph{distinct} \code{x} values in increasing order, see
  the \bold{Details} above.}
  \item{y}{the fitted values corresponding to \code{x}.}
  \item{w}{the weights used at the unique values of \code{x}.}
  \item{yin}{the y values used at the unique \code{y} values.}
  \item{lev}{leverages, the diagonal values of the smoother matrix.}
  \item{cv.crit}{cross-validation score, \dQuote{generalized} or true, depending
    on \code{cv}.}
  \item{pen.crit}{penalized criterion}
  \item{crit}{the criterion value minimized in the underlying
    \code{.Fortran} routine \file{sslvrg}.}
  \item{df}{equivalent degrees of freedom used.  Note that (currently)
    this value may become quite unprecise when the true \code{df} is
    between and 1 and 2.
  }
  \item{spar}{the value of \code{spar} computed or given.}
  \item{lambda}{the value of \eqn{\lambda} corresponding to \code{spar},
    see the details above.}
  \item{iparms}{named integer(3) vector where \code{..$ipars["iter"]}
    gives number of spar computing iterations used.}
  \item{fit}{list for use by \code{\link{predict.smooth.spline}}, with
    components
    \describe{
      \item{knot:}{the knot sequence (including the repeated boundary
	knots).}
      \item{nk:}{number of coefficients or number of \dQuote{proper}
	knots plus 2.}
      \item{coef:}{coefficients for the spline basis used.}
      \item{min, range:}{numbers giving the corresponding quantities of
	\code{x}.}
    }
  }
  \item{call}{the matched call.}
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical Models in S}, Wadsworth \& Brooks/Cole.

  Green, P. J. and Silverman, B. W. (1994)
  \emph{Nonparametric Regression and Generalized Linear Models:
    A Roughness Penalty Approach.} Chapman and Hall.

  Hastie, T. J. and Tibshirani, R. J. (1990)
  \emph{Generalized Additive Models.}  Chapman and Hall.
}
\author{
  \R implementation by B. D. Ripley and Martin Maechler
  (\code{spar/lambda}, etc).

  This function is based on code in the \code{GAMFIT} Fortran program by
  T. Hastie and R. Tibshirani (\url{http://lib.stat.cmu.edu/general/}),
  which makes use of spline code by Finbarr O'Sullivan.  Its design
  parallels the \code{smooth.spline} function of Chambers \& Hastie (1992).
}
\seealso{
  \code{\link{predict.smooth.spline}} for evaluating the spline
  and its derivatives.
}
\examples{
data(cars)
attach(cars)
plot(speed, dist, main = "data(cars)  &  smoothing splines")
cars.spl <- smooth.spline(speed, dist)
(cars.spl)
## This example has duplicate points, so avoid cv=TRUE
\dontshow{
  all(cars.spl $ w == table(speed)) # TRUE (weights = multiplicities)
  str(cars.spl, digits=5, vec.len=6)
  cars.spl$fit
}
lines(cars.spl, col = "blue")
lines(smooth.spline(speed, dist, df=10), lty=2, col = "red")
legend(5,120,c(paste("default [C.V.] => df =",round(cars.spl$df,1)),
               "s( * , df = 10)"), col = c("blue","red"), lty = 1:2,
       bg='bisque')
detach()

##-- artificial example
y18 <- c(1:3,5,4,7:3,2*(2:5),rep(10,4))
xx  <- seq(1,length(y18), len=201)
(s2  <- smooth.spline(y18)) # GCV
(s02 <- smooth.spline(y18, spar = 0.2))
plot(y18, main=deparse(s2$call), col.main=2)
lines(s2, col = "gray"); lines(predict(s2, xx), col = 2)
lines(predict(s02, xx), col = 3); mtext(deparse(s02$call), col = 3)

## The following shows the problematic behavior of 'spar' searching:
(s2  <- smooth.spline(y18,          con=list(trace=TRUE,tol=1e-6, low= -1.5)))
(s2m <- smooth.spline(y18, cv=TRUE, con=list(trace=TRUE,tol=1e-6, low= -1.5)))
## both above do quite similarly (Df = 8.5 +- 0.2)

\dontshow{
% not for the user
  s2. <- smooth.spline(y18, cv=TRUE,con=list(trace=TRUE, tol=1e-6,low= -3,maxit=20))
  s2. ## Intel-Linux: Df ~= (even! > ) 18 : interpolating -- much smaller PRESS
     ## {others, e.g., may end quite differently!}
  lines(predict(s2., xx), col = 4)
  mtext(deparse(s2.$call,200), side= 1, line= -1, cex= 0.8, col= 4)

  sdf8 <- smooth.spline(y18, df = 8, con=list(trace=TRUE))
  sdf8 ; sdf8$df - 8

  try(smooth.spline(y18, spar = 50)) #>> error : spar 'way too large'
}
}
\keyword{smooth}
\eof
\name{smoothEnds}
\title{End Points Smoothing (for Running Medians)}
\alias{smoothEnds}
\usage{
smoothEnds(y, k = 3)
}
\description{
  Smooth end points of a vector \code{y} using subsequently smaller
  medians and Tukey's end point rule at the very end. (of odd span),
}
\arguments{
  \item{y}{dependent variable to be smoothed (vector).}
  \item{k}{width of largest median window; must be odd.}
}
\value{vector of smoothed values, the same length as \code{y}.
}
\details{
  \code{smoothEnds} is used to only do the \dQuote{end point smoothing},
  i.e., change at most the observations closer to the beginning/end
  than half the window \code{k}.  The first and last value are computed using
  \dQuote{\emph{Tukey's end point rule}}, i.e.,
  \code{sm[1] = median(y[1], sm[2], 3*sm[2] - 2*sm[3])}.
}
\references{
  John W. Tukey (1977)
  \emph{Exploratory Data Analysis}, Addison.

  Velleman, P.F., and Hoaglin, D.C. (1981)
  \emph{ABC of EDA (Applications, Basics, and Computing of Exploratory
    Data Analysis)}; Duxbury.
}
\author{Martin Maechler}
\seealso{
  \code{\link{runmed}(*, end.rule = "median")} which calls
  \code{smoothEnds()}.
}
\examples{
y <- ys <- (-20:20)^2
y [c(1,10,21,41)] <-  c(100, 30, 400, 470)
s7k <- runmed(y,7, end = "keep")
s7. <- runmed(y,7, end = "const")
s7m <- runmed(y,7)
col3 <- c("midnightblue","blue","steelblue")
plot(y, main = "Running Medians -- runmed(*, k=7, end.rule = X)")
lines(ys, col = "light gray")
matlines(cbind(s7k,s7.,s7m), lwd= 1.5, lty = 1, col = col3)
legend(1,470, paste("end.rule",c("keep","constant","median"),sep=" = "),
       col = col3, lwd = 1.5, lty = 1)

stopifnot(identical(s7m, smoothEnds(s7k, 7)))
}
\keyword{smooth}
\keyword{robust}
\eof
% $Id: sortedXyData.Rd,v 1.1 2003/12/11 07:16:06 ripley Exp $
\name{sortedXyData}
\title{Create a sortedXyData object}
\alias{sortedXyData}
\alias{sortedXyData.default}
\usage{
sortedXyData(x, y, data)
}
\arguments{
 \item{x}{ a numeric vector or an expression that will evaluate in
   \code{data} to a numeric vector }
 \item{y}{ a numeric vector or an expression that will evaluate in
   \code{data} to a numeric vector }
 \item{data}{ an optional data frame in which to evaluate expressions
   for \code{x} and \code{y}, if they are given as expressions }
}
\description{
  This is a constructor function for the class of \code{sortedXyData}
  objects.  These objects are mostly used in the \code{initial}
  function for a self-starting nonlinear regression model, which will be
  of the \code{selfStart} class.
}
\value{
  A \code{sortedXyData} object. This is a data frame with exactly
  two numeric columns, named \code{x} and \code{y}.  The rows are
  sorted so the \code{x} column is in increasing order.  Duplicate
  \code{x} values are eliminated by averaging the corresponding \code{y}
  values.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{\code{\link{selfStart}}, \code{\link{NLSstClosestX}},
  \code{\link{NLSstLfAsymptote}}, \code{\link{NLSstRtAsymptote}}
}
\examples{
data( DNase )
DNase.2 <- DNase[ DNase$Run == "2", ]
sortedXyData( expression(log(conc)), expression(density), DNase.2 )
}
\keyword{manip}
\eof
\name{spec.ar}
\alias{spec.ar}
\title{Estimate Spectral Density of a Time Series from AR Fit}
\usage{
spec.ar(x, n.freq, order = NULL, plot = TRUE, na.action = na.fail,
        method = "yule-walker", \dots)
}
\arguments{
  \item{x}{A univariate (not yet:or multivariate) time series or the
    result of a fit by \code{\link{ar}}.}
  \item{n.freq}{The number of points at which to plot.}
  \item{order}{The order of the AR model to be fitted.  If omitted,
    the order is chosen by AIC.}
  \item{plot}{Plot the periodogram?}
  \item{na.action}{\code{NA} action function.}
  \item{method}{\code{method} for \code{ar} fit.}
  \item{\dots}{Graphical arguments passed to \code{\link{plot.spec}}.}
}
\description{
  Fits an AR model to \code{x} (or uses the existing fit) and computes
  (and by default plots) the spectral density of the fitted model.
}
\value{
  An object of class \code{"spec"}.
  The result is returned invisibly if \code{plot} is true.
}
\references{
  Thompson, D.J. (1990) Time series analysis of Holocene climate data.
  \emph{Phil. Trans. Roy. Soc. A} \bold{330}, 601--616.

  Venables, W.N. and Ripley, B.D. (2002) \emph{Modern Applied
    Statistics with S.} Fourth edition. Springer. (Especially
  page 402.)
}
\section{Warning}{
  Some authors, for example Thomson (1990), warn strongly
  that AR spectra can be misleading.
}

\seealso{\code{\link{ar}}, \code{\link{spectrum}}.}

\note{The multivariate case is not yet implemented.}

\examples{
data(lh)
spec.ar(lh)

data(UKLungDeaths)
spec.ar(ldeaths)
spec.ar(ldeaths, method="burg")
}
\keyword{ts}
\eof
\name{spec.pgram}
\alias{spec.pgram}
\title{Estimate Spectral Density of a Time Series by a Smoothed
    Periodogram}
\usage{
spec.pgram(x, spans = NULL, kernel, taper = 0.1,
           pad = 0, fast = TRUE, demean = FALSE, detrend =TRUE,
           plot = TRUE, na.action = na.fail, \dots)
}
\arguments{
  \item{x}{univariate or multivariate time series.}
  \item{spans}{vector of odd integers giving the widths of modified
    Daniell smoothers to be used to smooth the periodogram.}
  \item{kernel}{alternatively, a kernel smoother of class
    \code{"tskernel"}.}
  \item{taper}{proportion of data to taper.  A split cosine bell taper
    is applied to this proportion of the data at the beginning and
    end of the series.}
  \item{pad}{proportion of data to pad. Zeros are added to the end of
    the series to increase its length by the proportion \code{pad}.}
  \item{fast}{logical; if \code{TRUE}, pad the series to a highly composite
    length.}
  \item{demean}{logical. If \code{TRUE}, subtract the mean of the
    series.}
  \item{detrend}{logical. If \code{TRUE}, remove a linear trend from
    the series. This will also remove the mean.}
  \item{plot}{plot the periodogram?}
  \item{na.action}{\code{NA} action function.}
  \item{\dots}{graphical arguments passed to \code{plot.spec}.}
}
\description{
  \code{spec.pgram} calculates the periodogram using a fast Fourier
  transform, and optionally smooths the result with a series of
  modified Daniell smoothers (moving averages giving half weight to
  the end values).
}
\details{
  The raw periodogram is not a consistent estimator of the spectral density,
  but adjacent values are asymptotically independent. Hence a consistent
  estimator can be derived by smoothing the raw periodogram, assuming that
  the spectral density is smooth.

  The series will be automatically padded with zeros until the series
  length is a highly composite number in order to help the Fast Fourier
  Transform. This is controlled by the \code{fast} and not the \code{pad}
  argument.

  The periodogram at zero is in theory zero as the mean of the series
  is removed (but this may be affected by tapering): it is replaced by
  an interpolation of adjacent values during smoothing, and no value
  is returned for that frequency.
}
\value{
  A list object of class \code{"spec"} (see \code{\link{spectrum}})
  with the following additional components:
  \item{kernel}{The \code{kernel} argument, or the kernel constructed
    from \code{spans}.}
  \item{df}{The distribution of the spectral density estimate can be
    approximated by a chi square distribution with \code{df} degrees
    of freedom.}
  \item{bandwidth}{The equivalent bandwidth of the kernel smoother as
    defined by Bloomfield (1976, page 201).}
  \item{taper}{The value of the \code{taper} argument.}
  \item{pad}{The value of the \code{pad} argument.}
  \item{detrend}{The value of the \code{detrend} argument.}
  \item{demean}{The value of the \code{demean} argument.}

  The result is returned invisibly if \code{plot} is true.
}

\references{
  Bloomfield, P. (1976) \emph{Fourier Analysis of Time Series: An
    Introduction.} Wiley.

  Brockwell, P.J. and Davis, R.A. (1991) \emph{Time Series: Theory and
    Methods.} Second edition. Springer.

  Venables, W.N. and Ripley, B.D. (2002) \emph{Modern Applied
    Statistics with S.} Fourth edition. Springer. (Especially
  pp. 392--7.)
}
\author{
  Originally Martyn Plummer; kernel smoothing by Adrian Trapletti,
  synthesis by B.D. Ripley
}
\seealso{\code{\link{spectrum}}, \code{\link{spec.taper}},
  \code{\link{plot.spec}}, \code{\link{fft}}}

\examples{
## Examples from Venables & Ripley
data(UKLungDeaths)
spectrum(ldeaths)
spectrum(ldeaths, spans = c(3,5))
spectrum(ldeaths, spans = c(5,7))
spectrum(mdeaths, spans = c(3,3))
spectrum(fdeaths, spans = c(3,3))

## bivariate example
mfdeaths.spc <- spec.pgram(ts.union(mdeaths, fdeaths), spans = c(3,3))
# plots marginal spectra: now plot coherency and phase
plot(mfdeaths.spc, plot.type = "coherency")
plot(mfdeaths.spc, plot.type = "phase")

## now impose a lack of alignment
mfdeaths.spc <- spec.pgram(ts.intersect(mdeaths, lag(fdeaths, 4)),
   spans = c(3,3), plot = FALSE)
plot(mfdeaths.spc, plot.type = "coherency")
plot(mfdeaths.spc, plot.type = "phase")

data(EuStockMarkets)
stocks.spc <- spectrum(EuStockMarkets, kernel("daniell", c(30,50)),
                       plot = FALSE)
plot(stocks.spc, plot.type = "marginal") # the default type
plot(stocks.spc, plot.type = "coherency")
plot(stocks.spc, plot.type = "phase")

data(BJsales)
sales.spc <- spectrum(ts.union(BJsales, BJsales.lead),
                      kernel("modified.daniell", c(5,7)))
plot(sales.spc, plot.type = "coherency")
plot(sales.spc, plot.type = "phase")
}
\keyword{ts}
\eof
\name{spec.taper}
\alias{spec.taper}
\title{Taper a Time Series by a Cosine Bell}
\usage{
spec.taper(x, p=0.1)
}
\arguments{
 \item{x}{A univariate or multivariate time series}
 \item{p}{The total proportion to be tapered, either a scalar or a
   vector of the length of the number of series.}
}
\description{
  Apply a cosine-bell taper to a time series.
}
\details{
  The cosine-bell taper is applied to the first and last \code{p[i]/2}
  observations of time series \code{x[, i]}.
}
\value{
  A new time series object.
}
\author{Kurt Hornik, B.D. Ripley}
\note{From package \pkg{MASS}.}

\seealso{
  \code{\link{spec.pgram}},
  \code{\link{cpgram}}
}

\keyword{ts}
\eof
\name{spectrum}
\alias{spectrum}
\alias{spec}
\title{Spectral Density Estimation}
\description{
  The \code{spectrum} function estimates the spectral density of a
  time series.
}
\usage{
spectrum(x, method = c("pgram", "ar"), plot = TRUE, na.action = na.fail,
         \dots)
}
%% spectrum() dispatches to spec.pgram() or spec.ar() which have quite
%%            different extra arguments
\synopsis{
spectrum(\dots, method = c("pgram", "ar"))
}
\arguments{
  \item{x}{A univariate or multivariate time series.}
  \item{method}{String specifying the method used to estimate the
    spectral density.  Allowed methods are \code{"pgram"} (the default)
    and \code{"ar"}.} 
  \item{plot}{logical. If \code{TRUE} then the spectral density is
    plotted.}
  \item{na.action}{\code{NA} action function.}
  \item{\dots}{Further arguments to specific spec methods or
    \code{plot.spec}.}
}
\value{
  An object of class \code{"spec"}, which is a list containing at
  least the following components:
  \item{freq}{vector of frequencies at which the spectral
    density is estimated. (Possibly approximate Fourier frequencies.)
    The units are the reciprocal of cycles per unit time (and not per
    observation spacing): see Details below.}

  \item{spec}{Vector (for univariate series) or matrix (for multivariate
    series) of estimates of the spectral density at frequencies
    corresponding to \code{freq}.}

  \item{coh}{\code{NULL} for univariate series. For multivariate time
    series, a matrix containing the \emph{squared} coherency
    between different
    series. Column \eqn{ i + (j - 1) * (j - 2)/2} of \code{coh}
    contains the squared coherency between columns \eqn{i} and \eqn{j}
    of \code{x}, where \eqn{i < j}.}

  \item{phase}{\code{NULL} for univariate series. For multivariate
    time series a matrix containing the cross-spectrum phase between
    different series. The format is the same as \code{coh}.}

  \item{series}{The name of the time series.}

  \item{snames}{For multivariate input, the names of the component series.}

  \item{method}{The method used to calculate the spectrum.}

  The result is returned invisibly if \code{plot} is true.
}
\details{
  \code{spectrum} is a wrapper function which calls the methods
  \code{\link{spec.pgram}} and \code{\link{spec.ar}}.
  
  The spectrum here is defined with scaling \code{1/frequency(x)},
  following S-PLUS.  This makes the spectral density a density over the
  range \code{(-frequency(x)/2, +frequency(x)/2]}, whereas a more common
  scaling is \eqn{2\pi} and range \eqn{(-0.5, 0.5]} (e.g., Bloomfield)
  or 1 and range \eqn{(-\pi, \pi]}.
  
  If available, a confidence interval will be plotted by
  \code{plot.spec}: this is asymmetric, and the width of the centre
  mark indicates the equivalent bandwidth.
}
\note{
  The default plot for objects of class \code{"spec"} is quite complex,
  including an error bar and default title, subtitle and axis
  labels.  The defaults can all be overridden by supplying the
  appropriate graphical parameters.
}
\references{
  Bloomfield, P. (1976) \emph{Fourier Analysis of Time Series: An
    Introduction.} Wiley.

  Brockwell, P. J. and Davis, R. A. (1991) \emph{Time Series: Theory and
    Methods.} Second edition. Springer.

  Venables, W. N. and Ripley, B. D. (2002) \emph{Modern Applied
    Statistics with S-PLUS.} Fourth edition. Springer. (Especially
  pages 392--7.)
}
\author{Martyn Plummer, B.D. Ripley}
\seealso{
  \code{\link{spec.ar}},
  \code{\link{spec.pgram}};
  \code{\link{plot.spec}}.
}
\examples{
## Examples from Venables & Ripley
## spec.pgram
par(mfrow=c(2,2))
data(lh)
spectrum(lh)
spectrum(lh, spans=3)
spectrum(lh, spans=c(3,3))
spectrum(lh, spans=c(3,5))

data(UKLungDeaths)
spectrum(ldeaths)
spectrum(ldeaths, spans=c(3,3))
spectrum(ldeaths, spans=c(3,5))
spectrum(ldeaths, spans=c(5,7))
spectrum(ldeaths, spans=c(5,7), log="dB", ci=0.8)

# for multivariate examples see the help for spec.pgram

## spec.ar
spectrum(lh, method="ar")
spectrum(ldeaths, method="ar")
}
\keyword{ts}
\eof
\name{splinefun}
\alias{spline}
\alias{splinefun}
\title{Interpolating Splines}
\description{
  Perform cubic spline interpolation of given data points, returning
  either a list of points obtained by the interpolation or a function
  performing the interpolation.
}
\usage{
splinefun(x, y = NULL, method = "fmm")

spline(x, y = NULL, n = 3*length(x), method = "fmm",
       xmin = min(x), xmax = max(x))
}
\arguments{
  \item{x,y}{vectors giving the coordinates of the points to be
    interpolated.  Alternatively a single plotting structure can be
    specified: see \code{\link{xy.coords}.}}
  \item{method}{specifies the type of spline to be used.  Possible
    values are \code{"fmm"}, \code{"natural"} and \code{"periodic"}.}
  \item{n}{interpolation takes place at \code{n} equally spaced points
    spanning the interval [\code{xmin}, \code{xmax}].}
  \item{xmin}{left-hand endpoint of the interpolation interval.}
  \item{xmax}{right-hand endpoint of the interpolation interval.}
}
\details{
  If \code{method = "fmm"}, the spline used is that of Forsythe, Malcolm
  and Moler (an exact cubic is fitted through the four points at each
  end of the data, and this is used to determine the end conditions).
  Natural splines are used when \code{method = "natural"}, and periodic
  splines when \code{method = "periodic"}.

  These interpolation splines can also be used for extrapolation, that is
  prediction at points outside the range of \code{x}.  Extrapolation
  makes little sense for \code{method = "fmm"}; for natural splines it
  is linear using the slope of the interpolating curve at the nearest
  data point.
}
\value{
  \code{spline} returns a list containing components \code{x} and
  \code{y} which give the ordinates where interpolation took place and
  the interpolated values.

  \code{splinefun} returns a function which will perform cubic spline
  interpolation of the given data points.  This is often more useful
  than \code{spline}.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.

  Forsythe, G. E., Malcolm, M. A. and Moler, C. B. (1977)
  \emph{Computer Methods for Mathematical Computations}.
}
\seealso{
  \code{\link{approx}} and \code{\link{approxfun}} for constant and
  linear interpolation.

  Package \pkg{splines}, especially \code{\link[splines]{interpSpline}}
  and \code{\link[splines]{periodicSpline}} for interpolation splines.
  That package also generates spline bases that can be used for
  regression splines.

  \code{\link{smooth.spline}} in package \pkg{modreg} for
  smoothing splines.
}
\examples{
op <- par(mfrow = c(2,1), mgp = c(2,.8,0), mar = .1+c(3,3,3,1))
n <- 9
x <- 1:n
y <- rnorm(n)
plot(x, y, main = paste("spline[fun](.) through", n, "points"))
lines(spline(x, y))
lines(spline(x, y, n = 201), col = 2)

y <- (x-6)^2
plot(x, y, main = "spline(.) -- 3 methods")
lines(spline(x, y, n = 201), col = 2)
lines(spline(x, y, n = 201, method = "natural"), col = 3)
lines(spline(x, y, n = 201, method = "periodic"), col = 4)
legend(6,25, c("fmm","natural","periodic"), col=2:4, lty=1)

f <- splinefun(x, y)
ls(envir = environment(f))
splinecoef <- eval(expression(z), envir = environment(f))
curve(f(x), 1, 10, col = "green", lwd = 1.5)
points(splinecoef, col = "purple", cex = 2)
par(op)
}
\keyword{math}
\keyword{dplot}
\eof
\name{start}
\title{Encode the Terminal Times of Time Series}
\usage{
start(x, \dots)
end(x, \dots)
}
\alias{start}
\alias{end}
%\alias{start.default}
%\alias{end.default}
\arguments{
    \item{x}{a univariate or multivariate time-series, or a vector or matrix.}
    \item{\dots}{extra arguments for future methods.}
}
\description{
    Extract and encode the times the first and last observations were
    taken. Provided only for compatibility with S version 2.
}
\details{
    These are generic functions, which will use the
    \code{\link{tsp}} attribute of \code{x} if it exists.
    Their default methods decode the start time from the original time
    units, so that for a monthly series \code{1995.5} is represented
    as \code{c(1995, 7)}. For a series of frequency \code{f}, time
    \code{n+i/f} is presented as \code{c(n, i+1)} (even for \code{i = 0}
    and \code{f = 1}).
}
\section{Warning}{
    The representation used by \code{start} and \code{end} has no
    meaning unless the frequency is supplied.
}
\seealso{
  \code{\link{ts}}, \code{\link{time}}, \code{\link{tsp}}.
}
\keyword{ts}
\eof
\name{stat.anova}
\title{GLM Anova Statistics}
\usage{
stat.anova(table, test = c("Chisq", "F", "Cp"), scale, df.scale, n)
}
\alias{stat.anova}
\arguments{
 \item{table}{numeric matrix as results from
   \code{\link{anova.glm}(\dots, test=NULL)}.}
 \item{test}{a character string, matching one of \code{"Chisq"},
   \code{"F"} or \code{"Cp"}.}
 \item{scale}{a weighted residual sum of squares.}
 \item{df.scale}{degrees of freedom corresponding to scale.}
 \item{n}{number of observations.}
}
\description{This is a utility function, used in \code{lm} and
  \code{glm} methods for \code{\link{anova}(\dots, test != NULL)}
  and should not be used by the average user.
}
\value{
  A matrix which is the original \code{table}, augmented by a column
  of test statistics, depending on the \code{test} argument.
}
\references{
  Hastie, T. J. and Pregibon, D. (1992)
  \emph{Generalized linear models.}
  Chapter 6 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\seealso{\code{\link{anova.lm}}, \code{\link{anova.glm}}.}
\examples{
##-- Continued from '?glm':
\dontshow{example("glm", echo = FALSE)}
print(ag <- anova(glm.D93))
stat.anova(ag$table, test = "Cp",
           scale = sum(resid(glm.D93, "pearson")^2)/4, df = 4, n = 9)
}
\keyword{regression}
\keyword{models}
\eof
\name{step}
\alias{step}
\title{
Choose a model by AIC in a Stepwise Algorithm
}
\description{
  Select a formula-based model by AIC.
}
\usage{
step(object, scope, scale = 0,
     direction = c("both", "backward", "forward"), 
     trace = 1, keep = NULL, steps = 1000, k = 2, \dots)
}
\arguments{
  \item{object}{
    an object representing a model of an appropriate class (mainly
    \code{"lm"} and \code{"glm"}).
    This is used as the initial model in the stepwise search.
  }
  \item{scope}{
    defines the range of models examined in the stepwise search.
    This should be either a single formula, or a list containing
    components \code{upper} and \code{lower}, both formulae.  See the
    details for how to specify the formulae and how they are used.
  }
  \item{scale}{
    used in the definition of the AIC statistic for selecting the models,
    currently only for \code{\link{lm}}, \code{\link{aov}} and
    \code{\link{glm}} models.
  }
  \item{direction}{
    the mode of stepwise search, can be one of \code{"both"},
    \code{"backward"}, or \code{"forward"}, with a default of \code{"both"}. 
    If the \code{scope} argument is missing the default for
    \code{direction} is \code{"backward"}.
  }
  \item{trace}{
    if positive, information is printed during the running of \code{step}.
    Larger values may give more detailed information.
  }
  \item{keep}{
    a filter function whose input is a fitted model object and the 
    associated \code{AIC} statistic, and whose output is arbitrary. 
    Typically \code{keep} will select a subset of the components of 
    the object and return them. The default is not to keep anything.
  }
  \item{steps}{
    the maximum number of steps to be considered.  The default is 1000
    (essentially as many as required).  It is typically used to stop the
    process early.
  }
  \item{k}{
    the multiple of the number of degrees of freedom used for the penalty.
    Only \code{k = 2} gives the genuine AIC: \code{k = log(n)} is sometimes
    referred to as BIC or SBC.
  }
  \item{\dots}{
    any additional arguments to \code{\link{extractAIC}}.
  }
}
\value{
  the stepwise-selected model is returned, with up to two additional
  components.  There is an \code{"anova"} component corresponding to the
  steps taken in the search, as well as a \code{"keep"} component if the
  \code{keep=} argument was supplied in the call. The
  \code{"Resid. Dev"} column of the analysis of deviance table refers
  to a constant minus twice the maximized log likelihood: it will be a
  deviance only in cases where a saturated model is well-defined
  (thus excluding \code{lm}, \code{aov} and \code{survreg} fits,
  for example).
}
\details{
  \code{step} uses \code{\link{add1}} and \code{\link{drop1}}
  repeatedly; it will work for any method for which they work, and that
  is determined by having a valid method for \code{\link{extractAIC}}.
  When the additive constant can be chosen so that AIC is equal to
  Mallows' \eqn{C_p}{Cp}, this is done and the tables are labelled
  appropriately.

  The set of models searched is determined by the \code{scope} argument.
  The right-hand-side of its \code{lower} component is always included
  in the model, and right-hand-side of the model is included in the
  \code{upper} component.  If \code{scope} is a single formula, it
  specifes the \code{upper} component, and the \code{lower} model is
  empty.  If \code{scope} is missing, the initial model is used as the
  \code{upper} model.

  Models specified by \code{scope} can be templates to update
  \code{object} as used by \code{\link{update.formula}}.
  
  There is a potential problem in using \code{\link{glm}} fits with a
  variable \code{scale}, as in that case the deviance is not simply
  related to the maximized log-likelihood. The \code{"glm"} method for
  function \code{\link{extractAIC}} makes the
  appropriate adjustment for a \code{gaussian} family, but may need to be
  amended for other cases. (The \code{binomial} and \code{poisson}
  families have fixed \code{scale} by default and do not correspond
  to a particular maximum-likelihood problem for variable \code{scale}.)
}
\note{
  This function differs considerably from the function in S, which uses a
  number of approximations and does not compute the correct AIC.

  This is a minimal implementation. Use \code{\link[MASS]{stepAIC}} for
  a wider range of object classes.
}
\section{Warning}{
  The model fitting must apply the models to the same dataset. This
  may be a problem if there are missing values and \R's default of
  \code{na.action = na.omit} is used.  We suggest you remove the
  missing values first.
}
\seealso{
  \code{\link[MASS]{stepAIC}}, \code{\link{add1}}, \code{\link{drop1}}
}
\references{
  Hastie, T. J. and Pregibon, D. (1992)
  \emph{Generalized linear models.}
  Chapter 6 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.}
  New York: Springer (4th ed).
}
\author{
  B. D. Ripley: \code{step} is a slightly simplified version of
  \code{\link[MASS]{stepAIC}} in package \pkg{MASS} (Venables \&
  Ripley, 2002 and earlier editions).

  The idea of a \code{step} function follows that described in Hastie \&
  Pregibon (1992); but the implementation in \R is more general.
}
\examples{
example(lm)
step(lm.D9)  

data(swiss)
summary(lm1 <- lm(Fertility ~ ., data = swiss))
slm1 <- step(lm1)
summary(slm1)
slm1$anova
}
\keyword{models}
\eof
\name{stepfun}
\alias{stepfun}
\alias{is.stepfun}
\alias{as.stepfun}
%\alias{as.stepfun.default}
\alias{print.stepfun}
\alias{summary.stepfun}
\alias{knots}
%\alias{knots.stepfun}
\title{Step Function Class}
\usage{
stepfun(x, y, f = as.numeric(right), ties = "ordered", right = FALSE)

is.stepfun(x)
knots(Fn, \dots)
as.stepfun(x, \dots)

\method{print}{stepfun}(x, digits = getOption("digits") - 2, \dots)

\method{summary}{stepfun}(object, \dots)
}
\arguments{
  \item{x}{numeric vector giving the knots or jump locations of the step
    function for \code{stepfun()}.  For the other functions, \code{x} is
    as \code{object} below.}
  \item{y}{numeric vector one longer than \code{x}, giving the heights of
    the function values \emph{between} the x values.}
  \item{f}{a number between 0 and 1, indicating how interpolation outside
    the given x values should happen.  See \code{\link{approxfun}}.}
  \item{ties}{Handling of tied \code{x} values. Either a function or
    the string \code{"ordered"}.  See  \code{\link{approxfun}}.}
  \item{right}{logical, indicating if the intervals should be closed on
    the right (and open on the left) or vice versa.}

  \item{Fn, object}{an \R object inheriting from \code{"stepfun"}.}
  \item{digits}{number of significant digits to use, see \code{\link{print}}.}
  \item{\dots}{potentially further arguments (required by the generic).}
}
\description{
  Given the vectors \eqn{(x_1,\ldots, x_n)}{(x[1],\ldots, x[n])} and
  \eqn{(y_0,y_1,\ldots, y_n)}{(y[0],y[1],\ldots, y[n])} (one value
  more!), \code{stepfun(x,y,\dots)} returns an interpolating
  \dQuote{step} function, say \code{fn}. I.e., \eqn{fn(t) =
    c}\eqn{_i}{[i]} (constant) for \eqn{t \in (x_i, x_{i+1})}{t in (
    x[i], x[i+1])} and at the abscissa values, if (by default)
  \code{right = FALSE}, \eqn{fn(x_i) = y_i}{fn(x[i]) = y[i]} and for
  \code{right = TRUE}, \eqn{fn(x_i) = y_{i-1}}{fn(x[i]) = y[i-1]}, for
  \eqn{i=1,\ldots,n}.

  The value of the constant \eqn{c_i}{c[i]} above depends on the
  \dQuote{continuity} parameter \code{f}.
  For the default, \code{right = FALSE, f = 0},
  \code{fn} is a \dQuote{cadlag} function, i.e., continuous at right,
  limit (\dQuote{the point}) at left.
  In general, \eqn{c_i}{c[i]} is interpolated in between the
  neighbouring \eqn{y} values,
  \eqn{c_i= (1-f) y_i + f\cdot y_{i+1}}{c[i] = (1-f)*y[i] + f*y[i+1]}.
  Therefore, for non-0 values of \code{f}, \code{fn} may no longer be a proper
  step function, since it can be discontinuous from both sides, unless
  \code{right = TRUE, f = 1} which is right-continuous.
}
\value{
  A function of class \code{"stepfun"}, say \code{fn}.
%
  There are methods available for summarizing (\code{"summary(.)"}),
  representing (\code{"print(.)"}) and plotting  (\code{"plot(.)"}, see
  \code{\link{plot.stepfun}}) \code{"stepfun"} objects.

  The \code{\link{environment}} of \code{fn} contains all the
  information needed;
  \itemize{
    \item{"x","y"}{the original arguments}
    \item{"n"}{number of knots (x values)}
    \item{"f"}{continuity parameter}
    \item{"yleft", "yright"}{the function values \emph{outside} the knots;}
    \item{"method"}{(always \code{== "constant"}, from
      \code{\link{approxfun}(.)}).}
  }
  The knots are also available by \code{\link{knots}(fn)}.
}
\author{
  Martin Maechler, \email{maechler@stat.math.ethz.ch} with some basic
  code from Thomas Lumley.
}
\seealso{\code{\link{ecdf}} for empirical distribution functions as
  special step functions and \code{\link{plot.stepfun}} for \emph{plotting}
  step functions.

  \code{\link{approxfun}} and \code{\link{splinefun}}.
}
\examples{
y0 <- c(1,2,4,3)
sfun0  <- stepfun(1:3, y0, f = 0)
sfun.2 <- stepfun(1:3, y0, f = .2)
sfun1  <- stepfun(1:3, y0, f = 1)
sfun1c <- stepfun(1:3, y0, right=TRUE)# hence f=1
sfun0
summary(sfun0)
summary(sfun.2)

## look at the internal structure:
unclass(sfun0)
ls(envir = environment(sfun0))

x0 <- seq(0.5,3.5, by = 0.25)
rbind(x=x0, f.f0 = sfun0(x0), f.f02= sfun.2(x0),
      f.f1 = sfun1(x0), f.f1c = sfun1c(x0))
## Identities :
stopifnot(identical(y0[-1], sfun0 (1:3)),# right = FALSE
          identical(y0[-4], sfun1c(1:3)))# right = TRUE
}
\keyword{dplot}
\eof
\name{stl}
\alias{stl}
% all methods -> ./stlmethods.Rd
\title{Seasonal Decomposition of Time Series by Loess}
\description{
  Decompose a time series into seasonal, trend and irregular components
  using \code{loess}, acronym STL.
}
\usage{
stl(x, s.window, s.degree = 0,
    t.window = NULL, t.degree = 1,
    l.window = nextodd(period), l.degree = t.degree, 
    s.jump = ceiling(s.window/10),
    t.jump = ceiling(t.window/10),
    l.jump = ceiling(l.window/10),
    robust = FALSE,
    inner = if(robust)  1 else 2,
    outer = if(robust) 15 else 0,
    na.action = na.fail)
}
\arguments{
  \item{x}{univariate time series to be decomposed.
    This should be an object of class \code{"ts"} with a frequency
    greater than one.}
  \item{s.window}{either the character string \code{"periodic"} or the span (in
    lags) of the loess window for seasonal extraction, which should
    be odd.  This has no default.}
  \item{s.degree}{degree of locally-fitted polynomial in seasonal
    extraction.  Should be zero or one.} 
  \item{t.window}{the span (in lags) of the loess window for trend
    extraction, which should be odd.  If \code{NULL}, the default,
    \code{nextodd(ceiling((1.5*period) / (1-(1.5/s.window))))}, is taken.}
  \item{t.degree}{degree of locally-fitted polynomial in trend
    extraction.  Should be zero or one.}
  \item{l.window}{the span (in lags) of the loess window of the low-pass
    filter used for each subseries.  Defaults to the smallest odd
    integer greater than or equal to \code{frequency(x)} which is
    recommended since it prevents competition between the trend and
    seasonal components.  If not an odd integer its given value is
    increased to the next odd one.}
  \item{l.degree}{degree of locally-fitted polynomial for the subseries
    low-pass filter.  Must be 0 or 1.}
  \item{s.jump, t.jump, l.jump}{integers at least one to increase speed of
    the respective smoother.  Linear interpolation happens between every
    \code{*.jump}th value.}
  \item{robust}{logical indicating if robust fitting be used in the
    \code{loess} procedure.}
  \item{inner}{integer; the number of \sQuote{inner} (backfitting)
    iterations; usually very few (2) iterations suffice.}
  \item{outer}{integer; the number of \sQuote{outer} robustness
    iterations.}
  \item{na.action}{action on missing values.}
}
\details{
  The seasonal component is found by \emph{loess} smoothing the
  seasonal sub-series (the series of all January values, \ldots); if
  \code{s.window = "periodic"} smoothing is effectively replaced by
  taking the mean. The seasonal values are removed, and the remainder
  smoothed to find the trend. The overall level is removed from the
  seasonal component and added to the trend component. This process is
  iterated a few times.  The \code{remainder} component is the
  residuals from the seasonal plus trend fit.

  Several methods for the resulting class \code{"stl"} objects, see,
  \code{\link{plot.stl}}.
}
\value{
  \code{stl} returns an object of class \code{"stl"} with components
  \item{time.series}{a multiple time series with columns
    \code{seasonal}, \code{trend} and \code{remainder}.}
  \item{weights}{the final robust weights (all one if fitting is not
    done robustly).}
  \item{call}{the matched call.}
  \item{win}{integer (length 3 vector) with the spans used for the \code{"s"},
    \code{"t"}, and \code{"l"} smoothers.}
  \item{deg}{integer (length 3) vector with the polynomial degrees for
    these smoothers.}
  \item{jump}{integer (length 3) vector with the \dQuote{jumps} (skips)
    used for these smoothers.}
  \item{ni}{number of \bold{i}nner iterations}
  \item{no}{number of \bold{o}uter robustness iterations}
}
\references{
  R. B. Cleveland, W. S. Cleveland, J.E.  McRae, and I. Terpenning (1990)
  STL:  A  Seasonal-Trend  Decomposition  Procedure Based on Loess.
  \emph{Journal of Official Statistics}, \bold{6}, 3--73.
}
\author{B.D. Ripley; Fortran code by Cleveland \emph{et al.} (1990) from
  \file{netlib}.}

\note{This is similar to but not identical to the \code{stl} function in
  S-PLUS. The \code{remainder} component given by S-PLUS is the sum of
  the \code{trend} and \code{remainder} series from this function.}

\seealso{
  \code{\link{plot.stl}} for \code{stl} methods;
  \code{\link{loess}} in package \pkg{stats} (which is not actually
  used in \code{stl}).
} 
\examples{
data(nottem)
plot(stl(nottem, "per"))
plot(stl(nottem, s.win = 4, t.win = 50, t.jump = 1))
    
data(co2)
plot(stllc <- stl(log(co2), s.window=21))
summary(stllc)
## linear trend, strict period.
plot(stl(log(co2), s.window="per", t.window=1000))

## Two STL plotted side by side :    
data(UKLungDeaths)
        stmd <- stl(mdeaths, s.w = "per") # un-robust
summary(stmR <- stl(mdeaths, s.w = "per", robust = TRUE))
op <- par(mar = c(0, 4, 0, 3), oma = c(5, 0, 4, 0), mfcol = c(4, 2))
plot(stmd, set.pars=NULL, labels = NULL,
     main = "stl(mdeaths, s.w = \"per\",  robust = FALSE / TRUE )")
plot(stmR, set.pars=NULL)
# mark the 'outliers' :    
(iO <- which(stmR $ weights  < 1e-8)) # 10 were considered outliers
sts <- stmR$time.series
points(time(sts)[iO], .8* sts[,"remainder"][iO], pch = 4, col = "red")
par(op)# reset    
}
\keyword{ts}
\eof
\name{stlmethods}
%\alias{print.stl}
%\alias{summary.stl}
\alias{plot.stl}
\title{Methods for STL Objects}
\description{
  Methods for objects of class \code{stl}, typically the result of
  \code{\link{stl}}.  The \code{plot} method does a multiple figure plot
  with some flexibility.
}
\usage{
\method{plot}{stl}(x, labels = colnames(X),
     set.pars = list(mar = c(0, 6, 0, 6), oma = c(6, 0, 4, 0),
                     tck = -0.01, mfrow = c(nplot, 1)),
     main = NULL, range.bars = TRUE, \dots)
%\method{print}{stl}(x, \dots)
%\method{summary}{stl}(object, digits = getOption("digits"), \dots)
}
\arguments{
  \item{x}{\code{\link{stl}} object.}
  \item{labels}{character of length 4 giving the names of the component
    time-series.}
  \item{set.pars}{settings for \code{\link{par}(.)} when setting up the plot.}
  \item{main}{plot main title.}
  \item{range.bars}{logical indicating if each plot should have a bar at
    its right side which are of equal heights in user coordinates.}
%  \item{digits}{significant figures used in printing.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\seealso{\code{\link{plot.ts}} and \code{\link{stl}}, particularly for
  examples.}
\keyword{ts}
\eof
\name{summary.aov}
\alias{summary.aov}
\alias{summary.aovlist}
\alias{print.summary.aov}
\alias{print.summary.aovlist}
\title{Summarize an Analysis of Variance Model}
\usage{
\method{summary}{aov}(object, intercept = FALSE, split,
         expand.split = TRUE, keep.zero.df = TRUE, \dots)

\method{summary}{aovlist}(object, \dots)
}
\arguments{
  \item{object}{An object of class \code{"aov"} or \code{"aovlist"}.}
  \item{intercept}{logical: should intercept terms be included?}
  \item{split}{an optional named list, with names corresponding to terms
    in the model.  Each component is itself a list with integer
    components giving contrasts whose contributions are to be summed.}
  \item{expand.split}{logical: should the split apply also to
    interactions involving the factor?}
  \item{keep.zero.df}{logical: should terms with no degrees of freedom
    be included?}
  \item{\dots}{Arguments to be passed to or from other methods,
    for \code{summary.aovlist} including those for \code{summary.aov}.}
}
\description{
  Summarize an analysis of variance model.
}
\value{
  An object of class \code{c("summary.aov", "listof")} or
  \code{"summary.aovlist"} respectively.
}
\note{
  The use of \code{expand.split = TRUE} is little tested: it is always
  possible to set it to \code{FALSE} and specify exactly all
  the splits required.
}
\seealso{
  \code{\link{aov}}, \code{\link{summary}}, \code{\link{model.tables}},
  \code{\link{TukeyHSD}}
}
\examples{
## From Venables and Ripley (2002) p.165.
N <- c(0,1,0,1,1,1,0,0,0,1,1,0,1,1,0,0,1,0,1,0,1,1,0,0)
P <- c(1,1,0,0,0,1,0,1,1,1,0,0,0,1,0,1,1,0,0,1,0,1,1,0)
K <- c(1,0,0,1,0,1,1,0,0,1,0,1,0,1,1,0,0,0,1,1,1,0,1,0)
yield <- c(49.5,62.8,46.8,57.0,59.8,58.5,55.5,56.0,62.8,55.8,69.5,55.0,
           62.0,48.8,45.5,44.2,52.0,51.5,49.8,48.8,57.2,59.0,53.2,56.0)
npk <- data.frame(block=gl(6,4), N=factor(N), P=factor(P),
                  K=factor(K), yield=yield)

( npk.aov <- aov(yield ~ block + N*P*K, npk) )
summary(npk.aov)
coefficients(npk.aov)

# Cochran and Cox (1957, p.164)
# 3x3 factorial with ordered factors, each is average of 12. 
CC <- data.frame(
    y = c(449, 413, 326, 409, 358, 291, 341, 278, 312)/12,
    P = ordered(gl(3, 3)), N = ordered(gl(3, 1, 9))
)
CC.aov <- aov(y ~ N * P, data = CC , weights = rep(12, 9))
summary(CC.aov)

# Split both main effects into linear and quadratic parts.
summary(CC.aov, split = list(N = list(L = 1, Q = 2), P = list(L = 1, Q = 2)))

# Split only the interaction
summary(CC.aov, split = list("N:P" = list(L.L = 1, Q = 2:4)))

# split on just one var
summary(CC.aov, split = list(P = list(lin = 1, quad = 2)))
summary(CC.aov, split = list(P = list(lin = 1, quad = 2)),
        expand.split=FALSE)}
\keyword{models}
\keyword{regression}
\eof
\name{summary.glm}
\alias{summary.glm}
%\alias{summary.glm.null}
\alias{print.summary.glm}
%\alias{print.summary.glm.null}
\title{Summarizing Generalized Linear Model Fits}
\usage{
\method{summary}{glm}(object, dispersion = NULL, correlation = FALSE,
       symbolic.cor = FALSE, \dots)

\method{print}{summary.glm}(x, digits = max(3, getOption("digits") - 3),
      symbolic.cor = x$symbolic.cor,
      signif.stars = getOption("show.signif.stars"), \dots)
}
\arguments{
  \item{object}{an object of class \code{"glm"}, usually, a result of a
    call to \code{\link{glm}}.}
  \item{x}{an object of class \code{"summary.glm"}, usually, a result of a
    call to \code{summary.glm}.}
  \item{dispersion}{the dispersion parameter for the fitting family.
    By default it is obtained from \code{object}.}
  \item{correlation}{logical; if \code{TRUE}, the correlation matrix of
    the estimated parameters is returned and printed.}
  \item{digits}{the number of significant digits to use when printing.}
  \item{symbolic.cor}{logical. If \code{TRUE}, print the correlations in
    a symbolic form (see \code{\link{symnum}}) rather than as numbers.}
  \item{signif.stars}{logical. If \code{TRUE}, \dQuote{significance stars}
    are printed for each coefficient.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  These functions are all \code{\link{methods}} for class \code{glm} or
  \code{summary.glm} objects.
}
\details{
  \code{print.summary.glm} tries to be smart about formatting the
  coefficients, standard errors, etc. and additionally gives
  \dQuote{significance stars} if \code{signif.stars} is \code{TRUE}.

  Aliased coefficients are omitted in the returned object but (as from \R
  1.8.0) restored by the \code{print} method.

  Correlations are printed to two decimal places (or symbolically): to
  see the actual correlations print \code{summary(object)$correlation}
  directly.
}
\value{
  \code{summary.glm} returns an object of class \code{"summary.glm"}, a
  list with components

  \item{call}{the component from \code{object}.}
  \item{family}{the component from \code{object}.}
  \item{deviance}{the component from \code{object}.}
  \item{contrasts}{the component from \code{object}.}
  \item{df.residual}{the component from \code{object}.}
  \item{null.deviance}{the component from \code{object}.}
  \item{df.null}{the component from \code{object}.}
  \item{deviance.resid}{the deviance residuals:
    see \code{\link{residuals.glm}}.}
  \item{coefficients}{the matrix of coefficients, standard errors,
    z-values and p-values.  Aliased coefficients are omitted.}
  \item{aliased}{named logical vector showing if the original
    coefficients are aliased.} 
  \item{dispersion}{either the supplied argument or the estimated
    dispersion if the latter in \code{NULL}}
  \item{df}{a 3-vector of the rank of the model and the number of
    residual degrees of freedom, plus number of non-aliased coefficients.}
  \item{cov.unscaled}{the unscaled (\code{dispersion = 1}) estimated covariance
    matrix of the estimated coefficients.}
  \item{cov.scaled}{ditto, scaled by \code{dispersion}.}
  \item{correlation}{(only if \code{correlation} is true.)  The estimated
    correlations of the estimated coefficients.}
  \item{symbolic.cor}{(only if \code{correlation} is true.)  The value
    of the argument \code{symbolic.cor}.}
}
\seealso{
  \code{\link{glm}}, \code{\link{summary}}.
}
\examples{
## --- Continuing the Example from  '?glm':%\code{\link{glm}}:
\dontshow{example("glm", echo = FALSE)}
summary(glm.D93)
}
\keyword{models}
\keyword{regression}

\eof
\name{summary.lm}
\alias{summary.lm}
% \alias{summary.lm.null}
\alias{summary.mlm}% mlm is *not* mentioned at all
\alias{print.summary.lm}
% \alias{print.summary.lm.null}
\title{Summarizing Linear Model Fits}
\usage{
\method{summary}{lm}(object, correlation = FALSE, symbolic.cor = FALSE, \dots)

\method{print}{summary.lm}(x, digits = max(3, getOption("digits") - 3),
      symbolic.cor = x$symbolic.cor,
      signif.stars = getOption("show.signif.stars"), \dots)
}
\arguments{
  \item{object}{an object of class \code{"lm"}, usually, a result of a
    call to \code{\link{lm}}.}
  \item{x}{an object of class \code{"summary.lm"}, usually, a result of a
    call to \code{summary.lm}.}
  \item{correlation}{logical; if \code{TRUE}, the correlation matrix of
    the estimated parameters is returned and printed.}
  \item{digits}{the number of significant digits to use when printing.}
  \item{symbolic.cor}{logical. If \code{TRUE}, print the correlations in
    a symbolic form (see \code{\link{symnum}}) rather than as numbers.}
  \item{signif.stars}{logical. If \code{TRUE}, \dQuote{significance stars}
    are printed for each coefficient.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
\code{summary} method for class \code{"lm"}.
}
\details{
  \code{print.summary.lm} tries to be smart about formatting the
  coefficients, standard errors, etc. and additionally gives
  \dQuote{significance stars} if \code{signif.stars} is \code{TRUE}.

  Correlations are printed to two decimal places (or symbolically): to
  see the actual correlations print \code{summary(object)$correlation}
  directly.
}
\value{
  The function \code{summary.lm} computes and returns a list of summary
  statistics of the fitted linear model given in \code{object}, using
  the components (list elements) \code{"call"} and \code{"terms"}
  from its argument, plus
  \item{residuals}{the \emph{weighted} residuals, the usual residuals
    rescaled by the square root of the weights specified in the call to
    \code{lm}.}
  \item{coefficients}{a \eqn{p \times 4}{p x 4} matrix with columns for
    the estimated coefficient, its standard error, t-statistic and
    corresponding (two-sided) p-value.  Aliased coefficients are omitted.}
  \item{aliased}{named logical vector showing if the original
    coefficients are aliased.} 
  \item{sigma}{the square root of the estimated variance of the random
    error
    \deqn{\hat\sigma^2 = \frac{1}{n-p}\sum_i{R_i^2},}{%
      sigma^2 = 1/(n-p) Sum(R[i]^2),}
    where \eqn{R_i}{R[i]} is the \eqn{i}-th residual, \code{residuals[i]}.}
  \item{df}{degrees of freedom, a 3-vector \eqn{(p, n-p, p*)}, the last
    being the number of non-aliased coefficients.}
  \item{fstatistic}{(for models including non-intercept terms)
    a 3-vector with the value of the F-statistic with
    its numerator and denominator degrees of freedom.}
  \item{r.squared}{\eqn{R^2}, the \dQuote{fraction of variance explained by
    the model},
    \deqn{R^2 = 1 - \frac{\sum_i{R_i^2}}{\sum_i(y_i- y^*)^2},}{%
      R^2 = 1 - Sum(R[i]^2) / Sum((y[i]- y*)^2),}
    where \eqn{y^*}{y*} is the mean of \eqn{y_i}{y[i]} if there is an
    intercept and zero otherwise.}
  \item{adj.r.squared}{the above \eqn{R^2} statistic
    \dQuote{\emph{adjusted}}, penalizing for higher \eqn{p}.}
  \item{cov.unscaled}{a \eqn{p \times p}{p x p} matrix of (unscaled)
    covariances of the \eqn{\hat\beta_j}{coef[j]}, \eqn{j=1, \dots, p}.}
  \item{correlation}{the correlation matrix corresponding to the above
    \code{cov.unscaled}, if \code{correlation = TRUE} is specified.}
  \item{symbolic.cor}{(only if \code{correlation} is true.)  The value
    of the argument \code{symbolic.cor}.}
}
\seealso{
  The model fitting function \code{\link{lm}}, \code{\link{summary}}.

  Function \code{\link{coef}} will extract the matrix of coefficients
  with standard errors, t-statistics and p-values.
}
\examples{
\dontshow{example("lm", echo = FALSE)}
##-- Continuing the  lm(.) example:
coef(lm.D90)# the bare coefficients
sld90 <- summary(lm.D90 <- lm(weight ~ group -1))# omitting intercept
sld90
coef(sld90)# much more
}
\keyword{regression}
\keyword{models}
\eof
\name{summary.manova}
\alias{summary.manova}
\alias{print.summary.manova}
\title{Summary Method for Multivariate Analysis of Variance}
\description{
  A \code{summary} method for class \code{"manova"}.
}
\usage{
\method{summary}{manova}(object,
        test = c("Pillai", "Wilks", "Hotelling-Lawley", "Roy"),
        intercept = FALSE, \dots)
}
\arguments{
  \item{object}{An object of class \code{"manova"} or an \code{aov}
    object with multiple responses.}
  \item{test}{The name of the test statistic to be used. Partial
    matching is used so the name can be abbreviated.}
  \item{intercept}{logical. If \code{TRUE}, the intercept term is
    included in the table.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\details{
  The \code{summary.manova} method uses a multivariate test statistic
  for the summary table.  Wilks' statistic is most popular in the
  literature, but the default Pillai-Bartlett statistic is recommended
  by Hand and Taylor (1987).
}
\value{
  A list with components
  \item{SS}{A named list of sums of squares and product matrices.}
  \item{Eigenvalues}{A matrix of eigenvalues.}
  \item{stats}{A matrix of the statistics, approximate F value,
    degrees of freedom and P value.}
}
\references{
  Krzanowski, W. J. (1988) \emph{Principles of Multivariate Analysis. A
    User's Perspective.} Oxford.

  Hand, D. J. and Taylor, C. C.  (1987)
  \emph{Multivariate Analysis of Variance and Repeated Measures.}
  Chapman and Hall.
}
\seealso{
  \code{\link{manova}}, \code{\link{aov}}
}

\examples{
## Example on producing plastic film from Krzanowski (1998, p. 381)
tear <- c(6.5, 6.2, 5.8, 6.5, 6.5, 6.9, 7.2, 6.9, 6.1, 6.3,
          6.7, 6.6, 7.2, 7.1, 6.8, 7.1, 7.0, 7.2, 7.5, 7.6)
gloss <- c(9.5, 9.9, 9.6, 9.6, 9.2, 9.1, 10.0, 9.9, 9.5, 9.4,
           9.1, 9.3, 8.3, 8.4, 8.5, 9.2, 8.8, 9.7, 10.1, 9.2)
opacity <- c(4.4, 6.4, 3.0, 4.1, 0.8, 5.7, 2.0, 3.9, 1.9, 5.7,
             2.8, 4.1, 3.8, 1.6, 3.4, 8.4, 5.2, 6.9, 2.7, 1.9)
Y <- cbind(tear, gloss, opacity)
rate <- factor(gl(2,10), labels=c("Low", "High"))
additive <- factor(gl(2, 5, len=20), labels=c("Low", "High"))

fit <- manova(Y ~ rate * additive)
summary.aov(fit)           # univariate ANOVA tables
summary(fit, test="Wilks") # ANOVA table of Wilks' lambda
}
\keyword{models}
\eof
\name{summary.princomp}
\alias{summary.princomp}
\alias{print.summary.princomp}
\title{Summary method for Principal Components Analysis}
\usage{
\method{summary}{princomp}(object, loadings = FALSE, cutoff = 0.1, \dots)

\method{print}{summary.princomp}(x, digits = 3, loadings = x$print.loadings,
      cutoff = x$cutoff, \dots)
}
\arguments{
  \item{object}{an object of class \code{"princomp"}, as
    from \code{princomp()}.}
  \item{loadings}{logical. Should loadings be included?}
  \item{cutoff}{numeric. Loadings below this cutoff in absolute value
    are shown as blank in the output.}
  \item{x}{an object of class "summary.princomp".}
  \item{digits}{the number of significant digits to be used in listing
    loadings.}
  \item{\dots}{arguments to be passed to or from other methods.}
}
\description{
  The \code{\link{summary}} method for class \code{"princomp"}.
}
\value{
  \code{object} with additional components \code{cutoff} and
  \code{print.loadings}.
}
\seealso{
  \code{\link{princomp}}
}
\examples{
data(USArrests)
summary(pc.cr <- princomp(USArrests, cor=TRUE))
print(summary(princomp(USArrests, cor=TRUE),
              loadings = TRUE, cutoff = 0.2), digits = 2)
}
\keyword{multivariate}
\eof
\name{sunspot}
\docType{data}
\alias{sunspot}
\alias{sunspot.month}
\alias{sunspot.year}
\title{
  Yearly Sunspot Data, 1700--1988, and
  Monthly Sunspot Data, 1749--1997
}
\description{
  Monthly and yearly number of sunspots. 
}
\usage{
data(sunspot)
}
\format{
  The univariate time series \code{sunspot.year} and
  \code{sunspot.month} contain 289 and 2988 observations, respectively.
  The objects are of class \code{"ts"}.
}
\seealso{
  \code{sunspot.month} is a longer version of \code{\link{sunspots}}
  that runs until 1988.
}
\source{
  Monthly data: Sunspot Index Data Center,
  World Data Center-C1 For Sunspot Index
  Royal Observatory of Belgium, Av. Circulaire, 3, B-1180 BRUSSELS
  \url{http://www.oma.be/KSB-ORB/SIDC/sidc_txt.html}

  Yearly data: H. Tong (1996)
  \emph{Non-Linear Time Series}. Clarendon Press, Oxford, p. 471. 
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\examples{
## Compare the monthly series 
data(sunspots)
data(sunspot)
plot (sunspot.month, main = "sunspot.month [stats]", col = 2)
lines(sunspots) # "very barely" see something

## Now look at the difference :
all(tsp(sunspots)     [c(1,3)] ==
    tsp(sunspot.month)[c(1,3)]) ## Start & Periodicity are the same
n1 <- length(sunspots)
table(eq <- sunspots == sunspot.month[1:n1]) #>  132  are different !
i <- which(!eq) 
rug(time(eq)[i])
s1 <- sunspots[i] ; s2 <- sunspot.month[i]
cbind(i = i, sunspots = s1, ss.month = s2,
      perc.diff = round(100*2*abs(s1-s2)/(s1+s2), 1))
}
\keyword{datasets}
\eof
\name{sunspots}
\docType{data}
\alias{sunspots}
\title{Monthly Sunspot Numbers, 1749--1983}
\description{
  Monthly mean relative sunspot numbers from 1749 to 1983.  Collected at
  Swiss Federal Observatory, Zurich until 1960, then Tokyo Astronomical
  Observatory.
}
\usage{data(sunspots)}
\format{
  A time series of monthly data from 1749 to 1983.
}
\source{
  Andrews, D. F. and Herzberg, A. M. (1985)
  \emph{Data: A Collection of Problems from Many Fields for the
    Student and Research Worker}.
  New York: Springer-Verlag.
}
\seealso{\code{\link[stats]{sunspot.month}} (package \pkg{stats}) has a
  longer (and a bit different) series.} 
\examples{
data(sunspots)
plot(sunspots, main = "sunspots data", xlab = "Year",
     ylab = "Monthly sunspot numbers")
}
\keyword{datasets}
\eof
\name{supsmu}
\alias{supsmu}
\title{Friedman's SuperSmoother}
\description{
  Smooth the (x, y) values by Friedman's \dQuote{super smoother}.
}
\usage{
supsmu(x, y, wt, span = "cv", periodic = FALSE, bass = 0)
}
\arguments{
  \item{x}{x values for smoothing}
  \item{y}{y values for smoothing}
  \item{wt}{case weights, by default all equal}
  \item{span}{the fraction of the observations in the span of the running
    lines smoother, or \code{"cv"} to choose this by leave-one-out
    cross-validation.}
  \item{periodic}{if \code{TRUE}, the x values are assumed to be in
    \code{[0, 1]} and of period 1.}
  \item{bass}{controls the smoothness of the fitted curve. Values of up
    to 10 indicate increasing smoothness.}
}
\details{
  \code{supsmu} is a running lines smoother which chooses between three
  spans for the lines. The running lines smoothers are symmetric, with
  \code{k/2} data points each side of the predicted point, and values of
  \code{k} as \code{0.5 * n}, \code{0.2 * n} and \code{0.05 * n}, where
  \code{n} is the number of data points.  If \code{span} is specified,
  a single smoother with span \code{span * n} is used.

  The best of the three smoothers is chosen by cross-validation for each
  prediction. The best spans are then smoothed by a running lines
  smoother and the final prediction chosen by linear interpolation. 
 
  The FORTRAN code says: \dQuote{For small samples (\code{n < 40}) or if
    there are substantial serial correlations between observations close
    in x-value, then a prespecified fixed span smoother (\code{span >
      0}) should be used.  Reasonable span values are 0.2 to 0.4.}
}
\value{
  A list with components
  \item{x}{the input values in increasing order with duplicates removed.}
  \item{y}{the corresponding y values on the fitted curve.}
}
\references{
  Friedman, J. H. (1984)
  SMART User's Guide.
  Laboratory for Computational Statistics, Stanford University Technical
  Report No. 1.

  Friedman, J. H. (1984)
  A variable span scatterplot smoother.
  Laboratory for Computational Statistics, Stanford University Technical
  Report No. 5.
}
\seealso{\code{\link{ppr}}}

\examples{
data(cars)
with(cars, {
    plot(speed, dist)
    lines(supsmu(speed, dist))
    lines(supsmu(speed, dist, bass = 7), lty = 2)
    })
}
\keyword{smooth}
\eof
\name{symnum}
\alias{symnum}
\title{Symbolic Number Coding}
\description{
  Symbolically encode a given numeric or logical vector or array.
}
\usage{
symnum(x, cutpoints=c(0.3, 0.6, 0.8, 0.9, 0.95),
       symbols=c(" ", ".", ",", "+", "*", "B"),
       legend = length(symbols) >= 3,
       na = "?", eps = 1e-5, corr = missing(cutpoints),
       show.max = if(corr) "1", show.min = NULL,
       abbr.colnames = has.colnames,
       lower.triangular = corr && is.numeric(x) && is.matrix(x),
       diag.lower.tri   = corr && !is.null(show.max))
}
\arguments{
  \item{x}{numeric or logical vector or array.}
  \item{cutpoints}{numeric vector whose values \code{cutpoints[j]}%
    \eqn{= c_j}{ == c[j]}  (\emph{after} augmentation, see \code{corr}
    below) are used for intervals.}
  \item{symbols}{character vector, one shorter than (the
    \emph{augmented}, see \code{corr} below) \code{cutpoints}.
    \code{symbols[j]}\eqn{= s_j}{ == s[j]} are used as \dQuote{code} for
    the (half open) interval \eqn{(c_j,c_{j+1}]}{(c[j], c[j+1]]}.

    For \code{logical} argument \code{x}, the default is
    \code{c(".","|")} (graphical 0 / 1 s).}
  \item{legend}{logical indicating if a \code{"legend"} attribute is
    desired.}
  \item{na}{character or logical. How \code{\link{NA}s} are coded.  If
    \code{na == FALSE}, \code{NA}s are coded invisibly, \emph{including}
    the \code{"legend"} attribute below, which otherwise mentions NA
    coding.}
  \item{eps}{absolute precision to be used at left and right boundary.}
  \item{corr}{logical.  If \code{TRUE}, \code{x} contains correlations.
    The cutpoints are augmented by \code{0} and \code{1} and
    \code{abs(x)} is coded.}
  \item{show.max}{if \code{TRUE}, or of mode \code{character}, the
    maximal cutpoint is coded especially.}
  \item{show.min}{if \code{TRUE}, or of mode \code{character}, the
    minimal cutpoint is coded especially.}
  \item{abbr.colnames}{logical, integer or \code{NULL} indicating how
    column names should be abbreviated (if they are); if \code{NULL}
    (or \code{FALSE} and \code{x} has no column names),
    the column names will all be empty, i.e., \code{""}; otherwise if
    \code{abbr.colnames} is false, they are left unchanged.  If
    \code{TRUE} or integer, existing column names will be abbreviated to
    \code{\link{abbreviate}(*, minlength = abbr.colnames)}.}
  \item{lower.triangular}{logical.  If \code{TRUE} and \code{x} is a
    matrix, only the \emph{lower triangular} part of the matrix is coded
    as non-blank.}
  \item{diag.lower.tri}{logical.  If \code{lower.triangular} \emph{and}
    this are \code{TRUE}, the \emph{diagonal} part of the matrix is
    shown.}
}
\value{
  An atomic character object of class \code{\link{noquote}} and the same
  dimensions as \code{x}.

  If \code{legend} (\code{TRUE} by default when there more than 2 classes),
  it has an attribute \code{"legend"} containing a legend
  of the returned character codes, in the form
  \deqn{c_1 s_1 c_2 s_2 \dots s_n c_{n+1}}{% the dots are UGLY in LaTeX Math
    c[1] \sQuote{s[1]} c[2] \sQuote{s[2]} \dots \sQuote{s[n]} c_[n+1]}
  where \eqn{c_j}{c[j]}\code{ = cutpoints[j]} and
  \eqn{s_j}{s[j]}\code{ = symbols[j]}.
}
\author{Martin Maechler \email{maechler@stat.math.ethz.ch}}
\seealso{\code{\link{as.character}}}
\examples{
ii <- 0:8; names(ii) <- ii
symnum(ii, cut= 2*(0:4), sym = c(".", "-", "+", "$"))
symnum(ii, cut= 2*(0:4), sym = c(".", "-", "+", "$"), show.max=TRUE)

symnum(1:12 \%\% 3 == 0)# use for logical

##-- Symbolic correlation matrices:
data(attitude)
symnum(cor(attitude), diag = FALSE)
symnum(cor(attitude), abbr.= NULL)
symnum(cor(attitude), abbr.= FALSE)
symnum(cor(attitude), abbr.= 2)

symnum(cor(rbind(1, rnorm(25), rnorm(25)^2)))
symnum(cor(matrix(rexp(30, 1), 5, 18))) # <<-- PATTERN ! --
symnum(cm1 <- cor(matrix(rnorm(90) ,  5, 18))) # < White Noise SMALL n
symnum(cm1, diag=FALSE)
symnum(cm2 <- cor(matrix(rnorm(900), 50, 18))) # < White Noise "BIG" n
symnum(cm2, lower=FALSE)

## NA's:
Cm <- cor(matrix(rnorm(60),  10, 6)); Cm[c(3,6), 2] <- NA
symnum(Cm, show.max=NULL)

## Graphical P-values (aka "significance stars"):
pval <- rev(sort(c(outer(1:6, 10^-(1:3)))))
symp <- symnum(pval, corr=FALSE,
               cutpoints = c(0,  .001,.01,.05, .1, 1),
               symbols = c("***","**","*","."," "))
noquote(cbind(P.val = format(pval), Signif= symp))
}
\keyword{utilities}
\keyword{character}
\eof
\name{t.test}
\alias{t.test}
\alias{t.test.default}
\alias{t.test.formula}
\title{Student's t-Test}
\description{
  Performs one and two sample t-tests on vectors of data.
}
\usage{
t.test(x, \dots)

\method{t.test}{default}(x, y = NULL, alternative = c("two.sided", "less", "greater"),
       mu = 0, paired = FALSE, var.equal = FALSE,
       conf.level = 0.95, \dots)

\method{t.test}{formula}(formula, data, subset, na.action, \dots)
}
\arguments{
  \item{x}{a numeric vector of data values.}
  \item{y}{an optional numeric vector data values.}
  \item{alternative}{a character string specifying the alternative
    hypothesis, must be one of \code{"two.sided"} (default),
    \code{"greater"} or \code{"less"}.  You can specify just the initial
    letter.}
  \item{mu}{a number indicating the true value of the mean (or
    difference in means if you are performing a two sample test).}
  \item{paired}{a logical indicating whether you want a paired
    t-test.}
  \item{var.equal}{a logical variable indicating whether to treat the
    two variances as being equal. If \code{TRUE} then the pooled
    variance is used to estimate the variance otherwise the Welch
    (or Satterthwaite) approximation to the degrees of freedom is used.}
  \item{conf.level}{confidence level of the interval.}
  \item{formula}{a formula of the form \code{lhs ~ rhs} where \code{lhs}
    is a numeric variable giving the data values and \code{rhs} a factor
    with two levels giving the corresponding groups.}
  \item{data}{an optional data frame containing the variables in the
    model formula.}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.  Defaults to
    \code{getOption("na.action")}.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\details{
  The formula interface is only applicable for the 2-sample tests.

  If \code{paired} is \code{TRUE} then both \code{x} and \code{y} must
  be specified and they must be the same length.  Missing values are
  removed (in pairs if \code{paired} is \code{TRUE}).  If
  \code{var.equal} is \code{TRUE} then the pooled estimate of the
  variance is used.  By default, if \code{var.equal} is \code{FALSE}
  then the variance is estimated separately for both groups and the
  Welch modification to the degrees of freedom is used.
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the t-statistic.}
  \item{parameter}{the degrees of freedom for the t-statistic.}
  \item{p.value}{the p-value for the test.}
  \item{conf.int}{a confidence interval for the mean appropriate to the
    specified alternative hypothesis.}
  \item{estimate}{the estimated mean or difference in means depending on
    whether it was a one-sample test or a two-sample test.}
  \item{null.value}{the specified hypothesized value of the mean or mean
    difference depending on whether it was a one-sample test or a
    two-sample test.}
  \item{alternative}{a character string describing the alternative
    hypothesis.}
  \item{method}{a character string indicating what type of t-test was
    performed.}
  \item{data.name}{a character string giving the name(s) of the data.}
}
\seealso{
  \code{\link{prop.test}}
}
\examples{
t.test(1:10,y=c(7:20))      # P = .00001855
t.test(1:10,y=c(7:20, 200)) # P = .1245    -- NOT significant anymore

## Classical example: Student's sleep data
data(sleep)
plot(extra ~ group, data = sleep)
## Traditional interface
with(sleep, t.test(extra[group == 1], extra[group == 2]))
## Formula interface
t.test(extra ~ group, data = sleep)
}
\keyword{htest}
\eof
\name{termplot}
\alias{termplot}
\title{Plot regression terms}
\description{
  Plots regression terms against their predictors, optionally with
  standard errors and partial residuals added.
}
\usage{
termplot(model, data=NULL, envir=environment(formula(model)),
         partial.resid=FALSE, rug=FALSE,
         terms=NULL, se=FALSE, xlabs=NULL, ylabs=NULL, main = NULL,
         col.term = 2, lwd.term = 1.5,
         col.se = "orange", lty.se = 2, lwd.se = 1,
         col.res = "gray", cex = 1, pch = par("pch"),
         col.smth = "darkred", lty.smth=2, span.smth=2/3,
         ask = interactive() && nb.fig < n.tms && .Device !="postscript",
         use.factor.levels=TRUE, smooth=NULL,
         \dots)
}
\arguments{
  \item{model}{fitted model object}
  \item{data}{data frame in which variables in \code{model} can be
    found}
  \item{envir}{environment in which variables in \code{model} can be found}
  \item{partial.resid}{logical; should partial residuals be plotted?}
  \item{rug}{add \link{rug}plots (jittered 1-d histograms) to the axes?}
  \item{terms}{which terms to plot (default \code{NULL} means all terms)}
  \item{se}{plot pointwise standard errors?}
  \item{xlabs}{vector of labels for the x axes}
  \item{ylabs}{vector of labels for the y axes}
  \item{main}{logical, or vector of main titles;  if \code{TRUE}, the
    model's call is taken as main title, \code{NULL} or \code{FALSE} mean
    no titles.}
  \item{col.term, lwd.term}{color and line width for the \dQuote{term curve},
    see \code{\link{lines}}.}
  \item{col.se, lty.se, lwd.se}{color, line type and line width for the
    \dQuote{twice-standard-error curve} when \code{se = TRUE}.}
  \item{col.res, cex, pch}{color, plotting character expansion and type
    for partial residuals, when \code{partial.resid = TRUE}, see
    \code{\link{points}}.}
   \item{ask}{logical; if \code{TRUE}, the user is \emph{ask}ed before
     each plot, see \code{\link{par}(ask=.)}.}
   \item{use.factor.levels}{Should x-axis ticks use factor levels or
     numbers for factor terms?}
   \item{smooth}{\code{NULL} or a function with the same arguments as
     \code{\link{panel.smooth}} to draw a smooth through the partial
     residuals for non-factor terms}
   \item{lty.smth,col.smth, span.smth}{Passed to \code{smooth}}
  \item{\dots}{other graphical parameters}
}
\details{
  The model object must have a \code{predict} method that accepts
  \code{type=terms}, eg \code{\link{glm}} in the \pkg{base} package,
  \code{\link[survival]{coxph}} and \code{\link[survival]{survreg}} in
  the \pkg{survival} package.

  For the \code{partial.resid=TRUE} option it must have a
  \code{\link{residuals}} method that accepts \code{type="partial"},
  which \code{\link{lm}} and \code{\link{glm}} do.

  The \code{data} argument should rarely be needed, but in some cases
  \code{termplot} may be unable to reconstruct the original data
  frame. Using \code{na.action=na.exclude} makes these problems less likely.

  Nothing sensible happens for interaction terms.
}
\seealso{For (generalized) linear models, \code{\link{plot.lm}} and
  \code{\link{predict.glm}}.}
\examples{
had.splines <- "package:splines" \%in\% search()
if(!had.splines) rs <- require(splines)
x <- 1:100
z <- factor(rep(LETTERS[1:4],25))
y <- rnorm(100,sin(x/10)+as.numeric(z))
model <- glm(y ~ ns(x,6) + z)

par(mfrow=c(2,2)) ## 2 x 2 plots for same model :
termplot(model, main = paste("termplot( ", deparse(model$call)," ...)"))
termplot(model, rug=TRUE)
termplot(model, partial=TRUE, se = TRUE, main = TRUE)
termplot(model, partial=TRUE, smooth=panel.smooth,span.smth=1/4)
if(!had.splines && rs) detach("package:splines")
}
\keyword{hplot}
\keyword{regression}
\eof
\name{terms}
\title{Model Terms}
\usage{
terms(x, \dots)
}
\alias{terms}
\alias{terms.terms}
\alias{terms.aovlist}
\alias{terms.default}
\alias{print.terms}
\description{
  The function \code{terms} is a generic function
  which can be used to extract \emph{terms} objects
  from various kinds of \R data objects.
}
\arguments{
  \item{x}{object used to select a method to dispatch.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\details{
  There are methods for classes \code{"aovlist"}, and \code{"terms"}
  \code{"formula"} (see \code{\link{terms.formula}}):
  the default method just extracts the \code{terms} component of the
  object (if any).
}
\value{
  An object of class \code{c("terms", "formula")} which contains the
  \emph{terms} representation of a symbolic model. See
  \code{\link{terms.object}} for its structure.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical models.}
  Chapter 2 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{terms.object}}, \code{\link{terms.formula}},
  \code{\link{lm}}, \code{\link{glm}}, \code{\link{formula}}.
}
%%--- MM: I would really like instructive examples here...
\keyword{models}
\eof
\name{terms.formula}
\alias{terms.formula}
\title{Construct a terms Object from a Formula}
\description{
  This function takes a formula and some optional arguments and
  constructs a terms object. The terms object can then be used to
  construct a \code{\link{model.matrix}}.
}
\usage{
\method{terms}{formula}(x, specials = NULL, abb = NULL, data = NULL, neg.out = TRUE,
     keep.order = FALSE, simplify = FALSE, ...)
}
\arguments{
  \item{x}{a formula.}
  \item{specials}{which functions in the formula should be marked as
    special in the \code{terms} object.}
  \item{abb}{Not implemented in \R.}
  \item{data}{a data frame from which the meaning of the special symbol
    \code{.} can be inferred. It is unused if there is no \code{.} in
    the formula.}
  \item{neg.out}{Not implemented in \R.}
  \item{keep.order}{a logical value indicating whether the terms should
    keep their positions. If \code{FALSE} the terms are reordered so
    that main effects come first, followed by the interactions,
    all second-order, all third-order and so on.  Effects of a given
    order are kept in the order specified.}
  \item{simplify}{should the formula be expanded and simplified, the
    pre-1.7.0 behaviour?}
  \item{\dots}{further arguments passed to or from other methods.}
}
\details{
  Not all of the options work in the same way that they do in S and not
  all are implemented.
}
\value{
  A \code{\link{terms.object}} object is returned.  The object itself is
  the re-ordered (unless \code{keep.order = TRUE}) formula.  In all
  cases variables within an interaction term in the formula are
  re-ordered by the ordering of the \code{"variables"} attribute, which
  is the order in which the variables occur in the formula.
}
\seealso{
  \code{\link{terms}}, \code{\link{terms.object}}
}
\keyword{models}
\eof
\name{terms.object}
\title{Description of Terms Objects}
\alias{terms.object}
\description{
  An object of class \code{\link{terms}} holds information about a
  model.  Usually the model was specified in terms of a
  \code{\link{formula}} and that formula was used to determine the terms
  object.
}
\value{
  The object itself is simply the formula supplied to the call of
  \code{\link{terms.formula}}.  The object has a number of attributes
  and they are used to construct the model frame:

  \item{factors}{A matrix of variables by terms showing which variables
    appear in which terms. The entries are 0 if the variable does not
    occur in the term, 1 if it does occur and should be coded by
    contrasts, and 2 if it occurs and should be coded via dummy
    variables for all levels (as when an intercept or lower-order term
    is missing).  If there are no terms other than an intercept and offsets,
    this is \code{numeric(0)}.}
  \item{term.labels}{A character vector containing the labels for each
    of the terms in the model, except for offsets.  Non-syntactic names
    will be quoted by backticks.}
  \item{variables}{A call to \code{list} of the variables in the model.}
  \item{intercept}{Either 0, indicating no intercept is to be fit, or 1
    indicating that an intercept is to be fit.}
  \item{order}{A vector of the same length as \code{term.labels}
    indicating the order of interaction for each term }
  \item{response}{The index of the variable (in variables) of the
    response (the left hand side of the formula). Zero, if there is no
    response.}
  \item{offset}{If the model contains \code{\link{offset}} terms there
    is an \code{offset} attribute indicating which variable(s) are offsets}
  \item{specials}{If the \code{specials} argument was given to
    \code{\link{terms.formula}} there is a \code{specials} attribute, a
    list of vectors indicating the terms that contain these special
    functions.}
  \item{dataClasses}{optional. A named character vector giving the classes
    (as given by \code{\link{.MFclass}} of the variables used in a fit.}

  The object has class \code{c("terms", "formula")}.
}
\note{
  These objects are different from those found in S.  In particular
  there is no \code{formula} attribute, instead the object is itself a
  formula.  Thus, the mode of a terms object is different as well. 

  Examples of the \code{specials} argument can be seen in the
  \code{\link{aov}} and \code{\link[survival]{coxph}} functions.
}
\seealso{
  \code{\link{terms}}, \code{\link{formula}}.
}
%%--- MM: I would really like instructive examples here...
\keyword{models}
\eof
\name{time}
\alias{time}
\alias{cycle}
\alias{frequency}
\alias{deltat}
\alias{time.default}
%\alias{time.ts}
%\alias{cycle.default}
%\alias{cycle.ts}
%\alias{frequency.default}
%\alias{deltat.default}
\title{Sampling Times of Time Series}
\description{
  \code{time} creates the vector of times at which a time series was sampled.

  \code{cycle} gives the positions in the cycle of each observation.

  \code{frequency} returns the number of samples per unit time and
  \code{deltat} the time interval between observations (see
  \code{\link{ts}}).
}
\usage{
time(x, \dots)
\method{time}{default}(x, offset=0, \dots)

cycle(x, \dots)
frequency(x, \dots)
deltat(x, \dots)
}
\arguments{
  \item{x}{a univariate or multivariate time-series, or a vector or matrix.}
  \item{offset}{can be used to indicate when sampling took place
    in the time unit. \code{0} (the default) indicates the start
    of the unit, \code{0.5} the middle and \code{1} the end of
    the interval.}
  \item{\dots}{extra arguments for future methods.}
}
\details{
  These are all generic functions, which will use the
  \code{\link{tsp}} attribute of \code{x} if it exists. \code{time}
  and \code{cycle} have methods for class \code{\link{ts}} that coerce
  the result to that class.
}
\seealso{
  \code{\link{ts}}, \code{\link{start}}, \code{\link{tsp}},
  \code{\link{window}}.

  \code{\link{date}} for clock time, \code{\link{system.time}}
  for CPU usage.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\examples{
data(presidents)
cycle(presidents)
# a simple series plot: c() makes the x and y arguments into vectors
plot(c(time(presidents)), c(presidents), type="l")
}
\keyword{ts}
\eof
\name{toeplitz}
\title{Form Symmetric Toeplitz Matrix}
\usage{
toeplitz(x)
}
\alias{toeplitz}
\arguments{
  \item{x}{the first row to form the Toeplitz matrix.}
}
\description{
  Forms a symmetric Toeplitz matrix given its first row.
}
\value{
  The Toeplitz matrix.
}
\author{A. Trapletti}
\examples{
x <- 1:5
toeplitz (x)
}
\keyword{ts}

\eof
\name{treering}
\docType{data}
\alias{treering}
\title{
  Yearly Treering Data, -6000--1979
}
\description{
  Contains normalized tree-ring widths in dimensionless units.
}
\usage{
data(treering)
}
\format{
  A univariate time series with 7981 observations. The object is of
  class \code{"ts"}.

  Each tree ring corresponds to one year.
}
\details{
  The data were recorded by Donald A. Graybill, 1980, from 
  Gt Basin Bristlecone Pine 2805M, 3726-11810 in Methuselah Walk, California.
}
\source{
  Time Series Data Library:
  \url{http://www-personal.buseco.monash.edu.au/~hyndman/TSDL/},
  series \file{CA535.DAT}
}
\references{
  For background on Bristlecone pines and Methuselah Walk, see
  \url{http://www.sonic.net/bristlecone/}; for some photos see
  \url{http://www.ltrr.arizona.edu/~hallman/sitephotos/meth.html}
}
\keyword{datasets}
\eof
\name{ts-defunct}
\alias{arima0.diag}
\title{Defunct functions in Package ts}
\description{
  The functions or variables listed here are no longer part of \R as
  they are not needed (any more).
}
\usage{
% Removed in 1.6.0
arima0.diag(\dots)
% </Removed 1.6.0>
}
\details{
  \code{arima0.diag} has been replaced by \code{\link{tsdiag.arima0}}.
}
\keyword{ts}
\keyword{internal}
\eof
\name{ts-methods}
\alias{diff.ts}
\alias{na.omit.ts}
\title{Methods for Time Series Objects}
\description{
  Methods for objects of class \code{"ts"}, typically the result of
  \code{\link{ts}}.
}
\usage{
\method{diff}{ts}(x, lag=1, differences=1, \dots)

\method{na.omit}{ts}(object, \dots)
}
\arguments{
  \item{x}{an object of class \code{"ts"} containing the values to be
    differenced.}
  \item{lag}{an integer indicating which lag to use.}
  \item{differences}{an integer indicating the order of the difference.}
  \item{object}{a univariate or multivariate time series.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\details{
  The \code{na.omit} method omits initial and final segments with
  missing values in one or more of the series.  \sQuote{Internal}
  missing values will lead to failure.
}
\value{
  For the \code{na.omit} method, a time series without missing values.
  The class of \code{object} will be preserved.
}
\seealso{
  \code{\link{diff}};
  \code{\link{na.omit}}, \code{\link{na.fail}},
  \code{\link{na.contiguous}}.
}
\keyword{ts}
\eof
\name{ts}
\title{Time-Series Objects}
\alias{ts}
\alias{as.ts}
\alias{is.ts}
\alias{Ops.ts}
\alias{cbind.ts}
\alias{is.mts}
\alias{[.ts}
\alias{t.ts}
\description{
  The function \code{ts} is used to create time-series objects.

  \code{as.ts} and \code{is.ts} coerce an object to a time-series and
  test whether an object is a time series.
}
\usage{
ts(data = NA, start = 1, end = numeric(0), frequency = 1,
   deltat = 1, ts.eps = getOption("ts.eps"), class = , names = )
as.ts(x)
is.ts(x)
}
\arguments{
  \item{data}{a numeric vector or matrix of the observed time-series
    values. A data frame will be coerced to a numeric matrix via
    \code{data.matrix}.}
  \item{start}{the time of the first observation.  Either a single
    number or a vector of two integers, which specify a natural time
    unit and a (1-based) number of samples into the time unit.  See
    the examples for the use of the second form.}
  \item{end}{the time of the last observation, specified in the same way
    as \code{start}.}
  \item{frequency}{the number of observations per unit of time.}
  \item{deltat}{the fraction of the sampling period between successive
    observations; e.g., 1/12 for monthly data.  Only one of
    \code{frequency} or \code{deltat} should be provided.}
  \item{ts.eps}{time series comparison tolerance.  Frequencies are
    considered equal if their absolute difference is less than
    \code{ts.eps}.}
  \item{class}{class to be given to the result, or none if \code{NULL}
    or \code{"none"}.  The default is \code{"ts"} for a single series,
    \code{c("mts", "ts")} for multiple series.}
  \item{names}{a character vector of names for the series in a multiple
    series: defaults to the colnames of \code{data}, or \code{Series 1},
    \code{Series 2}, \dots.}
  \item{x}{an arbitrary \R object.}
}
\details{
  The function \code{ts} is used to create time-series objects.  These
  are vector or matrices with class of \code{"ts"} (and additional
  attributes) which represent data which has been sampled at equispaced
  points in time.  In the matrix case, each column of the matrix
  \code{data} is assumed to contain a single (univariate) time series.
  Time series must have an least one observation, and although they need
  not be numeric there is very limited support for non-numeric series.

  Class \code{"ts"} has a number of methods.  In particular arithmetic
  will attempt to align time axes, and subsetting to extract subsets of
  series can be used (e.g., \code{EuStockMarkets[, "DAX"]}).  However,
  subsetting the first (or only) dimension will return a matrix or
  vector, as will matrix subsetting.  There is a method for \code{\link{t}}
  that transposes the series as a matrix (a one-column matrix if a vector)
  and hence returns a result that does not inherit from class \code{"ts"}.

  The value of argument \code{frequency} is used when the series is
  sampled an integral number of times in each unit time interval.  For
  example, one could use a value of \code{7} for \code{frequency} when
  the data are sampled daily, and the natural time period is a week, or
  \code{12} when the data are sampled monthly and the natural time
  period is a year.  Values of \code{4} and \code{12} are assumed in
  (e.g.) \code{print} methods to imply a quarterly and monthly series
  respectively.

  \code{as.ts} will use the \code{\link{tsp}} attribute of the object if
  it has one to set the start and end times and frequency.

  \code{is.ts} tests if an object is a time series. It is generic: you
  can write methods to handle specific classes of objects,
  see \link{InternalMethods}.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{tsp}},
  \code{\link{frequency}},
  \code{\link{start}},
  \code{\link{end}},
  \code{\link{time}},
  \code{\link{window}};
  \code{\link{print.ts}}, the print method for time series objects;
  \code{\link{plot.ts}}, the plot method for time series objects.
}
\examples{
ts(1:10, frequency = 4, start = c(1959, 2)) # 2nd Quarter of 1959
print( ts(1:10, freq = 7, start = c(12, 2)), calendar = TRUE) # print.ts(.)
## Using July 1954 as start date:
gnp <- ts(cumsum(1 + round(rnorm(100), 2)),
          start = c(1954, 7), frequency = 12)
plot(gnp) # using 'plot.ts' for time-series plot

## Multivariate
z <- ts(matrix(rnorm(300), 100, 3), start=c(1961, 1), frequency=12)
class(z)
plot(z)
plot(z, plot.type="single", lty=1:3)

## A phase plot:
data(nhtemp)
plot(nhtemp, c(nhtemp[-1], NA), cex = .8, col="blue",
     main = "Lag plot of New Haven temperatures")
## a clearer way to do this would be
\dontrun{
plot(nhtemp, lag(nhtemp, 1), cex = .8, col="blue",
     main = "Lag plot of New Haven temperatures")
}}
\keyword{ts}
\eof
\name{ts.plot}
\alias{ts.plot}
\title{Plot Multiple Time Series}
\usage{
ts.plot(\dots, gpars = list())
}
\arguments{
  \item{\dots}{one or more univariate or multivariate time series.}
  \item{gpars}{list of named graphics parameters to be passed to the
    plotting functions.  Those commonly used can be supplied directly in
    \code{\dots}.}
}
\description{
  Plot several time series on a common plot. Unlike
  \code{\link{plot.ts}} the series can have a different time bases,
  but they should have the same frequency.
}
\note{
  Although this can be used for a single time series, \code{plot} is
  easier to use and is preferred.
}
\value{
  None.
}

\seealso{\code{\link{plot.ts}}}

\examples{
data(UKLungDeaths)
ts.plot(ldeaths, mdeaths, fdeaths,
        gpars=list(xlab="year", ylab="deaths", lty=c(1:3)))
}
\keyword{ts}
\eof
\name{ts.union}
\alias{ts.union}
\alias{ts.intersect}
\title{Bind Two or More Time Series}
\usage{
ts.intersect(\dots, dframe = FALSE)
ts.union(\dots, dframe = FALSE)
}
\arguments{
  \item{\dots}{two or more univariate or multivariate time series, or
    objects which can coerced to time series.}
  \item{dframe}{logical; if \code{TRUE} return the result as a data
    frame.}
}
\description{
  Bind time series which have a common frequency. \code{ts.union} pads
  with \code{NA}s to the total time coverage, \code{ts.intersect}
  restricts to the time covered by all the series.
}
\details{
  As a special case, \code{\dots} can contain vectors or matrices of the
  same length as the combined time series of the time series present, as
  well as those of a single row.
}
\value{
  A time series object if \code{dframe} is \code{FALSE}, otherwise a
  data frame.
}
\seealso{\code{\link{cbind}}.}
\examples{
data(UKLungDeaths)
ts.union(mdeaths, fdeaths)
cbind(mdeaths, fdeaths) # same as the previous line
ts.intersect(window(mdeaths, 1976), window(fdeaths, 1974, 1978))
data(BJsales)
sales1 <- ts.union(BJsales, lead = BJsales.lead)
ts.intersect(sales1, lead3 = lag(BJsales.lead, -3))
}
\keyword{ts}
\eof
\name{tsSmooth}
\alias{tsSmooth}
\alias{tsSmooth.StructTS}
\title{Use Fixed-Interval Smoothing on Time Series}
\description{
  Performs fixed-interval smoothing on a univariate time series via a
  state-space model.  Fixed-interval smoothing gives the best estimate
  of the state at each time point based on the whole observed series.
}
\usage{
tsSmooth(object, \dots)
}
\arguments{
  \item{object}{a time-series fit.  Currently only class
    \code{"\link{StructTS}"} is supported}
  \item{\dots}{possible arguments for future methods.}
}
\value{
  A time series, with as many dimensions as the state space and results
  at each time point of the original series.  (For seasonal models, only
  the current seasonal component is returned.)
}

\references{
  Durbin, J. and Koopman, S. J. (2001) \emph{Time Series Analysis by
    State Space Methods.}  Oxford University Press.
}

\author{ B. D. Ripley }
 
\seealso{
  \code{\link{KalmanSmooth}}, \code{\link{StructTS}}.

  For examples consult \code{\link{AirPassengers}},
  \code{\link{JohnsonJohnson}} and \code{\link{Nile}}.
}
\keyword{ts}

\eof
\name{tsdiag}
\alias{tsdiag}
\alias{tsdiag.arima0}
\alias{tsdiag.Arima}
\alias{tsdiag.StructTS}
\title{Diagnostic Plots for Time-Series Fits}
\description{
  A generic function to plot time-series diagnostics.
}
\usage{
tsdiag(object, gof.lag, \dots)
}
\arguments{
  \item{object}{a fitted time-series model}
  \item{gof.lag}{the maximum number of lags for a Portmanteau
    goodness-of-fit test}
  \item{\dots}{further arguments to be passed to particular methods}
}
\details{
  This is a generic function. It will generally plot the residuals,
  often standadized, the autocorrelation function of the residuals, and
  the p-values of a Portmanteau test for all lags up to \code{gof.lag}.

  The methods for \code{\link{arima}} and \code{\link{StructTS}} objects
  plots residuals scaled by the estimate of their (individual) variance,
  and use the Ljung--Box version of the portmanteau test.
}
\value{
  None. Diagnostics are plotted.
}

\seealso{
  \code{\link{arima}}, \code{\link{StructTS}}, \code{\link{Box.test}}
}

\examples{
\dontrun{data(lh)
fit <- arima(lh, c(1,0,0))
tsdiag(fit)

## see also examples(arima)

data(JohnsonJohnson)
(fit <- StructTS(log10(JohnsonJohnson), type="BSM"))
tsdiag(fit)
}}
\keyword{ts}

\eof
\name{tsp}
\title{Tsp Attribute of Time-Series-like Objects}
\alias{tsp}
\alias{tsp<-}
\alias{hasTsp}
\usage{
tsp(x)
tsp(x) <- value
hasTsp(x)
}
\arguments{
    \item{x}{a vector or matrix or univariate or multivariate time-series.}
    \item{value}{a numeric vector of length 3 or \code{NULL}.}
}
\description{
    \code{tsp} returns the \code{tsp} attribute (or \code{NULL}).
    It is included for compatibility with S version 2. \code{tsp<-}
    sets the \code{tsp} attribute. \code{hasTsp} ensures \code{x} has a
    \code{tsp} attribute, by adding one if needed.
}
\details{
    The \code{tsp} attribute was previously described here as
    \code{c(start(x), end(x), frequency(x))}, but this is incorrect.
    It gives the start time \emph{in time units}, the end time and
    the frequency.

    Assignments are checked for consistency.    

    Assigning \code{NULL} which removes the \code{tsp} attribute
    \emph{and} any \code{"ts"} class of \code{x}.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
    \code{\link{ts}}, \code{\link{time}}, \code{\link{start}}.
}
\keyword{ts}
\eof
\name{uniroot}
\title{One Dimensional Root (Zero) Finding}
\usage{
uniroot(f, interval, lower = min(interval), upper = max(interval),
        tol = .Machine$double.eps^0.25, maxiter = 1000, \dots)
}
\alias{uniroot}
\arguments{
  \item{f}{the function for which the root is sought.}
  \item{interval}{a vector containing the end-points of the interval
    to be searched for the root.}
  \item{lower}{the lower end point of the interval to be searched.}
  \item{upper}{the upper end point of the interval to be searched.}
  \item{tol}{the desired accuracy (convergence tolerance).}
  \item{maxiter}{the maximum number of iterations.}
  \item{\dots}{additional arguments to \code{f}.}
}
\description{
  The function \code{uniroot} searches the interval from \code{lower}
  to \code{upper} for a root (i.e., zero) of the function \code{f} with
  respect to its first argument.
}
\details{
  Either \code{interval} or both \code{lower} and \code{upper} must be
  specified.  The function uses Fortran subroutine \file{"zeroin"} (from
  Netlib) based on algorithms given in the reference below.

  If the algorithm does not converge in \code{maxiter} steps, a warning
  is printed and the current approximation is returned.
}
\value{
  A list with four components: \code{root} and \code{f.root} give the
  location of the root and the value of the function evaluated at that
  point. \code{iter} and \code{estim.prec} give the number of iterations
  used and an approximate estimated precision for \code{root}.
}
\references{
  Brent, R. (1973)
  \emph{Algorithms for Minimization without Derivatives.}
  Englewood Cliffs, NJ: Prentice-Hall.
}
\seealso{
  \code{\link{polyroot}} for all complex roots of a polynomial;
  \code{\link{optimize}}, \code{\link{nlm}}.
}
\examples{
f <- function (x,a) x - a
str(xmin <- uniroot(f, c(0, 1), tol = 0.0001, a = 1/3))
str(uniroot(function(x) x*(x^2-1) + .5, low = -2, up = 2, tol = 0.0001),
    dig = 10)
str(uniroot(function(x) x*(x^2-1) + .5, low = -2, up =2 , tol = 1e-10 ),
    dig = 10)

## Find the smallest value x for which exp(x) > 0 (numerically):
r <- uniroot(function(x) 1e80*exp(x) -1e-300,,-1000,0, tol=1e-20)
str(r, digits= 15)##> around -745.1332191

exp(r$r)	# = 0, but not for r$r * 0.999...
minexp <- r$r * (1 - .Machine$double.eps)
exp(minexp)	# typically denormalized
}
\keyword{optimize}
\eof
\name{update}
\title{Update and Re-fit a Model Call}
\usage{
update(object, \dots)

\method{update}{default}(object, formula., \dots, evaluate = TRUE)
}
\alias{update}
\alias{update.default}
\arguments{
 \item{object}{An existing fit from a model function such as \code{lm},
   \code{glm} and many others.}
 \item{formula.}{Changes to the formula -- see \code{update.formula} for
   details.}
 \item{\dots}{Additional arguments to the call, or arguments with
   changed values. Use \code{name=NULL} to remove the argument \code{name}.}
 \item{evaluate}{If true evaluate the new call else return the call.}
}
\description{
\code{update} will update and (by default) re-fit a model. It does this
by extracting the call stored in the object, updating the call and (by
default) evaluating that call. Sometimes it is useful to call
\code{update} with only one argument, for example if the data frame has
been corrected.
}
\value{
  If \code{evaluate = TRUE} the fitted object, otherwise the updated call.
}
\seealso{
  \code{\link{update.formula}}
}
\references{
  Chambers, J. M. (1992)
  \emph{Linear models.}
  Chapter 4 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\examples{
oldcon <- options(contrasts = c("contr.treatment", "contr.poly"))
## Annette Dobson (1990) "An Introduction to Generalized Linear Models".
## Page 9: Plant Weight Data.
ctl <- c(4.17,5.58,5.18,6.11,4.50,4.61,5.17,4.53,5.33,5.14)
trt <- c(4.81,4.17,4.41,3.59,5.87,3.83,6.03,4.89,4.32,4.69)
group <- gl(2, 10, 20, labels = c("Ctl", "Trt"))
weight <- c(ctl, trt)
lm.D9 <- lm(weight ~ group)
lm.D9
summary(lm.D90 <- update(lm.D9, . ~ . - 1))
options(contrasts = c("contr.helmert", "contr.poly"))
update(lm.D9)
options(oldcon)
}
\keyword{models}
\eof
\name{update.formula}
\alias{update.formula}
\title{Model Updating}
\usage{
\method{update}{formula}(old, new, ...)
}
\arguments{
  \item{old}{a model formula to be updated.}
  \item{new}{a formula giving a template which specifies how to update.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  \code{update.formula} is used to update model formulae.
  This typically involves adding or dropping terms,
  but updates can be more general.
}
\details{
  The function works by first identifying the \emph{left-hand side}
  and \emph{right-hand side} of the \code{old} formula.
  It then examines the \code{new} formula and substitutes
  the \emph{lhs} of the \code{old} formula for any occurence
  of "\code{.}" on the left of \code{new}, and substitutes
  the \emph{rhs} of the \code{old} formula for any occurence
  of "\code{.}" on the right of \code{new}.
}
\value{
  The updated formula is returned.
}
\seealso{
  \code{\link{terms}}, \code{\link{model.matrix}}.
}
\examples{
update(y ~ x,    ~ . + x2) #> y ~ x + x2
update(y ~ x, log(.) ~ . ) #> log(y) ~ x
}
\keyword{models}
\eof
\name{uspop}
\docType{data}
\alias{uspop}
\title{Populations Recorded by the US Census}
\description{
  This data set gives the population of the United States (in millions)
  as recorded by the decennial census for the period 1790--1970.
}
\usage{data(uspop)}
\format{A time series of 19 values.}
\source{
  McNeil, D. R. (1977)
  \emph{Interactive Data Analysis}.
  New York: Wiley.
}
% FIXME: missing \references{}
\examples{
data(uspop)
plot(uspop, log = "y", main = "uspop data", xlab = "Year",
     ylab = "U.S. Population (millions)")
}
\keyword{datasets}
\eof
\name{var.test}
\alias{var.test}
\alias{var.test.default}
\alias{var.test.formula}
\title{F Test to Compare Two Variances}
\description{
  Performs an F test to compare the variances of two samples from normal
  populations.
}
\usage{
var.test(x, \dots)

\method{var.test}{default}(x, y, ratio = 1, alternative = c("two.sided", "less", "greater"),
         conf.level = 0.95, \dots)

\method{var.test}{formula}(formula, data, subset, na.action, \dots)
}
\arguments{
  \item{x, y}{numeric vectors of data values, or fitted linear model
    objects (inheriting from class \code{"lm"}).}
  \item{ratio}{the hypothesized ratio of the population variances of
    \code{x} and \code{y}.}
  \item{alternative}{a character string specifying the alternative
    hypothesis, must be one of \code{"two.sided"} (default),
    \code{"greater"} or \code{"less"}.  You can specify just the initial
    letter.}
  \item{conf.level}{confidence level for the returned confidence
    interval.}
  \item{formula}{a formula of the form \code{lhs ~ rhs} where \code{lhs}
    is a numeric variable giving the data values and \code{rhs} a factor
    with two levels giving the corresponding groups.}
  \item{data}{an optional data frame containing the variables in the
    model formula.}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.  Defaults to
    \code{getOption("na.action")}.}
  \item{\dots}{further arguments to be passed to or from methods.}  
}
\details{
  The null hypothesis is that the ratio of the variances of the
  populations from which \code{x} and \code{y} were drawn, or in the
  data to which the linear models \code{x} and \code{y} were fitted, is
  equal to \code{ratio}.
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the F test statistic.}
  \item{parameter}{the degrees of the freedom of the F distribtion of
    the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{conf.int}{a confidence interval for the ratio of the population
    variances.}
  \item{estimate}{the ratio of the sample variances of \code{x} and
    \code{y}.}
  \item{null.value}{the ratio of population variances under the null.}
  \item{alternative}{a character string describing the alternative
    hypothesis.}
  \item{method}{the character string
    \code{"F test to compare two variances"}.}
  \item{data.name}{a character string giving the names of the data.}
}
\seealso{
  \code{\link{bartlett.test}} for testing homogeneity of variances in
  more than two samples from normal distributions;
  \code{\link{ansari.test}} and \code{\link{mood.test}} for two rank
  based (nonparametric) two-sample tests for difference in scale.
}
\examples{
x <- rnorm(50, mean = 0, sd = 2)
y <- rnorm(30, mean = 1, sd = 1)
var.test(x, y)                  # Do x and y have the same variance?
var.test(lm(x ~ 1), lm(y ~ 1))  # The same.
}
\keyword{htest}
\eof
\name{varimax}
\alias{promax}
\alias{varimax}
\title{Rotation Methods for Factor Analysis}
\description{
  These functions \sQuote{rotate} loading matrices in factor analysis.
}
\usage{
varimax(x, normalize = TRUE, eps = 1e-5)
promax(x, m = 4)
}
\arguments{
  \item{x}{A loadings matrix, with \eqn{p} rows and \eqn{k < p} columns}
  \item{m}{The power used the target for \code{promax}.  Values of 2 to
    4 are recommended.}
  \item{normalize}{logical. Should Kaiser normalization be performed?
    If so the rows of \code{x} are re-scaled to unit length before
    rotation, and scaled back afterwards.}
  \item{eps}{The tolerance for stopping: the relative change in the sum
    of singular values.}
}
\details{
  These seek a \sQuote{rotation} of the factors \code{x \%*\% T} that
  aims to clarify the structure of the loadings matrix.  The matrix
  \code{T} is a rotation (possibly with reflection) for \code{varimax},
  but a general linear transformation for \code{promax}, with the
  variance of the factors being preserved.
}
\value{
  A list with components
  \item{loadings}{The \sQuote{rotated} loadings matrix,
    \code{x \%*\% rotmat}, of class \code{"loadings"}.}
  \item{rotmat}{The \sQuote{rotation} matrix.}
}
\references{
  Hendrickson, A. E. and White, P. O. (1964) Promax: a quick method for
  rotation to orthogonal oblique structure. \emph{British Journal of
    Statistical Psychology}, \bold{17}, 65--70.

  Horst, P. (1965) \emph{Factor Analysis of Data Matrices.} Holt,
  Rinehart and Winston.  Chapter 10.

  Kaiser, H. F. (1958) The varimax criterion for analytic rotation in
  factor analysis. \emph{Psychometrika} \bold{23}, 187--200.

  Lawley, D. N. and Maxwell, A. E. (1971) \emph{Factor Analysis as a
    Statistical Method}. Second edition. Butterworths.
}
\seealso{\code{\link{factanal}}, \code{\link{Harman74.cor}}.}

\examples{
data(swiss)
## varimax with normalize = TRUE is the default
fa <- factanal( ~., 2, data = swiss)
varimax(loadings(fa), normalize = FALSE)
promax(loadings(fa))
}
\keyword{multivariate}
\eof
\name{vcov}
\alias{vcov}
\alias{vcov.lm}
\alias{vcov.glm}
\alias{vcov.lme}
\alias{vcov.gls}
\alias{vcov.coxph}
\alias{vcov.survreg}
\title{
Calculate Variance-Covariance Matrix for a Fitted Model Object
}
\description{
Returns the variance-covariance matrix of the main parameters of
a fitted model object.
}
\usage{
vcov(object, \dots)
}
\arguments{
  \item{object}{
    a fitted model object.
  }
  \item{\dots}{
    additional arguments for method functions.  For the
    \code{\link{glm}} method this can be used to pass a
    \code{dispersion} parameter.
  }
}
\value{
  A matrix of the estimated covariances between the parameter estimates
  in the linear or non-linear predictor of the model.
}
\details{
This is a generic function.
Functions with names beginning in \code{vcov.} will be
methods for this function.
Classes with methods for this function include:
\code{lm}, \code{glm}, \code{nls}, \code{lme}, \code{gls}, \code{coxph}
and \code{survreg}
}
\keyword{models}
\keyword{nonlinear}
\eof
\name{weighted.mean}
\title{Weighted Arithmetic Mean}
\usage{
weighted.mean(x, w, na.rm=FALSE)
}
\alias{weighted.mean}
\arguments{
  \item{x}{a numeric vector containing the values
    whose mean is to be computed.}
  \item{w}{a vector of weights the same length as \code{x} giving
    the weights to use for each element of \code{x}.}
  \item{na.rm}{a logical value indicating whether \code{NA}
    values in \code{x} should be stripped before the computation proceeds.}
}
\description{
  Compute a weighted mean of a numeric vector.
}
\details{
  If \code{w} is missing then all elements of \code{x} are
  given the same weight.
  
  Missing values in \code{w} are not handled.
}
\seealso{\code{\link{mean}}
}
\examples{
## GPA from Siegel 1994
wt <- c(5,  5,  4,  1)/15
x <- c(3.7,3.3,3.5,2.8)
xm <- weighted.mean(x,wt)
}
\keyword{univar}
\eof
\name{weighted.residuals}
\alias{weighted.residuals}
\title{Compute Weighted Residuals}
\usage{
weighted.residuals(obj, drop0 = TRUE)
}
\arguments{
  \item{obj}{\R object, typically of class \code{\link{lm}} or
    \code{\link{glm}}.}
  \item{drop0}{logical.  If \code{TRUE}, drop all cases with
    \code{\link{weights} == 0}.}
}
\description{
  Computed weighted residuals from a linear model fit.
}
\details{
  Weighted residuals are the usual residuals \eqn{R_i}{Ri}, multiplied
  by \eqn{\sqrt{w_i}}{wi^0.5}, where \eqn{w_i}{wi} are the
  \code{weights} as specified in \code{\link{lm}}'s call.

  Dropping cases with weights zero is compatible with
  \code{\link{influence}} and related functions.
}
\value{
  Numeric vector of length \eqn{n'}, where \eqn{n'} is the number of
  of non-0 weights (\code{drop0 = TRUE}) or the number of
  observations, otherwise.
}
\seealso{\code{\link{residuals}}, \code{\link{lm.influence}}, etc.}
\examples{
example("lm")
all.equal(weighted.residuals(lm.D9),
          residuals(lm.D9))
x <- 1:10
w <- 0:9
y <- rnorm(x)
weighted.residuals(lmxy <- lm(y ~ x, weights = w))
weighted.residuals(lmxy, drop0 = FALSE)
}
\keyword{regression}

\eof
\name{wilcox.test}
\alias{wilcox.test}
\alias{wilcox.test.default}
\alias{wilcox.test.formula}
\concept{Mann-Whitney Test}
\title{Wilcoxon Rank Sum and Signed Rank Tests}
\description{
  Performs one and two sample Wilcoxon tests on vectors of data; the
  latter is also known as \sQuote{Mann-Whitney} test.
}
\usage{
wilcox.test(x, \dots)

\method{wilcox.test}{default}(x, y = NULL, alternative = c("two.sided", "less", "greater"),
            mu = 0, paired = FALSE, exact = NULL, correct = TRUE,
            conf.int = FALSE, conf.level = 0.95, \dots)

\method{wilcox.test}{formula}(formula, data, subset, na.action, \dots)
}
\arguments{
  \item{x}{numeric vector of data values.}
  \item{y}{an optional numeric vector of data values.}
  \item{alternative}{a character string specifying the alternative
    hypothesis, must be one of \code{"two.sided"} (default),
    \code{"greater"} or \code{"less"}.  You can specify just the initial
    letter.}
  \item{mu}{a number specifying an optional location parameter.}
  \item{paired}{a logical indicating whether you want a paired test.}
  \item{exact}{a logical indicating whether an exact p-value should be
    computed.}
  \item{correct}{a logical indicating whether to apply continuity
    correction in the normal approximation for the p-value.}
  \item{conf.int}{a logical indicating whether a confidence interval
    should be computed.}
  \item{conf.level}{confidence level of the interval.}
  \item{formula}{a formula of the form \code{lhs ~ rhs} where \code{lhs}
    is a numeric variable giving the data values and \code{rhs} a factor
    with two levels giving the corresponding groups.}
  \item{data}{an optional data frame containing the variables in the
    model formula.}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.  Defaults to
    \code{getOption("na.action")}.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\details{
  The formula interface is only applicable for the 2-sample tests.

  If only \code{x} is given, or if both \code{x} and \code{y} are given
  and \code{paired} is \code{TRUE}, a Wilcoxon signed rank test of the
  null that the distribution of \code{x} (in the one sample case) or of
  \code{x-y} (in the paired two sample case) is symmetric about
  \code{mu} is performed.

  Otherwise, if both \code{x} and \code{y} are given and \code{paired}
  is \code{FALSE}, a Wilcoxon rank sum test (equivalent to the
  Mann-Whitney test) is carried out.  In this case, the null hypothesis
  is that the location of the distributions of \code{x} and \code{y}
  differ by \code{mu}.

  By default (if \code{exact} is not specified), an exact p-value is
  computed if the samples contain less than 50 finite values and there
  are no ties.  Otherwise, a normal approximation is used.

  Optionally (if argument \code{conf.int} is true), a nonparametric
  confidence interval and an estimator for the pseudomedian (one-sample
  case) or for the difference of the location parameters \code{x-y} is
  computed.  (The pseudomedian of a distribution \eqn{F} is the median
  of the distribution of \eqn{(u+v)/2}, where \eqn{u} and \eqn{v} are
  independent, each with distribution \eqn{F}.  If \eqn{F} is symmetric,
  then the pseudomedian and median coincide.  See Hollander & Wolfe
  (1973), page 34.)  If exact p-values are available, an exact
  confidence interval is obtained by the algorithm described in Bauer
  (1972), and the Hodges-Lehmann estimator is employed.  Otherwise, the
  returned confidence interval and point estimate are based on normal
  approximations.
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the test statistic with a name
    describing it.}
  \item{parameter}{the parameter(s) for the exact distribution of the
    test statistic.}
  \item{p.value}{the p-value for the test.}
  \item{null.value}{the location parameter \code{mu}.}
  \item{alternative}{a character string describing the alternative
    hypothesis.}
  \item{method}{the type of test applied.}
  \item{data.name}{a character string giving the names of the data.}
  \item{conf.int}{a confidence interval for the location parameter.
    (Only present if argument \code{conf.int = TRUE}.)}
  \item{estimate}{an estimate of the location parameter.
    (Only present if argument \code{conf.int = TRUE}.)}
}
\references{
  Myles Hollander & Douglas A. Wolfe (1973),
  \emph{Nonparametric statistical inference}.
  New York: John Wiley & Sons.
  Pages 27--33 (one-sample), 68--75 (two-sample).

  David F. Bauer (1972),
  Constructing confidence sets using rank statistics.
  \emph{Journal of the American Statistical Association}
  \bold{67}, 687--690.
}
\seealso{
  \code{\link{kruskal.test}} for testing homogeneity in location
  parameters in the case of two or more samples;
  \code{\link{t.test}} for a parametric alternative under normality
  assumptions.
}
\examples{
## One-sample test.
## Hollander & Wolfe (1973), 29f.
## Hamilton depression scale factor measurements in 9 patients with
##  mixed anxiety and depression, taken at the first (x) and second
##  (y) visit after initiation of a therapy (administration of a
##  tranquilizer).
x <- c(1.83,  0.50,  1.62,  2.48, 1.68, 1.88, 1.55, 3.06, 1.30)
y <- c(0.878, 0.647, 0.598, 2.05, 1.06, 1.29, 1.06, 3.14, 1.29)
wilcox.test(x, y, paired = TRUE, alternative = "greater")
wilcox.test(y - x, alternative = "less")    # The same.
wilcox.test(y - x, alternative = "less",
            exact = FALSE, correct = FALSE) # H&W large sample
                                            # approximation

## Two-sample test.
## Hollander & Wolfe (1973), 69f.
## Permeability constants of the human chorioamnion (a placental
##  membrane) at term (x) and between 12 to 26 weeks gestational
##  age (y).  The alternative of interest is greater permeability
##  of the human chorioamnion for the term pregnancy.
x <- c(0.80, 0.83, 1.89, 1.04, 1.45, 1.38, 1.91, 1.64, 0.73, 1.46)
y <- c(1.15, 0.88, 0.90, 0.74, 1.21)
wilcox.test(x, y, alternative = "g")        # greater
wilcox.test(x, y, alternative = "greater",
            exact = FALSE, correct = FALSE) # H&W large sample
                                            # approximation

wilcox.test(rnorm(10), rnorm(10, 2), conf.int = TRUE)

## Formula interface.
data(airquality)
boxplot(Ozone ~ Month, data = airquality)
wilcox.test(Ozone ~ Month, data = airquality,
            subset = Month \%in\% c(5, 8))
}
\keyword{htest}
\eof
\name{window}
\alias{window}
\alias{window.default}
\alias{window.ts}
\title{Time Windows}
\usage{
window(x, \dots)

\method{window}{ts}(x, \dots)

\method{window}{default}(x, start = NULL, end = NULL,
      frequency = NULL, deltat = NULL, extend = FALSE, \dots)
}
\arguments{
  \item{x}{a time-series or other object.}
  \item{start}{the start time of the period of interest.}
  \item{end}{the end time of the period of interest.}
  \item{frequency, deltat}{the new frequency can be specified by either
    (or both if they are consistent).}
  \item{extend}{logical. If true, the \code{start} and \code{end} values
    are allowed to extend the series. If false, attempts to extend the
    series give a warning and are ignored.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  \code{window} is a generic function which
  extracts the subset of the object \code{x}
  observed between the times \code{start} and \code{end}. If a
  frequency is specified, the series is then re-sampled at the new
  frequency.
}
\details{
  The start and end times can be specified as for \code{\link{ts}}. If
  there is no observation at the new \code{start} or \code{end},
  the immediately following (\code{start}) or preceding (\code{end})
  observation time is used.
}
\value{
  The value depends on the method. \code{window.default} will return a
  vector or matrix with an appropriate \code{\link{tsp}} attribute.

  \code{window.ts} differs from \code{window.default} only in
  ensuring the result is a \code{ts} object.

  If \code{extend = TRUE} the series will be padded with \code{NA} if
  needed.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{\code{\link{time}}, \code{\link{ts}}.}
\examples{
data(presidents)
window(presidents, 1960, c(1969,4)) # values in the 1960's
window(presidents, deltat=1)  # All Qtr1s
window(presidents, start=c(1945,3), deltat=1)  # All Qtr3s
window(presidents, 1944, c(1979,2), extend=TRUE)
}
\keyword{ts}
\eof
\name{xtabs}
\alias{xtabs}
\alias{print.xtabs}
\title{Cross Tabulation}
\description{
  Create a contingency table from cross-classifying factors, usually
  contained in a data frame, using a formula interface.
}
\usage{
xtabs(formula = ~., data = parent.frame(), subset, na.action,
      exclude = c(NA, NaN), drop.unused.levels = FALSE)
}
\arguments{
  \item{formula}{a formula object with the cross-classifying variables,
    separated by \code{+}, on the right hand side.  Interactions are not
    allowed.  On the left hand side, one may optionally give a vector or
    a matrix of counts; in the latter case, the columns are interpreted
    as corresponding to the levels of a variable.  This is useful if the
    data has already been tabulated, see the examples below.}
  \item{data}{a data frame, list or environment containing the variables
    to be cross-tabulated.}
  \item{subset}{an optional vector specifying a subset of observations
    to be used.}
  \item{na.action}{a function which indicates what should happen when
    the data contain \code{NA}s.}
  \item{exclude}{a vector of values to be excluded when forming the
    set of levels of the classifying factors.}
  \item{drop.unused.levels}{a logical indicating whether to drop unused
    levels in the classifying factors.  If this is \code{FALSE} and
    there are unused levels, the table will contain zero marginals, and
    a subsequent chi-squared test for independence of the factors will
    not work.}
}
\details{
  There is a \code{summary} method for contingency table objects created
  by \code{table} or \code{xtabs}, which gives basic information and
  performs a chi-squared test for independence of factors (note that the
  function \code{\link[stats]{chisq.test}} in package \pkg{ctest}
  currently only handles 2-d tables).

  If a left hand side is given in \code{formula}, its entries are simply
  summed over the cells corresponding to the right hand side; this also
  works if the lhs does not give counts.
}
\value{
  A contingency table in array representation of class \code{c("xtabs",
    "table")}, with a \code{"call"} attribute storing the matched call.
}
\seealso{
  \code{\link{table}} for \dQuote{traditional} cross-tabulation, and
  \code{\link{as.data.frame.table}} which is the inverse operation of
  \code{xtabs} (see the \code{DF} example below).
}
\examples{
data(esoph)
## 'esoph' has the frequencies of cases and controls for all levels of
## the variables 'agegp', 'alcgp', and 'tobgp'.
xtabs(cbind(ncases, ncontrols) ~ ., data = esoph)
## Output is not really helpful ... flat tables are better:
ftable(xtabs(cbind(ncases, ncontrols) ~ ., data = esoph))
## In particular if we have fewer factors ...
ftable(xtabs(cbind(ncases, ncontrols) ~ agegp, data = esoph))

data(UCBAdmissions)
## This is already a contingency table in array form.
DF <- as.data.frame(UCBAdmissions)
## Now 'DF' is a data frame with a grid of the factors and the counts
## in variable 'Freq'.
DF
## Nice for taking margins ...
xtabs(Freq ~ Gender + Admit, DF)
## And for testing independence ...
summary(xtabs(Freq ~ ., DF))

data(warpbreaks)
## Create a nice display for the warp break data.
warpbreaks$replicate <- rep(1:9, len = 54)
ftable(xtabs(breaks ~ wool + tension + replicate, data = warpbreaks))
}
\keyword{category}
\eof
\name{C}
\title{Sets Contrasts for a Factor}
\usage{
C(object, contr, how.many, \dots)
}
\alias{C}
\arguments{
  \item{object}{a factor or ordered factor}
  \item{contr}{which contrasts to use. Can be a matrix with one row for
    each level of the factor or a suitable function like
    \code{contr.poly} or a character string giving the name of the function}
  \item{how.many}{the number of contrasts to set, by default one less
    than \code{nlevels(object)}.}
  \item{\dots}{additional arguments for the function \code{contr}.}
}
\description{
  Sets the \code{"contrasts"} attribute for the factor.
}
\details{
  For compatibility with S, \code{contr} can be \code{treatment},
  \code{helmert}, \code{sum} or \code{poly} (without quotes) as shorthand
  for \code{contr.treatment} and so on.
}
\value{
  The factor \code{object} with the \code{"contrasts"} attribute set.
}
\seealso{
  \code{\link{contrasts}}, \code{\link{contr.sum}}, etc.
}
\references{
  Chambers, J. M. and Hastie, T. J. (1992)
  \emph{Statistical models.}
  Chapter 2 of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\examples{
## reset contrasts to defaults
options(contrasts=c("contr.treatment", "contr.poly"))
data(warpbreaks)
attach(warpbreaks)
tens <- C(tension, poly, 1)
attributes(tens)
detach()
## tension SHOULD be an ordered factor, but as it is not we can use
aov(breaks ~ wool + tens + tension, data=warpbreaks)

## show the use of ...  The default contrast is contr.treatment here
summary(lm(breaks ~ wool + C(tension, base=2), data=warpbreaks))


data(esoph) # following on from help(esoph)
model3 <- glm(cbind(ncases, ncontrols) ~ agegp + C(tobgp, , 1) +
     C(alcgp, , 1), data = esoph, family = binomial())
summary(model3)
}
\keyword{models}
\eof
\name{CO2}
\docType{data}
\alias{CO2}
\title{Carbon Dioxide uptake in grass plants}
\description{
  The \code{CO2} data frame has 84 rows and 5 columns of data from an
  experiment on the cold tolerance of the grass species
  \emph{Echinochloa crus-galli}.
}
\usage{data(CO2)}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Plant}{
      an ordered factor with levels
      \code{Qn1} < \code{Qn2} < \code{Qn3} < \dots < \code{Mc1}
      giving a unique identifier for each plant.
    }
    \item{Type}{
      a factor with levels
      \code{Quebec} 
      \code{Mississippi}
      giving the origin of the plant
    }
    \item{Treatment}{
      a factor with levels
      \code{nonchilled} 
      \code{chilled}
    }
    \item{conc}{
      a numeric vector of ambient carbon dioxide concentrations (mL/L).
    }
    \item{uptake}{
      a numeric vector of carbon dioxide uptake rates
      (\eqn{\mu\mbox{mol}/m^2}{umol/m^2} sec).
    }
  }
}
\details{
  The \eqn{CO_2}{CO2} uptake of six plants from Quebec and six plants
  from Mississippi was measured at several levels of ambient
  \eqn{CO_2}{CO2} concentration.  Half the plants of each type were
  chilled overnight before the experiment was conducted.
}
\source{
  Potvin, C., Lechowicz, M. J. and Tardif, S. (1990)
  \dQuote{The statistical analysis of ecophysiological response curves
  obtained from experiments involving repeated measures}, \emph{Ecology},
  \bold{71}, 1389--1400.

  Pinheiro, J. C. and Bates, D. M. (2000)
  \emph{Mixed-effects Models in S and S-PLUS}, Springer.
}
\examples{
data(CO2)
coplot(uptake ~ conc | Plant, data = CO2, show = FALSE, type = "b")
## fit the data for the first plant
fm1 <- nls(uptake ~ SSasymp(conc, Asym, lrc, c0),
   data = CO2, subset = Plant == 'Qn1')
summary(fm1)
## fit each plant separately
fmlist <- list()
for (pp in levels(CO2$Plant)) {
  fmlist[[pp]] <- nls(uptake ~ SSasymp(conc, Asym, lrc, c0),
      data = CO2, subset = Plant == pp)
}
## check the coefficients by plant
sapply(fmlist, coef)
}
\keyword{datasets}
\keyword{nonlinear}
\eof
\name{zcbind}
\alias{.cbind.ts}
\title{Bind Two or More Time Series}
\description{
  Bind Two or More Time Series which have common frequency.
}
\usage{
.cbind.ts(sers, nmsers, dframe = FALSE, union = TRUE)
}
\arguments{
  \item{sers}{a list of two or more univariate or multivariate time
    series, or objects which can coerced to time series.}
  \item{nmsers}{a character vector of the same length as \code{sers}
    with the names for the time series.}
  \item{dframe}{logical; if \code{TRUE} return the result as a data
    frame.}
  \item{union}{logical; if \code{TRUE}, act as \code{ts.union} or
    \code{ts.intersect}.}
}
\details{
  This is an internal function which is not to be called by the user.
}
\keyword{internal}
\eof
