/******************************************************************************/
/* smooth_gtk1_engine.c - Style Functions and Engine Hooks for GTK1 Engine    */
/******************************************************************************/
/* Smooth Theme Engine                                                        */
/* Copyright (C) 2002-2004 Andrew Johnson                                     */
/*                                                                            */
/* This library is free software; you can redistribute it and/or              */
/* modify it under the terms of the GNU Lesser General Public                 */
/* License as published by the Free Software Foundation; either               */
/* version 2.1 of the License, or (at your option) any later version.         */
/*                                                                            */
/* This library is distributed in the hope that it will be useful,            */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of             */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          */
/* Lesser General Public License for more details.                            */
/*                                                                            */
/* You should have received a copy of the GNU Lesser General Public           */
/* License along with this library; if not, write to the Free Software        */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  */
/*                                                                            */
/* Author(s): Andrew Johnson <ajgenius@ajgenius.us>                           */
/******************************************************************************/
/* Portions Based on GTK+                                                     */
/*   Peter Mattis <petm@xcf.berkeley.edu>                                     */
/*   Spencer Kimball <spencer@xcf.berkeley.edu>                               */
/*   Josh MacDonald <jmacd@xcf.berkeley.edu>                                  */
/*                                                                            */
/* Portions Based on the EnGradient Engine                                    */
/*   Andrew Cattau                                                            */
/******************************************************************************/
#include "smooth_gtk1_engine.h"
#include "smooth_gtk1_drawing.h"
#include "smooth_gtk1_misc.h"
#include "smooth_gtk1_rc.h"
#include "smooth_gtk1_patches.h"

/**********************************/
/* Register & Initialize RC Style */
/**********************************/
static void
smooth_rc_style_merge (GtkRcStyle * dest,
                       GtkRcStyle * src)
{
	SmoothRcStyle *src_data = SMOOTH_RC_DATA (src);
	SmoothRcStyle *dest_data = SMOOTH_RC_DATA (dest);

	if (!dest_data) 
	{
		dest_data = NEW_THEME_DATA(dest);
		dest_data->xthickness       = src_data->xthickness;
		dest_data->ythickness       = src_data->ythickness;
		dest_data->paned_handle_size = src_data->paned_handle_size;

		dest_data->refcount         = 1;
        }
	smooth_gtkrc_style_merge (dest_data, src_data);

	SET_THEME_DATA(dest, dest_data);
}

static guint 
smooth_rc_style_parse (GScanner * scanner, 
                       GtkRcStyle * rc_style)
{
  static GQuark scope_id = 0;
  SmoothRcStyle *smooth_style = NEW_THEME_DATA(rc_style);
  
  guint old_scope;
  guint token;
  guint i;
  
  smooth_style->refcount = 1;
  smooth_rc_style_init (smooth_style);

  /* Set up a new scope in this scanner */
  if (!scope_id)
    {
      scope_id = g_quark_from_string ("theme_engine");
    }

  /* If we bail out due to errors, we don't reset the scope, so the
   * error messaging code can make sense of our tokens. */
  old_scope = g_scanner_set_scope (scanner, scope_id);

  /* Check if we already added our symbols to this scope */
  if (!g_scanner_lookup_symbol (scanner, theme_symbols[0].name))
    {
      g_scanner_freeze_symbol_table (scanner);
      for (i = 0; i < n_theme_symbols; i++)
	{
	  g_scanner_scope_add_symbol (scanner, scope_id, theme_symbols[i].name, 
				      GINT_TO_POINTER (theme_symbols[i].token));
	}
      g_scanner_thaw_symbol_table (scanner);
    }

  if (!g_scanner_lookup_symbol (scanner, gtk1_theme_symbols[0].name))
    {
      g_scanner_freeze_symbol_table (scanner);
      for (i = 0; i < n_gtk1_theme_symbols; i++)
	{
	  g_scanner_scope_add_symbol (scanner, scope_id, gtk1_theme_symbols[i].name, 
				      GINT_TO_POINTER (gtk1_theme_symbols[i].token));
	}
      g_scanner_thaw_symbol_table (scanner);
    }

  /* Read to go, now parse the top level */
  token = g_scanner_peek_next_token (scanner);
  while (token != G_TOKEN_RIGHT_CURLY)
    {
      token = smooth_gtkrc_parse(scanner, smooth_style, token);

      if (token != G_TOKEN_NONE)
	{
          g_free (smooth_style);
	  return token;
	}

      token = g_scanner_peek_next_token (scanner);
    }

  g_scanner_get_next_token(scanner);

  SET_THEME_DATA(rc_style, smooth_style);
  
  g_scanner_set_scope (scanner, old_scope);

  return G_TOKEN_NONE;
}

/***************************************/
/* Initialize Drawing Style            */
/***************************************/
GtkStyleClass smooth_default_class = {
  2,
  2,
  smooth_draw_hline,
  smooth_draw_vline,
  smooth_draw_shadow,
  smooth_draw_polygon,
  smooth_draw_arrow,
  smooth_draw_diamond,
  smooth_draw_oval,
  smooth_draw_string,
  smooth_draw_box,
  smooth_draw_flat_box,
  smooth_draw_check,
  smooth_draw_option,
  smooth_draw_cross,
  smooth_draw_ramp,
  smooth_draw_tab,
  smooth_draw_shadow_gap,
  smooth_draw_box_gap,
  smooth_draw_extension,
  smooth_draw_focus,
  smooth_draw_slider,
  smooth_draw_handle
};


/* Theme Style Function Overrides */

void
smooth_style_duplicate (GtkStyle * dest, GtkStyle * src)
{
  SmoothRcStyle *dest_data = NEW_THEME_DATA(dest); 
  
  dest_data->refcount = 1;
  dest->klass = &smooth_default_class;
  smooth_rc_style_init(dest_data);
  SET_THEME_DATA(dest, dest_data);
}

void
smooth_style_realize (GtkStyle * style)
{
}

void
smooth_style_unrealize (GtkStyle * style)
{
}

static void
theme_data_ref (SmoothRcStyle * theme_data)
{
  if (theme_data) theme_data->refcount++;
}

static void part_finalize (smooth_part_style *part)
{
  gint i;
  for (i=0; i < 5; i++) {
    if (part->fill.file_name[i])
            g_string_free(part->fill.file_name[i], TRUE);
  }
}

static void
theme_data_unref (SmoothRcStyle * theme_data)
{
  if (theme_data) theme_data->refcount--;
  if ((theme_data) && (theme_data->refcount == 0)) {
  	gint i;
        for (i=0; i < 5; i++) {
          if (theme_data->fill.file_name[i])
            g_string_free(theme_data->fill.file_name[i], TRUE);

          if (theme_data->focus.pattern[i])
            g_free(theme_data->focus.pattern[i]);
        }
        part_finalize(THEME_PART(&theme_data->grip));
        part_finalize(THEME_PART(&theme_data->check));
        part_finalize(THEME_PART(&theme_data->option));
        part_finalize(THEME_PART(&theme_data->trough));
        part_finalize(&theme_data->progress);
        part_finalize(THEME_PART(&theme_data->button));
        part_finalize(&theme_data->button.button_default);
        part_finalize(THEME_PART(&theme_data->tabs));
        part_finalize(&theme_data->tabs.active_tab);
    GDKFinalizeColorCube(&theme_data->colors);
    g_free (theme_data);
    theme_data = NULL;
  }  
}

static void
smooth_rc_style_destroy (GtkRcStyle * rc_style)
{
  theme_data_unref (SMOOTH_RC_DATA(rc_style));
}

static void
smooth_style_destroy (GtkStyle * style)
{
  theme_data_unref (THEME_DATA(style));
}

static void
smooth_rc_style_to_style(GtkStyle * style, 
                         GtkRcStyle * rc_style)
{
  SmoothRcStyle *data = SMOOTH_RC_DATA(rc_style);

  style->klass = &smooth_default_class;
  style->engine_data = data;
  
  if (data->xthickness >= 0)
  style->klass->xthickness = data->xthickness;

  if (data->ythickness >= 0)
  style->klass->ythickness = data->ythickness;
  
  theme_data_ref (data);
}

static void
theme_set_background (GtkStyle * style,
		      GdkWindow * window, 
		      GtkStateType state_type)
{
  GdkPixmap *pixmap;
  gint parent_relative;

  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  if (style->bg_pixmap[state_type]) {
    if (style->bg_pixmap[state_type] == (GdkPixmap *) GDK_PARENT_RELATIVE) {
      pixmap = NULL;
      parent_relative = TRUE;
    } else {
      pixmap = style->bg_pixmap[state_type];
      parent_relative = FALSE;
    }

    gdk_window_set_back_pixmap (window, pixmap, parent_relative);
  } else {
    gdk_window_set_background (window, &style->bg[state_type]);
  }
}

/****************/
/* Engine Hooks */
/****************/

G_MODULE_EXPORT void
theme_init (GtkThemeEngine * engine)
{
  gdk_rgb_init();
  engine->parse_rc_style = smooth_rc_style_parse;
  engine->merge_rc_style = smooth_rc_style_merge;
  engine->rc_style_to_style = smooth_rc_style_to_style;
  engine->duplicate_style = smooth_style_duplicate;
  engine->realize_style = smooth_style_realize;
  engine->unrealize_style = smooth_style_unrealize;
  engine->destroy_rc_style = smooth_rc_style_destroy;
  engine->destroy_style = smooth_style_destroy;
  engine->set_background = theme_set_background;

  patches_install();
  SmoothDrawingInterfaceInitialize();
}

G_MODULE_EXPORT void
theme_exit (void)
{
  patches_uninstall();
  SmoothDrawingInterfaceFinalize();
}

/* The following function will be called by GTK+ when the module
 * is loaded and checks to see if we are compatible with the
 * version of GTK+ that loads us.
*/
G_MODULE_EXPORT const gchar *g_module_check_init (GModule * module);

const gchar *

g_module_check_init (GModule * module)
{
  return gtk_check_version (GTK_MAJOR_VERSION,
			    GTK_MINOR_VERSION,
			    GTK_MICRO_VERSION - GTK_INTERFACE_AGE);
}
