/* gtk-flat-theme - A gtk+ theme engine.
   Copyright (C) 2000  Mark Slicker <jamess1@wwnet.com>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gmodule.h>

#include "flat_theme.h"

/* Theme functions to export */
void                theme_init(GtkThemeEngine * engine);
void                theme_exit(void);

/* Exported vtable from th_draw */

extern GtkStyleClass flat_default_class;

/* internals */

/* external theme functions called */


/* Information about a single RC style
 */


static void
theme_data_ref (FlatThemeData *theme_data)
{
  theme_data->refcount++;
}

static void
theme_data_unref (FlatThemeData *theme_data)
{
  theme_data->refcount--;
  if (theme_data->refcount == 0) {
    int i;
    for (i = 0; i < 5; i++) {
      gdk_pixmap_unref (theme_data->nw[i]);
      gdk_pixmap_unref (theme_data->nw[i]);
      gdk_pixmap_unref (theme_data->nw[i]);
      gdk_pixmap_unref (theme_data->nw[i]);
    }
    g_free (theme_data);
  }
}

/* external theme functions called */

static guint
theme_parse_rc_style(GScanner * scanner,
		     GtkRcStyle * rc_style)
{
  FlatThemeData *data;
  guint               token;

  token = g_scanner_peek_next_token(scanner);
  while (token != G_TOKEN_RIGHT_CURLY)
    {
      switch (token)
	{
	default:
	  g_scanner_get_next_token(scanner);
	  token = G_TOKEN_RIGHT_CURLY;
	  break;
	}

      if (token != G_TOKEN_NONE)
	return token;

      token = g_scanner_peek_next_token(scanner);
    }

  g_scanner_get_next_token(scanner);

  rc_style->engine_data = data = g_new (FlatThemeData, 1);
  theme_data_ref (data);

  return G_TOKEN_NONE;
}

static void
theme_merge_rc_style(GtkRcStyle * dest,
		     GtkRcStyle * src)
{
}

static void
theme_rc_style_to_style(GtkStyle * style,
			GtkRcStyle * rc_style)
{
  style->klass = &flat_default_class;
}

static void
theme_duplicate_style(GtkStyle * dest,
		      GtkStyle * src)
{
  dest->klass = src->klass;
}

static void
fill_color (GdkColor *color, GdkColormap *cmap)
{
  GdkVisual *visual = gdk_colormap_get_visual (cmap);
  gulong pixel = color->pixel;

  switch (visual->type) {
  case GDK_VISUAL_STATIC_GRAY:
  case GDK_VISUAL_GRAYSCALE:
  case GDK_VISUAL_STATIC_COLOR:
  case GDK_VISUAL_PSEUDO_COLOR:
    color->red = cmap->colors[pixel].red;
    color->green = cmap->colors[pixel].green;
    color->blue = cmap->colors[pixel].blue;
    break;
  case GDK_VISUAL_TRUE_COLOR:
  case GDK_VISUAL_DIRECT_COLOR:
    color->red = (pixel & visual->red_mask) >> visual->red_shift << (16 - visual->red_prec);
    color->green = (pixel & visual->green_mask) >> visual->green_shift << (16 - visual->green_prec);
    color->blue = (pixel & visual->blue_mask) >> visual->blue_shift << (16 - visual->blue_prec);
    break;
  }
}

static void
blend_images (GdkImage *c, GdkImage *a, GdkImage *b, float *bias_array,
              int width, int height, GdkColormap *cmap)
{
  int x, y;
  GdkColor ca, cb, cc;

  for (y = 0; y < height; y++) {
    for (x = 0; x < width; x++) {
      float bias = bias_array[y*height+x];
      ca.pixel = gdk_image_get_pixel (a, x, y);
      cb.pixel = gdk_image_get_pixel (b, x, y);
      fill_color (&ca, cmap);
      fill_color (&cb, cmap);
      cc.red = (gushort)(ca.red * (1.0f - bias) + cb.red * bias);
      cc.green = (gushort)(ca.green * (1.0f - bias) + cb.green * bias);
      cc.blue = (gushort)(ca.blue * (1.0f - bias) + cb.blue * bias);
      gdk_color_alloc (cmap, &cc);
      gdk_image_put_pixel (c, x, y, cc.pixel); 
    }
  }
}

static void
copy_image (GdkImage *b, GdkImage *a, int xsrc, int ysrc,
            int xdest, int ydest, int width, int height)
{
  int x, y;
  gulong pixel;

  for (y = 0; y < height; y++) {
    for (x = 0; x < width; x++) {
      pixel = gdk_image_get_pixel (a, xsrc+x, ysrc+y);
      gdk_image_put_pixel (b, xdest+x, xdest+y, pixel); 
    }
  }
}

static void
set_image_color (GdkImage *a, GdkColor *c, int width, int height)
{
  int x, y;
  gulong pixel = c->pixel;

  for (y = 0; y < height; y++) {
    for (x = 0; x < width; x++) {
      gdk_image_put_pixel (a, x, y, pixel); 
    }
  }
}

static float corner_bias_a[4][4][4] = {
  {{          0, .196078431f, .690106078f, .960784314f},
   {.196078431f, .960784314f,           0,           0},
   {.690106078f,           0,           0,           0},
   {.960784314f,           0,           0,           0}},
  {0},
  {0},
  {0}
};

static float corner_bias_b[4][4][4] = {
  {{0,           0,           0,          0},
   {0,           0, .415686275f, .050980392},
   {0, .415686275f,           0,          0},
   {0, .050980392f,           0,          0}},
  {0},
  {0},
  {0}
};

static float corner_bias_c[4][4][4] = {
  {{0,    0,    0,    0},
   {0,    0, 1.0f, 1.0f},
   {0, 1.0f, 1.0f, 1.0f},
   {0, 1.0f, 1.0f, 1.0f}},
  {0},
  {0},
  {0}
};

static float focused_corner_bias_a[4][4][4] = {
  {{.960784314f, 0, 0, 0},
   {0}, {0}, {0}},
  {0}, {0}, {0}
};

static float focused_corner_bias_b[4][4][4] = {
  {{   0,        1.0f,        1.0f, 1.0f},
   {1.0f, .623529412f, .082352941f,    0},
   {1.0f, .082352941f,           0,    0},
   {1.0f,           0,           0,    0}},
  {0}, {0}, {0}
};

static float focused_corner_bias_c[4][4][4] = {
  {{0,    1.0f, 1.0f, 1.0f},
   {1.0f, 1.0f, 1.0f, 1.0f},
   {1.0f, 1.0f, 1.0f, 1.0f},
   {1.0f, 1.0f, 1.0f, 1.0f}},
  {0}, {0}, {0}
};

static float radio_bias[12][12] = {
  {0, 0, 0, .454901961f, .796078431f, .976470588f, 0, 0, 0, 0, 0, 0},
  {0, .133333333f, .796078431f, .650980392f, .247058824f, .031372549, 0, 0, 0, 0, 0, 0},
  {0, .796078431f, .450980392f, 0, 0, 0, 0, 0, 0, 0, 0, 0},
  {.454901961f, .650980392f, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
  {.796078431f, .247058824f, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
  {.976470588f, .031372549f, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
  {0}, {0}, {0}, {0}, {0}
};

static float radio_on_bias[12][12] = {
  {0},
  {0},
  {0},
  {0, 0, 0, 0, .584213725f, .949019608f, 0, 0, 0, 0, 0, 0},
  {0, 0, 0, .584213725f, 1.0f, 1.0f, 0, 0, 0, 0, 0, 0},
  {0, 0, 0, .949019608f, 1.0f, 1.0f, 0, 0, 0, 0, 0, 0},
  {0}, {0}, {0}, {0}, {0}
};

#define flip(b, a, l, x, y)			\
do {						\
  for (i = 0; i < l; i++)			\
    for (j = 0; j < l; j++)			\
      b[i][j] = a[y?l-i-1:i][x?l-j-1:j];	\
} while (0)

static void
init ()
{
  int i, j;

  /* make the other corner biases */
  flip (corner_bias_a[1], corner_bias_a[0], 4, 0, 1);
  flip (corner_bias_a[2], corner_bias_a[0], 4, 1, 0);
  flip (corner_bias_a[3], corner_bias_a[0], 4, 1, 1);
  flip (corner_bias_b[1], corner_bias_b[0], 4, 0, 1);
  flip (corner_bias_b[2], corner_bias_b[0], 4, 1, 0);
  flip (corner_bias_b[3], corner_bias_b[0], 4, 1, 1);
  flip (corner_bias_c[1], corner_bias_c[0], 4, 0, 1);
  flip (corner_bias_c[2], corner_bias_c[0], 4, 1, 0);
  flip (corner_bias_c[3], corner_bias_c[0], 4, 1, 1);
  flip (focused_corner_bias_a[1], focused_corner_bias_a[0], 4, 0, 1);
  flip (focused_corner_bias_a[2], focused_corner_bias_a[0], 4, 1, 0);
  flip (focused_corner_bias_a[3], focused_corner_bias_a[0], 4, 1, 1);
  flip (focused_corner_bias_b[1], focused_corner_bias_b[0], 4, 0, 1);
  flip (focused_corner_bias_b[2], focused_corner_bias_b[0], 4, 1, 0);
  flip (focused_corner_bias_b[3], focused_corner_bias_b[0], 4, 1, 1);
  flip (focused_corner_bias_c[1], focused_corner_bias_c[0], 4, 0, 1);
  flip (focused_corner_bias_c[2], focused_corner_bias_c[0], 4, 1, 0);
  flip (focused_corner_bias_c[3], focused_corner_bias_c[0], 4, 1, 1);
  
  /* fill in the rest of the radio biases*/
  for (i = 0; i < 6; i++) {
    for (j = 0; j < 6; j++) {
      radio_bias[6+i][j] = radio_bias[5-i][j];
      radio_bias[i][6+j] = radio_bias[i][5-j];
      radio_bias[6+i][6+j] = radio_bias[5-i][5-j];
      radio_on_bias[6+i][j] = radio_on_bias[5-i][j];
      radio_on_bias[i][6+j] = radio_on_bias[i][5-j];
      radio_on_bias[6+i][6+j] = radio_on_bias[5-i][5-j];
    }
  }
}

static void
theme_realize_style(GtkStyle *style)
{
  static first = TRUE;
  FlatThemeData *data = style->engine_data;
  GdkImage *bgimage, *fgimage;
  GdkColormap *cmap = style->colormap;
  GdkGC *gc = style->white_gc;
  GdkImage *a, *b, *c, *d, *e;
  GdkVisual *visual = gdk_colormap_get_visual (cmap);
  GdkPixmap *p[4];
  int i, j, width, height;

  if (first) {
    init ();
    first = FALSE;
  }

  a = gdk_image_new (GDK_IMAGE_NORMAL, visual, 12, 12);
  b = gdk_image_new (GDK_IMAGE_NORMAL, visual, 12, 12);
  c = gdk_image_new (GDK_IMAGE_NORMAL, visual, 4, 4);
  d = gdk_image_new (GDK_IMAGE_NORMAL, visual, 4, 4);
  e = gdk_image_new (GDK_IMAGE_NORMAL, visual, 4, 4);


  if (!data)
    style->engine_data = data = g_new (FlatThemeData, 1);

  if (style->bg_pixmap[GTK_STATE_NORMAL]) {
    gdk_window_get_geometry (style->bg_pixmap[GTK_STATE_NORMAL],
                             NULL, NULL, &width, &height, NULL);
    bgimage = gdk_image_get (style->bg_pixmap[GTK_STATE_NORMAL],
                             0, 0, width, height);
  } else
    bgimage = NULL;
 
  for (i = 0; i < 5; i++) {
    if (style->bg_pixmap[i]) {
      gdk_window_get_geometry (style->bg_pixmap[i],
                               NULL, NULL, &width, &height, NULL);
      fgimage = gdk_image_get (style->bg_pixmap[i],
                               0, 0, width, height);
    } else
      fgimage = NULL;
    /* draw rounded corners */
    p[0] = data->nw[i] = gdk_pixmap_new (NULL, 4, 4, style->depth);
    p[1] = data->sw[i] = gdk_pixmap_new (NULL, 4, 4, style->depth);
    p[2] = data->ne[i] = gdk_pixmap_new (NULL, 4, 4, style->depth);
    p[3] = data->se[i] = gdk_pixmap_new (NULL, 4, 4, style->depth);

    for (j = 0; j < 4; j++) {
      if (bgimage)
        copy_image (a, bgimage, 0, 0, 0, 0, 4, 4);
      else
        set_image_color (a, &style->bg[GTK_STATE_NORMAL], 4, 4);
    
      set_image_color (b, &style->fg[i], 4, 4);
      blend_images (c, a, b, &corner_bias_a[j][0][0], 4, 4, cmap);

      if (fgimage)
        copy_image (a, fgimage, 0, 0, 0, 0, 4, 4);
      else
        set_image_color (a, &style->bg[i], 4, 4);

      blend_images (d, a, b, &corner_bias_b[j][0][0], 4, 4, cmap);
      blend_images (e, c, d, &corner_bias_c[j][0][0], 4, 4, cmap);
      gdk_draw_image (p[j], style->white_gc,
                      e, 0, 0, 0, 0, 4, 4);
    }

    /* draw rounded corner focus */
    p[0] = data->fnw[i] = gdk_pixmap_new (NULL, 4, 4, style->depth);
    p[1] = data->fsw[i] = gdk_pixmap_new (NULL, 4, 4, style->depth);
    p[2] = data->fne[i] = gdk_pixmap_new (NULL, 4, 4, style->depth);
    p[3] = data->fse[i] = gdk_pixmap_new (NULL, 4, 4, style->depth);

    for (j = 0; j < 4; j++) {
      if (bgimage)
        copy_image (a, bgimage, 1, 1, 0, 0, 4, 4);
      else
        set_image_color (a, &style->bg[GTK_STATE_NORMAL], 4, 4);
    
      set_image_color (b, &style->fg[i], 4, 4);
      blend_images (c, a, b, &focused_corner_bias_a[j][0][0], 4, 4, cmap);

      if (fgimage)
        copy_image (a, fgimage, 1, 1, 0, 0, 4, 4);
      else
        set_image_color (a, &style->bg[i], 4, 4);

      blend_images (d, a, b, &focused_corner_bias_b[j][0][0], 4, 4, cmap);
      blend_images (e, c, d, &focused_corner_bias_c[j][0][0], 4, 4, cmap);
      gdk_draw_image (p[j], style->white_gc,
                      e, 0, 0, 0, 0, 4, 4);
    }

    /* draw radio buttons */
    data->radio[i] = gdk_pixmap_new (NULL, 12, 12, style->depth);
    data->radio_on[i] = gdk_pixmap_new (NULL, 12, 12, style->depth);
    set_image_color (b, &style->fg[i], 12, 12);
    if (fgimage) {
      blend_images (fgimage, fgimage, b, &radio_bias[0][0], 12, 12, cmap);
      gdk_draw_image (data->radio[i], style->white_gc,
                      fgimage, 0, 0, 0, 0, 12, 12);
      blend_images (fgimage, fgimage, b, &radio_on_bias[0][0], 12, 12, cmap);
      gdk_draw_image (data->radio_on[i], style->white_gc,
                      fgimage, 0, 0, 0, 0, 12, 12);
    } else {
      set_image_color (a, &style->bg[i], 12, 12);
      blend_images (a, a, b, &radio_bias[0][0], 12, 12, cmap);
      gdk_draw_image (data->radio[i], style->white_gc,
                      a, 0, 0, 0, 0, 12, 12);
      blend_images (a, a, b, &radio_on_bias[0][0], 12, 12, cmap);
      gdk_draw_image (data->radio_on[i], style->white_gc,
                      a, 0, 0, 0, 0, 12, 12);
    }

    if (fgimage)
      gdk_image_destroy (fgimage);
  }

  if (bgimage)
    gdk_image_destroy (bgimage);

  gdk_image_destroy (a);
  gdk_image_destroy (b);
  gdk_image_destroy (c);
  gdk_image_destroy (d);
  gdk_image_destroy (e);
}

static void
theme_unrealize_style(GtkStyle * style)
{
}

static void
theme_destroy_rc_style(GtkRcStyle * rc_style)
{
}

static void
theme_destroy_style(GtkStyle * style)
{
}

/******************************************************************/
void
theme_init(GtkThemeEngine * engine)
{
  engine->parse_rc_style = theme_parse_rc_style;
  engine->merge_rc_style = theme_merge_rc_style;
  engine->rc_style_to_style = theme_rc_style_to_style;
  engine->duplicate_style = theme_duplicate_style;
  engine->realize_style = theme_realize_style;
  engine->unrealize_style = theme_unrealize_style;
  engine->destroy_rc_style = theme_destroy_rc_style;
  engine->destroy_style = theme_destroy_style;
  engine->set_background = NULL;
}

void
theme_exit(void)
{
}

/* The following function will be called by GTK+ when the module
 * is loaded and checks to see if we are compatible with the
 * version of GTK+ that loads us.
 */
G_MODULE_EXPORT const gchar* g_module_check_init (GModule *module);
const gchar*
g_module_check_init (GModule *module)
{
  return gtk_check_version (GTK_MAJOR_VERSION,
			    GTK_MINOR_VERSION,
			    GTK_MICRO_VERSION - GTK_INTERFACE_AGE);
}
