/*
 * vinagre-main.c
 * This file is part of vinagre
 *
 * Copyright (C) 2007,2008,2009 - Jonh Wendell <wendell@bani.com.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef ENABLE_INTROSPECTION
#include <girepository.h>
#endif

#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <locale.h>

#include "vinagre-window.h"
#include "vinagre-utils.h"
#include "vinagre-prefs.h"
#include "vinagre-cache-prefs.h"
#include "vinagre-debug.h"
#include "vinagre-ssh.h"
#include "vinagre-options.h"
#include "vinagre-plugins-engine.h"

#ifdef HAVE_TELEPATHY
#include "vinagre-tubes-manager.h"
#endif

#ifdef VINAGRE_ENABLE_AVAHI
#include "vinagre-mdns.h"
#endif

static gboolean startup_called = FALSE;
static GtkWindow *window = NULL;
static GOptionContext *context = NULL;

#ifdef HAVE_TELEPATHY
static VinagreTubesManager *vinagre_tubes_manager = NULL;
#endif

static void
app_init (GtkApplication *app)
{
  vinagre_debug_init ();
  vinagre_debug_message (DEBUG_APP, "Startup");

  vinagre_cache_prefs_init ();

  window = GTK_WINDOW (vinagre_window_new ());
  gtk_window_set_application (window, app);
  gtk_widget_show (GTK_WIDGET (window));

  vinagre_utils_handle_debug ();

#ifdef HAVE_TELEPATHY
   vinagre_tubes_manager = vinagre_tubes_manager_new (window);
#endif

  /* fake call, just to ensure this symbol will be present at vinagre.so */
  vinagre_ssh_connect (NULL, NULL, -1, NULL, NULL, NULL, NULL, NULL);
}

static void
app_startup (GApplication *app,
	     void         *user_data)
{
  /* We don't do anything here, as we need to know the options
   * when we set everything up.
   * Note that this will break D-Bus activation of the application */
  startup_called = TRUE;
}

static int
app_command_line (GApplication            *app,
		  GApplicationCommandLine *command_line,
		  void                    *user_data)
{
  GError *error = NULL;
  int argc;
  char **argv;

  argv = g_application_command_line_get_arguments (command_line, &argc);

  g_option_context_parse (context, &argc, &argv, &error);

  /* Don't create another window if we're remote.
   * We can't use g_application_get_is_remote() because it's not registered yet */
  if (startup_called != FALSE)
    {
      app_init (GTK_APPLICATION (app));
      startup_called = FALSE;
    }
  else
    {
      gtk_window_present_with_time (window, GDK_CURRENT_TIME);
    }

  vinagre_options_process_command_line (GTK_APPLICATION (app), window, &optionstate);

  g_strfreev (argv);
  return 0;
}

int main (int argc, char **argv) {
  GtkApplication *app;
  VinagreProtocol *extension;
  GHashTable *extensions;
  GHashTableIter iter;
  int res;

  g_set_prgname ("vinagre");
  g_type_init ();
  g_set_application_name (_("Remote Desktop Viewer"));
  optionstate.new_window = FALSE;

  /* i18n */
  setlocale (LC_ALL, "");
  bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  /* Setup command line options */
  context = g_option_context_new (_("- Remote Desktop Viewer"));
  g_option_context_add_main_entries (context, all_options, GETTEXT_PACKAGE);
  g_option_context_set_translation_domain(context, GETTEXT_PACKAGE);
  g_option_context_add_group (context, gtk_get_option_group (TRUE));

#ifdef ENABLE_INTROSPECTION
  g_option_context_add_group (context, g_irepository_get_option_group ());
#endif

  extensions = vinagre_plugins_engine_get_plugins_by_protocol (vinagre_plugins_engine_get_default ());

  g_hash_table_iter_init (&iter, extensions);
  while (g_hash_table_iter_next (&iter, NULL, (gpointer *)&extension))
    {
      GSList *groups, *l;

      groups = vinagre_protocol_get_context_groups (extension);
      for (l = groups; l; l = l->next)
	g_option_context_add_group (context, (GOptionGroup *)l->data);
      g_slist_free (groups);
    }

  app = gtk_application_new ("org.gnome.vinagre", G_APPLICATION_HANDLES_COMMAND_LINE);
  /* https://bugzilla.gnome.org/show_bug.cgi?id=634990 */
  /* g_application_set_option_context (G_APPLICATION (app), context); */
  g_signal_connect (app,
		    "command-line",
		    G_CALLBACK (app_command_line),
		    NULL);
  g_signal_connect (app,
		    "startup",
		    G_CALLBACK (app_startup),
		    NULL);
  res = g_application_run (G_APPLICATION (app), argc, argv);

  if (res == 0)
    {
      #ifdef HAVE_TELEPATHY
	g_object_unref (vinagre_tubes_manager);
      #endif

      g_object_unref (vinagre_bookmarks_get_default ());
      g_object_unref (vinagre_prefs_get_default ());
      vinagre_cache_prefs_finalize ();

    #ifdef VINAGRE_ENABLE_AVAHI
	g_object_unref (vinagre_mdns_get_default ());
    #endif
    }

  g_object_unref (app);
  g_option_context_free (context);

  return res;
}
/* vim: set ts=8: */
