/*
 * Copyright (C) 2008 Nokia Corporation, all rights reserved.
 * Copyright (C) 2008 Zeeshan Ali (Khattak) <zeeshanak@gnome.org>.
 *
 * Author: Zeeshan Ali (Khattak) <zeeshanak@gnome.org>
 *
 * This file is part of Rygel.
 *
 * Rygel is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Rygel is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "rygel-plugin-loader.h"
#include <gmodule.h>
#include <gio/gio.h>
#include <stdlib.h>
#include <string.h>

typedef RygelPlugin* (*RygelPluginLoaderLoadPluginFunc) (void* user_data);



enum  {
	RYGEL_PLUGIN_LOADER_DUMMY_PROPERTY
};
static void rygel_plugin_loader_load_plugins_from_dir (RygelPluginLoader* self, GFile* dir);
static void rygel_plugin_loader_load_plugin_from_file (RygelPluginLoader* self, const char* file_path);
static gboolean rygel_plugin_loader_is_dir (GFile* file);
static gpointer rygel_plugin_loader_parent_class = NULL;
static int _vala_strcmp0 (const char * str1, const char * str2);



/* Plugin loading functions*/
void rygel_plugin_loader_load_plugins (RygelPluginLoader* self) {
	GFile* dir;
	gboolean _tmp0;
	g_return_if_fail (self != NULL);
	g_assert (g_module_supported ());
	dir = g_file_new_for_path (PLUGIN_DIR);
	_tmp0 = FALSE;
	if (dir != NULL) {
		_tmp0 = rygel_plugin_loader_is_dir (dir);
	} else {
		_tmp0 = FALSE;
	}
	g_assert (_tmp0);
	rygel_plugin_loader_load_plugins_from_dir (self, dir);
	(dir == NULL) ? NULL : (dir = (g_object_unref (dir), NULL));
}


static void rygel_plugin_loader_load_plugins_from_dir (RygelPluginLoader* self, GFile* dir) {
	GError * inner_error;
	GFileEnumerator* enumerator;
	GFileInfo* info;
	g_return_if_fail (self != NULL);
	g_return_if_fail (dir != NULL);
	inner_error = NULL;
	enumerator = NULL;
	{
		char* attributes;
		GFileEnumerator* _tmp0;
		GFileEnumerator* _tmp1;
		attributes = g_strdup (G_FILE_ATTRIBUTE_STANDARD_NAME "," G_FILE_ATTRIBUTE_STANDARD_TYPE "," G_FILE_ATTRIBUTE_STANDARD_CONTENT_TYPE);
		_tmp0 = g_file_enumerate_children (dir, attributes, G_FILE_QUERY_INFO_NONE, NULL, &inner_error);
		if (inner_error != NULL) {
			attributes = (g_free (attributes), NULL);
			goto __catch7_g_error;
			goto __finally7;
		}
		_tmp1 = NULL;
		enumerator = (_tmp1 = _tmp0, (enumerator == NULL) ? NULL : (enumerator = (g_object_unref (enumerator), NULL)), _tmp1);
		attributes = (g_free (attributes), NULL);
	}
	goto __finally7;
	__catch7_g_error:
	{
		GError * error;
		error = inner_error;
		inner_error = NULL;
		{
			char* _tmp2;
			_tmp2 = NULL;
			g_critical ("rygel-plugin-loader.vala:59: Error listing contents of directory '%s': %s\n", _tmp2 = g_file_get_path (dir), error->message);
			_tmp2 = (g_free (_tmp2), NULL);
			(error == NULL) ? NULL : (error = (g_error_free (error), NULL));
			(enumerator == NULL) ? NULL : (enumerator = (g_object_unref (enumerator), NULL));
			return;
		}
	}
	__finally7:
	if (inner_error != NULL) {
		(enumerator == NULL) ? NULL : (enumerator = (g_object_unref (enumerator), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	info = NULL;
	while (TRUE) {
		GFileInfo* _tmp3;
		GFileInfo* _tmp4;
		const char* _tmp5;
		char* file_name;
		char* _tmp6;
		char* _tmp7;
		char* file_path;
		GFile* file;
		GFileType file_type;
		const char* _tmp8;
		char* content_type;
		const char* mime;
		_tmp3 = g_file_enumerator_next_file (enumerator, NULL, &inner_error);
		if (inner_error != NULL) {
			(enumerator == NULL) ? NULL : (enumerator = (g_object_unref (enumerator), NULL));
			(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return;
		}
		_tmp4 = NULL;
		if (!((info = (_tmp4 = _tmp3, (info == NULL) ? NULL : (info = (g_object_unref (info), NULL)), _tmp4)) != NULL)) {
			break;
		}
		_tmp5 = NULL;
		file_name = (_tmp5 = g_file_info_get_name (info), (_tmp5 == NULL) ? NULL : g_strdup (_tmp5));
		_tmp6 = NULL;
		_tmp7 = NULL;
		file_path = (_tmp7 = g_build_filename (_tmp6 = g_file_get_path (dir), file_name, NULL), _tmp6 = (g_free (_tmp6), NULL), _tmp7);
		file = g_file_new_for_path (file_path);
		file_type = g_file_info_get_file_type (info);
		_tmp8 = NULL;
		content_type = (_tmp8 = g_file_info_get_content_type (info), (_tmp8 == NULL) ? NULL : g_strdup (_tmp8));
		mime = g_content_type_get_mime_type (content_type);
		if (file_type == G_FILE_TYPE_DIRECTORY) {
			/* Recurse into directories*/
			rygel_plugin_loader_load_plugins_from_dir (self, file);
		} else {
			if (_vala_strcmp0 (mime, "application/x-sharedlib") == 0) {
				/* Seems like we found a plugin*/
				rygel_plugin_loader_load_plugin_from_file (self, file_path);
			}
		}
		file_name = (g_free (file_name), NULL);
		file_path = (g_free (file_path), NULL);
		(file == NULL) ? NULL : (file = (g_object_unref (file), NULL));
		content_type = (g_free (content_type), NULL);
	}
	(enumerator == NULL) ? NULL : (enumerator = (g_object_unref (enumerator), NULL));
	(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
}


static void rygel_plugin_loader_load_plugin_from_file (RygelPluginLoader* self, const char* file_path) {
	GModule* module;
	void* function;
	RygelPluginLoaderLoadPluginFunc _tmp0;
	void* load_plugin_target;
	RygelPluginLoaderLoadPluginFunc load_plugin;
	RygelPlugin* plugin;
	g_return_if_fail (self != NULL);
	g_return_if_fail (file_path != NULL);
	module = g_module_open (file_path, G_MODULE_BIND_LOCAL);
	if (module == NULL) {
		g_debug ("rygel-plugin-loader.vala:90: Failed to load plugin from path: '%s'\n", file_path);
		(module == NULL) ? NULL : (module = (g_module_close (module), NULL));
		return;
	}
	function = NULL;
	g_module_symbol (module, "load_plugin", &function);
	load_plugin = (_tmp0 = (RygelPluginLoaderLoadPluginFunc) function, load_plugin_target = NULL, _tmp0);
	if (load_plugin == NULL) {
		g_warning ("rygel-plugin-loader.vala:101: Failed to load plugin from path: '%s'\n", file_path);
		(module == NULL) ? NULL : (module = (g_module_close (module), NULL));
		return;
	}
	g_debug ("rygel-plugin-loader.vala:106: Loaded plugin: '%s'\n", g_module_name (module));
	plugin = load_plugin (load_plugin_target);
	if (plugin != NULL) {
		/* We don't want our modules to ever unload*/
		g_module_make_resident (module);
		g_signal_emit_by_name (self, "plugin-available", plugin);
	}
	(module == NULL) ? NULL : (module = (g_module_close (module), NULL));
	(plugin == NULL) ? NULL : (plugin = (g_object_unref (plugin), NULL));
}


static gboolean rygel_plugin_loader_is_dir (GFile* file) {
	GError * inner_error;
	GFileInfo* file_info;
	gboolean _tmp4;
	g_return_val_if_fail (file != NULL, FALSE);
	inner_error = NULL;
	file_info = NULL;
	{
		GFileInfo* _tmp0;
		GFileInfo* _tmp1;
		_tmp0 = g_file_query_info (file, G_FILE_ATTRIBUTE_STANDARD_TYPE, G_FILE_QUERY_INFO_NONE, NULL, &inner_error);
		if (inner_error != NULL) {
			goto __catch8_g_error;
			goto __finally8;
		}
		_tmp1 = NULL;
		file_info = (_tmp1 = _tmp0, (file_info == NULL) ? NULL : (file_info = (g_object_unref (file_info), NULL)), _tmp1);
	}
	goto __finally8;
	__catch8_g_error:
	{
		GError * error;
		error = inner_error;
		inner_error = NULL;
		{
			char* _tmp2;
			gboolean _tmp3;
			_tmp2 = NULL;
			g_critical ("rygel-plugin-loader.vala:125: Failed to query content type for '%s'\n", _tmp2 = g_file_get_path (file));
			_tmp2 = (g_free (_tmp2), NULL);
			return (_tmp3 = FALSE, (error == NULL) ? NULL : (error = (g_error_free (error), NULL)), (file_info == NULL) ? NULL : (file_info = (g_object_unref (file_info), NULL)), _tmp3);
		}
	}
	__finally8:
	if (inner_error != NULL) {
		(file_info == NULL) ? NULL : (file_info = (g_object_unref (file_info), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return FALSE;
	}
	return (_tmp4 = g_file_info_get_file_type (file_info) == G_FILE_TYPE_DIRECTORY, (file_info == NULL) ? NULL : (file_info = (g_object_unref (file_info), NULL)), _tmp4);
}


/**
 * Responsible for plugin loading. Probes for shared library files in a specific
 * directry and tries to grab a function with a specific name and signature,
 * calls it and expects a Plugin instance in return.
 */
RygelPluginLoader* rygel_plugin_loader_construct (GType object_type) {
	RygelPluginLoader * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


RygelPluginLoader* rygel_plugin_loader_new (void) {
	return rygel_plugin_loader_construct (RYGEL_TYPE_PLUGIN_LOADER);
}


static void rygel_plugin_loader_class_init (RygelPluginLoaderClass * klass) {
	rygel_plugin_loader_parent_class = g_type_class_peek_parent (klass);
	g_signal_new ("plugin_available", RYGEL_TYPE_PLUGIN_LOADER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_marshal_VOID__OBJECT, G_TYPE_NONE, 1, RYGEL_TYPE_PLUGIN);
}


static void rygel_plugin_loader_instance_init (RygelPluginLoader * self) {
}


GType rygel_plugin_loader_get_type (void) {
	static GType rygel_plugin_loader_type_id = 0;
	if (rygel_plugin_loader_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (RygelPluginLoaderClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) rygel_plugin_loader_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (RygelPluginLoader), 0, (GInstanceInitFunc) rygel_plugin_loader_instance_init, NULL };
		rygel_plugin_loader_type_id = g_type_register_static (G_TYPE_OBJECT, "RygelPluginLoader", &g_define_type_info, 0);
	}
	return rygel_plugin_loader_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




