# -*- coding: utf-8 -*-

# Copyright (C) 2004-2006 Johan Svedberg <johan@svedberg.com>

# This file is part of ontv.

# ontv is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# ontv is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with ontv; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import datetime

class TimeSpan:
    """A TimeSpan represents a span of time, for example the running time of a
    specific TV program."""

    def __init__(self, start, stop):
        """Create a TimeSpan from a start time and an stop time."""
        self.start = start
        self.stop = stop

    def spans_time(self, time):
        """Check if this TimeSpan includes a specific time."""
        return time > self.start and time < self.stop

    def spans_now(self):
        """Check if this TimeSpan includes the current time."""
        return self.spans_time(datetime.datetime.now())

    def duration(self):
        """Calculate the distance (TimeDelta) from the start of this TimeSpan,
        to the stop of this TimeSpan."""
        return self.stop - self.start

    def time_until_start_from(self, base):
        """Calculate the distance (TimeDelta) from a specific time, to the
        start of this TimeSpan."""
        return self.start - base

    def time_until_start(self):
        """Calculate the distance (TimeDelta) from the current time, to the
        start of this TimeSpan."""
        return self.time_until_start_from(datetime.datetime.now())

    def time_from_start_from(self, base):
        """Calculate the distance (TimeDelta) from the start of this TimeSpan,
        to a specific time."""
        return base - self.start

    def time_from_start(self):
        """Calculate the distance (TimeDelta) from the start of this TimeSpan,
        to the current time."""
        return self.time_from_start_from(datetime.datetime.now())

    def time_until_stop_from(self, base):
        """Calculate the distance (TimeDelta) from a specific time, to the stop
        of this TimeSpan."""
        return self.stop - base

    def time_until_stop(self):
        """Calculate the distance (TimeDelta) from the current time, to the
        stop of this TimeSpan."""
        return self.time_until_stop_from(datetime.datetime.now())

# vim: set sw=4 et sts=4 tw=79 fo+=l:
