/* gnome-db-qfield.c
 *
 * Copyright (C) 2003 - 2005  Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-qfield.h"
#include "gnome-db-query.h"
#include "gnome-db-field.h"
#include "gnome-db-qf-all.h"
#include "gnome-db-qf-field.h"
#include "gnome-db-qf-value.h"
#include "gnome-db-qf-func.h"
#include "gnome-db-qf-agg.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-ref-base.h"
#include "gnome-db-server.h"

#include "gnome-db-query-parsing.h"

/* 
 * Main static functions 
 */
static void gnome_db_qfield_class_init (GnomeDbQfieldClass *class);
static void gnome_db_qfield_init (GnomeDbQfield *qfield);
static void gnome_db_qfield_dispose (GObject *object);
static void gnome_db_qfield_finalize (GObject *object);

static void gnome_db_qfield_set_property    (GObject              *object,
				       guint                 param_id,
				       const GValue         *value,
				       GParamSpec           *pspec);
static void gnome_db_qfield_get_property    (GObject              *object,
				       guint                 param_id,
				       GValue               *value,
				       GParamSpec           *pspec);

static void attach_to_query (GnomeDbQfield *qfield, GnomeDbQuery *query);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	LAST_SIGNAL
};

static gint gnome_db_qfield_signals[LAST_SIGNAL] = { };

/* properties */
enum
{
	PROP_0,
	PROP
};


struct _GnomeDbQfieldPrivate
{
	gchar     *alias;
	gboolean   visible;
	gboolean   internal;
};

/* module error */
GQuark gnome_db_qfield_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_qfield_error");
	return quark;
}


guint
gnome_db_qfield_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbQfieldClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_qfield_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbQfield),
			0,
			(GInstanceInitFunc) gnome_db_qfield_init
		};
		
		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbQfield", &info, 0);
	}
	return type;
}

static void
gnome_db_qfield_class_init (GnomeDbQfieldClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);



	/* virtual functions */
	class->copy = NULL;
	class->is_equal = NULL;

	object_class->dispose = gnome_db_qfield_dispose;
	object_class->finalize = gnome_db_qfield_finalize;

	/* Properties */
	object_class->set_property = gnome_db_qfield_set_property;
	object_class->get_property = gnome_db_qfield_get_property;
	g_object_class_install_property (object_class, PROP,
					 g_param_spec_pointer ("prop", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

static void
gnome_db_qfield_init (GnomeDbQfield *qfield)
{
	qfield->priv = g_new0 (GnomeDbQfieldPrivate, 1);
	qfield->priv->alias = NULL;
	qfield->priv->visible = TRUE;
	qfield->priv->internal = FALSE;
}

/**
 * gnome_db_qfield_new_from_xml
 * @query: a #GnomeDbQuery object
 * @node: an XML node corresponding to a GNOME_DB_QFIELD tag
 * @error: location to store error, or %NULL
 *
 * This is an object factory which does create instances of class inheritants of the #GnomeDbDfield class.
 * Ths #GnomeDbQfield object MUST then be attached to @query
 * 
 * Returns: the newly created object
 */
GObject   *
gnome_db_qfield_new_from_xml (GnomeDbQuery *query, xmlNodePtr node, GError **error)
{
	GObject *obj = NULL;
	gchar *prop;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (node, NULL);
	g_return_val_if_fail (!strcmp (node->name, "GNOME_DB_QF"), NULL);

	prop = xmlGetProp (node, "type");
	if (prop) {
		gboolean done = FALSE;
		if (!strcmp (prop, "ALL")) {
			gchar *target;

			done = TRUE;
			target = xmlGetProp (node, "target");
			if (target) {
				obj = gnome_db_qf_all_new_with_xml_id (query, target);
				g_free (target);
			}
			else {
				g_set_error (error,
					     GNOME_DB_QF_ALL_ERROR,
					     GNOME_DB_QF_ALL_XML_LOAD_ERROR,
					     _("Missing 'target' attribute in <GNOME_DB_QF>"));
				return NULL;
			}
		}

		if (!done && !strcmp (prop, "FIELD")) {
			gchar *target, *field;

			done = TRUE;
			target = xmlGetProp (node, "target");
			field = xmlGetProp (node, "object");
			if (target && field) 
				obj = gnome_db_qf_field_new_with_xml_ids (query, target, field);

			if (target)
				g_free (target);
			if (field)
				g_free (field);

			if (!obj) {
				g_set_error (error,
					     GNOME_DB_QF_ALL_ERROR,
					     GNOME_DB_QF_ALL_XML_LOAD_ERROR,
					     _("Missing 'target' attribute in <GNOME_DB_QF>"));
				return NULL;
			}

		}

		if (!done && !strcmp (prop, "AGG")) {
			gchar *agg;

			agg = xmlGetProp (node, "object");
			if (agg) {
				obj = gnome_db_qf_agg_new_with_xml_id (query, agg);
				g_free (agg);
			}
			
			if (!obj) {
				g_set_error (error,
					     GNOME_DB_QF_ALL_ERROR,
					     GNOME_DB_QF_ALL_XML_LOAD_ERROR,
					     _("Missing 'object' attribute in <GNOME_DB_QF>"));
				return NULL;
			}
		}

		if (!done && !strcmp (prop, "FUNC")) {
			gchar *func;

			func = xmlGetProp (node, "object");
			if (func) {
				obj = gnome_db_qf_func_new_with_xml_id (query, func);
				g_free (func);
			}
			
			if (!obj) {
				g_set_error (error,
					     GNOME_DB_QF_ALL_ERROR,
					     GNOME_DB_QF_ALL_XML_LOAD_ERROR,
					     _("Missing 'object' attribute in <GNOME_DB_QF>"));
				return NULL;
			}
		}

		if (!done && !strcmp (prop, "VAL")) {
			gchar *srvt;

			done = TRUE;
			srvt = xmlGetProp (node, "srv_type");
			if (srvt) {
				GnomeDbServerDataType *dt;

				dt = gnome_db_server_get_data_type_by_xml_id (gnome_db_dict_get_server (gnome_db_base_get_dict (GNOME_DB_BASE (query))),
									srvt);
				if (dt)
					obj = gnome_db_qf_value_new (query, dt);
				else {
					g_set_error (error,
						     GNOME_DB_QF_ALL_ERROR,
						     GNOME_DB_QF_ALL_XML_LOAD_ERROR,
						     _("Can't find data type %s for query field"), srvt);
					return NULL;
				}
				g_free (srvt);
			}
			else {
				g_set_error (error,
					     GNOME_DB_QF_ALL_ERROR,
					     GNOME_DB_QF_ALL_XML_LOAD_ERROR,
					     _("Missing 'srv_type' attribute for VALUE query field"));
				return NULL;
			}
					
		}

		g_free (prop);

		if (obj) {
			attach_to_query (GNOME_DB_QFIELD (obj), query);
			if (!gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (obj), node, error)) {
				g_object_unref (obj);
				return NULL;
			}
		}
		else 
			g_set_error (error,
				     GNOME_DB_QF_ALL_ERROR,
				     GNOME_DB_QF_ALL_XML_LOAD_ERROR,
				     _("Missing Implementation in loading <GNOME_DB_QF>"));
	}
	else {
		g_set_error (error,
			     GNOME_DB_QFIELD_ERROR,
			     GNOME_DB_QFIELD_XML_LOAD_ERROR,
			     _("Unknown value for 'type' attribute in <GNOME_DB_QF>"));
		return NULL;
	}

	return obj;
}


/**
 * gnome_db_qfield_new_copy
 * @orig: a #GnomeDbQfield to copy
 *
 * This is a copy constructor
 *
 * Returns: the new object
 */
GObject *
gnome_db_qfield_new_copy (GnomeDbQfield *orig)
{
	GnomeDbQfieldClass *class;
	GObject *obj;
	GnomeDbQuery *query;
	GnomeDbQfield *newfield;

	g_return_val_if_fail (orig && IS_GNOME_DB_QFIELD (orig), NULL);
	g_return_val_if_fail (orig->priv, NULL);
	g_object_get (G_OBJECT (orig), "query", &query, NULL);
	g_return_val_if_fail (query, NULL);

	class = GNOME_DB_QFIELD_CLASS (G_OBJECT_GET_CLASS (orig));
	g_return_val_if_fail (class->copy, NULL);

	obj = (class->copy) (orig);
	newfield = GNOME_DB_QFIELD (obj);
	newfield->priv->visible = orig->priv->visible;
	newfield->priv->internal = orig->priv->internal;

	attach_to_query (GNOME_DB_QFIELD (obj), query);

	return obj;
}

/**
 * gnome_db_qfield_new_from_sql
 * @query: a #GnomeDbQuery object
 * @sqlfield: a SQL statement representing a query field
 * @error: location to store error, or %NULL
 *
 * Creates a new #GnomeDbQfield from its SQL representation
 *
 * Returns: a new #GnomeDbQfield object, or %NULL if an error occured
 */
GObject *
gnome_db_qfield_new_from_sql (GnomeDbQuery *query, const gchar *sqlfield, GError **error)
{
	gchar *sql;
	sql_statement *result;
	GnomeDbQfield *qfield = NULL;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (sqlfield && *sqlfield, NULL);

	sql = g_strconcat ("SELECT ", sqlfield, NULL);
	result = sql_parse_with_error (sql, error);
	if (result) {
		GList *list = ((sql_select_statement *) result->statement)->fields;
		if (list) {
			if (g_slist_next (list))
				g_set_error (error, GNOME_DB_QUERY_ERROR, GNOME_DB_QUERY_PARSE_ERROR,
					     _("More than one field to parse in '%s'"), sqlfield);
			else {
				ParseData *pdata;
				
				pdata = parse_data_new (query);
				parse_data_compute_targets_hash (query, pdata);
				qfield = (GnomeDbQfield *) parsed_create_global_query_field (query, FALSE, pdata, 
											(sql_field *) list->data,
											FALSE, NULL, NULL, error);
				parse_data_destroy (pdata);
			}
		}
		else 
			g_set_error (error, GNOME_DB_QUERY_ERROR, GNOME_DB_QUERY_PARSE_ERROR,
				     _("No field to parse in '%s'"), sqlfield);
		sql_destroy (result);
	}
	else {
		if (error && !(*error))
			g_set_error (error, GNOME_DB_QUERY_ERROR, GNOME_DB_QUERY_PARSE_ERROR,
				     _("Error parsing '%s'"), sqlfield);
	}
	g_free (sql);
	
	return G_OBJECT (qfield);
}

static void
attach_to_query (GnomeDbQfield *qfield, GnomeDbQuery *query)
{
	g_object_set (G_OBJECT (qfield), "query", query, NULL);
}


static void
gnome_db_qfield_dispose (GObject *object)
{
	GnomeDbQfield *qfield;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_QFIELD (object));

	qfield = GNOME_DB_QFIELD (object);
	if (qfield->priv) {
		if (qfield->priv->alias) {
			g_free (qfield->priv->alias);
			qfield->priv->alias = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_qfield_finalize (GObject   * object)
{
	GnomeDbQfield *qfield;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_QFIELD (object));

	qfield = GNOME_DB_QFIELD (object);
	if (qfield->priv) {
		g_free (qfield->priv);
		qfield->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_qfield_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	GnomeDbQfield *qfield;

	qfield = GNOME_DB_QFIELD (object);
	if (qfield->priv) {
		switch (param_id) {
		case PROP:
			break;
		}
	}
}

static void
gnome_db_qfield_get_property (GObject              *object,
			guint                 param_id,
			GValue               *value,
			GParamSpec           *pspec)
{
	GnomeDbQfield *qfield;

	qfield = GNOME_DB_QFIELD (object);
        if (qfield->priv) {
                switch (param_id) {
                case PROP:
                        break;
                }
        }
}


/**
 * gnome_db_qfield_set_alias
 * @qfield: a #GnomeDbQfield object
 * @alias: the alias to set @qfield to
 *
 * Sets @qfield's alias
 */
void
gnome_db_qfield_set_alias (GnomeDbQfield *qfield, const gchar *alias)
{
	g_return_if_fail (qfield && IS_GNOME_DB_QFIELD (qfield));
	g_return_if_fail (qfield->priv);

	if (qfield->priv->alias) {
		g_free (qfield->priv->alias);
		qfield->priv->alias = NULL;
	}
	
	if (alias)
		qfield->priv->alias = g_strdup (alias);
	gnome_db_base_changed (GNOME_DB_BASE (qfield));
}

/**
 * gnome_db_qfield_get_alias
 * @qfield: a #GnomeDbQfield object
 *
 * Get @qfield's alias
 *
 * Returns: the alias
 */
const gchar *
gnome_db_qfield_get_alias (GnomeDbQfield *qfield)
{
	g_return_val_if_fail (qfield && IS_GNOME_DB_QFIELD (qfield), NULL);
	g_return_val_if_fail (qfield->priv, NULL);

	return qfield->priv->alias;
}


/**
 * gnome_db_qfield_set_visible
 * @qfield: a #GnomeDbQfield object
 * @visible:
 *
 * Sets the visibility of @qfield. A visible field will appear in the query's 
 * corresponding (virtual) entity, whereas a non visible one will be hidden (and
 * possibly not taking part in the query).
 */
void
gnome_db_qfield_set_visible (GnomeDbQfield *qfield, gboolean visible)
{
	GnomeDbQuery *query;

	g_return_if_fail (qfield && IS_GNOME_DB_QFIELD (qfield));
	g_return_if_fail (qfield->priv);
	g_object_get (G_OBJECT (qfield), "query", &query, NULL);
	g_return_if_fail (query);

	if (qfield->priv->visible != visible) {
		qfield->priv->visible = visible;
		if (visible)
			g_signal_emit_by_name (G_OBJECT (query), "field_added", GNOME_DB_FIELD (qfield));
		else
			g_signal_emit_by_name (G_OBJECT (query), "field_removed", GNOME_DB_FIELD (qfield));
		gnome_db_base_changed (GNOME_DB_BASE (query));
	}
}


/**
 * gnome_db_qfield_is_visible
 * @qfield: a #GnomeDbQfield object
 *
 * Returns: TRUE if @field is visible
 */
gboolean
gnome_db_qfield_is_visible (GnomeDbQfield *qfield)
{
	g_return_val_if_fail (qfield && IS_GNOME_DB_QFIELD (qfield), FALSE);
	g_return_val_if_fail (qfield->priv, FALSE);

	return qfield->priv->visible;
}

/**
 * gnome_db_qfield_set_internal
 * @qfield: a #GnomeDbQfield object
 * @internal:
 *
 * Sets weather @qfield is internal or not. Internal fields in a query are fields added
 * or changed by libgnomedb itself, such fields may or may not be visible.
 */
void
gnome_db_qfield_set_internal (GnomeDbQfield *qfield, gboolean internal)
{
	g_return_if_fail (qfield && IS_GNOME_DB_QFIELD (qfield));
	g_return_if_fail (qfield->priv);

	qfield->priv->internal = internal;
}


/**
 * gnome_db_qfield_is_internal
 * @qfield: a #GnomeDbQfield object
 *
 * Returns: TRUE if @field is internal
 */
gboolean
gnome_db_qfield_is_internal (GnomeDbQfield *qfield)
{
	g_return_val_if_fail (qfield && IS_GNOME_DB_QFIELD (qfield), FALSE);
	g_return_val_if_fail (qfield->priv, FALSE);

	return qfield->priv->internal;
}

/**
 * gnome_db_qfield_get_data_type
 * @qfield: a #GnomeDbQfield object
 *
 * Get the #GnomeDbServerDataType represented by the @qfield object: for a function it returns
 * the return type, for a value, it returns its type, etc.
 *
 * Returns: the data type, or %NULL if @qfield does not have a data type.
 */
GnomeDbServerDataType *
gnome_db_qfield_get_data_type (GnomeDbQfield *qfield)
{
	g_return_val_if_fail (qfield && IS_GNOME_DB_QFIELD (qfield), NULL);
	g_return_val_if_fail (qfield->priv, NULL);

	/* it is assumed that qfield really implements the GnomeDbField interface */
	return gnome_db_field_get_data_type (GNOME_DB_FIELD (qfield));
}

/**
 * gnome_db_qfield_get_parameters
 * @qfield: a #GnomeDbQfield object
 *
 * Get a list of all the parameters needed to @qfield to be
 * rendered as a valid statement
 *
 * Returns: a new list of parameters for @qfield
 */
GSList *
gnome_db_qfield_get_parameters (GnomeDbQfield *qfield)
{
	GnomeDbQfieldClass *class;

	g_return_val_if_fail (qfield && IS_GNOME_DB_QFIELD (qfield), NULL);
	g_return_val_if_fail (qfield->priv, NULL);
	class = GNOME_DB_QFIELD_CLASS (G_OBJECT_GET_CLASS (qfield));

	if (class->get_params)
		return (class->get_params) (qfield);
	else
		return NULL;
}

/**
 * gnome_db_qfield_is_equal
 * @qfield1: a #GnomeDbQfield object
 * @qfield2: a #GnomeDbQfield object
 *
 * Compares the @qfield1 and @qfield2. The name and aliases of the two fields are
 * not compared, only the contents of the fields are.
 *
 * Returns: TRUE if they are equal and FALSE otherwise
 */
gboolean
gnome_db_qfield_is_equal (GnomeDbQfield *qfield1, GnomeDbQfield *qfield2)
{
	GnomeDbQfieldClass *class1, *class2;
	GnomeDbQuery *q1, *q2;

	g_return_val_if_fail (qfield1 && IS_GNOME_DB_QFIELD (qfield1), FALSE);
	g_return_val_if_fail (qfield2 && IS_GNOME_DB_QFIELD (qfield2), FALSE);
	g_return_val_if_fail (qfield1->priv, FALSE);
	g_return_val_if_fail (qfield2->priv, FALSE);
	
	if (qfield1 == qfield2)
		return TRUE;

	g_object_get (G_OBJECT (qfield1), "query", &q1, NULL);
	g_object_get (G_OBJECT (qfield2), "query", &q2, NULL);
	g_return_val_if_fail (q1, FALSE);
	g_return_val_if_fail (q2, FALSE);

	if (q1 != q2)
		return FALSE;

	class1 = GNOME_DB_QFIELD_CLASS (G_OBJECT_GET_CLASS (qfield1));
	class2 = GNOME_DB_QFIELD_CLASS (G_OBJECT_GET_CLASS (qfield2));
	if (class1 != class2)
		return FALSE;

	g_return_val_if_fail (class1->is_equal, FALSE);

	return (class1->is_equal) (qfield1, qfield2);
}


/**
 * gnome_db_qfield_is_list
 * @qfield: a #GnomeDbQfield object
 *
 * Tells if @qfield can potentially represent a list of values.
 *
 * Returns: TRUE if @field can be a list of values
 */
gboolean
gnome_db_qfield_is_list (GnomeDbQfield *qfield)
{
	GnomeDbQfieldClass *class;

	g_return_val_if_fail (qfield && IS_GNOME_DB_QFIELD (qfield), FALSE);
	g_return_val_if_fail (qfield->priv, FALSE);

	class = GNOME_DB_QFIELD_CLASS (G_OBJECT_GET_CLASS (qfield));
	if (class->is_list)
		return (class->is_list) (qfield);
	else
		return FALSE;
}

