/* GNOME-DB Components
 * Copyright (C) 2000 - 2005 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *      Vivien Malerba <malerba@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib/gi18n-lib.h>
#include <string.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkstock.h>
#include <libgnomedb/libgnomedb.h>
#include <libgnomeui/gnome-client.h>
#include <libgnomeui/gnome-window-icon.h>
#include "database-capplet.h"

GtkWindow *main_window;


static void
druid_finished_cb (GnomeDbDsnConfigDruid *druid, gboolean error, gpointer user_data)
{
	const GdaDataSourceInfo *dsn_info;

	if (!error) {
		dsn_info = gnome_db_dsn_config_druid_get_dsn (druid);
		if (dsn_info)
			gda_config_save_data_source_info (dsn_info);
		else
			gnome_db_show_error (_("No valid data source info was created"));
	}

	gtk_widget_destroy (GTK_WIDGET (user_data));
}

static void
file_new_cb (GtkAction *action, gpointer user_data)
{
	GtkWidget *druid, *dialog;
	GdkPixbuf *icon;

	/* create the window to contain the druid */
	dialog = gtk_dialog_new_with_buttons (_("New data source"), GTK_WINDOW (main_window),
					      0, NULL);

	icon = gdk_pixbuf_new_from_file (LIBGNOMEDB_ICONSDIR "/gnome-db.png", NULL);
	if (icon) {
		gtk_window_set_icon (GTK_WINDOW (dialog), icon);
		g_object_unref (icon);
	}
	gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);

	druid = gnome_db_dsn_config_druid_new ();
	g_signal_connect (G_OBJECT (druid), "finished",
			  G_CALLBACK (druid_finished_cb), dialog);
	gtk_widget_show (druid);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), druid, TRUE, TRUE, 0);

	/* run the dialog */
	gtk_widget_show (dialog);
}

static void
file_properties_cb (GtkAction *action, gpointer user_data)
{
	database_capplet_edit_properties (GTK_WIDGET (user_data));
}

static void
window_closed_cb (GtkAction *action, gpointer user_data)
{
	gnome_db_main_quit ();
}

static void
edit_delete_cb (GtkAction *action, gpointer user_data)
{
	database_capplet_delete (GTK_WIDGET (user_data));
}

static void
edit_select_all_cb (GtkAction *action, gpointer user_data)
{
	database_capplet_select_all (GTK_WIDGET (user_data));
}

static void
edit_deselect_all_cb (GtkAction *action, gpointer user_data)
{
	database_capplet_deselect_all (GTK_WIDGET (user_data));
}

static void
view_data_sources_cb (GtkAction *action, gpointer user_data)
{
	database_capplet_switch (GTK_WIDGET (user_data), DSN_PAGE);
}

static void
view_providers_cb (GtkAction *action, gpointer user_data)
{
	database_capplet_switch (GTK_WIDGET (user_data), PROVIDER_PAGE);
}

static void
about_cb (GtkAction *action, gpointer user_data)
{
	GdkPixbuf *icon;
	GtkWidget *dialog;
	const gchar *authors[] = {
		"Rodrigo Moya <rodrigo@gnome-db.org>",
		"Vivien Malerba <malerba@gnome-db.org>",
		"Carlos Perello Marin <carlos@gnome-db.org>",
		"Gonzalo Paniagua Javier <gonzalo@gnome-db.org>",
		"Laurent Sansonetti <lrz@gnome.org>",
		NULL
	};
	const gchar *documenters[] = {
		"Rodrigo Moya <rodrigo@gnome-db.org>",
		NULL
	};
	const gchar *translator_credits =
		"Christian Rose <menthos@menthos.com> Swedish translations\n" \
		"Kjartan Maraas <kmaraas@online.no> Norwegian translation\n";

	icon = gdk_pixbuf_new_from_file (LIBGNOMEDB_ICONSDIR "/gnome-db.png", NULL);

	dialog = gtk_about_dialog_new ();
	gtk_about_dialog_set_name (GTK_ABOUT_DIALOG (dialog), _("Database Access Properties"));
	gtk_about_dialog_set_version (GTK_ABOUT_DIALOG (dialog), PACKAGE_VERSION);
	gtk_about_dialog_set_copyright (GTK_ABOUT_DIALOG (dialog), "(C) 1998-2005 GNOME Foundation");
	gtk_about_dialog_set_comments (GTK_ABOUT_DIALOG (dialog), _("Database services for the GNOME Desktop"));
	gtk_about_dialog_set_license (GTK_ABOUT_DIALOG (dialog), "GNU Lesser General Public License");
	gtk_about_dialog_set_website (GTK_ABOUT_DIALOG (dialog), "http://www.gnome-db.org");
	gtk_about_dialog_set_authors (GTK_ABOUT_DIALOG (dialog), authors);
	gtk_about_dialog_set_documenters (GTK_ABOUT_DIALOG (dialog), documenters);
	gtk_about_dialog_set_translator_credits (GTK_ABOUT_DIALOG (dialog), translator_credits);
	gtk_about_dialog_set_logo (GTK_ABOUT_DIALOG (dialog), icon);
	g_signal_connect (G_OBJECT (dialog), "destroy",
			  G_CALLBACK (gtk_widget_destroyed),
			  &dialog);
	gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (main_window));
	gtk_widget_show (dialog);

}

static GtkActionEntry ui_actions[] = {
	{ "Database", NULL, "_Data source", NULL, "Database", NULL },
	{ "DatabaseClose", GTK_STOCK_CLOSE, "_Close", NULL, "Close this window", G_CALLBACK (window_closed_cb) },
	{ "DatabaseNew", GTK_STOCK_NEW, "_New", NULL, "Create new entry", G_CALLBACK (file_new_cb) },
	/* { "DatabaseSaveConnection", GTK_STOCK_SAVE, "_Save", NULL, "Save connection parameters to file", G_CALLBACK () }, */
	{ "DatabaseProperties", GTK_STOCK_PROPERTIES, "_Properties", NULL, "Edit properties for selected item", G_CALLBACK (file_properties_cb) },
	{ "Edit", NULL, "_Edit", NULL, "Edit", NULL },
	{ "EditDelete", GTK_STOCK_DELETE, "_Delete", NULL, "Delete selected items", G_CALLBACK (edit_delete_cb) },
	{ "EditSelectAll", NULL, "_Select all", NULL, "Select all items", G_CALLBACK (edit_select_all_cb) },
	{ "EditClearSelection", NULL, "_Clear selection", NULL, "Deselect all items", G_CALLBACK (edit_deselect_all_cb) },
	{ "ViewDataSources", NULL, "_DataSources", NULL, "View data sources", G_CALLBACK (view_data_sources_cb) },
	{ "ViewDataProviders", NULL, "_Providers", NULL, "View providers", G_CALLBACK (view_providers_cb) },
	{ "About", NULL, "_About", NULL, "About", NULL },
	{ "HelpAbout", GTK_STOCK_ABOUT, "_About", NULL, "About GNOME-DB", G_CALLBACK (about_cb) }
};

static const gchar *ui_actions_info =
        "<ui>"
	"  <menubar name='MenuBar'>"
	"    <menu name='Database' action='Database'>"
	"      <menuitem name='DatabaseNew' action= 'DatabaseNew'/>"
	"      <menuitem name='DatabaseProperties' action= 'DatabaseProperties'/>"
	"      <separator/>"
	"      <menuitem name='DatabaseClose' action= 'DatabaseClose'/>"
	"    </menu>"
	"    <menu name='Edit' action='Edit'>"
	"      <menuitem name='EditDelete' action= 'EditDelete'/>"
	"      <menuitem name='EditSelectAll' action= 'EditSelectAll'/>"
	"      <menuitem name='EditClearSelection' action= 'EditClearSelection'/>"
	"    </menu>"
	"    <menu name='About' action='About'>"
	"      <menuitem name='HelpAbout' action= 'HelpAbout'/>"
	"    </menu>"
	"  </menubar>"
        "  <toolbar  name='ToolBar'>"
        "    <toolitem action='DatabaseNew'/>"
        "    <toolitem action='DatabaseClose'/>"
        "    <toolitem action='DatabaseProperties'/>"
        "  </toolbar>"
        "</ui>";

static void
prepare_menu (GtkBox *vbox, GtkWidget *capplet)
{
	GtkActionGroup *actions;
        GtkUIManager *ui;
        GtkWidget *menubar;

        actions = gtk_action_group_new ("Actions");
        gtk_action_group_add_actions (actions, ui_actions, G_N_ELEMENTS (ui_actions), capplet);

        ui = gtk_ui_manager_new ();
        gtk_ui_manager_insert_action_group (ui, actions, 0);
        gtk_ui_manager_add_ui_from_string (ui, ui_actions_info, -1, NULL);

        menubar = gtk_ui_manager_get_widget (ui, "/MenuBar");
        gtk_box_pack_start (vbox, menubar, FALSE, FALSE, 0);
	gtk_widget_show (menubar);
}


static void
create_main_window (gpointer user_data)
{
	GtkWidget *window, *vbox;
	GtkWidget *capplet;
	GdkPixbuf *icon;

	/* create the main window */
	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	main_window = GTK_WINDOW (window);
	gtk_window_set_title (GTK_WINDOW (window), _("Database Access Properties"));
	gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
	gtk_window_set_default_size (GTK_WINDOW (window), 400, 500);
	g_signal_connect (G_OBJECT (window), "destroy",
			  G_CALLBACK (window_closed_cb), NULL);

	/* icon */
	icon = gdk_pixbuf_new_from_file (LIBGNOMEDB_ICONSDIR "/gnome-db.png", NULL);
	if (icon) {
		gtk_window_set_icon (GTK_WINDOW (window), icon);
		g_object_unref (icon);
	}

	/* menu and contents */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (window), vbox);
	gtk_widget_show (vbox);

	capplet = database_capplet_new ();	
	prepare_menu (GTK_BOX (vbox), capplet);

        gtk_container_set_border_width (GTK_CONTAINER (capplet), 6);
	gtk_box_pack_start (GTK_BOX (vbox), capplet, TRUE, TRUE, 0);
	gtk_widget_show (capplet);

	gtk_widget_show (window);
}

int
main (int argc, char *argv[])
{
	GnomeClient *session_client;

	bindtextdomain (GETTEXT_PACKAGE, LIBGNOMEDB_LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	gnome_db_init ("gnome-database-properties", PACKAGE_VERSION, argc, argv);

	/* session management */
	session_client = gnome_master_client ();
	gnome_client_set_restart_style (session_client, GNOME_RESTART_NEVER);

	/* application loop */
	gnome_db_main_run ((GdaInitFunc) create_main_window, NULL);

	return 0;
}
