/* gnome-db-server-info.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-server.h"
#include "gnome-db-server-info.h"

static GnomeDbServerInfo *none_info_build ();
static GnomeDbServerInfo *postgres_info_build ();
static GnomeDbServerInfo *mysql_info_build ();

GnomeDbServerInfo *
gnome_db_server_info_get (const gchar *provider_id)
{
	if (!provider_id || ! (*provider_id) )
		return none_info_build ();

	if (!strcmp (provider_id, "PostgreSQL")) 
		return postgres_info_build ();

	if (!strcmp (provider_id, "MySQL")) 
		return mysql_info_build ();
}

void
gnome_db_server_info_free (GnomeDbServerInfo *info)
{
	g_free (info->value_test_data_types);
	g_free (info->value_test_gda_types);
	g_free (info);
}

static GnomeDbServerInfo *
postgres_info_build ()
{
	GnomeDbServerInfo *info;

	info = g_new0 (GnomeDbServerInfo, 1);
	info->provider_name = "PostgreSQL";

	info->is_case_insensitive = TRUE;

	info->implicit_data_types_casts = TRUE;
	
	info->value_nb_tests_types = 5;
	info->value_test_data_types = g_new0 (gchar *, 5);
	info->value_test_gda_types = g_new0 (GdaValueType, 5);
	info->value_test_data_types[0] = "int4";
	info->value_test_data_types[1] = "bool";
	info->value_test_data_types[2] = "time";
	info->value_test_data_types[3] = "timestamp";
	info->value_test_data_types[4] = "varchar";
	info->value_test_gda_types[0] = GDA_VALUE_TYPE_INTEGER;
	info->value_test_gda_types[1] = GDA_VALUE_TYPE_BOOLEAN;
	info->value_test_gda_types[2] = GDA_VALUE_TYPE_TIME;
	info->value_test_gda_types[3] = GDA_VALUE_TYPE_TIMESTAMP;
	info->value_test_gda_types[4] = GDA_VALUE_TYPE_STRING;
	
	return info;
}

static GnomeDbServerInfo *
mysql_info_build ()
{
	GnomeDbServerInfo *info;

	info = g_new0 (GnomeDbServerInfo, 1);
	info->provider_name = "MySQL";

	info->is_case_insensitive = TRUE;

	info->implicit_data_types_casts = TRUE;
	
	info->value_nb_tests_types = 5;
	info->value_test_data_types = g_new0 (gchar *, 5);
	info->value_test_gda_types = g_new0 (GdaValueType, 5);
	info->value_test_data_types[0] = "int";
	info->value_test_data_types[1] = "tinyint";
	info->value_test_data_types[2] = "time";
	info->value_test_data_types[3] = "timestamp";
	info->value_test_data_types[4] = "varchar";
	info->value_test_gda_types[0] = GDA_VALUE_TYPE_INTEGER;
	info->value_test_gda_types[1] = GDA_VALUE_TYPE_BOOLEAN;
	info->value_test_gda_types[2] = GDA_VALUE_TYPE_TIME;
	info->value_test_gda_types[3] = GDA_VALUE_TYPE_TIMESTAMP;
	info->value_test_gda_types[4] = GDA_VALUE_TYPE_STRING;
	
	return info;
}


/*
 * Creates a GnomeDbServerInfo structure to be used by a GnomeDbServer when
 * no data source has been set
 */
static
GnomeDbServerInfo *none_info_build ()
{
	GnomeDbServerInfo *info;
	GdaValueType gda_type;

	info = g_new0 (GnomeDbServerInfo, 1);
	info->provider_name = NULL;

	info->is_case_insensitive = TRUE;
	info->implicit_data_types_casts = TRUE;
	
	info->value_nb_tests_types = GDA_VALUE_TYPE_UNKNOWN;
	info->value_test_data_types = g_new0 (gchar *, GDA_VALUE_TYPE_UNKNOWN);
	info->value_test_gda_types = g_new0 (GdaValueType, GDA_VALUE_TYPE_UNKNOWN);

	for (gda_type = GDA_VALUE_TYPE_NULL; gda_type < GDA_VALUE_TYPE_UNKNOWN; gda_type ++) {
		info->value_test_data_types [gda_type] = gda_type_to_string (gda_type);
		info->value_test_gda_types [gda_type] = gda_type;
	}
	
	return info;	
}
