/* gnome-db-dict.c
 *
 * Copyright (C) 2003 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-dict.h"
#include "marshal.h"
#include <libxml/parser.h>
#include <libxml/tree.h>
#include "gnome-db-base.h"
#include "gnome-db-server.h"
#include "gnome-db-database.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-query.h"
#include "gnome-db-referer.h"
#include "gnome-db-entity.h"
#include "gnome-db-table.h"
#include "graph/gnome-db-graph.h"
#include "graph/gnome-db-graph-query.h"
#include "gnome-db-custom-layout.h"

/* 
 * Main static functions 
 */
static void gnome_db_dict_class_init (GnomeDbDictClass * class);
static void gnome_db_dict_init (GnomeDbDict * srv);
static void gnome_db_dict_dispose (GObject   * object);
static void gnome_db_dict_finalize (GObject   * object);

static void gnome_db_dict_set_property (GObject              *object,
				  guint                 param_id,
				  const GValue         *value,
				  GParamSpec           *pspec);
static void gnome_db_dict_get_property (GObject              *object,
				  guint                 param_id,
				  GValue               *value,
				  GParamSpec           *pspec);

static gboolean gnome_db_dict_load_queries (GnomeDbDict *dict, xmlNodePtr queries, GError **error);
static gboolean gnome_db_dict_load_graphs (GnomeDbDict *dict, xmlNodePtr graphs, GError **error);
static gboolean gnome_db_dict_load_custom_layouts (GnomeDbDict *dict, xmlNodePtr layouts, GError **error);

static void query_id_changed_cb (GnomeDbQuery *query, GnomeDbDict *dict);
static void query_nullified_cb (GnomeDbQuery *query, GnomeDbDict *dict);
static void query_weak_ref_notify (GnomeDbDict *dict, GnomeDbQuery *query);
static void updated_query_cb (GnomeDbQuery *query, GnomeDbDict *dict);

static void graph_id_changed_cb (GnomeDbGraph *graph, GnomeDbDict *dict);
static void graph_nullified_cb (GnomeDbGraph *graph, GnomeDbDict *dict);
static void graph_weak_ref_notify (GnomeDbDict *dict, GnomeDbGraph *graph);
static void updated_graph_cb (GnomeDbGraph *graph, GnomeDbDict *dict);

static void layout_id_changed_cb (GnomeDbCustomLayout *layout, GnomeDbDict *dict);
static void layout_nullified_cb (GnomeDbCustomLayout *layout, GnomeDbDict *dict);
static void layout_weak_ref_notify (GnomeDbDict *dict, GnomeDbCustomLayout *layout);
static void updated_layout_cb (GnomeDbCustomLayout *layout, GnomeDbDict *dict);

static void dict_changed (GnomeDbDict *dict, gpointer data);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	QUERY_ADDED,
	QUERY_REMOVED,
	QUERY_UPDATED,
	GRAPH_ADDED,
	GRAPH_REMOVED,
	GRAPH_UPDATED,
	LAYOUT_ADDED,
	LAYOUT_REMOVED,
	LAYOUT_UPDATED,
	CHANGED,
	LAST_SIGNAL
};

static gint gnome_db_dict_signals[LAST_SIGNAL] = { 0, 0, 0, 0, 0, 0, 0 };

/* properties */
enum
{
	PROP_0,
	PROP_SERIAL_QUERY,
	PROP_SERIAL_GRAPH,
	PROP_SERIAL_LAYOUT
};


struct _GnomeDbDictPrivate
{
	guint              serial_query; /* counter */
	guint              serial_graph; /* counter */
	guint              serial_layout; /* counter */

	GSList            *assumed_queries;   /* list of GnomeDbQuery objects */
	GSList            *all_queries;

	GSList            *assumed_graphs;   /* list of GnomeDbGraph objects */
	GSList            *all_graphs;

	GSList            *assumed_layouts;   /* list of GnomeDbCustomLayout objects */
	GSList            *all_layouts;

	GnomeDbDatabase        *database;
	GnomeDbServer          *srv;

	gchar             *xml_filename;
};

/* module error */
GQuark gnome_db_dict_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_dict_error");
	return quark;
}


guint
gnome_db_dict_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbDictClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_dict_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbDict),
			0,
			(GInstanceInitFunc) gnome_db_dict_init
		};
		
		type = g_type_register_static (G_TYPE_OBJECT, "GnomeDbDict", &info, 0);
	}
	return type;
}

static void
gnome_db_dict_class_init (GnomeDbDictClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_dict_signals[QUERY_ADDED] =
		g_signal_new ("query_added",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbDictClass, query_added),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_dict_signals[QUERY_REMOVED] =
		g_signal_new ("query_removed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbDictClass, query_removed),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_dict_signals[QUERY_UPDATED] =
		g_signal_new ("query_updated",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbDictClass, query_updated),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_dict_signals[GRAPH_ADDED] =
		g_signal_new ("graph_added",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbDictClass, graph_added),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_dict_signals[GRAPH_REMOVED] =
		g_signal_new ("graph_removed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbDictClass, graph_removed),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_dict_signals[GRAPH_UPDATED] =
		g_signal_new ("graph_updated",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbDictClass, graph_updated),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_dict_signals[LAYOUT_ADDED] =
		g_signal_new ("layout_added",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbDictClass, layout_added),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_dict_signals[LAYOUT_REMOVED] =
		g_signal_new ("layout_removed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbDictClass, layout_removed),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_dict_signals[LAYOUT_UPDATED] =
		g_signal_new ("layout_updated",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbDictClass, layout_updated),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_dict_signals[CHANGED] =
		g_signal_new ("changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbDictClass, changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->query_added = (void (*) (GnomeDbDict *, GnomeDbQuery *)) dict_changed;
	class->query_removed = (void (*) (GnomeDbDict *, GnomeDbQuery *)) dict_changed;
	class->query_updated = (void (*) (GnomeDbDict *, GnomeDbQuery *)) dict_changed;
	class->graph_added = (void (*) (GnomeDbDict *, GnomeDbGraph *)) dict_changed;
	class->graph_removed = (void (*) (GnomeDbDict *, GnomeDbGraph *)) dict_changed;
	class->graph_updated = (void (*) (GnomeDbDict *, GnomeDbGraph *)) dict_changed;
	class->layout_added = (void (*) (GnomeDbDict *, GnomeDbCustomLayout *)) dict_changed;
	class->layout_removed = (void (*) (GnomeDbDict *, GnomeDbCustomLayout *)) dict_changed;
	class->layout_updated = (void (*) (GnomeDbDict *, GnomeDbCustomLayout *)) dict_changed;
	class->changed = NULL;

	/* Properties */
	object_class->set_property = gnome_db_dict_set_property;
	object_class->get_property = gnome_db_dict_get_property;
	g_object_class_install_property (object_class, PROP_SERIAL_QUERY,
					 g_param_spec_uint ("query_serial", NULL, NULL, 
							    1, G_MAXUINT, 1, G_PARAM_READABLE));
	g_object_class_install_property (object_class, PROP_SERIAL_GRAPH,
					 g_param_spec_uint ("graph_serial", NULL, NULL, 
							    1, G_MAXUINT, 1, G_PARAM_READABLE));
	g_object_class_install_property (object_class, PROP_SERIAL_LAYOUT,
					 g_param_spec_uint ("layout_serial", NULL, NULL, 
							    1, G_MAXUINT, 1, G_PARAM_READABLE));

	object_class->dispose = gnome_db_dict_dispose;
	object_class->finalize = gnome_db_dict_finalize;
}

/* data is unused */
static void
dict_changed (GnomeDbDict *dict, gpointer data)
{
#ifdef debug_signal
	g_print (">> 'CHANGED' from %s()\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (dict), gnome_db_dict_signals[CHANGED], 0);
#ifdef debug_signal
	g_print ("<< 'CHANGED' from %s()\n", __FUNCTION__);
#endif	
}

static void
gnome_db_dict_init (GnomeDbDict * dict)
{
	dict->priv = g_new0 (GnomeDbDictPrivate, 1);
	dict->priv->serial_query = 1;
	dict->priv->serial_graph = 1;
	dict->priv->serial_layout = 1;
	dict->priv->assumed_queries = NULL;
	dict->priv->all_queries = NULL;
	dict->priv->assumed_graphs = NULL;
	dict->priv->all_graphs = NULL;
	dict->priv->assumed_layouts = NULL;
	dict->priv->all_layouts = NULL;
	dict->priv->database = NULL;
	dict->priv->srv = NULL;
	dict->priv->xml_filename = NULL;
}

/**
 * gnome_db_dict_new
 *
 * Create a new #GnomeDbDict object.
 *
 * Returns: the newly created object.
 */
GObject   *
gnome_db_dict_new ()
{
	GObject   *obj;
	GnomeDbDict *dict;

	obj = g_object_new (GNOME_DB_DICT_TYPE, NULL);
	dict = GNOME_DB_DICT (obj);

	/* Server and database objects creation */
	dict->priv->srv = GNOME_DB_SERVER (gnome_db_server_new (dict));
	dict->priv->database = GNOME_DB_DATABASE (gnome_db_database_new (dict));

	return obj;
}


static void
gnome_db_dict_dispose (GObject   * object)
{
	GnomeDbDict *dict;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_DICT (object));

	dict = GNOME_DB_DICT (object);
	if (dict->priv) {
		GSList *list;

		/* layouts */
		list = dict->priv->all_layouts;
		while (list) {
			g_object_weak_unref (G_OBJECT (list->data), (GWeakNotify) layout_weak_ref_notify, dict);
			
			g_signal_handlers_disconnect_by_func (G_OBJECT (list->data), 
							      G_CALLBACK (layout_id_changed_cb), dict);
			list = g_slist_next (list);
		}

		if (dict->priv->all_layouts) {
			g_slist_free (dict->priv->all_layouts);
			dict->priv->all_layouts = NULL;
		}

		while (dict->priv->assumed_layouts)
			gnome_db_base_nullify (GNOME_DB_BASE (dict->priv->assumed_layouts->data));


		/* graphs */
		list = dict->priv->all_graphs;
		while (list) {
			g_object_weak_unref (G_OBJECT (list->data), (GWeakNotify) graph_weak_ref_notify, dict);
			
			g_signal_handlers_disconnect_by_func (G_OBJECT (list->data), 
							      G_CALLBACK (graph_id_changed_cb), dict);
			list = g_slist_next (list);
		}

		if (dict->priv->all_graphs) {
			g_slist_free (dict->priv->all_graphs);
			dict->priv->all_graphs = NULL;
		}

		while (dict->priv->assumed_graphs)
			gnome_db_base_nullify (GNOME_DB_BASE (dict->priv->assumed_graphs->data));

		/* queries */
		while (dict->priv->assumed_queries)
			gnome_db_base_nullify (GNOME_DB_BASE (dict->priv->assumed_queries->data));
		
		list = dict->priv->all_queries;
		while (list) {
			g_object_weak_unref (G_OBJECT (list->data), (GWeakNotify) query_weak_ref_notify, dict);
			
			g_signal_handlers_disconnect_by_func (G_OBJECT (list->data), 
							      G_CALLBACK (query_id_changed_cb), dict);
			gnome_db_base_nullify (GNOME_DB_BASE (list->data));
			list = g_slist_next (list);
		}

		if (dict->priv->all_queries) {
			g_slist_free (dict->priv->all_queries);
			dict->priv->all_queries = NULL;
		}

		/* database */
		if (dict->priv->database) {
			g_object_unref (G_OBJECT (dict->priv->database));
			dict->priv->database = NULL;
		}
		
		/* server */
		if (dict->priv->srv) {
			g_object_unref (G_OBJECT (dict->priv->srv));
			dict->priv->srv = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_dict_finalize (GObject   * object)
{
	GnomeDbDict *dict;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_DICT (object));

	dict = GNOME_DB_DICT (object);
	if (dict->priv) {
		if (dict->priv->xml_filename) {
			g_free (dict->priv->xml_filename);
			dict->priv->xml_filename = NULL;
		}

		g_free (dict->priv);
		dict->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}

static void 
gnome_db_dict_set_property (GObject              *object,
		      guint                 param_id,
		      const GValue         *value,
		      GParamSpec           *pspec)
{
	GnomeDbDict *gnome_db_dict;

	gnome_db_dict = GNOME_DB_DICT (object);
	if (gnome_db_dict->priv) {
		switch (param_id) {
		case PROP_SERIAL_QUERY:
		case PROP_SERIAL_GRAPH:
		case PROP_SERIAL_LAYOUT:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
		}
	}
}

static void
gnome_db_dict_get_property (GObject              *object,
		       guint                 param_id,
		       GValue               *value,
		       GParamSpec           *pspec)
{
	GnomeDbDict *gnome_db_dict;
	gnome_db_dict = GNOME_DB_DICT (object);
	
	if (gnome_db_dict->priv) {
		switch (param_id) {
		case PROP_SERIAL_QUERY:
			g_value_set_uint (value, gnome_db_dict->priv->serial_query++);
			break;
		case PROP_SERIAL_GRAPH:
			g_value_set_uint (value, gnome_db_dict->priv->serial_graph++);
			break;
		case PROP_SERIAL_LAYOUT:
			g_value_set_uint (value, gnome_db_dict->priv->serial_layout++);
			break;
		}	
	}
}


static void xml_validity_error_func (void *ctx, const char *msg, ...);

/**
 * gnome_db_dict_load_xml_file
 * @dict: a #GnomeDbDict object
 * @xmlfile: the name of the file to which the XML will be written to
 * @error: location to store error, or %NULL
 *
 * Loads an XML file which respects the Libgnomedb DTD, and creates all the necessary
 * objects that are defined within the XML file. During the creation of the other
 * objects, all the normal signals are emitted.
 *
 * If the GnomeDbDict object already has some contents, then it is first of all
 * nullified (to return its state as when it was first created).
 *
 * If an error occurs during loading then the GnomeDbDict object is left as empty
 * as when it is first created.
 *
 * Returns: TRUE if loading was successfull and FALSE otherwise.
 */
gboolean
gnome_db_dict_load_xml_file (GnomeDbDict *dict, const gchar *xmlfile, GError **error)
{
	xmlDocPtr doc;
	xmlNodePtr node, subnode;

	g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), FALSE);
	g_return_val_if_fail (dict->priv, FALSE);
	g_return_val_if_fail (xmlfile && *xmlfile, FALSE);

	if (! g_file_test (xmlfile, G_FILE_TEST_EXISTS)) {
		g_set_error (error,
			     GNOME_DB_DICT_ERROR,
			     GNOME_DB_DICT_LOAD_FILE_NOT_EXIST_ERROR,
			     "File '%s' does not exist", xmlfile);
		return FALSE;
	}

	doc = xmlParseFile (xmlfile);

	if (doc) {
		/* doc validation */
		xmlValidCtxtPtr validc;

		validc = g_new0 (xmlValidCtxt, 1);
		validc->userData = dict;
		validc->error = xml_validity_error_func;
		validc->warning = NULL; 
		xmlDoValidityCheckingDefaultValue = 1;
		if (! xmlValidateDocument (validc, doc)) {
			gchar *str;

			xmlFreeDoc (doc);
			g_free (validc);
			str = g_object_get_data (G_OBJECT (dict), "xmlerror");
			if (str) {
				g_set_error (error,
					     GNOME_DB_DICT_ERROR,
					     GNOME_DB_DICT_FILE_LOAD_ERROR,
					     "File '%s' does not dictorm to DTD:\n%s", xmlfile, str);
				g_free (str);
				g_object_set_data (G_OBJECT (dict), "xmlerror", NULL);
			}
			else 
				g_set_error (error,
					     GNOME_DB_DICT_ERROR,
					     GNOME_DB_DICT_FILE_LOAD_ERROR,
					     "File '%s' does not dictorm to DTD", xmlfile);

			return FALSE;
		}
		g_free (validc);
	}
	else {
		g_set_error (error,
			     GNOME_DB_DICT_ERROR,
			     GNOME_DB_DICT_FILE_LOAD_ERROR,
			     "Can't load file '%s'", xmlfile);
		return FALSE;
	}

	/* doc is now OK */
	node = xmlDocGetRootElement (doc);
	if (strcmp (node->name, "GNOME_DB_DICT")) {
		g_set_error (error,
			     GNOME_DB_DICT_ERROR,
			     GNOME_DB_DICT_FILE_LOAD_ERROR,
			     "XML file '%s' does not have any <GNOME_DB_DICT> node", xmlfile);
		return FALSE;
	}
	subnode = node->children;
	
	if (!subnode) {
		g_set_error (error,
			     GNOME_DB_DICT_ERROR,
			     GNOME_DB_DICT_FILE_LOAD_ERROR,
			     "XML file '%s': <GNOME_DB_DICT> does not have any children",
			     xmlfile);
		return FALSE;
	}

	/* GnomeDbServer object */
	if (xmlNodeIsText (subnode)) 
		subnode = subnode->next;

	if (strcmp (subnode->name, "GNOME_DB_SERVER")) {
		g_set_error (error,
			     GNOME_DB_DICT_ERROR,
			     GNOME_DB_DICT_FILE_LOAD_ERROR,
			     "XML file '%s': <GNOME_DB_SERVER> not first child of <GNOME_DB_DICT>",
			     xmlfile);
		return FALSE;
	}
	if (!gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (dict->priv->srv), subnode, error))
		return FALSE;
	
	/* GnomeDbDatabase object */
	subnode = subnode->next;
	if (xmlNodeIsText (subnode)) 
		subnode = subnode->next;
	if (!subnode || strcmp (subnode->name, "GNOME_DB_DATABASE")) {
		g_set_error (error,
			     GNOME_DB_DICT_ERROR,
			     GNOME_DB_DICT_FILE_LOAD_ERROR,
			     "XML file '%s': <GNOME_DB_DATABASE> not second child of <GNOME_DB_DICT>",
			     xmlfile);
		return FALSE;
	}
	if (!gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (dict->priv->database), subnode, error))
		return FALSE;

	/* GnomeDbQuery objects */
	if (subnode) {
		subnode = subnode->next;
		if (xmlNodeIsText (subnode)) 
			subnode = subnode->next;
		if (!subnode || strcmp (subnode->name, "GNOME_DB_QUERIES")) {
			g_set_error (error,
				     GNOME_DB_DICT_ERROR,
				     GNOME_DB_DICT_FILE_LOAD_ERROR,
				     "XML file '%s': <GNOME_DB_QUERIES> not 3rd child of <GNOME_DB_DICT>",
				     xmlfile);
			return FALSE;
		}
		if (!gnome_db_dict_load_queries (dict, subnode, error))
			return FALSE;
	}

	/* GnomeDbGraph objects */
	if (subnode) {
		subnode = subnode->next;
		if (xmlNodeIsText (subnode)) 
			subnode = subnode->next;
		if (subnode && strcmp (subnode->name, "GNOME_DB_GRAPHS")) {
			g_set_error (error,
				     GNOME_DB_DICT_ERROR,
				     GNOME_DB_DICT_FILE_LOAD_ERROR,
				     "XML file '%s': <GNOME_DB_GRAPHS> not 4th child of <GNOME_DB_DICT>",
				     xmlfile);
			return FALSE;
		}
		if (subnode && !gnome_db_dict_load_graphs (dict, subnode, error))
			return FALSE;
	}

	/* GnomeDbCustomLayout objects */
	if (subnode) {
		subnode = subnode->next;
		if (xmlNodeIsText (subnode)) 
			subnode = subnode->next;
		if (subnode && strcmp (subnode->name, "GNOME_DB_LAYOUTS")) {
			g_set_error (error,
				     GNOME_DB_DICT_ERROR,
				     GNOME_DB_DICT_FILE_LOAD_ERROR,
				     "XML file '%s': <GNOME_DB_LAYOUTS> not 5th child of <GNOME_DB_DICT>",
				     xmlfile);
			return FALSE;
		}
		if (subnode && !gnome_db_dict_load_custom_layouts (dict, subnode, error))
			return FALSE;
	}

	xmlFreeDoc (doc);

	return TRUE;
}

static gboolean
gnome_db_dict_load_queries (GnomeDbDict *dict, xmlNodePtr queries, GError **error)
{
	xmlNodePtr qnode = queries->children;
	gboolean allok = TRUE;
	
	while (qnode && allok) {
		if (!strcmp (qnode->name, "GNOME_DB_QUERY")) {
			GnomeDbQuery *query;

			query = GNOME_DB_QUERY (gnome_db_query_new (dict));
			allok = gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (query), qnode, error);
			gnome_db_dict_assume_query (dict, query);
			g_object_unref (G_OBJECT (query));
		}
		qnode = qnode->next;
	}

	if (allok) {
		GSList *list = dict->priv->assumed_queries;
		while (list) {
			gnome_db_referer_activate (GNOME_DB_REFERER (list->data));
			list= g_slist_next (list);
		}
	}

	return allok;
}

static gboolean
gnome_db_dict_load_graphs (GnomeDbDict *dict, xmlNodePtr graphs, GError **error)
{
	xmlNodePtr qnode = graphs->children;
	gboolean allok = TRUE;
	
	while (qnode && allok) {
		if (!strcmp (qnode->name, "GNOME_DB_GRAPH")) {
			GnomeDbGraph *graph = NULL;
			gchar *prop;

			prop = xmlGetProp (qnode, "type");
			if (prop) {
				gchar *oprop = xmlGetProp (qnode, "object");
				if (!oprop && (*prop == 'Q')) {
					allok = FALSE;
					g_set_error (error,
						     GNOME_DB_DICT_ERROR,
						     GNOME_DB_DICT_FILE_LOAD_ERROR,
						     _("GNOME_DB_GRAPH of type 'Q' must have an 'object' attribute"));
				}
				
				if (allok && (*prop == 'Q')) {
					GnomeDbQuery *query = gnome_db_dict_get_query_by_xml_id (dict, oprop);
					if (!query) {
						allok = FALSE;
						g_set_error (error,
							     GNOME_DB_DICT_ERROR,
							     GNOME_DB_DICT_FILE_LOAD_ERROR,
							     _("GNOME_DB_GRAPH of type 'Q' must have valid 'object' attribute"));
					}
					else 
						graph = GNOME_DB_GRAPH (gnome_db_graph_query_new (query));
				}
				g_free (oprop);
			}
			g_free (prop);

			if (allok) {
				if (!graph)
					graph = GNOME_DB_GRAPH (gnome_db_graph_new (dict, GNOME_DB_GRAPH_DB_RELATIONS)); /* type is checked next line */
				allok = gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (graph), qnode, error);
				gnome_db_dict_assume_graph (dict, graph);
				g_object_unref (G_OBJECT (graph));
			}
		}
		qnode = qnode->next;
	}

	if (0 && allok) { /* the GnomeDbGraph does not implement the GnomeDbReferer interface for now */
		GSList *list = dict->priv->assumed_graphs;
		while (list) {
			gnome_db_referer_activate (GNOME_DB_REFERER (list->data));
			list= g_slist_next (list);
		}
	}

	return allok;
}

static gboolean
gnome_db_dict_load_custom_layouts (GnomeDbDict *dict, xmlNodePtr layouts, GError **error)
{
	xmlNodePtr qnode = layouts->children;
	gboolean allok = TRUE;

	while (qnode && allok) {
		GnomeDbCustomLayout *cl;

		if (! xmlNodeIsText (qnode)) {
			cl = (GnomeDbCustomLayout *) gnome_db_custom_layout_new (dict);
			allok = gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (cl), qnode, error);
			gnome_db_dict_assume_layout (dict, cl);
			g_object_unref (G_OBJECT (cl));
		}
		    
		qnode = qnode->next;
	}

	if (0 && allok) { /* the GnomeDbGraph does not implement the GnomeDbReferer interface for now */
		GSList *list = dict->priv->assumed_layouts;
		while (list) {
			gnome_db_referer_activate (GNOME_DB_REFERER (list->data));
			list= g_slist_next (list);
		}
	}

	return allok;
}

/*
 * function called when an error occurred during the document validation
 */
static void
xml_validity_error_func (void *ctx, const char *msg, ...)
{
        xmlValidCtxtPtr pctxt;
        va_list args;
        gchar *str, *str2, *newerr;
	GnomeDbDict *dict;

	pctxt = (xmlValidCtxtPtr) ctx;
	/* FIXME: it looks like libxml does set ctx to userData... */
	/*dict = GNOME_DB_DICT (pctxt->userData);*/
	dict = GNOME_DB_DICT (pctxt);
	str2 = g_object_get_data (G_OBJECT (dict), "xmlerror");

        va_start (args, msg);
        str = g_strdup_vprintf (msg, args);
        va_end (args);
	
	if (str2) {
		newerr = g_strdup_printf ("%s\n%s", str2, str);
		g_free (str2);
	}
	else
		newerr = g_strdup (str);
        g_free (str);
	g_object_set_data (G_OBJECT (dict), "xmlerror", newerr);
}


/**
 * gnome_db_dict_save_xml_file
 * @dict: a #GnomeDbDict object
 * @xmlfile: the name of the file to which the XML will be written to
 * @error: location to store error, or %NULL
 *
 * Saves the contents of a GnomeDbDict object to a file which is given as argument.
 *
 * Returns: TRUE if saving was successfull and FALSE otherwise.
 */
gboolean
gnome_db_dict_save_xml_file (GnomeDbDict *dict, const gchar *xmlfile, GError **error)
{
	gboolean retval = TRUE;
	xmlDocPtr doc;
#define XML_LIBGNOME_DB_DTD_FILE DTDINSTALLDIR"/libgnomedb-dict.dtd"

	g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), FALSE);
	g_return_val_if_fail (dict->priv, FALSE);
		
	doc = xmlNewDoc ("1.0");
	if (doc) {
		xmlNodePtr topnode, node;

		/* DTD insertion */
                xmlCreateIntSubset(doc, "GNOME_DB_DICT", NULL, XML_LIBGNOME_DB_DTD_FILE);

		/* Top node */
		topnode = xmlNewDocNode (doc, NULL, "GNOME_DB_DICT", NULL);
		xmlDocSetRootElement (doc, topnode);

		/* GnomeDbServer */
		node = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (dict->priv->srv), error);
		if (node)
			xmlAddChild (topnode, node);
		else 
			/* error handling */
			retval = FALSE;
		
		/* GnomeDbDatabase */
		if (retval) {
			node = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (dict->priv->database), error);
			if (node)
				xmlAddChild (topnode, node);
			else 
				/* error handling */
				retval = FALSE;
		}
		
		/* GnomeDbQuery objects */
		if (retval) {
			GSList *list;
			node = xmlNewChild (topnode, NULL, "GNOME_DB_QUERIES", NULL);
			list = dict->priv->assumed_queries;
			while (list) {
				if (!gnome_db_query_get_parent_query (GNOME_DB_QUERY (list->data))) {
					/* We only add nodes for queries which do not have any parent query */
					xmlNodePtr qnode;
					
					qnode = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
					if (qnode)
						xmlAddChild (node, qnode);
					else 
						/* error handling */
						retval = FALSE;
				}
				list = g_slist_next(list);
			}
		}

		/* GnomeDbGraph objects */
		if (retval) {
			GSList *list;
			node = xmlNewChild (topnode, NULL, "GNOME_DB_GRAPHS", NULL);
			list = dict->priv->assumed_graphs;
			while (list) {
				xmlNodePtr qnode;
				
				qnode = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
				if (qnode)
					xmlAddChild (node, qnode);
				else 
					/* error handling */
					retval = FALSE;

				list = g_slist_next(list);
			}
		}

		/* GnomeDbCustomLayout objects */
		if (retval) {
			GSList *list;
			node = xmlNewChild (topnode, NULL, "GNOME_DB_LAYOUTS", NULL);
			list = dict->priv->assumed_layouts;
			while (list) {
				xmlNodePtr qnode;
				
				qnode = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
				if (qnode)
					xmlAddChild (node, qnode);
				else 
					/* error handling */
					retval = FALSE;

				list = g_slist_next(list);
			}
		}

		/* actual writing to file */
		if (retval) {
			gint i;
			i = xmlSaveFormatFile (xmlfile, doc, TRUE);
			if (i == -1) {
				/* error handling */
				g_set_error (error, GNOME_DB_DICT_ERROR, GNOME_DB_DICT_FILE_SAVE_ERROR,
					     _("Error writing XML file %s"), xmlfile);
				retval = FALSE;
			}
		}

		/* getting rid of the doc */
		xmlFreeDoc (doc);
	}
	else {
		/* error handling */
		g_set_error (error, GNOME_DB_DICT_ERROR, GNOME_DB_DICT_FILE_SAVE_ERROR,
			     _("Can't allocate memory for XML structure."));
		retval = FALSE;
	}

	return retval;
}


/**
 * gnome_db_dict_declare_query
 * @dict: a #GnomeDbDict object
 * @query: a #GnomeDbQuery object
 *
 * Declares the existence of a new query to @dict. All the #GnomeDbQuery objects MUST
 * be declared to the corresponding #GnomeDbDict object for the library to work correctly.
 * Once @query has been declared, @dict does not hold any reference to @query. If @dict
 * must hold such a reference, then use gnome_db_dict_assume_query().
 *
 * This functions is called automatically from each gnome_db_query_new* function, and it should not be necessary
 * to call it except for classes extending the #GnomeDbQuery class.
 */
void
gnome_db_dict_declare_query (GnomeDbDict *dict, GnomeDbQuery *query)
{
	g_return_if_fail (dict && IS_GNOME_DB_DICT (dict));
	g_return_if_fail (dict->priv);
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
	
	/* we don't take any reference on query */
	if (g_slist_find (dict->priv->all_queries, query))
		return;	
	else {
		dict->priv->all_queries = g_slist_append (dict->priv->all_queries, query);
		g_object_weak_ref (G_OBJECT (query), (GWeakNotify) query_weak_ref_notify, dict);

		/* make sure the dict->priv->serial_query value is always 1 above this query's id */
		query_id_changed_cb (query, dict);
		g_signal_connect (G_OBJECT (query), "id_changed",
				  G_CALLBACK (query_id_changed_cb), dict);
	}
}

static void
query_id_changed_cb (GnomeDbQuery *query, GnomeDbDict *dict)
{
	if (dict->priv->serial_query <= gnome_db_base_get_id (GNOME_DB_BASE (query)))
		dict->priv->serial_query = gnome_db_base_get_id (GNOME_DB_BASE (query)) + 1;
}


static void
query_weak_ref_notify (GnomeDbDict *dict, GnomeDbQuery *query)
{
	dict->priv->all_queries = g_slist_remove (dict->priv->all_queries, query);
	g_signal_handlers_disconnect_by_func (G_OBJECT (query),
					      G_CALLBACK (query_id_changed_cb), dict);
}

static void
updated_query_cb (GnomeDbQuery *query, GnomeDbDict *dict)
{
#ifdef debug_signal
	g_print (">> 'QUERY_UPDATED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (dict), "query_updated", query);
#ifdef debug_signal
	g_print ("<< 'QUERY_UPDATED' from %s\n", __FUNCTION__);
#endif	
}

/**
 * gnome_db_dict_assume_query
 * @dict: a #GnomeDbDict object
 * @query: a #GnomeDbQuery object
 *
 * Force @dict to manage @query: it will get a reference to it.
 */
void
gnome_db_dict_assume_query (GnomeDbDict *dict, GnomeDbQuery *query)
{
	g_return_if_fail (dict && IS_GNOME_DB_DICT (dict));
	g_return_if_fail (dict->priv);
	g_return_if_fail (query && IS_GNOME_DB_QUERY (query));
	
	if (g_slist_find (dict->priv->assumed_queries, query)) {
		g_warning ("GnomeDbQuery %p already assumed!", query);
		return;
	}

	gnome_db_dict_declare_query (dict, query);

	dict->priv->assumed_queries = g_slist_append (dict->priv->assumed_queries, query);
	g_object_ref (G_OBJECT (query));
	gnome_db_base_connect_nullify (query,
				 G_CALLBACK (query_nullified_cb), dict);
	g_signal_connect (G_OBJECT (query), "changed",
			  G_CALLBACK (updated_query_cb), dict);

#ifdef debug_signal
	g_print (">> 'QUERY_ADDED' from gnome_db_dict_assume_query\n");
#endif
	g_signal_emit (G_OBJECT (dict), gnome_db_dict_signals[QUERY_ADDED], 0, query);
#ifdef debug_signal
	g_print ("<< 'QUERY_ADDED' from gnome_db_dict_assume_query\n");
#endif
}


/* called when a GnomeDbQuery is "nullified" */
static void 
query_nullified_cb (GnomeDbQuery *query, GnomeDbDict *dict)
{
	gnome_db_dict_unassume_query (dict, query);
}


/**
 * gnome_db_dict_unassume_query
 * @dict: a #GnomeDbDict object
 * @query: a #GnomeDbQuery object
 *
 * Forces @dict to lose a reference it has on @query
 */
void
gnome_db_dict_unassume_query (GnomeDbDict *dict, GnomeDbQuery *query)
{
	g_return_if_fail (dict && IS_GNOME_DB_DICT (dict));
	g_return_if_fail (dict->priv);

	if (g_slist_find (dict->priv->assumed_queries, query)) {
		dict->priv->assumed_queries = g_slist_remove (dict->priv->assumed_queries, query);
		g_signal_handlers_disconnect_by_func (G_OBJECT (query),
						      G_CALLBACK (query_nullified_cb), dict);
		g_signal_handlers_disconnect_by_func (G_OBJECT (query),
						      G_CALLBACK (updated_query_cb), dict);
#ifdef debug_signal
		g_print (">> 'QUERY_REMOVED' from gnome_db_dict_unassume_query\n");
#endif
		g_signal_emit (G_OBJECT (dict), gnome_db_dict_signals[QUERY_REMOVED], 0, query);
#ifdef debug_signal
		g_print ("<< 'QUERY_REMOVED' from gnome_db_dict_unassume_query\n");
#endif
		g_object_unref (G_OBJECT (query));
	}
}


/**
 * gnome_db_dict_get_queries
 * @dict: a #GnomeDbDict object
 *
 * Get a list of all the non interdependant queries managed by @dict
 * (only queries with no parent query are listed)
 *
 * Returns: a new list of #GnomeDbQuery objects
 */
GSList *
gnome_db_dict_get_queries (GnomeDbDict *dict)
{
	GSList *list, *retval = NULL;

	g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), NULL);
	g_return_val_if_fail (dict->priv, NULL);

	list = dict->priv->assumed_queries;
	while (list) {
		if (! gnome_db_query_get_parent_query (GNOME_DB_QUERY (list->data)))
			retval = g_slist_append (retval, list->data);
		list = g_slist_next (list);
	}

	return retval;
}

/**
 * gnome_db_dict_get_query_by_xml_id
 * @dict: a #GnomeDbDict object
 * @xml_id: the XML Id of the query being searched
 *
 * Find a #GnomeDbQuery object from its XML Id
 *
 * Returns: the #GnomeDbQuery object, or NULL if not found
 */
GnomeDbQuery *
gnome_db_dict_get_query_by_xml_id (GnomeDbDict *dict, const gchar *xml_id)
{
	GnomeDbQuery *query = NULL;

	GSList *list;
        gchar *str;

        g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), NULL);
        g_return_val_if_fail (dict->priv, NULL);

        list = dict->priv->all_queries;
        while (list && !query) {
                str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
                if (!strcmp (str, xml_id))
                        query = GNOME_DB_QUERY (list->data);
                g_free (str);
                list = g_slist_next (list);
        }

        return query;
}

/**
 * gnome_db_dict_declare_graph
 * @dict: a #GnomeDbDict object
 * @graph: a #GnomeDbGraph object
 *
 * Declares the existence of a new graph to @dict. All the #GnomeDbGraph objects MUST
 * be declared to the corresponding #GnomeDbDict object for the library to work correctly.
 * Once @graph has been declared, @dict does not hold any reference to @graph. If @dict
 * must hold such a reference, then use gnome_db_dict_assume_graph().
 *
 * This functions is called automatically from each gnome_db_graph_new* function, and it should not be necessary
 * to call it except for classes extending the #GnomeDbGraph class.
 */
void
gnome_db_dict_declare_graph (GnomeDbDict *dict, GnomeDbGraph *graph)
{
	g_return_if_fail (dict && IS_GNOME_DB_DICT (dict));
	g_return_if_fail (dict->priv);
	g_return_if_fail (graph && IS_GNOME_DB_GRAPH (graph));
	
	/* we don't take any reference on graph */
	if (g_slist_find (dict->priv->all_graphs, graph))
		return;	
	else {
		dict->priv->all_graphs = g_slist_append (dict->priv->all_graphs, graph);
		g_object_weak_ref (G_OBJECT (graph), (GWeakNotify) graph_weak_ref_notify, dict);

		/* make sure the dict->priv->serial_graph value is always 1 above this graph's id */
		graph_id_changed_cb (graph, dict);
		g_signal_connect (G_OBJECT (graph), "id_changed",
				  G_CALLBACK (graph_id_changed_cb), dict);
	}
}

static void
graph_id_changed_cb (GnomeDbGraph *graph, GnomeDbDict *dict)
{
	if (dict->priv->serial_graph <= gnome_db_base_get_id (GNOME_DB_BASE (graph)))
		dict->priv->serial_graph = gnome_db_base_get_id (GNOME_DB_BASE (graph)) + 1;
}


static void
graph_weak_ref_notify (GnomeDbDict *dict, GnomeDbGraph *graph)
{
	dict->priv->all_graphs = g_slist_remove (dict->priv->all_graphs, graph);
	g_signal_handlers_disconnect_by_func (G_OBJECT (graph),
					      G_CALLBACK (graph_id_changed_cb), dict);
}

static void
updated_graph_cb (GnomeDbGraph *graph, GnomeDbDict *dict)
{
#ifdef debug_signal
	g_print (">> 'GRAPH_UPDATED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (dict), "graph_updated", graph);
#ifdef debug_signal
	g_print ("<< 'GRAPH_UPDATED' from %s\n", __FUNCTION__);
#endif
}



/**
 * gnome_db_dict_assume_graph
 * @dict: a #GnomeDbDict object
 * @graph: a #GnomeDbGraph object
 *
 * Force @dict to manage @graph: it will get a reference to it.
 */
void
gnome_db_dict_assume_graph (GnomeDbDict *dict, GnomeDbGraph *graph)
{
	g_return_if_fail (dict && IS_GNOME_DB_DICT (dict));
	g_return_if_fail (dict->priv);
	g_return_if_fail (graph && IS_GNOME_DB_GRAPH (graph));
	
	if (g_slist_find (dict->priv->assumed_graphs, graph)) {
		g_warning ("GnomeDbGraph %p already assumed!", graph);
		return;
	}

	gnome_db_dict_declare_graph (dict, graph);

	dict->priv->assumed_graphs = g_slist_append (dict->priv->assumed_graphs, graph);
	g_object_ref (G_OBJECT (graph));
	gnome_db_base_connect_nullify (graph, 
				 G_CALLBACK (graph_nullified_cb), dict);
	g_signal_connect (G_OBJECT (graph), "changed",
			  G_CALLBACK (updated_graph_cb), dict);

#ifdef debug_signal
	g_print (">> 'GRAPH_ADDED' from gnome_db_dict_assume_graph\n");
#endif
	g_signal_emit (G_OBJECT (dict), gnome_db_dict_signals[GRAPH_ADDED], 0, graph);
#ifdef debug_signal
	g_print ("<< 'GRAPH_ADDED' from gnome_db_dict_assume_graph\n");
#endif
}


/* called when a GnomeDbGraph is "nullified" */
static void 
graph_nullified_cb (GnomeDbGraph *graph, GnomeDbDict *dict)
{
	gnome_db_dict_unassume_graph (dict, graph);
}


/**
 * gnome_db_dict_unassume_graph
 * @dict: a #GnomeDbDict object
 * @graph: a #GnomeDbGraph object
 *
 * Forces @dict to lose a reference it has on @graph
 */
void
gnome_db_dict_unassume_graph (GnomeDbDict *dict, GnomeDbGraph *graph)
{
	g_return_if_fail (dict && IS_GNOME_DB_DICT (dict));
	g_return_if_fail (dict->priv);

	if (g_slist_find (dict->priv->assumed_graphs, graph)) {
		dict->priv->assumed_graphs = g_slist_remove (dict->priv->assumed_graphs, graph);
		g_signal_handlers_disconnect_by_func (G_OBJECT (graph),
						      G_CALLBACK (graph_nullified_cb), dict);
		g_signal_handlers_disconnect_by_func (G_OBJECT (graph),
						      G_CALLBACK (updated_graph_cb), dict);
#ifdef debug_signal
		g_print (">> 'GRAPH_REMOVED' from gnome_db_dict_unassume_graph\n");
#endif
		g_signal_emit (G_OBJECT (dict), gnome_db_dict_signals[GRAPH_REMOVED], 0, graph);
#ifdef debug_signal
		g_print ("<< 'GRAPH_REMOVED' from gnome_db_dict_unassume_graph\n");
#endif
		g_object_unref (G_OBJECT (graph));
	}
}


/**
 * gnome_db_dict_get_graphs
 * @dict: a #GnomeDbDict object
 * @type_of_graphs: the requested type of graphs
 *
 * Get a list of the graphs managed by @dict, which are of the
 * requested type.
 *
 * Returns: a new list of #GnomeDbGraph objects
 */
GSList *
gnome_db_dict_get_graphs (GnomeDbDict *dict, GnomeDbGraphType type_of_graphs)
{
	g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), NULL);
	g_return_val_if_fail (dict->priv, NULL);

	if (dict->priv->assumed_graphs) {
		GSList *list, *retval = NULL;

		list = dict->priv->assumed_graphs;
		while (list) {
			if (gnome_db_graph_get_graph_type (GNOME_DB_GRAPH (list->data)) == type_of_graphs)
				retval = g_slist_prepend (retval, list->data);
			list = g_slist_next (list);
		}
		return g_slist_reverse (retval);
	}
	else
		return NULL;
}

/**
 * gnome_db_dict_get_graph_by_xml_id
 * @dict: a #GnomeDbDict object
 * @xml_id: the XML Id of the graph being searched
 *
 * Find a #GnomeDbGraph object from its XML Id
 *
 * Returns: the #GnomeDbGraph object, or NULL if not found
 */
GnomeDbGraph *
gnome_db_dict_get_graph_by_xml_id (GnomeDbDict *dict, const gchar *xml_id)
{
	GnomeDbGraph *graph = NULL;

	GSList *list;
        gchar *str;

        g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), NULL);
        g_return_val_if_fail (dict->priv, NULL);

        list = dict->priv->all_graphs;
        while (list && !graph) {
                str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
                if (!strcmp (str, xml_id))
                        graph = GNOME_DB_GRAPH (list->data);
                g_free (str);
                list = g_slist_next (list);
        }

        return graph;
}

/**
 * gnome_db_dict_get_graph_for_object
 * @dict: a #GnomeDbDict object
 * @obj: a #Gobject object
 *
 * Find a #GnomeDbGraph object guiven the object it is related to.
 *
 * Returns: the #GnomeDbGraph object, or NULL if not found
 */
GnomeDbGraph *
gnome_db_dict_get_graph_for_object (GnomeDbDict *dict, GObject *obj)
{
	GnomeDbGraph *graph = NULL;

	GSList *list;
	GObject *ref_obj;

        g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), NULL);
        g_return_val_if_fail (dict->priv, NULL);

        list = dict->priv->all_graphs;
        while (list && !graph) {
		g_object_get (G_OBJECT (list->data), "ref_object", &ref_obj, NULL);
		if (ref_obj == obj)
                        graph = GNOME_DB_GRAPH (list->data);
                list = g_slist_next (list);
        }

        return graph;
}

/**
 * gnome_db_dict_declare_layout
 * @dict: a #GnomeDbDict object
 * @layout: a #GnomeDbLayout object
 *
 * Declares the existence of a new layout to @dict. All the #GnomeDbCustomLayout objects MUST
 * be declared to the corresponding #GnomeDbDict object for the library to work correctly.
 * Once @layout has been declared, @dict does not hold any reference to @layout. If @dict
 * must hold such a reference, then use gnome_db_dict_assume_layout().
 *
 * This functions is called automatically from each gnome_db_layout_new* function, and it should not be necessary
 * to call it except for classes extending the #GnomeDbCustomLayout class.
 */
void
gnome_db_dict_declare_layout (GnomeDbDict *dict, GnomeDbCustomLayout *layout)
{
	g_return_if_fail (dict && IS_GNOME_DB_DICT (dict));
	g_return_if_fail (dict->priv);
	g_return_if_fail (layout && IS_GNOME_DB_CUSTOM_LAYOUT (layout));
	
	/* we don't take any reference on layout */
	if (g_slist_find (dict->priv->all_layouts, layout))
		return;	
	else {
		dict->priv->all_layouts = g_slist_append (dict->priv->all_layouts, layout);
		g_object_weak_ref (G_OBJECT (layout), (GWeakNotify) layout_weak_ref_notify, dict);

		/* make sure the dict->priv->serial_layout value is always 1 above this layout's id */
		layout_id_changed_cb (layout, dict);
		g_signal_connect (G_OBJECT (layout), "id_changed",
				  G_CALLBACK (layout_id_changed_cb), dict);
	}
}

/**
 * gnome_db_dict_assume_layout
 * @dict: a #GnomeDbDict object
 * @layout: a #GnomeDbCustomLayout object
 *
 * Force @dict to manage @layout: it will get a reference to it.
 */
void
gnome_db_dict_assume_layout (GnomeDbDict *dict, GnomeDbCustomLayout *layout)
{
	g_return_if_fail (dict && IS_GNOME_DB_DICT (dict));
	g_return_if_fail (dict->priv);
	g_return_if_fail (layout && IS_GNOME_DB_CUSTOM_LAYOUT (layout));
	
	if (g_slist_find (dict->priv->assumed_layouts, layout)) {
		g_warning ("GnomeDbCustomLayout %p already assumed!", layout);
		return;
	}

	gnome_db_dict_declare_layout (dict, layout);

	dict->priv->assumed_layouts = g_slist_append (dict->priv->assumed_layouts, layout);
	g_object_ref (G_OBJECT (layout));
	gnome_db_base_connect_nullify (layout,
				 G_CALLBACK (layout_nullified_cb), dict);
	g_signal_connect (G_OBJECT (layout), "changed",
			  G_CALLBACK (updated_layout_cb), dict);

#ifdef debug_signal
	g_print (">> 'LAYOUT_ADDED' from gnome_db_dict_assume_layout\n");
#endif
	g_signal_emit (G_OBJECT (dict), gnome_db_dict_signals[LAYOUT_ADDED], 0, layout);
#ifdef debug_signal
	g_print ("<< 'LAYOUT_ADDED' from gnome_db_dict_assume_layout\n");
#endif
}

/**
 * gnome_db_dict_unassume_layout
 * @dict: a #GnomeDbDict object
 * @layout: a #GnomeDbCustomLayout object
 *
 * Forces @dict to lose a reference it has on @layout
 */
void
gnome_db_dict_unassume_layout (GnomeDbDict *dict, GnomeDbCustomLayout *layout)
{
	g_return_if_fail (dict && IS_GNOME_DB_DICT (dict));
	g_return_if_fail (dict->priv);

	if (g_slist_find (dict->priv->assumed_layouts, layout)) {
		dict->priv->assumed_layouts = g_slist_remove (dict->priv->assumed_layouts, layout);
		g_signal_handlers_disconnect_by_func (G_OBJECT (layout),
						      G_CALLBACK (layout_nullified_cb), dict);
		g_signal_handlers_disconnect_by_func (G_OBJECT (layout),
						      G_CALLBACK (updated_layout_cb), dict);
#ifdef debug_signal
		g_print (">> 'LAYOUT_REMOVED' from gnome_db_dict_unassume_layout\n");
#endif
		g_signal_emit (G_OBJECT (dict), gnome_db_dict_signals[LAYOUT_REMOVED], 0, layout);
#ifdef debug_signal
		g_print ("<< 'LAYOUT_REMOVED' from gnome_db_dict_unassume_layout\n");
#endif
		g_object_unref (G_OBJECT (layout));
	}
}

/**
 * gnome_db_dict_get_layouts
 * @dict: a #GnomeDbDict object
 *
 * Get a list of all the layouts managed by @dict
 *
 * Returns: a new list of #GnomeDbCustomLayout objects
 */
GSList *
gnome_db_dict_get_layouts (GnomeDbDict *dict)
{
	g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), NULL);
	g_return_val_if_fail (dict->priv, NULL);

	if (dict->priv->assumed_layouts)
		return g_slist_copy (dict->priv->assumed_layouts);
	else
		return NULL;
}

/**
 * gnome_db_dict_get_layout_by_xml_id
 * @dict: a #GnomeDbDict object
 * @xml_id: the XML Id of the layout being searched
 *
 * Find a #GnomeDbCustomLayout object from its XML Id
 *
 * Returns: the #GnomeDbCustomLayout object, or NULL if not found
 */
GnomeDbCustomLayout *
gnome_db_dict_get_layout_by_xml_id (GnomeDbDict *dict, const gchar *xml_id)
{
	GnomeDbCustomLayout *layout = NULL;

	GSList *list;
        gchar *str;

        g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), NULL);
        g_return_val_if_fail (dict->priv, NULL);

        list = dict->priv->all_layouts;
        while (list && !layout) {
                str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
                if (!strcmp (str, xml_id))
                        layout = GNOME_DB_CUSTOM_LAYOUT (list->data);
                g_free (str);
                list = g_slist_next (list);
        }

        return layout;
}


static void
layout_id_changed_cb (GnomeDbCustomLayout *layout, GnomeDbDict *dict)
{
	if (dict->priv->serial_layout <= gnome_db_base_get_id (GNOME_DB_BASE (layout)))
		dict->priv->serial_layout = gnome_db_base_get_id (GNOME_DB_BASE (layout)) + 1;
}

static void
layout_nullified_cb (GnomeDbCustomLayout *layout, GnomeDbDict *dict)
{
	gnome_db_dict_unassume_layout (dict, layout);
}

static void
layout_weak_ref_notify (GnomeDbDict *dict, GnomeDbCustomLayout *layout)
{
	dict->priv->all_layouts = g_slist_remove (dict->priv->all_layouts, layout);
	g_signal_handlers_disconnect_by_func (G_OBJECT (layout),
					      G_CALLBACK (layout_id_changed_cb), dict);
}

static void
updated_layout_cb (GnomeDbCustomLayout *layout, GnomeDbDict *dict)
{
#ifdef debug_signal
	g_print (">> 'LAYOUT_UPDATED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (dict), "layout_updated", layout);
#ifdef debug_signal
	g_print ("<< 'LAYOUT_UPDATED' from %s\n", __FUNCTION__);
#endif	
}


/**
 * gnome_db_dict_get_server
 * @dict: a #GnomeDbDict object
 *
 * Fetch a pointer to the GnomeDbServer used by the GnomeDbDict object.
 *
 * Returns: a pointer to the GnomeDbServer
 */
GnomeDbServer *
gnome_db_dict_get_server (GnomeDbDict *dict)
{
	g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), NULL);
	g_return_val_if_fail (dict->priv, NULL);

	return dict->priv->srv;
}

/**
 * gnome_db_dict_get_database
 * @dict: a #GnomeDbDict object
 *
 * Fetch a pointer to the GnomeDbDatabase used by the GnomeDbDict object.
 *
 * Returns: a pointer to the GnomeDbDatabase
 */
GnomeDbDatabase *
gnome_db_dict_get_database (GnomeDbDict *dict)
{
	g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), NULL);
	g_return_val_if_fail (dict->priv, NULL);

	return dict->priv->database;
}

#ifdef debug
/**
 * gnome_db_dict_dump
 * @dict: a #GnomeDbDict object
 *
 * Dumps the whole dictionary managed by the GnomeDbDict object
 */
void
gnome_db_dict_dump (GnomeDbDict *dict)
{
	GSList *list;

	g_return_if_fail (dict && IS_GNOME_DB_DICT (dict));
	g_return_if_fail (dict->priv);

	g_print ("\n----------------- DUMPING START -----------------\n");
	g_print (D_COL_H1 "GnomeDbDict %p\n" D_COL_NOR, dict);
	if (dict->priv->srv) 
		gnome_db_server_dump (dict->priv->srv, 0);

	if (dict->priv->database)
		gnome_db_base_dump (GNOME_DB_BASE (dict->priv->database), 0);

	list = dict->priv->assumed_queries;
	if (list)
		g_print ("Queries:\n");
	else
		g_print ("No Query defined\n");
	while (list) {
		if (!gnome_db_query_get_parent_query (GNOME_DB_QUERY (list->data)))
			gnome_db_base_dump (GNOME_DB_BASE (list->data), 0);
		list = g_slist_next (list);
	}

	list = dict->priv->assumed_graphs;
	if (list)
		g_print ("Graphs:\n");
	else
		g_print ("No Graph defined\n");
	while (list) {
		gnome_db_base_dump (GNOME_DB_BASE (list->data), 0);
		list = g_slist_next (list);
	}

	list = dict->priv->assumed_layouts;
	if (list)
		g_print ("Layouts:\n");
	else
		g_print ("No Layout defined\n");
	while (list) {
		gnome_db_base_dump (GNOME_DB_BASE (list->data), 0);
		list = g_slist_next (list);
	}

	g_print ("----------------- DUMPING END -----------------\n\n");
}
#endif


/**
 * gnome_db_dict_compute_xml_filename
 * @dict: a #GnomeDbDict object
 * @datasource: a data source
 * @app_id: an extra identification, or %NULL
 * @error: location to store error, or %NULL
 *
 * Get the prefered filename which represents the data dictionary associated to the @datasource data source.
 * Using the returned value in conjunction with gnome_db_dict_load_xml_file() and gnome_db_dict_save_xml_file() has
 * the advantage of letting the library handle file naming onventions.
 *
 * The @app_id argument allows to give an extra identification to the request, when some special features
 * must be saved but not interfere with the default dictionary.
 *
 * Returns: a new string
 */
gchar *
gnome_db_dict_compute_xml_filename (GnomeDbDict *dict, const gchar *datasource, const gchar *app_id, GError **error)
{
	gchar *str;
	gboolean with_error = FALSE;

/* REM: for now @dict is not directly used, but another version could make use of it, so we still keep it as
 *      an argument */

/* same as for libgda */
#define LIBGDA_USER_DICTIG_DIR G_DIR_SEPARATOR_S ".libgda"

	g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), NULL);
	g_return_val_if_fail (dict->priv, NULL);
	g_return_val_if_fail (datasource && *datasource, NULL);

	if (!app_id)
		str = g_strdup_printf ("%s%sDICT_%s_default.xml", g_get_home_dir (), LIBGDA_USER_DICTIG_DIR G_DIR_SEPARATOR_S,
				       datasource);
	else
		str = g_strdup_printf ("%s%sDICT_%s_%s.xml", g_get_home_dir (), LIBGDA_USER_DICTIG_DIR G_DIR_SEPARATOR_S,
				       datasource, app_id);

	/* create an empty file with that name */
	if (!g_file_test (str, G_FILE_TEST_EXISTS)) {
		gchar *dirpath;
		FILE *fp;
		
		dirpath = g_strdup_printf ("%s%s", g_get_home_dir (), LIBGDA_USER_DICTIG_DIR);
		if (!g_file_test (dirpath, G_FILE_TEST_IS_DIR)){
			if (mkdir (dirpath, 0700)) {
				g_set_error (error,
					     GNOME_DB_DICT_ERROR,
					     GNOME_DB_DICT_FILE_LOAD_ERROR,
					     _("Error creating directory %s"), dirpath);
				with_error = TRUE;
			}
		}
		g_free (dirpath);

		/* fp = fopen (str, "wt"); */
/* 		if (fp == NULL) { */
/* 			g_set_error (error, */
/* 				     GNOME_DB_DICT_ERROR, */
/* 				     GNOME_DB_DICT_FILE_LOAD_ERROR, */
/* 				     _("Unable to create the dictionary file %s"), str); */
/* 			with_error = TRUE; */
/* 		} */
/* 		else */
/* 			fclose (fp); */
	}

	if (with_error) {
		g_free (str);
		str = NULL;
	}

	return str;
}


/**
 * gnome_db_dict_set_xml_filename
 * @dict: a #GnomeDbDict object
 * @xmlfile: a file name
 *
 * Sets the filename @dict will use when gnome_db_dict_save_xml() and gnome_db_dict_load_xml() are called.
 */
void
gnome_db_dict_set_xml_filename (GnomeDbDict *dict, const gchar *xmlfile)
{
	g_return_if_fail (dict && IS_GNOME_DB_DICT (dict));
	g_return_if_fail (dict->priv);

	if (dict->priv->xml_filename) {
		g_free (dict->priv->xml_filename);
		dict->priv->xml_filename = NULL;
	}
	
	if (xmlfile)
		dict->priv->xml_filename = g_strdup (xmlfile);
}

/**
 * gnome_db_dict_get_xml_filename
 * @dict: a #GnomeDbDict object
 *
 * Get the filename @dict will use when gnome_db_dict_save_xml() and gnome_db_dict_load_xml() are called.
 *
 * Returns: the filename, or %NULL if none have been set.
 */
const gchar *
gnome_db_dict_get_xml_filename (GnomeDbDict *dict)
{
	g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), NULL);
	g_return_val_if_fail (dict->priv, NULL);

	return dict->priv->xml_filename;
}

/**
 * gnome_db_dict_load_xml
 * @dict: a #GnomeDbDict object
 * @error: location to store error, or %NULL
 * 
 * Loads an XML file which respects the Libgnomedb DTD, and creates all the necessary
 * objects that are defined within the XML file. During the creation of the other
 * objects, all the normal signals are emitted.
 *
 * If the GnomeDbDict object already has some contents, then it is first of all
 * nullified (to return its state as when it was first created).
 *
 * If an error occurs during loading then the GnomeDbDict object is left as empty
 * as when it is first created.
 *
 * The file loaded is the one specified using gnome_db_dict_set_xml_filename()
 *
 * Returns: TRUE if loading was successfull and FALSE otherwise.
 */
gboolean
gnome_db_dict_load_xml (GnomeDbDict *dict, GError **error)
{
	g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), FALSE);
	g_return_val_if_fail (dict->priv, FALSE);

	return gnome_db_dict_load_xml_file (dict, dict->priv->xml_filename, error);
}

/**
 * gnome_db_dict_save_xml
 * @dict: a #GnomeDbDict object
 * @error: location to store error, or %NULL
 *
 * Saves the contents of a GnomeDbDict object to a file which is specified using the
 * gnome_db_dict_set_xml_filename() method.
 *
 * Returns: TRUE if saving was successfull and FALSE otherwise.
 */
gboolean
gnome_db_dict_save_xml (GnomeDbDict *dict, GError **error)
{
	g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), FALSE);
	g_return_val_if_fail (dict->priv, FALSE);

	return gnome_db_dict_save_xml_file (dict, dict->priv->xml_filename, error);
}


/**
 * gnome_db_dict_get_entities_fk_constraints
 * @dict: a #GnomeDbDict object
 * @entity1: an object implementing the #GnomeDbEntity interface
 * @entity2: an object implementing the #GnomeDbEntity interface
 * @entity1_has_fk: TRUE if the returned constraints are the one for which @entity1 contains the foreign key
 *
 * Get a list of all the constraints which represent a foreign constrains, between
 * @entity1 and @entity2. If @entity1 and @entity2 are #GnomeDbTable objects, then the
 * constraints are the ones from the database.
 *
 * Constraints are represented as #GnomeDbConstraint objects.
 *
 * Returns: a new list of the constraints
 */
GSList *
gnome_db_dict_get_entities_fk_constraints (GnomeDbDict *dict, GnomeDbEntity *entity1, GnomeDbEntity *entity2,
				     gboolean entity1_has_fk)
{
	GSList *retval = NULL;

	g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), NULL);
	g_return_val_if_fail (dict->priv, NULL);
	g_return_val_if_fail (entity1 && IS_GNOME_DB_ENTITY (entity1), NULL);
	g_return_val_if_fail (entity2 && IS_GNOME_DB_ENTITY (entity2), NULL);
	if (entity1 == entity2)
		return NULL;

	if (IS_GNOME_DB_TABLE (entity1)) {
		if (IS_GNOME_DB_TABLE (entity2)) 
			retval = gnome_db_database_get_tables_fk_constraints (dict->priv->database, 
									GNOME_DB_TABLE (entity1), GNOME_DB_TABLE (entity2),
									entity1_has_fk);
	}

	return retval;
}
