/* gnome-db-query.h
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#ifndef __GNOME_DB_QUERY_H_
#define __GNOME_DB_QUERY_H_

#include "gnome-db-base.h"
#include "gnome-db-decl.h"
#include "gnome-db-enums.h"
#include <libgda/libgda.h>

G_BEGIN_DECLS

#define GNOME_DB_QUERY_TYPE          (gnome_db_query_get_type())
#define GNOME_DB_QUERY(obj)          G_TYPE_CHECK_INSTANCE_CAST (obj, gnome_db_query_get_type(), GnomeDbQuery)
#define GNOME_DB_QUERY_CLASS(klass)  G_TYPE_CHECK_CLASS_CAST (klass, gnome_db_query_get_type (), GnomeDbQueryClass)
#define IS_GNOME_DB_QUERY(obj)       G_TYPE_CHECK_INSTANCE_TYPE (obj, gnome_db_query_get_type ())

/* interfaces:
 * GnomeDbEntity
 * GnomeDbXmlStorage
 * GnomeDbReferer
 * GnomeDbRenderer
 */

/* error reporting */
extern GQuark gnome_db_query_error_quark (void);
#define GNOME_DB_QUERY_ERROR gnome_db_query_error_quark ()

/* different possible types for a query */
typedef enum {
        GNOME_DB_QUERY_TYPE_SELECT,
	GNOME_DB_QUERY_TYPE_INSERT,
	GNOME_DB_QUERY_TYPE_UPDATE,
	GNOME_DB_QUERY_TYPE_DELETE,
        GNOME_DB_QUERY_TYPE_UNION,
        GNOME_DB_QUERY_TYPE_INTERSECT,
	GNOME_DB_QUERY_TYPE_EXCEPT,
        GNOME_DB_QUERY_TYPE_NON_PARSED_SQL
} GnomeDbQueryType;

enum
{
	GNOME_DB_QUERY_XML_LOAD_ERROR,
	GNOME_DB_QUERY_META_DATA_UPDATE,
	GNOME_DB_QUERY_FIELDS_ERROR,
	GNOME_DB_QUERY_TARGETS_ERROR,
	GNOME_DB_QUERY_RENDER_ERROR,
	GNOME_DB_QUERY_PARSE_ERROR,
	GNOME_DB_QUERY_SYNTAX_ERROR,
	GNOME_DB_QUERY_STRUCTURE_ERROR,
	GNOME_DB_QUERY_SQL_ANALYSE_ERROR
};


/* struct for the object's data */
struct _GnomeDbQuery
{
	GnomeDbBase                object;
	GnomeDbQueryPrivate       *priv;
};

/* struct for the object's class */
struct _GnomeDbQueryClass
{
	GnomeDbBaseClass                    class;

	/* signals */
	void   (*type_changed)         (GnomeDbQuery *query);
	void   (*condition_changed)    (GnomeDbQuery *query);

	void   (*target_added)         (GnomeDbQuery *query, GnomeDbTarget *target);
	void   (*target_removed)       (GnomeDbQuery *query, GnomeDbTarget *target);
	void   (*target_updated)       (GnomeDbQuery *query, GnomeDbTarget *target);

	void   (*join_added)           (GnomeDbQuery *query, GnomeDbJoin *join);
	void   (*join_removed)         (GnomeDbQuery *query, GnomeDbJoin *join);
	void   (*join_updated)         (GnomeDbQuery *query, GnomeDbJoin *join);
	
	void   (*sub_query_added)      (GnomeDbQuery *query, GnomeDbQuery *sub_query);
	void   (*sub_query_removed)    (GnomeDbQuery *query, GnomeDbQuery *sub_query);
	void   (*sub_query_updated)    (GnomeDbQuery *query, GnomeDbQuery *sub_query);
};

guint             gnome_db_query_get_type               (void);
GObject          *gnome_db_query_new                    (GnomeDbDict *dict);
GObject          *gnome_db_query_new_copy               (GnomeDbQuery *orig, GHashTable *replacements);
GObject          *gnome_db_query_new_from_sql           (GnomeDbDict *dict, const gchar *sql, GError **error);

void              gnome_db_query_set_query_type         (GnomeDbQuery *query, GnomeDbQueryType type);
GnomeDbQueryType  gnome_db_query_get_query_type         (GnomeDbQuery *query);
const gchar      *gnome_db_query_get_query_type_string  (GnomeDbQuery *query);
gboolean          gnome_db_query_is_select_query        (GnomeDbQuery *query);
gboolean          gnome_db_query_is_modif_query         (GnomeDbQuery *query);
GnomeDbQuery     *gnome_db_query_get_parent_query       (GnomeDbQuery *query);
gboolean          gnome_db_query_is_equal_to            (GnomeDbQuery *query, GnomeDbQuery *compare_to);

/* if SQL queries */
void              gnome_db_query_set_sql_text           (GnomeDbQuery *query, const gchar *sql, GError **error);
const gchar      *gnome_db_query_get_sql_text           (GnomeDbQuery *query);

/* for other types of queries */
GSList           *gnome_db_query_get_sub_queries        (GnomeDbQuery *query);
void              gnome_db_query_add_sub_query          (GnomeDbQuery *query, GnomeDbQuery *sub_query);
void              gnome_db_query_del_sub_query          (GnomeDbQuery *query, GnomeDbQuery *sub_query);

void              gnome_db_query_add_param_source       (GnomeDbQuery *query, GnomeDbQuery *param_source);
void              gnome_db_query_del_param_source       (GnomeDbQuery *query, GnomeDbQuery *param_source);
const GSList     *gnome_db_query_get_param_sources      (GnomeDbQuery *query);

gboolean          gnome_db_query_add_target             (GnomeDbQuery *query, GnomeDbTarget *target, GError **error);
void              gnome_db_query_del_target             (GnomeDbQuery *query, GnomeDbTarget *target);
GSList           *gnome_db_query_get_targets            (GnomeDbQuery *query);
GnomeDbTarget    *gnome_db_query_get_target_by_xml_id   (GnomeDbQuery *query, const gchar *xml_id);
GnomeDbTarget    *gnome_db_query_get_target_by_alias    (GnomeDbQuery *query, const gchar *alias_or_name);
GSList           *gnome_db_query_get_target_pkfields    (GnomeDbQuery *query, GnomeDbTarget *target);

GSList           *gnome_db_query_get_joins              (GnomeDbQuery *query);
GnomeDbJoin      *gnome_db_query_get_join_by_targets    (GnomeDbQuery *query, GnomeDbTarget *target1, GnomeDbTarget *target2);
gboolean          gnome_db_query_add_join               (GnomeDbQuery *query, GnomeDbJoin *join);
void              gnome_db_query_del_join               (GnomeDbQuery *query, GnomeDbJoin *join);

void              gnome_db_query_set_condition          (GnomeDbQuery *query, GnomeDbCondition *cond);
GnomeDbCondition *gnome_db_query_get_condition          (GnomeDbQuery *query);
GSList           *gnome_db_query_get_main_conditions    (GnomeDbQuery *query);
void              gnome_db_query_append_condition       (GnomeDbQuery *query, GnomeDbCondition *cond, gboolean append_as_and);

void              gnome_db_query_set_order_by_field     (GnomeDbQuery *query, GnomeDbQfield *field, gint order, gboolean ascendant);
gint              gnome_db_query_get_order_by_field     (GnomeDbQuery *query, GnomeDbQfield *field, gboolean *ascendant);

/* utility functions */
GnomeDbQfield    *gnome_db_query_add_field_from_sql     (GnomeDbQuery *query, const gchar *field, GError **error);

GSList           *gnome_db_query_get_all_fields         (GnomeDbQuery *query);
GnomeDbQfield    *gnome_db_query_get_field_by_sql_naming        (GnomeDbQuery *query, const gchar *sql_naming);
GnomeDbQfield    *gnome_db_query_get_field_by_ref_field         (GnomeDbQuery *query, GnomeDbTarget *target, GnomeDbField *ref_field, 
								 GnomeDbFieldState field_state);
GnomeDbQfield    *gnome_db_query_get_first_field_for_target     (GnomeDbQuery *query, GnomeDbTarget *target);
GSList           *gnome_db_query_expand_all_field               (GnomeDbQuery *query, GnomeDbTarget *target);
void              gnome_db_query_order_fields_using_join_conds  (GnomeDbQuery *query);
GSList           *gnome_db_query_get_fields_by_target           (GnomeDbQuery *query, GnomeDbTarget *target, gboolean visible_fields_only);

void              gnome_db_query_declare_condition      (GnomeDbQuery *query, GnomeDbCondition *cond);
void              gnome_db_query_undeclare_condition    (GnomeDbQuery *query, GnomeDbCondition *cond);

G_END_DECLS

#endif
