/* gnome-db-base.c
 *
 * Copyright (C) 2003 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-base.h"
#include "marshal.h"
#include "gnome-db-dict.h"

/* 
 * Main static functions 
 */
static void gnome_db_base_class_init (GnomeDbBaseClass * class);
static void gnome_db_base_init (GnomeDbBase * srv);
static void gnome_db_base_dispose (GObject   * object);
static void gnome_db_base_finalize (GObject   * object);

static void gnome_db_base_set_property    (GObject              *object,
				    guint                 param_id,
				    const GValue         *value,
				    GParamSpec           *pspec);
static void gnome_db_base_get_property    (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	CHANGED,
	ID_CHANGED,
	NAME_CHANGED,
	DESCR_CHANGED,
	OWNER_CHANGED,
	NULLIFIED,
	LAST_SIGNAL
};

static gint gnome_db_base_signals[LAST_SIGNAL] = { 0, 0, 0, 0};

/* properties */
enum
{
	PROP_0,
	PROP_DICT,
	PROP_BLOCK_CHANGED
};


struct _GnomeDbBasePrivate
{
	GnomeDbDict            *dict;     /* property: NOT NULL to use GnomeDbBase object */
	guint              id;	     /* 0 or UNIQUE, comes from DictManager->id_serial */

	gchar             *name;
	gchar             *descr;
	gchar             *owner;
	
	gboolean           nullified;/* TRUE if the object has been "nullified" */
	gboolean           changed_locked;
};

guint
gnome_db_base_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbBaseClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_base_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbBase),
			0,
			(GInstanceInitFunc) gnome_db_base_init
		};
		
		type = g_type_register_static (G_TYPE_OBJECT, "GnomeDbBase", &info, 0);
	}
	return type;
}

static void
gnome_db_base_class_init (GnomeDbBaseClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_base_signals[CHANGED] =
		g_signal_new ("changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbBaseClass, changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	gnome_db_base_signals[NAME_CHANGED] =
		g_signal_new ("name_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbBaseClass, name_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	gnome_db_base_signals[ID_CHANGED] =
		g_signal_new ("id_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbBaseClass, id_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	gnome_db_base_signals[DESCR_CHANGED] =
		g_signal_new ("descr_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbBaseClass, descr_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	gnome_db_base_signals[OWNER_CHANGED] =
		g_signal_new ("owner_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbBaseClass, owner_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	gnome_db_base_signals[NULLIFIED] =
		g_signal_new ("nullified",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbBaseClass, nullified),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->changed = NULL;
	class->name_changed = gnome_db_base_changed;
	class->id_changed = gnome_db_base_changed;
	class->descr_changed = gnome_db_base_changed;
	class->owner_changed = gnome_db_base_changed;
	class->nullified = NULL;


	/* virtual functions */
	class->signal_changed = NULL;
#ifdef debug
	class->dump = NULL;
#endif

	object_class->dispose = gnome_db_base_dispose;
	object_class->finalize = gnome_db_base_finalize;

	/* Properties */
	object_class->set_property = gnome_db_base_set_property;
	object_class->get_property = gnome_db_base_get_property;
	g_object_class_install_property (object_class, PROP_DICT,
					 g_param_spec_pointer ("dict", NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_BLOCK_CHANGED,
                                         g_param_spec_boolean ("changed_blocked", NULL, NULL, FALSE,
                                                               G_PARAM_READABLE | G_PARAM_WRITABLE));
}

static void
gnome_db_base_init (GnomeDbBase * base)
{
	base->priv = g_new0 (GnomeDbBasePrivate, 1);
	base->priv->dict = NULL;
	base->priv->nullified = FALSE;
	base->priv->id = 0;

	base->priv->name = NULL;
	base->priv->descr = NULL;
	base->priv->owner = NULL;

	base->priv->changed_locked = FALSE;
}

/**
 * gnome_db_base_new
 *
 * Creates a new GnomeDbBase object. This object class is normally not instantiated
 * directly but through child classes objects' intantiation
 * 
 * Returns: the newly created object
 */
GObject   *
gnome_db_base_new ()
{
	GObject   *obj;

	GnomeDbBase *base;
	obj = g_object_new (GNOME_DB_BASE_TYPE, NULL);
	base = GNOME_DB_BASE (obj);

	return obj;
}


static void
gnome_db_base_dispose (GObject   * object)
{
	GnomeDbBase *base;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_BASE (object));

	base = GNOME_DB_BASE (object);
	if (base->priv) {
		if (! base->priv->nullified)
			gnome_db_base_nullify (base);

		if (base->priv->dict) {
			g_object_remove_weak_pointer (G_OBJECT (base->priv->dict),
						      (gpointer) & (base->priv->dict));
			base->priv->dict = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_base_finalize (GObject   * object)
{
	GnomeDbBase *base;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_BASE (object));

	base = GNOME_DB_BASE (object);
	if (base->priv) {
		if (! base->priv->nullified) {
			g_warning ("GnomeDbBase::finalize(%p) not nullified!\n", base);
		}

		if (base->priv->name)
			g_free (base->priv->name);
		if (base->priv->descr)
			g_free (base->priv->descr);
		if (base->priv->owner)
			g_free (base->priv->owner);

		g_free (base->priv);
		base->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_base_set_property (GObject              *object,
		      guint                 param_id,
		      const GValue         *value,
		      GParamSpec           *pspec)
{
	gpointer ptr;
	GnomeDbBase *base;

	base = GNOME_DB_BASE (object);
	if (base->priv) {
		switch (param_id) {
		case PROP_DICT:
			ptr = g_value_get_pointer (value);
			gnome_db_base_set_dict (base, GNOME_DB_DICT (ptr));
			break;
		case PROP_BLOCK_CHANGED:
			if (g_value_get_boolean (value))
				gnome_db_base_block_changed (base);
			else
				gnome_db_base_unblock_changed (base);
		}
	}
}

static void
gnome_db_base_get_property (GObject              *object,
		      guint                 param_id,
		      GValue               *value,
		      GParamSpec           *pspec)
{
	GnomeDbBase *base;

	base = GNOME_DB_BASE (object);
	if (base->priv) {
		switch (param_id) {
		case PROP_DICT:
			g_value_set_pointer (value, base->priv->dict);
			break;
		case PROP_BLOCK_CHANGED:
			g_value_set_boolean (value, base->priv->changed_locked);
		}
	}
}

/**
 * gnome_db_base_set_dict
 * @base: a #GnomeDbBase object
 * @dict: the #GnomeDbDict to which @base is attached
 * 
 * Sets the #GnomeDbDict to which @base is attached to. It has no other
 * action than to store it to easily be able to retreive it.
 */
void
gnome_db_base_set_dict (GnomeDbBase *base, GnomeDbDict *dict)
{
	g_return_if_fail (base && IS_GNOME_DB_BASE (base));
	g_return_if_fail (base->priv);
	g_return_if_fail (dict && IS_GNOME_DB_DICT (dict));

	if (base->priv->dict) {
		g_object_remove_weak_pointer (G_OBJECT (base->priv->dict),
					      (gpointer) & (base->priv->dict));
		base->priv->dict = NULL;
	}

	base->priv->dict = dict;
	g_object_add_weak_pointer (G_OBJECT (base->priv->dict),
				   (gpointer) & (base->priv->dict));
}

/**
 * gnome_db_base_get_dict
 * @base: a #GnomeDbBase object
 *
 * Fetch the corresponding #GnomeDbDict object.
 *
 * Returns: the #GnomeDbDict object to which @base is attached to
 */ 
GnomeDbDict *
gnome_db_base_get_dict (GnomeDbBase *base)
{
	g_return_val_if_fail (base && IS_GNOME_DB_BASE (base), NULL);
	g_return_val_if_fail (base->priv, NULL);

	return base->priv->dict;
}


/**
 * gnome_db_base_set_id
 * @base: a #GnomeDbBase object
 * @id: 
 * 
 * Sets the id of the object. The id is set to 0 by default (which means the object does
 * not use the id at all).
 * WARNING: the id is not checked which means no search is
 * made to see if the id is already used
 */
void
gnome_db_base_set_id (GnomeDbBase *base, guint id)
{
	g_return_if_fail (base && IS_GNOME_DB_BASE (base));
	g_return_if_fail (base->priv);

	if (base->priv->id != id) {
		base->priv->id = id;
#ifdef debug_signal
		g_print (">> 'ID_CHANGED' from %s %p\n", G_OBJECT_TYPE_NAME (base), base);
#endif
		g_signal_emit (G_OBJECT (base), gnome_db_base_signals[ID_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'ID_CHANGED' from %s %p\n", G_OBJECT_TYPE_NAME (base), base);
#endif
	}
}

/**
 * gnome_db_base_set_name
 * @base: a #GnomeDbBase object
 * @name: 
 *
 * Sets the name of the GnomeDbBase object. If the name is changed, then the 
 * "name_changed" signal is emitted.
 * 
 */
void
gnome_db_base_set_name (GnomeDbBase *base, const gchar *name)
{
	gboolean changed = FALSE;

	g_return_if_fail (base && IS_GNOME_DB_BASE (base));
	g_return_if_fail (base->priv);

	if (name) {
		if (base->priv->name) {
			changed = strcmp (base->priv->name, name) ? TRUE : FALSE;
			g_free (base->priv->name);
		}
		else
			changed = TRUE;
		base->priv->name = g_strdup (name);
	}

	if (changed) {
#ifdef debug_signal
		g_print (">> 'NAME_CHANGED' from %s %p\n", G_OBJECT_TYPE_NAME (base), base);
#endif
		g_signal_emit (G_OBJECT (base), gnome_db_base_signals[NAME_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'NAME_CHANGED' from %s %p\n", G_OBJECT_TYPE_NAME (base), base);
#endif
	}
}


/**
 * gnome_db_base_set_description
 * @base: a #GnomeDbBase object
 * @descr: 
 *
 * Sets the description of the GnomeDbBase object. If the description is changed, then the 
 * "descr_changed" signal is emitted.
 * 
 */
void
gnome_db_base_set_description (GnomeDbBase *base, const gchar *descr)
{
	gboolean changed = FALSE;

	g_return_if_fail (base && IS_GNOME_DB_BASE (base));
	g_return_if_fail (base->priv);

	if (descr) {
		if (base->priv->descr) {
			changed = strcmp (base->priv->descr, descr) ? TRUE : FALSE;
			g_free (base->priv->descr);
		}
		else
			changed = TRUE;
		base->priv->descr = g_strdup (descr);
	}

	if (changed) {
#ifdef debug_signal
		g_print (">> 'DESCR_CHANGED' from gnome_db_base_set_descr (%s)\n", base->priv->descr);
#endif
		g_signal_emit (G_OBJECT (base), gnome_db_base_signals[DESCR_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'DESCR_CHANGED' from gnome_db_base_set_descr (%s)\n", base->priv->descr);
#endif
	}
}


/**
 * gnome_db_base_set_owner
 * @base: a #GnomeDbBase object
 * @owner: 
 *
 * Sets the owner of the GnomeDbBase object. If the owner is changed, then the 
 * "owner_changed" signal is emitted.
 * 
 */
void
gnome_db_base_set_owner (GnomeDbBase *base, const gchar *owner)
{
	gboolean changed = FALSE;

	g_return_if_fail (base && IS_GNOME_DB_BASE (base));
	g_return_if_fail (base->priv);

	if (owner) {
		if (base->priv->owner) {
			changed = strcmp (base->priv->owner, owner) ? TRUE : FALSE;
			g_free (base->priv->owner);
		}
		else
			changed = TRUE;
		base->priv->owner = g_strdup (owner);
	}

	if (changed) {
#ifdef debug_signal
		g_print (">> 'OWNER_CHANGED' from gnome_db_base_set_descr (%s)\n", base->priv->descr);
#endif
		g_signal_emit (G_OBJECT (base), gnome_db_base_signals[OWNER_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'OWNER_CHANGED' from gnome_db_base_set_descr (%s)\n", base->priv->descr);
#endif
	}
}

/**
 * gnome_db_base_get_id
 * @base: a #GnomeDbBase object
 * 
 * Fetch the id of the GnomeDbBase object.
 * 
 * Returns: the id.
 */
guint
gnome_db_base_get_id (GnomeDbBase *base)
{
	g_return_val_if_fail (base && IS_GNOME_DB_BASE (base), 0);
	g_return_val_if_fail (base->priv, 0);
		
	return base->priv->id;
}


/**
 * gnome_db_base_get_name
 * @base: a #GnomeDbBase object
 * 
 * Fetch the name of the GnomeDbBase object.
 * 
 * Returns: the object's name.
 */
const gchar *
gnome_db_base_get_name (GnomeDbBase *base)
{
	g_return_val_if_fail (base && IS_GNOME_DB_BASE (base), NULL);
	g_return_val_if_fail (base->priv, NULL);

	return base->priv->name;
}


/**
 * gnome_db_base_get_description
 * @base: a #GnomeDbBase object
 * 
 * Fetch the description of the GnomeDbBase object.
 * 
 * Returns: the object's description.
 */
const gchar *
gnome_db_base_get_description (GnomeDbBase *base)
{
	g_return_val_if_fail (base && IS_GNOME_DB_BASE (base), NULL);
	g_return_val_if_fail (base->priv, NULL);

	return base->priv->descr;
}

/**
 * gnome_db_base_get_owner
 * @base: a #GnomeDbBase object
 * 
 * Fetch the owner of the GnomeDbBase object.
 * 
 * Returns: the object's owner.
 */
const gchar *
gnome_db_base_get_owner (GnomeDbBase *base)
{
	g_return_val_if_fail (base && IS_GNOME_DB_BASE (base), NULL);
	g_return_val_if_fail (base->priv, NULL);

	return base->priv->owner;
}



/**
 * gnome_db_base_nullify
 * @base: a #GnomeDbBase object
 *
 * Force the @base object to be destroyed, even if we don't have a reference on it
 * (we can't call g_object_unref() then) and even if the object is referenced
 * multiple times by other objects.
 *
 * The "nullified" signal is then emitted to tell the other reference holders that the object
 * must be destroyed and that they should give back their reference (using g_object_unref()).
 * However if a reference is still present, the object will not actually be destroyed and will
 * still be alive, but its state may not be deterministic.
 *
 * This is usefull because sometimes objects need to disappear even if they are referenced by other
 * objects. This usage is the same as with the gtk_widget_destroy() function on widgets.
 */
void
gnome_db_base_nullify (GnomeDbBase *base)
{
	g_return_if_fail (base && IS_GNOME_DB_BASE (base));
	g_return_if_fail (base->priv);

	if (!base->priv->nullified) {
		GnomeDbBaseClass *class;
		class = GNOME_DB_BASE_CLASS (G_OBJECT_GET_CLASS (base));
		
		base->priv->nullified = TRUE;
#ifdef debug_signal
		g_print (">> 'NULLIFIED' from %s %p\n", G_OBJECT_TYPE_NAME (base), base);
#endif
		g_signal_emit (G_OBJECT (base), gnome_db_base_signals[NULLIFIED], 0);
#ifdef debug_signal
		g_print ("<< 'NULLIFIED' from %p\n", base);
#endif
	}
	else 
		g_warning ("GnomeDbBase::nullify called on already nullified object %p, "
			   "of type %s (refcount=%d)\n", base, G_OBJECT_TYPE_NAME (base), G_OBJECT (base)->ref_count);
}

/**
 * gnome_db_base_nullify_check
 * @base: a #GnomeDbBase object
 *
 * Checks that the object has been nullified, and if not, then calls gnome_db_base_nullify() on it.
 * This is usefull for objects inheriting the #GnomeDbBase object to be called first line in their
 * dispose() method.
 */
void
gnome_db_base_nullify_check (GnomeDbBase *base)
{
	g_return_if_fail (base && IS_GNOME_DB_BASE (base));

	if (base->priv && !base->priv->nullified)
		gnome_db_base_nullify (base);
}

/**
 * gnome_db_base_nullified
 * @base: a #GnomeDbBase object
 *
 * Returns:
 */
gboolean
gnome_db_base_nullified (GnomeDbBase *base)
{
	g_return_val_if_fail (base && IS_GNOME_DB_BASE (base), TRUE);
	g_return_val_if_fail (base->priv, TRUE);
	
	return base->priv->nullified;
}

/**
 * gnome_db_base_changed
 * @base: a #GnomeDbBase object
 *
 * Force emission of the "changed" signal, except if gnome_db_base_block_changed() has
 * been called.
 */
void
gnome_db_base_changed (GnomeDbBase *base)
{
	g_return_if_fail (base && IS_GNOME_DB_BASE (base));
	g_return_if_fail (base->priv);

	if (base->priv->changed_locked)
		return;

#ifdef debug_signal
	g_print (">> 'CHANGED' from %s %p\n", G_OBJECT_TYPE_NAME (base), base);
#endif
	g_signal_emit (G_OBJECT (base), gnome_db_base_signals[CHANGED], 0);
#ifdef debug_signal
	g_print ("<< 'CHANGED' from %s %p\n", G_OBJECT_TYPE_NAME (base), base);
#endif
}

/**
 * gnome_db_base_block_changed
 * @base: a #GnomeDbBase object
 *
 * No "changed" signal will be emitted.
 */
void
gnome_db_base_block_changed (GnomeDbBase *base)
{
	GnomeDbBaseClass *class;

	g_return_if_fail (base && IS_GNOME_DB_BASE (base));
	g_return_if_fail (base->priv);

	base->priv->changed_locked = TRUE;

	class = GNOME_DB_BASE_CLASS (G_OBJECT_GET_CLASS (base));
	if (class->signal_changed)
		(*class->signal_changed) (base, TRUE);
}

/**
 * gnome_db_base_unblock_changed
 * @base: a #GnomeDbBase object
 *
 * The "changed" signal will again be emitted.
 */
void
gnome_db_base_unblock_changed (GnomeDbBase *base)
{
	GnomeDbBaseClass *class;

	g_return_if_fail (base && IS_GNOME_DB_BASE (base));
	g_return_if_fail (base->priv);

	base->priv->changed_locked = FALSE;

	class = GNOME_DB_BASE_CLASS (G_OBJECT_GET_CLASS (base));
	if (class->signal_changed)
		(*class->signal_changed) (base, FALSE);
}

#ifdef debug
/**
 * gnome_db_base_dump
 * @base: a #GnomeDbBase object
 * @offset: the offset (in caracters) at which the dump will start
 *
 * Writes a textual description of the object to STDOUT. This function only
 * exists if libergeant is compiled with the "--enable-debug" option. This is
 * a virtual function.
 */
void
gnome_db_base_dump (GnomeDbBase *base, guint offset)
{
	gchar *str;
	guint i;

	g_return_if_fail (base);
	g_return_if_fail (IS_GNOME_DB_BASE (base));

	/* string for the offset */
	str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;
	
	/* dump */
	if (base->priv) {
		GnomeDbBaseClass *class;
		class = GNOME_DB_BASE_CLASS (G_OBJECT_GET_CLASS (base));
		if (class->dump)
			(class->dump) (base, offset);
		else 
			g_print ("%s" D_COL_H1 "%s %p (name=%s, id=%d)\n" D_COL_NOR, 
				 str, G_OBJECT_TYPE_NAME (base), base, base->priv->name,
				 base->priv->id);
	}
	else
		g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, base);

	g_free (str);
}
#endif
