/* The Cairo CSS Drawing Library.
 * Copyright (C) 2008 Robert Staudinger
 *
 * This  library is free  software; you can  redistribute it and/or
 * modify it  under  the terms  of the  GNU Lesser  General  Public
 * License  as published  by the Free  Software  Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed  in the hope that it will be useful,
 * but  WITHOUT ANY WARRANTY; without even  the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License  along  with  this library;  if not,  write to  the Free
 * Software Foundation, Inc., 51  Franklin St, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include <string.h>
#include "ccd-background.h"

ccd_background_t * 
ccd_background_new (void)
{
	return g_new0 (ccd_background_t, 1);
}

void
ccd_background_free (ccd_background_t *self)
{
	g_free (self);
}

static bool
bg_color_parse (ccd_background_color_t	 *self,
		CRTerm const		**values)
{
	if (!*values) {
		return false;
	}

	self->spec = ccd_color_parse (&self->color, values);
	return self->spec == CCD_PROPERTY_SPEC_SET;
}

static bool
bg_image_parse (ccd_background_image_t	 *self,
		CRTerm const		**values)
{
	if (!*values) {
		return false;
	}

	self->spec = ccd_image_parse (&self->image, values);
	return self->spec == CCD_PROPERTY_SPEC_SET;
}

bool
ccd_background_parse (ccd_background_t	*self,
		      char const	*property,
		      CRTerm const	*values)
{
	bool ret;

	ret = false;

	if (0 == strcmp ("background", property)) {

		ret |= bg_color_parse (&self->bg_color, &values);
		ret |= bg_image_parse (&self->bg_image, &values);
		return ret;
	}

	if (0 == strcmp ("background-color", property)) {

		return bg_color_parse (&self->bg_color, &values);
	}

	if (0 == strcmp ("background-image", property)) {

		return bg_image_parse (&self->bg_image, &values);
	}

	return false;
}

void
ccd_background_draw (ccd_background_color_t const	*bg_color, 
		     ccd_background_image_t const	*bg_image,
		     cairo_t				*cr,
		     int32_t				 x,
		     int32_t				 y, 
		     int32_t				 width,
		     int32_t				 height)
{
	double dx;
	double dy;

	cairo_save (cr);

	if (bg_color && bg_color->spec != CCD_PROPERTY_SPEC_UNSET) {

		cairo_set_source_rgb (cr, bg_color->color.red, 
					  bg_color->color.green, 
					  bg_color->color.blue);
		cairo_rectangle (cr, x, y, width, height);
		cairo_fill (cr);
	}

	if (bg_image && bg_image->spec != CCD_PROPERTY_SPEC_UNSET) {

		cairo_pattern_t *pattern;

		dx = (double) width / 
		     cairo_image_surface_get_width (bg_image->image.surface);
		dy = (double) height /
		     cairo_image_surface_get_height (bg_image->image.surface);
		cairo_scale (cr, dx, dy);

		cairo_set_source_surface (cr, bg_image->image.surface,
					  x / dx, y / dy);

		pattern = cairo_get_source (cr);
		cairo_pattern_set_extend (pattern, CAIRO_EXTEND_PAD);

		cairo_rectangle (cr, x / dx, y / dy, width / dx, height / dy);
		cairo_fill (cr);
	}

	cairo_restore (cr);
}

void
ccd_background_dump (ccd_background_t const *self)
{
	printf (CCD_PROPERTY_DUMP_PREFIX "background-color: ");
	ccd_background_color_dump (&self->bg_color);
	printf (";\n");

	printf (CCD_PROPERTY_DUMP_PREFIX "background-image: ");
	ccd_background_image_dump (&self->bg_image);
	printf (";\n");
}

void
ccd_background_color_dump (ccd_background_color_t const *self)
{
	switch (self->spec) {
	case CCD_PROPERTY_SPEC_UNSET:
	case CCD_PROPERTY_SPEC_NONE:
		ccd_property_spec_dump (self->spec);
		break;
	case CCD_PROPERTY_SPEC_INHERIT:
		ccd_property_spec_dump (self->spec);
		/* fall thru */
	case CCD_PROPERTY_SPEC_SET:
		ccd_color_dump (&self->color);
		break;
	}
}

void
ccd_background_image_dump (ccd_background_image_t const *self)
{
	switch (self->spec) {
	case CCD_PROPERTY_SPEC_UNSET:
	case CCD_PROPERTY_SPEC_NONE:
		ccd_property_spec_dump (self->spec);
		break;
	case CCD_PROPERTY_SPEC_INHERIT:
		ccd_property_spec_dump (self->spec);
		/* fall thru */
	case CCD_PROPERTY_SPEC_SET:
		ccd_image_dump (&self->image);
		break;
	}
}

