/* The Cairo CSS Drawing Library.
 * Copyright (C) 2008 Robert Staudinger
 *
 * This  library is free  software; you can  redistribute it and/or
 * modify it  under  the terms  of the  GNU Lesser  General  Public
 * License  as published  by the Free  Software  Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed  in the hope that it will be useful,
 * but  WITHOUT ANY WARRANTY; without even  the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License  along  with  this library;  if not,  write to  the Free
 * Software Foundation, Inc., 51  Franklin St, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include <string.h>
#include "ccd-style.h"

/**
 * ccd_style_init:
 * @self:	a ccd_style_t.
 *
 * Reset the object's internal state.
 **/
void
ccd_style_init (ccd_style_t *self)
{
	memset (self, 0, sizeof (*self));
}

/**
 * ccd_style_draw_horizontal_line:
 * @self:	a ccd_style_t.
 * @cr:		the target to draw onto.
 * @x1:		the starting x coordinate.
 * @x2:		the ending x coordinate.
 * @y:		the y coordinate.
 *
 * Draw a horizontal line using this style instance. This takes the `top'
 * border from the CSS, if specified.
 **/
void
ccd_style_draw_line (ccd_style_t const	*self,
		     cairo_t		*cr, 
		     int32_t 		 x1,
		     int32_t 		 x2,
		     int32_t 		 y1,
		     int32_t		 y2)
{
	ccd_border_stroke_t const	*stroke;
	double				 off;

	if (y1 == y2) {
		/* Horizontal: try to use top, then bottom border. 
		 * Fallback based on flags is flawed, however. */
		stroke = self->top.flags ? &self->top : &self->bottom;
		if (stroke) {
			/* Unlike borders, lines are not drawn inside the box, 
			 * account for that. */
			off = stroke->width / 2.;

			ccd_border_draw (NULL, stroke, NULL, NULL, 
						cr, x1, y1 - off, x2 - x1, 0);

		}
	} else {
		/* Vertical: try to use left, then right border.
		 * Fallback based on flags is flawed, however. */
		stroke = self->left.flags ? &self->left : &self->right;
		if (stroke) {
			/* Unlike borders, lines are not drawn inside the box, 
			 * account for that. */
			off = stroke->width / 2.;

			ccd_border_draw (stroke, NULL, NULL, NULL, 
						cr, x1 - off, y1, 0, y2 - y1);
		}
	}

}

/**
 * ccd_style_draw_outline:
 * @self:	a ccd_style_t.
 * @cr:		the target to draw onto.
 * @x:		the starting x coordinate.
 * @y:		the starting y coordinate.
 * @width:	width of the outline to draw.
 * @height:	height of the outline to draw.
 *
 * Draw an outline using this style instance. Information about how to draw
 * this style's background is diregarded.
 **/
void
ccd_style_draw_outline (ccd_style_t const	*self, 
			cairo_t			*cr, 
			int32_t			 x,
			int32_t			 y,
			int32_t			 width,
			int32_t			 height)
{
	ccd_border_draw (&self->left, &self->top, &self->right, &self->bottom, 
				cr, x, y, width, height);
}

/**
 * ccd_style_draw_rectangle:
 * @self:	a ccd_style_t.
 * @cr:		the target to draw onto.
 * @x:		the starting x coordinate.
 * @y:		the starting y coordinate.
 * @width:	width of the outline to draw.
 * @height:	height of the outline to draw.
 *
 * Draw a rectangle using this style instance.
 **/
void
ccd_style_draw_rectangle (ccd_style_t const	*self,
			  cairo_t		*cr, 
			  int32_t		 x,
			  int32_t		 y,
			  int32_t		 width,
			  int32_t		 height)
{
	ccd_background_draw (self->bg_color, self->bg_image,
			     cr, x, y, width, height);

	ccd_border_draw (&self->left, &self->top, &self->right, &self->bottom, 
			 cr, x, y, width, height);
}

/**
 * ccd_style_dump:
 * @self:	a ccd_style_t.
 *
 * Print informations about the internal state of this object.
 **/
void
ccd_style_dump (ccd_style_t const *self)
{
	if (self->bg_color)
		ccd_background_color_dump (self->bg_color);
	if (self->bg_image)
		ccd_background_image_dump (self->bg_image);
	if (self->left.flags)
		ccd_border_stroke_dump (&self->left);
	if (self->top.flags)
		ccd_border_stroke_dump (&self->top);
	if (self->right.flags)
		ccd_border_stroke_dump (&self->right);
	if (self->bottom.flags)
		ccd_border_stroke_dump (&self->bottom);
}

