/* vi:set ts=8 sts=0 sw=8:
 * $Id: project.c,v 1.12 1998/12/06 17:40:33 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <gtk/gtk.h>
#include "win.h"
#include "doc.h"
#include "file.h"
#include "msgbar.h"
#include "msgbox.h"
#include "recent.h"
#include "prjbar.h"
#include "project.h"

#ifdef WANT_PROJECT

/*** local types ***/
#define PRJ_HEADER	":GNP+ prj file:"


/*** local function prototypes ***/
static int prj_header_read(FILE *fp, win_t *w);


/*** global function defintions ***/
/*
 * PUBLIC: prj_close_cb
 *
 *
 */
void
prj_close_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)(cbdata);
	prj_t *prj;

	g_assert(w != NULL);
	doc_close_all_cb(wgt, cbdata);
	prj = (prj_t *)(w->prj);
	if (prj) {
		/* XXX - maybe save the project before closing? */
		g_free(prj->name);
		g_free(prj->path);
		g_free(prj);
		w->prj = NULL;
		prjbar_update(w);
	}
} /* prj_close_cb */


/*
 * PUBLIC: prj_open_cb
 *
 *
 */
void
prj_open_cb(GtkWidget *wgt, gpointer cbdata)
{
	doc_open_prj((win_t *)cbdata);
} /* prj_open_cb */


/*
 * PUBLIC: prj_open
 *
 * reads a project (.prj) file and opens all files of the project.
 */
void
prj_open(GtkWidget *wgt, gpointer cbdata)
{
	char line[MAXPATH], *fname, buf[MAXPATH];
	FILE *fp;
	int err, cnt;
	prj_t *prj;
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	fname = gtk_file_selection_get_filename(GTK_FILE_SELECTION(w->filesel));
	if (fname == NULL)
		goto out;

	if ((fp = fopen(fname, "r")) == NULL) {
		g_snprintf(line, MAXPATH,
				"prj_open: could not open '%s'", fname);
		perror(line);
		goto out;
	}

	err = file_lock(fname, fp, FALSE, FALSE, TRUE);
	prj = (prj_t *)g_malloc0(sizeof(prj_t));
	w->prj = prj;

	if ((err = prj_header_read(fp, w))) {
		g_snprintf(line, MAXPATH,
				"prj_open: could read header for '%s'", fname);
		perror(line);
		g_free(w->prj);
		w->prj = NULL;
		return;
	}

#if 0
	/*
	 * if there's only one document, and it's an empty "Untitled"
	 * doc, then remove it before adding the one we want
	 */
	doc_t *d;
	d = doc_current(w);
	if (w->numdoc == 1 && !d->changed && strcmp(d->fname, UNTITLED) == 0) {
		doc_close_common(w);
	}
#endif

	/* now read in the files */
	/* XXX - do something with cnt */
	gtk_widget_hide(w->nb);
	cnt = 0;
	while (fgets(line, MAXPATH, fp)) {
		cnt++;
		line[strlen(line) - 1] = '\0';	/* strip \n */
		if (line[0] != '/') {
			g_snprintf(buf, MAXPATH, "%s%s", prj->path, line);
			doc_new(w, buf, TRUE);
			recent_list_add(w, buf);
		} else {
			doc_new(w, line, TRUE);
			recent_list_add(w, line);
		}
	}
	gtk_widget_show(w->nb);

	err = file_unlock(fname, fp);
	fclose(fp);

	msgbox_printf("Project '%s' opened", prj->name);
	msgbar_printf(w, "Project '%s' opened", prj->name);
	prjbar_update(w);

out:
	gtk_widget_destroy(w->filesel);
	w->filesel = NULL;
} /* prj_open */


static int
prj_header_read(FILE *fp, win_t *w)
{
	char line[MAXPATH], *p;
	prj_t *prj = (prj_t *)(w->prj);

	if (fgets(line, MAXPATH, fp) == NULL) {
		perror("prj_open: could not read version info");
		goto out;
	}

	if ((p = strstr(line, PRJ_HEADER)) == NULL) {
		p = (char *)g_malloc(strlen(line) + 64);
		g_snprintf(p, strlen(line) + 64,
				"prj_open: version not found ('%s')", line);
		perror(p);
		g_free(p);
		goto out;
	}
	p += strlen(PRJ_HEADER);
	prj->_rev = atoi(p);

	if (fgets(line, MAXPATH, fp) == NULL) {
		perror("prj_open: could not read project name");
		goto out;
	}
	line[strlen(line) - 1] = '\0';	/* strip \n */
	for (p = line + 1; isspace((int)(*p)); p++)
		;
	prj->name = g_strdup(p);

	if (fgets(line, MAXPATH, fp) == NULL) {
		perror("prj_open: could get number of files");
		goto out;
	}
	p = line + 1;
	prj->numfiles = atoi(p);

	if (fgets(line, MAXPATH, fp) == NULL) {
		perror("prj_open: could get project version");
		goto out;
	}
	p = line + 1;
	prj->version = atoi(p);

	if (fgets(line, MAXPATH, fp) == NULL) {
		perror("prj_open: could not read project path");
		goto out;
	}
	for (p = line + 1; isspace((int)(*p)); p++)
		;
	prj->path = g_strdup(p);	 /* we want the extra char */
	prj->path[strlen(prj->path) - 1] = '\0';	/* strip \n */
	if (prj->path[strlen(prj->path) - 1] != '/')
		strcat(prj->path, "/");			/* append '/' */

	GNPDBG_PROJECT(("prj->_rev = %d\n", prj->_rev));
	GNPDBG_PROJECT(("prj->name = '%s'\n", prj->name));
	GNPDBG_PROJECT(("prj->numfiles = %d\n", prj->numfiles));
	GNPDBG_PROJECT(("prj->version = %d\n", prj->version));
	GNPDBG_PROJECT(("prj->path = '%s'\n", prj->path));

	return ESUCCESS;
out:
	gtk_widget_destroy(w->filesel);
	w->filesel = NULL;
	return errno;
} /* prj_header_read */


#endif	/* WANT_PROJECT */
/* the end */
