; Aisleriot - sol.scm
; Copyright (C) 1998 Jonathan Blandford <jrb@mit.edu>
;
; This game is free software; you can redistribute it and/or
; modify it under the terms of the GNU Library General Public
; License as published by the Free Software Foundation; either
; version 2 of the License, or (at your option) any later version.
;
; This library is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
; Library General Public License for more details.
;
; You should have received a copy of the GNU Library General Public
; License along with this library; if not, write to the Free
; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;; Constants:

(define jack 11)
(define queen 12)
(define king 13)
(define ace 1)

(define club 0)
(define diamond 1)
(define heart 2)
(define spade 3)

(define black 0)
(define red 1)

(define down 0)
(define right 1)

;; Global variables:

(define DECK '())

;; NEW-GAME PROCEDURES
; these may be used in game files during the new-game procedure.

; This procedure MUST be called at the start of the new-game procedure.
(define (initialize-playing-area)
  (reset-surface)
  (set! FLIP-COUNTER 0)
  (set! SLOTS 0)
  (set! HORIZPOS (get-horiz-start))
  (set! VERTPOS (get-vert-start))
  (set! IN-GAME #f)
  (set! MOVE '())
  (set! HISTORY '()))

; create a 52 card deck (puts list of cards into DECK)
(define (make-standard-deck)
  (if (= ace 14)
      (set! DECK (make-standard-deck-list-ace-high 2 club))
      (set! DECK (make-standard-deck-list-ace-low ace club))))

; create a double deck of 104 cards (puts list of cards into DECK)
(define (make-standard-double-deck)
  (if (= ace 14)
      (set! DECK (append (make-standard-deck-list-ace-high 2 club) (make-standard-deck-list-ace-high 2 club)))
      (set! DECK (append (make-standard-deck-list-ace-low ace club) (make-standard-deck-list-ace-low ace club)))))

; shuffle the card list in DECK
(define (shuffle-deck)
  (let* ((vec (list->vector DECK))
	 (len (vector-length vec)))
    (set! DECK (shuffle-deck-helper vec 0 len))))
					     
; The playing area is divided into slots, where cards can be placed.
; Each slot can hold any amount of cards.  The slots are identified 
; using numbers assigned in order of their creation. The deck of cards
; held in DECK should be assigned to one of the slots on creation.
; (You may then create another deck and place it in a later slot).
;
; The slots are added to the board from left to right until the
; add-carriage-return-slot procedure is called, which starts a new line.
; A space may be added using the add-blank-slot procedure. These false
; slots are not assigned identifiers.

(define (add-blank-slot)
  (get-and-increment-position))

(define (add-carriage-return-slot)
  (linefeed-position))

; The real slots come in three varieties:
; A slot in which only the topmost card is visible:
(define (add-normal-slot cards)
  (add-slot (set-tag! (new-slot cards 
				(list 'normal (get-and-increment-position))))))

; A slot in which all the cards are visible, arranged as an overlapped pile:
; (either proceeding to the right or down).
(define (add-extended-slot cards direction)
  (if (= right direction)
      (add-slot (set-tag! (new-slot cards 
				    (list 'expanded-right 
					  (get-and-increment-position)))))
      (add-slot (set-tag! (new-slot cards 
				    (list 'expanded 
					  (get-and-increment-position)))))))

; A slot in only the n topmost cards are visible:
(define (add-partially-extended-slot cards direction n)
  (if (= right direction)
      (add-slot (set-tag! (new-slot cards 
				    (list 'partially-expanded-right 
					  (get-and-increment-position) n))))
      (add-slot (set-tag! (new-slot cards 
				    (list 'partially-expanded 
					  (get-and-increment-position) n))))))

; Cards may be dealt off one slot (usually the one containing the deck)
; and onto a list of other slots using these procedures:
(define (deal-cards target-slot-id slot-list)
  (if (not (null? slot-list))
      (begin 
	(add-card! (car slot-list) (remove-card target-slot-id))
	(deal-cards target-slot-id (cdr slot-list)))))

(define (deal-cards-face-up target-slot-id slot-list)
  (if (not (null? slot-list))
      (begin 
	(add-card! (car slot-list) (make-visible (remove-card target-slot-id)))
	(deal-cards-face-up target-slot-id (cdr slot-list)))))

;; GENERAL GAME PROCEDURES
; these may be used in game files at any time.

;; Procedures that change slot contents:

; turn the top card of a slot over (face up to face down and vice versa)
(define (flip-top-card slot-id)
  (add-card! slot-id (flip-card (remove-card slot-id))))

; turn the top card of a slot face side up 
(define (make-visible-top-card slot-id)
  (add-card! slot-id (make-visible (remove-card slot-id))))

; add a card onto the top of a slot
(define (add-card! slot-id card)
  (set-cards! slot-id (cons card (get-cards slot-id))))

; add a list of cards onto the top of a slot
(define (add-cards! slot-id cards)
  (set-cards! slot-id (append cards (get-cards slot-id))))

; remove (and return) the top card from a slot
(define (remove-card slot-id)
  (let ((cards (get-cards slot-id)))
    (set-cards! slot-id (cdr cards))
    (car cards)))

;; Utilities

; deal a card from the stock-slot to the waste-slot.
; when the stock slot is empty than the waste slot will be flipped back
; onto the stock unless the flip limit has been reached.
; an optional forth argument indicates the number of cards to deal.
(define (flip-stock stock-slot waste-slot flip-limit . rest)
  (if (empty-slot? stock-slot)
      (and (< FLIP-COUNTER flip-limit)
	   (set! FLIP-COUNTER (+ 1 FLIP-COUNTER))
	   (flip-deck stock-slot waste-slot))
      (let loop ((i (if (null? rest) 1 (car rest))))
	(and (> i 0)
	     (not (empty-slot? stock-slot))
	     (add-card! waste-slot (flip-card (remove-card stock-slot)))
	     (loop (- i 1))))))

; turn the cards in the waste slot over and add them to the stock-slot.
(define (flip-deck stock-slot waste-slot)
  (and (not (empty-slot? waste-slot))
       (add-card! stock-slot (flip-card (remove-card waste-slot)))
       (flip-deck stock-slot waste-slot)))

;; Procedures for manipulating cards:

; NB: In order to use these procedures you must remove the cards 
;     from their slots and then replace them after applying the procedure 
;     (as in the make-top-card-visible procedure above)
(define (flip-card card)
  (list (car card) (cadr card) (not (caddr card))))

(define (make-visible card)
  (list (car card) (cadr card) #t))

;; Procedures that provide information only:

; card procedures
(define (is-visible? card)
  (caddr card))

(define (get-suit card) 
  (cadr card))

(define (get-color card)
  (cond ((eq? (get-suit card) club) black)
	((eq? (get-suit card) spade) black)
	((eq? (get-suit card) heart) red)
	((eq? (get-suit card) diamond) red)
	(#t "Unknown color")))

(define (get-value card) 
  (car card))

; slot procedures
(define (get-cards slot-id)
  (cadr (get-slot slot-id)))

(define (empty-slot? slot-id)
  (null? (get-cards slot-id)))

(define (get-top-card slot-id)
  (let ((cards (get-cards slot-id)))
    (if (null? cards)
	'()
	(car cards))))

;; Utilities - need more of these:
(define (cards-eq? card1 card2)
  (and (eq? (get-value card1) (get-value card2))
       (eq? (get-suit card1) (get-suit card2))))

(define (is-red? card)
  (eq? red (get-color card)))

(define (is-black? card)
  (eq? black (get-color card)))

; use to compare two cards when aces are treated as high:
(define (ace-high-order value)
  (remainder (+ 11 value) 13))

(define (check-same-suit-list card-list)
  (or (< (length card-list) 2)
      (and (= (get-suit (car card-list)) (get-suit (cadr card-list)))
	   (check-same-suit-list (cdr card-list)))))

(define (check-straight-descending-list card-list)
  (or (< (length card-list) 2)
      (and (= (get-value (car card-list)) (- (get-value (cadr card-list)) 1))
	   (check-straight-descending-list (cdr card-list)))))

; debugging aid:
(define (display-list . objs)
  (map display objs) (newline))

; hint procedures: these will be superceeded shortly.
(define (get-value-name value)
  (cond ((eq? value ace) "ace")
	((eq? value 2) "two")
	((eq? value 3) "three")
	((eq? value 4) "four")
	((eq? value 5) "five")
	((eq? value 6) "six")
	((eq? value 7) "seven")
	((eq? value 8) "eight")
	((eq? value 9) "nine")
	((eq? value 10) "ten")
	((eq? value jack) "jack")
	((eq? value queen) "queen")
	((eq? value king) "king")
	(#t "Unknown value")))

(define (get-suit-name suit)
  (cond ((eq? suit club) "clubs")
	((eq? suit spade) "spades")
	((eq? suit heart) "hearts")
	((eq? suit diamond) "diamonds")
	(#t "Unknown suit")))

(define (get-name card)
  (string-append (get-value-name (get-value card)) 
		 " of "
		 (get-suit-name (get-suit card))))

;; Deprecated procedures:
; probably better to stick with ace = 1 to avoid confusion.
(define (set-ace-low)  (set! ace 1))
(define (set-ace-high) (set! ace 14))

; use add-cards! instead - the name of this procedure is misleading
(define (move-n-cards! start-slot end-slot cards)
  (add-cards! end-slot cards))

; you probably do not want to use this as it throws away the cards:
(define (remove-n-cards slot-id n)
  (set-cards! slot-id (nthcdr n (get-cards slot-id))))

; use deal-cards instead, it is more consistent with the rest of the API.
(define (deal-cards-from-deck deck slot-list)
  (if (not (null? slot-list))
      (begin 
	(add-card! (car slot-list) (car deck))
	(deal-cards-from-deck (cdr deck) (cdr slot-list)))))

(define (deal-cards-face-up-from-deck deck slot-list)
  (if (not (null? slot-list))
      (begin 
	(add-card! (car slot-list) (make-visible (car deck)))
	(deal-cards-face-up-from-deck (cdr deck) (cdr slot-list)))))

;; INTERNAL procedures
; The procedures in the rest of this file should not be used by games!!!
; Perhaps they should be in a separate file ???

; global variables
(define FLIP-COUNTER 0)
(define SLOTS 0)
(define HORIZPOS 0)
(define VERTPOS 0)
(define MOVE '())
(define HISTORY '())
(define FUTURE '())
(define IN-GAME #f)
(define MOVE-MODE '())

; called from C:
(define (start-game)
  (set! IN-GAME #t))

; called from C:
(define (end-move)
  (set! HISTORY (cons MOVE HISTORY))
  (set! FUTURE '())
  (set! MOVE '()))

; called from C:
(define (undo)
  (if (not (null? HISTORY))
      (let ((move (car HISTORY)))
	(set! MOVE-MODE 'undo)
	(eval (cons 'begin move))
	(set! FUTURE (cons move FUTURE)) 
	(set! HISTORY (cdr HISTORY))))) 
	
; called from C:
(define (redo)
  (if (not (null? FUTURE))
      (let ((move (car FUTURE)))
	(set! MOVE-MODE 'redo)
	(eval (cons 'begin (reverse move)))
	(set! HISTORY (cons move HISTORY)) 
	(set! FUTURE (cdr FUTURE))))) 

(define (sub-move slot-id cards1 cards2)
  (if (eq? MOVE-MODE 'redo)
      (set-cards-c! slot-id cards1)
      (set-cards-c! slot-id cards2)))

; called from C:
(define (record-move slot-id new_cards old_cards)
  (set! MOVE (cons (list 'sub-move 
			 slot-id 
			 (list 'quote new_cards)
			 (list 'quote old_cards)) MOVE)))

; called from C:
(define (discard-move)
  (set! MOVE '()))

; Used directly by some naughty code so we have to do redo / undo at
; this level until that code is changed to use the proper API:
(define (set-cards! slot-id new_cards)
  (if IN-GAME 
      (record-move slot-id new_cards (get-cards slot-id)))
  (set-cards-c! slot-id new_cards))

(define (make-card value suit)
  (list value suit #f))

(define (make-standard-deck-list-ace-high value suit)
  (if (eq? ace value)
      (if (eq? spade suit)
	  (list (make-card ace spade))
	  (cons (make-card value suit) 
		(make-standard-deck-list-ace-high 2 (+ 1 suit))))
      (cons (make-card value suit) 
	    (make-standard-deck-list-ace-high (+ 1 value) suit))))

(define (make-standard-deck-list-ace-low value suit)
  (if (eq? king value)
      (if (eq? spade suit)
	  (list (make-card king spade))
	  (cons (make-card value suit) 
		(make-standard-deck-list-ace-low 1 (+ 1 suit))))
      (cons (make-card value suit) 
	    (make-standard-deck-list-ace-low (+ 1 value) suit))))

(define (shuffle-deck-helper deck ref1 len)
  (if (zero? len)
      '()
      (let* ((ref2 (+ ref1 (random len)))
	     (val-at-ref2 (vector-ref deck ref2)))
	(vector-set! deck ref2 (vector-ref deck ref1))
	(cons val-at-ref2 (shuffle-deck-helper deck (+ ref1 1) (- len 1))))))

(define (new-slot deck placement)
  (list #f deck placement))

(define (set-tag! slot)
  (set! SLOTS (+ 1 SLOTS))
  (cons (- SLOTS 1) (cdr slot)))

(define (get-and-increment-position)
  (let ((retval (list HORIZPOS VERTPOS)))
    (set! HORIZPOS (+ HORIZPOS (get-horiz-offset)))
    retval))

(define (linefeed-position)
  (set! HORIZPOS (get-horiz-start))
  (set! VERTPOS (+ VERTPOS (get-vert-offset))))

; common lisp procedure not provided in guile 1.3
(define (nthcdr n lst)
  (if (zero? n) lst (nthcdr (+ -1 n) (cdr lst))))
