/* Copyright 2014 Yorba Foundation
 *
 * This software is licensed under the GNU Lesser General Public License
 * (version 2.1 or later).  See the COPYING file in this distribution.
 */

namespace California.Toolkit {

/**
 * A caching read-ahead model for Gtk.Stack.
 *
 * StackModel allows for items of any type to be stored in sorted order and presented in a Gtk.Stack
 * via presentation Gtk.Widgets generated by the caller for each item.  Gtk.Stack (and
 * Gtk.Container) do not have a notion of ordering, so StackModel "fakes" a sense of ordering by
 * configuring the Gtk.Stack prior to each transition to make it look like one presentation widget
 * is spatially above/below or left/right to the widget being transitioned to.
 *
 * StackModel also caches presentation widgets.  A {@link TrimPresentationFromCache} callback can
 * be supplied to selectively remove widgets from the cache, while a
 * {@link EnsurePresentationInCache} callback can be supplied to enforce locality.
 *
 * If caching and read-ahead are used, the Gtk.Stack is probably not well-suited for a
 * Gtk.StackSwitcher, since items may come and go at almost any time.  It's for this reason that
 * {@link ModelPresentation} returns an id but not a title for the widget.
 *
 * @see Deck
 */

public class StackModel<G> : BaseObject {
    public const string PROP_STACK = "stack";
    public const string PROP_VISIBLE_ITEM = "visible-item";
    
    /**
     * Transition type for spatial transitions according to ordering.
     */
    public enum OrderedTransitionType {
        CROSSFADE,
        SLIDE_LEFT_RIGHT,
        SLIDE_UP_DOWN;
        
        /**
         * Returns the Gtk.StackTransitionType that matches the {@link OrderedTransitionType} for
         * the direction implied by the comparison result.
         *
         * Negative values are to the left or up, positive values are to the right or down.
         * There is no direction for crossfading.  Zero means equal, returning NONE unless the
         * ordered type is CROSSFADE.
         */
        public Gtk.StackTransitionType to_stack_transition(int compare) {
            if (compare == 0)
                return (this == CROSSFADE) ? Gtk.StackTransitionType.CROSSFADE : Gtk.StackTransitionType.NONE;
            
            switch (this) {
                case CROSSFADE:
                    return Gtk.StackTransitionType.CROSSFADE;
                
                case SLIDE_LEFT_RIGHT:
                    return (compare < 0) ? Gtk.StackTransitionType.SLIDE_LEFT : Gtk.StackTransitionType.SLIDE_RIGHT;
                
                case SLIDE_UP_DOWN:
                    return (compare < 0) ? Gtk.StackTransitionType.SLIDE_UP : Gtk.StackTransitionType.SLIDE_DOWN;
                
                default:
                    assert_not_reached();
            }
        }
    }
    
    /**
     * Callback to convert the item into a child widget for the {@link stack}.
     *
     * The callback may also return an identifier for the widget, which may be used to reference
     * it later in the stack.  Note that {@link StackModel} doesn't store or track this identifier.
     */
    public delegate Gtk.Widget ModelPresentation<G>(G item, out string? id);
    
    /**
     * Callback for determining if the presentation Gtk.Widget for an item should be kept in the
     * cache.
     *
     * Returns true if the widget associated with the item should be removed from the cache.
     * visible_item indicates which item is currently being presented to the user.
     */
    public delegate bool TrimPresentationFromCache<G>(G item, G? visible_item);
    
    /**
     * Callback for maintaining read-ahead presentation Gtk.Widgets in the cache.
     *
     * The caller should return a collection of items that should be introduced into the cache,
     * if not already present.  Presentation widgets will be generated for the items to ensure
     * they're ready for display.
     *
     * This is used as a read-ahead mechanism as well as a way for the caller to enforce cache
     * locality.  It can be used, for example, to guarantee that certain items are always stored
     * in the cache, such as a "home" page, as well as the next and previous ''n'' items.
     *
     * visible_item indicates which item is currently being presented to the user.
     */
    public delegate Gee.Collection<G>? EnsurePresentationInCache<G>(G? visible_item);
    
    /**
     * The Gtk.Stack the {@link StackModel} is backing.
     */
    public Gtk.Stack stack { get; private set; }
    
    /**
     * The current visible item in the {@link stack}.
     */
    public G? visible_item { get; private set; default = null; }
    
    private OrderedTransitionType ordered_transition_type;
    private unowned ModelPresentation<G> model_presentation;
    private unowned TrimPresentationFromCache<G>? trim_from_cache;
    private unowned EnsurePresentationInCache<G>? ensure_in_cache;
    private unowned CompareDataFunc<G>? comparator;
    private Gee.HashMap<G, Gtk.Widget?> items;
    private bool in_balance_cache = false;
    private bool stack_destroyed = false;
    private Scheduled? scheduled_balance_cache = null;
    
    public StackModel(Gtk.Stack stack,
        OrderedTransitionType ordered_transition_type,
        ModelPresentation<G> model_presentation,
        TrimPresentationFromCache<G>? trim_from_cache = null,
        EnsurePresentationInCache<G>? ensure_in_cache = null,
        CompareDataFunc<G>? comparator = null,
        owned Gee.HashDataFunc<G>? hash_func = null,
        owned Gee.EqualDataFunc<G>? equal_func = null) {
        
        this.stack = stack;
        this.ordered_transition_type = ordered_transition_type;
        this.model_presentation = model_presentation;
        this.trim_from_cache = trim_from_cache;
        this.ensure_in_cache = ensure_in_cache;
        this.comparator = comparator;
        
        items = new Gee.HashMap<G, Gtk.Widget?>((owned) hash_func, (owned) equal_func);
        
        stack.remove.connect(on_stack_removed);
        stack.notify["visible-child"].connect(on_stack_child_visible);
        stack.destroy.connect(on_stack_destroyed);
    }
    
    ~StackModel() {
        stack.remove.disconnect(on_stack_removed);
        stack.notify["visible-child"].disconnect(on_stack_child_visible);
        stack.destroy.disconnect(on_stack_destroyed);
    }
    
    /**
     * Add the item to the {@link StackModel}.
     *
     * This will not necessarily make the item visible (in particular, only if the {@link stack}
     * is already empty).  Use {@link show_item} for that.
     *
     * Returns true if the item was added, false otherwise (already present).
     */
    public bool add(G item) {
        if (items.has_key(item))
            return false;
        
        items.set(item, null);
        
        // don't need to balance the cache; "visible-child" will do that automatically when
        // show() is called
        
        return true;
    }
    
    /**
     * Removes the item from the {@link StackModel}.
     *
     * If the item is already visible in the {@link stack}, the Gtk.Stack will itself determine
     * which widget will take its place.  If this is undesirable, call {@link show} ''before''
     * removing the item.
     *
     * Returns true if the item was removed, false otherwise (not present).
     */
    public bool remove(G item) {
        Gtk.Widget? presentation;
        if (!items.unset(item, out presentation))
            return false;
        
        // remove from stack, let "removed" signal handler do the rest
        if (presentation != null)
            presentation.destroy();
        
        return true;
    }
    
    /**
     * Clear all items from the {@link StackModel}.
     */
    public void clear() {
        foreach (G item in items.keys.to_array())
            remove(item);
    }
    
    /**
     * Show the item using the specified transition.
     *
     * If the item was not already present in {@link StackModel}, it will be added.
     *
     * @see add
     */
    public void show(G item) {
        add(item);
        
        Gtk.Widget presentation = ensure_presentation_exists(item);
        
        if (visible_item == null) {
            stack.transition_type = Gtk.StackTransitionType.NONE;
        } else {
            stack.transition_type = ordered_transition_type.to_stack_transition(
                item_comparator(visible_item, item));
        }
        
        stack.set_visible_child(presentation);
    }
    
    private void on_stack_removed(Gtk.Widget child) {
        // remove from cache, if present
        bool found = false;
        Gee.MapIterator<G, Gtk.Widget?> iter = items.map_iterator();
        while (iter.next()) {
            if (iter.get_value() == child) {
                found = true;
                iter.set_value(null);
                
                break;
            }
        }
        
        // only destroy widget if found (otherwise added externally from StackModel, so not ours
        // to break)
        if (found) {
            child.destroy();
            balance_cache("on_stack_removed");
        }
    }
    
    private void on_stack_child_visible() {
        if (stack.visible_child == null) {
            visible_item = null;
            
            return;
        }
        
        // find item for widget ... obviously for larger stacks a reverse mapping (perhaps with
        // get/set_data()) would be preferable, but this will do for now
        Gee.MapIterator<G, Gtk.Widget?> iter = items.map_iterator();
        while (iter.next()) {
            if (iter.get_value() == stack.visible_child) {
                visible_item = iter.get_key();
                
                // to avoid stutter, only balance the cache when the transition has completed,
                // which (apparently) it has not when this change is made (probably made at start
                // of transition, not the end) ... "transition-running" property would be useful
                // here, but that's not available until GTK 3.12
                scheduled_balance_cache = new Scheduled.once_at_idle(() => {
                    balance_cache("on_stack_child_visible");
                }, Priority.LOW);
                
                return;
            }
        }
        
        // nothing found
        visible_item = null;
    }
    
    private void on_stack_destroyed() {
        stack_destroyed = true;
    }
    
    private Gtk.Widget ensure_presentation_exists(G item) {
        Gtk.Widget? presentation = items.get(item);
        if (presentation != null)
            return presentation;
        
        // item -> presentation widget and identifier
        string? id;
        presentation = model_presentation(item, out id);
        presentation.show_all();
        
        // mappings
        items.set(item, presentation);
        
        // add to stack using identifier
        if (id != null)
            stack.add_named(presentation, id);
        else
            stack.add(presentation);
        
        return presentation;
    }
    
    private void balance_cache(string why) {
        // don't balance the cache if the stack is destroyed or if already balancing the cache
        if (stack_destroyed || in_balance_cache)
            return;
        
        in_balance_cache = true;
        
        // trim existing widgets from cache
        if (trim_from_cache != null) {
            // use a copy of the keys to avoid iterating over the map while it's being altered
            // (in particular, by one of our own signal handlers)
            foreach (G key in items.keys.to_array()) {
                Gtk.Widget? presentation = items[key];
                if (presentation != null && trim_from_cache(key, visible_item)) {
                    // set_value before removing from stack to prevent our signal handler from
                    // unsetting underneath us and causing iterator stamp problems
                    items[key] = null;
                    presentation.destroy();
                }
            }
        }
        
        // read-ahead (add any widgets the user requires)
        if (ensure_in_cache != null) {
            Gee.Collection<G>? ensure_items = ensure_in_cache(visible_item);
            if (ensure_items != null && ensure_items.size > 0) {
                foreach (G ensure_item in ensure_items)
                    ensure_presentation_exists(ensure_item);
            }
        }
        
        in_balance_cache = false;
    }
    
    private int item_comparator(G a, G b) {
        if (comparator != null)
            return comparator(a, b);
        
        return Gee.Functions.get_compare_func_for(typeof(G))(a, b);
    }
    
    public override string to_string() {
        return "StackModel (%d items)".printf(items.size);
    }
}

}

